/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.server.issue.index;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.OptionalLong;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import javax.annotation.CheckForNull;
import javax.annotation.Nullable;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.index.query.BoolQueryBuilder;
import org.elasticsearch.index.query.MatchAllQueryBuilder;
import org.elasticsearch.index.query.QueryBuilder;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.indices.TermsLookup;
import org.elasticsearch.search.aggregations.AbstractAggregationBuilder;
import org.elasticsearch.search.aggregations.AggregationBuilder;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.BucketOrder;
import org.elasticsearch.search.aggregations.HasAggregations;
import org.elasticsearch.search.aggregations.bucket.filter.FilterAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.filter.InternalFilter;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramAggregationBuilder;
import org.elasticsearch.search.aggregations.bucket.histogram.DateHistogramInterval;
import org.elasticsearch.search.aggregations.bucket.histogram.ExtendedBounds;
import org.elasticsearch.search.aggregations.bucket.terms.IncludeExclude;
import org.elasticsearch.search.aggregations.bucket.terms.InternalTerms;
import org.elasticsearch.search.aggregations.bucket.terms.StringTerms;
import org.elasticsearch.search.aggregations.bucket.terms.Terms;
import org.elasticsearch.search.aggregations.bucket.terms.TermsAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.max.InternalMax;
import org.elasticsearch.search.aggregations.metrics.min.Min;
import org.elasticsearch.search.aggregations.metrics.sum.SumAggregationBuilder;
import org.elasticsearch.search.aggregations.metrics.valuecount.InternalValueCount;
import org.elasticsearch.search.sort.FieldSortBuilder;
import org.joda.time.DateTimeZone;
import org.joda.time.Duration;
import org.joda.time.ReadableDuration;
import org.sonar.api.issue.Issue;
import org.sonar.api.rule.Severity;
import org.sonar.api.rules.RuleType;
import org.sonar.api.utils.System2;
import org.sonar.core.util.stream.MoreCollectors;
import org.sonar.db.rule.RuleDefinitionDto;
import org.sonar.server.es.BaseDoc;
import org.sonar.server.es.EsClient;
import org.sonar.server.es.EsUtils;
import org.sonar.server.es.IndexType;
import org.sonar.server.es.SearchOptions;
import org.sonar.server.es.Sorting;
import org.sonar.server.es.StickyFacetBuilder;
import org.sonar.server.issue.index.BranchStatistics;
import org.sonar.server.issue.index.IssueIndexDefinition;
import org.sonar.server.issue.index.IssueQuery;
import org.sonar.server.issue.index.ProjectStatistics;
import org.sonar.server.issue.index.SecurityStandardCategoryStatistics;
import org.sonar.server.permission.index.AuthorizationDoc;
import org.sonar.server.permission.index.WebAuthorizationTypeSupport;
import org.sonar.server.security.SecurityStandardHelper;
import org.sonar.server.user.UserSession;
import org.sonar.server.view.index.ViewIndexDefinition;

public class IssueIndex {
    public static final String FACET_PROJECTS = "projects";
    public static final String FACET_ASSIGNED_TO_ME = "assigned_to_me";
    private static final int DEFAULT_FACET_SIZE = 15;
    private static final int MAX_FACET_SIZE = 100;
    private static final String AGG_VULNERABILITIES = "vulnerabilities";
    private static final String AGG_SEVERITIES = "severities";
    private static final String AGG_TO_REVIEW_SECURITY_HOTSPOTS = "toReviewSecurityHotspots";
    private static final String AGG_IN_REVIEW_SECURITY_HOTSPOTS = "inReviewSecurityHotspots";
    private static final String AGG_REVIEWED_SECURITY_HOTSPOTS = "reviewedSecurityHotspots";
    private static final String AGG_CWES = "cwes";
    private static final BoolQueryBuilder NON_RESOLVED_VULNERABILITIES_FILTER = QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"type", (String)RuleType.VULNERABILITY.name())).mustNot((QueryBuilder)QueryBuilders.existsQuery((String)"resolution"));
    private static final BoolQueryBuilder IN_REVIEW_HOTSPOTS_FILTER = QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"type", (String)RuleType.SECURITY_HOTSPOT.name())).filter((QueryBuilder)QueryBuilders.termQuery((String)"status", (String)"IN_REVIEW")).mustNot((QueryBuilder)QueryBuilders.existsQuery((String)"resolution"));
    private static final BoolQueryBuilder TO_REVIEW_HOTSPOTS_FILTER = QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"type", (String)RuleType.SECURITY_HOTSPOT.name())).filter((QueryBuilder)QueryBuilders.termQuery((String)"status", (String)"TO_REVIEW")).mustNot((QueryBuilder)QueryBuilders.existsQuery((String)"resolution"));
    private static final BoolQueryBuilder REVIEWED_HOTSPOTS_FILTER = QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"type", (String)RuleType.SECURITY_HOTSPOT.name())).filter((QueryBuilder)QueryBuilders.termQuery((String)"status", (String)"REVIEWED")).filter((QueryBuilder)QueryBuilders.termQuery((String)"resolution", (String)"FIXED"));
    private static final String SUBSTRING_MATCH_REGEXP = ".*%s.*";
    private static final String FACET_SUFFIX_MISSING = "_missing";
    private static final String IS_ASSIGNED_FILTER = "__isAssigned";
    private static final SumAggregationBuilder EFFORT_AGGREGATION = (SumAggregationBuilder)AggregationBuilders.sum((String)"effort").field("effort");
    private static final BucketOrder EFFORT_AGGREGATION_ORDER = BucketOrder.aggregation((String)"effort", (boolean)false);
    private static final Duration TWENTY_DAYS = Duration.standardDays((long)20L);
    private static final Duration TWENTY_WEEKS = Duration.standardDays((long)140L);
    private static final Duration TWENTY_MONTHS = Duration.standardDays((long)600L);
    private static final String AGG_COUNT = "count";
    private final Sorting sorting;
    private final EsClient client;
    private final System2 system;
    private final UserSession userSession;
    private final WebAuthorizationTypeSupport authorizationTypeSupport;

    public IssueIndex(EsClient client, System2 system, UserSession userSession, WebAuthorizationTypeSupport authorizationTypeSupport) {
        this.client = client;
        this.system = system;
        this.userSession = userSession;
        this.authorizationTypeSupport = authorizationTypeSupport;
        this.sorting = new Sorting();
        this.sorting.add("STATUS", "status");
        this.sorting.add("SEVERITY", "severityValue");
        this.sorting.add("CREATION_DATE", "issueCreatedAt");
        this.sorting.add("UPDATE_DATE", "issueUpdatedAt");
        this.sorting.add("CLOSE_DATE", "issueClosedAt");
        this.sorting.add("FILE_LINE", "project");
        this.sorting.add("FILE_LINE", "filePath");
        this.sorting.add("FILE_LINE", "line");
        this.sorting.add("FILE_LINE", "severityValue").reverse();
        this.sorting.add("FILE_LINE", "key");
        this.sorting.addDefault("issueCreatedAt").reverse();
        this.sorting.addDefault("project");
        this.sorting.addDefault("filePath");
        this.sorting.addDefault("line");
        this.sorting.addDefault("key");
    }

    public SearchResponse search(IssueQuery query, SearchOptions options) {
        SearchRequestBuilder requestBuilder = this.client.prepareSearch(IssueIndexDefinition.TYPE_ISSUE.getMainType());
        this.configureSorting(query, requestBuilder);
        IssueIndex.configurePagination(options, requestBuilder);
        IssueIndex.configureRouting(query, options, requestBuilder);
        MatchAllQueryBuilder esQuery = QueryBuilders.matchAllQuery();
        BoolQueryBuilder esFilter = QueryBuilders.boolQuery();
        Map<String, QueryBuilder> filters = this.createFilters(query);
        for (QueryBuilder filter : filters.values()) {
            if (filter == null) continue;
            esFilter.must(filter);
        }
        if (esFilter.hasClauses()) {
            requestBuilder.setQuery((QueryBuilder)QueryBuilders.boolQuery().must((QueryBuilder)esQuery).filter((QueryBuilder)esFilter));
        } else {
            requestBuilder.setQuery((QueryBuilder)esQuery);
        }
        this.configureStickyFacets(query, options, filters, (QueryBuilder)esQuery, requestBuilder);
        requestBuilder.addAggregation((AggregationBuilder)EFFORT_AGGREGATION);
        requestBuilder.setFetchSource(false);
        return (SearchResponse)requestBuilder.get();
    }

    private static void configureRouting(IssueQuery query, SearchOptions options, SearchRequestBuilder requestBuilder) {
        Collection<String> uuids = query.projectUuids();
        if (!uuids.isEmpty() && options.getFacets().isEmpty()) {
            requestBuilder.setRouting((String[])uuids.stream().map(AuthorizationDoc::idOf).toArray(String[]::new));
        }
    }

    private static void configurePagination(SearchOptions options, SearchRequestBuilder esSearch) {
        esSearch.setFrom(options.getOffset()).setSize(options.getLimit());
    }

    private Map<String, QueryBuilder> createFilters(IssueQuery query) {
        HashMap<String, QueryBuilder> filters = new HashMap<String, QueryBuilder>();
        filters.put("__indexType", (QueryBuilder)QueryBuilders.termQuery((String)"indexType", (String)IssueIndexDefinition.TYPE_ISSUE.getName()));
        filters.put("__authorization", this.createAuthorizationFilter());
        if (BooleanUtils.isTrue((Boolean)query.assigned())) {
            filters.put(IS_ASSIGNED_FILTER, (QueryBuilder)QueryBuilders.existsQuery((String)"assignee"));
        } else if (BooleanUtils.isFalse((Boolean)query.assigned())) {
            filters.put(IS_ASSIGNED_FILTER, (QueryBuilder)QueryBuilders.boolQuery().mustNot((QueryBuilder)QueryBuilders.existsQuery((String)"assignee")));
        }
        String isResolved = "__isResolved";
        if (BooleanUtils.isTrue((Boolean)query.resolved())) {
            filters.put(isResolved, (QueryBuilder)QueryBuilders.existsQuery((String)"resolution"));
        } else if (BooleanUtils.isFalse((Boolean)query.resolved())) {
            filters.put(isResolved, (QueryBuilder)QueryBuilders.boolQuery().mustNot((QueryBuilder)QueryBuilders.existsQuery((String)"resolution")));
        }
        filters.put("key", IssueIndex.createTermsFilter("key", query.issueKeys()));
        filters.put("assignee", IssueIndex.createTermsFilter("assignee", query.assignees()));
        filters.put("language", IssueIndex.createTermsFilter("language", query.languages()));
        filters.put("tags", IssueIndex.createTermsFilter("tags", query.tags()));
        filters.put("type", IssueIndex.createTermsFilter("type", query.types()));
        filters.put("resolution", IssueIndex.createTermsFilter("resolution", query.resolutions()));
        filters.put("authorLogin", IssueIndex.createTermsFilter("authorLogin", query.authors()));
        filters.put("ruleId", IssueIndex.createTermsFilter("ruleId", query.rules().stream().map(RuleDefinitionDto::getId).collect(Collectors.toList())));
        filters.put("status", IssueIndex.createTermsFilter("status", query.statuses()));
        filters.put("organization", IssueIndex.createTermFilter("organization", query.organizationUuid()));
        filters.put("owaspTop10", IssueIndex.createTermsFilter("owaspTop10", query.owaspTop10()));
        filters.put("sansTop25", IssueIndex.createTermsFilter("sansTop25", query.sansTop25()));
        filters.put("cwe", IssueIndex.createTermsFilter("cwe", query.cwe()));
        IssueIndex.addSeverityFilter(query, filters);
        filters.put("sonarsourceSecurity", IssueIndex.createTermsFilter("sonarsourceSecurity", query.sonarsourceSecurity()));
        IssueIndex.addComponentRelatedFilters(query, filters);
        this.addDatesFilter(filters, query);
        IssueIndex.addCreatedAfterByProjectsFilter(filters, query);
        return filters;
    }

    private static void addSeverityFilter(IssueQuery query, Map<String, QueryBuilder> filters) {
        QueryBuilder severityFieldFilter = IssueIndex.createTermsFilter("severity", query.severities());
        if (severityFieldFilter != null) {
            filters.put("severity", (QueryBuilder)QueryBuilders.boolQuery().must(severityFieldFilter).mustNot((QueryBuilder)QueryBuilders.termQuery((String)"type", (String)RuleType.SECURITY_HOTSPOT.name())));
        }
    }

    private static void addComponentRelatedFilters(IssueQuery query, Map<String, QueryBuilder> filters) {
        IssueIndex.addCommonComponentRelatedFilters(query, filters);
        if (query.viewUuids().isEmpty()) {
            IssueIndex.addBranchComponentRelatedFilters(query, filters);
        } else {
            IssueIndex.addViewRelatedFilters(query, filters);
        }
    }

    private static void addCommonComponentRelatedFilters(IssueQuery query, Map<String, QueryBuilder> filters) {
        QueryBuilder componentFilter = IssueIndex.createTermsFilter("component", query.componentUuids());
        QueryBuilder projectFilter = IssueIndex.createTermsFilter("project", query.projectUuids());
        QueryBuilder moduleRootFilter = IssueIndex.createTermsFilter("modulePath", query.moduleRootUuids());
        QueryBuilder moduleFilter = IssueIndex.createTermsFilter("module", query.moduleUuids());
        QueryBuilder directoryFilter = IssueIndex.createTermsFilter("dirPath", query.directories());
        QueryBuilder fileFilter = IssueIndex.createTermsFilter("component", query.fileUuids());
        if (BooleanUtils.isTrue((Boolean)query.onComponentOnly())) {
            filters.put("component", componentFilter);
        } else {
            filters.put("project", projectFilter);
            filters.put("__module", moduleRootFilter);
            filters.put("module", moduleFilter);
            filters.put("dirPath", directoryFilter);
            filters.put("component", fileFilter != null ? fileFilter : componentFilter);
        }
    }

    private static void addBranchComponentRelatedFilters(IssueQuery query, Map<String, QueryBuilder> filters) {
        if (BooleanUtils.isTrue((Boolean)query.onComponentOnly())) {
            return;
        }
        QueryBuilder branchFilter = IssueIndex.createTermFilter("branch", query.branchUuid());
        filters.put("__is_main_branch", IssueIndex.createTermFilter("isMainBranch", Boolean.toString(query.isMainBranch())));
        filters.put("branch", branchFilter);
    }

    private static void addViewRelatedFilters(IssueQuery query, Map<String, QueryBuilder> filters) {
        boolean onApplicationBranch;
        if (BooleanUtils.isTrue((Boolean)query.onComponentOnly())) {
            return;
        }
        Collection<String> viewUuids = query.viewUuids();
        String branchUuid = query.branchUuid();
        boolean bl = onApplicationBranch = branchUuid != null && !viewUuids.isEmpty();
        if (onApplicationBranch) {
            filters.put("__view", IssueIndex.createViewFilter(Collections.singletonList(query.branchUuid())));
        } else {
            filters.put("__is_main_branch", IssueIndex.createTermFilter("isMainBranch", Boolean.toString(true)));
            filters.put("__view", IssueIndex.createViewFilter(viewUuids));
        }
    }

    @CheckForNull
    private static QueryBuilder createViewFilter(Collection<String> viewUuids) {
        if (viewUuids.isEmpty()) {
            return null;
        }
        BoolQueryBuilder viewsFilter = QueryBuilders.boolQuery();
        for (String viewUuid : viewUuids) {
            IndexType.IndexMainType mainType = ViewIndexDefinition.TYPE_VIEW;
            viewsFilter.should((QueryBuilder)QueryBuilders.termsLookupQuery((String)"branch", (TermsLookup)new TermsLookup(mainType.getIndex().getName(), mainType.getType(), viewUuid, FACET_PROJECTS)));
        }
        return viewsFilter;
    }

    private static AggregationBuilder addEffortAggregationIfNeeded(IssueQuery query, AggregationBuilder aggregation) {
        if (IssueIndex.hasQueryEffortFacet(query)) {
            aggregation.subAggregation((AggregationBuilder)EFFORT_AGGREGATION);
        }
        return aggregation;
    }

    private static boolean hasQueryEffortFacet(IssueQuery query) {
        return "effort".equals(query.facetMode());
    }

    private static AggregationBuilder createSeverityFacet(IssueQuery query, Map<String, QueryBuilder> filters, QueryBuilder queryBuilder) {
        String fieldName = Facet.SEVERITIES.getFieldName();
        String facetName = Facet.SEVERITIES.getName();
        StickyFacetBuilder stickyFacetBuilder = IssueIndex.newStickyFacetBuilder(query, filters, queryBuilder);
        BoolQueryBuilder facetFilter = stickyFacetBuilder.getStickyFacetFilter(new String[]{fieldName}).mustNot((QueryBuilder)QueryBuilders.termQuery((String)"type", (String)RuleType.SECURITY_HOTSPOT.name()));
        FilterAggregationBuilder facetTopAggregation = stickyFacetBuilder.buildTopFacetAggregation(fieldName, facetName, facetFilter, Facet.SEVERITIES.getSize());
        return AggregationBuilders.global((String)facetName).subAggregation((AggregationBuilder)facetTopAggregation);
    }

    private static AggregationBuilder createAssigneesFacet(IssueQuery query, Map<String, QueryBuilder> filters, QueryBuilder queryBuilder) {
        String fieldName = Facet.ASSIGNEES.getFieldName();
        String facetName = Facet.ASSIGNEES.getName();
        HashMap assigneeFilters = Maps.newHashMap(filters);
        assigneeFilters.remove(IS_ASSIGNED_FILTER);
        assigneeFilters.remove(fieldName);
        StickyFacetBuilder stickyFacetBuilder = IssueIndex.newStickyFacetBuilder(query, assigneeFilters, queryBuilder);
        BoolQueryBuilder facetFilter = stickyFacetBuilder.getStickyFacetFilter(new String[]{fieldName});
        FilterAggregationBuilder facetTopAggregation = stickyFacetBuilder.buildTopFacetAggregation(fieldName, facetName, facetFilter, Facet.ASSIGNEES.getSize());
        if (!query.assignees().isEmpty()) {
            facetTopAggregation = stickyFacetBuilder.addSelectedItemsToFacet(fieldName, facetName, facetTopAggregation, t -> t, query.assignees().toArray());
        }
        facetTopAggregation.subAggregation(IssueIndex.addEffortAggregationIfNeeded(query, (AggregationBuilder)AggregationBuilders.missing((String)(facetName + FACET_SUFFIX_MISSING)).field(fieldName)));
        return AggregationBuilders.global((String)facetName).subAggregation((AggregationBuilder)facetTopAggregation);
    }

    private static AggregationBuilder createResolutionFacet(IssueQuery query, Map<String, QueryBuilder> filters, QueryBuilder esQuery) {
        String fieldName = Facet.RESOLUTIONS.getFieldName();
        String facetName = Facet.RESOLUTIONS.getName();
        HashMap resolutionFilters = Maps.newHashMap(filters);
        resolutionFilters.remove("__isResolved");
        resolutionFilters.remove(fieldName);
        StickyFacetBuilder stickyFacetBuilder = IssueIndex.newStickyFacetBuilder(query, resolutionFilters, esQuery);
        BoolQueryBuilder facetFilter = stickyFacetBuilder.getStickyFacetFilter(new String[]{fieldName});
        FilterAggregationBuilder facetTopAggregation = stickyFacetBuilder.buildTopFacetAggregation(fieldName, facetName, facetFilter, Facet.RESOLUTIONS.getSize());
        facetTopAggregation = stickyFacetBuilder.addSelectedItemsToFacet(fieldName, facetName, facetTopAggregation, t -> t, new Object[0]);
        facetTopAggregation.subAggregation(IssueIndex.addEffortAggregationIfNeeded(query, (AggregationBuilder)AggregationBuilders.missing((String)(facetName + FACET_SUFFIX_MISSING)).field(fieldName)));
        return AggregationBuilders.global((String)facetName).subAggregation((AggregationBuilder)facetTopAggregation);
    }

    @CheckForNull
    private static QueryBuilder createTermsFilter(String field, Collection<?> values) {
        return values.isEmpty() ? null : QueryBuilders.termsQuery((String)field, values);
    }

    @CheckForNull
    private static QueryBuilder createTermFilter(String field, @Nullable String value) {
        return value == null ? null : QueryBuilders.termQuery((String)field, (String)value);
    }

    private void configureSorting(IssueQuery query, SearchRequestBuilder esRequest) {
        this.createSortBuilders(query).forEach(arg_0 -> ((SearchRequestBuilder)esRequest).addSort(arg_0));
    }

    private List<FieldSortBuilder> createSortBuilders(IssueQuery query) {
        String sortField = query.sort();
        if (sortField != null) {
            boolean asc = BooleanUtils.isTrue((Boolean)query.asc());
            return this.sorting.fill(sortField, asc);
        }
        return this.sorting.fillDefault();
    }

    private QueryBuilder createAuthorizationFilter() {
        return this.authorizationTypeSupport.createQueryFilter();
    }

    private void addDatesFilter(Map<String, QueryBuilder> filters, IssueQuery query) {
        Date createdAt;
        IssueQuery.PeriodStart createdAfter = query.createdAfter();
        Date createdBefore = query.createdBefore();
        this.validateCreationDateBounds(createdBefore, createdAfter != null ? createdAfter.date() : null);
        if (createdAfter != null) {
            filters.put("__createdAfter", (QueryBuilder)QueryBuilders.rangeQuery((String)"issueCreatedAt").from((Object)BaseDoc.dateToEpochSeconds((Date)createdAfter.date()), createdAfter.inclusive()));
        }
        if (createdBefore != null) {
            filters.put("__createdBefore", (QueryBuilder)QueryBuilders.rangeQuery((String)"issueCreatedAt").lt((Object)BaseDoc.dateToEpochSeconds((Date)createdBefore)));
        }
        if ((createdAt = query.createdAt()) != null) {
            filters.put("__createdAt", (QueryBuilder)QueryBuilders.termQuery((String)"issueCreatedAt", (long)BaseDoc.dateToEpochSeconds((Date)createdAt)));
        }
    }

    private static void addCreatedAfterByProjectsFilter(Map<String, QueryBuilder> filters, IssueQuery query) {
        Map<String, IssueQuery.PeriodStart> createdAfterByProjectUuids = query.createdAfterByProjectUuids();
        BoolQueryBuilder boolQueryBuilder = QueryBuilders.boolQuery();
        createdAfterByProjectUuids.forEach((projectUuid, createdAfterDate) -> boolQueryBuilder.should((QueryBuilder)QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"project", (String)projectUuid)).filter((QueryBuilder)QueryBuilders.rangeQuery((String)"issueCreatedAt").from((Object)BaseDoc.dateToEpochSeconds((Date)createdAfterDate.date()), createdAfterDate.inclusive()))));
        filters.put("createdAfterByProjectUuids", (QueryBuilder)boolQueryBuilder);
    }

    private void validateCreationDateBounds(@Nullable Date createdBefore, @Nullable Date createdAfter) {
        Preconditions.checkArgument((createdAfter == null || createdAfter.before(new Date(this.system.now())) ? 1 : 0) != 0, (Object)"Start bound cannot be in the future");
        Preconditions.checkArgument((createdAfter == null || createdBefore == null || createdAfter.before(createdBefore) ? 1 : 0) != 0, (Object)"Start bound cannot be larger or equal to end bound");
    }

    private void configureStickyFacets(IssueQuery query, SearchOptions options, Map<String, QueryBuilder> filters, QueryBuilder esQuery, SearchRequestBuilder esSearch) {
        if (!options.getFacets().isEmpty()) {
            StickyFacetBuilder stickyFacetBuilder = IssueIndex.newStickyFacetBuilder(query, filters, esQuery);
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.STATUSES, new Object[0]);
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.PROJECT_UUIDS, query.projectUuids().toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.MODULE_UUIDS, query.moduleUuids().toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.DIRECTORIES, query.directories().toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.FILE_UUIDS, query.fileUuids().toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.LANGUAGES, query.languages().toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.RULES, query.rules().stream().map(RuleDefinitionDto::getId).toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.AUTHORS, query.authors().toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.AUTHOR, query.authors().toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.TAGS, query.tags().toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.TYPES, query.types().toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.OWASP_TOP_10, query.owaspTop10().toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.SANS_TOP_25, query.sansTop25().toArray());
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.CWE, query.cwe().toArray());
            if (options.getFacets().contains(AGG_SEVERITIES)) {
                esSearch.addAggregation(IssueIndex.createSeverityFacet(query, filters, esQuery));
            }
            IssueIndex.addSimpleStickyFacetIfNeeded(options, stickyFacetBuilder, esSearch, Facet.SONARSOURCE_SECURITY, query.sonarsourceSecurity().toArray());
            if (options.getFacets().contains("resolutions")) {
                esSearch.addAggregation(IssueIndex.createResolutionFacet(query, filters, esQuery));
            }
            if (options.getFacets().contains("assignees")) {
                esSearch.addAggregation(IssueIndex.createAssigneesFacet(query, filters, esQuery));
            }
            if (options.getFacets().contains("createdAt")) {
                this.getCreatedAtFacet(query, filters, esQuery).ifPresent(arg_0 -> ((SearchRequestBuilder)esSearch).addAggregation(arg_0));
            }
            this.addAssignedToMeFacetIfNeeded(esSearch, options, query, filters, esQuery);
        }
    }

    private Optional<AggregationBuilder> getCreatedAtFacet(IssueQuery query, Map<String, QueryBuilder> filters, QueryBuilder esQuery) {
        boolean startInclusive;
        long startTime;
        IssueQuery.PeriodStart createdAfter = query.createdAfter();
        if (createdAfter == null) {
            OptionalLong minDate = this.getMinCreatedAt(filters, esQuery);
            if (!minDate.isPresent()) {
                return Optional.empty();
            }
            startTime = minDate.getAsLong();
            startInclusive = true;
        } else {
            startTime = createdAfter.date().getTime();
            startInclusive = createdAfter.inclusive();
        }
        Date createdBefore = query.createdBefore();
        long endTime = createdBefore == null ? this.system.now() : createdBefore.getTime();
        Duration timeSpan = new Duration(startTime, endTime);
        DateHistogramInterval bucketSize = DateHistogramInterval.YEAR;
        if (timeSpan.isShorterThan((ReadableDuration)TWENTY_DAYS)) {
            bucketSize = DateHistogramInterval.DAY;
        } else if (timeSpan.isShorterThan((ReadableDuration)TWENTY_WEEKS)) {
            bucketSize = DateHistogramInterval.WEEK;
        } else if (timeSpan.isShorterThan((ReadableDuration)TWENTY_MONTHS)) {
            bucketSize = DateHistogramInterval.MONTH;
        }
        DateHistogramAggregationBuilder dateHistogram = ((DateHistogramAggregationBuilder)((DateHistogramAggregationBuilder)((DateHistogramAggregationBuilder)AggregationBuilders.dateHistogram((String)Facet.CREATED_AT.getName()).field(Facet.CREATED_AT.getFieldName())).dateHistogramInterval(bucketSize).minDocCount(0L).format("yyyy-MM-dd'T'HH:mm:ssZ")).timeZone(DateTimeZone.forOffsetMillis((int)this.system.getDefaultTimeZone().getRawOffset()))).extendedBounds(new ExtendedBounds(Long.valueOf(startInclusive ? startTime : startTime + 1L), Long.valueOf(endTime - 1L)));
        IssueIndex.addEffortAggregationIfNeeded(query, (AggregationBuilder)dateHistogram);
        return Optional.of(dateHistogram);
    }

    private OptionalLong getMinCreatedAt(Map<String, QueryBuilder> filters, QueryBuilder esQuery) {
        String facetNameAndField = Facet.CREATED_AT.getFieldName();
        SearchRequestBuilder esRequest = this.client.prepareSearch(IssueIndexDefinition.TYPE_ISSUE.getMainType()).setSize(0);
        BoolQueryBuilder esFilter = QueryBuilders.boolQuery();
        filters.values().stream().filter(Objects::nonNull).forEach(arg_0 -> ((BoolQueryBuilder)esFilter).must(arg_0));
        if (esFilter.hasClauses()) {
            esRequest.setQuery((QueryBuilder)QueryBuilders.boolQuery().must(esQuery).filter((QueryBuilder)esFilter));
        } else {
            esRequest.setQuery(esQuery);
        }
        esRequest.addAggregation((AggregationBuilder)AggregationBuilders.min((String)facetNameAndField).field(facetNameAndField));
        Min minValue = (Min)((SearchResponse)esRequest.get()).getAggregations().get(facetNameAndField);
        double actualValue = minValue.getValue();
        if (Double.isInfinite(actualValue)) {
            return OptionalLong.empty();
        }
        return OptionalLong.of((long)actualValue);
    }

    private void addAssignedToMeFacetIfNeeded(SearchRequestBuilder builder, SearchOptions options, IssueQuery query, Map<String, QueryBuilder> filters, QueryBuilder queryBuilder) {
        String uuid = this.userSession.getUuid();
        if (!options.getFacets().contains(Facet.ASSIGNED_TO_ME.getName()) || StringUtils.isEmpty((String)uuid)) {
            return;
        }
        String fieldName = Facet.ASSIGNED_TO_ME.getFieldName();
        String facetName = Facet.ASSIGNED_TO_ME.getName();
        StickyFacetBuilder assignedToMeFacetBuilder = IssueIndex.newStickyFacetBuilder(query, filters, queryBuilder);
        BoolQueryBuilder facetFilter = assignedToMeFacetBuilder.getStickyFacetFilter(new String[]{IS_ASSIGNED_FILTER, fieldName});
        FilterAggregationBuilder facetTopAggregation = (FilterAggregationBuilder)AggregationBuilders.filter((String)(facetName + "__filter"), (QueryBuilder)facetFilter).subAggregation(IssueIndex.addEffortAggregationIfNeeded(query, (AggregationBuilder)((TermsAggregationBuilder)AggregationBuilders.terms((String)(facetName + "__terms")).field(fieldName)).includeExclude(new IncludeExclude(EsUtils.escapeSpecialRegexChars((String)uuid), null))));
        builder.addAggregation((AggregationBuilder)AggregationBuilders.global((String)facetName).subAggregation((AggregationBuilder)facetTopAggregation));
    }

    private static StickyFacetBuilder newStickyFacetBuilder(IssueQuery query, Map<String, QueryBuilder> filters, QueryBuilder esQuery) {
        if (IssueIndex.hasQueryEffortFacet(query)) {
            return new StickyFacetBuilder(esQuery, filters, (AbstractAggregationBuilder)EFFORT_AGGREGATION, EFFORT_AGGREGATION_ORDER);
        }
        return new StickyFacetBuilder(esQuery, filters);
    }

    private static void addSimpleStickyFacetIfNeeded(SearchOptions options, StickyFacetBuilder stickyFacetBuilder, SearchRequestBuilder esSearch, Facet facet, Object ... selectedValues) {
        if (options.getFacets().contains(facet.getName())) {
            esSearch.addAggregation(stickyFacetBuilder.buildStickyFacet(facet.getFieldName(), facet.getName(), facet.getSize(), selectedValues));
        }
    }

    public List<String> searchTags(IssueQuery query, @Nullable String textQuery, int size) {
        Terms terms = this.listTermsMatching("tags", query, textQuery, BucketOrder.key((boolean)true), size);
        return EsUtils.termsKeys((Terms)terms);
    }

    public Map<String, Long> countTags(IssueQuery query, int maxNumberOfTags) {
        Terms terms = this.listTermsMatching("tags", query, null, BucketOrder.count((boolean)false), maxNumberOfTags);
        return EsUtils.termsToMap((Terms)terms);
    }

    public List<String> searchAuthors(IssueQuery query, @Nullable String textQuery, int maxNumberOfAuthors) {
        Terms terms = this.listTermsMatching("authorLogin", query, textQuery, BucketOrder.key((boolean)true), maxNumberOfAuthors);
        return EsUtils.termsKeys((Terms)terms);
    }

    private Terms listTermsMatching(String fieldName, IssueQuery query, @Nullable String textQuery, BucketOrder termsOrder, int size) {
        SearchRequestBuilder requestBuilder = this.client.prepareSearch(IssueIndexDefinition.TYPE_ISSUE.getMainType()).setSize(0);
        requestBuilder.setQuery((QueryBuilder)QueryBuilders.boolQuery().must((QueryBuilder)QueryBuilders.matchAllQuery()).filter((QueryBuilder)this.createBoolFilter(query)));
        TermsAggregationBuilder aggreg = ((TermsAggregationBuilder)AggregationBuilders.terms((String)"_ref").field(fieldName)).size(size).order(termsOrder).minDocCount(1L);
        if (textQuery != null) {
            aggreg.includeExclude(new IncludeExclude(String.format(SUBSTRING_MATCH_REGEXP, EsUtils.escapeSpecialRegexChars((String)textQuery)), null));
        }
        SearchResponse searchResponse = (SearchResponse)requestBuilder.addAggregation((AggregationBuilder)aggreg).get();
        return (Terms)searchResponse.getAggregations().get("_ref");
    }

    private BoolQueryBuilder createBoolFilter(IssueQuery query) {
        BoolQueryBuilder boolQuery = QueryBuilders.boolQuery();
        for (QueryBuilder filter : this.createFilters(query).values()) {
            if (filter == null) continue;
            boolQuery.must(filter);
        }
        return boolQuery;
    }

    public List<ProjectStatistics> searchProjectStatistics(List<String> projectUuids, List<Long> froms, @Nullable String assigneeUuid) {
        Preconditions.checkState((projectUuids.size() == froms.size() ? 1 : 0) != 0, (String)"Expected same size for projectUuids (had size %s) and froms (had size %s)", (Object[])new Object[]{projectUuids.size(), froms.size()});
        if (projectUuids.isEmpty()) {
            return Collections.emptyList();
        }
        SearchRequestBuilder request = this.client.prepareSearch(IssueIndexDefinition.TYPE_ISSUE.getMainType()).setQuery((QueryBuilder)QueryBuilders.boolQuery().mustNot((QueryBuilder)QueryBuilders.existsQuery((String)"resolution")).filter((QueryBuilder)QueryBuilders.termQuery((String)"assignee", (String)assigneeUuid)).mustNot((QueryBuilder)QueryBuilders.termQuery((String)"type", (String)RuleType.SECURITY_HOTSPOT.name()))).setSize(0);
        IntStream.range(0, projectUuids.size()).forEach(i -> {
            String projectUuid = (String)projectUuids.get(i);
            long from = (Long)froms.get(i);
            request.addAggregation((AggregationBuilder)AggregationBuilders.filter((String)projectUuid, (QueryBuilder)QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)"project", (String)projectUuid)).filter((QueryBuilder)QueryBuilders.rangeQuery((String)"issueCreatedAt").gte((Object)BaseDoc.epochMillisToEpochSeconds((long)from)))).subAggregation((AggregationBuilder)((TermsAggregationBuilder)((TermsAggregationBuilder)AggregationBuilders.terms((String)"branchUuid").field("branch")).subAggregation((AggregationBuilder)AggregationBuilders.count((String)AGG_COUNT).field("key"))).subAggregation((AggregationBuilder)AggregationBuilders.max((String)"maxFuncCreatedAt").field("issueCreatedAt"))));
        });
        SearchResponse response = (SearchResponse)request.get();
        return (List)response.getAggregations().asList().stream().map(x -> (InternalFilter)x).flatMap(projectBucket -> ((StringTerms)projectBucket.getAggregations().get("branchUuid")).getBuckets().stream().flatMap(branchBucket -> {
            long count = ((InternalValueCount)branchBucket.getAggregations().get(AGG_COUNT)).getValue();
            if (count < 1L) {
                return Stream.empty();
            }
            long lastIssueDate = (long)((InternalMax)branchBucket.getAggregations().get("maxFuncCreatedAt")).getValue();
            return Stream.of(new ProjectStatistics(branchBucket.getKeyAsString(), count, lastIssueDate));
        })).collect(MoreCollectors.toList((int)projectUuids.size()));
    }

    public List<BranchStatistics> searchBranchStatistics(String projectUuid, List<String> branchUuids) {
        if (branchUuids.isEmpty()) {
            return Collections.emptyList();
        }
        SearchRequestBuilder request = this.client.prepareSearch(IssueIndexDefinition.TYPE_ISSUE.getMainType()).setRouting(AuthorizationDoc.idOf((String)projectUuid)).setQuery((QueryBuilder)QueryBuilders.boolQuery().must((QueryBuilder)QueryBuilders.termsQuery((String)"branch", branchUuids)).mustNot((QueryBuilder)QueryBuilders.existsQuery((String)"resolution")).must((QueryBuilder)QueryBuilders.termQuery((String)"isMainBranch", (String)Boolean.toString(false)))).setSize(0).addAggregation((AggregationBuilder)((TermsAggregationBuilder)AggregationBuilders.terms((String)"branchUuids").field("branch")).size(branchUuids.size()).subAggregation((AggregationBuilder)AggregationBuilders.terms((String)"types").field("type")));
        SearchResponse response = (SearchResponse)request.get();
        return (List)((StringTerms)response.getAggregations().get("branchUuids")).getBuckets().stream().map(bucket -> new BranchStatistics(bucket.getKeyAsString(), (Map)((StringTerms)bucket.getAggregations().get("types")).getBuckets().stream().collect(MoreCollectors.uniqueIndex(StringTerms.Bucket::getKeyAsString, InternalTerms.Bucket::getDocCount)))).collect(MoreCollectors.toList((int)branchUuids.size()));
    }

    public List<SecurityStandardCategoryStatistics> getSansTop25Report(String projectUuid, boolean isViewOrApp, boolean includeCwe) {
        SearchRequestBuilder request = this.prepareNonClosedVulnerabilitiesAndHotspotSearch(projectUuid, isViewOrApp);
        Stream.of("insecure-interaction", "risky-resource", "porous-defenses").forEach(sansCategory -> request.addAggregation(IssueIndex.createAggregation("sansTop25", sansCategory, includeCwe, Optional.of(SecurityStandardHelper.SANS_TOP_25_CWE_MAPPING))));
        return IssueIndex.processSecurityReportSearchResults(request, includeCwe);
    }

    public List<SecurityStandardCategoryStatistics> getSonarSourceReport(String projectUuid, boolean isViewOrApp, boolean includeCwe) {
        SearchRequestBuilder request = this.prepareNonClosedVulnerabilitiesAndHotspotSearch(projectUuid, isViewOrApp);
        Stream.concat(SecurityStandardHelper.SONARSOURCE_CWE_MAPPING.keySet().stream(), Stream.of("others")).forEach(sonarsourceCategory -> request.addAggregation(IssueIndex.createAggregation("sonarsourceSecurity", sonarsourceCategory, includeCwe, Optional.of(SecurityStandardHelper.SONARSOURCE_CWE_MAPPING))));
        return IssueIndex.processSecurityReportSearchResults(request, includeCwe);
    }

    public List<SecurityStandardCategoryStatistics> getOwaspTop10Report(String projectUuid, boolean isViewOrApp, boolean includeCwe) {
        SearchRequestBuilder request = this.prepareNonClosedVulnerabilitiesAndHotspotSearch(projectUuid, isViewOrApp);
        IntStream.rangeClosed(1, 10).mapToObj(i -> "a" + i).forEach(owaspCategory -> request.addAggregation(IssueIndex.createAggregation("owaspTop10", owaspCategory, includeCwe, Optional.empty())));
        return IssueIndex.processSecurityReportSearchResults(request, includeCwe);
    }

    private static AggregationBuilder createAggregation(String categoryField, String category, boolean includeCwe, Optional<Map<String, Set<String>>> categoryToCwesMap) {
        return IssueIndex.addSecurityReportSubAggregations((AggregationBuilder)AggregationBuilders.filter((String)category, (QueryBuilder)QueryBuilders.boolQuery().filter((QueryBuilder)QueryBuilders.termQuery((String)categoryField, (String)category))), includeCwe, categoryToCwesMap.map(m -> (Set)m.get(category)));
    }

    private static List<SecurityStandardCategoryStatistics> processSecurityReportSearchResults(SearchRequestBuilder request, boolean includeCwe) {
        SearchResponse response = (SearchResponse)request.get();
        return (List)response.getAggregations().asList().stream().map(c -> IssueIndex.processSecurityReportIssueSearchResults((InternalFilter)c, includeCwe)).collect(MoreCollectors.toList());
    }

    private static SecurityStandardCategoryStatistics processSecurityReportIssueSearchResults(InternalFilter categoryBucket, boolean includeCwe) {
        List<Object> children = new ArrayList<SecurityStandardCategoryStatistics>();
        if (includeCwe) {
            Stream stream = ((StringTerms)categoryBucket.getAggregations().get(AGG_CWES)).getBuckets().stream();
            children = stream.map(cweBucket -> IssueIndex.processSecurityReportCategorySearchResults((HasAggregations)cweBucket, cweBucket.getKeyAsString(), null)).collect(Collectors.toList());
        }
        return IssueIndex.processSecurityReportCategorySearchResults((HasAggregations)categoryBucket, categoryBucket.getName(), children);
    }

    private static SecurityStandardCategoryStatistics processSecurityReportCategorySearchResults(HasAggregations categoryBucket, String categoryName, @Nullable List<SecurityStandardCategoryStatistics> children) {
        List severityBuckets = ((StringTerms)((InternalFilter)categoryBucket.getAggregations().get(AGG_VULNERABILITIES)).getAggregations().get(AGG_SEVERITIES)).getBuckets();
        long vulnerabilities = severityBuckets.stream().mapToLong(b -> ((InternalValueCount)b.getAggregations().get(AGG_COUNT)).getValue()).sum();
        OptionalInt severityRating = severityBuckets.stream().filter(b -> ((InternalValueCount)b.getAggregations().get(AGG_COUNT)).getValue() != 0L).mapToInt(b -> Severity.ALL.indexOf(b.getKeyAsString()) + 1).max();
        long toReviewSecurityHotspots = ((InternalValueCount)((InternalFilter)categoryBucket.getAggregations().get(AGG_TO_REVIEW_SECURITY_HOTSPOTS)).getAggregations().get(AGG_COUNT)).getValue();
        long inReviewSecurityHotspots = ((InternalValueCount)((InternalFilter)categoryBucket.getAggregations().get(AGG_IN_REVIEW_SECURITY_HOTSPOTS)).getAggregations().get(AGG_COUNT)).getValue();
        long reviewedSecurityHotspots = ((InternalValueCount)((InternalFilter)categoryBucket.getAggregations().get(AGG_REVIEWED_SECURITY_HOTSPOTS)).getAggregations().get(AGG_COUNT)).getValue();
        return new SecurityStandardCategoryStatistics(categoryName, vulnerabilities, severityRating, inReviewSecurityHotspots, toReviewSecurityHotspots, reviewedSecurityHotspots, children);
    }

    private static AggregationBuilder addSecurityReportSubAggregations(AggregationBuilder categoriesAggs, boolean includeCwe, Optional<Set<String>> cwesInCategory) {
        AggregationBuilder aggregationBuilder = IssueIndex.addSecurityReportIssueCountAggregations(categoriesAggs);
        if (includeCwe) {
            TermsAggregationBuilder cwesAgg = ((TermsAggregationBuilder)AggregationBuilders.terms((String)AGG_CWES).field("cwe")).size(100);
            cwesInCategory.ifPresent(set -> cwesAgg.includeExclude(new IncludeExclude(set.toArray(new String[0]), new String[0])));
            categoriesAggs.subAggregation(IssueIndex.addSecurityReportIssueCountAggregations((AggregationBuilder)cwesAgg));
        }
        return aggregationBuilder;
    }

    private static AggregationBuilder addSecurityReportIssueCountAggregations(AggregationBuilder categoryAggs) {
        return categoryAggs.subAggregation((AggregationBuilder)AggregationBuilders.filter((String)AGG_VULNERABILITIES, (QueryBuilder)NON_RESOLVED_VULNERABILITIES_FILTER).subAggregation((AggregationBuilder)((TermsAggregationBuilder)AggregationBuilders.terms((String)AGG_SEVERITIES).field("severity")).subAggregation((AggregationBuilder)AggregationBuilders.count((String)AGG_COUNT).field("key")))).subAggregation((AggregationBuilder)AggregationBuilders.filter((String)AGG_TO_REVIEW_SECURITY_HOTSPOTS, (QueryBuilder)TO_REVIEW_HOTSPOTS_FILTER).subAggregation((AggregationBuilder)AggregationBuilders.count((String)AGG_COUNT).field("key"))).subAggregation((AggregationBuilder)AggregationBuilders.filter((String)AGG_IN_REVIEW_SECURITY_HOTSPOTS, (QueryBuilder)IN_REVIEW_HOTSPOTS_FILTER).subAggregation((AggregationBuilder)AggregationBuilders.count((String)AGG_COUNT).field("key"))).subAggregation((AggregationBuilder)AggregationBuilders.filter((String)AGG_REVIEWED_SECURITY_HOTSPOTS, (QueryBuilder)REVIEWED_HOTSPOTS_FILTER).subAggregation((AggregationBuilder)AggregationBuilders.count((String)AGG_COUNT).field("key")));
    }

    private SearchRequestBuilder prepareNonClosedVulnerabilitiesAndHotspotSearch(String projectUuid, boolean isViewOrApp) {
        BoolQueryBuilder componentFilter = QueryBuilders.boolQuery();
        if (isViewOrApp) {
            IndexType.IndexMainType mainType = ViewIndexDefinition.TYPE_VIEW;
            componentFilter.filter((QueryBuilder)QueryBuilders.termsLookupQuery((String)"branch", (TermsLookup)new TermsLookup(mainType.getIndex().getName(), mainType.getType(), projectUuid, FACET_PROJECTS)));
        } else {
            componentFilter.filter((QueryBuilder)QueryBuilders.termQuery((String)"branch", (String)projectUuid));
        }
        return this.client.prepareSearch(IssueIndexDefinition.TYPE_ISSUE.getMainType()).setQuery((QueryBuilder)componentFilter.should((QueryBuilder)NON_RESOLVED_VULNERABILITIES_FILTER).should((QueryBuilder)TO_REVIEW_HOTSPOTS_FILTER).should((QueryBuilder)IN_REVIEW_HOTSPOTS_FILTER).should((QueryBuilder)REVIEWED_HOTSPOTS_FILTER).minimumShouldMatch(1)).setSize(0);
    }

    public static enum Facet {
        SEVERITIES("severities", "severity", Severity.ALL.size()),
        STATUSES("statuses", "status", Issue.STATUSES.size()),
        RESOLUTIONS("resolutions", "resolution", Issue.RESOLUTIONS.size() + 1),
        TYPES("types", "type", RuleType.values().length),
        LANGUAGES("languages", "language", 100),
        RULES("rules", "ruleId", 100),
        TAGS("tags", "tags", 100),
        AUTHORS("authors", "authorLogin", 100),
        AUTHOR("author", "authorLogin", 100),
        PROJECT_UUIDS("projects", "project", 100),
        MODULE_UUIDS("moduleUuids", "module", 100),
        FILE_UUIDS("fileUuids", "component", 100),
        DIRECTORIES("directories", "dirPath", 100),
        ASSIGNEES("assignees", "assignee", 100),
        ASSIGNED_TO_ME("assigned_to_me", "assignee", 1),
        OWASP_TOP_10("owaspTop10", "owaspTop10", 15),
        SANS_TOP_25("sansTop25", "sansTop25", 15),
        CWE("cwe", "cwe", 15),
        CREATED_AT("createdAt", "issueCreatedAt", 15),
        SONARSOURCE_SECURITY("sonarsourceSecurity", "sonarsourceSecurity", 15);

        private final String name;
        private final String fieldName;
        private final int size;

        private Facet(String name, String fieldName, int size) {
            this.name = name;
            this.fieldName = fieldName;
            this.size = size;
        }

        public String getName() {
            return this.name;
        }

        public String getFieldName() {
            return this.fieldName;
        }

        public int getSize() {
            return this.size;
        }

        public static Facet of(String name) {
            return Arrays.stream(Facet.values()).filter(f -> f.getName().equals(name)).reduce((a, b) -> {
                throw new IllegalStateException("Multiple facets with same name: " + (Object)a + ", " + (Object)b);
            }).orElseThrow(() -> new IllegalArgumentException(String.format("Facet name '%s' hasn't been found", name)));
        }
    }
}

