/*
 * COPYRIGHT (c) International Business Machines Corp. 2011-2017
 *
 * This program is provided under the terms of the Common Public License,
 * version 1.0 (CPL-1.0). Any use, reproduction or distribution for this
 * software constitutes recipient's acceptance of CPL-1.0 terms which can be
 * found in the file LICENSE file or at
 * https://opensource.org/licenses/cpl1.0.php
 */

#include "pkcs11types.h"

#define MAX_KEY_SIZE  150
#define MAX_DATA_SIZE 512
#define MAX_HASH_SIZE 64
#define MAX_CHUNKS 8

CK_ULONG four = 4;
CK_ULONG ten = 10;
CK_ULONG sixteen = 16;
CK_ULONG twentyfour = 24;
CK_ULONG thirtytwo = 32;

struct DIGEST_TEST_VECTOR {
    CK_BYTE data[512];
    CK_ULONG data_len;
    CK_BYTE hash[512];
    CK_ULONG hash_len;
    int chunks[MAX_CHUNKS];
    int num_chunks;
};


/** The hash function RIPEMD-160
http://homes.esat.kuleuven.be/~bosselae/ripemd160.html
 **/
struct DIGEST_TEST_VECTOR ripemd128_digest_test_vector[] = {
    {
        .data = "",
        .data_len = 0,
        .hash = {0xcd,0xf2,0x62,0x13,0xa1,0x50,0xdc,0x3e,0xcb,
                 0x61,0x0f,0x18,0xf6,0xb3,0x8b,0x46},
        .hash_len = 16,
    }
};

/** The hash function RIPEMD-160
http://homes.esat.kuleuven.be/~bosselae/ripemd160.html
 **/
struct DIGEST_TEST_VECTOR ripemd160_digest_test_vector[] = {
    {
        .data = "",
        .data_len = 0,
        .hash =	{0x9c,0x11,0x85,0xa5,0xc5,0xe9,0xfc,0x54,0x61,
                 0x28,0x08,0x97,0x7e,0xe8,0xf5,0x48,0xb2,0x25,
                 0x8d,0x31},
        .hash_len = 20,
    }
};

/** The MD2 Message-Digest Algorithm
http://tools.ietf.org/html/rfc1319
A.5 Test suite
 **/
struct DIGEST_TEST_VECTOR md2_digest_test_vector[] = {
    {
        .data = "",
        .data_len = 0,
        .hash =	{0x83,0x50,0xe5,0xa3,0xe2,0x4c,0x15,0x3d,0xf2,
                 0x27,0x5c,0x9f,0x80,0x69,0x27,0x73},
        .hash_len = 16,
    }, {
        .data = "a",
        .data_len = 1,
        .hash =	{0x32,0xec,0x01,0xec,0x4a,0x6d,0xac,0x72,0xc0,
                 0xab,0x96,0xfb,0x34,0xc0,0xb5,0xd1},
        .hash_len = 16,
    }, {
        .data = "abc",
        .data_len = 3,
        .hash =	{0xda,0x85,0x3b,0x0d,0x3f,0x88,0xd9,0x9b,0x30,
                 0x28,0x3a,0x69,0xe6,0xde,0xd6,0xbb},
        .hash_len = 16,
    }, {
        .data = "message digest",
        .data_len = 14,
        .hash =	{0xab,0x4f,0x49,0x6b,0xfb,0x2a,0x53,0x0b,0x21,
                 0x9f,0xf3,0x30,0x31,0xfe,0x06,0xb0},
        .hash_len = 20,
    }, {
        .data = "abcdefghijklmnopqrstuvwxyz",
        .data_len = 26,
        .hash = {0x4e,0x8d,0xdf,0xf3,0x65,0x02,0x92,0xab,0x5a,
                 0x41,0x08,0xc3,0xaa,0x47,0x94,0x0b},
        .hash_len = 20,
    }, {
        .data =
            "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
        .data_len = 62,
        .hash =	{0xda,0x33,0xde,0xf2,0xa4,0x2d,0xf1,0x39,0x75,
                 0x35,0x28,0x46,0xc3,0x03,0x38,0xcd},
        .hash_len = 20,
    }, {
        .data = "12345678901234567890123456789012345678901234567890123456789012"
                "345678901234567890",
        .data_len = 80,
        .hash =	{0xd5,0x97,0x6f,0x79,0xd8,0x3d,0x3a,0x0d,0xc9,
                 0x80,0x6c,0x3c,0x66,0xf3,0xef,0xd8},
        .hash_len = 20,
    }
};

/** The MD5 Message-Digest Algorithm
http://tools.ietf.org/html/rfc1321
A.5 Test suite
 **/
struct DIGEST_TEST_VECTOR md5_digest_test_vector[] = {
    {
        .data = "",
        .data_len = 0,
        .hash =	{0xd4,0x1d,0x8c,0xd9,0x8f,0x00,0xb2,0x04,0xe9,
                 0x80,0x09,0x98,0xec,0xf8,0x42,0x7e},
        .hash_len = 16,
    }, {
        .data = "a",
        .data_len = 1,
        .hash = {0x0c,0xc1,0x75,0xb9,0xc0,0xf1,0xb6,0xa8,0x31,
                 0xc3,0x99,0xe2,0x69,0x77,0x26,0x61},
        .hash_len = 16,
    }, {
        .data = "abc",
        .data_len = 3,
        .hash =	{0x90,0x01,0x50,0x98,0x3c,0xd2,0x4f,0xb0,0xd6,
                 0x96,0x3f,0x7d,0x28,0xe1,0x7f,0x72},
        .hash_len = 16,
    }
};

/** SHA test Vectors for Hashing Byte-Oriented Messages
http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
SHA1ShortMsg.rsp and SHA1LongMsg.rsp

FIPS PUB 180-1
http://www.itl.nist.gov/fipspubs/fip180-1.htm
 **/
struct DIGEST_TEST_VECTOR sha1_digest_test_vector[] = {
    {	// #0
        .data = "",
        .data_len = 0,
        .hash =	{0xda,0x39,0xa3,0xee,0x5e,0x6b,0x4b,0x0d,0x32,
                 0x55,0xbf,0xef,0x95,0x60,0x18,0x90,0xaf,0xd8,
                 0x07,0x09},
        .hash_len = 20,
    }, {	// #1
        .data = "abc",
        .data_len = 3,
        .hash =	{0xA9,0x99,0x3E,0x36,0x47,0x06,0x81,0x6A,0xBA,
                 0x3E,0x25,0x71,0x78,0x50,0xC2,0x6C,0x9C,0xD0,
                 0xD8,0x9D},
        .hash_len = 20,
    }, {	// #2
        .data = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        .data_len = 56,
        .hash =	{0x84,0x98,0x3E,0x44,0x1C,0x3B,0xD2,0x6E,0xBA,
                 0xAE,0x4A,0xA1,0xF9,0x51,0x29,0xE5,0xE5,0x46,
                 0x70,0xF1},
        .hash_len = 20,
        .chunks = {28, 0, 28},
        .num_chunks = 3,
    }, {	// #3
        .data = {0x03,0x21,0x79,0x4b,0x73,0x94,0x18,0xc2,0x4e,
                 0x7c,0x2e,0x56,0x52,0x74,0x79,0x1c,0x4b,0xe7,
                 0x49,0x75,0x2a,0xd2,0x34,0xed,0x56,0xcb,0x0a,
                 0x63,0x47,0x43,0x0c,0x6b},
        .data_len = 32,
        .hash = {0xb8,0x99,0x62,0xc9,0x4d,0x60,0xf6,0xa3,0x32,
                 0xfd,0x60,0xf6,0xf0,0x7d,0x4f,0x03,0x2a,0x58,
                 0x6b,0x76},
        .hash_len = 20,
        .num_chunks = 5,
        .chunks = {8,8,8,-1,8},
    }, {	// #4
        .data = {0x45,0x92,0x7e,0x32,0xdd,0xf8,0x01,0xca,0xf3,
                 0x5e,0x18,0xe7,0xb5,0x07,0x8b,0x7f,0x54,0x35,
                 0x27,0x82,0x12,0xec,0x6b,0xb9,0x9d,0xf8,0x84,
                 0xf4,0x9b,0x32,0x7c,0x64,0x86,0xfe,0xae,0x46,
                 0xba,0x18,0x7d,0xc1,0xcc,0x91,0x45,0x12,0x1e,
                 0x14,0x92,0xe6,0xb0,0x6e,0x90,0x07,0x39,0x4d,
                 0xc3,0x3b,0x77,0x48,0xf8,0x6a,0xc3,0x20,0x7c,
                 0xfe},
            .data_len = 64,
            .hash = {0xa7,0x0c,0xfb,0xfe,0x75,0x63,0xdd,0x0e,0x66,
                     0x5c,0x7c,0x67,0x15,0xa9,0x6a,0x8d,0x75,0x69,
                     0x50,0xc0},
            .hash_len = 20,
    }, {	// #5
        .data = {0x7c,0x9c,0x67,0x32,0x3a,0x1d,0xf1,0xad,0xbf,
                 0xe5,0xce,0xb4,0x15,0xea,0xef,0x01,0x55,0xec,
                 0xe2,0x82,0x0f,0x4d,0x50,0xc1,0xec,0x22,0xcb,
                 0xa4,0x92,0x8a,0xc6,0x56,0xc8,0x3f,0xe5,0x85,
                 0xdb,0x6a,0x78,0xce,0x40,0xbc,0x42,0x75,0x7a,
                 0xba,0x7e,0x5a,0x3f,0x58,0x24,0x28,0xd6,0xca,
                 0x68,0xd0,0xc3,0x97,0x83,0x36,0xa6,0xef,0xb7,
                 0x29,0x61,0x3e,0x8d,0x99,0x79,0x01,0x62,0x04,
                 0xbf,0xd9,0x21,0x32,0x2f,0xdd,0x52,0x22,0x18,
                 0x35,0x54,0x44,0x7d,0xe5,0xe6,0xe9,0xbb,0xe6,
                 0xed,0xf7,0x6d,0x7b,0x71,0xe1,0x8d,0xc2,0xe8,
                 0xd6,0xdc,0x89,0xb7,0x39,0x83,0x64,0xf6,0x52,
                 0xfa,0xfc,0x73,0x43,0x29,0xaa,0xfa,0x3d,0xcd,
                 0x45,0xd4,0xf3,0x1e,0x38,0x8e,0x4f,0xaf,0xd7,
                 0xfc,0x64,0x95,0xf3,0x7c,0xa5,0xcb,0xab,0x7f,
                 0x54,0xd5,0x86,0x46,0x3d,0xa4,0xbf,0xea,0xa3,
                 0xba,0xe0,0x9f,0x7b,0x8e,0x92,0x39,0xd8,0x32,
                 0xb4,0xf0,0xa7,0x33,0xaa,0x60,0x9c,0xc1,0xf8,
                 0xd4},
        .data_len = 163,
        .hash = {0xd8,0xfd,0x6a,0x91,0xef,0x3b,0x6c,0xed,0x05,
                 0xb9,0x83,0x58,0xa9,0x91,0x07,0xc1,0xfa,0xc8,
                 0xc8,0x07},
        .hash_len = 20,
        .num_chunks = 4,
        .chunks = {50, 50, 50, 13},
    }, {	// #6
        .data = {0x6c,0xb7,0x0d,0x19,0xc0,0x96,0x20,0x0f,0x92,
                 0x49,0xd2,0xdb,0xc0,0x42,0x99,0xb0,0x08,0x5e,
                 0xb0,0x68,0x25,0x75,0x60,0xbe,0x3a,0x30,0x7d,
                 0xbd,0x74,0x1a,0x33,0x78,0xeb,0xfa,0x03,0xfc,
                 0xca,0x61,0x08,0x83,0xb0,0x7f,0x7f,0xea,0x56,
                 0x3a,0x86,0x65,0x71,0x82,0x24,0x72,0xda,0xde,
                 0x8a,0x0b,0xec,0x4b,0x98,0x20,0x2d,0x47,0xa3,
                 0x44,0x31,0x29,0x76,0xa7,0xbc,0xb3,0x96,0x44,
                 0x27,0xea,0xcb,0x5b,0x05,0x25,0xdb,0x22,0x06,
                 0x65,0x99,0xb8,0x1b,0xe4,0x1e,0x5a,0xda,0xf1,
                 0x57,0xd9,0x25,0xfa,0xc0,0x4b,0x06,0xeb,0x6e,
                 0x01,0xde,0xb7,0x53,0xba,0xbf,0x33,0xbe,0x16,
                 0x16,0x2b,0x21,0x4e,0x8d,0xb0,0x17,0x21,0x2f,
                 0xaf,0xa5,0x12,0xcd,0xc8,0xc0,0xd0,0xa1,0x5c,
                 0x10,0xf6,0x32,0xe8,0xf4,0xf4,0x77,0x92,0xc6,
                 0x4d,0x3f,0x02,0x60,0x04,0xd1,0x73,0xdf,0x50,
                 0xcf,0x0a,0xa7,0x97,0x60,0x66,0xa7,0x9a,0x8d,
                 0x78,0xde,0xee,0xec,0x95,0x1d,0xab,0x7c,0xc9,
                 0x0f,0x68,0xd1,0x6f,0x78,0x66,0x71,0xfe,0xba,
                 0x0b,0x7d,0x26,0x9d,0x92,0x94,0x1c,0x4f,0x02,
                 0xf4,0x32,0xaa,0x5c,0xe2,0xaa,0xb6,0x19,0x4d,
                 0xcc,0x6f,0xd3,0xae,0x36,0xc8,0x43,0x32,0x74,
                 0xef,0x6b,0x1b,0xd0,0xd3,0x14,0x63,0x6b,0xe4,
                 0x7b,0xa3,0x8d,0x19,0x48,0x34,0x3a,0x38,0xbf,
                 0x94,0x06,0x52,0x3a,0x0b,0x2a,0x8c,0xd7,0x8e,
                 0xd6,0x26,0x6e,0xe3,0xc9,0xb5,0xc6,0x06,0x20,
                 0xb3,0x08,0xcc,0x6b,0x3a,0x73,0xc6,0x06,0x0d,
                 0x52,0x68,0xa7,0xd8,0x2b,0x6a,0x33,0xb9,0x3a,
                 0x6f,0xd6,0xfe,0x1d,0xe5,0x52,0x31,0xd1,0x2c,
                 0x97},
            .data_len = 262,
            .hash = {0x4a,0x75,0xa4,0x06,0xf4,0xde,0x5f,0x9e,0x11,
                     0x32,0x06,0x9d,0x66,0x71,0x7f,0xc4,0x24,0x37,
                     0x63,0x88},
            .hash_len = 20,
            .num_chunks = 5,
            .chunks = {75, 75, 75, 37, 0},
    },
};

/** SHA test Vectors for Hashing Byte-Oriented Messages
http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
SHA256ShortMsg.rsp & SHA256LongMsg.rsp

http://csrc.nist.gov/groups/ST/toolkit/examples.html
 **/
struct DIGEST_TEST_VECTOR sha224_digest_test_vector[] = {
    {
        .data = {0x00},
        .data_len = 0,
        .hash = {0xd1,0x4a,0x02,0x8c,0x2a,0x3a,0x2b,0xc9,0x47,
                 0x61,0x02,0xbb,0x28,0x82,0x34,0xc4,0x15,0xa2,
                 0xb0,0x1f,0x82,0x8e,0xa6,0x2a,0xc5,0xb3,0xe4,
                 0x2f},
        .hash_len = 28,
    }, {
        .data = "abc",
        .data_len = 3,
        .hash = {0x23,0x09,0x7d,0x22,0x34,0x05,0xd8,0x22,0x86,
                 0x42,0xa4,0x77,0xbd,0xa2,0x55,0xb3,0x2a,0xad,
                 0xbc,0xe4,0xbd,0xa0,0xb3,0xf7,0xe3,0x6c,0x9d,
                 0xa7},
        .hash_len = 28,
    }, {
        .data = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        .data_len = 56,
        .hash = {0x75,0x38,0x8b,0x16,0x51,0x27,0x76,0xcc,0x5d,
                 0xba,0x5d,0xa1,0xfd,0x89,0x01,0x50,0xb0,0xc6,
                 0x45,0x5c,0xb4,0xf5,0x8b,0x19,0x52,0x52,0x25,
                 0x25},
        .hash_len = 28,
        .chunks = {28, 0, 28},
        .num_chunks = 3,
    }, {
        .data = {0xfe,0x1f,0x0f,0xb0,0x2c,0x90,0x11,0xf4,0xc8,
                 0xc5,0x90,0x59,0x34,0xed,0x15,0x13,0x67,0x71,
                 0x73,0x7c,0xe3,0x1c,0x58,0x59,0xe6,0x7f,0x23,
                 0x5f,0xe5,0x94,0xf5,0xf6},
        .data_len = 32,
        .hash = {0xbb,0xea,0xac,0xc6,0x32,0xc2,0xa3,0xdb,0x2a,
                 0x9b,0x47,0xf1,0x57,0xab,0x54,0xaa,0x27,0x77,
                 0x6c,0x6e,0x74,0xcf,0x0b,0xca,0xa9,0x1b,0x06,
                 0xd5},
        .hash_len = 28,
        .chunks = {16, -1, 16, 0},
        .num_chunks = 4,
    }, {
        .data = {0xa3,0x31,0x0b,0xa0,0x64,0xbe,0x2e,0x14,0xad,
                 0x32,0x27,0x6e,0x18,0xcd,0x03,0x10,0xc9,0x33,
                 0xa6,0xe6,0x50,0xc3,0xc7,0x54,0xd0,0x24,0x3c,
                 0x6c,0x61,0x20,0x78,0x65,0xb4,0xb6,0x52,0x48,
                 0xf6,0x6a,0x08,0xed,0xf6,0xe0,0x83,0x26,0x89,
                 0xa9,0xdc,0x3a,0x2e,0x5d,0x20,0x95,0xee,0xea,
                 0x50,0xbd,0x86,0x2b,0xac,0x88,0xc8,0xbd,0x31,
                 0x8d},
        .data_len = 64,
        .hash = {0xb2,0xa5,0x58,0x6d,0x9c,0xbf,0x0b,0xaa,0x99,
                 0x91,0x57,0xb4,0xaf,0x06,0xd8,0x8a,0xe0,0x8d,
                 0x7c,0x9f,0xaa,0xb4,0xbc,0x1a,0x96,0x82,0x9d,
                 0x65},
        .hash_len = 28,
        .chunks = {20, 14, 30},
        .num_chunks = 3,
    }, {
        .data = {0xf1,0x49,0xe4,0x1d,0x84,0x8f,0x59,0x27,0x6c,
                 0xfd,0xdd,0x74,0x3b,0xaf,0xa9,0xa9,0x0e,0x1e,
                 0xe4,0xa2,0x63,0xa1,0x18,0x14,0x2b,0x33,0xe3,
                 0x70,0x21,0x76,0xef,0x0a,0x59,0xf8,0x23,0x7a,
                 0x1c,0xb5,0x1b,0x42,0xf3,0xde,0xd6,0xb2,0x02,
                 0xd9,0xaf,0x09,0x97,0x89,0x8f,0xdd,0x03,0xcf,
                 0x60,0xbd,0xa9,0x51,0xc5,0x14,0x54,0x7a,0x08,
                 0x50,0xce,0xc2,0x54,0x44,0xae,0x2f,0x24,0xcb,
                 0x71,0x1b,0xfb,0xaf,0xcc,0x39,0x56,0xc9,0x41,
                 0xd3,0xde,0x69,0xf1,0x55,0xe3,0xf8,0xb1,0x0f,
                 0x06,0xdb,0x5f,0x37,0x35,0x9b,0x77,0x2d,0xdd,
                 0x43,0xe1,0x03,0x5a,0x0a,0x0d,0x3d,0xb3,0x32,
                 0x42,0xd5,0x84,0x30,0x33,0x83,0x3b,0x0d,0xd4,
                 0x3b,0x87,0x0c,0x6b,0xf6,0x0e,0x8d,0xea,0xb5,
                 0x5f,0x31,0x7c,0xc3,0x27,0x3f,0x5e,0x3b,0xa7,
                 0x47,0xf0,0xcb,0x65,0x05,0x0c,0xb7,0x22,0x87,
                 0x96,0x21,0x0d,0x92,0x54,0x87,0x36,0x43,0x00,
                 0x8d,0x45,0xf2,0x9c,0xfd,0x6c,0x5b,0x06,0x0c,
                 0x9a},
        .data_len = 163,
        .hash = {0x9d,0xb6,0xdc,0x3a,0x23,0xab,0xd7,0xb6,0xc3,
                 0xd7,0x2c,0x38,0xf4,0x84,0x3c,0x7d,0xe4,0x8a,
                 0x71,0xd0,0xba,0x91,0xa8,0x6b,0x18,0x39,0x3e,
                 0x5f},
        .hash_len = 28,
        .chunks = {54, 54, 54, 1, 0},
        .num_chunks = 5,
    }
};


/** SHA test Vectors for Hashing Byte-Oriented Messages
http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
SHA256ShortMsg.rsp & SHA256LongMsg.rsp

http://csrc.nist.gov/groups/ST/toolkit/examples.html
 **/
struct DIGEST_TEST_VECTOR sha256_digest_test_vector[] = {
    {
        .data = {0x00},
        .data_len = 0,
        .hash = {0xe3,0xb0,0xc4,0x42,0x98,0xfc,0x1c,0x14,0x9a,
                 0xfb,0xf4,0xc8,0x99,0x6f,0xb9,0x24,0x27,0xae,
                 0x41,0xe4,0x64,0x9b,0x93,0x4c,0xa4,0x95,0x99,
                 0x1b,0x78,0x52,0xb8,0x55},
        .hash_len = 32,
    }, {
        .data = "abc",
        .data_len = 3,
        .hash = {0xba,0x78,0x16,0xbf,0x8f,0x01,0xcf,0xea,0x41,
                 0x41,0x40,0xde,0x5d,0xae,0x22,0x23,0xb0,0x03,
                 0x61,0xa3,0x96,0x17,0x7a,0x9c,0xb4,0x10,0xff,
                 0x61,0xf2,0x00,0x15,0xad},
        .hash_len = 32,
    }, {
        .data = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        .data_len = 56,
        .hash = {0x24,0x8d,0x6a,0x61,0xd2,0x06,0x38,0xb8,0xe5,
                 0xc0,0x26,0x93,0x0c,0x3e,0x60,0x39,0xa3,0x3c,
                 0xe4,0x59,0x64,0xff,0x21,0x67,0xf6,0xec,0xed,
                 0xd4,0x19,0xdb,0x06,0xc1},
        .hash_len = 32,
        .chunks = {28, 0, 28},
        .num_chunks = 3,
    }, {
        .data = {0x09,0xfc,0x1a,0xcc,0xc2,0x30,0xa2,0x05,0xe4,
                 0xa2,0x08,0xe6,0x4a,0x8f,0x20,0x42,0x91,0xf5,
                 0x81,0xa1,0x27,0x56,0x39,0x2d,0xa4,0xb8,0xc0,
                 0xcf,0x5e,0xf0,0x2b,0x95},
        .data_len = 32,
        .hash = {0x4f,0x44,0xc1,0xc7,0xfb,0xeb,0xb6,0xf9,0x60,
                 0x18,0x29,0xf3,0x89,0x7b,0xfd,0x65,0x0c,0x56,
                 0xfa,0x07,0x84,0x4b,0xe7,0x64,0x89,0x07,0x63,
                 0x56,0xac,0x18,0x86,0xa4},
        .hash_len = 32,
        .chunks = {16, -1, 16, 0},
        .num_chunks = 4,
    }, {
        .data = {0x5a,0x86,0xb7,0x37,0xea,0xea,0x8e,0xe9,0x76,
                 0xa0,0xa2,0x4d,0xa6,0x3e,0x7e,0xd7,0xee,0xfa,
                 0xd1,0x8a,0x10,0x1c,0x12,0x11,0xe2,0xb3,0x65,
                 0x0c,0x51,0x87,0xc2,0xa8,0xa6,0x50,0x54,0x72,
                 0x08,0x25,0x1f,0x6d,0x42,0x37,0xe6,0x61,0xc7,
                 0xbf,0x4c,0x77,0xf3,0x35,0x39,0x03,0x94,0xc3,
                 0x7f,0xa1,0xa9,0xf9,0xbe,0x83,0x6a,0xc2,0x85,
                 0x09},
        .data_len = 64,
        .hash = {0x42,0xe6,0x1e,0x17,0x4f,0xbb,0x38,0x97,0xd6,
                 0xdd,0x6c,0xef,0x3d,0xd2,0x80,0x2f,0xe6,0x7b,
                 0x33,0x19,0x53,0xb0,0x61,0x14,0xa6,0x5c,0x77,
                 0x28,0x59,0xdf,0xc1,0xaa},
        .hash_len = 32,
        .chunks = {20, 14, 30},
        .num_chunks = 3,
    }, {
        .data = {0x45,0x11,0x01,0x25,0x0e,0xc6,0xf2,0x66,0x52,
                 0x24,0x9d,0x59,0xdc,0x97,0x4b,0x73,0x61,0xd5,
                 0x71,0xa8,0x10,0x1c,0xdf,0xd3,0x6a,0xba,0x3b,
                 0x58,0x54,0xd3,0xae,0x08,0x6b,0x5f,0xdd,0x45,
                 0x97,0x72,0x1b,0x66,0xe3,0xc0,0xdc,0x5d,0x8c,
                 0x60,0x6d,0x96,0x57,0xd0,0xe3,0x23,0x28,0x3a,
                 0x52,0x17,0xd1,0xf5,0x3f,0x2f,0x28,0x4f,0x57,
                 0xb8,0x5c,0x8a,0x61,0xac,0x89,0x24,0x71,0x1f,
                 0x89,0x5c,0x5e,0xd9,0x0e,0xf1,0x77,0x45,0xed,
                 0x2d,0x72,0x8a,0xbd,0x22,0xa5,0xf7,0xa1,0x34,
                 0x79,0xa4,0x62,0xd7,0x1b,0x56,0xc1,0x9a,0x74,
                 0xa4,0x0b,0x65,0x5c,0x58,0xed,0xfe,0x0a,0x18,
                 0x8a,0xd2,0xcf,0x46,0xcb,0xf3,0x05,0x24,0xf6,
                 0x5d,0x42,0x3c,0x83,0x7d,0xd1,0xff,0x2b,0xf4,
                 0x62,0xac,0x41,0x98,0x00,0x73,0x45,0xbb,0x44,
                 0xdb,0xb7,0xb1,0xc8,0x61,0x29,0x8c,0xdf,0x61,
                 0x98,0x2a,0x83,0x3a,0xfc,0x72,0x8f,0xae,0x1e,
                 0xda,0x2f,0x87,0xaa,0x2c,0x94,0x80,0x85,0x8b,
                 0xec},
        .data_len = 163,
        .hash = {0x3c,0x59,0x3a,0xa5,0x39,0xfd,0xcd,0xae,0x51,
                 0x6c,0xdf,0x2f,0x15,0x00,0x0f,0x66,0x34,0x18,
                 0x5c,0x88,0xf5,0x05,0xb3,0x97,0x75,0xfb,0x9a,
                 0xb1,0x37,0xa1,0x0a,0xa2},
        .hash_len = 32,
        .chunks = {54, 54, 54, 1, 0},
        .num_chunks = 5,
    }
};

/** SHA test Vectors for Hashing Byte-Oriented Messages
http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
SHA384ShortMsg.rsp & SHA384LongMsg.rsp

http://csrc.nist.gov/groups/ST/toolkit/examples.html
 **/
struct DIGEST_TEST_VECTOR sha384_digest_test_vector[] = {
    {
        .data = {0x00},
        .data_len = 0,
        .hash = {0x38,0xb0,0x60,0xa7,0x51,0xac,0x96,0x38,0x4c,
                 0xd9,0x32,0x7e,0xb1,0xb1,0xe3,0x6a,0x21,0xfd,
                 0xb7,0x11,0x14,0xbe,0x07,0x43,0x4c,0x0c,0xc7,
                 0xbf,0x63,0xf6,0xe1,0xda,0x27,0x4e,0xde,0xbf,
                 0xe7,0x6f,0x65,0xfb,0xd5,0x1a,0xd2,0xf1,0x48,
                 0x98,0xb9,0x5b},
        .hash_len = 48,
    },{
        .data = "abc",
        .data_len = 3,
        .hash = {0xcb,0x00,0x75,0x3f,0x45,0xa3,0x5e,0x8b,0xb5,
                 0xa0,0x3d,0x69,0x9a,0xc6,0x50,0x07,0x27,0x2c,
                 0x32,0xab,0x0e,0xde,0xd1,0x63,0x1a,0x8b,0x60,
                 0x5a,0x43,0xff,0x5b,0xed,0x80,0x86,0x07,0x2b,
                 0xa1,0xe7,0xcc,0x23,0x58,0xba,0xec,0xa1,0x34,
                 0xc8,0x25,0xa7},
        .hash_len = 48,
    },{
        .data = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklm"
                "noijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
        .data_len = 112,
        .hash = {0x09,0x33,0x0c,0x33,0xf7,0x11,0x47,0xe8,0x3d,
                 0x19,0x2f,0xc7,0x82,0xcd,0x1b,0x47,0x53,0x11,
                 0x1b,0x17,0x3b,0x3b,0x05,0xd2,0x2f,0xa0,0x80,
                 0x86,0xe3,0xb0,0xf7,0x12,0xfc,0xc7,0xc7,0x1a,
                 0x55,0x7e,0x2d,0xb9,0x66,0xc3,0xe9,0xfa,0x91,
                 0x74,0x60,0x39},
        .hash_len = 48,
        .chunks = {24, 40, 0, 48},
        .num_chunks =	4,
    }, {
        .data = {0xbe,0x01,0xe5,0x20,0xe6,0x9f,0x04,0x17,0x4c,
                 0xcf,0x95,0x45,0x5b,0x1c,0x81,0x44,0x52,0x98,
                 0x26,0x4d,0x9a,0xdc,0x49,0x58,0x57,0x4a,0x52,
                 0x84,0x3d,0x95,0xb8,0xba},
        .data_len = 32,
        .hash = {0xc5,0xcf,0x54,0xb8,0xe3,0x10,0x5b,0x1c,0x7b,
                 0xf7,0xa4,0x37,0x54,0xd9,0x15,0xb0,0x94,0x7f,
                 0x28,0xb6,0xdc,0x94,0xa0,0x19,0x18,0x29,0x29,
                 0xb5,0xc8,0x48,0xe1,0x14,0x41,0xc9,0xe4,0xe9,
                 0x0c,0x74,0x49,0xf4,0xc3,0xcd,0x12,0x95,0x4f,
                 0x0f,0x5d,0x99},
        .hash_len = 48,
        .chunks = {16, -1, 16, 0},
        .num_chunks = 4,
    }, {
        .data = {0x93,0x03,0x5d,0x3a,0x13,0xae,0x1b,0x06,0xdd,
                 0x03,0x3e,0x76,0x4a,0xca,0x01,0x24,0x96,0x1d,
                 0xa7,0x9c,0x36,0x6c,0x6c,0x75,0x6b,0xc4,0xbc,
                 0xc1,0x18,0x50,0xa3,0xa8,0xd1,0x20,0x85,0x4f,
                 0x34,0x29,0x0f,0xff,0x7c,0x8d,0x6d,0x83,0x53,
                 0x1d,0xbd,0xd1,0xe8,0x1c,0xc4,0xed,0x42,0x46,
                 0xe0,0x0b,0xd4,0x11,0x3e,0xf4,0x51,0x33,0x4d,
                 0xaa},
        .data_len = 64,
        .hash = {0x8d,0x46,0xcc,0x84,0xb6,0xc2,0xde,0xb2,0x06,
                 0xaa,0x5c,0x86,0x17,0x98,0x79,0x87,0x51,0xa2,
                 0x6e,0xe7,0x4b,0x1d,0xaf,0x3a,0x55,0x7c,0x41,
                 0xae,0xbd,0x65,0xad,0xc0,0x27,0x55,0x9f,0x7c,
                 0xd9,0x2b,0x25,0x5b,0x37,0x4c,0x83,0xbd,0x55,
                 0x56,0x8b,0x45},
        .hash_len = 48,
        .chunks = {20, 14, 30},
        .num_chunks = 3
    }, {
        .data = {0x3b,0xf5,0x2c,0xc5,0xee,0x86,0xb9,0xa0,0x19,
                 0x0f,0x39,0x0a,0x5c,0x03,0x66,0xa5,0x60,0xb5,
                 0x57,0x00,0x0d,0xbe,0x51,0x15,0xfd,0x9e,0xe1,
                 0x16,0x30,0xa6,0x27,0x69,0x01,0x15,0x75,0xf1,
                 0x58,0x81,0x19,0x8f,0x22,0x78,0x76,0xe8,0xfe,
                 0x68,0x5a,0x69,0x39,0xbc,0x8b,0x89,0xfd,0x48,
                 0xa3,0x4e,0xc5,0xe7,0x1e,0x13,0x14,0x62,0xb2,
                 0x88,0x67,0x94,0xdf,0xfa,0x68,0xcc,0xc6,0xd5,
                 0x64,0x73,0x3e,0x67,0xff,0xef,0x25,0xe6,0x27,
                 0xc6,0xf4,0xb5,0x46,0x07,0x96,0xe3,0xbc,0xe6,
                 0x7b,0xf5,0x8c,0xa6,0xe8,0xe5,0x55,0xbc,0x91,
                 0x6a,0x85,0x31,0x69,0x7a,0xc9,0x48,0xb9,0x0d,
                 0xc8,0x61,0x6f,0x25,0x10,0x1d,0xb9,0x0b,0x50,
                 0xc3,0xd3,0xdb,0xc9,0xe2,0x1e,0x42,0xff,0x38,
                 0x71,0x87},
        .data_len = 128,
        .hash = {0x12,0xb6,0xcb,0x35,0xed,0xa9,0x2e,0xe3,0x73,
                 0x56,0xdd,0xee,0x77,0x78,0x1a,0x17,0xb3,0xd9,
                 0x0e,0x56,0x38,0x24,0xa9,0x84,0xfa,0xff,0xc6,
                 0xfd,0xd1,0x69,0x3b,0xd7,0x62,0x60,0x39,0x63,
                 0x55,0x63,0xcf,0xc3,0xb9,0xa2,0xb0,0x0f,0x9c,
                 0x65,0xee,0xfd},
        .hash_len = 48,
        .chunks = {48, 48, -1, 32},
        .num_chunks = 4,
    }, {
        .data = {0x62,0xc6,0xa1,0x69,0xb9,0xbe,0x02,0xb3,0xd7,
                 0xb4,0x71,0xa9,0x64,0xfc,0x0b,0xcc,0x72,0xb4,
                 0x80,0xd2,0x6a,0xec,0xb2,0xed,0x46,0x0b,0x7f,
                 0x50,0x01,0x6d,0xda,0xf0,0x4c,0x51,0x21,0x87,
                 0x83,0xf3,0xaa,0xdf,0xdf,0xf5,0xa0,0x4d,0xed,
                 0x03,0x0d,0x7b,0x3f,0xb7,0x37,0x6b,0x61,0xba,
                 0x30,0xb9,0x0e,0x2d,0xa9,0x21,0xa4,0x47,0x07,
                 0x40,0xd6,0x3f,0xb9,0x9f,0xa1,0x6c,0xc8,0xed,
                 0x81,0xab,0xaf,0x8c,0xe4,0x01,0x6e,0x50,0xdf,
                 0x81,0xda,0x83,0x20,0x70,0x37,0x2c,0x24,0xa8,
                 0x08,0x90,0xaa,0x3a,0x26,0xfa,0x67,0x57,0x10,
                 0xb8,0xfb,0x71,0x82,0x66,0x24,0x9d,0x49,0x6f,
                 0x31,0x3c,0x55,0xd0,0xba,0xda,0x10,0x1f,0x8f,
                 0x56,0xee,0xcc,0xee,0x43,0x45,0xa8,0xf9,0x8f,
                 0x60,0xa3,0x66,0x62,0xcf,0xda,0x79,0x49,0x00,
                 0xd1,0x2f,0x94,0x14,0xfc,0xbd,0xfd,0xeb,0x85,
                 0x38,0x8a,0x81,0x49,0x96,0xb4,0x7e,0x24,0xd5,
                 0xc8,0x08,0x6e,0x7a,0x8e,0xdc,0xc5,0x3d,0x29,
                 0x9d,0x0d,0x03,0x3e,0x6b,0xb6,0x0c,0x58,0xb8,
                 0x3d,0x6e,0x8b,0x57,0xf6,0xc2,0x58,0xd6,0x08,
                 0x1d,0xd1,0x0e,0xb9,0x42,0xfd,0xf8,0xec,0x15,
                 0x7e,0xc3,0xe7,0x53,0x71,0x23,0x5a,0x81,0x96,
                 0xeb,0x9d,0x22,0xb1,0xde,0x3a,0x2d,0x30,0xc2,
                 0xab,0xbe,0x0d,0xb7,0x65,0x0c,0xf6,0xc7,0x15,
                 0x9b,0xac,0xbe,0x29,0xb3,0xa9,0x3c,0x92,0x10,
                 0x05,0x08},
        .data_len = 227,
        .hash = {0x07,0x30,0xe1,0x84,0xe7,0x79,0x55,0x75,0x56,
                 0x9f,0x87,0x03,0x02,0x60,0xbb,0x8e,0x54,0x49,
                 0x8e,0x0e,0x5d,0x09,0x6b,0x18,0x28,0x5e,0x98,
                 0x8d,0x24,0x5b,0x6f,0x34,0x86,0xd1,0xf2,0x44,
                 0x7d,0x5f,0x85,0xbc,0xbe,0x59,0xd5,0x68,0x9f,
                 0xc4,0x94,0x25},
        .hash_len = 48,
        .chunks = {30, 30, 30, 30, 30, 30, 30, 17},
        .num_chunks = 8,
    }, {
        .data = {0xff,0xd6,0x78,0x90,0xff,0x77,0xf3,0x44,0xad,
                 0x4f,0x06,0x7d,0xf2,0xf4,0xff,0x1d,0xb8,0xf5,
                 0x41,0xc7,0xa2,0xbd,0x9a,0xe9,0xfa,0xba,0xd0,
                 0xfa,0xeb,0xbf,0x7d,0x00,0xf0,0xa7,0x1d,0x56,
                 0x8c,0x3c,0x66,0xac,0x3c,0x57,0xd8,0x4f,0xaa,
                 0x48,0x94,0xab,0x23,0x77,0x71,0x0e,0x4b,0x4c,
                 0x4d,0xae,0x0f,0x4d,0xa1,0xee,0xdc,0x86,0x58,
                 0xdd,0x0e,0x2e,0xe2,0xff,0xac,0x87,0x84,0x51,
                 0x52,0x06,0xf2,0x87,0x6e,0xb4,0x1f,0x98,0xaf,
                 0xd4,0x54,0x7c,0xbb,0xc6,0x80,0x34,0x21,0x2b,
                 0xcf,0x0c,0x8e,0x4a,0x7d,0x1d,0x43,0xb3,0xed,
                 0x15,0xc6,0x21,0xf5,0x3b,0xd8,0xa5,0x7c,0xad,
                 0xa8,0x01,0x48,0xec,0x46,0x52,0x11,0x9b,0x5a,
                 0xf3,0xda,0x84,0x16,0x9d,0x81,0xdc,0x69,0xd3,
                 0x94,0xc8,0x76,0x7d,0x66,0x20,0x44,0xd3,0x62,
                 0x72,0xb7,0x7c,0xa0,0x4a,0xbf,0xf7,0xb6,0xb0,
                 0xcf,0x3b,0xd1,0xf3,0x91,0x9a,0x04,0xa5,0xd8,
                 0xeb,0xdf,0xe7,0xd6,0xe8,0x44,0xe7,0x8f,0xd5,
                 0x76,0xa6,0x8d,0x63,0x73,0xff,0xd5,0xd3,0x84,
                 0xe5,0x1b,0x5e,0x12,0xec,0x32,0xd5,0xbb,0x0a,
                 0xc6,0x85,0xa5,0x9f,0x4d,0x5d,0x12,0xb4,0x3b,
                 0x53,0x35,0x80,0x75,0x03,0x45,0x31,0x09,0x99,
                 0xcf,0xe9,0x1c,0xf2,0x50,0x06,0x24,0xfe,0x03,
                 0xa6,0x57,0x69,0xf8,0x6a,0x62,0x7a,0x66,0x7b,
                 0x5f,0x3b,0x42,0xcb,0x01,0xda,0x10,0x9e,0x12,
                 0x4f,0xfa,0x48,0x20,0x3f,0x1f,0x38,0x73,0x20,
                 0x2d,0x35,0x42,0x9f,0x32,0xe8,0x26,0x3e,0xaf,
                 0x9b,0xce,0x42,0xef,0x40,0xf5,0xcc,0x96,0xb5,
                 0x91,0x46,0x7d,0x46,0x4d,0x00,0xbd,0x74,0x3a,
                 0x1b,0x0a,0xf4,0xc1,0xa7,0x43,0xfb,0xdd,0x08,
                 0x46,0xb9,0x87,0x9e,0x09,0x23,0x71,0xa5,0xe7,
                 0xf6,0xf6,0x59,0x37,0xf9,0x51,0x5e,0x23,0x82,
                 0x0e,0x60,0xb8,0x3b,0xbf,0xf7,0x39,0x26,0xf0,
                 0xcd,0xb9,0xdf,0x5d,0x02,0xe8,0x22,0x62,0xcf,
                 0x2e,0x8c,0xb2,0x6a,0xf6,0xa6,0x4c,0x2a,0x4d,
                 0x1f,0xab,0xec,0xab,0x59,0x3d,0xb5,0x10,0x28,
                 0x17,0x99},
        .data_len = 326,
        .hash = {0x13,0x96,0xfe,0xa9,0x5c,0xe0,0xc1,0xc1,0xc2,
                 0x24,0xb5,0x0a,0x07,0xdd,0x71,0x97,0xf1,0xd6,
                 0x2b,0x99,0x3c,0x7f,0xe9,0xe1,0xcc,0x1a,0x56,
                 0x10,0x19,0x20,0xd4,0xb0,0xfe,0xce,0xf5,0x87,
                 0xfb,0xcd,0x56,0xb8,0x54,0xc8,0xc9,0xda,0x95,
                 0x13,0x2f,0x02},
        .hash_len = 48,
        .chunks = {163, 163},
        .num_chunks = 2,
    }
};

/** SHA test Vectors for Hashing Byte-Oriented Messages
http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
SHA512ShortMsg.rsp & SHA512LongMsg.rsp

http://csrc.nist.gov/groups/ST/toolkit/examples.html
 **/
struct DIGEST_TEST_VECTOR sha512_digest_test_vector[] = {
    {
        .data =	{0x00},
        .data_len = 0,
        .hash = {0xcf,0x83,0xe1,0x35,0x7e,0xef,0xb8,0xbd,0xf1,
                 0x54,0x28,0x50,0xd6,0x6d,0x80,0x07,0xd6,0x20,
                 0xe4,0x05,0x0b,0x57,0x15,0xdc,0x83,0xf4,0xa9,
                 0x21,0xd3,0x6c,0xe9,0xce,0x47,0xd0,0xd1,0x3c,
                 0x5d,0x85,0xf2,0xb0,0xff,0x83,0x18,0xd2,0x87,
                 0x7e,0xec,0x2f,0x63,0xb9,0x31,0xbd,0x47,0x41,
                 0x7a,0x81,0xa5,0x38,0x32,0x7a,0xf9,0x27,0xda,
                 0x3e},
        .hash_len = 64,
    },{
        .data = "abc",
        .data_len = 3,
        .hash = {0xdd,0xaf,0x35,0xa1,0x93,0x61,0x7a,0xba,0xcc,
                 0x41,0x73,0x49,0xae,0x20,0x41,0x31,0x12,0xe6,
                 0xfa,0x4e,0x89,0xa9,0x7e,0xa2,0x0a,0x9e,0xee,
                 0xe6,0x4b,0x55,0xd3,0x9a,0x21,0x92,0x99,0x2a,
                 0x27,0x4f,0xc1,0xa8,0x36,0xba,0x3c,0x23,0xa3,
                 0xfe,0xeb,0xbd,0x45,0x4d,0x44,0x23,0x64,0x3c,
                 0xe8,0x0e,0x2a,0x9a,0xc9,0x4f,0xa5,0x4c,0xa4,
                 0x9f},
        .hash_len = 64,
    }, {
        .data = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklm"
                "noijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
        .data_len = 112,
        .hash = {0x8e,0x95,0x9b,0x75,0xda,0xe3,0x13,0xda,0x8c,
                 0xf4,0xf7,0x28,0x14,0xfc,0x14,0x3f,0x8f,0x77,
                 0x79,0xc6,0xeb,0x9f,0x7f,0xa1,0x72,0x99,0xae,
                 0xad,0xb6,0x88,0x90,0x18,0x50,0x1d,0x28,0x9e,
                 0x49,0x00,0xf7,0xe4,0x33,0x1b,0x99,0xde,0xc4,
                 0xb5,0x43,0x3a,0xc7,0xd3,0x29,0xee,0xb6,0xdd,
                 0x26,0x54,0x5e,0x96,0xe5,0x5b,0x87,0x4b,0xe9,
                 0x09},
        .hash_len = 64,
        .chunks = {24, 40, 0, 48},
        .num_chunks = 4,
    }, {
        .data = {0xfd,0x22,0x03,0xe4,0x67,0x57,0x4e,0x83,0x4a,
                 0xb0,0x7c,0x90,0x97,0xae,0x16,0x45,0x32,0xf2,
                 0x4b,0xe1,0xeb,0x5d,0x88,0xf1,0xaf,0x77,0x48,
                 0xce,0xff,0x0d,0x2c,0x67,0xa2,0x1f,0x4e,0x40,
                 0x97,0xf9,0xd3,0xbb,0x4e,0x9f,0xbf,0x97,0x18,
                 0x6e,0x0d,0xb6,0xdb,0x01,0x00,0x23,0x0a,0x52,
                 0xb4,0x53,0xd4,0x21,0xf8,0xab,0x9c,0x9a,0x60,
                 0x43,0xaa,0x32,0x95,0xea,0x20,0xd2,0xf0,0x6a,
                 0x2f,0x37,0x47,0x0d,0x8a,0x99,0x07,0x5f,0x1b,
                 0x8a,0x83,0x36,0xf6,0x22,0x8c,0xf0,0x8b,0x59,
                 0x42,0xfc,0x1f,0xb4,0x29,0x9c,0x7d,0x24,0x80,
                 0xe8,0xe8,0x2b,0xce,0x17,0x55,0x40,0xbd,0xfa,
                 0xd7,0x75,0x2b,0xc9,0x5b,0x57,0x7f,0x22,0x95,
                 0x15,0x39,0x4f,0x3a,0xe5,0xce,0xc8,0x70,0xa4,
                 0xb2,0xf8},
        .data_len = 128,
        .hash = {0xa2,0x1b,0x10,0x77,0xd5,0x2b,0x27,0xac,0x54,
                 0x5a,0xf6,0x3b,0x32,0x74,0x6c,0x6e,0x3c,0x51,
                 0xcb,0x0c,0xb9,0xf2,0x81,0xeb,0x9f,0x35,0x80,
                 0xa6,0xd4,0x99,0x6d,0x5c,0x99,0x17,0xd2,0xa6,
                 0xe4,0x84,0x62,0x7a,0x9d,0x5a,0x06,0xfa,0x1b,
                 0x25,0x32,0x7a,0x9d,0x71,0x0e,0x02,0x73,0x87,
                 0xfc,0x3e,0x07,0xd7,0xc4,0xd1,0x4c,0x60,0x86,
                 0xcc},
        .hash_len = 64,
        .chunks = {64, 64, -1},
        .num_chunks = 3,
    }, {
        .data = {0xc1,0xca,0x70,0xae,0x12,0x79,0xba,0x0b,0x91,
                 0x81,0x57,0x55,0x8b,0x49,0x20,0xd6,0xb7,0xfb,
                 0xa8,0xa0,0x6b,0xe5,0x15,0x17,0x0f,0x20,0x2f,
                 0xaf,0xd3,0x6f,0xb7,0xf7,0x9d,0x69,0xfa,0xd7,
                 0x45,0xdb,0xa6,0x15,0x05,0x68,0xdb,0x1e,0x2b,
                 0x72,0x85,0x04,0x11,0x3e,0xea,0xc3,0x4f,0x52,
                 0x7f,0xc8,0x2f,0x22,0x00,0xb4,0x62,0xec,0xbf,
                 0x5d},
        .data_len = 64,
        .hash = {0x04,0x6e,0x46,0x62,0x39,0x12,0xb3,0x93,0x2b,
                 0x8d,0x66,0x2a,0xb4,0x25,0x83,0x42,0x38,0x43,
                 0x20,0x63,0x01,0xb5,0x8b,0xf2,0x0a,0xb6,0xd7,
                 0x6f,0xd4,0x7f,0x1c,0xbb,0xcf,0x42,0x1d,0xf5,
                 0x36,0xec,0xd7,0xe5,0x6d,0xb5,0x35,0x4e,0x7e,
                 0x0f,0x98,0x82,0x2d,0x21,0x29,0xc1,0x97,0xf6,
                 0xf0,0xf2,0x22,0xb8,0xec,0x52,0x31,0xf3,0x96,
                 0x7d},
        .hash_len = 64,
        .chunks = {20, 14, 30},
        .num_chunks = 3,
    }, {
        .data = {0x4f,0x05,0x60,0x09,0x50,0x66,0x4d,0x51,0x90,
                 0xa2,0xeb,0xc2,0x9c,0x9e,0xdb,0x89,0xc2,0x00,
                 0x79,0xa4,0xd3,0xe6,0xbc,0x3b,0x27,0xd7,0x5e,
                 0x34,0xe2,0xfa,0x3d,0x02,0x76,0x85,0x02,0xbd,
                 0x69,0x79,0x00,0x78,0x59,0x8d,0x5f,0xcf,0x3d,
                 0x67,0x79,0xbf,0xed,0x12,0x84,0xbb,0xe5,0xad,
                 0x72,0xfb,0x45,0x60,0x15,0x18,0x1d,0x95,0x87,
                 0xd6,0xe8,0x64,0xc9,0x40,0x56,0x4e,0xaa,0xfb,
                 0x4f,0x2f,0xea,0xd4,0x34,0x6e,0xa0,0x9b,0x68,
                 0x77,0xd9,0x34,0x0f,0x6b,0x82,0xeb,0x15,0x15,
                 0x88,0x08,0x72,0x21,0x3d,0xa3,0xad,0x88,0xfe,
                 0xba,0x9f,0x4f,0x13,0x81,0x7a,0x71,0xd6,0xf9,
                 0x0a,0x1a,0x17,0xc4,0x3a,0x15,0xc0,0x38,0xd9,
                 0x88,0xb5,0xb2,0x9e,0xdf,0xfe,0x2d,0x6a,0x06,
                 0x28,0x13,0xce,0xdb,0xe8,0x52,0xcd,0xe3,0x02,
                 0xb3,0xe3,0x3b,0x69,0x68,0x46,0xd2,0xa8,0xe3,
                 0x6b,0xd6,0x80,0xef,0xcc,0x6c,0xd3,0xf9,0xe9,
                 0xa4,0xc1,0xae,0x8c,0xac,0x10,0xcc,0x52,0x44,
                 0xd1,0x31,0x67,0x71,0x40,0x39,0x91,0x76,0xed,
                 0x46,0x70,0x00,0x19,0xa0,0x04,0xa1,0x63,0x80,
                 0x6f,0x7f,0xa4,0x67,0xfc,0x4e,0x17,0xb4,0x61,
                 0x7b,0xbd,0x76,0x41,0xaa,0xff,0x7f,0xf5,0x63,
                 0x96,0xba,0x8c,0x08,0xa8,0xbe,0x10,0x0b,0x33,
                 0xa2,0x0b,0x5d,0xaf,0x13,0x4a,0x2a,0xef,0xa5,
                 0xe1,0xc3,0x49,0x67,0x70,0xdc,0xf6,0xba,0xa4,
                 0xf7,0xbb},
        .data_len = 227,
        .hash = {0xa9,0xdb,0x49,0x0c,0x70,0x8c,0xc7,0x25,0x48,
                 0xd7,0x86,0x35,0xaa,0x7d,0xa7,0x9b,0xb2,0x53,
                 0xf9,0x45,0xd7,0x10,0xe5,0xcb,0x67,0x7a,0x47,
                 0x4e,0xfc,0x7c,0x65,0xa2,0xaa,0xb4,0x5b,0xc7,
                 0xca,0x11,0x13,0xc8,0xce,0x0f,0x3c,0x32,0xe1,
                 0x39,0x9d,0xe9,0xc4,0x59,0x53,0x5e,0x88,0x16,
                 0x52,0x1a,0xb7,0x14,0xb2,0xa6,0xcd,0x20,0x05,
                 0x25},
        .hash_len = 64,
        .chunks = {30, 30, 30, 30, 30, 30, 30, 17},
        .num_chunks = 8,
    }, {
        .data = {0xd5,0xe3,0x78,0xae,0x9f,0xc2,0x64,0x8f,0x4a,
                 0x13,0xbb,0xec,0x4b,0x09,0x35,0xaf,0xb4,0xf8,
                 0x22,0xf5,0xfe,0x0d,0x50,0x63,0x05,0x3d,0x2f,
                 0xbd,0x54,0x7b,0x33,0xb4,0xa3,0x2e,0x7a,0x00,
                 0x9e,0xe2,0xaf,0xaf,0xe8,0x3d,0x2e,0xbd,0x60,
                 0x35,0x68,0xe4,0xa3,0x81,0x89,0xb5,0xd2,0x4d,
                 0x59,0xe8,0x95,0x32,0x60,0xf1,0x5f,0x65,0x4e,
                 0xd4,0xf4,0x2f,0x9a,0x39,0x29,0x9d,0x68,0xc3,
                 0xeb,0x78,0xb0,0x9e,0x83,0x77,0x9d,0x57,0x18,
                 0xb4,0x33,0xf1,0x76,0x5d,0x35,0x35,0x0e,0xac,
                 0x46,0x49,0x3d,0x19,0x4e,0x84,0xd1,0xce,0x1f,
                 0x81,0xc9,0x5b,0x59,0x72,0x5c,0xab,0x8a,0xb7,
                 0x3d,0x36,0x9a,0xb0,0x1e,0x79,0x67,0xcf,0x73,
                 0xa3,0xac,0xf1,0x78,0x92,0x27,0xee,0x75,0xfd,
                 0xfb,0x6e,0x40,0xf3,0x53,0xff,0x04,0x84,0x48,
                 0x65,0x42,0xbe,0x05,0x31,0x15,0xdb,0x28,0x96,
                 0xba,0xb8,0x6c,0x77,0x4f,0x89,0x85,0xc4,0xdb,
                 0xcc,0x4c,0x07,0x8f,0x7b,0x1c,0x3a,0x4c,0x86,
                 0x7c,0xdc,0x65,0x80,0xfe,0x44,0xa5,0x98,0x67,
                 0x34,0x94,0xcc,0x0f,0xb1,0xf6,0x59,0x8b,0x12,
                 0x95,0x76,0x8a,0x58,0x40,0x41,0xfd,0xbd,0x14,
                 0xfa,0x7b,0x90,0xfa,0x6f,0xe3,0x3f,0x71,0xb7,
                 0x43,0xb6,0x8e,0x23,0xf8,0xe7,0x40,0x72,0x17,
                 0xaa,0xd9,0x44,0x0c,0xc8,0xca,0xd2,0x81,0x52,
                 0xae,0xdb,0x82,0x38,0x8b,0xe2,0xde,0x16,0x54,
                 0x96,0xd0,0x51,0xb2,0x92,0xde,0x63,0x03,0x46,
                 0x02,0x73,0xa4,0x35,0x08,0x29,0x6b,0x62,0x37,
                 0xc0,0x78,0x04,0x33,0x5d,0x2e,0x81,0x22,0x9f,
                 0x7c,0x9a,0x0e,0x77,0x61,0xe3,0x8a,0x3a,0xaf,
                 0x77,0x99,0xf4,0x0f,0xe9,0xcb,0x00,0x45,0x7e,
                 0xa9,0xd5,0xb5,0x99,0x53,0x23,0x26,0x76,0x68,
                 0x1f,0xc7,0x1b,0x26,0x1a,0x6f,0x8c,0xd3,0x59,
                 0x29,0x3f,0x5b,0x21,0xf0,0xcf,0x3a,0x11,0xb7,
                 0xf4,0x9c,0xb5,0xad,0xb3,0xc3,0x57,0xbe,0xd2,
                 0xaa,0x18,0x5d,0x8f,0xe8,0x40,0x81,0x92,0xd6,
                 0xd3,0xed,0x1f,0xf4,0x65,0xb5,0x90,0x89,0x2e,
                 0xfe,0x03},
        .data_len = 326,
        .hash = {0xa7,0x0c,0x75,0xb9,0xb1,0xf0,0xac,0x2e,0xd2,
                 0xc2,0x79,0x77,0x63,0xac,0x9a,0x66,0x01,0xd9,
                 0x5f,0x46,0x88,0x9b,0x00,0xfc,0x3d,0xda,0xe4,
                 0xd0,0xac,0x69,0x23,0x75,0x0a,0x10,0x8d,0x79,
                 0xeb,0x76,0x4e,0x77,0xac,0x07,0xb7,0xcb,0x5c,
                 0x01,0xcb,0x4b,0x37,0x47,0xdc,0xf6,0x9b,0xa3,
                 0xb3,0x5c,0x51,0xfb,0x99,0x5d,0xa2,0x63,0x2e,
                 0x70},
        .hash_len = 64,
        .chunks = {163, 163},
        .num_chunks = 2,
    }
};

/** SHA test Vectors for Hashing Byte-Oriented Messages
    http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
    SHA512_224ShortMsg.rsp & SHA512_224LongMsg.rsp

    http://csrc.nist.gov/groups/ST/toolkit/examples.html
**/
struct DIGEST_TEST_VECTOR sha512_224_digest_test_vector[] = {
    {
        .data = {0x00},
        .data_len = 0,
        .hash = {0x6e,0xd0,0xdd,0x02,0x80,0x6f,0xa8,0x9e,0x25,
                 0xde,0x06,0x0c,0x19,0xd3,0xac,0x86,0xca,0xbb,
                 0x87,0xd6,0xa0,0xdd,0xd0,0x5c,0x33,0x3b,0x84,
                 0xf4},
        .hash_len = 28,
    }, {
        .data = {0x49,0x76,0x04 },
        .data_len = 3,
        .hash = {0xa9,0xc3,0x45,0xd5,0x8a,0x95,0x9a,0xf2,0x0a,
                 0x42,0xc8,0x4e,0x28,0x52,0x3b,0xa4,0x7e,0x3b,
                 0xf8,0xfa,0xd8,0xe8,0xc3,0xf3,0x2b,0x7a,0x72,
                 0xae},
        .hash_len = 28,
    }, {
        .data = {0x4c,0xd2,0x73,0x24,0xc2,0x83,0x64,0x87,0x3c,
                 0x6d,0xdb,0xc3,0xe3,0xa7,0xe2,0xcd,0xa9,0xe8,
                 0xa7,0x2a,0xd2,0xf7,0x22,0x01,0xb2,0x62,0xf8,
                 0x74,0xb8,0x73,0x9f,0x30,0xab,0x60,0xc3,0x43,
                 0x34,0xc2,0xe9,0x2f,0x9d,0x48,0x53,0x3c,0xd8,
                 0xad,0x23,0x12,0xc3,0xe7,0xc3,0x86,0xaa,0xa2,
                 0x83,0xb5,0x0d,0xec,0x84,0x4f,0xa4,0x32,0xd6,
                 0x36},
        .data_len = 64,
        .hash = {0xce,0xd6,0x08,0x17,0x61,0xff,0x52,0x59,0xf1,
                 0x32,0xaa,0x83,0x1b,0x7a,0x1b,0x43,0x2d,0x09,
                 0x3f,0xc8,0x57,0xda,0x0e,0xee,0xb8,0x2b,0xe7,
                 0x1f},
        .hash_len = 28,
        .chunks = {20, 14, 30},
        .num_chunks = 3,
    }, {
        .data = {0xd2,0x4d,0xf7,0x5a,0x00,0xcf,0x92,0x67,0x7b,
                 0xb4,0x1a,0x62,0x0f,0xae,0x51,0x97,0x23,0x93,
                 0x7e,0xbf,0xe1,0xf7,0xb4,0x30,0x97,0x00,0x56,
                 0x50,0x5d,0x76,0xdb,0x4f,0xf9,0x1a,0xcf,0x16,
                 0xff,0x39,0x1a,0x7a,0x3d,0x80,0x85,0xb6,0x55,
                 0x12,0x7a,0x18,0xac,0xd8,0x0b,0xfa,0x83,0x18,
                 0x37,0xf4,0x64,0x4a,0x68,0x50,0xc0,0x27,0x3f,
                 0xbe,0xd6,0x02,0x94,0x49,0xd6,0x5b,0xb9,0x8a,
                 0x47,0xb2,0xff,0x1c,0xa6,0x99,0x7c,0x50,0x50,
                 0x0d,0x0b,0x21,0xa2,0x06,0x93,0x6a,0x5e,0x4d,
                 0x8d,0x56,0x50,0x8e,0xc0,0x18,0x32,0xae,0x4f,
                 0xdd,0xce,0x5e,0xf6,0xff,0x62,0xf1,0x91,0x7c,
                 0x48,0x6a,0xde,0xa6},
        .data_len = 112,
        .hash = {0x5c,0xae,0x12,0xea,0x96,0x52,0x26,0x9e,0xa2,
                 0xaa,0xfc,0x65,0x6c,0xb8,0x34,0x24,0x74,0x6e,
                 0xa1,0xd5,0xd4,0x91,0xf9,0xa1,0x59,0x59,0x4b,
                 0x2a},
        .hash_len = 28,
        .chunks = {24, 40, 0, 48},
        .num_chunks = 4,
    }, {
        .data = {0x9e,0x12,0x78,0x70,0xbe,0x24,0x31,0xbc,0xb4,
                 0xf4,0xeb,0x4e,0xfd,0x5c,0x2a,0x6c,0x58,0x70,
                 0xc5,0x5e,0x7a,0x5e,0x3b,0x75,0x03,0x99,0x4a,
                 0x4c,0xb1,0x36,0xbe,0x4e,0xd3,0x96,0x88,0x78,
                 0x01,0x45,0x0f,0x60,0x0b,0x22,0xcb,0x77,0x2f,
                 0xc0,0x0f,0x8b,0x8f,0x0d,0x26,0x90,0xe2,0x31,
                 0xa2,0x9f,0x69,0xb9,0xf1,0x3f,0x24,0xf5,0x31,
                 0xe4,0x47,0x9e,0x45,0xb5,0xe8,0xbc,0x29,0x92,
                 0xfa,0xc7,0x82,0x56,0x7e,0x0d,0x7a,0x59,0xf8,
                 0x53,0xca,0x3a,0x20,0xbf,0x18,0xdb,0xdb,0xf6,
                 0x84,0xac,0x69,0x81,0x7e,0x2d,0xe0,0x75,0xda,
                 0xae,0xd9,0x53,0x26,0x59,0x69,0x2d,0x3b,0x73,
                 0x53,0x0a,0x12,0xdf,0x7b,0x8c,0xd9,0xe4,0x9e,
                 0xd0,0x46,0x30,0x41,0x96,0x2c,0x1c,0xe7,0xa2,
                 0x4c,0x31},
        .data_len = 128,
        .hash = {0x7e,0x2c,0xf6,0x22,0x66,0x23,0x53,0x57,0x84,
                 0xc5,0x9c,0xd6,0xa7,0xb2,0x7d,0xac,0x60,0xee,
                 0x23,0xfd,0xce,0x8a,0x80,0x4d,0xbd,0x6d,0xfe,
                 0xdd},
        .hash_len = 28,
        .chunks = {64, 64, -1},
        .num_chunks = 3,
    }, {
        .data = {0x96,0x25,0xae,0x61,0x8e,0xa6,0x33,0xfd,0x7a,
                 0xe5,0xb2,0x0c,0xea,0xfd,0x6b,0x1f,0x3a,0xb1,
                 0xa6,0xaa,0x20,0xad,0xed,0x66,0x81,0x0e,0x78,
                 0xf3,0x89,0x25,0xe9,0xc2,0xfa,0x78,0x3a,0x32,
                 0xc4,0x0a,0xf3,0xf9,0xd7,0xdd,0xa0,0xc6,0x35,
                 0xb4,0x82,0x25,0x4b,0x1d,0x85,0xa2,0x81,0xaf,
                 0x72,0x31,0x10,0x91,0x66,0xcd,0x13,0x3c,0x83,
                 0x60,0xe2,0x81,0xe5,0xe3,0x9b,0xcd,0xd7,0xc6,
                 0x01,0xac,0x47,0x92,0x8a,0x8c,0x78,0xcd,0xb3,
                 0xc4,0xf7,0x1e,0x97,0xd4,0xd0,0xb1,0xc0,0xee,
                 0x01,0xdd,0x3d,0xb6,0x2f,0x04,0xf4,0x47,0x98,
                 0xbb,0x3a,0x76,0x49,0x2b,0xa1,0x5a,0x91,0xb7,
                 0x11,0x0c,0xb5,0xe0,0x1b,0xab,0xe5,0x65,0x89,
                 0xa3,0x6f,0xae,0x3a,0x2f,0x33,0x6a,0x2d,0x1d,
                 0x57,0x78,0xdb,0xd2,0x3c,0x03,0xca,0x8d,0xb0,
                 0xf2,0x5f,0xf0,0x65,0x7f,0xf4,0xbc,0xa1,0x25,
                 0x2a,0xdc,0x38,0xc0,0x80,0xa5,0xb8,0xf0,0x25,
                 0x5c,0xe3,0xbe,0x0b,0xf8,0x62,0x82,0x3d,0x2a,
                 0xb7,0x04,0x72,0x9b,0x74,0xe1,0xe2,0x75,0xaa,
                 0x30,0x58,0x24,0xa5,0x66,0x89,0x5e,0xd6,0x77,
                 0xa4,0x60,0x11,0x3e,0x2a,0x7b,0xf9,0x1f,0x00,
                 0xd0,0xb8,0xeb,0xc3,0x58,0xf3,0x03,0x5b,0x27,
                 0xfc,0xc1,0xd3,0xf1,0x4a,0x13,0x67,0xcd,0x27,
                 0x69,0xdf,0x39,0xa9,0xd2,0x1c,0x5e,0xe3,0x61,
                 0xf1,0x96,0x5c,0xd6,0x34,0x2c,0xc1,0x7a,0x14,
                 0x63,0xd6},
        .data_len = 227,
        .hash = {0x72,0x64,0x0a,0x79,0xfb,0xb1,0xcf,0xb2,0x6e,
                 0x09,0xb4,0xb3,0x53,0x85,0x38,0x9e,0xd6,0x33,
                 0xa5,0x5e,0x09,0x29,0x06,0xd0,0x1a,0x71,0x86,
                 0xe1},
        .hash_len = 28,
        .chunks = {30, 30, 30, 30, 30, 30, 30, 17},
        .num_chunks = 8,
    }, {
        .data = {0x0c,0x20,0x65,0x08,0x01,0xdf,0x97,0x65,0x48,
                 0x2e,0x7e,0x79,0xf6,0xa0,0xef,0x0d,0xc5,0xdb,
                 0x8a,0x94,0x64,0x9d,0xa8,0x56,0x11,0xfd,0x8e,
                 0xcd,0x6a,0xf4,0xe7,0x4f,0x80,0x1b,0x71,0xee,
                 0x2e,0xb6,0xf6,0x9d,0xa9,0xf2,0x7e,0x94,0x57,
                 0x6b,0xc3,0xf5,0xe5,0xf1,0xe5,0xaf,0x94,0x8c,
                 0x5f,0xa2,0xd3,0xbd,0x3a,0x05,0x13,0x07,0x80,
                 0x05,0xd7,0xae,0xf4,0xaf,0x0a,0xbf,0x00,0x10,
                 0x96,0x50,0xef,0x9b,0x32,0x09,0x5c,0xcb,0x7b,
                 0xe4,0xea,0x09,0x47,0x30,0x36,0xcf,0x36,0xdc,
                 0x0a,0x0e,0x8a,0xf2,0xd4,0xcd,0x38,0x62,0x56,
                 0x13,0x3e,0x01,0x9e,0x08,0x38,0xd1,0x39,0x57,
                 0x2e,0x83,0xa4,0xd6,0xb1,0x3a,0x03,0x69,0x26,
                 0x92,0x15,0x24,0x61,0xe1,0x68,0xaf,0x76,0xbd,
                 0xc9,0xef,0xcb,0xbe,0xff,0xe2,0x72,0x84,0xb9,
                 0xcb,0xb6,0xfc,0xc4,0x90,0xf4,0xe0,0x81,0x60,
                 0xad,0x7d,0x97,0x98,0xd8,0xb9,0xe5,0xe8,0x7d,
                 0x54,0xbb,0xb2,0x7f,0x6e,0x49,0xb2,0xba,0x4a,
                 0x57,0xe4,0x4d,0x51,0x37,0xa2,0x39,0x5a,0xf8,
                 0x0d,0x2c,0x28,0xdd,0xaf,0x97,0x34,0x0d,0x5f,
                 0x06,0xad,0x8c,0x64,0xb1,0xbe,0x6a,0xdd,0x4e,
                 0x0f,0x3b,0x60,0x2a,0x8e,0xc7,0xe1,0x78,0x6f,
                 0x02,0xd5,0xe9,0x71,0x3e,0x98,0x7d,0x64,0x9f,
                 0x0c,0x98,0xe8,0xcf,0x84,0x58,0x77,0xf4,0xe0,
                 0x26,0xa6,0xdf,0xf9,0x8b,0xa7,0x86,0x3f,0x45,
                 0x13,0xb5,0xa0,0x9b,0xe6,0x00,0xac,0xed,0xcc,
                 0xfc,0xf7,0x8f,0x9b,0xaf,0xbf,0xd8,0x20,0x44,
                 0xcf,0x75,0x1e,0x03,0xa3,0x03,0x94,0x81,0xcd,
                 0xde,0x3f,0x14,0x98,0x33,0x60,0x8a,0xea,0x5d,
                 0xbf,0x1b,0x6f,0x6e,0xf0,0x9b,0xf3,0x0b,0xd5,
                 0xf5,0x10,0x7e,0x64,0xd4,0x0a,0x06,0xaa,0xa3,
                 0xf7,0x71,0x10,0xb8,0x23,0xed,0xde,0xcc,0xb2,
                 0x1c,0xfd,0xaf,0x87,0x66,0x01,0x39,0x65,0x93,
                 0x67,0x84,0x54,0x99,0x1a,0x58,0x47,0xb2,0x3c,
                 0xe7,0xfb,0xb2,0x24,0xbb,0x22,0x98,0xa3,0xec,
                 0xc1,0x63,0x8c,0x96,0xd7,0x27,0x73,0x8a,0x3f,
                 0x5f,0xb4},
        .data_len = 326,
        .hash = {0xaf,0xf2,0x28,0x7a,0xb9,0x78,0xd6,0x04,0xba,
                 0xb6,0xfe,0x2b,0x5f,0xb4,0x98,0xd1,0x4c,0x4b,
                 0x16,0x9f,0x68,0x80,0x64,0xe1,0x96,0x40,0xb5,
                 0x97},
        .hash_len = 28,
        .chunks = {163, 163},
        .num_chunks = 2,
    }
};

/** SHA test Vectors for Hashing Byte-Oriented Messages
    http://csrc.nist.gov/groups/STM/cavp/documents/shs/shabytetestvectors.zip
    SHA512_256ShortMsg.rsp & SHA512_256LongMsg.rsp

    http://csrc.nist.gov/groups/ST/toolkit/examples.html
**/
struct DIGEST_TEST_VECTOR sha512_256_digest_test_vector[] = {
    {
        .data = {0x00},
        .data_len = 0,
        .hash = {0xc6,0x72,0xb8,0xd1,0xef,0x56,0xed,0x28,0xab,
                 0x87,0xc3,0x62,0x2c,0x51,0x14,0x06,0x9b,0xdd,
                 0x3a,0xd7,0xb8,0xf9,0x73,0x74,0x98,0xd0,0xc0,
                 0x1e,0xce,0xf0,0x96,0x7a},
        .hash_len = 32,
    }, {
        .data = {0x6f,0x63,0xb4},
        .data_len = 3,
        .hash = {0xa9,0xe2,0x42,0x7c,0xec,0x31,0x4b,0x28,0x14,
                 0xaa,0xba,0x87,0x03,0x94,0x85,0xfc,0x8d,0x3a,
                 0xde,0x99,0x2f,0xa1,0xd9,0xac,0xbb,0x7f,0x67,
                 0x69,0x46,0x0a,0x73,0x17},
        .hash_len = 32,
    }, {
        .data = {0xd2,0xbc,0x0c,0xe7,0x21,0x7f,0xf2,0xe9,0x44,
                 0xe1,0xae,0x47,0xad,0x58,0x73,0xbf,0x39,0x1f,
                 0x1b,0x0c,0xc0,0x7f,0x61,0x51,0xeb,0x4c,0x50,
                 0xbb,0x45,0xb2,0xfb,0x62,0x95,0x32,0x6f,0x71,
                 0x6c,0xe7,0xe6,0x87,0xfa,0x0e,0x3d,0x5d,0x25,
                 0xc5,0xa8,0xa8,0xdd,0x13,0xa5,0x41,0xa9,0x29,
                 0x2e,0x83,0x86,0xe7,0x33,0xf4,0xf2,0xa2,0x47,
                 0x28},
        .data_len = 64,
        .hash = {0x9c,0x1f,0xea,0x57,0x86,0x70,0x2d,0x02,0x7b,
                 0xb5,0xb6,0x6b,0x3f,0xa9,0x2d,0xe3,0x46,0x21,
                 0xa8,0x62,0x69,0x82,0xec,0x21,0xc0,0xec,0xf8,
                 0xda,0xa7,0x9d,0xea,0x05},
        .hash_len = 32,
        .chunks = {20, 14, 30},
        .num_chunks = 3,
    }, {
        .data = {0x92,0xb2,0x3c,0x0b,0xc4,0xd8,0xd0,0x7d,0x22,
                 0xe2,0x88,0x12,0x71,0x0d,0xff,0x06,0xcb,0x9b,
                 0xbe,0xce,0xa2,0xc9,0x60,0xac,0x02,0x00,0xf4,
                 0x80,0x16,0x4f,0xa2,0xe1,0xee,0x19,0x92,0x6c,
                 0x7f,0x0b,0x09,0x5c,0xec,0x51,0xd5,0x5c,0x04,
                 0x0a,0xec,0x99,0x0b,0xf9,0x50,0x1a,0xbd,0x7d,
                 0x35,0x54,0x90,0xc3,0x66,0xf9,0x3a,0x3a,0xe5,
                 0x12,0x73,0x47,0xd1,0x4d,0xfc,0x3b,0x8d,0x98,
                 0xe0,0x82,0x1f,0xee,0xfa,0x1c,0xd6,0x71,0xb7,
                 0x52,0x30,0xba,0x1d,0xa1,0xfa,0x6d,0x0c,0xfb,
                 0xb9,0x10,0xc4,0x2f,0x49,0x1d,0xa8,0xa5,0xc4,
                 0x55,0x42,0x4e,0xa6,0x58,0x86,0xdb,0x2e,0x73,
                 0x5b,0x2d,0x07,0xb9},
        .data_len = 112,
        .hash = {0x55,0xa0,0x59,0x7f,0x11,0xff,0x71,0xc4,0x26,
                 0x20,0x17,0x15,0xbe,0xb5,0x85,0xf2,0x54,0xbb,
                 0x31,0xc1,0xdb,0xad,0xe5,0x33,0xf0,0x4e,0x49,
                 0x9c,0x33,0x91,0xff,0x79},
        .hash_len = 32,
        .chunks = {24, 40, 0, 48},
        .num_chunks = 4,
    }, {
        .data = {0xbc,0x81,0x73,0xc8,0x78,0xca,0x60,0xe9,0xa0,
                 0xf8,0x23,0xf9,0xa5,0x89,0xd4,0xff,0x84,0x54,
                 0x7b,0x38,0x9b,0x11,0x7f,0xb6,0xbb,0x1b,0x61,
                 0x4e,0x7e,0x75,0xa9,0xb1,0xdb,0x0b,0x21,0xd9,
                 0xf7,0x3b,0x42,0xa7,0x3e,0x94,0xec,0xca,0xb3,
                 0xde,0x5a,0xe2,0x84,0x5a,0x54,0xe5,0xe2,0x4b,
                 0xa6,0xc2,0x0f,0xb4,0xd2,0x45,0xb9,0x64,0x02,
                 0x3b,0x86,0x30,0x40,0xd6,0xf0,0x80,0xe9,0x53,
                 0x53,0x0d,0x5f,0xd9,0x44,0xe8,0xff,0xa5,0x25,
                 0xbf,0x53,0x64,0xf6,0x5c,0x88,0xe0,0x6e,0x6e,
                 0x22,0xdf,0x4b,0x8c,0xee,0x48,0xe6,0x77,0x38,
                 0x88,0x0a,0x9f,0x3f,0x34,0x06,0xe9,0xe6,0xf0,
                 0x01,0xb0,0xac,0x8f,0x8e,0x0a,0xde,0x7c,0x81,
                 0x4c,0x0c,0x58,0x00,0xd0,0xb9,0xe4,0xdd,0xf5,
                 0x56,0x22},
        .data_len = 128,
        .hash = {0xf6,0x91,0xd0,0x1e,0xe9,0xab,0x67,0x5f,0x38,
                 0x72,0x31,0x3b,0x77,0xe6,0xa4,0x54,0x3c,0x71,
                 0xe3,0xe8,0x9a,0xa9,0x4c,0x48,0xf9,0x1d,0x6e,
                 0xe7,0xfa,0x1a,0xb4,0xfb},
        .hash_len = 32,
        .chunks = {64, 64, -1},
        .num_chunks = 3,
    }, {
        .data = {0x97,0xe0,0x03,0x90,0x3b,0xb9,0x71,0xa5,0x23,
                 0xce,0x0c,0x82,0xbd,0xa5,0xd6,0x73,0x3c,0x76,
                 0xb9,0x0d,0xeb,0x30,0x75,0x59,0xc1,0xbd,0xdd,
                 0x35,0x36,0x87,0x43,0xf6,0x56,0x3b,0x31,0x52,
                 0x14,0xcd,0x5a,0x7e,0xe0,0xbc,0xcf,0x93,0x7c,
                 0x97,0x76,0x36,0x0b,0xc0,0xb9,0x78,0x6b,0x70,
                 0x7b,0xfb,0xc4,0xfb,0x50,0x57,0x61,0x55,0xed,
                 0xbb,0xbf,0xd5,0xdd,0xd8,0xe4,0x3a,0x76,0xfa,
                 0xf2,0xec,0x0c,0x78,0xfc,0x84,0x64,0x4f,0x18,
                 0x8d,0x6b,0x0a,0xb6,0x8c,0x28,0xe5,0x30,0x3f,
                 0xf0,0x31,0xa2,0x23,0xd9,0xfa,0xfb,0x38,0x71,
                 0xe8,0x54,0x08,0xaf,0x63,0x81,0xe6,0x29,0xfa,
                 0xe6,0x74,0x88,0x06,0x8c,0x68,0x39,0x8a,0x75,
                 0x8f,0x66,0x5e,0x2c,0x12,0x25,0x8d,0x9f,0xf8,
                 0xef,0xfb,0x31,0xec,0x53,0x4b,0x0c,0x40,0xeb,
                 0xff,0xb4,0x33,0x90,0xe1,0xe2,0x6f,0xca,0xa2,
                 0x8f,0xd6,0x8a,0xc2,0x4f,0x7e,0x1c,0xaf,0xe0,
                 0xfa,0x57,0x31,0x03,0xdc,0x17,0x05,0x8a,0x77,
                 0xed,0xc9,0xb3,0xea,0x14,0x18,0xb4,0x5a,0xa7,
                 0xf5,0x97,0x7e,0x12,0x6d,0x48,0x61,0xc7,0x78,
                 0xed,0x63,0x32,0x21,0x75,0x81,0xee,0xe6,0x74,
                 0xd7,0x39,0x62,0x2e,0x63,0xa5,0x29,0xf1,0x0c,
                 0x11,0xf4,0xa9,0xe3,0xd8,0xfe,0xae,0xa8,0x48,
                 0xad,0xe0,0x90,0x56,0x75,0xf6,0x45,0x8f,0xfa,
                 0x13,0x2f,0x52,0x74,0x9a,0xf2,0x3d,0x58,0x44,
                 0x38,0xe5},
        .data_len = 227,
        .hash = {0x00,0xce,0x3b,0x59,0x2d,0x4e,0x1a,0x65,0xf7,
                 0x80,0xdf,0x35,0x1f,0xa7,0xb2,0xc0,0x1b,0x49,
                 0xdf,0x4e,0xa9,0x13,0xc3,0xfa,0xb2,0x42,0x97,
                 0xf5,0x79,0x1b,0x18,0xe5},
        .hash_len = 32,
        .chunks = {30, 30, 30, 30, 30, 30, 30, 17},
        .num_chunks = 8,
    }, {
        .data = {0x77,0x3e,0x10,0x98,0xb7,0x25,0xab,0x1a,0x74,
                 0x65,0xc6,0x78,0x92,0xa3,0x84,0x14,0x7b,0xf0,
                 0xc3,0x27,0x14,0xee,0xab,0x05,0xf1,0x34,0x87,
                 0xa3,0xc5,0xf4,0xd4,0x56,0x1c,0xdb,0x98,0xdd,
                 0x4c,0x39,0xf6,0xa2,0xa8,0x62,0xfe,0x3d,0xf8,
                 0x54,0xca,0x3a,0x26,0x9a,0xc6,0x1c,0x3a,0x70,
                 0x4f,0xa1,0x88,0x27,0x80,0x48,0x28,0xcb,0x48,
                 0x11,0xa7,0x04,0xc0,0x84,0xfc,0x3a,0xbc,0xf4,
                 0xbe,0xfc,0xa1,0x05,0x94,0xa3,0x76,0x6c,0xec,
                 0x32,0x3d,0xf6,0xf0,0x81,0x83,0xcc,0xb5,0x9b,
                 0x36,0xf5,0xb6,0x64,0xb7,0x1c,0x82,0x7f,0x30,
                 0x94,0xb1,0x6e,0x28,0x99,0x1c,0xfb,0x54,0xd9,
                 0x4e,0xd7,0xb0,0x33,0x35,0x82,0xcf,0xbe,0x1d,
                 0x6a,0x6f,0x0d,0xe0,0x57,0x51,0xb1,0x56,0x06,
                 0x48,0x00,0x13,0x14,0x8f,0x15,0x52,0x1f,0x5f,
                 0x18,0x2c,0x27,0xc1,0xc0,0x0e,0x3a,0xa2,0x15,
                 0x69,0x50,0x25,0x7b,0xab,0x74,0x3d,0xec,0x6f,
                 0x24,0x7a,0x85,0xf0,0xfb,0x23,0x44,0xc4,0x8d,
                 0x86,0x10,0xc7,0x93,0x8c,0xf9,0x55,0x48,0x90,
                 0xc2,0xbc,0x12,0x71,0x9c,0xf0,0x65,0xe6,0x35,
                 0x81,0xe4,0x12,0xf1,0xcb,0xca,0x59,0x77,0x6d,
                 0x89,0x71,0x70,0xfc,0xb1,0xbc,0x8b,0xf8,0x12,
                 0xd5,0xc5,0xfe,0x25,0x69,0xe7,0x40,0xa8,0x48,
                 0x50,0x33,0x89,0xbb,0xf4,0x87,0x05,0x19,0xa5,
                 0x5c,0x11,0x95,0x92,0xb3,0xf9,0x5a,0x0d,0x22,
                 0x47,0xda,0x91,0xcd,0xe6,0x62,0x03,0x91,0x87,
                 0xf2,0x38,0x82,0x44,0x4d,0xb8,0x98,0x83,0x4c,
                 0xc1,0xa5,0x1e,0x77,0x8f,0x8a,0xc1,0xd6,0xdb,
                 0xab,0x93,0x05,0xa2,0xa0,0x18,0x87,0x27,0x2e,
                 0x56,0x5c,0x3b,0x75,0x36,0xd6,0xbc,0x6a,0x2e,
                 0xa8,0x5c,0x88,0x1d,0x40,0xa3,0xc3,0x76,0x57,
                 0x38,0xbe,0xa1,0xc6,0x5c,0x3d,0x9e,0x9b,0xb7,
                 0xdd,0xec,0x6c,0x57,0x03,0xa9,0xeb,0xf6,0x12,
                 0xfa,0xbb,0x7b,0xe1,0xad,0x82,0xc4,0x6c,0x1d,
                 0x40,0xa3,0xc9,0x56,0xdf,0x26,0x5e,0x7a,0x0b,
                 0x1c,0x52,0x64,0x43,0xd4,0x41,0x8a,0x30,0x28,
                 0x34,0x28},
        .data_len = 326,
        .hash = {0xf3,0xe5,0x24,0xf6,0x45,0x88,0x1c,0x45,0x31,
                 0x80,0x30,0x62,0x4b,0xaf,0x25,0x4f,0xc6,0xb3,
                 0x38,0x9d,0x7b,0xb7,0xa2,0x2a,0xa5,0x5e,0xf5,
                 0x28,0xc2,0xfd,0x3c,0xf0},
        .hash_len = 32,
        .chunks = {163, 163},
        .num_chunks = 2,
    }
};

/** SHA3 test Vectors for Hashing Byte-Oriented Messages
NIST SHA3-224 test vectors
http://csrc.nist.gov/groups/STM/cavp/secure-hashing.html
 **/
struct DIGEST_TEST_VECTOR sha3_224_digest_test_vector[] = {
    {
        .data =      { 0x00 },
        .data_len = 0,
        .hash =      { 0x6b,0x4e,0x03,0x42,0x36,0x67,0xdb,0xb7,0x3b,
                       0x6e,0x15,0x45,0x4f,0x0e,0xb1,0xab,0xd4,0x59,
                       0x7f,0x9a,0x1b,0x07,0x8e,0x3f,0x5b,0x5a,0x6b,
                       0xc7 },
        .hash_len = 28,
    }, {
        .data =      { 0x01 },
        .data_len = 1,
        .hash =      { 0x48,0x82,0x86,0xd9,0xd3,0x27,0x16,0xe5,0x88,
                       0x1e,0xa1,0xee,0x51,0xf3,0x6d,0x36,0x60,0xd7,
                       0x0f,0x0d,0xb0,0x3b,0x3f,0x61,0x2c,0xe9,0xed,
                       0xa4 },
        .hash_len = 28,
    }, {
        .data =      { 0xbf,0x58,0x31 },
        .data_len = 3,
        .hash =      { 0x1b,0xb3,0x6b,0xeb,0xde,0x5f,0x3c,0xb6,0xd8,
                       0xe4,0x67,0x2a,0xcf,0x6e,0xec,0x87,0x28,0xf3,
                       0x1a,0x54,0xda,0xcc,0x25,0x60,0xda,0x2a,0x00,
                       0xcc },
        .hash_len = 28,
    }, {
        .data =      { 0xdd,0x0f,0x85,0xb5,0x5f,0xdf,0x56,0xba,0x25,
                       0x4e,0x06,0xf8,0xc2,0xb6,0x50,0xcc,0x6b,0x86,
                       0xbf,0x28,0xa1,0x4d,0x71,0x40,0x11,0x14,0x1a,
                       0x86,0xb8,0xf1,0x4b,0xd9 },
        .data_len = 32,
        .hash =      { 0x0f,0xe9,0x24,0x69,0x29,0x7c,0x2c,0x34,0x91,
                       0x1e,0xae,0x42,0x47,0x10,0xdb,0x6d,0x31,0x20,
                       0x47,0x89,0x8b,0x97,0x56,0xed,0xc5,0xc2,0xde,
                       0xb2 },
        .hash_len = 28,
        .chunks = { 16, -1, 16, 0 },
        .num_chunks = 4,
    }, {
        .data =      { 0x6b,0x2b,0x92,0x58,0x41,0x46,0xa4,0x33,0xbe,
                       0xe8,0xb9,0x47,0xcc,0x1f,0x35,0xb6,0x17,0xb7,
                       0x3f,0x5b,0x1e,0x03,0x76,0xac,0x8b,0xda,0xdf,
                       0xe5,0xbf,0xdf,0x22,0x63,0xb2,0x05,0xf7,0x4d,
                       0xfa,0x53,0xdb,0x7a,0x29,0xe5,0x07,0x8f,0x5c,
                       0x34,0xa2,0x68,0x11,0x97,0x36,0xba,0x39,0x09,
                       0x61,0xf6 },
        .data_len = 56,
        .hash =      { 0x13,0x2c,0xfa,0x7e,0x71,0xfe,0x09,0x91,0xab,
                       0xbd,0x88,0xef,0x58,0x8a,0xc9,0x5a,0xc9,0x28,
                       0x9b,0x1d,0x77,0x5b,0x42,0x03,0x35,0x67,0xdd,
                       0x33 },
        .hash_len = 28,
        .chunks = { 28, 0, 28 },
        .num_chunks = 3,
    }, {
        .data =      { 0x0e,0xef,0x94,0x7f,0x1e,0x4f,0x01,0xcd,0xb5,
                       0x48,0x1c,0xa6,0xea,0xa2,0x5f,0x2c,0xac,0xa4,
                       0xc4,0x01,0x61,0x28,0x88,0xfe,0xce,0xf5,0x2e,
                       0x28,0x37,0x48,0xc8,0xdf,0xc7,0xb4,0x72,0x59,
                       0x32,0x2c,0x1f,0x4f,0x98,0x5f,0x98,0xf6,0xad,
                       0x44,0xc1,0x31,0x17,0xf5,0x1e,0x05,0x17,0xc0,
                       0x97,0x4d,0x6c,0x7b,0x78,0xaf,0x74,0x19,0xbc,
                       0xce,0x95,0x7b,0x8b,0xc1,0xdb,0x88,0x01,0xc5,
                       0xe2,0x80,0x31,0x2e,0xf7,0x8d,0x6a,0xa4,0x7a,
                       0x9c,0xb9,0x8b,0x86,0x6a,0xae,0xc3,0xd5,0xe2,
                       0x63,0x92,0xdd,0xa6,0xbb,0xde,0x3f,0xec,0xe8,
                       0xa0,0x62,0x8b,0x30,0x95,0x5b,0x55,0xf0,0x37,
                       0x11,0xa8,0xe1,0xeb,0x9e,0x40,0x9a,0x7c,0xf8,
                       0x4f,0x56,0xc8,0xd0,0xd0,0xf8,0xb9,0xba,0x18,
                       0x4c,0x77,0x8f,0xae,0x90,0xdc,0x0f,0x5c,0x33,
                       0x29,0xcb,0x86,0xdc,0xf7,0x43,0xbb,0xae },
        .data_len = 143,
        .hash =      { 0x98,0xec,0x52,0xc2,0x1c,0xb9,0x88,0xb1,0x43,
                       0x4b,0x16,0x53,0xdd,0x4a,0xc8,0x06,0xd1,0x18,
                       0xde,0x6a,0xf1,0xbb,0x47,0x1c,0x16,0x57,0x7c,
                       0x34 },
        .hash_len = 28,
        .chunks = { 71, 71, 1, 0 },
        .num_chunks = 4,
    }
};

/** SHA3 test Vectors for Hashing Byte-Oriented Messages
NIST SHA3-256 test vectors
http://csrc.nist.gov/groups/STM/cavp/secure-hashing.html
 **/
struct DIGEST_TEST_VECTOR sha3_256_digest_test_vector[] = {
    {
        .data =      { 0x00 },
        .data_len = 0,
        .hash =      { 0xa7,0xff,0xc6,0xf8,0xbf,0x1e,0xd7,0x66,0x51,
                       0xc1,0x47,0x56,0xa0,0x61,0xd6,0x62,0xf5,0x80,
                       0xff,0x4d,0xe4,0x3b,0x49,0xfa,0x82,0xd8,0x0a,
                       0x4b,0x80,0xf8,0x43,0x4a },
        .hash_len = 32,
    }, {
        .data =      { 0xe9 },
        .data_len = 1,
        .hash =      { 0xf0,0xd0,0x4d,0xd1,0xe6,0xcf,0xc2,0x9a,0x44,
                       0x60,0xd5,0x21,0x79,0x68,0x52,0xf2,0x5d,0x9e,
                       0xf8,0xd2,0x8b,0x44,0xee,0x91,0xff,0x5b,0x75,
                       0x9d,0x72,0xc1,0xe6,0xd6 },
        .hash_len = 32,
    }, {
        .data =      { 0xb0,0x53,0xfa },
        .data_len = 3,
        .hash =      { 0x9d,0x0f,0xf0,0x86,0xcd,0x0e,0xc0,0x6a,0x68,
                       0x2c,0x51,0xc0,0x94,0xdc,0x73,0xab,0xdc,0x49,
                       0x20,0x04,0x29,0x23,0x44,0xbd,0x41,0xb8,0x2a,
                       0x60,0x49,0x8c,0xcf,0xdb },
        .hash_len = 32,
    }, {
        .data =      { 0xc1,0x78,0xce,0x0f,0x72,0x0a,0x6d,0x73,0xc6,
                       0xcf,0x1c,0xaa,0x90,0x5e,0xe7,0x24,0xd5,0xba,
                       0x94,0x1c,0x2e,0x26,0x28,0x13,0x6e,0x3a,0xad,
                       0x7d,0x85,0x37,0x33,0xba },
        .data_len = 32,
        .hash =      { 0x64,0x53,0x7b,0x87,0x89,0x28,0x35,0xff,0x09,
                       0x63,0xef,0x9a,0xd5,0x14,0x5a,0xb4,0xcf,0xce,
                       0x5d,0x30,0x3a,0x0c,0xb0,0x41,0x5b,0x3b,0x03,
                       0xf9,0xd1,0x6e,0x7d,0x6b },
        .hash_len = 32,
        .chunks = { 16, -1, 16, 0 },
        .num_chunks = 4,
    }, {
        .data =      { 0x00,0xff,0x6c,0x96,0xb7,0xaa,0x3c,0xf2,0x7d,
                       0x03,0x6c,0xf2,0x0a,0xf7,0x03,0x14,0x34,0x11,
                       0x32,0x52,0x57,0x4b,0xda,0x9c,0xf9,0x24,0x4d,
                       0x85,0xae,0xf2,0x59,0x3d,0x3a,0x7a,0x83,0xbf,
                       0xf6,0xbe,0x90,0x4b,0x75,0x16,0x4a,0x17,0x66,
                       0x82,0x80,0x42,0xbc,0x3f,0x4f,0x09,0x0d,0x98,
                       0xa0,0x3d },
        .data_len = 56,
        .hash =      { 0xd0,0x00,0xea,0xfc,0xa3,0x48,0x15,0x78,0x3b,
                       0xed,0x9b,0x05,0x0c,0x69,0x01,0xc9,0x7f,0x2e,
                       0x77,0xd4,0x77,0x1a,0x0e,0xd7,0x24,0xdd,0x8f,
                       0x6f,0xf1,0x44,0x87,0x91 },
        .hash_len = 32,
        .chunks = { 28, 0, 28 },
        .num_chunks = 3,
    }, {
        .data =      { 0xb1,0xf6,0x07,0x65,0x09,0x93,0x84,0x32,0x14,
                       0x5b,0xb1,0x5d,0xbe,0x1a,0x7b,0x2e,0x00,0x79,
                       0x34,0xbe,0x5f,0x75,0x39,0x08,0xb5,0x0f,0xd2,
                       0x43,0x33,0x45,0x59,0x70,0xa7,0x42,0x9f,0x2f,
                       0xfb,0xd2,0x8b,0xd6,0xfe,0x18,0x04,0xc4,0x68,
                       0x83,0x11,0xf3,0x18,0xfe,0x3f,0xcd,0x9f,0x67,
                       0x44,0x41,0x02,0x43,0xe1,0x15,0xbc,0xb0,0x0d,
                       0x7e,0x03,0x9a,0x4f,0xee,0x4c,0x32,0x6c,0x2d,
                       0x11,0x9c,0x42,0xab,0xd2,0xe8,0xf4,0x15,0x5a,
                       0x44,0x47,0x26,0x43,0x70,0x4c,0xc0,0xbc,0x72,
                       0x40,0x3b,0x8a,0x8a,0xb0,0xfd,0x4d,0x68,0xe0,
                       0x4a,0x05,0x9d,0x6e,0x5e,0xd4,0x50,0x33,0xb9,
                       0x06,0x32,0x6a,0xbb,0x4e,0xb4,0x14,0x70,0x52,
                       0x77,0x9b,0xad,0x6a,0x03,0xb5,0x5c,0xa5,0xbd,
                       0x8b,0x14,0x0e,0x13,0x1b,0xed,0x2d,0xfa,0xda },
        .data_len = 135,
        .hash =      { 0xf8,0x2d,0x96,0x02,0xb2,0x31,0xd3,0x32,0xd9,
                       0x02,0xcb,0x64,0x36,0xb1,0x5a,0xef,0x89,0xac,
                       0xc5,0x91,0xcb,0x86,0x26,0x23,0x3c,0xed,0x20,
                       0xc0,0xa6,0xe8,0x0d,0x7a },
        .hash_len = 32,
        .chunks = { 67, 67, 1, 0 },
        .num_chunks = 4,
    }
};

/** SHA3 test Vectors for Hashing Byte-Oriented Messages
NIST SHA3-224 test vectors
http://csrc.nist.gov/groups/STM/cavp/secure-hashing.html
 **/
struct DIGEST_TEST_VECTOR sha3_384_digest_test_vector[] = {
    {
        .data =      { 0x00 },
        .data_len = 0,
        .hash =      { 0x0c,0x63,0xa7,0x5b,0x84,0x5e,0x4f,0x7d,0x01,
                       0x10,0x7d,0x85,0x2e,0x4c,0x24,0x85,0xc5,0x1a,
                       0x50,0xaa,0xaa,0x94,0xfc,0x61,0x99,0x5e,0x71,
                       0xbb,0xee,0x98,0x3a,0x2a,0xc3,0x71,0x38,0x31,
                       0x26,0x4a,0xdb,0x47,0xfb,0x6b,0xd1,0xe0,0x58,
                       0xd5,0xf0,0x04 },
        .hash_len = 48,
    }, {
        .data =      { 0x80 },
        .data_len = 1,
        .hash =      { 0x75,0x41,0x38,0x48,0x52,0xe1,0x0f,0xf1,0x0d,
                       0x5f,0xb6,0xa7,0x21,0x3a,0x4a,0x6c,0x15,0xcc,
                       0xc8,0x6d,0x8b,0xc1,0x06,0x8a,0xc0,0x4f,0x69,
                       0x27,0x71,0x42,0x94,0x4f,0x4e,0xe5,0x0d,0x91,
                       0xfd,0xc5,0x65,0x53,0xdb,0x06,0xb2,0xf5,0x03,
                       0x9c,0x8a,0xb7 },
        .hash_len = 48,
    }, {
        .data =      { 0x6a,0xb7,0xd6 },
        .data_len = 3,
        .hash =      { 0xea,0x12,0xd6,0xd3,0x2d,0x69,0xad,0x21,0x54,
                       0xa5,0x7e,0x0e,0x1b,0xe4,0x81,0xa4,0x5a,0xdd,
                       0x73,0x9e,0xe7,0xdd,0x6e,0x2a,0x27,0xe5,0x44,
                       0xb6,0xc8,0xb5,0xad,0x12,0x26,0x54,0xbb,0xf9,
                       0x51,0x34,0xd5,0x67,0x98,0x71,0x56,0x29,0x5d,
                       0x5e,0x57,0xdb },
        .hash_len = 48,
    }, {
        .data =      { 0x61,0x89,0x59,0x7e,0x01,0x98,0xa1,0x8c,0x65,
                       0xfa,0x0b,0xdd,0x07,0x97,0xf1,0x30,0x37,0xc7,
                       0x5c,0x40,0x58,0xb7,0xd3,0x45,0x4c,0x0f,0x71,
                       0xbd,0x2d,0xd1,0x3b,0x6c },
        .data_len = 32,
        .hash =      { 0x11,0x06,0x30,0xca,0x76,0x31,0xb7,0x62,0x0e,
                       0x6b,0xee,0x6e,0xd6,0xe9,0x29,0x09,0x89,0x65,
                       0x57,0x19,0x36,0xc3,0x48,0x29,0x48,0x49,0x83,
                       0xeb,0xa9,0x53,0x2b,0x81,0x75,0x52,0x8c,0x22,
                       0x8c,0x57,0x43,0x94,0x53,0xf0,0x27,0xa4,0xf7,
                       0xc8,0x3c,0xa3 },
        .hash_len = 48,
        .chunks = { 16, -1, 16, 0 },
        .num_chunks = 4,
    }, {
        .data =      { 0x26,0x8c,0x7b,0x3a,0x84,0x84,0x9f,0xec,0x5c,
                       0x76,0x9b,0xc4,0xad,0x37,0x7d,0xea,0x10,0xc9,
                       0xd2,0x0c,0x91,0xdd,0x17,0xfd,0xbd,0x96,0x70,
                       0xa2,0xfc,0x90,0x9d,0x0e,0x21,0x21,0x29,0xec,
                       0x40,0xde,0xe4,0x1d,0xbf,0x61,0x94,0xa3,0xb0,
                       0x4a,0xe8,0xbe,0x5e,0x84,0xad,0x54,0x26,0xca,
                       0x44,0x96 },
        .data_len = 56,
        .hash =      { 0x0d,0xfc,0x6f,0xfc,0xf4,0xa3,0x87,0xec,0x09,
                       0xff,0x86,0x2c,0x61,0x39,0xa6,0xf7,0xac,0x77,
                       0xab,0xb2,0xb5,0xe1,0xf6,0xdc,0x81,0x4e,0xb7,
                       0x15,0x25,0xf8,0x65,0x7a,0xc7,0x4a,0x76,0x97,
                       0xc2,0x97,0x5c,0x70,0xa5,0x43,0xaf,0x0e,0x22,
                       0x7d,0x03,0xca },
        .hash_len = 48,
        .chunks = { 28, 0, 28 },
        .num_chunks = 3,
    }, {
        .data =      { 0x6c,0x36,0x14,0x76,0x52,0xe7,0x1b,0x56,0x0b,
                       0xec,0xbc,0xa1,0xe7,0x65,0x6c,0x81,0xb4,0xf7,
                       0x0b,0xec,0xe2,0x63,0x21,0xd5,0xe5,0x5e,0x67,
                       0xa3,0xdb,0x9d,0x89,0xe2,0x6f,0x2f,0x2a,0x38,
                       0xfd,0x0f,0x28,0x9b,0xf7,0xfa,0x22,0xc2,0x87,
                       0x7e,0x38,0xd9,0x75,0x54,0x12,0x79,0x4c,0xef,
                       0x24,0xd7,0xb8,0x55,0x30,0x3c,0x33,0x2e,0x0c,
                       0xb5,0xe0,0x1a,0xa5,0x0b,0xb7,0x48,0x44,0xf5,
                       0xe3,0x45,0x10,0x8d,0x68,0x11,0xd5,0x01,0x09,
                       0x78,0x03,0x8b,0x69,0x9f,0xfa,0xa3,0x70,0xde,
                       0x84,0x73,0xf0,0xcd,0xa3,0x8b,0x89,0xa2,0x8e,
                       0xd6,0xca,0xba,0xf6 },
        .data_len = 103,
        .hash =      { 0xb1,0x31,0x91,0x92,0xdf,0x11,0xfa,0xa0,0x0d,
                       0x3c,0x4b,0x06,0x8b,0xec,0xc8,0xf1,0xba,0x3b,
                       0x00,0xe0,0xd1,0xff,0x1f,0x93,0xc1,0x1a,0x36,
                       0x63,0x52,0x2f,0xdb,0x92,0xab,0x3c,0xca,0x38,
                       0x96,0x34,0x68,0x7c,0x63,0x2e,0x0a,0x4b,0x5a,
                       0x26,0xce,0x92 },
        .hash_len = 48,
        .chunks = { 51, 51, 1, 0 },
        .num_chunks = 4,
    }
};

/** SHA3 test Vectors for Hashing Byte-Oriented Messages
NIST SHA3-224 test vectors
http://csrc.nist.gov/groups/STM/cavp/secure-hashing.html
 **/
struct DIGEST_TEST_VECTOR sha3_512_digest_test_vector[] = {
    {
        .data =      { 0x00 },
        .data_len = 0,
        .hash =      { 0xa6,0x9f,0x73,0xcc,0xa2,0x3a,0x9a,0xc5,0xc8,
                       0xb5,0x67,0xdc,0x18,0x5a,0x75,0x6e,0x97,0xc9,
                       0x82,0x16,0x4f,0xe2,0x58,0x59,0xe0,0xd1,0xdc,
                       0xc1,0x47,0x5c,0x80,0xa6,0x15,0xb2,0x12,0x3a,
                       0xf1,0xf5,0xf9,0x4c,0x11,0xe3,0xe9,0x40,0x2c,
                       0x3a,0xc5,0x58,0xf5,0x00,0x19,0x9d,0x95,0xb6,
                       0xd3,0xe3,0x01,0x75,0x85,0x86,0x28,0x1d,0xcd,
                       0x26 },
        .hash_len = 64,
    }, {
        .data =      { 0xe5 },
        .data_len = 1,
        .hash =      { 0x15,0x02,0x40,0xba,0xf9,0x5f,0xb3,0x6f,0x8c,
                       0xcb,0x87,0xa1,0x9a,0x41,0x76,0x7e,0x7a,0xed,
                       0x95,0x12,0x50,0x75,0xa2,0xb2,0xdb,0xba,0x6e,
                       0x56,0x5e,0x1c,0xe8,0x57,0x5f,0x2b,0x04,0x2b,
                       0x62,0xe2,0x9a,0x04,0xe9,0x44,0x03,0x14,0xa8,
                       0x21,0xc6,0x22,0x41,0x82,0x96,0x4d,0x8b,0x55,
                       0x7b,0x16,0xa4,0x92,0xb3,0x80,0x6f,0x4c,0x39,
                       0xc1 },
        .hash_len = 64,
    }, {
        .data =      { 0x37,0xd5,0x18 },
        .data_len = 3,
        .hash =      { 0x4a,0xa9,0x6b,0x15,0x47,0xe6,0x40,0x2c,0x0e,
                       0xee,0x78,0x1a,0xca,0xa6,0x60,0x79,0x7e,0xfe,
                       0x26,0xec,0x00,0xb4,0xf2,0xe0,0xae,0xc4,0xa6,
                       0xd1,0x06,0x88,0xdd,0x64,0xcb,0xd7,0xf1,0x2b,
                       0x3b,0x6c,0x7f,0x80,0x2e,0x20,0x96,0xc0,0x41,
                       0x20,0x8b,0x92,0x89,0xae,0xc3,0x80,0xd1,0xa7,
                       0x48,0xfd,0xfc,0xd4,0x12,0x85,0x53,0xd7,0x81,
                       0xe3 },
        .hash_len = 64,
    }, {
        .data =      { 0x7c,0x16,0x88,0x21,0x7b,0x31,0x32,0x78,0xb9,
                       0xea,0xe8,0xed,0xcf,0x8a,0xa4,0x27,0x16,0x14,
                       0x29,0x6d,0x0c,0x1e,0x89,0x16,0xf9,0xe0,0xe9,
                       0x40,0xd2,0x8b,0x88,0xc5 },
        .data_len = 32,
        .hash =      { 0x62,0x7b,0xa4,0xde,0x74,0xd0,0x5b,0xb6,0xdf,
                       0x89,0x91,0x11,0x2e,0x4d,0x37,0x3b,0xfc,0xed,
                       0x37,0xac,0xde,0x13,0x04,0xe0,0xf6,0x64,0xf2,
                       0x9f,0xa1,0x26,0xcb,0x49,0x7c,0x8a,0x1b,0x71,
                       0x7b,0x99,0x29,0x12,0x08,0x83,0xec,0x88,0x98,
                       0x96,0x8e,0x46,0x49,0x01,0x3b,0x76,0x0a,0x21,
                       0x80,0xa9,0xdc,0x0f,0xc9,0xb2,0x7f,0x5b,0x7f,
                       0x3b },
        .hash_len = 64,
        .chunks = { 16, -1, 16, 0 },
        .num_chunks = 4,
    }, {
        .data =      { 0x30,0x2f,0xa8,0x4f,0xda,0xa8,0x20,0x81,0xb1,
                       0x19,0x2b,0x84,0x7b,0x81,0xdd,0xea,0x10,0xa9,
                       0xf0,0x5a,0x0f,0x04,0x13,0x8f,0xd1,0xda,0x84,
                       0xa3,0x9b,0xa5,0xe1,0x8e,0x18,0xbc,0x3c,0xea,
                       0x06,0x2e,0x6d,0xf9,0x2f,0xf1,0xac,0xe8,0x9b,
                       0x3c,0x5f,0x55,0x04,0x31,0x30,0x10,0x8a,0xbf,
                       0x63,0x1e },
        .data_len = 56,
        .hash =      { 0x8c,0x8e,0xaa,0xe9,0xa4,0x45,0x64,0x3a,0x37,
                       0xdf,0x34,0xcf,0xa6,0xa7,0xf0,0x9d,0xec,0xca,
                       0xb2,0xa2,0x22,0xc4,0x21,0xd2,0xfc,0x57,0x4b,
                       0xbc,0x56,0x41,0xe5,0x04,0x35,0x43,0x91,0xe8,
                       0x1e,0xb5,0x13,0x02,0x80,0xb1,0x22,0x68,0x12,
                       0x55,0x6d,0x47,0x4e,0x95,0x1b,0xb7,0x8d,0xbd,
                       0xd9,0xb7,0x7d,0x19,0xf6,0x47,0xe2,0xe7,0xd7,
                       0xbe },
        .hash_len = 64,
        .chunks = { 28, 0, 28 },
        .num_chunks = 3,
    }, {
        .data =      { 0xb0,0xde,0x04,0x30,0xc2,0x00,0xd7,0x4b,0xf4,
                       0x1e,0xa0,0xc9,0x2f,0x8f,0x28,0xe1,0x1b,0x68,
                       0x00,0x6a,0x88,0x4e,0x0d,0x4b,0x0d,0x88,0x45,
                       0x33,0xee,0x58,0xb3,0x8a,0x43,0x8c,0xc1,0xa7,
                       0x57,0x50,0xb6,0x43,0x4f,0x46,0x7e,0x2d,0x0c,
                       0xd9,0xaa,0x40,0x52,0xce,0xb7,0x93,0x29,0x1b,
                       0x93,0xef,0x83,0xfd,0x5d,0x86,0x20,0x45,0x6c,
                       0xe1,0xaf,0xf2,0x94,0x1b,0x36,0x05,0xa4 },
        .data_len = 71,
        .hash =      { 0x9e,0x9e,0x46,0x9c,0xa9,0x22,0x6c,0xd0,0x12,
                       0xf5,0xc9,0xcc,0x39,0xc9,0x6a,0xdc,0x22,0xf4,
                       0x20,0x03,0x0f,0xce,0xe3,0x05,0xa0,0xed,0x27,
                       0x97,0x4e,0x3c,0x80,0x27,0x01,0x60,0x3d,0xac,
                       0x87,0x3a,0xe4,0x47,0x6e,0x9c,0x3d,0x57,0xe5,
                       0x55,0x24,0x48,0x3f,0xc0,0x1a,0xda,0xef,0x87,
                       0xda,0xa9,0xe3,0x04,0x07,0x8c,0x59,0x80,0x27,
                       0x57 },
        .hash_len = 64,
        .chunks = { 35, 35, 1, 0 },
        .num_chunks = 4,
    }
};

struct digest_test_suite_info {
    const char *name;
    unsigned int tvcount;
    struct DIGEST_TEST_VECTOR *tv;
    CK_MECHANISM mech;
};

#define NUM_DIGEST_TEST_SUITES 15
struct digest_test_suite_info digest_test_suites[] = {
    {
        .name = "SHA-1",
        .tvcount = 7,
        .tv = sha1_digest_test_vector,
        .mech = {CKM_SHA_1, 0, 0},
    }, {
        .name = "SHA-224",
        .tvcount = 6,
        .tv = sha224_digest_test_vector,
        .mech = {CKM_SHA224, 0, 0},
    }, {
        .name = "SHA-256",
        .tvcount = 6,
        .tv = sha256_digest_test_vector,
        .mech = {CKM_SHA256, 0, 0},
    }, {
        .name = "SHA-384",
        .tvcount = 8,
        .tv = sha384_digest_test_vector,
        .mech = {CKM_SHA384, 0, 0},
    }, {
        .name = "SHA-512",
        .tvcount = 7,
        .tv = sha512_digest_test_vector,
        .mech = {CKM_SHA512, 0, 0},
    }, {
        .name = "SHA-512/224",
        .tvcount = 7,
        .tv = sha512_224_digest_test_vector,
        .mech = {CKM_SHA512_224, 0, 0},
    }, {
        .name = "SHA-512/256",
        .tvcount = 7,
        .tv = sha512_256_digest_test_vector,
        .mech = {CKM_SHA512_256, 0, 0},
    }, {
        .name = "MD2",
        .tvcount = 7,
        .tv = md2_digest_test_vector,
        .mech = {CKM_MD2, 0, 0},
    }, {
        .name = "MD5",
        .tvcount = 3,
        .tv = md5_digest_test_vector,
        .mech = {CKM_MD5, 0, 0},
    }, {
        .name = "RIPEMD-128",
        .tvcount = 1,
        .tv = ripemd128_digest_test_vector,
        .mech = {CKM_RIPEMD128, 0, 0},
    }, {
        .name = "RIPEMD-160",
        .tvcount = 1,
        .tv = ripemd160_digest_test_vector,
        .mech = {CKM_RIPEMD160, 0, 0},
    }, {
        .name = "IBM-SHA3-224",
        .tvcount = 6,
        .tv = sha3_224_digest_test_vector,
        .mech = {CKM_IBM_SHA3_224, 0, 0},
    }, {
        .name = "IBM-SHA3-256",
        .tvcount = 6,
        .tv = sha3_256_digest_test_vector,
        .mech = {CKM_IBM_SHA3_256, 0, 0},
    }, {
        .name = "IBM-SHA3-384",
        .tvcount = 6,
        .tv = sha3_384_digest_test_vector,
        .mech = {CKM_IBM_SHA3_384, 0, 0},
    }, {
        .name = "IBM-SHA3-512",
        .tvcount = 6,
        .tv = sha3_512_digest_test_vector,
        .mech = {CKM_IBM_SHA3_512, 0, 0},
    },
};

struct HMAC_TEST_VECTOR {
    /* test vector inputs */
    CK_BYTE key[512];
    CK_ULONG key_len;
    CK_BYTE data[512];
    CK_ULONG data_len;
    CK_ULONG mac_len;
    CK_BYTE mac[512];
    int chunks[MAX_CHUNKS];
    int num_chunks;
};

/*  MD5 HMAC test vectors from RFC2202:
 *  http://www.faqs.org/rfcs/rfc2202.html
 */
struct HMAC_TEST_VECTOR md5_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b},
        .key_len = 16,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 16,
        .mac = {0x92,0x94,0x72,0x7a,0x36,0x38,0xBB,0x1C,0x13,
                0xf4,0x8E,0xf8,0x15,0x8b,0xfc,0x9d}
    }, {
        .key = {'J','e','f','e'},
        .key_len = 4,
        .data =	{"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 16,
        .mac = {0x75,0x0c,0x78,0x3e,0x6a,0xb0,0xb5,0x03,0xea,
                0xa8,0x6e,0x31,0x0a,0x5d,0xb7,0x38}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 16,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 16,
        .mac = {0x56,0xbe,0x34,0x52,0x1d,0x14,0x4c,0x88,0xdb,
                0xb8,0xc7,0x33,0xf0,0xe8,0xb3,0xf6}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 16,
        .mac = {0x69,0x7e,0xaf,0x0a,0xca,0x3a,0x3a,0xea,0x3a,
                0x75,0x16,0x47,0x46,0xff,0xaa,0x79}
    }, {
        .key = {0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
                0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c},
        .key_len = 16,
        .data = {"Test With Truncation"},
        .data_len = 20,
        .mac_len = 16,
        .mac = {0x56,0x46,0x1e,0xf2,0x34,0x2e,0xdc,0x00,0xf9,
                0xba,0xb9,0x95,0x69,0x0e,0xfd,0x4c}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 80,
        .data = {"Test Using Larger Than Block-Size Key - Hash Key First"},
        .data_len = 54,
        .mac_len = 16,
        .mac = {0x6b,0x1a,0xb7,0xfe,0x4b,0xd7,0xbf,0x8f,0x0b,
                0x62,0xe6,0xce,0x61,0xb9,0xd0,0xcd}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 80,
        .data = {"Test Using Larger Than Block-Size Key and Larger Than One "
                 "Block-Size Data" },
        .data_len = 73,
        .mac_len = 16,
        .mac = {0x6f,0x63,0x0f,0xad,0x67,0xcd,0xa0,0xee,0x1f,
                0xb1,0xf5,0x62,0xdb,0x3a,0xa5,0x3e}
    },
};

/*  SHA1 HMAC test vectors from RFC2202:
 *  http://www.faqs.org/rfcs/rfc2202.html
 */
struct HMAC_TEST_VECTOR sha1_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 20,
        .mac = {0xb6,0x17,0x31,0x86,0x55,0x05,0x72,0x64,0xe2,
                0x8b,0xc0,0xb6,0xfb,0x37,0x8c,0x8e,0xf1,0x46,
                0xbe,0x00}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 20,
        .mac = {0xef,0xfc,0xdf,0x6a,0xe5,0xeb,0x2f,0xa2,0xd2,
                0x74,0x16,0xd5,0xf1,0x84,0xdf,0x9c,0x25,0x9a,
                0x7c,0x79}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa},
        .key_len = 20,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 20,
        .mac = {0x12,0x5d,0x73,0x42,0xb9,0xac,0x11,0xcd,0x91,
                0xa3,0x9a,0xf4,0x8a,0xa1,0x7b,0x4f,0x63,0xf1,
                0x75,0xd3}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19 },
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 20,
        .mac = {0x4c,0x90,0x07,0xf4,0x02,0x62,0x50,0xc6,0xbc,
                0x84,0x14,0xf9,0xbf,0x50,0xc8,0x6c,0x2d,0x72,
                0x35,0xda}
    }, {
        .key = {0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
                0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
                0x0c,0x0c},
        .key_len = 20,
        .data =	{"Test With Truncation"},
        .data_len = 20,
        .mac_len = 20,
        .mac = {0x4c,0x1a,0x03,0x42,0x4b,0x55,0xe0,0x7f,0xe7,
                0xf2,0x7b,0xe1,0xd5,0x8b,0xb9,0x32,0x4a,0x9a,
                0x5a,0x04}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 80,
        .data = {"Test Using Larger Than Block-Size Key - Hash Key First"},
        .data_len = 54,
        .mac_len = 20,
        .mac = {0xaa,0x4a,0xe5,0xe1,0x52,0x72,0xd0,0x0e,0x95,
                0x70,0x56,0x37,0xce,0x8a,0x3b,0x55,0xed,0x40,
                0x21,0x12}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 80,
        .data = {"Test Using Larger Than Block-Size Key and Larger Than One "
                 "Block-Size Data"},
        .data_len = 73,
        .mac_len = 20,
        .mac = {0xe8,0xe9,0x9d,0x0f,0x45,0x23,0x7d,0x78,0x6d,
                0x6b,0xba,0xa7,0x96,0x5c,0x78,0x08,0xbb,0xff,
                0x1a,0x91}
    },
};

/*  Test vectors for HMAC SHA224 from RFC4231:
 *  https://tools.ietf.org/html/rfc4231
 */
struct HMAC_TEST_VECTOR sha224_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 28,
        .mac = {0x89,0x6f,0xb1,0x12,0x8a,0xbb,0xdf,0x19,0x68,
                0x32,0x10,0x7c,0xd4,0x9d,0xf3,0x3f,0x47,0xb4,
                0xb1,0x16,0x99,0x12,0xba,0x4f,0x53,0x68,0x4b,
                0x22}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 28,
        .mac = {0xa3,0x0e,0x01,0x09,0x8b,0xc6,0xdb,0xbf,0x45,
                0x69,0x0f,0x3a,0x7e,0x9e,0x6d,0x0f,0x8b,0xbe,
                0xa2,0xa3,0x9e,0x61,0x48,0x00,0x8f,0xd0,0x5e,
                0x44}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa},
        .key_len = 20,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 28,
        .mac = {0x7f,0xb3,0xcb,0x35,0x88,0xc6,0xc1,0xf6,0xff,
                0xa9,0x69,0x4d,0x7d,0x6a,0xd2,0x64,0x93,0x65,
                0xb0,0xc1,0xf6,0x5d,0x69,0xd1,0xec,0x83,0x33,
                0xea}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 28,
        .mac = {0x6c,0x11,0x50,0x68,0x74,0x01,0x3c,0xac,0x6a,
                0x2a,0xbc,0x1b,0xb3,0x82,0x62,0x7c,0xec,0x6a,
                0x90,0xd8,0x6e,0xfc,0x01,0x2d,0xe7,0xaf,0xec,
                0x5a}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x65,0x73,0x74,0x20,0x55,0x73,0x69,0x6e,
                 0x67,0x20,0x4c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x54,0x68,0x61,0x6e,0x20,0x42,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x53,0x69,0x7a,0x65,0x20,0x4b,0x65,
                 0x79,0x20,0x2d,0x20,0x48,0x61,0x73,0x68,0x20,
                 0x4b,0x65,0x79,0x20,0x46,0x69,0x72,0x73,0x74},
        .data_len = 54,
        .mac_len = 28,
        .mac = {0x95,0xe9,0xa0,0xdb,0x96,0x20,0x95,0xad,0xae,
                0xbe,0x9b,0x2d,0x6f,0x0d,0xbc,0xe2,0xd4,0x99,
                0xf1,0x12,0xf2,0xd2,0xb7,0x27,0x3f,0xa6,0x87,
                0x0e}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x68,0x69,0x73,0x20,0x69,0x73,0x20,0x61,
                 0x20,0x74,0x65,0x73,0x74,0x20,0x75,0x73,0x69,
                 0x6e,0x67,0x20,0x61,0x20,0x6c,0x61,0x72,0x67,
                 0x65,0x72,0x20,0x74,0x68,0x61,0x6e,0x20,0x62,
                 0x6c,0x6f,0x63,0x6b,0x2d,0x73,0x69,0x7a,0x65,
                 0x20,0x6b,0x65,0x79,0x20,0x61,0x6e,0x64,0x20,
                 0x61,0x20,0x6c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x74,0x68,0x61,0x6e,0x20,0x62,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x73,0x69,0x7a,0x65,0x20,0x64,0x61,
                 0x74,0x61,0x2e,0x20,0x54,0x68,0x65,0x20,0x6b,
                 0x65,0x79,0x20,0x6e,0x65,0x65,0x64,0x73,0x20,
                 0x74,0x6f,0x20,0x62,0x65,0x20,0x68,0x61,0x73,
                 0x68,0x65,0x64,0x20,0x62,0x65,0x66,0x6f,0x72,
                 0x65,0x20,0x62,0x65,0x69,0x6e,0x67,0x20,0x75,
                 0x73,0x65,0x64,0x20,0x62,0x79,0x20,0x74,0x68,
                 0x65,0x20,0x48,0x4d,0x41,0x43,0x20,0x61,0x6c,
                 0x67,0x6f,0x72,0x69,0x74,0x68,0x6d,0x2e},
        .data_len = 152,
        .mac_len = 28,
        .mac = {0x3a,0x85,0x41,0x66,0xac,0x5d,0x9f,0x02,0x3f,
                0x54,0xd5,0x17,0xd0,0xb3,0x9d,0xbd,0x94,0x67,
                0x70,0xdb,0x9c,0x2b,0x95,0xc9,0xf6,0xf5,0x65,
                0xd1}
    }
};

/*  Test vectors for HMAC SHA256 from RFC4868:
 *  http://tools.ietf.org/html/rfc4868
 */
struct HMAC_TEST_VECTOR sha256_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 32,
        .mac = {0xb0,0x34,0x4c,0x61,0xd8,0xdb,0x38,0x53,0x5c,
                0xa8,0xaf,0xce,0xaf,0x0b,0xf1,0x2b,0x88,0x1d,
                0xc2,0x00,0xc9,0x83,0x3d,0xa7,0x26,0xe9,0x37,
                0x6c,0x2e,0x32,0xcf,0xf7}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 32,
        .mac = {0x5b,0xdc,0xc1,0x46,0xbf,0x60,0x75,0x4e,0x6a,
                0x04,0x24,0x26,0x08,0x95,0x75,0xc7,0x5a,0x00,
                0x3f,0x08,0x9d,0x27,0x39,0x83,0x9d,0xec,0x58,
                0xb9,0x64,0xec,0x38,0x43}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa},
        .key_len = 20,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 32,
        .mac = {0x77,0x3e,0xa9,0x1e,0x36,0x80,0x0e,0x46,0x85,
                0x4d,0xb8,0xeb,0xd0,0x91,0x81,0xa7,0x29,0x59,
                0x09,0x8b,0x3e,0xf8,0xc1,0x22,0xd9,0x63,0x55,
                0x14,0xce,0xd5,0x65,0xfe}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 32,
        .mac = {0x82,0x55,0x8a,0x38,0x9a,0x44,0x3c,0x0e,0xa4,
                0xcc,0x81,0x98,0x99,0xf2,0x08,0x3a,0x85,0xf0,
                0xfa,0xa3,0xe5,0x78,0xf8,0x07,0x7a,0x2e,0x3f,
                0xf4,0x67,0x29,0x66,0x5b}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x65,0x73,0x74,0x20,0x55,0x73,0x69,0x6e,
                 0x67,0x20,0x4c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x54,0x68,0x61,0x6e,0x20,0x42,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x53,0x69,0x7a,0x65,0x20,0x4b,0x65,
                 0x79,0x20,0x2d,0x20,0x48,0x61,0x73,0x68,0x20,
                 0x4b,0x65,0x79,0x20,0x46,0x69,0x72,0x73,0x74},
        .data_len = 54,
        .mac_len = 32,
        .mac = {0x60,0xe4,0x31,0x59,0x1e,0xe0,0xb6,0x7f,0x0d,
                0x8a,0x26,0xaa,0xcb,0xf5,0xb7,0x7f,0x8e,0x0b,
                0xc6,0x21,0x37,0x28,0xc5,0x14,0x05,0x46,0x04,
                0x0f,0x0e,0xe3,0x7f,0x54}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x68,0x69,0x73,0x20,0x69,0x73,0x20,0x61,
                 0x20,0x74,0x65,0x73,0x74,0x20,0x75,0x73,0x69,
                 0x6e,0x67,0x20,0x61,0x20,0x6c,0x61,0x72,0x67,
                 0x65,0x72,0x20,0x74,0x68,0x61,0x6e,0x20,0x62,
                 0x6c,0x6f,0x63,0x6b,0x2d,0x73,0x69,0x7a,0x65,
                 0x20,0x6b,0x65,0x79,0x20,0x61,0x6e,0x64,0x20,
                 0x61,0x20,0x6c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x74,0x68,0x61,0x6e,0x20,0x62,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x73,0x69,0x7a,0x65,0x20,0x64,0x61,
                 0x74,0x61,0x2e,0x20,0x54,0x68,0x65,0x20,0x6b,
                 0x65,0x79,0x20,0x6e,0x65,0x65,0x64,0x73,0x20,
                 0x74,0x6f,0x20,0x62,0x65,0x20,0x68,0x61,0x73,
                 0x68,0x65,0x64,0x20,0x62,0x65,0x66,0x6f,0x72,
                 0x65,0x20,0x62,0x65,0x69,0x6e,0x67,0x20,0x75,
                 0x73,0x65,0x64,0x20,0x62,0x79,0x20,0x74,0x68,
                 0x65,0x20,0x48,0x4d,0x41,0x43,0x20,0x61,0x6c,
                 0x67,0x6f,0x72,0x69,0x74,0x68,0x6d,0x2e},
        .data_len = 152,
        .mac_len = 32,
        .mac = {0x9b,0x09,0xff,0xa7,0x1b,0x94,0x2f,0xcb,0x27,
                0x63,0x5f,0xbc,0xd5,0xb0,0xe9,0x44,0xbf,0xdc,
                0x63,0x64,0x4f,0x07,0x13,0x93,0x8a,0x7f,0x51,
                0x53,0x5c,0x3a,0x35,0xe2}
    }
};

/*  Test vectors for HMAC SHA384 from RFC4868:
 *  http://tools.ietf.org/html/rfc4868
 */
struct HMAC_TEST_VECTOR sha384_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 48,
        .mac = {0xaf,0xd0,0x39,0x44,0xd8,0x48,0x95,0x62,0x6b,
                0x08,0x25,0xf4,0xab,0x46,0x90,0x7f,0x15,0xf9,
                0xda,0xdb,0xe4,0x10,0x1e,0xc6,0x82,0xaa,0x03,
                0x4c,0x7c,0xeb,0xc5,0x9c,0xfa,0xea,0x9e,0xa9,
                0x07,0x6e,0xde,0x7f,0x4a,0xf1,0x52,0xe8,0xb2,
                0xfa,0x9c,0xb6}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 48,
        .mac = {0xaf,0x45,0xd2,0xe3,0x76,0x48,0x40,0x31,0x61,
                0x7f,0x78,0xd2,0xb5,0x8a,0x6b,0x1b,0x9c,0x7e,
                0xf4,0x64,0xf5,0xa0,0x1b,0x47,0xe4,0x2e,0xc3,
                0x73,0x63,0x22,0x44,0x5e,0x8e,0x22,0x40,0xca,
                0x5e,0x69,0xe2,0xc7,0x8b,0x32,0x39,0xec,0xfa,
                0xb2,0x16,0x49}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa},
        .key_len = 20,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 48,
        .mac = {0x88,0x06,0x26,0x08,0xd3,0xe6,0xad,0x8a,0x0a,
                0xa2,0xac,0xe0,0x14,0xc8,0xa8,0x6f,0x0a,0xa6,
                0x35,0xd9,0x47,0xac,0x9f,0xeb,0xe8,0x3e,0xf4,
                0xe5,0x59,0x66,0x14,0x4b,0x2a,0x5a,0xb3,0x9d,
                0xc1,0x38,0x14,0xb9,0x4e,0x3a,0xb6,0xe1,0x01,
                0xa3,0x4f,0x27}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 48,
        .mac = {0x3e,0x8a,0x69,0xb7,0x78,0x3c,0x25,0x85,0x19,
                0x33,0xab,0x62,0x90,0xaf,0x6c,0xa7,0x7a,0x99,
                0x81,0x48,0x08,0x50,0x00,0x9c,0xc5,0x57,0x7c,
                0x6e,0x1f,0x57,0x3b,0x4e,0x68,0x01,0xdd,0x23,
                0xc4,0xa7,0xd6,0x79,0xcc,0xf8,0xa3,0x86,0xc6,
                0x74,0xcf,0xfb}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x65,0x73,0x74,0x20,0x55,0x73,0x69,0x6e,
                 0x67,0x20,0x4c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x54,0x68,0x61,0x6e,0x20,0x42,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x53,0x69,0x7a,0x65,0x20,0x4b,0x65,
                 0x79,0x20,0x2d,0x20,0x48,0x61,0x73,0x68,0x20,
                 0x4b,0x65,0x79,0x20,0x46,0x69,0x72,0x73,0x74},
        .data_len = 54,
        .mac_len = 48,
        .mac = {0x4e,0xce,0x08,0x44,0x85,0x81,0x3e,0x90,0x88,
                0xd2,0xc6,0x3a,0x04,0x1b,0xc5,0xb4,0x4f,0x9e,
                0xf1,0x01,0x2a,0x2b,0x58,0x8f,0x3c,0xd1,0x1f,
                0x05,0x03,0x3a,0xc4,0xc6,0x0c,0x2e,0xf6,0xab,
                0x40,0x30,0xfe,0x82,0x96,0x24,0x8d,0xf1,0x63,
                0xf4,0x49,0x52}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x68,0x69,0x73,0x20,0x69,0x73,0x20,0x61,
                 0x20,0x74,0x65,0x73,0x74,0x20,0x75,0x73,0x69,
                 0x6e,0x67,0x20,0x61,0x20,0x6c,0x61,0x72,0x67,
                 0x65,0x72,0x20,0x74,0x68,0x61,0x6e,0x20,0x62,
                 0x6c,0x6f,0x63,0x6b,0x2d,0x73,0x69,0x7a,0x65,
                 0x20,0x6b,0x65,0x79,0x20,0x61,0x6e,0x64,0x20,
                 0x61,0x20,0x6c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x74,0x68,0x61,0x6e,0x20,0x62,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x73,0x69,0x7a,0x65,0x20,0x64,0x61,
                 0x74,0x61,0x2e,0x20,0x54,0x68,0x65,0x20,0x6b,
                 0x65,0x79,0x20,0x6e,0x65,0x65,0x64,0x73,0x20,
                 0x74,0x6f,0x20,0x62,0x65,0x20,0x68,0x61,0x73,
                 0x68,0x65,0x64,0x20,0x62,0x65,0x66,0x6f,0x72,
                 0x65,0x20,0x62,0x65,0x69,0x6e,0x67,0x20,0x75,
                 0x73,0x65,0x64,0x20,0x62,0x79,0x20,0x74,0x68,
                 0x65,0x20,0x48,0x4d,0x41,0x43,0x20,0x61,0x6c,
                 0x67,0x6f,0x72,0x69,0x74,0x68,0x6d,0x2e},
        .data_len = 152,
        .mac_len = 48,
        .mac = {0x66,0x17,0x17,0x8e,0x94,0x1f,0x02,0x0d,0x35,
                0x1e,0x2f,0x25,0x4e,0x8f,0xd3,0x2c,0x60,0x24,
                0x20,0xfe,0xb0,0xb8,0xfb,0x9a,0xdc,0xce,0xbb,
                0x82,0x46,0x1e,0x99,0xc5,0xa6,0x78,0xcc,0x31,
                0xe7,0x99,0x17,0x6d,0x38,0x60,0xe6,0x11,0x0c,
                0x46,0x52,0x3e}
    }
};

/*  Test vectors for HMAC SHA512 from RFC4868:
 *  http://tools.ietf.org/html/rfc4868
 */
struct HMAC_TEST_VECTOR sha512_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 64,
        .mac = {0x87,0xaa,0x7c,0xde,0xa5,0xef,0x61,0x9d,0x4f,
                0xf0,0xb4,0x24,0x1a,0x1d,0x6c,0xb0,0x23,0x79,
                0xf4,0xe2,0xce,0x4e,0xc2,0x78,0x7a,0xd0,0xb3,
                0x05,0x45,0xe1,0x7c,0xde,0xda,0xa8,0x33,0xb7,
                0xd6,0xb8,0xa7,0x02,0x03,0x8b,0x27,0x4e,0xae,
                0xa3,0xf4,0xe4,0xbe,0x9d,0x91,0x4e,0xeb,0x61,
                0xf1,0x70,0x2e,0x69,0x6c,0x20,0x3a,0x12,0x68,
                0x54}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 64,
        .mac = {0x16,0x4b,0x7a,0x7b,0xfc,0xf8,0x19,0xe2,0xe3,
                0x95,0xfb,0xe7,0x3b,0x56,0xe0,0xa3,0x87,0xbd,
                0x64,0x22,0x2e,0x83,0x1f,0xd6,0x10,0x27,0x0c,
                0xd7,0xea,0x25,0x05,0x54,0x97,0x58,0xbf,0x75,
                0xc0,0x5a,0x99,0x4a,0x6d,0x03,0x4f,0x65,0xf8,
                0xf0,0xe6,0xfd,0xca,0xea,0xb1,0xa3,0x4d,0x4a,
                0x6b,0x4b,0x63,0x6e,0x07,0x0a,0x38,0xbc,0xe7,
                0x37}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa},
        .key_len = 20,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 64,
        .mac = {0xfa,0x73,0xb0,0x08,0x9d,0x56,0xa2,0x84,0xef,
                0xb0,0xf0,0x75,0x6c,0x89,0x0b,0xe9,0xb1,0xb5,
                0xdb,0xdd,0x8e,0xe8,0x1a,0x36,0x55,0xf8,0x3e,
                0x33,0xb2,0x27,0x9d,0x39,0xbf,0x3e,0x84,0x82,
                0x79,0xa7,0x22,0xc8,0x06,0xb4,0x85,0xa4,0x7e,
                0x67,0xc8,0x07,0xb9,0x46,0xa3,0x37,0xbe,0xe8,
                0x94,0x26,0x74,0x27,0x88,0x59,0xe1,0x32,0x92,
                0xfb}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 64,
        .mac = {0xb0,0xba,0x46,0x56,0x37,0x45,0x8c,0x69,0x90,
                0xe5,0xa8,0xc5,0xf6,0x1d,0x4a,0xf7,0xe5,0x76,
                0xd9,0x7f,0xf9,0x4b,0x87,0x2d,0xe7,0x6f,0x80,
                0x50,0x36,0x1e,0xe3,0xdb,0xa9,0x1c,0xa5,0xc1,
                0x1a,0xa2,0x5e,0xb4,0xd6,0x79,0x27,0x5c,0xc5,
                0x78,0x80,0x63,0xa5,0xf1,0x97,0x41,0x12,0x0c,
                0x4f,0x2d,0xe2,0xad,0xeb,0xeb,0x10,0xa2,0x98,
                0xdd}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x65,0x73,0x74,0x20,0x55,0x73,0x69,0x6e,
                 0x67,0x20,0x4c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x54,0x68,0x61,0x6e,0x20,0x42,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x53,0x69,0x7a,0x65,0x20,0x4b,0x65,
                 0x79,0x20,0x2d,0x20,0x48,0x61,0x73,0x68,0x20,
                 0x4b,0x65,0x79,0x20,0x46,0x69,0x72,0x73,0x74},
        .data_len = 54,
        .mac_len = 64,
        .mac = {0x80,0xb2,0x42,0x63,0xc7,0xc1,0xa3,0xeb,0xb7,
                0x14,0x93,0xc1,0xdd,0x7b,0xe8,0xb4,0x9b,0x46,
                0xd1,0xf4,0x1b,0x4a,0xee,0xc1,0x12,0x1b,0x01,
                0x37,0x83,0xf8,0xf3,0x52,0x6b,0x56,0xd0,0x37,
                0xe0,0x5f,0x25,0x98,0xbd,0x0f,0xd2,0x21,0x5d,
                0x6a,0x1e,0x52,0x95,0xe6,0x4f,0x73,0xf6,0x3f,
                0x0a,0xec,0x8b,0x91,0x5a,0x98,0x5d,0x78,0x65,
                0x98}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x68,0x69,0x73,0x20,0x69,0x73,0x20,0x61,
                 0x20,0x74,0x65,0x73,0x74,0x20,0x75,0x73,0x69,
                 0x6e,0x67,0x20,0x61,0x20,0x6c,0x61,0x72,0x67,
                 0x65,0x72,0x20,0x74,0x68,0x61,0x6e,0x20,0x62,
                 0x6c,0x6f,0x63,0x6b,0x2d,0x73,0x69,0x7a,0x65,
                 0x20,0x6b,0x65,0x79,0x20,0x61,0x6e,0x64,0x20,
                 0x61,0x20,0x6c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x74,0x68,0x61,0x6e,0x20,0x62,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x73,0x69,0x7a,0x65,0x20,0x64,0x61,
                 0x74,0x61,0x2e,0x20,0x54,0x68,0x65,0x20,0x6b,
                 0x65,0x79,0x20,0x6e,0x65,0x65,0x64,0x73,0x20,
                 0x74,0x6f,0x20,0x62,0x65,0x20,0x68,0x61,0x73,
                 0x68,0x65,0x64,0x20,0x62,0x65,0x66,0x6f,0x72,
                 0x65,0x20,0x62,0x65,0x69,0x6e,0x67,0x20,0x75,
                 0x73,0x65,0x64,0x20,0x62,0x79,0x20,0x74,0x68,
                 0x65,0x20,0x48,0x4d,0x41,0x43,0x20,0x61,0x6c,
                 0x67,0x6f,0x72,0x69,0x74,0x68,0x6d,0x2e},
        .data_len = 152,
        .mac_len = 64,
        .mac = {0xe3,0x7b,0x6a,0x77,0x5d,0xc8,0x7d,0xba,0xa4,
                0xdf,0xa9,0xf9,0x6e,0x5e,0x3f,0xfd,0xde,0xbd,
                0x71,0xf8,0x86,0x72,0x89,0x86,0x5d,0xf5,0xa3,
                0x2d,0x20,0xcd,0xc9,0x44,0xb6,0x02,0x2c,0xac,
                0x3c,0x49,0x82,0xb1,0x0d,0x5e,0xeb,0x55,0xc3,
                0xe4,0xde,0x15,0x13,0x46,0x76,0xfb,0x6d,0xe0,
                0x44,0x60,0x65,0xc9,0x74,0x40,0xfa,0x8c,0x6a,
                0x58}
    }
};

/*
 * Test vectors for HMAC SHA512/224:
 */
struct HMAC_TEST_VECTOR sha512_224_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 28,
        .mac = {0xb2,0x44,0xba,0x01,0x30,0x7c,0x0e,0x7a,0x8c,
                0xca,0xad,0x13,0xb1,0x06,0x7a,0x4c,0xf6,0xb9,
                0x61,0xfe,0x0c,0x6a,0x20,0xbd,0xa3,0xd9,0x20,
                0x39}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 28,
        .mac = {0x4a,0x53,0x0b,0x31,0xa7,0x9e,0xbc,0xce,0x36,
                0x91,0x65,0x46,0x31,0x7c,0x45,0xf2,0x47,0xd8,
                0x32,0x41,0xdf,0xb8,0x18,0xfd,0x37,0x25,0x4b,
                0xde}
    }
};

/*
 * Test vectors for HMAC SHA512/256:
 */
struct HMAC_TEST_VECTOR sha512_256_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 32,
        .mac = {0x9f,0x91,0x26,0xc3,0xd9,0xc3,0xc3,0x30,0xd7,
                0x60,0x42,0x5c,0xa8,0xa2,0x17,0xe3,0x1f,0xea,
                0xe3,0x1b,0xfe,0x70,0x19,0x6f,0xf8,0x16,0x42,
                0xb8,0x68,0x40,0x2e,0xab}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 32,
        .mac = {0x6d,0xf7,0xb2,0x46,0x30,0xd5,0xcc,0xb2,0xee,
                0x33,0x54,0x07,0x08,0x1a,0x87,0x18,0x8c,0x22,
                0x14,0x89,0x76,0x8f,0xa2,0x02,0x05,0x13,0xb2,
                0xd5,0x93,0x35,0x94,0x56}
    }
};

/*  Test vectors for HMAC SHA3-224:
 */
struct HMAC_TEST_VECTOR sha3_224_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 28,
        .mac = {0x3b,0x16,0x54,0x6b,0xbc,0x7b,0xe2,0x70,0x6a,
                0x03,0x1d,0xca,0xfd,0x56,0x37,0x3d,0x98,0x84,
                0x36,0x76,0x41,0xd8,0xc5,0x9a,0xf3,0xc8,0x60,
                0xf7}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 28,
        .mac = {0x7f,0xdb,0x8d,0xd8,0x8b,0xd2,0xf6,0x0d,0x1b,
                0x79,0x86,0x34,0xad,0x38,0x68,0x11,0xc2,0xcf,
                0xc8,0x5b,0xfa,0xf5,0xd5,0x2b,0xba,0xce,0x5e,
                0x66}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa},
        .key_len = 20,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 28,
        .mac = {0x67,0x6c,0xfc,0x7d,0x16,0x15,0x36,0x38,0x78,
                0x03,0x90,0x69,0x2b,0xe1,0x42,0xd2,0xdf,0x7c,
                0xe9,0x24,0xb9,0x09,0xc0,0xc0,0x8d,0xbf,0xdc,
                0x1a}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 28,
        .mac = {0xa9,0xd7,0x68,0x5a,0x19,0xc4,0xe0,0xdb,0xd9,
                0xdf,0x25,0x56,0xcc,0x8a,0x7d,0x2a,0x77,0x33,
                0xb6,0x76,0x25,0xce,0x59,0x4c,0x78,0x27,0x0e,
                0xeb}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x65,0x73,0x74,0x20,0x55,0x73,0x69,0x6e,
                 0x67,0x20,0x4c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x54,0x68,0x61,0x6e,0x20,0x42,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x53,0x69,0x7a,0x65,0x20,0x4b,0x65,
                 0x79,0x20,0x2d,0x20,0x48,0x61,0x73,0x68,0x20,
                 0x4b,0x65,0x79,0x20,0x46,0x69,0x72,0x73,0x74},
        .data_len = 54,
        .mac_len = 28,
        .mac = {0xb4,0xa1,0xf0,0x4c,0x00,0x28,0x7a,0x9b,0x7f,
                0x60,0x75,0xb3,0x13,0xd2,0x79,0xb8,0x33,0xbc,
                0x8f,0x75,0x12,0x43,0x52,0xd0,0x5f,0xb9,0x99,
                0x5f}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x68,0x69,0x73,0x20,0x69,0x73,0x20,0x61,
                 0x20,0x74,0x65,0x73,0x74,0x20,0x75,0x73,0x69,
                 0x6e,0x67,0x20,0x61,0x20,0x6c,0x61,0x72,0x67,
                 0x65,0x72,0x20,0x74,0x68,0x61,0x6e,0x20,0x62,
                 0x6c,0x6f,0x63,0x6b,0x2d,0x73,0x69,0x7a,0x65,
                 0x20,0x6b,0x65,0x79,0x20,0x61,0x6e,0x64,0x20,
                 0x61,0x20,0x6c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x74,0x68,0x61,0x6e,0x20,0x62,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x73,0x69,0x7a,0x65,0x20,0x64,0x61,
                 0x74,0x61,0x2e,0x20,0x54,0x68,0x65,0x20,0x6b,
                 0x65,0x79,0x20,0x6e,0x65,0x65,0x64,0x73,0x20,
                 0x74,0x6f,0x20,0x62,0x65,0x20,0x68,0x61,0x73,
                 0x68,0x65,0x64,0x20,0x62,0x65,0x66,0x6f,0x72,
                 0x65,0x20,0x62,0x65,0x69,0x6e,0x67,0x20,0x75,
                 0x73,0x65,0x64,0x20,0x62,0x79,0x20,0x74,0x68,
                 0x65,0x20,0x48,0x4d,0x41,0x43,0x20,0x61,0x6c,
                 0x67,0x6f,0x72,0x69,0x74,0x68,0x6d,0x2e},
        .data_len = 152,
        .mac_len = 28,
        .mac = {0x05,0xd8,0xcd,0x6d,0x00,0xfa,0xea,0x8d,0x1e,
                0xb6,0x8a,0xde,0x28,0x73,0x0b,0xbd,0x3c,0xba,
                0xb6,0x92,0x9f,0x0a,0x08,0x6b,0x29,0xcd,0x62,
                0xa0}
    }
};

/*  Test vectors for HMAC SHA3-256:
 */
struct HMAC_TEST_VECTOR sha3_256_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 32,
        .mac = {0xba,0x85,0x19,0x23,0x10,0xdf,0xfa,0x96,0xe2,
                0xa3,0xa4,0x0e,0x69,0x77,0x43,0x51,0x14,0x0b,
                0xb7,0x18,0x5e,0x12,0x02,0xcd,0xcc,0x91,0x75,
                0x89,0xf9,0x5e,0x16,0xbb}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 32,
        .mac = {0xc7,0xd4,0x07,0x2e,0x78,0x88,0x77,0xae,0x35,
                0x96,0xbb,0xb0,0xda,0x73,0xb8,0x87,0xc9,0x17,
                0x1f,0x93,0x09,0x5b,0x29,0x4a,0xe8,0x57,0xfb,
                0xe2,0x64,0x5e,0x1b,0xa5}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa},
        .key_len = 20,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 32,
        .mac = {0x84,0xec,0x79,0x12,0x4a,0x27,0x10,0x78,0x65,
                0xce,0xdd,0x8b,0xd8,0x2d,0xa9,0x96,0x5e,0x5e,
                0xd8,0xc3,0x7b,0x0a,0xc9,0x80,0x05,0xa7,0xf3,
                0x9e,0xd5,0x8a,0x42,0x07}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 32,
        .mac = {0x57,0x36,0x6a,0x45,0xe2,0x30,0x53,0x21,0xa4,
                0xbc,0x5a,0xa5,0xfe,0x2e,0xf8,0xa9,0x21,0xf6,
                0xaf,0x82,0x73,0xd7,0xfe,0x7b,0xe6,0xcf,0xed,
                0xb3,0xf0,0xae,0xa6,0xd7}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x65,0x73,0x74,0x20,0x55,0x73,0x69,0x6e,
                 0x67,0x20,0x4c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x54,0x68,0x61,0x6e,0x20,0x42,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x53,0x69,0x7a,0x65,0x20,0x4b,0x65,
                 0x79,0x20,0x2d,0x20,0x48,0x61,0x73,0x68,0x20,
                 0x4b,0x65,0x79,0x20,0x46,0x69,0x72,0x73,0x74},
        .data_len = 54,
        .mac_len = 32,
        .mac = {0xed,0x73,0xa3,0x74,0xb9,0x6c,0x00,0x52,0x35,
                0xf9,0x48,0x03,0x2f,0x09,0x67,0x4a,0x58,0xc0,
                0xce,0x55,0x5c,0xfc,0x1f,0x22,0x3b,0x02,0x35,
                0x65,0x60,0x31,0x2c,0x3b}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x68,0x69,0x73,0x20,0x69,0x73,0x20,0x61,
                 0x20,0x74,0x65,0x73,0x74,0x20,0x75,0x73,0x69,
                 0x6e,0x67,0x20,0x61,0x20,0x6c,0x61,0x72,0x67,
                 0x65,0x72,0x20,0x74,0x68,0x61,0x6e,0x20,0x62,
                 0x6c,0x6f,0x63,0x6b,0x2d,0x73,0x69,0x7a,0x65,
                 0x20,0x6b,0x65,0x79,0x20,0x61,0x6e,0x64,0x20,
                 0x61,0x20,0x6c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x74,0x68,0x61,0x6e,0x20,0x62,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x73,0x69,0x7a,0x65,0x20,0x64,0x61,
                 0x74,0x61,0x2e,0x20,0x54,0x68,0x65,0x20,0x6b,
                 0x65,0x79,0x20,0x6e,0x65,0x65,0x64,0x73,0x20,
                 0x74,0x6f,0x20,0x62,0x65,0x20,0x68,0x61,0x73,
                 0x68,0x65,0x64,0x20,0x62,0x65,0x66,0x6f,0x72,
                 0x65,0x20,0x62,0x65,0x69,0x6e,0x67,0x20,0x75,
                 0x73,0x65,0x64,0x20,0x62,0x79,0x20,0x74,0x68,
                 0x65,0x20,0x48,0x4d,0x41,0x43,0x20,0x61,0x6c,
                 0x67,0x6f,0x72,0x69,0x74,0x68,0x6d,0x2e},
        .data_len = 152,
        .mac_len = 32,
        .mac = {0x65,0xc5,0xb0,0x6d,0x4c,0x3d,0xe3,0x2a,0x7a,
                0xef,0x87,0x63,0x26,0x1e,0x49,0xad,0xb6,0xe2,
                0x29,0x3e,0xc8,0xe7,0xc6,0x1e,0x8d,0xe6,0x17,
                0x01,0xfc,0x63,0xe1,0x23}
    }
};

/*  Test vectors for HMAC SHA3-384:
 */
struct HMAC_TEST_VECTOR sha3_384_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 48,
        .mac = {0x68,0xd2,0xdc,0xf7,0xfd,0x4d,0xdd,0x0a,0x22,
                0x40,0xc8,0xa4,0x37,0x30,0x5f,0x61,0xfb,0x73,
                0x34,0xcf,0xb5,0xd0,0x22,0x6e,0x1b,0xc2,0x7d,
                0xc1,0x0a,0x2e,0x72,0x3a,0x20,0xd3,0x70,0xb4,
                0x77,0x43,0x13,0x0e,0x26,0xac,0x7e,0x3d,0x53,
                0x28,0x86,0xbd}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 48,
        .mac = {0xf1,0x10,0x1f,0x8c,0xbf,0x97,0x66,0xfd,0x67,
                0x64,0xd2,0xed,0x61,0x90,0x3f,0x21,0xca,0x9b,
                0x18,0xf5,0x7c,0xf3,0xe1,0xa2,0x3c,0xa1,0x35,
                0x08,0xa9,0x32,0x43,0xce,0x48,0xc0,0x45,0xdc,
                0x00,0x7f,0x26,0xa2,0x1b,0x3f,0x5e,0x0e,0x9d,
                0xf4,0xc2,0x0a}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa},
        .key_len = 20,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 48,
        .mac = {0x27,0x5c,0xd0,0xe6,0x61,0xbb,0x8b,0x15,0x1c,
                0x64,0xd2,0x88,0xf1,0xf7,0x82,0xfb,0x91,0xa8,
                0xab,0xd5,0x68,0x58,0xd7,0x2b,0xab,0xb2,0xd4,
                0x76,0xf0,0x45,0x83,0x73,0xb4,0x1b,0x6a,0xb5,
                0xbf,0x17,0x4b,0xec,0x42,0x2e,0x53,0xfc,0x31,
                0x35,0xac,0x6e}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 48,
        .mac = {0x3a,0x5d,0x7a,0x87,0x97,0x02,0xc0,0x86,0xbc,
                0x96,0xd1,0xdd,0x8a,0xa1,0x5d,0x9c,0x46,0x44,
                0x6b,0x95,0x52,0x13,0x11,0xc6,0x06,0xfd,0xc4,
                0xe3,0x08,0xf4,0xb9,0x84,0xda,0x2d,0x0f,0x94,
                0x49,0xb3,0xba,0x84,0x25,0xec,0x7f,0xb8,0xc3,
                0x1b,0xc1,0x36}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x65,0x73,0x74,0x20,0x55,0x73,0x69,0x6e,
                 0x67,0x20,0x4c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x54,0x68,0x61,0x6e,0x20,0x42,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x53,0x69,0x7a,0x65,0x20,0x4b,0x65,
                 0x79,0x20,0x2d,0x20,0x48,0x61,0x73,0x68,0x20,
                 0x4b,0x65,0x79,0x20,0x46,0x69,0x72,0x73,0x74},
        .data_len = 54,
        .mac_len = 48,
        .mac = {0x0f,0xc1,0x95,0x13,0xbf,0x6b,0xd8,0x78,0x03,
                0x70,0x16,0x70,0x6a,0x0e,0x57,0xbc,0x52,0x81,
                0x39,0x83,0x6b,0x9a,0x42,0xc3,0xd4,0x19,0xe4,
                0x98,0xe0,0xe1,0xfb,0x96,0x16,0xfd,0x66,0x91,
                0x38,0xd3,0x3a,0x11,0x05,0xe0,0x7c,0x72,0xb6,
                0x95,0x3b,0xcc}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x68,0x69,0x73,0x20,0x69,0x73,0x20,0x61,
                 0x20,0x74,0x65,0x73,0x74,0x20,0x75,0x73,0x69,
                 0x6e,0x67,0x20,0x61,0x20,0x6c,0x61,0x72,0x67,
                 0x65,0x72,0x20,0x74,0x68,0x61,0x6e,0x20,0x62,
                 0x6c,0x6f,0x63,0x6b,0x2d,0x73,0x69,0x7a,0x65,
                 0x20,0x6b,0x65,0x79,0x20,0x61,0x6e,0x64,0x20,
                 0x61,0x20,0x6c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x74,0x68,0x61,0x6e,0x20,0x62,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x73,0x69,0x7a,0x65,0x20,0x64,0x61,
                 0x74,0x61,0x2e,0x20,0x54,0x68,0x65,0x20,0x6b,
                 0x65,0x79,0x20,0x6e,0x65,0x65,0x64,0x73,0x20,
                 0x74,0x6f,0x20,0x62,0x65,0x20,0x68,0x61,0x73,
                 0x68,0x65,0x64,0x20,0x62,0x65,0x66,0x6f,0x72,
                 0x65,0x20,0x62,0x65,0x69,0x6e,0x67,0x20,0x75,
                 0x73,0x65,0x64,0x20,0x62,0x79,0x20,0x74,0x68,
                 0x65,0x20,0x48,0x4d,0x41,0x43,0x20,0x61,0x6c,
                 0x67,0x6f,0x72,0x69,0x74,0x68,0x6d,0x2e},
        .data_len = 152,
        .mac_len = 48,
        .mac = {0x02,0x6f,0xdf,0x6b,0x50,0x74,0x1e,0x37,0x38,
                0x99,0xc9,0xf7,0xd5,0x40,0x6d,0x4e,0xb0,0x9f,
                0xc6,0x66,0x56,0x36,0xfc,0x1a,0x53,0x00,0x29,
                0xdd,0xf5,0xcf,0x3c,0xa5,0xa9,0x00,0xed,0xce,
                0x01,0xf5,0xf6,0x1e,0x2f,0x40,0x8c,0xdf,0x2f,
                0xd3,0xe7,0xe8}
    }
};

/*  Test vectors for HMAC SHA3-512:
 */
struct HMAC_TEST_VECTOR sha3_512_hmac_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 64,
        .mac = {0xeb,0x3f,0xbd,0x4b,0x2e,0xaa,0xb8,0xf5,0xc5,
                0x04,0xbd,0x3a,0x41,0x46,0x5a,0xac,0xec,0x15,
                0x77,0x0a,0x7c,0xab,0xac,0x53,0x1e,0x48,0x2f,
                0x86,0x0b,0x5e,0xc7,0xba,0x47,0xcc,0xb2,0xc6,
                0xf2,0xaf,0xce,0x8f,0x88,0xd2,0x2b,0x6d,0xc6,
                0x13,0x80,0xf2,0x3a,0x66,0x8f,0xd3,0x88,0x8b,
                0xb8,0x05,0x37,0xc0,0xa0,0xb8,0x64,0x07,0x68,
                0x9e}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 64,
        .mac = {0x5a,0x4b,0xfe,0xab,0x61,0x66,0x42,0x7c,0x7a,
                0x36,0x47,0xb7,0x47,0x29,0x2b,0x83,0x84,0x53,
                0x7c,0xdb,0x89,0xaf,0xb3,0xbf,0x56,0x65,0xe4,
                0xc5,0xe7,0x09,0x35,0x0b,0x28,0x7b,0xae,0xc9,
                0x21,0xfd,0x7c,0xa0,0xee,0x7a,0x0c,0x31,0xd0,
                0x22,0xa9,0x5e,0x1f,0xc9,0x2b,0xa9,0xd7,0x7d,
                0xf8,0x83,0x96,0x02,0x75,0xbe,0xb4,0xe6,0x20,
                0x24}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa},
        .key_len = 20,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 64,
        .mac = {0x30,0x9e,0x99,0xf9,0xec,0x07,0x5e,0xc6,0xc6,
                0xd4,0x75,0xed,0xa1,0x18,0x06,0x87,0xfc,0xf1,
                0x53,0x11,0x95,0x80,0x2a,0x99,0xb5,0x67,0x74,
                0x49,0xa8,0x62,0x51,0x82,0x85,0x1c,0xb3,0x32,
                0xaf,0xb6,0xa8,0x9c,0x41,0x13,0x25,0xfb,0xcb,
                0xcd,0x42,0xaf,0xcb,0x7b,0x6e,0x5a,0xab,0x7e,
                0xa4,0x2c,0x66,0x0f,0x97,0xfd,0x85,0x84,0xbf,
                0x03}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 64,
        .mac = {0xb2,0x7e,0xab,0x1d,0x6e,0x8d,0x87,0x46,0x1c,
                0x29,0xf7,0xf5,0x73,0x9d,0xd5,0x8e,0x98,0xaa,
                0x35,0xf8,0xe8,0x23,0xad,0x38,0xc5,0x49,0x2a,
                0x20,0x88,0xfa,0x02,0x81,0x99,0x3b,0xbf,0xff,
                0x9a,0x0e,0x9c,0x6b,0xf1,0x21,0xae,0x9e,0xc9,
                0xbb,0x09,0xd8,0x4a,0x5e,0xba,0xc8,0x17,0x18,
                0x2e,0xa9,0x74,0x67,0x3f,0xb1,0x33,0xca,0x0d,
                0x1d}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x65,0x73,0x74,0x20,0x55,0x73,0x69,0x6e,
                 0x67,0x20,0x4c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x54,0x68,0x61,0x6e,0x20,0x42,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x53,0x69,0x7a,0x65,0x20,0x4b,0x65,
                 0x79,0x20,0x2d,0x20,0x48,0x61,0x73,0x68,0x20,
                 0x4b,0x65,0x79,0x20,0x46,0x69,0x72,0x73,0x74},
        .data_len = 54,
        .mac_len = 64,
        .mac = {0x00,0xf7,0x51,0xa9,0xe5,0x06,0x95,0xb0,0x90,
                0xed,0x69,0x11,0xa4,0xb6,0x55,0x24,0x95,0x1c,
                0xdc,0x15,0xa7,0x3a,0x5d,0x58,0xbb,0x55,0x21,
                0x5e,0xa2,0xcd,0x83,0x9a,0xc7,0x9d,0x2b,0x44,
                0xa3,0x9b,0xaf,0xab,0x27,0xe8,0x3f,0xde,0x9e,
                0x11,0xf6,0x34,0x0b,0x11,0xd9,0x91,0xb1,0xb9,
                0x1b,0xf2,0xee,0xe7,0xfc,0x87,0x24,0x26,0xc3,
                0xa4}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 131,
        .data = {0x54,0x68,0x69,0x73,0x20,0x69,0x73,0x20,0x61,
                 0x20,0x74,0x65,0x73,0x74,0x20,0x75,0x73,0x69,
                 0x6e,0x67,0x20,0x61,0x20,0x6c,0x61,0x72,0x67,
                 0x65,0x72,0x20,0x74,0x68,0x61,0x6e,0x20,0x62,
                 0x6c,0x6f,0x63,0x6b,0x2d,0x73,0x69,0x7a,0x65,
                 0x20,0x6b,0x65,0x79,0x20,0x61,0x6e,0x64,0x20,
                 0x61,0x20,0x6c,0x61,0x72,0x67,0x65,0x72,0x20,
                 0x74,0x68,0x61,0x6e,0x20,0x62,0x6c,0x6f,0x63,
                 0x6b,0x2d,0x73,0x69,0x7a,0x65,0x20,0x64,0x61,
                 0x74,0x61,0x2e,0x20,0x54,0x68,0x65,0x20,0x6b,
                 0x65,0x79,0x20,0x6e,0x65,0x65,0x64,0x73,0x20,
                 0x74,0x6f,0x20,0x62,0x65,0x20,0x68,0x61,0x73,
                 0x68,0x65,0x64,0x20,0x62,0x65,0x66,0x6f,0x72,
                 0x65,0x20,0x62,0x65,0x69,0x6e,0x67,0x20,0x75,
                 0x73,0x65,0x64,0x20,0x62,0x79,0x20,0x74,0x68,
                 0x65,0x20,0x48,0x4d,0x41,0x43,0x20,0x61,0x6c,
                 0x67,0x6f,0x72,0x69,0x74,0x68,0x6d,0x2e},
        .data_len = 152,
        .mac_len = 64,
        .mac = {0x38,0xa4,0x56,0xa0,0x04,0xbd,0x10,0xd3,0x2c,
                0x9a,0xb8,0x33,0x66,0x84,0x11,0x28,0x62,0xc3,
                0xdb,0x61,0xad,0xcc,0xa3,0x18,0x29,0x35,0x5e,
                0xaf,0x46,0xfd,0x5c,0x73,0xd0,0x6a,0x1f,0x0d,
                0x13,0xfe,0xc9,0xa6,0x52,0xfb,0x38,0x11,0xb5,
                0x77,0xb1,0xb1,0xd1,0xb9,0x78,0x9f,0x97,0xae,
                0x5b,0x83,0xc6,0xf4,0x4d,0xfc,0xf1,0xd6,0x7e,
                0xba}
    }
};

/*  MD5 HMAC test vectors from RFC2202:
 *  http://www.faqs.org/rfcs/rfc2202.html
 *  Note: These are same test vectors as md5_hmac_test_vector but
 *        the mac_len is different.
 */
struct HMAC_TEST_VECTOR md5_hmac_general_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b},
        .key_len = 16,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 4,
        .mac = {0x92,0x94,0x72,0x7a,0x36,0x38,0xBB,0x1C,0x13,
                0xf4,0x8E,0xf8,0x15,0x8b,0xfc,0x9d}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 4,
        .mac = {0x75,0x0c,0x78,0x3e,0x6a,0xb0,0xb5,0x03,0xea,
                0xa8,0x6e,0x31,0x0a,0x5d,0xb7,0x38}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 16,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 4,
        .mac = {0x56,0xbe,0x34,0x52,0x1d,0x14,0x4c,0x88,0xdb,
                0xb8,0xc7,0x33,0xf0,0xe8,0xb3,0xf6}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 4,
        .mac = {0x69,0x7e,0xaf,0x0a,0xca,0x3a,0x3a,0xea,0x3a,
                0x75,0x16,0x47,0x46,0xff,0xaa,0x79}
    }, {
        .key = {0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
                0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c},
        .key_len = 16,
        .data = {"Test With Truncation"},
        .data_len = 20,
        .mac_len = 4,
        .mac = {0x56,0x46,0x1e,0xf2,0x34,0x2e,0xdc,0x00,0xf9,
                0xba,0xb9,0x95,0x69,0x0e,0xfd,0x4c}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 80,
        .data = {"Test Using Larger Than Block-Size Key - Hash Key First"},
        .data_len = 54,
        .mac_len = 4,
        .mac = {0x6b,0x1a,0xb7,0xfe,0x4b,0xd7,0xbf,0x8f,0x0b,
                0x62,0xe6,0xce,0x61,0xb9,0xd0,0xcd}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 80,
        .data = {"Test Using Larger Than Block-Size Key and Larger Than One "
                 "Block-Size Data"},
        .data_len = 73,
        .mac_len = 4,
        .mac = {0x6f,0x63,0x0f,0xad,0x67,0xcd,0xa0,0xee,0x1f,
                0xb1,0xf5,0x62,0xdb,0x3a,0xa5,0x3e}
    },
};

/*  SHA1 HMAC test vectors from RFC2202:
 *  http://www.faqs.org/rfcs/rfc2202.html
 *  Note: These are same test vectors as sha1_hmac_test_vector but
 *        the mac_len is different.
 */
struct HMAC_TEST_VECTOR sha1_hmac_general_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 10,
        .mac = {0xb6,0x17,0x31,0x86,0x55,0x05,0x72,0x64,0xe2,
                0x8b,0xc0,0xb6,0xfb,0x37,0x8c,0x8e,0xf1,0x46,
                0xbe,0x00}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 10,
        .mac = {0xef,0xfc,0xdf,0x6a,0xe5,0xeb,0x2f,0xa2,0xd2,
                0x74,0x16,0xd5,0xf1,0x84,0xdf,0x9c,0x25,0x9a,
                0x7c,0x79}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa},
        .key_len = 20,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 10,
        .mac = {0x12,0x5d,0x73,0x42,0xb9,0xac,0x11,0xcd,0x91,
                0xa3,0x9a,0xf4,0x8a,0xa1,0x7b,0x4f,0x63,0xf1,
                0x75,0xd3}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 10,
        .mac = {0x4c,0x90,0x07,0xf4,0x02,0x62,0x50,0xc6,0xbc,
                0x84,0x14,0xf9,0xbf,0x50,0xc8,0x6c,0x2d,0x72,
                0x35,0xda}
    }, {
        .key = {0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
                0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,0x0c,
                0x0c,0x0c},
        .key_len = 20,
        .data = {"Test With Truncation"},
        .data_len = 20,
        .mac_len = 10,
        .mac = {0x4c,0x1a,0x03,0x42,0x4b,0x55,0xe0,0x7f,0xe7,
                0xf2,0x7b,0xe1,0xd5,0x8b,0xb9,0x32,0x4a,0x9a,
                0x5a,0x04}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 80,
        .data = {"Test Using Larger Than Block-Size Key - Hash Key First"},
        .data_len = 54,
        .mac_len = 10,
        .mac = {0xaa,0x4a,0xe5,0xe1,0x52,0x72,0xd0,0x0e,0x95,
                0x70,0x56,0x37,0xce,0x8a,0x3b,0x55,0xed,0x40,
                0x21,0x12}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 80,
        .data = {"Test Using Larger Than Block-Size Key and Larger Than One "
                 "Block-Size Data"},
        .data_len = 73,
        .mac_len = 10,
        .mac = {0xe8,0xe9,0x9d,0x0f,0x45,0x23,0x7d,0x78,0x6d,
                0x6b,0xba,0xa7,0x96,0x5c,0x78,0x08,0xbb,0xff,
                0x1a,0x91}
    }
};

/*  Test vectors for HMAC SHA256 from RFC4868:
 *  http://tools.ietf.org/html/rfc4868
 *  Note: These are same test vectors as sha256_hmac_test_vector but
 *        the mac_len is different.
 */
struct HMAC_TEST_VECTOR sha224_hmac_general_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 16,
        .mac = {0x89,0x6f,0xb1,0x12,0x8a,0xbb,0xdf,0x19,0x68,
                0x32,0x10,0x7c,0xd4,0x9d,0xf3,0x3f}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 16,
        .mac = {0xa3,0x0e,0x01,0x09,0x8b,0xc6,0xdb,0xbf,0x45,
                0x69,0x0f,0x3a,0x7e,0x9e,0x6d,0x0f}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa},
        .key_len = 20,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 16,
        .mac = {0x7f,0xb3,0xcb,0x35,0x88,0xc6,0xc1,0xf6,0xff,
                0xa9,0x69,0x4d,0x7d,0x6a,0xd2,0x64}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19},
        .key_len = 25,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 16,
        .mac = {0x6c,0x11,0x50,0x68,0x74,0x01,0x3c,0xac,0x6a,
                0x2a,0xbc,0x1b,0xb3,0x82,0x62,0x7c}
    }
};


struct HMAC_TEST_VECTOR sha256_hmac_general_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b},
        .key_len = 32,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 16,
        .mac = {0x19,0x8a,0x60,0x7e,0xb4,0x4b,0xfb,0xc6,0x99,
                0x03,0xa0,0xf1,0xcf,0x2b,0xbd,0xc5}
    }, {
        .key = {0x4a,0x65,0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,
                0x65,0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,0x65,
                0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,0x65,0x66,
                0x65,0x4a,0x65,0x66,0x65},
        .key_len = 32,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 16,
        .mac = {0x16,0x7f,0x92,0x85,0x88,0xc5,0xcc,0x2e,0xef,
                0x8e,0x30,0x93,0xca,0xa0,0xe8,0x7c},
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa},
        .key_len = 32,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 16,
        .mac = {0xcd,0xcb,0x12,0x20,0xd1,0xec,0xcc,0xea,0x91,
                0xe5,0x3a,0xba,0x30,0x92,0xf9,0x62}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,
                0x1c,0x1d,0x1e,0x1f,0x20},
        .key_len = 32,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 16,
        .mac = {0x37,0x2e,0xfc,0xf9,0xb4,0x0b,0x35,0xc2,0x11,
                0x5b,0x13,0x46,0x90,0x3d,0x2e,0xf4}
    }
};

/*  Test vectors for HMAC SHA384 from RFC4868:
 *  http://tools.ietf.org/html/rfc4868
 *  Note: These are same test vectors as sha384_hmac_test_vector but
 *        the mac_len is different.
 */
struct HMAC_TEST_VECTOR sha384_hmac_general_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b},
        .key_len = 48,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 24,
        .mac = {0xb6,0xa8,0xd5,0x63,0x6f,0x5c,0x6a,0x72,0x24,
                0xf9,0x97,0x7d,0xcf,0x7e,0xe6,0xc7,0xfb,0x6d,
                0x0c,0x48,0xcb,0xde,0xe9,0x73}
    }, {
        .key = {0x4a,0x65,0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,
                0x65,0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,0x65,
                0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,0x65,0x66,
                0x65,0x4a,0x65,0x66,0x65,0x4a,0x65,0x66,0x65,
                0x4a,0x65,0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,
                0x65,0x66,0x65},
        .key_len = 48,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 24,
        .mac = {0x2c,0x73,0x53,0x97,0x4f,0x18,0x42,0xfd,0x66,
                0xd5,0x3c,0x45,0x2c,0xa4,0x21,0x22,0xb2,0x8c,
                0x0b,0x59,0x4c,0xfb,0x18,0x4d}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa},
        .key_len = 48,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 24,
        .mac = {0x80,0x9f,0x43,0x9b,0xe0,0x02,0x74,0x32,0x1d,
                0x4a,0x53,0x86,0x52,0x16,0x4b,0x53,0x55,0x4a,
                0x50,0x81,0x84,0xa0,0xc3,0x16}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,
                0x1c,0x1d,0x1e,0x1f,0x20,0x0a,0x0b,0x0c,0x0d,
                0x0e,0x0f,0x10,0x11,0x12,0x13,0x14,0x15,0x16,
                0x17,0x18,0x19},
        .key_len = 48,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 24,
        .mac = {0x5b,0x54,0x00,0x85,0xc6,0xe6,0x35,0x80,0x96,
                0x53,0x2b,0x24,0x93,0x60,0x9e,0xd1,0xcb,0x29,
                0x8f,0x77,0x4f,0x87,0xbb,0x5c}
    }
};

/*  Test vectors for HMAC SHA512 from RFC4868:
 *  http://tools.ietf.org/html/rfc4868
 *  Note: These are same test vectors as sha512_hmac_test_vector but
 *        the mac_len is different.
 */
struct HMAC_TEST_VECTOR sha512_hmac_general_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b},
        .key_len = 64,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 32,
        .mac = {0x63,0x7e,0xdc,0x6e,0x01,0xdc,0xe7,0xe6,0x74,
                0x2a,0x99,0x45,0x1a,0xae,0x82,0xdf,0x23,0xda,
                0x3e,0x92,0x43,0x9e,0x59,0x0e,0x43,0xe7,0x61,
                0xb3,0x3e,0x91,0x0f,0xb8}
    }, {
        .key = {0x4a,0x65,0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,
                0x65,0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,0x65,
                0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,0x65,0x66,
                0x65,0x4a,0x65,0x66,0x65,0x4a,0x65,0x66,0x65,
                0x4a,0x65,0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,
                0x65,0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,0x65,
                0x66,0x65,0x4a,0x65,0x66,0x65,0x4a,0x65,0x66,
                0x65},
        .key_len = 64,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 32,
        .mac = {0xcb,0x37,0x09,0x17,0xae,0x8a,0x7c,0xe2,0x8c,
                0xfd,0x1d,0x8f,0x47,0x05,0xd6,0x14,0x1c,0x17,
                0x3b,0x2a,0x93,0x62,0xc1,0x5d,0xf2,0x35,0xdf,
                0xb2,0x51,0xb1,0x54,0x54}
    }, {
        .key = {0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,0xaa,
                0xaa},
        .key_len = 64,
        .data = {0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,0xDD,
                 0xDD,0xDD,0xDD,0xDD,0xDD},
        .data_len = 50,
        .mac_len = 32,
        .mac = {0x2e,0xe7,0xac,0xd7,0x83,0x62,0x4c,0xa9,0x39,
                0x87,0x10,0xf3,0xee,0x05,0xae,0x41,0xb9,0xf9,
                0xb0,0x51,0x0c,0x87,0xe4,0x9e,0x58,0x6c,0xc9,
                0xbf,0x96,0x17,0x33,0xd8}
    }, {
        .key = {0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,
                0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,0x10,0x11,0x12,
                0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1a,0x1b,
                0x1c,0x1d,0x1e,0x1f,0x20,0x21,0x22,0x23,0x24,
                0x25,0x26,0x27,0x28,0x29,0x2a,0x2b,0x2c,0x2d,
                0x2e,0x2f,0x30,0x31,0x32,0x33,0x34,0x35,0x36,
                0x37,0x38, 0x39,0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,
                0x40},
        .key_len = 64,
        .data = {0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,0xCD,
                 0xCD,0xCD,0xCD,0xCD,0xCD},
        .data_len = 50,
        .mac_len = 32,
        .mac = {0x5e,0x66,0x88,0xe5,0xa3,0xda,0xec,0x82,0x6c,
                0xa3,0x2e,0xae,0xa2,0x24,0xef,0xf5,0xe7,0x00,
                0x62,0x89,0x47,0x47,0x0e,0x13,0xad,0x01,0x30,
                0x25,0x61,0xba,0xb1,0x08}
    }
};

/*
 * Test vectors for HMAC SHA512/224:
 */
struct HMAC_TEST_VECTOR sha512_224_hmac_general_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 16,
        .mac = {0xb2,0x44,0xba,0x01,0x30,0x7c,0x0e,0x7a,0x8c,
                0xca,0xad,0x13,0xb1,0x06,0x7a,0x4c}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 16,
        .mac = {0x4a,0x53,0x0b,0x31,0xa7,0x9e,0xbc,0xce,0x36,
                0x91,0x65,0x46,0x31,0x7c,0x45,0xf2}
    }
};

/*
 * Test vectors for HMAC SHA512/256:
 */
struct HMAC_TEST_VECTOR sha512_256_hmac_general_test_vector[] = {
    {
        .key = {0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,0x0b,
                0x0b,0x0b},
        .key_len = 20,
        .data = {"Hi There"},
        .data_len = 8,
        .mac_len = 16,
        .mac = {0x9f,0x91,0x26,0xc3,0xd9,0xc3,0xc3,0x30,0xd7,
                0x60,0x42,0x5c,0xa8,0xa2,0x17,0xe3}
    }, {
        .key = {'J', 'e', 'f', 'e'},
        .key_len = 4,
        .data = {"what do ya want for nothing?"},
        .data_len = 28,
        .mac_len = 16,
        .mac = {0x6d,0xf7,0xb2,0x46,0x30,0xd5,0xcc,0xb2,0xee,
                0x33,0x54,0x07,0x08,0x1a,0x87,0x18}
    }
};

/* FIPS HMAC test vectors from
 * http://csrc.nist.gov/groups/STM/cavp/documents/mac/hmactestvectors.zip
 * Notes:
 * 	The test vectors have various mac sizes, so suitable for HMAC
 * 	GENERAL mechanisms.
 *	The second number in parentheisis is the "Count" in the HMAC.rsp.
 */

struct HMAC_TEST_VECTOR fips_sha1_hmac_test_vector[] = {
    {	// 0 (45)
        .key = {0x59,0x78,0x59,0x28,0xd7,0x25,0x16,0xe3,0x12,0x72},
        .key_len = 10,
        .data = {0xa3,0xce,0x88,0x99,0xdf,0x10,0x22,0xe8,0xd2,
                 0xd5,0x39,0xb4,0x7b,0xf0,0xe3,0x09,0xc6,0x6f,
                 0x84,0x09,0x5e,0x21,0x43,0x8e,0xc3,0x55,0xbf,
                 0x11,0x9c,0xe5,0xfd,0xcb,0x4e,0x73,0xa6,0x19,
                 0xcd,0xf3,0x6f,0x25,0xb3,0x69,0xd8,0xc3,0x8f,
                 0xf4,0x19,0x99,0x7f,0x0c,0x59,0x83,0x01,0x08,
                 0x22,0x36,0x06,0xe3,0x12,0x23,0x48,0x3f,0xd3,
                 0x9e,0xde,0xaa,0x4d,0x3f,0x0d,0x21,0x19,0x88,
                 0x62,0xd2,0x39,0xc9,0xfd,0x26,0x07,0x41,0x30,
                 0xff,0x6c,0x86,0x49,0x3f,0x52,0x27,0xab,0x89,
                 0x5c,0x8f,0x24,0x4b,0xd4,0x2c,0x7a,0xfc,0xe5,
                 0xd1,0x47,0xa2,0x0a,0x59,0x07,0x98,0xc6,0x8e,
                 0x70,0x8e,0x96,0x49,0x02,0xd1,0x24,0xda,0xde,
                 0xcd,0xbd,0xa9,0xdb,0xd0,0x05,0x1e,0xd7,0x10,
                 0xe9,0xbf},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x3c,0x81,0x62,0x58,0x9a,0xaf,0xae,0xe0,0x24,
                0xfc,0x9a,0x5c,0xa5,0x0d,0xd2,0x33,0x6f,0xe3,
                0xeb,0x28},
        .chunks	= {25, 25, 25, 25, 3, 25},
        .num_chunks =	6,
    }, {	// 1 (46)
        .key = {0xc5,0x21,0x09,0xc9,0xd0,0xda,0x92,0x58,0xeb,0x73},
        .key_len = 10,
        .data = {0x52,0xb1,0x13,0x61,0x4b,0x80,0xb9,0x70,0x51,
                 0x0f,0x65,0xa2,0x5d,0x46,0xed,0xc0,0x23,0xd9,
                 0xc7,0xb8,0xe7,0xca,0x7c,0x41,0x92,0x30,0x59,
                 0xc2,0x05,0x36,0x68,0x70,0xad,0x66,0x9f,0xb7,
                 0x57,0x28,0x56,0xdc,0x46,0x85,0xff,0xe0,0x83,
                 0x31,0x11,0xa7,0x75,0xc9,0x45,0x5a,0xb1,0x59,
                 0x05,0x09,0x13,0x21,0x21,0x95,0x0e,0x99,0xc5,
                 0xcd,0x40,0xb2,0xa8,0xd7,0x4a,0x5f,0x85,0xd2,
                 0xde,0x54,0xcf,0xb9,0x1a,0x0d,0xa1,0x8a,0x14,
                 0x13,0xf4,0xa8,0xb6,0x7b,0x14,0x7e,0xcc,0xaf,
                 0x55,0x66,0x5b,0x71,0x01,0xc9,0x34,0x1c,0x96,
                 0x87,0xca,0x2d,0x2e,0x99,0x41,0x03,0x3f,0xf5,
                 0xc7,0xe3,0x84,0xb1,0x27,0x3f,0x3b,0x6c,0x9b,
                 0x38,0x91,0xea,0xe2,0x61,0x5b,0xfe,0x93,0xc6,
                 0x06,0xad},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x2f,0xec,0xb4,0x66,0xbc,0x92,0x0f,0x61,0x0e,
                0x3e,0xae,0x99,0x49,0xe0,0x0f,0x45,0x4a,0x71,
                0x4a,0xb5},
        .chunks = {54, 0, 54, -1, 20},
        .num_chunks = 5,
    }, {	// 2 (105)
        .key = {0x89,0x58,0x68,0xf1,0x96,0x95,0xc1,0xf5,0xa2,
                0x6d,0x8a,0xe3,0x39,0xc5,0x67,0xe5,0xab,0x43,
                0xb0,0xfc,0xc8,0x05,0x60,0x50,0xe9,0x92,0x2e,
                0xc5,0x30,0x10,0xf9,0xce},
        .key_len = 32,
        .data = {0x88,0x3e,0x6c,0xa2,0xb1,0x9e,0xf5,0x46,0x40,
                 0xbb,0x83,0x33,0xf8,0x5a,0x93,0x80,0xe1,0x72,
                 0x11,0xf6,0xee,0x3d,0x1d,0xc7,0xdc,0x8f,0x0e,
                 0x7c,0x5d,0x67,0xb7,0x30,0x76,0xc3,0xea,0xfc,
                 0x26,0xb9,0x3b,0xb2,0x48,0xc4,0x06,0xce,0xba,
                 0x5c,0xb4,0xa9,0xbf,0xc9,0x39,0xf0,0xa2,0x38,
                 0xe1,0x55,0x9d,0x0f,0x4d,0x84,0xf8,0x7e,0xb8,
                 0x59,0x75,0x56,0x80,0x50,0xec,0x1f,0xe1,0x3d,
                 0x33,0x65,0x03,0x3d,0x40,0x52,0x37,0xec,0x92,
                 0x82,0x7d,0xd8,0xcd,0x12,0x4b,0x36,0xa4,0xfa,
                 0x89,0xd4,0xfb,0x9d,0xe0,0x4f,0x4d,0x9f,0x34,
                 0x86,0x4c,0xf7,0x6f,0x4e,0xc8,0x45,0x81,0x68,
                 0xd2,0x65,0xa5,0xb0,0x21,0x44,0xe5,0x96,0xb5,
                 0xf2,0xe0,0xd2,0xb9,0xf9,0xcb,0x54,0xae,0xee,
                 0xb6,0x7a},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x37,0x4c,0x88,0xf4,0x48,0x0f,0x5e,0x8a,0xaa,
                0x9f,0x44,0x8b,0x77,0x75,0x57,0xc5,0x00,0x65,
                0xe9,0xac},
    }, {	// 3 (106)
        .key = {0x95,0x0f,0xb0,0xcd,0xe3,0x0f,0x34,0xf5,0x97,
                0xaf,0x5c,0xaa,0x2b,0x16,0xfc,0x86,0xa5,0xc3,
                0xef,0x06,0x5d,0x36,0xff,0xdd,0x06,0xec,0x04,
                0x8e,0xec,0x91,0x50,0x39},
        .key_len = 32,
        .data = {0xe4,0x63,0x62,0x65,0x06,0x14,0x4c,0xec,0xe5,
                 0x5d,0xfb,0x7a,0xa2,0x2e,0xb2,0x1e,0xa3,0xa4,
                 0x27,0x7d,0x89,0x2c,0x21,0x17,0x62,0xea,0x45,
                 0xcc,0x20,0x5c,0x2d,0x9e,0x4b,0x3a,0xbb,0xb8,
                 0xf2,0xa1,0xad,0xb0,0xe7,0x71,0x71,0x09,0x2c,
                 0xf4,0x3a,0xfc,0xa8,0xc0,0x53,0x77,0x1e,0xde,
                 0xb4,0x67,0x60,0x2b,0xd3,0x33,0xc0,0xff,0xbc,
                 0x88,0xc8,0x0d,0x64,0x5c,0x2b,0x8a,0x3a,0x2d,
                 0xfa,0x92,0x00,0x8a,0x1b,0xc7,0xd9,0xd5,0xf8,
                 0x3b,0xa3,0x47,0x74,0x90,0x86,0x34,0x23,0x5d,
                 0xcd,0x91,0xba,0xd4,0xf5,0xb3,0xc4,0xa2,0x04,
                 0x59,0x97,0x17,0x1d,0xed,0x87,0x87,0x50,0x07,
                 0x59,0xf0,0xb6,0x33,0xfb,0xdc,0xbe,0xf4,0x72,
                 0x89,0xc2,0x09,0x13,0x48,0xde,0xee,0xf6,0x23,
                 0x01,0xa6},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x8c,0x90,0x48,0x0e,0xa6,0x41,0x45,0x53,0xdf,
                0x17,0xe5,0x3c,0xf9,0x6d,0xcb,0x16,0x6b,0x94,
                0xbe,0x35},
    }, { // 4 (165)
        .key = {0xb9,0x57,0x5f,0x4d,0x5e,0xcc,0x0f,0x4f,0x62,
                0xe4,0xa0,0x55,0x6b,0xb8,0x94,0x64,0xba,0x97,
                0xd4,0x57,0x0e,0x55,0xac,0xd4,0xc5,0xe5,0x17,
                0x7e,0x45,0x2a,0x3d,0x6c,0x9a,0x0b,0x3a,0xdb,
                0x60,0xc6,0x21,0x1f,0xe4,0x86,0x40,0xe0,0x86,
                0x37,0xa6,0x82,0x62,0x99,0xe3,0xe5,0x2f,0x93,
                0x0f,0x4f,0x66,0xcb,0x0e,0xa6,0xa7,0x73,0x11,
                0xe3},
        .key_len = 64,
        .data = {0x8c,0x83,0x87,0xf4,0xae,0x2c,0xa1,0xa6,0xdd,
                 0x13,0xd2,0x9e,0x93,0x58,0x0b,0x1c,0xdf,0x62,
                 0x68,0xda,0x66,0xcf,0x58,0x9c,0xa8,0xb1,0xff,
                 0x08,0x84,0xf7,0xd8,0xb8,0xfe,0x29,0x9f,0x8e,
                 0x41,0x59,0x6e,0x47,0xe0,0x56,0x26,0x53,0x61,
                 0x22,0x10,0xe4,0xfc,0xa6,0xc4,0x46,0xa0,0xa5,
                 0x4a,0x6e,0x37,0xef,0x80,0xd5,0x2b,0xd7,0xbb,
                 0x87,0x29,0xe6,0xb1,0x76,0x25,0xd1,0x97,0x15,
                 0x9e,0xa9,0x86,0x22,0x23,0x52,0x23,0xc3,0x16,
                 0x36,0x7f,0xd5,0xb0,0x3a,0x3c,0x81,0x45,0xf2,
                 0xf2,0x10,0xc9,0x10,0xd0,0x00,0x94,0x23,0x87,
                 0x57,0x62,0x7e,0x63,0x37,0x9e,0x75,0xbb,0xb3,
                 0xe0,0xd0,0x8c,0xe1,0xb4,0x79,0x61,0x30,0x9d,
                 0x78,0x76,0xfc,0x59,0x21,0x1c,0x60,0x67,0x8c,
                 0x5f,0x4c},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x15,0xaf,0x23,0x33,0x16,0x48,0x17,0x14,0x99,
                0xb5,0x80,0x42,0xdb,0xe7,0xb2,0xd5,0xdf,0x72,
                0xd1,0x52},
    }, {	// 5 (166)
        .key = {0xd2,0x91,0xad,0xbf,0x05,0xb0,0x65,0x96,0xc2,
                0xf3,0x6f,0x41,0xa8,0xcd,0x80,0x70,0x37,0x0c,
                0x42,0xf6,0x87,0xb8,0xa6,0xcc,0x3a,0x3e,0x7b,
                0x59,0xaf,0xcd,0x40,0xf0,0x78,0x01,0x36,0x9b,
                0x0f,0xbf,0xba,0x17,0xc4,0x60,0xd2,0x1f,0xfa,
                0x11,0x06,0xee,0x93,0x79,0x71,0xff,0xa9,0x9d,
                0x17,0x17,0x7f,0x01,0x79,0x85,0xb7,0x10,0x67,
                0xa8},
        .key_len = 64,
        .data = {0x50,0xbc,0xdf,0x31,0x38,0x9e,0xad,0xac,0x5b,
                 0xb8,0x19,0x7e,0xe9,0x49,0xf2,0x86,0x4e,0xde,
                 0x28,0x4c,0x07,0xd0,0x39,0xa0,0xb4,0x0e,0xed,
                 0x7e,0x6f,0x1c,0x43,0x35,0x5d,0x5c,0xab,0xc8,
                 0x82,0x8d,0x75,0x95,0xda,0x91,0x8a,0x34,0xa5,
                 0x73,0x5a,0xa2,0x02,0xa8,0x15,0x9f,0xbf,0x95,
                 0x1e,0x54,0x70,0x52,0xbd,0x39,0xbe,0xae,0x14,
                 0x36,0x02,0x73,0x54,0x09,0x13,0xeb,0x30,0xe7,
                 0x5b,0xa2,0x92,0x66,0x31,0x6e,0x8d,0x9a,0x63,
                 0xad,0x94,0x7e,0x11,0xce,0xe9,0x96,0xc2,0x13,
                 0x57,0xd3,0xb1,0x94,0x24,0xb7,0x68,0x88,0x42,
                 0xb9,0x90,0xc0,0xc5,0xeb,0x08,0x74,0x9a,0xda,
                 0x34,0x42,0x75,0xb6,0x98,0x74,0x0b,0xb3,0xa5,
                 0x82,0x82,0xae,0xd2,0xd7,0x25,0x14,0xef,0xd8,
                 0x5d,0x00},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x5f,0x7a,0x57,0xd4,0x2e,0x3e,0xbb,0xcb,0x85,
                0xb0,0x85,0x65,0x30,0x4d,0xab,0x94,0x1d,0x62,
                0x34,0xf3},
    }, {	// 6 (225)
        .key = {0xf0,0xda,0xe6,0xd8,0x75,0x30,0x76,0xb1,0x89,
                0x5c,0x01,0x26,0x2c,0xa9,0xb5,0x76,0x33,0xeb,
                0x28,0xb3,0xf9,0x63,0xa7,0xc7,0x52,0xe2,0xcb,
                0xb4,0xc0,0x31,0x4c,0x20,0xea,0xb1,0x1a,0x10,
                0x49,0x3f,0xaa,0xf4,0x25,0x5a,0x8e,0xe4,0xc0,
                0x88,0x49,0x29,0xd1,0xf5,0x61,0xff,0x33,0x5e,
                0xb6,0x99,0xdf,0x2d,0x11,0x66,0x18,0xe6,0x00,
                0x93,0xe5,0xc1,0xe2,0xd1,0xc4,0x99},
        .key_len = 70,
        .data = {0x61,0xcb,0x9e,0x1f,0x1e,0x4b,0x3a,0x3b,0x3b,
                 0xdf,0xf8,0xcd,0x5f,0x24,0x56,0x6b,0x98,0x7f,
                 0x75,0xc8,0xa0,0x53,0x77,0x85,0x5f,0x77,0x2b,
                 0x49,0xb0,0xe7,0xec,0x13,0x68,0xb9,0xc6,0xcf,
                 0x95,0x53,0xdb,0x28,0x03,0xdc,0x05,0x9e,0x05,
                 0xf0,0xbd,0xd8,0x71,0x98,0x3c,0x3b,0xed,0x79,
                 0xdf,0xbb,0x69,0x4b,0xd0,0xf1,0xed,0x8d,0xe3,
                 0x6e,0x95,0x77,0xbe,0x50,0xda,0x31,0x3d,0x13,
                 0x12,0x42,0x15,0xa9,0x3a,0x4b,0xb7,0xcc,0xf4,
                 0xf5,0x77,0x93,0xcc,0x28,0xed,0x43,0xbf,0x7e,
                 0x9b,0x68,0xfe,0xf7,0xd1,0x25,0xef,0xee,0xce,
                 0xc9,0x75,0x4b,0x28,0xa2,0x71,0xfb,0x6e,0x16,
                 0x89,0x9d,0x0b,0xef,0x28,0x7e,0x6d,0xf7,0xc5,
                 0xc8,0x67,0xc5,0x69,0xf6,0xd4,0xd6,0x6b,0x8b,
                 0x7e,0xe0},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x62,0xac,0x95,0x6a,0xda,0x19,0xf0,0x4b,0xe5,
                0x0c,0x23,0xf2,0x32,0x8a,0x32,0x47,0x7c,0xd5,
                0x8f,0xb9},
    }, {	// 7 (226)
        .key = {0x65,0xaf,0x1f,0x17,0xcd,0x7f,0xda,0xa5,0x23,
                0xb9,0xb7,0xa9,0x82,0x9d,0x49,0x7c,0xac,0x73,
                0x03,0xd4,0x50,0xc5,0x9e,0x98,0x88,0xcb,0xba,
                0xf3,0xa6,0x27,0xc8,0xa8,0x30,0xd3,0x27,0xa5,
                0x29,0x57,0x8d,0xda,0x92,0x3f,0xa9,0x4b,0x31,
                0xcc,0x07,0x64,0x91,0xea,0x33,0x8d,0x4a,0x62,
                0x21,0xff,0x82,0x51,0xcc,0xd6,0xb4,0xd9,0x1e,
                0x67,0xb1,0x16,0x10,0xd3,0xe4,0x53},
        .key_len = 70,
        .data = {0x9a,0xb4,0x66,0x7b,0x2d,0xf7,0xeb,0x4b,0xe8,
                 0x86,0x3a,0xa5,0x3e,0x9b,0xf9,0xaf,0x8b,0xae,
                 0x0f,0xc0,0x9d,0xe9,0x4f,0x73,0x73,0xdc,0x56,
                 0xfa,0x44,0x72,0xb6,0xb5,0xc4,0x23,0x54,0x03,
                 0xa2,0x6c,0x0e,0x59,0x55,0x7c,0xa1,0x91,0x18,
                 0x31,0xca,0x84,0x33,0x42,0xac,0xda,0x7d,0xbe,
                 0x72,0x21,0x1f,0xb5,0x35,0x1d,0x9a,0x34,0x20,
                 0x5f,0x0c,0x77,0xd2,0x19,0xaf,0x5b,0x03,0x31,
                 0xa2,0x12,0x6b,0x94,0xec,0x1a,0xdf,0xcd,0xbe,
                 0x70,0xbe,0xd6,0xf8,0x01,0x8b,0x2e,0xef,0x61,
                 0xdb,0x2b,0x6d,0xbf,0x72,0x92,0xfa,0x19,0xa9,
                 0x65,0x5a,0xac,0x13,0xfc,0x57,0xaf,0x5f,0x57,
                 0xc1,0x40,0x80,0xb3,0xb2,0x9f,0x0c,0x5b,0x16,
                 0x9a,0xe2,0xc1,0x6b,0x48,0x10,0xcd,0xc6,0xfa,
                 0xf4,0x75},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0xa2,0x79,0xd0,0x55,0xe2,0xd7,0x33,0x06,0xa8,
                0x18,0x73,0x44,0xfc,0x32,0xcb,0x0b,0x5b,0x80,
                0xcd,0x35},
    }, {	// 8 (285)
        .key = {0x4d,0xdd,0x00,0xd0,0xab,0x6a,0xab,0x21,0x00,
                0xce,0x97,0x54,0xc3,0xb3,0x98,0x7c,0x06,0xf7,
                0xe5,0x86,0x56,0x01,0x1d,0x26,0xe3,0x51,0x87,
                0x11,0xe1,0x5b,0x9e,0x6d,0x2d,0x96,0xcd,0x85,
                0x34,0xd0,0x77,0xc2,0x11,0xc4,0x3a,0xd7,0xf5,
                0xee,0x75,0x3b,0xcc,0x9e,0x07,0xdc,0x1d,0x4c,
                0x5a,0x12,0x32,0x2b,0xa1,0xd1,0x7a,0x00,0x5d,
                0x24,0x2b,0x35,0x26,0xd6,0x2b,0x29,0xa8,0x72,
                0x31,0xcb,0xec,0x6f,0x28,0x67,0xd9,0xa4},
        .key_len = 80,
        .data = {0x28,0xbe,0x0d,0x9e,0x62,0xdc,0x89,0xe2,0xa9,
                 0x13,0x06,0x4c,0x0d,0x3d,0xbf,0xb3,0x5a,0x0c,
                 0x77,0x66,0xf7,0x56,0x74,0x1b,0x0e,0xaf,0xcc,
                 0x28,0xed,0x3d,0xdf,0xf6,0xad,0xc8,0x25,0xb2,
                 0x11,0x11,0x2a,0x45,0xb0,0x65,0xd6,0x87,0x57,
                 0x71,0xf2,0xaf,0xa9,0x58,0xe8,0x0f,0x08,0x03,
                 0xca,0xfe,0xb9,0xb9,0x96,0x15,0x42,0xef,0xb9,
                 0x9e,0x17,0x61,0xd1,0x49,0x76,0x61,0xb7,0x21,
                 0x90,0x6f,0xbd,0xbf,0xe9,0x0b,0x34,0xbd,0x01,
                 0xc7,0x32,0x6e,0x34,0xa0,0x92,0xcc,0xdf,0x8e,
                 0x3b,0xb2,0xc4,0x5a,0xa6,0x4c,0xb0,0xb0,0x9a,
                 0xcb,0x5b,0x75,0x3a,0x5d,0x8f,0x5a,0x42,0x5c,
                 0x8c,0xb2,0x8e,0xc5,0xac,0x81,0xdc,0xed,0x43,
                 0xd5,0xd2,0x6f,0xc9,0x59,0x43,0x69,0x3b,0x27,
                 0xae,0xe8},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x39,0x32,0x38,0xd3,0xaf,0xdb,0x7d,0x97,0x0b,
                0x96,0x6d,0x37,0x4f,0xe0,0x97,0xec,0x87,0x97,
                0xa8,0x70},
    }, {	// 9 (286)
        .key = {0x7a,0x31,0x55,0x3b,0x05,0xe9,0x6a,0x8d,0xa0,
                0xa4,0xd5,0xb8,0x1a,0x85,0x7d,0x19,0x2a,0xfb,
                0x6a,0xab,0xb1,0xf1,0x27,0xd7,0x40,0x45,0x6a,
                0x8e,0xda,0x7c,0xf6,0x96,0xfb,0xb4,0xc1,0x21,
                0xd8,0xd9,0x52,0xa4,0xe9,0x1c,0x6e,0xe6,0xa5,
                0xa1,0xf3,0x58,0x8d,0x78,0x04,0xa4,0x6b,0xcf,
                0x66,0x88,0xdc,0x66,0x2a,0xe5,0x0c,0x43,0x8d,
                0x13,0xc1,0xa6,0x1c,0x78,0x9b,0x3f,0x1c,0x59,
                0x9a,0x9f,0x28,0xef,0xe0,0xed,0x1c,0xbe},
        .key_len = 80,
        .data = {0xfb,0x09,0x1d,0xdd,0x95,0xb1,0x00,0xdf,0xcf,
                 0x89,0x2d,0x78,0xe5,0xe7,0x70,0xd3,0xa3,0x7b,
                 0x8c,0x38,0x85,0xdf,0x80,0x3c,0x1d,0x6f,0x09,
                 0x35,0xb5,0x5b,0x68,0xf1,0x36,0xfb,0x65,0xa8,
                 0x48,0x62,0x94,0x2e,0xbb,0x35,0xd7,0x6d,0x26,
                 0xbe,0x24,0x13,0xcd,0x3c,0x89,0x88,0xc8,0x7d,
                 0x6d,0x23,0x62,0xaf,0x18,0x9d,0xc0,0x74,0x76,
                 0xc6,0xc3,0x34,0x17,0x76,0x2e,0xb7,0x7b,0xc7,
                 0x0c,0xf3,0x8d,0x81,0x4c,0x22,0x6d,0xd6,0xaf,
                 0x18,0x72,0x50,0xe4,0xd4,0x70,0x07,0xf1,0x55,
                 0x36,0x17,0xd4,0xaf,0x5b,0x51,0x6a,0x5d,0x3b,
                 0x31,0x91,0xd9,0x3c,0x10,0x89,0x6a,0x56,0x9b,
                 0xa1,0x3d,0xd2,0x84,0x0f,0xb8,0x51,0x78,0x1f,
                 0x0b,0x11,0x50,0x90,0x08,0x6c,0x8b,0x3a,0x34,
                 0xa1,0xfc},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x0f,0xdd,0x3f,0x83,0x6d,0xd7,0xe5,0xc5,0x06,
                0xab,0x21,0xad,0xde,0x9a,0xe5,0xdc,0x09,0xcb,
                0x35,0x9d},
    }
};

struct HMAC_TEST_VECTOR fips_sha224_hmac_test_vector[] = {
    {   // 0 (60)
        .key = {0xcf,0x12,0x75,0x79,0xd6,0xb2,0xb0,0xb3,0xa6,
                0x07,0xa6,0x31,0x4b,0xf8,0x73,0x30,0x61,0xc3,
                0x2a,0x04,0x35,0x93,0x19,0x55,0x27,0x54,0x4f,
                0x87,0x53,0xc6,0x5c,0x7a,0x70,0xd0,0x58,0x74,
                0xf7,0x18,0x27,0x5b,0x88,0xd0,0xfa,0x28,0x8b,
                0xd3,0x19,0x98,0x13,0xf0},
        .key_len = 50,
        .data = {0xfa,0x7e,0x18,0xcc,0x54,0x43,0x98,0x1f,0x22,
                 0xc0,0xa5,0xab,0xa2,0x11,0x79,0x15,0xf8,0x9c,
                 0x77,0x81,0xc3,0x4f,0x61,0xf9,0xf4,0x29,0xcb,
                 0x13,0xe0,0xfc,0xd0,0xce,0x94,0x71,0x03,0xbe,
                 0x68,0x4c,0xa8,0x69,0xd7,0xf1,0x25,0xf0,0x8d,
                 0x27,0xb3,0xf2,0xc2,0x1d,0x59,0xad,0xc7,0xab,
                 0x1b,0x66,0xde,0xd9,0x6f,0x0b,0x4f,0xa5,0xf0,
                 0x18,0xb8,0x01,0x56,0xb7,0xa5,0x1c,0xa6,0x2b,
                 0x60,0xe2,0xa6,0x6e,0x0b,0xc6,0x94,0x19,0xeb,
                 0xbf,0x17,0x85,0x07,0x90,0x76,0x30,0xf2,0x4d,
                 0x08,0x62,0xe5,0x1b,0xec,0x10,0x10,0x37,0xf9,
                 0x00,0x32,0x3a,0xf8,0x2e,0x68,0x9b,0x11,0x6f,
                 0x42,0x75,0x84,0x54,0x1c,0x8a,0x9a,0x51,0xac,
                 0x89,0xda,0x1e,0xd7,0x8c,0x7f,0x5e,0xc9,0xe5,
                 0x2a,0x7f},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0x35,0x4f,0x87,0xe9,0x8d,0x27,0x64,0x46,0x83,
                0x6e,0xa0,0x43,0x0c,0xe4,0x52,0x92,0x72,0xa0,
                0x17,0xc2,0x90,0x03,0x9a,0x9d,0xfe,0xa4,0x34,
                0x9b},
        .chunks = {32, 0, 64, 32},
        .num_chunks = 4,
    }, {    // 1 (61)
        .key = {0x82,0xa2,0x89,0xb0,0x91,0x1f,0x55,0xc5,0x32,
                0xe2,0x57,0x18,0x66,0xfa,0x35,0x4d,0x97,0x39,
                0x48,0xec,0x7b,0x89,0x9d,0x57,0x3b,0x83,0x3f,
                0x00,0xd5,0x3e,0xb8,0xd8,0xbf,0x65,0xd5,0x0b,
                0xcd,0x8d,0xaa,0xea,0x54,0xc5,0xd6,0x28,0x90,
                0x6e,0x08,0x4a,0x43,0x4d},
        .key_len = 50,
        .data = {0xd0,0x6d,0x29,0x6c,0xad,0x7d,0x42,0x8b,0x56,
                 0xb2,0x5c,0x53,0xfd,0xfc,0xf5,0x58,0xb5,0xbc,
                 0x42,0x11,0xed,0x31,0xcf,0xe6,0x37,0x32,0x67,
                 0x8a,0x4f,0x23,0x33,0x8b,0x58,0x22,0x25,0x28,
                 0x68,0x81,0x98,0x62,0x47,0x79,0x30,0x56,0x74,
                 0x1c,0x08,0xc9,0x69,0x8c,0x0c,0x05,0x20,0x6b,
                 0xa3,0xe4,0xd6,0x92,0x92,0x2a,0x0f,0x06,0x1d,
                 0x17,0xda,0x27,0x6e,0x3d,0x56,0x2b,0x3b,0x90,
                 0x75,0xcb,0xa4,0xbc,0x00,0x3c,0x5a,0xbb,0xdc,
                 0x61,0x06,0xa6,0x8b,0x3f,0xe9,0xcb,0xf5,0xb1,
                 0xbf,0x01,0x69,0x5d,0xea,0x38,0xdf,0xe6,0xcc,
                 0x54,0x80,0x64,0x75,0x3c,0x68,0x11,0x7e,0x2d,
                 0xaa,0x44,0x34,0x55,0x94,0xa4,0xfa,0x9d,0x35,
                 0x95,0xe8,0xc6,0x1d,0xf7,0xb8,0xb7,0x64,0x10,
                 0xe3,0x15},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0x55,0x07,0xc4,0xc7,0xc1,0x27,0x04,0x6c,0x04,
                0xd1,0x61,0xfd,0xc7,0x5e,0xc2,0xf5,0x12,0x2e,
                0xd4,0x4e,0x86,0x67,0x9c,0x39,0xc6,0x9c,0x5a,
                0xe8},
        .chunks = {64, -1, 64},
        .num_chunks = 3,
    }, {    // 2 (135)
        .key = {0xb3,0x4a,0x6e,0xba,0x59,0xe6,0x32,0xe3,0xc3,
                0x34,0xdf,0x58,0x2f,0xd0,0xb0,0x3a,0xca,0x7f,
                0x64,0x1c,0x69,0x14,0x92,0x0a,0x79,0x94,0x3d,
                0xca,0x39,0x98,0xa8,0x61,0x27,0xf3,0x6b,0xda,
                0xb7,0x95,0xc7,0x42,0x4b,0xaf,0x37,0xf7,0x60,
                0x18,0x47,0x23,0x05,0xf0,0xa9,0x83,0x92,0x83,
                0x86},
        .key_len = 55,
        .data = {0xd0,0x3d,0xd9,0x4e,0x43,0xd9,0x6c,0xa4,0x59,
                 0x92,0x8a,0xa9,0x6d,0x2b,0x81,0xc3,0x5e,0xd5,
                 0x45,0x66,0xe3,0x3e,0x66,0x35,0x1f,0xca,0x40,
                 0x6b,0x76,0x20,0x72,0x7a,0x9e,0xe9,0x91,0xf2,
                 0xf9,0xd4,0x1d,0xa3,0x22,0xde,0xb3,0x06,0xc6,
                 0xd0,0x85,0xd9,0xb5,0x09,0x08,0x0a,0x38,0x7d,
                 0xec,0xd6,0xa6,0xfe,0x51,0x32,0x32,0xbb,0x38,
                 0x6a,0x07,0x06,0x37,0x08,0x97,0x5a,0x72,0xf7,
                 0x2f,0x9c,0xc6,0xe8,0xcf,0xa1,0x47,0xf5,0x3a,
                 0xf1,0xed,0x84,0x49,0xca,0x8a,0x6f,0x84,0x68,
                 0xff,0x62,0xf3,0x84,0xb0,0x84,0x32,0x1b,0x35,
                 0x59,0xc4,0x70,0x53,0xe7,0xa9,0x54,0x2a,0x17,
                 0x33,0xa5,0xaf,0x5f,0x15,0x15,0x5d,0x9e,0xbd,
                 0x2c,0x28,0x47,0xe4,0x91,0xb3,0xc2,0x63,0x85,
                 0xc2,0x0f},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0x1c,0x49,0x44,0x84,0x3c,0x3e,0xe7,0xf9,0x8a,
                0xb5,0x29,0x87,0xe0,0xd2,0xd4,0x49,0x4a,0xa7,
                0x25,0x48,0xbf,0xdc,0x36,0x02,0xdb,0x45,0xb5,
                0xf4},
    }, {    // 3 (136)
        .key = {0x2f,0xd5,0x86,0x3a,0xb5,0xac,0x01,0x09,0xcd,
                0x1c,0xc8,0xa6,0x59,0x8e,0x75,0xd8,0x58,0x11,
                0xa8,0x4d,0x0d,0xf1,0x4c,0xd5,0x5e,0x8b,0x1c,
                0xce,0x7a,0x5f,0x65,0xdf,0xbe,0x67,0x0d,0xea,
                0xda,0xa8,0xd4,0x3b,0x2f,0x06,0xda,0x06,0x7c,
                0x5c,0x62,0x10,0xba,0xcc,0xd5,0xac,0x44,0x54,
                0x0a},
        .key_len = 55,
        .data = {0x85,0xc0,0x2d,0x7c,0xfa,0xb2,0x9f,0x8a,0xdf,
                 0x0f,0xa5,0x5e,0xf3,0x67,0x22,0xa0,0x47,0x57,
                 0xc8,0x86,0x50,0x53,0xd2,0xaf,0x3b,0xa2,0xf6,
                 0x4e,0x80,0xaa,0x95,0x8a,0xba,0x6e,0x36,0x25,
                 0xb6,0x55,0x32,0x5c,0xca,0x2d,0xb0,0x0f,0x68,
                 0x6f,0xd4,0x22,0xf2,0xc5,0x34,0x23,0xd0,0xc9,
                 0x8c,0x2d,0xc1,0x10,0xb2,0x0c,0x6e,0x67,0xcc,
                 0xa1,0x45,0x5c,0xc0,0x88,0x84,0x01,0xec,0xf9,
                 0x94,0xec,0x18,0xec,0x99,0x82,0xa8,0x81,0x47,
                 0x76,0x16,0x9e,0xf7,0x8c,0xa0,0xda,0xfa,0xa3,
                 0x3e,0x9a,0x2d,0xf2,0xd7,0x79,0xcd,0x92,0xb4,
                 0xee,0x8d,0x3c,0x35,0x29,0xe6,0x55,0xc3,0x3d,
                 0xaf,0x27,0x05,0x84,0xed,0x72,0x57,0x3f,0xec,
                 0x23,0x78,0x7e,0x8f,0x63,0x82,0x40,0xe4,0xd3,
                 0x20,0xda},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0xea,0xbb,0xa9,0xf3,0x5b,0xa3,0x9c,0xfb,0x92,
                0x83,0x39,0x0d,0x54,0x25,0x68,0x7c,0xdd,0x70,
                0xd4,0xcb,0x1f,0xea,0x43,0x39,0x25,0x64,0x7c,
                0x79},
    }, {    // 4 (210)
        .key = {0xf3,0xaa,0x51,0xee,0x90,0xaa,0x06,0xe9,0x8e,
                0x23,0x88,0xdf,0x7a,0x3a,0xf2,0xcd,0x69,0x7f,
                0x2a,0x52,0x8f,0x2a,0x14,0x14,0x0d,0xa4,0x05,
                0x60,0x0b,0x4a,0x7b,0x10,0x07,0x6d,0x0b,0xb2,
                0x6c,0x9d,0x9a,0xe6,0x67,0x27,0xd0,0x08,0xf6,
                0xdc,0xca,0x0f,0x42,0x14,0x0f,0xb5,0x2e,0xf3,
                0xc6,0xd9,0x30,0xeb,0x26,0x21,0x60,0x82,0x2d,
                0xe3},
        .key_len = 64,
        .data = {0x2c,0x07,0x0e,0x5e,0xb0,0xa7,0xef,0xbf,0xc4,
                 0x0b,0x23,0x43,0x14,0xc0,0x55,0xfc,0x43,0x6c,
                 0xe1,0xe2,0x30,0x05,0x39,0xb3,0x78,0x42,0x87,
                 0xfd,0x3c,0x4f,0x94,0x78,0x24,0xc5,0xe8,0x9a,
                 0xa3,0xd9,0x33,0x66,0x7d,0xd4,0xeb,0x85,0x87,
                 0xc3,0x37,0x97,0xae,0x6f,0x0c,0xcb,0x3b,0x8f,
                 0x95,0xad,0x56,0x3d,0xd9,0x40,0xc8,0xa7,0x98,
                 0x4d,0x05,0x07,0x75,0xfa,0x69,0xd5,0x5b,0x9e,
                 0xc2,0x3a,0x19,0xd4,0x0f,0xc9,0x4f,0xcf,0x87,
                 0x6a,0x8e,0xed,0xdd,0x96,0xbb,0x8e,0xc3,0xca,
                 0xb4,0x26,0xad,0x35,0x36,0x91,0xd7,0xb4,0xff,
                 0xa7,0x89,0x01,0x2f,0x11,0x9a,0x28,0xf4,0x5b,
                 0x33,0x3a,0x66,0x49,0xd5,0x44,0xb0,0x6f,0x9d,
                 0x82,0x71,0xfc,0xc7,0xd1,0x62,0xd7,0x72,0x63,
                 0x38,0xed},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0x69,0x5c,0x32,0x8d,0xc8,0x58,0xa4,0x6b,0xe7,
                0xac,0x8b,0x8d,0xeb,0xf5,0x8d,0xd9,0xa9,0xbd,
                0x72,0xd7,0xa4,0x08,0xa4,0x3e,0x6b,0xbc,0x69,
                0xa3},
    }, {    // 5 (211)
        .key = {0xa2,0x7f,0x5b,0xa4,0xa0,0xd5,0xa8,0x0c,0xb4,
                0xef,0x9b,0xb1,0x8d,0x4f,0xfc,0x4a,0xd4,0x87,
                0x68,0x0c,0xb5,0xa8,0xf6,0xc6,0x9c,0xfe,0x1b,
                0x29,0x3c,0xfa,0xcb,0x67,0xd6,0x70,0xe1,0x01,
                0xb8,0x03,0xc1,0xd9,0x04,0xdd,0x3c,0x8f,0xb2,
                0xa3,0xb4,0x0a,0xd8,0xa7,0x80,0x55,0x29,0x60,
                0x91,0xad,0xbc,0x18,0x56,0xd8,0x10,0xe5,0xde,
                0xb8},
        .key_len = 64,
        .data = {0x80,0x2e,0xd2,0x88,0x73,0xc2,0x16,0x05,0x62,
                 0xfb,0x91,0x62,0x24,0x6b,0x60,0xd7,0x5e,0x2f,
                 0xfb,0x8c,0xec,0xc5,0xbb,0x08,0x83,0x1e,0x9a,
                 0xec,0x94,0xf4,0x3b,0xe1,0x73,0x5e,0x2a,0xcf,
                 0xb8,0xa2,0x64,0x50,0xe6,0x4a,0x1a,0x64,0x4a,
                 0x7a,0xb2,0x66,0x26,0x80,0x0f,0x3e,0x0f,0xad,
                 0x12,0xd9,0x94,0x63,0x60,0xd6,0x20,0x70,0x66,
                 0x82,0x22,0x9a,0x9b,0x07,0x6a,0x7b,0x05,0xe0,
                 0xd6,0x90,0xff,0x90,0x2f,0x8c,0x39,0x94,0x4e,
                 0xf2,0x00,0x96,0x83,0xe2,0xc0,0xe3,0x37,0x33,
                 0xc7,0x1f,0xcf,0xc3,0x4b,0x38,0xd4,0x0b,0xaf,
                 0x98,0x51,0xf7,0x02,0x9f,0x7d,0xf4,0xc9,0x50,
                 0x9a,0x4a,0xfe,0x4a,0x1d,0xac,0x58,0xfa,0x0f,
                 0x0d,0xcd,0x10,0xb3,0xb6,0x56,0x2f,0x69,0x96,
                 0xf2,0xe9},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0xee,0x63,0xe8,0x6a,0xac,0x85,0xf3,0x69,0x61,
                0xd6,0x97,0x45,0x12,0x88,0xcf,0x78,0x00,0x41,
                0xea,0xfe,0x46,0x2c,0xa6,0x55,0x70,0x35,0xd9,
                0x2a},
    }, {    // 6 (285)
        .key = {0x5f,0x44,0x8c,0x3d,0x3d,0xf6,0xce,0xab,0x97,
                0x35,0x68,0x19,0xda,0x0d,0x45,0x96,0x62,0xb2,
                0xd5,0xa7,0x36,0x6a,0x5d,0x46,0xf2,0xa6,0x91,
                0x2a,0x04,0x72,0x64,0x49,0x1b,0x10,0x1f,0x9c,
                0xdd,0xe0,0xb0,0x22,0xd9,0x84,0x45,0x27,0xbc,
                0x40,0x1a,0xc6,0xc7,0xa7,0x8f,0xad,0x80,0x74,
                0x7e,0x20,0xa8,0xb6,0xcb,0x41,0x6e,0x30,0x3d,
                0x8e,0xe2,0xff},
        .key_len = 66,
        .data = {0xd5,0x2f,0xce,0x9d,0x18,0x22,0x00,0x1f,0x8a,
                 0x25,0x2e,0x34,0xb4,0xad,0xc8,0x18,0x31,0xf9,
                 0xda,0x37,0x0f,0xec,0x92,0x9d,0x79,0x1c,0xfd,
                 0xae,0x93,0x27,0x11,0x17,0xa6,0x46,0xf0,0x04,
                 0x7d,0x24,0x6b,0x94,0x0f,0xf0,0xc5,0x89,0x5e,
                 0xb0,0xba,0x45,0x9e,0xd9,0xc0,0xf2,0x9a,0x1d,
                 0x8d,0x00,0x54,0x85,0xd9,0xd4,0xeb,0xbf,0x65,
                 0xbe,0xc2,0xb9,0x3c,0x2c,0xdf,0x6f,0x39,0x1e,
                 0x91,0x01,0xc9,0xda,0x56,0x08,0xf2,0xe1,0xfb,
                 0x2a,0x95,0x21,0x05,0xec,0xf0,0x6c,0x50,0xc8,
                 0x66,0x29,0xf6,0x80,0x83,0x4a,0x72,0xb1,0x3b,
                 0x7e,0x06,0xab,0x72,0xa7,0x5a,0x80,0xc2,0x8e,
                 0x65,0x5b,0x78,0xd5,0x90,0x13,0xdf,0x2e,0x19,
                 0x00,0xc2,0xcb,0xb6,0x0e,0x2c,0x16,0x7c,0x0f,
                 0x72,0xa1},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0x1b,0x34,0x59,0xef,0x88,0x67,0x1f,0x03,0x1b,
                0xe6,0x02,0x52,0x70,0x12,0x31,0xfe,0xc1,0x71,
                0x32,0xa0,0xba,0xa7,0x5d,0x13,0x93,0x8f,0xbc,
                0x9a},
    }, {    // 7 (286)
        .key = {0x08,0xe1,0x3b,0xab,0x53,0x26,0x36,0x9a,0x27,
                0x06,0xc2,0xb3,0xe7,0xe8,0xf9,0xb7,0x1c,0xbe,
                0x56,0x46,0xbd,0x37,0x1a,0xad,0xae,0x35,0xef,
                0xff,0x86,0x81,0xcc,0x67,0x95,0xbe,0x18,0xbc,
                0x26,0x9b,0x12,0x56,0x01,0x4d,0x70,0x20,0x34,
                0x3d,0x46,0xef,0x13,0xfd,0x2a,0x12,0x7e,0x81,
                0x96,0x2b,0x62,0x8c,0x8e,0x3d,0x82,0x68,0x82,
                0x05,0x78,0xda},
        .key_len = 66,
        .data = {0x9e,0xe9,0x13,0x62,0x40,0xb9,0xc4,0x31,0x74,
                 0x7b,0xa3,0x63,0xa9,0xb2,0xb8,0x36,0x3b,0x1d,
                 0x57,0xfb,0x45,0x38,0x98,0xbc,0xb7,0x06,0x08,
                 0x21,0xa8,0x0e,0x9e,0x94,0xed,0xa3,0xf1,0xa4,
                 0xea,0x69,0xa7,0xa7,0xa8,0x15,0x31,0xa4,0x28,
                 0x74,0xb6,0x70,0xf7,0xaf,0x4c,0x16,0x03,0xcd,
                 0x6e,0x7c,0xad,0x79,0xab,0x44,0x1f,0x06,0x9b,
                 0xf1,0xe0,0xb0,0x17,0x4b,0xa5,0x25,0xa9,0x04,
                 0x6b,0x44,0x42,0x9b,0xc2,0x24,0x2b,0x81,0x6c,
                 0x58,0x3e,0x7b,0x26,0x71,0x56,0x47,0xc6,0xc5,
                 0x04,0x82,0x86,0x6f,0x84,0xc9,0xa0,0x97,0xef,
                 0x1f,0x1b,0xf4,0xb1,0x8e,0xe4,0x8e,0x3e,0x11,
                 0x20,0xc9,0x01,0xb2,0xc1,0x9f,0x95,0xf0,0x57,
                 0x2d,0x38,0x63,0x29,0x71,0x7d,0xa3,0x85,0x52,
                 0x41,0x65},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0xf6,0xf7,0x42,0xb7,0x0c,0x95,0xce,0x5f,0x69,
                0xfa,0x8a,0xb7,0x27,0x00,0x4e,0xf5,0x46,0xd6,
                0xde,0x9d,0x8f,0x05,0xad,0x9f,0x84,0xa0,0x21,
                0x0a},
    }, {    // 8 (360)
        .key = {0x4b,0x0f,0x8f,0xda,0x08,0x01,0x7b,0x10,0xb6,
                0x47,0xfc,0xd6,0xcd,0x04,0xf7,0x87,0x0c,0x92,
                0xb2,0x68,0x75,0x74,0xf2,0x38,0x99,0x8c,0x60,
                0x08,0x15,0x8e,0x31,0x4d,0x5d,0xb5,0x06,0x34,
                0xb8,0xb5,0x11,0x35,0x8c,0xf0,0x7a,0xeb,0xdc,
                0xec,0x01,0x23,0x0f,0x05,0xe4,0x33,0xf3,0x5f,
                0x03,0x8d,0x01,0x1f,0x42,0x93,0xe3,0xdb,0x2f,
                0xad,0x33},
        .key_len = 65,
        .data = {0x29,0x17,0xcd,0x57,0x24,0x31,0x9d,0xcb,0x5c,
                 0x08,0xd9,0x17,0xb6,0x7f,0x25,0x62,0x8d,0x15,
                 0x54,0x3f,0xf7,0x17,0xd1,0x82,0x49,0x15,0x3d,
                 0x51,0xdd,0x92,0x59,0x7e,0x12,0xff,0x27,0x14,
                 0x95,0xeb,0x4c,0x2f,0xdf,0x74,0xb9,0x11,0xff,
                 0x01,0x8a,0x73,0x9a,0x33,0x2e,0x19,0x30,0x18,
                 0xc9,0xa9,0xa0,0xa2,0xd6,0xbd,0xef,0x58,0x11,
                 0x37,0x45,0x4c,0x94,0xd5,0x38,0x4d,0x40,0x05,
                 0x4d,0x5d,0x5e,0xfa,0xbc,0x66,0x86,0xff,0x74,
                 0x28,0xc0,0x0b,0x5f,0x76,0xea,0x96,0xe7,0xa2,
                 0x5a,0xac,0xb9,0x36,0xc4,0x40,0xc9,0xe4,0x5d,
                 0xc2,0x96,0xc0,0x40,0xf4,0xaf,0xad,0x11,0xf9,
                 0x76,0x15,0xe1,0xae,0x24,0xde,0x52,0x3e,0x0d,
                 0x99,0xfc,0xf1,0x26,0xfc,0x0f,0x45,0xc4,0x92,
                 0x39,0x40},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0x45,0x3b,0x23,0x73,0xce,0x46,0x59,0x58,0x55,
                0xf6,0x03,0x0d,0x24,0x3d,0x16,0x46,0x29,0x38,
                0x89,0x41,0x6f,0x05,0x37,0x3e,0x78,0x12,0x6f,
                0x59},
    }, {    // 9 (361)
        .key = {0x28,0x9e,0x2a,0xec,0xe1,0x04,0x86,0x74,0xbd,
                0x57,0x1a,0x4d,0xac,0x53,0x04,0x33,0x51,0xa7,
                0x89,0x64,0xbd,0xa6,0xfd,0xf6,0x7e,0x3f,0xe3,
                0x6d,0xd8,0x2f,0x56,0x8d,0x52,0x75,0xa8,0x6d,
                0x75,0x06,0x48,0xcd,0xd0,0xc3,0xe9,0xa9,0x3f,
                0xa8,0xe0,0x0f,0x6d,0x46,0xce,0x2d,0x98,0x73,
                0x80,0xae,0x94,0x40,0x13,0x47,0x13,0x2b,0x5c,
                0x8c,0x23},
        .key_len = 65,
        .data = {0xb4,0x53,0xfc,0x9e,0x54,0x12,0x2b,0x18,0x14,
                 0x83,0x00,0x5c,0x41,0x53,0xcb,0x8b,0x47,0xce,
                 0xf4,0x7a,0x74,0x99,0xe3,0x07,0xfe,0x9f,0x1e,
                 0xae,0x48,0x4d,0xb1,0x57,0x61,0x02,0xd3,0x72,
                 0xcd,0xc1,0xc6,0x46,0x89,0x2a,0xff,0x86,0x49,
                 0xa8,0x87,0x26,0x04,0xeb,0xa1,0x6c,0xb2,0x99,
                 0xad,0x4a,0x55,0x40,0x4c,0xeb,0x36,0x90,0xc5,
                 0x8c,0x71,0xc7,0xa8,0x8d,0xb3,0x6c,0xb1,0xc8,
                 0x43,0x40,0xee,0x21,0x3f,0x72,0x24,0x5e,0x29,
                 0x12,0x70,0x3a,0xa2,0xc8,0x2a,0xb4,0x74,0xc6,
                 0x0e,0xad,0xd6,0xfb,0x9e,0xb2,0xec,0x89,0xc2,
                 0x17,0x8e,0x7a,0xb2,0x45,0x4a,0xae,0x15,0x54,
                 0xac,0x18,0x6b,0xdd,0xbc,0x2b,0xa8,0x09,0xc9,
                 0x8e,0x21,0xee,0x65,0x17,0xac,0x1c,0xb1,0xf7,
                 0x03,0x72},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0x01,0xda,0xfb,0x71,0x6c,0x50,0xb0,0x94,0x57,
                0xec,0xc8,0x77,0x81,0x41,0xca,0x84,0xb6,0x45,
                0xa2,0x32,0xa4,0xa3,0x9e,0xd0,0x23,0x0c,0x73,
                0x49},
    }
};

struct HMAC_TEST_VECTOR fips_sha256_hmac_test_vector[] = {
    {	// 0 (30)
        .key = {0x97,0x79,0xd9,0x12,0x06,0x42,0x79,0x7f,0x17,
                0x47,0x02,0x5d,0x5b,0x22,0xb7,0xac,0x60,0x7c,
                0xab,0x08,0xe1,0x75,0x8f,0x2f,0x3a,0x46,0xc8,
                0xbe,0x1e,0x25,0xc5,0x3b,0x8c,0x6a,0x8f,0x58,
                0xff,0xef,0xa1,0x76},
        .key_len = 40,
        .data = {0xb1,0x68,0x9c,0x25,0x91,0xea,0xf3,0xc9,0xe6,
                 0x60,0x70,0xf8,0xa7,0x79,0x54,0xff,0xb8,0x17,
                 0x49,0xf1,0xb0,0x03,0x46,0xf9,0xdf,0xe0,0xb2,
                 0xee,0x90,0x5d,0xcc,0x28,0x8b,0xaf,0x4a,0x92,
                 0xde,0x3f,0x40,0x01,0xdd,0x9f,0x44,0xc4,0x68,
                 0xc3,0xd0,0x7d,0x6c,0x6e,0xe8,0x2f,0xac,0xea,
                 0xfc,0x97,0xc2,0xfc,0x0f,0xc0,0x60,0x17,0x19,
                 0xd2,0xdc,0xd0,0xaa,0x2a,0xec,0x92,0xd1,0xb0,
                 0xae,0x93,0x3c,0x65,0xeb,0x06,0xa0,0x3c,0x9c,
                 0x93,0x5c,0x2b,0xad,0x04,0x59,0x81,0x02,0x41,
                 0x34,0x7a,0xb8,0x7e,0x9f,0x11,0xad,0xb3,0x04,
                 0x15,0x42,0x4c,0x6c,0x7f,0x5f,0x22,0xa0,0x03,
                 0xb8,0xab,0x8d,0xe5,0x4f,0x6d,0xed,0x0e,0x3a,
                 0xb9,0x24,0x5f,0xa7,0x95,0x68,0x45,0x1d,0xfa,
                 0x25,0x8e},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x76,0x9f,0x00,0xd3,0xe6,0xa6,0xcc,0x1f,0xb4,
                0x26,0xa1,0x4a,0x4f,0x76,0xc6,0x46,0x2e,0x61,
                0x49,0x72,0x6e,0x0d,0xee,0x0e,0xc0,0xcf,0x97,
                0xa1,0x66,0x05,0xac,0x8b},
        .chunks = {32, 0, 64, 32},
        .num_chunks = 4,
    }, {	// 1 (31)
        .key = {0x09,0x67,0x5f,0x2d,0xcc,0x47,0x83,0xb5,0x99,
                0xf1,0x8f,0xb7,0x65,0x58,0x36,0x68,0xa0,0xfd,
                0x8a,0xe4,0x09,0x6f,0x6f,0xcd,0xc6,0x0d,0x4f,
                0x35,0xb4,0x13,0x0f,0xbe,0xfc,0xd5,0x42,0xff,
                0xe7,0x45,0x9d,0x2a},
        .key_len = 40,
        .data = {0x0c,0xf2,0x19,0x8c,0x31,0x37,0x6f,0x5c,0x89,
                 0x15,0x66,0x01,0x37,0x72,0x5f,0x2b,0xbc,0x18,
                 0x0a,0x98,0x6e,0x5a,0x7b,0xda,0x27,0xfa,0x81,
                 0x59,0x3a,0x4a,0x33,0x9b,0xab,0x92,0xcb,0xc3,
                 0x9f,0xb2,0xb8,0x58,0x11,0x08,0xee,0x48,0xc7,
                 0x94,0x81,0x2d,0x84,0x5a,0x72,0xce,0x80,0x08,
                 0xc9,0xe9,0x15,0xd9,0xe3,0x30,0xbb,0xb9,0x0e,
                 0x91,0x36,0xaa,0x53,0xba,0x0e,0x66,0x93,0xdd,
                 0x40,0x46,0xd6,0xb0,0x33,0x62,0xdf,0xb9,0xed,
                 0xfa,0x04,0xc8,0x87,0x15,0x3c,0xc5,0xde,0x67,
                 0x7a,0xab,0x8c,0x78,0x39,0xd5,0x17,0x03,0x58,
                 0x79,0x67,0x9c,0x29,0x72,0x7e,0x96,0xc5,0x42,
                 0x63,0x24,0xa2,0x57,0x5f,0xbe,0x67,0x8d,0x6c,
                 0xc7,0xfe,0xf5,0xeb,0x6c,0xeb,0xd5,0x95,0xcf,
                 0xdd,0xef},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x6b,0x14,0x2d,0x4d,0xfe,0x21,0x7f,0x18,0x81,
                0xaa,0x0e,0x64,0x83,0xb2,0x71,0xdd,0x5d,0x43,
                0xf7,0x0b,0x85,0x60,0x59,0x53,0xa0,0xfe,0xf2,
                0x72,0xdd,0xde,0x46,0xca},
        .chunks = {64, -1, 64},
        .num_chunks = 3,
    }, { 	// 2 (75)
        .key = {0xb7,0x63,0x26,0x3d,0xc4,0xfc,0x62,0xb2,0x27,
                0xcd,0x3f,0x6b,0x4e,0x9e,0x35,0x8c,0x21,0xca,
                0x03,0x6c,0xe3,0x96,0xab,0x92,0x59,0xc1,0xbe,
                0xdd,0x2f,0x5c,0xd9,0x02,0x97,0xdc,0x70,0x3c,
                0x33,0x6e,0xca,0x3e,0x35,0x8a,0x4d,0x6d,0xc5},
        .key_len = 45,
        .data = {0x53,0xcb,0x09,0xd0,0xa7,0x88,0xe4,0x46,0x6d,
                 0x01,0x58,0x8d,0xf6,0x94,0x5d,0x87,0x28,0xd9,
                 0x36,0x3f,0x76,0xcd,0x01,0x2a,0x10,0x30,0x8d,
                 0xad,0x56,0x2b,0x6b,0xe0,0x93,0x36,0x48,0x92,
                 0xe8,0x39,0x7a,0x8d,0x86,0xf1,0xd8,0x1a,0x20,
                 0x96,0xcf,0xc8,0xa1,0xbb,0xb2,0x6a,0x1a,0x75,
                 0x52,0x5f,0xfe,0xbf,0xcf,0x16,0x91,0x1d,0xad,
                 0xd0,0x9e,0x80,0x2a,0xa8,0x68,0x6a,0xcf,0xd1,
                 0xe4,0x52,0x46,0x20,0x25,0x4a,0x6b,0xca,0x18,
                 0xdf,0xa5,0x6e,0x71,0x41,0x77,0x56,0xe5,0xa4,
                 0x52,0xfa,0x9a,0xe5,0xae,0xc5,0xdc,0x71,0x59,
                 0x1c,0x11,0x63,0x0e,0x9d,0xef,0xec,0x49,0xa4,
                 0xec,0xf8,0x5a,0x14,0xf6,0x0e,0xb8,0x54,0x65,
                 0x78,0x99,0x97,0x2e,0xa5,0xbf,0x61,0x59,0xcb,
                 0x95,0x47},
        .data_len =128,
        .mac_len = 32,
        .mac = {0x73,0x73,0x01,0xde,0xa9,0x3d,0xb6,0xbc,0xba,
                0xdd,0x7b,0xf7,0x96,0x69,0x39,0x61,0x31,0x7c,
                0xa6,0x80,0xb3,0x80,0x41,0x6f,0x12,0xf4,0x66,
                0xf0,0x65,0x26,0xb3,0x6b},
    }, {	// 3 (76)
        .key = {0x9f,0xe4,0x2d,0xfa,0xc9,0x2a,0x4a,0x13,0x6f,
                0xa7,0xc9,0xf6,0xe3,0x31,0xb5,0xd3,0xa6,0x1a,
                0xa7,0x30,0x35,0xb5,0x3a,0x8d,0x25,0x17,0xbe,
                0x43,0x72,0x1b,0x31,0xb2,0x15,0xa9,0x6b,0x9b,
                0xd4,0x37,0x98,0xcb,0x5e,0x8f,0xeb,0xfa,0x97},
        .key_len = 45,
        .data = {0xf9,0x66,0x0f,0xb7,0x84,0xc1,0x4b,0x5f,0xbe,
                 0xc2,0x80,0x52,0x6a,0x69,0xc2,0x29,0x4f,0xba,
                 0x12,0xae,0xa1,0x63,0x78,0x9b,0xbe,0x9f,0x52,
                 0xa5,0x1b,0x5a,0xeb,0xb9,0x7d,0x96,0x4f,0x86,
                 0x6c,0x0d,0x5e,0x3b,0xe4,0x18,0x20,0x92,0x4f,
                 0xcf,0x58,0x0d,0xb0,0x72,0x5c,0x7f,0x21,0x08,
                 0x23,0xcf,0x7f,0x45,0xa0,0xf9,0x64,0xb1,0x4e,
                 0x55,0x55,0x07,0x0d,0x1c,0x3d,0xdb,0x2c,0x28,
                 0x1a,0x80,0xc7,0xfb,0xf7,0x29,0x53,0x03,0x1a,
                 0x4e,0x77,0x1d,0x7e,0x52,0x1d,0x57,0x84,0x62,
                 0xca,0xfa,0xe5,0xa0,0x2a,0xc8,0xeb,0x81,0xf0,
                 0x82,0xe1,0x73,0xdd,0xad,0xc8,0xc4,0x1d,0x96,
                 0x4b,0xbf,0xda,0x94,0xf5,0x18,0x0c,0x8d,0xa2,
                 0x8a,0x8e,0xbb,0x33,0xbe,0x77,0xb0,0x86,0x6f,
                 0xa7,0x98},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x77,0x86,0xc1,0x55,0xd1,0x0c,0x74,0x1b,0x63,
                0xec,0x65,0x0b,0x7b,0x1a,0xa3,0xbf,0xd7,0x1a,
                0xc7,0x18,0x81,0xad,0x06,0xae,0x98,0xfb,0x08,
                0x2f,0x17,0xe0,0xca,0xa0},
    }, {	// 4 (120)
        .key = {0x99,0x28,0x68,0x50,0x4d,0x25,0x64,0xc4,0xfb,
                0x47,0xbc,0xbd,0x4a,0xe4,0x82,0xd8,0xfb,0x0e,
                0x8e,0x56,0xd7,0xb8,0x18,0x64,0xe6,0x19,0x86,
                0xa0,0xe2,0x56,0x82,0xda,0xeb,0x5b,0x50,0x17,
                0x7c,0x09,0x5e,0xdc,0x9e,0x97,0x1d,0xa9,0x5c,
                0x32,0x10,0xc3,0x76,0xe7,0x23,0x36,0x5a,0xc3,
                0x3d,0x1b,0x4f,0x39,0x18,0x17,0xf4,0xc3,0x51,
                0x24},
        .key_len = 64,
        .data = {0xed,0x4f,0x26,0x9a,0x88,0x51,0xeb,0x31,0x54,
                 0x77,0x15,0x16,0xb2,0x72,0x28,0x15,0x52,0x00,
                 0x77,0x80,0x49,0xb2,0xdc,0x19,0x63,0xf3,0xac,
                 0x32,0xba,0x46,0xea,0x13,0x87,0xcf,0xbb,0x9c,
                 0x39,0x15,0x1a,0x2c,0xc4,0x06,0xcd,0xc1,0x3c,
                 0x3c,0x98,0x60,0xa2,0x7e,0xb0,0xb7,0xfe,0x8a,
                 0x72,0x01,0xad,0x11,0x55,0x2a,0xfd,0x04,0x1e,
                 0x33,0xf7,0x0e,0x53,0xd9,0x7c,0x62,0xf1,0x71,
                 0x94,0xb6,0x61,0x17,0x02,0x8f,0xa9,0x07,0x1c,
                 0xc0,0xe0,0x4b,0xd9,0x2d,0xe4,0x97,0x2c,0xd5,
                 0x4f,0x71,0x90,0x10,0xa6,0x94,0xe4,0x14,0xd4,
                 0x97,0x7a,0xbe,0xd7,0xca,0x6b,0x90,0xba,0x61,
                 0x2d,0xf6,0xc3,0xd4,0x67,0xcd,0xed,0x85,0x03,
                 0x25,0x98,0xa4,0x85,0x46,0x80,0x4f,0x9c,0xf2,
                 0xec,0xfe},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x2f,0x83,0x21,0xf4,0x16,0xb9,0xbb,0x24,0x9f,
                0x11,0x3b,0x13,0xfc,0x12,0xd7,0x0e,0x16,0x68,
                0xdc,0x33,0x28,0x39,0xc1,0x0d,0xaa,0x57,0x17,
                0x89,0x6c,0xb7,0x0d,0xdf},
    }, {	// 5 (121)
        .key = {0xce,0xab,0x39,0x8e,0x41,0x07,0x48,0x3e,0xde,
                0x64,0xce,0x10,0x7c,0x92,0x70,0xe6,0x02,0x27,
                0x78,0xb6,0x1f,0x6a,0x25,0x8d,0x3b,0x70,0x45,
                0xd4,0xad,0x85,0x06,0xd3,0x2e,0xce,0x0a,0x73,
                0x8d,0x2c,0xb9,0x48,0xa5,0x62,0xdb,0xce,0x8d,
                0x7b,0x66,0xf3,0x0e,0x66,0x94,0xd6,0x5a,0xe4,
                0x39,0xcf,0xfa,0xa4,0x54,0xaf,0x09,0xab,0xe4,
                0x49},
        .key_len = 64,
        .data = {0x6d,0xde,0x9a,0xe8,0x67,0xe2,0xfe,0xb3,0x67,
                 0x00,0x8a,0x97,0x5d,0x78,0x53,0xed,0x8f,0x89,
                 0x69,0x0f,0x3c,0x87,0xa1,0x10,0x7f,0x2e,0x98,
                 0xaa,0x77,0x36,0xf4,0x77,0xa5,0x27,0xed,0x64,
                 0x95,0x6f,0x0d,0x64,0xc1,0xb2,0x33,0x61,0xb2,
                 0x61,0xde,0x78,0x68,0x8e,0xa8,0x65,0xfc,0xff,
                 0x11,0x3c,0x84,0x81,0x7e,0x5b,0x37,0x7e,0x82,
                 0x9c,0xd2,0xd2,0x5b,0xcf,0x3a,0xdb,0xc0,0x67,
                 0x62,0xcf,0xda,0x73,0x6f,0x53,0x90,0xd0,0x1a,
                 0x49,0x07,0x9d,0x56,0xe9,0x69,0xf0,0x33,0x13,
                 0xe6,0xc7,0x03,0xe3,0xf9,0x42,0xbb,0x87,0xed,
                 0x0f,0x9c,0x4d,0x9f,0x25,0x12,0x00,0x85,0xb5,
                 0xdc,0x75,0xef,0x5d,0x6d,0x61,0x8d,0xa0,0x92,
                 0x6d,0x32,0x93,0x56,0x8d,0xd7,0xd8,0x23,0x8d,
                 0xe3,0xd0},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x2d,0x3a,0x76,0x05,0x95,0xf3,0xfb,0x19,0x29,
                0x3c,0xc6,0xd2,0x36,0x51,0x22,0x2a,0x9f,0x5a,
                0x4f,0x02,0x28,0x44,0x57,0xa9,0xc1,0xed,0x4c,
                0x43,0xac,0x99,0x3c,0xa5},
    }, {	// 6 (165)
        .key = {0xc0,0x9e,0x29,0x07,0x1c,0x40,0x5d,0x5e,0x82,
                0x0d,0x34,0x5a,0x46,0xdb,0xbf,0x1e,0x0f,0x82,
                0x02,0xe9,0x2d,0xe3,0xed,0x3e,0x2d,0x29,0x8e,
                0x43,0xaa,0x4f,0x84,0x68,0x66,0xe3,0xb7,0x48,
                0x99,0x09,0x46,0xd4,0x88,0xc2,0xc1,0xae,0x5a,
                0x6e,0x99,0xd3,0x27,0x90,0xd4,0x7d,0x53,0xd2,
                0x05,0x48,0x1a,0x49,0x7c,0x93,0x6b,0xf9,0xba,
                0x29,0xfa,0x9c,0x28,0x21,0x91,0x9f},
        .key_len = 70,
        .data = {0xea,0x72,0x40,0x52,0x99,0x80,0x07,0x6d,0x3b,
                 0x02,0x8a,0x08,0x3e,0xbc,0x4e,0x24,0xef,0xda,
                 0xa0,0x6c,0x9c,0x84,0xd7,0x6b,0xf5,0xb2,0xd9,
                 0xfd,0xb8,0x42,0xe1,0x03,0x8e,0x48,0x7f,0x5b,
                 0x30,0xa5,0xe0,0x10,0xcd,0xdb,0x4f,0xcd,0xb0,
                 0x1f,0xfc,0x98,0x1e,0xb0,0xfc,0xbc,0x7d,0x68,
                 0x92,0x07,0xbc,0x90,0xad,0x36,0xee,0xf9,0xb1,
                 0xae,0x38,0x48,0x7a,0x6d,0xee,0x92,0x9f,0x3f,
                 0xf9,0x29,0xf3,0x35,0x7c,0xb5,0x52,0x53,0xb7,
                 0x86,0x9a,0x89,0x2b,0x28,0xf7,0xe5,0xfe,0x38,
                 0x64,0x06,0xa2,0x77,0x6e,0xd4,0xb2,0x1d,0x3b,
                 0x6e,0x1c,0x70,0xcc,0x64,0x85,0x94,0x7f,0x27,
                 0xe9,0xa5,0xd8,0xbd,0x82,0x03,0x80,0xb9,0xec,
                 0xed,0x8e,0x6b,0x86,0x52,0x06,0x54,0x1b,0xe3,
                 0x9f,0xdc},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x49,0xae,0x1c,0x4a,0x7a,0x57,0x0f,0xde,0x47,
                0xf7,0x51,0x7a,0xb1,0x88,0x98,0xb1,0xb9,0x91,
                0xd0,0x3c,0xfc,0xf8,0xc4,0x5b,0xb3,0x61,0x5b,
                0x5f,0x75,0x5d,0xa6,0x82},
    }, {	// 7 (166)
        .key = {0xbc,0xe5,0x0c,0xdf,0xff,0x84,0x38,0x85,0xd4,
                0xf3,0x64,0xd6,0x9f,0x93,0xbf,0x58,0xa2,0x32,
                0x2c,0x70,0x7b,0x82,0xe8,0x78,0xee,0xc9,0x6d,
                0x11,0xe5,0xdb,0x97,0xbb,0xb5,0x46,0x06,0xa3,
                0xa3,0xcc,0xc3,0xbb,0xa7,0x16,0x26,0x10,0x70,
                0xa6,0xf7,0x59,0xa7,0x0e,0xd3,0xcb,0x78,0x5f,
                0xd1,0x35,0x4f,0xe5,0x66,0x48,0xdf,0x11,0x86,
                0x36,0x69,0xb7,0x0c,0x80,0x3b,0x7a},
        .key_len = 70,
        .data = {0x93,0xb7,0xef,0x0e,0x47,0x0d,0xdf,0xac,0x6a,
                 0xef,0x93,0xc0,0xdc,0xd3,0x7b,0x8f,0x1c,0x4b,
                 0xaf,0x5e,0xad,0xd9,0x78,0xe3,0xbf,0x05,0x12,
                 0xfa,0x0b,0xae,0xb0,0x99,0xff,0x9e,0xc1,0x06,
                 0x1b,0x61,0x72,0x47,0x9b,0x56,0x74,0xdb,0x56,
                 0x06,0xff,0xa7,0xe6,0xb5,0x17,0x33,0x09,0x37,
                 0x0e,0x16,0x47,0x05,0x4a,0xaf,0xd5,0x90,0x48,
                 0x16,0xba,0xd5,0xe1,0x52,0x30,0x32,0xcc,0xcd,
                 0x4d,0x78,0x65,0x05,0xe2,0x41,0xac,0x83,0xa4,
                 0x84,0x91,0x11,0x89,0x66,0x6f,0x28,0x75,0x53,
                 0xd6,0xa8,0x16,0x4e,0x8d,0xcb,0x0c,0x85,0xd7,
                 0x5c,0x4e,0x29,0xf6,0x24,0xc9,0x7c,0xee,0xa6,
                 0x4a,0x2c,0x8b,0x0c,0x9d,0xdf,0xa5,0x60,0xf7,
                 0x0f,0xa3,0xff,0x91,0x18,0x3e,0x4b,0x96,0x8f,
                 0x88,0xa1},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x37,0xf9,0xf3,0x29,0x18,0x30,0x82,0x10,0x84,
                0x9d,0xfe,0xbf,0x8d,0xd4,0x56,0x80,0x4b,0xab,
                0xd6,0x84,0x5a,0xf0,0x72,0x18,0xf9,0xd9,0xbe,
                0x9d,0xf9,0x74,0x3d,0x55},
    }, {	// 8 (210)
        .key = {0x81,0x57,0x43,0x23,0xc9,0x73,0x54,0x07,0x19,
                0xd1,0x92,0x83,0x3d,0xdb,0x51,0xf1,0x3a,0x52,
                0xdc,0xba,0xe2,0x94,0xae,0xbe,0xa5,0x1b,0xe5,
                0xf6,0xaa,0x47,0xf3,0x57,0x1f,0x5d,0x97,0xfa,
                0xcd,0xcf,0x0c,0x7b,0xef,0xbe,0x80,0x9f,0x44,
                0xbd,0xc7,0x39,0x63,0xd8,0x51,0x4e,0x4f,0xd5,
                0x59,0x77,0x4b,0xb9,0x60,0x87,0xef,0x8e,0xda,
                0x6e,0x7c,0x64,0x27,0x5d,0x6d,0x96,0xc4,0x2b,
                0x4e,0x4e},
        .key_len = 74,
        .data = {0xb9,0xe9,0x44,0xe0,0xb4,0x2d,0x0f,0xf4,0x54,
                 0xf7,0xf8,0xaa,0x24,0xf0,0x0e,0x9e,0xe0,0x39,
                 0x05,0x8c,0xe4,0x09,0x41,0x11,0xe3,0x97,0x31,
                 0xb6,0xdc,0x3a,0xde,0x2a,0x4a,0xce,0xc4,0xcf,
                 0x9c,0x5b,0xe0,0x78,0xe4,0xf1,0x0a,0x72,0xd3,
                 0xd6,0x85,0xc1,0xe5,0xe4,0xd5,0xab,0xd9,0x2c,
                 0xd0,0x7b,0x64,0xdf,0xf8,0x7f,0x26,0x6f,0x08,
                 0x53,0xdd,0xf1,0xcd,0x61,0xd9,0xc6,0x37,0xa9,
                 0xb0,0x7a,0xb0,0xbe,0x32,0xec,0xac,0x11,0x9f,
                 0xaf,0x82,0x72,0x18,0xb1,0x7a,0xd4,0x54,0x1a,
                 0x27,0x51,0x94,0x77,0xf7,0x6e,0xd9,0x18,0x08,
                 0x9f,0x54,0xb6,0x3d,0x0e,0x1e,0x5a,0x92,0x98,
                 0x29,0x79,0xac,0x18,0x77,0x64,0xb5,0xe9,0x89,
                 0xe0,0x66,0xa6,0x1b,0x10,0x65,0x34,0x0e,0x9c,
                 0xd2,0x03},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x51,0x4b,0xd1,0x84,0x95,0xf6,0xde,0x0e,0x23,
                0x70,0x54,0xb8,0xe3,0xba,0x1a,0x74,0xc3,0xfa,
                0xda,0x42,0x79,0xad,0x6b,0x85,0x50,0xf3,0xa1,
                0x47,0x12,0xc5,0x28,0xdf},
    }, {	// 9 (211)
        .key = {0x44,0xf7,0x1c,0x23,0x17,0xcd,0xe5,0x21,0x51,
                0xc8,0x42,0x60,0xd1,0xd3,0xc0,0x4a,0x28,0xcc,
                0x15,0xce,0x5b,0x38,0x02,0xb2,0xe5,0x35,0x7e,
                0x2b,0xfc,0xaf,0x10,0xab,0x15,0xd7,0x7d,0xfa,
                0xaa,0xd1,0xa3,0x88,0x3b,0xad,0xa5,0x02,0x93,
                0x99,0x48,0x23,0x4c,0x55,0x9d,0xcd,0x95,0xe7,
                0xe1,0x58,0x33,0x8f,0xa1,0x2a,0xc6,0xfd,0x21,
                0x87,0x4e,0xc2,0xff,0xab,0xed,0x05,0x14,0x16,
                0xef,0x77},
        .key_len = 74,
        .data = {0x2a,0xc0,0xbb,0x05,0x24,0xc2,0x2b,0x90,0x2d,
                 0xe3,0x4c,0xe6,0x4e,0x61,0x72,0xd1,0xb2,0x07,
                 0x4e,0x15,0x9f,0x51,0x7a,0xb1,0xab,0xd1,0x52,
                 0x62,0x2c,0xd1,0x06,0x69,0xf0,0x3a,0xed,0x8e,
                 0x2e,0xb5,0x1c,0x65,0xbd,0x0f,0x38,0xd0,0x84,
                 0xe2,0x88,0xc5,0x32,0x72,0x4e,0x51,0x2f,0xd5,
                 0x58,0xdd,0xd2,0x57,0xd2,0xb1,0xd4,0x1c,0x5e,
                 0xb6,0x04,0x07,0x67,0x80,0x3d,0xdb,0xb1,0x8b,
                 0x95,0xa0,0x35,0xc5,0xd8,0x49,0x2d,0x4d,0x35,
                 0x93,0x6b,0x7b,0x36,0x30,0xee,0x20,0xf6,0x25,
                 0xb7,0x0f,0x8e,0x71,0xd9,0xdc,0xd0,0xef,0xd0,
                 0xe3,0x38,0x7d,0x13,0x8c,0x1f,0x5e,0xed,0xce,
                 0x32,0xdd,0x88,0xf2,0x23,0x33,0x4b,0x9a,0x9e,
                 0xab,0x65,0x01,0x7f,0x04,0xaa,0x84,0x42,0x17,
                 0x9f,0x62},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0xca,0x00,0x53,0xd5,0x1f,0x6c,0xf6,0xf9,0x99,
                0x8f,0xf1,0xe0,0xdb,0x00,0xb9,0x0e,0x82,0xc7,
                0xb1,0x8c,0xb5,0x37,0x7a,0xcc,0x8e,0xbe,0x9a,
                0xfe,0x20,0xda,0x1c,0x3d},
    }
};

struct HMAC_TEST_VECTOR fips_sha384_hmac_test_vector[] = {
    {	// 0	(45)
        .key = {0x5e,0xab,0x0d,0xfa,0x27,0x31,0x12,0x60,0xd7,0xbd,
                0xdc,0xf7,0x71,0x12,0xb2,0x3d,0x8b,0x42,0xeb,0x7a,
                0x5d,0x72,0xa5,0xa3,0x18,0xe1,0xba,0x7e,0x79,0x27,
                0xf0,0x07,0x9d,0xbb,0x70,0x13,0x17,0xb8,0x7a,0x33,
                0x40,0xe1,0x56,0xdb,0xce,0xe2,0x8e,0xc3,0xa8,0xd9},
        .key_len = 50,
        .data = {0xf4,0x13,0x80,0x12,0x3c,0xcb,0xec,0x4c,0x52,0x7b,
                 0x42,0x56,0x52,0x64,0x11,0x91,0xe9,0x0a,0x17,0xd4,
                 0x5e,0x2f,0x62,0x06,0xcf,0x01,0xb5,0xed,0xbe,0x93,
                 0x2d,0x41,0xcc,0x8a,0x24,0x05,0xc3,0x19,0x56,0x17,
                 0xda,0x2f,0x42,0x05,0x35,0xee,0xd4,0x22,0xac,0x60,
                 0x40,0xd9,0xcd,0x65,0x31,0x42,0x24,0xf0,0x23,0xf3,
                 0xba,0x73,0x0d,0x19,0xdb,0x98,0x44,0xc7,0x1c,0x32,
                 0x9c,0x8d,0x9d,0x73,0xd0,0x4d,0x8c,0x5f,0x24,0x4a,
                 0xea,0x80,0x48,0x82,0x92,0xdc,0x80,0x3e,0x77,0x24,
                 0x02,0xe7,0x2d,0x2e,0x9f,0x1b,0xab,0xa5,0xa6,0x00,
                 0x4f,0x00,0x06,0xd8,0x22,0xb0,0xb2,0xd6,0x5e,0x9e,
                 0x4a,0x30,0x2d,0xd4,0xf7,0x76,0xb4,0x7a,0x97,0x22,
                 0x50,0x05,0x1a,0x70,0x1f,0xab,0x2b,0x70},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x7c,0xf5,0xa0,0x61,0x56,0xad,0x3d,0xe5,0x40,0x5a,
                0x5d,0x26,0x1d,0xe9,0x02,0x75,0xf9,0xbb,0x36,0xde,
                0x45,0x66,0x7f,0x84,0xd0,0x8f,0xbc,0xb3,0x08,0xca,
                0x8f,0x53,0xa4,0x19,0xb0,0x7d,0xea,0xb3,0xb5,0xf8,
                0xea,0x23,0x1c,0x5b,0x03,0x6f,0x88,0x75},
    }, {	// 1 (46)
        .key = {0xf8,0x69,0x02,0xe5,0xe5,0xdb,0x47,0x8e,0xc6,0xe2,
                0x78,0x69,0x27,0x28,0xa8,0x12,0xc4,0xcd,0x87,0x45,
                0xf9,0x0a,0x7d,0x9f,0x79,0x15,0xf5,0xa9,0x43,0x45,
                0xfc,0x12,0xd2,0x77,0x0a,0x3c,0x94,0xb0,0x1f,0xfb,
                0x9e,0x04,0x12,0x99,0x9e,0xb6,0x26,0x1d,0x11,0xa0},
        .key_len = 50,
        .data = {0xe0,0xbc,0xac,0xbe,0x96,0xda,0xd6,0xf6,0x0e,0x51,
                 0x12,0x9f,0x35,0xac,0xd0,0x3e,0x12,0x27,0x6a,0x91,
                 0xfa,0x13,0xfc,0x15,0x03,0x7c,0x75,0xca,0xbb,0x0a,
                 0xee,0x3a,0x19,0x25,0x3b,0xb8,0xb3,0x5c,0xc0,0xe6,
                 0x32,0x08,0x86,0x7a,0x03,0x2c,0x8f,0x41,0x50,0xa0,
                 0x66,0x64,0x2f,0x6f,0xf9,0xea,0x19,0x7d,0xab,0x7e,
                 0x9d,0x6d,0xa6,0x72,0x55,0xc1,0x6e,0x05,0x1a,0x43,
                 0xbc,0xe1,0x74,0xa4,0x89,0xe8,0x54,0x64,0x69,0x30,
                 0x06,0xf1,0x1a,0x4c,0x61,0x13,0x5d,0xce,0x41,0x87,
                 0x04,0x09,0x37,0xeb,0x4d,0x1c,0x7e,0xda,0x6e,0x2c,
                 0x31,0x57,0x71,0xf0,0xbc,0x6f,0x42,0x73,0x91,0x1a,
                 0x07,0x15,0x1c,0x63,0xaf,0xd3,0xf8,0xc8,0xce,0xc9,
                 0x63,0xe4,0xa8,0xf5,0xef,0x4b,0x8b,0x3e},
        .data_len = 128,
        .mac_len = 48,
        .mac =	{0x4b,0xb4,0xeb,0x2d,0xb2,0xcc,0x92,0x1b,0x15,0x9b,
                 0x78,0xa2,0xbb,0x9e,0xdc,0x16,0x08,0xbb,0x2a,0x1c,
                 0xa9,0x87,0x3b,0x41,0x1a,0xe3,0x0a,0x63,0x38,0x6e,
                 0x46,0x2f,0x9f,0x69,0xd9,0xf5,0xfc,0x83,0x8f,0xf1,
                 0x81,0x87,0x48,0xaa,0xb7, 0x4d,0xa9,0x4f},
    }, {	// 2	(105)
        .key = {0xbf,0xe6,0xbb,0x4c,0x9b,0x17,0x1b,0x93,0xd2,0x8e,
                0x9f,0x8f,0x86,0xb8,0x8d,0xbe,0x50,0x9c,0x66,0xee,
                0xd4,0x18,0x18,0xa1,0x98,0x6d,0x75,0xb6,0x16,0xfe,
                0xe4,0x46,0x0f,0x54,0x56,0xcd,0x23,0x66,0x7c,0x8a,
                0x9f,0x17,0x38,0x28,0x96,0x01,0x51,0x9d,0x33,0x71,
                0x6a,0x53,0x4d,0xb2,0x35},
        .key_len = 55,
        .data = {0x5b,0x7a,0x07,0x8f,0x98,0x0b,0xb8,0x91,0x97,0x43,
                 0xbb,0xce,0x52,0xfd,0x0b,0xa3,0xc2,0x20,0x83,0xd2,
                 0xb0,0x25,0x4e,0x28,0xc8,0xd3,0xa0,0x5d,0xef,0x4d,
                 0xa3,0x3b,0xd6,0x4f,0xb5,0x02,0xcf,0xb5,0xd0,0x0c,
                 0xe0,0x3d,0x49,0xad,0x16,0x8d,0xbe,0x5d,0x1c,0x78,
                 0x4a,0x19,0x0c,0x7d,0xfa,0x06,0x85,0x90,0x85,0x58,
                 0xfe,0x1e,0x37,0x72,0x5a,0x4b,0x2f,0x4e,0xbc,0x7e,
                 0xca,0x20,0x9c,0x1f,0x5f,0x36,0x1b,0x9f,0x2d,0x23,
                 0x93,0xb9,0x91,0x1c,0x73,0xf8,0x7d,0xa2,0x4a,0x7a,
                 0x25,0x62,0x21,0xf3,0xfb,0x59,0x0e,0xf4,0xde,0x3b,
                 0x06,0x6e,0x8e,0x16,0xf3,0x72,0x64,0x32,0x06,0x3a,
                 0x40,0x3d,0x4f,0x6d,0xc2,0xa4,0x8b,0x9f,0xbd,0x44,
                 0x3d,0x17,0xe8,0x42,0x00,0xd6,0xd7,0x37},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0xe8,0x2e,0xeb,0x7f,0x4b,0x74,0x15,0xa4,0xc9,0x5d,
                0xc8,0x2c,0x46,0xbb,0x59,0x71,0x5f,0xda,0x4e,0x0b,
                0xda,0xf6,0x4a,0x7f,0xb3,0xaf,0x3c,0x70,0x58,0xec,
                0x7d,0x2a,0x17,0x2b,0x82,0x93,0x05,0x7b,0x72,0xf9,
                0x66,0x44,0x54,0xe7,0xde,0xe1,0x1d,0x95},
    }, {	// 3	(106)
        .key = {0x4c,0xf5,0x4e,0xb8,0xcf,0x7b,0xd4,0x21,0xdd,0xb0,
                0x58,0x6a,0xc4,0xfa,0xb9,0xc4,0x78,0xcd,0xae,0xdd,
                0x89,0xcc,0x5a,0x19,0x53,0x32,0x21,0x1f,0x75,0x71,
                0xb9,0x98,0x84,0x19,0x84,0x33,0x00,0xfa,0x1d,0xed,
                0x86,0x8d,0x31,0x8f,0x48,0x90,0x90,0x78,0xbb,0xf1,
                0x83,0x9c,0x8f,0xed,0x61},
        .key_len = 55,
        .data = {0xd2,0x2f,0x19,0x4a,0x1a,0xf3,0x3c,0xd8,0xcd,0xff,
                 0xe9,0x96,0x7f,0x67,0x7a,0xcb,0x68,0x50,0x0d,0x6c,
                 0xbb,0xf7,0x7a,0x3f,0x34,0xf5,0x88,0x40,0xf0,0xc1,
                 0x60,0x44,0x82,0x76,0x41,0xdc,0x43,0xd6,0x76,0x7c,
                 0xe9,0x8f,0x85,0xdd,0x5c,0xbe,0xaa,0x9f,0xc5,0xb2,
                 0x83,0x33,0xe7,0xf2,0x0d,0xf8,0xb2,0x81,0xcf,0xa4,
                 0x15,0x60,0x55,0xe6,0x15,0x55,0xe0,0x4a,0x1c,0xeb,
                 0x5c,0x5c,0x93,0xba,0x92,0x10,0xb2,0xe8,0x9f,0x61,
                 0x97,0xf0,0xa5,0x39,0x96,0xa2,0xc0,0x91,0xd1,0x6c,
                 0x3c,0xd9,0x08,0xd7,0x05,0x9a,0xb2,0x54,0x5e,0x5a,
                 0x4c,0x39,0xd6,0xc0,0xf1,0x07,0x78,0xf8,0x2b,0xee,
                 0x43,0x59,0x09,0x93,0xda,0x45,0x71,0x10,0x7c,0x51,
                 0xb8,0x3c,0x35,0xa6,0x70,0x2e,0x56,0xa8},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x83,0x0b,0x4a,0x79,0x8f,0x85,0xc4,0x48,0xb3,0xd5,
                0x4a,0xbf,0xee,0x61,0xb3,0x76,0x59,0x7f,0x65,0x66,
                0x6d,0x83,0xa2,0x10,0x52,0xcb,0x3f,0x44,0x66,0xf4,
                0x47,0x47,0x43,0x16,0x07,0xbc,0x65,0x9c,0x91,0xcb,
                0x52,0x03,0x08,0xfb,0xf4,0xfc,0xdb,0x58},
    }, {	// 4	(165)
        .key = {0xb6,0x19,0xd9,0xd0,0x74,0x61,0xc1,0x1b,0xc9,0xfb,
                0x66,0x11,0x7d,0x61,0xed,0x90,0x00,0x13,0x66,0xbb,
                0xff,0xdb,0xff,0x58,0x35,0x56,0x77,0x75,0x84,0xb0,
                0xd6,0x52,0x44,0xaf,0x5c,0x7b,0xdb,0xf3,0xb7,0x35,
                0x8d,0x7c,0x79,0x1b,0x96,0x6c,0xc8,0x09,0x76,0x0e,
                0x57,0x39,0x8d,0x18,0x96,0xac,0xe7,0x2d,0x26,0xcc,
                0x59,0xa6,0x90,0x4f,0xcd,0x92,0x36,0x5e,0xda,0xfb,
                0x8a,0xf7,0x98,0x6c,0x7d,0x90,0xb2,0xaf,0x3b,0xfd,
                0xbc,0xdb,0x15,0x93,0xc7,0x8f,0xbe,0x8e,0x33,0x78,
                0xbb,0xb0,0xc5,0x19,0x15,0x2b,0xf9,0xcb,0x51,0xc1,
                0x9a,0x02,0xa1,0x2a,0x8f,0xd3,0x5c,0xb6,0xf8,0xb3,
                0xac,0x33,0x7a,0x82,0x87,0x11,0xd6,0xc8,0xe0,0xc4,
                0xc1,0x3e,0x1e,0x6a,0xf0,0x90,0xcd,0xae},
        .key_len = 128,
        .data = {0x5a,0x81,0xe7,0x11,0xad,0xfe,0x50,0x77,0xdd,0x8c,
                 0x8b,0x57,0xc9,0x5e,0x8e,0x1f,0x3d,0xe3,0x9f,0x4f,
                 0xc4,0x48,0xc5,0x23,0xbd,0x3e,0x7c,0x72,0xb1,0xfd,
                 0xac,0xd6,0xe4,0x89,0xdc,0x0d,0x2a,0x34,0xa3,0x9f,
                 0xfc,0x64,0x60,0xc1,0xcb,0x96,0x2b,0x7a,0x94,0xa3,
                 0x0c,0x04,0xb5,0x42,0x6a,0x75,0xff,0xcf,0xc6,0x9f,
                 0x0c,0x4b,0xa9,0x34,0xd3,0xa3,0xda,0x2e,0x79,0x35,
                 0xd5,0x6d,0x6b,0x90,0x79,0xa2,0xa9,0x7b,0x01,0x6d,
                 0x65,0x3a,0x35,0xc2,0xcc,0x0c,0xe1,0x91,0x24,0xf8,
                 0x87,0xa6,0x17,0xc9,0x51,0xce,0x4e,0x58,0x49,0x3b,
                 0x42,0x09,0xcc,0x29,0x4f,0x98,0x3c,0xc2,0x0b,0x16,
                 0xf6,0x3f,0xd5,0x2e,0x84,0x51,0xb1,0xad,0x13,0xbf,
                 0x53,0x42,0x27,0x50,0x79,0x81,0x8d,0xeb},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x8b,0x42,0x58,0xbe,0x4c,0x09,0x4a,0xa4,0x05,0x6f,
                0x33,0x2e,0xde,0x8c,0x73,0x37,0x72,0x66,0x4b,0x08,
                0x8b,0xa2,0x2e,0xf8,0xca,0xae,0x7c,0xef,0xd7,0x7e,
                0xce,0xb3,0x5e,0x83,0xaf,0x8d,0x9c,0x12,0x83,0xcb,
                0xbf,0xfe,0x4a,0x37,0x2b,0x69,0x9c,0x21},
    }, {	// 5	(166)
        .key = {0xe4,0x88,0x25,0xa5,0x50,0x3a,0x6a,0xfe,0x0b,0xf9,
                0xa2,0x40,0xc6,0x7f,0x27,0xac,0xd4,0xa8,0xf6,0x99,
                0x38,0x34,0x64,0x5e,0x03,0xc8,0x0c,0x72,0xdd,0x37,
                0x0c,0xd2,0xe1,0x00,0x71,0xa3,0xae,0x18,0xef,0x19,
                0xba,0xe9,0xd6,0x97,0xea,0x9a,0x41,0x18,0x60,0x91,
                0x90,0xcd,0x95,0x36,0x19,0x07,0xa7,0xfa,0x1b,0x58,
                0xf4,0x99,0xf3,0xf5,0xe7,0x9b,0x93,0x5f,0x12,0x21,
                0x2f,0x43,0x7d,0xde,0x39,0x9e,0x3e,0x64,0x90,0x24,
                0x4a,0xa1,0xf5,0xe3,0x8b,0xa9,0xbe,0x24,0x33,0xb6,
                0xce,0x92,0x4f,0x6c,0xc4,0x9e,0x9f,0x62,0x73,0x21,
                0xa5,0xdf,0x93,0x43,0xfc,0xe1,0xb5,0x9d,0xeb,0x64,
                0x7d,0x9a,0x3a,0xe0,0x0b,0x23,0x44,0x14,0xba,0x7b,
                0x4e,0x02,0x0d,0x67,0x17,0x3b,0xe6,0x93},
        .key_len = 128,
        .data = {0x85,0x61,0x86,0x5a,0xc2,0xce,0x12,0x83,0x27,0x46,
                 0xf8,0x25,0x84,0xa4,0xb9,0x8e,0x7f,0x4c,0x3a,0xe2,
                 0x41,0x0e,0x18,0x19,0x6f,0x4e,0x3b,0x47,0x5c,0x62,
                 0xae,0x20,0x7d,0x3c,0xad,0xbb,0x1d,0x49,0x00,0x96,
                 0x51,0x98,0x88,0xdb,0x2f,0x3f,0x18,0xe1,0x3b,0xfb,
                 0x86,0xf6,0x22,0x16,0x01,0x5c,0xab,0x8e,0xa4,0x91,
                 0xea,0x73,0x4c,0xd3,0xb7,0x91,0xa7,0xe4,0x5e,0x4f,
                 0x8e,0x0b,0x98,0xd7,0x95,0x5b,0xba,0x77,0xe0,0x37,
                 0x2d,0x47,0x38,0x16,0x1e,0x0d,0x5d,0x84,0x76,0x5d,
                 0x9e,0x6a,0x0d,0x05,0xa8,0x8e,0x1a,0xa8,0x9c,0x5d,
                 0xef,0xa8,0x64,0xe9,0xe3,0x49,0x46,0x2e,0x8f,0x14,
                 0xb9,0x99,0x3d,0x7a,0x78,0xcb,0x9d,0xba,0xd6,0x9a,
                 0xba,0x05,0x51,0x58,0x2d,0xdf,0x69,0x58},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0xec,0x78,0x0a,0x91,0x5e,0xc7,0xde,0xeb,0xa2,0xc8,
                0xc9,0xe2,0xab,0x15,0xc9,0x76,0x2a,0x3e,0xb1,0x8f,
                0xaf,0xa2,0xd4,0x8a,0x55,0x4a,0xe1,0xfe,0x6c,0x44,
                0x59,0xda,0x1a,0x54,0xe2,0xd5,0x8b,0xdf,0x06,0xfe,
                0xa0,0x74,0x00,0x98,0xee,0xbb,0xb6,0x99},
    }, {	// 6	(225)
        .key = {0x74,0xf4,0x1a,0x6b,0x1c,0x4e,0x57,0x13,0x49,0x95,
                0x57,0xd6,0xf7,0xe8,0x89,0xf8,0xa8,0xce,0x2e,0x44,
                0x4e,0x82,0x61,0xfe,0x6a,0x8e,0x55,0x18,0x76,0x9b,
                0xdf,0xa8,0x81,0x88,0x34,0x9a,0x19,0xb9,0xf3,0xa2,
                0x6d,0xb2,0x66,0x75,0xb3,0xe4,0x05,0x39,0xc8,0xc6,
                0x3b,0x3a,0x16,0x28,0x6d,0xde,0xbb,0xc5,0x39,0xdb,
                0xe8,0x17,0xfb,0xa7,0x86,0x6f,0x96,0x31,0x20,0x44,
                0x71,0xce,0xfd,0xcb,0xbf,0x76,0x8c,0xc9,0x04,0x30,
                0x06,0xa6,0xd4,0xcb,0x4e,0xc2,0xde,0xcf,0x1c,0x0c,
                0x2a,0xb3,0x5a,0xd0,0x9f,0x50,0xce,0xd0,0xc8,0x96,
                0xfa,0x97,0xd8,0x7e,0x40,0x0a,0xeb,0x3f,0x4a,0x40,
                0x8e,0xc5,0xa9,0x93,0x82,0x5f,0xbc,0xf7,0xbd,0xb8,
                0xd4,0x8b,0xb2,0x08,0x95,0x6e,0xd2,0x8b,0xa0,0xd4},
        .key_len = 130,
        .data = {0x9a,0x12,0x14,0x82,0xc7,0x77,0x5a,0x8b,0x5f,0xda,
                 0xf1,0xc2,0xfb,0x7d,0xe1,0xa8,0x6e,0xf9,0x31,0xb1,
                 0xa8,0x8c,0xf2,0x3d,0xdb,0xb4,0x7f,0xc9,0xdc,0xfd,
                 0x02,0x67,0xcb,0x17,0x3a,0x6b,0xf6,0x2b,0x7c,0x68,
                 0xfb,0x6f,0xf8,0x5b,0x2d,0xf9,0x3e,0x25,0x39,0xd1,
                 0x01,0x3f,0x0a,0x49,0x1a,0xa9,0xe9,0x91,0xcf,0x23,
                 0xe9,0x86,0x56,0xa0,0x82,0xcb,0x95,0xf8,0x7c,0x1b,
                 0x2c,0xdd,0x0e,0xdd,0xb5,0x10,0x48,0xf9,0x4a,0xd4,
                 0xae,0xeb,0x48,0xa4,0x26,0x16,0x53,0x21,0x14,0x5a,
                 0x9b,0x4e,0xc3,0xe8,0x5d,0xff,0x07,0x55,0xac,0x8f,
                 0x20,0xee,0x71,0xd2,0xe2,0x4c,0xb1,0x4a,0x13,0x28,
                 0x0e,0x9e,0x15,0x70,0x91,0x47,0xc4,0x99,0xa6,0x8d,
                 0xa2,0x38,0x68,0xb2,0x32,0xcc,0x1f,0x6d},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0xb0,0xda,0x90,0xc0,0x43,0x49,0x35,0x11,0xd9,0x4f,
                0x22,0xfa,0xc3,0x5b,0x59,0x62,0x74,0x9c,0x49,0x97,
                0x2f,0xb4,0x35,0x71,0xb8,0x47,0x87,0x64,0xdf,0xfc,
                0x1c,0x25,0xe3,0xa7,0x52,0x3f,0xd4,0x05,0x33,0x8a,
                0x04,0x8d,0x38,0xdd,0x1b,0x75,0x51,0x1d},
    }, {	// 7	(226)
        .key = {0xd8,0x7f,0xb6,0xba,0x27,0x21,0x5e,0x5c,0xb6,0x5c,
                0x3b,0x5b,0x34,0xac,0x2a,0x32,0x03,0x7f,0x30,0xe1,
                0xf7,0xea,0x60,0x3d,0x5a,0x9b,0xff,0x8a,0x33,0x0f,
                0xe7,0x4b,0xc7,0x05,0x29,0x59,0x61,0x32,0xf6,0x33,
                0x4f,0x36,0xc0,0x95,0x2d,0xcf,0x9c,0x4c,0x66,0x4c,
                0xeb,0x48,0xf7,0x45,0x39,0xf3,0x76,0x8a,0x65,0xc1,
                0x53,0x59,0x02,0x08,0x5f,0xd4,0xfe,0x13,0x8a,0xb1,
                0x81,0x72,0xf1,0x34,0x18,0x93,0x18,0x5a,0x13,0x97,
                0x73,0x58,0x2c,0x5e,0x2c,0x43,0x69,0xe4,0x20,0x11,
                0x43,0xd1,0x2b,0xc0,0x07,0x4b,0xa5,0xd5,0x7d,0x0f,
                0x2c,0x08,0xc8,0xc0,0xa4,0x3e,0x8d,0x7e,0x7d,0xb7,
                0x57,0xbb,0x34,0x89,0x3a,0x4a,0x1d,0x4d,0xb7,0xb9,
                0x5f,0x18,0xe0,0xe1,0x40,0xad,0xbc,0xbb,0xa3,0xf0},
        .key_len = 130,
        .data = {0x9e,0x1a,0x5d,0x9f,0x23,0x6e,0xf9,0x3f,0x2c,0xda,
                 0x60,0x48,0x91,0x66,0xc8,0x2d,0xce,0x32,0x23,0x27,
                 0x04,0x66,0x44,0xcc,0x40,0x6b,0x42,0xe3,0x00,0x5c,
                 0x21,0x77,0xf3,0xb7,0xaf,0x2a,0x01,0x59,0xad,0xcc,
                 0x8b,0xa9,0x2f,0x2c,0xf4,0x13,0x46,0x2e,0x60,0xb8,
                 0xdb,0x1e,0xbb,0x63,0xde,0x44,0xfe,0xbf,0xa1,0xb9,
                 0xad,0xc8,0x7e,0x79,0xa4,0x80,0xc0,0xb8,0x14,0xe3,
                 0xc1,0x7a,0xc9,0x1c,0x4f,0x5e,0xae,0xf9,0x54,0xba,
                 0x92,0x9d,0xb6,0xed,0x2c,0x75,0x7d,0xf1,0x5d,0x6d,
                 0x34,0x30,0xb6,0x63,0x91,0x99,0x3a,0xdb,0x58,0xf2,
                 0x65,0xf5,0x7c,0x70,0x6d,0x9d,0x87,0x85,0xc7,0x02,
                 0x3d,0xf9,0xed,0x49,0x7c,0x3c,0x5f,0x82,0x67,0xfb,
                 0xe7,0xdb,0xc4,0xf1,0x22,0x13,0xa1,0x00},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x3c,0x7c,0xee,0x96,0x02,0x21,0xc9,0xd9,0xf7,0x46,
                0x4a,0xeb,0x70,0xd1,0x98,0xbd,0x60,0x41,0x4d,0xc3,
                0xff,0xbf,0xa7,0xa2,0x22,0x7a,0x3a,0x37,0x5e,0xbb,
                0x8f,0x64,0x48,0xe5,0x24,0x70,0x6e,0x1e,0x3a,0xe9,
                0x55,0x41,0xbd,0xce,0xf2,0xb3,0x1d,0x9f},
    }, {	// 8	(285)
        .key = {0xf0,0x5d,0x56,0x1f,0x5a,0xd7,0x04,0x03,0x26,0x4c,
                0x5e,0x0a,0x0e,0xdc,0x12,0xfd,0x47,0x3b,0x19,0xc0,
                0xb4,0x0f,0x8c,0xd8,0x5a,0x99,0xba,0x2a,0x14,0x98,
                0x77,0x05,0x87,0x6a,0xb7,0x63,0x59,0x75,0x5b,0x6c,
                0x9e,0xc5,0x4a,0x3c,0x93,0xf6,0xc4,0xe6,0x8f,0x55,
                0xf3,0xb9,0x36,0x42,0xc3,0xc2,0xf0,0xd9,0xf4,0x91,
                0x9a,0xd1,0x6e,0x40,0x7b,0xa3,0xd4,0xb2,0x79,0xef,
                0x5b,0x19,0x8c,0x1c,0xdd,0xbb,0x74,0x40,0x29,0xf5,
                0xa7,0x3f,0x9e,0x80,0x8e,0x36,0xf8,0xf3,0xf0,0x1a,
                0x69,0x89,0xaf,0x9c,0xec,0x25,0xb2,0x50,0xd6,0x93,
                0x22,0x0f,0xad,0x11,0xd9,0x9a,0x3e,0x0e,0x17,0x7f,
                0xea,0x31,0x77,0x41,0x41,0x9d,0x22,0xb3,0xd2,0x74,
                0x43,0xa5,0x40,0x99,0xbb,0xc2,0x99,0xbb,0x15,0xb9,
                0xe4,0x8f,0xbc,0x9b,0xf9,0x5c,0x6b,0x84,0x96,0xbd,
                0xe6,0x7e,0xae,0xa3,0xe8},
        .key_len = 145,
        .data = {0x77,0xd3,0xf3,0xe6,0x47,0xe6,0x77,0x66,0xe5,0xf4,
                 0xcf,0x1b,0xce,0x5f,0x63,0x1b,0xd5,0x75,0xdd,0xbd,
                 0x02,0xf2,0x96,0x43,0xa0,0xc6,0x4d,0xbd,0x92,0x19,
                 0x1f,0x2a,0xe6,0x8d,0xb3,0xdf,0xad,0xc3,0xb6,0x2d,
                 0x09,0x20,0x87,0x3e,0x87,0xd1,0x33,0x40,0xaf,0x0c,
                 0xa3,0xc5,0xda,0x99,0x14,0x6a,0x44,0x92,0xc8,0xb7,
                 0x62,0x67,0xfb,0x47,0x76,0x24,0x19,0x29,0x60,0xf7,
                 0x2e,0x85,0xb7,0xed,0x9e,0x83,0x18,0xfc,0x16,0x68,
                 0xbe,0x46,0xc2,0x03,0x53,0x9c,0xc1,0x47,0x06,0x41,
                 0xd6,0x39,0xde,0xf1,0x60,0x0d,0x4e,0x22,0x8c,0x8b,
                 0x09,0x8a,0xc9,0xb8,0x17,0xe1,0x7c,0xb3,0x29,0xe8,
                 0xf5,0xdd,0x2a,0xaa,0xa2,0x3c,0x16,0x02,0x83,0x22,
                 0x0f,0x5d,0xde,0x09,0xae,0xc1,0x34,0xc2},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x72,0x4e,0x5d,0x2d,0x51,0xd9,0x8c,0x15,0xce,0x2e,
                0x78,0xf8,0x61,0xd7,0xb6,0xf8,0x95,0x28,0x82,0xe9,
                0xd9,0x3d,0x40,0x85,0x0b,0x78,0xa2,0x3e,0x63,0x2c,
                0x4e,0x14,0xa2,0x22,0xab,0x37,0x26,0xb1,0xa0,0xaa,
                0x7c,0x6b,0x2c,0xd6,0x60,0x82,0xed,0x95},
    }, {	// 9	(286)
        .key = {0x95,0xec,0xe1,0xc8,0xae,0x5e,0x94,0xd1,0x6e,0xc9,
                0x98,0x3b,0x10,0x89,0xa3,0x73,0x95,0xad,0x5b,0x1d,
                0x66,0x09,0x16,0xc1,0x3c,0x87,0xe4,0xc1,0x3d,0xbe,
                0xcf,0x8f,0x68,0xc6,0x61,0x1c,0x32,0x4a,0x67,0x94,
                0x71,0xde,0xf5,0x48,0x7a,0x93,0xaa,0xec,0x86,0xc9,
                0x35,0x02,0x5b,0x45,0x18,0x96,0x28,0x84,0xac,0x2c,
                0xb0,0x4e,0x66,0xf7,0xaa,0x8e,0x58,0x4b,0x68,0x60,
                0xfb,0x55,0xb8,0x6c,0x2b,0x0a,0x08,0x73,0x73,0x5d,
                0xcd,0x27,0x8b,0xb5,0x25,0x40,0x1f,0x9e,0xba,0xcc,
                0xd2,0xbe,0xea,0xc6,0x83,0x0c,0x26,0xeb,0xcf,0x3c,
                0x98,0xc9,0xd7,0x7d,0x09,0x19,0x43,0x67,0x01,0x4e,
                0x87,0x2f,0x30,0x6e,0x64,0x1e,0x0c,0x21,0xb2,0x41,
                0xbc,0x08,0x5e,0x61,0x35,0x4f,0xaf,0x35,0xa3,0x86,
                0xcd,0xd7,0x0a,0xac,0x83,0x75,0x2d,0x8d,0x44,0x49,
                0xaf,0x4f,0x6c,0xcb,0x78},
        .key_len = 145,
        .data = {0x18,0xdb,0xab,0x9f,0x86,0xb9,0xd7,0x0b,0xbd,0xeb,
                 0x01,0x8f,0x6a,0x76,0xea,0x7a,0xf2,0x3e,0xb2,0xff,
                 0x11,0x1e,0x9b,0xe3,0xc1,0x38,0x11,0x79,0x5d,0x8a,
                 0xe7,0xd0,0x06,0xc3,0xe4,0x2b,0x46,0x54,0x7e,0xb1,
                 0xf3,0xc9,0xe5,0x66,0x56,0x5a,0x43,0x5a,0x8d,0xbd,
                 0x42,0x21,0x2e,0x3f,0xd0,0x82,0x2d,0x13,0x1f,0x73,
                 0x00,0xea,0xef,0x46,0x00,0xc4,0x0f,0x1d,0x13,0x05,
                 0x21,0xa3,0x88,0xcb,0x9f,0xfe,0x42,0x7f,0x1b,0xff,
                 0x19,0xaa,0xcb,0x9c,0x7d,0x0a,0x44,0xa1,0x5c,0xe6,
                 0x86,0xa2,0x46,0x9e,0x39,0x34,0xd0,0x86,0x36,0x5d,
                 0x36,0xf4,0x49,0x48,0x44,0x98,0x35,0x3d,0x76,0x0c,
                 0xf9,0xd1,0x5e,0xac,0x52,0x5a,0x46,0xa8,0x81,0xa6,
                 0x17,0x58,0x4e,0xed,0x79,0xcf,0x4d,0x03},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x2b,0xe1,0xbd,0x6a,0x76,0x6e,0x30,0x79,0x21,0x54,
                0xcd,0xa0,0x0a,0xf9,0x7c,0xc5,0x12,0xe8,0x14,0x13,
                0xe0,0xfb,0x76,0x16,0x98,0xf3,0x9a,0x26,0xce,0xcc,
                0x3f,0xac,0xe6,0xf9,0xa9,0x8b,0x7c,0x49,0x60,0x51,
                0x26,0xdf,0xa5,0xaa,0x8d,0xe1,0xad,0x72},
    }
};

struct HMAC_TEST_VECTOR fips_sha512_hmac_test_vector[] = {
    {	// 0	(60)
        .key = {0x57,0xc2,0xeb,0x67,0x7b,0x50,0x93,0xb9,0xe8,0x29,
                0xea,0x4b,0xab,0xb5,0x0b,0xde,0x55,0xd0,0xad,0x59,
                0xfe,0xc3,0x4a,0x61,0x89,0x73,0x80,0x2b,0x2a,0xd9,
                0xb7,0x8e,0x26,0xb2,0x04,0x5d,0xda,0x78,0x4d,0xf3,
                0xff,0x90,0xae,0x0f,0x2c,0xc5,0x1c,0xe3,0x9c,0xf5,
                0x48,0x67,0x32,0x0a,0xc6,0xf3,0xba,0x2c,0x6f,0x0d,
                0x72,0x36,0x04,0x80,0xc9,0x66,0x14,0xae,0x66,0x58,
                0x1f,0x26,0x6c,0x35,0xfb,0x79,0xfd,0x28,0x77,0x4a,
                0xfd,0x11,0x3f,0xa5,0x18,0x7e,0xff,0x92,0x06,0xd7,
                0xcb,0xe9,0x0d,0xd8,0xbf,0x67,0xc8,0x44,0xe2,0x02},
        .key_len = 100,
        .data =	{0x24,0x23,0xdf,0xf4,0x8b,0x31,0x2b,0xe8,0x64,0xcb,
                 0x34,0x90,0x64,0x1f,0x79,0x3d,0x2b,0x9f,0xb6,0x8a,
                 0x77,0x63,0xb8,0xe2,0x98,0xc8,0x6f,0x42,0x24,0x5e,
                 0x45,0x40,0xeb,0x01,0xae,0x4d,0x2d,0x45,0x00,0x37,
                 0x0b,0x18,0x86,0xf2,0x3c,0xa2,0xcf,0x97,0x01,0x70,
                 0x4c,0xad,0x5b,0xd2,0x1b,0xa8,0x7b,0x81,0x1d,0xaf,
                 0x7a,0x85,0x4e,0xa2,0x4a,0x56,0x56,0x5c,0xed,0x42,
                 0x5b,0x35,0xe4,0x0e,0x1a,0xcb,0xeb,0xe0,0x36,0x03,
                 0xe3,0x5d,0xcf,0x4a,0x10,0x0e,0x57,0x21,0x84,0x08,
                 0xa1,0xd8,0xdb,0xcc,0x3b,0x99,0x29,0x6c,0xfe,0xa9,
                 0x31,0xef,0xe3,0xeb,0xd8,0xf7,0x19,0xa6,0xd9,0xa1,
                 0x54,0x87,0xb9,0xad,0x67,0xea,0xfe,0xdf,0x15,0x55,
                 0x9c,0xa4,0x24,0x45,0xb0,0xf9,0xb4,0x2e},
        .data_len = 128,
        .mac = {0x33,0xc5,0x11,0xe9,0xbc,0x23,0x07,0xc6,0x27,0x58,
                0xdf,0x61,0x12,0x5a,0x98,0x0e,0xe6,0x4c,0xef,0xeb,
                0xd9,0x09,0x31,0xcb,0x91,0xc1,0x37,0x42,0xd4,0x71,
                0x4c,0x06,0xde,0x40,0x03,0xfa,0xf3,0xc4,0x1c,0x06,
                0xae,0xfc,0x63,0x8a,0xd4,0x7b,0x21,0x90,0x6e,0x6b,
                0x10,0x48,0x16,0xb7,0x2d,0xe6,0x26,0x9e,0x04,0x5a,
                0x1f,0x44,0x29,0xd4},
        .mac_len = 64,
    }, {	// 1	(61)
        .key = {0x7c,0x98,0x91,0x2c,0x74,0x42,0x13,0x62,0xe1,0x12,
                0xa2,0xf9,0x8f,0xed,0x9b,0xab,0xe0,0x05,0x7f,0xc7,
                0x78,0xb4,0x45,0x32,0x39,0xaa,0xf5,0xac,0x72,0x4b,
                0x72,0x55,0x53,0x53,0x97,0x70,0xa5,0xbc,0x86,0x66,
                0xb8,0xe1,0x3d,0x0e,0x9c,0xe3,0x6b,0x2b,0x93,0x4c,
                0x81,0x37,0xc7,0xf2,0x0b,0x5f,0x39,0x1f,0x41,0xce,
                0xfa,0xee,0xd9,0x2e,0x9d,0xf8,0x20,0x6c,0xec,0x30,
                0x49,0xbc,0xda,0x0c,0x05,0xde,0xb9,0xe6,0x54,0x9f,
                0xad,0xa1,0x9a,0xa2,0x61,0x8f,0xf5,0x60,0xf8,0x92,
                0xce,0x6e,0x47,0x82,0xae,0xff,0x41,0xcf,0x53,0xa9},
        .key_len = 100,
        .data = {0x74,0xe8,0x93,0x6d,0x83,0xbf,0x3f,0x16,0xb8,0xd0,
                 0x3f,0xb7,0x33,0x84,0xed,0x8f,0x46,0xbd,0x32,0x34,
                 0x3f,0x5d,0xf8,0x35,0x81,0x07,0xe2,0xfd,0xda,0x29,
                 0x3a,0xfa,0x10,0x3a,0x2b,0xff,0xbd,0x40,0x30,0xe7,
                 0x5d,0x96,0xcc,0x7c,0xa6,0xec,0x7c,0x97,0x18,0x8f,
                 0xea,0x88,0xd4,0xeb,0x63,0xb7,0xb1,0x4e,0x8b,0x8c,
                 0x8d,0xee,0x4f,0x8d,0xe1,0x2e,0x1c,0xc6,0x98,0x1d,
                 0x4e,0x6e,0x22,0x3f,0xec,0xc7,0xc4,0x91,0x92,0x46,
                 0x32,0xc7,0xae,0xf4,0x5f,0xd8,0xef,0x14,0x94,0xbc,
                 0xfb,0x06,0xc0,0x74,0x61,0x6b,0x0f,0x4c,0xce,0x8a,
                 0xbd,0x5d,0x83,0xf3,0x2d,0x55,0x06,0x61,0x35,0x7b,
                 0x18,0xe5,0xbc,0xed,0xe8,0x41,0x88,0x2c,0x86,0x92,
                 0x51,0xdb,0x9a,0x33,0x1a,0xc4,0x56,0xdd},
        .data_len = 128,
        .mac = {0x4c,0xc2,0x88,0x18,0x48,0x6b,0xb9,0xb1,0xb5,0x2e,
                0x33,0x3d,0xde,0x71,0xf7,0x3a,0xcc,0x22,0x74,0x88,
                0x45,0x3f,0xd9,0x07,0xc6,0xb5,0x1d,0x34,0x9d,0x67,
                0xaf,0x1d,0xf2,0x9a,0x9f,0x22,0x55,0x32,0xce,0x04,
                0xf5,0x03,0x95,0xfe,0xd5,0x65,0xe9,0x8d,0x78,0x97,
                0x86,0x26,0xdf,0x93,0x46,0x2d,0x3f,0x01,0x2f,0x73,
                0x73,0x34,0x72,0x98},
        .mac_len = 64,
    }, {	// 2	(135)
        .key = {0x13,0xfb,0x1e,0xd6,0x38,0x9f,0x32,0xd1,0xde,0x31,
                0x39,0xcb,0x04,0xbc,0xdd,0x53,0x52,0x5c,0x98,0x89,
                0xb8,0x53,0x79,0xd3,0x53,0x5a,0x25,0xd2,0x90,0x35,
                0x1c,0x95,0x93,0x8a,0x3d,0x0c,0xda,0xf3,0x8d,0xbf,
                0x1d,0x52,0x34,0xbf,0x79,0x65,0xc8,0xdd,0xce,0x9a,
                0xce,0x1b,0x66,0x24,0x7e,0x60,0xd7,0x4e,0xc7,0x70,
                0x2a,0x0f,0x93,0x1a,0x3c,0xdf,0x4c,0xb4,0x65,0xca,
                0x9f,0xc4,0x58,0xc3,0x80,0x00,0x4a,0x3a,0x6e,0x79,
                0x57,0xf1,0xf8,0x13,0x21,0x0b,0x80,0x38,0xba,0x66,
                0x3f,0xcd,0xc4,0x2a,0x89,0x65,0xd6,0xa2,0x52,0xb5,
                0x22,0x4b,0xf2,0x49,0x55,0x2b,0x25,0x75,0xbf,0x64,
                0x56,0x8d,0xb4,0x09,0x1d,0x58,0x32,0x30,0x06,0xc3,
                0xc3,0x49,0x94,0xd3,0xa5},
        .key_len = 125,
        .data = {0x88,0xad,0x81,0x2f,0xd3,0x4e,0x55,0xc8,0x09,0xe8,
                 0x17,0x19,0x96,0x04,0xb4,0xa7,0xf7,0xfe,0xae,0x42,
                 0xcd,0xc4,0xc9,0xe9,0x30,0xdb,0x08,0xe8,0x45,0xa3,
                 0xd7,0x43,0x13,0xdb,0x8a,0x57,0x92,0x67,0x06,0xbf,
                 0x05,0x51,0xbe,0x75,0x8a,0x0f,0xe2,0x39,0xf0,0x04,
                 0xd2,0x37,0xc8,0x49,0xd9,0xe4,0xbf,0xac,0x18,0x29,
                 0x2b,0xf9,0xc0,0xc3,0xe3,0x79,0x85,0xea,0x54,0xb9,
                 0x4f,0x30,0xd1,0x8c,0x32,0xad,0x2b,0x53,0xa0,0x59,
                 0x82,0x7c,0xdd,0xb9,0x5a,0x49,0xb4,0xbe,0xf1,0xd3,
                 0x69,0xea,0xd1,0x4e,0xee,0xb4,0xa1,0x8e,0x59,0x2e,
                 0x40,0xca,0x96,0xe5,0x15,0xa1,0x59,0x08,0xa0,0x5a,
                 0x57,0xcd,0x55,0x70,0xb6,0x11,0xab,0x4e,0xc2,0x3f,
                 0x70,0x57,0xe1,0x72,0x5f,0x29,0xc9,0xde},
        .data_len = 128,
        .mac = {0xa4,0x81,0xe7,0x13,0xcd,0xc8,0x1c,0xa5,0xaf,0xa0,
                0xef,0xcb,0x16,0xe3,0x5c,0xd2,0x0d,0x01,0xaa,0x44,
                0x99,0x58,0xfd,0x2e,0xae,0xde,0x2e,0x25,0xa5,0xba,
                0x54,0x0b,0xea,0xfb,0xa2,0xfa,0xb4,0xad,0xfe,0xf2,
                0xe1,0x46,0xb4,0xc1,0xb2,0xa1,0x83,0x2e,0x93,0xdd,
                0x37,0x3d,0x63,0xfa,0x90,0xbb,0x61,0x49,0x0f,0x65,
                0x68,0x19,0x1f,0x65},
        .mac_len = 64,
    }, {	// 3	(136)
        .key = {0xfd,0x50,0x70,0x36,0x22,0x96,0xc4,0x0d,0x65,0xb1,
                0x05,0xd5,0xab,0x46,0x53,0xfe,0x34,0xe0,0x20,0x05,
                0x16,0x93,0x3f,0x3e,0xea,0xe0,0x3e,0xd0,0xc5,0xd9,
                0xf6,0x01,0x6a,0x85,0x60,0xb4,0xbd,0x86,0xab,0x2f,
                0x7b,0xf9,0x8b,0x22,0x29,0x9e,0xd3,0xe5,0x4a,0x39,
                0x46,0x02,0xd5,0x38,0xaa,0xf3,0xe6,0x95,0x1f,0x2d,
                0xb4,0xfe,0xaf,0x5d,0xc3,0x34,0x26,0xf1,0x5b,0xb1,
                0x24,0xda,0x38,0x8d,0x70,0x90,0x83,0xa2,0x8f,0x57,
                0x01,0xef,0x96,0xc2,0x8b,0x3a,0x3c,0x75,0xbe,0xf9,
                0x33,0x2e,0xf3,0x73,0xb9,0x07,0x71,0x23,0x6a,0xf5,
                0xe2,0x5d,0x58,0x95,0x04,0x34,0x5d,0x28,0xa1,0x9a,
                0xb0,0xdb,0xc1,0xc9,0xb7,0x4d,0x1e,0xe2,0x1c,0x4b,
                0xd8,0xd4,0x23,0xde,0x6a},
        .key_len = 125,
        .data = {0x8d,0x2e,0x68,0xd7,0xe9,0x84,0x6c,0xfa,0x30,0xd9,
                 0x31,0xa3,0x8e,0xfb,0x59,0xbc,0xce,0xd5,0x3a,0x14,
                 0x16,0x4b,0x31,0x63,0xd2,0x65,0x38,0x88,0xee,0xb0,
                 0xbb,0x14,0x48,0xe1,0xa8,0x0c,0x65,0xbc,0xc6,0xeb,
                 0x63,0x34,0x47,0xe7,0x2e,0xd4,0xa0,0x75,0xf7,0x5d,
                 0x98,0x0f,0xe2,0xb1,0x9f,0x35,0xff,0xef,0x62,0xb2,
                 0x7c,0xe0,0x9c,0x20,0x19,0x92,0x2f,0xae,0xdb,0x42,
                 0x73,0x21,0x05,0x7f,0xce,0x19,0x44,0x8d,0x85,0x96,
                 0x2a,0x08,0xd1,0xba,0xdd,0xc9,0x36,0xd1,0x11,0x0e,
                 0x10,0x8e,0x33,0xd4,0x6f,0x97,0xe7,0x88,0x24,0x45,
                 0xb5,0xdf,0x1c,0xa4,0xff,0x03,0xed,0xc2,0x37,0xef,
                 0xaf,0x26,0x4f,0x1c,0x0d,0x9e,0x70,0x5d,0x9b,0x3e,
                 0xee,0x07,0x6b,0xa5,0x7c,0x56,0xdb,0x82},
        .data_len = 128,
        .mac = {0xb6,0xca,0xd1,0xca,0x5b,0xa5,0x05,0x49,0x8a,0x8f,
                0x66,0xa9,0x42,0x2b,0xf5,0x39,0x42,0x6a,0x8a,0x55,
                0x33,0x4f,0xab,0x9c,0x6b,0x9e,0x08,0xe3,0xa5,0x17,
                0x9d,0x15,0x7d,0x1e,0xfa,0x0f,0x91,0xd5,0xc5,0xe2,
                0x6f,0xfa,0x43,0xf5,0xc1,0xcb,0x7c,0xa5,0xf9,0x06,
                0xce,0x4f,0x0e,0xfc,0xf4,0xe8,0x71,0x82,0x0b,0x83,
                0x53,0xe8,0x90,0xe4},
        .mac_len = 64,
    }, {	// 4	(210)
        .key = {0xe9,0xe4,0x48,0x0d,0x1c,0x4a,0x62,0x1e,0x0c,0x4e,
                0x15,0x05,0x99,0x25,0x56,0x34,0x7a,0x7a,0xb3,0x4f,
                0xd2,0xb2,0x89,0x91,0x04,0x74,0x76,0x6c,0xc9,0x69,
                0x11,0x6f,0x80,0x40,0xd9,0x6d,0xc5,0xf6,0x6c,0xdc,
                0x44,0x54,0xfa,0x7b,0xcf,0xb9,0xf8,0x38,0xaf,0x19,
                0x19,0x50,0x38,0x46,0x7a,0xb8,0xa1,0x6e,0x1c,0xbc,
                0x12,0xe5,0x98,0xe6,0xfd,0x25,0x0e,0x21,0xb2,0x14,
                0x5f,0x1e,0x2e,0x85,0x9c,0xf7,0x34,0x00,0xbe,0x12,
                0xa0,0xc6,0x97,0x49,0xf7,0x10,0x08,0x47,0x42,0x98,
                0x75,0x35,0x1d,0x5a,0x76,0x97,0x0b,0x9c,0xcf,0x70,
                0x0c,0x2c,0xa3,0xad,0x72,0xe9,0xe4,0xc0,0xf0,0x84,
                0x0e,0x8c,0xf4,0x88,0x15,0x81,0x36,0x98,0x9b,0x08,
                0x91,0xf8,0x67,0x21,0x13,0x50,0x13,0x4a},
        .key_len = 128,
        .data = {0xb8,0x2e,0xef,0xb2,0x08,0x1b,0xd1,0x4d,0xab,0x0e,
                 0x9e,0x34,0x52,0x48,0xa3,0x4a,0xde,0x73,0xf3,0x29,
                 0x18,0x86,0xb9,0x1e,0xa3,0xe8,0xcc,0x74,0x2f,0xd8,
                 0x84,0xf6,0xee,0x0c,0xcd,0xaf,0x4c,0x98,0x79,0xf4,
                 0xdb,0x12,0xdb,0xa5,0x8c,0xf4,0x91,0xaf,0x25,0x41,
                 0xa1,0xd5,0xef,0x6c,0xc8,0xb1,0xaf,0x75,0x0e,0xf5,
                 0xd8,0x55,0x9e,0xf7,0xff,0x9c,0xd5,0x6d,0x8f,0x59,
                 0x99,0x74,0xbe,0x3a,0xec,0xd8,0xc0,0xf4,0xc0,0x8f,
                 0x3a,0xe5,0x0d,0x86,0xf9,0xf8,0x22,0xa1,0xe4,0xca,
                 0x39,0xfd,0x2f,0x0b,0x4d,0x78,0xd2,0x26,0x30,0x73,
                 0x3a,0x24,0xd8,0xd6,0x3e,0xcd,0xf9,0x55,0x54,0x11,
                 0xda,0xf2,0x05,0xa7,0x61,0xc3,0x9e,0xf4,0x6f,0xf6,
                 0x29,0x2e,0x74,0x12,0x9b,0xc1,0x3a,0x7f},
        .data_len = 128,
        .mac = {0x90,0x09,0x3b,0xdc,0xc4,0x5d,0xa7,0x33,0x8b,0xd2,
                0xef,0xe9,0x2e,0x30,0x93,0x3b,0x14,0xf7,0x55,0x82,
                0x73,0x9c,0x74,0x7f,0x75,0x72,0xb3,0x27,0x0b,0x10,
                0x4f,0x33,0xaf,0x0c,0x93,0x9e,0x3c,0x8a,0xe5,0x3b,
                0x20,0x66,0xfc,0x8c,0x97,0xcc,0xf3,0x87,0x85,0xcd,
                0x2e,0xc3,0xd7,0x9e,0x69,0x46,0x49,0x9d,0x36,0x12,
                0x1e,0x44,0xa3,0xe7},
        .mac_len = 64,
    }, {	// 5 (211)
        .key = {0xd3,0xfb,0xd6,0xfe,0x4e,0x35,0x6a,0xc1,0xc8,0xc1,
                0x20,0xd4,0x32,0xd7,0x20,0x4d,0x9d,0x57,0x9b,0x2a,
                0x5a,0x5d,0x0c,0x8b,0x60,0x16,0xbd,0x1e,0xef,0xd3,
                0x8d,0xda,0x73,0x5c,0xf2,0xf0,0xab,0x87,0x3a,0xfe,
                0x0a,0x09,0x16,0x86,0x5e,0x8b,0x58,0xa0,0xaf,0x01,
                0xfc,0xeb,0x6a,0x37,0x65,0xc9,0xbf,0xac,0xea,0xcc,
                0x47,0xa4,0x91,0x6b,0xea,0x79,0x1a,0xfa,0x00,0x32,
                0x40,0xd9,0xb6,0x56,0x3b,0xeb,0xb3,0x03,0x89,0x49,
                0xfc,0x3a,0xee,0x38,0x15,0x7d,0xba,0x59,0x6a,0x9c,
                0x4a,0x20,0xed,0xcc,0xd1,0x87,0xff,0xf9,0x59,0x04,
                0x94,0x5d,0x04,0xb8,0x92,0x52,0x98,0xe9,0x7b,0x64,
                0x3a,0xb2,0x4c,0xab,0x7a,0xf9,0xa5,0x58,0x90,0xa2,
                0x29,0x8d,0xe5,0x02,0x28,0x72,0xd6,0x97},
        .key_len = 128,
        .data = {0xb9,0x67,0xc7,0xd9,0xc0,0xa9,0x41,0xf0,0x2e,0x87,
                 0x72,0x3c,0xf2,0x82,0xea,0xda,0x43,0x47,0xb2,0x81,
                 0x93,0xd3,0xe0,0xbf,0xbe,0xda,0x69,0x85,0x88,0x6a,
                 0x37,0xe6,0x46,0xcc,0x7b,0x1c,0xdb,0xab,0x45,0xcc,
                 0xe6,0x77,0x52,0x8b,0x3a,0x0c,0x24,0xa0,0x8f,0x8f,
                 0x58,0x0b,0x77,0x99,0x35,0xc7,0x93,0x98,0x81,0x4d,
                 0x06,0x72,0x98,0x59,0x2a,0x6b,0xbf,0xf0,0x82,0x48,
                 0xb5,0xa2,0xf0,0xb4,0x8b,0x0d,0x28,0xe4,0xb6,0xa2,
                 0x65,0x77,0x63,0xac,0x5b,0xa0,0x0a,0x8d,0x6c,0x86,
                 0x46,0x4b,0x1e,0xeb,0xe4,0x4c,0xcd,0x0c,0x39,0x5e,
                 0x9d,0xc9,0xb9,0xfb,0xb3,0x06,0xc6,0xca,0xa5,0x51,
                 0xc6,0x68,0x2e,0xc5,0x78,0x69,0x27,0x2e,0x88,0x9a,
                 0xb2,0x6e,0x61,0x89,0xb9,0x1f,0x42,0x48},
        .data_len = 128,
        .mac = {0xbc,0x9a,0x83,0xd7,0x82,0xe5,0x0b,0xa5,0xa8,0x01,
                0x14,0x6f,0x8d,0xa3,0x90,0x95,0xd9,0x23,0x87,0xd7,
                0x59,0xeb,0x4a,0xd5,0x2b,0xbd,0x9e,0x99,0xd9,0xf6,
                0x8f,0x4a,0x0f,0x6f,0x64,0x70,0xc6,0x53,0xc4,0x59,
                0x79,0xc2,0xe1,0x95,0x43,0x80,0x4c,0xed,0x59,0x2e,
                0xe9,0xc5,0x3e,0xb6,0x8a,0x5b,0x1b,0x77,0x46,0xed,
                0x40,0x3e,0xbe,0x67},
        .mac_len = 64,
    }, {	// 6	(285)
        .key = {0x95,0xaf,0x10,0x92,0x0d,0xc7,0x88,0x26,0x9e,0x70,
                0xb8,0x56,0x0b,0x73,0x13,0x5c,0xf7,0xf6,0xf5,0xb0,
                0x4a,0x50,0x2c,0x7b,0xd6,0x1c,0xb7,0x4f,0x3b,0x8c,
                0xcd,0x16,0x07,0x01,0x22,0x49,0x22,0xd8,0x65,0x63,
                0x6a,0x86,0x0d,0x94,0x9a,0xe7,0x55,0xb9,0x70,0xd3,
                0x85,0x8c,0x0f,0xf3,0x74,0x18,0xa2,0xd2,0x4b,0x71,
                0x42,0x37,0x8b,0xa1,0x1a,0xb3,0x52,0xe5,0xc8,0x76,
                0xda,0x1a,0x07,0x66,0x42,0x72,0x8b,0x73,0x91,0x6b,
                0x2d,0x24,0xf8,0x02,0x48,0x76,0x57,0x23,0x63,0xe7,
                0x03,0x65,0x10,0xce,0xc7,0xf4,0x13,0xed,0x28,0xce,
                0xc7,0x49,0xed,0x33,0xbe,0x3a,0xdf,0x56,0xa8,0xbe,
                0xce,0x59,0x76,0x12,0xd4,0x78,0xbf,0x84,0xde,0x85,
                0x62,0x83,0x67,0x94,0x6d,0xf8,0x87,0xf7,0x3d,0xd9,
                0x2d,0x6d,0xe7,0xfa,0xa8,0x96,0xd7,0x27,0x6d},
        .key_len = 139,
        .data = {0x61,0xd9,0x1f,0x31,0x7a,0x90,0x2e,0xa0,0x94,0x4e,
                 0x11,0xe9,0x2e,0x66,0x57,0xa5,0x89,0xe1,0x7a,0xbc,
                 0x02,0x7f,0xcd,0x86,0x9f,0xf8,0xb0,0x30,0xe8,0x87,
                 0x06,0x62,0xf8,0xa9,0xe9,0x1e,0xd3,0x23,0x9c,0xec,
                 0xfa,0x42,0xc0,0x34,0x3d,0x66,0xcb,0xeb,0xd1,0xc2,
                 0xb7,0x71,0xa2,0x5d,0xf7,0xba,0xea,0x5c,0xaf,0xad,
                 0x03,0x84,0x24,0xc9,0x7a,0xfb,0x72,0x0e,0x64,0x4e,
                 0x7d,0x1b,0xf5,0xb8,0x29,0x94,0x4e,0xa2,0xce,0xc6,
                 0x97,0x66,0xe6,0x8e,0x4e,0x58,0x09,0x76,0xde,0x07,
                 0x1c,0x22,0x74,0xc0,0xc5,0xeb,0x0e,0x54,0x21,0xc9,
                 0xd5,0x1b,0xba,0x76,0xac,0x39,0xb3,0xd0,0x09,0x20,
                 0x46,0x80,0x03,0x57,0x71,0xd9,0xad,0x79,0xeb,0x02,
                 0xa3,0x80,0x5d,0x58,0xe2,0x43,0xcf,0x0e},
        .data_len = 128,
        .mac = {0x6e,0x98,0x9e,0xc9,0xcb,0xf0,0x10,0xad,0x66,0x91,
                0xa6,0x72,0xff,0x4c,0xa9,0x0a,0x00,0x27,0x5f,0x9b,
                0xa4,0xc8,0x1c,0xd1,0x47,0xcc,0x50,0x6e,0x1d,0xbc,
                0x8b,0xc9,0x3b,0x1d,0x96,0xa3,0x75,0xe4,0x93,0x50,
                0x3c,0x0a,0xc6,0x97,0xf7,0xc4,0x5e,0x4f,0xad,0xf1,
                0x38,0x24,0x2d,0xf7,0xe0,0x6e,0x67,0x7d,0xe2,0x45,
                0xaf,0xa9,0x77,0x80},
        .mac_len = 64,
    }, {	// 7	(286)
        .key = {0x27,0xe6,0xc9,0xf2,0x70,0xb9,0x85,0x5c,0x96,0x58,
                0xad,0x0e,0x3d,0x6c,0x9a,0x11,0x1a,0x62,0x4f,0x66,
                0xfa,0x64,0xa4,0x9a,0x06,0x88,0xa4,0x9b,0x45,0x47,
                0x33,0xca,0x62,0x30,0xf4,0x51,0xb0,0xdd,0x69,0xb7,
                0x6b,0x27,0x5c,0xb2,0x41,0x96,0x7e,0x3c,0x10,0x1b,
                0x4f,0xe8,0xf2,0x02,0x3d,0x77,0x77,0x22,0x10,0xa6,
                0x31,0x57,0x85,0x4b,0x76,0x32,0x39,0xa0,0x61,0xee,
                0xc9,0xdf,0x1a,0xa6,0x38,0x0f,0x57,0xc6,0x91,0x1d,
                0x23,0xc0,0xcd,0x2e,0xdf,0x00,0xf6,0x34,0x86,0x21,
                0x8d,0xbf,0x35,0x61,0x2a,0x17,0xea,0x52,0x62,0x87,
                0x8b,0xd3,0xed,0xfb,0x2b,0x3f,0x08,0xce,0x8a,0xe4,
                0x19,0xdd,0xda,0xb7,0x92,0xe0,0xc9,0x45,0x17,0xfa,
                0xbb,0xed,0xe3,0x8e,0x57,0x4d,0x68,0x55,0x46,0xfa,
                0x35,0xad,0x37,0x74,0x1d,0x34,0x27,0x59,0x96},
        .key_len = 139,
        .data = {0xdf,0x24,0x27,0x9b,0xf8,0x27,0x7a,0xd1,0x09,0x19,
                 0x72,0xb8,0x25,0x94,0xd8,0x46,0x77,0xe5,0x4f,0xe5,
                 0xd6,0x57,0x86,0xd1,0x9a,0xb5,0xb2,0xc1,0xae,0x0a,
                 0x3c,0xc9,0xe7,0xab,0xb6,0x7f,0x94,0x77,0x14,0x5d,
                 0x57,0x5e,0x19,0x66,0x33,0x20,0x0f,0x0c,0xe5,0x57,
                 0xbb,0x52,0x78,0xb8,0x90,0x2e,0x14,0x96,0x23,0x31,
                 0x17,0xa7,0xdf,0x69,0x66,0x0b,0xfa,0x87,0x06,0x8a,
                 0xa7,0x3d,0xe6,0x1e,0x8e,0xea,0xff,0xb1,0x79,0x79,
                 0x9f,0x27,0x50,0x86,0x02,0x9f,0x47,0xc3,0x23,0xf6,
                 0x56,0x9b,0xd1,0x8d,0xea,0x15,0x05,0x4d,0xda,0xfa,
                 0x73,0xe8,0x9c,0x3a,0x5f,0x61,0xb9,0x8c,0xb2,0xce,
                 0x7e,0x55,0x4d,0x5d,0xf4,0xcb,0x9d,0x95,0x13,0x5a,
                 0x70,0xde,0x33,0x47,0x07,0x44,0xc3,0x93},
        .data_len = 128,
        .mac = {0xe6,0xf6,0x06,0x12,0x75,0xa8,0x93,0x45,0xf5,0x46,
                0x3c,0xfa,0x19,0x8d,0x52,0x8e,0x14,0x04,0x7d,0x47,
                0x8f,0x69,0xad,0x7a,0x73,0x43,0x2f,0x18,0xf8,0x8b,
                0xc6,0x8a,0x1b,0x8a,0xba,0x2c,0x3b,0x02,0x5c,0x93,
                0xb2,0x5d,0xeb,0x8f,0x40,0x37,0x63,0xa5,0x50,0x24,
                0x40,0x8a,0x97,0xa9,0x03,0xe9,0x5f,0x0c,0xb6,0x17,
                0x8e,0x7b,0xe3,0x89},
        .mac_len = 64,
    }, {	// 8	(360)
        .key = {0x01,0xb9,0x5a,0x88,0x79,0x27,0xce,0x31,0xb1,0x24,
                0x23,0x91,0xbb,0xd0,0x09,0x65,0xeb,0x77,0xa9,0x03,
                0xd4,0xb8,0x39,0x9b,0x72,0xe6,0xce,0xbd,0xa9,0xae,
                0x72,0x1b,0xee,0xfa,0x77,0x91,0x45,0x16,0x0b,0x62,
                0x6b,0x11,0x0c,0xc5,0x54,0x67,0x1d,0xa0,0xd8,0xdc,
                0xf9,0x93,0xa9,0xab,0x07,0x38,0x88,0xe0,0x2f,0xa9,
                0xb8,0x03,0xed,0x43,0xb3,0xf6,0xa3,0xaa,0x1d,0x20,
                0x34,0x0d,0xf6,0xcc,0xce,0xac,0x13,0xcb,0x07,0x97,
                0xcf,0x61,0x2c,0xb8,0xfe,0x5f,0xd5,0x13,0x22,0x8c,
                0xbd,0x4d,0xe2,0x49,0xd1,0x6b,0xb7,0x75,0x87,0xdd,
                0xe9,0x8f,0x71,0xbb,0xba,0x1a,0x12,0x4e,0xe0,0x46,
                0xf0,0xd2,0x39,0xcc,0xea,0x7a,0xbb,0x1a,0xcc,0xb5,
                0xaa,0xb0,0x21,0xb0,0x0d,0xca,0x49,0x1c,0x62,0x3f,
                0xcb,0x31,0x91,0xa9,0xec,0xf3,0x1f,0xc6,0x80,0xb4,
                0xa4,0x1e},
        .key_len = 142,
        .data = {0x63,0x2a,0xfa,0x8e,0x79,0xb1,0x4b,0x2a,0x36,0x04,
                 0xf5,0x85,0x5d,0x2b,0xf1,0x82,0xd3,0xc5,0x6d,0x68,
                 0x53,0xf2,0x1f,0xe4,0x62,0x71,0xda,0x52,0x86,0x06,
                 0x5f,0x38,0xb3,0x1f,0x75,0x13,0x06,0xb6,0x3c,0x57,
                 0xb6,0x79,0xbe,0xb1,0x47,0x29,0xc7,0x8f,0x00,0x40,
                 0xf7,0xe2,0xa0,0xd6,0x15,0x22,0x4d,0xc5,0xa6,0x93,
                 0xcd,0x0c,0xbe,0xc8,0xf8,0x71,0x17,0x65,0x6d,0x6b,
                 0x60,0x29,0x85,0x3e,0xd7,0x2b,0x85,0x68,0x1a,0x63,
                 0x18,0x3c,0x3a,0x6d,0xfc,0xcd,0x12,0x8a,0xfb,0x0d,
                 0xd7,0xe8,0x1d,0x36,0xf0,0x23,0x1c,0x69,0x07,0x0b,
                 0x18,0x95,0x60,0xa8,0x8c,0x9b,0x69,0x7b,0x81,0xb0,
                 0x93,0x07,0x01,0x02,0x61,0x90,0xcf,0x9e,0xbe,0x23,
                 0x55,0x91,0x94,0xd6,0xde,0x4d,0x9a,0x51},
        .data_len = 128,
        .mac = {0x21,0x0a,0xd4,0x5c,0xa2,0xfd,0x1f,0x10,0x5c,0x0a,
                0x18,0xf9,0x93,0x77,0x4f,0x93,0x3e,0xce,0x57,0xac,
                0xe4,0xda,0x61,0x96,0x89,0xe1,0xcb,0x8b,0x49,0x1a,
                0x18,0x9c,0xc6,0xe4,0xee,0x19,0x54,0xa3,0x22,0x01,
                0x07,0x2e,0x70,0xf9,0x34,0x83,0x7c,0x0f,0xb6,0xe2,
                0x39,0xb4,0xfd,0xfb,0xd2,0x6e,0xbf,0x11,0xb9,0xa9,
                0x19,0xea,0xfd,0x09},
        .mac_len = 64,
    }, {	// 9	(361)
        .key = {0x61,0x09,0x6f,0x4f,0xe5,0x34,0x04,0x88,0x91,0x6d,
                0xe2,0x93,0xbe,0x38,0xcc,0x3a,0xe0,0xc8,0x77,0x67,
                0x0c,0x71,0x36,0x37,0xb7,0x60,0xd7,0x4f,0xc1,0x8a,
                0xc7,0x73,0xb2,0xe2,0x7d,0x55,0x43,0xcf,0x16,0xaa,
                0x20,0xdd,0x3d,0x83,0xec,0xb3,0x4e,0xdb,0x85,0x45,
                0xbb,0x6c,0x8a,0x4a,0xae,0xc8,0x1b,0xf1,0xf0,0xa4,
                0xe0,0xcf,0x09,0x77,0x4d,0x1c,0xa9,0x44,0x24,0x20,
                0x46,0xb3,0x3b,0xe8,0x07,0x67,0x7f,0x3d,0xe1,0x8c,
                0x39,0xd7,0x00,0xaf,0x90,0xcd,0x68,0xd3,0x4f,0x50,
                0xdc,0xc1,0xe9,0x99,0xfe,0x9f,0xbb,0x20,0xb9,0xc4,
                0x90,0x0f,0xdc,0xcb,0x6a,0xf6,0x07,0xe6,0x80,0xc0,
                0xcb,0x75,0x83,0xe6,0x0d,0xd8,0x25,0xe2,0xab,0x81,
                0xdc,0xe7,0x63,0x4d,0xe3,0xcf,0xf0,0x14,0x83,0x55,
                0x75,0x7f,0x90,0x84,0x1f,0x19,0x36,0x6f,0x06,0xa9,
                0xf6,0x23},
        .key_len = 142,
        .data = {0x67,0xe7,0x04,0x04,0x6f,0x98,0xcb,0x5a,0xa9,0x7d,
                 0xa9,0x5b,0x19,0x14,0x73,0x91,0xf0,0x57,0x88,0xf8,
                 0x11,0x36,0x6b,0x0e,0xce,0x44,0xb1,0x2a,0xf2,0xb1,
                 0x1e,0x0e,0x05,0x78,0x0b,0xbf,0xcb,0xd9,0x0a,0x95,
                 0x0e,0x0a,0xcd,0x8e,0x9d,0x2a,0x44,0xe7,0x95,0x76,
                 0x06,0xee,0xdf,0xbf,0xf2,0x12,0xfa,0x1c,0x16,0x3c,
                 0xfb,0xdc,0xd0,0x62,0xd2,0xbe,0x32,0x59,0xce,0x65,
                 0xab,0xea,0x64,0x06,0xe4,0x29,0x2c,0x64,0xe9,0x02,
                 0x2c,0xfe,0x89,0x15,0x59,0x86,0xff,0xc4,0x5b,0x96,
                 0xd2,0x89,0x91,0x9f,0xf9,0x8d,0x55,0x22,0x43,0x77,
                 0x81,0x22,0xf6,0x82,0x31,0xd9,0xb6,0xd3,0xcb,0xaa,
                 0xa9,0x09,0x3d,0x57,0xd9,0x15,0x86,0x74,0xda,0x4c,
                 0x78,0x1b,0xac,0xba,0xbc,0xe2,0xe2,0xba},
        .data_len = 128,
        .mac = {0x9a,0x2d,0x14,0x7e,0x50,0x82,0x71,0x57,0xf3,0x86,
                0x6e,0x86,0x8c,0x1c,0xca,0x9f,0x08,0x15,0x79,0xc9,
                0x2f,0x25,0xda,0x8c,0xeb,0xc9,0xed,0x24,0x99,0x28,
                0xc8,0x2b,0xea,0xd3,0x9d,0x48,0x0e,0xcb,0xb5,0xb5,
                0xd0,0xe0,0x75,0x50,0x29,0xae,0xbf,0x3e,0x02,0x06,
                0x98,0x4f,0x3e,0xa8,0x3f,0x4d,0x63,0x72,0xf4,0x45,
                0x33,0x90,0xe0,0x70},
        .mac_len = 64,
    },
};

struct HMAC_TEST_VECTOR fips_sha1_hmac_general_test_vector[] = {
    {	// 0 (0)
        .key = {0x82,0xf3,0xb6,0x9a,0x1b,0xff,0x4d,0xe1,0x5c,0x33},
        .key_len = 10,
        .data = {0xfc,0xd6,0xd9,0x8b,0xef,0x45,0xed,0x68,0x50,
                 0x80,0x6e,0x96,0xf2,0x55,0xfa,0x0c,0x81,0x14,
                 0xb7,0x28,0x73,0xab,0xe8,0xf4,0x3c,0x10,0xbe,
                 0xa7,0xc1,0xdf,0x70,0x6f,0x10,0x45,0x8e,0x6d,
                 0x4e,0x1c,0x92,0x01,0xf0,0x57,0xb8,0x49,0x2f,
                 0xa1,0x0f,0xe4,0xb5,0x41,0xd0,0xfc,0x9d,0x41,
                 0xef,0x83,0x9a,0xcf,0xf1,0xbc,0x76,0xe3,0xfd,
                 0xfe,0xbf,0x22,0x35,0xb5,0xbd,0x03,0x47,0xa9,
                 0xa6,0x30,0x3e,0x83,0x15,0x2f,0x9f,0x8d,0xb9,
                 0x41,0xb1,0xb9,0x4a,0x8a,0x1c,0xe5,0xc2,0x73,
                 0xb5,0x5d,0xc9,0x4d,0x99,0xa1,0x71,0x37,0x79,
                 0x69,0x23,0x41,0x34,0xe7,0xda,0xd1,0xab,0x4c,
                 0x8e,0x46,0xd1,0x8d,0xf4,0xdc,0x01,0x67,0x64,
                 0xcf,0x95,0xa1,0x1a,0xc4,0xb4,0x91,0xa2,0x64,
                 0x6b,0xe1},
        .data_len = 128,
        .mac_len = 10,
        .mac = {0x1b,0xa0,0xe6,0x6c,0xf7,0x2e,0xfc,0x34,0x92,0x07},
        .chunks = {64, 64},
        .num_chunks = 2,
    }, {	// 1 (1)
        .key = {0x47,0x66,0xe6,0xfe,0x5d,0xff,0xc9,0x8a,0x5c,0x50},
        .key_len = 10,
        .data = {0xd6,0x8b,0x82,0x8a,0x15,0x3f,0x51,0x98,0xc0,
                 0x05,0xee,0x36,0xc0,0xaf,0x2f,0xf9,0x2e,0x84,
                 0x90,0x75,0x17,0xf0,0x1d,0x9b,0x7c,0x79,0x93,
                 0x46,0x9d,0xf5,0xc2,0x10,0x78,0xfa,0x35,0x6a,
                 0x8c,0x97,0x15,0xec,0xe2,0x41,0x4b,0xe9,0x4e,
                 0x10,0xe5,0x47,0xf3,0x2c,0xbb,0x8d,0x05,0x82,
                 0x52,0x3e,0xd3,0xbb,0x00,0x66,0x04,0x6e,0x51,
                 0x72,0x20,0x94,0xaa,0x44,0x53,0x3d,0x2c,0x87,
                 0x6e,0x82,0xdb,0x40,0x2f,0xbb,0x00,0xa6,0xc2,
                 0xf2,0xcc,0x34,0x87,0x97,0x3d,0xfc,0x16,0x74,
                 0x46,0x3e,0x81,0xe4,0x2a,0x39,0xd9,0x40,0x29,
                 0x41,0xf3,0x9b,0x5e,0x12,0x6b,0xaf,0xe8,0x64,
                 0xea,0x16,0x48,0xc0,0xa5,0xbe,0x0a,0x91,0x26,
                 0x97,0xa8,0x7e,0x4f,0x8e,0xab,0xf7,0x9c,0xbf,
                 0x13,0x0e},
        .data_len = 128,
        .mac_len = 10,
        .mac = {0x00,0x7e,0x45,0x04,0x04,0x1a,0x12,0xf9,0xe3,0x45},
        .chunks = {32, 32, 32, 32},
        .num_chunks = 4,
    }, {	// 2 (15)
        .key = {0x13,0x79,0xa7,0xaf,0xcc,0x09,0x05,0xa5,0xfc,0x81},
        .key_len = 10,
        .data = {0x96,0xfa,0x56,0x19,0xfa,0xc6,0x48,0x84,0x3d,
                 0xb7,0x88,0xcb,0x8e,0x90,0xdc,0x6f,0xfd,0x6e,
                 0xfe,0x13,0x32,0xab,0xf0,0x81,0x5f,0x03,0x90,
                 0xee,0x73,0xf5,0x6c,0x7f,0x91,0x6c,0xd7,0x0c,
                 0xc0,0x9f,0x3d,0x23,0xe4,0x36,0xb3,0x50,0xed,
                 0xae,0xd2,0x9b,0x4e,0xfe,0xc6,0x53,0xb0,0x7b,
                 0xa2,0x0a,0xe8,0xf9,0xf6,0xe1,0x27,0x33,0xa4,
                 0x06,0x71,0x6d,0xef,0x7a,0x51,0x57,0xd5,0x18,
                 0xca,0x35,0x9f,0xd3,0x90,0x3d,0xb6,0x3f,0x79,
                 0x40,0xb8,0x53,0x2e,0x8d,0xcb,0x6d,0x26,0x13,
                 0x32,0x96,0xd5,0xc5,0x1e,0x07,0x20,0x43,0xc6,
                 0xed,0x15,0xb6,0xb9,0x6a,0xd9,0xfb,0x73,0xdc,
                 0xe1,0x05,0x2f,0x61,0x65,0x7c,0xfd,0x9b,0x12,
                 0xaa,0x14,0xb0,0x00,0x98,0x69,0x95,0xe3,0x74,
                 0x81,0x8d},
        .data_len = 128,
        .mac_len = 12,
        .mac = {0x42,0x53,0x7b,0x22,0x52,0x0a,0x08,0x55,0x77,
                0x58,0x76,0x16},
        .chunks = {32, 32, 64},
        .num_chunks = 3,
    }, {	// 3 (16)
        .key = {0x80,0xa0,0xdb,0x49,0xd0,0x39,0xb3,0x16,0xae,0x12},
        .key_len = 10,
        .data = {0x91,0xf8,0xec,0x84,0x8d,0x6f,0x81,0x14,0x31,
                 0xcb,0xde,0xee,0x15,0x0b,0x93,0xaf,0x6f,0x67,
                 0x8b,0xe9,0x9c,0x90,0x3f,0x81,0xfc,0x38,0x29,
                 0x55,0x03,0xd5,0x7c,0x22,0x8d,0xa2,0x12,0xa6,
                 0x72,0xe7,0xa6,0x01,0x5b,0x7b,0x43,0x61,0xd4,
                 0x87,0xfc,0xde,0xa2,0x8c,0xde,0xa3,0x56,0xa8,
                 0x23,0x4f,0x22,0x15,0xa8,0x9b,0xec,0xf2,0xa2,
                 0x3c,0xa1,0x46,0x8c,0x0b,0xcc,0x42,0x64,0x63,
                 0x67,0xc6,0x16,0xca,0xf0,0x27,0x39,0xd4,0xc0,
                 0x30,0xf9,0x45,0x99,0x66,0x54,0x76,0x7e,0x90,
                 0x8a,0xfa,0xc7,0x77,0xce,0x80,0x74,0xeb,0x42,
                 0xfb,0xc2,0x06,0x22,0x01,0xfc,0xb5,0x3f,0x71,
                 0x94,0x73,0xb0,0x59,0x72,0x58,0xc4,0x17,0x8c,
                 0x53,0x3b,0xbe,0xb7,0xb4,0xb5,0xbb,0xbc,0xed,
                 0x6a,0xb8},
        .data_len = 128,
        .mac_len = 12,
        .mac = {0xec,0xae,0x13,0x83,0x22,0xd2,0xd4,0x08,0x6a,
                0xa2,0xbe,0xc6},
        .chunks = {128},
        .num_chunks = 1,
    }, {	// 4 (30)
        .key = {0xf5,0x73,0x1a,0x6e,0x89,0x25,0xf7,0x43,0x06,0xfa},
        .key_len = 10,
        .data = {0x44,0xc7,0xcc,0x06,0xad,0x29,0x0f,0x3a,0x54,
                 0xa9,0x70,0xb6,0x40,0x01,0x4c,0xb5,0xd1,0xe6,
                 0x18,0x23,0x52,0x45,0x99,0x01,0xcd,0xcd,0x57,
                 0x0c,0x23,0xad,0x4f,0x99,0x5b,0x9f,0xe8,0xc4,
                 0x3b,0x25,0x28,0xc9,0x15,0x12,0x28,0xb2,0xe4,
                 0x4d,0xc5,0x33,0x98,0xd2,0x99,0xd2,0xad,0xf9,
                 0x2a,0x4a,0x02,0xfb,0x60,0x32,0xe9,0xb2,0x3d,
                 0xda,0x7a,0xa0,0xc8,0x76,0x2e,0x33,0x4a,0x7e,
                 0xa9,0x47,0xbd,0x54,0xd6,0xed,0x82,0x28,0x39,
                 0x6b,0x52,0x19,0x81,0x84,0x77,0x9c,0x5d,0xf9,
                 0x3c,0x22,0x91,0x4f,0xa2,0xf5,0x49,0xd3,0x54,
                 0x63,0xad,0xdc,0xdd,0x1f,0xb5,0x50,0x19,0xe4,
                 0x3f,0x69,0xe9,0x5b,0x5f,0xb9,0x2b,0x3f,0xf6,
                 0x6c,0xea,0xbf,0x86,0xce,0xd1,0x24,0x44,0x0d,
                 0xe6,0xb3},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x73,0xb0,0x83,0xd8,0xbe,0x0d,0x19,0xee,0x7a,
                0x69,0x7f,0x9e,0x5d,0x76,0x36,0x2f},
        .chunks = {32, 0, 64, 32},
        .num_chunks = 4,
    }, {	// 5 (31)
        .key = {0x29,0x05,0x66,0xd7,0x77,0xb0,0xee,0xe9,0x84,0xfa},
        .key_len = 10,
        .data = {0x78,0x7f,0xda,0xa9,0x0a,0x2d,0xe3,0x93,0x7e,
                 0x79,0x42,0xe6,0x71,0x1f,0x16,0x5a,0x89,0xb9,
                 0xe0,0x77,0xfe,0x32,0x2c,0xab,0x59,0x7d,0x74,
                 0x9a,0x7c,0x87,0x41,0xb5,0xe3,0x6a,0x93,0x0e,
                 0x29,0xe3,0x83,0x6a,0xce,0x06,0x27,0x98,0x37,
                 0x30,0xb6,0x02,0xf6,0x3e,0xec,0x82,0x4c,0xfc,
                 0xb0,0x77,0xec,0xe0,0xf5,0x17,0x02,0xf9,0xde,
                 0x07,0x74,0x22,0x25,0x29,0x68,0x7b,0xbd,0xb5,
                 0x06,0x1a,0xb6,0x8b,0x7f,0xfd,0x62,0xc7,0x4e,
                 0x43,0xb6,0x96,0xbe,0x9c,0xf2,0x49,0xac,0xff,
                 0x85,0xa8,0x8e,0x9b,0x2a,0x89,0xb4,0x0f,0x58,
                 0xa1,0xce,0xdd,0xd9,0x99,0xaf,0x1c,0xb8,0x64,
                 0x50,0x6e,0x61,0xd1,0x18,0x32,0x04,0x5c,0x5a,
                 0xfb,0x3a,0x4a,0x20,0x40,0xeb,0xf5,0x27,0x55,
                 0x6f,0x64},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xd7,0x2b,0x37,0x0a,0x1d,0x82,0x90,0x10,0x51,
                0x73,0xc8,0x3a,0xee,0xdb,0x83,0x58 },
        .chunks = {64, -1, 64},
        .num_chunks = 3,
    }, {	//6 (45)
        .key = {0x59,0x78,0x59,0x28,0xd7,0x25,0x16,0xe3,0x12,0x72},
        .key_len = 10,
        .data = {0xa3,0xce,0x88,0x99,0xdf,0x10,0x22,0xe8,0xd2,
                 0xd5,0x39,0xb4,0x7b,0xf0,0xe3,0x09,0xc6,0x6f,
                 0x84,0x09,0x5e,0x21,0x43,0x8e,0xc3,0x55,0xbf,
                 0x11,0x9c,0xe5,0xfd,0xcb,0x4e,0x73,0xa6,0x19,
                 0xcd,0xf3,0x6f,0x25,0xb3,0x69,0xd8,0xc3,0x8f,
                 0xf4,0x19,0x99,0x7f,0x0c,0x59,0x83,0x01,0x08,
                 0x22,0x36,0x06,0xe3,0x12,0x23,0x48,0x3f,0xd3,
                 0x9e,0xde,0xaa,0x4d,0x3f,0x0d,0x21,0x19,0x88,
                 0x62,0xd2,0x39,0xc9,0xfd,0x26,0x07,0x41,0x30,
                 0xff,0x6c,0x86,0x49,0x3f,0x52,0x27,0xab,0x89,
                 0x5c,0x8f,0x24,0x4b,0xd4,0x2c,0x7a,0xfc,0xe5,
                 0xd1,0x47,0xa2,0x0a,0x59,0x07,0x98,0xc6,0x8e,
                 0x70,0x8e,0x96,0x49,0x02,0xd1,0x24,0xda,0xde,
                 0xcd,0xbd,0xa9,0xdb,0xd0,0x05,0x1e,0xd7,0x10,
                 0xe9,0xbf},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x3c,0x81,0x62,0x58,0x9a,0xaf,0xae,0xe0,0x24,
                0xfc,0x9a,0x5c,0xa5,0x0d,0xd2,0x33,0x6f,0xe3,
                0xeb,0x28},
        .chunks	= {25, 25, 25, 25, 3, 25},
        .num_chunks = 6,
    }, {	// 7 (46)
        .key = {0xc5,0x21,0x09,0xc9,0xd0,0xda,0x92,0x58,0xeb,0x73},
        .key_len = 10,
        .data = {0x52,0xb1,0x13,0x61,0x4b,0x80,0xb9,0x70,0x51,
                 0x0f,0x65,0xa2,0x5d,0x46,0xed,0xc0,0x23,0xd9,
                 0xc7,0xb8,0xe7,0xca,0x7c,0x41,0x92,0x30,0x59,
                 0xc2,0x05,0x36,0x68,0x70,0xad,0x66,0x9f,0xb7,
                 0x57,0x28,0x56,0xdc,0x46,0x85,0xff,0xe0,0x83,
                 0x31,0x11,0xa7,0x75,0xc9,0x45,0x5a,0xb1,0x59,
                 0x05,0x09,0x13,0x21,0x21,0x95,0x0e,0x99,0xc5,
                 0xcd,0x40,0xb2,0xa8,0xd7,0x4a,0x5f,0x85,0xd2,
                 0xde,0x54,0xcf,0xb9,0x1a,0x0d,0xa1,0x8a,0x14,
                 0x13,0xf4,0xa8,0xb6,0x7b,0x14,0x7e,0xcc,0xaf,
                 0x55,0x66,0x5b,0x71,0x01,0xc9,0x34,0x1c,0x96,
                 0x87,0xca,0x2d,0x2e,0x99,0x41,0x03,0x3f,0xf5,
                 0xc7,0xe3,0x84,0xb1,0x27,0x3f,0x3b,0x6c,0x9b,
                 0x38,0x91,0xea,0xe2,0x61,0x5b,0xfe,0x93,0xc6,
                 0x06,0xad},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x2f,0xec,0xb4,0x66,0xbc,0x92,0x0f,0x61,0x0e,
                0x3e,0xae,0x99,0x49,0xe0,0x0f,0x45,0x4a,0x71,
                0x4a,0xb5},
        .chunks = {54, 0, 54, -1, 20},
        .num_chunks = 5,
    },  {	// 8 (60)
        .key = {0x19,0x1a,0x70,0x0f,0x3d,0xc5,0x60,0xa5,0x89,
                0xf9,0xc2,0xca,0x78,0x4e,0x97,0x0c,0xb1,0xe5,
                0x52,0xa0,0xe6,0xb3,0xdf,0x54,0xfc,0x1c,0xe3,
                0xc5,0x6c,0xc4,0x46,0xd2},
        .key_len = 32,
        .data = {0x19,0x48,0xc7,0x12,0x0a,0x06,0x18,0xc5,0x44,
                 0xa3,0x9e,0x59,0x57,0x40,0x8b,0x89,0x22,0x0a,
                 0xe3,0x98,0xec,0x05,0x30,0x39,0xb0,0x09,0x78,
                 0xad,0xb7,0x0a,0x6c,0x2b,0x6c,0x9c,0xe2,0x84,
                 0x6d,0xb5,0x85,0x07,0xde,0xb5,0xcb,0xa2,0x02,
                 0xa5,0x28,0x4b,0x0c,0xbc,0x82,0x9e,0x32,0x28,
                 0xe4,0xc8,0x04,0x0b,0x76,0xa3,0xfc,0xc3,0xad,
                 0x22,0x56,0x6e,0xbf,0xf0,0x21,0xad,0x5a,0x54,
                 0x97,0xa9,0x95,0x58,0xaa,0x54,0x27,0x2a,0xdf,
                 0xf2,0xd6,0xc2,0x5f,0xd7,0x33,0xc5,0x4c,0x72,
                 0x85,0xaa,0x51,0x8a,0x03,0x1b,0x7d,0xc8,0x46,
                 0x9e,0x51,0x76,0xfd,0x74,0x17,0x86,0xe3,0xc1,
                 0x76,0xd6,0xee,0xee,0x44,0xb2,0xc9,0x4c,0x9b,
                 0x9b,0x85,0xfa,0x2f,0x46,0x8c,0x08,0xde,0xe8,
                 0xd6,0xdc},
        .data_len = 128,
        .mac_len = 10,
        .mac = {0x40,0x24,0x93,0xfa,0xc2,0x6c,0x24,0x54,0xd0,0xcb},
    }, {	// 9 (61)
        .key = {0xdc,0xb4,0x63,0xa1,0x3a,0xe3,0x37,0x41,0x41,
                0x51,0xa3,0x1a,0xa0,0xc3,0xe8,0xba,0xb3,0xee,
                0x78,0x1b,0x9f,0x3a,0xaa,0x86,0x9d,0xc5,0xb1,
                0xb1,0x96,0xab,0xcf,0x2b},
        .key_len = 32,
        .data = {0x44,0xc9,0xbf,0x3a,0xe8,0xf1,0x4c,0xc9,0xd6,
                 0x93,0x5d,0xed,0xa3,0xc2,0x4d,0xe6,0x9c,0x67,
                 0xf0,0x88,0x5a,0x87,0xc8,0x99,0x96,0xc4,0x7c,
                 0x7b,0x3e,0x27,0x85,0x0a,0xc7,0x1c,0x2b,0xc8,
                 0xc6,0xbe,0xb0,0x38,0xba,0x55,0xcb,0x87,0x2c,
                 0x1d,0x58,0x71,0xfb,0x4a,0x4d,0x63,0xf1,0x48,
                 0xf0,0xdd,0x99,0x47,0x47,0x1b,0x55,0xf7,0xd0,
                 0xf4,0xab,0x90,0x73,0x02,0xe0,0x16,0xb5,0x03,
                 0xc8,0xdb,0x2e,0x7f,0xdc,0x45,0x3d,0xac,0x8d,
                 0xd1,0xfa,0x8e,0xd8,0x58,0x6c,0x62,0x1b,0x92,
                 0xfd,0x3d,0x27,0xd8,0x2a,0xf1,0x96,0x2e,0x7f,
                 0x30,0x5f,0x80,0xc3,0xf4,0xa7,0x2c,0x70,0x1d,
                 0xda,0xc1,0x66,0x5c,0xfb,0x06,0xdf,0x51,0x38,
                 0x3f,0xa6,0xf0,0xc2,0xab,0x84,0x29,0xdb,0x51,
                 0xfb,0xc8},
        .data_len = 128,
        .mac_len = 10,
        .mac = {0xb9,0x6d,0xe3,0xa2,0x19,0xd7,0x66,0x14,0xaa,0xa4},
    }, {	// 10 (75)
        .key = {0xa1,0x27,0x94,0x05,0x7d,0xe3,0xb3,0xea,0x42,
                0x6f,0xbe,0x01,0x95,0xee,0x17,0xb4,0x87,0x3e,
                0xf7,0xe6,0xba,0x87,0xb2,0x2b,0xc6,0x14,0x3c,
                0x38,0xda,0x62,0xec,0x98},
        .key_len = 32,
        .data = {0xd1,0x99,0x87,0x5b,0xb7,0x07,0x1c,0x43,0x4a,
                 0xb2,0x36,0xe6,0xd1,0x0f,0x84,0x05,0x97,0x8f,
                 0xca,0x25,0x9f,0x7c,0x34,0x93,0x94,0x24,0xea,
                 0xa6,0xff,0x3a,0xe4,0x44,0xbd,0x79,0x00,0xa7,
                 0xaf,0x8a,0x51,0x61,0xb3,0x28,0xba,0x9e,0xd3,
                 0x82,0xbc,0xaa,0xbd,0xe1,0x8d,0xb3,0x73,0x8a,
                 0x6a,0xcf,0x44,0xe6,0x2d,0x41,0xfb,0xe0,0x22,
                 0xf8,0x56,0x8f,0x17,0x58,0xba,0x15,0xb2,0x3d,
                 0x24,0xc7,0x08,0x3d,0x63,0x8e,0x6a,0x2e,0x85,
                 0x8c,0x82,0xe8,0x8f,0x03,0xa0,0x4c,0x71,0x73,
                 0x4e,0x86,0x38,0x03,0x2a,0x8e,0x86,0x22,0xf5,
                 0xf5,0x3f,0x6e,0xe7,0xde,0x86,0xd5,0x45,0x4b,
                 0xe8,0xfa,0x36,0x9a,0xd6,0xda,0xd3,0x4f,0x59,
                 0xaf,0x7d,0x13,0x01,0x15,0x73,0xfd,0x1f,0x6b,
                 0xa3,0x11},
        .data_len = 128,
        .mac_len = 12,
        .mac = {0x44,0x5a,0xa9,0x2b,0x03,0x2c,0x6b,0x65,0xb2,
                0x8a,0x65,0x41},
    }, {	// 11 (76)
        .key = {0x2a,0x43,0x2b,0x46,0x2e,0xbb,0x78,0x83,0x50,
                0x08,0xb4,0xaa,0x8a,0x92,0xb4,0x0f,0x6f,0xe9,
                0xdc,0x53,0xa9,0x63,0x35,0x2e,0xa5,0x07,0xc0,
                0x6c,0x8d,0xa9,0x0a,0x36},
        .key_len = 32,
        .data = {0xac,0x76,0xa7,0xdb,0x96,0x4e,0x9f,0xad,0x2f,
                 0x98,0xc1,0x8c,0x06,0xf9,0x29,0xf2,0x3b,0x62,
                 0x17,0xee,0x35,0xef,0x45,0x25,0x92,0x0f,0x77,
                 0x17,0x64,0xe6,0x53,0xa3,0x9a,0xef,0x73,0xcd,
                 0xbc,0xe6,0xb9,0xc0,0xdc,0xe5,0xe2,0x0f,0xc9,
                 0xcd,0x5e,0x40,0x85,0xe7,0x5f,0x8b,0xf9,0xcb,
                 0x31,0xdf,0xe8,0x81,0xc9,0x26,0x22,0xe7,0xa0,
                 0xca,0xfa,0x52,0xc2,0x78,0xf9,0x78,0x21,0x24,
                 0xd4,0x8e,0x30,0x4d,0x9c,0xad,0xad,0x82,0x35,
                 0x7a,0xbe,0x25,0x09,0x06,0x40,0x6f,0xfd,0xf3,
                 0x5c,0xb4,0xa5,0xd9,0x5b,0xe8,0xb3,0xe7,0xbb,
                 0x63,0xb6,0xce,0x82,0xe1,0x01,0xda,0xd2,0xcd,
                 0xe8,0x62,0xbe,0xbf,0x33,0x63,0x5c,0x43,0xcc,
                 0x68,0x1b,0xdc,0xbb,0xad,0x57,0x48,0x54,0x83,
                 0x2b,0x06},
        .data_len = 128,
        .mac_len = 12,
        .mac = {0x2f,0x8e,0x18,0xb7,0x5c,0xb3,0x74,0x02,0xd6,
                0xe8,0x73,0x55},
    }, {	// 12 (90)
        .key = {0x58,0x10,0x24,0x23,0xa4,0x16,0x8f,0xa6,0x0a,
                0x5a,0xa7,0xf7,0x90,0x92,0xd5,0x23,0x26,0xc9,
                0x8e,0x22,0xee,0x5f,0x3d,0xff,0xdb,0x52,0x7d,
                0x39,0x7d,0xbb,0x8c,0x68},
        .key_len = 32,
        .data = {0x48,0x0b,0xe7,0x58,0xa9,0xb7,0xba,0x9a,0xf0,
                 0x01,0xbf,0x21,0xdb,0x00,0xc4,0x51,0xcf,0xd6,
                 0x6f,0x06,0xc9,0xd8,0xd5,0xd6,0x98,0xef,0x47,
                 0x97,0x4a,0x3d,0x6f,0x21,0xe4,0x04,0x9d,0x55,
                 0x56,0xc4,0x5b,0x5f,0xad,0xa4,0x47,0x37,0x8b,
                 0x13,0x22,0x6e,0xd4,0xaf,0x24,0x27,0xab,0x66,
                 0x92,0x64,0x9d,0xdb,0x93,0x83,0x1b,0x0b,0x40,
                 0x08,0x2e,0x30,0xfa,0x9c,0x66,0xe6,0x00,0x56,
                 0x14,0x8c,0x40,0x3a,0xb8,0xed,0x6e,0xff,0xbd,
                 0x1f,0x54,0x16,0x64,0xac,0x69,0xe7,0xff,0xf0,
                 0xa4,0x5e,0x5f,0xc2,0x92,0xa6,0x8f,0x57,0xa7,
                 0x34,0xc3,0x62,0xd2,0x08,0x8b,0x80,0x53,0x2f,
                 0x4c,0xd4,0xd1,0x8d,0xf1,0xee,0xa7,0xd9,0xde,
                 0xf2,0x80,0xe9,0x25,0xf6,0x23,0x30,0xfd,0xab,
                 0x90,0x85},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x4c,0x41,0xbe,0xa8,0x23,0xee,0x67,0x91,0xe8,
                0x36,0x36,0xbf,0x75,0x2c,0x12,0x40},
    }, {	// 13 (91)
        .key = {0x81,0x6a,0xa4,0xc3,0xee,0x06,0x63,0x10,0xac,
                0x1e,0x66,0x66,0xcf,0x83,0x0c,0x37,0x53,0x55,
                0xc3,0xc8,0xba,0x18,0xcf,0xe1,0xf5,0x0a,0x48,
                0xc9,0x88,0xb4,0x62,0x72},
        .key_len = 32,
        .data = {0x22,0x02,0x48,0xf5,0xe6,0xd7,0xa4,0x93,0x35,
                 0xb3,0xf9,0x13,0x74,0xf1,0x8b,0xb8,0xb0,0xff,
                 0x5e,0x8b,0x9a,0x58,0x53,0xf3,0xcf,0xb2,0x93,
                 0x85,0x5d,0x78,0x30,0x1d,0x83,0x7a,0x0a,0x2e,
                 0xb9,0xe4,0xf0,0x56,0xf0,0x6c,0x08,0x36,0x1b,
                 0xd0,0x71,0x80,0xee,0x80,0x26,0x51,0xe6,0x97,
                 0x26,0xc2,0x89,0x10,0xd2,0xba,0xef,0x37,0x96,
                 0x06,0x81,0x5d,0xcb,0xab,0x01,0xd0,0xdc,0x7a,
                 0xcb,0x0b,0xa8,0xe6,0x5a,0x29,0x28,0x13,0x0d,
                 0xa0,0x52,0x2f,0x2b,0x2b,0x3d,0x05,0x26,0x08,
                 0x85,0xcf,0x1c,0x64,0xf1,0x4c,0xa3,0x14,0x53,
                 0x13,0xc6,0x85,0xb0,0x27,0x4b,0xf6,0xa1,0xcb,
                 0x38,0xe4,0xf9,0x98,0x95,0xc6,0xa8,0xcc,0x72,
                 0xfb,0xe0,0xe5,0x2c,0x01,0x76,0x6f,0xed,0xe7,
                 0x8a,0x1a},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x17,0xcb,0x2e,0x9e,0x98,0xb7,0x48,0xb5,0xae,
                0x0f,0x70,0x78,0xea,0x55,0x19,0xe5},
    }, {	// 14 (105)
        .key = {0x89,0x58,0x68,0xf1,0x96,0x95,0xc1,0xf5,0xa2,
                0x6d,0x8a,0xe3,0x39,0xc5,0x67,0xe5,0xab,0x43,
                0xb0,0xfc,0xc8,0x05,0x60,0x50,0xe9,0x92,0x2e,
                0xc5,0x30,0x10,0xf9,0xce},
        .key_len = 32,
        .data = {0x88,0x3e,0x6c,0xa2,0xb1,0x9e,0xf5,0x46,0x40,
                 0xbb,0x83,0x33,0xf8,0x5a,0x93,0x80,0xe1,0x72,
                 0x11,0xf6,0xee,0x3d,0x1d,0xc7,0xdc,0x8f,0x0e,
                 0x7c,0x5d,0x67,0xb7,0x30,0x76,0xc3,0xea,0xfc,
                 0x26,0xb9,0x3b,0xb2,0x48,0xc4,0x06,0xce,0xba,
                 0x5c,0xb4,0xa9,0xbf,0xc9,0x39,0xf0,0xa2,0x38,
                 0xe1,0x55,0x9d,0x0f,0x4d,0x84,0xf8,0x7e,0xb8,
                 0x59,0x75,0x56,0x80,0x50,0xec,0x1f,0xe1,0x3d,
                 0x33,0x65,0x03,0x3d,0x40,0x52,0x37,0xec,0x92,
                 0x82,0x7d,0xd8,0xcd,0x12,0x4b,0x36,0xa4,0xfa,
                 0x89,0xd4,0xfb,0x9d,0xe0,0x4f,0x4d,0x9f,0x34,
                 0x86,0x4c,0xf7,0x6f,0x4e,0xc8,0x45,0x81,0x68,
                 0xd2,0x65,0xa5,0xb0,0x21,0x44,0xe5,0x96,0xb5,
                 0xf2,0xe0,0xd2,0xb9,0xf9,0xcb,0x54,0xae,0xee,
                 0xb6,0x7a},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x37,0x4c,0x88,0xf4,0x48,0x0f,0x5e,0x8a,0xaa,
                0x9f,0x44,0x8b,0x77,0x75,0x57,0xc5,0x00,0x65,
                0xe9,0xac},
    }, {	// 15 (106)
        .key = {0x95,0x0f,0xb0,0xcd,0xe3,0x0f,0x34,0xf5,0x97,
                0xaf,0x5c,0xaa,0x2b,0x16,0xfc,0x86,0xa5,0xc3,
                0xef,0x06,0x5d,0x36,0xff,0xdd,0x06,0xec,0x04,
                0x8e,0xec,0x91,0x50,0x39},
        .key_len = 32,
        .data = {0xe4,0x63,0x62,0x65,0x06,0x14,0x4c,0xec,0xe5,
                 0x5d,0xfb,0x7a,0xa2,0x2e,0xb2,0x1e,0xa3,0xa4,
                 0x27,0x7d,0x89,0x2c,0x21,0x17,0x62,0xea,0x45,
                 0xcc,0x20,0x5c,0x2d,0x9e,0x4b,0x3a,0xbb,0xb8,
                 0xf2,0xa1,0xad,0xb0,0xe7,0x71,0x71,0x09,0x2c,
                 0xf4,0x3a,0xfc,0xa8,0xc0,0x53,0x77,0x1e,0xde,
                 0xb4,0x67,0x60,0x2b,0xd3,0x33,0xc0,0xff,0xbc,
                 0x88,0xc8,0x0d,0x64,0x5c,0x2b,0x8a,0x3a,0x2d,
                 0xfa,0x92,0x00,0x8a,0x1b,0xc7,0xd9,0xd5,0xf8,
                 0x3b,0xa3,0x47,0x74,0x90,0x86,0x34,0x23,0x5d,
                 0xcd,0x91,0xba,0xd4,0xf5,0xb3,0xc4,0xa2,0x04,
                 0x59,0x97,0x17,0x1d,0xed,0x87,0x87,0x50,0x07,
                 0x59,0xf0,0xb6,0x33,0xfb,0xdc,0xbe,0xf4,0x72,
                 0x89,0xc2,0x09,0x13,0x48,0xde,0xee,0xf6,0x23,
                 0x01,0xa6},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x8c,0x90,0x48,0x0e,0xa6,0x41,0x45,0x53,0xdf,
                0x17,0xe5,0x3c,0xf9,0x6d,0xcb,0x16,0x6b,0x94,
                0xbe,0x35},
    }, {	// 16 (120)
        .key = {0x26,0x8b,0x0e,0x1f,0x11,0x00,0x52,0xaa,0xa2,
                0xee,0xe3,0x27,0xe3,0x4a,0xb3,0x49,0x02,0x98,
                0x06,0xda,0xf7,0x02,0x30,0x68,0x67,0xa7,0xa0,
                0x3b,0xc8,0x35,0x1d,0x8a,0xc7,0xba,0x50,0xee,
                0xe6,0xb7,0x83,0x16,0x6a,0x77,0xa8,0xbd,0x74,
                0x9e,0x9d,0xd9,0x6e,0x05,0xae,0x15,0xa8,0xc5,
                0x5c,0x82,0x43,0x92,0x5c,0x89,0x4f,0x4b,0xe3,
                0x25},
        .key_len = 64,
        .data = {0x7c,0x88,0x1d,0xe0,0x03,0x88,0xa0,0x0f,0x8c,
                 0xee,0xa8,0x87,0xb8,0xe8,0x7e,0xf7,0xce,0xb2,
                 0x3e,0xa0,0x5d,0xad,0x95,0x06,0x23,0xb0,0xca,
                 0xeb,0x2e,0xa2,0xfb,0x7d,0x41,0x49,0xaa,0xcf,
                 0x79,0x5d,0x78,0x86,0x30,0xe1,0x2f,0xd5,0x22,
                 0xb3,0x06,0xab,0xce,0x61,0x21,0x2a,0x20,0x3e,
                 0x58,0x5c,0x4c,0xb5,0x39,0x21,0xfd,0xde,0x50,
                 0x6c,0xaf,0x4f,0xa6,0xaf,0x59,0x35,0x87,0x94,
                 0x50,0xa3,0x88,0xee,0x68,0x29,0xc9,0xef,0x5c,
                 0xa9,0x78,0x9b,0x70,0x66,0x96,0x7c,0x54,0x5e,
                 0xfe,0x98,0x4c,0xda,0xa3,0xa0,0x8e,0x43,0x19,
                 0x6a,0xeb,0x37,0x57,0xa1,0xb2,0xdc,0xbb,0xbc,
                 0xd2,0x74,0x4e,0x2c,0x3e,0x32,0x4a,0xda,0x96,
                 0x4c,0xd9,0xd0,0x03,0x52,0x20,0x36,0x63,0xbe,
                 0x7c,0x81},
        .data_len = 128,
        .mac_len = 10,
        .mac = {0xe0,0x6c,0x08,0x6d,0x34,0x34,0xd7,0x95,0x95,0xd3},
    }, {	// 17 (121)
        .key = {0x77,0xc1,0x92,0x47,0x22,0x53,0x68,0x5d,0x52,
                0xa6,0xfc,0x39,0x3b,0xb7,0xa9,0xd5,0xbd,0x73,
                0xf5,0xaf,0x2b,0x6e,0x74,0x20,0x50,0xd7,0xea,
                0xe9,0xb4,0xac,0xb0,0x0f,0x1b,0x2a,0x59,0xea,
                0x4f,0x88,0x94,0x78,0x1f,0xe4,0x54,0xf7,0xa8,
                0x7e,0x2f,0xb2,0xd3,0x24,0x04,0x1b,0x1f,0xed,
                0xe1,0x1a,0xa1,0x2a,0x24,0xa5,0x49,0x9a,0xe0,
                0x91},
        .key_len = 64,
        .data = {0x83,0x7d,0xc1,0x90,0xbf,0x0a,0x96,0xd9,0xc7,
                 0x87,0x9d,0x8d,0x99,0x8c,0x5c,0x21,0xa2,0x63,
                 0x47,0x51,0x80,0xbc,0x9c,0x70,0x0c,0xa2,0x8c,
                 0xfc,0x98,0xae,0x9b,0x75,0x75,0x7b,0x49,0x6f,
                 0xb9,0x59,0xf2,0xe7,0x3e,0x46,0xf3,0xd3,0xee,
                 0x1a,0x0e,0xfc,0x3e,0x01,0x10,0x10,0xf9,0x2e,
                 0xb0,0xf3,0x3f,0xce,0xbb,0x57,0xcd,0x3b,0x6e,
                 0x8c,0x7f,0x73,0x23,0x99,0x12,0xc8,0x31,0x8b,
                 0x2f,0xd9,0x0d,0x0d,0xa5,0xc0,0xb5,0x39,0xf7,
                 0x8d,0x4e,0xae,0x16,0xf4,0x0b,0xe3,0x6f,0x42,
                 0x52,0xbb,0x28,0x95,0x1a,0x59,0xa7,0x4d,0x98,
                 0x35,0x55,0xbe,0x1a,0x6f,0xa1,0x27,0x33,0x64,
                 0x47,0xe8,0x18,0x80,0xd2,0xef,0x4a,0x53,0x5f,
                 0x74,0x75,0xe6,0xa5,0xe6,0x98,0x4f,0x32,0x25,
                 0x67,0x83},
        .data_len = 128,
        .mac_len = 10,
        .mac = {0x2d,0x0f,0x6c,0x93,0x5a,0x06,0xd9,0xd4,0x8e,0x10},
    }, {	// 18 (135)
        .key = {0xb2,0x44,0xd3,0x05,0xbf,0xd5,0x34,0xde,0x7b,
                0x05,0xb6,0x6c,0xda,0x0b,0x7b,0xd3,0xc2,0x41,
                0x49,0x56,0xb5,0x36,0x46,0x11,0xb0,0xfe,0xff,
                0xea,0x53,0xcd,0xaf,0xc5,0x41,0xc5,0xbf,0xf7,
                0xca,0x0b,0x89,0xfd,0xc8,0x20,0x61,0x6f,0xc6,
                0x6f,0xd6,0x2f,0x68,0x22,0x35,0xe6,0x07,0x3a,
                0x4f,0xb1,0x9b,0xdf,0x7c,0x17,0xde,0xf4,0xe0,
                0x3f},
        .key_len = 64,
        .data = {0xf5,0x4c,0x5e,0x14,0xa2,0x9a,0xbb,0x69,0x9f,
                 0xea,0x35,0x04,0xf4,0xb9,0xa0,0x77,0xbd,0x40,
                 0xa4,0xdd,0x72,0xa6,0x1c,0xb5,0x6c,0x75,0xbd,
                 0xf0,0xa5,0x4b,0xf8,0x48,0xc0,0xd2,0x21,0xd4,
                 0x49,0xf1,0xd0,0xd9,0x3d,0x44,0x88,0xe4,0xcd,
                 0xca,0x96,0x15,0x5f,0xde,0x3c,0xbe,0xd6,0x69,
                 0x0f,0x2d,0x13,0x55,0x9e,0xc5,0xbb,0x45,0x54,
                 0x54,0x3b,0x83,0xa0,0xa0,0x0a,0x39,0x52,0x43,
                 0x2e,0xe5,0x49,0xb9,0x02,0x07,0x4b,0xb8,0x36,
                 0x1c,0x34,0xbf,0x17,0xd0,0x53,0xf2,0x11,0x70,
                 0x11,0x25,0x72,0x9e,0xd3,0x37,0x70,0x48,0x22,
                 0xa1,0x6e,0xdb,0x0a,0x4e,0x7b,0xb3,0xbf,0xae,
                 0x1c,0xd7,0x87,0x06,0x4b,0xe3,0xd3,0x0a,0xbf,
                 0x45,0xaf,0xad,0x6e,0xac,0x5d,0x38,0x51,0xbe,
                 0x3d,0x99},
        .data_len = 128,
        .mac_len = 12,
        .mac = {0xe6,0x85,0xc2,0x6a,0x4e,0xf7,0x66,0xa1,0xac,
                0x24,0x4b,0xf7},
    }, {	// 19 (136)
        .key = {0xf3,0xcb,0x2c,0xba,0xaf,0xe6,0x28,0x1e,0xbb,
                0x54,0x6a,0xf8,0x8c,0x05,0x2e,0x66,0x58,0xa5,
                0x84,0x07,0xcd,0x7b,0xa3,0x05,0x02,0x91,0x80,
                0x52,0xae,0x15,0x9f,0x31,0x98,0xff,0x29,0xf9,
                0x4e,0xf4,0x40,0x15,0x1a,0x6a,0x8f,0x50,0x32,
                0x0e,0x25,0x50,0x2f,0x62,0x83,0x5f,0xc0,0xab,
                0xf3,0x72,0xa0,0x0a,0x1c,0x63,0xc5,0xe9,0xd4,
                0x82},
        .key_len = 64,
        .data = {0x8f,0x63,0x60,0x70,0xd8,0xc5,0xc1,0xf9,0x79,
                 0x73,0x4a,0xe3,0x6a,0xcf,0xe6,0x3f,0x0c,0x08,
                 0x17,0x53,0x1a,0x3f,0x8d,0xe1,0xdd,0xe9,0xf7,
                 0xad,0xa0,0x75,0x19,0x39,0x64,0x2e,0x1e,0xd3,
                 0xd5,0x62,0x30,0xd1,0x7c,0xc4,0x47,0x1c,0x35,
                 0x0f,0x3e,0xeb,0xe4,0xec,0x2c,0xd1,0x64,0x16,
                 0xf1,0xfa,0xc0,0xbc,0x0f,0xb2,0xa6,0x27,0xbc,
                 0x26,0x18,0x9c,0x35,0x6f,0x65,0x84,0x54,0xcc,
                 0x58,0xca,0x65,0x2f,0xaf,0x85,0x36,0xfc,0xce,
                 0xd7,0x6d,0x0d,0xb5,0x14,0x1e,0xf9,0x30,0x27,
                 0x9d,0x96,0x4d,0x32,0x91,0xbc,0x13,0x75,0x4a,
                 0x4c,0x71,0x71,0x55,0x71,0x75,0x4d,0x4d,0x26,
                 0xbf,0x78,0xf3,0xf9,0x34,0x90,0x81,0x0e,0xf7,
                 0x83,0x3c,0x66,0x95,0xf4,0x49,0x61,0x7f,0xe0,
                 0xc1,0x82},
        .data_len = 128,
        .mac_len = 12,
        .mac = {0x3b,0xf0,0xf6,0xf4,0xac,0x75,0x7a,0xfb,0x9d,
                0xea,0xfd,0xb3},
    }, {	// 20 (150)
        .key = {0x8e,0xcc,0xd4,0x67,0xd8,0x75,0x83,0x9c,0xb4,
                0xb0,0xa0,0x17,0x0a,0x97,0x6f,0x60,0x56,0x87,
                0x68,0x59,0xfb,0x24,0x2f,0x69,0xd9,0x9d,0xc6,
                0xda,0x21,0x32,0x02,0x80,0x68,0xf3,0x3b,0x9c,
                0xfb,0xca,0x48,0xff,0x73,0xbb,0xaa,0x73,0x89,
                0x6b,0x08,0x56,0x2b,0xdf,0xdc,0x88,0xcf,0x87,
                0x6b,0x88,0x07,0x7b,0xfa,0xd9,0x55,0x04,0x3f,
                0xab},
        .key_len = 64,
        .data = {0xa6,0x7b,0x1d,0xc3,0x63,0x3d,0x30,0xc4,0xef,
                 0x2b,0xf3,0x18,0x5f,0xd4,0x48,0x65,0xd2,0xaf,
                 0x5e,0x72,0x01,0x5c,0xdf,0x8c,0x18,0x2e,0x6b,
                 0x28,0xc5,0xe7,0x46,0xc9,0x8e,0xc2,0x4d,0x24,
                 0x67,0xb7,0x2f,0x82,0x84,0xfa,0xd9,0x67,0x6c,
                 0xc5,0x32,0x71,0x4f,0x57,0x09,0x82,0x99,0x3d,
                 0x4b,0x22,0xc7,0xd0,0x7a,0x1e,0x79,0xff,0x5a,
                 0x75,0xc9,0x4e,0xee,0x75,0xdc,0x1f,0xa2,0x22,
                 0xb6,0x30,0xca,0xd7,0x53,0x66,0x4b,0x30,0xf3,
                 0xc9,0x98,0x26,0xb5,0xcf,0xe1,0x7c,0x67,0xdd,
                 0x87,0x5b,0x9d,0x0b,0xd2,0x39,0x00,0x28,0xe6,
                 0xff,0xe9,0xfe,0xf3,0x6a,0x2f,0xd6,0xad,0xb1,
                 0x3d,0x3f,0xfc,0x69,0x67,0x0c,0xf4,0xa6,0x7e,
                 0x9c,0x07,0x64,0xa1,0x5e,0x79,0x25,0x57,0x93,
                 0x15,0xdb},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xf3,0x5a,0x43,0x23,0xca,0xb7,0xad,0xe7,0x16,
                0x8c,0x8b,0x9f,0x72,0x76,0x74,0x4e },
    }, {	// 21 (151)
        .key = {0xb4,0x88,0x33,0x2a,0x10,0xf2,0xbc,0x7d,0x90,
                0x42,0xa1,0x93,0x3d,0xa8,0x5d,0xcc,0x89,0x25,
                0x04,0xbe,0x3e,0xa8,0xd5,0x7b,0xb5,0x78,0x0f,
                0x16,0x48,0xd1,0x07,0x63,0x09,0xd2,0x76,0xff,
                0xb5,0x97,0x17,0x90,0xe3,0xa2,0x72,0x4e,0x81,
                0x7f,0xf2,0xc3,0x81,0xa7,0x3e,0xce,0xd0,0xa6,
                0xc6,0xee,0x88,0x79,0x9c,0xbd,0x66,0x3a,0x86,
                0xbb},
        .key_len = 64,
        .data = {0xa9,0x17,0x4a,0x67,0x60,0x3a,0x4d,0x5f,0xba,
                 0xa8,0xcf,0xb5,0x62,0xf0,0x73,0x93,0xab,0xad,
                 0xbc,0x80,0xd1,0xb5,0x72,0x31,0x82,0x93,0x47,
                 0xa2,0x9c,0x38,0xba,0x66,0x39,0xed,0x3c,0x3c,
                 0xe9,0x8c,0x91,0xe2,0x3e,0xf0,0x7a,0x2e,0x8e,
                 0xaa,0x91,0x5a,0xf4,0xf5,0x74,0xa0,0x98,0xed,
                 0x25,0x06,0x30,0xfb,0xb1,0x7c,0xc7,0x94,0x10,
                 0x24,0xbd,0x23,0x4d,0xf1,0x10,0x43,0xe7,0x73,
                 0xd9,0x32,0x76,0xf1,0x1a,0x82,0x91,0xb9,0xb6,
                 0x12,0xf0,0xb4,0xc1,0x3d,0xce,0x3d,0xfa,0x51,
                 0x91,0x33,0x96,0x43,0xad,0x4d,0x40,0xa1,0xc6,
                 0xae,0x5d,0xc7,0x15,0xba,0x94,0x56,0x0c,0x27,
                 0x8e,0xe2,0x3d,0x57,0xfa,0xeb,0x78,0xe5,0xd5,
                 0x0f,0x33,0x7e,0xe8,0x7d,0x2f,0xf2,0x92,0xad,
                 0x59,0x8a},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x59,0xa1,0x16,0xa2,0x49,0xea,0xca,0xff,0xc5,
                0x44,0x98,0x95,0x77,0x87,0xf8,0xf4},
    }, { // 22 (165)
        .key = {0xb9,0x57,0x5f,0x4d,0x5e,0xcc,0x0f,0x4f,0x62,
                0xe4,0xa0,0x55,0x6b,0xb8,0x94,0x64,0xba,0x97,
                0xd4,0x57,0x0e,0x55,0xac,0xd4,0xc5,0xe5,0x17,
                0x7e,0x45,0x2a,0x3d,0x6c,0x9a,0x0b,0x3a,0xdb,
                0x60,0xc6,0x21,0x1f,0xe4,0x86,0x40,0xe0,0x86,
                0x37,0xa6,0x82,0x62,0x99,0xe3,0xe5,0x2f,0x93,
                0x0f,0x4f,0x66,0xcb,0x0e,0xa6,0xa7,0x73,0x11,
                0xe3},
        .key_len = 64,
        .data = {0x8c,0x83,0x87,0xf4,0xae,0x2c,0xa1,0xa6,0xdd,
                 0x13,0xd2,0x9e,0x93,0x58,0x0b,0x1c,0xdf,0x62,
                 0x68,0xda,0x66,0xcf,0x58,0x9c,0xa8,0xb1,0xff,
                 0x08,0x84,0xf7,0xd8,0xb8,0xfe,0x29,0x9f,0x8e,
                 0x41,0x59,0x6e,0x47,0xe0,0x56,0x26,0x53,0x61,
                 0x22,0x10,0xe4,0xfc,0xa6,0xc4,0x46,0xa0,0xa5,
                 0x4a,0x6e,0x37,0xef,0x80,0xd5,0x2b,0xd7,0xbb,
                 0x87,0x29,0xe6,0xb1,0x76,0x25,0xd1,0x97,0x15,
                 0x9e,0xa9,0x86,0x22,0x23,0x52,0x23,0xc3,0x16,
                 0x36,0x7f,0xd5,0xb0,0x3a,0x3c,0x81,0x45,0xf2,
                 0xf2,0x10,0xc9,0x10,0xd0,0x00,0x94,0x23,0x87,
                 0x57,0x62,0x7e,0x63,0x37,0x9e,0x75,0xbb,0xb3,
                 0xe0,0xd0,0x8c,0xe1,0xb4,0x79,0x61,0x30,0x9d,
                 0x78,0x76,0xfc,0x59,0x21,0x1c,0x60,0x67,0x8c,
                 0x5f,0x4c},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x15,0xaf,0x23,0x33,0x16,0x48,0x17,0x14,0x99,
                0xb5,0x80,0x42,0xdb,0xe7,0xb2,0xd5,0xdf,0x72,
                0xd1,0x52},
    }, {	// 23 (166)
        .key = {0xd2,0x91,0xad,0xbf,0x05,0xb0,0x65,0x96,0xc2,
                0xf3,0x6f,0x41,0xa8,0xcd,0x80,0x70,0x37,0x0c,
                0x42,0xf6,0x87,0xb8,0xa6,0xcc,0x3a,0x3e,0x7b,
                0x59,0xaf,0xcd,0x40,0xf0,0x78,0x01,0x36,0x9b,
                0x0f,0xbf,0xba,0x17,0xc4,0x60,0xd2,0x1f,0xfa,
                0x11,0x06,0xee,0x93,0x79,0x71,0xff,0xa9,0x9d,
                0x17,0x17,0x7f,0x01,0x79,0x85,0xb7,0x10,0x67,
                0xa8},
        .key_len = 64,
        .data = {0x50,0xbc,0xdf,0x31,0x38,0x9e,0xad,0xac,0x5b,
                 0xb8,0x19,0x7e,0xe9,0x49,0xf2,0x86,0x4e,0xde,
                 0x28,0x4c,0x07,0xd0,0x39,0xa0,0xb4,0x0e,0xed,
                 0x7e,0x6f,0x1c,0x43,0x35,0x5d,0x5c,0xab,0xc8,
                 0x82,0x8d,0x75,0x95,0xda,0x91,0x8a,0x34,0xa5,
                 0x73,0x5a,0xa2,0x02,0xa8,0x15,0x9f,0xbf,0x95,
                 0x1e,0x54,0x70,0x52,0xbd,0x39,0xbe,0xae,0x14,
                 0x36,0x02,0x73,0x54,0x09,0x13,0xeb,0x30,0xe7,
                 0x5b,0xa2,0x92,0x66,0x31,0x6e,0x8d,0x9a,0x63,
                 0xad,0x94,0x7e,0x11,0xce,0xe9,0x96,0xc2,0x13,
                 0x57,0xd3,0xb1,0x94,0x24,0xb7,0x68,0x88,0x42,
                 0xb9,0x90,0xc0,0xc5,0xeb,0x08,0x74,0x9a,0xda,
                 0x34,0x42,0x75,0xb6,0x98,0x74,0x0b,0xb3,0xa5,
                 0x82,0x82,0xae,0xd2,0xd7,0x25,0x14,0xef,0xd8,
                 0x5d,0x00},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x5f,0x7a,0x57,0xd4,0x2e,0x3e,0xbb,0xcb,0x85,
                0xb0,0x85,0x65,0x30,0x4d,0xab,0x94,0x1d,0x62,
                0x34,0xf3},
    }, {	// 24 (180)
        .key = {0x15,0x0d,0x3a,0xa3,0x09,0xa3,0x66,0x9a,0xf9,
                0x9a,0x70,0xf2,0xce,0xc5,0x2d,0x3d,0xa1,0x6b,
                0x1c,0x13,0x7f,0xf7,0x46,0x62,0x69,0xf2,0x68,
                0x05,0x9f,0x2f,0x54,0x98,0x1f,0x45,0x95,0x8b,
                0x68,0x42,0x52,0x76,0x83,0x9e,0x75,0xac,0x44,
                0x6e,0x0b,0x13,0xce,0xda,0xee,0x33,0x55,0xd1,
                0xa2,0x8c,0x28,0xfc,0x7e,0x2d,0xee,0xf0,0x0c,
                0x82,0x2f,0xa7,0xb2,0x6e,0x17,0x31},
        .key_len = 70,
        .data = {0x07,0x35,0x5a,0xc8,0x18,0xce,0x6b,0x46,0xd3,
                 0x41,0x63,0xae,0xec,0x45,0xab,0x17,0x2d,0x4b,
                 0x85,0x0b,0x0d,0xbb,0x42,0xe6,0x83,0x81,0xb6,
                 0x7f,0x1c,0xc8,0xe9,0x0a,0x4c,0x05,0x0f,0x3d,
                 0x01,0x38,0xba,0xb2,0x7e,0x6f,0x4f,0x8d,0x67,
                 0x8b,0xb6,0x5e,0x18,0x46,0x56,0x49,0x3b,0x75,
                 0x41,0x64,0x9a,0x8b,0xab,0x60,0x31,0x5f,0xa1,
                 0x6c,0x88,0x2f,0xf8,0x56,0x40,0xe4,0x83,0xf3,
                 0xeb,0x97,0x89,0xc2,0x21,0x55,0x75,0xcc,0xd0,
                 0x1f,0xd0,0xce,0xd3,0x35,0x6d,0x9a,0xc6,0x95,
                 0xe3,0xbb,0x19,0xbe,0x40,0x58,0x64,0xb9,0xfc,
                 0x5b,0xfa,0x5a,0x2c,0xd1,0xc1,0xc4,0xf8,0x94,
                 0x41,0x2b,0x4f,0x28,0xfa,0xde,0xda,0xe4,0xfb,
                 0x84,0x2e,0x52,0xb0,0xa5,0x45,0xd8,0xfc,0x6d,
                 0x2f,0x97},
        .data_len = 128,
        .mac_len = 10,
        .mac = {0xc7,0x3d,0x3c,0xf2,0xbd,0x6c,0x5c,0x9d,0xcb,0x91},
    }, {	// 25 (181)
        .key = {0xc9,0xc8,0xb8,0x91,0xb8,0x25,0x67,0x75,0x7d,
                0xbf,0x1a,0x15,0xb3,0x17,0x62,0x8d,0x98,0xc4,
                0x86,0xdb,0xbe,0x5e,0xd4,0xe6,0x04,0x9a,0x35,
                0xbf,0xc5,0xb6,0x04,0x26,0x4f,0x18,0x20,0x50,
                0x97,0x32,0x40,0xe7,0x2b,0xa8,0x87,0x53,0x67,
                0xb5,0x59,0x38,0xec,0xcb,0x6c,0x3f,0x4e,0x79,
                0x22,0x1a,0x0d,0x92,0x16,0xc2,0xc7,0x8c,0xf4,
                0x03,0xab,0x26,0x8f,0x3b,0x31,0x4d},
        .key_len = 70,
        .data = {0x17,0x92,0x59,0x52,0xaf,0x30,0x95,0x9b,0x1a,
                 0x5a,0x13,0x6f,0xf1,0x1b,0x3d,0xe1,0x0d,0xb6,
                 0xe4,0xce,0xe1,0x9f,0x31,0x08,0x0d,0xcb,0xde,
                 0xb4,0x31,0x29,0xa5,0xf1,0xff,0x71,0xf9,0xbb,
                 0x95,0x1c,0xf5,0x0e,0x09,0xb3,0x92,0x4e,0x45,
                 0x4d,0x1c,0xe6,0x15,0x54,0xe7,0x30,0x7e,0x87,
                 0x3e,0x95,0x52,0x45,0x9c,0xf5,0x01,0x08,0x1f,
                 0x48,0xb2,0x30,0x39,0x86,0x92,0x02,0xa9,0xc5,
                 0x6c,0xf0,0xa9,0xa1,0x7b,0x1a,0x69,0xe1,0x7c,
                 0x16,0xbd,0x58,0x06,0xec,0x12,0x08,0x1e,0x65,
                 0xa7,0x8e,0x07,0x86,0xfa,0xba,0x57,0x57,0x80,
                 0x7d,0x50,0xe9,0x98,0x08,0x6c,0x96,0xc2,0x32,
                 0x3a,0x8b,0x0c,0x1a,0x69,0x84,0xce,0x0e,0x22,
                 0xd7,0x97,0xac,0x9c,0xb4,0x67,0x47,0xea,0xab,
                 0x1f,0x8d},
        .data_len = 128,
        .mac_len = 10,
        .mac = {0x3b,0x89,0xbc,0x8d,0x9f,0x3f,0xbe,0xdb,0x86,0xa8},
    }, {	// 26 (195)
        .key = {0x08,0x6d,0x40,0xb5,0xbb,0xe7,0x5d,0xfa,0x59,
                0x05,0x54,0x5f,0x83,0xbc,0xd5,0x2d,0x71,0x2f,
                0x09,0x2f,0xce,0x2c,0x0f,0x5c,0xc9,0xfa,0xac,
                0xb5,0x69,0x52,0x3e,0x71,0x20,0xab,0xf2,0x58,
                0xa4,0xbb,0x37,0x6d,0xfa,0x3a,0x73,0xcf,0xd3,
                0xe9,0xf4,0xe1,0x1c,0xd3,0x29,0xa9,0xd1,0xd2,
                0x12,0x76,0x12,0x56,0xf5,0xc6,0x78,0x62,0x53,
                0x66,0xa9,0xd7,0x1a,0xdb,0x2a,0xf5},
        .key_len = 70,
        .data = {0x33,0xfc,0xb8,0xef,0xf4,0x17,0x86,0x63,0x44,
                 0x63,0x2d,0x0f,0x9e,0x81,0x98,0xc4,0xdb,0xee,
                 0x1c,0x13,0x9e,0xda,0xfe,0xbd,0xef,0x37,0x35,
                 0x6b,0x26,0x10,0x72,0x9f,0x0b,0x1c,0x5e,0xeb,
                 0x3b,0x93,0x22,0x61,0xce,0x40,0x2d,0x4a,0x36,
                 0xd8,0x31,0x1b,0x6a,0x8a,0x6f,0xa4,0x45,0xd7,
                 0x35,0x8b,0x28,0xa4,0xa5,0xf9,0xe7,0x8d,0xb7,
                 0x93,0xe3,0x7d,0x82,0xac,0x73,0x7b,0xb7,0xb8,
                 0x89,0xc7,0x6e,0x04,0x92,0x26,0x25,0xa5,0x9d,
                 0x7a,0x05,0xaf,0xc0,0x95,0x68,0xa7,0xb7,0x4f,
                 0x99,0x3a,0xcf,0xd6,0xda,0x2e,0x03,0x46,0xac,
                 0x9a,0x64,0x7a,0x4a,0x52,0xbe,0x21,0x77,0xa6,
                 0x78,0x14,0x79,0x4c,0xbc,0xe7,0x66,0x9a,0xd8,
                 0xbd,0x9e,0xf8,0xe4,0x61,0x99,0x96,0xa5,0x93,
                 0xe3,0x5a},
        .data_len = 128,
        .mac_len = 12,
        .mac = {0x14,0xad,0x91,0x5c,0x81,0x90,0x56,0x7f,0x88,
                0x91,0x60,0xf9},
    }, {	// 27 (196)
        .key = {0x57,0x44,0x61,0x8f,0xe8,0xe5,0xc1,0xe4,0xca,
                0xd9,0x5c,0xf4,0x35,0x05,0xcc,0x03,0x2d,0xf1,
                0xcf,0xe5,0x04,0x34,0xed,0x13,0x20,0x2d,0x5b,
                0xfe,0xfe,0xf4,0x20,0xa3,0x77,0x90,0x76,0x60,
                0x42,0x6b,0x73,0x06,0xbb,0x03,0xe8,0x2f,0xe2,
                0xe1,0x8a,0xd2,0xa7,0xcf,0x4f,0x14,0x65,0x46,
                0x1b,0x61,0xac,0x26,0x9c,0xbc,0x43,0xa9,0x72,
                0x53,0x6d,0x9a,0x94,0x57,0x6c,0xc2},
        .key_len = 70,
        .data = {0x90,0xa0,0x2b,0xc5,0xf2,0x6d,0x2c,0xcc,0x03,
                 0x0b,0x15,0x03,0xc6,0xc7,0x12,0xb8,0xe6,0xef,
                 0x4b,0x41,0xec,0x33,0xb8,0x87,0xb4,0x51,0x37,
                 0xc1,0x22,0xf2,0xdc,0x82,0x11,0xce,0x88,0xf6,
                 0x8c,0x17,0xbd,0x68,0x41,0x15,0xb0,0x08,0x32,
                 0x0e,0xa0,0xec,0xae,0x68,0x67,0x54,0x80,0x11,
                 0x4f,0x32,0x66,0x1f,0x26,0xea,0xc5,0xb4,0x95,
                 0x56,0x9a,0x25,0xad,0x0d,0xb4,0x5b,0xc3,0xe5,
                 0x21,0x79,0x7e,0xb6,0xe6,0xbe,0x2e,0x61,0xf3,
                 0xae,0x5f,0x11,0x55,0x6c,0xaf,0xc1,0xae,0x6b,
                 0xdc,0xff,0xe2,0x45,0x21,0xef,0x14,0xeb,0xc3,
                 0x92,0xd1,0xff,0xe7,0x48,0x8a,0x7e,0xa6,0x94,
                 0x48,0xa2,0x63,0x20,0x9b,0x07,0x5c,0x01,0xd3,
                 0x0c,0x80,0x3b,0x73,0x7c,0x81,0x88,0xe3,0x6e,
                 0x29,0x55},
        .data_len = 128,
        .mac_len = 12,
        .mac = {0x43,0xbf,0x10,0x01,0xad,0x1f,0x5c,0x5a,0xdf,
                0x0f,0x59,0xc2},
    }, {	// 28 (211)
        .key = {0x4a,0x25,0xe3,0xa8,0x8e,0xae,0x86,0x48,0x51,
                0xb4,0xc6,0xd0,0x1c,0x6b,0x98,0xb7,0x99,0xa7,
                0x0f,0x0c,0xa4,0x9f,0x18,0x60,0xa4,0xf1,0x67,
                0xdf,0x1c,0xe7,0xb1,0xc0,0x7d,0xf9,0x1c,0xe0,
                0x3f,0x93,0xf4,0xa9,0x2f,0x18,0x9f,0x39,0x0b,
                0x26,0xd3,0xc0,0x4c,0x1c,0x06,0x2a,0x43,0xd9,
                0x26,0xff,0x67,0xc7,0x8b,0x87,0xee,0x19,0x2a,
                0x31,0x9a,0x50,0x0b,0x35,0xd6,0x04},
        .key_len = 70,
        .data = {0x72,0x72,0xef,0xf0,0xb2,0x89,0x64,0xa1,0xaa,
                 0xbf,0xa0,0x8f,0x37,0x52,0x7a,0x86,0x07,0x04,
                 0x3f,0xed,0xf3,0x1b,0xa6,0xee,0x8f,0xad,0x05,
                 0xd8,0xff,0x1a,0xc4,0xc1,0x0c,0xda,0x12,0x6f,
                 0x77,0x79,0xd8,0x79,0x8c,0xdf,0xeb,0xa9,0xfb,
                 0xd5,0x86,0xa5,0xe4,0xc5,0xf7,0xce,0x31,0xc1,
                 0x98,0x69,0x28,0xc7,0x01,0xfd,0x40,0x44,0x7c,
                 0xfb,0x34,0xd6,0xba,0xa4,0x57,0x56,0xc4,0x28,
                 0x27,0x16,0x33,0x0b,0x24,0x67,0xa4,0xcd,0xe3,
                 0x5f,0x67,0xca,0x5e,0xd9,0x77,0x5f,0x8e,0xbc,
                 0xaf,0x4e,0x3c,0x81,0x3a,0x64,0x14,0xef,0x4c,
                 0x59,0xfb,0x29,0x0f,0xf7,0xa2,0xeb,0xe1,0x7e,
                 0x5b,0x11,0xbc,0x48,0x2c,0x59,0xf5,0xa9,0x22,
                 0x69,0x2a,0x19,0xe8,0x14,0x76,0x95,0x98,0xd9,
                 0xe6,0x42},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x76,0x12,0x2c,0x55,0x82,0xfe,0xa3,0xb4,0xf5,
                0x91,0x81,0xcb,0x1d,0x83,0xa5,0xee},
    }, {	// 29 (212)
        .key = {0x13,0xe8,0xb6,0x56,0x8b,0x1d,0x83,0xee,0x06,
                0x23,0x52,0x23,0xca,0xf6,0xbe,0x6e,0x76,0x89,
                0x7f,0xfc,0x95,0x0a,0x9a,0x0f,0x74,0x68,0xd5,
                0xa2,0x31,0x13,0x6e,0x4c,0x15,0x03,0x0c,0x66,
                0x23,0xfb,0xf6,0x70,0xf1,0x0f,0x83,0xb1,0xb7,
                0x64,0xd2,0x1e,0xa6,0x37,0xba,0x7d,0x7b,0x20,
                0x04,0xca,0x53,0x98,0xd8,0xda,0xc1,0xba,0x76,
                0x3e,0x1e,0x46,0x27,0x6a,0x20,0xeb},
        .key_len = 70,
        .data = {0xc2,0xc1,0xad,0x60,0x4e,0x21,0xc2,0xc8,0x69,
                 0x19,0x3d,0x67,0x97,0xae,0x65,0x7e,0xe7,0x40,
                 0x64,0x9c,0x78,0x05,0xee,0xb8,0x3c,0xb6,0x23,
                 0x7d,0xfc,0x88,0xb7,0xe5,0x9d,0x5e,0x50,0x09,
                 0xa1,0x3d,0x2f,0x38,0xf1,0x00,0x13,0x46,0xd9,
                 0x4d,0x5a,0x26,0x54,0xc7,0x6a,0xbb,0x8a,0x85,
                 0x4f,0xec,0x97,0xc4,0xa5,0xf7,0x8e,0xd8,0xb9,
                 0x07,0xbd,0x69,0xeb,0x08,0x33,0xdb,0x57,0xba,
                 0x80,0x0e,0xb4,0x04,0xbc,0x48,0x7b,0x8c,0xcb,
                 0x6f,0x4c,0x84,0xde,0x7c,0x8f,0xc7,0x3d,0x2c,
                 0x57,0x24,0x45,0xf8,0x8b,0xf9,0xac,0x48,0x47,
                 0x04,0x0d,0xe4,0x80,0x77,0xa0,0xab,0xe7,0x4a,
                 0x48,0x87,0x10,0xd5,0xd4,0xa0,0xd4,0x9e,0x7e,
                 0xd0,0xf4,0x70,0xb8,0x58,0xfe,0xad,0x29,0xd1,
                 0x75,0xe4},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x87,0xae,0x09,0x52,0x13,0x2a,0x3b,0x05,0x83,
                0x31,0x79,0x97,0xe5,0x90,0x7a,0xe4},
    }, {	// 30 (225)
        .key = {0xf0,0xda,0xe6,0xd8,0x75,0x30,0x76,0xb1,0x89,
                0x5c,0x01,0x26,0x2c,0xa9,0xb5,0x76,0x33,0xeb,
                0x28,0xb3,0xf9,0x63,0xa7,0xc7,0x52,0xe2,0xcb,
                0xb4,0xc0,0x31,0x4c,0x20,0xea,0xb1,0x1a,0x10,
                0x49,0x3f,0xaa,0xf4,0x25,0x5a,0x8e,0xe4,0xc0,
                0x88,0x49,0x29,0xd1,0xf5,0x61,0xff,0x33,0x5e,
                0xb6,0x99,0xdf,0x2d,0x11,0x66,0x18,0xe6,0x00,
                0x93,0xe5,0xc1,0xe2,0xd1,0xc4,0x99},
        .key_len = 70,
        .data = {0x61,0xcb,0x9e,0x1f,0x1e,0x4b,0x3a,0x3b,0x3b,
                 0xdf,0xf8,0xcd,0x5f,0x24,0x56,0x6b,0x98,0x7f,
                 0x75,0xc8,0xa0,0x53,0x77,0x85,0x5f,0x77,0x2b,
                 0x49,0xb0,0xe7,0xec,0x13,0x68,0xb9,0xc6,0xcf,
                 0x95,0x53,0xdb,0x28,0x03,0xdc,0x05,0x9e,0x05,
                 0xf0,0xbd,0xd8,0x71,0x98,0x3c,0x3b,0xed,0x79,
                 0xdf,0xbb,0x69,0x4b,0xd0,0xf1,0xed,0x8d,0xe3,
                 0x6e,0x95,0x77,0xbe,0x50,0xda,0x31,0x3d,0x13,
                 0x12,0x42,0x15,0xa9,0x3a,0x4b,0xb7,0xcc,0xf4,
                 0xf5,0x77,0x93,0xcc,0x28,0xed,0x43,0xbf,0x7e,
                 0x9b,0x68,0xfe,0xf7,0xd1,0x25,0xef,0xee,0xce,
                 0xc9,0x75,0x4b,0x28,0xa2,0x71,0xfb,0x6e,0x16,
                 0x89,0x9d,0x0b,0xef,0x28,0x7e,0x6d,0xf7,0xc5,
                 0xc8,0x67,0xc5,0x69,0xf6,0xd4,0xd6,0x6b,0x8b,
                 0x7e,0xe0},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x62,0xac,0x95,0x6a,0xda,0x19,0xf0,0x4b,0xe5,
                0x0c,0x23,0xf2,0x32,0x8a,0x32,0x47,0x7c,0xd5,
                0x8f,0xb9},
    }, {	// 31 (226)
        .key = {0x65,0xaf,0x1f,0x17,0xcd,0x7f,0xda,0xa5,0x23,
                0xb9,0xb7,0xa9,0x82,0x9d,0x49,0x7c,0xac,0x73,
                0x03,0xd4,0x50,0xc5,0x9e,0x98,0x88,0xcb,0xba,
                0xf3,0xa6,0x27,0xc8,0xa8,0x30,0xd3,0x27,0xa5,
                0x29,0x57,0x8d,0xda,0x92,0x3f,0xa9,0x4b,0x31,
                0xcc,0x07,0x64,0x91,0xea,0x33,0x8d,0x4a,0x62,
                0x21,0xff,0x82,0x51,0xcc,0xd6,0xb4,0xd9,0x1e,
                0x67,0xb1,0x16,0x10,0xd3,0xe4,0x53},
        .key_len = 70,
        .data = {0x9a,0xb4,0x66,0x7b,0x2d,0xf7,0xeb,0x4b,0xe8,
                 0x86,0x3a,0xa5,0x3e,0x9b,0xf9,0xaf,0x8b,0xae,
                 0x0f,0xc0,0x9d,0xe9,0x4f,0x73,0x73,0xdc,0x56,
                 0xfa,0x44,0x72,0xb6,0xb5,0xc4,0x23,0x54,0x03,
                 0xa2,0x6c,0x0e,0x59,0x55,0x7c,0xa1,0x91,0x18,
                 0x31,0xca,0x84,0x33,0x42,0xac,0xda,0x7d,0xbe,
                 0x72,0x21,0x1f,0xb5,0x35,0x1d,0x9a,0x34,0x20,
                 0x5f,0x0c,0x77,0xd2,0x19,0xaf,0x5b,0x03,0x31,
                 0xa2,0x12,0x6b,0x94,0xec,0x1a,0xdf,0xcd,0xbe,
                 0x70,0xbe,0xd6,0xf8,0x01,0x8b,0x2e,0xef,0x61,
                 0xdb,0x2b,0x6d,0xbf,0x72,0x92,0xfa,0x19,0xa9,
                 0x65,0x5a,0xac,0x13,0xfc,0x57,0xaf,0x5f,0x57,
                 0xc1,0x40,0x80,0xb3,0xb2,0x9f,0x0c,0x5b,0x16,
                 0x9a,0xe2,0xc1,0x6b,0x48,0x10,0xcd,0xc6,0xfa,
                 0xf4,0x75},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0xa2,0x79,0xd0,0x55,0xe2,0xd7,0x33,0x06,0xa8,
                0x18,0x73,0x44,0xfc,0x32,0xcb,0x0b,0x5b,0x80,
                0xcd,0x35},
    }, {	// 32 (240)
        .key = {0x13,0xfb,0x1e,0xd6,0x38,0x9f,0x32,0xd1,0xde,
                0x31,0x39,0xcb,0x04,0xbc,0xdd,0x53,0x52,0x5c,
                0x98,0x89,0xb8,0x53,0x79,0xd3,0x53,0x5a,0x25,
                0xd2,0x90,0x35,0x1c,0x95,0x93,0x8a,0x3d,0x0c,
                0xda,0xf3,0x8d,0xbf,0x1d,0x52,0x34,0xbf,0x79,
                0x65,0xc8,0xdd,0xce,0x9a,0xce,0x1b,0x66,0x24,
                0x7e,0x60,0xd7,0x4e,0xc7,0x70,0x2a,0x0f,0x93,
                0x1a,0x3c,0xdf,0x4c,0xb4,0x65,0xca,0x9f,0xc4,
                0x58,0xc3,0x80,0x00,0x4a,0x3a,0x6e,0x79},
        .key_len = 80,
        .data = {0x0c,0x36,0xca,0x43,0xe7,0xc1,0x13,0xed,0x9f,
                 0xb7,0x16,0x70,0xb3,0xea,0x73,0xbf,0xd6,0x92,
                 0x8c,0x83,0x9f,0x36,0xdb,0x1a,0x82,0xd0,0x8a,
                 0xe0,0xff,0x2c,0x3d,0xae,0x19,0x91,0x33,0xa1,
                 0x0a,0xa3,0x8d,0x1d,0x35,0x88,0xed,0x11,0x5c,
                 0x4a,0x43,0x7c,0x13,0x7c,0xe4,0x30,0x74,0x21,
                 0xdd,0xd6,0x15,0xc9,0x86,0x32,0x37,0xfd,0x5a,
                 0xa8,0x40,0xdd,0x05,0xff,0x6c,0x08,0xbf,0x66,
                 0xbf,0xbc,0xd9,0xb4,0x3e,0x3f,0x95,0xf4,0x5e,
                 0x7d,0x3b,0x21,0xbd,0xf2,0x69,0x2e,0x10,0xca,
                 0xab,0x49,0x5c,0x47,0x4b,0x61,0x6a,0x64,0x6b,
                 0xe6,0x75,0xb8,0x50,0xd0,0x25,0x9c,0x01,0xe2,
                 0xc1,0x90,0x11,0x30,0xa0,0xdb,0xb9,0xdf,0xe0,
                 0x72,0x2a,0x2c,0x5b,0x1b,0x20,0xaf,0xd7,0xd2,
                 0xbb,0xe1},
        .data_len = 128,
        .mac_len = 10,
        .mac = {0x76,0x53,0xdc,0x1c,0xa2,0xb7,0x0f,0x05,0x86,0x14},
    }, {	// 33 (241)
        .key = {0x5c,0xf5,0x9e,0x34,0xf1,0xae,0x4e,0xd7,0x32,
                0xa9,0x5c,0xee,0x65,0xeb,0x49,0x4c,0x1f,0x7e,
                0x89,0xe1,0xa2,0x72,0x7c,0xde,0x68,0x22,0x9f,
                0x1a,0x00,0xb9,0x04,0xb5,0x19,0xf4,0xff,0xfb,
                0xdd,0x29,0x23,0x8b,0x80,0x88,0x6c,0xb8,0x18,
                0xa1,0xbe,0x2f,0xaf,0x26,0x8e,0xda,0x96,0xf2,
                0xdf,0x05,0xfd,0x4b,0x71,0xc0,0xc1,0x64,0x35,
                0x84,0x85,0x26,0x03,0x19,0x04,0x30,0x8f,0xb6,
                0xa5,0x1d,0x9a,0x6b,0x51,0x05,0x65,0xbc },
        .key_len = 80,
        .data = {0xab,0x5d,0xa4,0xa6,0x4f,0xbb,0xf3,0xc6,0x0f,
                 0x5a,0xb1,0xf7,0x77,0x6e,0xd6,0xa5,0x57,0x51,
                 0xe3,0x9a,0x5e,0xc8,0x19,0x67,0xea,0x88,0xe9,
                 0x06,0x1f,0xf9,0xad,0xbd,0x37,0x39,0x95,0x45,
                 0x18,0x64,0xe4,0x2c,0x2c,0x13,0x5c,0x78,0x6d,
                 0x22,0xf6,0x8d,0xbf,0xb7,0xd7,0x51,0x83,0x7f,
                 0x80,0x8d,0x69,0x3b,0x45,0x97,0x85,0x7c,0x00,
                 0x2e,0xa6,0xaa,0x06,0xa5,0xe3,0x4b,0x5a,0x44,
                 0x76,0x82,0x21,0xeb,0xce,0xd6,0x56,0xf8,0xdf,
                 0x35,0xbf,0x6b,0xbd,0x39,0x20,0x48,0x69,0xaa,
                 0xae,0x3d,0xea,0x43,0xc6,0x85,0xa0,0xb9,0xdf,
                 0x0c,0xd6,0xf9,0xbe,0xd4,0x96,0xb1,0xe9,0x97,
                 0xc1,0x13,0x5d,0xae,0x5f,0xd6,0x83,0x31,0x33,
                 0x7d,0x61,0x60,0x92,0xdb,0x0d,0x41,0x76,0xd7,
                 0x68,0x8b},
        .data_len = 128,
        .mac_len = 10,
        .mac =  {0x8d,0xb9,0x4b,0xaa,0xaf,0x03,0xa5,0x1a,0xcc,0x87},
    }, { 	// 34 (255)
        .key = {0x24,0xd8,0x93,0x8c,0x16,0x44,0xcb,0xb0,0x80,
                0xc4,0x50,0x55,0x39,0xe4,0x4c,0x8a,0x61,0x56,
                0x7c,0xa7,0x44,0x43,0x36,0x3b,0x80,0xdf,0xaa,
                0x46,0x6b,0x40,0x68,0xa9,0xaf,0x70,0x22,0xda,
                0x37,0xc1,0xb3,0xdc,0x4f,0x60,0x61,0x6f,0x06,
                0x2d,0x5f,0x84,0xd7,0xca,0x96,0xf3,0x89,0xf2,
                0xa6,0x70,0x54,0x0d,0x27,0xbc,0x45,0x01,0x34,
                0x18,0xe4,0x4a,0x2a,0xff,0x13,0x4d,0xad,0x14,
                0x39,0xe9,0xec,0x5a,0xa0,0x50,0x26,0xa3},
        .key_len = 80,
        .data = {0xc0,0xb1,0x84,0xc7,0xb9,0xe4,0xcb,0x8d,0xd1,
                 0x9a,0xf3,0x77,0x30,0x65,0x16,0xc5,0x63,0xb3,
                 0xb8,0x78,0xba,0xa2,0x50,0xc1,0xee,0x16,0x05,
                 0xb9,0x07,0x08,0xb5,0x52,0x7d,0x21,0x3b,0x8e,
                 0x9e,0x87,0xf2,0xef,0x2f,0xf7,0x75,0x2e,0x56,
                 0x14,0xa9,0x30,0xb8,0xfe,0xfe,0x35,0xde,0x27,
                 0xf1,0x53,0xdd,0x62,0xd6,0x23,0x36,0x3d,0xd4,
                 0xba,0xfb,0x91,0x31,0xda,0x33,0x57,0xcf,0x6a,
                 0x80,0xbd,0xf7,0x24,0xff,0x7a,0x56,0x8e,0x70,
                 0x5e,0x45,0x2b,0x97,0x2d,0x4e,0xf2,0xe1,0xad,
                 0xeb,0xff,0x4b,0xfe,0x90,0x89,0x80,0x2a,0xec,
                 0x14,0x41,0xfd,0x6d,0xe7,0x0a,0x17,0x02,0xc1,
                 0xf3,0x3f,0x24,0xc8,0xd4,0xfa,0x17,0xc2,0xac,
                 0x5c,0x6d,0x87,0x44,0x1f,0xcd,0xb6,0x0f,0xf2,
                 0xf2,0xa8},
        .data_len = 128,
        .mac_len = 12,
        .mac = {0x1d,0x1d,0x12,0xf4,0xff,0x4e,0x0d,0xeb,0xb7,
                0x15,0xb9,0xcb},
    }, {	// 35 (256)
        .key = {0x4b,0xdc,0x4b,0x88,0x62,0x95,0x68,0x99,0x37,
                0x3d,0x3d,0xf4,0xda,0x72,0x81,0xc0,0xea,0x2b,
                0xdd,0x57,0x63,0x40,0x59,0xef,0xb8,0x2d,0x15,
                0x7a,0x22,0x13,0x39,0xcb,0x37,0xff,0x2e,0xf9,
                0xbe,0x6f,0x0f,0x08,0xc2,0x12,0x5a,0xc6,0xe5,
                0xd0,0xec,0xf4,0xf7,0x0a,0x2c,0xa6,0xc7,0x23,
                0x86,0xed,0x39,0x3f,0x1b,0xb2,0x99,0x4a,0xb6,
                0xe5,0x2f,0x3d,0x02,0xd8,0x14,0x9c,0xfb,0xe5,
                0x44,0x43,0xa3,0x57,0xf3,0x63,0xf6,0x88},
        .key_len = 80,
        .data = {0x28,0xaa,0xb2,0xe4,0xa0,0xe5,0x5c,0x11,0xd5,
                 0x50,0x3c,0x4d,0xca,0xb5,0x84,0x54,0x5c,0x49,
                 0x23,0xa6,0x1b,0x31,0x3c,0x2c,0x5a,0x44,0xd6,
                 0x1d,0x82,0x13,0xd5,0x23,0xac,0x26,0x29,0xba,
                 0x6e,0x89,0x45,0xd9,0xf4,0x88,0xd2,0xd5,0x53,
                 0xb6,0xa5,0x82,0x1b,0x34,0xef,0x9b,0x2b,0x2f,
                 0xb4,0x64,0xca,0xab,0x7f,0x8d,0xf3,0x7f,0x53,
                 0x5a,0xef,0xa1,0xe4,0x01,0x2a,0xa4,0x07,0x54,
                 0x3f,0x7f,0x68,0x9f,0x55,0x90,0x7b,0xd4,0xae,
                 0xe1,0xb5,0xe5,0x7d,0xa9,0xfb,0x72,0xf8,0x16,
                 0x5b,0xa4,0xaf,0x49,0xfa,0x59,0x1c,0xa3,0x4d,
                 0x81,0x7b,0x3f,0x8c,0xc7,0xdc,0xbf,0x64,0x75,
                 0x76,0x4c,0xed,0x91,0x3e,0xd8,0xdb,0x4c,0xb8,
                 0xa6,0xf8,0x9e,0x0d,0x0d,0xd2,0x2a,0x5f,0x79,
                 0xb0,0x67},
        .data_len = 128,
        .mac_len = 12,
        .mac = {0xb3,0xeb,0xb5,0x67,0xbe,0xf1,0xfe,0xa5,0xd4,
                0xf9,0x54,0xbb},
    }, {	// 36 (270)
        .key = {0xb5,0x43,0x8e,0x38,0x45,0xf3,0x9a,0xfe,0x7d,
                0xeb,0x0f,0xcf,0xb8,0x6e,0x2d,0xbe,0x4f,0xbc,
                0x48,0x9f,0x55,0xf0,0x1c,0x0f,0x84,0x29,0x61,
                0xb5,0x76,0xe8,0x9f,0xc7,0x19,0xb9,0x44,0xcf,
                0x5d,0x16,0xf4,0xaf,0x2f,0x88,0x20,0xe2,0xab,
                0x0f,0xda,0x06,0x8d,0xc4,0xe7,0x97,0xe9,0xbd,
                0x16,0xfe,0x1d,0x31,0xd1,0xca,0x03,0xdc,0xf2,
                0x3d,0x6b,0xa5,0xd8,0x0a,0xc8,0x7f,0xb9,0x5d,
                0x29,0x8d,0x39,0x1c,0x6b,0x89,0x3c,0x6c},
        .key_len = 80,
        .data = {0x8e,0xcd,0xcd,0x81,0x76,0xd8,0xa1,0x64,0xf6,
                 0x25,0x97,0x33,0xbc,0x77,0xef,0x78,0x3b,0x48,
                 0xd4,0x0c,0xff,0xc5,0x47,0x35,0x3d,0x19,0x59,
                 0x12,0xaf,0xee,0x9d,0x39,0x9e,0x31,0xdd,0x9e,
                 0x41,0x16,0x0c,0xb7,0x45,0x5d,0x7c,0xdd,0xad,
                 0xd3,0x51,0xf6,0xdc,0x1b,0x36,0x51,0xf0,0xae,
                 0x4e,0xd1,0x52,0x21,0x6d,0x4e,0x8b,0xa7,0x89,
                 0x38,0x5a,0xd6,0x6b,0x7d,0x03,0xae,0xaa,0xad,
                 0xe9,0xd7,0xda,0x5d,0x5f,0x2a,0x01,0xc9,0xbc,
                 0x73,0x4a,0xbd,0xad,0x75,0xfe,0xb5,0xd0,0x2f,
                 0xaf,0x43,0x7e,0x5e,0xb7,0xb1,0xe8,0x43,0xe1,
                 0xe7,0x65,0xa6,0x65,0x90,0x0a,0x1b,0x1a,0x79,
                 0x7c,0x84,0xe7,0x39,0x02,0xd7,0x7a,0x17,0xde,
                 0x22,0x3d,0x28,0xde,0xcc,0x86,0xb8,0x2e,0x1d,
                 0x0f,0xeb},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xa9,0x5c,0xf7,0xbb,0x2f,0x67,0x98,0x34,0x69,
                0xd4,0xfc,0x48,0x9e,0x31,0x92,0xd3},
    }, {	// 37 (271)
        .key = {0x95,0xf2,0xc1,0x50,0x9d,0xff,0x6d,0x16,0x2e,
                0xdd,0x5d,0xe3,0x2d,0xed,0x42,0x38,0x66,0xdf,
                0xda,0x68,0x2b,0xc7,0xb7,0x50,0x3e,0x73,0x41,
                0x42,0xf2,0xfc,0xfe,0x42,0x8c,0x9c,0x11,0x75,
                0xef,0xbf,0x01,0xd6,0x79,0x5d,0xbc,0x2b,0x28,
                0x86,0xdc,0x38,0x01,0x3f,0x28,0x32,0xb2,0x8c,
                0x5e,0x76,0x76,0xce,0x30,0x7b,0x39,0x4f,0x8c,
                0x05,0xfd,0x1c,0x20,0x9c,0x7c,0x13,0x1e,0x3d,
                0x0e,0x3c,0x3c,0x4f,0xce,0x5d,0x00,0xd8},
        .key_len = 80,
        .data = {0x1c,0x43,0x96,0xf7,0xb7,0xf9,0x22,0x8e,0x83,
                 0x2a,0x13,0x69,0x20,0x02,0xba,0x2a,0xff,0x43,
                 0x9d,0xcb,0x7f,0xdd,0xbf,0xd4,0x56,0xc0,0x22,
                 0xd1,0x33,0xee,0x89,0x03,0xa2,0xd4,0x82,0x56,
                 0x2f,0xda,0xa4,0x93,0xce,0x39,0x16,0xd7,0x7a,
                 0x0c,0x51,0x44,0x1d,0xab,0x26,0xf6,0xb0,0x34,
                 0x02,0x38,0xa3,0x6a,0x71,0xf8,0x7f,0xc3,0xe1,
                 0x79,0xca,0xbc,0xa9,0x48,0x2b,0x70,0x49,0x71,
                 0xce,0x69,0xf3,0xf2,0x0a,0xb6,0x4b,0x70,0x41,
                 0x3d,0x6c,0x29,0x08,0x53,0x2b,0x2a,0x88,0x8a,
                 0x9f,0xc2,0x24,0xca,0xe1,0x36,0x5d,0xa4,0x10,
                 0xb6,0xf2,0xe2,0x98,0x90,0x4b,0x63,0xb4,0xa4,
                 0x17,0x26,0x32,0x18,0x35,0xa4,0x77,0x4d,0xd0,
                 0x63,0xc2,0x11,0xcf,0xc8,0xb5,0x16,0x6c,0x2d,
                 0x11,0xa2},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x0a,0x06,0x07,0x35,0xb4,0x79,0x9e,0xeb,0x20,
                0x4c,0x52,0x03,0xe6,0x17,0xa7,0x76},
    }, {	// 38 (285)
        .key = {0x4d,0xdd,0x00,0xd0,0xab,0x6a,0xab,0x21,0x00,
                0xce,0x97,0x54,0xc3,0xb3,0x98,0x7c,0x06,0xf7,
                0xe5,0x86,0x56,0x01,0x1d,0x26,0xe3,0x51,0x87,
                0x11,0xe1,0x5b,0x9e,0x6d,0x2d,0x96,0xcd,0x85,
                0x34,0xd0,0x77,0xc2,0x11,0xc4,0x3a,0xd7,0xf5,
                0xee,0x75,0x3b,0xcc,0x9e,0x07,0xdc,0x1d,0x4c,
                0x5a,0x12,0x32,0x2b,0xa1,0xd1,0x7a,0x00,0x5d,
                0x24,0x2b,0x35,0x26,0xd6,0x2b,0x29,0xa8,0x72,
                0x31,0xcb,0xec,0x6f,0x28,0x67,0xd9,0xa4},
        .key_len = 80,
        .data = {0x28,0xbe,0x0d,0x9e,0x62,0xdc,0x89,0xe2,0xa9,
                 0x13,0x06,0x4c,0x0d,0x3d,0xbf,0xb3,0x5a,0x0c,
                 0x77,0x66,0xf7,0x56,0x74,0x1b,0x0e,0xaf,0xcc,
                 0x28,0xed,0x3d,0xdf,0xf6,0xad,0xc8,0x25,0xb2,
                 0x11,0x11,0x2a,0x45,0xb0,0x65,0xd6,0x87,0x57,
                 0x71,0xf2,0xaf,0xa9,0x58,0xe8,0x0f,0x08,0x03,
                 0xca,0xfe,0xb9,0xb9,0x96,0x15,0x42,0xef,0xb9,
                 0x9e,0x17,0x61,0xd1,0x49,0x76,0x61,0xb7,0x21,
                 0x90,0x6f,0xbd,0xbf,0xe9,0x0b,0x34,0xbd,0x01,
                 0xc7,0x32,0x6e,0x34,0xa0,0x92,0xcc,0xdf,0x8e,
                 0x3b,0xb2,0xc4,0x5a,0xa6,0x4c,0xb0,0xb0,0x9a,
                 0xcb,0x5b,0x75,0x3a,0x5d,0x8f,0x5a,0x42,0x5c,
                 0x8c,0xb2,0x8e,0xc5,0xac,0x81,0xdc,0xed,0x43,
                 0xd5,0xd2,0x6f,0xc9,0x59,0x43,0x69,0x3b,0x27,
                 0xae,0xe8},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x39,0x32,0x38,0xd3,0xaf,0xdb,0x7d,0x97,0x0b,
                0x96,0x6d,0x37,0x4f,0xe0,0x97,0xec,0x87,0x97,
                0xa8,0x70},
    }, {	// 39 (286)
        .key = {0x7a,0x31,0x55,0x3b,0x05,0xe9,0x6a,0x8d,0xa0,
                0xa4,0xd5,0xb8,0x1a,0x85,0x7d,0x19,0x2a,0xfb,
                0x6a,0xab,0xb1,0xf1,0x27,0xd7,0x40,0x45,0x6a,
                0x8e,0xda,0x7c,0xf6,0x96,0xfb,0xb4,0xc1,0x21,
                0xd8,0xd9,0x52,0xa4,0xe9,0x1c,0x6e,0xe6,0xa5,
                0xa1,0xf3,0x58,0x8d,0x78,0x04,0xa4,0x6b,0xcf,
                0x66,0x88,0xdc,0x66,0x2a,0xe5,0x0c,0x43,0x8d,
                0x13,0xc1,0xa6,0x1c,0x78,0x9b,0x3f,0x1c,0x59,
                0x9a,0x9f,0x28,0xef,0xe0,0xed,0x1c,0xbe},
        .key_len = 80,
        .data = {0xfb,0x09,0x1d,0xdd,0x95,0xb1,0x00,0xdf,0xcf,
                 0x89,0x2d,0x78,0xe5,0xe7,0x70,0xd3,0xa3,0x7b,
                 0x8c,0x38,0x85,0xdf,0x80,0x3c,0x1d,0x6f,0x09,
                 0x35,0xb5,0x5b,0x68,0xf1,0x36,0xfb,0x65,0xa8,
                 0x48,0x62,0x94,0x2e,0xbb,0x35,0xd7,0x6d,0x26,
                 0xbe,0x24,0x13,0xcd,0x3c,0x89,0x88,0xc8,0x7d,
                 0x6d,0x23,0x62,0xaf,0x18,0x9d,0xc0,0x74,0x76,
                 0xc6,0xc3,0x34,0x17,0x76,0x2e,0xb7,0x7b,0xc7,
                 0x0c,0xf3,0x8d,0x81,0x4c,0x22,0x6d,0xd6,0xaf,
                 0x18,0x72,0x50,0xe4,0xd4,0x70,0x07,0xf1,0x55,
                 0x36,0x17,0xd4,0xaf,0x5b,0x51,0x6a,0x5d,0x3b,
                 0x31,0x91,0xd9,0x3c,0x10,0x89,0x6a,0x56,0x9b,
                 0xa1,0x3d,0xd2,0x84,0x0f,0xb8,0x51,0x78,0x1f,
                 0x0b,0x11,0x50,0x90,0x08,0x6c,0x8b,0x3a,0x34,
                 0xa1,0xfc},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x0f,0xdd,0x3f,0x83,0x6d,0xd7,0xe5,0xc5,0x06,
                0xab,0x21,0xad,0xde,0x9a,0xe5,0xdc,0x09,0xcb,
                0x35,0x9d},
    }
};

struct HMAC_TEST_VECTOR fips_sha224_hmac_general_test_vector[] = {
    {   // 0 (0)
        .key = {0x37,0x14,0x70,0x78,0x39,0xda,0xf7,0x91,0x22,
                0xc7,0x82,0x41,0x63,0x51,0x38,0x5e,0x88,0xa8,
                0x1d,0x31,0xc9,0xf6,0x41,0xd8,0xdc,0xe5,0x38,
                0xe9,0x0e,0x63,0xc9,0x58,0x92,0xa2,0xea,0x9b,
                0x19,0x62,0xed,0x0b,0xa3,0x72,0xf4,0x8e,0x94,
                0x74,0xaa,0x73,0x0a,0xe2},
        .key_len = 50,
        .data = {0x41,0x18,0x43,0xa2,0x13,0x87,0x84,0x6f,0x3b,
                 0x9e,0xd5,0xfc,0x54,0x5a,0xca,0xdf,0xa5,0xb7,
                 0x03,0x86,0xf6,0x2d,0xa4,0xd9,0xa2,0x7b,0x04,
                 0x1b,0xee,0xa3,0xaa,0x11,0x99,0x36,0x75,0x67,
                 0xb4,0xd1,0x1a,0x4f,0xb4,0xe8,0xd4,0x6b,0xc6,
                 0xc2,0x56,0xed,0x62,0xc5,0x05,0xfd,0x23,0xf4,
                 0x64,0x5b,0xd6,0xb6,0xcf,0x45,0xd1,0xd9,0x6d,
                 0x9b,0x86,0xd6,0x60,0x41,0x57,0x57,0x3e,0xc5,
                 0xac,0xf6,0xc5,0x41,0x43,0x48,0xca,0x83,0xc8,
                 0x1a,0x73,0x6c,0xa6,0xfa,0xa6,0x96,0x1c,0xfa,
                 0xc1,0x39,0x93,0xb0,0x8c,0x50,0x2f,0x81,0x6c,
                 0xf7,0xa4,0x20,0xd9,0x18,0x4b,0x51,0x11,0x46,
                 0x75,0xf3,0x0e,0xe9,0xff,0x3d,0xb6,0x9c,0x26,
                 0x48,0x53,0xd3,0x9d,0xcd,0x42,0xc1,0xdd,0x31,
                 0xef,0x79},
        .data_len = 128,
        .mac_len = 14,
        .mac = {0x33,0xf1,0x7a,0xc8,0xa5,0xc6,0xb5,0x25,0xdb,
                0x8b,0x86,0x44,0xb6,0xab},
        .chunks = {64, 64},
        .num_chunks = 2,
    }, {    // 1 (1)
        .key = {0xce,0x4c,0x92,0x6c,0x09,0x22,0xba,0x36,0x26,
                0x9a,0x20,0xd6,0x0d,0xcf,0x08,0xd4,0x3a,0x1c,
                0xea,0x12,0x0f,0x26,0x6a,0xf7,0x6f,0x1c,0x8a,
                0xcd,0x88,0x3d,0x1f,0x68,0xf0,0x9b,0x82,0x09,
                0xf4,0x1f,0x87,0x82,0x2d,0xce,0xb3,0x9a,0x54,
                0x4a,0xa9,0xb2,0x56,0x9c},
        .key_len = 50,
        .data = {0x41,0x0a,0xc8,0x70,0x3f,0x31,0x2f,0xe4,0x26,
                 0xf0,0xd1,0xa6,0x2d,0x36,0x2d,0x44,0x78,0x25,
                 0x2f,0x11,0xdf,0xc7,0x0b,0x78,0xf0,0xfc,0x6c,
                 0x91,0x37,0xb7,0xce,0xf2,0xbd,0x6a,0x28,0xc8,
                 0xce,0xbf,0xcf,0x26,0xff,0x89,0x97,0x9c,0x70,
                 0xfa,0x3b,0x0c,0x4f,0x16,0xff,0xb9,0x9d,0x67,
                 0x93,0xa1,0x35,0xf3,0x3b,0x0d,0x0a,0x6b,0x2d,
                 0x66,0xb0,0xa5,0x3a,0x4a,0x1e,0x4c,0xb1,0xdf,
                 0xb2,0xd7,0x81,0x67,0x75,0xb7,0x9d,0x15,0xa1,
                 0xd5,0xa5,0x1f,0x60,0x3b,0xc2,0x15,0xd7,0x11,
                 0xd2,0x71,0x63,0xcc,0xc5,0x6f,0x22,0xa2,0x46,
                 0x1f,0xb6,0x41,0x9e,0xb8,0x45,0xc0,0xd6,0x2f,
                 0xc4,0xd6,0x1c,0x08,0x76,0x4a,0x69,0x6d,0xaf,
                 0xaa,0x60,0x7d,0xde,0x40,0x78,0xe6,0xca,0x42,
                 0x5d,0xb6},
        .data_len = 128,
        .mac_len = 14,
        .mac = {0x90,0x08,0x42,0xbb,0x91,0x6f,0xb0,0xdf,0xf7,
                0xab,0xe3,0x7e,0xd8,0xf6},
        .chunks = {32, 32, 32, 32},
        .num_chunks = 4,
    }, {    // 2 (15)
        .key = {0x49,0x55,0x39,0xa6,0x81,0x41,0xfc,0x09,0x93,
                0x93,0xad,0x40,0x55,0x5a,0x70,0xeb,0xb4,0x5e,
                0x3d,0x37,0xf9,0x57,0x3f,0xb1,0x4b,0x5c,0x7a,
                0x5c,0x75,0x9e,0xb1,0x00,0xea,0x56,0x87,0xc6,
                0x06,0xfc,0xe4,0x02,0x97,0xba,0x9a,0x50,0x9c,
                0x20,0x49,0xe2,0x4d,0x19},
        .key_len = 50,
        .data = {0xf2,0xc2,0x98,0xf6,0x2c,0xcd,0x8e,0x10,0x26,
                 0x45,0x35,0x2f,0xd2,0x64,0xaf,0x76,0x17,0x84,
                 0xc2,0x2a,0x77,0x31,0x94,0x67,0xef,0x83,0xb2,
                 0x11,0x1e,0xaa,0x57,0x0c,0xaa,0xcb,0xec,0xb0,
                 0xe0,0x0b,0xcb,0x77,0x9a,0xe9,0x48,0xf9,0x66,
                 0xd0,0xf2,0x1b,0xe4,0xec,0x16,0x83,0x70,0x3e,
                 0x85,0x4f,0x01,0xeb,0x97,0x06,0x51,0xda,0x70,
                 0xce,0x3f,0x7d,0x82,0xe0,0x71,0xef,0x53,0x3a,
                 0x3d,0x40,0x81,0x93,0xca,0x0d,0x03,0x94,0x72,
                 0x53,0xb1,0x45,0x6b,0x84,0x07,0xc6,0xa7,0xdb,
                 0x26,0x3f,0xf8,0xed,0x1b,0x19,0x78,0x82,0x64,
                 0x6f,0x28,0xa2,0xb0,0xbf,0xd7,0xf3,0xe7,0xe6,
                 0x46,0x42,0x6a,0x4b,0x89,0x5f,0x9c,0x5a,0xb0,
                 0x2e,0xa1,0x34,0xa7,0xfb,0x66,0xa4,0x5b,0x0e,
                 0xe1,0x56},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x04,0x40,0xb3,0xfc,0xd0,0xdd,0xf5,0x07,0x71,
                0x7e,0xae,0x86,0xbe,0x2b,0x0a,0x69},
        .chunks = {32, 32, 64},
        .num_chunks = 3,
    }, {    // 3 (16)
        .key = {0x60,0xcb,0xa4,0xff,0xa6,0x39,0xda,0x15,0x80,
                0x9f,0x0c,0x93,0x07,0x42,0xba,0x0a,0x5d,0x50,
                0xa6,0x47,0xeb,0x18,0x35,0x25,0xed,0x79,0x17,
                0xa6,0x31,0x9b,0x2a,0x42,0x1b,0xde,0x9c,0x7f,
                0xed,0x10,0x51,0xc2,0x45,0xe3,0x84,0x65,0xe8,
                0x9e,0xb0,0x09,0xbd,0x12},
        .key_len = 50,
        .data = {0x1a,0xc2,0x64,0x9d,0xc5,0x45,0x8f,0x43,0x9f,
                 0x94,0x49,0x5d,0xb8,0x4e,0x48,0x19,0x9a,0xa8,
                 0x7c,0xf5,0xda,0xb2,0x46,0x24,0xec,0xdc,0xe5,
                 0x33,0x3b,0xb8,0x31,0xec,0x79,0x01,0x8b,0x7b,
                 0x34,0xdd,0x14,0xcf,0xab,0x5f,0x1d,0x43,0xdf,
                 0x50,0xdf,0x7e,0xed,0x2f,0x6c,0x97,0x9a,0x76,
                 0xe6,0x7d,0x30,0x1f,0xef,0x97,0x89,0x67,0x60,
                 0xa7,0x69,0xd2,0x3e,0xf4,0x11,0x6f,0xea,0xa8,
                 0x10,0x5d,0x97,0x77,0x5d,0x8e,0x01,0x81,0xa3,
                 0xcb,0x5d,0x7a,0xdc,0xf8,0x8b,0x08,0x21,0x9a,
                 0x93,0x6c,0xc0,0x96,0x4f,0x65,0x90,0x3b,0x65,
                 0x51,0x81,0x83,0x79,0x96,0x77,0x96,0xa0,0x40,
                 0x59,0x90,0x17,0xec,0x64,0x84,0x35,0xe4,0xc6,
                 0xc1,0x9e,0x8b,0x68,0x54,0xba,0xfc,0x0b,0xce,
                 0x65,0xee},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x48,0x9b,0x50,0xad,0x20,0x00,0x2f,0xdc,0x27,
                0x65,0xad,0xa1,0x13,0x4b,0xad,0xda},
        .chunks = {128},
        .num_chunks = 1,
    }, {    // 4 (30)
        .key = {0x76,0xcc,0x58,0x37,0x87,0x82,0xbb,0xc0,0x17,
                0x8f,0x9b,0x4f,0x1f,0x9c,0x4a,0xfc,0x23,0xa6,
                0x25,0xc5,0x00,0xaf,0x74,0xde,0xe4,0xb3,0x91,
                0x48,0x9e,0xdd,0x4d,0xa5,0x18,0xfb,0x8f,0x4b,
                0x21,0x30,0x3e,0x97,0x70,0xe8,0x61,0x83,0xe6,
                0x11,0x85,0x7b,0x14,0x89},
        .key_len = 50,
        .data = {0xba,0xa0,0xac,0xc0,0x88,0x14,0xf2,0x45,0x3a,
                 0x81,0x6e,0xce,0xe2,0xf7,0xf8,0xa8,0x31,0xbc,
                 0xd3,0xd7,0xa3,0xaa,0x83,0xa1,0xf7,0xd2,0xde,
                 0x51,0xe2,0x65,0x66,0x35,0x04,0xd5,0xd6,0xe9,
                 0x1b,0x37,0xfb,0xd7,0x77,0xec,0xc5,0xff,0xae,
                 0x80,0x9e,0xf4,0x90,0xd2,0xfa,0x27,0x5d,0x75,
                 0x30,0x68,0x36,0x57,0x88,0xaa,0xd2,0xa0,0x95,
                 0x07,0x72,0xdd,0xc5,0x04,0x17,0x5e,0x36,0x15,
                 0xf2,0xdc,0xcb,0x64,0x1c,0x60,0x80,0xce,0x87,
                 0x5f,0xfa,0x2e,0x35,0x2a,0x63,0xdf,0xa7,0x86,
                 0x74,0x7c,0x99,0x96,0xd3,0xea,0xec,0x02,0xa0,
                 0x99,0xe4,0x52,0x4a,0x06,0x93,0x9b,0xa8,0x85,
                 0x95,0x5e,0x45,0x3d,0xfe,0xad,0xfb,0x74,0x91,
                 0x8f,0xc3,0x0d,0x33,0xa6,0xcb,0x4e,0x9f,0x0f,
                 0xc9,0xe7},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0xf2,0x55,0xc7,0x06,0x5a,0x6a,0xd7,0xac,0x49,
                0xaa,0x68,0x31,0xe5,0xe0,0xfa,0x8b,0x12,0x8a,
                0x5f,0xf7},
        .chunks = {32, 0, 64, 32},
        .num_chunks = 4,
    }, {    // 5 (31)
        .key = {0x49,0x18,0x62,0x90,0x58,0xbb,0x50,0x84,0x7e,
                0x7a,0xb9,0x96,0xa8,0x27,0x7c,0x35,0x51,0xd6,
                0xa7,0x4a,0x91,0xd9,0x97,0xcb,0x57,0xef,0x61,
                0x8c,0xf2,0xb8,0x71,0x59,0xd7,0x6c,0x71,0x43,
                0x9e,0xcd,0x6d,0x37,0x4d,0x5a,0x34,0xd5,0xd9,
                0xc6,0x01,0x7e,0xea,0xa8},
        .key_len = 50,
        .data = {0x38,0x7b,0xac,0x63,0xcf,0x86,0x5f,0x06,0xe9,
                 0x6a,0x5a,0x62,0x32,0xb7,0x3b,0xa8,0xe8,0x43,
                 0xcf,0x80,0xdf,0x1f,0x36,0x9f,0xc9,0xc9,0x41,
                 0x9c,0xb4,0xa3,0x1f,0xff,0xd4,0xa4,0x39,0x9c,
                 0x83,0x4e,0x5a,0x1f,0x86,0xe0,0x13,0xe7,0x92,
                 0xcc,0x5e,0x18,0x0e,0x6b,0x81,0xeb,0x1b,0xf0,
                 0x35,0x49,0xd2,0x9b,0x08,0x99,0xcc,0x2f,0x00,
                 0xd6,0xe8,0xa5,0x81,0xe4,0x48,0x22,0x70,0x24,
                 0x9d,0x27,0x99,0x46,0xa5,0x07,0x4e,0x5e,0xf2,
                 0x9f,0x51,0x3a,0xec,0x41,0xf5,0xf3,0x19,0xf1,
                 0x7f,0xc3,0x9c,0xd4,0xd2,0x16,0xfc,0xe5,0x4c,
                 0xe4,0x27,0xea,0xeb,0x3d,0xb7,0xa7,0xb4,0x26,
                 0x73,0x51,0x55,0xa3,0x19,0x45,0x09,0x15,0x5b,
                 0xfa,0xbc,0xc6,0x27,0x34,0x70,0xc3,0x2b,0x7b,
                 0xdf,0xea},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0xb1,0x75,0x6c,0x19,0x6d,0xc4,0x40,0x75,0x21,
                0xf3,0xc2,0x1d,0xe5,0x53,0xec,0xc2,0x1c,0x50,
                0x76,0xf5},
        .chunks = {64, -1, 64},
        .num_chunks = 3,
    }, {    // 6 (45)
        .key = {0xff,0x91,0xe9,0xe7,0x48,0xc9,0x4f,0xd0,0x81,
                0x84,0x2c,0x50,0xfc,0x5e,0xfc,0x03,0xb9,0xe5,
                0x66,0x06,0x2d,0x4e,0x1c,0x52,0x43,0x67,0x03,
                0x48,0xa1,0xcd,0x94,0x8c,0x2c,0x5f,0x82,0x17,
                0xce,0xf2,0x9c,0x88,0x77,0x47,0xf6,0xbb,0x61,
                0x3a,0x43,0x9d,0x99,0x3b},
        .key_len = 50,
        .data = {0x9c,0xe6,0x6b,0xe0,0xe1,0x6f,0x03,0xba,0xae,
                 0x35,0x67,0xae,0xb7,0xae,0x84,0x00,0xfe,0x60,
                 0x14,0x99,0x99,0x9c,0x7b,0x5a,0xb6,0x68,0xef,
                 0xb0,0xdc,0xbd,0xdc,0x69,0x74,0xf3,0x87,0xc6,
                 0x87,0x79,0xf1,0xd1,0xc9,0xc9,0xfe,0xf0,0xd7,
                 0x9b,0xd6,0xbb,0xbd,0x59,0x8c,0x0b,0xbb,0xd4,
                 0xfe,0x53,0x49,0x35,0xfc,0x34,0x58,0x36,0xac,
                 0x4b,0xdb,0x92,0x2c,0x4e,0x86,0xb9,0x7a,0x57,
                 0xd5,0xc9,0x91,0x7f,0x51,0xba,0xd5,0xaf,0x0f,
                 0xd8,0xb1,0xb3,0x79,0x77,0x7f,0x90,0x50,0xe2,
                 0xa8,0x18,0xf2,0x94,0x0c,0xbb,0xd9,0xab,0xa4,
                 0xa0,0x65,0x99,0x65,0xf5,0xdb,0x1d,0x68,0x83,
                 0xad,0x72,0x49,0x85,0xfc,0xc6,0xcd,0xba,0x5b,
                 0xed,0xc7,0xb9,0xd6,0x57,0x3c,0x85,0x33,0x3f,
                 0xc5,0x61},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x77,0xee,0x73,0x67,0x64,0x7f,0xa4,0x10,0xa0,
                0x9e,0x96,0x59,0xb6,0x88,0xcb,0xb9,0x2a,0x51,
                0xb4,0x79,0xfd,0xa8,0x95,0xc6},
        .chunks = {54, 0, 54, -1, 20},
        .num_chunks = 5,
    }, {    // 7 (46)
        .key = {0x3c,0x17,0xd3,0x27,0x44,0x95,0xdc,0xc8,0x6f,
                0x27,0x22,0x39,0x8d,0xb6,0x02,0x37,0xfc,0x70,
                0xfc,0x0e,0x63,0xb3,0x0a,0xa4,0xa3,0x2c,0x30,
                0xb9,0x0b,0x40,0x55,0x6d,0xcc,0xaa,0x51,0x03,
                0xac,0x66,0x47,0xe4,0xfe,0xce,0x35,0xe7,0xd1,
                0x04,0xc9,0xcf,0x68,0x8f},
        .key_len = 50,
        .data = {0x1b,0xd9,0x6d,0xb9,0x6e,0x74,0x1d,0x63,0x7e,
                 0x5c,0x63,0xd6,0x97,0xf8,0x1a,0x4d,0x99,0xf8,
                 0x44,0x96,0xac,0x38,0x5a,0x15,0x30,0xe7,0xe0,
                 0x0c,0xf3,0xe8,0x3b,0x1a,0xa7,0x78,0xc7,0x01,
                 0x93,0x12,0x25,0xac,0x88,0x8f,0xd2,0x70,0x17,
                 0x92,0xfe,0x92,0x01,0x23,0x9b,0x09,0xef,0x9a,
                 0xb4,0x86,0x63,0xbf,0x00,0xb2,0xef,0x2f,0xb7,
                 0xb6,0xa5,0x02,0xa4,0x01,0xce,0xaa,0x7c,0x45,
                 0xdf,0x1e,0x6e,0xc3,0xa8,0x39,0xfb,0xc9,0xcf,
                 0x5c,0x08,0x79,0x7d,0x5b,0x31,0xcb,0x57,0x97,
                 0x61,0xd6,0xb0,0x67,0x9f,0x4b,0xfc,0xbd,0x2b,
                 0x42,0x88,0x06,0xca,0x39,0x69,0x51,0x5e,0x59,
                 0xb3,0xb4,0xb9,0xfc,0xeb,0xeb,0xb3,0x6d,0xbe,
                 0x43,0x7a,0xee,0xf2,0xb7,0x2a,0x00,0x9e,0x58,
                 0x9e,0xde},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0xcd,0xf3,0x67,0xa1,0xb5,0x73,0x71,0x3f,0x1f,
                0xb0,0xea,0xb4,0xfa,0x6d,0xbc,0x13,0x4e,0x1d,
                0x3b,0x90,0x14,0xec,0xe5,0x10},
        .chunks = {0, 128},
        .num_chunks = 2,
    }, {    // 8 (59)
        .key = {0x10,0x9f,0xaf,0xbb,0x90,0x12,0xbe,0x97,0x20,
                0x36,0x15,0x85,0xc1,0x58,0xdf,0xd0,0x1e,0xc6,
                0x46,0xcf,0xc2,0x30,0x42,0x6a,0x89,0x54,0xae,
                0xdb,0x54,0xdb,0xa1,0xb2,0x01,0x7d,0xfb,0x1c,
                0x9b,0x6c,0x3e,0x64,0xcd,0xf0,0xa0,0xba,0x9d,
                0x4e,0x5e,0x34,0x57,0xef},
        .key_len = 50,
        .data = {0xf2,0xa1,0xc2,0x04,0xbb,0xd1,0xcc,0x55,0xc3,
                 0x09,0x72,0xbc,0xc2,0xb5,0xbc,0x33,0x97,0xc1,
                 0x37,0x26,0xd3,0x0b,0x98,0x83,0x7b,0x18,0xa3,
                 0x11,0x0a,0x06,0x4b,0x02,0xb6,0xc1,0x1b,0xcb,
                 0xf4,0x7a,0xd4,0x33,0x3e,0x55,0x58,0xaa,0x14,
                 0x62,0xc6,0xdd,0x41,0xcf,0xe7,0x87,0x5f,0x5c,
                 0xc9,0x8a,0x9d,0x0e,0xa9,0x3e,0xc0,0x28,0x16,
                 0x09,0x68,0x57,0x54,0x30,0x95,0x2d,0xff,0xec,
                 0x69,0xd0,0xf0,0xc1,0x30,0xf4,0x65,0x0b,0x9d,
                 0x06,0x5b,0xda,0x56,0xaa,0x3f,0xff,0xb4,0x68,
                 0xc6,0x67,0xc3,0xa2,0x1a,0xa8,0x91,0xca,0x36,
                 0xda,0xa5,0xd2,0x93,0xa8,0xcd,0xe3,0x04,0xa6,
                 0x1d,0x51,0xde,0xe6,0x34,0xcf,0x5d,0xc6,0xc1,
                 0xdf,0x43,0x0b,0x46,0x8d,0xc2,0x34,0x1c,0x5d,
                 0x9c,0x08},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0xab,0xfb,0x36,0xe5,0xd7,0x72,0x30,0x9a,0xdf,
                0x06,0x08,0x36,0x72,0x69,0x58,0x20,0x67,0xa3,
                0x0f,0x8d,0xb7,0x10,0xd3,0xb2},
    }, {    // 9 (60)
        .key = {0xcf,0x12,0x75,0x79,0xd6,0xb2,0xb0,0xb3,0xa6,
                0x07,0xa6,0x31,0x4b,0xf8,0x73,0x30,0x61,0xc3,
                0x2a,0x04,0x35,0x93,0x19,0x55,0x27,0x54,0x4f,
                0x87,0x53,0xc6,0x5c,0x7a,0x70,0xd0,0x58,0x74,
                0xf7,0x18,0x27,0x5b,0x88,0xd0,0xfa,0x28,0x8b,
                0xd3,0x19,0x98,0x13,0xf0},
        .key_len = 50,
        .data = {0xfa,0x7e,0x18,0xcc,0x54,0x43,0x98,0x1f,0x22,
                 0xc0,0xa5,0xab,0xa2,0x11,0x79,0x15,0xf8,0x9c,
                 0x77,0x81,0xc3,0x4f,0x61,0xf9,0xf4,0x29,0xcb,
                 0x13,0xe0,0xfc,0xd0,0xce,0x94,0x71,0x03,0xbe,
                 0x68,0x4c,0xa8,0x69,0xd7,0xf1,0x25,0xf0,0x8d,
                 0x27,0xb3,0xf2,0xc2,0x1d,0x59,0xad,0xc7,0xab,
                 0x1b,0x66,0xde,0xd9,0x6f,0x0b,0x4f,0xa5,0xf0,
                 0x18,0xb8,0x01,0x56,0xb7,0xa5,0x1c,0xa6,0x2b,
                 0x60,0xe2,0xa6,0x6e,0x0b,0xc6,0x94,0x19,0xeb,
                 0xbf,0x17,0x85,0x07,0x90,0x76,0x30,0xf2,0x4d,
                 0x08,0x62,0xe5,0x1b,0xec,0x10,0x10,0x37,0xf9,
                 0x00,0x32,0x3a,0xf8,0x2e,0x68,0x9b,0x11,0x6f,
                 0x42,0x75,0x84,0x54,0x1c,0x8a,0x9a,0x51,0xac,
                 0x89,0xda,0x1e,0xd7,0x8c,0x7f,0x5e,0xc9,0xe5,
                 0x2a,0x7f},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0x35,0x4f,0x87,0xe9,0x8d,0x27,0x64,0x46,0x83,
                0x6e,0xa0,0x43,0x0c,0xe4,0x52,0x92,0x72,0xa0,
                0x17,0xc2,0x90,0x03,0x9a,0x9d,0xfe,0xa4,0x34,
                0x9b},
    }, {    // 10 (75)
        .key = {0xb2,0xff,0x28,0x2d,0x91,0x3a,0x31,0x26,0xaa,
                0x23,0x1a,0xc4,0x06,0xb1,0xcb,0xde,0x6e,0x98,
                0xf8,0xea,0xd0,0x1e,0xa3,0x71,0xe4,0x2b,0xe3,
                0xc0,0x0e,0xa0,0xe3,0xe1,0x87,0xea,0x2d,0xc9,
                0xb8,0xd7,0x29,0xd3,0xe6,0xa8,0xe0,0x6d,0x14,
                0x67,0x06,0x11,0x25,0x7b,0x38,0x78,0x42,0x79,
                0x82},
        .key_len = 55,
        .data = {0xbc,0x68,0x7c,0x26,0xe4,0x2b,0xc5,0xd7,0x71,
                 0xe3,0xc1,0xba,0x81,0xc6,0x1c,0xdf,0xb5,0x81,
                 0x1d,0x7d,0x36,0x2e,0xce,0x89,0xf6,0x80,0x5d,
                 0x87,0x39,0x41,0xc2,0xa1,0x53,0xff,0xa2,0xf6,
                 0xb3,0x02,0x73,0x34,0x20,0x81,0xe2,0x69,0x43,
                 0x15,0x71,0x16,0x66,0x6d,0x65,0x86,0x7b,0x44,
                 0xdf,0x5d,0x60,0x90,0x49,0x74,0x61,0xfc,0xf3,
                 0xe4,0xff,0x99,0xfe,0x61,0xef,0xd0,0x7a,0xca,
                 0x66,0x9c,0xcb,0xfd,0x94,0xf6,0x51,0xf0,0x6e,
                 0x90,0x48,0xfe,0xb1,0xc5,0xec,0x7e,0x24,0xd0,
                 0xe1,0x9a,0x1d,0x3d,0xd3,0x5f,0x46,0x9e,0x5a,
                 0x10,0xa1,0x7f,0x20,0xc0,0xca,0x9c,0x45,0x1f,
                 0x7d,0x51,0x02,0x95,0x57,0x51,0x8d,0x8d,0xe1,
                 0x9b,0x3d,0x14,0xca,0xaa,0xef,0xa2,0x74,0xec,
                 0xa8,0x37},
        .data_len = 128,
        .mac_len = 14,
        .mac = {0xe5,0x01,0xec,0x9c,0x4f,0x64,0xa8,0xbd,0x00,
                0x45,0xec,0x1f,0x5d,0x1a},
    }, {    // 11 (76)
        .key = {0xac,0x37,0xe2,0x6c,0xdf,0x42,0x16,0x13,0x63,
                0x8a,0x00,0x51,0xf8,0x20,0x7c,0x60,0x7c,0xbb,
                0x77,0x4d,0x5f,0x5a,0x01,0xb4,0x57,0x2b,0xa4,
                0xcb,0x55,0xac,0xf4,0xa1,0xb7,0x10,0x68,0xa8,
                0xbf,0xe4,0x68,0x72,0x44,0xf2,0xa7,0xb8,0xa7,
                0x66,0xb3,0xa0,0x09,0x2f,0xcb,0x3f,0xbb,0xa8,
                0xf0},
        .key_len = 55,
        .data = {0x31,0xf7,0x10,0x73,0x70,0x7b,0x06,0x4c,0xf9,
                 0xe9,0xe5,0x79,0x52,0x7f,0x7f,0xd7,0xf5,0xa0,
                 0x3b,0x80,0xa3,0xd8,0x0b,0xef,0xfc,0x1b,0x9e,
                 0x65,0x4d,0x4d,0xd6,0x8c,0xa9,0x08,0x7c,0x5a,
                 0x3b,0x99,0x33,0x39,0xda,0x4b,0xaa,0x65,0xde,
                 0xe9,0x16,0x95,0x32,0x7e,0xe0,0xdd,0xab,0x5f,
                 0x9a,0xd3,0x98,0x6b,0x38,0xf1,0x9b,0x92,0x7d,
                 0x3a,0xb7,0x94,0xb3,0x20,0xa9,0x0b,0x9f,0xa6,
                 0x7c,0xac,0xb4,0xc2,0x58,0xca,0x87,0x05,0x9b,
                 0x1b,0x83,0x7d,0x2a,0xbf,0xde,0xdb,0x0a,0x7e,
                 0xcb,0xd1,0x2a,0x52,0xf2,0xbe,0x83,0x25,0x1d,
                 0x8d,0x63,0xdb,0x34,0xda,0x06,0x93,0x6b,0x57,
                 0x67,0x2a,0xba,0x6f,0x55,0x01,0x8e,0x29,0xc9,
                 0x01,0xbe,0x4e,0x77,0xdc,0xc0,0xce,0xa3,0x95,
                 0x1d,0x60},
        .data_len = 128,
        .mac_len = 14,
        .mac = {0x98,0x19,0x5b,0x20,0x59,0xde,0x04,0x57,0xa5,
                0x3c,0x81,0xc4,0x87,0x86},
    }, {    // 12 (90)
        .key = {0x21,0xb7,0x02,0x6d,0xf9,0xd8,0xe4,0x35,0x96,
                0x6d,0x29,0x9e,0xf1,0xd4,0xc2,0xa4,0xa9,0x88,
                0x55,0x27,0xb1,0xfd,0x8f,0xed,0x46,0xcb,0xba,
                0xe7,0x25,0x0f,0x92,0xb5,0xfa,0x0e,0x54,0x6b,
                0x1b,0x1c,0x19,0x96,0x76,0xbd,0x62,0x7a,0xc9,
                0x09,0xf4,0x9e,0x11,0x55,0xe6,0x8b,0x44,0x0f,
                0xf8},
        .key_len = 55,
        .data = {0x6f,0xd5,0xc7,0x32,0x17,0x27,0x29,0x3f,0x9e,
                 0xcc,0xcd,0xd7,0x83,0x55,0xce,0x3f,0x49,0x45,
                 0x75,0x2d,0x20,0x22,0xfc,0x7d,0xc1,0xcc,0x97,
                 0x25,0x9a,0x0d,0xb5,0x6b,0x2b,0x70,0x82,0x56,
                 0xb0,0x57,0xb1,0x62,0x00,0x48,0x7a,0x77,0xb5,
                 0x88,0xde,0x1c,0x8d,0xeb,0x32,0xe9,0x10,0xea,
                 0xdb,0x3a,0x73,0xf6,0x57,0x2b,0xbd,0x0c,0x7d,
                 0xc0,0x28,0x8a,0x05,0x63,0xca,0x41,0x4a,0x94,
                 0x74,0xcb,0x4f,0xdc,0xdb,0x1e,0x00,0x02,0xf5,
                 0x62,0xbb,0x82,0xa6,0x00,0x0f,0x3b,0x9c,0xfc,
                 0xc4,0x68,0x78,0x84,0x1b,0xd1,0x2e,0xc9,0x56,
                 0x43,0xae,0x0d,0x7b,0x3d,0xc4,0xfd,0x56,0xec,
                 0x3c,0x42,0xbd,0x59,0x4b,0x53,0x18,0xe2,0x2a,
                 0x8d,0x20,0xe7,0x04,0x85,0xe6,0xc5,0x12,0x9d,
                 0x17,0x08},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xb7,0xdf,0x20,0xfb,0x02,0xdc,0x06,0x3c,0x82,
                0x50,0xc1,0x0f,0x2e,0x0f,0x0a,0x91},
    }, {    // 13 (91)
        .key = {0x75,0xde,0xc6,0x97,0xbc,0xf3,0x0b,0x7b,0x34,
                0x5d,0x97,0xbc,0x02,0x7f,0xcd,0x80,0xc5,0x4a,
                0x6f,0xd1,0x6e,0xbc,0x11,0x43,0x52,0xa7,0xcd,
                0xd6,0x7d,0xb4,0x39,0x1d,0x49,0xa7,0x96,0xa0,
                0x30,0xe5,0x4a,0x80,0xdf,0x33,0x91,0x59,0x8b,
                0xc0,0x6f,0x6c,0x88,0x66,0xe9,0xfe,0x18,0x2d,
                0x74},
        .key_len = 55,
        .data = {0x7f,0xc3,0xaa,0x4f,0x04,0xea,0x53,0x39,0xd7,
                 0xf8,0xd6,0xee,0x41,0x64,0xdb,0x43,0x4f,0x52,
                 0x05,0x25,0xca,0xba,0x80,0xdd,0x69,0xc7,0xd9,
                 0x92,0x46,0x7d,0x13,0x10,0x8f,0x9f,0xf7,0xc8,
                 0x7e,0xd0,0xab,0x17,0x8b,0xf8,0x86,0xcb,0xe0,
                 0xe9,0x23,0xe3,0x0d,0x43,0xb4,0x96,0xfd,0x94,
                 0x99,0x97,0x28,0xc4,0x32,0x98,0x0b,0xb1,0xf1,
                 0x0d,0xd5,0x16,0xb3,0x32,0x64,0xb1,0x50,0x4f,
                 0x6f,0xa5,0x70,0x3a,0x05,0x1d,0xa1,0x1e,0x36,
                 0x32,0xa7,0x7c,0x10,0x0f,0x13,0x6f,0xcd,0x52,
                 0x18,0x34,0x4a,0xb6,0x99,0xe9,0xbb,0xef,0x86,
                 0x95,0x9e,0xd6,0x69,0x69,0xed,0xc6,0x3c,0x46,
                 0x98,0x46,0x7e,0x7f,0x32,0xe8,0x86,0xdf,0xea,
                 0x71,0x16,0xee,0xbe,0x39,0x33,0x70,0xeb,0xd5,
                 0x34,0x2f},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x12,0x2a,0x00,0xcc,0xa9,0x91,0xda,0x9c,0x55,
                0xa5,0xe0,0x33,0x15,0x61,0x34,0x60},
    }, {    // 14 (105)
        .key = {0xb9,0x08,0x26,0x98,0x10,0xb3,0x54,0xc1,0xba,
                0x7e,0xd9,0xc8,0x2b,0xdd,0xe5,0x67,0x3f,0x0c,
                0x55,0x5c,0xce,0x53,0xe3,0xca,0x81,0x89,0x9a,
                0xdc,0xe4,0xd7,0x8e,0xab,0xbc,0xbe,0xc9,0x62,
                0x9a,0x5b,0x88,0xa0,0xb0,0xea,0xa8,0x1d,0xe4,
                0x8a,0x5b,0x91,0x23,0x3f,0x1f,0xd1,0xf8,0x3d,
                0xf1},
        .key_len = 55,
        .data = {0xeb,0x39,0x1d,0x78,0x04,0x74,0x1b,0xb2,0xfc,
                 0xe4,0x06,0x7e,0x2e,0xc6,0x3a,0x98,0xfd,0x41,
                 0x4c,0x11,0xc5,0x92,0x9e,0x94,0x12,0x66,0x30,
                 0xee,0x10,0x87,0xb9,0xcd,0x43,0xbb,0xf7,0xc8,
                 0x78,0xea,0x43,0xa0,0xfc,0x7e,0x68,0xc6,0x0b,
                 0x0f,0xb5,0x1d,0x8b,0x40,0x51,0x00,0xfb,0xe4,
                 0xcf,0x87,0xa6,0x66,0x02,0x06,0x1c,0x67,0x13,
                 0xd7,0xe2,0xa8,0xd0,0x5a,0x69,0x2b,0x47,0x39,
                 0x3d,0x95,0x94,0xf1,0x7f,0xd7,0xf7,0x8e,0x95,
                 0x0d,0x2a,0xb5,0x20,0xa6,0xf1,0xe8,0x2e,0xc6,
                 0xf2,0x06,0xb2,0xe8,0xc7,0x11,0x31,0xc8,0x52,
                 0x34,0xbd,0x80,0x50,0x05,0x27,0xf1,0x31,0x07,
                 0x71,0x64,0x28,0x73,0x82,0x71,0x7e,0x38,0x1d,
                 0x21,0x2b,0x40,0x14,0x4f,0xa7,0xf5,0xb9,0x54,
                 0xfe,0x04},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0xaf,0x9d,0x53,0xa7,0x6a,0xfb,0x25,0x04,0xb7,
                0x6b,0x96,0x59,0xd4,0x04,0x11,0x24,0xcc,0xa9,
                0x05,0x6d},
    }, {    // 15 (106)
        .key = {0xc5,0xf1,0xb3,0xcb,0xdd,0x47,0x1b,0x97,0xa9,
                0x94,0x5f,0x90,0xb7,0xc3,0x0c,0x18,0xb0,0x61,
                0xcc,0x44,0x26,0x53,0x45,0x33,0x8c,0xc4,0x3e,
                0x9e,0x86,0x62,0x03,0xb1,0x64,0xca,0x0b,0x19,
                0xe8,0x40,0xd9,0x5f,0x50,0xa4,0x20,0x74,0x90,
                0xa1,0x2a,0x8d,0xe0,0x28,0x7a,0x30,0x76,0x1d,
                0xb1},
        .key_len = 55,
        .data = {0x4f,0x24,0x59,0x09,0xbf,0x3d,0x0f,0x29,0xdb,
                 0xb5,0x83,0x77,0x32,0x2f,0xdf,0x46,0x5c,0x2d,
                 0xaa,0xd2,0x32,0x79,0x9a,0xc8,0x3f,0xc5,0xa9,
                 0x09,0x71,0x8e,0x6a,0xab,0x14,0x46,0x9e,0x41,
                 0x01,0xb1,0x03,0x5d,0x13,0x90,0x6c,0x53,0xdb,
                 0x7d,0x16,0x3f,0x95,0x16,0x3a,0x49,0x5a,0x5d,
                 0xdf,0x27,0x97,0x5f,0xf2,0xe4,0xbd,0x14,0x98,
                 0xe0,0x3a,0x87,0xc3,0x19,0xa9,0x36,0x04,0xa8,
                 0x45,0xbc,0xa0,0x8f,0xa4,0xad,0xe5,0xb1,0xa2,
                 0xc2,0xbe,0x22,0x9b,0x36,0x86,0xa1,0xdd,0x6f,
                 0x1a,0x63,0x5c,0xe1,0x32,0x13,0xa8,0x30,0xf1,
                 0x16,0x1d,0x94,0xba,0x13,0x59,0x82,0x46,0xb3,
                 0xf4,0x89,0xf3,0xa5,0x6b,0x7a,0x5f,0xe6,0x5c,
                 0xcb,0x36,0x3f,0xf9,0x3d,0x48,0x0f,0xb7,0xbd,
                 0xdf,0x4a},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x6b,0x30,0x31,0x24,0x13,0x75,0x7c,0x67,0xa1,
                0xa1,0x7e,0xc1,0x58,0x35,0xe1,0xb7,0xe6,0x58,
                0x65,0x77},
    }, {    // 16 (120)
        .key = {0x0d,0xf4,0x0c,0xc5,0x47,0xac,0xed,0x4a,0x85,
                0xca,0x48,0x57,0x40,0x35,0x7c,0xa5,0xdd,0x07,
                0x45,0xcb,0x27,0x36,0xd5,0x05,0xfc,0x73,0x27,
                0x33,0x46,0x52,0x90,0x25,0x2e,0xac,0x97,0x15,
                0x8c,0x42,0x06,0x75,0xa1,0xa2,0x41,0xec,0x08,
                0x4f,0xd7,0x9b,0xea,0x1c,0x57,0xa4,0x72,0xd9,
                0x65},
        .key_len = 55,
        .data = {0x95,0x9b,0xb5,0xd8,0x4a,0xd0,0x5a,0x55,0xca,
                 0xad,0xef,0x57,0xc7,0xea,0xd8,0xfc,0xf9,0x33,
                 0xa2,0xef,0x6e,0x2c,0xf3,0x8c,0x22,0xb1,0xc1,
                 0x45,0xcf,0x62,0xc3,0x44,0x06,0x5f,0x25,0x51,
                 0x7f,0x50,0x80,0xf7,0xae,0xa9,0x09,0xe2,0x96,
                 0x2c,0x09,0x8c,0xe2,0xaf,0x8d,0x19,0x32,0x1f,
                 0x5e,0x7f,0xb3,0xc5,0x33,0xe9,0xf2,0xc8,0x26,
                 0x5e,0xf2,0xf3,0x65,0x9d,0x7c,0xb9,0x2a,0x4c,
                 0xb6,0x76,0xed,0x52,0xf7,0xf8,0xd8,0x82,0xda,
                 0x3e,0x7b,0x79,0x91,0x7f,0xc1,0xc9,0x7c,0x6f,
                 0x22,0x57,0xfa,0xbd,0x1b,0x01,0x80,0xe0,0xdb,
                 0x84,0xe8,0x13,0x40,0x9d,0x0b,0x9e,0x01,0x6f,
                 0x91,0x58,0x2d,0x25,0x08,0x6a,0xd4,0xd8,0x34,
                 0x55,0x03,0xb3,0xa6,0x37,0xf6,0x55,0x90,0x4a,
                 0x8a,0x76},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x3b,0x47,0xb7,0x8b,0x7d,0xd5,0xe4,0xc0,0x09,
                0xa8,0x21,0x67,0x9e,0x8f,0x7d,0x84,0xd2,0x04,
                0x07,0x40,0x41,0xbd,0xa9,0xe4},
    }, {    // 17 (121)
        .key = {0x5a,0xd5,0x01,0x55,0xc4,0x65,0x26,0xa9,0xbe,
                0xd2,0x0d,0xac,0xcc,0xa0,0xf7,0x29,0xdf,0x0d,
                0xff,0x63,0x79,0xa2,0x58,0x4f,0x11,0x09,0xd7,
                0x00,0xb9,0x1e,0xb1,0x63,0xfe,0x48,0x85,0x2c,
                0x10,0x0a,0x9d,0x10,0x23,0xd5,0x71,0x35,0x14,
                0x21,0x6c,0xe7,0x50,0xbc,0xf3,0x8c,0xf2,0x61,
                0xa4},
        .key_len = 55,
        .data = {0xc3,0x3d,0x3d,0x1f,0x44,0x27,0x76,0xd5,0x46,
                 0xf4,0xd5,0xa2,0x5b,0x7d,0x23,0x40,0x2a,0x5f,
                 0xd6,0x5e,0x6e,0xf3,0x33,0x3a,0x42,0x81,0xb5,
                 0x72,0x9b,0xb0,0xb2,0x18,0x16,0x41,0x3c,0x04,
                 0xf4,0xfd,0x46,0x6e,0x62,0xec,0x07,0xbc,0xfe,
                 0xc8,0xa4,0x98,0x98,0xb1,0x09,0x46,0x35,0x22,
                 0x17,0x34,0x5a,0x24,0x05,0xd3,0x87,0xc2,0x00,
                 0xa6,0x01,0xbc,0x15,0x99,0xd4,0x80,0x51,0x93,
                 0x19,0x55,0x27,0x4e,0x75,0xda,0x11,0x67,0xe2,
                 0xaf,0x7a,0xb1,0xb2,0x72,0xdc,0xcf,0xd1,0xed,
                 0x26,0x02,0x4a,0x8d,0x60,0x3f,0x0c,0x16,0x91,
                 0x85,0xef,0x96,0xe1,0x6d,0xf2,0x98,0xfb,0x03,
                 0xc6,0x99,0xb5,0xd4,0x91,0xc0,0xf3,0x48,0xec,
                 0xc9,0xc8,0x32,0x2d,0x43,0xdf,0xa6,0xec,0xfb,
                 0x1a,0x0b},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0xc6,0x05,0xa8,0x4b,0x7f,0x4d,0x75,0x71,0xc4,
                0xf3,0xbc,0xee,0x28,0x3d,0x45,0xe2,0x7a,0x58,
                0x13,0xd8,0x00,0xfa,0x9d,0xc9},
    }, {    // 18 (135)
        .key = {0xb3,0x4a,0x6e,0xba,0x59,0xe6,0x32,0xe3,0xc3,
                0x34,0xdf,0x58,0x2f,0xd0,0xb0,0x3a,0xca,0x7f,
                0x64,0x1c,0x69,0x14,0x92,0x0a,0x79,0x94,0x3d,
                0xca,0x39,0x98,0xa8,0x61,0x27,0xf3,0x6b,0xda,
                0xb7,0x95,0xc7,0x42,0x4b,0xaf,0x37,0xf7,0x60,
                0x18,0x47,0x23,0x05,0xf0,0xa9,0x83,0x92,0x83,
                0x86},
        .key_len = 55,
        .data = {0xd0,0x3d,0xd9,0x4e,0x43,0xd9,0x6c,0xa4,0x59,
                 0x92,0x8a,0xa9,0x6d,0x2b,0x81,0xc3,0x5e,0xd5,
                 0x45,0x66,0xe3,0x3e,0x66,0x35,0x1f,0xca,0x40,
                 0x6b,0x76,0x20,0x72,0x7a,0x9e,0xe9,0x91,0xf2,
                 0xf9,0xd4,0x1d,0xa3,0x22,0xde,0xb3,0x06,0xc6,
                 0xd0,0x85,0xd9,0xb5,0x09,0x08,0x0a,0x38,0x7d,
                 0xec,0xd6,0xa6,0xfe,0x51,0x32,0x32,0xbb,0x38,
                 0x6a,0x07,0x06,0x37,0x08,0x97,0x5a,0x72,0xf7,
                 0x2f,0x9c,0xc6,0xe8,0xcf,0xa1,0x47,0xf5,0x3a,
                 0xf1,0xed,0x84,0x49,0xca,0x8a,0x6f,0x84,0x68,
                 0xff,0x62,0xf3,0x84,0xb0,0x84,0x32,0x1b,0x35,
                 0x59,0xc4,0x70,0x53,0xe7,0xa9,0x54,0x2a,0x17,
                 0x33,0xa5,0xaf,0x5f,0x15,0x15,0x5d,0x9e,0xbd,
                 0x2c,0x28,0x47,0xe4,0x91,0xb3,0xc2,0x63,0x85,
                 0xc2,0x0f},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0x1c,0x49,0x44,0x84,0x3c,0x3e,0xe7,0xf9,0x8a,
                0xb5,0x29,0x87,0xe0,0xd2,0xd4,0x49,0x4a,0xa7,
                0x25,0x48,0xbf,0xdc,0x36,0x02,0xdb,0x45,0xb5,
                0xf4},
    }, {    // 19 (136)
        .key = {0x2f,0xd5,0x86,0x3a,0xb5,0xac,0x01,0x09,0xcd,
                0x1c,0xc8,0xa6,0x59,0x8e,0x75,0xd8,0x58,0x11,
                0xa8,0x4d,0x0d,0xf1,0x4c,0xd5,0x5e,0x8b,0x1c,
                0xce,0x7a,0x5f,0x65,0xdf,0xbe,0x67,0x0d,0xea,
                0xda,0xa8,0xd4,0x3b,0x2f,0x06,0xda,0x06,0x7c,
                0x5c,0x62,0x10,0xba,0xcc,0xd5,0xac,0x44,0x54,
                0x0a},
        .key_len = 55,
        .data = {0x85,0xc0,0x2d,0x7c,0xfa,0xb2,0x9f,0x8a,0xdf,
                 0x0f,0xa5,0x5e,0xf3,0x67,0x22,0xa0,0x47,0x57,
                 0xc8,0x86,0x50,0x53,0xd2,0xaf,0x3b,0xa2,0xf6,
                 0x4e,0x80,0xaa,0x95,0x8a,0xba,0x6e,0x36,0x25,
                 0xb6,0x55,0x32,0x5c,0xca,0x2d,0xb0,0x0f,0x68,
                 0x6f,0xd4,0x22,0xf2,0xc5,0x34,0x23,0xd0,0xc9,
                 0x8c,0x2d,0xc1,0x10,0xb2,0x0c,0x6e,0x67,0xcc,
                 0xa1,0x45,0x5c,0xc0,0x88,0x84,0x01,0xec,0xf9,
                 0x94,0xec,0x18,0xec,0x99,0x82,0xa8,0x81,0x47,
                 0x76,0x16,0x9e,0xf7,0x8c,0xa0,0xda,0xfa,0xa3,
                 0x3e,0x9a,0x2d,0xf2,0xd7,0x79,0xcd,0x92,0xb4,
                 0xee,0x8d,0x3c,0x35,0x29,0xe6,0x55,0xc3,0x3d,
                 0xaf,0x27,0x05,0x84,0xed,0x72,0x57,0x3f,0xec,
                 0x23,0x78,0x7e,0x8f,0x63,0x82,0x40,0xe4,0xd3,
                 0x20,0xda},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0xea,0xbb,0xa9,0xf3,0x5b,0xa3,0x9c,0xfb,0x92,
                0x83,0x39,0x0d,0x54,0x25,0x68,0x7c,0xdd,0x70,
                0xd4,0xcb,0x1f,0xea,0x43,0x39,0x25,0x64,0x7c,
                0x79},
    }, {    // 20 (150)
        .key = {0xf4,0x29,0xce,0x00,0x3a,0x8a,0x56,0xac,0xa9,
                0x28,0xf8,0x8b,0xf9,0xad,0x4e,0x22,0xb2,0x4e,
                0x43,0xfa,0x42,0x95,0xca,0x8d,0x4c,0x64,0xd0,
                0x67,0x44,0x37,0x6e,0x85,0x39,0xa9,0x64,0x01,
                0xf1,0x3a,0xc3,0xf1,0x3b,0x4f,0xe9,0x8c,0x81,
                0x7a,0xb2,0x8e,0x9d,0xfc,0xed,0xde,0xed,0x88,
                0x0d,0x6e,0xaa,0x4a,0x21,0x63,0x75,0x28,0x07,
                0x77},
        .key_len = 64,
        .data = {0x71,0x95,0xbe,0xf6,0xe6,0x3a,0x04,0xc5,0xd4,
                 0xa5,0x65,0xff,0x52,0xe0,0xd3,0x99,0x17,0x19,
                 0xd3,0xd6,0xea,0x48,0x8e,0x0a,0x59,0x12,0x25,
                 0xa4,0xae,0xed,0x46,0x6b,0x1f,0x86,0xfd,0x08,
                 0x45,0x60,0xcc,0xc5,0xd0,0xbd,0xfd,0x94,0x78,
                 0xc1,0x37,0x53,0x37,0xd5,0x50,0x4e,0xb5,0x4a,
                 0x62,0x06,0x04,0x3e,0x21,0x31,0x8d,0xc4,0x6b,
                 0x01,0x4a,0x3f,0x21,0xa9,0x36,0x0f,0x36,0x1e,
                 0x02,0x70,0x05,0x1a,0x0a,0x9e,0xe8,0x67,0x87,
                 0x3e,0x06,0x5a,0x06,0xf9,0x6c,0x9e,0x19,0x6a,
                 0x60,0x32,0x4c,0x3a,0xf1,0xb5,0x5a,0x89,0xd3,
                 0xe2,0xa1,0x90,0x6a,0x57,0xd4,0x85,0x25,0x9a,
                 0xf0,0xbf,0x0d,0x5d,0x71,0x1b,0x9e,0x01,0xc5,
                 0x9e,0xd5,0x5a,0x01,0x5c,0x8a,0xb2,0xd3,0x98,
                 0x00,0x10},
        .data_len = 128,
        .mac_len = 14,
        .mac = {0xf5,0xe7,0xc5,0x7d,0x68,0xeb,0x37,0x62,0x6c,
                0x4d,0x14,0xe9,0x14,0xc7},
    }, {    // 21 (151)
        .key = {0x99,0xe9,0xa3,0xe7,0x71,0xc6,0x1c,0x89,0xa9,
                0x04,0xf0,0xa1,0xe2,0x0e,0xf0,0x8f,0x92,0xd6,
                0x50,0x83,0xb3,0xbd,0xff,0x87,0xb8,0xb7,0x34,
                0xc0,0xc5,0xaa,0x4a,0xf0,0x1f,0x18,0xcd,0x40,
                0xe2,0xfe,0xd5,0x3d,0xf2,0xfc,0x92,0x20,0x33,
                0xb5,0x29,0x52,0xfd,0x79,0xbe,0x2f,0xca,0x22,
                0xd2,0x0b,0x32,0x3b,0xd0,0x72,0x1a,0x68,0xe2,
                0xb4},
        .key_len = 64,
        .data = {0xea,0x75,0x33,0x86,0xd4,0x48,0x44,0xfc,0xbd,
                 0xec,0x77,0x3f,0xa1,0x7d,0x50,0x55,0xba,0x5b,
                 0x78,0xd1,0xa4,0xc5,0x06,0x03,0xbe,0xaa,0xfe,
                 0x6c,0xd3,0xbb,0x17,0x74,0xfa,0xc5,0xc2,0x60,
                 0xba,0x3e,0x67,0x7e,0x72,0x1e,0xf2,0xa2,0xf8,
                 0xf0,0x8e,0xd6,0x81,0x28,0xb0,0x4e,0xc1,0xb7,
                 0x58,0x9a,0x53,0xdd,0x94,0x1f,0x5a,0x3e,0x45,
                 0xc8,0x69,0x39,0x25,0xf9,0xa0,0x75,0x09,0xc5,
                 0x18,0xbc,0x64,0x60,0x82,0x0d,0x0d,0xd7,0x0a,
                 0xec,0x42,0xfe,0x82,0xfd,0xec,0xaa,0xbc,0xd1,
                 0x24,0x21,0x37,0x00,0xf7,0xcf,0xfe,0x78,0x66,
                 0x3b,0xce,0x14,0xb1,0x39,0x77,0x71,0xd6,0x60,
                 0x84,0xfd,0x54,0xbe,0x1e,0x50,0x34,0xcb,0x2e,
                 0xc0,0x4a,0x9c,0xc2,0x2f,0xb9,0x59,0xb2,0x28,
                 0xb5,0xf1},
        .data_len = 128,
        .mac_len = 14,
        .mac = {0x12,0x51,0x02,0x95,0xbb,0x35,0x34,0x36,0x9b,
                0xf5,0xd8,0x30,0xb1,0xad},
    }, {    // 22 (165)
        .key = {0xf0,0x0f,0x23,0x44,0xb9,0xff,0xa8,0xef,0x49,
                0xdb,0xb6,0x23,0xd0,0x1c,0xac,0x56,0x3a,0x13,
                0x69,0x76,0x0f,0x37,0x57,0xdb,0x97,0xbe,0xa9,
                0x7a,0x41,0x6a,0xe0,0x5b,0x45,0xa6,0x94,0x4d,
                0x89,0xd1,0x71,0xd8,0xc2,0xda,0x80,0x73,0xdb,
                0x0d,0xa7,0xf7,0x9a,0x2c,0x74,0x54,0x81,0xc2,
                0x16,0x82,0xf2,0xb4,0xac,0xbb,0x97,0x19,0xc3,
                0xe8},
        .key_len = 64,
        .data = {0x88,0x2d,0x04,0x1c,0x2a,0x6f,0x05,0x05,0xef,
                 0xe2,0xa6,0x6c,0x17,0x92,0x59,0x52,0xaf,0x30,
                 0x95,0x9b,0x1a,0x5a,0x13,0x6f,0xf1,0x1b,0x3d,
                 0xe1,0x0d,0xb6,0xe4,0xce,0xe1,0x9f,0x31,0x08,
                 0x0d,0xcb,0xde,0xb4,0x31,0x29,0xa5,0xf1,0xff,
                 0x71,0xf9,0xbb,0x95,0x1c,0xf5,0x0e,0x09,0xb3,
                 0x92,0x4e,0x45,0x4d,0x1c,0xe6,0x15,0x54,0xe7,
                 0x30,0x7e,0x87,0x3e,0x95,0x52,0x45,0x9c,0xf5,
                 0x01,0x08,0x1f,0x48,0xb2,0x30,0x39,0x86,0x92,
                 0x02,0xa9,0xc5,0x6c,0xf0,0xa9,0xa1,0x7b,0x1a,
                 0x69,0xe1,0x7c,0x16,0xbd,0x58,0x06,0xec,0x12,
                 0x08,0x1e,0x65,0xa7,0x8e,0x07,0x86,0xfa,0xba,
                 0x57,0x57,0x80,0x7d,0x50,0xe9,0x98,0x08,0x6c,
                 0x96,0xc2,0x32,0x3a,0x8b,0x0c,0x1a,0x69,0x84,
                 0xce,0x0e},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xab,0x15,0x00,0x58,0x48,0x0e,0xfe,0x9b,0x92,
                0x43,0xbf,0x6d,0x59,0xb1,0xea,0x0f},
    }, {    // 23 (166)
        .key = {0x41,0xc9,0x0a,0xec,0xc4,0x6f,0xc8,0xe1,0x93,
                0x38,0x0e,0x6e,0x14,0x7f,0x1b,0xe2,0x2a,0x2c,
                0x85,0x8b,0xb6,0x2c,0xdb,0xcf,0x5e,0x1e,0x88,
                0x78,0x8d,0x4f,0xfa,0x50,0xba,0xe9,0x04,0xee,
                0xee,0x67,0x81,0xcf,0x80,0x4c,0x3b,0xa9,0x23,
                0x15,0x0b,0xfb,0x24,0x6c,0x41,0xe9,0xec,0x9e,
                0x45,0x47,0xcc,0x4f,0x7f,0xf4,0xfa,0x6d,0x75,
                0x69},
        .key_len = 64,
        .data = {0x92,0x24,0x50,0x54,0x1f,0xe8,0xfc,0xa6,0x6a,
                 0x8b,0xcd,0x46,0x91,0x3c,0x86,0xfa,0x15,0x0b,
                 0x44,0x7c,0x99,0xa0,0x61,0xce,0xe7,0x2d,0x99,
                 0xce,0x34,0xa1,0x6b,0x0a,0xa5,0x1f,0xc2,0xd5,
                 0x12,0xae,0xd5,0x9b,0x09,0x32,0x4c,0x71,0x16,
                 0x9b,0xa4,0xf0,0x41,0x5c,0xd4,0x44,0xd9,0x1e,
                 0x31,0x80,0x70,0xb6,0x8f,0x34,0x27,0x55,0xa9,
                 0x42,0x2c,0xff,0xc3,0xed,0x80,0x3f,0x8d,0x33,
                 0xfe,0x18,0x4b,0x5d,0x99,0x3e,0x33,0xd4,0xe5,
                 0x87,0x16,0x36,0xd9,0x9c,0x43,0xa9,0xd3,0xf4,
                 0xa9,0x70,0xdc,0x03,0x3b,0xbc,0x2d,0xae,0x99,
                 0xc4,0xf3,0x03,0xec,0x17,0x50,0x27,0x11,0x31,
                 0xa2,0x8f,0xfb,0x4d,0x07,0x73,0x52,0x7b,0x21,
                 0x80,0x60,0xf2,0x16,0xce,0x2e,0xa7,0xdb,0x11,
                 0xea,0x1f},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xf8,0x4d,0x32,0x38,0x59,0x67,0x0d,0x90,0x05,
                0xe2,0x83,0x06,0x5a,0x76,0xae,0xfc},
    }, {    // 24 (180)
        .key = {0x0a,0xd0,0x88,0x72,0xef,0x79,0xc9,0x40,0x18,
                0x89,0x35,0x69,0x95,0x64,0xf1,0x8e,0x5c,0xb7,
                0x46,0xe7,0x8d,0xae,0xff,0x1f,0xfa,0x68,0x12,
                0x65,0xe6,0xa8,0x83,0xc0,0x0d,0xcd,0x86,0xa2,
                0xe5,0x04,0xe4,0xc1,0x24,0xbc,0x9f,0x22,0x82,
                0x5a,0xd9,0x76,0xc0,0x2e,0xf2,0xaa,0xe9,0xa6,
                0xf1,0xba,0xd8,0xc2,0x25,0xc6,0xfa,0xc8,0x0c,
                0x67},
        .key_len = 64,
        .data = {0xca,0xdf,0xc1,0xa6,0x87,0xa6,0xb4,0x36,0xe1,
                 0xbd,0x3f,0x6f,0xeb,0xbf,0x19,0x78,0x41,0xa6,
                 0xeb,0xae,0xe0,0x49,0xbd,0x44,0x3b,0x24,0xae,
                 0x2c,0x8b,0x58,0xe9,0x3b,0x66,0xf2,0xd4,0x88,
                 0x93,0x70,0x8b,0x15,0x1d,0x82,0x8c,0xb1,0xa6,
                 0x0d,0xb4,0x1c,0xfe,0x75,0x6a,0x6d,0xf4,0x1c,
                 0x1a,0xb1,0x6c,0x26,0xe5,0xd5,0xb2,0x80,0x96,
                 0xc6,0x0e,0x20,0xd8,0xd6,0x99,0x8e,0xa7,0x62,
                 0x4e,0xa3,0x05,0xb1,0x64,0x90,0xee,0x20,0xdc,
                 0x23,0x8d,0x7c,0x56,0x46,0xb0,0xb0,0x28,0xc9,
                 0x7a,0xa0,0xb1,0x61,0x57,0x54,0xeb,0xf0,0x35,
                 0x59,0x40,0x06,0xf2,0x64,0x9f,0xa8,0x10,0x39,
                 0xce,0x51,0xb0,0xcb,0x02,0x37,0xbe,0xa5,0xad,
                 0xcf,0x69,0x88,0x97,0x93,0xd6,0x56,0x30,0x03,
                 0x15,0x2e},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x1f,0x60,0xfc,0xa9,0x69,0x4f,0x82,0x54,0x77,
                0x8f,0x18,0xca,0xb6,0xc2,0x86,0xa0,0x4f,0x89,
                0xb3,0x71},
    }, {    // 25 (181)
        .key = {0x59,0x88,0xc7,0x94,0xc1,0xf1,0xe8,0x5d,0x23,
                0xd6,0x5b,0xe0,0x40,0xc0,0x12,0x9b,0xb8,0xa6,
                0xbb,0xcc,0xd8,0x6c,0x3b,0x1e,0xb3,0xa9,0x58,
                0x87,0x74,0xad,0xb5,0x71,0xf2,0xc3,0x04,0x18,
                0x85,0xb3,0x77,0x33,0x19,0x8b,0x77,0xd6,0x80,
                0x9f,0x99,0x97,0x0d,0xcf,0xce,0xf0,0x5e,0x08,
                0xda,0xe4,0x79,0x0e,0x07,0xe5,0x1b,0x78,0x1a,
                0xf6},
        .key_len = 64,
        .data = {0x40,0xae,0x14,0x83,0x42,0x14,0xdf,0x6b,0xf1,
                 0x10,0x10,0xa3,0x21,0x33,0xc6,0x63,0x7b,0x9b,
                 0x79,0xd3,0xd7,0xc3,0x80,0x7a,0xed,0x9f,0xd6,
                 0xf9,0x2d,0xe9,0x1d,0xf9,0xeb,0xb6,0xe3,0x62,
                 0xbf,0x4c,0x25,0xcd,0x2c,0x37,0x11,0x3d,0x7f,
                 0x9a,0x10,0x09,0xfa,0xc4,0x04,0x1d,0x32,0x03,
                 0xce,0x63,0xb6,0xbd,0x14,0x7b,0xf1,0xbc,0x41,
                 0xb2,0x52,0xca,0xcd,0x32,0xaa,0x7e,0xbe,0x76,
                 0xd5,0x55,0x6f,0x01,0x9c,0x04,0x90,0x50,0xda,
                 0x0d,0x6e,0x0d,0xde,0x91,0x74,0x85,0x11,0x65,
                 0xf2,0x5c,0xd8,0x68,0x1e,0x7f,0xbc,0x13,0x59,
                 0x5b,0xc9,0x51,0xb3,0x4d,0x33,0x91,0xb0,0x59,
                 0x40,0x1a,0x4d,0x1e,0x61,0x9f,0x3e,0x09,0xa2,
                 0x14,0x7a,0xca,0x45,0xc0,0xf9,0x04,0xaa,0x92,
                 0xbc,0x49},
        .data_len = 128,
        .mac_len = 20,
        .mac = {0x25,0xc4,0xde,0x6c,0x4e,0x70,0x53,0xec,0x00,
                0xc2,0x9d,0x0e,0x95,0xb5,0x1c,0xcc,0x78,0xc1,
                0xf1,0x10},
    }, {    // 26 (195)
        .key = {0xc7,0x34,0x1f,0xb6,0x18,0xef,0x24,0x48,0xc1,
                0x1c,0x67,0x61,0x11,0x9e,0xd1,0x97,0x56,0x56,
                0x4c,0x78,0xf6,0xe9,0x7b,0x72,0xbb,0xc4,0xa8,
                0xf0,0x6f,0x5e,0xfe,0xc6,0xa9,0xbd,0xf8,0xcf,
                0xe6,0x3c,0x19,0xd4,0xdb,0x8b,0xde,0xb0,0x0a,
                0xf7,0xf1,0xb2,0x94,0xec,0x9f,0x43,0x4e,0xb8,
                0x8c,0xc9,0x16,0x91,0xa5,0xcb,0x68,0xda,0x57,
                0x55},
        .key_len = 64,
        .data = {0xe2,0xdf,0x37,0xf7,0x50,0xaa,0xdc,0x39,0x3a,
                 0x1d,0x30,0x72,0x2f,0x9c,0xda,0x3d,0xaf,0xb6,
                 0xfd,0xc3,0x90,0xfa,0xba,0x2f,0xa1,0x0c,0xf7,
                 0x62,0x8e,0xf8,0x5f,0xe7,0x03,0xc5,0xaa,0x29,
                 0x14,0x4e,0x1f,0xe3,0xdd,0x4d,0x57,0xd9,0x65,
                 0xb9,0x43,0xc6,0xc9,0x52,0x71,0x95,0x96,0x7a,
                 0x8a,0xd3,0x76,0x46,0x20,0x65,0x0f,0xd5,0xa6,
                 0x76,0x28,0xe0,0xc0,0xb4,0x0d,0x93,0xbe,0xbc,
                 0x35,0x9b,0x62,0xd2,0x23,0x81,0xea,0xd6,0x8a,
                 0x36,0x4a,0x73,0xb4,0xef,0xba,0x9a,0x24,0x32,
                 0x70,0x2f,0x7b,0x31,0xaf,0x02,0x98,0x58,0x00,
                 0x00,0x03,0x22,0xc0,0xab,0x8c,0x30,0xda,0xc7,
                 0x2f,0x9c,0xd6,0x56,0x3b,0x8b,0x13,0xc7,0xb4,
                 0x01,0x50,0x6b,0xf5,0xfa,0x71,0x28,0x99,0xb6,
                 0x80,0x71},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x0a,0x94,0xe0,0x61,0xe9,0x65,0xf1,0x09,0x81,
                0x02,0x96,0x6f,0x5e,0xf8,0xf0,0x31,0x07,0x6b,
                0x13,0xd8,0xb5,0xb3,0x62,0xd0},
    }, {    // 27 (196)
        .key = {0x7d,0x40,0x85,0xba,0xfc,0xc1,0x42,0x00,0xf7,
                0xad,0x29,0x8c,0xd0,0xed,0x4c,0xb8,0xc9,0xc8,
                0xa9,0x1c,0x6c,0xb3,0xac,0x02,0x2a,0x6a,0xf6,
                0x1a,0x42,0x3f,0x5d,0x70,0xe4,0xb2,0x34,0x9a,
                0x96,0x43,0xcc,0x36,0xd8,0x84,0x3a,0x94,0x03,
                0x72,0x5b,0xaf,0x77,0x05,0x16,0x62,0xf5,0x06,
                0xe1,0xd8,0x4a,0x21,0x87,0x45,0x2e,0x3b,0x28,
                0xd3},
        .key_len = 64,
        .data = {0x19,0x2a,0xdf,0x7a,0x27,0x91,0x33,0x1e,0x99,
                 0x5c,0xa9,0x76,0xa8,0x36,0x50,0x21,0xfe,0x66,
                 0x61,0x49,0xb6,0xf6,0xd0,0xe0,0x83,0x14,0x22,
                 0x29,0x2f,0xb8,0x7e,0x7b,0x48,0x87,0x06,0x42,
                 0x16,0xf8,0x68,0x46,0xcf,0xb5,0xaf,0x30,0x0e,
                 0x44,0x60,0x4a,0x82,0xe6,0xc2,0x86,0xe3,0xbf,
                 0x7d,0xc2,0x2b,0x38,0x66,0x4a,0x02,0xa2,0x9a,
                 0x9e,0x7a,0x1c,0xca,0x76,0xfc,0xb6,0xd5,0x04,
                 0x83,0xd6,0x9c,0x79,0x07,0xdb,0x12,0xca,0x3d,
                 0x01,0x92,0xb4,0x7d,0x84,0x76,0xfe,0x03,0x09,
                 0xe5,0xfe,0x5b,0x39,0xec,0x5e,0x17,0xaf,0xd0,
                 0x20,0xc3,0x14,0xa3,0xe0,0x1b,0x88,0x16,0x59,
                 0x7b,0x44,0x8e,0x2b,0x96,0x7e,0x7a,0xdc,0x6b,
                 0x8d,0xe3,0x1c,0x75,0x4e,0x44,0x4d,0x22,0xc6,
                 0x9d,0x42},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x0d,0x47,0xd7,0x80,0x8c,0x1f,0xc2,0xd4,0x0a,
                0x5f,0xb6,0x7f,0x75,0xb0,0xd5,0x54,0x92,0x9e,
                0x40,0x94,0xfc,0x45,0xee,0x88},
    }, {    // 28 (210)
        .key = {0xf3,0xaa,0x51,0xee,0x90,0xaa,0x06,0xe9,0x8e,
                0x23,0x88,0xdf,0x7a,0x3a,0xf2,0xcd,0x69,0x7f,
                0x2a,0x52,0x8f,0x2a,0x14,0x14,0x0d,0xa4,0x05,
                0x60,0x0b,0x4a,0x7b,0x10,0x07,0x6d,0x0b,0xb2,
                0x6c,0x9d,0x9a,0xe6,0x67,0x27,0xd0,0x08,0xf6,
                0xdc,0xca,0x0f,0x42,0x14,0x0f,0xb5,0x2e,0xf3,
                0xc6,0xd9,0x30,0xeb,0x26,0x21,0x60,0x82,0x2d,
                0xe3},
        .key_len = 64,
        .data = {0x2c,0x07,0x0e,0x5e,0xb0,0xa7,0xef,0xbf,0xc4,
                 0x0b,0x23,0x43,0x14,0xc0,0x55,0xfc,0x43,0x6c,
                 0xe1,0xe2,0x30,0x05,0x39,0xb3,0x78,0x42,0x87,
                 0xfd,0x3c,0x4f,0x94,0x78,0x24,0xc5,0xe8,0x9a,
                 0xa3,0xd9,0x33,0x66,0x7d,0xd4,0xeb,0x85,0x87,
                 0xc3,0x37,0x97,0xae,0x6f,0x0c,0xcb,0x3b,0x8f,
                 0x95,0xad,0x56,0x3d,0xd9,0x40,0xc8,0xa7,0x98,
                 0x4d,0x05,0x07,0x75,0xfa,0x69,0xd5,0x5b,0x9e,
                 0xc2,0x3a,0x19,0xd4,0x0f,0xc9,0x4f,0xcf,0x87,
                 0x6a,0x8e,0xed,0xdd,0x96,0xbb,0x8e,0xc3,0xca,
                 0xb4,0x26,0xad,0x35,0x36,0x91,0xd7,0xb4,0xff,
                 0xa7,0x89,0x01,0x2f,0x11,0x9a,0x28,0xf4,0x5b,
                 0x33,0x3a,0x66,0x49,0xd5,0x44,0xb0,0x6f,0x9d,
                 0x82,0x71,0xfc,0xc7,0xd1,0x62,0xd7,0x72,0x63,
                 0x38,0xed},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0x69,0x5c,0x32,0x8d,0xc8,0x58,0xa4,0x6b,0xe7,
                0xac,0x8b,0x8d,0xeb,0xf5,0x8d,0xd9,0xa9,0xbd,
                0x72,0xd7,0xa4,0x08,0xa4,0x3e,0x6b,0xbc,0x69,
                0xa3},
    }, {    // 29 (211)
        .key = {0xa2,0x7f,0x5b,0xa4,0xa0,0xd5,0xa8,0x0c,0xb4,
                0xef,0x9b,0xb1,0x8d,0x4f,0xfc,0x4a,0xd4,0x87,
                0x68,0x0c,0xb5,0xa8,0xf6,0xc6,0x9c,0xfe,0x1b,
                0x29,0x3c,0xfa,0xcb,0x67,0xd6,0x70,0xe1,0x01,
                0xb8,0x03,0xc1,0xd9,0x04,0xdd,0x3c,0x8f,0xb2,
                0xa3,0xb4,0x0a,0xd8,0xa7,0x80,0x55,0x29,0x60,
                0x91,0xad,0xbc,0x18,0x56,0xd8,0x10,0xe5,0xde,
                0xb8},
        .key_len = 64,
        .data = {0x80,0x2e,0xd2,0x88,0x73,0xc2,0x16,0x05,0x62,
                 0xfb,0x91,0x62,0x24,0x6b,0x60,0xd7,0x5e,0x2f,
                 0xfb,0x8c,0xec,0xc5,0xbb,0x08,0x83,0x1e,0x9a,
                 0xec,0x94,0xf4,0x3b,0xe1,0x73,0x5e,0x2a,0xcf,
                 0xb8,0xa2,0x64,0x50,0xe6,0x4a,0x1a,0x64,0x4a,
                 0x7a,0xb2,0x66,0x26,0x80,0x0f,0x3e,0x0f,0xad,
                 0x12,0xd9,0x94,0x63,0x60,0xd6,0x20,0x70,0x66,
                 0x82,0x22,0x9a,0x9b,0x07,0x6a,0x7b,0x05,0xe0,
                 0xd6,0x90,0xff,0x90,0x2f,0x8c,0x39,0x94,0x4e,
                 0xf2,0x00,0x96,0x83,0xe2,0xc0,0xe3,0x37,0x33,
                 0xc7,0x1f,0xcf,0xc3,0x4b,0x38,0xd4,0x0b,0xaf,
                 0x98,0x51,0xf7,0x02,0x9f,0x7d,0xf4,0xc9,0x50,
                 0x9a,0x4a,0xfe,0x4a,0x1d,0xac,0x58,0xfa,0x0f,
                 0x0d,0xcd,0x10,0xb3,0xb6,0x56,0x2f,0x69,0x96,
                 0xf2,0xe9},
        .data_len = 128,
        .mac_len = 28,
        .mac = {0xee,0x63,0xe8,0x6a,0xac,0x85,0xf3,0x69,0x61,
                0xd6,0x97,0x45,0x12,0x88,0xcf,0x78,0x00,0x41,
                0xea,0xfe,0x46,0x2c,0xa6,0x55,0x70,0x35,0xd9,
                0x2a},
    }
};


struct HMAC_TEST_VECTOR fips_sha256_hmac_general_test_vector[] = {
    {	// 0 (0)
        .key = {0x6f,0x35,0x62,0x8d,0x65,0x81,0x34,0x35,0x53,
                0x4b,0x5d,0x67,0xfb,0xdb,0x54,0xcb,0x33,0x40,
                0x3d,0x04,0xe8,0x43,0x10,0x3e,0x63,0x99,0xf8,
                0x06,0xcb,0x5d,0xf9,0x5f,0xeb,0xbd,0xd6,0x12,
                0x36,0xf3,0x32,0x45},
        .key_len = 40,
        .data = {0x75,0x2c,0xff,0x52,0xe4,0xb9,0x07,0x68,0x55,
                 0x8e,0x53,0x69,0xe7,0x5d,0x97,0xc6,0x96,0x43,
                 0x50,0x9a,0x5e,0x59,0x04,0xe0,0xa3,0x86,0xcb,
                 0xe4,0xd0,0x97,0x0e,0xf7,0x3f,0x91,0x8f,0x67,
                 0x59,0x45,0xa9,0xae,0xfe,0x26,0xda,0xea,0x27,
                 0x58,0x7e,0x8d,0xc9,0x09,0xdd,0x56,0xfd,0x04,
                 0x68,0x80,0x5f,0x83,0x40,0x39,0xb3,0x45,0xf8,
                 0x55,0xcf,0xe1,0x9c,0x44,0xb5,0x5a,0xf2,0x41,
                 0xff,0xf3,0xff,0xcd,0x80,0x45,0xcd,0x5c,0x28,
                 0x8e,0x6c,0x4e,0x28,0x4c,0x37,0x20,0x57,0x0b,
                 0x58,0xe4,0xd4,0x7b,0x8f,0xee,0xed,0xc5,0x2f,
                 0xd1,0x40,0x1f,0x69,0x8a,0x20,0x9f,0xcc,0xfa,
                 0x3b,0x4c,0x0d,0x9a,0x79,0x7b,0x04,0x6a,0x27,
                 0x59,0xf8,0x2a,0x54,0xc4,0x1c,0xcd,0x7b,0x5f,
                 0x59,0x2b},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x05,0xd1,0x24,0x3e,0x64,0x65,0xed,0x96,0x20,
                0xc9,0xae,0xc1,0xc3,0x51,0xa1,0x86},
        .chunks = {64, 64},
        .num_chunks = 2,
    }, {	// 1 (1)
        .key = {0x17,0xb5,0x28,0x58,0xe3,0xe1,0x35,0xbe,0x44,
                0x40,0xd7,0xdf,0x0c,0xa9,0x96,0xf4,0x1c,0xcb,
                0x78,0xb7,0xd8,0xcc,0x19,0x24,0xd8,0x30,0xfe,
                0x81,0xe0,0xfd,0x27,0x9c,0x13,0x1c,0xe3,0x54,
                0x63,0x03,0xe9,0x5a},
        .key_len = 40,
        .data = {0xe0,0xef,0xf0,0x0f,0x3c,0x46,0xe9,0x6c,0x8d,
                 0x5b,0xd1,0x81,0x28,0x3e,0x46,0x05,0x34,0x8e,
                 0x3f,0xa1,0x0b,0x47,0x94,0x5d,0xe3,0xdc,0xc1,
                 0x59,0xae,0x86,0xe7,0xbd,0x3f,0xdb,0x13,0xf2,
                 0xad,0xa2,0xc3,0x13,0xfc,0xe6,0xa6,0x9e,0xfa,
                 0x49,0xa4,0x70,0x68,0x9b,0x1e,0xf0,0x5a,0xab,
                 0x77,0x8a,0xe1,0x5d,0xd3,0x5f,0xe6,0xfd,0x1e,
                 0x3a,0x59,0xd3,0x51,0xc6,0x8c,0xf8,0xf0,0xff,
                 0xd9,0x68,0xd7,0xe7,0x8b,0x57,0x37,0x7a,0xfc,
                 0xc9,0xdc,0xe3,0xfa,0x5d,0xb1,0xf0,0x6f,0x69,
                 0x85,0xc4,0x41,0x4c,0x0f,0xcc,0x78,0x00,0x30,
                 0xf4,0x9f,0xef,0x79,0x1a,0x6c,0x08,0xed,0xc2,
                 0xa3,0x11,0x08,0x0c,0x37,0x3f,0x00,0xe4,0xb2,
                 0x04,0x4a,0x79,0xd8,0x28,0x60,0xf0,0x87,0x1b,
                 0xc2,0x59},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xc4,0x06,0x14,0x27,0x76,0x4f,0x97,0x94,0x68,
                0xac,0x42,0x28,0x91,0xde,0xa9,0xca},
        .chunks = {32, 32, 32, 32},
        .num_chunks = 4,
    }, {	// 2 (15)
        .key = {0x14,0xd4,0x5c,0xa2,0xa3,0xd4,0x97,0x7d,0xab,
                0x2b,0x7d,0x44,0x2c,0x6f,0x9e,0x57,0xce,0x34,
                0x8e,0x0a,0x6a,0x80,0x8b,0xb3,0xcc,0x7f,0x60,
                0x02,0xb8,0x77,0x89,0x91,0x2a,0xfd,0x98,0xbc,
                0xe2,0x6a,0xd8,0xb3},
        .key_len = 40,
        .data = {0x0c,0x29,0x4a,0x31,0x8b,0x7c,0x1e,0x88,0x46,
                 0x49,0xfe,0x54,0xe4,0xa8,0x72,0x85,0xe4,0x2f,
                 0x86,0x8e,0x3d,0x0a,0x85,0x19,0x41,0x4e,0x05,
                 0xf9,0xc7,0x8b,0x23,0x60,0x89,0xa1,0x10,0x52,
                 0xcb,0xd4,0xcd,0x59,0x3e,0x22,0x32,0x7b,0x23,
                 0xd3,0x35,0x69,0xb3,0x53,0x69,0xf9,0xbf,0x3d,
                 0xc5,0xd6,0x94,0xb8,0xa7,0x76,0x21,0x06,0x18,
                 0x4d,0x5c,0x5a,0x52,0x41,0xe1,0xea,0x80,0x5d,
                 0xdc,0x46,0xc4,0xc9,0x2a,0xe8,0x7e,0xfa,0xbb,
                 0x0c,0xcc,0x26,0x3b,0xc2,0x4d,0xfb,0xf1,0x41,
                 0x2b,0x90,0xe7,0x7e,0x58,0x9c,0x4b,0xfd,0x17,
                 0xe6,0x15,0xe7,0xbf,0xfc,0xea,0x5e,0xbb,0x28,
                 0x40,0x0d,0xd6,0xa0,0xc4,0x03,0xb6,0xfd,0xf8,
                 0xc1,0xa5,0xee,0x21,0x91,0x98,0x2e,0x60,0x1a,
                 0x69,0xb3},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x28,0xf1,0xb6,0x63,0x21,0x30,0x43,0xc4,0xd4,
                0xfb,0x31,0x2b,0xd3,0x6d,0x85,0xfb,0xe6,0x2c,
                0x80,0x08,0xce,0x82,0xaa,0xbc},
        .chunks = {32, 32, 64},
        .num_chunks = 3,
    }, {	//3 (16)
        .key = {0x2a,0x04,0x66,0xdd,0x51,0x5d,0x2f,0x48,0xfe,
                0xc5,0xe7,0x8e,0x22,0xbb,0x22,0xc6,0x06,0xb0,
                0x9e,0x81,0x84,0x69,0x1c,0x51,0x77,0xa4,0x6e,
                0x8c,0x70,0xfe,0xd2,0x4d,0xab,0x14,0x7e,0xbc,
                0x41,0xe9,0x7c,0x8f},
        .key_len = 40,
        .data = {0xd6,0x08,0x12,0x43,0x30,0x98,0xc4,0x46,0x23,
                 0x15,0x91,0x53,0xde,0x7c,0xd2,0x72,0x1b,0x34,
                 0x9f,0x68,0x5c,0x43,0x38,0x8a,0x74,0xc2,0xa3,
                 0xd0,0x4a,0x8e,0x97,0x2a,0xda,0x41,0x99,0x17,
                 0x7c,0x61,0x65,0x73,0x69,0xd7,0x8f,0x90,0x7b,
                 0xa2,0x6a,0x89,0x34,0xcc,0x29,0xd3,0x02,0x9d,
                 0x44,0x15,0xc1,0x10,0x1e,0x3a,0x82,0x83,0xe4,
                 0xc4,0x8b,0xb2,0xb8,0x63,0x9f,0xe6,0x0f,0xc6,
                 0x7f,0x6a,0x57,0xb1,0xb0,0x3f,0xde,0x50,0x7f,
                 0x10,0xef,0xcb,0x43,0x68,0x3e,0x1a,0xe2,0x23,
                 0x85,0x1b,0x96,0x23,0x70,0xe1,0xf1,0x44,0xb7,
                 0x4f,0x1f,0x91,0x89,0xe6,0x6c,0xb8,0x31,0xdc,
                 0x05,0xbb,0xf4,0x6e,0x03,0xe9,0x38,0x77,0xa5,
                 0x0d,0xec,0x40,0xdd,0xe5,0x23,0x9a,0x0f,0xd5,
                 0x02,0x2a},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0X7c,0x2e,0x5f,0x1f,0xdb,0xda,0x3c,0x15,0x35,
                0x36,0xec,0x71,0x36,0x09,0x1e,0xba,0x0b,0xa5,
                0x25,0xb9,0x50,0xbf,0xc8,0x4f},
        .chunks = {128},
        .num_chunks = 1,
    }, {	// 4 (30)
        .key = {0x97,0x79,0xd9,0x12,0x06,0x42,0x79,0x7f,0x17,
                0x47,0x02,0x5d,0x5b,0x22,0xb7,0xac,0x60,0x7c,
                0xab,0x08,0xe1,0x75,0x8f,0x2f,0x3a,0x46,0xc8,
                0xbe,0x1e,0x25,0xc5,0x3b,0x8c,0x6a,0x8f,0x58,
                0xff,0xef,0xa1,0x76},
        .key_len = 40,
        .data = {0xb1,0x68,0x9c,0x25,0x91,0xea,0xf3,0xc9,0xe6,
                 0x60,0x70,0xf8,0xa7,0x79,0x54,0xff,0xb8,0x17,
                 0x49,0xf1,0xb0,0x03,0x46,0xf9,0xdf,0xe0,0xb2,
                 0xee,0x90,0x5d,0xcc,0x28,0x8b,0xaf,0x4a,0x92,
                 0xde,0x3f,0x40,0x01,0xdd,0x9f,0x44,0xc4,0x68,
                 0xc3,0xd0,0x7d,0x6c,0x6e,0xe8,0x2f,0xac,0xea,
                 0xfc,0x97,0xc2,0xfc,0x0f,0xc0,0x60,0x17,0x19,
                 0xd2,0xdc,0xd0,0xaa,0x2a,0xec,0x92,0xd1,0xb0,
                 0xae,0x93,0x3c,0x65,0xeb,0x06,0xa0,0x3c,0x9c,
                 0x93,0x5c,0x2b,0xad,0x04,0x59,0x81,0x02,0x41,
                 0x34,0x7a,0xb8,0x7e,0x9f,0x11,0xad,0xb3,0x04,
                 0x15,0x42,0x4c,0x6c,0x7f,0x5f,0x22,0xa0,0x03,
                 0xb8,0xab,0x8d,0xe5,0x4f,0x6d,0xed,0x0e,0x3a,
                 0xb9,0x24,0x5f,0xa7,0x95,0x68,0x45,0x1d,0xfa,
                 0x25,0x8e},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x76,0x9f,0x00,0xd3,0xe6,0xa6,0xcc,0x1f,0xb4,
                0x26,0xa1,0x4a,0x4f,0x76,0xc6,0x46,0x2e,0x61,
                0x49,0x72,0x6e,0x0d,0xee,0x0e,0xc0,0xcf,0x97,
                0xa1,0x66,0x05,0xac,0x8b},
        .chunks = {32, 0, 64, 32},
        .num_chunks = 4,
    }, {	// 5 (31)
        .key = {0x09,0x67,0x5f,0x2d,0xcc,0x47,0x83,0xb5,0x99,
                0xf1,0x8f,0xb7,0x65,0x58,0x36,0x68,0xa0,0xfd,
                0x8a,0xe4,0x09,0x6f,0x6f,0xcd,0xc6,0x0d,0x4f,
                0x35,0xb4,0x13,0x0f,0xbe,0xfc,0xd5,0x42,0xff,
                0xe7,0x45,0x9d,0x2a},
        .key_len = 40,
        .data = {0x0c,0xf2,0x19,0x8c,0x31,0x37,0x6f,0x5c,0x89,
                 0x15,0x66,0x01,0x37,0x72,0x5f,0x2b,0xbc,0x18,
                 0x0a,0x98,0x6e,0x5a,0x7b,0xda,0x27,0xfa,0x81,
                 0x59,0x3a,0x4a,0x33,0x9b,0xab,0x92,0xcb,0xc3,
                 0x9f,0xb2,0xb8,0x58,0x11,0x08,0xee,0x48,0xc7,
                 0x94,0x81,0x2d,0x84,0x5a,0x72,0xce,0x80,0x08,
                 0xc9,0xe9,0x15,0xd9,0xe3,0x30,0xbb,0xb9,0x0e,
                 0x91,0x36,0xaa,0x53,0xba,0x0e,0x66,0x93,0xdd,
                 0x40,0x46,0xd6,0xb0,0x33,0x62,0xdf,0xb9,0xed,
                 0xfa,0x04,0xc8,0x87,0x15,0x3c,0xc5,0xde,0x67,
                 0x7a,0xab,0x8c,0x78,0x39,0xd5,0x17,0x03,0x58,
                 0x79,0x67,0x9c,0x29,0x72,0x7e,0x96,0xc5,0x42,
                 0x63,0x24,0xa2,0x57,0x5f,0xbe,0x67,0x8d,0x6c,
                 0xc7,0xfe,0xf5,0xeb,0x6c,0xeb,0xd5,0x95,0xcf,
                 0xdd,0xef},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x6b,0x14,0x2d,0x4d,0xfe,0x21,0x7f,0x18,0x81,
                0xaa,0x0e,0x64,0x83,0xb2,0x71,0xdd,0x5d,0x43,
                0xf7,0x0b,0x85,0x60,0x59,0x53,0xa0,0xfe,0xf2,
                0x72,0xdd,0xde,0x46,0xca},
        .chunks = {64, -1, 64},
        .num_chunks = 3,
    }, {	// 6 (45)
        .key = {0xf9,0x87,0xeb,0x83,0xa3,0xfd,0x6d,0x94,0xeb,
                0xf3,0x62,0x6b,0x7d,0x34,0xfe,0xc2,0x3e,0xe0,
                0x6c,0x63,0xdf,0xb4,0x07,0x8c,0xb3,0x8b,0xcc,
                0x97,0xbd,0x25,0x0f,0xda,0x0e,0x28,0x6e,0xcd,
                0x4e,0x64,0x04,0x6a,0x98,0x5b,0xdf,0xda,0x8b},
        .key_len = 45,
        .data = {0x39,0x0a,0x9d,0xc2,0xea,0x20,0x22,0x1c,0x59,
                 0x93,0xc5,0x81,0x89,0x2e,0xb4,0xb0,0x43,0x64,
                 0x29,0x4f,0xad,0x91,0x9c,0x45,0x1e,0x83,0x37,
                 0x65,0x31,0x39,0x8a,0x4c,0x18,0xea,0x80,0x8c,
                 0x33,0x4a,0x91,0x0a,0xe1,0x08,0x3a,0xa4,0x97,
                 0x9b,0xaa,0x17,0x2f,0x3e,0xbf,0x20,0x82,0x39,
                 0x30,0xe2,0x38,0x63,0x0c,0x88,0xdf,0xe5,0x63,
                 0x2b,0x3b,0x40,0x42,0xf6,0xdd,0x92,0xe5,0x88,
                 0xf7,0x15,0x29,0x99,0x6f,0xe8,0x40,0xe1,0x32,
                 0x12,0xa8,0x35,0xcb,0xc4,0x5e,0xf4,0x34,0xde,
                 0x4f,0xa1,0xec,0xb5,0x0f,0xd1,0x49,0x13,0xcd,
                 0x48,0x10,0x80,0x87,0x5f,0x43,0xc0,0x7a,0xa9,
                 0x3a,0x9d,0xdd,0xd5,0xf5,0xe7,0xce,0xd6,0xb1,
                 0xb8,0x8d,0x42,0xb9,0xfc,0xe8,0xf8,0x7f,0x31,
                 0xf6,0x06},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x0b,0x3b,0x22,0x0e,0xe7,0xa4,0xfd,0xcb,0x0d,
                0x17,0xa5,0xc8,0xb5,0x95,0xb9,0x81},
        .chunks = {54, 0, 54, -1, 20},
        .num_chunks = 5,
    }, {	// 7 (46)
        .key = {0xef,0x25,0x71,0x32,0xb7,0xbe,0x12,0x4e,0xa0,
                0x88,0x6d,0x58,0x77,0x65,0xe8,0xe7,0x03,0x57,
                0x95,0x9c,0xf3,0x9e,0xbf,0x62,0x14,0x20,0xc3,
                0xf3,0xc7,0x0e,0x21,0x9f,0xb3,0xc5,0xd3,0x49,
                0xb7,0xf2,0xde,0xb2,0x22,0xfa,0x26,0xfa,0x27},
        .key_len = 45,
        .data = {0xf9,0x07,0x68,0x95,0x4c,0xdc,0xbd,0x57,0x05,
                 0xf9,0xd3,0x18,0xfc,0xa6,0x59,0x17,0x87,0xaf,
                 0x84,0x0a,0x92,0x1f,0xbd,0x06,0xf2,0x4b,0x97,
                 0x9e,0xf6,0x12,0x03,0x4f,0x3f,0x64,0xc7,0x1c,
                 0xd2,0x01,0x2c,0x75,0x6c,0x83,0xf7,0x5d,0x16,
                 0x9f,0x9b,0xcc,0xf8,0xa8,0xad,0x52,0x72,0x54,
                 0x98,0xfe,0x69,0xc3,0x92,0x7e,0xdf,0xbd,0xcf,
                 0x87,0xc7,0x3c,0xf4,0x78,0x17,0x2a,0xce,0x3a,
                 0x1e,0x6b,0x44,0x6a,0x18,0x1e,0x8a,0xba,0x00,
                 0x20,0x98,0x94,0xa5,0xd2,0xdb,0x01,0x00,0x1d,
                 0x2a,0xca,0xc5,0xb3,0xfb,0xdd,0x38,0x97,0xd7,
                 0xf1,0x42,0xdf,0x0b,0x6d,0xc4,0xb9,0xa1,0x86,
                 0x2b,0xac,0x8e,0xa8,0x45,0x20,0x2d,0x18,0x53,
                 0x21,0xec,0xd7,0x5f,0x60,0x46,0xc9,0xcf,0x7a,
                 0xf1,0x16},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xa1,0x7d,0x0e,0x0f,0x02,0x11,0x84,0xa3,0x93,
                0x72,0x22,0xde,0x81,0xbe,0x62,0x7c},
        .chunks = {0, 128},
        .num_chunks = 2,
    }, {	// 8 (59)
        .key = {0x91,0x17,0xcf,0x3c,0xe9,0xf5,0xc6,0xe1,0x97,
                0x52,0xbf,0x0b,0x1c,0xf8,0x6a,0x78,0xce,0x3a,
                0xdb,0xba,0x87,0xda,0xe1,0x39,0x9a,0x2a,0x93,
                0x7b,0x0b,0x72,0x2b,0xa3,0xff,0x92,0x18,0x38,
                0x71,0xe8,0x4e,0x28,0x27,0x74,0xe1,0x0d,0xe4},
        .key_len = 45,
        .data = {0x93,0x5a,0x3c,0x27,0x24,0x9d,0xcf,0x92,0xae,
                 0xda,0xc8,0xdc,0x76,0xd2,0x2f,0xf7,0x74,0x2e,
                 0x5c,0xee,0x57,0x71,0x17,0x78,0xc9,0x2a,0xfd,
                 0xcd,0xf3,0x6e,0x26,0xb8,0x44,0x85,0x04,0xee,
                 0x6e,0xe4,0x8e,0x9e,0xb2,0x5b,0x9e,0x49,0x5e,
                 0x90,0x98,0xd4,0x94,0xac,0x4d,0xdc,0x4c,0x54,
                 0x1f,0x49,0x9c,0xdb,0x65,0x26,0x38,0xb6,0x11,
                 0xb0,0x35,0x30,0x90,0xac,0x12,0x5f,0xf1,0xfe,
                 0xf8,0x56,0x4a,0x78,0x41,0x9c,0x57,0xf0,0x38,
                 0xdd,0x65,0x95,0x1f,0xe0,0x6e,0x83,0x77,0xb9,
                 0x86,0x94,0x7b,0x40,0x75,0x79,0xee,0xc1,0xa6,
                 0x0a,0x16,0xf5,0x40,0xdb,0x09,0x31,0x92,0x10,
                 0x27,0xde,0xb4,0x72,0xe8,0x29,0x6b,0xc2,0xd8,
                 0xfb,0x4e,0x4d,0xdf,0x2c,0x27,0xc0,0xc6,0xf4,
                 0x9c,0x3e},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x4c,0xd0,0x95,0xce,0x64,0x1f,0x21,0x7f,0x8b,
                0x5f,0x35,0x51,0x52,0xee,0xd0,0x0b,0x1d,0x9f,
                0xd7,0x21,0xa0,0x8d,0xc5,0xa0},
    }, {	// 9 (60)
        .key = {0x36,0x3b,0x32,0xac,0xcf,0xa5,0x93,0xe4,0x54,
                0xcc,0x3e,0xc8,0x3b,0x9d,0x77,0x5a,0x0d,0xd0,
                0x27,0xb0,0x17,0xca,0x2f,0xf8,0x63,0xc1,0xfc,
                0xb9,0xe6,0x21,0x5b,0x5c,0xfb,0x2e,0x8f,0xea,
                0x10,0xeb,0xa2,0x17,0x9f,0x3b,0xf8,0x80,0x61},
        .key_len = 45,
        .data = {0x54,0x85,0x64,0xe5,0xb7,0x37,0x04,0x26,0xd5,
                 0x75,0xbb,0xe8,0x17,0x5b,0x48,0xc2,0x44,0xde,
                 0xdc,0xef,0x3d,0xaf,0x72,0x52,0xec,0x62,0x5f,
                 0xb7,0x77,0xd0,0x2a,0x5c,0xb9,0xba,0x9d,0xb0,
                 0xf2,0xaf,0x1c,0x5a,0xbd,0x2f,0x36,0x7d,0x43,
                 0x10,0x7a,0x3a,0xaf,0x21,0x8c,0x77,0xe2,0x0e,
                 0x78,0xdf,0x67,0x83,0x45,0x2a,0xa9,0x94,0xce,
                 0x9f,0x63,0x5d,0xcd,0xd7,0x59,0xe5,0x39,0xc3,
                 0x46,0x49,0xd2,0xf1,0x15,0x16,0xfa,0x0a,0x53,
                 0xf6,0xc6,0xa0,0xe5,0x8f,0x55,0x26,0xf6,0xa8,
                 0x60,0x40,0x34,0x8d,0x13,0x3e,0x3c,0xb5,0x1b,
                 0xe2,0x52,0xa3,0x01,0x6a,0x56,0x0a,0xb6,0xca,
                 0xf3,0x34,0x6f,0x3a,0x1a,0xa4,0xb2,0xf0,0xaf,
                 0xfb,0xb1,0x2f,0x82,0x18,0xd8,0x80,0x80,0x83,
                 0xa2,0x40},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x64,0x6a,0xbb,0xd4,0x26,0x25,0x5d,0x2e,0x36,
                0x9b,0x7a,0xc9,0xeb,0x3c,0x3a,0xf1,0x9c,0x71,
                0x85,0xec,0xd2,0x8b,0xd8,0x2c},
    }, { 	// 10 (75)
        .key = {0xb7,0x63,0x26,0x3d,0xc4,0xfc,0x62,0xb2,0x27,
                0xcd,0x3f,0x6b,0x4e,0x9e,0x35,0x8c,0x21,0xca,
                0x03,0x6c,0xe3,0x96,0xab,0x92,0x59,0xc1,0xbe,
                0xdd,0x2f,0x5c,0xd9,0x02,0x97,0xdc,0x70,0x3c,
                0x33,0x6e,0xca,0x3e,0x35,0x8a,0x4d,0x6d,0xc5},
        .key_len = 45,
        .data = {0x53,0xcb,0x09,0xd0,0xa7,0x88,0xe4,0x46,0x6d,
                 0x01,0x58,0x8d,0xf6,0x94,0x5d,0x87,0x28,0xd9,
                 0x36,0x3f,0x76,0xcd,0x01,0x2a,0x10,0x30,0x8d,
                 0xad,0x56,0x2b,0x6b,0xe0,0x93,0x36,0x48,0x92,
                 0xe8,0x39,0x7a,0x8d,0x86,0xf1,0xd8,0x1a,0x20,
                 0x96,0xcf,0xc8,0xa1,0xbb,0xb2,0x6a,0x1a,0x75,
                 0x52,0x5f,0xfe,0xbf,0xcf,0x16,0x91,0x1d,0xad,
                 0xd0,0x9e,0x80,0x2a,0xa8,0x68,0x6a,0xcf,0xd1,
                 0xe4,0x52,0x46,0x20,0x25,0x4a,0x6b,0xca,0x18,
                 0xdf,0xa5,0x6e,0x71,0x41,0x77,0x56,0xe5,0xa4,
                 0x52,0xfa,0x9a,0xe5,0xae,0xc5,0xdc,0x71,0x59,
                 0x1c,0x11,0x63,0x0e,0x9d,0xef,0xec,0x49,0xa4,
                 0xec,0xf8,0x5a,0x14,0xf6,0x0e,0xb8,0x54,0x65,
                 0x78,0x99,0x97,0x2e,0xa5,0xbf,0x61,0x59,0xcb,
                 0x95,0x47},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x73,0x73,0x01,0xde,0xa9,0x3d,0xb6,0xbc,0xba,
                0xdd,0x7b,0xf7,0x96,0x69,0x39,0x61,0x31,0x7c,
                0xa6,0x80,0xb3,0x80,0x41,0x6f,0x12,0xf4,0x66,
                0xf0,0x65,0x26,0xb3,0x6b},
    }, {	// 11 (76)
        .key = {0x9f,0xe4,0x2d,0xfa,0xc9,0x2a,0x4a,0x13,0x6f,
                0xa7,0xc9,0xf6,0xe3,0x31,0xb5,0xd3,0xa6,0x1a,
                0xa7,0x30,0x35,0xb5,0x3a,0x8d,0x25,0x17,0xbe,
                0x43,0x72,0x1b,0x31,0xb2,0x15,0xa9,0x6b,0x9b,
                0xd4,0x37,0x98,0xcb,0x5e,0x8f,0xeb,0xfa,0x97},
        .key_len = 45,
        .data = {0xf9,0x66,0x0f,0xb7,0x84,0xc1,0x4b,0x5f,0xbe,
                 0xc2,0x80,0x52,0x6a,0x69,0xc2,0x29,0x4f,0xba,
                 0x12,0xae,0xa1,0x63,0x78,0x9b,0xbe,0x9f,0x52,
                 0xa5,0x1b,0x5a,0xeb,0xb9,0x7d,0x96,0x4f,0x86,
                 0x6c,0x0d,0x5e,0x3b,0xe4,0x18,0x20,0x92,0x4f,
                 0xcf,0x58,0x0d,0xb0,0x72,0x5c,0x7f,0x21,0x08,
                 0x23,0xcf,0x7f,0x45,0xa0,0xf9,0x64,0xb1,0x4e,
                 0x55,0x55,0x07,0x0d,0x1c,0x3d,0xdb,0x2c,0x28,
                 0x1a,0x80,0xc7,0xfb,0xf7,0x29,0x53,0x03,0x1a,
                 0x4e,0x77,0x1d,0x7e,0x52,0x1d,0x57,0x84,0x62,
                 0xca,0xfa,0xe5,0xa0,0x2a,0xc8,0xeb,0x81,0xf0,
                 0x82,0xe1,0x73,0xdd,0xad,0xc8,0xc4,0x1d,0x96,
                 0x4b,0xbf,0xda,0x94,0xf5,0x18,0x0c,0x8d,0xa2,
                 0x8a,0x8e,0xbb,0x33,0xbe,0x77,0xb0,0x86,0x6f,
                 0xa7,0x98},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x77,0x86,0xc1,0x55,0xd1,0x0c,0x74,0x1b,0x63,
                0xec,0x65,0x0b,0x7b,0x1a,0xa3,0xbf,0xd7,0x1a,
                0xc7,0x18,0x81,0xad,0x06,0xae,0x98,0xfb,0x08,
                0x2f,0x17,0xe0,0xca,0xa0},
    }, {	// 12 (90)
        .key = {0x79,0xf8,0x77,0x34,0xc4,0x6c,0x5a,0x11,0xd8,
                0x6a,0xed,0xea,0xd2,0x2e,0xd3,0xea,0x01,0x57,
                0x7a,0xd4,0xec,0xdf,0x42,0x96,0x96,0x50,0xe1,
                0x20,0x00,0x35,0x06,0x76,0xf0,0xcf,0x3c,0x04,
                0xf1,0x0a,0x11,0x33,0x9b,0xaf,0x78,0x39,0x14,
                0xdb,0x6d,0x35,0xd7,0xb0,0xd7,0x7b,0xb4,0x4a,
                0xb2,0x2c,0x18,0xf5,0x6d,0x0b,0x8f,0x9d,0x91,
                0x8b},
        .key_len = 64,
        .data = {0x50,0x9a,0x0a,0x45,0xa1,0x51,0x2b,0x50,0x72,
                 0x47,0x4b,0x29,0x7f,0x9c,0x1a,0x8c,0x24,0x89,
                 0x00,0x16,0x14,0x44,0x68,0x50,0x4e,0x24,0x5f,
                 0xe9,0x4d,0x06,0x5d,0x43,0x7f,0xef,0x62,0x32,
                 0xf9,0xf3,0x45,0x00,0x69,0x55,0x49,0xb4,0x4c,
                 0xef,0xf2,0x93,0x61,0xd4,0x17,0xe8,0x5d,0x35,
                 0x37,0x01,0xe0,0x81,0x11,0x7a,0xa8,0xd0,0x6e,
                 0xbe,0x05,0x82,0x42,0xca,0x8c,0x23,0xf3,0x34,
                 0x10,0x92,0xf9,0x6c,0xce,0x63,0xa7,0x43,0xe8,
                 0x81,0x48,0xa9,0x15,0x18,0x6e,0xbb,0x96,0xb2,
                 0x87,0xfd,0x6c,0xa0,0xb1,0xe3,0xc8,0x9b,0xd0,
                 0x97,0xc3,0xab,0xdd,0xf6,0x4f,0x48,0x81,0xdb,
                 0x6d,0xbf,0xe2,0xa1,0xa1,0xd8,0xbd,0xe3,0xa3,
                 0xb6,0xb5,0x86,0x58,0xfe,0xea,0xfa,0x00,0x3c,
                 0xce,0xbc},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xb5,0xb0,0xc4,0x30,0x28,0xe8,0x16,0x28,0xdc,
                0xe8,0x25,0x17,0xfa,0x36,0xaa,0x29},
    }, {	// 13 (91)
        .key = {0xea,0xe2,0x55,0xd9,0xe0,0x83,0x26,0x8f,0x89,
                0x64,0x29,0xce,0x36,0x64,0x55,0x02,0xaf,0xf9,
                0xdb,0xea,0xca,0x71,0x59,0xf9,0x3c,0x7d,0x51,
                0xfd,0xae,0xef,0xdb,0xfe,0x14,0xc3,0x96,0x69,
                0x3a,0x5c,0xe4,0x6e,0x9f,0x11,0x57,0xa6,0x87,
                0xe8,0x66,0xf9,0x4c,0xa1,0x65,0xbf,0xf5,0xf7,
                0xb4,0x25,0x09,0x22,0x36,0xd2,0xa6,0xa0,0x04,
                0xcb},
        .key_len = 64,
        .data = {0xc2,0x8f,0x6a,0x09,0xce,0x07,0x6e,0xf2,0x70,
                 0x45,0x89,0x67,0xfe,0x19,0xd4,0x6e,0x6f,0x6b,
                 0x2c,0xbe,0xb6,0x36,0x2b,0xdc,0x4f,0xd5,0x56,
                 0x84,0x17,0x7e,0x98,0x4a,0x60,0x0c,0xf0,0x81,
                 0x45,0x01,0x66,0x5c,0x3b,0xcb,0x43,0x53,0xe9,
                 0x46,0x81,0xc8,0x3a,0x83,0x81,0xeb,0xb0,0xc8,
                 0xfc,0xdb,0xfb,0xd7,0x3c,0x0e,0xca,0x73,0x8c,
                 0xf2,0xe1,0x21,0xed,0xd4,0x6b,0x2c,0x0a,0x02,
                 0x92,0xeb,0x6e,0x2c,0x4e,0x46,0xf5,0x10,0x7a,
                 0x77,0x80,0x57,0x2d,0x0e,0xed,0xb9,0x47,0x38,
                 0x47,0x68,0x4a,0x40,0x39,0xac,0x6c,0x56,0xc9,
                 0xca,0xea,0x90,0x43,0x2b,0x9e,0x2e,0x72,0xba,
                 0xd4,0x22,0x16,0x8e,0x5a,0xd0,0x93,0xc9,0xd6,
                 0x12,0xe7,0xc0,0x5c,0x7f,0xde,0x5c,0x40,0xed,
                 0x89,0xc0},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xb8,0x40,0x03,0xc4,0x17,0xa4,0x72,0xfd,0x29,
                0x35,0x34,0x19,0x62,0x74,0x43,0x30},
    }, {	// 14 (105)
        .key = {0xa5,0xfd,0x99,0xca,0x57,0xc1,0xfe,0xc8,0x15,
                0x9a,0x79,0x87,0x92,0x42,0x6d,0x29,0x6f,0xa1,
                0xb1,0x7d,0x53,0x92,0x41,0xde,0x3d,0xea,0x33,
                0x58,0x19,0xb7,0xed,0x0d,0x92,0xc5,0x96,0xd7,
                0x28,0x67,0xca,0x2f,0x82,0x73,0x92,0x4e,0x05,
                0x8f,0x93,0x91,0xa5,0xab,0x85,0x22,0xfb,0xcf,
                0xe7,0xd5,0x98,0x17,0xf1,0x50,0x9a,0xfc,0xcb,
                0x6f},
        .key_len = 64,
        .data = {0x5c,0xf3,0xa5,0x20,0x2d,0xf8,0x70,0x6f,0x6b,
                 0xff,0x5b,0xf2,0x59,0x0d,0xe3,0x7c,0x90,0x2c,
                 0x7f,0xfd,0x4e,0x6c,0x8e,0xa6,0x11,0x28,0x8e,
                 0x4e,0x65,0x8a,0x8e,0x15,0xfa,0x51,0xe6,0x47,
                 0xf9,0xd2,0x25,0x83,0x98,0x3d,0x4b,0x1c,0xed,
                 0x22,0x39,0xbf,0xff,0x34,0x65,0x56,0x23,0x4c,
                 0xd2,0x2d,0x86,0xb1,0x40,0x53,0x06,0x96,0xa0,
                 0x44,0x46,0xe4,0xca,0xc4,0x01,0x3a,0x72,0x0e,
                 0x9e,0x32,0x58,0x2e,0x05,0xe7,0xc0,0xac,0xb2,
                 0xb4,0x22,0x6a,0x07,0x3e,0x22,0xcf,0xe7,0xb4,
                 0xc2,0x25,0x80,0x55,0xd7,0x40,0x68,0x33,0xba,
                 0x61,0xec,0x37,0x3f,0x5a,0xa5,0x66,0xeb,0xf2,
                 0x4c,0x62,0x61,0x8a,0xce,0x34,0x1e,0x01,0xa3,
                 0x48,0x66,0xd6,0x5c,0xb9,0x7e,0x8c,0x7c,0xd0,
                 0x1c,0x53},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x2c,0x2b,0xc8,0xc8,0x70,0x17,0xf2,0x04,0xc9,
                0x58,0xab,0xd9,0xaa,0xb2,0xbe,0xb6,0xac,0x67,
                0x78,0x1d,0x8d,0x9d,0x80,0x4c},
    }, {	// 15 (106)
        .key = {0x30,0xbc,0x3e,0x32,0x1a,0x89,0x78,0xe2,0x35,
                0xfa,0x1b,0x55,0x00,0x64,0xb8,0x2e,0xaa,0x0c,
                0x10,0x75,0x25,0xea,0xcc,0x82,0x7c,0xad,0x6f,
                0x1d,0x66,0xff,0x88,0xe3,0x1b,0x09,0x2c,0xec,
                0x66,0x3a,0xa3,0xaa,0xfc,0x44,0x62,0x14,0x0c,
                0x68,0x39,0x04,0x17,0xf4,0xce,0xde,0x02,0x0a,
                0x4a,0x73,0x6a,0xa2,0x52,0x25,0x37,0xd2,0x39,
                0x4b},
        .key_len = 64,
        .data = {0xc1,0x26,0x3b,0xe4,0x23,0xe7,0x88,0x8e,0xac,
                 0xec,0xcf,0xef,0x26,0xf0,0xb5,0xaa,0xef,0xe0,
                 0x3f,0x3c,0xe7,0x32,0xdd,0xe9,0x8c,0x78,0xa7,
                 0xf6,0x64,0x35,0xe6,0x19,0x9c,0xef,0xd6,0x2e,
                 0xee,0x85,0xaa,0x2b,0xc8,0xc3,0xd1,0x56,0xaa,
                 0x34,0x78,0xb6,0xcf,0x37,0x50,0xc7,0x11,0x55,
                 0x91,0x72,0x07,0xd2,0x3f,0x3b,0x70,0x82,0xac,
                 0xbd,0xd4,0xde,0x3e,0x53,0x68,0x57,0x72,0x19,
                 0x33,0xeb,0x21,0x13,0x6f,0xf5,0x02,0xab,0x32,
                 0x49,0x71,0x61,0x4d,0x80,0x6e,0xbe,0x74,0x91,
                 0xe9,0x89,0xa0,0xa2,0x3d,0x3e,0xb2,0x1d,0xfa,
                 0xbc,0x59,0x05,0xe7,0x3e,0x35,0x8b,0x47,0x8c,
                 0x3d,0xdc,0x5c,0x73,0x5e,0x3e,0x2a,0x72,0x64,
                 0x5b,0x7d,0xb6,0x1e,0xdc,0x2d,0x49,0xbd,0x3a,
                 0xa1,0x86},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0xd7,0x22,0xb5,0x7c,0x48,0x12,0x8b,0x37,0xba,
                0x38,0x77,0x0c,0xbf,0x46,0x60,0x69,0x77,0x57,
                0xba,0xb9,0x5c,0x00,0xc4,0x84},
    }, {	// 16 (120)
        .key = {0x99,0x28,0x68,0x50,0x4d,0x25,0x64,0xc4,0xfb,
                0x47,0xbc,0xbd,0x4a,0xe4,0x82,0xd8,0xfb,0x0e,
                0x8e,0x56,0xd7,0xb8,0x18,0x64,0xe6,0x19,0x86,
                0xa0,0xe2,0x56,0x82,0xda,0xeb,0x5b,0x50,0x17,
                0x7c,0x09,0x5e,0xdc,0x9e,0x97,0x1d,0xa9,0x5c,
                0x32,0x10,0xc3,0x76,0xe7,0x23,0x36,0x5a,0xc3,
                0x3d,0x1b,0x4f,0x39,0x18,0x17,0xf4,0xc3,0x51,
                0x24},
        .key_len = 64,
        .data = {0xed,0x4f,0x26,0x9a,0x88,0x51,0xeb,0x31,0x54,
                 0x77,0x15,0x16,0xb2,0x72,0x28,0x15,0x52,0x00,
                 0x77,0x80,0x49,0xb2,0xdc,0x19,0x63,0xf3,0xac,
                 0x32,0xba,0x46,0xea,0x13,0x87,0xcf,0xbb,0x9c,
                 0x39,0x15,0x1a,0x2c,0xc4,0x06,0xcd,0xc1,0x3c,
                 0x3c,0x98,0x60,0xa2,0x7e,0xb0,0xb7,0xfe,0x8a,
                 0x72,0x01,0xad,0x11,0x55,0x2a,0xfd,0x04,0x1e,
                 0x33,0xf7,0x0e,0x53,0xd9,0x7c,0x62,0xf1,0x71,
                 0x94,0xb6,0x61,0x17,0x02,0x8f,0xa9,0x07,0x1c,
                 0xc0,0xe0,0x4b,0xd9,0x2d,0xe4,0x97,0x2c,0xd5,
                 0x4f,0x71,0x90,0x10,0xa6,0x94,0xe4,0x14,0xd4,
                 0x97,0x7a,0xbe,0xd7,0xca,0x6b,0x90,0xba,0x61,
                 0x2d,0xf6,0xc3,0xd4,0x67,0xcd,0xed,0x85,0x03,
                 0x25,0x98,0xa4,0x85,0x46,0x80,0x4f,0x9c,0xf2,
                 0xec,0xfe},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x2f,0x83,0x21,0xf4,0x16,0xb9,0xbb,0x24,0x9f,
                0x11,0x3b,0x13,0xfc,0x12,0xd7,0x0e,0x16,0x68,
                0xdc,0x33,0x28,0x39,0xc1,0x0d,0xaa,0x57,0x17,
                0x89,0x6c,0xb7,0x0d,0xdf},
    }, {	// 17 (121)
        .key = {0xce,0xab,0x39,0x8e,0x41,0x07,0x48,0x3e,0xde,
                0x64,0xce,0x10,0x7c,0x92,0x70,0xe6,0x02,0x27,
                0x78,0xb6,0x1f,0x6a,0x25,0x8d,0x3b,0x70,0x45,
                0xd4,0xad,0x85,0x06,0xd3,0x2e,0xce,0x0a,0x73,
                0x8d,0x2c,0xb9,0x48,0xa5,0x62,0xdb,0xce,0x8d,
                0x7b,0x66,0xf3,0x0e,0x66,0x94,0xd6,0x5a,0xe4,
                0x39,0xcf,0xfa,0xa4,0x54,0xaf,0x09,0xab,0xe4,
                0x49},
        .key_len = 64,
        .data = {0x6d,0xde,0x9a,0xe8,0x67,0xe2,0xfe,0xb3,0x67,
                 0x00,0x8a,0x97,0x5d,0x78,0x53,0xed,0x8f,0x89,
                 0x69,0x0f,0x3c,0x87,0xa1,0x10,0x7f,0x2e,0x98,
                 0xaa,0x77,0x36,0xf4,0x77,0xa5,0x27,0xed,0x64,
                 0x95,0x6f,0x0d,0x64,0xc1,0xb2,0x33,0x61,0xb2,
                 0x61,0xde,0x78,0x68,0x8e,0xa8,0x65,0xfc,0xff,
                 0x11,0x3c,0x84,0x81,0x7e,0x5b,0x37,0x7e,0x82,
                 0x9c,0xd2,0xd2,0x5b,0xcf,0x3a,0xdb,0xc0,0x67,
                 0x62,0xcf,0xda,0x73,0x6f,0x53,0x90,0xd0,0x1a,
                 0x49,0x07,0x9d,0x56,0xe9,0x69,0xf0,0x33,0x13,
                 0xe6,0xc7,0x03,0xe3,0xf9,0x42,0xbb,0x87,0xed,
                 0x0f,0x9c,0x4d,0x9f,0x25,0x12,0x00,0x85,0xb5,
                 0xdc,0x75,0xef,0x5d,0x6d,0x61,0x8d,0xa0,0x92,
                 0x6d,0x32,0x93,0x56,0x8d,0xd7,0xd8,0x23,0x8d,
                 0xe3,0xd0},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x2d,0x3a,0x76,0x05,0x95,0xf3,0xfb,0x19,0x29,
                0x3c,0xc6,0xd2,0x36,0x51,0x22,0x2a,0x9f,0x5a,
                0x4f,0x02,0x28,0x44,0x57,0xa9,0xc1,0xed,0x4c,
                0x43,0xac,0x99,0x3c,0xa5},
    }, {	// 18 (135)
        .key = {0x07,0xc3,0x58,0xed,0x1d,0xf3,0xb0,0x6d,0x47,
                0xb5,0xec,0x76,0x3a,0xfa,0x07,0xa6,0x67,0x7c,
                0xa3,0xa7,0x22,0x52,0x4e,0x61,0x03,0xc1,0x05,
                0x6d,0x8c,0x56,0xf6,0xcd,0x0d,0x31,0x8a,0xdb,
                0xc5,0xa4,0xa3,0x80,0x4a,0xfd,0x23,0xa6,0x2b,
                0x9f,0xad,0xf0,0xd3,0x58,0xaf,0xa8,0xb0,0xee,
                0xa0,0xf9,0x95,0xfb,0x86,0x5e,0x5d,0xfb,0xbc,
                0x5a,0xd2,0xa4,0xf2,0x6a,0xcd,0x76},
        .key_len = 70,
        .data = {0x2a,0xa1,0xd9,0x4e,0xc8,0x3c,0xe7,0xc3,0xc7,
                 0x5c,0x6b,0xc8,0x47,0x75,0x9b,0x08,0x52,0x34,
                 0xfd,0x44,0xb4,0x07,0xd8,0xf8,0x0d,0xdf,0xe9,
                 0x3c,0x24,0x35,0x56,0xe8,0x7e,0x4b,0xe8,0xfb,
                 0x30,0xb4,0x74,0x3e,0xf1,0x16,0x9a,0x24,0x73,
                 0x2f,0xb2,0xf5,0xf4,0x16,0x04,0x2b,0x10,0xc3,
                 0x37,0x1d,0xd9,0xd2,0x0d,0xda,0x29,0x84,0x4d,
                 0x58,0x37,0x07,0x00,0xce,0x69,0xf7,0xdf,0x5e,
                 0x69,0x24,0x0d,0xf7,0x7b,0x96,0x02,0x7a,0x0e,
                 0xce,0xc7,0x1b,0x90,0x4f,0x69,0x0b,0x87,0x5d,
                 0xa8,0x54,0xde,0x05,0xef,0x04,0x7c,0x5d,0x89,
                 0x8d,0x1c,0x0d,0x11,0x6c,0x58,0x0e,0x2a,0x09,
                 0x06,0xb2,0x71,0xde,0xc8,0xe5,0xb0,0xdc,0xdf,
                 0xb2,0x55,0x0a,0x40,0x09,0x22,0x70,0xea,0xbf,
                 0x25,0x33},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xb3,0xa1,0x89,0xa1,0x7e,0x8d,0x9e,0x98,0x6c,
                0xd3,0x1b,0xbe,0x01,0xb4,0x9f,0xb3},
    }, {	// 19 (136)
        .key = {0xab,0x8d,0xfb,0xa4,0x41,0x4e,0x69,0x86,0x51,
                0x3a,0x97,0x67,0xaf,0x5e,0xae,0xd9,0x72,0x08,
                0x11,0xc4,0xb3,0x80,0x40,0xb9,0x91,0xf3,0xfd,
                0x82,0x78,0xb0,0xad,0xfe,0xa4,0x97,0x00,0x2c,
                0xe0,0xcd,0xd4,0x85,0x94,0xb5,0x57,0x8f,0xfe,
                0x1c,0x6c,0xaf,0xc0,0xb4,0x51,0x3e,0x9b,0xc4,
                0x7e,0xe0,0x7a,0x1d,0xd0,0x11,0xb2,0x50,0xe6,
                0x01,0x88,0x1e,0xcc,0xa2,0xf4,0x30},
        .key_len = 70,
        .data = {0xd1,0xa7,0x08,0x6d,0x13,0x4c,0x11,0xa8,0xa3,
                 0x20,0x4e,0x01,0x9f,0x52,0x84,0x3e,0x89,0xf2,
                 0xd0,0x1a,0x02,0xa8,0x8a,0x94,0xd4,0xa6,0x6e,
                 0x8d,0x36,0xdb,0xfe,0x92,0x4c,0x69,0x22,0xf7,
                 0xee,0x5a,0x12,0x25,0xaa,0x8e,0x75,0x34,0x0c,
                 0xf8,0xcb,0xbd,0x1c,0x0b,0x08,0xe9,0x29,0x6e,
                 0x81,0xce,0xc5,0xf7,0x0c,0xfc,0x11,0xd7,0x63,
                 0x52,0x3b,0x12,0xca,0x17,0x44,0x33,0xf2,0x46,
                 0x07,0x3d,0x1c,0x28,0x77,0xe4,0x81,0x28,0x28,
                 0xfd,0xf2,0xe4,0x11,0x34,0xbc,0x80,0x90,0xfd,
                 0xce,0x3f,0xae,0xcd,0x1e,0x54,0xa5,0x89,0x48,
                 0xf5,0x9f,0x3f,0x78,0xb2,0xc1,0x14,0x8b,0x05,
                 0x68,0x7d,0x71,0x2a,0xb2,0xb2,0xd6,0x30,0x41,
                 0x60,0x01,0x51,0x3b,0x9e,0xfc,0x7f,0x95,0x23,
                 0xf5,0x3f},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x7a,0xea,0x0e,0x2d,0x93,0xe9,0xa6,0xa3,0x00,
                0x41,0x17,0xad,0x4a,0x4a,0x72,0xa3},
    }, {	// 20 (150)
        .key = {0x05,0x8f,0x60,0x4e,0x53,0x05,0x1a,0x0f,0x85,
                0x50,0xde,0x16,0xb7,0x24,0x5f,0xda,0xd3,0xda,
                0x63,0x9a,0x6c,0xc3,0xc8,0x4e,0xea,0xbc,0xc5,
                0xdd,0xe8,0x02,0x73,0x90,0xda,0x48,0x8c,0xc7,
                0xf3,0x07,0x72,0xeb,0x46,0x16,0x73,0xa3,0x2b,
                0x7a,0x4b,0x4b,0xe4,0x7f,0xea,0xa2,0x80,0x08,
                0x78,0xc2,0x00,0x23,0x97,0x56,0xb9,0xe0,0xe8,
                0x07,0xf9,0x64,0xd0,0x37,0xed,0x39},
        .key_len = 70,
        .data = {0xa7,0x41,0x00,0xcf,0x30,0xcd,0x26,0x41,0x6e,
                 0x98,0x78,0x73,0x9d,0xfd,0xb3,0xc1,0xfa,0x56,
                 0x9d,0x64,0x27,0xca,0x8e,0xe9,0xd0,0x66,0x30,
                 0xe1,0x8f,0x6f,0x83,0xdb,0x0d,0xf7,0x24,0x8f,
                 0x6b,0xaf,0xce,0x5c,0xe0,0xfc,0x21,0xf5,0xa3,
                 0x4d,0xa2,0x57,0x0b,0xab,0x04,0xfe,0xf4,0x92,
                 0xa6,0x58,0x66,0xff,0x5c,0x7a,0x71,0xca,0x72,
                 0x12,0x5b,0x36,0xee,0x9c,0xfe,0xc7,0x16,0xd9,
                 0x6b,0x53,0x32,0x7d,0xd3,0x5c,0x93,0x28,0xa8,
                 0x9d,0xd4,0x98,0xff,0xe3,0x60,0x1d,0x39,0x1e,
                 0x34,0x4d,0xe2,0xb8,0xe7,0xf8,0xd9,0x25,0xe7,
                 0x5f,0xb1,0xbc,0x05,0xa0,0x58,0xc5,0x34,0x75,
                 0xf6,0xd3,0x8d,0x1e,0x18,0x54,0x97,0x9c,0x0e,
                 0x66,0xc6,0x20,0x91,0xec,0x41,0xc3,0xaa,0xe1,
                 0xe8,0x77},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x08,0xe3,0xa1,0x71,0x8c,0x6d,0x1c,0xde,0xf2,
                0xc0,0xc6,0x76,0x60,0xf7,0xc1,0xe8,0xa4,0x59,
                0x63,0xe5,0xff,0xed,0x54,0xa7},
    }, {	// 21 (151)
        .key = {0x98,0x6e,0x0d,0x3c,0x3e,0x76,0x45,0xe4,0x93,
                0xd3,0x59,0x62,0x29,0x1d,0x97,0x9d,0xdf,0x09,
                0xe8,0xa6,0x10,0xd5,0xa7,0x3d,0x0a,0xe7,0xb3,
                0x97,0xc2,0xb1,0xc3,0x5e,0xc6,0xd7,0xfa,0xfa,
                0x72,0x94,0xbc,0x0f,0x67,0x5a,0xbf,0x46,0x39,
                0xb8,0x65,0x51,0x68,0x81,0x49,0x29,0x92,0x2b,
                0x17,0x9a,0xe6,0x75,0xa2,0x02,0xdc,0x4c,0x30,
                0x56,0x23,0xf0,0x18,0x65,0xdb,0x53},
        .key_len = 70,
        .data = {0x72,0xc2,0x1b,0xe6,0xf0,0xc4,0xdf,0x7c,0xc8,
                 0xa5,0x3f,0x92,0x26,0xf3,0x61,0x46,0xf9,0xec,
                 0x5b,0xea,0x9c,0x94,0xf3,0xb7,0xb6,0x04,0xa8,
                 0xbf,0x5f,0x05,0xf7,0x24,0x84,0xdd,0xd7,0x88,
                 0x8c,0x69,0x86,0xc4,0x3b,0x6c,0x87,0xdd,0xd7,
                 0x27,0xec,0x34,0x8a,0x2a,0xd1,0xfc,0x08,0x69,
                 0x29,0xf1,0x71,0x92,0xbd,0x47,0x79,0x9e,0x71,
                 0xe1,0xc6,0xa7,0xc9,0xc4,0x9a,0xf9,0xad,0xcb,
                 0xb1,0x6b,0x69,0x9c,0x6d,0xf0,0xf8,0xda,0x30,
                 0x69,0x82,0x9d,0x09,0xbd,0x23,0x1f,0x94,0x2c,
                 0xee,0xb8,0x1b,0xe0,0x32,0x0c,0x01,0xc5,0xfb,
                 0x83,0x61,0x9b,0xdc,0xf9,0xf2,0x4a,0xec,0xb7,
                 0x2e,0x75,0x0f,0xa2,0xb3,0x51,0x77,0xb3,0xe9,
                 0xb8,0x6a,0xa7,0xe5,0x79,0x45,0xf8,0x8d,0xf3,
                 0xc1,0x0b},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0xb5,0x79,0xea,0xf7,0x70,0x69,0x76,0x15,0x2b,
                0x16,0x22,0xc1,0x7f,0xc4,0x7c,0x5d,0xb3,0x80,
                0x2a,0xa3,0xf4,0x6f,0x6a,0x3e},
    }, {	// 22 (165)
        .key = {0xc0,0x9e,0x29,0x07,0x1c,0x40,0x5d,0x5e,0x82,
                0x0d,0x34,0x5a,0x46,0xdb,0xbf,0x1e,0x0f,0x82,
                0x02,0xe9,0x2d,0xe3,0xed,0x3e,0x2d,0x29,0x8e,
                0x43,0xaa,0x4f,0x84,0x68,0x66,0xe3,0xb7,0x48,
                0x99,0x09,0x46,0xd4,0x88,0xc2,0xc1,0xae,0x5a,
                0x6e,0x99,0xd3,0x27,0x90,0xd4,0x7d,0x53,0xd2,
                0x05,0x48,0x1a,0x49,0x7c,0x93,0x6b,0xf9,0xba,
                0x29,0xfa,0x9c,0x28,0x21,0x91,0x9f},
        .key_len = 70,
        .data = {0xea,0x72,0x40,0x52,0x99,0x80,0x07,0x6d,0x3b,
                 0x02,0x8a,0x08,0x3e,0xbc,0x4e,0x24,0xef,0xda,
                 0xa0,0x6c,0x9c,0x84,0xd7,0x6b,0xf5,0xb2,0xd9,
                 0xfd,0xb8,0x42,0xe1,0x03,0x8e,0x48,0x7f,0x5b,
                 0x30,0xa5,0xe0,0x10,0xcd,0xdb,0x4f,0xcd,0xb0,
                 0x1f,0xfc,0x98,0x1e,0xb0,0xfc,0xbc,0x7d,0x68,
                 0x92,0x07,0xbc,0x90,0xad,0x36,0xee,0xf9,0xb1,
                 0xae,0x38,0x48,0x7a,0x6d,0xee,0x92,0x9f,0x3f,
                 0xf9,0x29,0xf3,0x35,0x7c,0xb5,0x52,0x53,0xb7,
                 0x86,0x9a,0x89,0x2b,0x28,0xf7,0xe5,0xfe,0x38,
                 0x64,0x06,0xa2,0x77,0x6e,0xd4,0xb2,0x1d,0x3b,
                 0x6e,0x1c,0x70,0xcc,0x64,0x85,0x94,0x7f,0x27,
                 0xe9,0xa5,0xd8,0xbd,0x82,0x03,0x80,0xb9,0xec,
                 0xed,0x8e,0x6b,0x86,0x52,0x06,0x54,0x1b,0xe3,
                 0x9f,0xdc},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x49,0xae,0x1c,0x4a,0x7a,0x57,0x0f,0xde,0x47,
                0xf7,0x51,0x7a,0xb1,0x88,0x98,0xb1,0xb9,0x91,
                0xd0,0x3c,0xfc,0xf8,0xc4,0x5b,0xb3,0x61,0x5b,
                0x5f,0x75,0x5d,0xa6,0x82},
    }, {	// 23 (166)
        .key = {0xbc,0xe5,0x0c,0xdf,0xff,0x84,0x38,0x85,0xd4,
                0xf3,0x64,0xd6,0x9f,0x93,0xbf,0x58,0xa2,0x32,
                0x2c,0x70,0x7b,0x82,0xe8,0x78,0xee,0xc9,0x6d,
                0x11,0xe5,0xdb,0x97,0xbb,0xb5,0x46,0x06,0xa3,
                0xa3,0xcc,0xc3,0xbb,0xa7,0x16,0x26,0x10,0x70,
                0xa6,0xf7,0x59,0xa7,0x0e,0xd3,0xcb,0x78,0x5f,
                0xd1,0x35,0x4f,0xe5,0x66,0x48,0xdf,0x11,0x86,
                0x36,0x69,0xb7,0x0c,0x80,0x3b,0x7a},
        .key_len = 70,
        .data = {0x93,0xb7,0xef,0x0e,0x47,0x0d,0xdf,0xac,0x6a,
                 0xef,0x93,0xc0,0xdc,0xd3,0x7b,0x8f,0x1c,0x4b,
                 0xaf,0x5e,0xad,0xd9,0x78,0xe3,0xbf,0x05,0x12,
                 0xfa,0x0b,0xae,0xb0,0x99,0xff,0x9e,0xc1,0x06,
                 0x1b,0x61,0x72,0x47,0x9b,0x56,0x74,0xdb,0x56,
                 0x06,0xff,0xa7,0xe6,0xb5,0x17,0x33,0x09,0x37,
                 0x0e,0x16,0x47,0x05,0x4a,0xaf,0xd5,0x90,0x48,
                 0x16,0xba,0xd5,0xe1,0x52,0x30,0x32,0xcc,0xcd,
                 0x4d,0x78,0x65,0x05,0xe2,0x41,0xac,0x83,0xa4,
                 0x84,0x91,0x11,0x89,0x66,0x6f,0x28,0x75,0x53,
                 0xd6,0xa8,0x16,0x4e,0x8d,0xcb,0x0c,0x85,0xd7,
                 0x5c,0x4e,0x29,0xf6,0x24,0xc9,0x7c,0xee,0xa6,
                 0x4a,0x2c,0x8b,0x0c,0x9d,0xdf,0xa5,0x60,0xf7,
                 0x0f,0xa3,0xff,0x91,0x18,0x3e,0x4b,0x96,0x8f,
                 0x88,0xa1},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x37,0xf9,0xf3,0x29,0x18,0x30,0x82,0x10,0x84,
                0x9d,0xfe,0xbf,0x8d,0xd4,0x56,0x80,0x4b,0xab,
                0xd6,0x84,0x5a,0xf0,0x72,0x18,0xf9,0xd9,0xbe,
                0x9d,0xf9,0x74,0x3d,0x55},
    }, {	// 24 (180)
        .key = {0x81,0x5c,0x2a,0x91,0x1a,0xaf,0x0f,0x84,0x98,
                0x70,0x61,0x10,0xa9,0x5e,0x6f,0x9c,0x26,0xc3,
                0xef,0x52,0xa3,0xb1,0x37,0x81,0x44,0x8c,0xb0,
                0x3f,0xd2,0xc8,0x87,0x52,0x0d,0xf4,0xa5,0x51,
                0x44,0xf8,0xe2,0x06,0x24,0x9b,0x75,0x17,0xce,
                0x48,0xaf,0xe5,0x2c,0x11,0xea,0xb5,0x84,0xf4,
                0xbc,0x0e,0x4d,0x5d,0x70,0x61,0x42,0xed,0xb6,
                0xf0,0xb6,0x7a,0x99,0xe8,0x27,0x57,0xb2,0xd0,
                0x15,0xd5},
        .key_len = 74,
        .data = {0x8b,0x7f,0xdf,0x79,0x2a,0x90,0x21,0x8f,0x91,
                 0x99,0x8b,0x08,0x47,0x56,0xf3,0x2f,0xf8,0x14,
                 0x88,0x46,0x6b,0xcd,0x66,0xce,0xb4,0x95,0x67,
                 0x02,0xab,0x34,0x3c,0xa5,0x9c,0x15,0xbd,0xfd,
                 0x40,0x5f,0x7e,0x20,0xec,0x61,0xa3,0x6e,0x09,
                 0x33,0xf5,0x5f,0xc4,0x9a,0x35,0x7f,0x06,0x2d,
                 0xb0,0xb6,0xa7,0xb6,0x13,0xcd,0xdf,0xdb,0x81,
                 0x2e,0xfd,0xfe,0xe3,0xeb,0x5b,0x61,0x7f,0x02,
                 0x91,0x8e,0xcd,0xe0,0xe9,0xf6,0x85,0x23,0x13,
                 0xd8,0xfd,0xa4,0x1a,0x64,0xb2,0xb5,0x97,0x21,
                 0x24,0xa7,0x25,0x8c,0xe8,0x90,0x14,0x02,0xf8,
                 0x4a,0x62,0xdf,0x4d,0xbf,0xe6,0xe8,0xb0,0x64,
                 0xcf,0xe6,0xcd,0x04,0x4d,0x94,0x89,0xbf,0x8e,
                 0xbb,0x95,0x52,0xec,0x9c,0x43,0x99,0x65,0x8e,
                 0x99,0x52},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0x79,0x66,0x44,0x0d,0xf7,0x9b,0x13,0xe9,0x5c,
                0x41,0x34,0x6e,0xb7,0x92,0xf3,0xec},
    }, {	// 25 (181)
        .key = {0x48,0x09,0xf3,0x1e,0x93,0x42,0x3c,0xab,0xf4,
                0x4c,0xdd,0xca,0xd2,0x3d,0xa7,0xd7,0xae,0xe7,
                0x34,0xd3,0x11,0xfc,0x7b,0xab,0xc2,0x76,0xa1,
                0xbd,0x3d,0x35,0x13,0x98,0x61,0xea,0xd1,0x03,
                0x69,0x35,0x0d,0x42,0x1d,0x0a,0xf4,0x94,0x49,
                0x59,0xcc,0x00,0x6f,0xee,0x3f,0x51,0xb9,0x96,
                0xf6,0x60,0x31,0x83,0x6a,0x91,0x34,0xf1,0xf7,
                0xa0,0x24,0x0a,0x33,0x9e,0x5e,0x07,0x7d,0x36,
                0x6c,0x99},
        .key_len = 74,
        .data = {0x6e,0x4a,0xbd,0x41,0x4d,0xca,0x21,0xa6,0xad,
                 0x43,0x31,0x46,0x98,0x62,0x73,0xe2,0xda,0x95,
                 0x2e,0xf6,0x13,0xcd,0x1f,0x9a,0x0a,0x83,0x6c,
                 0xa6,0x44,0xf9,0xde,0x19,0xd6,0xc2,0x4a,0xbc,
                 0x77,0x84,0x50,0x02,0xd9,0xfd,0x48,0x33,0x3a,
                 0x44,0x7a,0xc9,0x36,0x51,0x8d,0x1b,0xdf,0xc0,
                 0x43,0x38,0x0f,0xd2,0x63,0x16,0xfd,0xb5,0xf6,
                 0xec,0x0f,0x05,0xb5,0xdc,0xef,0x92,0xc3,0xd5,
                 0xe1,0x64,0x98,0xb8,0x54,0xfc,0x3d,0xb9,0xb6,
                 0xdd,0xbf,0x09,0x8d,0x4b,0xde,0xb2,0xc4,0x53,
                 0x05,0xc2,0x42,0x0b,0x7f,0xab,0xc2,0x1b,0xe7,
                 0xea,0xde,0x7c,0xe0,0xe7,0x6c,0x80,0x07,0x1c,
                 0x0e,0x13,0x26,0x7a,0x05,0x40,0xab,0x08,0x46,
                 0xf7,0x58,0xce,0xd0,0x0d,0x3b,0xf1,0x3c,0x84,
                 0xe1,0x1f},
        .data_len = 128,
        .mac_len = 16,
        .mac = {0xd7,0xba,0xa0,0x11,0x7d,0x00,0x8a,0xf7,0x86,
                0xc2,0xba,0xcb,0x38,0xb9,0xd3,0x86},
    }, {	// 26 (195)
        .key = {0xfe,0xe6,0x03,0x25,0x85,0x82,0xe3,0xa3,0xe8,
                0xfe,0xb8,0x86,0x59,0x9d,0x4a,0xc4,0x05,0xa1,
                0x63,0x4c,0x32,0x0e,0x85,0xea,0x8a,0xb0,0xdc,
                0x6b,0xb6,0x5f,0x72,0x01,0x2f,0x82,0xa2,0xe9,
                0x51,0xd2,0xcf,0x4a,0xb2,0x61,0x56,0x61,0xb1,
                0xda,0xc0,0xdb,0x52,0x0a,0x3d,0x82,0x49,0x9f,
                0x4e,0x1c,0x54,0x30,0xc1,0x90,0xce,0x7e,0xe2,
                0x4b,0x82,0xfa,0xf0,0xe2,0xbd,0x87,0xce,0xf9,
                0xa7,0x80},
        .key_len = 74,
        .data = {0x67,0x54,0x1f,0x77,0xf4,0xe4,0x0d,0x14,0x30,
                 0x35,0x46,0x25,0x05,0xde,0x14,0xa0,0x21,0x24,
                 0xb9,0x92,0xec,0x1d,0x00,0x64,0xbd,0x15,0x18,
                 0x5d,0x4d,0x30,0xa2,0x69,0x6c,0x51,0x09,0x19,
                 0xf2,0x3b,0x12,0xea,0xf9,0xf6,0xb4,0xca,0x49,
                 0x75,0x29,0xd8,0x14,0x75,0x45,0x6c,0xe4,0xa8,
                 0x07,0x57,0xd1,0x13,0x6e,0x6c,0xf7,0xb4,0x8d,
                 0x3f,0x27,0x69,0xe2,0x2c,0xdd,0x0d,0xe4,0x9b,
                 0x72,0xe4,0xdb,0x83,0x93,0x39,0xf4,0x2d,0xf2,
                 0x45,0x95,0x3b,0x3b,0x53,0xee,0xe8,0x4a,0x22,
                 0xd1,0x91,0x9b,0x8b,0xc3,0x75,0x02,0x63,0x53,
                 0xb9,0x9c,0xa3,0xaa,0xaf,0x05,0xc6,0x64,0x57,
                 0xcb,0x73,0x9e,0x26,0x23,0x5c,0x50,0x07,0xdb,
                 0x66,0xde,0xa0,0x90,0x0a,0xe9,0xd6,0x21,0xfb,
                 0x6b,0x93},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0xf9,0x14,0xc8,0x42,0xb7,0x8c,0x3b,0x91,0xfe,
                0x66,0x26,0x27,0x2c,0x04,0xf6,0xbf,0xa3,0x9c,
                0x58,0x6d,0x48,0x23,0xce,0x0e},
    }, {	// 27 (196)
        .key = {0x83,0x2f,0x87,0xd5,0x96,0x44,0x9a,0xec,0xa6,
                0x56,0xe0,0xe0,0xb4,0xae,0x92,0xdc,0xd1,0x6a,
                0x66,0x88,0x90,0x20,0xa9,0xd2,0xbb,0xc4,0x8e,
                0xee,0x45,0xcc,0xc6,0x9b,0x80,0x91,0x50,0xa9,
                0x90,0xf9,0x93,0xb8,0x20,0x53,0xaa,0x42,0x53,
                0x82,0xff,0xdc,0xfd,0x5e,0x1b,0xb8,0x14,0x57,
                0xbc,0x6f,0x61,0x5c,0x28,0xfd,0x7b,0xfb,0xc2,
                0x0d,0xf6,0xc9,0xdb,0x78,0xd8,0x04,0xca,0x08,
                0x4c,0x77},
        .key_len = 74,
        .data = {0x78,0x2a,0xc1,0x6b,0xcd,0x74,0x4e,0xc0,0x16,
                 0xff,0xb6,0xb0,0x14,0xe0,0xc8,0x98,0x3d,0xfd,
                 0xe2,0x31,0xfa,0x72,0xc3,0x12,0x12,0x34,0x9a,
                 0x77,0x66,0xf4,0x62,0x40,0xe0,0x47,0x72,0x3d,
                 0xa6,0x03,0x50,0xa8,0x93,0xec,0xc7,0xf3,0xe7,
                 0x90,0x39,0xc5,0x3d,0x6f,0x36,0x3f,0xbe,0x5f,
                 0x4c,0x83,0x95,0x2f,0x21,0x77,0xa2,0x8b,0xc0,
                 0xc6,0x73,0x1f,0x31,0x28,0x70,0x00,0x4c,0xe4,
                 0x55,0x47,0xce,0x93,0xe6,0xff,0xad,0x26,0xde,
                 0x41,0xa9,0x2a,0x28,0x9d,0x24,0x4b,0x51,0xbc,
                 0x33,0x17,0x3e,0x44,0xf5,0x05,0x1a,0xfc,0x24,
                 0xb6,0x93,0x31,0xe9,0x7a,0x46,0x58,0xf5,0x16,
                 0x77,0xf4,0xcd,0xc5,0x06,0xba,0x65,0x7c,0x9e,
                 0xf3,0xf1,0x72,0x30,0x23,0xf8,0xe0,0xa0,0xe8,
                 0xaa,0x05},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0xc6,0x8f,0x21,0x5b,0x05,0x98,0x81,0xc9,0xf9,
                0x71,0x17,0xb3,0xc6,0xd9,0xd6,0xde,0xea,0x2e,
                0x09,0x45,0xe3,0xe1,0x97,0x2d},
    }, {	// 28 (210)
        .key = {0x81,0x57,0x43,0x23,0xc9,0x73,0x54,0x07,0x19,
                0xd1,0x92,0x83,0x3d,0xdb,0x51,0xf1,0x3a,0x52,
                0xdc,0xba,0xe2,0x94,0xae,0xbe,0xa5,0x1b,0xe5,
                0xf6,0xaa,0x47,0xf3,0x57,0x1f,0x5d,0x97,0xfa,
                0xcd,0xcf,0x0c,0x7b,0xef,0xbe,0x80,0x9f,0x44,
                0xbd,0xc7,0x39,0x63,0xd8,0x51,0x4e,0x4f,0xd5,
                0x59,0x77,0x4b,0xb9,0x60,0x87,0xef,0x8e,0xda,
                0x6e,0x7c,0x64,0x27,0x5d,0x6d,0x96,0xc4,0x2b,
                0x4e,0x4e},
        .key_len = 74,
        .data = {0xb9,0xe9,0x44,0xe0,0xb4,0x2d,0x0f,0xf4,0x54,
                 0xf7,0xf8,0xaa,0x24,0xf0,0x0e,0x9e,0xe0,0x39,
                 0x05,0x8c,0xe4,0x09,0x41,0x11,0xe3,0x97,0x31,
                 0xb6,0xdc,0x3a,0xde,0x2a,0x4a,0xce,0xc4,0xcf,
                 0x9c,0x5b,0xe0,0x78,0xe4,0xf1,0x0a,0x72,0xd3,
                 0xd6,0x85,0xc1,0xe5,0xe4,0xd5,0xab,0xd9,0x2c,
                 0xd0,0x7b,0x64,0xdf,0xf8,0x7f,0x26,0x6f,0x08,
                 0x53,0xdd,0xf1,0xcd,0x61,0xd9,0xc6,0x37,0xa9,
                 0xb0,0x7a,0xb0,0xbe,0x32,0xec,0xac,0x11,0x9f,
                 0xaf,0x82,0x72,0x18,0xb1,0x7a,0xd4,0x54,0x1a,
                 0x27,0x51,0x94,0x77,0xf7,0x6e,0xd9,0x18,0x08,
                 0x9f,0x54,0xb6,0x3d,0x0e,0x1e,0x5a,0x92,0x98,
                 0x29,0x79,0xac,0x18,0x77,0x64,0xb5,0xe9,0x89,
                 0xe0,0x66,0xa6,0x1b,0x10,0x65,0x34,0x0e,0x9c,
                 0xd2,0x03},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x51,0x4b,0xd1,0x84,0x95,0xf6,0xde,0x0e,0x23,
                0x70,0x54,0xb8,0xe3,0xba,0x1a,0x74,0xc3,0xfa,
                0xda,0x42,0x79,0xad,0x6b,0x85,0x50,0xf3,0xa1,
                0x47,0x12,0xc5,0x28,0xdf},
    }, {	// 29 (211)
        .key = {0x44,0xf7,0x1c,0x23,0x17,0xcd,0xe5,0x21,0x51,
                0xc8,0x42,0x60,0xd1,0xd3,0xc0,0x4a,0x28,0xcc,
                0x15,0xce,0x5b,0x38,0x02,0xb2,0xe5,0x35,0x7e,
                0x2b,0xfc,0xaf,0x10,0xab,0x15,0xd7,0x7d,0xfa,
                0xaa,0xd1,0xa3,0x88,0x3b,0xad,0xa5,0x02,0x93,
                0x99,0x48,0x23,0x4c,0x55,0x9d,0xcd,0x95,0xe7,
                0xe1,0x58,0x33,0x8f,0xa1,0x2a,0xc6,0xfd,0x21,
                0x87,0x4e,0xc2,0xff,0xab,0xed,0x05,0x14,0x16,
                0xef,0x77},
        .key_len = 74,
        .data = {0x2a,0xc0,0xbb,0x05,0x24,0xc2,0x2b,0x90,0x2d,
                 0xe3,0x4c,0xe6,0x4e,0x61,0x72,0xd1,0xb2,0x07,
                 0x4e,0x15,0x9f,0x51,0x7a,0xb1,0xab,0xd1,0x52,
                 0x62,0x2c,0xd1,0x06,0x69,0xf0,0x3a,0xed,0x8e,
                 0x2e,0xb5,0x1c,0x65,0xbd,0x0f,0x38,0xd0,0x84,
                 0xe2,0x88,0xc5,0x32,0x72,0x4e,0x51,0x2f,0xd5,
                 0x58,0xdd,0xd2,0x57,0xd2,0xb1,0xd4,0x1c,0x5e,
                 0xb6,0x04,0x07,0x67,0x80,0x3d,0xdb,0xb1,0x8b,
                 0x95,0xa0,0x35,0xc5,0xd8,0x49,0x2d,0x4d,0x35,
                 0x93,0x6b,0x7b,0x36,0x30,0xee,0x20,0xf6,0x25,
                 0xb7,0x0f,0x8e,0x71,0xd9,0xdc,0xd0,0xef,0xd0,
                 0xe3,0x38,0x7d,0x13,0x8c,0x1f,0x5e,0xed,0xce,
                 0x32,0xdd,0x88,0xf2,0x23,0x33,0x4b,0x9a,0x9e,
                 0xab,0x65,0x01,0x7f,0x04,0xaa,0x84,0x42,0x17,
                 0x9f,0x62},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0xca,0x00,0x53,0xd5,0x1f,0x6c,0xf6,0xf9,0x99,
                0x8f,0xf1,0xe0,0xdb,0x00,0xb9,0x0e,0x82,0xc7,
                0xb1,0x8c,0xb5,0x37,0x7a,0xcc,0x8e,0xbe,0x9a,
                0xfe,0x20,0xda,0x1c,0x3d},
    }
};

struct HMAC_TEST_VECTOR fips_sha384_hmac_general_test_vector[] = {
    {	// 0	(0)
        .key = {0xf1,0x6a,0xd7,0x37,0x90,0xca,0x39,0xc7,0xf9,0x85,
                0x6c,0x44,0x83,0x20,0x2e,0x7f,0x8e,0x0c,0x82,0x83,
                0xc7,0xd5,0x0d,0x6d,0xa7,0x9c,0xc0,0x7d,0x3d,0xc7,
                0xb7,0x6c,0x2e,0xf7,0x61,0x00,0xfa,0x3a,0xe2,0xdf,
                0x80,0x83,0xb5,0xa1,0xc5,0x57,0x96,0x28,0xf1,0xc8},
        .key_len = 50,
        .data =	{0x98,0x70,0x00,0x76,0x54,0xeb,0xc3,0xd2,0x8f,0x88,
                 0x3b,0xb8,0x32,0xe0,0xb3,0x17,0x00,0xf9,0x23,0xd9,
                 0xc9,0xb1,0x01,0x68,0xe0,0x60,0x59,0x71,0xcf,0xb9,
                 0x20,0xe8,0x48,0xf1,0xc6,0x4c,0x5f,0x24,0x0a,0x2c,
                 0xf7,0xf4,0x12,0xea,0x7a,0x73,0xbb,0xbf,0xce,0x43,
                 0x2e,0xff,0x84,0xfb,0xb4,0x9e,0x52,0xcd,0xcb,0xf4,
                 0xc3,0x66,0x79,0xbd,0x2d,0x16,0xe0,0x64,0xe4,0x31,
                 0x13,0x81,0xad,0xb5,0x28,0xa0,0x75,0x2c,0x8e,0x44,
                 0x43,0xd4,0xa1,0x2b,0x6c,0xfe,0x7c,0xd4,0x06,0xb4,
                 0x0e,0x3f,0x9e,0x9e,0x71,0xf4,0x2e,0x27,0x76,0x46,
                 0x49,0xdb,0x85,0xd9,0x99,0x13,0xa4,0x62,0x8b,0xd5,
                 0xd5,0xae,0x49,0xf6,0xa5,0xe6,0xe9,0x81,0x02,0x11,
                 0xe3,0x5d,0x4d,0xda,0xc9,0x29,0xb0,0x93},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x79,0xe2,0x4a,0x20,0x3b,0xf4,0x20,0x74,0xe7,0x2c,
                0x8b,0x4a,0x02,0x22,0xaf,0xac,0xe3,0xe8,0xce,0x7b,
                0x40,0x04,0xce,0xc2},
    }, {	// 1	(1)
        .key = {0xa5,0x70,0x9b,0xa5,0x52,0x9c,0xb9,0xa1,0xa2,0x27,
                0xf0,0xbe,0x44,0x8e,0x11,0x9a,0x35,0x6f,0x92,0xe1,
                0x3e,0xfc,0x34,0x63,0xbe,0xaa,0xe4,0x6a,0xa9,0x29,
                0xdf,0x4a,0xd1,0x99,0x1a,0x39,0x64,0xfb,0xe1,0x61,
                0xb6,0xe5,0xbe,0x34,0x41,0x7a,0x9c,0x00,0xeb,0x9a},
        .key_len = 50,
        .data = {0x4f,0x56,0x9d,0x60,0x40,0x56,0x63,0xff,0xd4,0x89,
                 0x37,0x77,0xcb,0xc3,0x71,0x55,0xd4,0x03,0xe2,0xb0,
                 0xf5,0x48,0x5d,0xa4,0x2c,0xa6,0x75,0x03,0x57,0x98,
                 0x89,0x46,0x51,0x98,0xfe,0xca,0x5e,0xed,0xcc,0x39,
                 0xc9,0xc5,0x3c,0x45,0xcb,0x83,0xf0,0x9d,0xaf,0x5a,
                 0x23,0x19,0x34,0x1b,0x32,0x38,0x33,0x4b,0x5b,0xcd,
                 0x81,0x79,0xc5,0xf5,0x17,0xce,0xc1,0x4c,0x70,0xe6,
                 0x50,0x61,0x33,0xde,0xe5,0x67,0x12,0xaf,0x6c,0x2d,
                 0xf2,0xba,0x8a,0x50,0x4c,0xa4,0x27,0xaf,0xd3,0x63,
                 0x2a,0x1f,0x57,0x99,0x83,0x60,0xe9,0x21,0x6f,0x50,
                 0x40,0xe8,0xf7,0x5f,0x5b,0xff,0xba,0x43,0x68,0xee,
                 0xed,0xed,0xe5,0x4a,0xa0,0xbb,0x05,0x8a,0x43,0xef,
                 0x55,0x16,0x68,0x60,0x9f,0xa1,0xcb,0x6f},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x24,0x7e,0xb5,0x1a,0x39,0x7b,0xa3,0x69,0xec,0xba,
                0x43,0xb9,0x5a,0x46,0xa9,0x33,0xcf,0xf0,0xb1,0x00,
                0x57,0x14,0xf0,0xe5},
    }, {	// 2	(15)
        .key = {0xc2,0xf2,0xf7,0x98,0x57,0x28,0xb6,0x77,0xa7,0xad,
                0x06,0x2d,0xd9,0x60,0x5a,0x2c,0x24,0xe7,0xcd,0xfa,
                0x86,0x98,0x6f,0x35,0xb9,0x9a,0xdc,0xd4,0x63,0x47,
                0x14,0xaf,0x8d,0xd5,0x86,0x42,0x56,0x36,0x6e,0xad,
                0xe8,0x3c,0x61,0x00,0xac,0x01,0x26,0xb6,0xba,0x86},
        .key_len = 50,
        .data = {0x35,0xaf,0x2e,0xa1,0x67,0xe5,0x6c,0x84,0x21,0xcd,
                 0xab,0x1b,0x9f,0xc9,0x9b,0xe4,0xb8,0x5f,0x74,0xc7,
                 0x06,0xd4,0x3a,0x49,0x47,0xfc,0x3f,0x02,0x03,0x50,
                 0xe9,0x51,0x70,0x41,0xb5,0x4e,0x92,0xcc,0x7c,0x00,
                 0xa6,0x4f,0xf6,0xd1,0xc1,0x9b,0x7c,0x3e,0xb5,0x4a,
                 0x12,0xd3,0x34,0x53,0xa4,0x57,0x38,0xdb,0x90,0x44,
                 0xa1,0x4e,0x65,0x7a,0x20,0xaf,0xea,0x33,0x55,0x2c,
                 0x63,0x3a,0x34,0xf6,0x0f,0x58,0xad,0x4f,0xf5,0x0f,
                 0x8c,0xe5,0xe1,0x8b,0x9e,0x5e,0xa9,0xd6,0x15,0x34,
                 0xb4,0x4b,0x2d,0xc3,0xbd,0x4d,0x10,0xa0,0xd5,0x39,
                 0xf7,0x2d,0xa7,0x98,0x93,0x6a,0x00,0x9a,0xab,0x0e,
                 0x8f,0xc0,0x06,0xd7,0xe9,0xd8,0x8b,0x1e,0xcf,0x2e,
                 0xa7,0xae,0xb4,0x01,0xef,0xd6,0x7a,0x34},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x6e,0x49,0xe4,0xaa,0x01,0xda,0x45,0xcc,0x5f,0xfb,
                0x71,0x56,0x9f,0x25,0x7e,0xcf,0x11,0x4f,0xe8,0x58,
                0xba,0x95,0x90,0xaf,0xc2,0x3a,0xfa,0xc9,0xc0,0xd6,
                0x7f,0x52},
    }, { 	//3	(16)
        .key = {0x8f,0x23,0x9b,0x06,0xfc,0x66,0x78,0xbe,0x26,0x30,
                0x7d,0xc7,0x02,0xf8,0x54,0xf6,0xa3,0xd0,0xd9,0x80,
                0xf6,0x45,0x73,0x04,0xaf,0x87,0xa5,0xcc,0x83,0xca,
                0xe0,0x50,0x98,0xeb,0x9c,0xfb,0x3a,0x57,0xa7,0x32,
                0xcf,0x29,0xbc,0x93,0x0d,0x92,0x57,0x7a,0x6a,0x7e},
        .key_len = 50,
        .data = {0x35,0xb1,0x27,0xb5,0x2a,0x9b,0x4a,0xec,0xe9,0x78,
                 0xad,0x17,0xaa,0xa7,0x00,0xb5,0x47,0xb1,0x7e,0xab,
                 0x59,0xda,0x27,0x81,0x9e,0xf6,0x50,0xce,0x9f,0x7e,
                 0x5e,0xf1,0x8f,0xe3,0xcf,0x02,0x75,0x22,0x7a,0x09,
                 0x8e,0x99,0x17,0x6d,0xec,0x19,0x01,0xaf,0x64,0x3c,
                 0x3b,0x57,0xa7,0xf9,0xb1,0x2f,0xaf,0x75,0xc1,0xb0,
                 0x5d,0x1c,0xfa,0xba,0x60,0xf1,0x24,0x88,0x58,0x22,
                 0x80,0xe2,0x3b,0xe0,0x51,0x94,0xf8,0x6d,0x9a,0x20,
                 0x5b,0x77,0x2a,0xb0,0x31,0xa4,0xd6,0x4e,0xac,0x6e,
                 0x06,0x57,0x09,0x31,0x73,0x2d,0x6f,0x82,0x2e,0x2d,
                 0x1b,0xc3,0xe5,0xe1,0xba,0xf4,0x62,0x76,0x16,0xcc,
                 0x54,0x70,0xf5,0x09,0x52,0x9c,0x3e,0x04,0x1d,0x46,
                 0x5e,0x88,0x25,0xad,0xea,0xe4,0x4f,0xb4},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x4a,0x13,0x78,0x36,0x50,0xcb,0x96,0xaa,0x0d,0xcb,
                0x4b,0xca,0x10,0xec,0x30,0xee,0x2a,0x9d,0x37,0x68,
                0xf2,0xb6,0xf1,0xdc,0x62,0x6f,0x99,0x54,0x5e,0xfd,
                0xef,0xe6},
    }, {	// 4 (30)
        .key = {0x1c,0xe7,0xe2,0x0a,0xbb,0xdc,0xd1,0x15,0x4d,0x4b,
                0x53,0x67,0x14,0xff,0x53,0x4a,0x01,0xb8,0xe8,0x8c,
                0x78,0xda,0x34,0xd6,0x53,0x63,0x8c,0x39,0x29,0x1f,
                0xd8,0x0a,0xd0,0x1f,0x3d,0xf0,0x20,0x67,0xfa,0x3b,
                0xfa,0xe7,0x90,0x77,0x89,0xad,0x26,0x41,0xc8,0x58},
        .key_len = 50,
        .data = {0x7c,0xd7,0x50,0xb5,0xc9,0xb2,0xbb,0xc3,0xee,0x95,
                 0x5a,0x4f,0x4f,0xa7,0xc9,0x56,0x84,0x6c,0x8b,0x1b,
                 0x52,0xea,0xa0,0x6f,0xd9,0x0a,0x5a,0x30,0x0e,0x42,
                 0x6c,0x10,0x6c,0x71,0x44,0x97,0xe7,0x0a,0x9b,0x6c,
                 0x22,0x75,0x4a,0xd0,0xe1,0xb2,0x5f,0x6b,0xc1,0x40,
                 0x70,0x4b,0x27,0x3d,0x2f,0x2a,0x76,0xce,0x3f,0xef,
                 0x85,0xc1,0x46,0x78,0x50,0x71,0x44,0x97,0xed,0xea,
                 0x23,0x5a,0xc2,0x4e,0x8f,0x90,0xf6,0x78,0x07,0x88,
                 0x25,0xde,0x34,0x1c,0x58,0xbc,0x7a,0xee,0x34,0x6f,
                 0xce,0xf2,0x71,0x1e,0xc7,0x2d,0x8e,0xa0,0xf7,0xbc,
                 0xc3,0x9a,0x7b,0x17,0x38,0xe8,0xd1,0x97,0x74,0x3f,
                 0xea,0x36,0x18,0x10,0x80,0x97,0xcd,0xaf,0xaa,0x46,
                 0x7b,0xb4,0xae,0x40,0xbc,0xa2,0x16,0xb6},
        .data_len = 128,
        .mac_len = 40,
        .mac = {0x72,0x9a,0x16,0x7f,0x19,0x86,0xca,0xc5,0x3f,0xf3,
                0xe1,0x11,0xff,0x82,0xf2,0xa7,0x7b,0x57,0x3d,0x08,
                0x63,0xe1,0xa3,0xae,0xaf,0x00,0x04,0x1a,0x03,0xe1,
                0x43,0x01,0x88,0xa2,0x02,0xbd,0xb7,0xe9,0xbf,0xd4},
    }, {	// 5 (31)
        .key = {0x36,0x2b,0xc4,0x40,0xe5,0xda,0xc1,0x6a,0x43,0x69,
                0x58,0x1c,0x0c,0xb5,0xbe,0x45,0xbf,0x4f,0x17,0x08,
                0x47,0x87,0x3d,0x6c,0xdb,0xc9,0xbd,0x55,0x23,0x2d,
                0x23,0xb3,0x9c,0x49,0x78,0xf9,0x3d,0x4a,0x08,0xd1,
                0x5b,0x43,0x69,0x0d,0xca,0xc4,0xb8,0xe1,0x45,0xaf},
        .key_len = 50,
        .data = {0x1a,0xf3,0xae,0xda,0xa8,0xfa,0xc5,0x51,0x57,0xf3,
                 0x06,0x42,0xa0,0x02,0x58,0x10,0x2d,0xbd,0x48,0x21,
                 0x98,0xe0,0xf1,0x34,0x76,0x41,0x1f,0xf5,0x94,0x06,
                 0xb4,0xce,0x80,0x15,0x4a,0x01,0x4b,0xcc,0x19,0xf4,
                 0x8e,0xf3,0x1b,0xcb,0xab,0xee,0x6f,0x3c,0x55,0x37,
                 0xfc,0x9f,0x53,0x0c,0x56,0x45,0x80,0x65,0xe5,0x0b,
                 0x17,0x29,0x44,0x2f,0x2d,0xa1,0xe7,0x62,0x7f,0x2d,
                 0x01,0x1e,0x6e,0x36,0xa4,0x39,0x48,0x63,0x2a,0xbb,
                 0xc9,0x10,0xd5,0xed,0xe2,0xfc,0xb2,0xb2,0xb8,0x41,
                 0xc3,0x1a,0xf0,0x8a,0x5c,0x35,0x2a,0x80,0xce,0x25,
                 0xcb,0x85,0x43,0x77,0x00,0xa5,0xe9,0xb4,0x00,0xc9,
                 0x53,0x32,0x91,0x2e,0x1c,0x30,0xcd,0x16,0xcd,0x22,
                 0x26,0xbe,0x00,0x4a,0xa8,0x8f,0xc6,0x88},
        .data_len = 128,
        .mac_len = 40,
        .mac = {0x64,0x67,0xe5,0xa6,0x90,0xbd,0x32,0xe1,0x57,0xcf,
                0x8c,0xe6,0x74,0xac,0x63,0x0b,0x74,0xce,0x32,0xa7,
                0x8e,0x8f,0x78,0x41,0x53,0x46,0xc0,0x0c,0x30,0x60,
                0xa4,0xa2,0x6c,0x40,0xc3,0xe2,0xea,0xbd,0x80,0xa7},
    }, {	// 6 (45)
        .key = {0x5e,0xab,0x0d,0xfa,0x27,0x31,0x12,0x60,0xd7,0xbd,
                0xdc,0xf7,0x71,0x12,0xb2,0x3d,0x8b,0x42,0xeb,0x7a,
                0x5d,0x72,0xa5,0xa3,0x18,0xe1,0xba,0x7e,0x79,0x27,
                0xf0,0x07,0x9d,0xbb,0x70,0x13,0x17,0xb8,0x7a,0x33,
                0x40,0xe1,0x56,0xdb,0xce,0xe2,0x8e,0xc3,0xa8,0xd9},
        .key_len = 50,
        .data = {0xf4,0x13,0x80,0x12,0x3c,0xcb,0xec,0x4c,0x52,0x7b,
                 0x42,0x56,0x52,0x64,0x11,0x91,0xe9,0x0a,0x17,0xd4,
                 0x5e,0x2f,0x62,0x06,0xcf,0x01,0xb5,0xed,0xbe,0x93,
                 0x2d,0x41,0xcc,0x8a,0x24,0x05,0xc3,0x19,0x56,0x17,
                 0xda,0x2f,0x42,0x05,0x35,0xee,0xd4,0x22,0xac,0x60,
                 0x40,0xd9,0xcd,0x65,0x31,0x42,0x24,0xf0,0x23,0xf3,
                 0xba,0x73,0x0d,0x19,0xdb,0x98,0x44,0xc7,0x1c,0x32,
                 0x9c,0x8d,0x9d,0x73,0xd0,0x4d,0x8c,0x5f,0x24,0x4a,
                 0xea,0x80,0x48,0x82,0x92,0xdc,0x80,0x3e,0x77,0x24,
                 0x02,0xe7,0x2d,0x2e,0x9f,0x1b,0xab,0xa5,0xa6,0x00,
                 0x4f,0x00,0x06,0xd8,0x22,0xb0,0xb2,0xd6,0x5e,0x9e,
                 0x4a,0x30,0x2d,0xd4,0xf7,0x76,0xb4,0x7a,0x97,0x22,
                 0x50,0x05,0x1a,0x70,0x1f,0xab,0x2b,0x70},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x7c,0xf5,0xa0,0x61,0x56,0xad,0x3d,0xe5,0x40,0x5a,
                0x5d,0x26,0x1d,0xe9,0x02,0x75,0xf9,0xbb,0x36,0xde,
                0x45,0x66,0x7f,0x84,0xd0,0x8f,0xbc,0xb3,0x08,0xca,
                0x8f,0x53,0xa4,0x19,0xb0,0x7d,0xea,0xb3,0xb5,0xf8,
                0xea,0x23,0x1c,0x5b,0x03,0x6f,0x88,0x75},
    }, {	// 7 (46)
        .key = {0xf8,0x69,0x02,0xe5,0xe5,0xdb,0x47,0x8e,0xc6,0xe2,
                0x78,0x69,0x27,0x28,0xa8,0x12,0xc4,0xcd,0x87,0x45,
                0xf9,0x0a,0x7d,0x9f,0x79,0x15,0xf5,0xa9,0x43,0x45,
                0xfc,0x12,0xd2,0x77,0x0a,0x3c,0x94,0xb0,0x1f,0xfb,
                0x9e,0x04,0x12,0x99,0x9e,0xb6,0x26,0x1d,0x11,0xa0},
        .key_len = 50,
        .data = {0xe0,0xbc,0xac,0xbe,0x96,0xda,0xd6,0xf6,0x0e,0x51,
                 0x12,0x9f,0x35,0xac,0xd0,0x3e,0x12,0x27,0x6a,0x91,
                 0xfa,0x13,0xfc,0x15,0x03,0x7c,0x75,0xca,0xbb,0x0a,
                 0xee,0x3a,0x19,0x25,0x3b,0xb8,0xb3,0x5c,0xc0,0xe6,
                 0x32,0x08,0x86,0x7a,0x03,0x2c,0x8f,0x41,0x50,0xa0,
                 0x66,0x64,0x2f,0x6f,0xf9,0xea,0x19,0x7d,0xab,0x7e,
                 0x9d,0x6d,0xa6,0x72,0x55,0xc1,0x6e,0x05,0x1a,0x43,
                 0xbc,0xe1,0x74,0xa4,0x89,0xe8,0x54,0x64,0x69,0x30,
                 0x06,0xf1,0x1a,0x4c,0x61,0x13,0x5d,0xce,0x41,0x87,
                 0x04,0x09,0x37,0xeb,0x4d,0x1c,0x7e,0xda,0x6e,0x2c,
                 0x31,0x57,0x71,0xf0,0xbc,0x6f,0x42,0x73,0x91,0x1a,
                 0x07,0x15,0x1c,0x63,0xaf,0xd3,0xf8,0xc8,0xce,0xc9,
                 0x63,0xe4,0xa8,0xf5,0xef,0x4b,0x8b,0x3e},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x4b,0xb4,0xeb,0x2d,0xb2,0xcc,0x92,0x1b,0x15,0x9b,
                0x78,0xa2,0xbb,0x9e,0xdc,0x16,0x08,0xbb,0x2a,0x1c,
                0xa9,0x87,0x3b,0x41,0x1a,0xe3,0x0a,0x63,0x38,0x6e,
                0x46,0x2f,0x9f,0x69,0xd9,0xf5,0xfc,0x83,0x8f,0xf1,
                0x81,0x87,0x48,0xaa,0xb7, 0x4d,0xa9,0x4f},
    }, {	// 8	(60)
        .key = {0x40,0xea,0xe6,0xb8,0xe3,0xab,0xea,0x17,0xc0,0x69,
                0xf0,0x88,0x26,0x49,0x57,0x7b,0x19,0x52,0xde,0x40,
                0xf4,0x7c,0x6a,0xc0,0x53,0x0a,0x03,0x6b,0x2f,0x1a,
                0x1f,0x71,0x4b,0x7b,0x23,0x35,0xcf,0xbe,0x27,0xff,
                0x33,0x90,0xf9,0xf0,0x5f,0x47,0x65,0x3c,0x11,0xba,
                0xb4,0x93,0x7e,0x56,0x73},
        .key_len = 55,
        .data = {0xd2,0x81,0x3f,0x53,0x1c,0xe4,0x93,0x1c,0xb2,0x90,
                 0x89,0x95,0x79,0xe6,0xc7,0x5b,0xea,0x8a,0x32,0x4d,
                 0xb8,0x75,0xb4,0x40,0xb2,0x46,0x3d,0xf5,0xec,0xd9,
                 0x07,0x48,0x19,0x1b,0x1f,0xa9,0x3c,0x1d,0x21,0xd0,
                 0x80,0x67,0x42,0xad,0x63,0x8e,0x94,0x9e,0x1a,0x09,
                 0x86,0xe5,0x31,0x40,0xaa,0x59,0x73,0xe6,0xbc,0x5b,
                 0x09,0x89,0xdf,0x0c,0xe6,0x67,0x29,0xbe,0x62,0x84,
                 0x62,0xa8,0x24,0xf9,0x09,0xdd,0x46,0x8f,0x98,0x7f,
                 0xb4,0x8c,0x0a,0x2f,0xd5,0xcd,0x99,0xc9,0x6e,0x15,
                 0xcc,0x4e,0xc3,0xa5,0xb1,0x22,0xff,0x0d,0x67,0x78,
                 0xd5,0x41,0xe0,0x0a,0x68,0xef,0xe5,0x0d,0x68,0x10,
                 0x5b,0x64,0x7e,0xbe,0xc4,0x14,0xeb,0x45,0x09,0xf8,
                 0x6c,0x7c,0x76,0xb6,0x60,0x56,0x06,0xf1},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x3f,0x25,0x54,0x80,0xd6,0x45,0x36,0x92,0x98,0xf7,
                0x72,0x4f,0x42,0xbc,0xa1,0xb9,0x84,0x23,0x38,0x52,
                0x77,0x7f,0xf7,0xf7},
    }, {	// 9	(61)
        .key = {0x4e,0x58,0x96,0x74,0x03,0x0c,0x40,0x67,0x9c,0x34,
                0x38,0x74,0xd6,0xa2,0xd6,0x25,0x63,0x55,0xeb,0x95,
                0x48,0x4e,0x4a,0xdd,0x84,0xa0,0x87,0xcf,0xf2,0xc7,
                0xfe,0xe7,0x70,0x3a,0x17,0x7e,0x41,0x44,0xc9,0x41,
                0xb0,0x0f,0x5d,0xe2,0xf6,0x02,0x75,0x0d,0x5e,0x4c,
                0x4c,0x9e,0xa7,0xc1,0x3f},
        .key_len = 55,
        .data = {0xe9,0xe8,0x35,0x61,0xcf,0x23,0xff,0xd4,0x4a,0x79,
                 0xee,0x76,0x54,0xc8,0xf3,0xc7,0x80,0x2a,0x5a,0x35,
                 0x8f,0x2f,0xfa,0x88,0x3e,0x69,0xaf,0x7d,0x63,0x2e,
                 0x0a,0xb1,0x38,0x99,0x46,0xc1,0xf7,0xd2,0x7e,0xb0,
                 0xa7,0x8f,0x1e,0x89,0x35,0xdb,0x98,0x45,0xc6,0x17,
                 0x58,0xee,0x4c,0x3e,0xf9,0x05,0x57,0x6d,0xb9,0x22,
                 0x2f,0xa2,0x2a,0xda,0x1f,0xc3,0x2b,0xe5,0x13,0xe3,
                 0x17,0x80,0x66,0xc2,0x3c,0x11,0xf5,0x92,0x8f,0x0a,
                 0x78,0x01,0x9d,0x0f,0x12,0x73,0xc5,0x5b,0x26,0x8f,
                 0xa5,0x60,0x6d,0xfe,0xd2,0xad,0x45,0x6f,0xcc,0x15,
                 0x4c,0xdf,0x31,0x0e,0x2e,0x17,0x30,0x57,0xbb,0x76,
                 0x41,0xde,0x3d,0xf0,0x13,0xe0,0x08,0x57,0xc6,0x52,
                 0x52,0xd9,0x5b,0x80,0x45,0xcb,0x69,0xf4},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0xa1,0x22,0x75,0x28,0xf8,0xc2,0x1c,0xf0,0x4c,0x7c,
                0x9e,0x6c,0x02,0x01,0x19,0xbb,0x6e,0xe9,0x07,0xa9,
                0xe1,0x10,0x7c,0x61},
    }, {	// 10	(75)
        .key = {0x88,0x60,0x41,0x8e,0x48,0xec,0x77,0xa2,0x29,0x2c,
                0x51,0x87,0x62,0x07,0x6b,0x7a,0x0c,0xc6,0x39,0x2c,
                0xe1,0xc9,0xee,0x17,0x43,0x78,0x9c,0x11,0x80,0x7d,
                0x9d,0x22,0x54,0x31,0x33,0x93,0xaf,0x53,0x6b,0x47,
                0xb9,0x00,0x47,0x4f,0x13,0xdf,0x1b,0xb8,0x60,0x9d,
                0x38,0xa7,0x05,0x99,0xc8},
        .key_len = 55,
        .data = {0x10,0x7f,0xd2,0xe4,0xbd,0x7a,0x19,0xa4,0xff,0x6f,
                 0x48,0x2d,0x62,0x89,0x6d,0xa5,0x83,0xc3,0x27,0x7e,
                 0x23,0xab,0x5e,0x53,0x7a,0x65,0x31,0x12,0xcd,0xf2,
                 0x30,0x60,0x43,0xb3,0xcc,0x39,0xf5,0x28,0x0b,0xd7,
                 0x44,0xfe,0x81,0xd6,0x6f,0x49,0x7b,0x95,0x65,0x0e,
                 0x7d,0xdf,0xd7,0x04,0xef,0xcb,0x92,0x9b,0x13,0xe0,
                 0x0c,0x3e,0x3a,0x7d,0x3c,0xd5,0x38,0x78,0xaf,0x8f,
                 0x15,0x06,0xd9,0xde,0x05,0xdb,0xa9,0xc3,0x9a,0x92,
                 0x60,0x4b,0x39,0x4e,0xa2,0x5a,0xcb,0xa2,0xcd,0xa7,
                 0xb4,0xae,0x8b,0x08,0x09,0x8b,0xa3,0xf0,0xfd,0xea,
                 0x15,0x35,0x9d,0xf7,0x65,0x17,0xbe,0x84,0x37,0x7f,
                 0x33,0x63,0x1c,0x84,0x43,0x13,0xac,0x33,0x5a,0xa0,
                 0xd5,0x90,0xfe,0xc4,0x72,0xd8,0x05,0x52},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x51,0x28,0x05,0xc9,0x80,0x6a,0x47,0x39,0xd0,0x4c,
                0x19,0x4a,0x1f,0x1b,0xe6,0x79,0xe9,0xe5,0x0e,0x31,
                0x3f,0xe6,0x3e,0xc5,0xd1,0x2c,0xfc,0x3c,0xf4,0xb0,
                0x70,0x73},
    }, {	// 11	(76)
        .key = {0xf1,0x57,0x76,0x97,0x6b,0x37,0x2a,0xbe,0x66,0x37,
                0x99,0x61,0xf0,0x78,0x73,0x38,0x76,0x0a,0x9a,0x75,
                0xef,0x51,0xec,0x49,0x57,0xad,0x5c,0xa9,0x5f,0x59,
                0x48,0x52,0x63,0x94,0x07,0x0b,0x9c,0xff,0xc1,0x2a,
                0x97,0x47,0x83,0x59,0xe5,0x03,0x92,0x9a,0x15,0xe0,
                0x00,0x89,0xdf,0xfb,0x7e},
        .key_len = 55,
        .data = {0xcf,0x85,0x77,0x54,0xd1,0x8e,0x6b,0x8b,0x32,0x94,
                 0x1d,0x69,0xfe,0x44,0x16,0xa1,0x28,0x91,0x0b,0x68,
                 0x20,0xfc,0x0d,0xda,0xa7,0x13,0x00,0x99,0xe3,0x38,
                 0x4e,0xb7,0xae,0xa4,0xdd,0xd6,0x34,0xac,0x3e,0x8d,
                 0xbd,0x42,0x27,0x0e,0xc7,0xbe,0x23,0x06,0x58,0xdf,
                 0x88,0xc5,0x92,0x0c,0xa9,0x9f,0x88,0xe0,0x4e,0x92,
                 0x50,0xe6,0x61,0x29,0x5a,0xa1,0xea,0x9f,0xff,0xd0,
                 0x3e,0x48,0x5d,0xef,0x72,0x2d,0x63,0x01,0x16,0xf6,
                 0x28,0x8d,0x20,0x0e,0x81,0xe7,0x27,0x01,0xd2,0xb0,
                 0xd2,0x29,0x24,0xa0,0x8f,0x89,0x78,0x83,0x88,0xf9,
                 0x5b,0x82,0xd3,0x84,0xbb,0xa4,0xe8,0x0e,0xf9,0x95,
                 0x59,0x39,0x3f,0xa5,0xbd,0x8a,0x14,0x13,0xed,0xc8,
                 0x2e,0x8c,0x74,0xa5,0x87,0xef,0x40,0xa1},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x6b,0x44,0x2d,0x4f,0x5c,0xd3,0xe4,0xbc,0x60,0x9b,
                0xd2,0x09,0x6d,0xb3,0x1f,0x2b,0x2e,0x1e,0x41,0x3a,
                0xab,0xd5,0xdb,0x0a,0xef,0xdc,0x59,0x98,0x13,0xf6,
                0xdd,0x1b},
    }, {	// 12	(90)
        .key = {0x30,0x4f,0x1d,0xe5,0xe8,0xfc,0xd7,0xae,0xe3,0x4d,
                0x5f,0xe5,0x12,0x7f,0xcf,0xca,0x0b,0xdd,0x11,0x2b,
                0xb0,0xd9,0xa4,0x1f,0x0b,0x5b,0x9c,0xf7,0x7d,0x59,
                0xeb,0x72,0x18,0xa8,0xe0,0x30,0x49,0x12,0xed,0x69,
                0xba,0xa8,0xad,0xdf,0x76,0x59,0x25,0x11,0x4f,0xc4,
                0x4b,0xb2,0x7d,0x4b,0xc4},
        .key_len =  55,
        .data = {0x0b,0x99,0x5e,0xb3,0xf8,0xd1,0xfb,0x4c,0x1b,0xe0,
                 0xa7,0xfb,0x36,0x4e,0x5d,0x1b,0x4e,0xdf,0x5e,0x3e,
                 0xba,0x5d,0xdd,0x14,0x7b,0x97,0xfc,0x8e,0xcb,0xaa,
                 0xf7,0x42,0xf8,0x7f,0x9f,0x12,0x73,0x95,0x0b,0x08,
                 0x24,0x01,0x8a,0x85,0x01,0xb3,0xdb,0x9b,0xdf,0xfa,
                 0xa1,0xb7,0x88,0x4b,0x11,0x83,0x0d,0x3e,0xee,0x0a,
                 0x5e,0xd9,0xb7,0x1e,0x17,0x11,0x1f,0xf6,0x9d,0x8e,
                 0xbd,0x1c,0x6a,0xaf,0x05,0x87,0xa5,0xce,0x77,0x03,
                 0xf6,0xc5,0x16,0xda,0x98,0xb0,0x1c,0xad,0xb0,0xf5,
                 0xec,0xa3,0xdd,0x82,0x48,0xc6,0x10,0x56,0xc4,0xa9,
                 0x9e,0x43,0x7a,0x4e,0x93,0xf2,0x00,0x48,0x4a,0x27,
                 0x97,0x1d,0x3a,0x46,0xa5,0xee,0x13,0x17,0x66,0x5a,
                 0x0a,0xc6,0xde,0x9f,0x70,0x2e,0x12,0x02},
        .data_len = 128,
        .mac_len = 40,
        .mac = {0x03,0xfc,0x9f,0xc8,0xd4,0xf1,0x86,0xe8,0x71,0x84,
                0x75,0xc6,0xa3,0xe8,0x23,0x89,0x16,0xef,0xa8,0x28,
                0xb5,0x40,0x42,0x93,0x2e,0x87,0x2b,0xff,0x0a,0x13,
                0x62,0xa6,0x75,0x05,0x63,0x79,0x7d,0x35,0x71,0xe8},
    }, {	// 13	(91)
        .key = {0x9e,0x1c,0x51,0xd3,0x5e,0x36,0x36,0xce,0xae,0xc4,
                0x4d,0x7f,0xf4,0x27,0xca,0x5e,0x98,0xb3,0xfb,0x8c,
                0x0c,0xa7,0x73,0x64,0x09,0x63,0x67,0x52,0x1b,0xd5,
                0x58,0xe8,0x5f,0x35,0xf2,0x2e,0x8b,0xf2,0x35,0x3a,
                0x30,0xec,0xd2,0x01,0x3c,0xe3,0xd8,0x6d,0xc3,0x2f,
                0x8f,0xb0,0xff,0xb2,0xa4},
        .key_len = 55,
        .data = {0xd7,0x77,0x21,0xf0,0xca,0x5a,0x83,0xee,0xa7,0x82,
                 0x10,0x73,0xd4,0x09,0x90,0xfb,0x6c,0xf0,0x0b,0x36,
                 0xf0,0x06,0x27,0x0b,0x39,0x0b,0xb1,0xde,0xb1,0x16,
                 0x79,0x0e,0xc3,0x34,0x63,0xc2,0x90,0x52,0xbb,0xe6,
                 0xe4,0x5d,0xc9,0x70,0x68,0xa7,0xa5,0xe8,0x19,0x8d,
                 0x4d,0x27,0xf8,0x57,0xf5,0x5f,0x03,0x5f,0x9e,0x5b,
                 0x65,0x76,0xea,0x08,0xea,0x83,0x2f,0x35,0xb5,0x6d,
                 0xca,0x97,0x35,0x3b,0xb6,0x10,0x55,0x7a,0x5d,0x30,
                 0xf3,0xdf,0xad,0xfd,0x94,0x2e,0x6f,0xef,0x56,0x5a,
                 0xd4,0x3f,0x26,0xee,0x51,0x62,0x32,0xab,0xad,0xd0,
                 0xa1,0x73,0x59,0x09,0x82,0x66,0xff,0xda,0x03,0x4a,
                 0x5d,0xdc,0xe4,0x30,0x54,0x3f,0x2b,0x54,0x3c,0xc5,
                 0x18,0x46,0x7b,0x11,0x5b,0x47,0x56,0x22},
        .data_len = 128,
        .mac_len = 40,
        .mac = {0x63,0x21,0xb0,0x2d,0x91,0x59,0x10,0x09,0x91,0x3f,
                0x81,0x70,0xfb,0x0b,0x5e,0xa6,0x79,0x3e,0xe8,0xbb,
                0x32,0xa3,0xe6,0x2f,0xbe,0x11,0xcb,0xee,0x2d,0x06,
                0x7d,0xba,0xe2,0x61,0x14,0x20,0xa0,0x3f,0xb0,0x03},
    }, {	// 14	(105)
        .key = {0xbf,0xe6,0xbb,0x4c,0x9b,0x17,0x1b,0x93,0xd2,0x8e,
                0x9f,0x8f,0x86,0xb8,0x8d,0xbe,0x50,0x9c,0x66,0xee,
                0xd4,0x18,0x18,0xa1,0x98,0x6d,0x75,0xb6,0x16,0xfe,
                0xe4,0x46,0x0f,0x54,0x56,0xcd,0x23,0x66,0x7c,0x8a,
                0x9f,0x17,0x38,0x28,0x96,0x01,0x51,0x9d,0x33,0x71,
                0x6a,0x53,0x4d,0xb2,0x35},
        .key_len = 55,
        .data = {0x5b,0x7a,0x07,0x8f,0x98,0x0b,0xb8,0x91,0x97,0x43,
                 0xbb,0xce,0x52,0xfd,0x0b,0xa3,0xc2,0x20,0x83,0xd2,
                 0xb0,0x25,0x4e,0x28,0xc8,0xd3,0xa0,0x5d,0xef,0x4d,
                 0xa3,0x3b,0xd6,0x4f,0xb5,0x02,0xcf,0xb5,0xd0,0x0c,
                 0xe0,0x3d,0x49,0xad,0x16,0x8d,0xbe,0x5d,0x1c,0x78,
                 0x4a,0x19,0x0c,0x7d,0xfa,0x06,0x85,0x90,0x85,0x58,
                 0xfe,0x1e,0x37,0x72,0x5a,0x4b,0x2f,0x4e,0xbc,0x7e,
                 0xca,0x20,0x9c,0x1f,0x5f,0x36,0x1b,0x9f,0x2d,0x23,
                 0x93,0xb9,0x91,0x1c,0x73,0xf8,0x7d,0xa2,0x4a,0x7a,
                 0x25,0x62,0x21,0xf3,0xfb,0x59,0x0e,0xf4,0xde,0x3b,
                 0x06,0x6e,0x8e,0x16,0xf3,0x72,0x64,0x32,0x06,0x3a,
                 0x40,0x3d,0x4f,0x6d,0xc2,0xa4,0x8b,0x9f,0xbd,0x44,
                 0x3d,0x17,0xe8,0x42,0x00,0xd6,0xd7,0x37},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0xe8,0x2e,0xeb,0x7f,0x4b,0x74,0x15,0xa4,0xc9,0x5d,
                0xc8,0x2c,0x46,0xbb,0x59,0x71,0x5f,0xda,0x4e,0x0b,
                0xda,0xf6,0x4a,0x7f,0xb3,0xaf,0x3c,0x70,0x58,0xec,
                0x7d,0x2a,0x17,0x2b,0x82,0x93,0x05,0x7b,0x72,0xf9,
                0x66,0x44,0x54,0xe7,0xde,0xe1,0x1d,0x95},
    }, {	// 15	(106)
        .key = {0x4c,0xf5,0x4e,0xb8,0xcf,0x7b,0xd4,0x21,0xdd,0xb0,
                0x58,0x6a,0xc4,0xfa,0xb9,0xc4,0x78,0xcd,0xae,0xdd,
                0x89,0xcc,0x5a,0x19,0x53,0x32,0x21,0x1f,0x75,0x71,
                0xb9,0x98,0x84,0x19,0x84,0x33,0x00,0xfa,0x1d,0xed,
                0x86,0x8d,0x31,0x8f,0x48,0x90,0x90,0x78,0xbb,0xf1,
                0x83,0x9c,0x8f,0xed,0x61},
        .key_len = 55,
        .data = {0xd2,0x2f,0x19,0x4a,0x1a,0xf3,0x3c,0xd8,0xcd,0xff,
                 0xe9,0x96,0x7f,0x67,0x7a,0xcb,0x68,0x50,0x0d,0x6c,
                 0xbb,0xf7,0x7a,0x3f,0x34,0xf5,0x88,0x40,0xf0,0xc1,
                 0x60,0x44,0x82,0x76,0x41,0xdc,0x43,0xd6,0x76,0x7c,
                 0xe9,0x8f,0x85,0xdd,0x5c,0xbe,0xaa,0x9f,0xc5,0xb2,
                 0x83,0x33,0xe7,0xf2,0x0d,0xf8,0xb2,0x81,0xcf,0xa4,
                 0x15,0x60,0x55,0xe6,0x15,0x55,0xe0,0x4a,0x1c,0xeb,
                 0x5c,0x5c,0x93,0xba,0x92,0x10,0xb2,0xe8,0x9f,0x61,
                 0x97,0xf0,0xa5,0x39,0x96,0xa2,0xc0,0x91,0xd1,0x6c,
                 0x3c,0xd9,0x08,0xd7,0x05,0x9a,0xb2,0x54,0x5e,0x5a,
                 0x4c,0x39,0xd6,0xc0,0xf1,0x07,0x78,0xf8,0x2b,0xee,
                 0x43,0x59,0x09,0x93,0xda,0x45,0x71,0x10,0x7c,0x51,
                 0xb8,0x3c,0x35,0xa6,0x70,0x2e,0x56,0xa8},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x83,0x0b,0x4a,0x79,0x8f,0x85,0xc4,0x48,0xb3,0xd5,
                0x4a,0xbf,0xee,0x61,0xb3,0x76,0x59,0x7f,0x65,0x66,
                0x6d,0x83,0xa2,0x10,0x52,0xcb,0x3f,0x44,0x66,0xf4,
                0x47,0x47,0x43,0x16,0x07,0xbc,0x65,0x9c,0x91,0xcb,
                0x52,0x03,0x08,0xfb,0xf4,0xfc,0xdb,0x58},
    }, {	// 16	(120)
        .key = {0xc6,0x3b,0xff,0x38,0x2d,0xe2,0xbd,0x2d,0x07,0x65,
                0x38,0xea,0x88,0xff,0x54,0x13,0xd1,0x19,0x69,0xf5,
                0x0a,0x0d,0xf1,0x6d,0xb1,0x2f,0x84,0x05,0x31,0x0e,
                0x07,0x61,0xb7,0xf7,0x20,0xda,0x41,0xbb,0xec,0x68,
                0xf8,0xb2,0xf5,0xc5,0xbf,0x00,0x5e,0xcf,0x0c,0x17,
                0x61,0x2f,0xf6,0x7e,0xfc,0x38,0x90,0xd0,0xe6,0x11,
                0x76,0x07,0xc8,0x17,0xa5,0xfa,0xaa,0x7d,0x90,0x25,
                0xab,0x35,0x70,0xa9,0xf6,0x14,0xdb,0x93,0xf1,0x31,
                0x98,0x61,0xb8,0x8e,0xb2,0xc3,0xc9,0xfa,0xcb,0x9e,
                0x01,0x35,0xb3,0x56,0xc7,0x56,0x39,0x4d,0x87,0x6a,
                0x41,0xa7,0x62,0x5e,0x17,0x51,0x23,0x1f,0x03,0x41,
                0x75,0xff,0x1e,0xff,0x54,0x5b,0x63,0x64,0xc2,0x7a,
                0x09,0xa1,0xbb,0xb9,0x11,0x84,0x6a,0xe5},
        .key_len = 128,
        .data = {0x99,0x2a,0x5b,0x8a,0x63,0x9a,0xe2,0xb2,0xf7,0xfc,
                 0x9e,0x13,0x53,0xa7,0x9e,0x52,0x1c,0xfd,0xc9,0x89,
                 0x90,0x93,0x72,0x90,0xbc,0x93,0x2c,0x7b,0xef,0x5e,
                 0xdf,0x63,0x6e,0x75,0x1b,0x6a,0x69,0x99,0xad,0xf9,
                 0x2e,0x31,0x70,0x4c,0x9d,0xed,0x66,0x31,0xdc,0xa9,
                 0x07,0x0c,0x4c,0x94,0xd9,0x1f,0xbb,0x91,0x41,0x08,
                 0xdb,0xdd,0x99,0x8b,0xf2,0xf2,0x82,0x92,0xd4,0xac,
                 0x7c,0x72,0x0f,0xab,0xb4,0x70,0x65,0xf8,0x1c,0x84,
                 0x7f,0xeb,0xc1,0x5d,0xdf,0x4c,0x5a,0xa4,0x17,0xb8,
                 0x1c,0x85,0x38,0x46,0xd6,0x6c,0x8e,0x6b,0x39,0x0c,
                 0x8a,0x1b,0x77,0xa6,0x00,0x31,0x11,0x88,0x93,0x11,
                 0xe9,0xd4,0x6d,0x8c,0x9f,0x82,0x33,0x04,0x1a,0xa8,
                 0x37,0xd0,0x65,0xf9,0xf0,0xe1,0xbd,0x8c},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x44,0x8a,0xbc,0xe3,0xc3,0x8e,0x7f,0x10,0x90,0x73,
                0xf1,0x51,0x34,0x55,0x21,0x4d,0x03,0x61,0xca,0x75,
                0x9c,0x77,0x54,0x52},
    }, {	// 17	(121)
        .key = {0x43,0x59,0xb0,0x93,0x28,0xdd,0xce,0x80,0xcc,0xf1,
                0xd3,0xec,0x54,0x37,0xab,0xa6,0xa1,0x1a,0xe7,0x89,
                0x77,0x5f,0x04,0xac,0xde,0xfc,0xf0,0xd8,0xc8,0x2e,
                0xca,0x3f,0xf5,0xc6,0xe9,0x6a,0x14,0xc3,0x21,0x74,
                0x2b,0x26,0x41,0x76,0x38,0x02,0xe0,0x42,0x19,0xd3,
                0x5a,0x54,0xa9,0x10,0x15,0x05,0x2c,0x04,0x09,0x02,
                0xed,0xd9,0x7f,0xfd,0x25,0xf6,0x18,0xa2,0x1f,0x8b,
                0x12,0xcd,0x9a,0x69,0xc7,0xfa,0x6f,0x18,0x76,0xfd,
                0x73,0x23,0x46,0xf3,0x9f,0xb7,0x88,0x78,0x6e,0x6c,
                0x3d,0x1a,0x87,0x63,0xd8,0x0e,0x9c,0x91,0x45,0x22,
                0x92,0x5a,0x29,0xf3,0xe2,0x62,0x6c,0x60,0x3f,0xa0,
                0xf5,0x3e,0x79,0xb4,0xb4,0x4d,0xa1,0x7a,0xe6,0x6b,
                0x6e,0xdf,0x94,0x08,0xdf,0xf3,0x5d,0xda},
        .key_len = 128,
        .data = {0xa5,0x71,0x95,0xbf,0xf0,0x00,0xd7,0x68,0xe3,0x9c,
                 0xe6,0xda,0xf6,0x6e,0x91,0xb3,0x1a,0x30,0xfe,0x94,
                 0x42,0x9d,0x4c,0x2f,0x22,0x25,0x76,0xa1,0x36,0xe6,
                 0x7b,0x03,0x07,0xd8,0xbc,0x3b,0xaa,0x47,0xa5,0x18,
                 0x89,0x87,0x8f,0x9f,0x66,0xe3,0xe5,0x9f,0x9c,0xd6,
                 0x86,0x8c,0xa8,0x7e,0x6b,0x89,0xe9,0x4d,0x8a,0xc7,
                 0xa4,0x02,0xfa,0x0e,0x4b,0xd7,0x57,0x99,0xff,0xfc,
                 0x68,0x27,0x53,0x45,0xff,0x4f,0x53,0x20,0x21,0x14,
                 0xc5,0xc9,0x67,0xb9,0xae,0xc1,0xa4,0xd7,0x18,0x7c,
                 0xc8,0xac,0x13,0x59,0x05,0xb6,0xfa,0xd8,0x30,0x80,
                 0xf7,0x08,0x69,0xbd,0xc9,0x3e,0xfe,0x93,0xc5,0x0c,
                 0x8d,0x39,0x1b,0x71,0x69,0xc4,0x5b,0x4e,0x3f,0x3e,
                 0x38,0x19,0xfa,0xf9,0x8b,0xd5,0xe3,0x22},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0xf2,0x15,0xc9,0x88,0x60,0x11,0x19,0x87,0x33,0x40,
                0xc4,0xcf,0xf6,0x06,0x3f,0xf9,0x7c,0xea,0xcb,0x3e,
                0xed,0xc4,0x0a,0xef},
    }, {	// 18	(135)
        .key = {0x14,0x16,0x13,0x40,0xbb,0xfa,0x1c,0x47,0x80,0x58,
                0x37,0x96,0xf7,0x07,0x31,0xd2,0x02,0xee,0xa4,0x42,
                0x97,0xbc,0xd4,0x28,0xc3,0xd9,0xd7,0x52,0xef,0x9c,
                0xf9,0xec,0x63,0xbe,0x5e,0x98,0x08,0x0e,0xe9,0xc1,
                0x72,0x67,0x5d,0x2b,0x1d,0xdf,0xc2,0xcf,0xf7,0x42,
                0x0b,0xe7,0x1f,0xbf,0xf5,0x45,0xed,0xe0,0x32,0xe8,
                0x32,0xc0,0xc7,0xd1,0xc1,0x78,0xb3,0x13,0x2e,0xda,
                0xd1,0x2a,0xd5,0x62,0xff,0x8d,0x1e,0x69,0x80,0x87,
                0x00,0x9c,0x9f,0x42,0xc4,0xad,0x95,0x25,0x0c,0x48,
                0xad,0x5f,0x13,0x49,0xa6,0xc4,0x36,0x2c,0x59,0xd9,
                0xb4,0xc4,0x9c,0x2a,0xb2,0x30,0x65,0xe4,0x82,0x0c,
                0x33,0x9f,0x24,0xe4,0xa9,0x7c,0x0d,0xee,0x7c,0x70,
                0x28,0xf8,0x90,0xdf,0x1b,0x9f,0x5e,0x6d},
        .key_len = 128,
        .data = {0x6f,0x24,0xfa,0x08,0xde,0x52,0x44,0xf3,0x01,0x73,
                 0x80,0x9f,0x1a,0x14,0x1a,0x9e,0x00,0xff,0xc2,0xa9,
                 0x14,0x5f,0x07,0xe6,0x77,0x26,0x27,0x6b,0x7a,0xac,
                 0x25,0xfe,0x56,0x98,0x1d,0x1e,0x1e,0x04,0xd5,0x48,
                 0xf1,0xdc,0x94,0x73,0x74,0x87,0x37,0xdd,0x7f,0xca,
                 0x81,0x09,0x17,0xe9,0xb3,0x08,0x9d,0x0f,0x5c,0xf9,
                 0x44,0xef,0x73,0xcc,0xc9,0xac,0xa3,0x4b,0x5e,0xf6,
                 0xe6,0x5a,0xe7,0x77,0x55,0x7d,0x68,0x6d,0x3f,0x9c,
                 0xbe,0x98,0x78,0x03,0x8e,0x56,0xf3,0xad,0x7c,0x0d,
                 0x93,0xc2,0x9d,0xc9,0x3f,0x5e,0x2e,0x26,0x35,0x94,
                 0x86,0x71,0xa0,0xb3,0x49,0x0a,0x6c,0xc7,0xdf,0x0c,
                 0x59,0x63,0x24,0x30,0x4e,0x9e,0x61,0xef,0xf1,0x5c,
                 0x7c,0xe7,0x74,0xcf,0x6b,0x80,0xb1,0x3d},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x64,0x2a,0xba,0xb1,0x3c,0x76,0x78,0xba,0xa4,0x46,
                0x0e,0x4b,0x6d,0xec,0x2f,0x4e,0xa8,0x41,0x23,0x99,
                0x95,0x76,0x25,0x0c,0x98,0x00,0x6a,0x8a,0x0a,0x06,
                0xeb,0x57},
    }, {	// 19	(136)
        .key = {0x8b,0xcd,0x18,0x2e,0x78,0xa9,0xdc,0x1d,0x38,0xff,
                0x52,0x95,0x86,0x32,0xa2,0x2b,0x73,0x9e,0x06,0x41,
                0xaa,0xcf,0x2e,0xd8,0xf8,0xf1,0xe4,0xa5,0x0c,0x88,
                0xec,0x66,0x7b,0x62,0x2e,0x76,0x07,0xc9,0x17,0x9f,
                0x20,0xfd,0x3c,0x30,0xab,0xe1,0x40,0x50,0x03,0xf4,
                0xf8,0x92,0x3d,0x83,0xce,0xca,0xb1,0x1d,0x63,0x1e,
                0xb5,0x48,0x79,0x60,0xac,0x72,0x0f,0x9b,0x40,0x2a,
                0xcd,0xeb,0xeb,0x90,0xa3,0x92,0xbc,0x0a,0xae,0x49,
                0x58,0x39,0x5b,0xd4,0x3f,0x2c,0xce,0xd9,0x50,0xd3,
                0x85,0xf2,0x90,0xb6,0x38,0x01,0x27,0xe6,0x04,0xc4,
                0xab,0x34,0xc9,0xa9,0xa1,0xa2,0xd1,0xe3,0x41,0x17,
                0xb2,0xbd,0x7a,0x57,0x75,0x2e,0x36,0x31,0xf7,0xae,
                0xdf,0xfb,0x90,0x49,0x22,0x3b,0xf3,0xf2},
        .key_len = 128,
        .data =	{0x80,0x3b,0x54,0xa0,0xa9,0xb4,0x4c,0xc9,0x35,0x34,
                 0x9e,0x9d,0x99,0xaf,0x7c,0x5a,0xa6,0x00,0x64,0x4e,
                 0xff,0x8b,0x3c,0x9d,0xd0,0x21,0xa0,0x3f,0xbd,0x24,
                 0x7b,0x48,0x19,0xeb,0xd4,0x6c,0x59,0x67,0xeb,0xc2,
                 0xc8,0x07,0x85,0xc8,0x7c,0xda,0x84,0xa8,0x88,0xf4,
                 0xba,0xb9,0x73,0x12,0xff,0x49,0xe9,0x81,0x81,0x9a,
                 0xb1,0x3b,0x5c,0x2a,0xdf,0x54,0x6b,0x37,0x4b,0x94,
                 0x5d,0x83,0x41,0x66,0x0b,0x55,0x7a,0xf0,0x08,0xc0,
                 0x4b,0x84,0x7a,0x27,0x1d,0x37,0x29,0x01,0x1d,0xcf,
                 0xd6,0xda,0x35,0xe3,0xce,0x9a,0x3a,0x3d,0xbf,0x0a,
                 0x67,0x83,0xc9,0x94,0x0a,0x17,0xd8,0x4b,0x7d,0x3b,
                 0x32,0x2b,0x58,0x79,0x4c,0xa1,0xe5,0x42,0xe2,0x4e,
                 0xd4,0xd5,0x46,0x08,0x30,0x62,0xf9,0x21},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0xbc,0x08,0x20,0x56,0x94,0xfe,0x5b,0xcd,0x78,0x6c,
                0x78,0x5a,0x07,0x31,0xb7,0xb7,0x37,0xa6,0x7b,0xea,
                0x10,0x52,0x8b,0xfa,0x33,0x44,0x8a,0x7e,0xa9,0x3d,
                0xff,0x0b},
    }, {	// 20	(150)
        .key = {0x66,0x58,0x12,0xa5,0x54,0xfe,0x08,0x43,0x39,0x55,
                0x3e,0x3c,0xc2,0x9d,0xfa,0x89,0x96,0x36,0x2e,0x29,
                0x43,0xc4,0x05,0x68,0x78,0x8b,0xbc,0x61,0x76,0x1b,
                0xb3,0xc2,0xc1,0x32,0xc4,0xcf,0x1b,0xde,0xd3,0xaa,
                0xb2,0xe2,0xa6,0xd1,0x99,0x5b,0xf7,0xe8,0x75,0xa3,
                0xc8,0xb9,0x79,0x76,0xf7,0x79,0x94,0x53,0x12,0x4c,
                0xe8,0x25,0x6c,0x0c,0x7f,0x23,0x71,0x46,0x39,0xf5,
                0x36,0x86,0x09,0x18,0x55,0xd5,0x61,0x83,0xf7,0x7f,
                0xeb,0x8b,0x32,0x1a,0x7a,0x04,0x96,0xc3,0x40,0xa9,
                0x02,0xab,0x41,0xbe,0xbc,0xd3,0x07,0xf3,0xc1,0x13,
                0x98,0x8c,0x5a,0x61,0xa5,0xbf,0xf0,0x50,0x04,0x5d,
                0x21,0xd7,0x61,0xb5,0x14,0x54,0x30,0xb6,0x09,0xd0,
                0xe5,0x53,0x34,0x85,0x68,0x2c,0xcb,0x9d},
        .key_len = 128,
        .data = {0x8d,0xf0,0xa3,0x67,0x32,0x78,0xe2,0x60,0x64,0xcb,
                 0x6f,0x68,0x80,0x22,0xac,0x2a,0x0f,0x2e,0x99,0x73,
                 0x41,0xb6,0xed,0xf2,0x97,0x81,0x66,0x3c,0xdf,0x76,
                 0x5d,0x12,0x65,0x02,0x9d,0xe7,0x68,0xba,0x75,0x9d,
                 0xcb,0x42,0x0c,0x90,0x0d,0x6d,0xf5,0xd5,0x7b,0xa5,
                 0x03,0xc4,0xa4,0x8e,0x5f,0xb3,0x0e,0xe6,0xd7,0x05,
                 0x27,0xb0,0x79,0x64,0x7e,0x91,0x61,0x4a,0x33,0x7a,
                 0xcf,0xc6,0xad,0x87,0x7d,0x7d,0x8a,0x27,0x2f,0xef,
                 0xdc,0x7e,0x8b,0xfb,0x92,0x07,0x2b,0xa5,0x34,0x7e,
                 0xf1,0x18,0xd4,0xfd,0x9a,0xe7,0x41,0x65,0x96,0x98,
                 0x7d,0xff,0x17,0x63,0x71,0x63,0x69,0x37,0xe0,0x99,
                 0x81,0xfe,0x9a,0x7f,0xd8,0x22,0xf2,0x6a,0x7a,0x50,
                 0x71,0x10,0x59,0x7c,0xcc,0xa6,0xe8,0x25},
        .data_len = 128,
        .mac_len = 40,
        .mac = {0x20,0x31,0x30,0x5f,0x71,0xc6,0x9a,0xe3,0xea,0x4d,
                0x55,0x47,0x27,0xf8,0x3c,0x7d,0x9c,0x48,0x57,0x65,
                0x96,0x8b,0x19,0x79,0x6b,0xb0,0x18,0x7c,0xe0,0x4a,
                0xa7,0x41,0x03,0x02,0xe2,0xfb,0x09,0xcf,0x4b,0x07},
    }, {	// 21	(151)
        .key = {0x71,0x08,0xc0,0xd1,0xe1,0x15,0xbf,0x9d,0x61,0x31,
                0xdc,0x37,0x05,0x2b,0x76,0x0b,0xdd,0xe7,0xb4,0x3a,
                0xd5,0xb1,0x82,0x8f,0xda,0xd1,0xd6,0xb6,0x3e,0x60,
                0x38,0xa8,0x5e,0x5a,0x81,0x6a,0x82,0xf4,0xe3,0xf7,
                0x04,0x2f,0x29,0x7b,0xb5,0xad,0x40,0xc1,0x7c,0xc3,
                0xf7,0xef,0x40,0xd1,0x03,0x71,0x08,0xce,0x46,0x33,
                0x61,0x27,0x51,0x13,0x01,0xca,0x27,0x96,0xa9,0x7d,
                0x43,0xd9,0x50,0x75,0xdd,0xcb,0x7d,0x24,0x6a,0x9a,
                0xf5,0x52,0x62,0x6b,0x96,0x6f,0x45,0x4e,0x83,0x28,
                0xe0,0x71,0x81,0x38,0xcd,0x94,0xa1,0x81,0x39,0xbc,
                0x20,0x5b,0xe9,0x2c,0x8b,0x2c,0x7f,0x91,0x26,0x39,
                0xfa,0x7d,0x8f,0xbb,0x7d,0x16,0x9f,0x36,0x51,0x10,
                0x91,0x06,0x2c,0xd8,0x66,0x3a,0xb4,0x12},
        .key_len = 128,
        .data = {0x2c,0x72,0x3d,0x78,0xa6,0x6c,0x53,0x64,0x94,0xcd,
                 0xf3,0x7d,0xa0,0xe4,0x3e,0x2e,0x17,0x1a,0x09,0xc7,
                 0x9d,0xd5,0x32,0x7e,0x20,0x9c,0x34,0xb4,0x0a,0x7b,
                 0xfa,0x79,0xbb,0xa9,0xf1,0x51,0x36,0x68,0x61,0x38,
                 0x1a,0x2d,0xfd,0xe5,0xd5,0x01,0xb9,0x4c,0x14,0x27,
                 0xdb,0x66,0x7d,0xff,0x55,0x34,0xa1,0x2a,0x52,0x02,
                 0x2c,0xef,0x0f,0xe9,0x5f,0xdd,0xec,0x97,0xc1,0xb2,
                 0xc3,0x53,0x11,0x7c,0x78,0x3b,0x7e,0xfe,0xd1,0xd0,
                 0x1b,0x55,0x88,0xb5,0x8d,0xdc,0x9f,0xb4,0x06,0x4c,
                 0xf4,0x02,0x78,0x28,0x15,0xc4,0x02,0x58,0x55,0xd1,
                 0xaf,0x13,0x20,0xba,0x5f,0x03,0x8b,0x08,0x05,0xa4,
                 0x2f,0xc4,0x13,0xee,0x38,0x3d,0x33,0x33,0xb9,0x05,
                 0x38,0x4a,0x43,0x3d,0x54,0xed,0xb5,0x12},
        .data_len = 128,
        .mac_len = 40,
        .mac = {0x82,0x6a,0x2a,0x11,0x38,0x0c,0x26,0x08,0x73,0x66,
                0x38,0x15,0xff,0x5e,0x02,0x20,0x1a,0x17,0xde,0xdc,
                0xa1,0xb2,0x0c,0x61,0x3d,0x0d,0xcc,0x01,0x90,0x95,
                0xb4,0x44,0xfa,0x0e,0x96,0xc8,0xb2,0xdf,0x54,0x33},
    }, {	// 22	(165)
        .key = {0xb6,0x19,0xd9,0xd0,0x74,0x61,0xc1,0x1b,0xc9,0xfb,
                0x66,0x11,0x7d,0x61,0xed,0x90,0x00,0x13,0x66,0xbb,
                0xff,0xdb,0xff,0x58,0x35,0x56,0x77,0x75,0x84,0xb0,
                0xd6,0x52,0x44,0xaf,0x5c,0x7b,0xdb,0xf3,0xb7,0x35,
                0x8d,0x7c,0x79,0x1b,0x96,0x6c,0xc8,0x09,0x76,0x0e,
                0x57,0x39,0x8d,0x18,0x96,0xac,0xe7,0x2d,0x26,0xcc,
                0x59,0xa6,0x90,0x4f,0xcd,0x92,0x36,0x5e,0xda,0xfb,
                0x8a,0xf7,0x98,0x6c,0x7d,0x90,0xb2,0xaf,0x3b,0xfd,
                0xbc,0xdb,0x15,0x93,0xc7,0x8f,0xbe,0x8e,0x33,0x78,
                0xbb,0xb0,0xc5,0x19,0x15,0x2b,0xf9,0xcb,0x51,0xc1,
                0x9a,0x02,0xa1,0x2a,0x8f,0xd3,0x5c,0xb6,0xf8,0xb3,
                0xac,0x33,0x7a,0x82,0x87,0x11,0xd6,0xc8,0xe0,0xc4,
                0xc1,0x3e,0x1e,0x6a,0xf0,0x90,0xcd,0xae},
        .key_len = 128,
        .data = {0x5a,0x81,0xe7,0x11,0xad,0xfe,0x50,0x77,0xdd,0x8c,
                 0x8b,0x57,0xc9,0x5e,0x8e,0x1f,0x3d,0xe3,0x9f,0x4f,
                 0xc4,0x48,0xc5,0x23,0xbd,0x3e,0x7c,0x72,0xb1,0xfd,
                 0xac,0xd6,0xe4,0x89,0xdc,0x0d,0x2a,0x34,0xa3,0x9f,
                 0xfc,0x64,0x60,0xc1,0xcb,0x96,0x2b,0x7a,0x94,0xa3,
                 0x0c,0x04,0xb5,0x42,0x6a,0x75,0xff,0xcf,0xc6,0x9f,
                 0x0c,0x4b,0xa9,0x34,0xd3,0xa3,0xda,0x2e,0x79,0x35,
                 0xd5,0x6d,0x6b,0x90,0x79,0xa2,0xa9,0x7b,0x01,0x6d,
                 0x65,0x3a,0x35,0xc2,0xcc,0x0c,0xe1,0x91,0x24,0xf8,
                 0x87,0xa6,0x17,0xc9,0x51,0xce,0x4e,0x58,0x49,0x3b,
                 0x42,0x09,0xcc,0x29,0x4f,0x98,0x3c,0xc2,0x0b,0x16,
                 0xf6,0x3f,0xd5,0x2e,0x84,0x51,0xb1,0xad,0x13,0xbf,
                 0x53,0x42,0x27,0x50,0x79,0x81,0x8d,0xeb},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x8b,0x42,0x58,0xbe,0x4c,0x09,0x4a,0xa4,0x05,0x6f,
                0x33,0x2e,0xde,0x8c,0x73,0x37,0x72,0x66,0x4b,0x08,
                0x8b,0xa2,0x2e,0xf8,0xca,0xae,0x7c,0xef,0xd7,0x7e,
                0xce,0xb3,0x5e,0x83,0xaf,0x8d,0x9c,0x12,0x83,0xcb,
                0xbf,0xfe,0x4a,0x37,0x2b,0x69,0x9c,0x21},
    }, {	// 23	(166)
        .key = {0xe4,0x88,0x25,0xa5,0x50,0x3a,0x6a,0xfe,0x0b,0xf9,
                0xa2,0x40,0xc6,0x7f,0x27,0xac,0xd4,0xa8,0xf6,0x99,
                0x38,0x34,0x64,0x5e,0x03,0xc8,0x0c,0x72,0xdd,0x37,
                0x0c,0xd2,0xe1,0x00,0x71,0xa3,0xae,0x18,0xef,0x19,
                0xba,0xe9,0xd6,0x97,0xea,0x9a,0x41,0x18,0x60,0x91,
                0x90,0xcd,0x95,0x36,0x19,0x07,0xa7,0xfa,0x1b,0x58,
                0xf4,0x99,0xf3,0xf5,0xe7,0x9b,0x93,0x5f,0x12,0x21,
                0x2f,0x43,0x7d,0xde,0x39,0x9e,0x3e,0x64,0x90,0x24,
                0x4a,0xa1,0xf5,0xe3,0x8b,0xa9,0xbe,0x24,0x33,0xb6,
                0xce,0x92,0x4f,0x6c,0xc4,0x9e,0x9f,0x62,0x73,0x21,
                0xa5,0xdf,0x93,0x43,0xfc,0xe1,0xb5,0x9d,0xeb,0x64,
                0x7d,0x9a,0x3a,0xe0,0x0b,0x23,0x44,0x14,0xba,0x7b,
                0x4e,0x02,0x0d,0x67,0x17,0x3b,0xe6,0x93},
        .key_len = 128,
        .data = {0x85,0x61,0x86,0x5a,0xc2,0xce,0x12,0x83,0x27,0x46,
                 0xf8,0x25,0x84,0xa4,0xb9,0x8e,0x7f,0x4c,0x3a,0xe2,
                 0x41,0x0e,0x18,0x19,0x6f,0x4e,0x3b,0x47,0x5c,0x62,
                 0xae,0x20,0x7d,0x3c,0xad,0xbb,0x1d,0x49,0x00,0x96,
                 0x51,0x98,0x88,0xdb,0x2f,0x3f,0x18,0xe1,0x3b,0xfb,
                 0x86,0xf6,0x22,0x16,0x01,0x5c,0xab,0x8e,0xa4,0x91,
                 0xea,0x73,0x4c,0xd3,0xb7,0x91,0xa7,0xe4,0x5e,0x4f,
                 0x8e,0x0b,0x98,0xd7,0x95,0x5b,0xba,0x77,0xe0,0x37,
                 0x2d,0x47,0x38,0x16,0x1e,0x0d,0x5d,0x84,0x76,0x5d,
                 0x9e,0x6a,0x0d,0x05,0xa8,0x8e,0x1a,0xa8,0x9c,0x5d,
                 0xef,0xa8,0x64,0xe9,0xe3,0x49,0x46,0x2e,0x8f,0x14,
                 0xb9,0x99,0x3d,0x7a,0x78,0xcb,0x9d,0xba,0xd6,0x9a,
                 0xba,0x05,0x51,0x58,0x2d,0xdf,0x69,0x58},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0xec,0x78,0x0a,0x91,0x5e,0xc7,0xde,0xeb,0xa2,0xc8,
                0xc9,0xe2,0xab,0x15,0xc9,0x76,0x2a,0x3e,0xb1,0x8f,
                0xaf,0xa2,0xd4,0x8a,0x55,0x4a,0xe1,0xfe,0x6c,0x44,
                0x59,0xda,0x1a,0x54,0xe2,0xd5,0x8b,0xdf,0x06,0xfe,
                0xa0,0x74,0x00,0x98,0xee,0xbb,0xb6,0x99},
    }, {	// 24	(180)
        .key = {0xe6,0x2c,0x2c,0xe5,0xff,0x8e,0x3d,0x46,0x5d,0x34,
                0x45,0x82,0xf2,0xdd,0x56,0x6e,0x0d,0x29,0xdd,0xd6,
                0x20,0x51,0x84,0xcb,0xc6,0x14,0x56,0x3a,0x04,0xf5,
                0x24,0xc3,0xcf,0xa3,0x28,0xc7,0xcf,0xb9,0xe9,0xc3,
                0x63,0xa7,0x5e,0xdd,0xe5,0xf0,0x56,0xbd,0x2f,0x97,
                0x18,0x9e,0x5e,0xd3,0x46,0xa5,0xd6,0x0a,0x07,0x71,
                0x86,0xfd,0x64,0xf3,0x6d,0x41,0xf3,0x80,0x5c,0xee,
                0xb3,0x24,0xfe,0x9b,0x38,0x3a,0x17,0x89,0x08,0x39,
                0x09,0x1e,0x44,0xd1,0x9c,0x95,0x8b,0x37,0xfe,0xf5,
                0x1b,0xbd,0x09,0x3f,0x39,0xa5,0xc5,0xec,0xd4,0x37,
                0x2e,0x96,0x65,0x11,0x37,0xf6,0x19,0xbf,0x0e,0x0e,
                0x32,0x8b,0xa2,0xa7,0xaa,0x96,0x63,0xfe,0x1a,0x28,
                0x48,0xbb,0xdb,0x45,0x17,0x22,0x6b,0x81,0xe1,0x8d},
        .key_len = 130,
        .data = {0x7a,0xc2,0x40,0xb4,0xbd,0xe6,0x4b,0x6b,0x51,0x42,
                 0x37,0xf1,0x22,0xdd,0x77,0x3a,0x6a,0xdc,0x2f,0x2d,
                 0x83,0x04,0xa4,0x49,0xfa,0x7b,0xf2,0x8e,0xe4,0xce,
                 0xfd,0x9b,0x75,0x38,0xa9,0x14,0xce,0x22,0x4e,0x46,
                 0x17,0xe6,0x66,0x0e,0xce,0xd8,0x89,0xf6,0x5e,0x87,
                 0x9c,0xa2,0xf4,0x40,0x45,0x6b,0x8e,0xd4,0xa1,0x49,
                 0x55,0x9b,0x3a,0xf6,0x2c,0xb9,0x33,0x50,0x89,0xa0,
                 0xc6,0x00,0x83,0xd7,0xf1,0x59,0x2d,0xf9,0x6b,0x82,
                 0x23,0x00,0xdd,0xd8,0x62,0xc3,0x4f,0x8e,0x5d,0xbe,
                 0xe0,0x96,0x4d,0x12,0xbb,0xc5,0xc8,0xcc,0x46,0x32,
                 0x95,0x6a,0x9d,0xdf,0xbf,0x9c,0x3e,0x9b,0xe8,0x47,
                 0x06,0x26,0xe0,0xcf,0x5c,0x97,0x76,0x96,0xac,0xfb,
                 0x8e,0xc0,0x17,0x3f,0x74,0x3a,0x68,0x50},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x98,0x4e,0x2d,0xc5,0xdd,0x32,0x74,0x86,0x5d,0x0a,
                0x65,0x8e,0x1f,0x6d,0x2e,0xb0,0x85,0x4d,0xb7,0x5e,
                0x1e,0x02,0x93,0x7b},
    }, {	// 25	(181)
        .key = {0x4b,0xb9,0x7c,0x7c,0x53,0x36,0x87,0x16,0x93,0x8c,
                0x83,0x65,0x19,0xe2,0x0d,0x64,0x84,0xa7,0xe6,0xa1,
                0xbc,0x79,0x46,0x40,0xe0,0x46,0x73,0x42,0x80,0xb1,
                0x91,0x08,0x0d,0xb7,0xbc,0x12,0x8d,0x92,0x54,0x0d,
                0x93,0xdd,0x98,0x0d,0x6f,0xfa,0x77,0x17,0xa2,0xb1,
                0xa2,0xe2,0x95,0x19,0xe6,0xde,0x8a,0xbc,0xb9,0x76,
                0xd2,0xf6,0x13,0x92,0x40,0x9e,0x7f,0x61,0xdf,0xe8,
                0x71,0x55,0xe0,0x8a,0xda,0x6f,0xd6,0x1e,0xfa,0xbc,
                0x26,0xc8,0x75,0x20,0xb8,0x18,0xaa,0x52,0xf3,0x32,
                0x40,0x28,0xe9,0x2c,0x01,0x15,0xce,0xb7,0xec,0xf0,
                0xaf,0x02,0x85,0xf6,0x60,0xdb,0x70,0x13,0xb7,0x18,
                0x3e,0x5d,0xf7,0x35,0x87,0xe1,0x8f,0x3e,0x39,0xa3,
                0x05,0xef,0x2f,0xe0,0x2d,0x1b,0x06,0x96,0x9e,0x4a},
        .key_len = 130,
        .data = {0x4f,0xe1,0xa8,0x51,0x0a,0xb3,0x6b,0x97,0xc6,0x13,
                 0xd3,0x09,0xb5,0xb7,0xcc,0xb2,0x43,0xb3,0x28,0xe7,
                 0x02,0x9d,0x3a,0x88,0xe3,0xef,0xa0,0x82,0xac,0xac,
                 0x60,0x98,0xa6,0x47,0x37,0x64,0x96,0xc0,0x2d,0x98,
                 0xef,0x10,0x6d,0x8a,0x46,0xc7,0x54,0xf0,0x06,0xf4,
                 0xc8,0xe7,0x65,0x45,0xcb,0x3b,0x97,0x6f,0x4f,0xe2,
                 0x41,0xd0,0x4c,0xc9,0x30,0x53,0x11,0xd4,0x4b,0x95,
                 0xd8,0x20,0xc6,0x46,0x9c,0x8b,0x99,0xd1,0x2f,0x76,
                 0x31,0x3f,0x87,0xc9,0x65,0x58,0x58,0x38,0xab,0x0c,
                 0xe9,0xc5,0x8c,0x12,0x20,0x86,0x55,0xcf,0xde,0xa9,
                 0x10,0x79,0x93,0xb5,0xb2,0x7c,0x09,0x12,0x96,0x1d,
                 0x84,0xcc,0x2a,0x5d,0x0d,0x94,0xe9,0x20,0x0b,0x08,
                 0x89,0x8e,0x13,0x34,0x75,0xba,0x01,0x58},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0xf2,0x72,0x23,0x78,0xf0,0x2b,0xf8,0xd1,0x04,0xb3,
                0xb5,0x0c,0x77,0xd1,0x32,0xfa,0x35,0xac,0x86,0xe4,
                0xda,0x34,0xf1,0xc8},
    }, {	// 26	(195)
        .key = {0xf3,0xac,0x44,0x22,0xcc,0x72,0x43,0x78,0x10,0x0d,
                0x75,0x15,0xdd,0xfb,0xf3,0xfe,0x34,0x00,0x02,0xb7,
                0x97,0x6c,0x43,0xac,0xd6,0x9c,0x2a,0xcf,0x26,0xc3,
                0xb1,0x81,0x73,0xeb,0x4e,0xb6,0xf7,0x36,0x22,0x54,
                0x0c,0x6a,0x73,0xdd,0x3e,0xac,0x5c,0x4e,0xa5,0x8c,
                0xc3,0x47,0x72,0x42,0x8c,0x6b,0xc7,0x37,0x0c,0x0a,
                0xcc,0xc8,0xc1,0xfe,0xff,0x46,0x40,0xd2,0xcb,0x41,
                0x6e,0x2a,0x5d,0x06,0xf3,0x5e,0xb3,0x66,0xec,0x69,
                0xf5,0xb9,0xe0,0x02,0x09,0x23,0xf6,0x08,0x62,0x16,
                0x65,0x23,0x18,0x18,0x2b,0xa9,0x3e,0xc7,0x02,0xbe,
                0x70,0x1a,0x90,0xc0,0xab,0xe9,0xde,0xe2,0x61,0xb0,
                0x0b,0x16,0xcd,0x90,0x42,0x31,0x85,0x96,0xe9,0x49,
                0x4e,0x40,0x1b,0x62,0x33,0x3d,0x59,0x4a,0xd9,0x75},
        .key_len = 130,
        .data = {0x04,0xa3,0xc1,0xe8,0x9e,0xeb,0xe7,0xb9,0x9e,0xd3,
                 0xbc,0xda,0xa3,0xca,0xfe,0xd8,0x95,0x6e,0xe8,0xda,
                 0x93,0xac,0xbf,0xde,0x2d,0x29,0xa8,0x45,0xd4,0xe1,
                 0xbc,0x92,0x8e,0x0f,0x5e,0x6f,0xef,0x4c,0xcc,0xf1,
                 0x44,0xfa,0xf5,0x1c,0x11,0xe3,0x8b,0xaa,0xbe,0x1e,
                 0x58,0x08,0x8d,0x33,0xd5,0xa2,0xcf,0x7e,0xf9,0x60,
                 0x58,0xd9,0x4f,0x70,0x30,0x75,0x4b,0x47,0x8b,0x09,
                 0xde,0xe2,0xfb,0x2f,0x48,0x52,0xe5,0x0a,0x2e,0x77,
                 0x32,0x2d,0xec,0x0c,0x46,0xb8,0x2c,0xe3,0x36,0xc4,
                 0xb8,0x72,0x35,0x02,0x8c,0x8f,0x50,0x9e,0x30,0x78,
                 0x5c,0x6a,0x44,0x16,0x2c,0x38,0x5c,0x83,0x07,0x87,
                 0x0f,0xc9,0x58,0x63,0x4b,0xed,0xa8,0x86,0xeb,0x2c,
                 0xca,0xc3,0x8c,0x84,0x55,0xa5,0x9c,0x76},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x07,0xf7,0x3d,0x06,0x6e,0xee,0x17,0x80,0xfe,0x94,
                0x88,0xca,0x2f,0x02,0x4e,0xd5,0x66,0x00,0x92,0x0f,
                0x6e,0x0c,0x72,0x86,0x40,0xd1,0xf6,0xb5,0x3b,0x24,
                0x00,0x2c},
    }, {	// 27	(196)
        .key = {0x6f,0x67,0xac,0xc5,0x6e,0x89,0x29,0xe4,0x91,0xdf,
                0x25,0x2d,0x3c,0x8d,0x49,0x7a,0x6f,0xe7,0xa6,0x62,
                0xf6,0xd6,0x91,0xbb,0xdf,0x7d,0xb1,0x5d,0x31,0x1c,
                0x56,0x29,0xdb,0x34,0x0c,0x4a,0xeb,0xec,0x71,0xbb,
                0x00,0xb3,0x43,0x09,0x02,0x27,0xbd,0x5b,0x10,0x35,
                0x24,0xaf,0xba,0x82,0x9d,0x66,0x71,0x0a,0x41,0x03,
                0x3b,0x08,0x73,0x30,0xac,0x15,0x71,0x02,0x11,0x93,
                0x2a,0x7d,0xee,0x4c,0x50,0x5b,0xab,0x57,0xad,0x09,
                0x8a,0x3c,0xba,0xf3,0xdc,0x57,0x6e,0x01,0x75,0x82,
                0x54,0x61,0x5d,0xee,0xe0,0x88,0xc8,0x52,0x03,0x73,
                0x48,0x48,0xfd,0x33,0x42,0xf3,0x73,0xa8,0x9f,0xe6,
                0xc1,0x8d,0xc3,0x41,0x91,0xb4,0xc3,0x1d,0xad,0x93,
                0xd2,0x2b,0x11,0x00,0xfd,0x97,0x45,0x39,0x19,0x33},
        .key_len = 130,
        .data = {0x4f,0x34,0xc8,0x74,0xa9,0x09,0xe1,0xa3,0xec,0x18,
                 0x69,0x23,0x61,0x16,0x99,0x5b,0xaa,0xaf,0xba,0x7b,
                 0x02,0xbc,0x8b,0x54,0xc6,0xbc,0xe7,0x6e,0x35,0x82,
                 0xa3,0x54,0x74,0x2b,0xb6,0x33,0xd4,0x53,0x9c,0xa3,
                 0x58,0x89,0xfc,0xc5,0x72,0xff,0x88,0x8e,0x0e,0x86,
                 0x24,0x62,0xd1,0xba,0x4b,0xe5,0xa3,0x7a,0xaf,0x0e,
                 0x6b,0x9c,0x20,0x7d,0x19,0xde,0xaf,0x0e,0xea,0x1f,
                 0x13,0xae,0xe7,0xcf,0x4c,0x6d,0xb0,0xa4,0x86,0xd5,
                 0x77,0x8e,0x3f,0x7a,0x4f,0xee,0xac,0xd3,0xa7,0x03,
                 0x59,0x48,0x11,0xa4,0x11,0x8c,0x49,0x35,0xfd,0x2d,
                 0x72,0xd4,0x0f,0x6a,0xa2,0xd3,0xa2,0x44,0xa1,0x6b,
                 0x5a,0xd8,0xee,0xae,0x52,0xeb,0x03,0xbe,0x76,0xc7,
                 0xda,0x3d,0x2d,0x46,0xb0,0x04,0x3c,0x2c},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x0c,0xd3,0xc2,0xf7,0xae,0x63,0x53,0xec,0x7d,0x70,
                0xce,0x93,0x2f,0x39,0x80,0xcb,0xaf,0xb7,0x71,0x59,
                0xb2,0xfb,0x7a,0x5c,0x85,0xa1,0xcb,0xc3,0xa5,0x66,
                0xba,0x86},
    }, {	// 28	(210)
        .key = {0xe4,0x86,0x31,0x6b,0x3a,0xc5,0xec,0x10,0x0f,0x43,
                0xc0,0xea,0xbd,0xbc,0x0a,0x32,0xb3,0xb9,0xbb,0x65,
                0x80,0x58,0x0a,0x33,0x2d,0x4f,0x66,0x98,0xd0,0x2f,
                0xaf,0x49,0x5e,0xe6,0xa5,0x51,0xc1,0x88,0xa1,0xfc,
                0x2a,0x4f,0x83,0xf4,0xa9,0xe0,0xeb,0xaa,0xfb,0xcf,
                0xac,0xf7,0xa3,0x66,0x7d,0x04,0x3c,0xa2,0xf2,0x67,
                0x0d,0x7f,0xab,0x8e,0xde,0x75,0xcc,0x61,0x0a,0x43,
                0xac,0x23,0xd6,0xeb,0xd2,0x99,0x01,0xfe,0x1d,0x6f,
                0x9c,0x19,0x42,0xd7,0x1c,0xb0,0x7c,0x73,0xae,0x0a,
                0x7e,0x75,0xca,0x8d,0xad,0x05,0x0b,0x3c,0x8c,0xe3,
                0x3e,0x7e,0x77,0x4d,0xb9,0x21,0x95,0x23,0xee,0x9e,
                0x08,0x49,0x3f,0x9f,0x66,0x4d,0x14,0xbe,0x6d,0x49,
                0x2b,0x90,0xf2,0x0b,0x30,0xe2,0x1b,0x74,0x8e,0x42},
        .key_len = 130,
        .data = {0x7a,0x04,0xf8,0x51,0xd5,0x0c,0xd1,0x35,0x25,0x6e,
                 0xf0,0x44,0xed,0x74,0x0a,0xb5,0x9e,0x96,0x45,0x65,
                 0xb0,0x40,0xed,0xbe,0xf0,0xd5,0x68,0xde,0x1c,0xf3,
                 0x6c,0xf5,0xad,0xf9,0x6f,0xef,0xf4,0xc6,0x5f,0x54,
                 0x68,0xc4,0x94,0x6c,0x3f,0x26,0x03,0xa6,0x3b,0x6d,
                 0xb4,0x3a,0xc7,0x31,0x76,0x0e,0x42,0x1e,0xd1,0xd7,
                 0x9b,0x3d,0x3c,0x80,0x1e,0x74,0x90,0xcf,0x8d,0x51,
                 0xbd,0x46,0x73,0x03,0xbb,0x47,0xb5,0xa9,0xc4,0x7c,
                 0x6a,0xd0,0xc1,0x76,0xec,0x36,0x02,0x94,0x2f,0xd4,
                 0x31,0x27,0x52,0x1c,0x89,0xd3,0x74,0x80,0x43,0x39,
                 0xc9,0x33,0x51,0xd2,0xed,0x33,0x4f,0x1e,0x78,0x87,
                 0xb7,0xff,0xd2,0xc5,0x54,0x5f,0x49,0xd8,0xf9,0x19,
                 0x60,0x00,0x72,0x17,0x6a,0x1a,0xbb,0xb8},
        .data_len = 128,
        .mac_len = 40,
        .mac = {0x24,0xed,0x01,0x15,0x18,0x82,0x5f,0x9d,0x39,0xd0,
                0x6a,0x25,0x23,0x27,0x15,0x21,0xdc,0x79,0x49,0xd1,
                0x54,0xd6,0xcb,0x37,0x8b,0xe2,0x0e,0xcc,0x22,0x81,
                0xb2,0xce,0xac,0xd3,0x49,0xa6,0x1a,0x28,0x06,0xb2},
    }, {	// 29	(211)
        .key = {0x0a,0xc0,0x1a,0x06,0x05,0xad,0xf7,0xc6,0x08,0x26,
                0x4e,0xbd,0x66,0x7c,0x38,0x79,0x0e,0x36,0x36,0x3e,
                0xbd,0x6b,0x0d,0x93,0x72,0x70,0xd4,0x00,0x23,0xb4,
                0x4b,0x17,0xae,0xe7,0x6e,0xb1,0x12,0x62,0x4a,0x7a,
                0xdf,0xc3,0x10,0xb0,0xeb,0xd7,0x68,0x27,0x47,0xbe,
                0x07,0x91,0x71,0x49,0x84,0xfc,0xcb,0xa7,0x67,0x9c,
                0x4c,0x41,0x84,0xcb,0x76,0xe2,0x87,0x4e,0x88,0x1b,
                0xcf,0xda,0xf4,0xe6,0x80,0xd6,0x13,0x89,0xd3,0x63,
                0x18,0xbd,0xb1,0x9a,0x43,0x10,0x81,0x14,0x57,0x88,
                0x3e,0xb0,0x4d,0x89,0xcc,0x90,0x4a,0xf8,0x8c,0x65,
                0xcf,0xa1,0x2e,0xed,0x2f,0xd6,0xc6,0xba,0x47,0x59,
                0x22,0x34,0x69,0x7d,0x5b,0xe1,0x99,0x87,0xab,0xe4,
                0xfc,0x5a,0x7e,0xc4,0x8d,0x54,0xcc,0x6f,0x12,0x73},
        .key_len = 130,
        .data = {0xff,0x56,0x11,0xcc,0x44,0x96,0x62,0xec,0xf2,0xa0,
                 0x42,0x87,0xa8,0x28,0xec,0x04,0x00,0xee,0x6c,0x4b,
                 0x15,0x36,0x4a,0xd8,0x42,0x78,0x68,0x0d,0x2c,0x58,
                 0x2d,0xcd,0x02,0xd8,0xe3,0x46,0x03,0xcd,0x5e,0x0e,
                 0x41,0x90,0xdf,0x72,0xa5,0xf5,0x38,0x0b,0x34,0x81,
                 0x30,0x92,0x90,0xd7,0x28,0xf4,0xc2,0x74,0xff,0xa9,
                 0x36,0x9c,0x34,0x42,0x07,0x94,0x4a,0x42,0x7e,0x12,
                 0x71,0x2f,0xd5,0xf2,0x62,0xe9,0x40,0x2a,0x8b,0x3a,
                 0x2e,0x00,0x6c,0xad,0xcb,0x7b,0x41,0xa4,0xca,0x17,
                 0xe1,0xa5,0x63,0xbc,0xe6,0xf5,0x97,0xf1,0x0e,0x68,
                 0xbb,0x4e,0xe1,0x77,0x34,0x2f,0x94,0x93,0x80,0xb0,
                 0x2e,0xb9,0x76,0xd5,0xe9,0x47,0xce,0x08,0xdb,0x0e,
                 0xe3,0xc9,0xd5,0xa8,0xb8,0xa1,0x8c,0x0b},
        .data_len = 128,
        .mac_len = 40,
        .mac = {0xb9,0x69,0x36,0xda,0xc4,0x73,0x27,0x7b,0x4f,0x7f,
                0x63,0x27,0x30,0xdd,0x16,0xa8,0xfb,0xba,0x0d,0xe0,
                0xac,0xc0,0x21,0x3c,0xc4,0x4e,0xfe,0x4a,0xf6,0xf3,
                0x81,0xaf,0x6c,0x0f,0xe5,0xbe,0x56,0x3e,0x70,0x72},
    }, {	// 30	(225)
        .key = {0x74,0xf4,0x1a,0x6b,0x1c,0x4e,0x57,0x13,0x49,0x95,
                0x57,0xd6,0xf7,0xe8,0x89,0xf8,0xa8,0xce,0x2e,0x44,
                0x4e,0x82,0x61,0xfe,0x6a,0x8e,0x55,0x18,0x76,0x9b,
                0xdf,0xa8,0x81,0x88,0x34,0x9a,0x19,0xb9,0xf3,0xa2,
                0x6d,0xb2,0x66,0x75,0xb3,0xe4,0x05,0x39,0xc8,0xc6,
                0x3b,0x3a,0x16,0x28,0x6d,0xde,0xbb,0xc5,0x39,0xdb,
                0xe8,0x17,0xfb,0xa7,0x86,0x6f,0x96,0x31,0x20,0x44,
                0x71,0xce,0xfd,0xcb,0xbf,0x76,0x8c,0xc9,0x04,0x30,
                0x06,0xa6,0xd4,0xcb,0x4e,0xc2,0xde,0xcf,0x1c,0x0c,
                0x2a,0xb3,0x5a,0xd0,0x9f,0x50,0xce,0xd0,0xc8,0x96,
                0xfa,0x97,0xd8,0x7e,0x40,0x0a,0xeb,0x3f,0x4a,0x40,
                0x8e,0xc5,0xa9,0x93,0x82,0x5f,0xbc,0xf7,0xbd,0xb8,
                0xd4,0x8b,0xb2,0x08,0x95,0x6e,0xd2,0x8b,0xa0,0xd4},
        .key_len = 130,
        .data = {0x9a,0x12,0x14,0x82,0xc7,0x77,0x5a,0x8b,0x5f,0xda,
                 0xf1,0xc2,0xfb,0x7d,0xe1,0xa8,0x6e,0xf9,0x31,0xb1,
                 0xa8,0x8c,0xf2,0x3d,0xdb,0xb4,0x7f,0xc9,0xdc,0xfd,
                 0x02,0x67,0xcb,0x17,0x3a,0x6b,0xf6,0x2b,0x7c,0x68,
                 0xfb,0x6f,0xf8,0x5b,0x2d,0xf9,0x3e,0x25,0x39,0xd1,
                 0x01,0x3f,0x0a,0x49,0x1a,0xa9,0xe9,0x91,0xcf,0x23,
                 0xe9,0x86,0x56,0xa0,0x82,0xcb,0x95,0xf8,0x7c,0x1b,
                 0x2c,0xdd,0x0e,0xdd,0xb5,0x10,0x48,0xf9,0x4a,0xd4,
                 0xae,0xeb,0x48,0xa4,0x26,0x16,0x53,0x21,0x14,0x5a,
                 0x9b,0x4e,0xc3,0xe8,0x5d,0xff,0x07,0x55,0xac,0x8f,
                 0x20,0xee,0x71,0xd2,0xe2,0x4c,0xb1,0x4a,0x13,0x28,
                 0x0e,0x9e,0x15,0x70,0x91,0x47,0xc4,0x99,0xa6,0x8d,
                 0xa2,0x38,0x68,0xb2,0x32,0xcc,0x1f,0x6d},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0xb0,0xda,0x90,0xc0,0x43,0x49,0x35,0x11,0xd9,0x4f,
                0x22,0xfa,0xc3,0x5b,0x59,0x62,0x74,0x9c,0x49,0x97,
                0x2f,0xb4,0x35,0x71,0xb8,0x47,0x87,0x64,0xdf,0xfc,
                0x1c,0x25,0xe3,0xa7,0x52,0x3f,0xd4,0x05,0x33,0x8a,
                0x04,0x8d,0x38,0xdd,0x1b,0x75,0x51,0x1d},
    }, {	// 31	(226)
        .key = {0xd8,0x7f,0xb6,0xba,0x27,0x21,0x5e,0x5c,0xb6,0x5c,
                0x3b,0x5b,0x34,0xac,0x2a,0x32,0x03,0x7f,0x30,0xe1,
                0xf7,0xea,0x60,0x3d,0x5a,0x9b,0xff,0x8a,0x33,0x0f,
                0xe7,0x4b,0xc7,0x05,0x29,0x59,0x61,0x32,0xf6,0x33,
                0x4f,0x36,0xc0,0x95,0x2d,0xcf,0x9c,0x4c,0x66,0x4c,
                0xeb,0x48,0xf7,0x45,0x39,0xf3,0x76,0x8a,0x65,0xc1,
                0x53,0x59,0x02,0x08,0x5f,0xd4,0xfe,0x13,0x8a,0xb1,
                0x81,0x72,0xf1,0x34,0x18,0x93,0x18,0x5a,0x13,0x97,
                0x73,0x58,0x2c,0x5e,0x2c,0x43,0x69,0xe4,0x20,0x11,
                0x43,0xd1,0x2b,0xc0,0x07,0x4b,0xa5,0xd5,0x7d,0x0f,
                0x2c,0x08,0xc8,0xc0,0xa4,0x3e,0x8d,0x7e,0x7d,0xb7,
                0x57,0xbb,0x34,0x89,0x3a,0x4a,0x1d,0x4d,0xb7,0xb9,
                0x5f,0x18,0xe0,0xe1,0x40,0xad,0xbc,0xbb,0xa3,0xf0},
        .key_len = 130,
        .data = {0x9e,0x1a,0x5d,0x9f,0x23,0x6e,0xf9,0x3f,0x2c,0xda,
                 0x60,0x48,0x91,0x66,0xc8,0x2d,0xce,0x32,0x23,0x27,
                 0x04,0x66,0x44,0xcc,0x40,0x6b,0x42,0xe3,0x00,0x5c,
                 0x21,0x77,0xf3,0xb7,0xaf,0x2a,0x01,0x59,0xad,0xcc,
                 0x8b,0xa9,0x2f,0x2c,0xf4,0x13,0x46,0x2e,0x60,0xb8,
                 0xdb,0x1e,0xbb,0x63,0xde,0x44,0xfe,0xbf,0xa1,0xb9,
                 0xad,0xc8,0x7e,0x79,0xa4,0x80,0xc0,0xb8,0x14,0xe3,
                 0xc1,0x7a,0xc9,0x1c,0x4f,0x5e,0xae,0xf9,0x54,0xba,
                 0x92,0x9d,0xb6,0xed,0x2c,0x75,0x7d,0xf1,0x5d,0x6d,
                 0x34,0x30,0xb6,0x63,0x91,0x99,0x3a,0xdb,0x58,0xf2,
                 0x65,0xf5,0x7c,0x70,0x6d,0x9d,0x87,0x85,0xc7,0x02,
                 0x3d,0xf9,0xed,0x49,0x7c,0x3c,0x5f,0x82,0x67,0xfb,
                 0xe7,0xdb,0xc4,0xf1,0x22,0x13,0xa1,0x00},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x3c,0x7c,0xee,0x96,0x02,0x21,0xc9,0xd9,0xf7,0x46,
                0x4a,0xeb,0x70,0xd1,0x98,0xbd,0x60,0x41,0x4d,0xc3,
                0xff,0xbf,0xa7,0xa2,0x22,0x7a,0x3a,0x37,0x5e,0xbb,
                0x8f,0x64,0x48,0xe5,0x24,0x70,0x6e,0x1e,0x3a,0xe9,
                0x55,0x41,0xbd,0xce,0xf2,0xb3,0x1d,0x9f},
    }, {	// 32	(240)
        .key = {0xdd,0x90,0x8e,0xc0,0x58,0xf1,0x37,0xa4,0x4d,0x76,
                0xc0,0x04,0x82,0x1a,0x47,0x50,0x35,0x76,0x15,0x82,
                0x0d,0xe3,0x2e,0x5d,0x51,0xdf,0x25,0xf2,0x58,0xb2,
                0x35,0x69,0x41,0x51,0x8e,0xfe,0xe4,0x82,0xed,0x4a,
                0x07,0x41,0x6f,0xc4,0xd6,0x62,0xa7,0x87,0x8d,0x79,
                0xee,0x56,0x78,0xf7,0xfa,0xdd,0x1d,0x95,0xb3,0x39,
                0xb8,0xf6,0x41,0xbb,0xe7,0x87,0x6a,0xe9,0xa7,0xab,
                0x1b,0xc6,0x7f,0x15,0x44,0x54,0xfb,0x74,0xe9,0x56,
                0x5c,0x56,0x77,0x5a,0x8e,0x46,0x54,0xf7,0x5a,0x38,
                0xb9,0x54,0xdd,0x28,0xc4,0xe9,0x39,0xfd,0xc9,0x8a,
                0x8a,0xb3,0xea,0xa1,0x1c,0xb9,0xe7,0xbb,0xdb,0x98,
                0x37,0x46,0x0a,0xd6,0x57,0x98,0x38,0x1a,0x62,0x34,
                0x70,0x90,0xe2,0x49,0xb1,0x8f,0xe5,0x7c,0x9d,0x7a,
                0x54,0xe7,0x75,0xe4,0x81,0x62,0x45,0xf7,0xff,0x01,
                0x5c,0x16,0xb6,0xde,0xff},
        .key_len = 145,
        .data = {0xb9,0x37,0x7d,0x9b,0xdd,0xf4,0x0c,0xe1,0x76,0x28,
                 0xb4,0x57,0x0a,0xce,0xd9,0xe4,0xb1,0x32,0xe6,0x55,
                 0xd4,0xa5,0x35,0xaf,0x35,0x75,0x2f,0xc3,0x2f,0x1c,
                 0xb4,0x04,0x40,0xb8,0xbd,0x96,0xc4,0xbb,0x3f,0xa7,
                 0x03,0xe4,0x53,0x02,0x6e,0x6e,0x95,0xe1,0x26,0x87,
                 0xc9,0x03,0xbe,0x03,0xc5,0xff,0x42,0x52,0x8b,0xd8,
                 0x78,0xaf,0xb5,0xd1,0x65,0x9b,0x16,0x83,0x13,0x8a,
                 0x9e,0x2c,0x92,0xdc,0x7e,0x4a,0x3d,0x0e,0x8d,0x69,
                 0x3e,0x32,0xea,0x39,0x55,0x9c,0xe3,0xe3,0xd5,0xdf,
                 0x16,0x9d,0xef,0xf8,0xd6,0x7d,0x32,0xc8,0xd1,0x8a,
                 0x53,0xc8,0xef,0x19,0x2a,0xf8,0x7d,0x57,0xfe,0x18,
                 0x8a,0x22,0x02,0x1b,0x91,0x1d,0x1f,0xd9,0x59,0x03,
                 0xf4,0x04,0x1a,0x3b,0x1c,0x5d,0xe1,0xad},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x43,0x5e,0x4a,0xc3,0x7f,0x87,0x31,0x52,0x15,0x1b,
                0xa0,0x89,0xe7,0xf3,0x92,0x5c,0xcf,0x37,0x46,0xdf,
                0x52,0x5b,0x41,0xcc},
    }, {	// 33	(241)
        .key = {0x74,0x38,0xc5,0x42,0x4a,0xe9,0x5a,0xcb,0x1a,0x77,
                0xf2,0x7f,0xcb,0x43,0x38,0xed,0xfc,0x77,0x7f,0xb0,
                0x33,0x9a,0x03,0x9e,0x37,0x61,0x72,0x42,0xba,0xc8,
                0xab,0x8d,0x3b,0x62,0xc5,0xc8,0x2b,0xed,0x53,0xcd,
                0x4f,0x2a,0xe6,0x77,0x65,0xec,0xd4,0x57,0x0a,0x6e,
                0x38,0xa8,0xdb,0xe9,0x3a,0x85,0xdb,0x66,0x91,0x5a,
                0x15,0xd1,0x46,0x99,0x82,0x50,0xba,0xae,0x2c,0xd3,
                0xea,0x34,0x94,0xeb,0xf2,0x69,0x51,0xdf,0xd0,0xdf,
                0xfb,0xfd,0x6b,0x75,0x47,0x2e,0xd4,0x86,0x73,0xcd,
                0xcb,0x60,0xe5,0xb9,0x85,0xf8,0x0f,0xa9,0xac,0xdc,
                0x95,0xc0,0xa8,0x68,0xb2,0x62,0x1d,0x3d,0xd8,0x45,
                0xb4,0xef,0x96,0xcb,0x1f,0xfe,0xbf,0x8f,0x57,0x08,
                0xc9,0x3d,0x28,0x3c,0x73,0xa8,0xf0,0x12,0xaa,0x16,
                0xa4,0x39,0xae,0xde,0x13,0xd1,0x71,0x36,0x6f,0xdb,
                0x40,0x46,0x09,0xee,0xa4},
        .key_len = 145,
        .data = {0xdc,0x64,0x30,0xd1,0x4e,0x67,0x5d,0xec,0x59,0x76,
                 0xe6,0x71,0xaf,0x07,0xb9,0xa4,0x53,0xa3,0x8d,0x6e,
                 0x5b,0x97,0xc9,0xb0,0xf5,0xed,0xa2,0xd7,0xa8,0x9a,
                 0x84,0xb1,0xdc,0xf9,0xb4,0x7f,0x9d,0x78,0x33,0x4b,
                 0x9c,0x92,0xe5,0x7b,0x76,0xfd,0xf2,0xa7,0x39,0x24,
                 0x6d,0xa8,0x25,0xd7,0x34,0xf1,0xaf,0x41,0x17,0x23,
                 0xcc,0x1b,0x3c,0xdb,0x6b,0x20,0xc1,0xce,0x43,0xc1,
                 0xa4,0x19,0xd0,0x19,0xe0,0xd2,0x98,0x23,0xd9,0xe3,
                 0xb3,0x2b,0xfa,0x18,0x8a,0x82,0x9d,0x76,0xdf,0x76,
                 0x2b,0xcf,0x0e,0x81,0x84,0x93,0x92,0xc1,0xa1,0xe9,
                 0xca,0xa8,0x78,0xfe,0xfd,0xf5,0x1f,0x9d,0x9d,0xe3,
                 0x35,0x01,0xc8,0xa0,0x7e,0xe7,0x14,0x00,0xcd,0xe7,
                 0x8c,0x73,0x27,0x03,0xf6,0x35,0x25,0x81},
        .data_len = 128,
        .mac_len = 24,
        .mac = {0x4f,0x9c,0x9a,0xb5,0x06,0xd2,0xb2,0x13,0x7d,0xef,
                0xee,0x27,0x7f,0xe2,0x05,0xc8,0x8c,0xeb,0x16,0xa6,
                0xeb,0x63,0xfa,0x9b},
    }, {	// 34	(255)
        .key = {0x7e,0xae,0x9b,0x4d,0xf8,0x1f,0xe3,0xe1,0x3a,0xa5,
                0xb2,0x91,0x14,0x9a,0x89,0x28,0x35,0xc3,0xa9,0x37,
                0xfb,0xd5,0x84,0xb8,0x8a,0x37,0x71,0xf1,0x1c,0x0a,
                0xa3,0x9b,0x98,0x67,0x5c,0x44,0x65,0xe7,0x5e,0xf2,
                0x80,0x69,0xa3,0x09,0xcc,0x0f,0x4d,0x6e,0xaa,0x8a,
                0xe0,0xed,0xca,0x98,0xaf,0xd8,0x41,0xca,0x94,0xf1,
                0xdb,0x8b,0xe4,0x8a,0xb2,0x5e,0xa2,0xeb,0xff,0x67,
                0xf3,0xf4,0x31,0x2d,0xd2,0x04,0x3c,0x9e,0x05,0x19,
                0x34,0x54,0x50,0x38,0x1a,0x16,0xe8,0x0f,0xf6,0xa2,
                0x2a,0xd9,0x25,0xa9,0x8d,0x82,0xf5,0x1c,0xd6,0x0b,
                0x6f,0xa6,0x8b,0x31,0x74,0x7a,0xa9,0x49,0x0c,0x4e,
                0x99,0x6f,0xfa,0xa7,0x61,0xdf,0x94,0x58,0x18,0xa3,
                0xd1,0x45,0x9c,0x6f,0xa4,0x6d,0x87,0xc9,0x88,0x28,
                0x5e,0x6c,0x5c,0xac,0x0c,0xef,0x72,0xd6,0x8f,0xa2,
                0xe3,0x28,0xf2,0xb9,0x25},
        .key_len = 145,
        .data = {0x2a,0xe0,0x9d,0xb7,0xa5,0x9a,0x2f,0xfe,0x19,0x9f,
                 0x37,0xe4,0xaf,0x4b,0x2d,0x51,0xbb,0x6c,0x3b,0x17,
                 0xd9,0x01,0xf1,0xd6,0x3b,0x42,0x3a,0x1d,0x41,0xe0,
                 0x8f,0xd1,0x4a,0x8d,0x6a,0xd3,0xcb,0xbf,0xfe,0xdd,
                 0x9e,0xb1,0x25,0xad,0x9c,0xab,0x9a,0x66,0x6d,0x84,
                 0x68,0x33,0x9f,0x65,0xa2,0x2d,0xf6,0xcf,0x55,0x7f,
                 0x03,0xe0,0xbb,0xc7,0xf8,0xd2,0xb1,0xc0,0xbb,0xc4,
                 0x6b,0x1f,0x39,0xbd,0x67,0xba,0xfd,0xee,0x9f,0xef,
                 0x01,0x37,0x03,0x44,0xc4,0xed,0x26,0x4b,0x9c,0xdb,
                 0x38,0xc9,0x4e,0x86,0x34,0x16,0xdc,0xb6,0x5a,0xc7,
                 0x9e,0x31,0xe0,0x8b,0x94,0xca,0xe5,0x53,0xfb,0x9a,
                 0xa6,0xb6,0x1e,0x8e,0xf4,0x93,0x6f,0x22,0xe6,0xf8,
                 0xd6,0x6b,0xcb,0x42,0x49,0x5b,0x62,0x86},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0xba,0x84,0xb3,0xd2,0x21,0xac,0x58,0x98,0x63,0xb7,
                0x87,0x60,0xbf,0x6b,0x98,0x39,0xd4,0x76,0xcf,0x8d,
                0x6b,0xf1,0x4f,0xaa,0x93,0x29,0xb3,0x0a,0x61,0x83,
                0x14,0x07},
    }, {	// 35 	(256)
        .key = {0x76,0x0c,0x1d,0xa0,0x63,0xe1,0x04,0xfa,0x69,0xd2,
                0x67,0x6a,0x86,0x73,0xdc,0x45,0x8c,0xde,0x62,0x98,
                0x24,0xa9,0x8d,0xbb,0x12,0x15,0xe3,0x29,0xbb,0x88,
                0x85,0x8e,0xe4,0x3e,0x12,0x22,0xbc,0x3b,0xc8,0x36,
                0x1d,0x84,0xec,0x0c,0x0f,0x8e,0x6c,0xec,0x8e,0xf7,
                0xc4,0x74,0x8d,0x74,0x1e,0x30,0x3a,0x0b,0xbd,0x6b,
                0x84,0x82,0x9b,0x44,0xda,0x17,0x51,0x04,0x36,0x71,
                0x38,0xae,0x5b,0xd7,0xce,0xd8,0x5b,0xa0,0x5b,0xba,
                0x5b,0xea,0xd4,0x85,0x9a,0x8f,0xff,0xf6,0xb0,0x55,
                0xd8,0x2a,0x14,0x65,0x34,0xb4,0xe1,0xff,0xa4,0xed,
                0xda,0x6b,0x9c,0x8b,0xab,0x33,0xd4,0x75,0x72,0x8f,
                0xef,0xd6,0x7d,0x21,0x5b,0x70,0x55,0xe2,0x78,0x34,
                0xfc,0x5d,0xfc,0x0f,0x74,0x1b,0x96,0x06,0x80,0x5e,
                0xd1,0x85,0xaf,0x12,0x3c,0xf4,0x95,0x00,0xc8,0x8c,
                0x5c,0x45,0x71,0x69,0x7c},
        .key_len = 145,
        .data = {0x42,0x14,0x50,0x74,0xb2,0x6d,0x88,0x40,0x32,0x7f,
                 0xdd,0xe4,0x97,0x9c,0xe4,0xf6,0x30,0xa4,0x8e,0xef,
                 0x9e,0x87,0x9f,0xd5,0xe6,0x05,0x61,0x17,0xc5,0xb8,
                 0x99,0xb9,0x44,0x55,0xba,0xb0,0x8b,0x2f,0x59,0x90,
                 0xfa,0xfc,0xe1,0x64,0x76,0xdc,0x88,0xdd,0x01,0x97,
                 0x81,0xb0,0x8e,0x26,0x73,0x39,0x13,0x76,0x2b,0x64,
                 0xdf,0x68,0x8e,0xcd,0x2e,0x1c,0xd1,0x26,0xfb,0x6f,
                 0xec,0xda,0x15,0xc2,0x4b,0xef,0x75,0xfe,0xc7,0x31,
                 0xcf,0xd4,0xd6,0x8d,0x67,0x4a,0xfe,0xbb,0x26,0x73,
                 0x90,0x20,0xed,0x7d,0x92,0xf2,0x29,0xf2,0xfd,0x9d,
                 0xa8,0x37,0x29,0x65,0xab,0x59,0x85,0x4d,0x3c,0x08,
                 0x8b,0x38,0xc1,0xe4,0xf3,0x4f,0xd2,0xf8,0x27,0xa2,
                 0x48,0x84,0xf8,0x12,0x96,0x00,0x8f,0x6e},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0x24,0xb4,0xb0,0xca,0x23,0x48,0xc2,0x19,0xc1,0x51,
                0x29,0x7f,0x22,0x0b,0x2d,0x19,0xde,0xf3,0xbc,0x0d,
                0x8d,0x67,0x71,0x74,0x81,0x45,0xd6,0xeb,0x1e,0xfe,
                0x5a,0x5f},
    }, {	// 36	(270)
        .key = {0x81,0x6a,0xa4,0xc3,0xee,0x06,0x63,0x10,0xac,0x1e,
                0x66,0x66,0xcf,0x83,0x0c,0x37,0x53,0x55,0xc3,0xc8,
                0xba,0x18,0xcf,0xe1,0xf5,0x0a,0x48,0xc9,0x88,0xb4,
                0x62,0x72,0xea,0xc0,0xf2,0xf0,0x2a,0x0a,0x2d,0x2f,
                0xd8,0xfa,0xb2,0xe6,0x23,0x10,0xcb,0xc7,0x3f,0xc0,
                0xef,0xf9,0x0a,0x81,0x8c,0xc7,0xee,0x21,0x04,0x9a,
                0x09,0x5b,0xf5,0x24,0x8b,0xab,0xbe,0xdf,0x60,0x61,
                0x3f,0xb6,0x10,0xf5,0xe0,0x01,0xe3,0x9a,0xd4,0xb7,
                0x48,0x15,0x9f,0xdf,0x77,0xd0,0xd5,0x48,0x56,0x2a,
                0x25,0x7a,0x99,0x12,0x97,0xec,0x2e,0xd4,0x12,0x78,
                0xd8,0x32,0x21,0x0b,0x1a,0xb7,0xf6,0xa1,0x56,0x48,
                0xb0,0x71,0x36,0xfc,0xd9,0x64,0x4b,0x2e,0x33,0xcc,
                0xf4,0xd0,0xb4,0x53,0x82,0x01,0xcf,0x31,0x99,0xe2,
                0xf5,0xa3,0xd9,0xa6,0x5c,0x98,0xbe,0xe9,0xff,0xad,
                0xc5,0x43,0xe6,0x5b,0xd1},
        .key_len = 145,
        .data = {0xd0,0xdb,0x3f,0xf9,0xcf,0x2d,0xa1,0x0d,0x1e,0xdf,
                 0xef,0x38,0x9b,0xa7,0x17,0x80,0xcb,0x49,0xe0,0x5d,
                 0xcc,0xb8,0x7c,0x08,0x8b,0x7e,0x60,0xf5,0x37,0x5f,
                 0x53,0x9e,0xf5,0x39,0xc5,0x83,0xd5,0x2f,0x63,0x6f,
                 0x04,0x06,0xe8,0xfc,0x44,0xcf,0x36,0x59,0x9a,0xc7,
                 0xa5,0x4b,0x46,0x7b,0x9b,0x72,0xf8,0x30,0x5d,0xce,
                 0x41,0x81,0x24,0x82,0x48,0x4a,0x74,0xd7,0xbe,0xc5,
                 0xa9,0x83,0x14,0xf3,0xd3,0x2c,0xeb,0x3e,0xc3,0x28,
                 0xcc,0x5c,0x66,0x87,0x19,0x76,0x52,0x53,0x43,0x8a,
                 0x9f,0x16,0x3e,0x0e,0xf8,0x9d,0x32,0xd6,0xf1,0xfc,
                 0x74,0x37,0x9b,0xd2,0xb4,0x6d,0x57,0xce,0x78,0x3a,
                 0xe7,0x72,0xc9,0xd0,0xcb,0x17,0x2b,0xf1,0xca,0x32,
                 0xd3,0x55,0xcf,0xd5,0x15,0x4c,0xf6,0x79},
        .data_len = 128,
        .mac_len = 40,
        .mac = {0x29,0x19,0x57,0xf7,0x98,0xa3,0x2c,0x65,0x95,0x4d,
                0x31,0x28,0x9a,0xac,0x24,0xd6,0x2e,0xa4,0x74,0x69,
                0xe5,0xb1,0x10,0x5c,0xca,0x37,0x2e,0xab,0xed,0x44,
                0x74,0x63,0x16,0xcb,0x24,0xfb,0x13,0xc2,0x48,0x53},
    }, {	// 37	(271)
        .key = {0xd9,0xaa,0x1a,0x8f,0x94,0x47,0x1a,0x4c,0x69,0xe7,
                0x1f,0xd2,0x56,0xdf,0x38,0xc6,0x00,0x92,0x4b,0x42,
                0xa5,0x95,0xad,0x1e,0x52,0x11,0xf0,0x6e,0x56,0x69,
                0xfc,0x4b,0xf6,0x1d,0x61,0xd7,0x62,0xef,0xec,0x7b,
                0xe8,0x44,0x79,0x73,0x68,0xcc,0x07,0x8d,0x08,0x65,
                0x12,0x2d,0x28,0x3d,0x1d,0x95,0x99,0x37,0x8e,0x63,
                0x09,0x91,0xbe,0x5b,0xc2,0xc3,0x51,0x67,0x95,0xaa,
                0xc3,0xdc,0x10,0xe5,0x44,0xff,0xf8,0x80,0xc6,0xc8,
                0xf3,0xcf,0x54,0xe6,0x84,0x9d,0xa5,0x02,0x3c,0x87,
                0xf1,0x48,0x2f,0xa2,0x03,0x24,0xa6,0x64,0x90,0x49,
                0x13,0xc1,0xc7,0xf8,0x4a,0x94,0x29,0x7f,0xed,0x41,
                0x9b,0x00,0x28,0x98,0xb9,0x73,0x9f,0xfd,0x6e,0xee,
                0xa5,0xc6,0xc1,0x24,0xaf,0xe5,0xf2,0x2d,0x08,0xc7,
                0xc6,0x85,0xc6,0x7c,0x1f,0x56,0xfc,0x17,0xcf,0xfa,
                0x31,0x58,0xae,0xbb,0x11},
        .key_len = 145,
        .data = {0xdc,0xda,0x25,0xe0,0x85,0x0a,0x55,0x53,0x68,0xec,
                 0x4a,0xdf,0xc4,0x13,0x5e,0x47,0xe4,0xf3,0x47,0x37,
                 0x4e,0x42,0xec,0xcd,0x3e,0xa4,0x78,0x57,0x52,0x86,
                 0xad,0x5f,0x87,0x4a,0x7c,0xe5,0x64,0xa5,0xd2,0xeb,
                 0x75,0x46,0xb4,0x37,0xfb,0xb9,0x8e,0x54,0xe2,0x8a,
                 0xc8,0xda,0xf3,0xad,0x9b,0xef,0x4b,0x1d,0x23,0x0e,
                 0x0c,0xc4,0xb5,0x07,0x52,0x9b,0x72,0x47,0x69,0x12,
                 0x1c,0x5a,0x3d,0xb2,0x29,0xae,0x11,0x01,0x16,0x9a,
                 0x74,0xa6,0xfe,0x9e,0x71,0xcd,0x6b,0xed,0x07,0xcd,
                 0x6c,0x28,0xd9,0x08,0xe8,0x6d,0x25,0x13,0x31,0x90,
                 0xfa,0x18,0xdc,0xa8,0xa6,0x78,0xd1,0xce,0x6e,0xb4,
                 0x9e,0x21,0x83,0x8d,0x11,0x0e,0x31,0x6a,0x05,0xce,
                 0x58,0xfb,0x7c,0x2f,0x20,0x1c,0xee,0x98},
        .data_len = 128,
        .mac_len = 40,
        .mac = {0x97,0x11,0x17,0x3e,0x1c,0x86,0xfc,0xdb,0x11,0xb7,
                0xc2,0x24,0xec,0xb7,0x6b,0xa3,0x90,0xc2,0xe1,0x21,
                0x8d,0x26,0xf3,0x53,0x98,0x0e,0x65,0x6a,0x08,0xbb,
                0xaa,0xd9,0xca,0x8e,0x8f,0x27,0xc0,0xae,0xf6,0xad},
    }, {	// 38	(285)
        .key = {0xf0,0x5d,0x56,0x1f,0x5a,0xd7,0x04,0x03,0x26,0x4c,
                0x5e,0x0a,0x0e,0xdc,0x12,0xfd,0x47,0x3b,0x19,0xc0,
                0xb4,0x0f,0x8c,0xd8,0x5a,0x99,0xba,0x2a,0x14,0x98,
                0x77,0x05,0x87,0x6a,0xb7,0x63,0x59,0x75,0x5b,0x6c,
                0x9e,0xc5,0x4a,0x3c,0x93,0xf6,0xc4,0xe6,0x8f,0x55,
                0xf3,0xb9,0x36,0x42,0xc3,0xc2,0xf0,0xd9,0xf4,0x91,
                0x9a,0xd1,0x6e,0x40,0x7b,0xa3,0xd4,0xb2,0x79,0xef,
                0x5b,0x19,0x8c,0x1c,0xdd,0xbb,0x74,0x40,0x29,0xf5,
                0xa7,0x3f,0x9e,0x80,0x8e,0x36,0xf8,0xf3,0xf0,0x1a,
                0x69,0x89,0xaf,0x9c,0xec,0x25,0xb2,0x50,0xd6,0x93,
                0x22,0x0f,0xad,0x11,0xd9,0x9a,0x3e,0x0e,0x17,0x7f,
                0xea,0x31,0x77,0x41,0x41,0x9d,0x22,0xb3,0xd2,0x74,
                0x43,0xa5,0x40,0x99,0xbb,0xc2,0x99,0xbb,0x15,0xb9,
                0xe4,0x8f,0xbc,0x9b,0xf9,0x5c,0x6b,0x84,0x96,0xbd,
                0xe6,0x7e,0xae,0xa3,0xe8},
        .key_len = 145,
        .data = {0x77,0xd3,0xf3,0xe6,0x47,0xe6,0x77,0x66,0xe5,0xf4,
                 0xcf,0x1b,0xce,0x5f,0x63,0x1b,0xd5,0x75,0xdd,0xbd,
                 0x02,0xf2,0x96,0x43,0xa0,0xc6,0x4d,0xbd,0x92,0x19,
                 0x1f,0x2a,0xe6,0x8d,0xb3,0xdf,0xad,0xc3,0xb6,0x2d,
                 0x09,0x20,0x87,0x3e,0x87,0xd1,0x33,0x40,0xaf,0x0c,
                 0xa3,0xc5,0xda,0x99,0x14,0x6a,0x44,0x92,0xc8,0xb7,
                 0x62,0x67,0xfb,0x47,0x76,0x24,0x19,0x29,0x60,0xf7,
                 0x2e,0x85,0xb7,0xed,0x9e,0x83,0x18,0xfc,0x16,0x68,
                 0xbe,0x46,0xc2,0x03,0x53,0x9c,0xc1,0x47,0x06,0x41,
                 0xd6,0x39,0xde,0xf1,0x60,0x0d,0x4e,0x22,0x8c,0x8b,
                 0x09,0x8a,0xc9,0xb8,0x17,0xe1,0x7c,0xb3,0x29,0xe8,
                 0xf5,0xdd,0x2a,0xaa,0xa2,0x3c,0x16,0x02,0x83,0x22,
                 0x0f,0x5d,0xde,0x09,0xae,0xc1,0x34,0xc2},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x72,0x4e,0x5d,0x2d,0x51,0xd9,0x8c,0x15,0xce,0x2e,
                0x78,0xf8,0x61,0xd7,0xb6,0xf8,0x95,0x28,0x82,0xe9,
                0xd9,0x3d,0x40,0x85,0x0b,0x78,0xa2,0x3e,0x63,0x2c,
                0x4e,0x14,0xa2,0x22,0xab,0x37,0x26,0xb1,0xa0,0xaa,
                0x7c,0x6b,0x2c,0xd6,0x60,0x82,0xed,0x95},
    }, {	// 39	(286)
        .key = {0x95,0xec,0xe1,0xc8,0xae,0x5e,0x94,0xd1,0x6e,0xc9,
                0x98,0x3b,0x10,0x89,0xa3,0x73,0x95,0xad,0x5b,0x1d,
                0x66,0x09,0x16,0xc1,0x3c,0x87,0xe4,0xc1,0x3d,0xbe,
                0xcf,0x8f,0x68,0xc6,0x61,0x1c,0x32,0x4a,0x67,0x94,
                0x71,0xde,0xf5,0x48,0x7a,0x93,0xaa,0xec,0x86,0xc9,
                0x35,0x02,0x5b,0x45,0x18,0x96,0x28,0x84,0xac,0x2c,
                0xb0,0x4e,0x66,0xf7,0xaa,0x8e,0x58,0x4b,0x68,0x60,
                0xfb,0x55,0xb8,0x6c,0x2b,0x0a,0x08,0x73,0x73,0x5d,
                0xcd,0x27,0x8b,0xb5,0x25,0x40,0x1f,0x9e,0xba,0xcc,
                0xd2,0xbe,0xea,0xc6,0x83,0x0c,0x26,0xeb,0xcf,0x3c,
                0x98,0xc9,0xd7,0x7d,0x09,0x19,0x43,0x67,0x01,0x4e,
                0x87,0x2f,0x30,0x6e,0x64,0x1e,0x0c,0x21,0xb2,0x41,
                0xbc,0x08,0x5e,0x61,0x35,0x4f,0xaf,0x35,0xa3,0x86,
                0xcd,0xd7,0x0a,0xac,0x83,0x75,0x2d,0x8d,0x44,0x49,
                0xaf,0x4f,0x6c,0xcb,0x78},
        .key_len = 145,
        .data = {0x18,0xdb,0xab,0x9f,0x86,0xb9,0xd7,0x0b,0xbd,0xeb,
                 0x01,0x8f,0x6a,0x76,0xea,0x7a,0xf2,0x3e,0xb2,0xff,
                 0x11,0x1e,0x9b,0xe3,0xc1,0x38,0x11,0x79,0x5d,0x8a,
                 0xe7,0xd0,0x06,0xc3,0xe4,0x2b,0x46,0x54,0x7e,0xb1,
                 0xf3,0xc9,0xe5,0x66,0x56,0x5a,0x43,0x5a,0x8d,0xbd,
                 0x42,0x21,0x2e,0x3f,0xd0,0x82,0x2d,0x13,0x1f,0x73,
                 0x00,0xea,0xef,0x46,0x00,0xc4,0x0f,0x1d,0x13,0x05,
                 0x21,0xa3,0x88,0xcb,0x9f,0xfe,0x42,0x7f,0x1b,0xff,
                 0x19,0xaa,0xcb,0x9c,0x7d,0x0a,0x44,0xa1,0x5c,0xe6,
                 0x86,0xa2,0x46,0x9e,0x39,0x34,0xd0,0x86,0x36,0x5d,
                 0x36,0xf4,0x49,0x48,0x44,0x98,0x35,0x3d,0x76,0x0c,
                 0xf9,0xd1,0x5e,0xac,0x52,0x5a,0x46,0xa8,0x81,0xa6,
                 0x17,0x58,0x4e,0xed,0x79,0xcf,0x4d,0x03},
        .data_len = 128,
        .mac_len = 48,
        .mac = {0x2b,0xe1,0xbd,0x6a,0x76,0x6e,0x30,0x79,0x21,0x54,
                0xcd,0xa0,0x0a,0xf9,0x7c,0xc5,0x12,0xe8,0x14,0x13,
                0xe0,0xfb,0x76,0x16,0x98,0xf3,0x9a,0x26,0xce,0xcc,
                0x3f,0xac,0xe6,0xf9,0xa9,0x8b,0x7c,0x49,0x60,0x51,
                0x26,0xdf,0xa5,0xaa,0x8d,0xe1,0xad,0x72},
    }
};

struct HMAC_TEST_VECTOR fips_sha512_hmac_general_test_vector[] = {
    {	// 0	(0)
        .key = {0x72,0x63,0x74,0xc4,0xb8,0xdf,0x51,0x75,0x10,0xdb,
                0x91,0x59,0xb7,0x30,0xf9,0x34,0x31,0xe0,0xcd,0x46,
                0x8d,0x4f,0x38,0x21,0xea,0xb0,0xed,0xb9,0x3a,0xbd,
                0x0f,0xba,0x46,0xab,0x4f,0x1e,0xf3,0x5d,0x54,0xfe,
                0xc3,0xd8,0x5f,0xa8,0x9e,0xf7,0x2f,0xf3,0xd3,0x5f,
                0x22,0xcf,0x5a,0xb6,0x9e,0x20,0x5c,0x10,0xaf,0xcd,
                0xf4,0xaa,0xf1,0x13,0x38,0xdb,0xb1,0x20,0x73,0x47,
                0x4f,0xdd,0xb5,0x56,0xe6,0x0b,0x8e,0xe5,0x2f,0x91,
                0x16,0x3b,0xa3,0x14,0x30,0x3e,0xe0,0xc9,0x10,0xe6,
                0x4e,0x87,0xfb,0xf3,0x02,0x21,0x4e,0xdb,0xe3,0xf2},
        .key_len = 100,
        .data = {0xac,0x93,0x96,0x59,0xdc,0x5f,0x66,0x8c,0x99,0x69,
                 0xc0,0x53,0x04,0x22,0xe3,0x41,0x7a,0x46,0x2c,0x8b,
                 0x66,0x5e,0x8d,0xb2,0x5a,0x88,0x3a,0x62,0x5f,0x7a,
                 0xa5,0x9b,0x89,0xc5,0xad,0x0e,0xce,0x57,0x12,0xca,
                 0x17,0x44,0x2d,0x17,0x98,0xc6,0xde,0xa2,0x5d,0x82,
                 0xc5,0xdb,0x26,0x0c,0xb5,0x9c,0x75,0xae,0x65,0x0b,
                 0xe5,0x65,0x69,0xc1,0xbd,0x2d,0x61,0x2c,0xc5,0x7e,
                 0x71,0x31,0x59,0x17,0xf1,0x16,0xbb,0xfa,0x65,0xa0,
                 0xae,0xb8,0xaf,0x78,0x40,0xee,0x83,0xd3,0xe7,0x10,
                 0x1c,0x52,0xcf,0x65,0x2d,0x27,0x73,0x53,0x1b,0x7a,
                 0x6b,0xdd,0x69,0x0b,0x84,0x6a,0x74,0x18,0x16,0xc8,
                 0x60,0x81,0x92,0x70,0x52,0x2a,0x5b,0x0c,0xdf,0xa1,
                 0xd7,0x36,0xc5,0x01,0xc5,0x83,0xd9,0x16},
        .data_len = 128,
        .mac_len = 32,
        .mac = {0xbd,0x3d,0x2d,0xf6,0xf9,0xd2,0x84,0xb4,0x21,0xa4,
                0x3e,0x5f,0x9c,0xb9,0x4b,0xc4,0xff,0x88,0xa8,0x82,
                0x43,0xf1,0xf0,0x13,0x3b,0xad,0x0f,0xb1,0x79,0x1f,
                0x65,0x69},
    }, {	// 1 (1)
        .key = {0xe2,0x45,0xbe,0x9a,0x9c,0x81,0x01,0x26,0x38,0x30,
                0xad,0x35,0x15,0xc1,0xc6,0xcb,0xf2,0x85,0xa7,0xe4,
                0xb3,0x62,0xeb,0xc0,0x62,0xcb,0x8e,0x7e,0x75,0xef,
                0x50,0xec,0x4f,0x31,0x5a,0x9e,0x09,0xd9,0x24,0x3d,
                0x71,0x09,0x96,0x22,0x53,0xf2,0x6e,0x23,0xf8,0x47,
                0xe1,0xad,0xed,0xf2,0x85,0x14,0x05,0x07,0x6e,0x26,
                0xa1,0xf6,0x97,0x06,0x2f,0x04,0x84,0x38,0xf1,0xfc,
                0x26,0xf8,0x00,0x21,0xff,0xd0,0x90,0x68,0x87,0x69,
                0x75,0xe4,0xcd,0xa2,0xe7,0x82,0x61,0xdf,0x82,0xf6,
                0x72,0xa3,0x90,0xf5,0x34,0x62,0x8b,0xa5,0x84,0x90},
        .key_len = 100,
        .data = {0x42,0x53,0x15,0xdd,0x8e,0xcc,0xd1,0x7a,0x84,0xc1,
                 0xaa,0x00,0xff,0x72,0x76,0x3f,0x99,0xdd,0xcb,0xc2,
                 0xc3,0x81,0xb8,0xb2,0x15,0x67,0xb2,0xf8,0xe2,0x63,
                 0xd1,0xa2,0x10,0x98,0x3d,0x88,0x26,0x3a,0xe3,0x2f,
                 0xa7,0x98,0x6c,0xed,0x9f,0x59,0x6f,0x4e,0x7b,0x05,
                 0xe5,0xb7,0x1d,0xc8,0xde,0x49,0x30,0x73,0x73,0x08,
                 0xb9,0xc4,0xfc,0x3d,0xef,0xe7,0x83,0x19,0x4d,0x3c,
                 0x78,0x9a,0xe5,0x5b,0xa5,0xb3,0xf7,0x56,0x65,0xa7,
                 0xc2,0x3e,0x11,0xb6,0x9a,0xe8,0xbc,0xfb,0x3b,0xf0,
                 0x20,0x95,0x5d,0xff,0xd7,0x05,0x89,0x4a,0xcf,0xd7,
                 0x2a,0x5b,0xf8,0x85,0xe7,0x14,0x3f,0x98,0x30,0xf1,
                 0xc0,0x10,0x17,0x8d,0x37,0x06,0x62,0x68,0xb8,0x90,
                 0xde,0xe7,0xa1,0xe5,0xf6,0x9c,0xcc,0xc1},
        .data_len = 128,
        .mac = {0xc5,0x80,0x1d,0x80,0xa1,0x43,0x91,0x72,0x0e,0x77,
                0xeb,0x7f,0xfb,0x1a,0x0b,0x21,0xa1,0x63,0x23,0x64,
                0x1c,0x9a,0x31,0x2b,0x05,0xfd,0xc3,0x4e,0x90,0x38,
                0x3c,0x85},
        .mac_len = 32,
    }, {	// 2 (15)
        .key = {0x6a,0xa1,0x14,0x66,0xa2,0xc9,0x6e,0x20,0x54,0x4c,
                0x5b,0x34,0xc9,0x1f,0x90,0xd1,0x7f,0x97,0x99,0xa5,
                0x7c,0x73,0xca,0x00,0xe2,0x1d,0x77,0x36,0xc4,0x2d,
                0x68,0x45,0x38,0x2f,0x87,0xb7,0xad,0xa6,0xdc,0xca,
                0x7f,0x51,0xbb,0xcf,0xc9,0xac,0x3c,0xd0,0x7b,0xba,
                0xec,0x75,0xac,0x02,0xc2,0x28,0x31,0xd7,0x55,0x78,
                0x6e,0xdd,0x4b,0xbb,0x6c,0xc9,0x15,0xf8,0x2c,0x68,
                0xda,0x04,0x3b,0xe3,0xb0,0xea,0x87,0x41,0x12,0x90,
                0x71,0x2d,0x8f,0x98,0x23,0xfa,0xbf,0x85,0x24,0xe4,
                0x7b,0x1f,0x29,0x99,0x4d,0x18,0x2f,0xf8,0x26,0xef },
        .key_len = 100,
        .data = {0x27,0x71,0xcd,0xfd,0xd7,0x7a,0xab,0xbc,0xc9,0x57,
                 0x40,0xbb,0xd9,0x40,0x75,0x87,0x6e,0xd3,0x02,0x4f,
                 0x0b,0x5a,0x99,0x49,0x21,0x4d,0x60,0x52,0x28,0x18,
                 0xa5,0xd5,0xed,0x1d,0xce,0xed,0x2e,0xfd,0x8e,0xce,
                 0x64,0x13,0x5f,0x61,0xe0,0x42,0x2c,0x2e,0x19,0xf1,
                 0x4d,0x7f,0x45,0x53,0xb9,0xc2,0x26,0xec,0x77,0x30,
                 0x27,0x51,0x56,0xac,0xa7,0x2f,0xde,0xee,0x95,0x8e,
                 0x86,0xe0,0x3d,0x57,0x28,0x48,0x61,0x14,0xb1,0xa8,
                 0x8c,0xf8,0x1d,0x62,0xa3,0x1f,0xa3,0x20,0xbd,0x16,
                 0x2c,0x73,0x34,0x9e,0x0a,0xbb,0xfd,0xad,0xe7,0x46,
                 0x34,0xb6,0xfb,0xbc,0xfb,0x2a,0x83,0xba,0x85,0x3f,
                 0x6e,0xbd,0x6e,0xf0,0x59,0x42,0x4e,0x45,0xbd,0x2b,
                 0x8d,0xd4,0x67,0x65,0x79,0x88,0x29,0xc8},
        .data_len = 128,
        .mac = {0x24,0x76,0xdf,0xac,0x04,0x3e,0x55,0x51,0x8d,0xdb,
                0xa3,0x12,0xe6,0x36,0x99,0xb2,0x2a,0x58,0x7f,0xbd,
                0xcc,0xcc,0xf3,0x98,0x12,0x07,0xef,0x5c,0xfc,0x2e,
                0x27,0xee,0xb6,0xa4,0x06,0x2b,0xda,0x6b,0x19,0xbf},
        .mac_len = 40,
    }, {	// 3	(16)
        .key = {0x3b,0xeb,0x75,0x37,0x3a,0x0a,0x02,0x10,0x7b,0x27,
                0x48,0xa7,0xb9,0xa3,0x73,0x8b,0x1a,0x9d,0x75,0xfa,
                0xb1,0x67,0xa6,0xf8,0x47,0x3a,0xad,0x9e,0x5b,0x28,
                0xa4,0xb5,0x67,0x53,0x77,0x83,0x65,0x5d,0x8d,0xe5,
                0x92,0x1f,0xab,0x9e,0x72,0x9e,0x15,0xef,0x14,0x3a,
                0xf6,0x68,0x97,0x29,0x01,0xfd,0xed,0x24,0x59,0x5c,
                0xdf,0x60,0xae,0x7e,0x51,0xc0,0xa9,0x7c,0xc8,0x49,
                0x8b,0xa5,0x29,0xd6,0x11,0xcd,0xc4,0x0d,0x9a,0xe7,
                0x90,0x91,0x8b,0xd7,0xb8,0x79,0xf3,0x81,0x57,0x02,
                0x6b,0x26,0x80,0x80,0x41,0xf0,0x82,0x9b,0xd5,0xcb},
        .key_len = 100,
        .data = {0x04,0x05,0x55,0x5d,0x89,0xa8,0x11,0x77,0x94,0xc6,
                 0x42,0x85,0xcd,0x30,0x04,0x7d,0x64,0x2a,0x1d,0x63,
                 0x5f,0x6e,0xeb,0x6b,0x33,0x73,0x9b,0x5e,0x92,0xee,
                 0x9c,0x3e,0x85,0x9a,0x53,0x2d,0xba,0x34,0x11,0x8b,
                 0x1f,0xca,0x42,0x5f,0xb0,0x36,0xe7,0xf8,0x8c,0xe6,
                 0x46,0xd4,0x4f,0xde,0xdf,0xc0,0x9c,0xff,0xa3,0x41,
                 0xf9,0x89,0x61,0x33,0xe0,0xdf,0x81,0xb0,0x12,0x8a,
                 0xf4,0x33,0x2f,0x81,0x6b,0xfd,0x84,0xa7,0xc7,0xb8,
                 0x24,0x96,0xd1,0x5b,0x3a,0x9b,0x2c,0x4d,0x9d,0x01,
                 0x88,0xcd,0xa5,0x93,0x17,0x91,0x3e,0x5c,0x45,0xad,
                 0x3a,0xb1,0xa4,0xb7,0xec,0x72,0x25,0x29,0xf2,0x8c,
                 0x8e,0x06,0x52,0xc2,0x28,0xc1,0xc9,0xae,0xb8,0x5a,
                 0x5f,0x02,0xb1,0xca,0x3f,0x14,0xa8,0xa1},
        .data_len = 128,
        .mac = {0xfa,0x58,0xf0,0xaa,0xd7,0xa3,0x11,0x6c,0xae,0xc2,
                0x19,0x3a,0x3d,0xff,0xb0,0xd1,0xc0,0xb3,0x17,0x2c,
                0x3b,0x59,0xa9,0xa6,0x2e,0xbb,0xf3,0x1d,0x21,0xc7,
                0x66,0x81,0x8f,0x1c,0xef,0xb2,0x60,0x52,0xaf,0x72},
        .mac_len = 40,
    }, {	// 4	(30)
        .key = {0x79,0x86,0x75,0x59,0xc9,0x91,0x9f,0x39,0x4c,0xde,
                0x76,0xb3,0x1f,0xe2,0x2f,0x79,0x3b,0x88,0x98,0x70,
                0x01,0x76,0x4e,0x11,0x15,0x91,0x30,0x0f,0x70,0xc3,
                0x13,0x39,0xb1,0xfc,0xb8,0xa3,0xb4,0x76,0xfb,0x00,
                0x66,0x3e,0x4c,0x53,0xd8,0xf0,0x63,0xc7,0x92,0xc1,
                0x30,0xda,0x29,0xc3,0x11,0xc1,0x14,0x65,0x48,0x08,
                0xbb,0x39,0xac,0x7a,0xd1,0xfb,0x9e,0x40,0xe4,0xce,
                0x3f,0x4d,0x32,0xc8,0xe1,0xf9,0xbf,0xb1,0xbb,0xeb,
                0xa4,0x08,0xe0,0x3d,0xaa,0x47,0x72,0xdb,0x17,0x69,
                0x12,0x2c,0xc8,0xa8,0x9c,0xf5,0x80,0x8b,0x11,0x7a},
        .key_len = 100,
        .data = {0xd1,0x99,0x8d,0x3b,0x3f,0xf5,0xbc,0x1c,0x8f,0x72,
                 0x4a,0xab,0x79,0x2d,0xe8,0xa3,0x5c,0x60,0xfb,0xa4,
                 0xeb,0x41,0xf2,0x34,0x22,0x27,0xd6,0x03,0x80,0xef,
                 0x22,0x73,0x70,0x9e,0xd6,0x57,0x08,0x47,0xc7,0xfb,
                 0xf5,0x30,0x39,0x66,0xb6,0x30,0xaf,0xdb,0x09,0x5b,
                 0x02,0xa6,0xa3,0xe2,0xb1,0x15,0x24,0x34,0x78,0xc4,
                 0x4a,0x69,0x24,0x1d,0x65,0xe1,0xa5,0x20,0x0b,0x28,
                 0x65,0x60,0x0c,0x1d,0xd8,0x4e,0xa8,0x54,0x55,0xba,
                 0x00,0xb8,0x24,0x97,0xcb,0x75,0xd8,0xff,0x4a,0x15,
                 0xde,0x77,0x7e,0xe8,0x57,0x73,0x98,0xaa,0x23,0x1d,
                 0xb1,0x60,0x3e,0x52,0xbb,0x83,0x50,0xed,0xb8,0x60,
                 0x7f,0x49,0x2a,0xbf,0x88,0x01,0xca,0xc3,0x0e,0xfa,
                 0x7c,0x11,0x32,0xa4,0x83,0x34,0x63,0x7c},
        .data_len = 128,
        .mac = {0xb6,0xb1,0x8c,0x4c,0x53,0x64,0xb4,0x7f,0xd7,0x8c,
                0x3d,0x32,0x1d,0xad,0xf6,0x0d,0xe1,0x87,0xfa,0x68,
                0xf5,0xd7,0xe5,0x5c,0xa7,0xfb,0xf7,0x1f,0x42,0xc7,
                0x42,0x41,0x1d,0x37,0x9c,0x8c,0x55,0x18,0x0d,0x3f,
                0x54,0xcb,0x97,0x47,0xb5,0x1a,0xef,0x7f},
        .mac_len = 48,
    }, {	// 5	(31)
        .key = {0xe9,0x16,0x8c,0x65,0x9c,0x63,0xb6,0xf4,0x05,0x23,
                0xc9,0x05,0x32,0xba,0xe7,0x43,0xf2,0x4f,0xeb,0x2e,
                0x94,0x81,0x4b,0x6d,0xf2,0x55,0x43,0x65,0xaf,0x73,
                0x30,0x0a,0xbc,0x93,0x3a,0xf5,0x21,0x32,0x35,0xb8,
                0xfa,0x89,0xc9,0x6e,0xbf,0xbf,0xd1,0x96,0xc9,0x5e,
                0x02,0x21,0x12,0x04,0xcd,0xc9,0x3d,0x5b,0x86,0xa5,
                0xd6,0x4a,0xc5,0xfa,0xda,0x6d,0x0d,0x55,0x7a,0xc3,
                0xab,0xd6,0x1c,0xa7,0xe1,0xcf,0xa3,0x02,0x44,0x8e,
                0xf0,0xbe,0x63,0x76,0xa8,0x7e,0xa9,0x55,0x38,0x8c,
                0x85,0xf1,0x17,0x12,0xc7,0xe4,0x4b,0x8e,0x4e,0xaa},
        .key_len = 100,
        .data = {0x42,0x65,0x7c,0x63,0x16,0xfb,0xc1,0xbc,0xd6,0x9a,
                 0x87,0xc7,0xdf,0x3a,0xb0,0x6d,0xcc,0x1e,0x47,0x1e,
                 0x97,0x64,0xc1,0x0a,0x00,0xdb,0x83,0x94,0x33,0x14,
                 0xa5,0x03,0xa5,0xbc,0xf6,0x7b,0x3f,0x28,0x40,0x2d,
                 0xbe,0x32,0xcd,0xca,0x5a,0x44,0x29,0x3d,0xa5,0xb2,
                 0x2a,0x99,0xb7,0x4d,0x2e,0x50,0x88,0x35,0xa4,0xa7,
                 0x9d,0xee,0xce,0x66,0xe7,0xb6,0xc4,0xd4,0xe8,0x16,
                 0xbd,0xe2,0x17,0xb8,0x8a,0xa7,0x86,0x78,0x63,0x0c,
                 0xc2,0xbd,0xe6,0xc9,0xfa,0x0a,0x5d,0x90,0x64,0xd0,
                 0x40,0x87,0xdd,0x87,0xfb,0xfc,0xd0,0xfc,0xf0,0x55,
                 0x58,0xed,0xc0,0x45,0xa9,0xd3,0xc6,0x46,0xac,0xff,
                 0xe5,0x43,0xe9,0xe8,0x84,0x94,0xaa,0xc4,0xd6,0x30,
                 0x5d,0x55,0x5f,0x1e,0x3b,0x06,0xb4,0x2d},
        .data_len = 128,
        .mac = {0x0e,0x9b,0x3a,0xfa,0xb5,0x07,0x82,0xf1,0x75,0x1c,
                0x80,0xb4,0x6c,0xed,0x53,0x1c,0x07,0xc9,0xae,0x4a,
                0xf5,0x0c,0x77,0xa0,0x58,0xa2,0xdb,0x31,0xda,0xbb,
                0x01,0x3b,0x71,0x9c,0x4c,0x22,0xf5,0xe5,0xad,0xea,
                0xfa,0xb7,0x21,0x54,0xcf,0x1f,0xbd,0x21},
        .mac_len = 48,
    }, {	// 6	(45)
        .key = {0x2c,0xa6,0x7a,0xe3,0xea,0x3c,0x84,0xa2,0x54,0x4c,
                0xa8,0x79,0x44,0x57,0x34,0x0e,0x1e,0x42,0x4a,0x8a,
                0xb3,0xaa,0xe2,0x92,0x65,0x77,0x12,0x79,0x8b,0xb4,
                0x8e,0xb4,0x17,0x9e,0x6b,0x8e,0x76,0xfa,0x28,0x1d,
                0xb7,0xac,0xee,0x74,0xf0,0x86,0x17,0x1a,0xdd,0x5e,
                0xee,0xbb,0xcb,0x63,0xb5,0x1e,0xb4,0xb1,0xed,0x57,
                0xac,0x22,0xd1,0x3e,0x7b,0x67,0x24,0x1f,0x8c,0x58,
                0x2c,0xb3,0x06,0x89,0xff,0x4f,0x38,0x1e,0xfd,0x5c,
                0x3a,0xe0,0x9e,0x07,0xd1,0x90,0x6e,0x39,0x94,0x7b,
                0x55,0xca,0x4d,0x4e,0x1c,0xf2,0xa2,0x2c,0x2d,0x00},
        .key_len = 100,
        .data = {0x84,0x37,0x1c,0x9f,0xf4,0x0b,0x74,0x5a,0xb5,0x30,
                 0x0b,0xc9,0x14,0x51,0x2c,0x14,0x68,0xf9,0x8b,0x33,
                 0x97,0x04,0xe2,0x04,0xdb,0x54,0xdd,0x9a,0xc6,0xc5,
                 0x34,0xd8,0x84,0x62,0x16,0x3f,0x61,0xd9,0xce,0x05,
                 0x8a,0x2c,0xa1,0x67,0x18,0xaa,0xf5,0x40,0x4a,0x1e,
                 0xb9,0xd9,0x12,0x46,0x00,0x03,0xd4,0x86,0xef,0x63,
                 0x6a,0xde,0xda,0x0e,0x6b,0x1a,0xaa,0x56,0xad,0x48,
                 0x82,0xe3,0x08,0x6f,0xa2,0x05,0x5d,0x7e,0x8b,0x48,
                 0xd7,0x83,0x63,0x49,0x71,0x1d,0x9c,0xc9,0xb9,0x34,
                 0xd1,0x00,0xd3,0xe7,0xf0,0x33,0xc0,0xc6,0x9e,0x89,
                 0xe1,0x42,0xd7,0x1b,0xb6,0x61,0xd5,0x4a,0xec,0xb7,
                 0x34,0x55,0x97,0x91,0x53,0x2b,0xbc,0x73,0xfd,0x7d,
                 0x12,0xc5,0x49,0x18,0x09,0xc9,0xbe,0x72},
        .data_len = 128,
        .mac = {0xea,0xa9,0xb6,0x97,0xc8,0x3b,0xc5,0x71,0xa0,0x29,
                0xa6,0xdf,0x1f,0x44,0x0e,0x8c,0x64,0x6a,0xc7,0x63,
                0x33,0x3d,0x5f,0x7a,0x57,0xa6,0xff,0xe7,0xcf,0xda,
                0x7f,0x90,0x47,0x50,0xee,0x4c,0x3a,0xe8,0x87,0xe1,
                0xc2,0x94,0x29,0x01,0x2d,0x0a,0xa7,0x36,0xa8,0x32,
                0x6b,0xa5,0xc9,0x74,0xfc,0xe2},
        .mac_len = 56,
    }, { 	//7	(46)
        .key = {0xa8,0xce,0xfe,0xfa,0x10,0x9f,0xac,0xc8,0x5c,0xd6,
                0xf6,0x5b,0x91,0xb4,0x2a,0x98,0x8a,0xf5,0x1f,0xce,
                0xed,0xfb,0xc5,0xf7,0x5d,0x28,0x00,0x3b,0xf1,0x8c,
                0xb7,0xb6,0xe5,0xc0,0x28,0x7b,0x90,0x11,0x7e,0xd9,
                0x96,0xe1,0xa5,0xdd,0x5f,0x4b,0x5d,0x17,0xc8,0x06,
                0x8d,0xa1,0x18,0x8c,0xda,0x5e,0x35,0x7b,0x79,0x80,
                0x18,0x3b,0x41,0x4d,0x7e,0xe1,0x05,0x22,0xe0,0x53,
                0x20,0xb8,0xa6,0xad,0x51,0xa6,0x98,0xe4,0xca,0x27,
                0x95,0xaa,0x83,0x1e,0x87,0x91,0xd4,0x57,0x96,0x72,
                0x43,0x10,0x9c,0x10,0x9b,0xb8,0xd9,0xf9,0x3e,0xf6},
        .key_len = 100,
        .data = {0x6b,0x8b,0xc0,0x97,0x1c,0x1c,0x64,0xe8,0xb8,0xdf,
                 0x91,0x39,0x7a,0xb8,0xf0,0xa1,0xf2,0x82,0x3c,0x00,
                 0x4d,0x48,0xa4,0xd8,0xd6,0xb8,0x70,0x5f,0xbd,0xd4,
                 0xe8,0xd2,0x17,0xeb,0x71,0x0b,0x27,0xc8,0xfa,0x56,
                 0xdc,0x29,0x96,0x81,0x9a,0x73,0x6a,0x32,0x3e,0xa3,
                 0xca,0x7d,0x5c,0x88,0x9f,0xb6,0xca,0x30,0x0c,0x84,
                 0x63,0xa0,0x51,0x37,0x05,0xc7,0xef,0x5c,0xdc,0xb5,
                 0x0d,0x8e,0xe3,0x09,0x1a,0x8f,0xa7,0xa8,0xb4,0x97,
                 0x4f,0xb5,0xc8,0xeb,0xd9,0xbd,0xfb,0xb2,0xa6,0x31,
                 0x49,0x04,0x39,0x1a,0xec,0x45,0x3c,0x08,0x80,0xb1,
                 0xc3,0x4b,0x64,0x37,0xd5,0x66,0x63,0x8b,0x29,0xc1,
                 0x94,0x77,0x2d,0x9e,0x7e,0x72,0x4c,0x4e,0x80,0x73,
                 0x71,0xa5,0x71,0x07,0xc7,0xab,0x83,0xa2},
        .data_len = 128,
        .mac = {0x9c,0xd0,0x0b,0x99,0xf7,0x3c,0xa0,0xe8,0xca,0xbb,
                0xa5,0x51,0x3b,0x57,0x56,0x98,0xfc,0xf7,0xa2,0x7a,
                0x39,0x6d,0xc3,0x3a,0xfb,0xab,0x98,0x72,0x58,0x9e,
                0xfa,0x82,0x6b,0x2c,0xb2,0xeb,0x66,0x1a,0xd8,0xc3,
                0x6f,0x7b,0x99,0xd6,0x23,0xf4,0x40,0xa0,0x4f,0x07,
                0x67,0xc2,0x50,0x0b,0x05,0x98},
        .mac_len = 56,
    }, { 	//8	(60)
        .key = {0x57,0xc2,0xeb,0x67,0x7b,0x50,0x93,0xb9,0xe8,0x29,
                0xea,0x4b,0xab,0xb5,0x0b,0xde,0x55,0xd0,0xad,0x59,
                0xfe,0xc3,0x4a,0x61,0x89,0x73,0x80,0x2b,0x2a,0xd9,
                0xb7,0x8e,0x26,0xb2,0x04,0x5d,0xda,0x78,0x4d,0xf3,
                0xff,0x90,0xae,0x0f,0x2c,0xc5,0x1c,0xe3,0x9c,0xf5,
                0x48,0x67,0x32,0x0a,0xc6,0xf3,0xba,0x2c,0x6f,0x0d,
                0x72,0x36,0x04,0x80,0xc9,0x66,0x14,0xae,0x66,0x58,
                0x1f,0x26,0x6c,0x35,0xfb,0x79,0xfd,0x28,0x77,0x4a,
                0xfd,0x11,0x3f,0xa5,0x18,0x7e,0xff,0x92,0x06,0xd7,
                0xcb,0xe9,0x0d,0xd8,0xbf,0x67,0xc8,0x44,0xe2,0x02},
        .key_len = 100,
        .data = {0x24,0x23,0xdf,0xf4,0x8b,0x31,0x2b,0xe8,0x64,0xcb,
                 0x34,0x90,0x64,0x1f,0x79,0x3d,0x2b,0x9f,0xb6,0x8a,
                 0x77,0x63,0xb8,0xe2,0x98,0xc8,0x6f,0x42,0x24,0x5e,
                 0x45,0x40,0xeb,0x01,0xae,0x4d,0x2d,0x45,0x00,0x37,
                 0x0b,0x18,0x86,0xf2,0x3c,0xa2,0xcf,0x97,0x01,0x70,
                 0x4c,0xad,0x5b,0xd2,0x1b,0xa8,0x7b,0x81,0x1d,0xaf,
                 0x7a,0x85,0x4e,0xa2,0x4a,0x56,0x56,0x5c,0xed,0x42,
                 0x5b,0x35,0xe4,0x0e,0x1a,0xcb,0xeb,0xe0,0x36,0x03,
                 0xe3,0x5d,0xcf,0x4a,0x10,0x0e,0x57,0x21,0x84,0x08,
                 0xa1,0xd8,0xdb,0xcc,0x3b,0x99,0x29,0x6c,0xfe,0xa9,
                 0x31,0xef,0xe3,0xeb,0xd8,0xf7,0x19,0xa6,0xd9,0xa1,
                 0x54,0x87,0xb9,0xad,0x67,0xea,0xfe,0xdf,0x15,0x55,
                 0x9c,0xa4,0x24,0x45,0xb0,0xf9,0xb4,0x2e},
        .data_len = 128,
        .mac = {0x33,0xc5,0x11,0xe9,0xbc,0x23,0x07,0xc6,0x27,0x58,
                0xdf,0x61,0x12,0x5a,0x98,0x0e,0xe6,0x4c,0xef,0xeb,
                0xd9,0x09,0x31,0xcb,0x91,0xc1,0x37,0x42,0xd4,0x71,
                0x4c,0x06,0xde,0x40,0x03,0xfa,0xf3,0xc4,0x1c,0x06,
                0xae,0xfc,0x63,0x8a,0xd4,0x7b,0x21,0x90,0x6e,0x6b,
                0x10,0x48,0x16,0xb7,0x2d,0xe6,0x26,0x9e,0x04,0x5a,
                0x1f,0x44,0x29,0xd4},
        .mac_len = 64,
    }, {	//9	(61)
        .key = {0x7c,0x98,0x91,0x2c,0x74,0x42,0x13,0x62,0xe1,0x12,
                0xa2,0xf9,0x8f,0xed,0x9b,0xab,0xe0,0x05,0x7f,0xc7,
                0x78,0xb4,0x45,0x32,0x39,0xaa,0xf5,0xac,0x72,0x4b,
                0x72,0x55,0x53,0x53,0x97,0x70,0xa5,0xbc,0x86,0x66,
                0xb8,0xe1,0x3d,0x0e,0x9c,0xe3,0x6b,0x2b,0x93,0x4c,
                0x81,0x37,0xc7,0xf2,0x0b,0x5f,0x39,0x1f,0x41,0xce,
                0xfa,0xee,0xd9,0x2e,0x9d,0xf8,0x20,0x6c,0xec,0x30,
                0x49,0xbc,0xda,0x0c,0x05,0xde,0xb9,0xe6,0x54,0x9f,
                0xad,0xa1,0x9a,0xa2,0x61,0x8f,0xf5,0x60,0xf8,0x92,
                0xce,0x6e,0x47,0x82,0xae,0xff,0x41,0xcf,0x53,0xa9},
        .key_len = 100,
        .data = {0x74,0xe8,0x93,0x6d,0x83,0xbf,0x3f,0x16,0xb8,0xd0,
                 0x3f,0xb7,0x33,0x84,0xed,0x8f,0x46,0xbd,0x32,0x34,
                 0x3f,0x5d,0xf8,0x35,0x81,0x07,0xe2,0xfd,0xda,0x29,
                 0x3a,0xfa,0x10,0x3a,0x2b,0xff,0xbd,0x40,0x30,0xe7,
                 0x5d,0x96,0xcc,0x7c,0xa6,0xec,0x7c,0x97,0x18,0x8f,
                 0xea,0x88,0xd4,0xeb,0x63,0xb7,0xb1,0x4e,0x8b,0x8c,
                 0x8d,0xee,0x4f,0x8d,0xe1,0x2e,0x1c,0xc6,0x98,0x1d,
                 0x4e,0x6e,0x22,0x3f,0xec,0xc7,0xc4,0x91,0x92,0x46,
                 0x32,0xc7,0xae,0xf4,0x5f,0xd8,0xef,0x14,0x94,0xbc,
                 0xfb,0x06,0xc0,0x74,0x61,0x6b,0x0f,0x4c,0xce,0x8a,
                 0xbd,0x5d,0x83,0xf3,0x2d,0x55,0x06,0x61,0x35,0x7b,
                 0x18,0xe5,0xbc,0xed,0xe8,0x41,0x88,0x2c,0x86,0x92,
                 0x51,0xdb,0x9a,0x33,0x1a,0xc4,0x56,0xdd},
        .data_len = 128,
        .mac = {0x4c,0xc2,0x88,0x18,0x48,0x6b,0xb9,0xb1,0xb5,0x2e,
                0x33,0x3d,0xde,0x71,0xf7,0x3a,0xcc,0x22,0x74,0x88,
                0x45,0x3f,0xd9,0x07,0xc6,0xb5,0x1d,0x34,0x9d,0x67,
                0xaf,0x1d,0xf2,0x9a,0x9f,0x22,0x55,0x32,0xce,0x04,
                0xf5,0x03,0x95,0xfe,0xd5,0x65,0xe9,0x8d,0x78,0x97,
                0x86,0x26,0xdf,0x93,0x46,0x2d,0x3f,0x01,0x2f,0x73,
                0x73,0x34,0x72,0x98},
        .mac_len = 64,
    }, {	// 10	(75)
        .key = {0xfd,0xe1,0x5b,0xa9,0x00,0xe6,0x64,0x8c,0x9d,0x84,
                0x71,0xf0,0x0d,0x9b,0x32,0xd7,0x1e,0x53,0x83,0xb9,
                0x37,0x0c,0x93,0x1c,0x96,0x94,0x45,0x65,0xdd,0x9d,
                0xd6,0xee,0xcd,0x6e,0xc1,0x58,0x51,0xd8,0xdf,0x23,
                0xbe,0x6c,0xd3,0x7b,0x59,0xeb,0xa5,0x55,0x1a,0xfe,
                0xad,0xbb,0xaf,0x27,0xb3,0x64,0xc4,0xf8,0x54,0x88,
                0x82,0x83,0xa7,0xf2,0x55,0x11,0x21,0x57,0xf3,0x17,
                0xb6,0xe6,0x9a,0x65,0x41,0x20,0x98,0xa1,0x26,0xea,
                0x11,0xf4,0x36,0xe1,0xfd,0x03,0xd3,0xed,0x70,0x21,
                0x54,0x19,0x1e,0xc2,0xab,0xa2,0x1e,0x33,0x44,0x4d,
                0xee,0x89,0x31,0xfe,0xb8,0xd8,0x85,0x87,0xd3,0xa5,
                0xfd,0xbe,0x8d,0x9d,0xe6,0xd2,0x28,0x87,0x3a,0xdf,
                0x22,0x77,0x0d,0xbf,0xba},
        .key_len = 125,
        .data = {0xcf,0x65,0x95,0x9c,0x76,0x08,0x26,0xbe,0xcc,0x0d,
                 0x3c,0x4c,0xf7,0x57,0x40,0xd8,0xc8,0xbe,0xbb,0x98,
                 0x35,0xb2,0x6b,0x21,0x0a,0x21,0x97,0x73,0xdb,0x9b,
                 0x9f,0x36,0x3d,0xb5,0xd7,0x43,0x36,0xab,0x95,0x66,
                 0xf1,0x49,0x89,0x62,0xb6,0x0b,0x4d,0x36,0x1a,0x83,
                 0x3d,0x9f,0x73,0xfb,0x89,0xde,0x8f,0x5c,0x89,0xbb,
                 0xa6,0x4e,0x50,0xca,0x8c,0xa1,0x83,0xed,0x2b,0xec,
                 0x6c,0x1a,0x31,0x73,0x4a,0x5f,0x06,0x4e,0xeb,0x23,
                 0x01,0xc8,0x7d,0xae,0xdb,0x06,0xa2,0xec,0x64,0xe9,
                 0xc2,0x6d,0x74,0x34,0xa5,0xb1,0x25,0xf2,0x41,0xa3,
                 0x3f,0x12,0xd0,0x63,0xf4,0x55,0x26,0x48,0xce,0x6a,
                 0x42,0x26,0xe5,0x57,0x23,0xcc,0x55,0x1a,0xdd,0xd1,
                 0xef,0x90,0xeb,0x91,0x2f,0x97,0xc7,0x72},
        .data_len = 128,
        .mac = {0xb1,0xa9,0x13,0x53,0x11,0xd3,0x2f,0x5b,0xd6,0xb7,
                0x3a,0xbf,0x54,0x08,0x8b,0x65,0xfc,0x1d,0x23,0xe4,
                0xbe,0xd1,0xd3,0x38,0x79,0xa7,0x07,0xc5,0xc9,0x3e,
                0x8f,0xba},
        .mac_len = 32,
    }, {	// 11	(76)
        .key = {0x18,0x51,0x9a,0xce,0x34,0x6e,0x2e,0x99,0x87,0xa2,
                0x50,0x38,0xd7,0x64,0x2b,0x7f,0xbe,0xbd,0x3a,0x49,
                0xb9,0x04,0x00,0x5f,0x7b,0xed,0xfa,0x9c,0x87,0xc2,
                0xd2,0x4f,0xfd,0xef,0xd5,0xe1,0xb4,0xc4,0xa7,0xb8,
                0x8a,0x77,0x35,0x5a,0xf8,0x68,0xec,0xd6,0xea,0xa9,
                0x29,0x60,0xcd,0xbe,0xf0,0x3b,0xe9,0xa1,0xe0,0xdc,
                0xf1,0x0a,0x66,0x4c,0xd8,0xec,0x32,0xce,0xe7,0x43,
                0xe7,0xc2,0x0d,0x0d,0x17,0xc9,0x76,0x96,0x6e,0x2a,
                0x5e,0xf9,0x2d,0x74,0x28,0xe4,0xda,0x65,0xaa,0x9b,
                0x36,0x0b,0xcc,0x40,0x70,0x29,0x51,0x51,0x98,0xdc,
                0xf4,0x72,0x56,0x7e,0xb7,0x6d,0xdc,0xcb,0x78,0x77,
                0x19,0x42,0xc5,0xb5,0x30,0xce,0xf1,0xfe,0x28,0xa6,
                0x67,0xe3,0xe5,0x72,0x3f},
        .key_len = 125,
        .data = {0x72,0x10,0x54,0x9b,0x3e,0xdd,0x9a,0x0a,0x70,0x0b,
                 0x06,0x1f,0x65,0xce,0x10,0x43,0x74,0x77,0xd9,0xe5,
                 0xdc,0x95,0xbb,0xc1,0xfa,0x61,0x25,0x62,0xc0,0x59,
                 0xa6,0xe5,0x62,0x2a,0x47,0x01,0x52,0xd3,0x1e,0x44,
                 0x6f,0x08,0x20,0x9f,0x7a,0xd4,0x37,0x25,0xc9,0x83,
                 0x95,0x10,0x3f,0xbf,0x47,0x05,0x7b,0xf9,0x0d,0x99,
                 0x50,0x0b,0x69,0x13,0x75,0x1b,0xf6,0x73,0x7e,0xc2,
                 0xfb,0xb4,0x0b,0x6d,0x40,0x4f,0x40,0x04,0x20,0x00,
                 0x75,0xca,0xe0,0xcc,0x2e,0x85,0x3f,0x43,0x4d,0xde,
                 0x4e,0x03,0x86,0x0a,0x82,0x7d,0x14,0xaa,0x08,0xfc,
                 0xaa,0xf0,0x58,0xe3,0xad,0x04,0x0d,0x35,0xa0,0xa6,
                 0xf4,0x5b,0xe0,0x14,0x63,0x22,0x91,0x2e,0xcc,0x04,
                 0xd8,0xd8,0x91,0xa8,0x4a,0xa0,0x1a,0xac},
        .data_len = 128,
        .mac = {0x2b,0x26,0x49,0xc3,0x99,0xf3,0x71,0x6b,0xf3,0x3f,
                0x79,0x37,0x57,0x9d,0xf1,0x2d,0xea,0xa8,0xed,0x00,
                0xf6,0x56,0xf8,0x24,0x04,0x68,0xa3,0xb0,0x22,0xd4,
                0x11,0xfb},
        .mac_len = 32,
    }, {	// 12	(90)
        .key = {0x04,0xe8,0x66,0xda,0x69,0xea,0x09,0x39,0xb0,0x2a,
                0x4e,0x9e,0x29,0x05,0x2f,0xe6,0xcf,0xd7,0xa5,0xf8,
                0x7d,0x65,0x79,0x4a,0x5e,0x78,0x56,0xa7,0xa6,0xcb,
                0x24,0x2f,0x7f,0x27,0x91,0x9f,0x46,0xcd,0xf0,0xd2,
                0xf8,0x14,0x47,0x88,0xe7,0x53,0xa3,0x67,0xb2,0x01,
                0xaf,0x3f,0x73,0x1b,0x85,0x92,0x3a,0xc6,0xc4,0x54,
                0xbb,0x36,0xe3,0xef,0x43,0xce,0xc5,0x8a,0xf1,0x89,
                0x8d,0x8b,0x22,0x98,0xb3,0x5a,0x2d,0x4d,0x58,0x68,
                0x51,0x37,0xd6,0x71,0xeb,0x8f,0x9c,0xfe,0xec,0xd2,
                0x39,0x2d,0x8b,0xb0,0xb6,0xb4,0x37,0x25,0x29,0x24,
                0xd0,0xe6,0x87,0x6b,0x16,0xfe,0xba,0x9d,0x62,0xb9,
                0xf3,0xf4,0x94,0xc1,0x42,0x15,0x4c,0x87,0x64,0x94,
                0x5d,0xe4,0xdc,0xbb,0x7e},
        .key_len = 125,
        .data = {0x2f,0x77,0xd8,0x33,0x1b,0x2b,0x92,0xc8,0x56,0xc8,
                 0x11,0x88,0x9b,0xab,0x8e,0xdf,0x75,0xc6,0x87,0x5c,
                 0x02,0x4d,0xa9,0x0b,0xf6,0xb2,0xf3,0xff,0xe2,0xd4,
                 0x19,0x2e,0xb7,0x74,0x26,0x82,0x86,0xe8,0x66,0x2c,
                 0x89,0x13,0x83,0x3c,0x67,0x94,0xee,0x6e,0xb4,0x3e,
                 0x80,0x47,0xb7,0xc8,0x62,0x61,0x71,0xc6,0x2a,0x04,
                 0xda,0xd8,0x46,0xf5,0x6e,0x22,0x9e,0x93,0xe8,0xfc,
                 0x75,0x1f,0x4e,0xea,0x90,0x5c,0x2d,0xce,0x9b,0x58,
                 0x26,0x5c,0xc8,0x89,0xa9,0xcf,0xb9,0x1b,0x01,0xda,
                 0xa0,0x89,0x91,0xe2,0xa5,0x6b,0x5d,0x6a,0x88,0x8f,
                 0xcc,0xcf,0x87,0x4a,0xac,0x35,0x82,0x10,0x76,0xc1,
                 0x5d,0x43,0xd3,0x09,0xa6,0x49,0x60,0xc8,0x77,0xe1,
                 0xae,0xd7,0x9e,0xb7,0x8e,0x58,0xfc,0x36},
        .data_len = 128,
        .mac = {0xbf,0x29,0x05,0x1b,0xe9,0x36,0xe6,0xa3,0x24,0xf1,
                0x49,0xda,0x16,0x82,0x36,0xa5,0xaf,0x75,0x84,0xad,
                0x0b,0x8a,0xb1,0xe7,0xc2,0x7c,0xe0,0x3c,0x02,0x76,
                0x88,0x85,0xe2,0x7d,0x06,0x5f,0x26,0x32,0x16,0x67},
        .mac_len = 40,
    }, {	// 13	(91)
        .key = {0xe5,0xcf,0x7e,0xde,0x64,0x0e,0xce,0x05,0xe6,0xe0,
                0x8e,0x64,0x35,0xfa,0x6e,0x75,0x2a,0xde,0xbb,0xe5,
                0x15,0xad,0xe1,0x00,0x5e,0x3c,0x2e,0x6b,0x6d,0x69,
                0xd8,0x11,0xc8,0xb0,0x42,0x5f,0x7b,0xf9,0x7b,0xb4,
                0xbd,0xb4,0x07,0x13,0xd0,0x28,0xe3,0x1c,0x29,0x08,
                0xc3,0x3a,0xd1,0x48,0x9e,0x1d,0x0b,0x2e,0x6c,0x6b,
                0x37,0xac,0x2f,0xb2,0xf6,0xed,0x30,0xa2,0x8f,0x2e,
                0x8b,0x79,0x92,0xcf,0xed,0xbe,0xbb,0xaa,0x9d,0x32,
                0x18,0xa3,0xb9,0x04,0x6e,0x80,0xc3,0x44,0xda,0xfc,
                0x5c,0x9a,0xb4,0x16,0x4e,0x38,0xb8,0xaf,0xd0,0x0d,
                0x68,0x54,0x06,0x3b,0xac,0x59,0xc8,0xcc,0xbc,0x27,
                0xa4,0xa0,0x3f,0xd6,0x26,0xaa,0xb5,0xff,0x56,0x5d,
                0x12,0xcb,0x83,0x60,0xab},
        .key_len = 125,
        .data = {0x0c,0x36,0xca,0x43,0xe7,0xc1,0x13,0xed,0x9f,0xb7,
                 0x16,0x70,0xb3,0xea,0x73,0xbf,0xd6,0x92,0x8c,0x83,
                 0x9f,0x36,0xdb,0x1a,0x82,0xd0,0x8a,0xe0,0xff,0x2c,
                 0x3d,0xae,0x19,0x91,0x33,0xa1,0x0a,0xa3,0x8d,0x1d,
                 0x35,0x88,0xed,0x11,0x5c,0x4a,0x43,0x7c,0x13,0x7c,
                 0xe4,0x30,0x74,0x21,0xdd,0xd6,0x15,0xc9,0x86,0x32,
                 0x37,0xfd,0x5a,0xa8,0x40,0xdd,0x05,0xff,0x6c,0x08,
                 0xbf,0x66,0xbf,0xbc,0xd9,0xb4,0x3e,0x3f,0x95,0xf4,
                 0x5e,0x7d,0x3b,0x21,0xbd,0xf2,0x69,0x2e,0x10,0xca,
                 0xab,0x49,0x5c,0x47,0x4b,0x61,0x6a,0x64,0x6b,0xe6,
                 0x75,0xb8,0x50,0xd0,0x25,0x9c,0x01,0xe2,0xc1,0x90,
                 0x11,0x30,0xa0,0xdb,0xb9,0xdf,0xe0,0x72,0x2a,0x2c,
                 0x5b,0x1b,0x20,0xaf,0xd7,0xd2,0xbb,0xe1},
        .data_len = 128,
        .mac = {0x71,0x9d,0xd7,0x99,0x84,0xd1,0xa7,0x4b,0xce,0xa4,
                0x6c,0xcb,0xba,0x7a,0xe0,0x9f,0xe2,0x46,0xa4,0x77,
                0x09,0xd9,0x93,0xd3,0x15,0x55,0xa2,0x0d,0x57,0xdb,
                0xd5,0xb1,0xbe,0x9f,0x8f,0xe5,0x54,0x73,0xdd,0xbf},
        .mac_len = 40,
    }, {	// 14	(105)
        .key = {0xa6,0x19,0x89,0x53,0x52,0x2d,0x47,0x66,0x73,0x02,
                0x62,0x8c,0xdc,0x70,0x5e,0x09,0x59,0x61,0x8c,0xb7,
                0xe6,0x36,0xde,0x92,0x1f,0x66,0xf9,0x7a,0xf8,0x68,
                0x8c,0x35,0xae,0xd4,0xe0,0xb4,0xfe,0xc5,0xb1,0x97,
                0x94,0x81,0x3d,0xf3,0xc6,0x5c,0x9a,0x52,0x82,0xd9,
                0x4c,0xfb,0x85,0x13,0x17,0x74,0xce,0x5b,0x12,0x46,
                0x53,0x23,0xfd,0x00,0xf2,0x1b,0xd4,0x7e,0xaa,0x99,
                0xa4,0x6b,0x0b,0x3e,0x9e,0x05,0xeb,0xd7,0x6a,0x20,
                0x5b,0x81,0xbe,0x6e,0xda,0x11,0x2e,0xfd,0xc8,0xb2,
                0x46,0x01,0x1d,0xd0,0xd6,0xd4,0x5a,0x35,0x8d,0x3b,
                0xc0,0x72,0xc9,0xeb,0xc0,0x81,0xae,0x4c,0xb4,0xa8,
                0x76,0x7c,0xcc,0xa0,0x07,0x97,0x4b,0xf7,0xcb,0x36,
                0xf3,0xb3,0xbc,0x35,0xbd},
        .key_len = 125,
        .data = {0x29,0x09,0x53,0x25,0x50,0xdb,0x49,0x40,0x48,0x5a,
                 0xd5,0xc1,0x90,0x5a,0x88,0xc7,0x60,0x84,0x80,0xca,
                 0xe0,0xb0,0x38,0x21,0x97,0x96,0xa4,0xc7,0x26,0xa6,
                 0x7e,0x5e,0x36,0x34,0xdb,0x74,0xaf,0xd8,0x01,0x06,
                 0x2a,0x15,0x7c,0x42,0xaa,0x38,0x6f,0x91,0x86,0x83,
                 0x29,0xd5,0xaa,0x8b,0xf8,0xef,0x00,0xdf,0x42,0x85,
                 0x28,0xea,0xd1,0x02,0x6f,0x1b,0x6f,0xde,0xff,0x43,
                 0xb3,0x1f,0x53,0x3e,0x1a,0x20,0xee,0xd5,0x59,0x91,
                 0x4d,0xe3,0xf2,0xbf,0x1a,0xb7,0x06,0x15,0xa2,0xba,
                 0x6a,0xe3,0x89,0x51,0xfd,0x5f,0xbc,0x05,0x38,0xea,
                 0xa8,0xe2,0x06,0x94,0xaa,0x1c,0xd6,0xe1,0xc6,0xf9,
                 0xef,0xce,0x9b,0xea,0x04,0x0f,0x96,0xfb,0x09,0x9b,
                 0x67,0x6e,0x45,0x6a,0xb1,0xa3,0xa7,0x7d},
        .data_len = 128,
        .mac = {0x9b,0x31,0x71,0x4d,0xf3,0x8d,0x74,0xda,0x1d,0x31,
                0xc2,0x01,0x01,0x82,0x3a,0x7a,0x51,0x12,0x95,0x95,
                0xe9,0x68,0x85,0xfe,0x4a,0x3c,0xfb,0x31,0xd5,0xe3,
                0x2c,0x63,0x2b,0x2f,0x0e,0x83,0x18,0xc2,0x33,0x92,
                0xc4,0xe1,0xf8,0x3f,0x18,0x0a,0xa9,0xf0},
        .mac_len = 48,
    }, {	// 15	(106)
        .key = {0x91,0xf9,0xe6,0x9e,0x2b,0xd3,0xa0,0xdc,0x72,0x40,
                0xd5,0x09,0xc7,0xec,0x14,0xc8,0x54,0x27,0xf7,0x9c,
                0xa0,0x30,0x57,0x4d,0x60,0xb4,0xbc,0x8d,0x91,0x92,
                0x17,0xdb,0xc3,0xe1,0xb4,0xa8,0xb8,0x34,0x6a,0xb8,
                0x2d,0x1c,0x15,0xcc,0xf8,0xef,0x46,0x7e,0x53,0xc8,
                0x38,0x6c,0x78,0xcf,0x06,0x98,0x6e,0xbb,0x1c,0x0f,
                0x22,0x95,0xeb,0xc9,0xb9,0xbe,0xe2,0xd1,0x25,0x33,
                0x99,0xa5,0xf6,0x10,0x4f,0xe0,0x73,0x53,0x96,0x16,
                0xea,0xe3,0x4d,0x00,0x44,0xd1,0xcf,0xd9,0xdc,0xdd,
                0x6a,0x07,0x92,0x3c,0x13,0xfe,0x1b,0x98,0x57,0x62,
                0x9b,0x59,0x95,0x6b,0x75,0x23,0x6b,0x8e,0x61,0x9f,
                0x6e,0x5a,0xc0,0x7f,0x1e,0xa0,0x2d,0xbc,0x19,0xd6,
                0x55,0x22,0x8e,0xbd,0x08},
        .key_len = 125,
        .data = {0x8e,0x9b,0x8a,0x2a,0xf5,0xbf,0x4d,0x8e,0xfd,0x51,
                 0xe3,0x32,0x23,0xe3,0x5e,0x69,0xc3,0x72,0x9c,0x2d,
                 0x3c,0xf6,0x84,0x59,0x50,0x38,0x8c,0x19,0xc9,0xe4,
                 0x7e,0x9e,0x62,0xdf,0x7d,0x16,0xe4,0xda,0x43,0xdb,
                 0x90,0x28,0xac,0xea,0xbd,0xcc,0x78,0x98,0xc2,0xd7,
                 0x4c,0x80,0x16,0xf1,0xfb,0xc0,0xb6,0x35,0x04,0x65,
                 0xc7,0x42,0x5c,0x23,0x7d,0x8e,0x6d,0x4a,0x3b,0xfe,
                 0x5e,0xf5,0xfc,0xb4,0x95,0x84,0xf1,0x29,0x7a,0x4d,
                 0x6b,0x7b,0x7e,0x8f,0xfc,0x08,0x5d,0xa7,0xd9,0x3b,
                 0x9f,0x87,0x83,0xa6,0x65,0x63,0xa7,0x51,0x62,0xad,
                 0x42,0x52,0x28,0x44,0x08,0x9e,0xa5,0xe9,0x07,0x1f,
                 0xed,0xf2,0x88,0x07,0x3a,0x42,0x36,0x63,0x30,0x78,
                 0x82,0xf3,0x66,0x67,0x45,0x39,0x23,0xc7},
        .data_len = 128,
        .mac = {0xb7,0x71,0xf3,0x9b,0x76,0xea,0x76,0x4e,0x7e,0x3b,
                0x48,0x8c,0xff,0x14,0xea,0x8e,0x33,0x7c,0xdb,0x17,
                0x3b,0xe0,0xd8,0x60,0x0d,0x2f,0x56,0x5f,0xb0,0x4c,
                0xee,0x85,0x01,0x1a,0x26,0xb5,0xb9,0x22,0x4b,0x30,
                0x16,0x2d,0xc3,0xba,0x48,0xc9,0xb1,0x21},
        .mac_len = 48,
    }, {	// 16	(121)
        .key = {0x2d,0x0c,0xb6,0xd4,0x5a,0x95,0x2e,0x76,0x96,0xba,
                0x75,0xba,0xba,0xc0,0x52,0xa0,0xd4,0x4c,0x5f,0xa7,
                0x7a,0x6d,0x01,0xed,0x5e,0xdc,0x9d,0x97,0x23,0x83,
                0x09,0x73,0x6f,0x3d,0x41,0xdc,0x42,0xdd,0x5c,0xdf,
                0x86,0x71,0xbe,0x9c,0xe7,0xc8,0x8e,0xee,0xe4,0x31,
                0x6b,0x4e,0x26,0x86,0x5a,0xf4,0x11,0x69,0x03,0x40,
                0x9a,0xce,0x1e,0x59,0x58,0xba,0x28,0x14,0xdd,0x49,
                0x5a,0x17,0x46,0xf6,0xdb,0x3e,0x11,0x17,0x35,0x3d,
                0x2f,0xa7,0x06,0x87,0x5a,0x48,0xf8,0x6f,0xa9,0x88,
                0xf2,0x8d,0x62,0x64,0x90,0x64,0x0f,0x0b,0xd1,0x41,
                0xd4,0xe4,0x10,0x66,0xff,0xd1,0x36,0x68,0xe1,0xb6,
                0x2d,0xd6,0xdd,0x35,0x98,0x1b,0xbe,0xbc,0xc1,0xa6,
                0x4d,0xc2,0xd2,0x48,0x24},
        .key_len = 125,
        .data = {0x74,0x15,0xf6,0x37,0x3b,0x8a,0x79,0x48,0x77,0xc6,
                 0x39,0xf0,0x09,0xb3,0xc4,0x91,0x97,0xf3,0x88,0xf9,
                 0x8b,0xd3,0xf0,0xee,0x5f,0x7a,0xd0,0x0a,0x19,0x67,
                 0x03,0xca,0x11,0x1a,0x53,0xe1,0xfa,0x10,0x98,0xd2,
                 0x20,0xf5,0x24,0xbd,0xef,0x16,0x5f,0x79,0xb5,0x15,
                 0xaf,0xe4,0xa9,0xe0,0x9b,0x77,0x2b,0xe3,0x2f,0x1e,
                 0x4d,0x7d,0xa2,0x46,0x93,0xb1,0x36,0x37,0xf7,0x1f,
                 0x60,0xa4,0x30,0x32,0x29,0x80,0x34,0x9a,0xd4,0x14,
                 0xfc,0xfd,0xc1,0x4f,0x87,0xe9,0x91,0x5d,0x21,0x0e,
                 0x8b,0x7b,0xe5,0xaa,0x3e,0x09,0x81,0x44,0x68,0xe0,
                 0x39,0x9d,0x17,0xe7,0x2f,0xe4,0x0e,0xe1,0xe1,0x29,
                 0x6a,0x89,0xf3,0x14,0x86,0xe1,0x2f,0xd7,0x1b,0xc7,
                 0xca,0x61,0xac,0xc9,0xe8,0xd4,0x21,0x3a},
        .data_len = 128,
        .mac = {0x76,0x91,0x2f,0xf0,0xf1,0x80,0xd6,0x2a,0x86,0xa2,
                0xbb,0xf8,0xe1,0xf8,0xd4,0x43,0x8e,0xd5,0xde,0xd0,
                0xcf,0xd3,0xbb,0xbb,0x43,0x03,0x84,0xa6,0x0f,0x18,
                0xd9,0xe9,0xcd,0xeb,0x7e,0x49,0xec,0x43,0xa6,0x13,
                0x67,0x68,0x6b,0x34,0x63,0x36,0x01,0xa5,0xae,0xdf,
                0x8e,0x3d,0x66,0x9b,0xe2,0x82},
        .mac_len = 56,
    }, {	// 17	(122)
        .key = {0x6e,0x1d,0x5f,0x58,0x1e,0xee,0x88,0x4d,0x33,0x0e,
                0x4c,0xd2,0x6b,0x51,0x00,0x7f,0x4e,0x30,0x09,0xb0,
                0x11,0x1c,0xe5,0x81,0xbb,0x12,0x6b,0xf6,0x86,0x70,
                0x53,0x79,0x88,0x07,0xf8,0x8a,0x92,0x71,0x5d,0xb2,
                0x59,0xa4,0x6e,0xb8,0xd3,0xc8,0x89,0x18,0xe4,0x46,
                0x3d,0xb1,0x3a,0x22,0x52,0xbe,0x76,0x8a,0x09,0x07,
                0x8c,0xaf,0xa4,0x59,0x6c,0xd3,0x29,0x33,0xe2,0xa3,
                0x64,0xc2,0xba,0xc8,0x2b,0x0f,0x29,0xb9,0xe6,0x78,
                0x68,0xd3,0x4e,0x5d,0xa2,0xab,0x74,0x1b,0x10,0xa3,
                0x05,0x3e,0x63,0xbb,0xcd,0xf2,0x62,0xaa,0x7b,0xbb,
                0xd4,0x2e,0xa4,0x66,0xa0,0xa0,0x0a,0x40,0x33,0xcb,
                0x1d,0x85,0x22,0xfe,0xc2,0x1b,0x10,0x60,0x54,0x80,
                0xd3,0x84,0xe8,0x02,0xd2},
        .key_len = 125,
        .data = {0x0b,0x8c,0xdd,0x9b,0xc2,0xde,0xc6,0x87,0x48,0xa7,
                 0x99,0xc1,0xc1,0x0a,0x41,0x99,0x07,0x0e,0xf5,0x7b,
                 0xa7,0x97,0x5d,0x7f,0x2d,0x95,0xf6,0x3b,0xa2,0x7b,
                 0xb7,0xdc,0x52,0xf3,0xf5,0x75,0xd3,0xcf,0x84,0x96,
                 0x42,0x43,0x1f,0x21,0xd0,0x52,0x09,0x59,0x7d,0x87,
                 0xf5,0x3c,0x24,0xe5,0x24,0x13,0xb8,0x20,0xcd,0x47,
                 0xcd,0x0e,0xc1,0x76,0x5a,0x58,0x4f,0xc6,0xbd,0x75,
                 0x65,0x79,0xff,0xd9,0xf9,0xc2,0x5a,0x69,0xa7,0xd7,
                 0xf8,0x33,0x9c,0x48,0xfe,0x5c,0xb2,0x8f,0xc9,0x59,
                 0xdc,0xe8,0x77,0x62,0xc5,0x63,0xaa,0x24,0xb4,0x37,
                 0x88,0xb7,0xb3,0x59,0x07,0xcf,0xf0,0x4d,0x67,0x36,
                 0x63,0x3d,0x00,0x7a,0xb1,0x48,0x3d,0xe3,0x51,0x1c,
                 0xe9,0xa7,0xed,0xc7,0xa9,0x61,0x0b,0xca},
        .data_len = 128,
        .mac = {0x29,0x3f,0xab,0xbc,0xfd,0xb8,0x55,0x28,0x93,0x77,
                0xf5,0x12,0xef,0xbc,0x37,0x01,0x52,0xcb,0x87,0x19,
                0x7d,0x12,0xf6,0x61,0xb6,0xca,0xe1,0x8b,0x8d,0x24,
                0x2c,0xbc,0x76,0x7a,0x6f,0x66,0x33,0x77,0x08,0x55,
                0x25,0xf0,0xf2,0x7b,0x39,0x00,0x57,0xdf,0xb9,0x9b,
                0x57,0xb7,0x3f,0x52,0xaa,0x2d},
        .mac_len = 56,
    }, {	// 18	(135)
        .key = {0x13,0xfb,0x1e,0xd6,0x38,0x9f,0x32,0xd1,0xde,0x31,
                0x39,0xcb,0x04,0xbc,0xdd,0x53,0x52,0x5c,0x98,0x89,
                0xb8,0x53,0x79,0xd3,0x53,0x5a,0x25,0xd2,0x90,0x35,
                0x1c,0x95,0x93,0x8a,0x3d,0x0c,0xda,0xf3,0x8d,0xbf,
                0x1d,0x52,0x34,0xbf,0x79,0x65,0xc8,0xdd,0xce,0x9a,
                0xce,0x1b,0x66,0x24,0x7e,0x60,0xd7,0x4e,0xc7,0x70,
                0x2a,0x0f,0x93,0x1a,0x3c,0xdf,0x4c,0xb4,0x65,0xca,
                0x9f,0xc4,0x58,0xc3,0x80,0x00,0x4a,0x3a,0x6e,0x79,
                0x57,0xf1,0xf8,0x13,0x21,0x0b,0x80,0x38,0xba,0x66,
                0x3f,0xcd,0xc4,0x2a,0x89,0x65,0xd6,0xa2,0x52,0xb5,
                0x22,0x4b,0xf2,0x49,0x55,0x2b,0x25,0x75,0xbf,0x64,
                0x56,0x8d,0xb4,0x09,0x1d,0x58,0x32,0x30,0x06,0xc3,
                0xc3,0x49,0x94,0xd3,0xa5},
        .key_len = 125,
        .data = {0x88,0xad,0x81,0x2f,0xd3,0x4e,0x55,0xc8,0x09,0xe8,
                 0x17,0x19,0x96,0x04,0xb4,0xa7,0xf7,0xfe,0xae,0x42,
                 0xcd,0xc4,0xc9,0xe9,0x30,0xdb,0x08,0xe8,0x45,0xa3,
                 0xd7,0x43,0x13,0xdb,0x8a,0x57,0x92,0x67,0x06,0xbf,
                 0x05,0x51,0xbe,0x75,0x8a,0x0f,0xe2,0x39,0xf0,0x04,
                 0xd2,0x37,0xc8,0x49,0xd9,0xe4,0xbf,0xac,0x18,0x29,
                 0x2b,0xf9,0xc0,0xc3,0xe3,0x79,0x85,0xea,0x54,0xb9,
                 0x4f,0x30,0xd1,0x8c,0x32,0xad,0x2b,0x53,0xa0,0x59,
                 0x82,0x7c,0xdd,0xb9,0x5a,0x49,0xb4,0xbe,0xf1,0xd3,
                 0x69,0xea,0xd1,0x4e,0xee,0xb4,0xa1,0x8e,0x59,0x2e,
                 0x40,0xca,0x96,0xe5,0x15,0xa1,0x59,0x08,0xa0,0x5a,
                 0x57,0xcd,0x55,0x70,0xb6,0x11,0xab,0x4e,0xc2,0x3f,
                 0x70,0x57,0xe1,0x72,0x5f,0x29,0xc9,0xde},
        .data_len = 128,
        .mac = {0xa4,0x81,0xe7,0x13,0xcd,0xc8,0x1c,0xa5,0xaf,0xa0,
                0xef,0xcb,0x16,0xe3,0x5c,0xd2,0x0d,0x01,0xaa,0x44,
                0x99,0x58,0xfd,0x2e,0xae,0xde,0x2e,0x25,0xa5,0xba,
                0x54,0x0b,0xea,0xfb,0xa2,0xfa,0xb4,0xad,0xfe,0xf2,
                0xe1,0x46,0xb4,0xc1,0xb2,0xa1,0x83,0x2e,0x93,0xdd,
                0x37,0x3d,0x63,0xfa,0x90,0xbb,0x61,0x49,0x0f,0x65,
                0x68,0x19,0x1f,0x65},
        .mac_len = 64,
    }, {	// 19	(136)
        .key = {0xfd,0x50,0x70,0x36,0x22,0x96,0xc4,0x0d,0x65,0xb1,
                0x05,0xd5,0xab,0x46,0x53,0xfe,0x34,0xe0,0x20,0x05,
                0x16,0x93,0x3f,0x3e,0xea,0xe0,0x3e,0xd0,0xc5,0xd9,
                0xf6,0x01,0x6a,0x85,0x60,0xb4,0xbd,0x86,0xab,0x2f,
                0x7b,0xf9,0x8b,0x22,0x29,0x9e,0xd3,0xe5,0x4a,0x39,
                0x46,0x02,0xd5,0x38,0xaa,0xf3,0xe6,0x95,0x1f,0x2d,
                0xb4,0xfe,0xaf,0x5d,0xc3,0x34,0x26,0xf1,0x5b,0xb1,
                0x24,0xda,0x38,0x8d,0x70,0x90,0x83,0xa2,0x8f,0x57,
                0x01,0xef,0x96,0xc2,0x8b,0x3a,0x3c,0x75,0xbe,0xf9,
                0x33,0x2e,0xf3,0x73,0xb9,0x07,0x71,0x23,0x6a,0xf5,
                0xe2,0x5d,0x58,0x95,0x04,0x34,0x5d,0x28,0xa1,0x9a,
                0xb0,0xdb,0xc1,0xc9,0xb7,0x4d,0x1e,0xe2,0x1c,0x4b,
                0xd8,0xd4,0x23,0xde,0x6a},
        .key_len = 125,
        .data = {0x8d,0x2e,0x68,0xd7,0xe9,0x84,0x6c,0xfa,0x30,0xd9,
                 0x31,0xa3,0x8e,0xfb,0x59,0xbc,0xce,0xd5,0x3a,0x14,
                 0x16,0x4b,0x31,0x63,0xd2,0x65,0x38,0x88,0xee,0xb0,
                 0xbb,0x14,0x48,0xe1,0xa8,0x0c,0x65,0xbc,0xc6,0xeb,
                 0x63,0x34,0x47,0xe7,0x2e,0xd4,0xa0,0x75,0xf7,0x5d,
                 0x98,0x0f,0xe2,0xb1,0x9f,0x35,0xff,0xef,0x62,0xb2,
                 0x7c,0xe0,0x9c,0x20,0x19,0x92,0x2f,0xae,0xdb,0x42,
                 0x73,0x21,0x05,0x7f,0xce,0x19,0x44,0x8d,0x85,0x96,
                 0x2a,0x08,0xd1,0xba,0xdd,0xc9,0x36,0xd1,0x11,0x0e,
                 0x10,0x8e,0x33,0xd4,0x6f,0x97,0xe7,0x88,0x24,0x45,
                 0xb5,0xdf,0x1c,0xa4,0xff,0x03,0xed,0xc2,0x37,0xef,
                 0xaf,0x26,0x4f,0x1c,0x0d,0x9e,0x70,0x5d,0x9b,0x3e,
                 0xee,0x07,0x6b,0xa5,0x7c,0x56,0xdb,0x82},
        .data_len = 128,
        .mac = {0xb6,0xca,0xd1,0xca,0x5b,0xa5,0x05,0x49,0x8a,0x8f,
                0x66,0xa9,0x42,0x2b,0xf5,0x39,0x42,0x6a,0x8a,0x55,
                0x33,0x4f,0xab,0x9c,0x6b,0x9e,0x08,0xe3,0xa5,0x17,
                0x9d,0x15,0x7d,0x1e,0xfa,0x0f,0x91,0xd5,0xc5,0xe2,
                0x6f,0xfa,0x43,0xf5,0xc1,0xcb,0x7c,0xa5,0xf9,0x06,
                0xce,0x4f,0x0e,0xfc,0xf4,0xe8,0x71,0x82,0x0b,0x83,
                0x53,0xe8,0x90,0xe4},
        .mac_len = 64,
    }, {	// 20	(150)
        .key = {0x39,0xca,0xde,0x38,0x95,0xb0,0x7a,0xbb,0x4c,0x10,
                0x0d,0x2a,0xc9,0x75,0x86,0x06,0x9e,0xfd,0x1c,0xfb,
                0xd3,0x5c,0x41,0x09,0x7e,0x23,0xd1,0xe1,0x94,0x43,
                0x90,0x92,0xff,0xcc,0xd3,0x64,0xf1,0xfb,0x7d,0x04,
                0x77,0x74,0x2d,0xe2,0x51,0xcc,0xb6,0xd8,0x01,0x4a,
                0xd0,0x0c,0x22,0xef,0x0d,0x17,0xfb,0xb3,0x0e,0x67,
                0x54,0x12,0xf6,0xe1,0x88,0xc7,0x1f,0xcb,0xdc,0xd8,
                0x0c,0xea,0x0f,0xd2,0x9f,0xe7,0x33,0x56,0x8c,0xf8,
                0xbb,0xd4,0x0e,0xbc,0xd4,0x97,0xd9,0x66,0xf9,0x02,
                0x4f,0x1d,0xc1,0x19,0xf6,0xa3,0xfc,0x43,0x2a,0x35,
                0xff,0x4d,0x0f,0xb3,0x3c,0xb3,0xca,0x01,0x16,0x1b,
                0xf1,0xad,0xf3,0x23,0x3f,0xd2,0x92,0x5c,0x8d,0x3c,
                0x9f,0x96,0xfd,0xcc,0x84,0x5f,0x79,0x1e},
        .key_len = 128,
        .data = {0xff,0xb6,0xb2,0x07,0x1a,0xde,0x0e,0xbb,0xec,0xdf,
                 0xe6,0xdc,0xc8,0xcc,0xd5,0x2f,0xae,0xbb,0x66,0xa1,
                 0x28,0x1b,0x1c,0x39,0x72,0x48,0xb6,0x4c,0xdd,0x5d,
                 0xb0,0xf0,0x75,0x4a,0x0d,0xb2,0xe2,0x26,0x54,0x8c,
                 0x8c,0xc6,0xd9,0x11,0x03,0x8e,0x3d,0xeb,0x9d,0x87,
                 0x38,0x87,0x61,0xb2,0x96,0x0f,0x23,0x9c,0x80,0x99,
                 0xf1,0x33,0xfd,0x9e,0xae,0xd8,0xb9,0xad,0xfd,0xb5,
                 0x04,0x60,0x61,0x51,0x1d,0x90,0xa2,0x61,0xb1,0xc5,
                 0x72,0xe2,0x9f,0xf0,0x50,0x64,0xfe,0x5d,0xad,0xa9,
                 0x61,0xe3,0x4d,0x6e,0x59,0x5e,0xf1,0x89,0x3c,0x5f,
                 0x33,0x34,0x25,0x21,0xb1,0xa9,0x3c,0xbf,0x6d,0xef,
                 0x38,0xa5,0x74,0xf3,0x22,0x44,0xbe,0xda,0x5b,0x0b,
                 0x56,0x96,0xe0,0x92,0xba,0xb7,0xcd,0x18},
        .data_len = 128,
        .mac = {0xd4,0x08,0x0a,0xb3,0x7e,0x75,0x98,0xc7,0x8f,0xb4,
                0xc5,0xe4,0x3e,0x95,0x48,0x30,0x78,0xc1,0xe1,0x5e,
                0x2b,0x01,0x5e,0xdb,0x30,0xa5,0xdf,0xe3,0xc7,0x39,
                0xe5,0xa9},
        .mac_len = 32,
    }, {	// 21	(151)
        .key = {0x12,0xdd,0x35,0x38,0x7a,0x7d,0x0c,0xcb,0xb9,0x1f,
                0x4f,0xc9,0xaf,0xb0,0x87,0xa3,0xd8,0x49,0xb4,0x6b,
                0xd2,0xaa,0xef,0xe7,0x19,0xe4,0x4e,0xe4,0x16,0x55,
                0x08,0xcc,0x58,0xd9,0xd9,0x7b,0x21,0x3f,0xa8,0x4f,
                0x24,0xfa,0x68,0x7e,0x8d,0x19,0x3c,0xad,0x42,0x7a,
                0xde,0xbe,0x68,0x3a,0x41,0xe9,0x2a,0x6f,0x75,0x0e,
                0xb5,0x1a,0xcc,0x49,0x87,0x57,0x3b,0xc8,0xea,0xd7,
                0x02,0xd9,0xbb,0x90,0x8f,0x47,0x70,0xb3,0xa0,0xe7,
                0x5a,0xdf,0xa9,0x6e,0x26,0x94,0x75,0x85,0xaa,0xea,
                0x0f,0x20,0xdf,0x83,0xe3,0x0b,0x29,0xbe,0x21,0xc5,
                0xdf,0x2d,0x62,0x92,0x07,0x4a,0x8e,0xcc,0x61,0x65,
                0xfd,0x7a,0xa8,0xe1,0xdc,0x24,0x14,0xa0,0xd3,0xe5,
                0x23,0xbc,0xe4,0xc1,0x99,0x78,0x97,0x39},
        .key_len = 128,
        .data = {0xfd,0x64,0xd1,0xd0,0x6d,0x7b,0x92,0xb7,0x7e,0x33,
                 0xe3,0x99,0x97,0xec,0xf3,0xfc,0xf6,0x74,0xa5,0x45,
                 0x3d,0x7d,0x36,0xce,0x2d,0x4e,0x21,0x38,0xb1,0xb8,
                 0x3f,0x03,0x1e,0x4c,0x33,0x5b,0xb9,0xcb,0x05,0x35,
                 0x7a,0xde,0x0d,0x9b,0x0f,0x07,0x13,0x44,0x24,0xd8,
                 0xb4,0x68,0xe2,0x63,0x07,0xd2,0xb5,0xf1,0x81,0x37,
                 0x53,0xf7,0x16,0xc6,0xfa,0x45,0xa1,0x00,0xf8,0x81,
                 0x86,0xca,0x71,0xd4,0x96,0xc1,0x5d,0xe3,0x33,0xcd,
                 0xd0,0x01,0xb4,0xf9,0x7e,0xa2,0x0b,0x82,0x7e,0x13,
                 0xa4,0x2d,0x4e,0x69,0x91,0xde,0x18,0xb8,0xe5,0x67,
                 0x33,0x85,0x5d,0x9e,0xed,0x7d,0x46,0xb2,0x88,0x67,
                 0x9a,0xbf,0x82,0x57,0x6d,0x9f,0x1a,0x24,0x87,0x04,
                 0x0f,0x1d,0x53,0xc3,0xab,0xf8,0xf8,0x73},
        .data_len = 128,
        .mac = {0x30,0x87,0x11,0x76,0xe1,0x7a,0x7e,0x99,0xa6,0xfe,
                0x06,0xed,0x5d,0x67,0x9f,0x1f,0x07,0x04,0x9a,0xd4,
                0x45,0xea,0x91,0x22,0xf7,0x36,0xe4,0xf6,0x71,0x51,
                0xaa,0x6a},
        .mac_len = 32,
    }, {	// 22	(165)
        .key = {0xd8,0x6f,0x48,0x54,0x18,0x03,0xcd,0x41,0x1f,0xc3,
                0x4a,0xce,0xab,0xc7,0x8c,0x66,0x0c,0x6c,0x83,0x06,
                0xd8,0x42,0x99,0x44,0xdf,0x25,0x42,0x91,0x61,0xa9,
                0xd4,0xbb,0x7f,0xbc,0x9a,0xb6,0xf0,0x86,0x23,0x9b,
                0xf5,0x94,0xe0,0xf1,0x8f,0x36,0xef,0x2c,0xef,0x25,
                0x31,0x73,0x2e,0x69,0xa2,0xa6,0x3e,0x7a,0xea,0xa0,
                0x21,0xa6,0x2f,0xb9,0x58,0xb2,0x2e,0x8f,0x9f,0xc4,
                0x8a,0x07,0xd7,0x81,0x63,0x8e,0x44,0x72,0x2b,0xa5,
                0xf8,0xac,0xb5,0x9c,0x8f,0xb0,0x8f,0x9d,0x1d,0x90,
                0x92,0xed,0x65,0x53,0xb7,0x6f,0x4c,0x9b,0x2d,0x6a,
                0x8d,0x85,0xa8,0x8b,0xad,0xd1,0xfb,0x2e,0x32,0x43,
                0xe9,0xd9,0x10,0x48,0x38,0x09,0x64,0xdd,0x08,0x32,
                0x64,0xad,0x63,0x01,0x96,0x79,0x15,0xe4},
        .key_len = 128,
        .data = {0x3c,0x28,0x3b,0xad,0xcb,0xc4,0xcc,0x09,0xc1,0x00,
                 0xff,0x01,0x6d,0x7e,0x3c,0x65,0x2f,0xb2,0x78,0x6a,
                 0x37,0x96,0x14,0xfc,0x8f,0x9f,0x01,0x55,0x5c,0xf0,
                 0x29,0xcf,0x61,0xcf,0x0a,0xf6,0xc4,0x55,0xa4,0xe2,
                 0x15,0x69,0x96,0xc4,0x8c,0xef,0x84,0xbe,0x92,0x3c,
                 0xbb,0xf8,0x83,0xcd,0x18,0xf0,0xb3,0x39,0x26,0x11,
                 0xaf,0x65,0x86,0x88,0xc5,0xf7,0x94,0x53,0xc6,0x0d,
                 0x47,0x9a,0x0a,0x2e,0x59,0x43,0xb5,0x81,0xa8,0xc1,
                 0x39,0x3c,0xdd,0x2c,0x1c,0x60,0x4b,0x97,0xfc,0xa4,
                 0x1a,0x9e,0xd0,0xae,0xa4,0x3e,0x70,0x89,0x1f,0xea,
                 0x58,0x54,0x7d,0xda,0xa8,0x37,0x90,0xa7,0x70,0x9c,
                 0x72,0x15,0x2b,0x9b,0x24,0x2f,0x89,0xb5,0x75,0x9a,
                 0x72,0xc6,0x25,0x23,0x47,0x35,0x4b,0x9a},
        .data_len = 128,
        .mac = {0xc0,0x1a,0x4e,0x9e,0x33,0x17,0x74,0xe5,0x49,0xa5,
                0x6d,0x23,0xce,0x49,0x59,0x24,0x9d,0x6a,0x0d,0xb0,
                0x6d,0x8c,0x42,0xa0,0x41,0x5a,0x99,0x69,0x7c,0x96,
                0xc2,0x37,0x98,0xf8,0xc1,0xd9,0x78,0xfd,0xce,0xaa},
        .mac_len = 40,
    }, {	// 23	(166)
        .key = {0x04,0x17,0xb7,0xfd,0xaf,0x3b,0x20,0x80,0x27,0x14,
                0x15,0x68,0x5f,0xd8,0xbf,0xc4,0x7c,0x2c,0x05,0x71,
                0x37,0xf2,0x0c,0x8e,0x83,0x62,0xfe,0x31,0x70,0x5c,
                0x0b,0x58,0x50,0xfc,0xae,0x23,0x18,0x70,0x32,0x40,
                0x09,0x06,0x74,0xa6,0x8f,0x89,0xea,0x86,0x69,0xce,
                0xe4,0x7e,0x0a,0x8a,0x12,0x11,0x3f,0x66,0x47,0x54,
                0x13,0x56,0x7c,0xe9,0xf0,0x26,0x18,0x31,0x06,0x56,
                0x6f,0xbe,0xf0,0x27,0x88,0x07,0x95,0xf6,0x23,0x60,
                0x4d,0xc9,0xfa,0x29,0x46,0x28,0x48,0x45,0x25,0x2e,
                0x88,0xce,0xf0,0xdf,0x7f,0x06,0x2b,0xbc,0x7c,0x91,
                0x41,0x95,0xdf,0xe2,0x26,0x9f,0xd8,0xf4,0x3b,0xc6,
                0x71,0x3a,0x72,0x44,0xcf,0xd8,0xb2,0x73,0xb4,0x13,
                0x7d,0xdf,0x49,0x06,0xd0,0xa0,0x1b,0x97},
        .key_len = 128,
        .data = {0xbd,0xb0,0xcf,0x79,0x96,0x62,0x48,0xa6,0x8d,0x3f,
                 0xa4,0xf9,0xa1,0x22,0xe4,0xc1,0xf1,0x78,0x4e,0x7c,
                 0x16,0x47,0x62,0xf7,0xb6,0x70,0x11,0xb7,0x56,0x31,
                 0x2a,0x98,0x70,0xed,0x15,0xb4,0xc6,0x30,0xc3,0x86,
                 0xf5,0xee,0xe5,0x50,0x4d,0x0b,0x5c,0xb9,0xe0,0x20,
                 0xf1,0xbb,0xd9,0x7d,0x47,0xbc,0x10,0x6a,0x0d,0xfb,
                 0xdb,0xb3,0x78,0x2e,0x26,0x63,0xf1,0x6a,0xc6,0xcf,
                 0x0b,0x42,0x0a,0xd6,0xb7,0x33,0x93,0x68,0xbb,0x40,
                 0x66,0x45,0x60,0xd9,0x40,0x07,0x6b,0x01,0x2a,0x63,
                 0x4f,0xb0,0x4c,0xac,0xa1,0x76,0x06,0x98,0xcc,0x62,
                 0x3d,0x47,0x06,0x22,0xf3,0x81,0x83,0x6b,0x1b,0x40,
                 0xe8,0x9c,0xab,0x6e,0x02,0x9e,0xf2,0xef,0xb8,0x0e,
                 0x2c,0xe2,0x3e,0x9e,0x54,0x24,0x8c,0xb8},
        .data_len = 128,
        .mac = {0x4f,0x88,0x23,0xe7,0x6c,0xf5,0x77,0x7c,0x21,0xdc,
                0x05,0xeb,0xe4,0x07,0x17,0xfe,0x55,0x9f,0xbf,0x9c,
                0xfa,0x82,0x74,0xcc,0x34,0xf7,0x42,0x96,0x2f,0x65,
                0x83,0xc7,0xa9,0x2b,0x26,0x8e,0x5f,0x6d,0x76,0xf3},
        .mac_len = 40,
    }, {	// 24	(180)
        .key = {0x3d,0x8e,0x7d,0x7e,0xc3,0x0c,0x16,0xb7,0x47,0x2e,
                0xe0,0x07,0x8b,0x04,0xbe,0x96,0xa9,0x8c,0xbe,0x06,
                0x49,0x1e,0xf0,0xf8,0x17,0x07,0x79,0xd1,0x75,0x75,
                0xe3,0xbe,0x6c,0x93,0xb7,0xf5,0xe9,0xf4,0x4e,0x3e,
                0x26,0x35,0xe4,0xb2,0x66,0xde,0xee,0xc3,0xe5,0x8a,
                0xad,0x7d,0x0f,0x48,0x04,0x0c,0xf0,0x40,0x87,0x7d,
                0xec,0x9a,0xd4,0xc9,0xb0,0x9b,0x26,0x0f,0x68,0x11,
                0x27,0x21,0x32,0xf2,0x06,0x42,0xd3,0x40,0xca,0x2e,
                0xaa,0x2f,0xe6,0x5b,0xe2,0x8a,0x0d,0x7f,0xeb,0x53,
                0x47,0x52,0x2a,0xaa,0x45,0x95,0xef,0xef,0xf1,0x53,
                0xa8,0x2b,0x4a,0x67,0x55,0xac,0x6f,0x3e,0x3c,0xf0,
                0xf9,0xde,0xde,0xc5,0x8e,0x27,0x29,0xad,0xb4,0x59,
                0xef,0x87,0xe2,0xd7,0x97,0x6c,0xdd,0x2b},
        .key_len = 128,
        .data = {0x1d,0x00,0xe4,0x40,0x02,0x6f,0x7e,0xfd,0x6d,0x88,
                 0x64,0xfb,0xa4,0x8a,0xa6,0x97,0xe6,0xc9,0x1c,0xe0,
                 0x4e,0x01,0x5f,0x93,0xfb,0xea,0x19,0x4a,0x6e,0x0c,
                 0x7f,0xf0,0x33,0xd5,0x1b,0x78,0xc4,0xe9,0xcf,0xce,
                 0xaa,0xb9,0x17,0x0e,0xe5,0x57,0x8e,0xf5,0x8e,0x89,
                 0xb4,0x95,0xbb,0x1c,0xee,0x4c,0xa3,0x7a,0xce,0xf6,
                 0x03,0x7f,0x95,0x62,0xc0,0x89,0xf6,0x03,0xcd,0xce,
                 0x1d,0xe8,0x4f,0xf3,0xf0,0xee,0xff,0xda,0x95,0x35,
                 0xd0,0xbb,0x34,0xd0,0xd3,0x76,0xfe,0xc1,0x57,0x27,
                 0x6e,0x24,0x54,0xfb,0xce,0xaa,0x0a,0x43,0xfe,0x49,
                 0xe5,0xc7,0x71,0x32,0xa4,0x5d,0x78,0x04,0xba,0xba,
                 0xc3,0x3f,0xf4,0x87,0x24,0xfb,0x5d,0xb8,0x97,0xda,
                 0x5f,0x19,0x80,0x3b,0x2a,0x29,0x33,0xb0},
        .data_len = 128,
        .mac = {0x3c,0x27,0x0d,0xf9,0x9f,0x8b,0x2a,0xb1,0x66,0xfd,
                0x29,0xd5,0xfb,0x34,0x7c,0x7a,0xe5,0xda,0xef,0x69,
                0x7f,0xc2,0x0b,0x40,0x8d,0xd6,0xf0,0x1b,0x15,0xc7,
                0x13,0xf1,0xc6,0x26,0x44,0x2d,0xbe,0x5d,0xaf,0xe3,
                0x28,0x7f,0x2e,0x2c,0x86,0x51,0xc3,0x32},
        .mac_len = 48,
    }, {	// 25	(181)
        .key = {0xed,0x8f,0xe5,0x23,0xdd,0xc1,0x93,0x92,0x62,0xfa,
                0xcd,0x6d,0x72,0x2b,0x56,0xbf,0xf6,0x67,0x83,0xd4,
                0x22,0xe8,0x37,0x56,0x02,0x53,0x11,0xb6,0x56,0x3c,
                0xe4,0xcd,0xa0,0xed,0x68,0xc4,0x7b,0x63,0x2f,0x31,
                0x2e,0x9a,0xda,0x82,0x81,0x0b,0x8c,0x9f,0xd0,0x72,
                0x9f,0xcb,0x5d,0xd4,0x96,0xcb,0xea,0xc1,0x53,0x4b,
                0x88,0xd7,0x8b,0x7d,0xaa,0x87,0x76,0x03,0x7d,0x1c,
                0xcb,0x0f,0x7c,0xf8,0xeb,0xc9,0x3f,0x21,0x5c,0xf1,
                0x93,0xfd,0x7a,0xc5,0xd0,0x5a,0x69,0x25,0x67,0xb1,
                0x4e,0xf5,0xbb,0xac,0xbf,0xc3,0x32,0xe5,0x78,0xa9,
                0x8b,0xf1,0x4f,0x75,0x72,0x9d,0x6a,0x08,0x51,0x77,
                0xaf,0xfc,0x6a,0x91,0x7c,0x8a,0x23,0x81,0x98,0xaa,
                0xa0,0xac,0x6a,0x7b,0x97,0xc5,0x69,0xe6},
        .key_len = 128,
        .data = {0xc0,0xff,0xbe,0x82,0xe2,0xaa,0xc8,0x7b,0xf2,0xcb,
                 0xaf,0x24,0x16,0x43,0xe0,0x0b,0x34,0xac,0x99,0x41,
                 0xaa,0x3f,0x43,0x5f,0x40,0xf4,0x02,0xc7,0x5a,0xea,
                 0x8a,0x2c,0x73,0x0a,0x34,0x55,0xc6,0xe8,0x51,0x1d,
                 0x4e,0xe9,0xbe,0xbf,0xf1,0xab,0xb9,0x50,0xf9,0xa1,
                 0xf2,0x8d,0xc3,0xfe,0xe5,0xd7,0xbb,0xd5,0x68,0x7c,
                 0x88,0x7e,0x80,0x38,0x83,0x3b,0x79,0xfc,0x6e,0x1b,
                 0x36,0xed,0x63,0x1f,0xc5,0xb0,0x0a,0x9c,0x36,0xe5,
                 0x0f,0xe0,0xae,0xf1,0xd3,0x18,0xb7,0x01,0x62,0x72,
                 0xda,0x4c,0xa6,0x7e,0x70,0x98,0xda,0xda,0xb5,0xff,
                 0x40,0x0e,0x1e,0xf3,0x17,0xb5,0xed,0x80,0xc8,0xde,
                 0x02,0xd1,0x60,0xb1,0xf4,0xf6,0x42,0x56,0x60,0xe4,
                 0x1e,0x12,0x81,0xbd,0x1d,0xb3,0x01,0x52},
        .data_len = 128,
        .mac = {0xb9,0x4c,0xd9,0x4b,0x82,0xf7,0xcb,0xc2,0xa9,0x92,
                0xb4,0x13,0x49,0x9e,0x94,0x1c,0x03,0x01,0xed,0x87,
                0x14,0xc5,0xe0,0x7b,0x3b,0xad,0xec,0x2b,0xe1,0x79,
                0x75,0x6f,0x99,0x6e,0x33,0x80,0x25,0xfb,0x35,0xdc,
                0x72,0x9f,0x96,0x71,0x7c,0x5e,0xf8,0xb3},
        .mac_len = 48,
    }, {	// 26	(195)
        .key = {0xfc,0x58,0x50,0xb2,0x5c,0xdb,0x1b,0x94,0x31,0x2f,
                0xe0,0x3f,0x7c,0xa6,0xbb,0x31,0x59,0x35,0xda,0xb4,
                0x79,0x1c,0x8b,0xa5,0xd2,0xc9,0x99,0x7a,0x95,0xfa,
                0xd9,0x84,0x61,0x23,0x00,0x70,0xb6,0x0f,0xb7,0x14,
                0xac,0xc9,0xf2,0x69,0xe5,0xbd,0x7d,0xd3,0xc0,0x1b,
                0xd7,0xa9,0xcf,0x7c,0x44,0xf1,0x8c,0xc8,0x1e,0x6b,
                0x47,0x57,0x6c,0xd2,0x63,0x56,0xb6,0x5e,0xc2,0x6d,
                0x17,0x8d,0x7e,0x9e,0x93,0x23,0xda,0x71,0xab,0x6f,
                0x78,0x4e,0x2f,0xd1,0xcc,0x2a,0x54,0x79,0xd0,0x66,
                0x06,0x71,0xf9,0xf3,0x7f,0xde,0x6a,0xbb,0x19,0x83,
                0x69,0x3c,0x60,0x86,0x0f,0x76,0x6c,0x77,0x43,0x08,
                0x6e,0x8e,0xe9,0x96,0x88,0x34,0x56,0x6d,0xe2,0xc5,
                0x07,0x6a,0xfa,0x95,0x93,0x18,0x3b,0x31},
        .key_len = 128,
        .data = {0x85,0xd0,0x9b,0xe1,0x0a,0xd0,0x30,0xa5,0x76,0xa8,
                 0x96,0xeb,0x7c,0x4c,0xbf,0x6d,0x3f,0x6a,0x74,0x60,
                 0x6f,0xfb,0x9a,0xa7,0xb5,0x19,0xb3,0x47,0x52,0x10,
                 0xa8,0x1b,0x03,0xb7,0xc5,0xc9,0x27,0x01,0x05,0xf4,
                 0x23,0xcf,0x90,0xb3,0x06,0x08,0x82,0x42,0x69,0xec,
                 0xee,0xf8,0x9a,0xa5,0x23,0x79,0xf9,0x1f,0x2d,0x2d,
                 0xa1,0x1a,0x9f,0xf1,0x4b,0x7e,0x90,0x0c,0x70,0xee,
                 0xe4,0x64,0xae,0x9f,0xcd,0xf2,0x6d,0x6b,0x90,0x73,
                 0xc2,0x21,0x8d,0x60,0x94,0x20,0x7a,0x21,0xfd,0x24,
                 0xef,0x7e,0xf2,0x5f,0x27,0x56,0x27,0xcc,0x05,0x93,
                 0x2d,0xc0,0x55,0x5d,0xdd,0xc2,0x55,0x8e,0xfa,0x9c,
                 0x73,0x66,0x12,0xb2,0x59,0x81,0xa5,0x93,0x27,0x05,
                 0x8a,0xce,0xff,0x20,0x8d,0x07,0xc7,0xb2},
        .data_len = 128,
        .mac = {0x7a,0x90,0xa3,0xf4,0xa4,0x7d,0xd2,0x62,0xf3,0x02,
                0xd2,0x57,0xd6,0xac,0xb9,0x16,0xff,0x81,0x92,0xb8,
                0xd2,0xd4,0x53,0x91,0x19,0xfd,0x72,0xc2,0xb7,0x81,
                0x21,0x94,0xcf,0xee,0xf3,0x4a,0x8c,0x3c,0xbd,0x32,
                0x05,0x7a,0x74,0x77,0x08,0x5c,0xaf,0x72,0x91,0xff,
                0x7f,0x83,0xac,0xdc,0x1d,0x7b},
        .mac_len = 56,
    }, {	// 27	(196)
        .key = {0xb1,0xa8,0x5a,0xae,0xe5,0x5a,0x9b,0x2b,0xf4,0xe2,
                0x28,0x4e,0x66,0x32,0x82,0xa9,0x5c,0x7e,0x56,0x44,
                0x89,0x38,0xa1,0x35,0x7f,0x3f,0x17,0x71,0x2c,0x54,
                0xc6,0xe2,0xa4,0xcd,0xf7,0xba,0x21,0x8d,0x55,0x06,
                0x47,0x33,0x51,0x86,0xad,0x92,0xc4,0xbc,0x9a,0x62,
                0xd6,0x43,0x0c,0x34,0x20,0x62,0xff,0xac,0x0d,0xe1,
                0xeb,0xea,0x86,0x1b,0x8f,0x49,0xce,0x55,0xed,0x4c,
                0xaa,0x4e,0x96,0xd5,0xc6,0x17,0x27,0x98,0x54,0x25,
                0x14,0xd0,0xf6,0xa5,0xa4,0x30,0xde,0x0f,0x3d,0x1d,
                0x33,0xfe,0xe2,0x1a,0xb7,0x3f,0xfa,0x84,0xa8,0x28,
                0x53,0x7f,0xe4,0x2b,0x66,0x30,0x50,0xab,0xdd,0x2d,
                0x79,0x0d,0x52,0x31,0x7c,0x13,0x5a,0x15,0x24,0x65,
                0xc2,0xfb,0x53,0x71,0x18,0xcd,0xa3,0x75},
        .key_len = 128,
        .data = {0x7f,0xd1,0x2b,0x61,0x60,0x30,0x47,0x9e,0x30,0x66,
                 0xee,0x0a,0x0f,0xa6,0xdc,0x7d,0x40,0xb5,0x0b,0x1a,
                 0x2c,0xe6,0xd6,0xdf,0xaa,0x48,0x5e,0x7a,0x7d,0x1e,
                 0x2d,0x5e,0x1e,0x8f,0x19,0x17,0xbd,0x9f,0x6b,0xda,
                 0x82,0x5e,0x41,0x61,0x18,0x5f,0x31,0xf6,0x6c,0x6c,
                 0x2f,0xcd,0xb2,0x78,0xb1,0x96,0xaa,0xd5,0x11,0x53,
                 0x21,0xa8,0x32,0x67,0xba,0x99,0x66,0xf0,0xee,0xcf,
                 0x8f,0x57,0x52,0x1b,0x85,0x26,0x18,0x73,0xb8,0x19,
                 0xd9,0xa3,0x17,0x77,0x92,0x3f,0x30,0xa2,0xec,0xdc,
                 0x98,0xed,0xc0,0x7f,0x8d,0xaf,0xda,0x56,0xda,0x96,
                 0xfd,0xa6,0xfb,0x3d,0x28,0x05,0x82,0x00,0x18,0xc9,
                 0xb9,0x0f,0x0e,0xc0,0x89,0xc1,0xd2,0x22,0x9a,0x2b,
                 0xf8,0xc3,0xbc,0xf2,0x62,0x8d,0x86,0x5d},
        .data_len = 128,
        .mac = {0xdf,0x5c,0x22,0x8b,0xeb,0xab,0xdd,0x4d,0x48,0xcd,
                0xa2,0x0a,0x86,0x9f,0x12,0xe6,0xd4,0x4f,0x1c,0x88,
                0x1a,0x28,0x83,0x2e,0xd4,0xd5,0xe4,0x04,0xd9,0x18,
                0x17,0xe3,0x96,0x9a,0x36,0x13,0x7f,0x6c,0x06,0x2e,
                0x4c,0x97,0x50,0x2f,0xd8,0x7f,0x48,0x94,0x48,0x08,
                0xa6,0x6b,0xe3,0xbe,0x39,0x23},
        .mac_len = 56,
    }, {	// 28	(210)
        .key = {0xe9,0xe4,0x48,0x0d,0x1c,0x4a,0x62,0x1e,0x0c,0x4e,
                0x15,0x05,0x99,0x25,0x56,0x34,0x7a,0x7a,0xb3,0x4f,
                0xd2,0xb2,0x89,0x91,0x04,0x74,0x76,0x6c,0xc9,0x69,
                0x11,0x6f,0x80,0x40,0xd9,0x6d,0xc5,0xf6,0x6c,0xdc,
                0x44,0x54,0xfa,0x7b,0xcf,0xb9,0xf8,0x38,0xaf,0x19,
                0x19,0x50,0x38,0x46,0x7a,0xb8,0xa1,0x6e,0x1c,0xbc,
                0x12,0xe5,0x98,0xe6,0xfd,0x25,0x0e,0x21,0xb2,0x14,
                0x5f,0x1e,0x2e,0x85,0x9c,0xf7,0x34,0x00,0xbe,0x12,
                0xa0,0xc6,0x97,0x49,0xf7,0x10,0x08,0x47,0x42,0x98,
                0x75,0x35,0x1d,0x5a,0x76,0x97,0x0b,0x9c,0xcf,0x70,
                0x0c,0x2c,0xa3,0xad,0x72,0xe9,0xe4,0xc0,0xf0,0x84,
                0x0e,0x8c,0xf4,0x88,0x15,0x81,0x36,0x98,0x9b,0x08,
                0x91,0xf8,0x67,0x21,0x13,0x50,0x13,0x4a},
        .key_len = 128,
        .data = {0xb8,0x2e,0xef,0xb2,0x08,0x1b,0xd1,0x4d,0xab,0x0e,
                 0x9e,0x34,0x52,0x48,0xa3,0x4a,0xde,0x73,0xf3,0x29,
                 0x18,0x86,0xb9,0x1e,0xa3,0xe8,0xcc,0x74,0x2f,0xd8,
                 0x84,0xf6,0xee,0x0c,0xcd,0xaf,0x4c,0x98,0x79,0xf4,
                 0xdb,0x12,0xdb,0xa5,0x8c,0xf4,0x91,0xaf,0x25,0x41,
                 0xa1,0xd5,0xef,0x6c,0xc8,0xb1,0xaf,0x75,0x0e,0xf5,
                 0xd8,0x55,0x9e,0xf7,0xff,0x9c,0xd5,0x6d,0x8f,0x59,
                 0x99,0x74,0xbe,0x3a,0xec,0xd8,0xc0,0xf4,0xc0,0x8f,
                 0x3a,0xe5,0x0d,0x86,0xf9,0xf8,0x22,0xa1,0xe4,0xca,
                 0x39,0xfd,0x2f,0x0b,0x4d,0x78,0xd2,0x26,0x30,0x73,
                 0x3a,0x24,0xd8,0xd6,0x3e,0xcd,0xf9,0x55,0x54,0x11,
                 0xda,0xf2,0x05,0xa7,0x61,0xc3,0x9e,0xf4,0x6f,0xf6,
                 0x29,0x2e,0x74,0x12,0x9b,0xc1,0x3a,0x7f},
        .data_len = 128,
        .mac = {0x90,0x09,0x3b,0xdc,0xc4,0x5d,0xa7,0x33,0x8b,0xd2,
                0xef,0xe9,0x2e,0x30,0x93,0x3b,0x14,0xf7,0x55,0x82,
                0x73,0x9c,0x74,0x7f,0x75,0x72,0xb3,0x27,0x0b,0x10,
                0x4f,0x33,0xaf,0x0c,0x93,0x9e,0x3c,0x8a,0xe5,0x3b,
                0x20,0x66,0xfc,0x8c,0x97,0xcc,0xf3,0x87,0x85,0xcd,
                0x2e,0xc3,0xd7,0x9e,0x69,0x46,0x49,0x9d,0x36,0x12,
                0x1e,0x44,0xa3,0xe7},
        .mac_len = 64,
    }, {	// 29 (211)
        .key = {0xd3,0xfb,0xd6,0xfe,0x4e,0x35,0x6a,0xc1,0xc8,0xc1,
                0x20,0xd4,0x32,0xd7,0x20,0x4d,0x9d,0x57,0x9b,0x2a,
                0x5a,0x5d,0x0c,0x8b,0x60,0x16,0xbd,0x1e,0xef,0xd3,
                0x8d,0xda,0x73,0x5c,0xf2,0xf0,0xab,0x87,0x3a,0xfe,
                0x0a,0x09,0x16,0x86,0x5e,0x8b,0x58,0xa0,0xaf,0x01,
                0xfc,0xeb,0x6a,0x37,0x65,0xc9,0xbf,0xac,0xea,0xcc,
                0x47,0xa4,0x91,0x6b,0xea,0x79,0x1a,0xfa,0x00,0x32,
                0x40,0xd9,0xb6,0x56,0x3b,0xeb,0xb3,0x03,0x89,0x49,
                0xfc,0x3a,0xee,0x38,0x15,0x7d,0xba,0x59,0x6a,0x9c,
                0x4a,0x20,0xed,0xcc,0xd1,0x87,0xff,0xf9,0x59,0x04,
                0x94,0x5d,0x04,0xb8,0x92,0x52,0x98,0xe9,0x7b,0x64,
                0x3a,0xb2,0x4c,0xab,0x7a,0xf9,0xa5,0x58,0x90,0xa2,
                0x29,0x8d,0xe5,0x02,0x28,0x72,0xd6,0x97},
        .key_len = 128,
        .data = {0xb9,0x67,0xc7,0xd9,0xc0,0xa9,0x41,0xf0,0x2e,0x87,
                 0x72,0x3c,0xf2,0x82,0xea,0xda,0x43,0x47,0xb2,0x81,
                 0x93,0xd3,0xe0,0xbf,0xbe,0xda,0x69,0x85,0x88,0x6a,
                 0x37,0xe6,0x46,0xcc,0x7b,0x1c,0xdb,0xab,0x45,0xcc,
                 0xe6,0x77,0x52,0x8b,0x3a,0x0c,0x24,0xa0,0x8f,0x8f,
                 0x58,0x0b,0x77,0x99,0x35,0xc7,0x93,0x98,0x81,0x4d,
                 0x06,0x72,0x98,0x59,0x2a,0x6b,0xbf,0xf0,0x82,0x48,
                 0xb5,0xa2,0xf0,0xb4,0x8b,0x0d,0x28,0xe4,0xb6,0xa2,
                 0x65,0x77,0x63,0xac,0x5b,0xa0,0x0a,0x8d,0x6c,0x86,
                 0x46,0x4b,0x1e,0xeb,0xe4,0x4c,0xcd,0x0c,0x39,0x5e,
                 0x9d,0xc9,0xb9,0xfb,0xb3,0x06,0xc6,0xca,0xa5,0x51,
                 0xc6,0x68,0x2e,0xc5,0x78,0x69,0x27,0x2e,0x88,0x9a,
                 0xb2,0x6e,0x61,0x89,0xb9,0x1f,0x42,0x48},
        .data_len = 128,
        .mac = {0xbc,0x9a,0x83,0xd7,0x82,0xe5,0x0b,0xa5,0xa8,0x01,
                0x14,0x6f,0x8d,0xa3,0x90,0x95,0xd9,0x23,0x87,0xd7,
                0x59,0xeb,0x4a,0xd5,0x2b,0xbd,0x9e,0x99,0xd9,0xf6,
                0x8f,0x4a,0x0f,0x6f,0x64,0x70,0xc6,0x53,0xc4,0x59,
                0x79,0xc2,0xe1,0x95,0x43,0x80,0x4c,0xed,0x59,0x2e,
                0xe9,0xc5,0x3e,0xb6,0x8a,0x5b,0x1b,0x77,0x46,0xed,
                0x40,0x3e,0xbe,0x67},
        .mac_len = 64,
    }, {	// 30	(225)
        .key = {0x9c,0xe6,0x6b,0xe0,0xe1,0x6f,0x03,0xba,0xae,0x35,
                0x67,0xae,0xb7,0xae,0x84,0x00,0xfe,0x60,0x14,0x99,
                0x99,0x9c,0x7b,0x5a,0xb6,0x68,0xef,0xb0,0xdc,0xbd,
                0xdc,0x69,0x74,0xf3,0x87,0xc6,0x87,0x79,0xf1,0xd1,
                0xc9,0xc9,0xfe,0xf0,0xd7,0x9b,0xd6,0xbb,0xbd,0x59,
                0x8c,0x0b,0xbb,0xd4,0xfe,0x53,0x49,0x35,0xfc,0x34,
                0x58,0x36,0xac,0x4b,0xdb,0x92,0x2c,0x4e,0x86,0xb9,
                0x7a,0x57,0xd5,0xc9,0x91,0x7f,0x51,0xba,0xd5,0xaf,
                0x0f,0xd8,0xb1,0xb3,0x79,0x77,0x7f,0x90,0x50,0xe2,
                0xa8,0x18,0xf2,0x94,0x0c,0xbb,0xd9,0xab,0xa4,0xa0,
                0x65,0x99,0x65,0xf5,0xdb,0x1d,0x68,0x83,0xad,0x72,
                0x49,0x85,0xfc,0xc6,0xcd,0xba,0x5b,0xed,0xc7,0xb9,
                0xd6,0x57,0x3c,0x85,0x33,0x3f,0xc5,0x61,0x77,0x26,
                0x35,0xe5,0xac,0x80,0x7c,0x52,0xe2,0xad,0xcb},
        .key_len = 139,
        .data = {0x64,0x56,0x64,0x3e,0x93,0x19,0x66,0x95,0xb4,0x84,
                 0xf8,0xa3,0x81,0x79,0x48,0x6c,0x3e,0x3b,0x57,0x7a,
                 0x9c,0xc8,0x00,0xd2,0xdc,0x69,0x36,0x28,0x37,0x87,
                 0x8d,0x4f,0x7e,0xc0,0xfb,0xf3,0xfe,0x3a,0xe0,0x8a,
                 0xa6,0x37,0x45,0x88,0x6c,0xea,0x61,0xd2,0xec,0x8a,
                 0x62,0x76,0x52,0xa4,0x6a,0x99,0x7b,0xb5,0xd7,0xb1,
                 0x57,0xf8,0xc7,0xf4,0x92,0x7d,0xdb,0x0f,0x73,0x7b,
                 0x3c,0x1c,0x04,0xe7,0xdc,0xce,0x73,0x45,0xff,0xef,
                 0xb8,0xbf,0xf9,0x0d,0x78,0x74,0x39,0x70,0x29,0x12,
                 0x86,0x4f,0x78,0xa7,0x89,0x43,0xe7,0xb4,0x8c,0xfb,
                 0xbb,0x84,0x81,0x32,0x15,0xbb,0x46,0xde,0x5f,0x32,
                 0x27,0xaa,0xe9,0x02,0xa5,0xa7,0xd4,0xdf,0x75,0x3e,
                 0x30,0xa8,0xcc,0x6a,0x61,0x3b,0xda,0x24},
        .data_len = 128,
        .mac = {0x27,0x97,0x3b,0x34,0x57,0xc6,0xbd,0x41,0x36,0xa3,
                0x3a,0xc6,0x1d,0x41,0xd5,0xdd,0x43,0x95,0xdb,0xd0,
                0x48,0x7d,0xa6,0x5a,0x0e,0x4e,0xb3,0x69,0xd9,0x48,
                0x2d,0x2b},
        .mac_len = 32,
    }, {	// 31	(226)
        .key = {0x86,0x47,0x45,0x38,0x89,0x47,0x6b,0x94,0x4e,0xaf,
                0x55,0xb9,0x7b,0x9a,0x7b,0xcc,0xec,0x87,0x36,0x57,
                0x55,0x67,0x80,0xfa,0x29,0xf4,0xfb,0x5e,0xbb,0x45,
                0x36,0x6b,0xa4,0x9f,0x2b,0xbb,0x64,0x8f,0x0c,0x4e,
                0x4c,0x35,0x3f,0x7f,0x9b,0xe3,0xa7,0x36,0xe7,0xe7,
                0x25,0x60,0xbe,0xa4,0x5e,0x9c,0x8e,0xe8,0xbf,0x37,
                0xc2,0x79,0xbf,0x5b,0x2e,0xf1,0x64,0x83,0xad,0xcc,
                0x09,0x32,0x08,0xc0,0x5e,0xe5,0x1a,0x4d,0xb0,0x46,
                0x32,0x94,0x6b,0xa2,0xb9,0x6c,0xdd,0x9d,0x15,0xb3,
                0x3c,0x25,0xcc,0xe2,0xeb,0xa4,0xed,0xe4,0xf9,0x7a,
                0xac,0x29,0xeb,0xaa,0x4c,0xf6,0xbb,0xd3,0x42,0xff,
                0x20,0x63,0x97,0x37,0x10,0x07,0x8e,0xe7,0x83,0x66,
                0x87,0xfd,0x7b,0x0e,0x23,0xa7,0x41,0xae,0xa9,0x21,
                0xf1,0x87,0xa8,0xcc,0x38,0x1d,0xde,0x7f,0x5d},
        .key_len = 139,
        .data = {0x8d,0x4f,0x4a,0x89,0x6a,0x5d,0x6f,0x68,0x1c,0x95,
                 0x1d,0xa1,0xee,0xe6,0x14,0x3c,0xd8,0x3a,0x27,0x17,
                 0x50,0xfb,0xa8,0x87,0x62,0xd5,0x21,0x42,0x03,0xbe,
                 0x44,0x7d,0xa3,0x34,0x25,0x51,0x01,0xc6,0xa7,0x63,
                 0x43,0xd6,0x34,0xc4,0x46,0x9d,0xb2,0x16,0x33,0x70,
                 0xb2,0xf1,0x53,0x41,0xea,0x85,0x24,0x68,0xe5,0x80,
                 0xac,0xa4,0xf9,0x32,0x0d,0x5c,0x7a,0xee,0x5a,0x2b,
                 0x2d,0xb4,0xa9,0x9c,0xff,0x0e,0x69,0x32,0xf7,0x38,
                 0xf6,0xac,0x6a,0x83,0x6b,0x86,0x6e,0xfb,0xb8,0xc3,
                 0x90,0x48,0xf4,0xea,0xca,0xd2,0xf4,0x4f,0xaf,0x29,
                 0x1c,0x93,0xe9,0xa3,0x75,0x6e,0xe5,0x47,0x00,0xac,
                 0xcd,0xe9,0x4a,0x76,0xb7,0x97,0x41,0xd3,0x1c,0x34,
                 0x46,0x6f,0x8b,0x63,0x83,0x9a,0x9e,0xa7},
        .data_len = 128,
        .mac = {0x54,0xa6,0xfe,0x35,0x15,0xe8,0x42,0x98,0xc9,0xe3,
                0xb1,0xf2,0xeb,0x6f,0x23,0x8f,0xad,0x2c,0xf6,0x1f,
                0x4e,0x2e,0x96,0x86,0x46,0x4e,0x08,0x7e,0xce,0x45,
                0x67,0xc3},
        .mac_len = 32,
    }, {	// 32	(240)
        .key = {0xf3,0xe8,0x1c,0x3d,0xcf,0xa5,0x30,0x5c,0x9f,0xf1,
                0x5e,0xdf,0x30,0x31,0x03,0xfd,0xed,0xf7,0xfc,0x73,
                0xaf,0x84,0x15,0x25,0xd7,0x8b,0xc4,0x09,0x9b,0x38,
                0x18,0x41,0x9d,0x71,0xec,0x87,0xc2,0x5c,0x60,0xce,
                0x41,0x21,0x8a,0x26,0xfa,0xf1,0x68,0xe1,0x0a,0x47,
                0x19,0xf4,0x9c,0x6d,0x4d,0xa0,0x01,0x43,0xfb,0x51,
                0x04,0x3c,0x52,0xc6,0x77,0xa9,0xf1,0x72,0x12,0x3a,
                0x5a,0x22,0x7e,0xc1,0xa4,0xe1,0xac,0x3a,0x71,0x18,
                0x6b,0x09,0x20,0xfa,0x3a,0x82,0x44,0x1f,0x5a,0xae,
                0x54,0x6f,0x28,0x4c,0xa4,0x42,0xaa,0x17,0x93,0xc6,
                0x68,0x47,0x06,0xa0,0x7d,0x5a,0x16,0xd6,0xed,0x14,
                0x06,0xec,0x39,0xc5,0x78,0x4d,0x31,0xca,0xe0,0xed,
                0x0a,0x56,0x38,0x2d,0xb0,0x73,0xf6,0xff,0x5d,0x43,
                0x3b,0x4a,0x6b,0x8c,0x4d,0x90,0x67,0x73,0x47},
        .key_len = 139,
        .data = {0x15,0x5e,0x66,0x93,0x6d,0x31,0x9f,0x16,0x9d,0xb5,
                 0x19,0xd5,0x17,0x63,0xf9,0xb4,0x3d,0xe5,0xd5,0x41,
                 0x77,0xd5,0x68,0x97,0x83,0xc8,0x8b,0x82,0x0d,0x8b,
                 0x61,0xe3,0x80,0xac,0xd1,0x56,0x1b,0x3c,0x39,0x13,
                 0x47,0xd4,0x60,0x12,0x28,0xd6,0xad,0x73,0x72,0xf5,
                 0x97,0x1c,0x7e,0xf8,0x5d,0xa4,0x9d,0xba,0xf7,0x70,
                 0xac,0x76,0x4c,0x1b,0xe8,0x41,0xa5,0x1b,0x04,0xd8,
                 0x62,0xa2,0x79,0x9c,0xec,0xc3,0x1e,0xdb,0xea,0xd6,
                 0xf8,0x51,0xb8,0x1d,0x53,0xef,0x14,0xa8,0x11,0xdb,
                 0x1b,0x75,0x43,0xb7,0x75,0xdc,0xf6,0x26,0xcc,0x2a,
                 0x4f,0x8c,0x82,0x8d,0xdb,0x16,0xa3,0x30,0x20,0xfb,
                 0x18,0xa6,0x78,0x01,0x1e,0x8c,0x1f,0x42,0xf7,0x6a,
                 0x8a,0x30,0xdf,0xdb,0xfa,0xd7,0x42,0xee},
        .data_len = 128,
        .mac = {0x62,0x51,0x62,0x8d,0x0e,0x65,0xb0,0xaa,0xc3,0x04,
                0x69,0x5a,0x20,0x59,0xfb,0x7c,0xdd,0x6e,0x78,0x7b,
                0x6d,0x37,0x87,0xea,0x54,0x4a,0x4a,0x53,0xe8,0x61,
                0xbf,0x54,0x67,0xd9,0xd3,0xfa,0xa8,0xcc,0xa1,0x3e},
        .mac_len = 40,
    }, {	// 33	(241)
        .key = {0xfb,0xbb,0xdd,0x42,0xe5,0xfc,0x63,0x19,0x68,0x98,
                0x5c,0x06,0x57,0xec,0x42,0xef,0x0d,0xb1,0x7a,0xf0,
                0x49,0x7e,0xcf,0xfe,0x3d,0x8e,0x1f,0xca,0xe1,0xc4,
                0x54,0xe8,0x8d,0xec,0x96,0x31,0xc7,0x44,0xc0,0x66,
                0x5a,0x3b,0x10,0x16,0xa8,0x78,0x22,0xc1,0x40,0x1e,
                0x1f,0x3d,0xb3,0x4f,0x32,0x59,0xa4,0xef,0x3b,0xc1,
                0x1f,0xaa,0xc8,0x21,0x2c,0x38,0x41,0x8d,0xa3,0xdf,
                0x62,0x21,0x41,0x3a,0xee,0xe4,0xfc,0x3a,0xa2,0xf2,
                0x94,0x6a,0x68,0xdb,0x6b,0x77,0xce,0x46,0x96,0xef,
                0xb6,0xda,0x34,0x49,0x4f,0x5e,0xfe,0x26,0x44,0x1e,
                0x20,0xd6,0x33,0x91,0x48,0x1e,0x60,0x3a,0xfc,0x38,
                0xab,0xcc,0x30,0x1c,0xbf,0x91,0x18,0xfe,0x8d,0x0c,
                0x3b,0x18,0xf9,0xde,0xc9,0xa9,0xd2,0xe6,0x2e,0x3e,
                0x08,0xac,0xe7,0x30,0x19,0x00,0x94,0x6e,0xcd},
        .key_len = 139,
        .data = {0x7a,0x03,0xff,0x37,0x37,0xa8,0xb2,0x6d,0xe4,0xf9,
                 0xfa,0x29,0x3b,0x94,0x89,0x9c,0xb9,0xd5,0xd9,0xb2,
                 0xac,0x9f,0xd5,0xf2,0x8c,0x59,0xd6,0xa7,0x8e,0x36,
                 0xd0,0x3d,0x77,0xba,0xce,0xed,0xae,0x7a,0x9b,0x9d,
                 0x96,0x23,0xc2,0x01,0x1a,0xbd,0xb9,0x07,0x8a,0x31,
                 0x5a,0x72,0xa5,0x09,0x92,0xc4,0xf7,0x78,0x5d,0x62,
                 0x65,0x9a,0xf2,0xf3,0x06,0xfc,0x3a,0x09,0x34,0x5f,
                 0x87,0x03,0xe3,0xb9,0x83,0x32,0x32,0x7d,0x67,0x3a,
                 0x40,0x1c,0x6d,0xbb,0x41,0xcc,0x87,0x31,0xd1,0x88,
                 0x51,0x19,0x87,0x58,0x44,0x56,0xce,0xd2,0x2d,0xd2,
                 0xf0,0xe1,0xde,0x68,0x74,0xc5,0x24,0x02,0xaa,0x5b,
                 0xf9,0xfe,0x84,0x9f,0xfa,0xd7,0xa7,0x6f,0x1b,0x01,
                 0xc2,0x92,0x99,0x14,0x1f,0xf8,0x30,0x2d},
        .data_len = 128,
        .mac = {0x0f,0x19,0xb0,0x0a,0x7f,0x9c,0x96,0xa0,0xd8,0x8f,
                0xba,0x43,0xcc,0x55,0xcb,0xd0,0x4c,0x0d,0xce,0x84,
                0x4a,0x94,0x53,0x20,0xc0,0x41,0xe3,0x6c,0x3f,0x8c,
                0x5b,0x5a,0xf5,0xeb,0x9f,0x38,0xed,0x7b,0x07,0x1c},
        .mac_len = 40,
    }, {	// 34	(255)
        .key = {0x77,0xb6,0x0a,0x49,0x89,0xc4,0x51,0x60,0xc1,0xe2,
                0xe4,0x19,0x78,0x53,0x0e,0x6b,0x5b,0x62,0xf9,0x9f,
                0x1c,0x48,0x0c,0x57,0x67,0x50,0x76,0xce,0x18,0x39,
                0x0b,0x61,0xab,0xdc,0x50,0x4f,0xf3,0x0f,0xd1,0xbd,
                0x0f,0xc8,0xbb,0xfd,0x9b,0x86,0xa6,0x0f,0x23,0xec,
                0xb3,0x68,0xdd,0xaf,0xd7,0xf3,0x97,0xb3,0x76,0x56,
                0xee,0x71,0x34,0x76,0xc1,0x9c,0x08,0x3d,0xe5,0x04,
                0x92,0x9b,0x1a,0xba,0xae,0xca,0x3e,0x75,0x49,0xd7,
                0xc1,0x34,0x73,0x5e,0x59,0x25,0xb6,0x95,0xf7,0xc6,
                0xa6,0x8b,0xf9,0x4a,0xba,0xb2,0xa1,0x3e,0x5a,0x9c,
                0xc6,0x6c,0x39,0x02,0xc9,0x00,0xe5,0x0a,0xcb,0xe9,
                0x9d,0xc9,0x1d,0x82,0x62,0x07,0xf8,0x72,0x50,0x43,
                0x6f,0xd1,0x2f,0xf7,0xa1,0x8c,0x46,0x1e,0x33,0x0a,
                0x6f,0xf2,0xfe,0x0f,0x71,0xfa,0x04,0xe1,0x89},
        .key_len = 139,
        .data = {0x53,0x4c,0x85,0xdd,0x76,0xba,0xf3,0xaa,0x0e,0x3c,
                 0xd3,0x1a,0xce,0x04,0x9d,0x93,0x1b,0x39,0xec,0x18,
                 0x78,0x9d,0x8d,0x10,0x42,0x6e,0xd6,0x49,0x9d,0x8a,
                 0x39,0x3c,0xae,0xd6,0x19,0x93,0x0b,0xdf,0xdb,0xe8,
                 0x6f,0xc2,0x41,0xd0,0xb3,0x4a,0xf3,0x18,0xf9,0x59,
                 0x5f,0x4e,0x2b,0x89,0xc3,0x83,0xa4,0x12,0x88,0x50,
                 0x2c,0xef,0xd2,0x17,0x2a,0x3c,0x55,0x8b,0x15,0xe3,
                 0x6a,0x73,0x2c,0x77,0x62,0xab,0x67,0x60,0x1a,0x6b,
                 0xad,0x39,0xcf,0xcb,0x47,0x76,0x34,0x87,0x95,0x4a,
                 0xc2,0x00,0xff,0xc8,0x50,0x84,0x2f,0x48,0xe3,0xcf,
                 0x7d,0x0c,0xc7,0xd2,0xab,0x23,0xdf,0xb3,0xd3,0x8e,
                 0x39,0x14,0x9d,0xa2,0xe5,0x98,0xb5,0xad,0x08,0xa3,
                 0x7d,0xdc,0x2c,0x62,0xff,0x5c,0xda,0x0c},
        .data_len = 128,
        .mac = {0x15,0x77,0x56,0x1f,0x5f,0x3c,0x0b,0xf5,0x23,0xcd,
                0x8c,0xc4,0x70,0xe9,0xc1,0xb9,0x50,0x7b,0xc6,0x99,
                0x1a,0x53,0xdf,0xe2,0x73,0x5f,0x03,0x3b,0xbf,0x88,
                0x1c,0x51,0x3b,0x31,0x53,0x17,0x25,0x02,0xc6,0xcf,
                0x01,0x94,0xfc,0x00,0x98,0x0e,0x1f,0xdc},
        .mac_len = 48,
    }, {	// 35	(256)
        .key = {0xa1,0x8a,0x27,0x74,0x8e,0xf3,0x9b,0x49,0xbe,0x98,
                0x4e,0x8d,0x18,0x52,0x01,0x10,0x00,0x8b,0xc8,0xa1,
                0xd5,0xae,0xb4,0x24,0xbe,0xdc,0xae,0xe5,0xa7,0xe1,
                0xa6,0x2c,0x86,0x66,0xee,0x12,0xe3,0x67,0xe0,0x92,
                0x97,0xe8,0xc7,0xe3,0xd4,0xe4,0xfd,0x05,0x65,0x87,
                0x50,0x9b,0x37,0x9d,0xaa,0xf8,0x19,0x49,0xf2,0x7c,
                0xc0,0xfa,0x2d,0x21,0x0e,0x9b,0xe9,0x51,0x94,0x0a,
                0xdb,0xfb,0x55,0xcc,0xc7,0xe5,0xcc,0xff,0xa0,0x44,
                0x31,0x8f,0xf1,0x8a,0xf9,0xad,0x7b,0x7f,0x9c,0x7d,
                0x1f,0x93,0x9a,0x0f,0xff,0x72,0xc0,0x91,0xe1,0xda,
                0xa7,0xc3,0xd4,0xa9,0x7f,0xab,0x15,0x3b,0x0a,0x89,
                0x33,0xf2,0xeb,0x0d,0x72,0x16,0x21,0xc8,0x6d,0xe0,
                0xcf,0xe1,0x00,0xd1,0x3e,0x09,0x65,0x48,0x24,0xb0,
                0x9d,0x54,0x27,0x79,0x12,0xc7,0x9d,0xec,0x7a},
        .key_len = 139,
        .data = {0x29,0xc4,0xae,0x34,0x84,0xdc,0x27,0xc0,0x36,0x03,
                 0x43,0xfd,0x0b,0x20,0x58,0xba,0x26,0x1a,0xc3,0xac,
                 0xf6,0xf8,0x7f,0xb5,0x66,0x47,0xf6,0x65,0x54,0xbc,
                 0x16,0xc2,0x45,0x1b,0xa8,0x44,0x57,0x57,0xdd,0x24,
                 0x77,0xfb,0x2a,0xd7,0xd3,0xc8,0x56,0xd5,0x92,0xa0,
                 0xd2,0x9e,0xc3,0xa3,0x48,0xff,0x94,0x97,0x76,0x91,
                 0xc5,0x8d,0x3d,0x84,0x5c,0xc1,0xf5,0x9a,0x99,0xc3,
                 0x04,0x76,0x2c,0xd4,0xaf,0x17,0xa6,0x93,0x30,0xa0,
                 0x2e,0xd9,0x08,0x5a,0x75,0xe1,0x96,0xfb,0xec,0xac,
                 0xe9,0x29,0x34,0xa3,0xd3,0x3a,0xd5,0x7f,0x3e,0x3e,
                 0x34,0x66,0xc3,0x3f,0xbf,0x56,0x79,0xc7,0x6b,0xc7,
                 0x0b,0xa3,0x60,0x8c,0x0e,0xe7,0xf2,0xfb,0x91,0x32,
                 0xd4,0x4d,0xf5,0x33,0x88,0x48,0xff,0x6c},
        .data_len = 128,
        .mac = {0x26,0xa3,0x7c,0x97,0xb7,0xc1,0x30,0x97,0xc5,0x73,
                0x5b,0xc3,0x1f,0x8b,0xdc,0xdc,0x27,0xdf,0xa9,0x6e,
                0x99,0x12,0x19,0xe7,0x08,0xc2,0x3d,0x1b,0x32,0xc0,
                0x4d,0x88,0xa8,0x9a,0x86,0xa1,0xc2,0x17,0x72,0x4f,
                0x4c,0xe5,0xbb,0x58,0x0a,0x1a,0x71,0xc0},
        .mac_len = 48,
    }, {	// 36	(270)
        .key = {0xa0,0x8e,0x14,0xc2,0x67,0x1e,0xf7,0x9f,0x81,0xf7,
                0x32,0xdf,0x2d,0xf3,0x83,0x5a,0xb0,0xcd,0x0d,0xb9,
                0xe1,0x90,0xfb,0x88,0xeb,0x4f,0x66,0x8c,0x4f,0x17,
                0x78,0x56,0x28,0x22,0xd2,0x18,0xb3,0x4d,0x2c,0xb8,
                0x0e,0xcc,0x3b,0x23,0x40,0x1d,0xd8,0xe4,0x7a,0x3a,
                0x5e,0xf5,0x9d,0x13,0x54,0xd4,0xfb,0x3b,0x4c,0xda,
                0x62,0xe2,0x0a,0xc9,0x57,0x03,0xe9,0xa4,0x9f,0xde,
                0x7b,0xc3,0x04,0xa9,0x70,0x26,0x55,0x83,0xb9,0x0a,
                0xaf,0xa9,0xed,0xba,0xfb,0xec,0xeb,0xa8,0xb8,0x63,
                0xc1,0xbc,0xfb,0x5d,0xcc,0xec,0xb3,0x99,0x21,0x0d,
                0x32,0xba,0x2c,0xe8,0xe9,0xf5,0x97,0x91,0x73,0x0d,
                0x6d,0xf4,0xd6,0x81,0x80,0xb1,0x05,0x04,0x08,0x2a,
                0xa9,0x2d,0x8c,0xd9,0x20,0x70,0x06,0xe8,0xc6,0x4d,
                0x42,0xfb,0x74,0xc7,0x87,0x51,0x47,0x18,0x79},
        .key_len = 139,
        .data = {0x0f,0xd5,0xe5,0x3b,0x97,0x97,0xcc,0x3e,0xa7,0x5f,
                 0xcb,0x7e,0x70,0xa9,0x3b,0x80,0xb5,0x40,0x91,0x76,
                 0x2b,0x0b,0xdf,0xb7,0x25,0x2a,0x9e,0x6d,0x70,0x42,
                 0xaa,0x8d,0x7c,0x14,0x8e,0x0d,0xbd,0x55,0x02,0x51,
                 0x51,0xb9,0xb2,0xe6,0xe3,0x52,0x41,0x72,0xcf,0x18,
                 0x8e,0xeb,0x87,0xc9,0x36,0x08,0x35,0xf3,0x5a,0xf5,
                 0xa2,0x4e,0xa2,0x4b,0x56,0x50,0x81,0x3f,0x01,0xfc,
                 0xbb,0x1a,0x19,0xfb,0x25,0xe9,0xc5,0xed,0xac,0x75,
                 0xfa,0x01,0x99,0x75,0xe3,0xc3,0x14,0x03,0x87,0x36,
                 0xe6,0x4d,0xa6,0x23,0x83,0x8d,0xe3,0xb0,0x47,0x3b,
                 0x29,0x34,0x00,0x80,0x47,0x4a,0xad,0xef,0xdd,0x25,
                 0x85,0x70,0x7c,0x23,0x3b,0x7c,0x09,0xa4,0x81,0x90,
                 0x62,0x1c,0xcb,0xdb,0x44,0x67,0x55,0x3a},
        .data_len = 128,
        .mac = {0xb0,0x7d,0x4e,0x89,0xd7,0xfb,0xca,0x2d,0xb3,0x35,
                0xe8,0x07,0xf9,0xec,0x13,0x1d,0xbc,0x75,0xb8,0x78,
                0xc5,0x13,0xf6,0xcf,0x65,0x95,0xd7,0x54,0x55,0x71,
                0xfd,0x13,0x61,0xd8,0x22,0x9d,0xfa,0x69,0x49,0xcd,
                0x97,0xde,0x56,0x44,0x62,0x93,0x7b,0x35,0xb9,0x23,
                0x20,0xe3,0xab,0x30,0xd9,0xd9},
        .mac_len = 56,
    }, {	// 37	(271)
        .key = {0xec,0x46,0x98,0xb6,0x8d,0x26,0xf2,0xfc,0x04,0x28,
                0xf4,0x13,0xa0,0xef,0x0d,0xc4,0xd6,0xb0,0xe6,0x23,
                0x3e,0x2e,0x8b,0x1c,0xde,0xe8,0xcb,0x4f,0xab,0x90,
                0xe1,0x1e,0x4d,0xc0,0x54,0x03,0x23,0xe9,0x1b,0x27,
                0x87,0x8c,0x05,0x31,0x53,0xac,0x58,0x5c,0xa3,0x83,
                0xb8,0xcd,0xdd,0x74,0x4b,0x23,0xef,0x41,0x1b,0x4f,
                0xd8,0x7e,0xbc,0xa6,0x6a,0x45,0x2e,0x34,0x4e,0x2c,
                0x04,0x54,0x48,0x74,0xc6,0x7e,0xbc,0x83,0x95,0x5f,
                0x72,0x94,0x0d,0x2f,0x96,0xae,0x70,0x3f,0x03,0xd2,
                0x00,0xd1,0xc1,0x79,0xac,0x2d,0xcb,0x3e,0xef,0xf1,
                0x16,0xd7,0xf6,0xa9,0xd0,0x49,0x01,0x9f,0xe5,0x5c,
                0x0b,0xf5,0xc8,0x4d,0xad,0xf0,0x70,0xc4,0x40,0x97,
                0xa1,0x05,0x42,0x7d,0x6c,0x6a,0xfa,0xdb,0xf9,0x11,
                0x5f,0xe1,0x84,0xd2,0x37,0x4e,0xa6,0x74,0x7c},
        .key_len = 139,
        .data = {0xf1,0x45,0x0e,0x17,0xbe,0xb5,0x7b,0xdc,0x8e,0x2a,
                 0xb1,0xb9,0xb6,0xb3,0x55,0x53,0x99,0x36,0x81,0xe8,
                 0xcc,0x08,0x0d,0x85,0x78,0xbb,0x0d,0x79,0x00,0x37,
                 0x9a,0x09,0xe4,0x0c,0xd6,0x65,0xe6,0x07,0x2a,0xdb,
                 0x6b,0x04,0xd2,0x4b,0x23,0x02,0x9c,0xdb,0xec,0x7d,
                 0xec,0xeb,0xf4,0xf8,0x04,0x4c,0x1e,0xd9,0x82,0xac,
                 0xa8,0x79,0x2a,0x55,0x0a,0x7c,0xe6,0x18,0x21,0x5e,
                 0x0b,0x83,0x8c,0x4f,0xde,0x5b,0x57,0x41,0x57,0x46,
                 0xd6,0x3f,0x25,0xc7,0xd4,0x00,0xf1,0x6f,0xee,0xbc,
                 0xe7,0x52,0x39,0x3e,0x73,0xb9,0x2b,0x3b,0x48,0x16,
                 0xb8,0xe2,0xa7,0x3d,0xbb,0xb4,0xde,0xd0,0x98,0x96,
                 0x0f,0xfb,0x1f,0x24,0x32,0x62,0xb4,0x49,0x5d,0x58,
                 0xad,0x0c,0x43,0x52,0xfc,0xad,0xfc,0x9b},
        .data_len = 128,
        .mac = {0x66,0x4a,0x4b,0x28,0x0e,0xde,0xf3,0x00,0x4c,0xa8,
                0x03,0x2a,0x42,0x42,0x06,0x08,0x3b,0xaf,0x4e,0xd3,
                0xf0,0x55,0xf1,0x95,0x8d,0x84,0xd7,0xde,0xb8,0xec,
                0x7e,0xab,0x7f,0x92,0x85,0xa3,0xb0,0xdc,0xe0,0x99,
                0x7c,0x07,0xb3,0x8e,0xae,0xe1,0x85,0x3a,0xa5,0xc6,
                0xde,0x79,0x89,0x33,0x8d,0x5a},
        .mac_len = 56,
    }, {	// 38	(285)
        .key = {0x95,0xaf,0x10,0x92,0x0d,0xc7,0x88,0x26,0x9e,0x70,
                0xb8,0x56,0x0b,0x73,0x13,0x5c,0xf7,0xf6,0xf5,0xb0,
                0x4a,0x50,0x2c,0x7b,0xd6,0x1c,0xb7,0x4f,0x3b,0x8c,
                0xcd,0x16,0x07,0x01,0x22,0x49,0x22,0xd8,0x65,0x63,
                0x6a,0x86,0x0d,0x94,0x9a,0xe7,0x55,0xb9,0x70,0xd3,
                0x85,0x8c,0x0f,0xf3,0x74,0x18,0xa2,0xd2,0x4b,0x71,
                0x42,0x37,0x8b,0xa1,0x1a,0xb3,0x52,0xe5,0xc8,0x76,
                0xda,0x1a,0x07,0x66,0x42,0x72,0x8b,0x73,0x91,0x6b,
                0x2d,0x24,0xf8,0x02,0x48,0x76,0x57,0x23,0x63,0xe7,
                0x03,0x65,0x10,0xce,0xc7,0xf4,0x13,0xed,0x28,0xce,
                0xc7,0x49,0xed,0x33,0xbe,0x3a,0xdf,0x56,0xa8,0xbe,
                0xce,0x59,0x76,0x12,0xd4,0x78,0xbf,0x84,0xde,0x85,
                0x62,0x83,0x67,0x94,0x6d,0xf8,0x87,0xf7,0x3d,0xd9,
                0x2d,0x6d,0xe7,0xfa,0xa8,0x96,0xd7,0x27,0x6d},
        .key_len = 139,
        .data = {0x61,0xd9,0x1f,0x31,0x7a,0x90,0x2e,0xa0,0x94,0x4e,
                 0x11,0xe9,0x2e,0x66,0x57,0xa5,0x89,0xe1,0x7a,0xbc,
                 0x02,0x7f,0xcd,0x86,0x9f,0xf8,0xb0,0x30,0xe8,0x87,
                 0x06,0x62,0xf8,0xa9,0xe9,0x1e,0xd3,0x23,0x9c,0xec,
                 0xfa,0x42,0xc0,0x34,0x3d,0x66,0xcb,0xeb,0xd1,0xc2,
                 0xb7,0x71,0xa2,0x5d,0xf7,0xba,0xea,0x5c,0xaf,0xad,
                 0x03,0x84,0x24,0xc9,0x7a,0xfb,0x72,0x0e,0x64,0x4e,
                 0x7d,0x1b,0xf5,0xb8,0x29,0x94,0x4e,0xa2,0xce,0xc6,
                 0x97,0x66,0xe6,0x8e,0x4e,0x58,0x09,0x76,0xde,0x07,
                 0x1c,0x22,0x74,0xc0,0xc5,0xeb,0x0e,0x54,0x21,0xc9,
                 0xd5,0x1b,0xba,0x76,0xac,0x39,0xb3,0xd0,0x09,0x20,
                 0x46,0x80,0x03,0x57,0x71,0xd9,0xad,0x79,0xeb,0x02,
                 0xa3,0x80,0x5d,0x58,0xe2,0x43,0xcf,0x0e},
        .data_len = 128,
        .mac = {0x6e,0x98,0x9e,0xc9,0xcb,0xf0,0x10,0xad,0x66,0x91,
                0xa6,0x72,0xff,0x4c,0xa9,0x0a,0x00,0x27,0x5f,0x9b,
                0xa4,0xc8,0x1c,0xd1,0x47,0xcc,0x50,0x6e,0x1d,0xbc,
                0x8b,0xc9,0x3b,0x1d,0x96,0xa3,0x75,0xe4,0x93,0x50,
                0x3c,0x0a,0xc6,0x97,0xf7,0xc4,0x5e,0x4f,0xad,0xf1,
                0x38,0x24,0x2d,0xf7,0xe0,0x6e,0x67,0x7d,0xe2,0x45,
                0xaf,0xa9,0x77,0x80},
        .mac_len = 64,
    }, {	// 39	(286)
        .key = {0x27,0xe6,0xc9,0xf2,0x70,0xb9,0x85,0x5c,0x96,0x58,
                0xad,0x0e,0x3d,0x6c,0x9a,0x11,0x1a,0x62,0x4f,0x66,
                0xfa,0x64,0xa4,0x9a,0x06,0x88,0xa4,0x9b,0x45,0x47,
                0x33,0xca,0x62,0x30,0xf4,0x51,0xb0,0xdd,0x69,0xb7,
                0x6b,0x27,0x5c,0xb2,0x41,0x96,0x7e,0x3c,0x10,0x1b,
                0x4f,0xe8,0xf2,0x02,0x3d,0x77,0x77,0x22,0x10,0xa6,
                0x31,0x57,0x85,0x4b,0x76,0x32,0x39,0xa0,0x61,0xee,
                0xc9,0xdf,0x1a,0xa6,0x38,0x0f,0x57,0xc6,0x91,0x1d,
                0x23,0xc0,0xcd,0x2e,0xdf,0x00,0xf6,0x34,0x86,0x21,
                0x8d,0xbf,0x35,0x61,0x2a,0x17,0xea,0x52,0x62,0x87,
                0x8b,0xd3,0xed,0xfb,0x2b,0x3f,0x08,0xce,0x8a,0xe4,
                0x19,0xdd,0xda,0xb7,0x92,0xe0,0xc9,0x45,0x17,0xfa,
                0xbb,0xed,0xe3,0x8e,0x57,0x4d,0x68,0x55,0x46,0xfa,
                0x35,0xad,0x37,0x74,0x1d,0x34,0x27,0x59,0x96},
        .key_len = 139,
        .data = {0xdf,0x24,0x27,0x9b,0xf8,0x27,0x7a,0xd1,0x09,0x19,
                 0x72,0xb8,0x25,0x94,0xd8,0x46,0x77,0xe5,0x4f,0xe5,
                 0xd6,0x57,0x86,0xd1,0x9a,0xb5,0xb2,0xc1,0xae,0x0a,
                 0x3c,0xc9,0xe7,0xab,0xb6,0x7f,0x94,0x77,0x14,0x5d,
                 0x57,0x5e,0x19,0x66,0x33,0x20,0x0f,0x0c,0xe5,0x57,
                 0xbb,0x52,0x78,0xb8,0x90,0x2e,0x14,0x96,0x23,0x31,
                 0x17,0xa7,0xdf,0x69,0x66,0x0b,0xfa,0x87,0x06,0x8a,
                 0xa7,0x3d,0xe6,0x1e,0x8e,0xea,0xff,0xb1,0x79,0x79,
                 0x9f,0x27,0x50,0x86,0x02,0x9f,0x47,0xc3,0x23,0xf6,
                 0x56,0x9b,0xd1,0x8d,0xea,0x15,0x05,0x4d,0xda,0xfa,
                 0x73,0xe8,0x9c,0x3a,0x5f,0x61,0xb9,0x8c,0xb2,0xce,
                 0x7e,0x55,0x4d,0x5d,0xf4,0xcb,0x9d,0x95,0x13,0x5a,
                 0x70,0xde,0x33,0x47,0x07,0x44,0xc3,0x93},
        .data_len = 128,
        .mac = {0xe6,0xf6,0x06,0x12,0x75,0xa8,0x93,0x45,0xf5,0x46,
                0x3c,0xfa,0x19,0x8d,0x52,0x8e,0x14,0x04,0x7d,0x47,
                0x8f,0x69,0xad,0x7a,0x73,0x43,0x2f,0x18,0xf8,0x8b,
                0xc6,0x8a,0x1b,0x8a,0xba,0x2c,0x3b,0x02,0x5c,0x93,
                0xb2,0x5d,0xeb,0x8f,0x40,0x37,0x63,0xa5,0x50,0x24,
                0x40,0x8a,0x97,0xa9,0x03,0xe9,0x5f,0x0c,0xb6,0x17,
                0x8e,0x7b,0xe3,0x89},
        .mac_len = 64,
    }, {	// 40	(300)
        .key = {0x1b,0xca,0xf9,0x6d,0xfd,0xba,0xab,0x10,0x28,0x39,
                0x68,0x65,0xbf,0xf3,0xfd,0x9a,0x87,0xa6,0x04,0x6e,
                0x91,0x30,0xaa,0xe9,0x1a,0xed,0xb5,0x4a,0xaf,0x3f,
                0xc5,0x42,0x95,0x6e,0xd9,0x5f,0x67,0xb3,0x16,0xcb,
                0x01,0x8f,0xc4,0x36,0x9d,0x61,0x9a,0x11,0x3c,0x38,
                0x3b,0xfd,0x48,0xe6,0x00,0x83,0x77,0x56,0xf8,0x05,
                0xf2,0x1f,0x2d,0xf4,0xb7,0x58,0x29,0x44,0x3b,0xaf,
                0xb9,0x1d,0x54,0xb5,0xf1,0xd4,0x57,0x7c,0x70,0x19,
                0x7d,0x64,0xe6,0x72,0x8c,0x32,0xd3,0xe8,0xb3,0x62,
                0x91,0xff,0xb5,0xbf,0x94,0xec,0x4a,0x30,0xf1,0x03,
                0xc2,0xd5,0x1d,0x6f,0xeb,0x52,0xf7,0x25,0x35,0x6b,
                0x09,0x1b,0x14,0xd2,0x0b,0xb4,0xa6,0x3c,0xa9,0x7f,
                0x96,0xba,0x87,0x22,0x40,0x47,0x81,0x91,0x15,0x9d,
                0x37,0x84,0x41,0x4e,0xc9,0x27,0x8c,0x0f,0xdf,0x13,
                0x7e,0x57},
        .key_len = 142,
        .data = {0x9d,0x1c,0x57,0xb3,0xe6,0x25,0xd6,0xa7,0xf3,0x84,
                 0x62,0x45,0x9c,0x44,0x2e,0xfd,0x5d,0x84,0x2c,0x0b,
                 0x57,0x6c,0x31,0x04,0x97,0x66,0x54,0xaf,0x9b,0x8a,
                 0x97,0x17,0x12,0x59,0xc9,0xde,0xe4,0xaf,0xaa,0xd4,
                 0x72,0xf3,0x51,0xc8,0x1f,0x91,0xcb,0x9f,0x0f,0xaa,
                 0x05,0xa3,0x2f,0x0a,0xeb,0x1d,0x75,0x5a,0xd2,0x25,
                 0x69,0x96,0x13,0xb9,0xe4,0xc2,0xf9,0x0d,0x17,0xd9,
                 0xde,0x22,0xd0,0x07,0x90,0x9a,0xf9,0x17,0xfa,0xcb,
                 0x61,0x2f,0x00,0x12,0xc0,0xbb,0xaa,0x69,0xf5,0xa7,
                 0x85,0x3c,0x91,0xd1,0x46,0x79,0x99,0xbe,0xb1,0xe7,
                 0x88,0x8c,0xb9,0xb8,0x94,0x27,0xe9,0x61,0x32,0xc0,
                 0x02,0xa9,0xdc,0x7b,0xe6,0x00,0x12,0x68,0xb4,0x13,
                 0x9f,0xb5,0x35,0x9d,0x1d,0x22,0x77,0x80},
        .data_len = 128,
        .mac = {0xca,0x7f,0x6a,0xbe,0xd1,0xc2,0x25,0x2f,0xb7,0xf0,
                0x87,0x06,0xd1,0x19,0xfe,0xdd,0x83,0xdf,0x9e,0x09,
                0xc1,0x2e,0x48,0x12,0x67,0xef,0x73,0x3b,0xc0,0x53,
                0xf9,0x45},
        .mac_len = 32,
    }, {	// 41 (301)
        .key = {0xf4,0x73,0x20,0x4e,0x11,0x13,0x98,0x1c,0x4b,0xba,
                0xe6,0xab,0x72,0x07,0xfa,0xbb,0x97,0xd6,0x11,0x2a,
                0x71,0x14,0x34,0x68,0x00,0xd2,0x02,0x2b,0x6f,0x97,
                0xb0,0x64,0x3f,0xfb,0x4e,0x71,0xa2,0x71,0x4c,0x94,
                0x25,0x65,0x3c,0xa4,0x92,0x87,0xd9,0x02,0x15,0x66,
                0x49,0xcf,0x9d,0x40,0x9f,0x5c,0x85,0x18,0xa4,0xe2,
                0x74,0x0a,0xd7,0x74,0x82,0xdc,0xb2,0x31,0x9c,0xd5,
                0x2f,0xbe,0x29,0xd2,0x8b,0xf4,0xed,0x96,0x4a,0xd5,
                0x2e,0x62,0x7b,0xc5,0x16,0xc5,0x3d,0xf7,0x59,0xb6,
                0x82,0x25,0x18,0xe0,0x13,0x84,0x04,0x3d,0xd5,0xf7,
                0xc6,0xe6,0x4a,0x82,0x9f,0x2a,0x5a,0x02,0x4b,0xdd,
                0x54,0x1c,0x2c,0xf2,0x30,0x74,0xa4,0x4d,0x17,0x1f,
                0xc3,0xd2,0x25,0x5a,0x22,0xc2,0x6d,0x7c,0x0e,0x47,
                0x3d,0x29,0xe9,0x4d,0xa4,0xc9,0xce,0x94,0x32,0x38,
                0x74,0x03},
        .key_len = 142,
        .data = {0x61,0xa9,0xc8,0x34,0x87,0xfd,0xb8,0xdf,0x53,0x6e,
                 0x2a,0x79,0xf5,0x77,0x7f,0x71,0x6e,0x2a,0x4c,0x92,
                 0xad,0xc7,0xa6,0x34,0xf8,0x8d,0x9d,0xb4,0x8e,0xd0,
                 0x06,0x21,0x8b,0x4c,0x7e,0x38,0x0b,0xb9,0xab,0x93,
                 0x92,0xa0,0xc5,0x1d,0xac,0x6c,0x55,0xc9,0xd0,0xc4,
                 0x3d,0x84,0x8a,0x18,0xf9,0x68,0xda,0x7a,0x24,0x32,
                 0xd7,0xb0,0x3e,0xf6,0x97,0x87,0xac,0x11,0x53,0x6a,
                 0x00,0x60,0xc7,0x32,0x7b,0xaf,0xd8,0x40,0xd0,0x74,
                 0x79,0xc9,0x2b,0xbe,0xcf,0x72,0x72,0x68,0x93,0x1d,
                 0xcb,0x92,0xb9,0x02,0x5e,0x51,0x4f,0x01,0x5c,0xa6,
                 0xa7,0x36,0x40,0xad,0xb8,0xcf,0x55,0x17,0xc2,0x1a,
                 0x78,0xda,0x31,0xe2,0x1f,0x27,0xc4,0xfc,0x9a,0x9d,
                 0xee,0x1d,0x99,0xbc,0x7d,0xef,0xef,0x9c},
        .data_len = 128,
        .mac = {0x9e,0xf8,0x22,0xb7,0x1d,0x2b,0x1f,0x44,0xc0,0xc4,
                0x78,0xd0,0xa1,0x6e,0x48,0xae,0x10,0x5f,0xc0,0x1e,
                0x4c,0x0c,0xc5,0x2e,0x9d,0xbc,0x68,0x21,0xa1,0xca,
                0x1e,0x3a},
        .mac_len = 32,
    }, {	// 42 (315)
        .key = {0x75,0x0d,0x16,0xba,0x01,0x45,0x22,0xc2,0xab,0x47,
                0x5e,0x86,0x38,0x53,0x5d,0x5e,0x72,0x9f,0xce,0x4a,
                0x28,0x4a,0xa4,0x87,0x77,0x91,0x75,0x60,0x48,0x45,
                0x20,0x47,0x1a,0x7c,0xb3,0x24,0x02,0x88,0x8c,0xed,
                0xef,0x25,0xbe,0x77,0x33,0xf6,0xb6,0x8c,0x3f,0x33,
                0x1e,0x3d,0xae,0x9c,0x01,0x79,0x86,0xb1,0x30,0xdf,
                0xba,0x24,0xe5,0xd1,0xf3,0x8f,0xa7,0xca,0x8f,0x63,
                0x6d,0xf1,0xea,0x20,0x06,0xee,0xdd,0xcc,0xb8,0xfc,
                0x85,0x9e,0x28,0x3f,0x46,0xfa,0x79,0x59,0x0f,0xa2,
                0xb8,0x38,0xdf,0x2a,0x78,0x31,0x88,0xac,0xb9,0x53,
                0xb3,0xbe,0x0f,0xf5,0x24,0xa1,0x00,0x92,0x3f,0xbb,
                0xad,0xd7,0xcb,0x47,0xb5,0xf9,0xf7,0x4c,0x56,0x4c,
                0x9a,0xcb,0xd5,0x12,0xbf,0x3d,0x20,0x90,0x61,0x3d,
                0xb3,0x68,0x73,0x81,0xb8,0x22,0x24,0x0e,0x72,0x0c,
                0x60,0xac},
        .key_len = 142,
        .data = {0x6e,0x9a,0xef,0x11,0x39,0x47,0xfa,0xfb,0x96,0x1e,
                 0xef,0xef,0x89,0x1e,0xa3,0xf8,0x3a,0xe0,0x18,0xa4,
                 0x56,0x70,0xff,0x6f,0xb3,0x5b,0x7b,0x80,0x33,0x09,
                 0x28,0x93,0xea,0x4d,0x5c,0x37,0x83,0x3b,0xad,0x39,
                 0xdf,0x3a,0xb3,0x60,0x00,0x5a,0x0f,0x5a,0x26,0xe7,
                 0xab,0xf7,0x37,0x3e,0x44,0x95,0x64,0xdf,0x26,0xdc,
                 0x0a,0xc4,0x37,0xab,0xd6,0xf1,0x37,0x51,0x2d,0x4d,
                 0x46,0x01,0xcb,0xb0,0xa7,0x80,0xaa,0x3d,0xc3,0x21,
                 0x25,0xf2,0x71,0xf3,0x5d,0x7d,0x74,0x39,0xdc,0x51,
                 0x45,0x1a,0x78,0xfa,0x14,0x9a,0x7a,0xad,0x09,0xc9,
                 0x00,0x24,0xcf,0x3a,0xa0,0xa9,0x53,0xb7,0x4e,0x70,
                 0xc9,0x33,0x80,0x29,0xf1,0x00,0x90,0xd6,0x98,0x43,
                 0x61,0xda,0x61,0xa4,0x45,0x53,0xc5,0x4a},
        .data_len = 128,
        .mac = {0x90,0x78,0x1f,0xaa,0x01,0x46,0x87,0x9a,0x36,0x45,
                0x31,0x9c,0x53,0x90,0xba,0xe2,0xd0,0xad,0x06,0x12,
                0xbf,0x2e,0x6e,0x7d,0xc7,0x41,0xe7,0xc7,0x4b,0x2a,
                0xb6,0x5c,0x52,0x04,0x03,0x41,0x0f,0x1e,0xca,0x27},
        .mac_len = 40,
    }, {	// 43	(316)
        .key = {0x3f,0x64,0xc8,0x15,0xf2,0x79,0x3e,0xd4,0x93,0x3e,
                0x37,0x49,0x50,0xf7,0x7e,0x68,0x5a,0xe3,0xdc,0x21,
                0xf3,0x09,0xd3,0x40,0xc9,0x35,0x8f,0x92,0x82,0x3d,
                0x1a,0x49,0x98,0x88,0xcf,0xa1,0x67,0xd1,0x11,0xc4,
                0x84,0x29,0x3d,0xeb,0x51,0x3b,0xdc,0xc7,0xce,0xd4,
                0x1e,0x1f,0xb2,0xa3,0x84,0xba,0xbb,0x3d,0xdc,0x7f,
                0x42,0x63,0x72,0xdb,0x12,0x88,0x85,0x63,0x7a,0x8e,
                0xf8,0xbb,0xa1,0xd1,0x4a,0xb6,0x1a,0x66,0xc8,0xbe,
                0xb8,0xa5,0x01,0x77,0xa4,0xe4,0x7b,0x9f,0xac,0x86,
                0xa4,0x39,0xde,0x35,0x0b,0xea,0x56,0x6b,0xd0,0xa3,
                0x54,0x37,0x5a,0x80,0x92,0x3e,0x47,0xf0,0xd4,0x00,
                0xa9,0x17,0xc0,0x5c,0x6f,0x70,0xf0,0x56,0x21,0xb7,
                0x46,0x19,0xd1,0x06,0x7f,0x38,0x4e,0x3e,0x0b,0x39,
                0x9a,0xa8,0x1c,0x54,0x9e,0xdc,0xfe,0x78,0x1d,0xe2,
                0xa6,0x8f},
        .key_len = 142,
        .data = {0xfe,0x7a,0xe0,0xcc,0x9e,0x99,0xc1,0xb0,0xa2,0xbb,
                 0x11,0xd7,0x1d,0xb4,0x29,0xcf,0xba,0x95,0x9e,0x65,
                 0x5b,0xc8,0x08,0x60,0xff,0x5c,0x08,0x6f,0xe9,0x9d,
                 0x89,0x5f,0xff,0xe8,0x45,0x91,0x70,0xbd,0x8b,0xac,
                 0x99,0x16,0x40,0x8f,0x2a,0xf7,0x25,0x8a,0x84,0x47,
                 0x57,0xcd,0x66,0xde,0x1c,0x20,0xc9,0xdf,0xdc,0xd5,
                 0x08,0x48,0x6e,0xfb,0x44,0xc7,0x96,0xd5,0x07,0xd0,
                 0xcb,0x8b,0x27,0xec,0x3d,0x1e,0xee,0x54,0x16,0xa1,
                 0x77,0xbe,0x00,0xa7,0x5b,0x77,0xe8,0xf8,0x8e,0xb2,
                 0x02,0x88,0x83,0x20,0x0e,0xa9,0x97,0xc0,0xd9,0x7f,
                 0xac,0xf9,0x06,0xb4,0x14,0xe6,0x0e,0x6e,0xb2,0xf1,
                 0x86,0x7e,0x5b,0xa3,0xa1,0xdb,0x82,0xba,0xf5,0x4e,
                 0x15,0x7c,0x82,0x6e,0xa2,0x4c,0xf0,0xcf},
        .data_len = 128,
        .mac = {0x4b,0xbb,0xa9,0xca,0x72,0xb7,0x20,0x35,0x6a,0x2c,
                0xe8,0x0f,0x32,0xed,0x65,0xf6,0x98,0x8b,0x87,0x69,
                0x72,0x14,0x13,0xae,0x44,0x01,0x4a,0x72,0xcf,0x34,
                0xef,0xc9,0x65,0xcf,0xae,0xb4,0x4b,0x29,0x44,0x45},
        .mac_len = 40,
    }, {	// 44	(330)
        .key = {0xf0,0x1a,0xaa,0x41,0x52,0x8e,0xbf,0xec,0xdb,0x9f,
                0xcd,0x42,0x94,0x80,0x47,0xbf,0x9c,0x23,0x66,0x86,
                0x1e,0x5d,0xc0,0xa4,0x80,0x45,0x1c,0x95,0xdd,0xdd,
                0x85,0x09,0xb6,0xf4,0x9d,0xe8,0xe4,0x4a,0x34,0x93,
                0xd1,0xf6,0x19,0x6d,0xe8,0x05,0x30,0x9a,0x4a,0x64,
                0xc5,0x06,0xc3,0x00,0x20,0xa9,0xfc,0x2b,0x5a,0x0a,
                0xf2,0x42,0x6f,0x02,0xa6,0x8c,0x93,0xc3,0x17,0x66,
                0xca,0x18,0x6d,0x9d,0xb3,0xef,0x77,0x1c,0xa4,0x14,
                0x57,0xc7,0xfc,0x3f,0x7c,0xa6,0xb5,0x51,0xd1,0x46,
                0x39,0xdb,0x34,0x5c,0x77,0xe0,0x2a,0xac,0x35,0xdc,
                0xd1,0x29,0xa8,0x04,0x16,0x5e,0x42,0xd0,0x27,0x0c,
                0xcb,0xb7,0x2c,0x15,0xb3,0x39,0x32,0x98,0xa7,0x23,
                0x67,0x5e,0x62,0x0f,0x8d,0x75,0xc6,0x2a,0xa9,0x1a,
                0x2b,0x3a,0xfb,0xdd,0xf7,0xdf,0x33,0xff,0x6e,0x18,
                0xe2,0x1f},
        .key_len = 142,
        .data = {0x7b,0xda,0x43,0xe9,0x04,0x79,0xf8,0x52,0x71,0x7f,
                 0x47,0x86,0xb2,0x67,0xca,0x87,0x4f,0xed,0x63,0x80,
                 0x5a,0x2b,0xd0,0x07,0xa2,0x02,0x7f,0x9f,0x29,0xbb,
                 0xa3,0x81,0xd8,0xee,0x87,0x9c,0x72,0xa3,0x22,0xda,
                 0xc7,0xb8,0xed,0xa5,0x2a,0x83,0xf1,0xaa,0x24,0xb7,
                 0x24,0xdc,0x6a,0x8c,0xf5,0xbc,0x0d,0x1a,0x26,0xd8,
                 0xf1,0x06,0xda,0x92,0x80,0x61,0xef,0x23,0x9f,0xcd,
                 0x8d,0xe0,0xe1,0xe2,0xff,0xbb,0xfb,0xd0,0xc2,0x0c,
                 0x79,0x45,0xdc,0x92,0xaf,0x91,0x6f,0xb4,0xf1,0x08,
                 0x8e,0x0d,0x07,0xa7,0x4f,0x28,0xdb,0xfb,0x22,0x88,
                 0x75,0x3f,0x61,0xec,0x29,0xc5,0x28,0xd1,0x0f,0x97,
                 0x66,0x37,0xb4,0x5d,0x34,0xa8,0x0f,0x81,0x66,0x77,
                 0xb7,0x92,0x60,0x46,0x53,0xe6,0x2c,0xca},
        .data_len = 128,
        .mac = {0x02,0x01,0xdb,0xe5,0xef,0x40,0x95,0xf4,0xd4,0xb3,
                0x30,0x6d,0x9b,0x2d,0x75,0x52,0xe0,0xd6,0x8c,0x59,
                0x1e,0xa8,0x83,0x68,0x19,0x5f,0x66,0x28,0x0b,0x80,
                0xba,0x0f,0x2f,0x6a,0xe4,0x36,0xc4,0x23,0xc5,0xc9,
                0x39,0x3a,0xa9,0xf4,0x58,0x8b,0x01,0x07},
        .mac_len = 48,
    }, {	// 45	(331)
        .key = {0x93,0x1b,0x7d,0x98,0xf5,0x80,0xe6,0xd2,0x27,0x8d,
                0x1b,0x67,0x11,0x17,0xe6,0x04,0x7a,0x59,0xed,0xcc,
                0xcd,0xa1,0x91,0xa8,0x1c,0x49,0x17,0xde,0x65,0xe9,
                0x05,0xe6,0x14,0xcb,0xcf,0x79,0xca,0x9f,0xb3,0xea,
                0x5d,0x70,0xe2,0xb9,0x20,0xd7,0xe0,0x66,0x64,0x6f,
                0x2d,0x83,0x3e,0x88,0x25,0x0d,0x8b,0x20,0x25,0xfc,
                0x32,0x0f,0xef,0x19,0xa9,0x81,0x50,0x10,0xbb,0xa9,
                0x00,0xc6,0x88,0xb4,0xc9,0xec,0xcd,0xfc,0xeb,0xd5,
                0x65,0x7f,0xc0,0x84,0x10,0x8f,0x9c,0x0a,0x74,0xcb,
                0xf7,0x0f,0x61,0x4d,0xce,0xae,0x59,0x25,0x46,0x86,
                0x50,0x06,0x93,0x0d,0xb0,0x40,0x18,0x28,0xa0,0xee,
                0xcf,0xf9,0x86,0x71,0xec,0xf8,0xca,0x1d,0xbd,0x46,
                0xde,0x31,0xd5,0x3e,0x7b,0x0d,0x69,0x4c,0x2d,0x9f,
                0xfa,0x02,0x11,0x1f,0x34,0x68,0xdd,0xfc,0x17,0x94,
                0x22,0x16},
        .key_len = 142,
        .data = {0x9f,0x84,0x8c,0xbe,0xf3,0xe2,0x9e,0x43,0x76,0x68,
                 0x25,0xa1,0xc3,0x8d,0xcc,0xa8,0xf8,0x4e,0xad,0xda,
                 0x22,0xd0,0x67,0x2b,0xf5,0x47,0x71,0x05,0xc1,0x16,
                 0xae,0x8f,0xa1,0x38,0x41,0x2f,0xf6,0xde,0xa2,0x4e,
                 0x13,0x59,0xf1,0x5b,0x3a,0x3b,0x8b,0x12,0xc8,0xb9,
                 0xdc,0xfe,0xac,0x54,0x74,0x3c,0x1d,0xb9,0x5c,0x83,
                 0x8d,0xe5,0xaa,0x61,0xd8,0x8c,0x53,0xc7,0xc2,0xbc,
                 0x41,0xc8,0xa0,0xbe,0xa5,0x9e,0x55,0x3d,0x8f,0xee,
                 0x80,0xa9,0xbc,0x4d,0xf4,0xde,0xae,0x02,0x69,0x01,
                 0x02,0x0d,0x71,0xab,0xff,0x69,0xad,0x4a,0x6b,0x8f,
                 0x40,0xc4,0xa2,0x3b,0x84,0x5d,0xe9,0x72,0xfc,0xda,
                 0xff,0xc4,0x8a,0xe6,0xf5,0xcf,0xe2,0xb6,0x40,0x04,
                 0x3f,0xe9,0x0d,0xae,0x55,0xb2,0xd4,0x2f},
        .data_len = 128,
        .mac = {0x95,0x75,0xd5,0x42,0xbc,0x82,0xea,0x5d,0xef,0xa5,
                0x06,0x98,0xa7,0xe7,0x7c,0x4b,0xc6,0x8f,0x47,0xfd,
                0x33,0x2c,0xb3,0xeb,0x52,0xf0,0x09,0x98,0x7a,0x7b,
                0xd2,0x89,0xad,0x38,0x37,0xee,0x50,0x36,0x02,0x64,
                0xc9,0x54,0x67,0xc7,0x6c,0x0e,0x8a,0xc3},
        .mac_len = 48,
    }, {	// 46	(345)
        .key = {0xc9,0x37,0xc7,0x38,0x77,0x38,0xe5,0xbe,0x87,0x61,
                0xa4,0x16,0x03,0xa2,0x04,0xcd,0x93,0x12,0x8f,0xda,
                0xa1,0x86,0x98,0xd3,0xbc,0x62,0xc1,0x61,0x3a,0xaf,
                0xc0,0xf2,0x22,0x6e,0x62,0xad,0x49,0x2c,0xcf,0xab,
                0xba,0xd7,0x11,0x64,0x5b,0xd0,0xd0,0x67,0xce,0x41,
                0x5f,0x4f,0x96,0xb1,0xfd,0xf2,0x7b,0xd6,0x54,0xdb,
                0x2f,0xaa,0x46,0xfb,0x31,0xc1,0xe9,0x9b,0xf2,0xd9,
                0xd0,0x95,0x37,0xb3,0x8d,0xc4,0xbe,0x21,0x73,0xb9,
                0x24,0x61,0xa1,0xaf,0x06,0xa9,0x3c,0xc1,0x67,0x03,
                0xfb,0x5b,0x51,0x5b,0xe9,0xef,0xbf,0xd6,0x66,0xe9,
                0xbb,0x66,0x6f,0xe4,0xc4,0x9c,0x20,0x1e,0x72,0xbd,
                0x77,0x29,0x5d,0x17,0x5b,0xe4,0x3a,0x95,0xdd,0xaa,
                0x47,0x92,0xbd,0xc9,0xea,0xeb,0x30,0x36,0x94,0x58,
                0xe0,0x7f,0xec,0xfc,0x35,0x00,0x2a,0x3a,0xd3,0x5a,
                0xc0,0xcb},
        .key_len = 142,
        .data = {0x92,0xcf,0x37,0x16,0xe2,0x2f,0x68,0xd3,0x02,0x11,
                 0x40,0x10,0x34,0xfc,0x38,0x39,0xc9,0x4f,0x47,0x36,
                 0x95,0xe0,0xf8,0x36,0xa0,0xe4,0xbf,0xab,0x1e,0x0f,
                 0x8d,0x95,0xa2,0xfc,0xa2,0x04,0x8e,0xce,0xa3,0xd8,
                 0xce,0x18,0xec,0xcc,0xc6,0xfa,0x27,0xb3,0x88,0x8d,
                 0x1d,0x01,0x4b,0x81,0xcd,0x8c,0x19,0x05,0xbb,0x94,
                 0xc7,0xe7,0x8b,0x01,0x2a,0xfd,0xce,0xed,0x88,0xc3,
                 0x1c,0x62,0x4d,0x45,0x63,0x65,0xff,0x40,0x7e,0x33,
                 0x74,0x84,0x0a,0xca,0x43,0x9c,0x75,0xab,0x78,0xb1,
                 0x57,0xdf,0x8c,0xac,0x06,0x38,0x94,0x9b,0x65,0xd9,
                 0xb8,0x56,0xfc,0xf8,0xe7,0x62,0x4e,0xe3,0xc9,0x15,
                 0x66,0x41,0xf5,0xfb,0xb4,0x9b,0x65,0x79,0x17,0x1b,
                 0x26,0xb0,0x3b,0x93,0x54,0xa4,0x25,0xc0},
        .data_len = 128,
        .mac = {0x49,0x04,0x50,0x77,0x2d,0x57,0xcd,0xe7,0xf8,0x6f,
                0x9d,0x68,0x5d,0x07,0x0c,0x23,0x79,0x4c,0xed,0xc1,
                0x13,0x14,0x3c,0x44,0x1d,0xbf,0x5b,0x4d,0x78,0x7f,
                0xe6,0xe6,0x19,0xb6,0xe6,0xa2,0x86,0xa3,0x91,0x3e,
                0x1d,0x71,0x3f,0x4b,0x28,0xaf,0x24,0x77,0xcf,0x34,
                0x53,0xc4,0x15,0x29,0xc2,0xec},
        .mac_len = 56,
    }, {	// 47	(346)
        .key = {0x49,0xa7,0x5f,0xaf,0x82,0x32,0x54,0x4d,0x78,0x23,
                0x4c,0x3f,0x4f,0xd9,0xbd,0x7b,0x35,0x99,0x62,0x17,
                0xf5,0xd5,0x79,0xb2,0xc9,0xae,0x2f,0x10,0x70,0xb7,
                0x59,0x04,0x81,0xfc,0x64,0x93,0xff,0x43,0x8f,0x56,
                0x2d,0x91,0x57,0x20,0xa4,0xf0,0x33,0xe1,0xa2,0x70,
                0x82,0xca,0xa5,0x6c,0xc2,0xb2,0x39,0x0e,0x02,0xb9,
                0x0e,0x15,0x9a,0x2b,0x4d,0xd6,0x19,0xa7,0x2c,0x81,
                0x7f,0x80,0xc1,0xa6,0xad,0x63,0x13,0xf8,0x19,0x3a,
                0x77,0x50,0xc6,0x51,0x1a,0xf5,0x74,0xb7,0xda,0x1a,
                0x14,0xe8,0x5a,0xa8,0x7e,0xa8,0x55,0x3e,0x66,0x32,
                0x32,0x76,0x8c,0x92,0xd9,0x6b,0x34,0x50,0xc6,0x78,
                0xd6,0x4e,0x5b,0x42,0x16,0xf3,0x7b,0xf8,0xf6,0x89,
                0x54,0xd9,0x4b,0xe4,0x7c,0xf7,0x6e,0x45,0x40,0x3b,
                0x33,0x7e,0x39,0xaa,0x1f,0x56,0x20,0xe6,0xd5,0x2e,
                0xe2,0xf8},
        .key_len = 142,
        .data = {0xe4,0x43,0x7f,0xae,0xaa,0x8e,0x07,0x1f,0x7f,0xd7,
                 0x1b,0x5d,0x9b,0x42,0xdb,0x56,0x43,0xd0,0xd7,0xad,
                 0x9e,0xe1,0x7f,0x5f,0xc5,0xf8,0xdc,0xa6,0x19,0xbf,
                 0x7e,0x1d,0xd9,0x4b,0xb6,0x25,0xf4,0xa0,0x05,0x88,
                 0x1b,0x78,0xd3,0x5d,0x1b,0x40,0x64,0x2c,0x6a,0x64,
                 0x09,0x52,0xff,0xcc,0x93,0xa6,0x67,0xa7,0x0d,0x5c,
                 0x16,0x67,0xab,0x2a,0xfb,0x0c,0xfd,0xaa,0x57,0xa6,
                 0x68,0x00,0xce,0x15,0xd7,0xbc,0xbf,0x14,0xae,0x02,
                 0xd1,0x76,0x00,0xe8,0xce,0xae,0x01,0x1f,0xb5,0x95,
                 0xbe,0xd6,0x0a,0x68,0x0b,0xdb,0x71,0x10,0xbd,0x03,
                 0x7b,0xa7,0x1a,0xcd,0xc0,0x37,0x8e,0x42,0x2e,0xc5,
                 0xa0,0x1f,0xa0,0x8e,0x23,0x36,0xb2,0x90,0xb1,0x12,
                 0xd4,0x4c,0x55,0xda,0x9e,0xcb,0x88,0x77},
        .data_len = 128,
        .mac = {0xa0,0x93,0xdf,0x05,0xd8,0xdf,0x35,0xbb,0x12,0xde,
                0x09,0xc9,0xae,0xdb,0x23,0xdc,0xfc,0xd5,0x17,0xdf,
                0x6f,0x0c,0xdd,0x18,0x86,0xd4,0xc7,0x36,0xc8,0x50,
                0x68,0xf9,0xf1,0x94,0x86,0x05,0x1d,0x22,0x3d,0xf4,
                0x85,0x09,0xa2,0xa6,0xe1,0xb5,0x0c,0xe2,0x24,0x29,
                0xef,0xb8,0xf0,0x79,0x78,0x23},
        .mac_len = 56,
    }, {	// 48	(360)
        .key = {0x01,0xb9,0x5a,0x88,0x79,0x27,0xce,0x31,0xb1,0x24,
                0x23,0x91,0xbb,0xd0,0x09,0x65,0xeb,0x77,0xa9,0x03,
                0xd4,0xb8,0x39,0x9b,0x72,0xe6,0xce,0xbd,0xa9,0xae,
                0x72,0x1b,0xee,0xfa,0x77,0x91,0x45,0x16,0x0b,0x62,
                0x6b,0x11,0x0c,0xc5,0x54,0x67,0x1d,0xa0,0xd8,0xdc,
                0xf9,0x93,0xa9,0xab,0x07,0x38,0x88,0xe0,0x2f,0xa9,
                0xb8,0x03,0xed,0x43,0xb3,0xf6,0xa3,0xaa,0x1d,0x20,
                0x34,0x0d,0xf6,0xcc,0xce,0xac,0x13,0xcb,0x07,0x97,
                0xcf,0x61,0x2c,0xb8,0xfe,0x5f,0xd5,0x13,0x22,0x8c,
                0xbd,0x4d,0xe2,0x49,0xd1,0x6b,0xb7,0x75,0x87,0xdd,
                0xe9,0x8f,0x71,0xbb,0xba,0x1a,0x12,0x4e,0xe0,0x46,
                0xf0,0xd2,0x39,0xcc,0xea,0x7a,0xbb,0x1a,0xcc,0xb5,
                0xaa,0xb0,0x21,0xb0,0x0d,0xca,0x49,0x1c,0x62,0x3f,
                0xcb,0x31,0x91,0xa9,0xec,0xf3,0x1f,0xc6,0x80,0xb4,
                0xa4,0x1e},
        .key_len = 142,
        .data = {0x63,0x2a,0xfa,0x8e,0x79,0xb1,0x4b,0x2a,0x36,0x04,
                 0xf5,0x85,0x5d,0x2b,0xf1,0x82,0xd3,0xc5,0x6d,0x68,
                 0x53,0xf2,0x1f,0xe4,0x62,0x71,0xda,0x52,0x86,0x06,
                 0x5f,0x38,0xb3,0x1f,0x75,0x13,0x06,0xb6,0x3c,0x57,
                 0xb6,0x79,0xbe,0xb1,0x47,0x29,0xc7,0x8f,0x00,0x40,
                 0xf7,0xe2,0xa0,0xd6,0x15,0x22,0x4d,0xc5,0xa6,0x93,
                 0xcd,0x0c,0xbe,0xc8,0xf8,0x71,0x17,0x65,0x6d,0x6b,
                 0x60,0x29,0x85,0x3e,0xd7,0x2b,0x85,0x68,0x1a,0x63,
                 0x18,0x3c,0x3a,0x6d,0xfc,0xcd,0x12,0x8a,0xfb,0x0d,
                 0xd7,0xe8,0x1d,0x36,0xf0,0x23,0x1c,0x69,0x07,0x0b,
                 0x18,0x95,0x60,0xa8,0x8c,0x9b,0x69,0x7b,0x81,0xb0,
                 0x93,0x07,0x01,0x02,0x61,0x90,0xcf,0x9e,0xbe,0x23,
                 0x55,0x91,0x94,0xd6,0xde,0x4d,0x9a,0x51},
        .data_len = 128,
        .mac = {0x21,0x0a,0xd4,0x5c,0xa2,0xfd,0x1f,0x10,0x5c,0x0a,
                0x18,0xf9,0x93,0x77,0x4f,0x93,0x3e,0xce,0x57,0xac,
                0xe4,0xda,0x61,0x96,0x89,0xe1,0xcb,0x8b,0x49,0x1a,
                0x18,0x9c,0xc6,0xe4,0xee,0x19,0x54,0xa3,0x22,0x01,
                0x07,0x2e,0x70,0xf9,0x34,0x83,0x7c,0x0f,0xb6,0xe2,
                0x39,0xb4,0xfd,0xfb,0xd2,0x6e,0xbf,0x11,0xb9,0xa9,
                0x19,0xea,0xfd,0x09},
        .mac_len = 64,
    }, {	// 49	(361)
        .key = {0x61,0x09,0x6f,0x4f,0xe5,0x34,0x04,0x88,0x91,0x6d,
                0xe2,0x93,0xbe,0x38,0xcc,0x3a,0xe0,0xc8,0x77,0x67,
                0x0c,0x71,0x36,0x37,0xb7,0x60,0xd7,0x4f,0xc1,0x8a,
                0xc7,0x73,0xb2,0xe2,0x7d,0x55,0x43,0xcf,0x16,0xaa,
                0x20,0xdd,0x3d,0x83,0xec,0xb3,0x4e,0xdb,0x85,0x45,
                0xbb,0x6c,0x8a,0x4a,0xae,0xc8,0x1b,0xf1,0xf0,0xa4,
                0xe0,0xcf,0x09,0x77,0x4d,0x1c,0xa9,0x44,0x24,0x20,
                0x46,0xb3,0x3b,0xe8,0x07,0x67,0x7f,0x3d,0xe1,0x8c,
                0x39,0xd7,0x00,0xaf,0x90,0xcd,0x68,0xd3,0x4f,0x50,
                0xdc,0xc1,0xe9,0x99,0xfe,0x9f,0xbb,0x20,0xb9,0xc4,
                0x90,0x0f,0xdc,0xcb,0x6a,0xf6,0x07,0xe6,0x80,0xc0,
                0xcb,0x75,0x83,0xe6,0x0d,0xd8,0x25,0xe2,0xab,0x81,
                0xdc,0xe7,0x63,0x4d,0xe3,0xcf,0xf0,0x14,0x83,0x55,
                0x75,0x7f,0x90,0x84,0x1f,0x19,0x36,0x6f,0x06,0xa9,
                0xf6,0x23},
        .key_len = 142,
        .data = {0x67,0xe7,0x04,0x04,0x6f,0x98,0xcb,0x5a,0xa9,0x7d,
                 0xa9,0x5b,0x19,0x14,0x73,0x91,0xf0,0x57,0x88,0xf8,
                 0x11,0x36,0x6b,0x0e,0xce,0x44,0xb1,0x2a,0xf2,0xb1,
                 0x1e,0x0e,0x05,0x78,0x0b,0xbf,0xcb,0xd9,0x0a,0x95,
                 0x0e,0x0a,0xcd,0x8e,0x9d,0x2a,0x44,0xe7,0x95,0x76,
                 0x06,0xee,0xdf,0xbf,0xf2,0x12,0xfa,0x1c,0x16,0x3c,
                 0xfb,0xdc,0xd0,0x62,0xd2,0xbe,0x32,0x59,0xce,0x65,
                 0xab,0xea,0x64,0x06,0xe4,0x29,0x2c,0x64,0xe9,0x02,
                 0x2c,0xfe,0x89,0x15,0x59,0x86,0xff,0xc4,0x5b,0x96,
                 0xd2,0x89,0x91,0x9f,0xf9,0x8d,0x55,0x22,0x43,0x77,
                 0x81,0x22,0xf6,0x82,0x31,0xd9,0xb6,0xd3,0xcb,0xaa,
                 0xa9,0x09,0x3d,0x57,0xd9,0x15,0x86,0x74,0xda,0x4c,
                 0x78,0x1b,0xac,0xba,0xbc,0xe2,0xe2,0xba},
        .data_len = 128,
        .mac = {0x9a,0x2d,0x14,0x7e,0x50,0x82,0x71,0x57,0xf3,0x86,
                0x6e,0x86,0x8c,0x1c,0xca,0x9f,0x08,0x15,0x79,0xc9,
                0x2f,0x25,0xda,0x8c,0xeb,0xc9,0xed,0x24,0x99,0x28,
                0xc8,0x2b,0xea,0xd3,0x9d,0x48,0x0e,0xcb,0xb5,0xb5,
                0xd0,0xe0,0x75,0x50,0x29,0xae,0xbf,0x3e,0x02,0x06,
                0x98,0x4f,0x3e,0xa8,0x3f,0x4d,0x63,0x72,0xf4,0x45,
                0x33,0x90,0xe0,0x70},
        .mac_len = 64,
    },
};

struct HMAC_TEST_SUITE_INFO {
    const char *name;
    unsigned int tvcount;
    struct HMAC_TEST_VECTOR *tv;
    CK_MECHANISM mech;
};

#define NUM_OF_FIPS_HMAC_TEST_SUITES 5
struct HMAC_TEST_SUITE_INFO fips_hmac_test_suites[] = {
    {
        .name = "SHA-1 HMAC",
        .tvcount = 10,
        .tv = fips_sha1_hmac_test_vector,
        .mech = { CKM_SHA_1_HMAC, 0, 0 },
    }, {
        .name = "SHA-224 HMAC",
        .tvcount = 10,
        .tv = fips_sha224_hmac_test_vector,
        .mech = { CKM_SHA224_HMAC, 0, 0 },
    }, {
        .name = "SHA-256 HMAC",
        .tvcount = 10,
        .tv = fips_sha256_hmac_test_vector,
        .mech = { CKM_SHA256_HMAC, 0, 0 },
    }, {
        .name = "SHA-384-HMAC",
        .tvcount = 10,
        .tv = fips_sha384_hmac_test_vector,
        .mech = {CKM_SHA384_HMAC, 0, 0 },
    }, {
        .name = "SHA-512-HMAC",
            .tvcount = 10,
            .tv = fips_sha512_hmac_test_vector,
            .mech = {CKM_SHA512_HMAC, 0, 0 },
    }
};

#define NUM_OF_FIPS_HMAC_GENERAL_TEST_SUITES 5
struct HMAC_TEST_SUITE_INFO fips_hmac_general_test_suites[] = {
    {
        .name = "SHA-1 HMAC GENERAL",
        .tvcount = 40,
        .tv = fips_sha1_hmac_general_test_vector,
        .mech = { CKM_SHA_1_HMAC_GENERAL, 0, 0 },
    },
    {
        .name = "SHA-224 HMAC GENERAL",
        .tvcount = 30,
        .tv = fips_sha224_hmac_general_test_vector,
        .mech = { CKM_SHA224_HMAC_GENERAL, 0, 0 },
    },
    {
        .name = "SHA-256 HMAC GENERAL",
        .tvcount = 30,
        .tv = fips_sha256_hmac_general_test_vector,
        .mech = { CKM_SHA256_HMAC_GENERAL, 0, 0 },
    },
    {
        .name = "SHA-384 HMAC GENERAL",
        .tvcount = 40,
        .tv = fips_sha384_hmac_general_test_vector,
        .mech = { CKM_SHA384_HMAC_GENERAL, 0, 0 },
    },
    {
        .name = "SHA-512 HMAC GENERAL",
        .tvcount = 50,
        .tv = fips_sha512_hmac_general_test_vector,
        .mech = { CKM_SHA512_HMAC_GENERAL, 0, 0 },
    }
};

#define NUM_OF_HMAC_TEST_SUITES 20
struct HMAC_TEST_SUITE_INFO hmac_test_suites[] = {
    {
        .name = "SHA-1 HMAC",
        .tvcount = 7,
        .tv = sha1_hmac_test_vector,
        .mech = { CKM_SHA_1_HMAC, 0, 0 },
    },
    {
        .name = "SHA-224 HMAC",
        .tvcount = 6,
        .tv = sha224_hmac_test_vector,
        .mech = { CKM_SHA224_HMAC, 0, 0 },
    },
    {
        .name = "SHA-256 HMAC",
        .tvcount = 6,
        .tv = sha256_hmac_test_vector,
        .mech = { CKM_SHA256_HMAC, 0, 0 },
    },
    {
        .name = "SHA-384 HMAC",
        .tvcount = 6,
        .tv = sha384_hmac_test_vector,
        .mech = { CKM_SHA384_HMAC, 0, 0 },
    },
    {
        .name = "SHA-512 HMAC",
        .tvcount = 6,
        .tv = sha512_hmac_test_vector,
        .mech = { CKM_SHA512_HMAC, 0, 0 },
    },
    {
        .name = "SHA-512/224 HMAC",
        .tvcount = 2,
        .tv = sha512_224_hmac_test_vector,
        .mech = { CKM_SHA512_224_HMAC, 0, 0 },
    },
    {
        .name = "SHA-512/256 HMAC",
        .tvcount = 2,
        .tv = sha512_256_hmac_test_vector,
        .mech = { CKM_SHA512_256_HMAC, 0, 0 },
    },
    {
        .name = "MD5 HMAC",
        .tvcount = 7,
        .tv = md5_hmac_test_vector,
        .mech = { CKM_MD5_HMAC, 0, 0 },
    },
    {
        .name = "SHA-1 HMAC General",
        .tvcount = 7,
        .tv = sha1_hmac_general_test_vector,
        .mech = { CKM_SHA_1_HMAC_GENERAL, &ten, sizeof(CK_ULONG) },
    },
    {
        .name = "SHA-224 HMAC General",
        .tvcount = 4,
        .tv = sha224_hmac_general_test_vector,
        .mech = { CKM_SHA224_HMAC_GENERAL, &sixteen, sizeof(CK_ULONG) },
    },
    {
        .name = "SHA-256 HMAC General",
        .tvcount = 4,
        .tv = sha256_hmac_general_test_vector,
        .mech = { CKM_SHA256_HMAC_GENERAL, &sixteen, sizeof(CK_ULONG) },
    },
    {
        .name = "SHA-384 HMAC General",
        .tvcount = 4,
        .tv = sha384_hmac_general_test_vector,
        .mech = {CKM_SHA384_HMAC_GENERAL,&twentyfour,sizeof(CK_ULONG)},
    },
    {
        .name = "SHA-512 HMAC General",
        .tvcount = 4,
        .tv = sha512_hmac_general_test_vector,
        .mech = { CKM_SHA512_HMAC_GENERAL,&thirtytwo,sizeof(CK_ULONG) },
    },
    {
        .name = "SHA-512/224 HMAC General",
        .tvcount = 2,
        .tv = sha512_224_hmac_general_test_vector,
        .mech = { CKM_SHA512_224_HMAC_GENERAL,&sixteen,sizeof(CK_ULONG) },
    },
    {
        .name = "SHA-512/256 HMAC General",
        .tvcount = 2,
        .tv = sha512_256_hmac_general_test_vector,
        .mech = { CKM_SHA512_256_HMAC_GENERAL,&sixteen,sizeof(CK_ULONG) },
    },
    {
        .name = "MD5 HMAC General",
        .tvcount = 7,
        .tv = md5_hmac_general_test_vector,
        .mech = { CKM_MD5_HMAC_GENERAL, &four, sizeof(CK_ULONG) },
    },
    {
        .name = "IBM-SHA3-224 HMAC",
        .tvcount = 6,
        .tv = sha3_224_hmac_test_vector,
        .mech = { CKM_IBM_SHA3_224_HMAC, 0, 0 },
    },
    {
        .name = "IBM-SHA3-256 HMAC",
        .tvcount = 6,
        .tv = sha3_256_hmac_test_vector,
        .mech = { CKM_IBM_SHA3_256_HMAC, 0, 0 },
    },
    {
        .name = "IBM-SHA3-384 HMAC",
        .tvcount = 6,
        .tv = sha3_384_hmac_test_vector,
        .mech = { CKM_IBM_SHA3_384_HMAC, 0, 0 },
    },
    {
        .name = "IBM-SHA3-512 HMAC",
        .tvcount = 6,
        .tv = sha3_512_hmac_test_vector,
        .mech = { CKM_IBM_SHA3_512_HMAC, 0, 0 },
    },

};
