/* SPDX-License-Identifier: GPL-2.0 */
#ifndef _VIRTIO_MMIO_DRAGONBALL_H
#define _VIRTIO_MMIO_DRAGONBALL_H
/*
 * The MMIO transport layer defined in the virtio specification has some
 * limitations, which may cause performance penalties when handling interrupts
 * or raising the doorbell. So a series of extensions to the virtio MMIO
 * transport layer has been proposed and implemented here to improve virtio
 * device performance.
 *
 * Ideally this should be implemented as extensions to the MMIO transport layer
 * instead of ulgily hard-coding.
 */

/*
 * MMIO device vendor ID with feature extensions for Dragonball.
 * The upper 24 bits are used as vendor id, the lower 8 bits are used as
 * feature bits.
 */
#define VIRTIO_MMIO_VENDOR_DRAGONBALL	0xdbfcdb00
#define VIRTIO_MMIO_VENDOR_DB_ID_MASK	0xffffff00

/* Optimization of the legacy interrupt status register
 * When this feature is present, the device won't generate interrupts for device
 * configuration change and will only generate interrupts for virtque data ready
 * events when the device is working in legacy_irq mode. So the legacy interrupt
 * handle doesn't need to read/clear the interrupt status register.
 */
#define VIRTIO_MMIO_VENDOR_DB_INTR_USED	0x00000001

/* Per-virtque interrupt(Message Signaled Interurpt).
 * When this feature is present, the virtio device implements n_virtque + 1
 * MSI interrupts. Unlike PCI MSI/MSIx, the MSI vectors have fixed relationship
 * with interrupt event sources. The first interrupt is for device configuraiton
 * change events, the second interrupt is for virtque 1, the third interrupt
 * is for virtque 2 and so on. There's no way for the OS to configure number
 * of enabled MSI vectors. The OS could only enable/disable MSI interrupts as
 * a whole.
 */
#define VIRTIO_MMIO_VENDOR_DB_MSI_INTR	0x00000002

/* Per-virtque doorbell, like PCI virtio devices.
 * When VIRTIO_MMIO_VENDOR_DB_DOORBELL is present, the VIRTIO_MMIO_QUEUE_NOTIFY
 * register becomes read-write. On reading, the lower 16-bit contains doorbell
 * base offset starting from the MMIO window base, and the upper 16-bit contains
 * scale for the offset. The notification register address for virque is:
 *	base + doorbell_base + doorbell_scale * queue_idx
 */
#define VIRTIO_MMIO_VENDOR_DB_DOORBELL	0x00000004

/*
 * When VIRTIO_MMIO_VENDOR_DB_PVDMA is set, pvDMA will be used for the device.
 */
#define VIRTIO_MMIO_VENDOR_DB_PVDMA	0x00000008

static inline bool check_vendor_dragonball(u32 id)
{
	return (id & VIRTIO_MMIO_VENDOR_DB_ID_MASK) ==
		VIRTIO_MMIO_VENDOR_DRAGONBALL;
}

static inline bool check_vendor_db_intr_used(u32 id)
{
	return check_vendor_dragonball(id) &&
		(id & VIRTIO_MMIO_VENDOR_DB_INTR_USED);
}

static inline bool check_vendor_db_msi(u32 id)
{
	return check_vendor_dragonball(id) &&
		(id & VIRTIO_MMIO_VENDOR_DB_MSI_INTR);
}

static inline bool check_vendor_db_doorbell(u32 id)
{
	return check_vendor_dragonball(id) &&
		(id & VIRTIO_MMIO_VENDOR_DB_DOORBELL);
}

static inline bool check_vendor_db_pvdma(u32 id)
{
	return check_vendor_dragonball(id) &&
		(id & VIRTIO_MMIO_VENDOR_DB_PVDMA);
}

#endif /* _VIRTIO_MMIO_DRAGONBALL_H */
