/* SPDX-License-Identifier: GPL-2.0 */
/*
 * Copyright (C) 2021 Alibaba Cloud. or its affiliates. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 */

#ifndef _DRAGONBALL_PLATFORM_FEATURE_H
#define _DRAGONBALL_PLATFORM_FEATURE_H

#include <linux/types.h>
#include <linux/irqreturn.h>
#include "dragonball.h"

/*
 * The `common_platform_feature` struct specifies the feature's info which
 * can get from dragonball vmm by feature's common probe logic.
 */
struct common_platform_feature {
	/* pointer to dragonball device */
	struct dragonball_device *dragonball_device;
	/* physical address of mmio space's base */
	phys_addr_t phys_base;
	/* logic address of mmio space's base */
	void __iomem *mmio_base;
	/* mmio space's size */
	resource_size_t mmio_size;
	/* logic address of config space's base */
	void __iomem *cfg_base;
	/* common capabilities between driver and device */
	u32 acked_caps;
	/* feature's version */
	u32 version;
	/* the number of msi irq */
	u32 msi_irq_nr;
	/* the base of msi irq */
	u32 msi_irq_base;
};

/*
 * The `platform_feature_driver` struct specifies the info and functions
 * that provided by the feature's driver.
 */
struct platform_feature_driver {
	/* the type of feature */
	enum dragonball_features feature_type;
	/* the capabilities of feature */
	u32 caps;
	/* driver's version */
	u32 version;
	/* negoitiate the version between feature and driver */
	int (*check_version)(struct common_platform_feature *feature);
	/* negoitiate the capabilitiy between feature and driver */
	u32 (*ack_cap)(struct common_platform_feature *feature, u32 caps);
	/* start the feature */
	int (*start)(struct common_platform_feature *feature);
	/* handle legacy irq */
	irqreturn_t (*legacy_irq)(struct common_platform_feature *feature);
};

extern int register_db_platform_feature(struct common_platform_feature *feature,
					struct platform_feature_driver driver);

extern int probe_db_platform_features(struct dragonball_device *db_dev);

extern irqreturn_t
db_features_handle_interrupt(struct dragonball_device *db_dev);

#endif
