// SPDX-License-Identifier: GPL-2.0

/*
 * Real-Time Scheduling Class (used by rt_overrider_credit module)
 */

#include <linux/sched.h>
#include "sched.h"

/*
 * qemu_head is a list which maintain all qemu task. This list will not
 * be released as the module is removed. when the rt_overroder_credit module
 * is rmmod, qemu_head can still preserve all credit information. It is a
 * very important character for rt_overrider_credit module upgrades.
 */
LIST_HEAD(qemu_head);

static struct rt_overrider_class *curr_rt_overrider;

bool is_curr_rt_overrider(void)
{
	return curr_rt_overrider != NULL;
}

void rt_overrider_task_proc_show(struct task_struct *p, struct seq_file *m)
{
	struct rt_overrider_class *rt_overrider;

	rcu_read_lock();
	rt_overrider = rcu_dereference(curr_rt_overrider);
	if (rt_overrider && rt_overrider->task_proc_show)
		rt_overrider->task_proc_show(p, m);
	rcu_read_unlock();
}

void rt_overrider_task_proc_write(struct task_struct *p, char *buf)
{
	struct rt_overrider_class *rt_overrider;

	rcu_read_lock();
	rt_overrider = rcu_dereference(curr_rt_overrider);
	if (rt_overrider && rt_overrider->task_proc_write)
		rt_overrider->task_proc_write(p, buf);
	rcu_read_unlock();
}

void rt_overrider_task_release(struct task_struct *p)
{
	struct rt_overrider_class *rt_overrider;

	rcu_read_lock();
	rt_overrider = rcu_dereference(curr_rt_overrider);
	if (rt_overrider && rt_overrider->task_release)
		rt_overrider->task_release(p);
	rcu_read_unlock();
}

int rt_overrider_prev_setsched(struct task_struct *p,
		   int policy, int old_policy)
{
	int ret = 0;
	struct rt_overrider_class *rt_overrider;

	rcu_read_lock();
	rt_overrider = rcu_dereference(curr_rt_overrider);
	if (rt_overrider && rt_overrider->prev_setsched)
		ret = rt_overrider->prev_setsched(p, policy, old_policy);
	rcu_read_unlock();
	return ret;
}

void rt_overrider_post_setsched(struct task_struct *p,
		   int policy, int old_policy, int ret)
{
	struct rt_overrider_class *rt_overrider;

	rcu_read_lock();
	rt_overrider = rcu_dereference(curr_rt_overrider);
	if (rt_overrider && rt_overrider->post_setsched)
		rt_overrider->post_setsched(p, policy, old_policy, ret);
	rcu_read_unlock();
}

void rt_overrider_balance_runtime(struct rt_rq *rt_rq)
{
	struct rt_overrider_class *rt_overrider;

	rcu_read_lock();
	rt_overrider = rcu_dereference(curr_rt_overrider);
	if (rt_overrider && rt_overrider->balance_runtime)
		rt_overrider->balance_runtime(rt_rq);
	rcu_read_unlock();
}

int register_rt_overrider(struct rt_overrider_class *rt_overrider)
{
	curr_rt_overrider = rt_overrider;
	return 0;
}
EXPORT_SYMBOL_GPL(register_rt_overrider);

void unregister_rt_overrider(struct rt_overrider_class *rt_overrider)
{
	curr_rt_overrider = NULL;
}
EXPORT_SYMBOL_GPL(unregister_rt_overrider);
