// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2020-2022 Alibaba Corporation. All rights reserved.
 * Author: Zelin Deng <zelin.deng@linux.alibaba.com>
 * Author: Guanjun <guanjun@linux.alibaba.com>
 * Author: Jiayu Ni <jiayu.ni@linux.alibaba.com>
 */

#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include "../utils/utils.h"
#include "rng.h"

struct rng_alg_entry *rng_alg_entries;

struct rng_result {
	struct completion completion;
	int err;
};

#define RNG_REQ_GET_COMPLETION(req)					\
	(&((struct rng_result *)(req)->data)->completion)		\

static inline struct rng_alg *find_rng_alg(const char *alg_name)
{
	struct rng_alg_entry *cur = rng_alg_entries;
	struct rng_alg *alg = NULL;

	while (cur) {
		if (!strncmp(alg_name, cur->alg->name,
			     strlen(cur->alg->name))) {
			alg = cur->alg;
			break;
		}
		cur = cur->next;
	}

	return alg;
}

struct rng_ctx *rng_alloc_ctx(const char *alg_name, uint32_t flag)
{
	struct rng_alg *alg;
	struct rng_ctx *ctx;

	alg = find_rng_alg(alg_name);
	if (!alg) {
		ycc_err("Failed to find alg:%s\n", alg_name);
		return NULL;
	}

	ctx = malloc(sizeof(struct rng_ctx) + alg->ctxsize);
	if (!ctx) {
		ycc_err("Failed to alloc ctx for alg:%s\n", alg_name);
		return NULL;
	}

	memset(ctx, 0, sizeof(struct rng_ctx) + alg->ctxsize);
	ctx->alg = alg;
	if (alg->init && alg->init(ctx)) {
		ycc_err("Failed to do init for alg:%s\n", alg_name);
		free(ctx);
		return NULL;
	}

	ctx->flags |= flag;

	/* reqsize & __ctx are initialized by real algrithm driver */
	return ctx;
}

void rng_free_ctx(struct rng_ctx *ctx)
{
	struct rng_alg *alg;

	if (ctx) {
		alg = ctx->alg;
		if (alg->exit)
			alg->exit(ctx);

		free(ctx);
	}
}

static void rng_complete(struct rng_ctx *ctx, int err)
{
	struct rng_result *result = (struct rng_result *)ctx->data;

	result->err = err;
	complete(&result->completion);
}

int rng_generate(struct rng_ctx *ctx, const uint8_t *src, uint32_t slen,
		 uint8_t *dst, uint32_t dlen)
{
	struct rng_alg *alg;
	int ret = -ENOMEM;

	alg = ctx->alg;

	ctx->data = malloc(sizeof(struct rng_result));
	if (!ctx->data)
		goto out;

	ctx->complete = rng_complete;
	init_completion(RNG_REQ_GET_COMPLETION(ctx));

	ctx->result = dst;
	ctx->len = dlen;

	ret = alg->generate(ctx);

	if (ret != -EINPROGRESS)
		goto free_data;

	/* SYNC for random number generator */
	ret = wait_for_completion_timeout(RNG_REQ_GET_COMPLETION(ctx),
					  COMPLETION_TIMEOUT_SECS);
	if (ret < 0)
		ycc_err("Rng result timeout\n");

free_data:
	free(ctx->data);
out:
	return ret;
}

int rng_seed(struct rng_ctx *ctx, const uint8_t *seed, unsigned int slen)
{
	return 0;
}

uint32_t rng_seedsize(struct rng_ctx *ctx)
{
	return 0;
}

void __attribute__((weak)) rng_register_algs(void) {}
void __attribute__((weak)) rng_unregister_algs(void) {}
