// SPDX-License-Identifier: GPL-2.0
#ifndef __YCC_RING_H
#define __YCC_RING_H

#include <linux/types.h>
#include <stdbool.h>
#include <stdint.h>

#define YCC_CMD_DESC_SIZE		64
#define YCC_RESP_DESC_SIZE		16

#define YITIAN_CPU_NUM_PER_DIE		64

#define YCC_RING_MIN_DESC		256
#define YCC_RING_MAX_DESC		4096
#define YCC_RING_DEFAULT_DESC		512

/* ring register offset */
#define REG_RING_CMD_BASE_ADDR_LO	0x00
#define REG_RING_CMD_BASE_ADDR_HI	0x04
#define REG_RING_CMD_WR_PTR		0x08
#define REG_RING_CMD_RD_PTR		0x0C
#define REG_RING_RSP_BASE_ADDR_LO	0x10
#define REG_RING_RSP_BASE_ADDR_HI	0x14
#define REG_RING_RSP_WR_PTR		0x18
#define REG_RING_RSP_RD_PTR		0x1C
#define REG_RING_CFG			0x20
#define REG_RING_TO_TH			0x24
#define REG_RING_STATUS			0x28
#define REG_RING_PENDING_CMD		0x2C
#define REG_RING_RSP_WR_SHADOW_PTR	0x30
#define REG_RING_RSP_AFULL_TH		0x34

#define CMD_RESP_TAG			"YCC response: "

#define CMD_ILLEGAL			0x15
#define CMD_YCC_UNDERATTACK		0x25
#define CMD_INVALID			0x35
#define CMD_ERROR			0x45
#define CMD_EXCESS			0x55
#define CMD_KEY_ERROR			0x65
#define CMD_VERIFY_ERROR		0x85
#define CMD_SUCCESS			0xa5
#define CMD_CANCELLED			0xff

#define CMD_INVALID_CONTENT_U8		0x7f
#define CMD_INVALID_CONTENT_U64		0x7f7f7f7f7f7f7f7fULL

#ifndef dma_addr_t
typedef __u64 dma_addr_t;
#endif
#ifndef atomic_t
typedef uint32_t atomic_t;
#endif
#ifndef __iomem
#define __iomem
#endif
#ifndef resource_size_t
typedef uint64_t resource_size_t;
#endif

#define YCC_CSR_WR(csr_base, csr_offset, val)		\
	(*(uint32_t *)((csr_base) + (csr_offset)) = (uint32_t)(val))
#define YCC_CSR_RD(csr_base, csr_offset)		\
	(*(uint32_t *)((csr_base) + (csr_offset)))

enum ring_type {
	FREE_RING,
	USER_RING,
	KERN_RING,
};

struct ycc_ring {
	uint8_t dev_id;
	uint16_t ring_id;
	atomic_t refcnt;
	void __iomem *csr_addr;
	resource_size_t csr_paddr;

	uint32_t max_desc;
	uint32_t irq_th;

	uint8_t type;

	void *cmd_base_vaddr;
	dma_addr_t cmd_base_paddr;
	uint16_t cmd_wr_ptr;
	uint16_t cmd_rd_ptr;
	void *resp_base_vaddr;
	dma_addr_t resp_base_paddr;
	uint16_t resp_wr_ptr;
	uint16_t resp_rd_ptr;

	pthread_t dequeue_thread;
	pthread_spinlock_t enqueue_lock;
};

struct ycc_flags {
	void *ptr;
	int (*ycc_done_callback)(void *ptr, uint16_t state);
};

struct ycc_resp_desc {
	uint64_t private_ptr;
	uint16_t state;
	uint8_t reserved[6];
};

struct ycc_skcipher_cmd {
	uint8_t cmd_id;
	uint8_t mode;
	uint64_t sptr:48;
	uint64_t dptr:48;
	uint32_t dlen;
	uint16_t key_idx; /* key used to decrypt kek */
	uint16_t kek_idx; /* encrypted key used to decrypt key_idx */
	uint64_t keyptr:48;
	uint8_t padding;
} __attribute__((packed));

struct ycc_aead_cmd {
	uint8_t cmd_id;
	uint8_t mode;
	uint64_t sptr:48; /* aad + payload */
	uint64_t dptr:48; /* encrypted/decrypted + tag */
	uint32_t dlen;
	uint16_t key_idx;
	uint16_t kek_idx;
	uint64_t keyptr:48;
	uint16_t aadlen;
	uint8_t taglen; /* authenc size */
} __attribute__((packed));

struct ycc_rng_cmd {
	uint8_t cmd_id;
	uint64_t dptr:48;
	uint32_t dlen;
} __attribute__((packed));

struct ycc_rsa_enc_cmd {
	uint8_t cmd_id;
	uint64_t sptr:48;
	uint16_t key_id;
	uint64_t keyptr:48; /* public key */
	uint16_t elen;
	uint16_t nlen;
	uint64_t dptr:48;
} __attribute__((packed));

struct ycc_rsa_dec_cmd {
	uint8_t cmd_id;
	uint64_t sptr:48;
	uint16_t key_id;
	uint16_t kek_id;
	uint64_t keyptr:48; /* private key */
	uint16_t elen;
	uint16_t nlen;
	uint64_t dptr:48;
} __attribute__((packed));

struct ycc_ecdsa_sign_cmd {
	uint8_t cmd_id;
	uint8_t curve_id;
	uint64_t sptr:48;
	uint8_t sig_len;
	uint16_t key_id;
	uint16_t kek_id;
	uint64_t keyptr:48;
	uint64_t dptr:48;
} __attribute__((packed));

struct ycc_ecdsa_verify_cmd {
	uint8_t cmd_id;
	uint8_t curve_id;
	uint64_t sptr:48;
	uint8_t sig_len;
	uint16_t key_id;
	uint64_t keyptr:48;
} __attribute__((packed));

struct ycc_ecc_gen_pub_cmd {
	uint8_t cmd_id;
	uint8_t curve_id;
	uint16_t key_idx;
	uint16_t kek_idx;
	uint64_t priv_ptr:48;
	uint64_t pub_ptr:48;
} __attribute__((packed));

struct ycc_ecc_gen_key_pair_cmd {
	uint8_t cmd_id;
	uint8_t curve_id;
	uint16_t kek_idx;
	uint64_t pin_ptr:48;
	uint64_t priv_ptr:48;
	uint64_t pub_ptr:48;
} __attribute__((packed));

struct ycc_ecdh_gen_shared_cmd {
	uint8_t cmd_id;
	uint8_t curve_id;
	uint16_t key_idx;
	uint16_t kek_idx;
	uint16_t out_kek_idx;
	uint64_t pub_ptr:48;
	uint64_t priv_ptr:48;
	uint64_t shared_ptr:48;
} __attribute__((packed));

typedef union {
	struct ycc_ecc_gen_pub_cmd gen_pub_cmd;
	struct ycc_ecc_gen_key_pair_cmd gen_key_pair_cmd;
	struct ycc_ecdh_gen_shared_cmd gen_shared_cmd;
} ycc_kpp_cmd_t;

typedef union {
	struct ycc_skcipher_cmd ske_cmd;
	struct ycc_aead_cmd aead_cmd;

	struct ycc_rng_cmd rng_cmd;

	struct ycc_rsa_enc_cmd rsa_enc_cmd;
	struct ycc_rsa_dec_cmd rsa_dec_cmd;
	struct ycc_ecdsa_sign_cmd ecdsa_sign_cmd;
	struct ycc_ecdsa_verify_cmd ecdsa_verify_cmd;

	ycc_kpp_cmd_t kpp_cmd;

	uint8_t padding[32];
} ycc_real_cmd_t;

struct ycc_cmd_desc {
	ycc_real_cmd_t cmd;
	uint64_t private_ptr;
	uint8_t reserved0[16];
	uint8_t reserved1[8];
} __attribute__((packed));

int ycc_enqueue(struct ycc_ring *ring, void *cmd);
void ycc_dequeue(struct ycc_ring *ring);
struct ycc_ring *ycc_crypto_get_ring(void);
void ycc_crypto_free_ring(struct ycc_ring *ring);
int ycc_init_ring(struct ycc_ring *ring, uint32_t max_desc);
void ycc_release_ring(struct ycc_ring *ring);

#endif
