// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2020-2022 Alibaba Corporation. All rights reserved.
 * Author: Zelin Deng <zelin.deng@linux.alibaba.com>
 * Author: Guanjun <guanjun@linux.alibaba.com>
 * Author: Jiayu Ni <jiayu.ni@linux.alibaba.com>
 */

#include <openssl/rsa.h>
#include <openssl/pem.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "ycc_uio.h"
#include "utils.h"
#include "kpp.h"

static bool inited;

/*
 * Destructor: It is must if driver initialize successfully.
 */
static void __attribute__((destructor)) drv_exit(void)
{
	if (inited)
		ycc_drv_exit();
}

int main(int argc, char *argv[])
{
	EC_KEY *ecPriv, *ecPub;
	EVP_PKEY *pubKey;
	struct kpp_ctx *cipher;
	struct kpp_req *req;
	unsigned char priv[128];
	unsigned char pub[256];
	FILE *fp;
	int ret;

	ret = ycc_drv_init(1);
	if (ret < 0)
		return ret;

	inited = true;
	ret = -1;
	pubKey = EVP_PKEY_new();
	ecPriv = EC_KEY_new();
	ecPub = EC_KEY_new();
	fp = fopen("./ecdh_priv_key.pem", "rb");
	if (!fp)
		goto out;

	PEM_read_ECPrivateKey(fp, &ecPriv, NULL, NULL);
	if (!ecPriv)
		goto close_fp;

	cipher = kpp_alloc_ctx("ecdh-nist-p256", CRYPTO_SYNC);
	if (!cipher)
		goto close_fp;

	req = kpp_alloc_req(cipher);
	if (!req)
		goto free_ctx;

	/* private key length is 32 bytes, public key length is 64 bytes */
	kpp_set_req(req, priv, 32, pub, 64);
	ret = kpp_generate_key_pair(req);
	if (ret < 0)
		goto free_req;

	hex_dump("KPP generated private key:", priv, 32);
	hex_dump("KPP generated public key:", pub, 64);

	ret = kpp_set_secret(cipher, (void *)priv, 32);
	if (ret < 0)
		goto free_req;

	ret = kpp_generate_public_key(req);
	if (!ret)
		hex_dump("KPP verified public key:", pub, 64);

free_req:
	kpp_free_req(req);
free_ctx:
	kpp_free_ctx(cipher);
close_fp:
	fclose(fp);
out:
	EVP_PKEY_free(pubKey);
	EC_KEY_free(ecPriv);
	EC_KEY_free(ecPub);
	return ret;
}
