// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2020-2022 Alibaba Corporation. All rights reserved.
 * Author: Zelin Deng <zelin.deng@linux.alibaba.com>
 * Author: Guanjun <guanjun@linux.alibaba.com>
 * Author: Jiayu Ni <jiayu.ni@linux.alibaba.com>
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "ycc_uio.h"
#include "utils.h"
#include "kpp.h"

struct kpp_test {
	const char *name;
	const char *secret;
	const char *ep;
	const char *es;
	const char *pp;
};

static const struct kpp_test kpp = {
	.name = "ecdh-nist-p256",
	.secret = "\x24\xd1\x21\xeb\xe5\xcf\x2d\x83"
		  "\xf6\x62\x1b\x6e\x43\x84\x3a\xa3"
		  "\x8b\xe0\x86\xc3\x20\x19\xda\x92"
		  "\x50\x53\x03\xe1\xc0\xea\xb8\x82",
	.ep = "\x1a\x7f\xeb\x52\x00\xbd\x3c\x31"
	      "\x7d\xb6\x70\xc1\x86\xa6\xc7\xc4"
	      "\x3b\xc5\x5f\x6c\x6f\x58\x3c\xf5"
	      "\xb6\x63\x82\x77\x33\x24\xa1\x5f"
	      "\x6a\xca\x43\x6f\xf7\x7e\xff\x02"
	      "\x37\x08\xcc\x40\x5e\x7a\xfd\x6a"
	      "\x6a\x02\x6e\x41\x87\x68\x38\x77"
	      "\xfa\xa9\x44\x43\x2d\xef\x09\xdf",
	.es = "\xea\x17\x6f\x7e\x6e\x57\x26\x38"
	      "\x8b\xfb\x41\xeb\xba\xc8\x6d\xa5"
	      "\xa8\x72\xd1\xff\xc9\x47\x3d\xaa"
	      "\x58\x43\x9f\x34\x0f\x8c\xf3\xc9",
	.pp = "\xcc\xb4\xda\x74\xb1\x47\x3f\xea"
	      "\x6c\x70\x9e\x38\x2d\xc7\xaa\xb7"
	      "\x29\xb2\x47\x03\x19\xab\xdd\x34"
	      "\xbd\xa8\x2c\x93\xe1\xa4\x74\xd9"
	      "\x64\x63\xf7\x70\x20\x2f\xa4\xe6"
	      "\x9f\x4a\x38\xcc\xc0\x2c\x49\x2f"
	      "\xb1\x32\xbb\xaf\x22\x61\xda\xcb"
	      "\x6f\xdb\xa9\xaa\xfc\x77\x81\xf3",
};

static bool inited;

/*
 * Destructor: It is must if driver initialize successfully.
 */
static void __attribute__((destructor)) drv_exit(void)
{
	if (inited)
		ycc_drv_exit();
}

int main(int argc, char *argv[])
{
	struct kpp_ctx *cipher;
	struct kpp_req *req;
	unsigned char priv[128];
	unsigned char pub[256];
	unsigned char shared[128];
	int ret;

	ret = ycc_drv_init(1);
	if (ret < 0)
		return ret;

	inited = true;
	memcpy(priv, kpp.secret, 32);
	memcpy(pub, kpp.pp, 64);

	cipher = kpp_alloc_ctx("ecdh-nist-p256", CRYPTO_SYNC);
	if (!cipher)
		goto out;

	req = kpp_alloc_req(cipher);
	if (!req)
		goto free_ctx;

	ret = kpp_set_secret(cipher, (void *)priv, 32);
	if (ret < 0)
		goto free_req;

	/* public key length is 64 bytes, shared key length is 32 bytes */
	kpp_set_req(req, pub, 64, shared, 32);

	ret = kpp_compute_shared_secret(req);
	if (!ret)
		hex_dump("KPP shared key:", shared, 32);

free_req:
	kpp_free_req(req);
free_ctx:
	kpp_free_ctx(cipher);
out:
	return ret;
}
