#ifndef __UDMA_ULIB_H__
#define __UDMA_ULIB_H__

#include <linux/version.h>
#include <linux/types.h>
#include <linux/ioctl.h>
#include <stdio.h>

#ifndef dma_addr_t
typedef __u64 dma_addr_t;
#endif

#ifndef atomic_t
typedef uint32_t atomic_t;
#endif

#ifndef DIV_ROUND_UP
#define DIV_ROUND_UP(x, y)	(((x) + (y) - 1) / (y))
#endif

#ifndef PAGE_SIZE
#define PAGE_SIZE		(4096)
#endif

enum slab_type {
	NORMAL_type	= 0,
	LARGE_type,
	HUGE_type,
};

/* Only support 5 page table levels */
#define PAGE_TABLE_LEVEL	5

/* udma minimum allocated granularity is 1024B */
#define UNIT_SIZE		(1024)

#define MAX_ALLOC_SIZE		(32768)  /* max malloc size 32K */

#define LARGE_ALLOC_SIZE	(1 << 22) /* large alloc size 4M */

/* udma normal memblock is 2MB */
#define MEMBLOCK_SIZE_NORMAL	(2*1024*1024)  /* normal */
#define MEMBLOCK_SIZE_LARGE	(4*1024*1024)  /* large */
#define MEMBLOCK_SIZE_HUGE	(32*1024*1024)  /* not supported */

#define MASK_SIZE_2M (0x200000UL - 1)

#define UNITS_NUM		(MEMBLOCK_SIZE_NORMAL / UNIT_SIZE)
#define BITMAP_LEN		(UNITS_NUM / sizeof(__u64) / 8)

/* Number of hash_key buckets is 4KB */
#define HASH_BUCKETS_SHIFT	12
#define HASH_BUCKETS		(1 << HASH_BUCKETS_SHIFT)
#define HASH_BUCKETS_MASK	(HASH_BUCKETS - 1)

#define YCC_CMD_MAGIC		'm'
#define YCC_CMD_MEM_ALLOC	(0)
#define YCC_CMD_MEM_FREE	(1)
#define YCC_CMD_LARGE_MEM_ALLOC	(2)
#define YCC_CMD_LARGE_MEM_FREE	(3)
#define YCC_CMD_GET_UIO_INFO	(4)

#define YCC_IOC_MEM_ALLOC	\
	_IOWR(YCC_CMD_MAGIC, YCC_CMD_MEM_ALLOC, struct ycc_udma_info)
#define YCC_IOC_MEM_FREE	\
	_IOWR(YCC_CMD_MAGIC, YCC_CMD_MEM_FREE, struct ycc_udma_info)
#define YCC_IOC_LARGE_MEM_ALLOC	\
	_IOWR(YCC_CMD_MAGIC, YCC_CMD_LARGE_MEM_ALLOC, struct ycc_udma_large_info)
#define YCC_IOC_LARGE_MEM_FREE	\
	_IOWR(YCC_CMD_MAGIC, YCC_CMD_LARGE_MEM_FREE, struct ycc_udma_large_info)
#define YCC_IOC_GET_UIO_INFO	\
	_IOR(YCC_CMD_MAGIC, YCC_CMD_GET_UIO_INFO, struct ycc_uio_info)

struct ycc_udma_large_info {
	dma_addr_t dma_addr;
	void *virt_addr;
};

struct ycc_udma_info {
	int node;
	size_t size;
	int type;
	void *virt_addr;
	dma_addr_t dma_addr;
	char reserved[256];
};

struct ycc_uio_info {
	/* Version information */
	int major;
	int minor;
	/* Reserved for feature bitmap */
	int reserved[8];
};

struct ycc_udma_ctrl {
	struct ycc_udma_info mem_info;
	__u16 slab_lock[4];		/* For bitmap and sizes */
	__u64 bitmap[BITMAP_LEN];
	__u16 sizes[UNITS_NUM];
	atomic_t alloc_cnts;		/* Allocated blk-unit cnt in this slab */

#ifdef CONFIG_ADDR_CHECK
	struct ycc_udma_info *prev_user;
	struct ycc_udma_info *next_user;
#endif
	struct ycc_udma_ctrl *prev_user_percpu;
	struct ycc_udma_ctrl *next_user_percpu;
};

struct slab_list {
	struct ycc_udma_ctrl *head;
	struct ycc_udma_ctrl *tail;
};

/*
 * List operations for '_kernel', '_user' and 'percpu' mode.
 */
#define ADD_ELEMENT_TO_HEAD_LIST(node, head, tail, mode)	\
	do {							\
		node->prev##mode = NULL;			\
		if (!head) {					\
			tail = node;				\
			node->next##mode = NULL;		\
		} else {					\
			node->next##mode = head;		\
			head->prev##mode = node;		\
		}						\
		head = node;					\
	} while (0)

#define ADD_ELEMENT_TO_TAIL_LIST(node, head, tail, mode)	\
	do {							\
		node->next##mode = NULL;			\
		if (!head) {					\
			head = node;				\
			node->prev##mode			\
		} else {					\
			node->prev##mode = head;		\
			head->next##mode = node;		\
		}						\
		tail = node;					\
	} while (0)

#define REMOVE_ELEMENT_FROM_HEAD_LIST(head_pptr, tail, mode)	\
	do {							\
		if (!*head_pptr)				\
			return;					\
		*head_pptr->prev##mode = NULL;			\
		*head_pptr->next##mode = NULL;			\
		tail->prev##mode = NULL;			\
		*head_pptr = tail;				\
	} while (0)						\

#define FOR_EACH_ENTRY(pos, head, tail, mode)			\
	for (pos = head; pos != NULL; pos->next##mode)		\

#define udma_dbg(format, arg...)		\
	do {								\
		if (unlikely(udma_debug_mode))	\
			printf("UDMA debug: %s "format, __func__, ##arg);		\
	} while (0)

#define udma_info(format, arg...)	printf("UDMA info: "format, ##arg)
#define udma_warn(format, arg...)	printf("UDMA warn: "format, ##arg)
#define udma_err(format, arg...)	printf("UDMA error: "format, ##arg)

int ycc_udma_init(void);
void ycc_udma_exit(void);
void *ycc_udma_malloc(size_t size);
void ycc_udma_free(void *ptr);
unsigned long virt_to_phys(void *ptr);

void *ycc_udma_large_malloc(void);
void ycc_udma_large_free(void);
unsigned long virt_to_phys_large(void *ptr);

#endif
