# Copyright 2016 The Closure Rules Authors. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Rule for testing contents of text files generated by rules."""

def _impl(ctx):
    exe = ctx.outputs.executable
    file_ = ctx.file.file
    content = ctx.attr.content
    regexp = ctx.attr.regexp
    matches = ctx.attr.matches
    if content and regexp:
        fail("Must specify one and only one of content or regexp")
    if content and matches != -1:
        fail("matches only makes sense with regexp")
    if not regexp:
        dat = ctx.actions.declare_file(exe.basename + ".dat")
        ctx.actions.write(
            output = dat,
            content = content,
        )
        ctx.actions.write(
            output = exe,
            content = "diff -u %s %s" % (dat.short_path, file_.short_path),
            is_executable = True,
        )
        return struct(runfiles = ctx.runfiles([exe, dat, file_]))
    if matches != -1:
        script = "[ %s == $(grep -c %s %s) ]" % (
            matches,
            repr(regexp),
            file_.short_path,
        )
    else:
        script = "grep %s %s" % (repr(regexp), file_.short_path)
    if ctx.attr.invert:
        script = "! " + script
    ctx.actions.write(
        output = exe,
        content = script,
        is_executable = True,
    )
    return struct(runfiles = ctx.runfiles([exe, file_]))

_file_test = rule(
    attrs = {
        "file": attr.label(
            mandatory = True,
            allow_single_file = True,
        ),
        "content": attr.string(default = ""),
        "regexp": attr.string(default = ""),
        "matches": attr.int(default = -1),
        "invert": attr.bool(),
    },
    executable = True,
    implementation = _impl,
    test = True,
)

def file_test(size = "small", **kwargs):
    _file_test(size = size, **kwargs)
