/**
 * Copyright (c) 2021 OceanBase
 * OceanBase CE is licensed under Mulan PubL v2.
 * You can use this software according to the terms and conditions of the Mulan PubL v2.
 * You may obtain a copy of Mulan PubL v2 at:
 *          http://license.coscl.org.cn/MulanPubL-2.0
 * THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
 * EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
 * MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
 * See the Mulan PubL v2 for more details.
 */

#define USING_LOG_PREFIX SHARE

#include "share/ob_column_checksum_error_operator.h"
#include "share/ob_server_struct.h"

namespace oceanbase
{
namespace share
{
using namespace oceanbase::common;
using namespace oceanbase::common::sqlclient;

///////////////////////////////////////////////////////////////////////////////

bool ObColumnChecksumErrorInfo::is_valid() const
{
  return (tenant_id_ != OB_INVALID_TENANT_ID) && (frozen_scn_.is_valid())
         && (data_table_id_ != OB_INVALID_ID) && (index_table_id_ != OB_INVALID_ID);
}

void ObColumnChecksumErrorInfo::reset()
{
  tenant_id_ = OB_INVALID_TENANT_ID;
  frozen_scn_.reset();
  is_global_index_ = false;
  data_table_id_ = OB_INVALID_ID;
  index_table_id_ = OB_INVALID_ID;
  data_tablet_id_.reset();
  index_tablet_id_.reset();
  column_id_ = OB_INVALID_ID;
  data_column_checksum_ = -1;
  index_column_checksum_ = -1;
}

///////////////////////////////////////////////////////////////////////////////

int ObColumnChecksumErrorOperator::insert_column_checksum_err_info(
    ObISQLClient &sql_client,
    const uint64_t tenant_id,
    const ObColumnChecksumErrorInfo &info)
{
  return insert_column_checksum_err_info_(sql_client, tenant_id, info);
}

int ObColumnChecksumErrorOperator::insert_column_checksum_err_info_(
    ObISQLClient &sql_client,
    const uint64_t tenant_id,
    const ObColumnChecksumErrorInfo &info)
{
    int ret = OB_SUCCESS;
  ObDMLSqlSplicer dml;
  int64_t affected_rows = 0;
  const uint64_t meta_tenant_id = gen_meta_tenant_id(tenant_id);
  ObDMLExecHelper exec(sql_client, meta_tenant_id);
  if (!info.is_valid()
      || (tenant_id != info.tenant_id_)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", KR(ret), K(tenant_id), K(info));
  } else if (OB_FAIL(dml.add_pk_column("tenant_id", tenant_id))
            || OB_FAIL(dml.add_pk_column("frozen_scn", info.frozen_scn_.get_val_for_inner_table_field()))
            || OB_FAIL(dml.add_pk_column("index_type", info.is_global_index_))
            || OB_FAIL(dml.add_pk_column("data_table_id", info.data_table_id_))
            || OB_FAIL(dml.add_pk_column("index_table_id", info.index_table_id_))
            || OB_FAIL(dml.add_pk_column("data_tablet_id", info.data_tablet_id_.id()))
            || OB_FAIL(dml.add_pk_column("index_tablet_id", info.index_tablet_id_.id()))
            || OB_FAIL(dml.add_column("column_id", info.column_id_))
            || OB_FAIL(dml.add_column("data_column_ckm", info.data_column_checksum_))
            || OB_FAIL(dml.add_column("index_column_ckm", info.index_column_checksum_))) {
    LOG_WARN("fail to add pk column", KR(ret), K(tenant_id), K(info));
  } else if (OB_FAIL(exec.exec_insert_update(OB_ALL_COLUMN_CHECKSUM_ERROR_INFO_TNAME, dml, affected_rows))) {
    LOG_WARN("fail to splice exec_insert_update", KR(ret), K(meta_tenant_id), K(info));
  } else if (affected_rows < 0 || affected_rows > 2) {
    // one ckm_error info may insert multi-times due to verifying checksum multi-times. if re-insert, cuz we
    // use 'on duplicate key update', the 'affected_rows' may be 0(not pk_column values unchanged)
    // or 2(not pk_column values changed)
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("unexpected affected rows", KR(ret), K(affected_rows));
  }
  return ret;
}

int ObColumnChecksumErrorOperator::delete_column_checksum_err_info(
    ObISQLClient &sql_client,
    const uint64_t tenant_id,
    const SCN &min_frozen_scn)
{
  int ret = OB_SUCCESS;
  ObSqlString sql;
  int64_t affected_rows = 0;
  const uint64_t meta_tenant_id = gen_meta_tenant_id(tenant_id);
  if (OB_UNLIKELY((!is_valid_tenant_id(tenant_id))) || (!min_frozen_scn.is_valid())) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", KR(ret), K(tenant_id), K(min_frozen_scn));
  } else if (OB_FAIL(sql.assign_fmt("DELETE FROM %s WHERE tenant_id = '%lu' AND frozen_scn < %lu",
             OB_ALL_COLUMN_CHECKSUM_ERROR_INFO_TNAME, tenant_id, min_frozen_scn.get_val_for_inner_table_field()))) {
    LOG_WARN("fail to assign sql", KR(ret), K(tenant_id), K(min_frozen_scn));
  } else if (OB_FAIL(sql_client.write(meta_tenant_id, sql.ptr(), affected_rows))) {
    LOG_WARN("fail to execute sql", KR(ret), K(meta_tenant_id), K(sql));
  } else {
    LOG_INFO("succ to delete column checksum error info", K(tenant_id), K(min_frozen_scn), K(affected_rows));
  }
  return ret;
}

int ObColumnChecksumErrorOperator::delete_column_checksum_err_info_by_scn(
    common::ObISQLClient &sql_client,
    const uint64_t tenant_id,
    const int64_t compaction_scn)
{
  int ret = OB_SUCCESS;
  ObSqlString sql;
  int64_t affected_rows = 0;
  const uint64_t meta_tenant_id = gen_meta_tenant_id(tenant_id);
  if (OB_UNLIKELY((!is_valid_tenant_id(tenant_id))) || compaction_scn <= 0) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", KR(ret), K(tenant_id), K(compaction_scn));
  } else if (OB_FAIL(sql.assign_fmt("DELETE FROM %s WHERE tenant_id = '%lu' AND frozen_scn = %lu",
             OB_ALL_COLUMN_CHECKSUM_ERROR_INFO_TNAME, tenant_id, compaction_scn))) {
    LOG_WARN("fail to assign sql", KR(ret), K(tenant_id), K(compaction_scn));
  } else if (OB_FAIL(sql_client.write(meta_tenant_id, sql.ptr(), affected_rows))) {
    LOG_WARN("fail to execute sql", KR(ret), K(meta_tenant_id), K(sql));
  } else {
    LOG_INFO("succ to delete column checksum error info", K(tenant_id), K(compaction_scn), K(affected_rows));
  }
  return ret;
}

int ObColumnChecksumErrorOperator::check_exist_ckm_error_table(const uint64_t tenant_id, const int64_t compaction_scn, bool &exist)
{
  int ret = OB_SUCCESS;
  exist = false;
  ObSqlString sql;
  if (OB_UNLIKELY(0 == tenant_id || compaction_scn <= 0)) {
    ret = OB_INVALID_ARGUMENT;
    LOG_WARN("invalid argument", KR(ret), K(tenant_id), K(compaction_scn));
  } else if (OB_ISNULL(GCTX.sql_proxy_)) {
    ret = OB_ERR_UNEXPECTED;
    LOG_WARN("sql proxy is null", KR(ret));
  } else {
    const uint64_t meta_tenant_id = gen_meta_tenant_id(tenant_id);
    int64_t exist_cnt = 0;
    ObMySQLResult *result = NULL;
    SMART_VAR(ObMySQLProxy::MySQLResult, res) {
      if (OB_FAIL(sql.append_fmt(
          "SELECT count(*) > 0 as c FROM %s WHERE tenant_id = '%lu' AND frozen_scn = %ld",
          OB_ALL_COLUMN_CHECKSUM_ERROR_INFO_TNAME, tenant_id, compaction_scn))) {
        LOG_WARN("fail to assign sql", KR(ret), K(tenant_id), K(compaction_scn));
      } else if (OB_FAIL(GCTX.sql_proxy_->read(res, meta_tenant_id, sql.ptr()))) {
        LOG_WARN("fail to execute sql", KR(ret), K(meta_tenant_id), K(sql));
      } else if (OB_ISNULL(result = res.get_result())) {
        ret = OB_ERR_UNEXPECTED;
        LOG_WARN("fail to get result", KR(ret), K(meta_tenant_id), K(sql));
      } else if (OB_FAIL(result->get_int("c", exist_cnt))) {
        LOG_WARN("failed to get int", KR(ret), K(compaction_scn));
      } else if (exist_cnt > 0) {
        LOG_INFO("exist ckm error info", KR(ret), K(exist_cnt));
        exist = true;
      }
    }
  }
  return ret;
}

} // namespace share
} // namespace oceanbase
