#
# Copyright (C) 2019  Red Hat, Inc.
#
# This copyrighted material is made available to anyone wishing to use,
# modify, copy, or redistribute it subject to the terms and conditions of
# the GNU General Public License v.2, or (at your option) any later version.
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY expressed or implied, including the implied warranties of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.  You should have received a copy of the
# GNU General Public License along with this program; if not, write to the
# Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
# 02110-1301, USA.  Any Red Hat trademarks that are incorporated in the
# source code or documentation are not subject to the GNU General Public
# License and may only be used or replicated with the express permission of
# Red Hat, Inc.
#
# Red Hat Author(s): Radek Vykydal <rvykydal@redhat.com>
#
import unittest
from unittest.mock import Mock, patch
import tempfile
import shutil
import os
from textwrap import dedent
from pyanaconda.core.kickstart.commands import NetworkData

from pyanaconda.modules.network.ifcfg import IFCFG_DIR, IfcfgFile, \
    get_ifcfg_files_paths, get_ifcfg_file, get_ifcfg_file_of_device, \
    get_slaves_from_ifcfgs, get_kickstart_network_data, get_master_slaves_from_ifcfgs

HWADDR_TO_IFACE = {
    "52:54:00:0c:77:e3": "ens6",
    "52:54:00:23:74:76": "ens12",
    "52:54:00:ac:d2:d3": "ens13",
    "52:54:00:97:8E:14": "ens16",
    "52:54:00:82:B9:BA": "ens17",
    "52:54:00:e1:1d:70": "ens23",
    "52:54:00:d4:23:f8": "ens24",
    "52:54:00:5E:26:B5": "ens27",
    "52:54:00:35:BF:0F": "ens28",
    "52:54:00:70:10:7f": "ens32",
    "52:54:00:E5:A0:29": "ens34",
}

UUID_TO_IFACE = {
    "98282424-e386-446a-b9b1-dc08460b68e2": "ens36",
    "abf92663-cfad-4cf0-acd3-9ca6f4d45962": "ens38",
}

UUID_TO_TEAM_CONFIG = {
    "ffe9e996-fc2f-41d0-9e46-cc91ad5236c2": "{\"runner\": {\"name\": \"activebackup\"}}",
    "d1b392aa-54a6-4007-85e9-0bbf35419c2d": "{\"runner\": {\"name\": \"roundrobin\"}}",
    "9b2b1a7d-204e-464c-bfb5-94b755564d6e": "{\"runner\": {\"name\": \"activebackup\"}}",
    "b3a3655c-86cf-4ce3-9968-b7df242ce705": "{\"runner\": {\"name\": \"activebackup\"}}",
    "3949fb93-bddb-4159-a2ef-2b84819e0d41": "{\"runner\": {\"name\": \"activebackup\"}}",
}

UUID_TO_TEAM_PORT_CONFIG = {
    "1d41bcf0-b190-4915-8a74-45b6848c03a4": "{\"prio\": -10, \"sticky\": true}",
    "b45d9605-d206-4977-84e5-460cbd87e116": "{\"prio\": 100}",
    "ca692742-bdb5-41c9-a267-bbbe5c7d17e3": "",
    "75ebd3e5-64c2-4fe0-b1f4-a2e20d4b3801": "",
    "67e3b437-466d-4484-ad86-f580802b2849": "{\"prio\": -10, \"sticky\": true}",
    "7dc1094f-f07b-43e1-b700-26cafffd6eb9": "{\"prio\": 100}",
    "41f1df54-48b5-41b3-b88c-255d7a0b7339": "{\"prio\": -10, \"sticky\": true}",
    "2e736df9-ebfc-4edc-8cf9-a13ef65274ee": "{\"prio\": 100}",
    "dbefddc1-f463-4a4c-bf4b-d951283ec490": "{\"prio\": -10, \"sticky\": true}",
    "a074b1f8-65ff-4a97-80b1-d40ef461baab": "{\"prio\": 100}",
}

MASTER0_UUID = "aae55749-e69b-4aa7-92bb-9e8957f26a55"
MASTER1_UUID = "923e8e5c-05e8-452a-9b79-3e3b05776880"
MASTER0_NAME = "master0"
MASTER1_NAME = "master1"

DEVNAME_TO_UUID = {
    MASTER1_NAME: MASTER1_UUID,
    MASTER0_NAME: MASTER0_UUID
}

IFCFG_FILES_REAL_ALL_TYPES = [
    ### Ethernet
    # TODO: ipv6 configuration
    # Ethernet generated by dracut initrd
    ("ifcfg-ens3",
     """
     # Generated by dracut initrd
     NAME="ens3"
     DEVICE="ens3"
     ONBOOT=yes
     NETBOOT=yes
     UUID="a95d101b-66de-44e5-9640-a06f2c422a14"
     IPV6INIT=yes
     BOOTPROTO=dhcp
     TYPE=Ethernet
     """,
     """
     network  --bootproto=dhcp --device=ens3 --ipv6=auto
     """),
    # Ethernet generated by parse-kickstart
    ("ifcfg-ens5",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     DEVICE="ens5"
     UUID="d191a0d9-e7ea-4351-baad-8ed7558a294e"
     ONBOOT="no"
     BOOTPROTO="static"
     IPADDR="192.168.141.131"
     NETMASK="255.255.255.0"
     GATEWAY="192.168.141.1"
     IPV6INIT="yes"
     """,
     """
     network  --bootproto=static --device=ens5 --gateway=192.168.141.1 --ip=192.168.141.131 --netmask=255.255.255.0 --onboot=off --ipv6=auto
     """),
    # Ethernet generated by parse-kickstart bound to mac
    ("ifcfg-ens6",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     HWADDR="52:54:00:0c:77:e3"
     UUID="2a4bed9e-9492-4ea6-95d2-c27650184a93"
     ONBOOT="yes"
     BOOTPROTO="static"
     IPADDR="192.168.141.132"
     NETMASK="255.255.255.0"
     GATEWAY="192.168.141.1"
     IPV6INIT="yes"
     """,
     # FIXME generating --bindto
     #network  --bootproto=static --device=ens6 --gateway=192.168.141.1 --ip=192.168.141.132 --netmask=255.255.255.0 --ipv6=auto --bindto mac
     """
     network  --bootproto=static --device=ens6 --gateway=192.168.141.1 --ip=192.168.141.132 --netmask=255.255.255.0 --ipv6=auto
     """),
    ### Bond
    # Bond created in parse-kickstart
    ("ifcfg-bond0",
     """
     # Generated by parse-kickstart
     DEVICE="bond0"
     UUID="88596748-226e-4665-9260-f9fc1ca3e11d"
     ONBOOT="yes"
     BOOTPROTO="dhcp"
     IPV6INIT="yes"
     TYPE="Bond"
     BONDING_MASTER="yes"
     NAME="Bond connection bond0"
     BONDING_OPTS="mode=active-backup primary=ens8"
     """,
     """
     network  --bootproto=dhcp --device=bond0 --ipv6=auto --bondslaves=ens8,ens9 --bondopts=mode=active-backup,primary=ens8
     """),
    ("ifcfg-bond0_slave_1",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bond0 slave 1"
     UUID="544b8596-2eeb-4be5-908c-8ce416d7d4d3"
     ONBOOT="yes"
     MASTER="88596748-226e-4665-9260-f9fc1ca3e11d"
     DEVICE="ens8"
     """,
     None),
    ("ifcfg-bond0_slave_2",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bond0 slave 2"
     UUID="ca7df81b-2882-4827-954e-64b59f615164"
     ONBOOT="yes"
     MASTER="88596748-226e-4665-9260-f9fc1ca3e11d"
     DEVICE="ens9"
     """,
     None),
    ("ifcfg-bond1",
     """
     # Generated by parse-kickstart
     DEVICE="bond1"
     UUID="2c470b94-3973-4fc9-a36f-4933e71e93b1"
     ONBOOT="no"
     BOOTPROTO="dhcp"
     IPV6INIT="yes"
     TYPE="Bond"
     BONDING_MASTER="yes"
     NAME="Bond connection bond1"
     BONDING_OPTS="mode=active-backup primary=ens10 miimon=100 fail_over_mac=2"
     """,
     """
     network  --bootproto=dhcp --device=bond1 --onboot=off --ipv6=auto --bondslaves=ens10,ens11 --bondopts=mode=active-backup,primary=ens10,miimon=100,fail_over_mac=2
     """),
    ("ifcfg-bond1_slave_1",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bond1 slave 1"
     UUID="96522af3-50b0-4b13-8ca2-32d2cb63ace9"
     ONBOOT="no"
     MASTER="2c470b94-3973-4fc9-a36f-4933e71e93b1"
     DEVICE="ens10"
     """,
     None),
    ("ifcfg-bond1_slave_2",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bond1 slave 2"
     UUID="e92da80f-4ba0-4cca-b4e7-153908a2041b"
     ONBOOT="no"
     MASTER="2c470b94-3973-4fc9-a36f-4933e71e93b1"
     DEVICE="ens11"
     """,
     None),
    # Bond created in parse-kickstart with slaves bound to mac
    ("ifcfg-bond2",
     """
     # Generated by parse-kickstart
     DEVICE="bond2"
     UUID="e5c2704c-1b61-41ea-838e-fb4669857749"
     ONBOOT="yes"
     BOOTPROTO="dhcp"
     IPV6INIT="yes"
     TYPE="Bond"
     BONDING_MASTER="yes"
     NAME="Bond connection bond2"
     BONDING_OPTS="mode=active-backup primary=ens12"
     """,
     # FIXME generating --bindto
     """
     network  --bootproto=dhcp --device=bond2 --ipv6=auto --bondslaves=ens12,ens13 --bondopts=mode=active-backup,primary=ens12
     """),
    # ens12
    ("ifcfg-bond2_slave_1",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bond2 slave 1"
     UUID="6ff22c13-cbe9-4d89-91b2-879f7d11a950"
     ONBOOT="yes"
     MASTER="e5c2704c-1b61-41ea-838e-fb4669857749"
     HWADDR="52:54:00:23:74:76"
     """,
     None),
    # ens13
    ("ifcfg-bond2_slave_2",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bond2 slave 2"
     UUID="5c66ad69-e4d2-4a12-af44-304b45eed629"
     ONBOOT="yes"
     MASTER="e5c2704c-1b61-41ea-838e-fb4669857749"
     HWADDR="52:54:00:ac:d2:d3"
     """,
     None),
    # Bond created in anaconda
    ("ifcfg-bond3",
     """
     BONDING_OPTS="mode=active-backup primary=ens14"
     TYPE=Bond
     BONDING_MASTER=yes
     PROXY_METHOD=none
     BROWSER_ONLY=no
     BOOTPROTO=dhcp
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6INIT=yes
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     IPV6_ADDR_GEN_MODE=stable-privacy
     NAME=bond3
     UUID=9d51ce5e-88b9-45da-ba7b-a9435771458d
     DEVICE=bond3
     ONBOOT=no
     """,
     """
     network  --bootproto=dhcp --device=bond3 --onboot=off --ipv6=auto --bondslaves=ens14,ens15 --bondopts=mode=active-backup,primary=ens14
     """),
    ("ifcfg-bond3_slave_1",
     """
     TYPE=Ethernet
     NAME="bond3 slave 1"
     UUID=324a8072-cb2f-4ad4-93af-2a3a943b7ed8
     DEVICE=ens14
     ONBOOT=no
     MASTER=bond3
     SLAVE=yes
     """,
     None),
    ("ifcfg-bond3_slave_2",
     """
     TYPE=Ethernet
     NAME="bond3 slave 2"
     UUID=980e64a4-69e9-4032-881c-90bee33946ae
     DEVICE=ens15
     ONBOOT=no
     MASTER=bond3
     SLAVE=yes
     """,
     None),
    # Bond created in anaconda with slaves bound to mac
    ("ifcfg-bond4",
     """
     BONDING_OPTS="mode=active-backup primary=ens16"
     TYPE=Bond
     BONDING_MASTER=yes
     PROXY_METHOD=none
     BROWSER_ONLY=no
     BOOTPROTO=dhcp
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6INIT=yes
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     IPV6_ADDR_GEN_MODE=stable-privacy
     NAME=bond4
     UUID=aea559a2-fdb8-40da-915e-3bf42568c416
     DEVICE=bond4
     ONBOOT=no
     """,
     # FIXME generating --bindto
     """
     network  --bootproto=dhcp --device=bond4 --onboot=off --ipv6=auto --bondslaves=ens16,ens17 --bondopts=mode=active-backup,primary=ens16
     """),
    # ens16
    ("ifcfg-bond4_slave_1",
     """
     HWADDR=52:54:00:97:8E:14
     TYPE=Ethernet
     NAME="bond4 slave 1"
     UUID=30d2979e-1d4b-46bd-89bf-0f9a725a4b5d
     ONBOOT=no
     MASTER=bond4
     SLAVE=yes
     """,
     None),
    # ens17
    ("ifcfg-bond4_slave_2",
     """
     HWADDR=52:54:00:82:B9:BA
     TYPE=Ethernet
     NAME="bond4 slave 2"
     UUID=5ead6a44-604c-4229-8034-acccc64a6f01
     ONBOOT=no
     MASTER=bond4
     SLAVE=yes
     """,
     None),
    ### Team
    # Team created in parse-kickstart
    ("ifcfg-team0",
     """
     # Generated by parse-kickstart
     DEVICE=team0
     UUID=ffe9e996-fc2f-41d0-9e46-cc91ad5236c2
     ONBOOT=no
     BOOTPROTO=dhcp
     IPV6INIT=yes
     DEVICETYPE=Team
     NAME="Team connection team0"
     TEAM_CONFIG="{\"runner\": {\"name\": \"activebackup\"}}"
     PROXY_METHOD=none
     BROWSER_ONLY=no
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     """,
     """
     network  --bootproto=dhcp --device=team0 --onboot=off --ipv6=auto --teamslaves="ens18'{\\"prio\\": -10, \\"sticky\\": true}',ens19'{\\"prio\\": 100}'" --teamconfig="{\\"runner\\": {\\"name\\": \\"activebackup\\"}}"
     """),
    ("ifcfg-team0_slave_1",
     """
     # Generated by parse-kickstart
     DEVICETYPE="TeamPort"
     NAME="team0 slave 1"
     UUID="1d41bcf0-b190-4915-8a74-45b6848c03a4"
     ONBOOT="no"
     TEAM_MASTER="team0"
     TEAM_PORT_CONFIG="{\"prio\": -10, \"sticky\": true}"
     DEVICE="ens18"
     """,
     None),
    ("ifcfg-team0_slave_2",
     """
     # Generated by parse-kickstart
     DEVICETYPE="TeamPort"
     NAME="team0 slave 2"
     UUID="b45d9605-d206-4977-84e5-460cbd87e116"
     ONBOOT="no"
     TEAM_MASTER="team0"
     TEAM_PORT_CONFIG="{\"prio\": 100}"
     DEVICE="ens19"
     """,
     None),
    ("ifcfg-team1",
     """
     # Generated by parse-kickstart
     DEVICE="team1"
     UUID="d1b392aa-54a6-4007-85e9-0bbf35419c2d"
     ONBOOT="no"
     BOOTPROTO="dhcp"
     IPV6INIT="yes"
     DEVICETYPE="Team"
     NAME="Team connection team1"
     TEAM_CONFIG="{\"runner\": {\"name\": \"roundrobin\"}}"
     """,
     """
     network  --bootproto=dhcp --device=team1 --onboot=off --ipv6=auto --teamslaves="ens20,ens21" --teamconfig="{\\"runner\\": {\\"name\\": \\"roundrobin\\"}}"
     """),
    ("ifcfg-team1_slave_1",
     """
     # Generated by parse-kickstart
     DEVICETYPE="TeamPort"
     NAME="team1 slave 1"
     UUID="ca692742-bdb5-41c9-a267-bbbe5c7d17e3"
     ONBOOT="no"
     TEAM_MASTER="team1"
     DEVICE="ens20"
     """,
     None),
    ("ifcfg-team1_slave_2",
     """
     # Generated by parse-kickstart
     DEVICETYPE="TeamPort"
     NAME="team1 slave 2"
     UUID="75ebd3e5-64c2-4fe0-b1f4-a2e20d4b3801"
     ONBOOT="no"
     TEAM_MASTER="team1"
     DEVICE="ens21"
     """,
     None),
    # Team created in parse-kickstart with slaves bound to mac
    ("ifcfg-team2",
     """
     # Generated by parse-kickstart
     DEVICE="team2"
     UUID="9b2b1a7d-204e-464c-bfb5-94b755564d6e"
     ONBOOT="yes"
     BOOTPROTO="dhcp"
     IPV6INIT="yes"
     DEVICETYPE="Team"
     NAME="Team connection team2"
     TEAM_CONFIG="{\"runner\": {\"name\": \"activebackup\"}}"
     """,
     # FIXME generating --bindto
     """
     network  --bootproto=dhcp --device=team2 --ipv6=auto --teamslaves="ens23'{\\"prio\\": -10, \\"sticky\\": true}',ens24'{\\"prio\\": 100}'" --teamconfig="{\\"runner\\": {\\"name\\": \\"activebackup\\"}}"
     """),
    # ens23
    ("ifcfg-team2_slave_1",
     """
     # Generated by parse-kickstart
     DEVICETYPE="TeamPort"
     NAME="team2 slave 1"
     UUID="67e3b437-466d-4484-ad86-f580802b2849"
     ONBOOT="no"
     TEAM_MASTER="team2"
     TEAM_PORT_CONFIG="{\"prio\": -10, \"sticky\": true}"
     HWADDR="52:54:00:e1:1d:70"
     """,
     None),
    # ens24
    ("ifcfg-team2_slave_2",
     """
     # Generated by parse-kickstart
     DEVICETYPE="TeamPort"
     NAME="team2 slave 2"
     UUID="7dc1094f-f07b-43e1-b700-26cafffd6eb9"
     ONBOOT="no"
     TEAM_MASTER="team2"
     TEAM_PORT_CONFIG="{\"prio\": 100}"
     HWADDR="52:54:00:d4:23:f8"
     """,
     None),
    # Team created in anaconda
    ("ifcfg-team3",
     """
     TEAM_CONFIG="{\"runner\": {\"name\": \"activebackup\"}}"
     PROXY_METHOD=none
     BROWSER_ONLY=no
     BOOTPROTO=dhcp
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6INIT=yes
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     IPV6_ADDR_GEN_MODE=stable-privacy
     NAME=team3
     UUID=b3a3655c-86cf-4ce3-9968-b7df242ce705
     DEVICE=team3
     ONBOOT=no
     DEVICETYPE=Team
     """,
     """
     network  --bootproto=dhcp --device=team3 --onboot=off --ipv6=auto --teamslaves="ens25'{\\"prio\\": -10, \\"sticky\\": true}',ens26'{\\"prio\\": 100}'" --teamconfig="{\\"runner\\": {\\"name\\": \\"activebackup\\"}}"
     """),
    ("ifcfg-team3_slave_1",
     """
     TEAM_PORT_CONFIG="{\"prio\": -10, \"sticky\": true}"
     NAME="team3 slave 1"
     UUID=41f1df54-48b5-41b3-b88c-255d7a0b7339
     DEVICE=ens25
     ONBOOT=no
     TEAM_MASTER=team3
     DEVICETYPE=TeamPort
     """,
     None),
    ("ifcfg-team3_slave_2",
     """
     TEAM_PORT_CONFIG="{\"prio\": 100}"
     NAME="team3 slave 2"
     UUID=2e736df9-ebfc-4edc-8cf9-a13ef65274ee
     DEVICE=ens26
     ONBOOT=no
     TEAM_MASTER=team3
     DEVICETYPE=TeamPort
     """,
     None),
    # Team created in anaconda with slaves bound to mac
    ("ifcfg-team4",
     """
     TEAM_CONFIG="{\"runner\": {\"name\": \"activebackup\"}}"
     PROXY_METHOD=none
     BROWSER_ONLY=no
     BOOTPROTO=dhcp
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6INIT=yes
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     IPV6_ADDR_GEN_MODE=stable-privacy
     NAME=team4
     UUID=3949fb93-bddb-4159-a2ef-2b84819e0d41
     DEVICE=team4
     ONBOOT=no
     DEVICETYPE=Team
     """,
     # FIXME generating --bindto
     """
     network  --bootproto=dhcp --device=team4 --onboot=off --ipv6=auto --teamslaves="ens27'{\\"prio\\": -10, \\"sticky\\": true}',ens28'{\\"prio\\": 100}'" --teamconfig="{\\"runner\\": {\\"name\\": \\"activebackup\\"}}"
     """),
    # ens27
    ("ifcfg-team4_slave_1",
     """
     HWADDR=52:54:00:5E:26:B5
     TEAM_PORT_CONFIG="{\"prio\": -10, \"sticky\": true}"
     NAME="team4 slave 1"
     UUID=dbefddc1-f463-4a4c-bf4b-d951283ec490
     ONBOOT=no
     TEAM_MASTER=team4
     DEVICETYPE=TeamPort
     """,
     None),
    # ens28
    ("ifcfg-team4_slave_2",
     """
     HWADDR=52:54:00:35:BF:0F
     TEAM_PORT_CONFIG="{\"prio\": 100}"
     NAME="team4 slave 2"
     UUID=a074b1f8-65ff-4a97-80b1-d40ef461baab
     ONBOOT=no
     TEAM_MASTER=team4
     DEVICETYPE=TeamPort
     """,
     None),
    ### Bridge
    # Bridge created in parse-kickstart
    ("ifcfg-bridge0",
     """
     # Generated by parse-kickstart
     DEVICE="bridge0"
     UUID="8d1a0dcc-5c21-4820-9d18-a54c405e2f62"
     ONBOOT="yes"
     BOOTPROTO="dhcp"
     IPV6INIT="yes"
     TYPE="Bridge"
     NAME="Bridge connection bridge0"
     """,
     """
     network  --bootproto=dhcp --device=bridge0 --ipv6=auto --bridgeslaves=ens30
     """),
    ("ifcfg-bridge0_slave_1",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bridge0 slave 1"
     UUID="96960118-f862-467c-92f8-6b5e34f53d1b"
     ONBOOT="yes"
     BRIDGE="bridge0"
     DEVICE="ens30"
     """,
     None),
    ("ifcfg-bridge1",
     """
     # Generated by parse-kickstart
     DEVICE="bridge1"
     UUID="6bbd88e3-5136-475c-b660-c30eb040f8d6"
     ONBOOT="no"
     BOOTPROTO="dhcp"
     IPV6INIT="yes"
     TYPE="Bridge"
     NAME="Bridge connection bridge1"
     """,
     """
     network  --bootproto=dhcp --device=bridge1 --onboot=off --ipv6=auto --bridgeslaves=ens31
     """),
    ("ifcfg-bridge1_slave_1",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bridge1 slave 1"
     UUID="72c35405-80e8-43a8-9cbf-afd520bdba51"
     ONBOOT="no"
     BRIDGE="bridge1"
     DEVICE="ens31"
     """,
     None),
    # Bridge created in parse-kickstart with slaves bound to mac
    ("ifcfg-bridge2",
     """
     # Generated by parse-kickstart
     DEVICE="bridge2"
     UUID="927685cf-8203-4d2e-82ae-c45f44117b6f"
     ONBOOT="yes"
     BOOTPROTO="dhcp"
     IPV6INIT="yes"
     TYPE="Bridge"
     NAME="Bridge connection bridge2"
     """,
     # FIXME generating --bindto
     """
     network  --bootproto=dhcp --device=bridge2 --ipv6=auto --bridgeslaves=ens32
     """),
    # ens32
    ("ifcfg-bridge2_slave_1",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bridge2 slave 1"
     UUID="9ce980bb-ff60-4e61-9a43-e72d5ba297c3"
     ONBOOT="yes"
     BRIDGE="bridge2"
     HWADDR="52:54:00:70:10:7f"
     """,
     None),
    # Bridge created in anaconda
    ("ifcfg-bridge3",
     """
     STP=yes
     BRIDGING_OPTS=priority=32768
     TYPE=Bridge
     PROXY_METHOD=none
     BROWSER_ONLY=no
     BOOTPROTO=dhcp
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6INIT=yes
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     IPV6_ADDR_GEN_MODE=stable-privacy
     NAME=bridge3
     UUID=ac34f70b-b41b-49ac-a940-7c3bdb4b878f
     DEVICE=bridge3
     ONBOOT=no
     """,
     # NOTE bridgeopts are not in kickstart
     #network  --bootproto=dhcp --device=bridge3 --onboot=no --bridgeslaves=ens33
     """
     network  --bootproto=dhcp --device=bridge3 --onboot=off --ipv6=auto --bridgeslaves=ens33 --bridgeopts=priority=32768,stp=yes
     """),
    ("ifcfg-bridge3_slave_1",
     """
     TYPE=Ethernet
     NAME="bridge3 slave 1"
     UUID=4cffe475-81ce-4989-861d-9cd9e08b575f
     DEVICE=ens33
     ONBOOT=no
     BRIDGE=bridge3
     """,
     None),
    # Bridge created in anaconda with slaves bound to mac
    ("ifcfg-bridge4",
     """
     STP=yes
     BRIDGING_OPTS=priority=32768
     TYPE=Bridge
     PROXY_METHOD=none
     BROWSER_ONLY=no
     BOOTPROTO=dhcp
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6INIT=yes
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     IPV6_ADDR_GEN_MODE=stable-privacy
     NAME=bridge4
     UUID=37792ba4-bd6c-45f9-84dc-3ec7ef995783
     DEVICE=bridge4
     ONBOOT=no
     """,
     # NOTE bridgeopts are not in kickstart
     #network  --bootproto=dhcp --device=bridge4 --onboot=no --bridgeslaves=ens34
     # FIXME generating --bindto
     """
     network  --bootproto=dhcp --device=bridge4 --onboot=off --ipv6=auto --bridgeslaves=ens34 --bridgeopts=priority=32768,stp=yes
     """),
    # ens34
    ("ifcfg-bridge4_slave_1",
     """
     HWADDR=52:54:00:E5:A0:29
     TYPE=Ethernet
     NAME="bridge4 slave 1"
     UUID=a8b68194-b45c-4ee8-9907-3361be553ac2
     ONBOOT=no
     BRIDGE=bridge4
     """,
     None),
    ### Vlan
    # Vlan created in nm-c-e, bound to device
    ("ifcfg-VLAN_connection_1",
     """
     VLAN=yes
     TYPE=Vlan
     PHYSDEV=ens35
     VLAN_ID=233
     REORDER_HDR=yes
     GVRP=no
     MVRP=no
     PROXY_METHOD=none
     BROWSER_ONLY=no
     BOOTPROTO=dhcp
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6INIT=yes
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     IPV6_PRIVACY=no
     IPV6_ADDR_GEN_MODE=stable-privacy
     NAME="VLAN connection 1"
     UUID=57a0f1c1-a8a3-4865-8b93-f96493bf5583
     ONBOOT=yes
     """,
     """
     network  --bootproto=dhcp --device=ens35 --ipv6=auto --vlanid=233
     """),
    # Vlan created in nm-c-e, bound to connection (uuid)
    ("ifcfg-ens36",
     """
     # Generated by dracut initrd
     NAME="ens36"
     DEVICE="ens36"
     ONBOOT=yes
     NETBOOT=yes
     UUID="98282424-e386-446a-b9b1-dc08460b68e2"
     IPV6INIT=yes
     BOOTPROTO=dhcp
     TYPE=Ethernet
     """,
     """
     network  --bootproto=dhcp --device=ens36 --ipv6=auto
     """),
    ("ifcfg-VLAN_connection_2",
     """
     VLAN=yes
     TYPE=Vlan
     PHYSDEV=98282424-e386-446a-b9b1-dc08460b68e2
     VLAN_ID=233
     REORDER_HDR=yes
     GVRP=no
     MVRP=no
     PROXY_METHOD=none
     BROWSER_ONLY=no
     BOOTPROTO=dhcp
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6INIT=yes
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     IPV6_PRIVACY=no
     IPV6_ADDR_GEN_MODE=stable-privacy
     NAME="VLAN connection 2"
     UUID=1e077522-5f5c-4504-9971-b1425452dbd0
     ONBOOT=yes
     """,
     """
     network  --bootproto=dhcp --device=ens36 --ipv6=auto --vlanid=233
     """),
    # Vlan created in nm-c-e, bound to device, named explicitly
    ("ifcfg-VLAN_connection_3",
     """
     VLAN=yes
     TYPE=Vlan
     PHYSDEV=ens37
     VLAN_ID=233
     REORDER_HDR=yes
     GVRP=no
     MVRP=no
     PROXY_METHOD=none
     BROWSER_ONLY=no
     BOOTPROTO=dhcp
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6INIT=yes
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     IPV6_PRIVACY=no
     IPV6_ADDR_GEN_MODE=stable-privacy
     NAME="VLAN connection 3"
     UUID=2cab7cbe-9df3-4313-9fe0-8a3899d8707e
     ONBOOT=yes
     DEVICE=vlan233
     """,
     """
     network  --bootproto=dhcp --device=ens37 --ipv6=auto --vlanid=233 --interfacename=vlan233
     """),
    # Vlan created in nm-c-e, bound to connection, named explicitly
    ("ifcfg-ens38",
     """
     # Generated by dracut initrd
     NAME="ens38"
     DEVICE="ens38"
     ONBOOT=yes
     NETBOOT=yes
     UUID="abf92663-cfad-4cf0-acd3-9ca6f4d45962"
     IPV6INIT=yes
     BOOTPROTO=dhcp
     TYPE=Ethernet
     """,
     """
     network  --bootproto=dhcp --device=ens38 --ipv6=auto
     """),
    ("ifcfg-VLAN_connection_4",
     """
     VLAN=yes
     TYPE=Vlan
     PHYSDEV=abf92663-cfad-4cf0-acd3-9ca6f4d45962
     VLAN_ID=234
     REORDER_HDR=yes
     GVRP=no
     MVRP=no
     PROXY_METHOD=none
     BROWSER_ONLY=no
     BOOTPROTO=dhcp
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6INIT=yes
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     IPV6_PRIVACY=no
     IPV6_ADDR_GEN_MODE=stable-privacy
     NAME="VLAN connection 4"
     UUID=0dbaf4df-74ac-46c4-8f81-3c303763e972
     ONBOOT=yes
     DEVICE=vlan234
     """,
     """
     network  --bootproto=dhcp --device=ens38 --ipv6=auto --vlanid=234 --interfacename=vlan234
     """),
    # Wireless
    ("ifcfg-Speckalm",
     """
     HWADDR=10:4A:7D:BE:68:74
     ESSID=Speckalm
     MODE=Managed
     KEY_MGMT=WPA-PSK
     SECURITYMODE=open
     MAC_ADDRESS_RANDOMIZATION=default
     TYPE=Wireless
     PROXY_METHOD=none
     BROWSER_ONLY=no
     BOOTPROTO=dhcp
     DEFROUTE=yes
     IPV4_FAILURE_FATAL=no
     IPV6INIT=yes
     IPV6_AUTOCONF=yes
     IPV6_DEFROUTE=yes
     IPV6_FAILURE_FATAL=no
     IPV6_ADDR_GEN_MODE=stable-privacy
     NAME=Speckalm
     UUID=b9b8a976-ce70-420d-b01a-0e3a3b64b1cb
     ONBOOT=yes
     """,
     None),
    # Vlan over bond created in parse-kickstart
    # FIXME - created by single network command
    # network --device bond0 --bootproto static --ip 192.168.141.130 --netmask 255.255.255.0 --gateway 192.168.141.1 --bondslaves=ens4,ens5 --bondopts=mode=active-backup,miimon-100,primary=ens4 --vlanid=222 --activate --onboot=no
    # Whould the two commands work as well?
    ("ifcfg-bond5",
     """
     # Generated by parse-kickstart
     TYPE="Bond"
     NAME="Bond connection bond5"
     UUID="e90295f3-bd63-4a21-b92d-78dd2923fe2d"
     ONBOOT="yes"
     BONDING_MASTER="yes"
     BONDING_OPTS="mode=active-backup miimon-100 primary=ens39"
     DEVICE="bond5"
     """,
     # FIXME
     #network  --device=bond5 --noipv4 --noipv6 --bondslaves=ens39,ens40 --bondopts=mode=active-backup,miimon-100,primary=ens39
     """
     network  --bootproto=dhcp --device=bond5 --noipv4 --noipv6 --bondslaves=ens39,ens40 --bondopts=mode=active-backup,miimon-100,primary=ens39
     """),
    ("ifcfg-bond5_slave_1",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bond5 slave 1"
     UUID="5e04989f-f97f-471e-8416-59bbea746e5e"
     ONBOOT="yes"
     MASTER="e90295f3-bd63-4a21-b92d-78dd2923fe2d"
     DEVICE="ens39"
     """,
     None),
    ("ifcfg-bond5_slave_2",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bond5 slave 2"
     UUID="e1d28630-2f6f-4e94-98a9-83facc8d4644"
     ONBOOT="yes"
     MASTER="e90295f3-bd63-4a21-b92d-78dd2923fe2d"
     DEVICE="ens40"
     """,
     None),
    ("ifcfg-bond5.222",
     """
     # Generated by parse-kickstart
     DEVICE="bond5.222"
     UUID="d6ad3440-d2e8-4797-96ba-185a30f74e6f"
     ONBOOT="yes"
     BOOTPROTO="static"
     IPADDR="192.168.141.130"
     NETMASK="255.255.255.0"
     GATEWAY="192.168.141.1"
     IPV6INIT="yes"
     TYPE="Vlan"
     NAME="VLAN connection bond5.222"
     VLAN="yes"
     VLAN_ID="222"
     PHYSDEV="bond5"
     """,
     """
     network  --bootproto=static --device=bond5 --gateway=192.168.141.1 --ip=192.168.141.130 --netmask=255.255.255.0 --ipv6=auto --vlanid=222
     """),
    # Bridged bond created in parse-kickstart
    ("ifcfg-bridge5",
     """
     # Generated by parse-kickstart
     DEVICE="bridge5"
     UUID="2f879653-20a3-421d-81ac-7c0c8f192850"
     ONBOOT="yes"
     BOOTPROTO="dhcp"
     IPV6INIT="yes"
     TYPE="Bridge"
     NAME="Bridge connection bridge5"
     """,
     """
     network  --bootproto=dhcp --device=bridge5 --ipv6=auto --bridgeslaves=bond6
     """),
    ("ifcfg-bond6",
     """
     # Generated by parse-kickstart
     DEVICE="bond6"
     UUID="d94466b9-0fcf-43e8-b1d6-51130b6d1c3b"
     ONBOOT="yes"
     IPV6INIT="yes"
     TYPE="Bond"
     BONDING_MASTER="yes"
     NAME="Bond connection bond6"
     BONDING_OPTS="mode=balance-tlb"
     BRIDGE=bridge5
     """,
     # FIXME
     #network  --device=bond6 --noipv4 --ipv6=auto --bondslaves=ens41,ens42 --bondopts=mode=balance-tlb
     """
     network  --bootproto=dhcp --device=bond6 --noipv4 --ipv6=auto --bondslaves=ens41,ens42 --bondopts=mode=balance-tlb
     """),
    ("ifcfg-bond6_slave_1",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bond6 slave 1"
     UUID="ec6a4359-8aa2-432b-89a9-ef7707480810"
     ONBOOT="yes"
     MASTER="d94466b9-0fcf-43e8-b1d6-51130b6d1c3b"
     DEVICE="ens41"
     """,
     None),
    ("ifcfg-bond6_slave_2",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     NAME="bond6 slave 2"
     UUID="262214c6-479f-4f2a-8e2f-4a97016834ab"
     ONBOOT="yes"
     MASTER="d94466b9-0fcf-43e8-b1d6-51130b6d1c3b"
     DEVICE="ens42"
     """,
     None),
    # Two ifcfgs for a device
    ("ifcfg-ens50",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     DEVICE="ens50"
     UUID="14f3e6d3-5d1a-4b49-9806-f3ef968123c4"
     ONBOOT="no"
     BOOTPROTO="static"
     IPADDR="192.168.141.131"
     NETMASK="255.255.255.0"
     GATEWAY="192.168.141.1"
     IPV6INIT="yes"
     """,
     """
     network  --bootproto=static --device=ens50 --gateway=192.168.141.1 --ip=192.168.141.131 --netmask=255.255.255.0 --onboot=off --ipv6=auto
     """),
    ("ifcfg-ens50-wannabe",
     """
     # Generated by parse-kickstart
     TYPE="Ethernet"
     DEVICE="ens50"
     UUID="8c3d117a-003e-4401-8edf-51eb5a8f4e13"
     ONBOOT="no"
     BOOTPROTO="dhcp"
     IPV6INIT="yes"
     """,
     """
     network  --bootproto=dhcp --device=ens50 --onboot=off --ipv6=auto
     """),
]


class IfcfgFileTestCase(unittest.TestCase):

    def setUp(self):
        self._root_dir = tempfile.mkdtemp(prefix="ifcfg-test-dir")
        self._ifcfg_dir = os.path.join(self._root_dir, IFCFG_DIR.lstrip("/"))
        os.makedirs(self._ifcfg_dir)

    def tearDown(self):
        shutil.rmtree(self._root_dir)

    def _dump_ifcfg_files(self, files_list):
        for file_name, content, _generated_ks in files_list:
            content = dedent(content).strip()
            with open(os.path.join(self._ifcfg_dir, file_name), "w") as f:
                f.write(content)

    def _check_ifcfg_files(self, files_list):
        for file_name, expected_content, _generated_ks in files_list:
            expected_content = dedent(expected_content).strip()
            with open(os.path.join(self._ifcfg_dir, file_name), "r") as f:
                content = f.read().strip()
            print("=============== EXPECTED CONTENT:")
            print(expected_content)
            print("================= ACTUAL CONTENT:")
            print(content)
            print("=================================")
            self.assertEqual(content, expected_content)

    def _get_ifcfg_file_path(self, file_name):
        return os.path.join(self._ifcfg_dir, file_name)

    def ifcfg_file_path_test(self):
        """Test IfcfgFile.path property."""
        file_name = "ifcfg-ens3"
        file_path = self._get_ifcfg_file_path(file_name)
        ifcfg = IfcfgFile(file_path)
        self.assertEqual(
            ifcfg.path,
            file_path
        )

    def ifcfg_file_uuid_test(self):
        """Test IfcfcgFile.uuid property."""
        file_name = "ifcfg-ens3"
        no_uuid_file_name = "ifcfg-ens4"
        uuid = "1d74da2f-7cdb-4df6-9bb6-16d9ccfa5146"
        input_ifcfg_files = [
            (file_name,
             """
             DEVICE=ens3
             UUID={}
             """.format(uuid),
             None),
            (no_uuid_file_name,
             """
             DEVICE=ens4
             """,
             None)
        ]
        self._dump_ifcfg_files(input_ifcfg_files)

        file_path = self._get_ifcfg_file_path(file_name)
        ifcfg = IfcfgFile(file_path)
        self.assertEqual(ifcfg.uuid, uuid)

        file_path = self._get_ifcfg_file_path(no_uuid_file_name)
        ifcfg = IfcfgFile(file_path)
        self.assertEqual(ifcfg.uuid, "")

    def ifcfg_file_read_write_test(self):
        """Test IfcfgFile read and write methods."""
        file_name = "ifcfg-ens3"

        # Non-existing ifcfg file can be instantiated but not read
        file_path = self._get_ifcfg_file_path(file_name)
        ifcfg = IfcfgFile(file_path)
        with self.assertRaises(FileNotFoundError):
            ifcfg.read()
        # The file will not be written if no value was changed
        ifcfg.write()
        with self.assertRaises(FileNotFoundError):
            ifcfg.read()
        # The file will be written if filename is supplied
        ifcfg.write(filename=file_path)
        ifcfg.read()

        # Read and write existing file
        # Order, comments, empty lines, and duplicates are preserved
        input_ifcfg_files = [
            (file_name,
             """
             BOOTPROTO="dhcp"
             # comment
             DEVICE="ens3"

             ONBOOT="yes"
             ONBOOT="yes"
             """,
             None)
        ]
        self._dump_ifcfg_files(input_ifcfg_files)

        file_path = self._get_ifcfg_file_path(file_name)
        ifcfg = IfcfgFile(file_path)
        ifcfg.read()
        ifcfg.write(filename=file_path)
        self._check_ifcfg_files(input_ifcfg_files)

    def ifcfg_file_is_from_kickstart_test(self):
        """Test IfcfgFile.is_from_kickstart."""
        file_name = "ifcfg-ens3"
        from_kickstart_file_name = "ifcfg-ens4"
        input_ifcfg_files = [
            (file_name,
             """
             DEVICE=\"ens3\"
             BOOTPROTO=dhcp
             """,
             None),
            (from_kickstart_file_name,
             """
             # Generated by parse-kickstart
             DEVICE=\"ens3\"
             BOOTPROTO=dhcp
             """,
             None)
        ]

        self._dump_ifcfg_files(input_ifcfg_files)

        file_path = self._get_ifcfg_file_path(file_name)
        ifcfg = IfcfgFile(file_path)
        self.assertFalse(ifcfg.is_from_kickstart)

        file_path = self._get_ifcfg_file_path(from_kickstart_file_name)
        ifcfg = IfcfgFile(file_path)
        self.assertTrue(ifcfg.is_from_kickstart)

    def ifcfg_file_get_test(self):
        """Test IfcfgFile.get method."""
        file_name = "ifcfg-ens3"
        input_ifcfg_files = [
            (file_name,
             """
             DEVICE="ens3"
             BOOTPROTO=dhcp
             """,
             None)
        ]
        self._dump_ifcfg_files(input_ifcfg_files)

        file_path = self._get_ifcfg_file_path(file_name)
        ifcfg = IfcfgFile(file_path)

        # Before read() the file is empty
        self.assertEqual(
            ifcfg.get("DEVICE"),
            ""
        )

        ifcfg.read()
        # Quoted value works
        self.assertEqual(
            ifcfg.get("DEVICE"),
            "ens3"
        )
        # Non-quoted value works
        self.assertEqual(
            ifcfg.get("BOOTPROTO"),
            "dhcp"
        )
        # Non-existing value returns empty string
        self.assertEqual(
            ifcfg.get("NONEXISTING"),
            ""
        )

    def ifcfg_file_set_unset_test(self):
        """Test IfcfgFile set and uset methods."""
        file_name = "ifcfg-ens3"
        # Non-existing ifcfg file can be instantiated but not read
        file_path = self._get_ifcfg_file_path(file_name)
        ifcfg = IfcfgFile(file_path)
        self.assertEqual(ifcfg.get("ONBOOT"), "")
        ifcfg.set(("ONBOOT", "yes"))
        self.assertEqual(ifcfg.get("ONBOOT"), "yes")
        ifcfg.set(("ONBOOT", "no"))
        self.assertEqual(ifcfg.get("ONBOOT"), "no")
        ifcfg.unset("ONBOOT")
        self.assertEqual(ifcfg.get("ONBOOT"), "")
        ifcfg.unset("NONEXISTING")
        ifcfg.set(("BOOTPROTO", "dhcp"), ("DEVICE", "ens3"))
        self.assertEqual(ifcfg.get("BOOTPROTO"), "dhcp")
        self.assertEqual(ifcfg.get("DEVICE"), "ens3")
        ifcfg.write()
        output_ifcfg_files = [
            (file_name,
             """
             BOOTPROTO="dhcp"
             DEVICE="ens3"
             """,
             None)
        ]
        self._check_ifcfg_files(output_ifcfg_files)

    def get_ifcfg_files_paths_test(self):
        """Test get_ifcfg_files_paths."""
        all_ifcfg_files = [
            ("ifcfg-ens3",
             """
             DEVICE="ens3"
             """,
             None),
            ("ifcfg-ens5",
             """
             DEVICE="ens5"
             """,
             None),
            ("ifcfg-lo",
             """
             DEVICE="lo"
             """,
             None),
            ("nonifcfg",
             """
             Lebenswelt
             """,
             None),
        ]
        self._dump_ifcfg_files(all_ifcfg_files)
        ifcfg_paths = get_ifcfg_files_paths(self._ifcfg_dir)
        self.assertEqual(len(ifcfg_paths), 2)
        ifcfg_files = [os.path.basename(path) for path in ifcfg_paths]
        self.assertIn("ifcfg-ens3", ifcfg_files)
        self.assertIn("ifcfg-ens5", ifcfg_files)

    def get_ifcfg_file_test(self):
        """Test get_ifcfg_file."""
        ifcfg_files = [
            ("ifcfg-ens3",
             """
             DEVICE="ens3"
             """,
             None),
            ("ifcfg-ens5",
             """
             DEVICE="ens5"
             """,
             None),
            ("ifcfg-Wiredconnection_1",
             """
             DEVICE="ens5"
             """,
             None),
            ("non-ifcfg-prefix",
             """
             DEVICE="ens5"
             ONBOOT="yes"
             """,
             None)
        ]
        self._dump_ifcfg_files(ifcfg_files)
        ifcfg = get_ifcfg_file([("DEVICE", "ens3")], root_path=self._root_dir)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-ens3")
        ifcfg = get_ifcfg_file([("ONBOOT", "yes")], root_path=self._root_dir)
        self.assertIsNone(ifcfg)
        ifcfg = get_ifcfg_file([("DEVICE", "ens7")], root_path=self._root_dir)
        self.assertIsNone(ifcfg)
        ifcfg = get_ifcfg_file([("DEVICE", "ens5")], root_path=self._root_dir)
        self.assertIn(
            os.path.basename(ifcfg.path),
            ["ifcfg-ens5", "ifcfg-Wiredconnection_1"]
        )

    @patch("pyanaconda.modules.network.ifcfg.get_iface_from_connection",
           lambda client, uuid: UUID_TO_IFACE[uuid])
    @patch("pyanaconda.modules.network.ifcfg.get_iface_from_hwaddr",
           lambda client, hwaddr: HWADDR_TO_IFACE[hwaddr])
    def get_ifcfg_file_of_device_test(self):
        """Test get_ifcfg_file_of_device."""
        self._dump_ifcfg_files(IFCFG_FILES_REAL_ALL_TYPES)

        nm_client = Mock()
        root_path = self._root_dir

        ### Ethernet

        # Ethernet generated by dracut initrd
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens3", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-ens3")
        # Ethernet generated by parse-kickstart
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens5", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-ens5")
        # Ethernet generated by parse-kickstart bound to mac
        # with device_hwaddr hint
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens6", root_path=root_path,
                                         device_hwaddr="52:54:00:0c:77:e3")
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-ens6")
        # Ethernet generated by parse-kickstart bound to mac
        # without device_hwaddr hint
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens6", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-ens6")

        ### Bond

        # Bond created in parse-kickstart
        # Bond devices are found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bond0", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bond0")
        ifcfg = get_ifcfg_file_of_device(nm_client, "bond1", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bond1")
        # Bond slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens8", root_path=root_path)
        self.assertIsNone(ifcfg)

        # Bond created in parse-kickstart with slaves bound to mac
        # Bond device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bond2", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bond2")
        # Bond slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens12", root_path=root_path)
        self.assertIsNone(ifcfg)

        # Bond created in anaconda
        # Bond device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bond3", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bond3")
        # Bond slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens14", root_path=root_path)
        self.assertIsNone(ifcfg)

        # Bond created in anaconda with slaves bound to mac
        # Bond device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bond4", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bond4")
        # Bond slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens16", root_path=root_path)
        self.assertIsNone(ifcfg)

        ### Team

        # Team created in parse-kickstart
        # Team devices are found
        ifcfg = get_ifcfg_file_of_device(nm_client, "team0", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-team0")
        ifcfg = get_ifcfg_file_of_device(nm_client, "team1", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-team1")
        # Team slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens18", root_path=root_path)
        self.assertIsNone(ifcfg)

        # Team created in parse-kickstart with slaves bound to mac
        # Team device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "team2", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-team2")
        # Team slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens23", root_path=root_path)
        self.assertIsNone(ifcfg)

        # Team created in anaconda
        # Team device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "team3", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-team3")
        # Team slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens25", root_path=root_path)
        self.assertIsNone(ifcfg)

        # Team created in anaconda with slaves bound to mac
        # Team device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "team4", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-team4")
        # Team slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens27", root_path=root_path)
        self.assertIsNone(ifcfg)

        ### Bridge

        # Bridge created in parse-kickstart
        # Bridge devices are found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bridge0", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bridge0")
        ifcfg = get_ifcfg_file_of_device(nm_client, "bridge1", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bridge1")
        # Bridge slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens30", root_path=root_path)
        self.assertIsNone(ifcfg)

        # Bridge created in parse-kickstart with slaves bound to mac
        # Bridge device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bridge2", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bridge2")
        # Bridge slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens32", root_path=root_path)
        self.assertIsNone(ifcfg)

        # Bridge created in anaconda
        # Bridge device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bridge3", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bridge3")
        # Bridge slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens33", root_path=root_path)
        self.assertIsNone(ifcfg)

        # Bridge created in anaconda with slaves bound to mac
        # Bridge device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bridge4", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bridge4")
        # Bridge slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens34", root_path=root_path)
        self.assertIsNone(ifcfg)

        ### Vlan
        # TODO: cases for ifcfgs generated by dracut or kickstart

        # Vlan created in nm-c-e, bound to device
        # Vlan device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens35.233", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-VLAN_connection_1")

        # Vlan created in nm-c-e, bound to connection (uuid)
        # Vlan device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens36.233", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-VLAN_connection_2")
        # Parent device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens36", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-ens36")

        # Vlan created in nm-c-e, bound to device, named explicitly
        # Vlan device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "vlan233", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-VLAN_connection_3")

        # Vlan created in nm-c-e, bound to connection, named explicitly
        # Vlan device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "vlan234", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-VLAN_connection_4")
        # Parent device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens38", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-ens38")

        ### Vlan over bond

        # Created in parse-kickstart
        # Vlan device if found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bond5.222", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bond5.222")
        # Bond device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bond5", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bond5")
        # Bond slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens39", root_path=root_path)
        self.assertIsNone(ifcfg)

        ### Bridged bond

        # Created in parse-kickstart
        # Bridge device if found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bridge5", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bridge5")
        # Bond device is found
        ifcfg = get_ifcfg_file_of_device(nm_client, "bond6", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-bond6")
        # Bond slaves are ignored
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens41", root_path=root_path)
        self.assertIsNone(ifcfg)


        # In case of more files for a device one is picked (unpredictably)
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens50", root_path=root_path)
        self.assertIn(
            os.path.basename(ifcfg.path),
            ["ifcfg-ens50", "ifcfg-ens50-wannabe"]
        )

    @patch("pyanaconda.modules.network.ifcfg.is_s390")
    def get_ifcfg_file_of_device_s390_test(self, is_s390):
        """Test get_ifcfg_file_of_device for s390."""

        nm_client = Mock()
        root_path = self._root_dir

        ifcfg_files = [
            ("ifcfg-ens3",
             """
             TYPE="Ethernet"
             NAME="ens3"
             """,
             None),
        ]
        self._dump_ifcfg_files(ifcfg_files)

        is_s390.return_value = False
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens3", root_path=root_path)
        self.assertIsNone(ifcfg)

        is_s390.return_value = True
        ifcfg = get_ifcfg_file_of_device(nm_client, "ens3", root_path=root_path)
        self.assertEqual(os.path.basename(ifcfg.path), "ifcfg-ens3")

    @patch("pyanaconda.modules.network.ifcfg.get_iface_from_hwaddr",
           lambda client, hwaddr: HWADDR_TO_IFACE[hwaddr])
    def get_slaves_from_ifcfgs_test(self):
        """Test get_slaves_from_ifcfgs."""
        self._dump_ifcfg_files(IFCFG_FILES_REAL_ALL_TYPES)

        nm_client = Mock()
        root_path = self._root_dir

        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "MASTER", ["ens5"],
                                   root_path=root_path),
            set()
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "MASTER", ["bond0", "88596748-226e-4665-9260-f9fc1ca3e11d"],
                                   root_path=root_path),
            {
                ("ens8", "544b8596-2eeb-4be5-908c-8ce416d7d4d3"),
                ("ens9", "ca7df81b-2882-4827-954e-64b59f615164")
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "MASTER", ["bond2", "e5c2704c-1b61-41ea-838e-fb4669857749"],
                                   root_path=root_path),
            {
                ("ens12", "6ff22c13-cbe9-4d89-91b2-879f7d11a950"),
                ("ens13", "5c66ad69-e4d2-4a12-af44-304b45eed629")
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "MASTER", ["bond3", "9d51ce5e-88b9-45da-ba7b-a9435771458d"],
                                   root_path=root_path),
            {
                ("ens14", "324a8072-cb2f-4ad4-93af-2a3a943b7ed8"),
                ("ens15", "980e64a4-69e9-4032-881c-90bee33946ae")
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "MASTER", ["bond4", "aea559a2-fdb8-40da-915e-3bf42568c416"],
                                   root_path=root_path),
            {
                ("ens16", "30d2979e-1d4b-46bd-89bf-0f9a725a4b5d"),
                ("ens17", "5ead6a44-604c-4229-8034-acccc64a6f01")
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "TEAM_MASTER", ["team0", "ffe9e996-fc2f-41d0-9e46-cc91ad5236c2"],
                                   root_path=root_path),
            {
                ("ens18", "1d41bcf0-b190-4915-8a74-45b6848c03a4"),
                ("ens19", "b45d9605-d206-4977-84e5-460cbd87e116")
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "TEAM_MASTER", ["team1", "d1b392aa-54a6-4007-85e9-0bbf35419c2d"],
                                   root_path=root_path),
            {
                ("ens20", "ca692742-bdb5-41c9-a267-bbbe5c7d17e3"),
                ("ens21", "75ebd3e5-64c2-4fe0-b1f4-a2e20d4b3801")
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "TEAM_MASTER", ["team2", "9b2b1a7d-204e-464c-bfb5-94b755564d6e"],
                                   root_path=root_path),
            {
                ("ens23", "67e3b437-466d-4484-ad86-f580802b2849"),
                ("ens24", "7dc1094f-f07b-43e1-b700-26cafffd6eb9")
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "TEAM_MASTER", ["team3", "b3a3655c-86cf-4ce3-9968-b7df242ce705"],
                                   root_path=root_path),
            {
                ("ens25", "41f1df54-48b5-41b3-b88c-255d7a0b7339"),
                ("ens26", "2e736df9-ebfc-4edc-8cf9-a13ef65274ee")
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "TEAM_MASTER", ["team4", "3949fb93-bddb-4159-a2ef-2b84819e0d41"],
                                   root_path=root_path),
            {
                ("ens27", "dbefddc1-f463-4a4c-bf4b-d951283ec490"),
                ("ens28", "a074b1f8-65ff-4a97-80b1-d40ef461baab")
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "BRIDGE", ["bridge0", "8d1a0dcc-5c21-4820-9d18-a54c405e2f62"],
                                   root_path=root_path),
            {
                ("ens30", "96960118-f862-467c-92f8-6b5e34f53d1b"),
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "BRIDGE", ["bridge1", "6bbd88e3-5136-475c-b660-c30eb040f8d6"],
                                   root_path=root_path),
            {
                ("ens31", "72c35405-80e8-43a8-9cbf-afd520bdba51"),
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "BRIDGE", ["bridge2", "927685cf-8203-4d2e-82ae-c45f44117b6f"],
                                   root_path=root_path),
            {
                ("ens32", "9ce980bb-ff60-4e61-9a43-e72d5ba297c3"),
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "BRIDGE", ["bridge3", "ac34f70b-b41b-49ac-a940-7c3bdb4b878f"],
                                   root_path=root_path),
            {
                ("ens33", "4cffe475-81ce-4989-861d-9cd9e08b575f"),
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "BRIDGE", ["bridge4", "37792ba4-bd6c-45f9-84dc-3ec7ef995783"],
                                   root_path=root_path),
            {
                ("ens34", "a8b68194-b45c-4ee8-9907-3361be553ac2"),
            }
        )
        # Vlan over bond created in parse-kickstart
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "MASTER", ["bond5", "e90295f3-bd63-4a21-b92d-78dd2923fe2d"],
                                   root_path=root_path),
            {
                ("ens39", "5e04989f-f97f-471e-8416-59bbea746e5e"),
                ("ens40", "e1d28630-2f6f-4e94-98a9-83facc8d4644")
            }
        )
        # Bridged bond created in parse-kickstart
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "MASTER", ["bond6", "d94466b9-0fcf-43e8-b1d6-51130b6d1c3b"],
                                   root_path=root_path),
            {
                ("ens41", "ec6a4359-8aa2-432b-89a9-ef7707480810"),
                ("ens42", "262214c6-479f-4f2a-8e2f-4a97016834ab")
            }
        )
        self.assertEqual(
            get_slaves_from_ifcfgs(nm_client,
                                   "BRIDGE", ["bridge5", "2f879653-20a3-421d-81ac-7c0c8f192850"],
                                   root_path=root_path),
            {
                ("bond6", "d94466b9-0fcf-43e8-b1d6-51130b6d1c3b"),
            }
        )

    @patch("pyanaconda.modules.network.ifcfg.get_team_port_config_from_connection",
           lambda client, uuid: UUID_TO_TEAM_PORT_CONFIG[uuid])
    @patch("pyanaconda.modules.network.ifcfg.get_team_config_from_connection",
           lambda client, uuid: UUID_TO_TEAM_CONFIG[uuid])
    @patch("pyanaconda.modules.network.ifcfg.get_iface_from_connection",
           lambda client, uuid: UUID_TO_IFACE[uuid])
    @patch("pyanaconda.modules.network.ifcfg.get_iface_from_hwaddr",
           lambda client, hwaddr: HWADDR_TO_IFACE[hwaddr])
    def get_kickstart_network_data_test(self):
        """Test get_kickstart_network_data."""

        self._dump_ifcfg_files(IFCFG_FILES_REAL_ALL_TYPES)
        nm_client = Mock()

        for ifcfg_file_name, _ifcfg_content, expected_ks in IFCFG_FILES_REAL_ALL_TYPES:
            ifcfg_file_path = self._get_ifcfg_file_path(ifcfg_file_name)
            ifcfg = IfcfgFile(ifcfg_file_path)
            generated_ks = get_kickstart_network_data(ifcfg, nm_client, NetworkData,
                                                      root_path=self._root_dir)
            if expected_ks:
                expected_ks = dedent(expected_ks).strip()
            if generated_ks:
                generated_ks = dedent(str(generated_ks)).strip()
            self.assertEqual(generated_ks, expected_ks)

    @patch("pyanaconda.modules.network.ifcfg.find_ifcfg_uuid_of_device",
           lambda client, device_name, root_path: DEVNAME_TO_UUID[device_name])
    def _get_master_slaves_from_ifcfgs_of_a_device_type(self, device_type_master_key):
        nm_client = Mock()
        other_master_uuid = "9fc96635-c7cc-4817-b680-20765bdd9155"
        not_found_uuid = "29c4abb0-f95d-447c-aa94-88f908efe31f"

        ifcfg_files = [
            ("ifcfg-ens3",
             """
             NAME=ens3
             UUID=7fe6a329-d8b2-4f7d-9abe-30e1a7ccca53
             {}={}
             """.format(device_type_master_key, MASTER0_UUID),
             None),
            ("ifcfg-ens5",
             """
             NAME=ens5
             UUID=f0851b85-72a8-42ab-918d-bb8c1ef532c4
             {}={}
             """.format(device_type_master_key, MASTER0_UUID),
             None),
            ("ifcfg-ens7",
             """
             NAME=ens7
             UUID=c34cb55c-da79-4c0f-b0c4-b957a533a456
             {}={}
             """.format(device_type_master_key, other_master_uuid),
             None),
            ("ifcfg-ens8",
             """
             NAME=ens8
             UUID=09f4ae3d-decc-4185-800b-d8bc284fb656
             {}={}
             """.format(device_type_master_key, other_master_uuid),
             None),
            ("ifcfg-ens10",
             """
             NAME=ens10
             UUID=7c95485c-05cb-48bc-9382-18475d028b65
             {}={}
             """.format(device_type_master_key, MASTER1_NAME),
             None),
            ("ifcfg-ens11",
             """
             NAME=ens11
             UUID=063a11d5-90c8-48e9-beae-31e26a2be367
             {}={}
             """.format(device_type_master_key, MASTER1_NAME),
             None),
            ("ifcfg-ens13",
             """
             NAME=ens13
             UUID=063a11d5-90c8-48e9-beae-21e26a2be367
             """,
             None),
        ]
        # Specify by master device name (looked up by uuid) name and uuid (found)
        self._dump_ifcfg_files(ifcfg_files)
        self.assertEqual(
            set(
                get_master_slaves_from_ifcfgs(nm_client, MASTER0_NAME,
                                              root_path=self._root_dir,
                                              uuid=MASTER0_UUID)
            ),
            set([
                ("ens3", "7fe6a329-d8b2-4f7d-9abe-30e1a7ccca53"),
                ("ens5", "f0851b85-72a8-42ab-918d-bb8c1ef532c4")
            ])
        )
        # Specify by master uuid (found)
        self._dump_ifcfg_files(ifcfg_files)
        self.assertEqual(
            set(
                get_master_slaves_from_ifcfgs(nm_client, None,
                                              root_path=self._root_dir,
                                              uuid=MASTER0_UUID)
            ),
            set([
                ("ens3", "7fe6a329-d8b2-4f7d-9abe-30e1a7ccca53"),
                ("ens5", "f0851b85-72a8-42ab-918d-bb8c1ef532c4")
            ])
        )
        # Specify by master uuid - not found
        self._dump_ifcfg_files(ifcfg_files)
        self.assertEqual(
            set(
                get_master_slaves_from_ifcfgs(nm_client, None,
                                              root_path=self._root_dir,
                                              uuid=not_found_uuid)
            ),
            set([
            ])
        )
        # Specify by master device name (uuid is not supplied, but looked up)
        self._dump_ifcfg_files(ifcfg_files)
        self.assertEqual(
            set(
                get_master_slaves_from_ifcfgs(nm_client, MASTER0_NAME,
                                              root_path=self._root_dir,
                                              uuid=None)
            ),
            set([
                ("ens3", "7fe6a329-d8b2-4f7d-9abe-30e1a7ccca53"),
                ("ens5", "f0851b85-72a8-42ab-918d-bb8c1ef532c4")
            ])
        )
        # Specify by master devname in ifcfg file
        self._dump_ifcfg_files(ifcfg_files)
        self.assertEqual(
            set(
                get_master_slaves_from_ifcfgs(nm_client, MASTER1_NAME,
                                              root_path=self._root_dir)
            ),
            set([
                ("ens10", "7c95485c-05cb-48bc-9382-18475d028b65"),
                ("ens11", "063a11d5-90c8-48e9-beae-31e26a2be367")
            ])
        )
        # Specify by master uuid - not found as slaves refer to master by name
        self._dump_ifcfg_files(ifcfg_files)
        self.assertEqual(
            set(
                get_master_slaves_from_ifcfgs(nm_client, None,
                                              root_path=self._root_dir,
                                              uuid=MASTER1_UUID)
            ),
            set([
            ])
        )

    def get_master_slaves_from_ifcfgs_test(self):
        """Test get_master_slaves_from_ifcfgs."""
        for master_key in ("MASTER", "TEAM_MASTER", "BRIDGE"):
            self._get_master_slaves_from_ifcfgs_of_a_device_type(master_key)
