package aQute.lib.telnet;

import java.io.*;

/**
 * The Telnet Protocol
 * 
 * The Telnet protocol is often thought of as simply providing a facility for
 * remote logins to computer via the Internet. This was its original purpose
 * although it can be used for many other purposes.
 * 
 * It is best understood in the context of a user with a simple terminal using
 * the local telnet program (known as the client program) to run a login session
 * on a remote computer where his communications needs are handled by a telnet
 * server program. It should be emphasised that the telnet server can pass on
 * the data it has received from the client to many other types of process
 * including a remote login server. It is described in RFC854 and was first
 * published in 1983.
 * 
 * The Network Virtual Terminal
 * 
 * Communication is established using the TCP/IP protocols and communication is
 * based on a set of facilities known as a Network Virtual Terminal (NVT). At
 * the user or client end the telnet client program is responsible for mapping
 * incoming NVT codes to the actual codes needed to operate the user's display
 * device and is also responsible for mapping user generated keyboard sequences
 * into NVT sequences.
 * 
 * The NVT uses 7 bit codes for characters, the display device, referred to as a
 * printer in the RFC, is only required to display the "standard" printing ASCII
 * characters represented by 7 bit codes and to recognise and process certain
 * control codes. The 7 bit characters are transmitted as 8 bit bytes with most
 * significant bit set to zero. An end-of-line is transmitted as the character
 * sequence CR (carriage return) followed by LF (line feed). If it is desired to
 * transmit an actual carriage return this is transmitted as a carriage return
 * followed by a NUL (all bits zero) character.
 * 
 * NVT ASCII is used by many other Internet protocols.
 * 
 * The following control codes are required to be understood by the Network
 * Virtual Terminal.
 * 
 * Name code Decimal Value Function NULL NUL 0 No operation Line Feed LF 10
 * Moves the printer to the next print line, keeping the same horizontal
 * position. Carriage Return CR 13 Moves the printer to the left margin of the
 * current line.
 * 
 * The following further control codes are optional but should have the
 * indicated defined effect on the display.
 * 
 * Name code Decimal Value Function BELL BEL 7 Produces an audible or visible
 * signal (which does NOT move the print head. Back Space BS 8 Moves the print
 * head one character position towards the left margin. [On a printing devices
 * this mechanism was commonly used to form composite characters by printing two
 * basic characters on top of each other.] Horizontal Tab HT 9 Moves the printer
 * to the next horizontal tab stop. It remains unspecified how either party
 * determines or establishes where such tab stops are located. Vertical Tab VT
 * 11 Moves the printer to the next vertical tab stop. It remains unspecified
 * how either party determines or establishes where such tab stops are located.
 * Form Feed FF 12 Moves the printer to the top of the next page, keeping the
 * same horizontal position. [On visual displays this commonly clears the screen
 * and moves the cursor to the top left corner.]
 * 
 * The NVT keyboard is specified as being capable of generating all 128 ASCII
 * codes by using keys, key combinations or key sequences.
 * 
 */

public class TelnetInputStream extends FilterInputStream {
	/**
	 * Commands The telnet protocol also specifies various commands that control
	 * the method and various details of the interaction between the client and
	 * server. These commands are incorporated within the data stream. The
	 * commands are distinguished by the use of various characters with the most
	 * significant bit set. Commands are always introduced by a character with
	 * the decimal code 255 known as an Interpret as command (IAC) character.
	 * The complete set of special characters is
	 */

	final static byte IAC = (byte) 255; // Interpret as command
	final static byte SE = (byte) 240; // End of subnegotiation parameters.
	final static byte NOP = (byte) 241; // No operation
	final static byte DM = (byte) 242; // Data
	// mark. Indicates the position of a Synch event within the data stream.
	// This should always be accompanied by a TCP urgent notification.
	final static byte BRK = (byte) 243; // Break. Indicates that the "break" or
	// "attention" key was hit.
	final static byte IP = (byte) 244; // Suspend, interrupt, or abort the
	// process which the NVT is connected.
	final static byte AO = (byte) 245; // Abort output. Allows the current
	// process to run to completion but do not send its output to the user. A
	final static byte AYT = (byte) 246; // Are you there. Send back to the
	// NVT some visible evidence that the AYT was received.
	final static byte EC = (byte) 247; // Erase character. The receiver should
	// delete the
	// last preceding undeleted character from the data stream.
	final static byte EL = (byte) 248; // Erase
	// line. Delete characters from the data stream back to but not including
	// the previous CRLF.
	final static byte GA = (byte) 249; // Go ahead. Used, under certain
	// circumstances, to
	// tell the other end that it can transmit.
	final static byte SB = (byte) 250;// Subnegotiation of the
	// indicated option follows.
	final static byte WILL = (byte) 251; // Indicates the desire to begin
	// performing, or confirmation that you are now performing, the indicated
	// option. WONT 252 Indicates the refusal to perform, or continue
	// performing, the indicated option.
	final static byte DO = (byte) 253;
	// Indicates the request that the
	// other party perform, or confirmation that you are expecting the other
	// party to perform, the indicated option.

	final static byte DONT = (byte) 254; // Indicates the demand that the
	// other party stop performing, or confirmation that you are no longer
	// expecting the other party to perform, the indicated option. IAC 255
	// Interpret as command

	/*
	 * 
	 * There are a variety of options that can be negotiated between a telnet
	 * client and server using commands at any stage during the connection. They
	 * are described in detail in separate RFCs. The following are the most
	 * important.
	 */
	final static byte ECHO = 1; // echo RFC 857
	final static byte SUPPRESS = 3; // go ahead RFC 858
	final static byte STATUS = 5; // status RFC 859
	final static byte TIMING_MARK = 6; // timing mark 860
	final static byte TERMINAL_TYPE = 24; // terminal type RFC 1091
	final static byte WINDOW_SIZE = 31; // window size RFC 1073
	final static byte TERMINAL_SPEED = 32; // terminal speed RFC 1079
	final static byte REMOTE_FLOW_CONTROL = 33; // remote flow control RFC 1372
	final static byte LINEMODE = 34; // linemode RFC 1184
	final static byte ENVIRONMENT_VARIABLES = 36; // environment variables
													// RFC 1408
	private OutputStream out;

	/**
	 * 
	 * Options are agreed by a process of negotiation which results in the
	 * client and server having a common view of various extra capabilities that
	 * affect the interchange and the operation of applications.
	 * 
	 * Either end of a telnet dialogue can enable or disable an option either
	 * locally or remotely. The initiator sends a 3 byte command of the form
	 * 
	 * IAC,<type of operation>,<option>
	 * 
	 * The following patterns of option negotiation are defined.
	 * 
	 * <pre>
	 * Sender Sent	Receiver Responds	Implication
     * WILL         DO                  The sender would like to 
     *                                  use a certain option if the receiver 
     *                                  can handle it. The receiver says it 
     *                                  can support the option. Option is now in effect
     * WILL         DONT                The sender would like to use a certain option 
     *                                  if the receiver can handle it. The receiver says it cannot support the option. Option is not in effect.
     * DO           WILL                The sender requests that the receiver use a 
     *                                  certain option. The receiver says it can support 
     *                                  the option. Option is now in effect.
     * DO           WONT	            The sender requests that the receiver use a 
     *                                  certain option. The receiver says it cannot 
     *                                  support the option. Option is not in effect.
     * WONT         DONT                The sender will not use a certain option. Option 
     *                                  disabled. DONT is only valid response.
     * DONT         WONT                The sender requests that the receiver does not 
     *                                  use a certain option. Option disabled. WONT is 
     *                                  only valid response.
	 * </pre>
	 * 
	 * For example if the sender wants the other end to suppress go-ahead it
	 * would send the byte sequence
	 * 
	 * 255(IAC),251(WILL),3
	 * 
	 * The final byte of the three byte sequence identifies the required action.
	 * 
	 * For some of the negotiable options values need to be communicated once
	 * support of the option has been agreed. This is done using sub-option
	 * negotiation. Values are communicated via an exchange of value query
	 * commands and responses in the following form.
	 * 
	 * IAC,SB,<option code number>,1,IAC,SE
	 * 
	 * and
	 * 
	 * IAC,SB,<option code>,0,<value>,IAC,SE
	 * 
	 * For example if the client wishes to identify the terminal type to the
	 * server the following exchange might take place
	 * 
	 * Client 255(IAC),251(WILL),24 Server 255(IAC),253(DO),24 Server
	 * 255(IAC),250(SB),24,1,255(IAC),240(SE) Client
	 * 255(IAC),250(SB),24,0,'V','T','2','2','0',255(IAC),240(SE)
	 * 
	 * The first exchange establishes that terminal type (option number 24) will
	 * be handled, the server then enquires of the client what value it wishes
	 * to associate with the terminal type. The sequence SB,24,1 implies
	 * sub-option negotiation for option type 24, value required (1). The IAC,SE
	 * sequence indicates the end of this request. The repsonse
	 * IAC,SB,24,0,'V'... implies sub-option negotiation for option type 24,
	 * value supplied (0), the IAC,SE sequence indicates the end of the response
	 * (and the supplied value).
	 * 
	 * The encoding of the value is specific to the option but a sequence of
	 * characters, as shown above, is common.
	 * 
	 * Telnet Negotiable Options
	 * 
	 * Many of those listed are self-evident, but some call for more comments.
	 * 
	 * Suppress Go Ahead
	 * 
	 * The original telnet implementation defaulted to "half duplex" operation.
	 * This means that data traffic could only go in one direction at a time and
	 * specific action is required to indicate the end of traffic in one
	 * direction and that traffic may now start in the other direction. [This
	 * similar to the use of "roger" and "over" by amateur and CB radio
	 * operators.] The specific action is the inclusion of a GA character in the
	 * data stream.
	 * 
	 * Modern links normally allow bi-directional operation and the "suppress go
	 * ahead" option is enabled. echo
	 * 
	 * The echo option is enabled, usually by the server, to indicate that the
	 * server will echo every character it receives. A combination of "suppress
	 * go ahead" and "echo" is called character at a time mode meaning that each
	 * character is separately transmitted and echoed.
	 * 
	 * There is an understanding known as kludge line mode which means that if
	 * either "suppress go ahead" or "echo" is enabled but not both then telnet
	 * operates in line at a time mode meaning that complete lines are assembled
	 * at each end and transmitted in one "go". linemode
	 * 
	 * This option replaces and supersedes the line mode kludge. remote flow
	 * control
	 * 
	 * This option controls where the special flow control effects of
	 * Ctrl-S/Ctrl-Q are implemented.
	 * 
	 * Telnet control functions
	 * 
	 * The telnet protocol includes a number of control functions. These are
	 * initiated in response to conditions detected by the client (usually
	 * certain special keys or key combinations) or server. The detected
	 * condition causes a special character to be incorporated in the data
	 * stream.
	 * 
	 * Interrupt Process
	 * 
	 * This is used by the client to cause the suspension or termination of the
	 * server process. Typically the user types Ctrl-C on the keyboard. An IP
	 * (244) character is included in the data stream. Abort Output
	 * 
	 * This is used to suppress the transmission of remote process output. An AO
	 * (238) character is included in the data stream. Are You There
	 * 
	 * This is used to trigger a visible response from the other end to confirm
	 * the operation of the link and the remote process. An AYT (246) character
	 * is incorporated in the data stream. Erase character
	 * 
	 * Sent to the display to tell it to delete the immediately preceding
	 * character from the display. An EC (247) character is incorporated in the
	 * data stream. Erase line
	 * 
	 * Causes the deletion of the current line of input. An EL (248) character
	 * is incorporated in the data stream. Data Mark
	 * 
	 * Some control functions such as AO and IP require immediate action and
	 * this may cause difficulties if data is held in buffers awaiting input
	 * requests from a (possibly misbehaving) remote process. To overcome this
	 * problem a DM (242) character is sent in a TCP Urgent segment, this tells
	 * the receiver to examine the data stream for "interesting" characters such
	 * as IP, AO and AYT. This is known as the telnet synch mechanism.
	 * 
	 * A DM not in a TCP Urgent segment has no effect.
	 * 
	 * The telnet command
	 * 
	 * On most Unix systems a telnet session can be initiated using the telnet
	 * command. Most users simply type
	 * 
	 * telnet <remote host>
	 * 
	 * but if the user just types telnet then various options and subcommands
	 * are available which can be used to study the behaviour of the session.
	 * 
	 * Here's an exmaple of a telnet session from scitsc to ccub
	 * 
	 * bash$ telnet telnet> toggle options Will show option processing. telnet>
	 * open ccub Trying 134.220.1.20 ... Connected to ccub.wlv.ac.uk. Escape
	 * character is '^]'. SENT do SUPPRESS GO AHEAD SENT will TERMINAL TYPE
	 * (reply) RCVD do TERMINAL TYPE (don't reply) RCVD will SUPPRESS GO AHEAD
	 * (don't reply) RCVD will ECHO (reply) SENT do ECHO (reply) RCVD do ECHO
	 * (reply) SENT wont ECHO (reply)
	 * 
	 * 
	 * UNIX(r) System V Release 4.0 (ccub)
	 * 
	 * RCVD dont ECHO (don't reply) login: Login timed out after 60 seconds
	 * Connection closed by foreign host.
	 * 
	 * 
	 * These pages were produced to support a communication systems module that
	 * is no longer taught. Further communication systems notes are available
	 * on-line. Author : Peter Burden
	 * 
	 */

	
	public TelnetInputStream(InputStream in, OutputStream out) {
		super(in);
		this.out = out;
	}
	
	public void echo(boolean enable) {
		
	}
}
