"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.pluginBuildTask = exports.pluginBuildRunner = exports.lintPlugin = exports.versions = exports.prepare = void 0;
const tslib_1 = require("tslib");
const eslint_1 = require("eslint");
const execa_1 = tslib_1.__importDefault(require("execa"));
const fs_1 = require("fs");
const globby_1 = tslib_1.__importDefault(require("globby"));
const path_1 = require("path");
const rimraf_1 = tslib_1.__importDefault(require("rimraf"));
const util_1 = require("util");
const useSpinner_1 = require("../utils/useSpinner");
const bundle_1 = require("./plugin/bundle");
const tests_1 = require("./plugin/tests");
const task_1 = require("./task");
const { access, copyFile } = fs_1.promises;
const { COPYFILE_EXCL } = fs_1.constants;
const rimraf = (0, util_1.promisify)(rimraf_1.default);
const bundlePlugin = (options) => (0, useSpinner_1.useSpinner)('Compiling...', () => (0, bundle_1.bundlePlugin)(options));
// @ts-ignore
const clean = () => (0, useSpinner_1.useSpinner)('Cleaning', () => rimraf(`${process.cwd()}/dist`));
const copyIfNonExistent = (srcPath, destPath) => copyFile(srcPath, destPath, COPYFILE_EXCL)
    .then(() => console.log(`Created: ${destPath}`))
    .catch((error) => {
    if (error.code !== 'EEXIST') {
        throw error;
    }
});
const prepare = () => (0, useSpinner_1.useSpinner)('Preparing', () => Promise.all([
    // Remove local dependencies for @grafana/data/node_modules
    // See: https://github.com/grafana/grafana/issues/26748
    rimraf((0, path_1.resolve)(__dirname, 'node_modules/@grafana/data/node_modules')),
    // Copy only if local tsconfig does not exist.  Otherwise this will work, but have odd behavior
    copyIfNonExistent((0, path_1.resolve)(__dirname, '../../config/tsconfig.plugin.local.json'), (0, path_1.resolve)(process.cwd(), 'tsconfig.json')),
    // Copy only if local prettierrc does not exist.  Otherwise this will work, but have odd behavior
    copyIfNonExistent((0, path_1.resolve)(__dirname, '../../config/prettier.plugin.rc.js'), (0, path_1.resolve)(process.cwd(), '.prettierrc.js')),
]));
exports.prepare = prepare;
const versions = () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    try {
        const nodeVersion = yield (0, execa_1.default)('node', ['--version']);
        console.log(`Using Node.js ${nodeVersion.stdout}`);
        const toolkitVersion = yield (0, execa_1.default)('grafana-toolkit', ['--version']);
        console.log(`Using @grafana/toolkit ${toolkitVersion.stdout}`);
    }
    catch (err) {
        console.log(`Error reading versions`, err);
    }
});
exports.versions = versions;
// @ts-ignore
const typecheckPlugin = () => (0, useSpinner_1.useSpinner)('Typechecking', () => (0, execa_1.default)('tsc', ['--noEmit']));
// @ts-ignore
const getStylesSources = () => (0, globby_1.default)((0, path_1.resolve)(process.cwd(), 'src/**/*.+(scss|css)'));
const lintPlugin = ({ fix } = {}) => (0, useSpinner_1.useSpinner)('Linting', () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    try {
        // Show a warning if the tslint file exists
        yield access((0, path_1.resolve)(process.cwd(), 'tslint.json'));
        console.log('\n');
        console.log('--------------------------------------------------------------');
        console.log('NOTE: @grafana/toolkit has migrated to use eslint');
        console.log('Update your configs to use .eslintrc rather than tslint.json');
        console.log('--------------------------------------------------------------');
    }
    catch (_a) {
        // OK: tslint does not exist
    }
    // @todo should remove this because the config file could be in a parent dir or within package.json
    const configFile = yield (0, globby_1.default)((0, path_1.resolve)(process.cwd(), '.eslintrc?(.cjs|.js|.json|.yaml|.yml)')).then((filePaths) => {
        if (filePaths.length > 0) {
            return filePaths[0];
        }
        else {
            return (0, path_1.resolve)(__dirname, '../../config/eslint.plugin.js');
        }
    });
    const eslint = new eslint_1.ESLint({
        extensions: ['.ts', '.tsx'],
        overrideConfigFile: configFile,
        fix,
        useEslintrc: false,
    });
    const results = yield eslint.lintFiles((0, path_1.resolve)(process.cwd(), 'src'));
    if (fix) {
        yield eslint_1.ESLint.outputFixes(results);
    }
    const { errorCount, warningCount } = results.reduce((acc, value) => {
        acc.errorCount += value.errorCount;
        acc.warningCount += value.warningCount;
        return acc;
    }, { errorCount: 0, warningCount: 0 });
    const formatter = yield eslint.loadFormatter('stylish');
    const resultText = formatter.format(results);
    if (errorCount > 0 || warningCount > 0) {
        console.log('\n');
        console.log(resultText);
        console.log('\n');
    }
    if (errorCount > 0) {
        throw new Error(`${errorCount} linting errors found in ${results.length} files`);
    }
}));
exports.lintPlugin = lintPlugin;
const pluginBuildRunner = ({ coverage, maxJestWorkers, preserveConsole, skipTest, skipLint, }) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    yield (0, exports.versions)();
    yield (0, exports.prepare)();
    if (!skipLint) {
        yield (0, exports.lintPlugin)({ fix: false });
    }
    if (!skipTest) {
        yield (0, tests_1.testPlugin)({ updateSnapshot: false, coverage, maxWorkers: maxJestWorkers, watch: false });
    }
    yield bundlePlugin({ watch: false, production: true, preserveConsole });
});
exports.pluginBuildRunner = pluginBuildRunner;
exports.pluginBuildTask = new task_1.Task('Build plugin', exports.pluginBuildRunner);
//# sourceMappingURL=plugin.build.js.map