"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BettererTaskLogger = void 0;
const tslib_1 = require("tslib");
const logger_1 = require("@betterer/logger");
const render_1 = require("@betterer/render");
const chalk_1 = (0, tslib_1.__importDefault)(require("chalk"));
const error_log_1 = require("../error-log");
const status_1 = require("./status");
const useTaskState_1 = require("./useTaskState");
const useTasksState_1 = require("./useTasksState");
/**
 * @public Ink component for rendering the output of a single {@link BettererTask | `BettererTask`}.
 * The output will update based on the status of the task. Once the task is finished, it will
 * output any logging and any errors (if the task failed).
 */
exports.BettererTaskLogger = (0, render_1.memo)(function BettererTaskLogger(props) {
    const { name, task } = props;
    const [tasksState] = (0, useTasksState_1.useTasks)();
    const [taskState, taskApi] = (0, useTaskState_1.useTaskState)();
    const { error, logs, status } = taskState;
    (0, render_1.useEffect)(() => {
        void (async () => {
            taskApi.reset();
            async function statusError(status) {
                await taskApi.status(['🔥', 'redBright', status]);
            }
            async function statusProgress(status) {
                await taskApi.status(['🤔', 'whiteBright', status]);
            }
            async function statusSuccess(status) {
                await taskApi.status(['✅', 'greenBright', status]);
            }
            async function logCode(codeInfo) {
                const codeFrame = (0, logger_1.code__)(codeInfo);
                await taskApi.log(['💻', 'whiteBright', codeFrame]);
            }
            async function logDebug(log) {
                await taskApi.log(['🤯', 'blueBright', log]);
            }
            async function logError(log) {
                await taskApi.log(['🔥', 'redBright', log]);
            }
            async function logInfo(log) {
                await taskApi.log(['💭', 'gray', log]);
            }
            async function logSuccess(log) {
                await taskApi.log(['✅', 'greenBright', log]);
            }
            async function logWarning(log) {
                await taskApi.log(['🚨', 'yellowBright', log]);
            }
            taskApi.start();
            try {
                const result = await task({
                    progress: statusProgress,
                    code: logCode,
                    debug: logDebug,
                    error: logError,
                    info: logInfo,
                    success: logSuccess,
                    warn: logWarning
                });
                if (typeof result === 'string') {
                    await statusSuccess(result);
                }
                else {
                    await statusSuccess('done!');
                }
                taskApi.stop();
            }
            catch (error) {
                await statusError(error.message);
                taskApi.error(error);
                process.exitCode = 1;
            }
        })();
    }, [name, task, taskApi]);
    return (render_1.React.createElement(render_1.Box, { flexDirection: "column" },
        status && render_1.React.createElement(status_1.BettererTaskStatus, { name: name, status: status }),
        tasksState.endTime != null && logs.length ? (render_1.React.createElement(render_1.Box, { flexDirection: "column" }, logs.map((log, index) => (render_1.React.createElement(render_1.Text, { key: index }, prependLogBlock(log)))))) : null,
        error && render_1.React.createElement(error_log_1.BettererErrorLog, { error: error })));
});
function prependLogBlock(log) {
    const [, colour, message] = log;
    return prependBlock(message, chalk_1.default[colour]('・'));
}
function prependBlock(message, block) {
    return message
        .toString()
        .split('\n')
        .map((line) => `${block} ${line}`)
        .join('\n');
}
//# sourceMappingURL=TaskLogger.js.map