"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createWorkerConfig = exports.createMergeConfig = exports.overrideConfig = exports.createConfig = void 0;
const tslib_1 = require("tslib");
const errors_1 = require("@betterer/errors");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const os = (0, tslib_1.__importStar)(require("os"));
const path = (0, tslib_1.__importStar)(require("path"));
const fs_1 = require("../fs");
const register_1 = require("./register");
const reporters_1 = require("../reporters");
const utils_1 = require("../utils");
const TOTAL_CPUS = os.cpus().length;
async function createConfig(options = {}, versionControl) {
    const tsconfigPath = resolveTsConfigPath(options);
    await (0, register_1.registerExtensions)(tsconfigPath);
    const baseConfig = await createBaseConfig(options, tsconfigPath, versionControl);
    const startConfig = createStartConfig(options);
    const watchConfig = createWatchConfig(options);
    const config = Object.assign(Object.assign(Object.assign({}, baseConfig), startConfig), watchConfig);
    modeConfig(config);
    if (config.tsconfigPath) {
        await validateFilePath({ tsconfigPath: config.tsconfigPath });
    }
    return config;
}
exports.createConfig = createConfig;
function overrideConfig(config, optionsOverride) {
    if (optionsOverride.filters) {
        validateStringRegExpArray({ filters: optionsOverride.filters });
        config.filters = toRegExps(toArray(optionsOverride.filters));
    }
    if (optionsOverride.ignores) {
        validateStringArray({ ignores: optionsOverride.ignores });
        config.ignores = toArray(optionsOverride.ignores);
    }
    if (optionsOverride.reporters) {
        const reporters = toArray(optionsOverride.reporters);
        config.reporter = (0, reporters_1.loadReporters)(reporters, config.cwd);
    }
}
exports.overrideConfig = overrideConfig;
function resolveTsConfigPath(options) {
    const cwd = options.cwd || process.cwd();
    const tsconfigPath = options.tsconfigPath || null;
    return tsconfigPath ? path.resolve(cwd, tsconfigPath) : null;
}
async function createBaseConfig(options, tsconfigPath, versionControl) {
    const cwd = options.cwd || process.cwd();
    const configPaths = options.configPaths ? toArray(options.configPaths) : ['./.betterer'];
    validateStringArray({ configPaths });
    const isDebug = !!process.env.BETTERER_DEBUG;
    const cache = !!options.cachePath || options.cache || false;
    const cachePath = options.cachePath || './.betterer.cache';
    const filters = toRegExps(toArray(options.filters));
    const reporters = toArray(options.reporters);
    const silent = isDebug || options.silent || false;
    const reporter = silent ? (0, reporters_1.loadSilentReporter)() : (0, reporters_1.loadReporters)(reporters, cwd);
    const resultsPath = options.resultsPath || './.betterer.results';
    validateBool({ cache });
    validateString({ cachePath });
    validateString({ cwd });
    validateStringRegExpArray({ filters });
    validateString({ resultsPath });
    validateBool({ silent });
    const workers = validateWorkers(options);
    const validatedConfigPaths = validateConfigPaths(cwd, configPaths);
    const versionControlPath = await versionControl.init(validatedConfigPaths, cwd);
    return {
        cache,
        cachePath: path.resolve(cwd, cachePath),
        cwd,
        configPaths: validatedConfigPaths,
        filters,
        reporter,
        resultsPath: path.resolve(cwd, resultsPath),
        tsconfigPath,
        versionControlPath,
        workers
    };
}
async function createMergeConfig(options) {
    const contents = toArray(options.contents);
    const cwd = options.cwd || process.cwd();
    const resultsPath = path.resolve(cwd, options.resultsPath || './.betterer.results');
    validateStringArray({ contents });
    await validateFilePath({ resultsPath });
    return {
        contents,
        resultsPath: path.resolve(cwd, resultsPath)
    };
}
exports.createMergeConfig = createMergeConfig;
async function createWorkerConfig(config) {
    await (0, register_1.registerExtensions)(config.tsconfigPath);
    return Object.assign(Object.assign({}, config), { reporter: (0, reporters_1.loadSilentReporter)(), workers: 1 });
}
exports.createWorkerConfig = createWorkerConfig;
function createStartConfig(options) {
    const ci = options.ci || false;
    const excludes = toRegExps(toArray(options.excludes)) || [];
    const includes = toArray(options.includes) || [];
    const precommit = options.precommit || false;
    const strict = options.strict || false;
    const update = options.update || false;
    validateBool({ ci });
    validateStringRegExpArray({ excludes });
    validateStringArray({ includes });
    validateBool({ precommit });
    validateBool({ strict });
    validateBool({ update });
    return {
        ci,
        excludes,
        includes,
        precommit,
        strict,
        update
    };
}
function createWatchConfig(options) {
    const ignores = toArray(options.ignores);
    const watch = options.watch || false;
    validateStringArray({ ignores });
    validateBool({ watch });
    return {
        ignores,
        watch
    };
}
function modeConfig(config) {
    // CI mode:
    if (config.ci) {
        config.precommit = false;
        config.strict = true;
        config.update = false;
        config.watch = false;
        return;
    }
    // Precommit mode:
    if (config.precommit) {
        config.ci = false;
        config.strict = true;
        config.update = false;
        config.watch = false;
        return;
    }
    // Strict mode:
    if (config.strict) {
        config.ci = false;
        config.precommit = false;
        config.update = false;
        config.watch = false;
        return;
    }
    // Update mode:
    if (config.update) {
        config.ci = false;
        config.precommit = false;
        config.strict = false;
        config.watch = false;
        return;
    }
    // Watch mode:
    if (config.watch) {
        config.ci = false;
        config.precommit = false;
        config.strict = true;
        config.update = false;
        return;
    }
}
function validateBool(config) {
    const [propertyName] = Object.keys(config);
    const value = config[propertyName];
    validate((0, utils_1.isBoolean)(value), `"${propertyName.toString()}" must be \`true\` or \`false\`. ${recieved(value)}`);
}
function validateNumber(config) {
    const [propertyName] = Object.keys(config);
    const value = config[propertyName];
    validate((0, utils_1.isNumber)(value), `"${propertyName.toString()}" must be a number. ${recieved(value)}`);
}
function validateString(config) {
    const [propertyName] = Object.keys(config);
    const value = config[propertyName];
    validate((0, utils_1.isString)(value), `"${propertyName.toString()}" must be a string. ${recieved(value)}`);
}
function validateStringOrArray(config) {
    const [propertyName] = Object.keys(config);
    const value = config[propertyName];
    validate((0, utils_1.isString)(value) || Array.isArray(value), `"${propertyName.toString()}" must be a string or an array. ${recieved(value)}`);
}
function validateStringArray(config) {
    const [propertyName] = Object.keys(config);
    const value = config[propertyName];
    validateStringOrArray(config);
    validate(!Array.isArray(value) || value.every((item) => (0, utils_1.isString)(item)), `"${propertyName.toString()}" must be an array of strings. ${recieved(value)}`);
}
function validateStringRegExpArray(config) {
    const [propertyName] = Object.keys(config);
    const value = config[propertyName];
    validateStringOrArray(config);
    validate(!Array.isArray(value) || value.every((item) => (0, utils_1.isString)(item) || (0, utils_1.isRegExp)(item)), `"${propertyName.toString()}" must be an array of strings or RegExps. ${recieved(value)}`);
}
function validateConfigPaths(cwd, configPaths) {
    return configPaths.map((configPath) => {
        const absoluteConfigPath = path.resolve(cwd, configPath);
        try {
            return require.resolve(absoluteConfigPath);
        }
        catch (error) {
            throw new errors_1.BettererError(`could not find config file at "${absoluteConfigPath}". 😔`, error);
        }
    });
}
async function validateFilePath(config) {
    const [propertyName] = Object.keys(config);
    const value = config[propertyName];
    validate(value == null || ((0, utils_1.isString)(value) && (await (0, fs_1.read)(value)) !== null), `"${propertyName.toString()}" must be a path to a file. ${recieved(value)}`);
}
function validateWorkers(options = {}) {
    if (options.workers === true || (0, utils_1.isUndefined)(options.workers)) {
        options.workers = TOTAL_CPUS >= 4 ? TOTAL_CPUS - 2 : false;
    }
    if (options.workers === false || options.workers === 0) {
        process.env.WORKER_REQUIRE = 'false';
        // When disabled, set workers to 1 so that BettererRunWorkerPoolΩ
        // can be instantiated correctly:
        options.workers = 1;
    }
    const { workers } = options;
    validateNumber({ workers });
    validate((0, utils_1.isNumber)(workers) && workers > 0 && workers <= TOTAL_CPUS, `"workers" must be more than zero and not more than the number of available CPUs (${TOTAL_CPUS}). To disable workers, set to \`false\`. ${recieved(workers)}`);
    return workers;
}
function validate(value, message) {
    // Wrap the AssertionError in a BettererError for logging:
    try {
        (0, assert_1.default)(value);
    }
    catch (_a) {
        throw new errors_1.BettererError(message);
    }
}
function recieved(value) {
    return `Recieved \`${JSON.stringify(value)}\`.`;
}
function toArray(value) {
    return Array.isArray(value) ? value : (0, utils_1.isUndefined)(value) ? [] : [value];
}
function toRegExps(value) {
    return value.map((v) => ((0, utils_1.isString)(v) ? new RegExp(v, 'i') : v));
}
//# sourceMappingURL=config.js.map