"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BettererResultsFileΩ = void 0;
const tslib_1 = require("tslib");
const assert_1 = (0, tslib_1.__importDefault)(require("assert"));
const fs_1 = require("../fs");
const parse_1 = require("./parse");
const print_1 = require("./print");
class BettererResultsFileΩ {
    constructor(_resultsPath, _versionControl, _baseline, _expected) {
        this._resultsPath = _resultsPath;
        this._versionControl = _versionControl;
        this._baseline = _baseline;
        this._expected = _expected;
    }
    static async create(resultsPath, versionControl) {
        const baseline = await (0, parse_1.parseResults)(resultsPath);
        const expected = baseline;
        return new BettererResultsFileΩ(resultsPath, versionControl, baseline, expected);
    }
    getChanged(runSummaries) {
        const missingRuns = Object.keys(this._expected).filter((name) => !runSummaries.find((runSummary) => runSummary.name === name));
        const changedRuns = runSummaries
            .filter((runSummary) => !runSummary.isNew && !runSummary.isFailed && !runSummary.isSkipped)
            .filter((runSummary) => runSummary.printed !== this._getResult(runSummary.name, this._expected))
            .map((runSummary) => runSummary.name);
        const newRuns = runSummaries
            .filter((runSummary) => runSummary.isNew && !runSummary.isComplete)
            .map((runSummary) => runSummary.name);
        const worseRuns = runSummaries.filter((runSummary) => runSummary.isWorse).map((runSummary) => runSummary.name);
        return [...missingRuns, ...changedRuns, ...newRuns, ...worseRuns];
    }
    getExpected(name) {
        const baseline = this._getResult(name, this._baseline);
        const expected = this._getResult(name, this._expected);
        return [baseline, expected];
    }
    hasResult(name) {
        return Object.hasOwnProperty.call(this._expected, name);
    }
    async sync() {
        this._expected = await (0, parse_1.parseResults)(this._resultsPath);
    }
    async writeNew(suiteSummary) {
        if (suiteSummary.new.length === 0) {
            return;
        }
        const printedNew = (0, print_1.printResults)(suiteSummary.new
            .filter((runSummary) => !runSummary.isComplete)
            .reduce((results, runSummary) => {
            results[runSummary.name] = { value: runSummary.printed };
            return results;
        }, this._expected));
        await (0, fs_1.write)(printedNew, this._resultsPath);
    }
    async write(suiteSummary, precommit) {
        const printedExpected = (0, print_1.printResults)(this._expected);
        const printedResult = (0, print_1.printResults)(suiteSummary.runSummaries
            .filter((runSummary) => runSummary.printed != null)
            .reduce((results, runSummary) => {
            results[runSummary.name] = { value: runSummary.printed };
            return results;
        }, {}));
        const shouldWrite = printedResult !== printedExpected;
        if (shouldWrite) {
            await (0, fs_1.write)(printedResult, this._resultsPath);
            if (precommit) {
                await this._versionControl.add(this._resultsPath);
            }
        }
    }
    _getResult(name, results) {
        const hasResult = Object.hasOwnProperty.call(results, name);
        (0, assert_1.default)(hasResult);
        const { value } = results[name];
        return value;
    }
}
exports.BettererResultsFileΩ = BettererResultsFileΩ;
//# sourceMappingURL=results-file.js.map