"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// Dependencies:
var typescript_1 = require("typescript");
var ast_1 = require("./ast");
var traverse_1 = require("./traverse");
// Constants:
var BRACKETS = {
    '<': '>',
    '[': ']',
    '(': ')',
    '{': '}',
    '\'': '\'',
    '"': '"'
};
var INTERNAL_DATA_REGEXP = /^__TSTEMPLATE_DATA_\d+;?$/;
var INTERNAL_SPREAD_REGEXP = /^__TSTEMPLATE_SPREAD_\d+;?$/;
var INTERNAL_IDENTIFIER_REGEXP = /^(__TSTEMPLATE_(DATA|SPREAD)_\d+);?$/;
var TEMPLATE_REGEXP = /([^\s,;]?)\s*?%(=?)\s*([\s\S]+?)\s*%\s*?([^\s,;]?)/g;
function template(tmpl, data) {
    return compile(tmpl)(data);
}
exports.template = template;
function compile(tmpl) {
    var identifiers = {};
    var index = 0;
    tmpl = tmpl.replace(TEMPLATE_REGEXP, function (match, open, isEval, codePart, close) {
        if (open) {
            var expectedClose = BRACKETS[open];
            if (!expectedClose || close && expectedClose !== close) {
                return match;
            }
        }
        if (isEval) {
            var isSpread = open !== '<' && open !== '\'' && open !== '"';
            var id = index++;
            var identifier = isSpread ? "__TSTEMPLATE_SPREAD_" + id : "__TSTEMPLATE_DATA_" + id;
            identifiers[identifier] = codePart;
            return isSpread ? (open + identifier + close) : identifier;
        }
        if (open !== '<') {
            return match;
        }
        return '';
    });
    var parsed = ast_1.ast(tmpl);
    return function (data) {
        return traverse_1.replace(parsed, {
            visit: function (node) {
                if (isInternalData(node)) {
                    var _a = node.getText().match(INTERNAL_IDENTIFIER_REGEXP), identifier = _a[1];
                    var match = data[identifiers[identifier]];
                    if (typeof match === 'string') {
                        return typescript_1.createLiteral(match);
                    }
                    var matches = Array.isArray(match) ? match : [match];
                    matches.forEach(function (m) {
                        traverse_1.traverse(m, {
                            enter: resetPos
                        });
                    });
                    return matches;
                }
            }
        });
    };
}
exports.compile = compile;
function resetPos(node) {
    node.pos = -1;
    node.end = -1;
}
function isInternalData(node) {
    var text = node.getText();
    return INTERNAL_DATA_REGEXP.test(text) || INTERNAL_SPREAD_REGEXP.test(text);
}
//# sourceMappingURL=template.js.map