import path, { dirname, isAbsolute, relative } from 'path';
import { builtinModules } from 'module';
import { promises } from 'fs';
import findUp from 'find-up';

/**
 * Determines if the `child` path is under the `parent` path.
 */
function isInDirectory(parent, child) {
    const relativePath = relative(parent, child);
    return !relativePath.startsWith('..') && !isAbsolute(relativePath);
}
/**
 * Iterates over package.json file paths recursively found in parent directories, starting from the
 * current working directory. If the current working directory is in a git repository, then package.json
 * files outside of the git repository will not be yielded.
 * @internal
 */
async function* findPackagePaths() {
    // Find git root if in git repository
    const gitDirectoryPath = await findUp('.git', { type: 'directory' });
    const gitRootPath = gitDirectoryPath === undefined
        ? undefined
        : dirname(gitDirectoryPath);
    function isInGitDirectory(path) {
        return gitRootPath === undefined || isInDirectory(gitRootPath, path);
    }
    let cwd = process.cwd();
    let packagePath;
    while ((packagePath = await findUp('package.json', { type: 'file', cwd })) &&
        isInGitDirectory(packagePath)) {
        yield packagePath;
        cwd = dirname(dirname(packagePath));
    }
}
/** @internal */
async function findDependencies({ packagePaths, keys, warnings }) {
    const dependencies = new Set();
    for await (const packagePath of packagePaths) {
        try {
            const pkg = JSON.parse((await promises.readFile(packagePath)).toString()) ?? {};
            // console.log(pkg)
            for (const key of keys) {
                const dependenciesToVersions = pkg[key] ?? {};
                for (const dependency of Object.keys(dependenciesToVersions)) {
                    dependencies.add(dependency);
                }
            }
        }
        catch {
            warnings.push(`Couldn't process '${packagePath}'. Make sure it is a valid JSON or use the 'packagePath' option`);
        }
    }
    return Array.from(dependencies);
}

/**
 * A Rollup plugin that automatically declares NodeJS built-in modules,
 * and optionally npm dependencies, as 'external'.
 */
function externals(options = {}) {
    // This will store all eventual warnings until we can display them.
    const warnings = [];
    // Consolidate options
    const config = {
        builtins: true,
        builtinsPrefix: 'add',
        packagePath: [],
        deps: true,
        devDeps: true,
        peerDeps: true,
        optDeps: true,
        include: [],
        exclude: [],
        prefixedBuiltins: 'strip',
        ...options
    };
    if ('prefixedBuiltins' in options) {
        warnings.push("The 'prefixedBuiltins' option is now deprecated, " +
            "please use 'builtinsPrefix' instead to silent this warning.");
    }
    else if ('builtinsPrefix' in options) {
        config.prefixedBuiltins = options.builtinsPrefix;
    }
    // Map the include and exclude options to arrays of regexes.
    const [include, exclude] = ['include', 'exclude'].map(option => []
        .concat(config[option])
        .reduce((result, entry, index) => {
        if (entry instanceof RegExp)
            result.push(entry);
        else if (typeof entry === 'string')
            result.push(new RegExp('^' + entry.replace(/[.*+?^${}()|[\]\\]/g, '\\$&') + '$'));
        else if (entry) {
            warnings.push(`Ignoring wrong entry type #${index} in '${option}' option: ${JSON.stringify(entry)}`);
        }
        return result;
    }, []));
    // A filter function to keep only non excluded dependencies.
    const isNotExcluded = (id) => !exclude.some(rx => rx.test(id));
    // The array of the final regexes.
    let externals = [];
    const isExternal = (id) => externals.some(rx => rx.test(id));
    // Support for builtin modules.
    const builtins = new Set(), alwaysSchemed = new Set();
    if (config.builtins) {
        const filtered = builtinModules.filter(b => isNotExcluded(b) && isNotExcluded('node:' + b));
        for (const builtin of filtered) {
            builtins.add(builtin);
            if (builtin.startsWith('node:'))
                alwaysSchemed.add(builtin);
            else
                builtins.add('node:' + builtin);
        }
    }
    return {
        name: 'node-externals',
        async buildStart() {
            // Begin with the include option as it has precedence over the other options.
            externals = [...include];
            // Find and filter dependencies, supporting potential import from a sub directory (e.g. 'lodash/map').
            const packagePaths = [].concat(config.packagePath);
            const dependencies = (await findDependencies({
                packagePaths: packagePaths.length > 0 ? packagePaths : findPackagePaths(),
                keys: [
                    config.deps && 'dependencies',
                    config.devDeps && 'devDependencies',
                    config.peerDeps && 'peerDependencies',
                    config.optDeps && 'optionalDependencies'
                ].filter(Boolean),
                warnings
            })).filter(isNotExcluded);
            if (dependencies.length > 0) {
                externals.push(new RegExp('^(?:' + dependencies.join('|') + ')(?:/.+)?$'));
            }
            // Issue the warnings we may have collected.
            let warning;
            while ((warning = warnings.shift())) {
                this.warn(warning);
            }
        },
        resolveId(importee) {
            // Ignore already resolved ids, relative imports and virtual modules.
            if (path.isAbsolute(importee) || /^(?:\0|\.{1,2}[\\/])/.test(importee))
                return null;
            // Handle builtins first.
            if (alwaysSchemed.has(importee))
                return false;
            if (builtins.has(importee)) {
                if (config.prefixedBuiltins === false)
                    return false;
                const stripped = importee.replace(/^node:/, '');
                const prefixed = 'node:' + stripped;
                return config.prefixedBuiltins === 'strip'
                    ? { id: stripped, external: true }
                    : { id: prefixed, external: true };
            }
            // Handle dependencies.
            return isExternal(importee) && isNotExcluded(importee)
                ? false
                : null;
        }
    };
}

export { externals as default, externals };
//# sourceMappingURL=index.js.map
