// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2020-2022 Alibaba Corporation. All rights reserved.
 * Author: Zelin Deng <zelin.deng@linux.alibaba.com>
 * Author: Guanjun <guanjun@linux.alibaba.com>
 * Author: Jiayu Ni <jiayu.ni@linux.alibaba.com>
 */

#define _GNU_SOURCE
#include <pthread.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include "../utils/utils.h"
#include "udma_ulib.h"
#include "ycc_ring.h"

static inline int ycc_ring_full(struct ycc_ring *ring)
{
	return ring->cmd_rd_ptr == (ring->cmd_wr_ptr + 1) % ring->max_desc;
}

static int ycc_ring_empty(struct ycc_ring *ring)
{
	return ring->resp_rd_ptr == ring->resp_wr_ptr;
}

static inline void ycc_check_cmd_state(uint16_t state)
{
	switch (state) {
	case CMD_SUCCESS:
		break;
	case CMD_ILLEGAL:
		ycc_dbg(CMD_RESP_TAG "Illegal cmd\n");
		break;
	case CMD_YCC_UNDERATTACK:
		ycc_dbg(CMD_RESP_TAG "Attack is detected\n");
		break;
	case CMD_INVALID:
		ycc_dbg(CMD_RESP_TAG "Invalid cmd\n");
		break;
	case CMD_ERROR:
		ycc_dbg(CMD_RESP_TAG "Cmd error\n");
		break;
	case CMD_EXCESS:
		ycc_dbg(CMD_RESP_TAG "Excess permission\n");
		break;
	case CMD_KEY_ERROR:
		ycc_dbg(CMD_RESP_TAG "Invalid internal key\n");
		break;
	case CMD_VERIFY_ERROR:
		ycc_dbg(CMD_RESP_TAG "Mac/tag verify failed\n");
		break;
	default:
		ycc_dbg(CMD_RESP_TAG "Unknown error\n");
		break;
	}
}

static void ycc_handle_resp(struct ycc_ring *ring, struct ycc_resp_desc *desc)
{
	struct ycc_flags *aflag;

	__sync_synchronize();

	aflag = (struct ycc_flags *)desc->private_ptr;
	if (!aflag || (uint64_t)aflag == CMD_INVALID_CONTENT_U64) {
		ycc_dbg(CMD_RESP_TAG "Invalid aflag\n");
		return;
	}

	ycc_check_cmd_state(desc->state);

	if (aflag->ycc_done_callback)
		aflag->ycc_done_callback(aflag->ptr, desc->state);

	memset(desc, CMD_INVALID_CONTENT_U8, sizeof(*desc));

	/* We should release the flag memory here */
	free(aflag);
}

/*
 * Submit command to ring's submission queue.
 */
int ycc_enqueue(struct ycc_ring *ring, void *cmd)
{
	struct ycc_cmd_desc *desc = (struct ycc_cmd_desc *)cmd;
	int ring_statue = 0;
	int ret = 0;

	if (!ring || !cmd)
		return -EINVAL;

	pthread_spin_lock(&ring->enqueue_lock);
	ring->cmd_rd_ptr = YCC_CSR_RD(ring->csr_addr, REG_RING_CMD_RD_PTR);
	if (ycc_ring_full(ring)) {
		ret = -EAGAIN;
		goto out;
	}

	memcpy(ring->cmd_base_vaddr + ring->cmd_wr_ptr * YCC_CMD_DESC_SIZE, cmd,
	       YCC_CMD_DESC_SIZE);

	/* Ensure that cmd_wr_ptr update after memcpy */
	__sync_synchronize();
	if (++ring->cmd_wr_ptr == ring->max_desc)
		ring->cmd_wr_ptr = 0;
	YCC_CSR_WR(ring->csr_addr, REG_RING_CMD_WR_PTR, ring->cmd_wr_ptr);

out:
	pthread_spin_unlock(&ring->enqueue_lock);
	return ret;
}

/*
 * Dequeue, read response descriptor
 */
void ycc_dequeue(struct ycc_ring *ring)
{
	struct ycc_resp_desc *resp;
	int cnt = 0;

	ring->resp_wr_ptr = YCC_CSR_RD(ring->csr_addr, REG_RING_RSP_WR_PTR);
	while (!ycc_ring_empty(ring)) {
		resp = (struct ycc_resp_desc *)ring->resp_base_vaddr +
			ring->resp_rd_ptr;
		ycc_handle_resp(ring, resp);

		cnt++;
		if (++ring->resp_rd_ptr == ring->max_desc)
			ring->resp_rd_ptr = 0;
	}

	if (cnt)
		YCC_CSR_WR(ring->csr_addr, REG_RING_RSP_RD_PTR, ring->resp_rd_ptr);
}

extern struct ycc_ring *ycc_select_ring(void);
struct ycc_ring *ycc_crypto_get_ring(void)
{
	return ycc_select_ring();
}

void ycc_crypto_free_ring(struct ycc_ring *ring)
{
	atomic_dec(&ring->refcnt);
}

int ycc_init_ring(struct ycc_ring *ring, uint32_t max_desc)
{
	uint32_t cmd_ring_size, resp_ring_size;
	uint32_t val = 0;
	int order = 0;
	void *base_addr;

	if (max_desc > YCC_RING_MAX_DESC)
		max_desc = YCC_RING_MAX_DESC;
	else if (max_desc < YCC_RING_MIN_DESC)
		max_desc = YCC_RING_MIN_DESC;

	ring->max_desc = max_desc;
	ring->cmd_wr_ptr = YCC_CSR_RD(ring->csr_addr, REG_RING_CMD_WR_PTR);
	ring->cmd_rd_ptr = YCC_CSR_RD(ring->csr_addr, REG_RING_CMD_RD_PTR);
	ring->resp_wr_ptr = YCC_CSR_RD(ring->csr_addr, REG_RING_RSP_WR_PTR);
	ring->resp_rd_ptr = YCC_CSR_RD(ring->csr_addr, REG_RING_RSP_RD_PTR);

	if (ring->cmd_wr_ptr != ring->cmd_rd_ptr ||
	    ring->cmd_wr_ptr != ring->resp_wr_ptr ||
	    ring->resp_wr_ptr != ring->resp_rd_ptr) {
		if (!ring->resp_rd_ptr && !ring->resp_wr_ptr && !ring->cmd_rd_ptr) {
			YCC_CSR_WR(ring->csr_addr, REG_RING_CMD_WR_PTR, 0);
			ring->cmd_wr_ptr = 0;
		} else {
			ycc_err("YCC something wrong with ring ptr, "
				"cmd_wr_ptr: %llx, cmd_rd_ptr: %llx, "
				"resp_wr_ptr: %llx, resp_rd_ptr: llx\n",
				ring->cmd_wr_ptr, ring->cmd_rd_ptr,
				ring->resp_wr_ptr, ring->resp_wr_ptr);
			return -1;
		}
	}

	cmd_ring_size = ring->max_desc * YCC_CMD_DESC_SIZE;
	resp_ring_size = ring->max_desc * YCC_RESP_DESC_SIZE;

	base_addr = ycc_udma_malloc(cmd_ring_size + resp_ring_size);
	if (!base_addr) {
		ycc_err("YCC failed to alloc ring memory\n");
		return -1;
	}

	ring->cmd_base_vaddr = base_addr;
	memset(ring->cmd_base_vaddr, CMD_INVALID_CONTENT_U8, cmd_ring_size);
	ring->cmd_base_paddr = virt_to_phys(ring->cmd_base_vaddr);

	ring->resp_base_vaddr = ring->cmd_base_vaddr + cmd_ring_size;
	memset(ring->resp_base_vaddr, CMD_INVALID_CONTENT_U8, resp_ring_size);
	ring->resp_base_paddr = ring->cmd_base_paddr + cmd_ring_size;

	while (max_desc >>= 1)
		order++;

	YCC_CSR_WR(ring->csr_addr, REG_RING_CFG, (order - 8) & 0x7);
	YCC_CSR_WR(ring->csr_addr, REG_RING_RSP_AFULL_TH, 0);
	YCC_CSR_WR(ring->csr_addr, REG_RING_CMD_BASE_ADDR_LO,
			(uint32_t)ring->cmd_base_paddr & 0xffffffff);
	YCC_CSR_WR(ring->csr_addr, REG_RING_CMD_BASE_ADDR_HI,
			((uint64_t)ring->cmd_base_paddr >> 32) & 0xffffffff);
	YCC_CSR_WR(ring->csr_addr, REG_RING_RSP_BASE_ADDR_LO,
			(uint32_t)ring->resp_base_paddr & 0xffffffff);
	YCC_CSR_WR(ring->csr_addr, REG_RING_RSP_BASE_ADDR_HI,
			((uint64_t)ring->resp_base_paddr >> 32) & 0xffffffff);

	ring->type = USER_RING;
	atomic_set(&ring->refcnt, 0);

	pthread_spin_init(&ring->enqueue_lock, 0);

	return 0;
}

void ycc_release_ring(struct ycc_ring *ring)
{
	ycc_udma_free(ring->cmd_base_vaddr);
	ring->cmd_base_vaddr = NULL;
	ring->resp_base_vaddr = NULL;
	ring->type = FREE_RING;
	pthread_spin_destroy(&ring->enqueue_lock);
}
