// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2020-2022 Alibaba Corporation. All rights reserved.
 * Author: Zelin Deng <zelin.deng@linux.alibaba.com>
 * Author: Guanjun <guanjun@linux.alibaba.com>
 * Author: Jiayu Ni <jiayu.ni@linux.alibaba.com>
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "ycc_uio.h"
#include "utils.h"
#include "pke.h"

struct akcipher_test {
	const char *name;
	const char *key;
	const char *priv;
	const char *msg;
	const char *sig;
	unsigned int key_len;
	unsigned int priv_len;
	unsigned int msg_len;
	unsigned int sig_len;
};

static const struct akcipher_test ecdsa = {
	.name = "ecdsa-nist-p521",
	.key = "\x01\xc6\x23\x4e\x52\x43\x13\x24"
	       "\x6c\x4b\x86\xac\xeb\xc8\xf7\xa5"
	       "\xa0\xff\x25\x76\x73\x9d\xcc\x16"
	       "\x96\x99\xfe\x03\x13\x52\xd4\xc7"
	       "\xd8\xc0\x98\x6a\xf1\xc5\xd1\x40"
	       "\xd5\xb1\x17\xb9\x73\xb7\x72\x62"
	       "\xd3\xe3\xd2\xaf\x43\x31\xc3\x55"
	       "\x41\x38\x76\x67\x90\xc4\x2d\x81"
	       "\xe6\xf5\x01\x9d\xa2\x23\x6b\x45"
	       "\xb5\xda\x57\x81\x83\x7d\xf2\x37"
	       "\xa5\xf2\x00\xf1\x72\x00\xe5\x34"
	       "\x2c\x14\xc4\x49\x3f\x1b\xfb\x37"
	       "\x38\x7c\x94\x1b\x4b\x93\x8e\xb4"
	       "\xec\x7c\x68\x61\x4d\x70\xf1\x99"
	       "\xc5\xff\xac\xc5\xad\x3d\xad\x7f"
	       "\x8b\xf8\xcf\xe9\x88\x61\x7f\x07"
	       "\x83\xad\x22\xcc",
	.key_len = 132,
	.msg = "\xd2\x6c\x40\x12\x21\xfb\x2e\xac"
	       "\x1a\x85\x4c\xdc\x38\x3c\x3a\xe1"
	       "\xe5\x71\x6e\xe5\x13\x42\x1e\x76"
	       "\xc9\x07\x6b\xf3\xa4\x9f\xbd\xd3",
	.msg_len = 32,
	.sig = "\x00\xe8\x91\x3a\x67\x62\x70\x09"
	       "\x2a\xde\x3f\xaa\x0a\xbb\x00\x5f"
	       "\x72\x2a\x59\x5d\xdb\x10\x7f\xfd"
	       "\x4b\xfd\x92\x83\x4a\xb9\x06\xe7"
	       "\x1e\xeb\xde\xdf\xcc\x1f\xb3\x93"
	       "\x72\x3a\xc5\xfc\xfd\xf9\xea\x50"
	       "\x97\x38\x79\x74\xca\x7b\x4a\x1d"
	       "\xee\x71\xbe\x15\xd2\x71\xcb\xc5"
	       "\x12\xf5\x00\xd9\x05\xee\x79\x51"
	       "\xd0\x09\x52\x67\x2f\x02\x69\x69"
	       "\xa2\x9f\xad\x1d\x75\xf8\x06\xc5"
	       "\x9e\x0b\x22\x77\x5d\xc1\xe5\xc1"
	       "\x67\x3d\xeb\x50\x77\xa1\x5e\xf4"
	       "\x48\x14\x32\xde\x70\x53\x64\x3e"
	       "\x21\x19\x4f\xee\xb4\xcb\x10\xd7"
	       "\x41\x4f\x28\xd7\xa5\x14\xeb\xef"
	       "\x5d\xe1\xa4\x04",
	.sig_len = 132,
};

static bool inited;

/*
 * Destructor: It is must if driver initialize successfully.
 */
static void __attribute__((destructor)) drv_exit(void)
{
	if (inited)
		ycc_drv_exit();
}

int main(int argc, char *argv[])
{
	struct akcipher_ctx *cipher;
	struct akcipher_req *req;
	unsigned char src[256];
	unsigned char dst[256];
	unsigned char key[256];
	int ret;

	ret = ycc_drv_init(1);
	if (ret < 0)
		return ret;

	inited = true;
	ret = -1;
	cipher = akcipher_alloc_ctx("ecdsa-nist-p521", CRYPTO_SYNC);
	if (!cipher)
		goto out;

	req = akcipher_alloc_req(cipher);
	if (!req)
		goto free_ctx;

	memcpy(src, ecdsa.msg, ecdsa.msg_len);
	memcpy(dst, ecdsa.sig, ecdsa.sig_len);
	memcpy(key, ecdsa.key, ecdsa.key_len);

	akcipher_set_req(req, src, dst, ecdsa.msg_len, ecdsa.sig_len, 0);

	ret = akcipher_set_pub_key(cipher, key, ecdsa.key_len);
	if (ret < 0)
		goto free_req;

	ret = akcipher_verify(req);
	if (!ret)
		printf("Sample code ecdsa-nist-p521 passed\n");

free_req:
	akcipher_free_req(req);
free_ctx:
	akcipher_free_ctx(cipher);
out:
	return ret;
}
