#package_name:utl_tcp
#auth: xinning.lf

CREATE OR REPLACE PACKAGE BODY utl_smtp AS

  TYPE TYPE_INDEX_TABLE IS TABLE OF BOOLEAN INDEX BY VARCHAR2(32767);
  TYPE TYPE_AUTH_SCHEMES_ARRAY IS TABLE OF VARCHAR2(32767) INDEX BY BINARY_INTEGER;
  ARRAY_AUTH_SCHEMES TYPE_AUTH_SCHEMES_ARRAY;

  STATE_IN_DATA         CONSTANT PLS_INTEGER := 1;
  TRANSIENT_ERRCODE_MIN CONSTANT PLS_INTEGER := 400;
  TRANSIENT_ERRCODE_MAX CONSTANT PLS_INTEGER := 499;
  PERMANENT_ERRCODE_MIN CONSTANT PLS_INTEGER := 500;
  PERMANENT_ERRCODE_MAX CONSTANT PLS_INTEGER := 599;
  SMTP_CHARSET          CONSTANT VARCHAR(100) := 'US7ASCII';

  PROCEDURE RAISE_ERROR_I(ERRCODE IN PLS_INTEGER,
                          MESG    IN VARCHAR2);
  PRAGMA INTERFACE(C, utl_smtp_raise);

  FUNCTION SET_SERVICE_NOT_AVAILABLE RETURN REPLY AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY.CODE := 421;
    V_REPLY.TEXT := 'Service not available';
    RETURN V_REPLY;
  END;

  PROCEDURE OPERATION_STATE_CHECK(C             IN OUT NOCOPY CONNECTION,
                                  CURR_STATE    IN            PLS_INTEGER) AS
  BEGIN

    IF ((CURR_STATE IS NULL) AND (C.PRIVATE_STATE IS NULL)) OR
        (CURR_STATE = STATE_IN_DATA AND C.PRIVATE_STATE = STATE_IN_DATA) THEN
       RETURN;
    END IF;
    RAISE INVALID_OPERATION;
  END;

  PROCEDURE DO_COMMAND(C    IN OUT NOCOPY CONNECTION,
                       CMD  IN VARCHAR2,
                       ARG  IN VARCHAR2 DEFAULT NULL) AS
    DUMMY_RET   PLS_INTEGER;
    CMD_STR VARCHAR(32767);
  BEGIN
    IF (ARG IS NOT NULL) THEN
       CMD_STR := CMD || ' ' || ARG;
    ELSE
       CMD_STR := CMD;
    END IF;
    DUMMY_RET := UTL_TCP.WRITE_LINE(C.PRIVATE_TCP_CON, CMD_STR);
  END;

  FUNCTION GET_REPLIES(C IN OUT NOCOPY CONNECTION) RETURN REPLIES AS
    LINE      VARCHAR2(1000);
    IDX       PLS_INTEGER := 1;
    V_REPLY   REPLY;
    V_REPLIES REPLIES;
    CODE_POS  PLS_INTEGER := 1;
    CODE_LEN  PLS_INTEGER := 3;
    TEXT_POS  PLS_INTEGER := 5;
    EMPTY_POS PLS_INTEGER := 4;
    EMPTY_LEN PLS_INTEGER := 1;
  BEGIN
    WHILE TRUE LOOP
      LINE := UTL_TCP.GET_LINE(C.PRIVATE_TCP_CON, TRUE);
      V_REPLY.CODE := SUBSTR(LINE, CODE_POS, CODE_LEN);
      V_REPLY.TEXT := SUBSTR(LINE, TEXT_POS);
      V_REPLIES(IDX) := V_REPLY;
      EXIT WHEN (SUBSTR(LINE, EMPTY_POS, EMPTY_LEN) = ' ');
      IDX := IDX + 1;
    END LOOP;
    RETURN V_REPLIES;
  EXCEPTION
     WHEN UTL_TCP.END_OF_INPUT OR UTL_TCP.NETWORK_ERROR THEN
       V_REPLY := SET_SERVICE_NOT_AVAILABLE();
       V_REPLIES(IDX) := V_REPLY;
       RETURN V_REPLIES;
  END;

  FUNCTION GET_REPLY(C IN OUT NOCOPY CONNECTION) RETURN REPLY AS
    V_REPLIES REPLIES;
  BEGIN
    V_REPLIES := GET_REPLIES(C);
    RETURN V_REPLIES(V_REPLIES.COUNT);
  END;

  FUNCTION IS_ERROR_REPLY_CODE(CODE IN PLS_INTEGER) RETURN BOOLEAN AS
  BEGIN
    RETURN ((CODE >= TRANSIENT_ERRCODE_MIN) AND (CODE <= PERMANENT_ERRCODE_MAX));
  END;

  PROCEDURE CHECK_ERROR_REPLY_CODE(V_REPLY IN REPLY) AS
  BEGIN
     IF ((V_REPLY.CODE >= TRANSIENT_ERRCODE_MIN) AND (V_REPLY.CODE <= TRANSIENT_ERRCODE_MAX)) THEN
        RAISE_ERROR_I(UTL_SMTP.TRANSIENT_ERROR_ERRCODE, V_REPLY.CODE || ' ' || V_REPLY.TEXT);
     ELSIF ((V_REPLY.CODE >= PERMANENT_ERRCODE_MIN) AND (V_REPLY.CODE <= PERMANENT_ERRCODE_MAX)) THEN
        RAISE_ERROR_I(UTL_SMTP.PERMANENT_ERROR_ERRCODE, V_REPLY.CODE || ' ' || V_REPLY.TEXT);
     END IF;
  END;

  FUNCTION OPEN_CONNECTION(HOST            IN  VARCHAR2,
                           PORT            IN  PLS_INTEGER DEFAULT 25,
                           C               OUT CONNECTION,
                           TX_TIMEOUT      IN  PLS_INTEGER DEFAULT NULL,
                           WALLET_PATH     IN  VARCHAR2 DEFAULT NULL,
                           WALLET_PASSWORD IN  VARCHAR2 DEFAULT NULL,
                           SECURE_CONNECTION_BEFORE_SMTP IN BOOLEAN DEFAULT FALSE,
                           SECURE_HOST     IN  VARCHAR2 DEFAULT NULL)
                           RETURN REPLY AS
    V_REPLY REPLY;
  BEGIN
    OPERATION_STATE_CHECK(C, NULL);
    C.PRIVATE_TCP_CON := UTL_TCP.OPEN_CONNECTION(REMOTE_HOST     => HOST,
                                                 REMOTE_PORT     => PORT,
                                                 CHARSET         => SMTP_CHARSET,
                                                 TX_TIMEOUT      => TX_TIMEOUT,
                                                 WALLET_PATH     => WALLET_PATH,
                                                 WALLET_PASSWORD => WALLET_PASSWORD);
    C.PRIVATE_STATE := NULL;
    C.TX_TIMEOUT    := TX_TIMEOUT;
    C.HOST          := HOST;
    C.PORT          := PORT;

    V_REPLY := GET_REPLY(C);
    RETURN V_REPLY;
  EXCEPTION
     WHEN UTL_TCP.NETWORK_ERROR THEN
       V_REPLY := SET_SERVICE_NOT_AVAILABLE();
       RETURN V_REPLY;
  END;

  FUNCTION OPEN_CONNECTION(HOST            IN  VARCHAR2,
                           PORT            IN  PLS_INTEGER DEFAULT 25,
                           TX_TIMEOUT      IN  PLS_INTEGER DEFAULT NULL,
                           WALLET_PATH     IN  VARCHAR2 DEFAULT NULL,
                           WALLET_PASSWORD IN  VARCHAR2 DEFAULT NULL,
                           SECURE_CONNECTION_BEFORE_SMTP IN BOOLEAN DEFAULT FALSE,
                           SECURE_HOST     IN  VARCHAR2 DEFAULT NULL)
                           RETURN CONNECTION
  AS
    C CONNECTION;
    V_REPLY REPLY;
  BEGIN
    V_REPLY := OPEN_CONNECTION(HOST, PORT, C, TX_TIMEOUT, WALLET_PATH, WALLET_PASSWORD,
                                           SECURE_CONNECTION_BEFORE_SMTP, SECURE_HOST);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
    RETURN C;
  END;

  FUNCTION COMMAND(C    IN OUT NOCOPY CONNECTION,
                   CMD  IN            VARCHAR2,
                   ARG  IN            VARCHAR2 DEFAULT NULL)
                   RETURN REPLY AS
    V_REPLIES REPLIES;
  BEGIN
    OPERATION_STATE_CHECK(C, NULL);
    V_REPLIES := COMMAND_REPLIES(C, CMD, ARG);
    RETURN V_REPLIES(V_REPLIES.COUNT);
  END;

  PROCEDURE COMMAND(C     IN OUT NOCOPY CONNECTION,
                    CMD   IN            VARCHAR2,
                    ARG   IN            VARCHAR2 DEFAULT NULL) AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY := COMMAND(C, CMD, ARG);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
  END;

  FUNCTION COMMAND_REPLIES(C     IN OUT NOCOPY CONNECTION,
                           CMD   IN            VARCHAR2,
                           ARG   IN            VARCHAR2 DEFAULT NULL)
                           RETURN REPLIES AS
    V_REPLY   REPLY;
    V_REPLIES  REPLIES;
  BEGIN
    OPERATION_STATE_CHECK(C, NULL);
    DO_COMMAND(C, CMD, ARG);
    RETURN GET_REPLIES(C);
  EXCEPTION
     WHEN UTL_TCP.NETWORK_ERROR THEN
       V_REPLY := SET_SERVICE_NOT_AVAILABLE();
       V_REPLIES(1)  := V_REPLY;
       RETURN V_REPLIES;
  END;

  FUNCTION HELO(C       IN OUT NOCOPY CONNECTION,
                DOMAIN  IN            VARCHAR2) RETURN REPLY AS
    CMD VARCHAR(100) := 'HELO';
  BEGIN
    RETURN COMMAND(C, CMD, DOMAIN);
  END;

  PROCEDURE HELO(C       IN OUT NOCOPY CONNECTION,
                 DOMAIN  IN            VARCHAR2) AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY := HELO(C, DOMAIN);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
  END;

  FUNCTION EHLO(C       IN OUT NOCOPY CONNECTION,
                DOMAIN  IN            VARCHAR2) RETURN REPLIES AS
    V_REPLIES REPLIES;
    CMD       VARCHAR(100) := 'EHLO';
    AUTH_POS  PLS_INTEGER := 6;
  BEGIN
    V_REPLIES := COMMAND_REPLIES(C, CMD, DOMAIN);
    FOR I IN 1..V_REPLIES.COUNT LOOP
      IF V_REPLIES(I).TEXT LIKE 'AUTH %' THEN
        ARRAY_AUTH_SCHEMES(C.PRIVATE_TCP_CON.PRIVATE_SD) := SUBSTR(V_REPLIES(I).TEXT, AUTH_POS);
      END IF;
    END LOOP;
    RETURN V_REPLIES;
  END;

  PROCEDURE EHLO(C       IN OUT NOCOPY CONNECTION,
                 DOMAIN  IN            VARCHAR2) AS
    V_REPLIES REPLIES;
  BEGIN
    V_REPLIES := EHLO(C, DOMAIN);
    CHECK_ERROR_REPLY_CODE(V_REPLIES(V_REPLIES.COUNT));
  END;

  FUNCTION BASE64_REMOVE_CRLF(STR VARCHAR2) RETURN VARCHAR2 AS
  BEGIN
    RETURN REPLACE(UTL_RAW.CAST_TO_VARCHAR2(UTL_ENCODE.BASE64_ENCODE(UTL_RAW.CAST_TO_RAW(STR))), UTL_TCP.CRLF);
  END;


  FUNCTION AUTH_BY_PLAIN(C        IN OUT NOCOPY CONNECTION,
                         USERNAME IN            VARCHAR2,
                         PASSWORD IN            VARCHAR2) RETURN REPLY AS
    V_REPLY REPLY;
    CMD     VARCHAR2(32767) := 'AUTH PLAIN';
  BEGIN
    V_REPLY := COMMAND(C, CMD);
    IF IS_ERROR_REPLY_CODE(V_REPLY.CODE) THEN
      RETURN V_REPLY;
    END IF;
    CMD := REPLACE(UTL_RAW.CAST_TO_VARCHAR2(UTL_ENCODE.BASE64_ENCODE(UTL_RAW.CONCAT(HEXTORAW('00'), UTL_RAW.CAST_TO_RAW(USERNAME), HEXTORAW('00'), UTL_RAW.CAST_TO_RAW(PASSWORD)))), UTL_TCP.CRLF);
    V_REPLY := COMMAND(C, CMD);
    RETURN V_REPLY;
  END;

  FUNCTION AUTH_BY_LOGIN(C        IN OUT NOCOPY CONNECTION,
                         USERNAME IN            VARCHAR2,
                         PASSWORD IN            VARCHAR2) RETURN REPLY AS
    V_REPLY REPLY;
    CMD     VARCHAR2(100) := 'AUTH LOGIN';
  BEGIN
    V_REPLY := COMMAND(C, CMD);
    IF IS_ERROR_REPLY_CODE(V_REPLY.CODE) THEN
      RETURN V_REPLY;
    END IF;
    V_REPLY := COMMAND(C, BASE64_REMOVE_CRLF(USERNAME));
    IF IS_ERROR_REPLY_CODE(V_REPLY.CODE) THEN
      RETURN V_REPLY;
    END IF;
    V_REPLY := COMMAND(C, BASE64_REMOVE_CRLF(PASSWORD));
    RETURN V_REPLY;
  END;

  FUNCTION GET_SCHEME_FROM_LIST(SCHEMES_STR IN OUT NOCOPY VARCHAR2) RETURN VARCHAR2 AS
    POS    PLS_INTEGER;
    SCHEME_STR VARCHAR2(32767);
  BEGIN
    SCHEMES_STR := LTRIM(SCHEMES_STR);
    POS     := INSTR(SCHEMES_STR, ' ');
    IF POS >= 1 THEN
      SCHEME_STR  := SUBSTR(SCHEMES_STR, 1, POS - 1);
      SCHEMES_STR := SUBSTR(SCHEMES_STR, POS + 1);
    ELSE
      SCHEME_STR  := SCHEMES_STR;
      SCHEMES_STR := NULL;
    END IF;
    RETURN SCHEME_STR;
  END;

  FUNCTION GET_SUPPORTED_SCHEMES(PRIVATE_SD IN PLS_INTEGER) RETURN TYPE_INDEX_TABLE AS
    SCHEME_STR         VARCHAR2(32767);
    SCHEMES_LIST       VARCHAR2(32767);
    SUPPORTED_SCHEMES  TYPE_INDEX_TABLE;
  BEGIN
    SCHEMES_LIST := ARRAY_AUTH_SCHEMES(PRIVATE_SD);
    WHILE TRUE LOOP
      SCHEME_STR := GET_SCHEME_FROM_LIST(SCHEMES_LIST);
      EXIT WHEN SCHEME_STR IS NULL;
      SUPPORTED_SCHEMES(SCHEME_STR) := TRUE;
    END LOOP;
    RETURN SUPPORTED_SCHEMES;
  END;

  FUNCTION AUTH(C        IN OUT NOCOPY CONNECTION,
                USERNAME IN            VARCHAR2,
                PASSWORD IN            VARCHAR2,
                SCHEMES  IN            VARCHAR2
                                       DEFAULT NON_CLEARTEXT_PASSWORD_SCHEMES)
                RETURN REPLY AS
    SCHEME_STR         VARCHAR2(32767);
    SCHEMES_LIST       VARCHAR2(32767);
    SUPPORTED_SCHEMES  TYPE_INDEX_TABLE;
  BEGIN
    IF ARRAY_AUTH_SCHEMES.EXISTS(C.PRIVATE_TCP_CON.PRIVATE_SD) THEN
      SUPPORTED_SCHEMES := GET_SUPPORTED_SCHEMES(C.PRIVATE_TCP_CON.PRIVATE_SD);
    END IF;

    SCHEMES_LIST := SCHEMES;
    WHILE TRUE LOOP
      SCHEME_STR := GET_SCHEME_FROM_LIST(SCHEMES_LIST);
      EXIT WHEN SCHEME_STR IS NULL;

      IF SCHEME_STR NOT IN ('PLAIN', 'LOGIN') THEN
        RAISE_ERROR_I(UNSUPPORTED_SCHEME_ERRCODE, SCHEME_STR);
      END IF;

      IF SUPPORTED_SCHEMES.EXISTS(SCHEME_STR) OR NOT ARRAY_AUTH_SCHEMES.EXISTS(C.PRIVATE_TCP_CON.PRIVATE_SD) THEN
        IF SCHEME_STR = 'PLAIN' THEN
            RETURN AUTH_BY_PLAIN(C, USERNAME, PASSWORD);
        ELSIF SCHEME_STR = 'LOGIN' THEN
            RETURN AUTH_BY_LOGIN(C, USERNAME, PASSWORD);
        END IF;
      END IF;
    END LOOP;
    RAISE NO_SUPPORTED_SCHEME;
  END;

  PROCEDURE AUTH(C        IN OUT NOCOPY CONNECTION,
                 USERNAME IN            VARCHAR2,
                 PASSWORD IN            VARCHAR2,
                 SCHEMES  IN            VARCHAR2
                                        DEFAULT NON_CLEARTEXT_PASSWORD_SCHEMES)
  AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY := AUTH(C, USERNAME, PASSWORD, SCHEMES);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
  END;


  FUNCTION GET_EMAIL_ADDRESS(ADDRESS IN VARCHAR2) RETURN VARCHAR2 AS
  BEGIN
    IF (ADDRESS NOT LIKE '<%>') THEN
      RETURN '<' || ADDRESS || '>';
    ELSE
      RETURN ADDRESS;
    END IF;
  END;

  FUNCTION MAIL(C          IN OUT NOCOPY CONNECTION,
                SENDER     IN            VARCHAR2,
                PARAMETERS IN            VARCHAR2 DEFAULT NULL) RETURN REPLY AS
    ARGUMENT VARCHAR2(512);
    CMD      VARCHAR2(100) := 'MAIL';
  BEGIN
    ARGUMENT := 'FROM:' || GET_EMAIL_ADDRESS(SENDER);
    IF (PARAMETERS IS NOT NULL) THEN
       ARGUMENT := ARGUMENT || ' ' || PARAMETERS;
    END IF;
    RETURN COMMAND(C, CMD, ARGUMENT);
  END;

  PROCEDURE MAIL(C          IN OUT NOCOPY CONNECTION,
                 SENDER     IN            VARCHAR2,
                 PARAMETERS IN            VARCHAR2 DEFAULT NULL) AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY := MAIL(C, SENDER, PARAMETERS);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
  END;


  FUNCTION RCPT(C          IN OUT NOCOPY CONNECTION,
                RECIPIENT  IN            VARCHAR2,
                PARAMETERS IN            VARCHAR2 DEFAULT NULL) RETURN REPLY AS
    ARGUMENT VARCHAR2(512);
    CMD      VARCHAR2(100) := 'RCPT';
  BEGIN
    ARGUMENT := 'TO:' || GET_EMAIL_ADDRESS(RECIPIENT);
    IF (PARAMETERS IS NOT NULL) THEN
       ARGUMENT := ARGUMENT || ' ' || PARAMETERS;
    END IF;
    RETURN COMMAND(C, CMD, ARGUMENT);
  END;

  PROCEDURE RCPT(C          IN OUT NOCOPY CONNECTION,
                 RECIPIENT  IN            VARCHAR2,
                 PARAMETERS IN            VARCHAR2 DEFAULT NULL) AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY := RCPT(C, RECIPIENT, PARAMETERS);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
  END;


  FUNCTION DATA(C     IN OUT NOCOPY CONNECTION,
                BODY  IN            VARCHAR2)
                RETURN REPLY AS
    V_REPLY REPLY;
  BEGIN
    OPEN_DATA(C);
    WRITE_DATA(C, BODY);
    V_REPLY := CLOSE_DATA(C);
    RETURN V_REPLY;
  END;

  PROCEDURE DATA(C     IN OUT NOCOPY CONNECTION,
                 BODY  IN            VARCHAR2) AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY := DATA(C, BODY);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
  END;


  FUNCTION OPEN_DATA(C IN OUT NOCOPY CONNECTION) RETURN REPLY AS
    V_REPLY REPLY;
    CMD     VARCHAR2(100) := 'DATA';
  BEGIN
    V_REPLY := COMMAND(C, CMD);
    C.PRIVATE_STATE := STATE_IN_DATA;
    RETURN V_REPLY;
  END;

  PROCEDURE OPEN_DATA(C IN OUT NOCOPY CONNECTION) AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY := OPEN_DATA(C);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
  END;


  PROCEDURE WRITE_DATA(C     IN OUT NOCOPY CONNECTION,
                       DATA  IN            VARCHAR2) AS
  BEGIN
    BEGIN
      WRITE_RAW_DATA(C, UTL_I18N.STRING_TO_RAW(DATA, SMTP_CHARSET));
    EXCEPTION
      WHEN UTL_TCP.NETWORK_ERROR THEN
       NULL;
    END;
  END;

  PROCEDURE REPLACE_SINGLE_DOT_I(SRC IN RAW, DST OUT NOCOPY RAW);
  PRAGMA INTERFACE(C, utl_smtp_escape_dot);

  PROCEDURE WRITE_RAW_DATA(C     IN OUT NOCOPY CONNECTION,
                           DATA  IN            RAW ) AS
    BUF         RAW(32767);
    TOTAL_LEN   PLS_INTEGER;
    WRITE_LINE  PLS_INTEGER;
    POS         PLS_INTEGER;
    DUMMY_RET   PLS_INTEGER;
    BUFFER_SIZE CONSTANT PLS_INTEGER := 8192;
  BEGIN
    OPERATION_STATE_CHECK(C, STATE_IN_DATA);
    TOTAL_LEN := UTL_RAW.LENGTH(DATA);
    POS := 1;
    WHILE (POS <= TOTAL_LEN) LOOP
      IF TOTAL_LEN - POS >= BUFFER_SIZE THEN
        WRITE_LINE := BUFFER_SIZE;
      ELSE
        WRITE_LINE := TOTAL_LEN - POS + 1;
      END IF;
      REPLACE_SINGLE_DOT_I(UTL_RAW.SUBSTR(DATA, POS, WRITE_LINE), BUF);
      DUMMY_RET := UTL_TCP.WRITE_RAW(C.PRIVATE_TCP_CON, BUF);
      POS := POS + WRITE_LINE;
    END LOOP;
  EXCEPTION
     WHEN UTL_TCP.NETWORK_ERROR THEN
       NULL;
  END;


  FUNCTION CLOSE_DATA(C IN OUT NOCOPY CONNECTION) RETURN REPLY AS
    V_REPLY       REPLY;
    DUMMY_RET   PLS_INTEGER;
    END_DOT     VARCHAR2(1) := '.';
  BEGIN
    OPERATION_STATE_CHECK(C, STATE_IN_DATA);
    DUMMY_RET := UTL_TCP.WRITE_LINE(C.PRIVATE_TCP_CON);
    DUMMY_RET := UTL_TCP.WRITE_LINE(C.PRIVATE_TCP_CON, END_DOT);
    C.PRIVATE_STATE := NULL;
    V_REPLY := GET_REPLY(C);
    RETURN V_REPLY;
  EXCEPTION
     WHEN UTL_TCP.NETWORK_ERROR THEN
       V_REPLY := SET_SERVICE_NOT_AVAILABLE();
       RETURN V_REPLY;
  END;

  PROCEDURE CLOSE_DATA(C IN OUT NOCOPY CONNECTION) AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY := CLOSE_DATA(C);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
  END;


  FUNCTION RSET(C IN OUT NOCOPY CONNECTION) RETURN REPLY AS
    CMD VARCHAR2(100) := 'RSET';
  BEGIN
    RETURN COMMAND(C, CMD);
  END;

  PROCEDURE RSET(C IN OUT NOCOPY CONNECTION) AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY := RSET(C);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
  END;


  FUNCTION VRFY(C          IN OUT NOCOPY CONNECTION,
                RECIPIENT  IN            VARCHAR2) RETURN REPLY AS
    CMD VARCHAR2(100) := 'VRFY';
  BEGIN
    RETURN COMMAND(C, CMD, RECIPIENT);
  END;


  FUNCTION HELP(C        IN OUT NOCOPY CONNECTION,
                COMMAND  IN            VARCHAR2 DEFAULT NULL) RETURN REPLIES AS
    CMD VARCHAR2(100) := 'HELP';
  BEGIN
    RETURN COMMAND_REPLIES(C, CMD, COMMAND);
  END;


  FUNCTION NOOP(C IN OUT NOCOPY CONNECTION) RETURN REPLY AS
    CMD VARCHAR2(100) := 'NOOP';
  BEGIN
    RETURN COMMAND(C, CMD);
  END;

  PROCEDURE NOOP(C IN OUT NOCOPY CONNECTION) AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY := NOOP(C);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
  END;


  FUNCTION QUIT(C IN OUT NOCOPY CONNECTION) RETURN REPLY AS
    V_REPLY REPLY;
    CMD     VARCHAR2(100) := 'QUIT';
  BEGIN
    V_REPLY := COMMAND(C, CMD);
    CLOSE_CONNECTION(C);
    RETURN V_REPLY;
  EXCEPTION
     WHEN UTL_TCP.NETWORK_ERROR THEN
       V_REPLY := SET_SERVICE_NOT_AVAILABLE();
       RETURN V_REPLY;
  END;

  PROCEDURE QUIT(C IN OUT NOCOPY CONNECTION) AS
    V_REPLY REPLY;
  BEGIN
    V_REPLY := QUIT(C);
    CHECK_ERROR_REPLY_CODE(V_REPLY);
  END;


  PROCEDURE CLOSE_CONNECTION(C IN OUT NOCOPY CONNECTION) AS
  BEGIN
    ARRAY_AUTH_SCHEMES.DELETE(C.PRIVATE_TCP_CON.PRIVATE_SD);
    UTL_TCP.CLOSE_CONNECTION(C.PRIVATE_TCP_CON);
    C.PRIVATE_STATE   := NULL;
    C.PRIVATE_TCP_CON := NULL;
    C.HOST            := NULL;
    C.PORT            := NULL;
    C.TX_TIMEOUT      := NULL;
  END;

END;
//
