/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.script;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.breaker.CircuitBreaker;
import org.elasticsearch.common.breaker.CircuitBreakingException;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.cache.RemovalListener;
import org.elasticsearch.common.cache.RemovalNotification;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.script.GeneralScriptException;
import org.elasticsearch.script.ScriptContext;
import org.elasticsearch.script.ScriptContextStats;
import org.elasticsearch.script.ScriptEngine;
import org.elasticsearch.script.ScriptException;
import org.elasticsearch.script.ScriptMetrics;
import org.elasticsearch.script.ScriptService;
import org.elasticsearch.script.ScriptStats;
import org.elasticsearch.script.ScriptType;

public class ScriptCache {
    private static final Logger logger = LogManager.getLogger(ScriptService.class);
    public static final CompilationRate UNLIMITED_COMPILATION_RATE = new CompilationRate(0, TimeValue.ZERO);
    private final Cache<CacheKey, Object> cache;
    private final ScriptMetrics scriptMetrics;
    final AtomicReference<TokenBucketState> tokenBucketState;
    final int cacheSize;
    final TimeValue cacheExpire;
    final CompilationRate rate;
    private final double compilesAllowedPerNano;
    private final String contextRateSetting;

    ScriptCache(int cacheMaxSize, TimeValue cacheExpire, CompilationRate maxCompilationRate, String contextRateSetting) {
        this.cacheSize = cacheMaxSize;
        this.cacheExpire = cacheExpire;
        this.contextRateSetting = contextRateSetting;
        CacheBuilder<CacheKey, Object> cacheBuilder = CacheBuilder.builder();
        if (this.cacheSize >= 0) {
            cacheBuilder.setMaximumWeight(this.cacheSize);
        }
        if (this.cacheExpire.getNanos() != 0L) {
            cacheBuilder.setExpireAfterAccess(this.cacheExpire);
        }
        logger.debug("using script cache with max_size [{}], expire [{}]", (Object)this.cacheSize, (Object)this.cacheExpire);
        this.cache = cacheBuilder.removalListener(new ScriptCacheRemovalListener()).build();
        this.rate = maxCompilationRate;
        this.compilesAllowedPerNano = (double)this.rate.count / (double)this.rate.time.nanos();
        this.scriptMetrics = new ScriptMetrics();
        this.tokenBucketState = new AtomicReference<TokenBucketState>(new TokenBucketState(this.rate.count));
    }

    <FactoryType> FactoryType compile(ScriptContext<FactoryType> context, ScriptEngine scriptEngine, String id, String idOrCode, ScriptType type, Map<String, String> options) {
        String lang = scriptEngine.getType();
        CacheKey cacheKey = new CacheKey(lang, idOrCode, context.name, options);
        try {
            return context.factoryClazz.cast(this.cache.computeIfAbsent(cacheKey, key -> {
                if (logger.isTraceEnabled()) {
                    logger.trace("context [{}]: compiling script, type: [{}], lang: [{}], options: [{}]", (Object)context.name, (Object)type, (Object)lang, (Object)options);
                }
                this.checkCompilationLimit();
                Object compiledScript = scriptEngine.compile(id, idOrCode, context, options);
                this.scriptMetrics.onCompilation();
                return compiledScript;
            }));
        }
        catch (ExecutionException executionException) {
            Throwable cause = executionException.getCause();
            if (cause instanceof ScriptException) {
                throw (ScriptException)cause;
            }
            if (cause instanceof Exception) {
                throw new GeneralScriptException("Failed to compile " + type + " script [" + id + "] using lang [" + lang + "]", cause);
            }
            ScriptCache.rethrow(cause);
            throw new AssertionError((Object)cause);
        }
    }

    static <T extends Throwable> void rethrow(Throwable t) throws T {
        throw t;
    }

    public ScriptStats stats() {
        return this.scriptMetrics.stats();
    }

    public ScriptContextStats stats(String context) {
        return this.scriptMetrics.stats(context);
    }

    void checkCompilationLimit() {
        if (this.rate.equals(UNLIMITED_COMPILATION_RATE)) {
            return;
        }
        TokenBucketState tokenBucketState = this.tokenBucketState.updateAndGet(current -> {
            long now = System.nanoTime();
            long timePassed = now - current.lastInlineCompileTime;
            double scriptsPerTimeWindow = current.availableTokens + (double)timePassed * this.compilesAllowedPerNano;
            if (scriptsPerTimeWindow > (double)this.rate.count) {
                scriptsPerTimeWindow = this.rate.count;
            }
            if (scriptsPerTimeWindow >= 1.0) {
                return new TokenBucketState(now, scriptsPerTimeWindow -= 1.0, true);
            }
            return new TokenBucketState(now, scriptsPerTimeWindow, false);
        });
        if (!tokenBucketState.tokenSuccessfullyTaken) {
            this.scriptMetrics.onCompilationLimit();
            throw new CircuitBreakingException("[script] Too many dynamic script compilations within, max: [" + this.rate + "]; please use indexed, or scripts with parameters instead; this limit can be changed by the [" + this.contextRateSetting + "] setting", CircuitBreaker.Durability.TRANSIENT);
        }
    }

    private class ScriptCacheRemovalListener
    implements RemovalListener<CacheKey, Object> {
        private ScriptCacheRemovalListener() {
        }

        @Override
        public void onRemoval(RemovalNotification<CacheKey, Object> notification) {
            if (logger.isDebugEnabled()) {
                logger.debug("removed [{}] from cache, reason: [{}]", notification.getValue(), (Object)notification.getRemovalReason());
            }
            ScriptCache.this.scriptMetrics.onCacheEviction();
        }
    }

    public static class CompilationRate {
        public final int count;
        public final TimeValue time;
        private final String source;

        public CompilationRate(Integer count, TimeValue time) {
            this.count = count;
            this.time = time;
            this.source = null;
        }

        public CompilationRate(Tuple<Integer, TimeValue> rate) {
            this(rate.v1(), rate.v2());
        }

        public CompilationRate(String value) {
            if (!value.contains("/") || value.startsWith("/") || value.endsWith("/")) {
                throw new IllegalArgumentException("parameter must contain a positive integer and a timevalue, i.e. 10/1m, but was [" + value + "]");
            }
            int idx = value.indexOf("/");
            String count = value.substring(0, idx);
            String time = value.substring(idx + 1);
            try {
                int rate = Integer.parseInt(count);
                if (rate < 0) {
                    throw new IllegalArgumentException("rate [" + rate + "] must be positive");
                }
                TimeValue timeValue = TimeValue.parseTimeValue(time, "script.max_compilations_rate");
                if (timeValue.nanos() <= 0L) {
                    throw new IllegalArgumentException("time value [" + time + "] must be positive");
                }
                if (timeValue.seconds() < 60L) {
                    throw new IllegalArgumentException("time value [" + time + "] must be at least on a one minute resolution");
                }
                this.count = rate;
                this.time = timeValue;
                this.source = value;
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("could not parse [" + count + "] as integer in value [" + value + "]", e);
            }
        }

        public Tuple<Integer, TimeValue> asTuple() {
            return new Tuple<Integer, TimeValue>(this.count, this.time);
        }

        public String toString() {
            return this.source != null ? this.source : this.count + "/" + this.time.toHumanReadableString(0);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CompilationRate that = (CompilationRate)o;
            return this.count == that.count && Objects.equals(this.time, that.time);
        }

        public int hashCode() {
            return Objects.hash(this.count, this.time);
        }
    }

    static class TokenBucketState {
        public final long lastInlineCompileTime;
        public final double availableTokens;
        public final boolean tokenSuccessfullyTaken;

        private TokenBucketState(double availableTokens) {
            this(System.nanoTime(), availableTokens, false);
        }

        private TokenBucketState(long lastInlineCompileTime, double availableTokens, boolean tokenSuccessfullyTaken) {
            this.lastInlineCompileTime = lastInlineCompileTime;
            this.availableTokens = availableTokens;
            this.tokenSuccessfullyTaken = tokenSuccessfullyTaken;
        }
    }

    private static final class CacheKey {
        final String lang;
        final String idOrCode;
        final String context;
        final Map<String, String> options;

        private CacheKey(String lang, String idOrCode, String context, Map<String, String> options) {
            this.lang = lang;
            this.idOrCode = idOrCode;
            this.context = context;
            this.options = options;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CacheKey cacheKey = (CacheKey)o;
            return Objects.equals(this.lang, cacheKey.lang) && Objects.equals(this.idOrCode, cacheKey.idOrCode) && Objects.equals(this.context, cacheKey.context) && Objects.equals(this.options, cacheKey.options);
        }

        public int hashCode() {
            return Objects.hash(this.lang, this.idOrCode, this.context, this.options);
        }
    }
}

