/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

/// Extra keys for events by name.
///
/// For user-defined `EventMetricType`s every event with extras
/// will get their own corresponding event extra data class.
public protocol EventExtras {
    /// Convert the event extras into 2 lists:
    ///
    /// 1. The list of extra key indices.
    ///    Unset keys will be skipped.
    /// 2. The list of extra values.
    func toExtraRecord() -> [String: String]
}

/// Default of no extra keys for events (for the new API).
///
/// An empty class for convenient use as the default set of extra keys
/// that an `EventMetricType` can accept.
public class NoExtras: EventExtras {
    public func toExtraRecord() -> [String: String] {
        return [String: String]()
    }
}

/// This implements the developer facing API for recording events.
///
/// Instances of this class type are automatically generated by the parsers at built time,
/// allowing developers to record events that were previously registered in the metrics.yaml file.
///
/// The Events API only exposes the `EventMetricType.record(extra:)` method, which takes care of validating the input
/// data and making sure that limits are enforced.
public class EventMetricType<ExtraObject: EventExtras> {
    let inner: EventMetric

    /// The public constructor used by automatically generated metrics.
    public init(_ meta: CommonMetricData, _ allowedExtraKeys: [String]? = nil) {
        self.inner = EventMetric(meta, allowedExtraKeys ?? [])
    }

    public func record(_ properties: ExtraObject? = nil) {
        self.recordInternal(extras: properties.map { $0.toExtraRecord() } ?? [:])
    }

    func recordInternal(extras: [String: String]) {
        self.inner.record(extras)
    }

    /// Returns the stored value for testing purposes only. This function will attempt to await the
    /// last task (if any) writing to the the metric's storage engine before returning a value.
    ///
    /// - parameters:
    ///     * pingName: represents the name of the ping to retrieve the metric for.
    ///                 Defaults to the first value in `sendInPings`.
    ///
    /// - returns:  value of the stored metric or `nil` if nothing was recorded.
    public func testGetValue(_ pingName: String? = nil) -> [RecordedEvent]? {
        return self.inner.testGetValue()
    }

    /// Returns the number of errors recorded for the given metric.
    ///
    /// - parameters:
    ///     * errorType: The type of error recorded.
    ///
    /// - returns: The number of errors recorded for the metric for the given error type.
    public func testGetNumRecordedErrors(_ errorType: ErrorType) -> Int32 {
        return self.inner.testGetNumRecordedErrors(errorType)
    }
}
