/**
 * `Monoid` extends the power of `Semigroup` by providing an additional `empty` value.
 *
 * ```ts
 * interface Semigroup<A> {
 *   readonly concat: (x: A, y: A) => A
 * }
 *
 * interface Monoid<A> extends Semigroup<A> {
 *   readonly empty: A
 * }
 * ```
 *
 * This `empty` value should be an identity for the `concat` operation, which means the following equalities hold for any choice of `x`.
 *
 * ```ts
 * concat(x, empty) = concat(empty, x) = x
 * ```
 *
 * Many types that form a `Semigroup` also form a `Monoid`, such as `number`s (with `0`) and `string`s (with `''`).
 *
 * ```ts
 * import { Monoid } from 'fp-ts/Monoid'
 *
 * const monoidString: Monoid<string> = {
 *   concat: (x, y) => x + y,
 *   empty: ''
 * }
 * ```
 *
 * *Adapted from https://typelevel.org/cats*
 *
 * @since 2.0.0
 */
import { Bounded } from './Bounded'
import { Endomorphism } from './Endomorphism'
import { ReadonlyRecord } from './ReadonlyRecord'
import * as Se from './Semigroup'
/**
 * @category type classes
 * @since 2.0.0
 */
export interface Monoid<A> extends Se.Semigroup<A> {
  readonly empty: A
}
/**
 * Get a monoid where `concat` will return the minimum, based on the provided bounded order.
 *
 * The `empty` value is the `top` value.
 *
 * @example
 * import * as N from 'fp-ts/number'
 * import * as M from 'fp-ts/Monoid'
 *
 * const M1 = M.min(N.Bounded)
 *
 * assert.deepStrictEqual(M1.concat(1, 2), 1)
 *
 * @category constructors
 * @since 2.10.0
 */
export declare const min: <A>(B: Bounded<A>) => Monoid<A>
/**
 * Get a monoid where `concat` will return the maximum, based on the provided bounded order.
 *
 * The `empty` value is the `bottom` value.
 *
 * @example
 * import * as N from 'fp-ts/number'
 * import * as M from 'fp-ts/Monoid'
 *
 * const M1 = M.max(N.Bounded)
 *
 * assert.deepStrictEqual(M1.concat(1, 2), 2)
 *
 * @category constructors
 * @since 2.10.0
 */
export declare const max: <A>(B: Bounded<A>) => Monoid<A>
/**
 * The dual of a `Monoid`, obtained by swapping the arguments of `concat`.
 *
 * @example
 * import { reverse } from 'fp-ts/Monoid'
 * import * as S from 'fp-ts/string'
 *
 * assert.deepStrictEqual(reverse(S.Monoid).concat('a', 'b'), 'ba')
 *
 * @category combinators
 * @since 2.10.0
 */
export declare const reverse: <A>(M: Monoid<A>) => Monoid<A>
/**
 * Given a struct of monoids returns a monoid for the struct.
 *
 * @example
 * import { struct } from 'fp-ts/Monoid'
 * import * as N from 'fp-ts/number'
 *
 * interface Point {
 *   readonly x: number
 *   readonly y: number
 * }
 *
 * const M = struct<Point>({
 *   x: N.MonoidSum,
 *   y: N.MonoidSum
 * })
 *
 * assert.deepStrictEqual(M.concat({ x: 1, y: 2 }, { x: 3, y: 4 }), { x: 4, y: 6 })
 *
 * @category combinators
 * @since 2.10.0
 */
export declare const struct: <A>(
  monoids: { [K in keyof A]: Monoid<A[K]> }
) => Monoid<{ readonly [K_1 in keyof A]: A[K_1] }>
/**
 * Given a tuple of monoids returns a monoid for the tuple.
 *
 * @example
 * import { tuple } from 'fp-ts/Monoid'
 * import * as B from 'fp-ts/boolean'
 * import * as N from 'fp-ts/number'
 * import * as S from 'fp-ts/string'
 *
 * const M1 = tuple(S.Monoid, N.MonoidSum)
 * assert.deepStrictEqual(M1.concat(['a', 1], ['b', 2]), ['ab', 3])
 *
 * const M2 = tuple(S.Monoid, N.MonoidSum, B.MonoidAll)
 * assert.deepStrictEqual(M2.concat(['a', 1, true], ['b', 2, false]), ['ab', 3, false])
 *
 * @category combinators
 * @since 2.10.0
 */
export declare const tuple: <A extends readonly unknown[]>(
  ...monoids: { [K in keyof A]: Monoid<A[K]> }
) => Monoid<Readonly<A>>
/**
 * Given a sequence of `as`, concat them and return the total.
 *
 * If `as` is empty, return the monoid `empty` value.
 *
 * @example
 * import { concatAll } from 'fp-ts/Monoid'
 * import * as N from 'fp-ts/number'
 *
 * assert.deepStrictEqual(concatAll(N.MonoidSum)([1, 2, 3]), 6)
 * assert.deepStrictEqual(concatAll(N.MonoidSum)([]), 0)
 *
 * @since 2.10.0
 */
export declare const concatAll: <A>(M: Monoid<A>) => (as: readonly A[]) => A
/**
 * Use [`Monoid`](./void.ts.html#monoid) instead.
 *
 * @category instances
 * @since 2.0.0
 * @deprecated
 */
export declare const monoidVoid: Monoid<void>
/**
 * Use [`tuple`](#tuple) instead.
 *
 * @category combinators
 * @since 2.0.0
 * @deprecated
 */
export declare const getTupleMonoid: <T extends ReadonlyArray<Monoid<any>>>(
  ...monoids: T
) => Monoid<
  {
    [K in keyof T]: T[K] extends Se.Semigroup<infer A> ? A : never
  }
>
/**
 * Use [`struct`](#struct) instead.
 *
 * @category combinators
 * @since 2.0.0
 * @deprecated
 */
export declare const getStructMonoid: <O extends ReadonlyRecord<string, any>>(
  monoids: {
    [K in keyof O]: Monoid<O[K]>
  }
) => Monoid<O>
/**
 * Use [`reverse`](#reverse) instead.
 *
 * @category combinators
 * @since 2.0.0
 * @deprecated
 */
export declare const getDualMonoid: <A>(M: Monoid<A>) => Monoid<A>
/**
 * Use [`max`](#max) instead.
 *
 * @category constructors
 * @since 2.0.0
 * @deprecated
 */
export declare const getJoinMonoid: <A>(B: Bounded<A>) => Monoid<A>
/**
 * Use [`min`](#min) instead.
 *
 * @category constructors
 * @since 2.0.0
 * @deprecated
 */
export declare const getMeetMonoid: <A>(B: Bounded<A>) => Monoid<A>
/**
 * Use [`concatAll`](#concatall) instead.
 *
 * @since 2.0.0
 * @deprecated
 */
export declare const fold: <A>(M: Monoid<A>) => (as: readonly A[]) => A
/**
 * Use [`MonoidAll`](./boolean.ts.html#monoidall) instead.
 *
 * @category instances
 * @since 2.0.0
 * @deprecated
 */
export declare const monoidAll: Monoid<boolean>
/**
 * Use [`MonoidAny`](./boolean.ts.html#monoidany) instead.
 *
 * @category instances
 * @since 2.0.0
 * @deprecated
 */
export declare const monoidAny: Monoid<boolean>
/**
 * Use [`getMonoid`](./function.ts.html#getmonoid) instead.
 *
 * @category instances
 * @since 2.0.0
 * @deprecated
 */
export declare const getFunctionMonoid: <M>(M: Monoid<M>) => <A = never>() => Monoid<(a: A) => M>
/**
 * Use [`getEndomorphismMonoid`](./function.ts.html#getendomorphismmonoid) instead.
 *
 * **Note**. The execution order in [`getEndomorphismMonoid`](./function.ts.html#getendomorphismmonoid) is reversed.
 *
 * @category instances
 * @since 2.0.0
 * @deprecated
 */
export declare const getEndomorphismMonoid: <A = never>() => Monoid<Endomorphism<A>>
/**
 * Use [`Monoid`](./string.ts.html#monoid) instead.
 *
 * @category instances
 * @since 2.0.0
 * @deprecated
 */
export declare const monoidString: Monoid<string>
/**
 * Use [`MonoidSum`](./number.ts.html#monoidsum) instead.
 *
 * @category instances
 * @since 2.0.0
 * @deprecated
 */
export declare const monoidSum: Monoid<number>
/**
 * Use [`MonoidProduct`](./number.ts.html#monoidproduct) instead.
 *
 * @category instances
 * @since 2.0.0
 * @deprecated
 */
export declare const monoidProduct: Monoid<number>
