/*
 * AuthenTec AES2660 driver for libfprint
 * Copyright (c) 2012 Vasily Khoruzhick <anarsoul@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#pragma once

#define AES2660_FRAME_SIZE 0x354

/* *INDENT-OFF* */

/* First init sequence, 0x07 cmd returns following before INIT1:
 * { 0x07, 0x05, 0x00, 0x91, 0x26, 0x21, 0x00, 0x00  }
 */

/* INIT1 */

static const unsigned char pkt13[] = {
  0x44, 0x02, 0x00, 0x04, 0x00,
  0x0d,
  0x23,
  0x42, 0xca, 0x00,
  0x00, 0x20, 0x7f, /* .#B... . */
  0x97, 0x37, 0xcf, 0xdf, 0x96, 0x26, 0xcf, 0xdf, /* .7...&.. */
  0x96, 0xff, 0x96, 0x9f, 0x22, 0x6d, 0x2e, 0x6a, /* ...."m.j */
  0xbf, 0x62, 0x23, 0x23, 0x83, 0xdd, 0x97, 0x26, /* .b##...& */
  0xaf, 0xdd, 0x97, 0x37, 0xaf, 0xa9, 0xcf, 0x70, /* ...7...p */
  0x18, 0xa9, 0xaf, 0x02, 0xef, 0x5d, 0xaf, 0xc5, /* .....].. */
  0xef, 0x2f, 0xff, 0xff, 0x96, 0xdf, 0x96, 0x69, /* ./.....i */
  0xef, 0x20, 0xff, 0xff, 0x96, 0xdf, 0x96, 0x1a, /* . ...... */
  0x27, 0x0b, 0xef, 0x06, 0xaf, 0xf1, 0x81, 0x3a, /* '......: */
  0xff, 0x3f, 0x1d, 0xc1, 0x2e, 0xa3, 0xbf, 0xc2, /* .?...... */
  0x2e, 0x00, 0xdf, 0xff, 0x2d, 0x02, 0x5a, 0xbf, /* ....-.Z. */
  0x22, 0xc1, 0x2e, 0xa3, 0xbf, 0x3c, 0x23, 0x57, /* "....<#W */
  0x23, 0x02, 0x2e, 0x30, 0x80, 0x05, 0x27, 0x12, /* #..0..'. */
  0xef, 0x99, 0xaf, 0x10, 0xef, 0x9b, 0xaf, 0x10, /* ........ */
  0xef, 0x55, 0xaf, 0x3d, 0x23, 0x30, 0xef, 0x20, /* .U.=#0.  */
  0xff, 0x1a, 0x27, 0x05, 0xef, 0x27, 0xff, 0x06, /* ..'..'.. */
  0x27, 0x05, 0x27, 0x3f, 0x80, 0x2b, 0x0d, 0x05, /* '.'?.+.. */
  0x27, 0xe5, 0x31, 0x05, 0x81, 0xe5, 0x39, 0x05, /* '.1...9. */
  0x27, 0xc8, 0x46, 0x0e, 0x5b, 0xd3, 0x45, 0xfd, /* '.F.[.E. */
  0x5a, 0x0f, 0x27, 0xd3, 0x45, 0xfa, 0x5a, 0x0f, /* Z.'.E.Z. */
  0x27, 0xd3, 0x45, 0xf7, 0x5a, 0x41, 0x2e, 0xa1, /* '.E.ZA.. */
  0xdf, 0x9f, 0x28, 0x41, 0x2e, 0xa1, 0xbf, 0x02, /* ..(A.... */
  0x2e, 0x57, 0x80, 0x05, 0x27, 0x9f, 0x22, 0xbf, /* .W..'.". */
  0x22, 0x1f, 0x20, 0xfd, 0x53, 0x3f, 0x20, 0x14, /* ". .S? . */
  0x1f, 0xfa, 0x55, 0x05, 0x27, 0x10, 0xef, 0xd0, /* ..U.'... */
  0x81, 0x65, 0xaf, 0x10, 0xef, 0xd0, 0x81      /* .e..... */
};

static const unsigned char pkt15[] = {
  0x42, 0x48, 0x00,
  0x64, 0x20, 0x9c, 0xaf, 0x10, /* BH.d ... */
  0xef, 0xd0, 0x81, 0x9a, 0xaf, 0x05, 0x27, 0x11, /* ......'. */
  0x00, 0x51, 0x87, 0x91, 0x81, 0xd1, 0x6c, 0xd2, /* .Q....l. */
  0x75, 0xe2, 0x80, 0xf2, 0x80, 0xf0, 0x57, 0xf4, /* u.....W. */
  0x7b, 0xf8, 0x7e, 0xfc, 0x8b, 0x10, 0xcf, 0x7c, /* {.~....| */
  0xff, 0x3f, 0x00, 0x20, 0xff, 0x00, 0x13, 0x1a, /* .?. .... */
  0x27, 0x07, 0x27, 0x05, 0x27, 0x01, 0x00, 0x11, /* '.'.'... */
  0x00, 0x21, 0x00, 0x31, 0x00, 0x31, 0x01, 0x32, /* .!.1.1.2 */
  0x01, 0x33, 0x01, 0x33, 0x03, 0x43, 0x03, 0x53, /* .3.3.C.S */
  0x03, 0x63, 0x03                              /* .c. */
};

static const unsigned char pkt17[] = {
  0x42, 0xca, 0x00,
  0x87, 0x20, 0x51, 0x85, 0x44, /* B... Q.D */
  0x1e, 0x03, 0x55, 0xc8, 0x46, 0x01, 0x5b, 0xe4, /* ..U.F.[. */
  0x2f, 0xc9, 0x50, 0x49, 0x1e, 0x16, 0x55, 0xe4, /* /.PI..U. */
  0x2f, 0xc5, 0x50, 0x02, 0xef, 0x02, 0x2e, 0x62, /* /.P....b */
  0x82, 0xfe, 0xcf, 0x08, 0xaf, 0xfe, 0xcf, 0x09, /* ........ */
  0xaf, 0xfe, 0xcf, 0x0b, 0xaf, 0x55, 0x81, 0x02, /* .....U.. */
  0xcf, 0x0f, 0x1c, 0x40, 0x18, 0x08, 0x42, 0x00, /* ...@..B. */
  0x5b, 0x20, 0x18, 0x3d, 0x23, 0x02, 0xaf, 0x10, /* [ .=#... */
  0xef, 0x55, 0xaf, 0x05, 0x27, 0x4a, 0x1e, 0x01, /* .U..'J.. */
  0x55, 0xe4, 0x2f, 0xac, 0x50, 0x4e, 0x1e, 0x01, /* U./.PN.. */
  0x55, 0x00, 0x2e, 0x0d, 0x7b, 0x4f, 0x1e, 0x01, /* U...{O.. */
  0x55, 0x02, 0x2e, 0xbd, 0x71, 0x51, 0x1e, 0x1a, /* U...qQ.. */
  0x55, 0xf9, 0xcf, 0x67, 0xaf, 0xcf, 0x1c, 0xf9, /* U..g.... */
  0xaf, 0xff, 0x86, 0xbf, 0x22, 0x09, 0xbf, 0xfe, /* ...."... */
  0xdf, 0x00, 0x2e, 0xd7, 0x8b, 0x01, 0xef, 0x3b, /* .......; */
  0xaf, 0x01, 0x2e, 0x63, 0x84, 0x3b, 0x80, 0x3b, /* ...c.;.; */
  0xcf, 0x1f, 0x99, 0x01, 0x2e, 0x07, 0x85, 0x00, /* ........ */
  0x2e, 0xa7, 0x8a, 0xbf, 0x22, 0xc1, 0x2e, 0x49, /* ...."..I */
  0xbf, 0x67, 0xcf, 0xf9, 0xaf, 0x05, 0x27, 0x54, /* .g....'T */
  0x1e, 0x32, 0x55, 0x04, 0xef, 0x20, 0x80, 0xe4, /* .2U.. .. */
  0x2f, 0x81, 0x5a, 0x23, 0x81, 0xfe, 0xcf, 0xfe, /* /.Z#.... */
  0xdf, 0xff, 0x96, 0xdf, 0x96, 0xfe, 0xcf, 0xfe, /* ........ */
  0xdf, 0xff, 0x96, 0xdf, 0x96, 0xdf, 0x22, 0x01, /* ......". */
  0x55, 0xff, 0x22, 0x06, 0x54, 0xc1, 0x97, 0x83, /* U.".T... */
  0x03, 0x0e, 0x52, 0x02, 0x5e, 0xc0, 0x97, 0x82, /* ..R.^... */
  0x03, 0x0a, 0x52, 0xc2, 0x97                  /* ..R.. */
};

static const unsigned char pkt19[] = {
  0x42, 0xca, 0x00,
  0xeb, 0x20, 0xc3, 0x9f, 0xcd, /* B... ... */
  0x80, 0x07, 0x80, 0x00, 0xef, 0xff, 0xaf, 0x00, /* ........ */
  0x2e, 0xa7, 0x8a, 0x04, 0x0c, 0x29, 0x81, 0x05, /* .....).. */
  0x27, 0xc0, 0x97, 0xc1, 0x9f, 0xcd, 0x80, 0x07, /* '....... */
  0x80, 0x01, 0xef, 0xff, 0xaf, 0x00, 0x2e, 0xa9, /* ........ */
  0x8a, 0xc2, 0x97, 0xc3, 0x9f, 0x80, 0x01, 0x41, /* .......A */
  0x01, 0x9f, 0x90, 0xff, 0x90, 0x02, 0x2e, 0xe3, /* ........ */
  0x70, 0x55, 0x1e, 0x16, 0x55, 0x54, 0x85, 0xe4, /* pU..UT.. */
  0x2f, 0x4d, 0x5b, 0x07, 0xef, 0x20, 0x80, 0xe4, /* /M[.. .. */
  0x2f, 0x49, 0x5a, 0xfe, 0xcf, 0x0a, 0xaf, 0xfe, /* /IZ..... */
  0xcf, 0xfe, 0xcf, 0x41, 0x2e, 0x44, 0xbf, 0xfe, /* ...A.D.. */
  0xcf, 0x41, 0x2e, 0x45, 0xbf, 0xfe, 0xcf, 0x52, /* .A.E...R */
  0xaf, 0xfe, 0xcf, 0x53, 0xaf, 0xfe, 0xcf, 0x5a, /* ...S...Z */
  0xaf, 0x05, 0x27, 0x58, 0x1e, 0x0d, 0x55, 0x02, /* ..'X..U. */
  0x44, 0x01, 0x5a, 0x00, 0x2e, 0x54, 0x79, 0xfe, /* D.Z..Ty. */
  0xcf, 0x75, 0xaf, 0xfe, 0xdf, 0x76, 0xbf, 0x41, /* .u...v.A */
  0x2e, 0xa4, 0xdf, 0xdf, 0x29, 0x41, 0x2e, 0xa4, /* ....)A.. */
  0xbf, 0x05, 0x27, 0x59, 0x1e, 0x03, 0x55, 0xfe, /* ..'Y..U. */
  0xcf, 0x10, 0xaf, 0x01, 0x2e, 0x94, 0x76, 0x63, /* ......vc */
  0x1e, 0x1a, 0x55, 0x54, 0x85, 0xe4, 0x2f, 0x1e, /* ..UT../. */
  0x5b, 0x0c, 0xef, 0x20, 0x80, 0xe4, 0x2f, 0x1a, /* [.. ../. */
  0x5a, 0xf0, 0xef, 0x00, 0xff, 0x19, 0x27, 0x08, /* Z.....'. */
  0xef, 0x06, 0xaf, 0xfe, 0xcf, 0x5f, 0x9f, 0x06, /* ....._.. */
  0x25, 0xfb, 0x55, 0xfe, 0xcf, 0xfe, 0xcf, 0x41, /* %.U....A */
  0x2e, 0x40, 0xbf, 0xfe, 0xcf, 0x41, 0x2e, 0x41, /* .@...A.A */
  0xbf, 0xfe, 0xcf, 0x41, 0x2e                  /* ...A. */
};

static const unsigned char pkt21[] = {
  0x42, 0xca, 0x00,
  0x4f, 0x21, 0x42, 0xbf, 0x05, /* B..O!B.. */
  0x27, 0x64, 0x1e, 0x31, 0x55, 0xb0, 0xef, 0x00, /* 'd.1U... */
  0xff, 0x19, 0x27, 0x0b, 0xef, 0x06, 0xaf, 0xfe, /* ..'..... */
  0xcf, 0x5f, 0x9f, 0x06, 0x25, 0xfb, 0x55, 0xfe, /* ._..%.U. */
  0xcf, 0x56, 0xaf, 0xfe, 0xcf, 0xae, 0xdf, 0x1f, /* .V...... */
  0x1c, 0xe0, 0x1d, 0x5f, 0x02, 0xae, 0xbf, 0xfe, /* ..._.... */
  0xcf, 0x41, 0x2e, 0x54, 0xbf, 0xfe, 0xcf, 0x41, /* .A.T...A */
  0x2e, 0x77, 0xbf, 0xfe, 0xcf, 0x41, 0x2e, 0x55, /* .w...A.U */
  0xbf, 0xfe, 0xcf, 0x41, 0x2e, 0x78, 0xbf, 0xfe, /* ...A.x.. */
  0xcf, 0xbb, 0xaf, 0xfe, 0xcf, 0xbc, 0xaf, 0xfe, /* ........ */
  0xcf, 0xbd, 0xaf, 0xfe, 0xcf, 0xbe, 0xaf, 0xfe, /* ........ */
  0xcf, 0x83, 0xaf, 0xfe, 0xcf, 0x84, 0xaf, 0xfe, /* ........ */
  0xcf, 0x81, 0xaf, 0xfe, 0xcf, 0x82, 0xaf, 0xfe, /* ........ */
  0xcf, 0x24, 0xaf, 0xfe, 0xcf, 0x02, 0x33, 0x05, /* .$....3. */
  0x27, 0x66, 0x1e, 0x0b, 0x55, 0xc5, 0xef, 0x2f, /* 'f..U../ */
  0xff, 0x1a, 0x27, 0x0b, 0xef, 0x06, 0xaf, 0xfe, /* ..'..... */
  0xcf, 0xfe, 0xdf, 0x06, 0x27, 0x01, 0x0e, 0x06, /* ....'... */
  0x25, 0xf9, 0x55, 0x05, 0x27, 0x73, 0x1e, 0x02, /* %.U.'s.. */
  0x55, 0x02, 0x2e, 0x5a, 0x88, 0x05, 0x27, 0x7d, /* U..Z..'} */
  0x1e, 0x0c, 0x55, 0x02, 0x2e, 0xb7, 0x81, 0xe3, /* ..U..... */
  0x2f, 0xb9, 0x5a, 0xfe, 0xcf, 0xfe, 0xdf, 0x1b, /* /.Z..... */
  0x27, 0xfe, 0xcf, 0x36, 0x03, 0xfe, 0xdf, 0x3f, /* '..6...? */
  0x02, 0x9f, 0x9d, 0x05, 0x27, 0x7e, 0x1e, 0x0c, /* ....'~.. */
  0x55, 0x02, 0x2e, 0xb7, 0x81, 0xe3, 0x2f, 0xaa, /* U...../. */
  0x5a, 0xfe, 0xcf, 0xfe, 0xdf, 0x1b, 0x27, 0x07, /* Z.....'. */
  0x80, 0xf6, 0x97, 0xff, 0xaf                  /* ..... */
};

static const unsigned char pkt23[] = {
  0x42, 0xca, 0x00,
  0xb3, 0x21, 0x00, 0x2e, 0xa7, /* B...!... */
  0x8a, 0x05, 0x27, 0x61, 0x74, 0xff, 0x96, 0xc6, /* ..'at... */
  0x2e, 0x01, 0xdf, 0x3f, 0x2c, 0xdd, 0x9f, 0x05, /* ...?,... */
  0x27, 0x01, 0xef, 0x20, 0x80, 0xe3, 0x2f, 0x96, /* '.. ../. */
  0x5a, 0xfe, 0xcf, 0xdf, 0x22, 0xe9, 0x2f, 0xf0, /* Z..."./. */
  0x54, 0x01, 0xff, 0x85, 0xef, 0xff, 0x96, 0xdf, /* T....... */
  0x96, 0x34, 0xff, 0x8d, 0xef, 0x19, 0x27, 0x03, /* .4....'. */
  0xcf, 0x7f, 0x1c, 0x5f, 0x9f, 0xdd, 0x97, 0x5f, /* ..._..._ */
  0x9f, 0xdd, 0x97, 0x5f, 0x9f, 0x05, 0xef, 0x06, /* ..._.... */
  0xaf, 0x9f, 0x22, 0x5f, 0x9f, 0x06, 0x25, 0xfc, /* .."_..%. */
  0x55, 0x5f, 0x9f, 0x06, 0x25, 0xfc, 0x55, 0x20, /* U_..%.U  */
  0xef, 0x08, 0xaf, 0x09, 0x23, 0x01, 0x2e, 0x06, /* ....#... */
  0x82, 0x51, 0x81, 0x03, 0xef, 0xfc, 0xaf, 0x84, /* .Q...... */
  0xef, 0xaa, 0xaf, 0x08, 0xef, 0xe3, 0xaf, 0x9f, /* ........ */
  0x22, 0xe2, 0xaf, 0x21, 0xef, 0x41, 0x2e, 0x47, /* "..!.A.G */
  0xbf, 0x27, 0xef, 0xf4, 0xaf, 0x2d, 0xff, 0xa7, /* .'...-.. */
  0xef, 0xe1, 0xbf, 0xe0, 0xaf, 0x08, 0xef, 0x10, /* ........ */
  0xaf, 0x01, 0x2e, 0x94, 0x86, 0xed, 0x80, 0x01, /* ........ */
  0x2e, 0x18, 0x87, 0x01, 0x2e, 0x1a, 0x87, 0x9b, /* ........ */
  0x22, 0x9b, 0x22, 0x34, 0xff, 0x95, 0xef, 0x19, /* "."4.... */
  0x27, 0x2d, 0xff, 0xa7, 0xef, 0x1a, 0x27, 0xc0, /* '-....'. */
  0xef, 0x06, 0xaf, 0xdc, 0x97, 0x5f, 0x9f, 0x06, /* ....._.. */
  0xdf, 0xc0, 0x1f, 0x03, 0x54, 0xdf, 0x22, 0x01, /* ....T.". */
  0x54, 0x00, 0x2e, 0xb9, 0x8b, 0x06, 0x25, 0xf4, /* T.....%. */
  0x55, 0x34, 0xff, 0x91, 0xef, 0x19, 0x27, 0xdd, /* U4....'. */
  0x97, 0xdd, 0x9f, 0x5f, 0x9f                  /* ..._. */
};

static const unsigned char pkt25[] = {
  0x42, 0xb2, 0x00,
  0x17, 0x22, 0x7f, 0x9f, 0xfd, /* B..."... */
  0x0d, 0xdf, 0x22, 0x01, 0x55, 0xff, 0x22, 0x01, /* ..".U.". */
  0x54, 0x01, 0xef, 0x1f, 0x98, 0x2f, 0xef, 0x41, /* T..../.A */
  0x2e, 0x47, 0xbf, 0x11, 0xef, 0xf4, 0xaf, 0x2d, /* .G.....- */
  0xff, 0xa7, 0xef, 0xe1, 0xbf, 0xe0, 0xaf, 0x01, /* ........ */
  0x2e, 0x18, 0x87, 0x01, 0x2e, 0x1a, 0x87, 0x9b, /* ........ */
  0x22, 0x9b, 0x22, 0x35, 0xff, 0x55, 0xef, 0x19, /* "."5.U.. */
  0x27, 0x2d, 0xff, 0xa7, 0xef, 0x1a, 0x27, 0xc0, /* '-....'. */
  0xef, 0x06, 0xaf, 0xdc, 0x97, 0xff, 0x1a, 0x5f, /* ......._ */
  0x9f, 0x06, 0xdf, 0xc0, 0x1f, 0x03, 0x54, 0xdf, /* ......T. */
  0x22, 0x01, 0x54, 0x00, 0x2e, 0xb9, 0x8b, 0x06, /* ".T..... */
  0x25, 0xf3, 0x55, 0x34, 0xff, 0x93, 0xef, 0x19, /* %.U4.... */
  0x27, 0xdd, 0x97, 0xdd, 0x9f, 0x5f, 0x9f, 0x7f, /* '...._.. */
  0x9f, 0xfb, 0x0d, 0xdf, 0x22, 0x01, 0x54, 0x01, /* ....".T. */
  0xef, 0x1f, 0x98, 0xff, 0x22, 0x01, 0x54, 0x01, /* ....".T. */
  0xef, 0x1f, 0x98, 0x34, 0xff, 0x8d, 0xef, 0x19, /* ...4.... */
  0x27, 0x01, 0x2e, 0x0c, 0x85, 0x00, 0x2e, 0xa7, /* '....... */
  0x8a, 0x55, 0x81, 0x02, 0xef, 0xfc, 0xaf, 0x88, /* .U...... */
  0xef, 0xaa, 0xaf, 0x00, 0xef, 0x41, 0x2e, 0x47, /* .....A.G */
  0xbf, 0x05, 0x27, 0xf9, 0x35, 0xc1, 0x2e, 0x4e, /* ..'.5..N */
  0xdf, 0x0f, 0x1d, 0x09, 0x27, 0x5f, 0x02, 0xc1, /* ....'_.. */
  0x2e, 0x4e, 0xbf, 0x05, 0x27, 0x9f, 0x22, 0x02, /* .N..'.". */
  0x2e, 0xa6, 0x85, 0x05, 0x27                  /* ....' */
};

static const unsigned char pkt27[] = {
  0x42, 0xca, 0x00,
  0x6f, 0x22, 0x51, 0x85, 0x01, /* B..o"Q.. */
  0x1e, 0x38, 0x55, 0x26, 0x40, 0x12, 0x5b, 0x26, /* .8U&@.[& */
  0x42, 0x2e, 0x5a, 0x26, 0x41, 0x0e, 0x5a, 0x26, /* B.Z&A.Z& */
  0xcf, 0x40, 0x1a, 0x26, 0xaf, 0x9f, 0x2d, 0x27, /* .@.&..-' */
  0x5a, 0x07, 0x80, 0x01, 0xef, 0xff, 0xaf, 0x26, /* Z......& */
  0x43, 0x01, 0x5b, 0x74, 0x25, 0x00, 0x55, 0x26, /* C.[t%.U& */
  0x32, 0x00, 0x2e, 0xa7, 0x7a, 0x02, 0x2e, 0xc0, /* 2...z... */
  0x85, 0x26, 0x40, 0x08, 0x5b, 0x41, 0x2e, 0x52, /* .&@.[A.R */
  0xdf, 0xdf, 0x2d, 0x04, 0x5b, 0x26, 0x43, 0x01, /* ..-.[&C. */
  0x5b, 0x74, 0x25, 0x00, 0x55, 0x26, 0x32, 0x41, /* [t%.U&2A */
  0x2e, 0x53, 0xdf, 0xc1, 0x2e, 0x79, 0xdf, 0xff, /* .S...y.. */
  0x96, 0xdf, 0x96, 0x41, 0x2e, 0x55, 0xdf, 0xc1, /* ...A.U.. */
  0x2e, 0x78, 0xdf, 0x9d, 0x01, 0x5d, 0x01, 0xff, /* .x...].. */
  0x22, 0x00, 0x54, 0xff, 0xef, 0x2e, 0xaf, 0x05, /* ".T..... */
  0x27, 0x07, 0x80, 0x9f, 0x22, 0xff, 0xaf, 0x00, /* '..."... */
  0x2e, 0xa7, 0x8a, 0x05, 0x27, 0x03, 0x1e, 0x3f, /* ....'..? */
  0x55, 0x02, 0x2e, 0xad, 0x85, 0xa1, 0xcf, 0xfe, /* U....... */
  0x1c, 0xa1, 0xaf, 0xed, 0x80, 0x31, 0x81, 0x02, /* .....1.. */
  0x2e, 0x51, 0x83, 0xff, 0x96, 0xdf, 0x96, 0xa1, /* .Q...... */
  0xcf, 0x01, 0x18, 0xa1, 0xaf, 0xed, 0x80, 0x64, /* .......d */
  0xef, 0xbf, 0x22, 0xa9, 0x80, 0x31, 0x81, 0x02, /* .."..1.. */
  0x2e, 0x51, 0x83, 0xff, 0x96, 0xdf, 0x96, 0x9b, /* .Q...... */
  0x22, 0x14, 0x1e, 0x03, 0x53, 0xff, 0x22, 0x01, /* "...S.". */
  0x55, 0x00, 0x20, 0x0a, 0x50, 0xc3, 0x97, 0xc4, /* U. .P... */
  0x9f, 0x81, 0x01, 0x42, 0x01, 0x03, 0x52, 0xfa, /* ...B..R. */
  0x1e, 0x03, 0x53, 0xff, 0x22                  /* ..S." */
};

static const unsigned char pkt29[] = {
  0x42, 0xca, 0x00,
  0xd3, 0x22, 0x01, 0x55, 0x00, /* B...".U. */
  0x20, 0xf2, 0x50, 0x00, 0x2e, 0xd8, 0x8d, 0x03, /*  .P..... */
  0xcf, 0x7f, 0x1c, 0xff, 0xaf, 0x05, 0xef, 0xff, /* ........ */
  0xaf, 0x00, 0xef, 0xff, 0xaf, 0xdd, 0x97, 0xff, /* ........ */
  0xaf, 0xc2, 0x97, 0xff, 0xaf, 0xc3, 0x97, 0xff, /* ........ */
  0xaf, 0xc0, 0x97, 0xff, 0xaf, 0xc1, 0x97, 0xff, /* ........ */
  0xaf, 0x04, 0x0c, 0x00, 0x2e, 0xa7, 0x8a, 0x05, /* ........ */
  0x27, 0x06, 0x1e, 0x01, 0x55, 0x0f, 0x27, 0x0f, /* '...U.'. */
  0x27, 0x0a, 0x1e, 0x08, 0x55, 0x02, 0x2e, 0xf6, /* '...U... */
  0x82, 0xa8, 0x77, 0x3b, 0x30, 0x07, 0x35, 0xff, /* ..w;0.5. */
  0xff, 0xc2, 0xbf, 0x0d, 0x27, 0x0c, 0x27, 0x0d, /* ....'.'. */
  0x1e, 0x02, 0x55, 0x54, 0x85, 0xf5, 0x5b, 0xbd, /* ..UT..[. */
  0x77, 0x12, 0x1e, 0x07, 0x55, 0xdc, 0xef, 0x05, /* w...U... */
  0xff, 0x1a, 0x27, 0x56, 0xdf, 0x3f, 0x20, 0x59, /* ..'V.? Y */
  0x80, 0x3c, 0xaf, 0x05, 0x27, 0x13, 0x1e, 0x01, /* .<..'... */
  0x55, 0x3c, 0x23, 0x05, 0x27, 0x1f, 0x1e, 0x06, /* U<#.'... */
  0x55, 0x37, 0x35, 0x08, 0xef, 0x40, 0x2e, 0xf0, /* U75..@.. */
  0xbf, 0x40, 0x2e, 0xf6, 0xbf, 0x05, 0x27, 0x20, /* .@....'  */
  0x1e, 0x06, 0x55, 0x37, 0x3d, 0x16, 0xef, 0x40, /* ..U7=..@ */
  0x2e, 0xf0, 0xbf, 0x40, 0x2e, 0xf6, 0xbf, 0x05, /* ...@.... */
  0x27, 0x2e, 0x1e, 0x0c, 0x55, 0x02, 0x2e, 0x57, /* '...U..W */
  0x80, 0x41, 0x2e, 0xa1, 0xdf, 0xdf, 0x2a, 0x41, /* .A....*A */
  0x2e, 0xa1, 0xbf, 0x02, 0x2e, 0x57, 0x80, 0x9b, /* .....W.. */
  0x22, 0xdf, 0x2f, 0xd3, 0x50, 0x05, 0x27, 0x2f, /* "./.P.'/ */
  0x1e, 0x0c, 0x55, 0x02, 0x2e, 0x57, 0x80, 0x41, /* ..U..W.A */
  0x2e, 0xa1, 0xdf, 0xdf, 0x2a                  /* ....* */
};

static const unsigned char pkt31[] = {
  0x42, 0x88, 0x00,
  0x37, 0x23, 0x41, 0x2e, 0xa1, /* B..7#A.. */
  0xbf, 0x02, 0x2e, 0x57, 0x80, 0x08, 0x18, 0x41, /* ...W...A */
  0x2e, 0xa1, 0xbf, 0x05, 0x27, 0x30, 0x1e, 0x0e, /* ....'0.. */
  0x55, 0x7e, 0xef, 0xff, 0xaf, 0x02, 0xef, 0xff, /* U~...... */
  0xaf, 0x00, 0xef, 0xff, 0xaf, 0xff, 0xef, 0x1f, /* ........ */
  0xff, 0x1a, 0x27, 0x07, 0x27, 0xff, 0xaf, 0xff, /* ..'.'... */
  0xbf, 0x00, 0x2e, 0xa7, 0x8a, 0x05, 0x27, 0xf7, /* ......'. */
  0x74, 0xc9, 0xcf, 0xf3, 0x1c, 0x04, 0x18, 0xc9, /* t....... */
  0xaf, 0x23, 0x81, 0x10, 0xff, 0xff, 0xef, 0xc1, /* .#...... */
  0x2e, 0x7c, 0xbf, 0x41, 0x2e, 0x7b, 0xbf, 0x02, /* .|.A.{.. */
  0xef, 0x41, 0x2e, 0x7a, 0xbf, 0xc9, 0x31, 0x18, /* .A.z..1. */
  0xef, 0xc0, 0xaf, 0x04, 0x27, 0x41, 0x2e, 0x7a, /* ....'A.z */
  0xdf, 0xdf, 0x2d, 0x08, 0x5b, 0x01, 0xef, 0x41, /* ..-.[..A */
  0x2e, 0x7a, 0xbf, 0xc0, 0x23, 0x18, 0xef, 0xc1, /* .z..#... */
  0xaf, 0xcb, 0xcf, 0xcc, 0xdf, 0x05, 0x27, 0xc9, /* ......'. */
  0x30, 0x01, 0xef, 0x41, 0x2e, 0x7a, 0xbf, 0xc0, /* 0..A.z.. */
  0x23, 0x18, 0xef, 0xc1, 0xaf, 0x9f, 0x22, 0xbf, /* #.....". */
  0x22, 0x05, 0x27                              /* ".' */
};

static const unsigned char pkt33[] = {
  0x42, 0xca, 0x00,
  0x7a, 0x23, 0x9b, 0x22, 0xff, /* B..z#.". */
  0x86, 0x02, 0x2e, 0x96, 0x83, 0xc2, 0x85, 0x00, /* ........ */
  0xef, 0x41, 0x2e, 0x76, 0xbf, 0x56, 0x87, 0x02, /* .A.v.V.. */
  0x2e, 0xed, 0x83, 0x78, 0x86, 0x02, 0x33, 0x07, /* ...x..3. */
  0x80, 0xdd, 0x97, 0xff, 0xaf, 0x00, 0x2e, 0xa7, /* ........ */
  0x8a, 0x05, 0x27, 0x6c, 0xcf, 0x6d, 0xdf, 0xbf, /* ..'l.m.. */
  0x01, 0x6c, 0xaf, 0x6d, 0x26, 0x00, 0x2e, 0x2c, /* .l.m&.., */
  0x55, 0x00, 0x2e, 0x47, 0x50, 0x9f, 0x22, 0x6b, /* U..GP."k */
  0xaf, 0x62, 0xaf, 0x6d, 0x2e, 0x6a, 0xbf, 0x01, /* .b.m.j.. */
  0x2e, 0x3f, 0x84, 0x00, 0xef, 0x41, 0x2e, 0x44, /* .?...A.D */
  0xbf, 0xbf, 0xef, 0x41, 0x2e, 0x45, 0xbf, 0x80, /* ...A.E.. */
  0xef, 0x0a, 0xaf, 0x01, 0x2e, 0x06, 0x82, 0x80, /* ........ */
  0x82, 0x84, 0x82, 0x6b, 0xcf, 0x10, 0xaf, 0x01, /* ...k.... */
  0x2e, 0x94, 0x86, 0x40, 0xef, 0x6d, 0xaf, 0x80, /* ...@.m.. */
  0xef, 0x6c, 0xaf, 0x01, 0x2e, 0xf9, 0x81, 0xed, /* .l...... */
  0x80, 0x01, 0x2e, 0x18, 0x87, 0x01, 0x2e, 0x1a, /* ........ */
  0x87, 0x07, 0x87, 0xff, 0x2f, 0xd1, 0x5a, 0x6c, /* ..../.Zl */
  0xcf, 0x6d, 0xdf, 0x3f, 0x00, 0x6c, 0xaf, 0x6d, /* .m.?.l.m */
  0x26, 0x03, 0x54, 0x6d, 0xdf, 0x0c, 0x82, 0x6d, /* &.Tm...m */
  0xbf, 0xeb, 0x50, 0x6b, 0xcf, 0x08, 0x1e, 0x11, /* ..Pk.... */
  0x5f, 0x6c, 0x24, 0x02, 0x53, 0xff, 0xef, 0x6c, /* _l$.S..l */
  0xaf, 0x10, 0x50, 0x6c, 0x24, 0x02, 0x53, 0xff, /* ..Pl$.S. */
  0xef, 0x6c, 0xaf, 0x0b, 0x50, 0x6b, 0xcf, 0x0a, /* .l..Pk.. */
  0x1e, 0x08, 0x55, 0x6c, 0x24, 0x02, 0x53, 0xff, /* ..Ul$.S. */
  0xef, 0x6c, 0xaf, 0x03, 0x50, 0x6c, 0x24, 0x01, /* .l..Pl$. */
  0x53, 0xff, 0xef, 0x6c, 0xaf                  /* S..l. */
};

static const unsigned char pkt35[] = {
  0x42, 0xca, 0x00,
  0xde, 0x23, 0x6b, 0xcf, 0xdf, /* B...#k.. */
  0x96, 0xb0, 0xef, 0x00, 0xff, 0x1d, 0x00, 0x00, /* ........ */
  0x13, 0x1b, 0x27, 0x6c, 0xcf, 0x9f, 0x9d, 0x6b, /* ..'l...k */
  0xcf, 0x1f, 0x20, 0x6b, 0xaf, 0x0a, 0x1e, 0xbc, /* .. k.... */
  0x5f, 0x05, 0x27, 0x01, 0x2e, 0x3f, 0x84, 0x55, /* _.'..?.U */
  0x82, 0x01, 0x2e, 0x06, 0x82, 0x73, 0x23, 0x0a, /* .....s#. */
  0xef, 0x10, 0xaf, 0x01, 0x2e, 0x94, 0x86, 0x80, /* ........ */
  0xef, 0x01, 0x2e, 0xf9, 0x81, 0xed, 0x80, 0x08, /* ........ */
  0xef, 0x06, 0xaf, 0x01, 0x2e, 0x18, 0x87, 0x01, /* ........ */
  0x2e, 0x1a, 0x87, 0x06, 0x25, 0xf9, 0x55, 0x02, /* ....%.U. */
  0x2e, 0x52, 0x84, 0x4e, 0x23, 0x40, 0xcf, 0xe0, /* .R.N#@.. */
  0xaf, 0x41, 0xdf, 0xe1, 0xbf, 0x41, 0x2e, 0x4e, /* .A...A.N */
  0xdf, 0xf7, 0x1c, 0x41, 0x2e, 0x4e, 0xbf, 0x01, /* ...A.N.. */
  0x2e, 0x18, 0x87, 0x01, 0x2e, 0x1a, 0x87, 0x45, /* .......E */
  0x82, 0x4e, 0x24, 0x4e, 0xcf, 0x20, 0x1e, 0x03, /* .N$N. .. */
  0x55, 0x01, 0xef, 0x1f, 0x90, 0x02, 0x2e, 0x87, /* U....... */
  0x73, 0x40, 0x2e, 0x8f, 0xdf, 0xdf, 0x22, 0xe5, /* s@....". */
  0x55, 0x4e, 0x86, 0x4e, 0x23, 0x4e, 0xcf, 0x52, /* UN.N#N.R */
  0x86, 0x59, 0x86, 0x3f, 0x9f, 0x4e, 0xcf, 0x52, /* .Y.?.N.R */
  0x86, 0x5f, 0x86, 0x1f, 0x9f, 0x4e, 0x24, 0x4e, /* ._...N$N */
  0xcf, 0x04, 0x1e, 0xf3, 0x55, 0xf8, 0x0f, 0x08, /* ....U... */
  0xef, 0x06, 0xaf, 0xf6, 0x97, 0xbc, 0x03, 0x01, /* ........ */
  0x5f, 0xff, 0x0f, 0xfc, 0x97, 0x06, 0x25, 0xf9, /* _.....%. */
  0x55, 0xf8, 0x0f, 0x08, 0xff, 0x06, 0xbf, 0xf6, /* U....... */
  0x9f, 0xdf, 0x01, 0xa1, 0x27, 0x7f, 0x21, 0x3f, /* ....'.!? */
  0x9f, 0x06, 0x25, 0xf8, 0x55                  /* ..%.U */
};

static const unsigned char pkt37[] = {
  0x42, 0xb2, 0x00,
  0x42, 0x24, 0xf8, 0x0f, 0x04, /* B..B$... */
  0xef, 0x06, 0xaf, 0xbb, 0xef, 0x00, 0xff, 0x1a, /* ........ */
  0x27, 0xfc, 0x9f, 0xfc, 0x97, 0x09, 0x27, 0x5f, /* '.....'_ */
  0x02, 0x3f, 0x97, 0x06, 0x25, 0xf8, 0x55, 0x81, /* .?..%.U. */
  0xcf, 0x24, 0xaf, 0x05, 0x27, 0x0a, 0xef, 0x73, /* .$..'..s */
  0x43, 0x02, 0x5b, 0x73, 0xcf, 0x09, 0x27, 0x0f, /* C.[s..'. */
  0x1c, 0x81, 0xaf, 0x73, 0xcf, 0x09, 0x27, 0x30, /* ...s..'0 */
  0x1c, 0xfc, 0xdf, 0xcf, 0x1d, 0x5f, 0x02, 0xfc, /* ....._.. */
  0xbf, 0x81, 0xcf, 0x10, 0xaf, 0x01, 0x2e, 0x94, /* ........ */
  0x86, 0x80, 0x82, 0x01, 0x2e, 0x34, 0x81, 0x40, /* .....4.@ */
  0xef, 0x6b, 0xaf, 0x80, 0xef, 0x82, 0xaf, 0x01, /* .k...... */
  0x2e, 0xf9, 0x81, 0xed, 0x80, 0x01, 0x2e, 0x18, /* ........ */
  0x87, 0x01, 0x2e, 0x1a, 0x87, 0x16, 0x82, 0x01, /* ........ */
  0x2e, 0x2f, 0x89, 0x0a, 0x5b, 0x73, 0x43, 0x19, /* ./..[sC. */
  0x5a, 0x81, 0xcf, 0x5e, 0xdf, 0xbf, 0x03, 0x15, /* Z..^.... */
  0x5f, 0x5f, 0x20, 0x81, 0xaf, 0xdf, 0x22, 0x11, /* __ ...". */
  0x54, 0xde, 0x50, 0x40, 0x2e, 0x8f, 0xdf, 0x1f, /* T.P@.... */
  0x2c, 0x00, 0x2e, 0x0c, 0x5a, 0x5f, 0x2c, 0x09, /* ,...Z_,. */
  0x5b, 0x82, 0xcf, 0x6b, 0xdf, 0x3f, 0x00, 0x82, /* [..k.?.. */
  0xaf, 0x6b, 0x26, 0xe7, 0x54, 0x6b, 0xdf, 0x0c, /* .k&.Tk.. */
  0x82, 0x6b, 0xbf, 0xd8, 0x50, 0x05, 0x27, 0x82, /* .k..P.'. */
  0xcf, 0x6b, 0xdf, 0xbf, 0x01, 0x82, 0xaf, 0x6b, /* .k.....k */
  0x26, 0xf4, 0x55, 0xdb, 0x50                  /* &.U.P */
};

static const unsigned char pkt39[] = {
  0x42, 0xca, 0x00,
  0x9a, 0x24, 0xfa, 0xcf, 0xdf, /* B...$... */
  0x96, 0x37, 0xcf, 0xdf, 0x96, 0x08, 0xcf, 0xdf, /* .7...... */
  0x96, 0x3d, 0xcf, 0xdf, 0x96, 0x77, 0xcf, 0xdf, /* .=...w.. */
  0x96, 0x01, 0x2e, 0x08, 0x82, 0xdd, 0x97, 0x77, /* .......w */
  0xaf, 0xdd, 0x97, 0x3d, 0xaf, 0xdd, 0x97, 0x08, /* ...=.... */
  0xaf, 0xdd, 0x97, 0x37, 0xaf, 0xdd, 0x97, 0xfa, /* ...7.... */
  0xaf, 0x9f, 0x22, 0x6d, 0x2e, 0x6d, 0xbf, 0xfb, /* .."m.m.. */
  0x39, 0x08, 0x45, 0x00, 0x5a, 0xfb, 0x31, 0x0a, /* 9.E.Z.1. */
  0x42, 0x00, 0x5b, 0xa2, 0x39, 0x02, 0xcf, 0x5c, /* B.[.9..\ */
  0xaf, 0xe1, 0xef, 0x95, 0xaf, 0x9f, 0x22, 0x61, /* ......"a */
  0xaf, 0x62, 0xaf, 0x69, 0xaf, 0x6a, 0xaf, 0x16, /* .b.i.j.. */
  0xaf, 0x96, 0xaf, 0x29, 0xaf, 0x28, 0xaf, 0xc1, /* ...).(.. */
  0x2e, 0x49, 0xdf, 0x3f, 0x2d, 0x03, 0x5b, 0xa0, /* .I.?-.[. */
  0xcf, 0x8f, 0x1c, 0x20, 0x18, 0xa0, 0xaf, 0xa1, /* ... .... */
  0xcf, 0xdf, 0x1c, 0xa1, 0xaf, 0xed, 0x80, 0x02, /* ........ */
  0x47, 0x06, 0x5a, 0x0a, 0x41, 0x04, 0x5a, 0x0a, /* G.Z.A.Z. */
  0xdf, 0x30, 0x1d, 0x30, 0x1f, 0x00, 0x54, 0x01, /* .0.0..T. */
  0xef, 0x44, 0xaf, 0x09, 0xef, 0x60, 0xaf, 0xa7, /* .D...`.. */
  0xef, 0x40, 0xaf, 0x2d, 0xef, 0x41, 0xaf, 0x0a, /* .@.-.A.. */
  0x40, 0x06, 0x5a, 0x10, 0xcf, 0x27, 0xaf, 0xe2, /* @.Z..'.. */
  0x23, 0x6e, 0x23, 0x08, 0xef, 0xe3, 0xaf, 0x6f, /* #n#....o */
  0xaf, 0xe2, 0x23, 0x08, 0xef, 0xe3, 0xaf, 0x79, /* ..#....y */
  0x82, 0x4d, 0xaf, 0xdf, 0x96, 0x64, 0xdf, 0x8b, /* .M...d.. */
  0x82, 0x93, 0xbf, 0xdd, 0x97, 0x77, 0xff, 0x8b, /* .....w.. */
  0x82, 0x94, 0xbf, 0x9e, 0x23, 0xe4, 0x23, 0x9f, /* ....#.#. */
  0x22, 0x6d, 0x2e, 0x6b, 0xbf                  /* "m.k. */
};

static const unsigned char pkt41[] = {
  0x42, 0x12, 0x00,
  0xfe, 0x24, 0x6d, 0x2e, 0x6a, /* B...$m.j */
  0xbf, 0x6d, 0x2e, 0x6c, 0xbf, 0xa5, 0xcf, 0xf3, /* .m.l.... */
  0x1c, 0xa5, 0xaf, 0x05, 0x27                  /* ....' */
};

static const unsigned char pkt43[] = {
  0x42, 0xca, 0x00,
  0x06, 0x25, 0xfe, 0xcf, 0x58, /* B...%..X */
  0xaf, 0xfe, 0xcf, 0xfe, 0xdf, 0x02, 0x2e, 0x9d, /* ........ */
  0x85, 0xfe, 0xcf, 0x3c, 0xaf, 0x07, 0x3d, 0x58, /* ...<..=X */
  0x40, 0x4e, 0x5a, 0x3c, 0x26, 0x4c, 0x55, 0x37, /* @NZ<&LU7 */
  0x45, 0x4a, 0x5a, 0x02, 0x2e, 0x1b, 0x86, 0x47, /* EJZ....G */
  0x5a, 0x02, 0x2e, 0xad, 0x85, 0x00, 0x2e, 0xb9, /* Z....... */
  0x8d, 0xfe, 0xef, 0x16, 0x80, 0x02, 0xcf, 0xdf, /* ........ */
  0x96, 0x61, 0xef, 0xc0, 0xaf, 0x04, 0x27, 0xc1, /* .a....'. */
  0x46, 0x04, 0x5b, 0xc0, 0x3e, 0x40, 0xef, 0xc1, /* F.[.>@.. */
  0xaf, 0x01, 0x2e, 0x8d, 0x8b, 0xd0, 0x41, 0x07, /* ......A. */
  0x5b, 0x02, 0x2e, 0x5d, 0x85, 0xf2, 0x5b, 0xae, /* [..]..[. */
  0x87, 0xdd, 0x97, 0x1f, 0x2d, 0x06, 0x5b, 0x03, /* ....-.[. */
  0x50, 0x01, 0x0c, 0x02, 0xcf, 0xc0, 0x1c, 0x01, /* P....... */
  0x55, 0x02, 0x2e, 0x40, 0x85, 0xc0, 0x23, 0x20, /* U..@..#  */
  0xef, 0xc1, 0xaf, 0x00, 0x2e, 0xd8, 0x8d, 0x05, /* ........ */
  0x27, 0x9b, 0x22, 0x37, 0x45, 0x05, 0x5a, 0x41, /* '."7E.ZA */
  0x2e, 0x52, 0xdf, 0xdf, 0x2d, 0x04, 0x5b, 0x00, /* .R..-.[. */
  0x28, 0x02, 0x50, 0x3b, 0x43, 0x00, 0x5b, 0x00, /* (.P;C.[. */
  0x28, 0xd3, 0xdf, 0xc6, 0x2e, 0x15, 0xbf, 0xd3, /* (....... */
  0x3c, 0x40, 0xef, 0x09, 0x80, 0xdd, 0x97, 0xff, /* <@...... */
  0xaf, 0x00, 0x2e, 0xa7, 0x8a, 0xc6, 0x2e, 0x15, /* ........ */
  0xdf, 0x10, 0x1d, 0xd3, 0xcf, 0x3f, 0x02, 0xd3, /* .....?.. */
  0xaf, 0x05, 0x27, 0xed, 0x83, 0x07, 0x45, 0x05, /* ..'...E. */
  0x27, 0x3d, 0x23, 0x07, 0x3d, 0x02, 0xcf, 0xdf, /* '=#.=... */
  0x96, 0x02, 0x2e, 0xad, 0x85, 0x02, 0x2e, 0x96, /* ........ */
  0x85, 0xfe, 0xef, 0x16, 0x80                  /* ..... */
};

static const unsigned char pkt45[] = {
  0x42, 0xca, 0x00,
  0x6a, 0x25, 0x05, 0xef, 0xc0, /* B..j%... */
  0xaf, 0x04, 0x27, 0xd0, 0x41, 0x03, 0x5b, 0x02, /* ..'.A.[. */
  0x2e, 0x5d, 0x85, 0xf9, 0x5b, 0xbb, 0x50, 0x27, /* .]..[.P' */
  0x82, 0x04, 0xef, 0xc1, 0xaf, 0x9b, 0x22, 0x02, /* ......". */
  0x2e, 0xd1, 0x85, 0xdd, 0x97, 0xdf, 0x22, 0x02, /* ......". */
  0x55, 0x58, 0x40, 0x02, 0x5a, 0x0b, 0x50, 0x58, /* UX@.Z.PX */
  0x40, 0x09, 0x5a, 0xdd, 0x97, 0x1f, 0x2d, 0xb3, /* @.Z...-. */
  0x5a, 0x02, 0x2e, 0x1c, 0x86, 0xb2, 0x5b, 0x9b, /* Z.....[. */
  0x22, 0x02, 0x2e, 0x4c, 0x85, 0xae, 0x50, 0x02, /* "..L..P. */
  0x2e, 0x1c, 0x86, 0xd5, 0x5b, 0x02, 0x2e, 0x40, /* ....[..@ */
  0x85, 0x58, 0x41, 0x02, 0x5b, 0x01, 0x0c, 0x02, /* .XA.[... */
  0x2e, 0x3a, 0x75, 0xcd, 0x50, 0x41, 0x2e, 0x52, /* .:u.PA.R */
  0xdf, 0xed, 0x1c, 0x04, 0x18, 0x41, 0x2e, 0x52, /* .....A.R */
  0xbf, 0x05, 0x27, 0x1a, 0x27, 0x56, 0xdf, 0x59, /* ..'.'V.Y */
  0x80, 0xdf, 0x22, 0x00, 0x54, 0x01, 0x16, 0x3f, /* ..".T..? */
  0x1e, 0x00, 0x5f, 0x3f, 0xef, 0x1f, 0x22, 0x1f, /* .._?..". */
  0x22, 0xaf, 0xdf, 0x03, 0x1d, 0x5f, 0x02, 0xaf, /* "...._.. */
  0xbf, 0x05, 0x27, 0x00, 0x2e, 0xa1, 0x8d, 0xae, /* ..'..... */
  0xdf, 0xe0, 0x19, 0xae, 0xbf, 0x57, 0x40, 0x00, /* .....W@. */
  0x5b, 0xa1, 0x32, 0x57, 0x43, 0x00, 0x5b, 0xae, /* [.2WC.[. */
  0x3f, 0xed, 0x80, 0x05, 0x27, 0x41, 0x2e, 0x52, /* ?...'A.R */
  0xdf, 0x03, 0x18, 0x41, 0x2e, 0x52, 0xbf, 0x05, /* ...A.R.. */
  0x27, 0xd3, 0xdf, 0xff, 0x96, 0x9b, 0x22, 0x02, /* '.....". */
  0x2e, 0xd1, 0x85, 0xd3, 0x3c, 0x07, 0x80, 0xdd, /* ....<... */
  0x97, 0xff, 0xaf, 0x00, 0x2e, 0xa7, 0x8a, 0xdd, /* ........ */
  0x9f, 0x10, 0x1d, 0xd3, 0xcf                  /* ..... */
};

static const unsigned char pkt47[] = {
  0x42, 0xb4, 0x00,
  0xce, 0x25, 0x3f, 0x02, 0xd3, /* B...%?.. */
  0xaf, 0x05, 0x27, 0x37, 0x45, 0x10, 0x5a, 0x02, /* ..'7E.Z. */
  0x2e, 0x1b, 0x86, 0x0d, 0x5a, 0x02, 0x2e, 0xad, /* ....Z... */
  0x85, 0x00, 0x2e, 0xbf, 0x8d, 0x00, 0x2e, 0xd3, /* ........ */
  0x8d, 0x41, 0x2e, 0x52, 0xdf, 0xdf, 0x2d, 0x00, /* .A.R..-. */
  0x5b, 0x00, 0x28, 0x00, 0x2e, 0xd8, 0x8d, 0x05, /* [.(..... */
  0x27, 0xf9, 0xcf, 0xdf, 0x96, 0xf9, 0x3d, 0x01, /* '.....=. */
  0x2e, 0x06, 0x82, 0x2d, 0xff, 0xa7, 0xef, 0xe1, /* ...-.... */
  0xbf, 0xe0, 0xaf, 0x09, 0xef, 0x10, 0xaf, 0x01, /* ........ */
  0x2e, 0x94, 0x86, 0xed, 0x80, 0x01, 0x2e, 0x18, /* ........ */
  0x87, 0x01, 0x2e, 0x1a, 0x87, 0x00, 0x2e, 0x16, /* ........ */
  0x82, 0x5b, 0x82, 0x00, 0x2e, 0x2d, 0x82, 0x26, /* .[...-.& */
  0x37, 0x00, 0x5a, 0x0c, 0x50, 0xff, 0x96, 0xdf, /* 7.Z.P... */
  0x96, 0x79, 0xcf, 0x7a, 0xdf, 0xff, 0x96, 0xdf, /* .y.z.... */
  0x96, 0xc2, 0x97, 0xc3, 0x9f, 0x9d, 0x01, 0x5d, /* .......] */
  0x01, 0x07, 0x53, 0xdd, 0x97, 0xdd, 0x9f, 0x79, /* ..S....y */
  0xaf, 0x7a, 0xbf, 0x3b, 0x3b, 0xdd, 0x97, 0xf9, /* .z.;;... */
  0xaf, 0x05, 0x27, 0xff, 0x22, 0x01, 0x55, 0x32, /* ..'.".U2 */
  0x1e, 0xf3, 0x5f, 0x02, 0x0c, 0x02, 0x2e, 0x0e, /* .._..... */
  0x86, 0x3b, 0x33, 0x00, 0x28, 0x05, 0x27, 0x0d, /* .;3.(.'. */
  0x27, 0x3c, 0x26, 0x07, 0x54, 0x3c, 0xcf, 0x3d, /* '<&.T<.= */
  0xdf, 0x3f, 0x20, 0x3d, 0xbf, 0xdf, 0x03, 0x01, /* .? =.... */
  0x52, 0x3d, 0x23, 0x0c, 0x27, 0x0d, 0x27      /* R=#.'.' */
};

static const unsigned char pkt49[] = {
  0x42, 0x1a, 0x00,
  0x27, 0x26, 0x09, 0x43, 0x04, /* B..'&.C. */
  0x5b, 0x00, 0xef, 0xf8, 0xaf, 0x32, 0xef, 0xfa, /* [....2.. */
  0xaf, 0x03, 0x50, 0x01, 0xef, 0xf8, 0xaf, 0x42, /* ..P....B */
  0xef, 0xfa, 0xaf, 0x05, 0x27                  /* ....' */
};

static const unsigned char pkt51[] = {
  0x42, 0x26, 0x00,
  0x33, 0x26, 0xd3, 0x45, 0xfd, /* B&.3&.E. */
  0x5a, 0x0f, 0x27, 0xd3, 0x45, 0xfa, 0x5a, 0x0f, /* Z.'.E.Z. */
  0x27, 0xd3, 0x45, 0xf7, 0x5a, 0x0f, 0x27, 0x41, /* '.E.Z.'A */
  0x2e, 0xa1, 0xdf, 0x9f, 0x28, 0x41, 0x2e, 0xa1, /* ....(A.. */
  0xbf, 0x02, 0x2e, 0x57, 0x80, 0x00, 0x2e, 0xcb, /* ...W.... */
  0x7a                                          /* z */
};

static const unsigned char pkt53[] = {
  0x42, 0x4e, 0x00,
  0x45, 0x26, 0x07, 0x3d, 0x02, /* BN.E&.=. */
  0x5b, 0xae, 0x87, 0xc2, 0x23, 0xd3, 0x3c, 0x02, /* [...#.<. */
  0x44, 0xdc, 0x2f, 0x43, 0x5a, 0x02, 0x45, 0xdc, /* D./CZ.E. */
  0x2f, 0xa4, 0x5b, 0x58, 0x23, 0x3c, 0x26, 0x02, /* /.[X#<&. */
  0x55, 0x02, 0x2e, 0x0e, 0x85, 0x00, 0x50, 0x09, /* U.....P. */
  0x8f, 0x07, 0x45, 0xeb, 0x5a, 0x41, 0x2e, 0x53, /* ..E.ZA.S */
  0xdf, 0xc1, 0x2e, 0x79, 0xdf, 0xff, 0x96, 0xdf, /* ...y.... */
  0x96, 0x41, 0x2e, 0x55, 0xdf, 0xc1, 0x2e, 0x78, /* .A.U...x */
  0xdf, 0x9d, 0x01, 0x5d, 0x01, 0xff, 0x22, 0x00, /* ...]..". */
  0x54, 0xff, 0xef, 0x2e, 0xaf, 0xdc, 0x2f, 0x89, /* T...../. */
  0x50                                          /* P */
};

static const unsigned char pkt55[] = {
  0x42, 0x86, 0x00,
  0x6b, 0x26, 0xa5, 0xcf, 0xed, /* B..k&... */
  0x2e, 0x6a, 0xdf, 0xf3, 0x1c, 0x3f, 0x02, 0xa5, /* .j...?.. */
  0xaf, 0x4a, 0xcf, 0xdf, 0x96, 0x10, 0xcf, 0x5f, /* .J....._ */
  0xdf, 0xbf, 0x03, 0x02, 0x5f, 0x5f, 0xcf, 0x10, /* ....__.. */
  0xaf, 0x04, 0x50, 0x5e, 0xdf, 0xbf, 0x03, 0x01, /* ..P^.... */
  0x53, 0x5e, 0xcf, 0x10, 0xaf, 0x01, 0x2e, 0x40, /* S^.....@ */
  0x88, 0x01, 0x2e, 0xe1, 0x86, 0xdd, 0x97, 0x0a, /* ........ */
  0x40, 0x18, 0x5a, 0x62, 0xcf, 0xdf, 0x22, 0x22, /* @.Zb.."" */
  0x54, 0xed, 0x2e, 0x6a, 0xdf, 0x10, 0xef, 0x00, /* T..j.... */
  0x1f, 0x06, 0x54, 0x0c, 0xef, 0x04, 0x1f, 0x03, /* ..T..... */
  0x54, 0x0b, 0xef, 0x08, 0x1f, 0x00, 0x54, 0x05, /* T.....T. */
  0xef, 0x62, 0xdf, 0x76, 0x80, 0x12, 0x82, 0x12, /* .b.v.... */
  0x82, 0x80, 0x10, 0x01, 0x13, 0x4a, 0xcf, 0x3f, /* .....J.? */
  0x00, 0x00, 0x53, 0xff, 0xef, 0x4a, 0xdf, 0xed, /* ..S..J.. */
  0x2e, 0x6e, 0xbf, 0x4a, 0xaf, 0x6d, 0x2e, 0x6f, /* .n.J.m.o */
  0xbf, 0xdf, 0x96, 0xbf, 0x01, 0x6d, 0x2e, 0x6d, /* .....m.m */
  0xbf, 0xdd, 0x97, 0x01, 0x2e, 0xf9, 0x81, 0x05, /* ........ */
  0x27                                          /* ' */
};

static const unsigned char pkt57[] = {
  0x42, 0x16, 0x00,
  0xad, 0x26, 0xd3, 0x45, 0x02, /* B...&.E. */
  0x5a, 0xd3, 0x45, 0x00, 0x5a, 0x05, 0x27, 0xd0, /* Z.E.Z.'. */
  0x41, 0xf8, 0x5b, 0xed, 0x83, 0xf6, 0x50, 0x05, /* A.[...P. */
  0x27                                          /* ' */
};

static const unsigned char pkt59[] = {
  0x42, 0xca, 0x00,
  0xb7, 0x26, 0xfd, 0x0c, 0x00, /* B...&... */
  0x2e, 0x3f, 0x8a, 0x39, 0x84, 0x05, 0xcf, 0xdf, /* .?.9.... */
  0x22, 0x00, 0x2e, 0xb4, 0x55, 0x04, 0xcf, 0x14, /* "...U... */
  0x1e, 0x00, 0x2e, 0xb0, 0x55, 0x00, 0x2e, 0xf5, /* ....U... */
  0x89, 0xbb, 0xef, 0x1f, 0x90, 0x91, 0xff, 0x7f, /* ........ */
  0x90, 0x14, 0xef, 0x00, 0x2e, 0x31, 0x8a, 0x14, /* .....1.. */
  0xef, 0x00, 0x2e, 0x39, 0x8a, 0x33, 0xef, 0x1f, /* ...9.3.. */
  0x90, 0x71, 0xff, 0x7f, 0x90, 0x18, 0xef, 0x00, /* .q...... */
  0x2e, 0x31, 0x8a, 0x00, 0x2e, 0xf5, 0x89, 0x40, /* .1.....@ */
  0xef, 0xfc, 0x9f, 0x5f, 0x20, 0xfc, 0x55, 0x01, /* ..._ .U. */
  0x2e, 0x67, 0x8d, 0x01, 0x2e, 0xfe, 0x8c, 0x00, /* .g...... */
  0x2e, 0xf5, 0x89, 0x02, 0x2e, 0x7a, 0x87, 0x0a, /* .....z.. */
  0xef, 0x00, 0x2e, 0x01, 0x8a, 0x02, 0x2e, 0x7a, /* .......z */
  0x87, 0x00, 0xff, 0x80, 0xef, 0x19, 0x27, 0x0a, /* ......'. */
  0xef, 0xdc, 0x9f, 0x7f, 0x9f, 0x5f, 0x20, 0xfb, /* ....._ . */
  0x55, 0x39, 0x84, 0x02, 0x2e, 0x7a, 0x87, 0x0a, /* U9...z.. */
  0xef, 0x00, 0x2e, 0x01, 0x8a, 0x02, 0x2e, 0x7a, /* .......z */
  0x87, 0x0a, 0xef, 0xdc, 0x9f, 0x7f, 0x9f, 0x5f, /* ......._ */
  0x20, 0xfb, 0x55, 0x02, 0x2e, 0x76, 0x87, 0x02, /*  .U..v.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, /* .D...?.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, /* .D...?.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, /* .D...?.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, /* .D...?.. */
  0x2e, 0x44, 0x88, 0x9d, 0xef, 0x1f, 0x90, 0xda, /* .D...... */
  0xff, 0x7f, 0x90, 0x0f, 0xef, 0x00, 0x2e, 0x31, /* .......1 */
  0x8a, 0x80, 0xef, 0x1f, 0x9f                  /* ..... */
};

static const unsigned char pkt61[] = {
  0x42, 0xca, 0x00,
  0x1b, 0x27, 0x9f, 0x22, 0x1f, /* B...'.". */
  0x9f, 0x1f, 0x9f, 0x1f, 0x9f, 0x1f, 0x9f, 0x1f, /* ........ */
  0x9f, 0x1f, 0x9f, 0x03, 0xef, 0x1f, 0x9f, 0xb8, /* ........ */
  0xef, 0x1f, 0x9f, 0x00, 0x2e, 0xf5, 0x89, 0x40, /* .......@ */
  0xef, 0xfc, 0x9f, 0x5f, 0x20, 0xfc, 0x55, 0x01, /* ..._ .U. */
  0x2e, 0xfe, 0x8c, 0x00, 0x2e, 0xff, 0x89, 0x02, /* ........ */
  0x2e, 0xb3, 0x87, 0x02, 0x2e, 0x44, 0x88, 0x02, /* .....D.. */
  0x2e, 0x3f, 0x88, 0x02, 0x2e, 0x44, 0x88, 0x02, /* .?...D.. */
  0x2e, 0x3f, 0x88, 0x02, 0x2e, 0x44, 0x88, 0x02, /* .?...D.. */
  0x2e, 0x3f, 0x88, 0x02, 0x2e, 0x44, 0x88, 0x02, /* .?...D.. */
  0x2e, 0x3f, 0x88, 0x02, 0x2e, 0x44, 0x88, 0x02, /* .?...D.. */
  0x2e, 0xb3, 0x87, 0x14, 0xef, 0xfc, 0x9f, 0x5f, /* ......._ */
  0x20, 0xfc, 0x55, 0x01, 0xff, 0xab, 0xef, 0x19, /*  .U..... */
  0x27, 0x02, 0x2e, 0xb3, 0x87, 0x11, 0xff, 0xbf, /* '....... */
  0x90, 0x51, 0xff, 0xc2, 0x97, 0xdf, 0x01, 0x7f, /* .Q...... */
  0x98, 0xfc, 0x9f, 0xbf, 0x98, 0x24, 0x28, 0xc2, /* .....$(. */
  0x9f, 0x7f, 0x20, 0xbf, 0x90, 0xf4, 0x55, 0x24, /* .. ...U$ */
  0x2a, 0x02, 0x2e, 0x7e, 0x87, 0x72, 0xef, 0xff, /* *..~.r.. */
  0xaf, 0x14, 0xef, 0xff, 0xaf, 0x00, 0xef, 0xff, /* ........ */
  0xaf, 0x00, 0xff, 0x80, 0xef, 0x19, 0x27, 0x14, /* ......'. */
  0xff, 0xfd, 0x97, 0xff, 0xaf, 0x7f, 0x20, 0xfb, /* ...... . */
  0x55, 0x00, 0x2e, 0xa9, 0x8a, 0x03, 0x0c, 0xd0, /* U....... */
  0xef, 0x07, 0xff, 0xa9, 0x80, 0x05, 0x27, 0x58, /* ......'X */
  0x85, 0x05, 0x27, 0x40, 0xef, 0x2d, 0xff, 0x1a, /* ..'@.-.. */
  0x27, 0x05, 0x27, 0x02, 0xef, 0x06, 0xff, 0x1a, /* '.'..... */
  0x27, 0x05, 0x27, 0x01, 0xff                  /* '.'.. */
};

static const unsigned char pkt63[] = {
  0x42, 0x72, 0x00,
  0x7f, 0x27, 0xab, 0xef, 0x19, /* Br..'... */
  0x27, 0x08, 0xef, 0x1f, 0x98, 0xe4, 0x29, 0x00, /* '.....). */
  0xff, 0xff, 0x96, 0x20, 0xff, 0xff, 0x96, 0x00, /* ... .... */
  0xff, 0xff, 0x96, 0x40, 0xff, 0xff, 0x96, 0x00, /* ...@.... */
  0xff, 0xff, 0x96, 0x60, 0xff, 0xff, 0x96, 0x00, /* ...`.... */
  0xff, 0xff, 0x96, 0x00, 0xff, 0xff, 0x96, 0x01, /* ........ */
  0xff, 0xff, 0x96, 0x20, 0xff, 0xff, 0x96, 0x00, /* ... .... */
  0xff, 0xff, 0x96, 0x02, 0xff, 0xff, 0x96, 0x02, /* ........ */
  0xff, 0xff, 0x96, 0x82, 0xff, 0xff, 0x96, 0x02, /* ........ */
  0xff, 0xff, 0x96, 0x02, 0x2e, 0xaa, 0x87, 0x46, /* .......F */
  0x2e, 0x01, 0xdf, 0x9f, 0x28, 0x46, 0x2e, 0x01, /* ....(F.. */
  0xbf, 0x05, 0x27, 0x0f, 0xef, 0x5f, 0x98, 0xdd, /* ..'.._.. */
  0x9f, 0xbf, 0x98, 0x24, 0x28, 0x5f, 0x20, 0xf9, /* ...$(_ . */
  0x55, 0x24, 0x2a, 0x05, 0x27, 0x7f, 0xef, 0x2d, /* U$*.'..- */
  0xff, 0x1b, 0x27, 0x05, 0x27                  /* ..'.' */
};

static const unsigned char pkt65[] = {
  0x42, 0xca, 0x00,
  0xb7, 0x27, 0x00, 0x2e, 0x3f, /* B...'..? */
  0x8a, 0x23, 0x81, 0x02, 0x2e, 0x4d, 0x88, 0x05, /* .#...M.. */
  0xcf, 0xdf, 0x22, 0x35, 0x55, 0x04, 0xcf, 0x2a, /* .."5U..* */
  0x1e, 0x32, 0x55, 0xfe, 0xdf, 0xfe, 0xcf, 0x00, /* .2U..... */
  0x1f, 0x1d, 0x55, 0x00, 0x1e, 0x0e, 0x54, 0xc6, /* ..U...T. */
  0x2e, 0x00, 0xdf, 0x3f, 0x20, 0xbf, 0x03, 0x16, /* ...? ... */
  0x55, 0xc6, 0x2e, 0x01, 0xdf, 0x3f, 0x2c, 0x12, /* U....?,. */
  0x5b, 0x46, 0x2e, 0x00, 0xbf, 0x00, 0x2e, 0x25, /* [F.....% */
  0x8a, 0x02, 0x2e, 0xe1, 0x77, 0x00, 0xef, 0x46, /* ....w..F */
  0x2e, 0x00, 0xbf, 0x46, 0x2e, 0x01, 0xdf, 0x1f, /* ...F.... */
  0x28, 0x46, 0x2e, 0x01, 0xbf, 0x00, 0x2e, 0x06, /* (F...... */
  0x8a, 0x02, 0x2e, 0xf7, 0x87, 0x05, 0x27, 0x46, /* ......'F */
  0x2e, 0x01, 0xdf, 0x1f, 0x2a, 0x46, 0x2e, 0x01, /* ....*F.. */
  0xbf, 0x41, 0x2e, 0xb6, 0xdf, 0x01, 0xff, 0xab, /* .A...... */
  0xef, 0x19, 0x27, 0x42, 0xff, 0x7f, 0x98, 0x9f, /* ..'B.... */
  0x98, 0x24, 0x28, 0x56, 0x85, 0x24, 0x2a, 0x05, /* .$(V.$*. */
  0x27, 0x58, 0x85, 0x05, 0x27, 0xfd, 0x0c, 0x00, /* 'X..'... */
  0x2e, 0xf5, 0x89, 0x14, 0xef, 0x00, 0x2e, 0x39, /* .......9 */
  0x8a, 0x02, 0x2e, 0x3b, 0x88, 0x02, 0x2e, 0x44, /* ...;...D */
  0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, 0x2e, 0x44, /* ...?...D */
  0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, 0x2e, 0x44, /* ...?...D */
  0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, 0x2e, 0x44, /* ...?...D */
  0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, 0x2e, 0x44, /* ...?...D */
  0x88, 0x46, 0x2e, 0x00, 0xdf, 0x1f, 0x9f, 0xe2, /* .F...... */
  0xef, 0x1f, 0x90, 0xe0, 0xff, 0x7f, 0x90, 0x0e, /* ........ */
  0xef, 0x00, 0x2e, 0x31, 0x8a                  /* ...1. */
};

static const unsigned char pkt67[] = {
  0x42, 0xca, 0x00,
  0x1b, 0x28, 0x80, 0xef, 0x1f, /* B...(... */
  0x9f, 0x9f, 0x22, 0x1f, 0x9f, 0x1f, 0x9f, 0x1f, /* .."..... */
  0x9f, 0x1f, 0x9f, 0x1f, 0x9f, 0x1f, 0x9f, 0x01, /* ........ */
  0xef, 0x1f, 0x9f, 0xb8, 0xef, 0x1f, 0x9f, 0x00, /* ........ */
  0x2e, 0xf5, 0x89, 0x40, 0xef, 0xfc, 0x9f, 0x5f, /* ...@..._ */
  0x20, 0xfc, 0x55, 0x03, 0x0c, 0x01, 0x2e, 0x67, /*  .U....g */
  0x8d, 0x01, 0x2e, 0xfe, 0x8c, 0x00, 0x2e, 0xff, /* ........ */
  0x89, 0x00, 0x2e, 0xfb, 0x89, 0x14, 0xef, 0x00, /* ........ */
  0x2e, 0x01, 0x8a, 0x05, 0x27, 0x40, 0xef, 0x2d, /* ....'@.- */
  0xff, 0x1a, 0x27, 0x05, 0x27, 0x16, 0x27, 0x04, /* ..'.'.'. */
  0x10, 0x00, 0x13, 0x1a, 0x27, 0x05, 0x27, 0xd9, /* ....'.'. */
  0x9f, 0x3f, 0x9f, 0xd8, 0x9f, 0x3f, 0x9f, 0xd7, /* .?...?.. */
  0x9f, 0x3f, 0x9f, 0xd6, 0x9f, 0x3f, 0x9f, 0x05, /* .?...?.. */
  0x27, 0x7f, 0xef, 0x2d, 0xff, 0x46, 0x2e, 0x1d, /* '..-.F.. */
  0xbf, 0xc6, 0x2e, 0x1c, 0xbf, 0x93, 0xef, 0x2d, /* .......- */
  0xff, 0x46, 0x2e, 0x1f, 0xbf, 0xc6, 0x2e, 0x1e, /* .F...... */
  0xbf, 0x05, 0x27, 0x00, 0x2e, 0x3f, 0x8a, 0x23, /* ..'..?.# */
  0x81, 0x02, 0x2e, 0x4d, 0x88, 0x02, 0x2e, 0x60, /* ...M...` */
  0x89, 0x08, 0xef, 0xfe, 0xdf, 0x3f, 0x9f, 0x5f, /* .....?._ */
  0x20, 0xfb, 0x55, 0x28, 0xef, 0xfe, 0xdf, 0x5f, /*  .U(..._ */
  0x20, 0xfc, 0x55, 0x00, 0x2e, 0xf5, 0x89, 0xfe, /*  .U..... */
  0xdf, 0x00, 0x2e, 0xf5, 0x89, 0x14, 0xff, 0xfe, /* ........ */
  0xcf, 0x1f, 0x9f, 0x7f, 0x20, 0xfb, 0x55, 0x23, /* .... .U# */
  0xff, 0xfe, 0xcf, 0x7f, 0x20, 0xfc, 0x55, 0x02, /* .... .U. */
  0x2e, 0x64, 0x89, 0x00, 0x2e, 0xf5, 0x89, 0x14, /* .d...... */
  0xff, 0xfc, 0x97, 0x1f, 0x97                  /* ..... */
};

static const unsigned char pkt69[] = {
  0x42, 0xca, 0x00,
  0x7f, 0x28, 0x7f, 0x20, 0xfb, /* B...(. . */
  0x55, 0x2c, 0xef, 0xbf, 0x22, 0x3f, 0x9f, 0x5f, /* U,.."?._ */
  0x20, 0xfc, 0x55, 0x00, 0x2e, 0xf5, 0x89, 0x40, /*  .U....@ */
  0xef, 0xf6, 0x9f, 0x36, 0x1b, 0x3f, 0x9f, 0x5f, /* ...6.?._ */
  0x20, 0xfa, 0x55, 0x01, 0x2e, 0x67, 0x8d, 0x01, /*  .U..g.. */
  0x2e, 0xfe, 0x8c, 0x00, 0x2e, 0xf5, 0x89, 0x02, /* ........ */
  0x2e, 0x5c, 0x89, 0xdc, 0x9f, 0x3f, 0x9f, 0xdc, /* .\...?.. */
  0x9f, 0x3f, 0x9f, 0xdc, 0x9f, 0x3f, 0x9f, 0xdc, /* .?...?.. */
  0x9f, 0x3f, 0x9f, 0xdc, 0x9f, 0x3f, 0x9f, 0xdc, /* .?...?.. */
  0x9f, 0x3f, 0x9f, 0xdc, 0x9f, 0x3f, 0x9f, 0xdc, /* .?...?.. */
  0x9f, 0x3f, 0x9f, 0x80, 0xff, 0x3f, 0x9f, 0x2f, /* .?...?./ */
  0xef, 0xbf, 0x22, 0x3f, 0x9f, 0x5f, 0x20, 0xfc, /* .."?._ . */
  0x55, 0x3f, 0x9f, 0x3f, 0x9f, 0x3f, 0x9f, 0x3f, /* U?.?.?.? */
  0x9f, 0x3f, 0x9f, 0x3f, 0x9f, 0x02, 0xff, 0x3f, /* .?.?...? */
  0x9f, 0x40, 0xff, 0x3f, 0x9f, 0x00, 0x2e, 0xf5, /* .@.?.... */
  0x89, 0x40, 0xef, 0xfc, 0x9f, 0x5f, 0x20, 0xfc, /* .@..._ . */
  0x55, 0x01, 0x2e, 0xfe, 0x8c, 0x00, 0x2e, 0xff, /* U....... */
  0x89, 0x00, 0x2e, 0xf5, 0x89, 0x14, 0xef, 0x00, /* ........ */
  0x2e, 0x01, 0x8a, 0x46, 0x2e, 0x18, 0xdf, 0xc6, /* ...F.... */
  0x2e, 0x17, 0xdf, 0x1a, 0x27, 0x02, 0x2e, 0x68, /* ....'..h */
  0x89, 0x14, 0xef, 0xfc, 0x9f, 0x5c, 0x1b, 0x3f, /* .....\.? */
  0x97, 0x5f, 0x20, 0xfa, 0x55, 0x2c, 0xef, 0x5c, /* ._ .U,.\ */
  0xff, 0x3f, 0x97, 0x5f, 0x20, 0xfc, 0x55, 0x00, /* .?._ .U. */
  0x2e, 0xff, 0x89, 0x02, 0x2e, 0x68, 0x89, 0x02, /* .....h.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, /* .D...?.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e                  /* .D... */
};

static const unsigned char pkt71[] = {
  0x42, 0xca, 0x00,
  0xe3, 0x28, 0x3f, 0x88, 0x02, /* B...(?.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, /* .D...?.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, /* .D...?.. */
  0x2e, 0x44, 0x88, 0x00, 0x2e, 0xf5, 0x89, 0x40, /* .D.....@ */
  0xef, 0xfc, 0x9f, 0x5f, 0x20, 0xfc, 0x55, 0x01, /* ..._ .U. */
  0x2e, 0x67, 0x8d, 0x01, 0x2e, 0xfe, 0x8c, 0x46, /* .g.....F */
  0x2e, 0x18, 0xdf, 0xc6, 0x2e, 0x17, 0xdf, 0x1a, /* ........ */
  0x27, 0x02, 0x2e, 0x68, 0x89, 0x14, 0xef, 0xfc, /* '..h.... */
  0x9f, 0x3f, 0x97, 0x5f, 0x20, 0xfb, 0x55, 0x80, /* .?._ .U. */
  0xff, 0x3f, 0x97, 0x23, 0xef, 0xbf, 0x22, 0x3f, /* .?.#.."? */
  0x97, 0x5f, 0x20, 0xfc, 0x55, 0x3f, 0x97, 0x3f, /* ._ .U?.? */
  0x97, 0x3f, 0x97, 0x3f, 0x97, 0x3f, 0x97, 0x3f, /* .?.?.?.? */
  0x97, 0x02, 0xff, 0x3f, 0x97, 0xa0, 0xff, 0x3f, /* ...?...? */
  0x97, 0x00, 0x2e, 0xf5, 0x89, 0x40, 0xef, 0xfc, /* .....@.. */
  0x9f, 0x5f, 0x20, 0xfc, 0x55, 0x01, 0x2e, 0xfe, /* ._ .U... */
  0x8c, 0x00, 0x2e, 0xff, 0x89, 0x00, 0x2e, 0xf5, /* ........ */
  0x89, 0x14, 0xef, 0x00, 0x2e, 0x01, 0x8a, 0x00, /* ........ */
  0x2e, 0xff, 0x89, 0x02, 0x2e, 0x68, 0x89, 0x02, /* .....h.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, /* .D...?.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, /* .D...?.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, /* .D...?.. */
  0x2e, 0x44, 0x88, 0x02, 0x2e, 0x3f, 0x88, 0x02, /* .D...?.. */
  0x2e, 0x44, 0x88, 0x17, 0x27, 0x5f, 0x20, 0x1b, /* .D..'_ . */
  0x27, 0xf6, 0x97, 0x0f, 0x1c, 0x54, 0x10, 0x2d, /* '....T.- */
  0xff, 0x1b, 0x27, 0x73, 0xef, 0xff, 0xaf, 0x84, /* ..'s.... */
  0xef, 0xff, 0xaf, 0x00, 0xef                  /* ..... */
};

static const unsigned char pkt73[] = {
  0x42, 0x4c, 0x00,
  0x47, 0x29, 0xff, 0xaf, 0xfc, /* BL.G)... */
  0x97, 0xff, 0xaf, 0xfc, 0x97, 0xff, 0xaf, 0xfc, /* ........ */
  0x97, 0xff, 0xaf, 0xfc, 0x97, 0xff, 0xaf, 0x80, /* ........ */
  0xef, 0xfe, 0xdf, 0xff, 0xbf, 0x5f, 0x20, 0xfb, /* ....._ . */
  0x55, 0x10, 0xef, 0xfe, 0xdf, 0x5f, 0x20, 0xfc, /* U...._ . */
  0x55, 0x00, 0x2e, 0xa9, 0x8a, 0x05, 0x27, 0x69, /* U.....'i */
  0xef, 0x2d, 0xff, 0x1a, 0x27, 0x05, 0x27, 0x69, /* .-..'.'i */
  0xef, 0x2d, 0xff, 0x1b, 0x27, 0x05, 0x27, 0x54, /* .-..'.'T */
  0xef, 0x2d, 0xff, 0x1a, 0x27, 0x05, 0x27, 0x54, /* .-..'.'T */
  0xef, 0x2d, 0xff, 0x1b, 0x27, 0x05, 0x27      /* .-..'.' */
};

static const unsigned char pkt75[] = {
  0x67, 0x05, 0x00, 0x00, 0x02, 0x2e, 0x00, 0x70, /* g......p */
  0x67, 0x05, 0x00, 0x02, 0x02, 0x2e, 0x87, 0x70, /* g......p */
  0x67, 0x05, 0x00, 0x01, 0x02, 0x2e, 0x6f, 0x72, /* g.....or */
  0x67, 0x05, 0x00, 0x03, 0x02, 0x2e, 0x7a, 0x73, /* g.....zs */
  0x67, 0x05, 0x00, 0x08, 0x02, 0x2e, 0x9a, 0x74, /* g......t */
  0x67, 0x05, 0x00, 0x07, 0x02, 0x2e, 0x06, 0x75, /* g......u */
  0x5a, 0x05, 0x00, 0x00, 0x55, 0x01, 0x27, 0x26, /* Z...U.'& */
  0x5a, 0x05, 0x00, 0x01, 0xc4, 0x0a, 0x33, 0x26, /* Z.....3& */
  0x5a, 0x05, 0x00, 0x02, 0xd9, 0x02, 0x45, 0x26, /* Z.....E& */
  0x5a, 0x05, 0x00, 0x03, 0x94, 0x16, 0x6b, 0x26, /* Z.....k& */
  0x5a, 0x05, 0x00, 0x04, 0xcc, 0x0a, 0xad, 0x26, /* Z......& */
  0x5a, 0x05, 0x00, 0x05, 0x3e, 0x0a, 0xb7, 0x26, /* Z...>..& */
  0x5a, 0x05, 0x00, 0x06, 0xf4, 0x09, 0xb7, 0x27, /* Z......' */
  0x0d,
  0x42, 0x04, 0x00,
  0x47, 0x29, 0xff, 0xaf /* .B..G).. */
};

/* INIT1 is over, 0x07 cmd returns
 * { 0x07, 0x05, 0x00, 0x91, 0x26, 0x21, 0x00, 0x3a }
 */
struct aesX660_cmd aes2660_init_1[] = {
  { .cmd = pkt13, .len = sizeof (pkt13) },
  { .cmd = pkt15, .len = sizeof (pkt15) },
  { .cmd = pkt17, .len = sizeof (pkt17) },
  { .cmd = pkt19, .len = sizeof (pkt19) },
  { .cmd = pkt21, .len = sizeof (pkt21) },
  { .cmd = pkt23, .len = sizeof (pkt23) },
  { .cmd = pkt25, .len = sizeof (pkt25) },
  { .cmd = pkt27, .len = sizeof (pkt27) },
  { .cmd = pkt29, .len = sizeof (pkt29) },
  { .cmd = pkt31, .len = sizeof (pkt31) },
  { .cmd = pkt33, .len = sizeof (pkt33) },
  { .cmd = pkt35, .len = sizeof (pkt35) },
  { .cmd = pkt37, .len = sizeof (pkt37) },
  { .cmd = pkt39, .len = sizeof (pkt39) },
  { .cmd = pkt41, .len = sizeof (pkt41) },
  { .cmd = pkt43, .len = sizeof (pkt43) },
  { .cmd = pkt45, .len = sizeof (pkt45) },
  { .cmd = pkt47, .len = sizeof (pkt47) },
  { .cmd = pkt49, .len = sizeof (pkt49) },
  { .cmd = pkt51, .len = sizeof (pkt51) },
  { .cmd = pkt53, .len = sizeof (pkt53) },
  { .cmd = pkt55, .len = sizeof (pkt55) },
  { .cmd = pkt57, .len = sizeof (pkt57) },
  { .cmd = pkt59, .len = sizeof (pkt59) },
  { .cmd = pkt61, .len = sizeof (pkt61) },
  { .cmd = pkt63, .len = sizeof (pkt63) },
  { .cmd = pkt65, .len = sizeof (pkt65) },
  { .cmd = pkt67, .len = sizeof (pkt67) },
  { .cmd = pkt69, .len = sizeof (pkt69) },
  { .cmd = pkt71, .len = sizeof (pkt71) },
  { .cmd = pkt73, .len = sizeof (pkt73) },
  { .cmd = pkt75, .len = sizeof (pkt75) },
};

/* Second init sequence
 * INIT2 */
static const unsigned char pkt187[] = {
  0x0d,
  0x23,
  0x42, 0xca, 0x00,
  0x00, 0x20, 0x7f, /* .#B... . */
  0x97, 0x37, 0xcf, 0xdf, 0x96, 0x26, 0xcf, 0xdf, /* .7...&.. */
  0x96, 0xff, 0x96, 0x9f, 0x22, 0x6d, 0x2e, 0x6a, /* ...."m.j */
  0xbf, 0x62, 0x23, 0x23, 0x83, 0xdd, 0x97, 0x26, /* .b##...& */
  0xaf, 0xdd, 0x97, 0x37, 0xaf, 0xa9, 0xcf, 0x70, /* ...7...p */
  0x18, 0xa9, 0xaf, 0x02, 0xef, 0x5d, 0xaf, 0xc5, /* .....].. */
  0xef, 0x2f, 0xff, 0xff, 0x96, 0xdf, 0x96, 0x6a, /* ./.....j */
  0xef, 0x20, 0xff, 0xff, 0x96, 0xdf, 0x96, 0x1a, /* . ...... */
  0x27, 0x0b, 0xef, 0x06, 0xaf, 0xf1, 0x81, 0x3a, /* '......: */
  0xff, 0x3f, 0x1d, 0x40, 0x19, 0xc1, 0x2e, 0xa3, /* .?.@.... */
  0xbf, 0xc2, 0x2e, 0x00, 0xdf, 0xff, 0x2d, 0x02, /* ......-. */
  0x5a, 0xbf, 0x22, 0xc1, 0x2e, 0xa3, 0xbf, 0x3c, /* Z."....< */
  0x23, 0x57, 0x23, 0x02, 0x2e, 0x31, 0x80, 0x05, /* #W#..1.. */
  0x27, 0x12, 0xef, 0x99, 0xaf, 0x10, 0xef, 0x9b, /* '....... */
  0xaf, 0x10, 0xef, 0x55, 0xaf, 0x3d, 0x23, 0x31, /* ...U.=#1 */
  0xef, 0x20, 0xff, 0x1a, 0x27, 0x05, 0xef, 0x27, /* . ..'..' */
  0xff, 0x06, 0x27, 0x05, 0x27, 0x3f, 0x80, 0x2b, /* ..'.'?.+ */
  0x0d, 0x05, 0x27, 0xe5, 0x31, 0x05, 0x81, 0xe5, /* ..'.1... */
  0x39, 0x05, 0x27, 0xc8, 0x46, 0x0e, 0x5b, 0xd3, /* 9.'.F.[. */
  0x45, 0xfd, 0x5a, 0x0f, 0x27, 0xd3, 0x45, 0xfa, /* E.Z.'.E. */
  0x5a, 0x0f, 0x27, 0xd3, 0x45, 0xf7, 0x5a, 0x41, /* Z.'.E.ZA */
  0x2e, 0xa1, 0xdf, 0x9f, 0x28, 0x41, 0x2e, 0xa1, /* ....(A.. */
  0xbf, 0x02, 0x2e, 0x58, 0x80, 0x05, 0x27, 0x9f, /* ...X..'. */
  0x22, 0xbf, 0x22, 0x1f, 0x20, 0xfd, 0x53, 0x3f, /* ".". .S? */
  0x20, 0x14, 0x1f, 0xfa, 0x55, 0x05, 0x27, 0x10, /*  ...U.'. */
  0xef, 0xd0, 0x81, 0x65, 0xaf, 0x10, 0xef      /* ...e... */
};

static const unsigned char pkt189[] = {
  0x42, 0x4a, 0x00,
  0x64, 0x20, 0xd0, 0x81, 0x9c, /* BJ.d ... */
  0xaf, 0x10, 0xef, 0xd0, 0x81, 0x9a, 0xaf, 0x05, /* ........ */
  0x27, 0x11, 0x00, 0x51, 0x87, 0x91, 0x81, 0xd1, /* '..Q.... */
  0x6c, 0xd2, 0x75, 0xe2, 0x80, 0xf2, 0x80, 0xf0, /* l.u..... */
  0x57, 0xf4, 0x7b, 0xf8, 0x7e, 0xfc, 0x8b, 0x10, /* W.{.~... */
  0xcf, 0x7d, 0xff, 0x3f, 0x00, 0x20, 0xff, 0x00, /* .}.?. .. */
  0x13, 0x1a, 0x27, 0x07, 0x27, 0x05, 0x27, 0x01, /* ..'.'.'. */
  0x00, 0x11, 0x00, 0x21, 0x00, 0x31, 0x00, 0x31, /* ...!.1.1 */
  0x01, 0x32, 0x01, 0x33, 0x01, 0x33, 0x03, 0x43, /* .2.3.3.C */
  0x03, 0x53, 0x03, 0x63, 0x03                  /* .S.c. */
};

static const unsigned char pkt191[] = {
  0x42, 0xca, 0x00,
  0x88, 0x20, 0x51, 0x85, 0x44, /* B... Q.D */
  0x1e, 0x03, 0x55, 0xc8, 0x46, 0x01, 0x5b, 0xe4, /* ..U.F.[. */
  0x2f, 0xc8, 0x50, 0x49, 0x1e, 0x16, 0x55, 0x37, /* /.PI..U7 */
  0x39, 0x0f, 0x27, 0x02, 0xef, 0x02, 0x2e, 0x2f, /* 9.'..../ */
  0x81, 0xfe, 0xcf, 0x08, 0xaf, 0xfe, 0xcf, 0x09, /* ........ */
  0xaf, 0xfe, 0xcf, 0x0b, 0xaf, 0x55, 0x81, 0x02, /* .....U.. */
  0xcf, 0x0f, 0x1c, 0x40, 0x18, 0x08, 0x42, 0x00, /* ...@..B. */
  0x5b, 0x20, 0x18, 0x3d, 0x23, 0x02, 0xaf, 0x10, /* [ .=#... */
  0xef, 0x55, 0xaf, 0x05, 0x27, 0x4a, 0x1e, 0x01, /* .U..'J.. */
  0x55, 0xe4, 0x2f, 0xab, 0x50, 0x4e, 0x1e, 0x01, /* U./.PN.. */
  0x55, 0x00, 0x2e, 0x0d, 0x7b, 0x55, 0x1e, 0x16, /* U...{U.. */
  0x55, 0x54, 0x85, 0xe4, 0x2f, 0xa2, 0x5b, 0x07, /* UT../.[. */
  0xef, 0x20, 0x80, 0xe4, 0x2f, 0x9e, 0x5a, 0xfe, /* . ../.Z. */
  0xcf, 0x0a, 0xaf, 0xfe, 0xcf, 0xfe, 0xcf, 0x41, /* .......A */
  0x2e, 0x44, 0xbf, 0xfe, 0xcf, 0x41, 0x2e, 0x45, /* .D...A.E */
  0xbf, 0xfe, 0xcf, 0x52, 0xaf, 0xfe, 0xcf, 0x53, /* ...R...S */
  0xaf, 0xfe, 0xcf, 0x5a, 0xaf, 0x05, 0x27, 0x58, /* ...Z..'X */
  0x1e, 0x0d, 0x55, 0x02, 0x44, 0x01, 0x5a, 0x00, /* ..U.D.Z. */
  0x2e, 0x54, 0x79, 0xfe, 0xcf, 0x75, 0xaf, 0xfe, /* .Ty..u.. */
  0xdf, 0x76, 0xbf, 0x41, 0x2e, 0xa4, 0xdf, 0xdf, /* .v.A.... */
  0x29, 0x41, 0x2e, 0xa4, 0xbf, 0x05, 0x27, 0x59, /* )A....'Y */
  0x1e, 0x03, 0x55, 0xfe, 0xcf, 0x10, 0xaf, 0x01, /* ..U..... */
  0x2e, 0x94, 0x76, 0x63, 0x1e, 0x1a, 0x55, 0x54, /* ..vc..UT */
  0x85, 0xe4, 0x2f, 0x73, 0x5b, 0x0c, 0xef, 0x20, /* ../s[..  */
  0x80, 0xe4, 0x2f, 0x6f, 0x5a, 0xf0, 0xef, 0x00, /* ../oZ... */
  0xff, 0x19, 0x27, 0x08, 0xef                  /* ..'.. */
};

static const unsigned char pkt193[] = {
  0x42, 0xa2, 0x00,
  0xec, 0x20, 0x06, 0xaf, 0xfe, /* B... ... */
  0xcf, 0x5f, 0x9f, 0x06, 0x25, 0xfb, 0x55, 0xfe, /* ._..%.U. */
  0xcf, 0xfe, 0xcf, 0x41, 0x2e, 0x40, 0xbf, 0xfe, /* ...A.@.. */
  0xcf, 0x41, 0x2e, 0x41, 0xbf, 0xfe, 0xcf, 0x41, /* .A.A...A */
  0x2e, 0x42, 0xbf, 0x05, 0x27, 0x66, 0x1e, 0x0b, /* .B..'f.. */
  0x55, 0xc5, 0xef, 0x2f, 0xff, 0x1a, 0x27, 0x0b, /* U../..'. */
  0xef, 0x06, 0xaf, 0xfe, 0xcf, 0xfe, 0xdf, 0x06, /* ........ */
  0x27, 0x01, 0x0e, 0x06, 0x25, 0xf9, 0x55, 0x05, /* '...%.U. */
  0x27, 0x7d, 0x1e, 0x0c, 0x55, 0x02, 0x2e, 0x29, /* '}..U..) */
  0x81, 0xe4, 0x2f, 0x47, 0x5a, 0xfe, 0xcf, 0xfe, /* ../GZ... */
  0xdf, 0x1b, 0x27, 0xfe, 0xcf, 0x36, 0x03, 0xfe, /* ..'..6.. */
  0xdf, 0x3f, 0x02, 0x9f, 0x9d, 0x05, 0x27, 0x7e, /* .?....'~ */
  0x1e, 0x0c, 0x55, 0x02, 0x2e, 0x29, 0x81, 0xe4, /* ..U..).. */
  0x2f, 0x38, 0x5a, 0xfe, 0xcf, 0xfe, 0xdf, 0x1b, /* /8Z..... */
  0x27, 0x07, 0x80, 0xf6, 0x97, 0xff, 0xaf, 0x00, /* '....... */
  0x2e, 0xa7, 0x8a, 0x05, 0x27, 0x61, 0x74, 0xff, /* ....'at. */
  0x96, 0xc6, 0x2e, 0x01, 0xdf, 0x3f, 0x2c, 0xdd, /* .....?,. */
  0x9f, 0x05, 0x27, 0xf9, 0x35, 0xc1, 0x2e, 0x4e, /* ..'.5..N */
  0xdf, 0x0f, 0x1d, 0x09, 0x27, 0x5f, 0x02, 0xc1, /* ....'_.. */
  0x2e, 0x4e, 0xbf, 0x05, 0x27, 0x9f, 0x22, 0x02, /* .N..'.". */
  0x2e, 0xe7, 0x8c, 0x05, 0x27                  /* ....' */
};

static const unsigned char pkt195[] = {
  0x42, 0xca, 0x00,
  0x3c, 0x21, 0x51, 0x85, 0x01, /* B..<!Q.. */
  0x1e, 0x38, 0x55, 0x26, 0x40, 0x12, 0x5b, 0x26, /* .8U&@.[& */
  0x42, 0x2e, 0x5a, 0x26, 0x41, 0x0e, 0x5a, 0x26, /* B.Z&A.Z& */
  0xcf, 0x40, 0x1a, 0x26, 0xaf, 0x9f, 0x2d, 0x27, /* .@.&..-' */
  0x5a, 0x07, 0x80, 0x01, 0xef, 0xff, 0xaf, 0x26, /* Z......& */
  0x43, 0x01, 0x5b, 0x74, 0x25, 0x00, 0x55, 0x26, /* C.[t%.U& */
  0x32, 0x00, 0x2e, 0xa7, 0x7a, 0x02, 0x2e, 0x01, /* 2...z... */
  0x8d, 0x26, 0x40, 0x08, 0x5b, 0x41, 0x2e, 0x52, /* .&@.[A.R */
  0xdf, 0xdf, 0x2d, 0x04, 0x5b, 0x26, 0x43, 0x01, /* ..-.[&C. */
  0x5b, 0x74, 0x25, 0x00, 0x55, 0x26, 0x32, 0x41, /* [t%.U&2A */
  0x2e, 0x53, 0xdf, 0xc1, 0x2e, 0x79, 0xdf, 0xff, /* .S...y.. */
  0x96, 0xdf, 0x96, 0x41, 0x2e, 0x55, 0xdf, 0xc1, /* ...A.U.. */
  0x2e, 0x78, 0xdf, 0x9d, 0x01, 0x5d, 0x01, 0xff, /* .x...].. */
  0x22, 0x00, 0x54, 0xff, 0xef, 0x2e, 0xaf, 0x05, /* ".T..... */
  0x27, 0x07, 0x80, 0x9f, 0x22, 0xff, 0xaf, 0x00, /* '..."... */
  0x2e, 0xa7, 0x8a, 0x05, 0x27, 0x06, 0x1e, 0x02, /* ....'... */
  0x55, 0x0f, 0x27, 0x0f, 0x27, 0x01, 0x70, 0x0a, /* U.'.'.p. */
  0x1e, 0x0b, 0x55, 0x02, 0x2e, 0x82, 0x81, 0xa8, /* ..U..... */
  0x77, 0x02, 0x46, 0x00, 0x5b, 0x08, 0x3b, 0x3b, /* w.F.[.;; */
  0x30, 0x07, 0x35, 0xff, 0xff, 0xc2, 0xbf, 0x0d, /* 0.5..... */
  0x27, 0x0c, 0x27, 0x0d, 0x1e, 0x02, 0x55, 0x54, /* '.'...UT */
  0x85, 0xf2, 0x5b, 0xbd, 0x77, 0x12, 0x1e, 0x07, /* ..[.w... */
  0x55, 0xdc, 0xef, 0x05, 0xff, 0x1a, 0x27, 0x56, /* U.....'V */
  0xdf, 0x3f, 0x20, 0x59, 0x80, 0x3c, 0xaf, 0x05, /* .? Y.<.. */
  0x27, 0x13, 0x1e, 0x01, 0x55, 0x3c, 0x23, 0x05, /* '...U<#. */
  0x27, 0x1f, 0x1e, 0x06, 0x55                  /* '...U */
};

static const unsigned char pkt197[] = {
  0x42, 0x60, 0x00,
  0xa0, 0x21, 0x37, 0x35, 0x08, /* B`..!75. */
  0xef, 0x40, 0x2e, 0xf0, 0xbf, 0x40, 0x2e, 0xf6, /* .@...@.. */
  0xbf, 0x05, 0x27, 0x20, 0x1e, 0x06, 0x55, 0x37, /* ..' ..U7 */
  0x3d, 0x16, 0xef, 0x40, 0x2e, 0xf0, 0xbf, 0x40, /* =..@...@ */
  0x2e, 0xf6, 0xbf, 0x05, 0x27, 0x22, 0x1e, 0x1b, /* ....'".. */
  0x55, 0x21, 0xef, 0x08, 0xaf, 0x00, 0xef, 0x09, /* U!...... */
  0xaf, 0x01, 0xef, 0x0b, 0xaf, 0x31, 0xef, 0x0a, /* .....1.. */
  0xaf, 0x81, 0xcf, 0x10, 0xaf, 0x01, 0x2e, 0x94, /* ........ */
  0x86, 0x82, 0xcf, 0x4a, 0xaf, 0x01, 0x2e, 0xf9, /* ...J.... */
  0x81, 0x08, 0xef, 0x9b, 0xaf, 0x99, 0xaf, 0x02, /* ........ */
  0xcf, 0x0f, 0x1c, 0x40, 0x18, 0x37, 0x31, 0x02, /* ...@.71. */
  0xaf, 0xe2, 0x23, 0x08, 0xef, 0xe3, 0xaf, 0x05, /* ..#..... */
  0x27, 0xf7, 0x74                              /* '.t */
};

static const unsigned char pkt199[] = {
  0x42, 0xca, 0x00,
  0xcf, 0x21, 0x9b, 0x22, 0xff, /* B...!.". */
  0x86, 0x02, 0x2e, 0xeb, 0x81, 0xc2, 0x85, 0x00, /* ........ */
  0xef, 0x41, 0x2e, 0x76, 0xbf, 0x56, 0x87, 0x02, /* .A.v.V.. */
  0x2e, 0x42, 0x82, 0x78, 0x86, 0x02, 0x33, 0x07, /* .B.x..3. */
  0x80, 0xdd, 0x97, 0xff, 0xaf, 0x00, 0x2e, 0xa7, /* ........ */
  0x8a, 0x05, 0x27, 0x6c, 0xcf, 0x6d, 0xdf, 0xbf, /* ..'l.m.. */
  0x01, 0x6c, 0xaf, 0x6d, 0x26, 0x00, 0x2e, 0x2c, /* .l.m&.., */
  0x55, 0x00, 0x2e, 0x47, 0x50, 0x9f, 0x22, 0x6b, /* U..GP."k */
  0xaf, 0x62, 0xaf, 0x6d, 0x2e, 0x6a, 0xbf, 0x01, /* .b.m.j.. */
  0x2e, 0x3f, 0x84, 0x00, 0xef, 0x41, 0x2e, 0x44, /* .?...A.D */
  0xbf, 0xbf, 0xef, 0x41, 0x2e, 0x45, 0xbf, 0x80, /* ...A.E.. */
  0xef, 0x0a, 0xaf, 0x01, 0x2e, 0x06, 0x82, 0x80, /* ........ */
  0x82, 0x84, 0x82, 0x6b, 0xcf, 0x10, 0xaf, 0x01, /* ...k.... */
  0x2e, 0x94, 0x86, 0x40, 0xef, 0x6d, 0xaf, 0x80, /* ...@.m.. */
  0xef, 0x6c, 0xaf, 0x01, 0x2e, 0xf9, 0x81, 0xed, /* .l...... */
  0x80, 0x01, 0x2e, 0x18, 0x87, 0x01, 0x2e, 0x1a, /* ........ */
  0x87, 0x07, 0x87, 0xff, 0x2f, 0xd1, 0x5a, 0x6c, /* ..../.Zl */
  0xcf, 0x6d, 0xdf, 0x3f, 0x00, 0x6c, 0xaf, 0x6d, /* .m.?.l.m */
  0x26, 0x03, 0x54, 0x6d, 0xdf, 0x0c, 0x82, 0x6d, /* &.Tm...m */
  0xbf, 0xeb, 0x50, 0x6b, 0xcf, 0x08, 0x1e, 0x11, /* ..Pk.... */
  0x5f, 0x6c, 0x24, 0x02, 0x53, 0xff, 0xef, 0x6c, /* _l$.S..l */
  0xaf, 0x10, 0x50, 0x6c, 0x24, 0x02, 0x53, 0xff, /* ..Pl$.S. */
  0xef, 0x6c, 0xaf, 0x0b, 0x50, 0x6b, 0xcf, 0x0a, /* .l..Pk.. */
  0x1e, 0x08, 0x55, 0x6c, 0x24, 0x02, 0x53, 0xff, /* ..Ul$.S. */
  0xef, 0x6c, 0xaf, 0x03, 0x50, 0x6c, 0x24, 0x01, /* .l..Pl$. */
  0x53, 0xff, 0xef, 0x6c, 0xaf                  /* S..l. */
};

static const unsigned char pkt201[] = {
  0x42, 0xca, 0x00,
  0x33, 0x22, 0x6b, 0xcf, 0xdf, /* B..3"k.. */
  0x96, 0xb0, 0xef, 0x00, 0xff, 0x1d, 0x00, 0x00, /* ........ */
  0x13, 0x1b, 0x27, 0x6c, 0xcf, 0x9f, 0x9d, 0x6b, /* ..'l...k */
  0xcf, 0x1f, 0x20, 0x6b, 0xaf, 0x0a, 0x1e, 0xbc, /* .. k.... */
  0x5f, 0x05, 0x27, 0x01, 0x2e, 0x3f, 0x84, 0x55, /* _.'..?.U */
  0x82, 0x01, 0x2e, 0x06, 0x82, 0x73, 0x23, 0x0a, /* .....s#. */
  0xef, 0x10, 0xaf, 0x01, 0x2e, 0x94, 0x86, 0x80, /* ........ */
  0xef, 0x01, 0x2e, 0xf9, 0x81, 0xed, 0x80, 0x08, /* ........ */
  0xef, 0x06, 0xaf, 0x01, 0x2e, 0x18, 0x87, 0x01, /* ........ */
  0x2e, 0x1a, 0x87, 0x06, 0x25, 0xf9, 0x55, 0x02, /* ....%.U. */
  0x2e, 0xa7, 0x82, 0x4e, 0x23, 0x40, 0xcf, 0xe0, /* ...N#@.. */
  0xaf, 0x41, 0xdf, 0xe1, 0xbf, 0x41, 0x2e, 0x4e, /* .A...A.N */
  0xdf, 0xf7, 0x1c, 0x41, 0x2e, 0x4e, 0xbf, 0x01, /* ...A.N.. */
  0x2e, 0x18, 0x87, 0x01, 0x2e, 0x1a, 0x87, 0x45, /* .......E */
  0x82, 0x4e, 0x24, 0x4e, 0xcf, 0x20, 0x1e, 0x03, /* .N$N. .. */
  0x55, 0x01, 0xef, 0x1f, 0x90, 0x02, 0x2e, 0xdc, /* U....... */
  0x71, 0x40, 0x2e, 0x8f, 0xdf, 0xdf, 0x22, 0xe5, /* q@....". */
  0x55, 0x4e, 0x86, 0x4e, 0x23, 0x4e, 0xcf, 0x52, /* UN.N#N.R */
  0x86, 0x59, 0x86, 0x3f, 0x9f, 0x4e, 0xcf, 0x52, /* .Y.?.N.R */
  0x86, 0x5f, 0x86, 0x1f, 0x9f, 0x4e, 0x24, 0x4e, /* ._...N$N */
  0xcf, 0x04, 0x1e, 0xf3, 0x55, 0xf8, 0x0f, 0x08, /* ....U... */
  0xef, 0x06, 0xaf, 0xf6, 0x97, 0xbc, 0x03, 0x01, /* ........ */
  0x5f, 0xff, 0x0f, 0xfc, 0x97, 0x06, 0x25, 0xf9, /* _.....%. */
  0x55, 0xf8, 0x0f, 0x08, 0xff, 0x06, 0xbf, 0xf6, /* U....... */
  0x9f, 0xdf, 0x01, 0xa1, 0x27, 0x7f, 0x21, 0x3f, /* ....'.!? */
  0x9f, 0x06, 0x25, 0xf8, 0x55                  /* ..%.U */
};

static const unsigned char pkt203[] = {
  0x42, 0xb2, 0x00,
  0x97, 0x22, 0xf8, 0x0f, 0x04, /* B..."... */
  0xef, 0x06, 0xaf, 0xbb, 0xef, 0x00, 0xff, 0x1a, /* ........ */
  0x27, 0xfc, 0x9f, 0xfc, 0x97, 0x09, 0x27, 0x5f, /* '.....'_ */
  0x02, 0x3f, 0x97, 0x06, 0x25, 0xf8, 0x55, 0x81, /* .?..%.U. */
  0xcf, 0x24, 0xaf, 0x05, 0x27, 0x0a, 0xef, 0x73, /* .$..'..s */
  0x43, 0x02, 0x5b, 0x73, 0xcf, 0x09, 0x27, 0x0f, /* C.[s..'. */
  0x1c, 0x81, 0xaf, 0x73, 0xcf, 0x09, 0x27, 0x30, /* ...s..'0 */
  0x1c, 0xfc, 0xdf, 0xcf, 0x1d, 0x5f, 0x02, 0xfc, /* ....._.. */
  0xbf, 0x81, 0xcf, 0x10, 0xaf, 0x01, 0x2e, 0x94, /* ........ */
  0x86, 0x80, 0x82, 0x01, 0x2e, 0x34, 0x81, 0x40, /* .....4.@ */
  0xef, 0x6b, 0xaf, 0x80, 0xef, 0x82, 0xaf, 0x01, /* .k...... */
  0x2e, 0xf9, 0x81, 0xed, 0x80, 0x01, 0x2e, 0x18, /* ........ */
  0x87, 0x01, 0x2e, 0x1a, 0x87, 0x16, 0x82, 0x01, /* ........ */
  0x2e, 0x2f, 0x89, 0x0a, 0x5b, 0x73, 0x43, 0x19, /* ./..[sC. */
  0x5a, 0x81, 0xcf, 0x5e, 0xdf, 0xbf, 0x03, 0x15, /* Z..^.... */
  0x5f, 0x5f, 0x20, 0x81, 0xaf, 0xdf, 0x22, 0x11, /* __ ...". */
  0x54, 0xde, 0x50, 0x40, 0x2e, 0x8f, 0xdf, 0x1f, /* T.P@.... */
  0x2c, 0x00, 0x2e, 0x0c, 0x5a, 0x5f, 0x2c, 0x09, /* ,...Z_,. */
  0x5b, 0x82, 0xcf, 0x6b, 0xdf, 0x3f, 0x00, 0x82, /* [..k.?.. */
  0xaf, 0x6b, 0x26, 0xe7, 0x54, 0x6b, 0xdf, 0x0c, /* .k&.Tk.. */
  0x82, 0x6b, 0xbf, 0xd8, 0x50, 0x05, 0x27, 0x82, /* .k..P.'. */
  0xcf, 0x6b, 0xdf, 0xbf, 0x01, 0x82, 0xaf, 0x6b, /* .k.....k */
  0x26, 0xf4, 0x55, 0xdb, 0x50                  /* &.U.P */
};

static const unsigned char pkt205[] = {
  0x42, 0x86, 0x00,
  0xef, 0x22, 0xa5, 0xcf, 0xed, /* B..."... */
  0x2e, 0x6a, 0xdf, 0xf3, 0x1c, 0x3f, 0x02, 0xa5, /* .j...?.. */
  0xaf, 0x4a, 0xcf, 0xdf, 0x96, 0x10, 0xcf, 0x5f, /* .J....._ */
  0xdf, 0xbf, 0x03, 0x02, 0x5f, 0x5f, 0xcf, 0x10, /* ....__.. */
  0xaf, 0x04, 0x50, 0x5e, 0xdf, 0xbf, 0x03, 0x01, /* ..P^.... */
  0x53, 0x5e, 0xcf, 0x10, 0xaf, 0x01, 0x2e, 0x40, /* S^.....@ */
  0x88, 0x01, 0x2e, 0xe1, 0x86, 0xdd, 0x97, 0x0a, /* ........ */
  0x40, 0x18, 0x5a, 0x62, 0xcf, 0xdf, 0x22, 0x22, /* @.Zb.."" */
  0x54, 0xed, 0x2e, 0x6a, 0xdf, 0x10, 0xef, 0x00, /* T..j.... */
  0x1f, 0x06, 0x54, 0x0c, 0xef, 0x04, 0x1f, 0x03, /* ..T..... */
  0x54, 0x0b, 0xef, 0x08, 0x1f, 0x00, 0x54, 0x05, /* T.....T. */
  0xef, 0x62, 0xdf, 0x76, 0x80, 0x12, 0x82, 0x12, /* .b.v.... */
  0x82, 0x80, 0x10, 0x01, 0x13, 0x4a, 0xcf, 0x3f, /* .....J.? */
  0x00, 0x00, 0x53, 0xff, 0xef, 0x4a, 0xdf, 0xed, /* ..S..J.. */
  0x2e, 0x6e, 0xbf, 0x4a, 0xaf, 0x6d, 0x2e, 0x6f, /* .n.J.m.o */
  0xbf, 0xdf, 0x96, 0xbf, 0x01, 0x6d, 0x2e, 0x6d, /* .....m.m */
  0xbf, 0xdd, 0x97, 0x01, 0x2e, 0xf9, 0x81, 0x05, /* ........ */
  0x27                                          /* ' */
};

static const unsigned char pkt207[] = {
  0x42, 0xca, 0x00,
  0x31, 0x23, 0x01, 0x2e, 0x48, /* B..1#..H */
  0x88, 0x02, 0x2e, 0x5c, 0x83, 0x37, 0x41, 0xdf, /* ...\.7A. */
  0x2f, 0x0c, 0x5a, 0x0a, 0x40, 0x1d, 0x5a, 0x02, /* /.Z.@.Z. */
  0x2e, 0xe2, 0x83, 0x02, 0x2e, 0x02, 0x84, 0x02, /* ........ */
  0x2e, 0x82, 0x83, 0x02, 0x2e, 0xa9, 0x83, 0x02, /* ........ */
  0x2e, 0x0f, 0x84, 0x37, 0x42, 0x09, 0x5a, 0x09, /* ...7B.Z. */
  0x46, 0x0f, 0x5b, 0x23, 0x26, 0x0d, 0x55, 0x22, /* F.[#&.U" */
  0xcf, 0xc6, 0x1e, 0x00, 0x2e, 0x8a, 0x52, 0x10, /* ......R. */
  0xcf, 0x27, 0xaf, 0x08, 0xef, 0x6f, 0xaf, 0x9f, /* .'...o.. */
  0x22, 0x6e, 0xaf, 0x6d, 0x2e, 0x6c, 0xbf, 0x6d, /* "n.m.l.m */
  0x2e, 0x6b, 0xbf, 0x6e, 0x26, 0x00, 0x55, 0x6e, /* .k.n&.Un */
  0x24, 0x05, 0x27, 0x9a, 0xcf, 0xdf, 0x96, 0xe0, /* $.'..... */
  0xef, 0x2c, 0xff, 0x19, 0x27, 0xfe, 0x0c, 0x9f, /* .,..'... */
  0x22, 0x00, 0xff, 0xe1, 0x22, 0x07, 0x55, 0x20, /* "...".U  */
  0x00, 0x05, 0x52, 0x82, 0x03, 0x03, 0x53, 0x02, /* ..R...S. */
  0x0d, 0x3f, 0x20, 0x0f, 0x1f, 0xf5, 0x55, 0x5f, /* .? ...U_ */
  0x90, 0x82, 0x01, 0x1f, 0x90, 0xc1, 0x97, 0xa0, /* ........ */
  0x01, 0x82, 0x01, 0xdf, 0x21, 0x80, 0x03, 0x02, /* ....!... */
  0x5e, 0x7f, 0x20, 0x00, 0x53, 0xbf, 0x22, 0x29, /* ^. .S.") */
  0xcf, 0xbf, 0x03, 0x01, 0x5e, 0x5f, 0x20, 0xdf, /* ....^_ . */
  0x9f, 0x28, 0xbf, 0x03, 0x0c, 0x05, 0x27, 0x62, /* .(....'b */
  0xcf, 0x6d, 0x2e, 0x6c, 0xbf, 0x61, 0x26, 0x20, /* .m.l.a&  */
  0x55, 0x29, 0xcf, 0x09, 0x1e, 0x08, 0x53, 0x02, /* U)....S. */
  0xff, 0x6d, 0x2e, 0x6c, 0xdf, 0xbf, 0x01, 0x00, /* .m.l.... */
  0x53, 0x9f, 0x22, 0x6d, 0x2e, 0x6c, 0xbf, 0x05, /* S."m.l.. */
  0x27, 0x28, 0xdf, 0x02, 0x1f                  /* '(... */
};

static const unsigned char pkt209[] = {
  0x42, 0xca, 0x00,
  0x95, 0x23, 0x11, 0x52, 0x0c, /* B...#.R. */
  0x1e, 0x0f, 0x52, 0x6d, 0x2e, 0x6c, 0xdf, 0x02, /* ..Rm.l.. */
  0x10, 0x30, 0x1e, 0x00, 0x5f, 0x30, 0xef, 0x6d, /* .0.._0.m */
  0x2e, 0x6c, 0xbf, 0x62, 0xdf, 0xbf, 0x01, 0xdf, /* .l.b.... */
  0x9f, 0x6e, 0xcf, 0xbf, 0x01, 0x00, 0x53, 0x9f, /* .n....S. */
  0x22, 0x6e, 0xaf, 0x05, 0x27, 0x6d, 0x2e, 0x6a, /* "n..'m.j */
  0xdf, 0x6d, 0x2e, 0x6b, 0xbf, 0x29, 0x50, 0x23, /* .m.k.)P# */
  0x26, 0x02, 0x55, 0x22, 0xcf, 0x20, 0x1e, 0x24, /* &.U". .$ */
  0x5f, 0x40, 0x2e, 0x62, 0xdf, 0xed, 0x2e, 0x6c, /* _@.b...l */
  0xdf, 0xbf, 0x03, 0x1e, 0x55, 0x61, 0x26, 0x1c, /* ....Ua&. */
  0x55, 0x6d, 0x2e, 0x6a, 0xdf, 0x00, 0x1e, 0x01, /* Um.j.... */
  0x55, 0x06, 0xff, 0x06, 0x50, 0x04, 0x1e, 0x01, /* U...P... */
  0x55, 0x07, 0xff, 0x02, 0x50, 0x08, 0x1e, 0x10, /* U...P... */
  0x55, 0x08, 0xff, 0x28, 0xcf, 0xbf, 0x03, 0x0c, /* U..(.... */
  0x52, 0x6d, 0x2e, 0x6b, 0xdf, 0x04, 0x10, 0x6d, /* Rm.k...m */
  0x2e, 0x6b, 0xbf, 0x03, 0xff, 0x04, 0x1e, 0xcf, /* .k...... */
  0x54, 0x03, 0xff, 0x08, 0x1e, 0xcc, 0x54, 0x06, /* T.....T. */
  0xff, 0xca, 0x50, 0x05, 0x27, 0x6c, 0x2e, 0xff, /* ..P.'l.. */
  0xdf, 0xdf, 0x22, 0x21, 0x55, 0x6c, 0x2e, 0xfe, /* .."!Ul.. */
  0xdf, 0xdf, 0x22, 0x1d, 0x55, 0x05, 0x27, 0x61, /* ..".U.'a */
  0x23, 0x6c, 0x2e, 0xff, 0xdf, 0xdf, 0x22, 0x17, /* #l....". */
  0x55, 0x6c, 0x2e, 0xfe, 0xdf, 0x65, 0xdf, 0xbf, /* Ul...e.. */
  0x03, 0x12, 0x5e, 0x3b, 0x43, 0x01, 0x5a, 0x3b, /* ..^;C.Z; */
  0x41, 0x0d, 0x5a, 0x29, 0xdf, 0xff, 0x96, 0x25, /* A.Z)...% */
  0xdf, 0x5d, 0x00, 0x10, 0x1f, 0x08, 0x53, 0x07, /* .]....S. */
  0x1f, 0x05, 0x53, 0x61, 0x24                  /* ..Sa$ */
};

static const unsigned char pkt211[] = {
  0x42, 0xca, 0x00,
  0xf9, 0x23, 0x3b, 0x41, 0x02, /* B...#;A. */
  0x5a, 0x03, 0x1f, 0x00, 0x53, 0x61, 0x24, 0x05, /* Z...Sa$. */
  0x27, 0xff, 0xef, 0x61, 0xaf, 0x05, 0x27, 0x3b, /* '..a..'; */
  0x43, 0x09, 0x5b, 0x28, 0xdf, 0x6e, 0xbf, 0x60, /* C.[(.n.` */
  0xef, 0xbf, 0x22, 0x1a, 0x27, 0x29, 0xdf, 0x6e, /* ..".').n */
  0xcf, 0xdf, 0x01, 0x59, 0x80, 0x6f, 0xaf, 0x05, /* ...Y.o.. */
  0x27, 0x10, 0xdf, 0x27, 0xbf, 0x61, 0xcf, 0xdf, /* '..'.a.. */
  0x22, 0x21, 0x54, 0x5f, 0x00, 0x00, 0x57, 0xbf, /* "!T_..W. */
  0x22, 0x5f, 0xcf, 0xdf, 0x03, 0x01, 0x5f, 0x61, /* "_...._a */
  0x23, 0x5f, 0xdf, 0x27, 0xbf, 0x61, 0xcf, 0xdf, /* #_.'.a.. */
  0x2d, 0x22, 0x5a, 0x02, 0x2e, 0x66, 0x84, 0xdf, /* -"Z..f.. */
  0x96, 0x6e, 0xdf, 0x02, 0x2e, 0x37, 0x84, 0x6e, /* .n...7.n */
  0xbf, 0xc0, 0x97, 0xed, 0x2e, 0x6c, 0xdf, 0x02, /* .....l.. */
  0x2e, 0x37, 0x84, 0x30, 0x1f, 0x00, 0x5f, 0x30, /* .7.0.._0 */
  0xff, 0xed, 0x2e, 0x6c, 0xbf, 0x6f, 0xdf, 0xdd, /* ...l.o.. */
  0x97, 0x02, 0x2e, 0x3c, 0x84, 0x6f, 0xaf, 0x05, /* ...<.o.. */
  0x27, 0x76, 0x80, 0x20, 0x10, 0x00, 0x13, 0x12, /* 'v. .... */
  0x82, 0x05, 0x27, 0xdf, 0x96, 0x9f, 0x22, 0xc3, /* ..'...". */
  0x81, 0x1a, 0x27, 0xdd, 0x9f, 0x59, 0x80, 0x05, /* ..'..Y.. */
  0x27, 0x02, 0x2e, 0x66, 0x84, 0xdf, 0x96, 0xed, /* '..f.... */
  0x2e, 0x6c, 0xdf, 0xc0, 0x97, 0x02, 0x2e, 0x3c, /* .l.....< */
  0x84, 0x6d, 0x2e, 0x6c, 0xbf, 0x6f, 0xdf, 0xc0, /* .m.l.o.. */
  0x97, 0x02, 0x2e, 0x37, 0x84, 0x6f, 0xbf, 0x6e, /* ...7.o.n */
  0xdf, 0xdd, 0x97, 0x02, 0x2e, 0x3c, 0x84, 0xed, /* .....<.. */
  0x2e, 0x6b, 0xdf, 0x04, 0x1f, 0x03, 0x5f, 0xfe, /* .k...._. */
  0x10, 0x08, 0x1f, 0x00, 0x54                  /* ....T */
};

static const unsigned char pkt213[] = {
  0x42, 0x24, 0x00,
  0x5d, 0x24, 0x02, 0x10, 0x6e, /* B$.]$..n */
  0xaf, 0x04, 0xef, 0x08, 0x1f, 0x00, 0x53, 0x9f, /* ......S. */
  0x22, 0x6d, 0x2e, 0x6b, 0xbf, 0x05, 0x27, 0x27, /* "m.k..'' */
  0xcf, 0x61, 0x47, 0x00, 0x5b, 0x1f, 0x20, 0x01, /* .aG.[. . */
  0x2e, 0x41, 0x88, 0xff, 0x97, 0x05, 0x27      /* .A....' */
};

static const unsigned char pkt215[] = {
  0x42, 0xca, 0x00,
  0x6e, 0x24, 0x5c, 0x46, 0x00, /* B..n$\F. */
  0x5b, 0xfb, 0x31, 0x02, 0x2e, 0x27, 0x87, 0x02, /* [.1..'.. */
  0x2e, 0x97, 0x86, 0x02, 0x2e, 0xaf, 0x86, 0x02, /* ........ */
  0x2e, 0xff, 0x86, 0x02, 0x2e, 0xf0, 0x85, 0x02, /* ........ */
  0x2e, 0xf1, 0x85, 0x40, 0xcf, 0x41, 0xdf, 0x45, /* ...@.A.E */
  0xaf, 0x46, 0xbf, 0x07, 0x38, 0x01, 0x5b, 0x02, /* .F..8.[. */
  0x2e, 0x8f, 0x85, 0x3b, 0x40, 0x00, 0x2e, 0x43, /* ...;@..C */
  0x5b, 0x5c, 0x47, 0x01, 0x5a, 0x02, 0x2e, 0xaf, /* [\G.Z... */
  0x86, 0x09, 0x46, 0x01, 0x5b, 0x02, 0x2e, 0xfe, /* ..F.[... */
  0x87, 0xc9, 0x30, 0x41, 0x2e, 0x7a, 0xdf, 0x1f, /* ..0A.z.. */
  0x28, 0x41, 0x2e, 0x7a, 0xbf, 0xfb, 0x39, 0xfa, /* (A.z..9. */
  0x33, 0xc8, 0x46, 0x0f, 0x5b, 0xd0, 0x46, 0xfd, /* 3.F.[.F. */
  0x5b, 0xd3, 0x45, 0xfd, 0x5a, 0xd3, 0x45, 0xfb, /* [.E.Z.E. */
  0x5a, 0xd3, 0x3c, 0x41, 0x2e, 0xa1, 0xdf, 0x9f, /* Z.<A.... */
  0x28, 0x41, 0x2e, 0xa1, 0xbf, 0x02, 0x2e, 0x58, /* (A.....X */
  0x80, 0x02, 0x2e, 0xb7, 0x74, 0x4e, 0x81, 0xd3, /* ....tN.. */
  0x45, 0xea, 0x5a, 0xd3, 0x3c, 0xc8, 0xcf, 0xdf, /* E.Z.<... */
  0x22, 0x02, 0x55, 0xc7, 0xcf, 0xdf, 0x22, 0x02, /* ".U...". */
  0x54, 0xd3, 0x33, 0xd3, 0x31, 0x4e, 0x81, 0xf7, /* T.3.1N.. */
  0xef, 0x16, 0x80, 0xf8, 0xcf, 0x9f, 0x1c, 0xf8, /* ........ */
  0xaf, 0xfb, 0x39, 0xfa, 0x33, 0x5c, 0x47, 0x03, /* ..9.3\G. */
  0x5a, 0x08, 0x43, 0xde, 0x2f, 0x15, 0x5a, 0x02, /* Z.C./.Z. */
  0x50, 0x3d, 0x23, 0xde, 0x2f, 0x11, 0x50, 0x02, /* P=#./.P. */
  0xcf, 0x0f, 0x1c, 0x10, 0x18, 0x02, 0xaf, 0xd9, /* ........ */
  0x72, 0x5c, 0x46, 0x01, 0x5a, 0x01, 0x2e, 0x71, /* r\F.Z..q */
  0x87, 0xd0, 0x41, 0x00, 0x5b                  /* ..A.[ */
};

static const unsigned char pkt217[] = {
  0x42, 0xca, 0x00,
  0xd2, 0x24, 0xed, 0x83, 0x5c, /* B...$..\ */
  0x47, 0x17, 0x5a, 0x52, 0x26, 0x04, 0x55, 0x53, /* G.ZR&.US */
  0x26, 0x02, 0x55, 0x07, 0x3a, 0x00, 0x2e, 0x20, /* &.U.:..  */
  0x7e, 0xc8, 0x47, 0x00, 0x2e, 0x09, 0x5b, 0x08, /* ~.G...[. */
  0xef, 0xc0, 0xaf, 0x07, 0x3a, 0x04, 0x27, 0x00, /* ....:.'. */
  0xef, 0xc0, 0xaf, 0x08, 0xef, 0xc1, 0xaf, 0x00, /* ........ */
  0x2e, 0x20, 0x7e, 0x07, 0x32, 0xc9, 0x31, 0x00, /* . ~.2.1. */
  0x2e, 0x20, 0x7e, 0x41, 0x2e, 0x7a, 0xdf, 0xdf, /* . ~A.z.. */
  0x2d, 0x0d, 0x5b, 0x10, 0xef, 0xc0, 0xaf, 0x07, /* -.[..... */
  0x3a, 0x04, 0x27, 0xc0, 0x23, 0x10, 0xef, 0xc1, /* :.'.#... */
  0xaf, 0x41, 0x2e, 0x72, 0xdf, 0x1f, 0x20, 0x41, /* .A.r.. A */
  0x2e, 0x72, 0xbf, 0x00, 0x2e, 0x20, 0x7e, 0x07, /* .r... ~. */
  0x32, 0xc9, 0x31, 0x41, 0x2e, 0x72, 0xdf, 0x1f, /* 2.1A.r.. */
  0x20, 0x41, 0x2e, 0x72, 0xbf, 0x00, 0x2e, 0x20, /*  A.r...  */
  0x7e, 0x00, 0x2e, 0x5b, 0x82, 0x45, 0xcf, 0x46, /* ~..[.E.F */
  0xdf, 0x19, 0x27, 0x02, 0x2e, 0x45, 0x86, 0x02, /* ..'..E.. */
  0x2e, 0x59, 0x86, 0x05, 0x27, 0x08, 0x82, 0x05, /* .Y..'... */
  0x27, 0x02, 0x2e, 0xcb, 0x87, 0x10, 0x5b, 0x49, /* '.....[I */
  0xef, 0xff, 0xaf, 0x05, 0xef, 0xff, 0xaf, 0x9f, /* ........ */
  0x22, 0xff, 0xaf, 0x10, 0xef, 0xff, 0xaf, 0x22, /* "......" */
  0xcf, 0xff, 0xaf, 0x23, 0xcf, 0xff, 0xaf, 0x9f, /* ...#.... */
  0x22, 0xff, 0xaf, 0xff, 0xaf, 0x02, 0x2e, 0x46, /* "......F */
  0x80, 0x0c, 0x40, 0x2a, 0x5b, 0x36, 0x80, 0x0d, /* ..@*[6.. */
  0x34, 0x10, 0x5a, 0x49, 0xcf, 0x5f, 0x9d, 0x49, /* 4.ZI._.I */
  0x38, 0x3f, 0x80, 0x49, 0xcf, 0x5f, 0x9d, 0x36, /* 8?.I._.6 */
  0x80, 0x9f, 0x22, 0x1f, 0x9c                  /* ..".. */
};

static const unsigned char pkt219[] = {
  0x42, 0xca, 0x00,
  0x36, 0x25, 0x5f, 0x9c, 0x1e, /* B..6%_.. */
  0xcf, 0x1f, 0x9d, 0x48, 0xcf, 0x1d, 0xdf, 0xdf, /* ...H.... */
  0x01, 0x9f, 0x22, 0x12, 0x50, 0x02, 0x2e, 0x3f, /* ..".P..? */
  0x86, 0xf5, 0x97, 0x49, 0xaf, 0x02, 0x2e, 0x69, /* ...I...i */
  0x87, 0x49, 0xcf, 0xf7, 0x1c, 0x5f, 0x9d, 0x44, /* .I..._.D */
  0xdf, 0x3f, 0x20, 0x44, 0xbf, 0xff, 0x9a, 0x8c, /* .? D.... */
  0x81, 0x20, 0x0d, 0x9f, 0x99, 0xe0, 0x0d, 0x1c, /* . ...... */
  0xcf, 0x1d, 0xdf, 0x9f, 0x9c, 0xff, 0x9c, 0x01, /* ........ */
  0x2e, 0x04, 0x85, 0x0c, 0x41, 0x01, 0x5b, 0x01, /* ....A.[. */
  0x2e, 0x06, 0x85, 0x0c, 0x42, 0x04, 0x5b, 0xe4, /* ....B.[. */
  0x40, 0x01, 0x5a, 0xe4, 0x3a, 0x00, 0x50, 0xe4, /* @.Z.:.P. */
  0x32, 0x0c, 0x43, 0x04, 0x5b, 0xe4, 0x42, 0x01, /* 2.C.[.B. */
  0x5a, 0xe4, 0x30, 0x00, 0x50, 0xe4, 0x38, 0x0c, /* Z.0.P.8. */
  0x44, 0x11, 0x5b, 0x18, 0x23, 0x1a, 0x23, 0x1b, /* D.[.#.#. */
  0x23, 0xe4, 0x42, 0x01, 0x5a, 0xe4, 0x38, 0x00, /* #.B.Z.8. */
  0x50, 0xe4, 0x30, 0x36, 0x80, 0x02, 0x2e, 0x40, /* P.06...@ */
  0x80, 0x02, 0x2e, 0x42, 0x80, 0xe4, 0x42, 0x01, /* ...B..B. */
  0x5a, 0xe4, 0x30, 0x00, 0x50, 0xe4, 0x38, 0x0c, /* Z.0.P.8. */
  0x45, 0x01, 0x5b, 0x02, 0x2e, 0x97, 0x85, 0x0c, /* E.[..... */
  0x46, 0x01, 0x5a, 0x02, 0x2e, 0x8f, 0x85, 0x0c, /* F.Z..... */
  0x43, 0x05, 0x5b, 0x0c, 0x45, 0x03, 0x5b, 0x0d, /* C.[.E.[. */
  0x45, 0x01, 0x5b, 0x0d, 0x37, 0x05, 0x27, 0x0c, /* E.[.7.'. */
  0x40, 0x00, 0x5b, 0x0d, 0x3f, 0x05, 0x27, 0x60, /* @.[.?.'` */
  0xcf, 0x33, 0x1c, 0x09, 0x27, 0x60, 0xdf, 0x0c, /* .3..'`.. */
  0x1d, 0x3f, 0x02, 0x60, 0xaf, 0x05, 0x27, 0x60, /* .?.`..'` */
  0xcf, 0x0f, 0x1c, 0xdf, 0x9f                  /* ..... */
};

static const unsigned char pkt221[] = {
  0x42, 0xca, 0x00,
  0x9a, 0x25, 0x0c, 0x1c, 0x08, /* B...%... */
  0x82, 0x03, 0x1d, 0x0f, 0x82, 0x5f, 0x02, 0x60, /* ....._.` */
  0xcf, 0xf0, 0x1c, 0x3f, 0x02, 0x60, 0xaf, 0x05, /* ...?.`.. */
  0x27, 0x60, 0xcf, 0x09, 0x27, 0x03, 0x1c, 0x00, /* '`..'... */
  0x1e, 0x02, 0x55, 0xa7, 0xef, 0x2d, 0xff, 0x06, /* ..U..-.. */
  0x50, 0x01, 0x1e, 0x02, 0x55, 0x1a, 0xef, 0x31, /* P...U..1 */
  0xff, 0x01, 0x50, 0x8d, 0xef, 0x34, 0xff, 0x40, /* ..P..4.@ */
  0xaf, 0x41, 0xbf, 0x19, 0x27, 0x05, 0x27, 0x2d, /* .A..'.'- */
  0xff, 0x64, 0xef, 0x19, 0x27, 0x01, 0xff, 0x5a, /* .d..'..Z */
  0xef, 0x1b, 0x27, 0x06, 0xef, 0x06, 0xaf, 0xfc, /* ..'..... */
  0x97, 0x5f, 0x9f, 0x06, 0x25, 0xfb, 0x55, 0x05, /* ._..%.U. */
  0x27, 0x10, 0xcf, 0x72, 0xaf, 0x0a, 0x40, 0x1a, /* '..r..@. */
  0x5a, 0x6d, 0x2e, 0x6b, 0xdf, 0x6d, 0x2e, 0x6a, /* Zm.k.m.j */
  0xbf, 0x6d, 0x2e, 0x6c, 0xdf, 0x62, 0xaf, 0x27, /* .m.l.b.' */
  0xcf, 0x10, 0xaf, 0x07, 0x43, 0x08, 0x5b, 0x3b, /* ....C.[; */
  0x31, 0x02, 0x5b, 0x3b, 0x36, 0x3b, 0x43, 0x03, /* 1.[;6;C. */
  0x5b, 0x6e, 0xcf, 0xe2, 0xaf, 0x6f, 0xcf, 0xe3, /* [n...o.. */
  0xaf, 0x01, 0x2e, 0x94, 0x86, 0x07, 0x40, 0x02, /* ......@. */
  0x5a, 0x61, 0x26, 0x00, 0x55, 0x07, 0x33, 0x80, /* Za&.U.3. */
  0x82, 0x02, 0x43, 0xdc, 0x2f, 0x9c, 0x5b, 0x01, /* ..C./.[. */
  0x2e, 0x34, 0x81, 0x05, 0x27, 0xa7, 0xcf, 0x5f, /* .4..'.._ */
  0x21, 0xa6, 0xcf, 0x5f, 0x21, 0x4a, 0xaf, 0x05, /* !.._!J.. */
  0x27, 0x05, 0x27, 0x08, 0x45, 0x13, 0x5a, 0x32, /* '.'.E.Z2 */
  0x80, 0xd2, 0xef, 0x5f, 0x9f, 0x29, 0xef, 0x5f, /* ..._.)._ */
  0x9f, 0x9f, 0x22, 0x5f, 0x9f, 0x18, 0xef, 0x5f, /* .."_..._ */
  0x9f, 0x08, 0xef, 0x5f, 0x9f                  /* ..._. */
};

static const unsigned char pkt223[] = {
  0x42, 0xca, 0x00,
  0xfe, 0x25, 0x10, 0xef, 0x5f, /* B...%.._ */
  0x9f, 0x26, 0xef, 0x06, 0xaf, 0x9f, 0x22, 0x5f, /* .&...."_ */
  0x9f, 0x06, 0x25, 0xfc, 0x55, 0x4e, 0x80, 0x05, /* ..%.UN.. */
  0x27, 0x0c, 0xcf, 0x03, 0x1c, 0x0b, 0x55, 0x0c, /* '.....U. */
  0xcf, 0x0c, 0x1c, 0x0c, 0x1e, 0x07, 0x54, 0x08, /* ......T. */
  0x1e, 0x03, 0x55, 0x0c, 0x45, 0x01, 0x5b, 0x0d, /* ..U.E.[. */
  0x44, 0x01, 0x5b, 0x0e, 0x33, 0x05, 0x27, 0x0e, /* D.[.3.'. */
  0x3b, 0x05, 0x27, 0x0d, 0xcf, 0x06, 0x1c, 0x0a, /* ;.'..... */
  0x55, 0x09, 0x46, 0x05, 0x5b, 0x02, 0x2e, 0x38, /* U.F.[..8 */
  0x86, 0x02, 0x5a, 0xf2, 0xcf, 0x0b, 0xaf, 0x12, /* ..Z..... */
  0x50, 0x02, 0xef, 0x0c, 0xaf, 0x0f, 0x50, 0x0a, /* P.....P. */
  0x41, 0x0d, 0x5b, 0x0c, 0x39, 0x0c, 0x46, 0x05, /* A.[.9.F. */
  0x5b, 0x36, 0x80, 0x02, 0x2e, 0x3f, 0x86, 0x02, /* [6...?.. */
  0x2e, 0x69, 0x87, 0x00, 0x50, 0x0c, 0x31, 0x0c, /* .i..P.1. */
  0x38, 0x01, 0x5b, 0x0d, 0x3f, 0x0d, 0x34, 0x05, /* 8.[.?.4. */
  0x27, 0x23, 0x26, 0x02, 0x55, 0x22, 0xcf, 0xc8, /* '#&.U".. */
  0x1e, 0x00, 0x5f, 0x0c, 0x27, 0x0d, 0x27, 0xf0, /* .._.'.'. */
  0x97, 0x47, 0xaf, 0xf1, 0x97, 0x48, 0xaf, 0x05, /* .G...H.. */
  0x27, 0x32, 0x80, 0x08, 0x46, 0x0f, 0x5a, 0xde, /* '2..F.Z. */
  0xef, 0x5f, 0x9f, 0x10, 0xef, 0x5f, 0x9f, 0x00, /* ._..._.. */
  0xef, 0x5f, 0x9f, 0x2d, 0xff, 0x54, 0xef, 0x1b, /* ._.-.T.. */
  0x27, 0x10, 0xef, 0x06, 0xaf, 0xfc, 0x97, 0x5f, /* '......_ */
  0x9f, 0x06, 0x25, 0xfb, 0x55, 0x4e, 0x80, 0x05, /* ..%.UN.. */
  0x27, 0x32, 0x80, 0x08, 0x47, 0xfb, 0x5a, 0xdf, /* '2..G.Z. */
  0xef, 0x5f, 0x9f, 0x06, 0xef, 0x5f, 0x9f, 0x00, /* ._..._.. */
  0xef, 0x5f, 0x9f, 0x2d, 0xff                  /* ._.-. */
};

static const unsigned char pkt225[] = {
  0x42, 0xca, 0x00,
  0x62, 0x26, 0x64, 0xef, 0x1b, /* B..b&d.. */
  0x27, 0x06, 0xef, 0x06, 0xaf, 0xea, 0x50, 0x2d, /* '.....P- */
  0xff, 0x64, 0xef, 0x19, 0x27, 0x0f, 0xef, 0x6b, /* .d..'..k */
  0xdf, 0xff, 0x96, 0xbf, 0x22, 0xff, 0x96, 0x7b, /* ...."..{ */
  0x00, 0xc1, 0x03, 0x6b, 0xbf, 0x0c, 0x53, 0xdf, /* ...k..S. */
  0x96, 0xc2, 0x97, 0xbf, 0x01, 0x5f, 0x90, 0xdd, /* ....._.. */
  0x97, 0x5f, 0x20, 0x02, 0x1e, 0xf3, 0x53, 0x02, /* ._ ...S. */
  0xef, 0x01, 0x0c, 0xdd, 0x9f, 0x6b, 0xbf, 0x15, /* .....k.. */
  0x50, 0xdf, 0x96, 0xc2, 0x97, 0xdf, 0x01, 0xc1, /* P....... */
  0x03, 0x05, 0x52, 0xdd, 0x97, 0x1f, 0x20, 0x6b, /* ..R... k */
  0xdf, 0xe0, 0x01, 0x6b, 0xbf, 0x00, 0x50, 0xdd, /* ...k..P. */
  0x97, 0x02, 0x0c, 0x6b, 0xdf, 0x28, 0x1f, 0x01, /* ...k.(.. */
  0x53, 0x28, 0xff, 0x6b, 0xbf, 0x68, 0x1f, 0x01, /* S(.k.h.. */
  0x5f, 0x68, 0xff, 0x6b, 0xbf, 0x05, 0x27, 0xe9, /* _h.k..'. */
  0x80, 0x02, 0x2e, 0xc4, 0x85, 0x02, 0x2e, 0xba, /* ........ */
  0x87, 0xed, 0x80, 0x02, 0x2e, 0xa4, 0x85, 0x02, /* ........ */
  0x2e, 0xea, 0x85, 0x01, 0x2e, 0xa9, 0x83, 0x5c, /* .......\ */
  0x47, 0x02, 0x5b, 0x01, 0x2e, 0xe7, 0x83, 0x01, /* G.[..... */
  0x50, 0x01, 0x2e, 0xe5, 0x83, 0x08, 0xef, 0x0f, /* P....... */
  0x80, 0x01, 0x2e, 0x18, 0x87, 0x05, 0x27, 0x07, /* ......'. */
  0x40, 0x01, 0x5b, 0x54, 0x23, 0x1f, 0x50, 0x02, /* @.[T#.P. */
  0x2e, 0x08, 0x85, 0x0d, 0xcf, 0x06, 0x1c, 0x0b, /* ........ */
  0x54, 0x02, 0x2e, 0xd4, 0x86, 0x02, 0x2e, 0x42, /* T......B */
  0x80, 0x01, 0x2e, 0x7b, 0x89, 0x02, 0x2e, 0x08, /* ...{.... */
  0x86, 0x02, 0x2e, 0x32, 0x87, 0x02, 0x2e, 0x94, /* ...2.... */
  0x87, 0x02, 0x2e, 0x19, 0x86                  /* ..... */
};

static const unsigned char pkt227[] = {
  0x42, 0xca, 0x00,
  0xc6, 0x26, 0x02, 0x2e, 0x22, /* B...&.." */
  0x87, 0x09, 0x46, 0x02, 0x5b, 0x2d, 0x82, 0x90, /* ..F.[-.. */
  0xaf, 0x91, 0xbf, 0x01, 0x2e, 0xbd, 0x84, 0x02, /* ........ */
  0x2e, 0x14, 0x85, 0x0c, 0x3e, 0xe8, 0x5a, 0x05, /* ....>.Z. */
  0x27, 0x2d, 0x82, 0x40, 0x82, 0x12, 0x82, 0x12, /* '-.@.... */
  0x82, 0x9d, 0xbf, 0xff, 0x97, 0x08, 0x10, 0x09, /* ........ */
  0x27, 0x0f, 0x1c, 0x5c, 0x46, 0x03, 0x5b, 0x02, /* '..\F.[. */
  0x1e, 0x04, 0x53, 0x02, 0xef, 0x02, 0x50, 0x03, /* ..S...P. */
  0x1e, 0x00, 0x53, 0x03, 0xef, 0x69, 0xaf, 0x3f, /* ..S..i.? */
  0x80, 0x1b, 0x0d, 0xe1, 0x97, 0x0f, 0x1c, 0x69, /* .......i */
  0xdf, 0xdf, 0x01, 0x00, 0x53, 0x00, 0xff, 0xe0, /* ....S... */
  0x97, 0x76, 0x80, 0x04, 0x10, 0x07, 0x82, 0x0f, /* .v...... */
  0x1e, 0x00, 0x5f, 0x0f, 0xef, 0x6a, 0xaf, 0xc1, /* .._..j.. */
  0x2e, 0x47, 0xdf, 0xf0, 0x1d, 0x5f, 0x02, 0xc1, /* .G..._.. */
  0x2e, 0x47, 0xbf, 0x10, 0x0d, 0x05, 0x27, 0x01, /* .G....'. */
  0x2e, 0x1a, 0x87, 0xf7, 0xef, 0x16, 0x80, 0x52, /* .......R */
  0x80, 0x16, 0x82, 0x02, 0x2e, 0xb7, 0x85, 0x55, /* .......U */
  0x82, 0x02, 0x2e, 0x60, 0x80, 0x01, 0x2e, 0x84, /* ...`.... */
  0x83, 0x26, 0x40, 0x00, 0x5b, 0x3b, 0x33, 0x5c, /* .&@.[;3\ */
  0x47, 0x09, 0x5b, 0x6c, 0x38, 0x3b, 0x43, 0x00, /* G.[l8;C. */
  0x5b, 0x6c, 0x30, 0x41, 0x2e, 0x71, 0xdf, 0xc1, /* [l0A.q.. */
  0x2e, 0x72, 0xdf, 0xbf, 0x03, 0x01, 0x55, 0x01, /* .r....U. */
  0x2e, 0x98, 0x83, 0x01, 0x2e, 0x63, 0x84, 0x01, /* .....c.. */
  0x2e, 0x71, 0x87, 0x05, 0x27, 0x3b, 0x40, 0x01, /* .q..';@. */
  0x5b, 0x02, 0xef, 0x0c, 0xaf, 0x05, 0x27, 0x52, /* [.....'R */
  0xcf, 0xcb, 0xaf, 0x53, 0xdf                  /* ...S. */
};

static const unsigned char pkt229[] = {
  0x42, 0xca, 0x00,
  0x2a, 0x27, 0xcc, 0xbf, 0xdf, /* B..*'... */
  0x22, 0x01, 0x55, 0xff, 0x22, 0x01, 0x54, 0x0e, /* ".U.".T. */
  0xef, 0xc9, 0xaf, 0x05, 0x27, 0x0e, 0xcf, 0x6e, /* ....'..n */
  0x1c, 0x49, 0xdf, 0x10, 0x1d, 0x3f, 0x02, 0x49, /* .I...?.I */
  0xaf, 0x0d, 0x44, 0x06, 0x5a, 0x0a, 0x41, 0x03, /* ..D.Z.A. */
  0x5b, 0x49, 0x43, 0x02, 0x5a, 0x48, 0x26, 0x00, /* [IC.ZH&. */
  0x55, 0x49, 0x30, 0x9f, 0x2c, 0x05, 0x5a, 0xdf, /* UI0.,.Z. */
  0x2c, 0x4d, 0x5a, 0x0e, 0x41, 0x16, 0x5b, 0x0c, /* ,MZ.A.[. */
  0x46, 0x14, 0x5a, 0x9f, 0x22, 0x1c, 0xaf, 0x21, /* F.Z."..! */
  0xaf, 0x02, 0x2e, 0xb4, 0x87, 0x03, 0x5a, 0x9f, /* ......Z. */
  0x2c, 0x05, 0x5a, 0x9f, 0x22, 0x04, 0x50, 0x9f, /* ,.Z.".P. */
  0x22, 0x0e, 0x3e, 0x0e, 0x3d, 0x00, 0x5b, 0x5f, /* ".>.=.[_ */
  0xef, 0x1d, 0xaf, 0x49, 0xcf, 0x87, 0x1c, 0x01, /* ...I.... */
  0x18, 0x49, 0xaf, 0x34, 0x50, 0x20, 0x26, 0x08, /* .I.4P &. */
  0x55, 0x1f, 0xcf, 0x48, 0xdf, 0xff, 0x2d, 0x00, /* U..H..-. */
  0x5b, 0xff, 0x21, 0x3f, 0x00, 0x00, 0x53, 0xff, /* [.!?..S. */
  0xef, 0x1f, 0xaf, 0x47, 0xdf, 0x1c, 0xcf, 0x3f, /* ...G...? */
  0x00, 0x1c, 0xaf, 0x1d, 0xcf, 0x48, 0xdf, 0x3f, /* .....H.? */
  0x00, 0x1d, 0xaf, 0xff, 0x22, 0x1f, 0x54, 0xff, /* ....".T. */
  0x2d, 0x03, 0x5b, 0xff, 0x1e, 0x19, 0x5b, 0x60, /* -.[...[` */
  0x10, 0x02, 0x50, 0x60, 0x1e, 0x0a, 0x52, 0x60, /* ..P`..R` */
  0x16, 0x1d, 0xaf, 0x49, 0x30, 0x49, 0x3c, 0x12, /* ...I0I<. */
  0x5b, 0x21, 0xcf, 0x47, 0xdf, 0x3f, 0x00, 0x1c, /* [!.G.?.. */
  0xaf, 0x21, 0x23, 0x0c, 0x50, 0x59, 0x1e, 0x0a, /* .!#.PY.. */
  0x52, 0x49, 0x34, 0x01, 0x5a, 0x21, 0x23, 0x06, /* RI4.Z!#. */
  0x50, 0x21, 0xcf, 0x47, 0xdf                  /* P!.G. */
};

static const unsigned char pkt231[] = {
  0x42, 0xca, 0x00,
  0x8e, 0x27, 0x3f, 0x00, 0x21, /* B...'?.! */
  0xaf, 0x01, 0x50, 0x06, 0x1e, 0xf4, 0x5f, 0x05, /* ..P..._. */
  0x27, 0x1e, 0x23, 0x0d, 0xcf, 0x06, 0x1c, 0x1a, /* '.#..... */
  0x54, 0x48, 0xcf, 0xdf, 0x2d, 0x00, 0x5b, 0xdf, /* TH..-.[. */
  0x21, 0x09, 0x27, 0x1f, 0x22, 0xbf, 0x22, 0x1a, /* !.'.".". */
  0x27, 0xc0, 0x2e, 0x96, 0xdf, 0x7f, 0x1f, 0x02, /* '....... */
  0x54, 0x3f, 0x20, 0xc0, 0x2e, 0x96, 0xbf, 0x59, /* T? ....Y */
  0x80, 0x40, 0x2e, 0x97, 0xbf, 0xc0, 0x2e, 0x95, /* .@...... */
  0xdf, 0x76, 0x80, 0x1e, 0xbf, 0x0c, 0x42, 0x02, /* .v....B. */
  0x5b, 0x9f, 0x22, 0x40, 0x2e, 0x96, 0xbf, 0x05, /* [."@.... */
  0x27, 0x0d, 0xcf, 0x06, 0x1c, 0x06, 0x1e, 0x00, /* '....... */
  0x54, 0x0d, 0x27, 0x0c, 0x27, 0x09, 0x46, 0x0d, /* T.'.'.F. */
  0x5b, 0x02, 0x2e, 0x38, 0x86, 0x07, 0x52, 0x0f, /* [..8..R. */
  0xcf, 0x01, 0x16, 0x02, 0x53, 0x02, 0x2e, 0xfe, /* ....S... */
  0x87, 0x04, 0x50, 0xf4, 0xaf, 0x02, 0x50, 0xf2, /* ..P...P. */
  0xcf, 0x02, 0x10, 0x0f, 0xaf, 0x05, 0x27, 0x5a, /* ......'Z */
  0xcf, 0xdf, 0x22, 0x10, 0x54, 0x3e, 0xdf, 0x3f, /* ..".T>.? */
  0x20, 0x3f, 0x1f, 0x0a, 0x52, 0xbf, 0x22, 0x3e, /*  ?..R."> */
  0xbf, 0x3d, 0xdf, 0x3f, 0x20, 0xdf, 0x03, 0x02, /* .=.? ... */
  0x52, 0xbf, 0x22, 0x3d, 0xbf, 0x0c, 0x27, 0x3d, /* R."=..'= */
  0xbf, 0x0d, 0x27, 0x3e, 0xbf, 0x0d, 0x27, 0x0d, /* ..'>..'. */
  0x27, 0xbf, 0x22, 0x3e, 0xbf, 0x3d, 0xbf, 0x05, /* '.">.=.. */
  0x27, 0x0f, 0x1c, 0x69, 0xdf, 0xdf, 0x01, 0x00, /* '..i.... */
  0x53, 0x00, 0xff, 0x05, 0x27, 0x2d, 0x82, 0x40, /* S...'-.@ */
  0x82, 0x12, 0x82, 0x12, 0x82, 0x9d, 0xbf, 0xff, /* ........ */
  0x97, 0x08, 0x10, 0x09, 0x27                  /* ....' */
};

static const unsigned char pkt233[] = {
  0x42, 0x24, 0x00,
  0xf2, 0x27, 0x0f, 0x1c, 0x5c, /* B$..'..\ */
  0x46, 0x03, 0x5b, 0x02, 0x1e, 0x04, 0x53, 0x02, /* F.[...S. */
  0xef, 0x02, 0x50, 0x03, 0x1e, 0x00, 0x53, 0x03, /* ..P...S. */
  0xef, 0x69, 0xaf, 0x05, 0x27, 0x14, 0xef, 0xf4, /* .i..'... */
  0xaf, 0x00, 0xef, 0xf5, 0xaf, 0x05, 0x27      /* ......' */
};

static const unsigned char pkt235[] = {
  0x42, 0xca, 0x00,
  0x03, 0x28, 0x0c, 0x42, 0x00, /* B...(.B. */
  0x5b, 0x18, 0x23, 0x00, 0xef, 0x0c, 0xaf, 0x0e, /* [.#..... */
  0xcf, 0xe0, 0x1c, 0x0e, 0xaf, 0x0a, 0x40, 0x02, /* ......@. */
  0x5a, 0x3b, 0x46, 0x00, 0x5a, 0x05, 0x27, 0x37, /* Z;F.Z.'7 */
  0x43, 0xfc, 0x5b, 0x08, 0xef, 0x0c, 0xaf, 0x0d, /* C.[..... */
  0x40, 0x0a, 0x5a, 0xee, 0xcf, 0x64, 0x1e, 0x01, /* @.Z..d.. */
  0x5e, 0x0c, 0x3b, 0x05, 0x27, 0x0d, 0x30, 0x0c, /* ^.;.'.0. */
  0x35, 0xbf, 0x22, 0xc0, 0x2e, 0x96, 0xbf, 0x05, /* 5."..... */
  0x27, 0x0e, 0x45, 0x02, 0x5b, 0x02, 0x2e, 0x30, /* '.E.[..0 */
  0x88, 0x01, 0x50, 0x02, 0x2e, 0x3a, 0x88, 0x0d, /* ..P..:.. */
  0x41, 0x24, 0x5a, 0x0d, 0x42, 0x14, 0x5a, 0x02, /* A$Z.B.Z. */
  0x2e, 0x2f, 0x79, 0x02, 0x2e, 0x8f, 0x88, 0x19, /* ./y..... */
  0xcf, 0xfe, 0x1e, 0x03, 0x5c, 0x0e, 0x3d, 0x0d, /* ....\.=. */
  0x39, 0x0d, 0x3b, 0x0c, 0x27, 0x0d, 0x27, 0x02, /* 9.;.'.'. */
  0x2e, 0xae, 0x88, 0x19, 0xcf, 0x02, 0x1e, 0xf9, /* ........ */
  0x5a, 0x0e, 0x3e, 0x0d, 0x3a, 0x0d, 0x33, 0x0c, /* Z.>.:.3. */
  0x27, 0xe4, 0x37, 0x02, 0x2e, 0x8b, 0x89, 0xe4, /* '.7..... */
  0x3f, 0x02, 0x2e, 0x80, 0x89, 0x0d, 0x45, 0x61, /* ?.....Ea */
  0x5b, 0x18, 0xcf, 0xdf, 0x22, 0x6b, 0x54, 0x0d, /* [..."kT. */
  0x3b, 0x02, 0x2e, 0xab, 0x78, 0x02, 0x2e, 0x8b, /* ;...x... */
  0x89, 0x02, 0x2e, 0x80, 0x89, 0x0d, 0x42, 0x07, /* ......B. */
  0x5a, 0x0d, 0x45, 0x53, 0x5b, 0x18, 0xcf, 0xdf, /* Z.ES[... */
  0x22, 0x5d, 0x54, 0x0d, 0x33, 0x02, 0x2e, 0xab, /* "]T.3... */
  0x78, 0x15, 0xcf, 0x1a, 0xaf, 0xe4, 0x37, 0x02, /* x.....7. */
  0x2e, 0x8b, 0x89, 0x15, 0xcf, 0x1b, 0xaf, 0xe4, /* ........ */
  0x3f, 0x0d, 0xcf, 0x60, 0x1c                  /* ?..`. */
};

static const unsigned char pkt237[] = {
  0x42, 0xca, 0x00,
  0x67, 0x28, 0xdf, 0x22, 0x01, /* B..g(.". */
  0x55, 0x02, 0x2e, 0xdf, 0x78, 0x16, 0xcf, 0x13, /* U...x... */
  0xdf, 0x2a, 0x1e, 0x00, 0x2e, 0x04, 0x5f, 0x2a, /* .*...._* */
  0x1f, 0x00, 0x2e, 0x01, 0x5f, 0x02, 0x2e, 0xdf, /* ...._... */
  0x78, 0xbf, 0x03, 0x40, 0x54, 0x20, 0x52, 0x02, /* x..@T R. */
  0x2e, 0x80, 0x89, 0x9f, 0x22, 0x1a, 0xaf, 0x18, /* ...."... */
  0xcf, 0x12, 0xdf, 0xbf, 0x03, 0x00, 0x53, 0x29, /* ......S) */
  0x50, 0x02, 0x2e, 0x8f, 0x88, 0x02, 0x2e, 0x74, /* P......t */
  0x89, 0x9f, 0x22, 0x1b, 0xaf, 0x02, 0x2e, 0x2e, /* .."..... */
  0x88, 0x02, 0x2e, 0xab, 0x78, 0x0e, 0xcf, 0x9f, /* ....x... */
  0x1c, 0x0e, 0xaf, 0x05, 0x27, 0x0e, 0xcf, 0x5f, /* ....'.._ */
  0x2d, 0x05, 0x5a, 0x9f, 0x2b, 0x5f, 0x29, 0x0e, /* -.Z.+_). */
  0xaf, 0x5f, 0xff, 0x1d, 0xbf, 0x1c, 0x23, 0x05, /* ._....#. */
  0x27, 0x9f, 0x22, 0x1b, 0xaf, 0x18, 0xcf, 0xdf, /* '."..... */
  0x22, 0x0c, 0x54, 0x18, 0xcf, 0x12, 0xdf, 0xbf, /* ".T..... */
  0x03, 0x00, 0x53, 0x07, 0x50, 0x02, 0x2e, 0xae, /* ..S.P... */
  0x88, 0x02, 0x2e, 0x7a, 0x89, 0x9f, 0x22, 0x1a, /* ...z..". */
  0xaf, 0x02, 0x2e, 0x38, 0x88, 0x02, 0x2e, 0xbc, /* ...8.... */
  0x88, 0x05, 0x27, 0x0e, 0xcf, 0x9f, 0x2d, 0x05, /* ..'...-. */
  0x5a, 0x9f, 0x29, 0x5f, 0x2b, 0x0e, 0xaf, 0x00, /* Z.)_+... */
  0xff, 0x1d, 0xbf, 0x1c, 0x23, 0x05, 0x27, 0x02, /* ....#.'. */
  0x2e, 0x64, 0x89, 0x02, 0x2e, 0xc4, 0x78, 0x0d, /* .d....x. */
  0x45, 0x20, 0x5b, 0x02, 0x2e, 0x64, 0x89, 0x18, /* E [..d.. */
  0xcf, 0x12, 0xdf, 0xbf, 0x03, 0x6a, 0x53, 0x0d, /* .....jS. */
  0x44, 0x02, 0x5b, 0x0c, 0x35, 0x02, 0x2e, 0xd6, /* D.[.5... */
  0x78, 0x17, 0xcf, 0xdf, 0x22                  /* x..." */
};

static const unsigned char pkt239[] = {
  0x42, 0xca, 0x00,
  0xcb, 0x28, 0x09, 0x55, 0x18, /* B...(.U. */
  0xcf, 0xdf, 0x22, 0x06, 0x55, 0x02, 0x2e, 0xdc, /* ..".U... */
  0x88, 0x0e, 0x3b, 0x0c, 0xdf, 0x24, 0x19, 0x0c, /* ..;..$.. */
  0xbf, 0x05, 0x27, 0x11, 0xcf, 0xff, 0x1e, 0x01, /* ..'..... */
  0x54, 0x1f, 0x20, 0x11, 0xaf, 0x05, 0x27, 0x1a, /* T. ...'. */
  0x23, 0x1b, 0x23, 0x05, 0x27, 0x02, 0x2e, 0xdc, /* #.#.'... */
  0x88, 0x02, 0x2e, 0xf2, 0x88, 0x02, 0x2e, 0x64, /* .......d */
  0x89, 0x02, 0x2e, 0xb4, 0x87, 0x37, 0x5a, 0x0a, /* .....7Z. */
  0xcf, 0xc0, 0x1c, 0xc0, 0x1e, 0x0e, 0x55, 0x9f, /* ......U. */
  0x22, 0x1f, 0xaf, 0x20, 0xaf, 0x19, 0xaf, 0x02, /* ".. .... */
  0x2e, 0x0f, 0x79, 0x0e, 0xcf, 0x05, 0x18, 0x0e, /* ..y..... */
  0xaf, 0x9f, 0x22, 0x47, 0xaf, 0x48, 0xaf, 0x17, /* .."G.H.. */
  0xaf, 0x18, 0xaf, 0x05, 0x27, 0x20, 0xcf, 0xdf, /* ....' .. */
  0x22, 0x06, 0x54, 0x05, 0x1e, 0x1b, 0x55, 0x1f, /* ".T...U. */
  0xdf, 0x80, 0x1f, 0x18, 0x5e, 0x02, 0x2e, 0x08, /* ....^... */
  0x79, 0x1f, 0xdf, 0x10, 0x1f, 0x13, 0x5e, 0x9f, /* y.....^. */
  0x22, 0x20, 0xaf, 0x1f, 0xaf, 0x0a, 0xcf, 0xc0, /* " ...... */
  0x1c, 0x80, 0x1e, 0x0a, 0x55, 0x0d, 0x41, 0x01, /* ....U.A. */
  0x5a, 0x0e, 0x35, 0x00, 0x50, 0x0e, 0x36, 0x0d, /* Z.5.P.6. */
  0xcf, 0x06, 0x18, 0x0d, 0xaf, 0x0e, 0x31, 0x9f, /* ......1. */
  0x22, 0x19, 0xaf, 0x02, 0x2e, 0x20, 0x79, 0x1f, /* ".... y. */
  0x20, 0x06, 0x1e, 0x00, 0x5e, 0x20, 0xaf, 0x0d, /*  ...^ .. */
  0x47, 0x0c, 0x5b, 0x51, 0xef, 0x0c, 0xaf, 0x0e, /* G.[Q.... */
  0xcf, 0xfa, 0x1c, 0x0e, 0xaf, 0x20, 0xcf, 0xdf, /* ..... .. */
  0x22, 0x03, 0x54, 0x06, 0x1e, 0x01, 0x54, 0x5f, /* ".T...T_ */
  0x20, 0x20, 0xaf, 0x05, 0x27                  /*   ..' */
};

static const unsigned char pkt241[] = {
  0x42, 0xca, 0x00,
  0x2f, 0x29, 0x0d, 0x44, 0x0b, /* B../).D. */
  0x5a, 0x0d, 0xcf, 0x06, 0x1c, 0x08, 0x54, 0x0c, /* Z.....T. */
  0x30, 0x48, 0xcf, 0xdf, 0x2d, 0x02, 0x5a, 0x9f, /* 0H..-.Z. */
  0x22, 0x1d, 0xaf, 0x01, 0x50, 0x5f, 0xef, 0x1d, /* "...P_.. */
  0xaf, 0x0c, 0x31, 0x0d, 0x45, 0x13, 0x5b, 0x0a, /* ..1.E.[. */
  0xcf, 0x30, 0x1c, 0x0e, 0x55, 0x0d, 0xcf, 0x06, /* .0..U... */
  0x1c, 0x06, 0x1e, 0x0d, 0x54, 0x11, 0xcf, 0x07, /* ....T... */
  0x1e, 0x03, 0x53, 0x03, 0xef, 0x12, 0xaf, 0x02, /* ..S..... */
  0x2e, 0x54, 0x79, 0x05, 0xef, 0x12, 0xaf, 0x02, /* .Ty..... */
  0x2e, 0x54, 0x79, 0x01, 0xef, 0x12, 0xaf, 0x18, /* .Ty..... */
  0x23, 0x9f, 0x22, 0x11, 0xaf, 0x0d, 0xcf, 0x1f, /* #."..... */
  0x1c, 0x0d, 0xaf, 0x0c, 0xdf, 0x24, 0x19, 0x0c, /* .....$.. */
  0xbf, 0x0a, 0xdf, 0x30, 0x1d, 0x20, 0x1f, 0x01, /* ...0. .. */
  0x55, 0xf9, 0x1c, 0x0d, 0xaf, 0x05, 0x27, 0x17, /* U.....'. */
  0xdf, 0xff, 0x21, 0x47, 0xbf, 0x18, 0xdf, 0x02, /* ..!G.... */
  0x2e, 0xb4, 0x87, 0x03, 0x5b, 0x0e, 0x46, 0x04, /* ....[.F. */
  0x5a, 0x02, 0x2e, 0x71, 0x79, 0x0d, 0x43, 0x00, /* Z..qy.C. */
  0x5a, 0xff, 0x21, 0x48, 0xbf, 0x05, 0x27, 0x19, /* Z.!H..'. */
  0xcf, 0x80, 0x1e, 0x01, 0x54, 0x5f, 0x20, 0x19, /* ....T_ . */
  0xaf, 0x05, 0x27, 0x19, 0xcf, 0x7f, 0x1e, 0x01, /* ..'..... */
  0x54, 0x1f, 0x20, 0x19, 0xaf, 0x05, 0x27, 0x13, /* T. ...'. */
  0xcf, 0x16, 0xaf, 0x15, 0xcf, 0x18, 0xaf, 0x14, /* ........ */
  0xcf, 0x17, 0xaf, 0x0d, 0x3d, 0x0d, 0x46, 0x00, /* ....=.F. */
  0x5b, 0x0d, 0x35, 0x05, 0x27, 0x14, 0x23, 0x15, /* [.5.'.#. */
  0x23, 0xff, 0xef, 0x13, 0xaf, 0x02, 0x2e, 0x98, /* #....... */
  0x89, 0x0d, 0x3e, 0x13, 0xcf                  /* ..>.. */
};

static const unsigned char pkt243[] = {
  0x42, 0xc0, 0x00,
  0x93, 0x29, 0x46, 0xff, 0xbf, /* B...)F.. */
  0x03, 0x00, 0x53, 0x0d, 0x36, 0x05, 0x27, 0x0d, /* ..S.6.'. */
  0xcf, 0x06, 0x1c, 0x06, 0x1e, 0x07, 0x55, 0xe4, /* ......U. */
  0x47, 0x02, 0x5a, 0x1a, 0xcf, 0x02, 0x2e, 0xa5, /* G.Z..... */
  0x79, 0x1b, 0xcf, 0x02, 0x2e, 0xa5, 0x79, 0x18, /* y.....y. */
  0xcf, 0x00, 0x1e, 0x04, 0x55, 0x5d, 0xef, 0x1b, /* ....U].. */
  0xff, 0x02, 0x2e, 0xc0, 0x79, 0x18, 0xcf, 0x01, /* ....y... */
  0x1e, 0x03, 0x55, 0x5e, 0xef, 0x1b, 0xff, 0x02, /* ..U^.... */
  0x2e, 0xc0, 0x79, 0x02, 0x1e, 0x03, 0x55, 0x61, /* ..y...Ua */
  0xef, 0x1b, 0xff, 0x02, 0x2e, 0xc0, 0x79, 0x03, /* ......y. */
  0x1e, 0x03, 0x55, 0x66, 0xef, 0x1b, 0xff, 0x02, /* ..Uf.... */
  0x2e, 0xc0, 0x79, 0x6d, 0xef, 0x1b, 0xff, 0x1a, /* ..ym.... */
  0x27, 0x07, 0x27, 0xe6, 0xaf, 0xe9, 0xbf, 0x7e, /* '.'....~ */
  0x81, 0xec, 0xcf, 0x13, 0xdf, 0xbf, 0x03, 0x19, /* ........ */
  0x54, 0x0a, 0x5e, 0x13, 0xaf, 0xe9, 0xcf, 0x0f, /* T.^..... */
  0x1c, 0x15, 0xaf, 0xe9, 0xcf, 0x09, 0x27, 0x0f, /* ......'. */
  0x1c, 0xdf, 0x2c, 0x00, 0x5b, 0xf0, 0x18, 0x14, /* ..,.[... */
  0xaf, 0x01, 0x0e, 0x07, 0x27, 0x88, 0x1f, 0xe8, /* ....'... */
  0x55, 0x0d, 0x3e, 0x13, 0xcf, 0x46, 0x1e, 0x04, /* U.>..F.. */
  0x53, 0x0d, 0x36, 0x13, 0xcf, 0x46, 0x1e, 0x00, /* S.6..F.. */
  0x5e, 0x0d, 0x36, 0x05, 0x27, 0xe9, 0xcf, 0x09, /* ^.6.'... */
  0x27, 0xdf, 0x2c, 0x00, 0x5b, 0xdf, 0x21, 0x0f, /* '.,.[.!. */
  0x1c, 0x14, 0xdf, 0xff, 0x2d, 0x00, 0x5b, 0xff, /* ....-.[. */
  0x21, 0xbf, 0x03, 0xe5, 0x53, 0xec, 0xcf, 0x02, /* !...S... */
  0x2e, 0xca, 0x79                              /* ..y */
};

static const unsigned char pkt245[] = {
  0x42, 0x66, 0x00,
  0x00, 0x2c, 0x01, 0x2e, 0x65, /* Bf..,..e */
  0x84, 0x08, 0x44, 0x2c, 0x5a, 0xff, 0x0d, 0x48, /* ..D,Z..H */
  0xef, 0x5f, 0x9a, 0x2a, 0xaf, 0x02, 0x2e, 0x75, /* ._.*...u */
  0x80, 0xdf, 0x96, 0x26, 0x40, 0x07, 0x5b, 0xa0, /* ...&@.[. */
  0xcf, 0x30, 0x1c, 0x04, 0x54, 0x20, 0x1e, 0x01, /* .0..T .. */
  0x55, 0x31, 0xff, 0x01, 0x50, 0x00, 0xff, 0x30, /* U1..P..0 */
  0x19, 0xbf, 0x9a, 0x2b, 0xbf, 0xa9, 0xcf, 0x70, /* ...+...p */
  0x1c, 0xa5, 0xdf, 0x0c, 0x1d, 0xff, 0x20, 0xff, /* ...... . */
  0x20, 0x5f, 0x02, 0xff, 0x9a, 0x2c, 0xbf, 0xdd, /*  _...,.. */
  0x97, 0x1f, 0x9b, 0x2d, 0xaf, 0x29, 0xcf, 0x09, /* ...-.).. */
  0x27, 0x28, 0xdf, 0x3f, 0x02, 0x9f, 0x9c, 0x6a, /* '(.?...j */
  0xcf, 0x09, 0x27, 0x69, 0xdf, 0x3f, 0x02, 0xdf, /* ..'i.?.. */
  0x9c, 0x6d, 0x2e, 0x6d, 0xdf, 0x1f, 0x9d, 0x05, /* .m.m.... */
  0x27                                          /* ' */
};

static const unsigned char pkt247[] = {
  0x42, 0x2c, 0x00,
  0x32, 0x2c, 0x08, 0x44, 0x11, /* B,.2,.D. */
  0x5a, 0x3f, 0x80, 0x28, 0x0d, 0xe2, 0x97, 0xdf, /* Z?.(.... */
  0x96, 0xe1, 0x97, 0xdf, 0x96, 0x01, 0x2e, 0xbf, /* ........ */
  0x84, 0xfd, 0x0d, 0x29, 0xcf, 0x09, 0x27, 0x28, /* ...)..'( */
  0xdf, 0x3f, 0x02, 0x5f, 0x9f, 0xdd, 0x97, 0x5f, /* .?._..._ */
  0x9f, 0xdd, 0x97, 0x5f, 0x9f, 0x05, 0x27      /* ..._..' */
};

static const unsigned char pkt249[] = {
  0x42, 0xca, 0x00,
  0x47, 0x2c, 0xfe, 0xcf, 0x58, /* B..G,..X */
  0xaf, 0xfe, 0xcf, 0xfe, 0xdf, 0x02, 0x2e, 0xde, /* ........ */
  0x8c, 0xfe, 0xcf, 0x3c, 0xaf, 0x07, 0x3d, 0x58, /* ...<..=X */
  0x40, 0x4e, 0x5a, 0x3c, 0x26, 0x4c, 0x55, 0x37, /* @NZ<&LU7 */
  0x45, 0x4a, 0x5a, 0x02, 0x2e, 0x5c, 0x8d, 0x47, /* EJZ..\.G */
  0x5a, 0x02, 0x2e, 0xee, 0x8c, 0x00, 0x2e, 0xb9, /* Z....... */
  0x8d, 0xfe, 0xef, 0x16, 0x80, 0x02, 0xcf, 0xdf, /* ........ */
  0x96, 0x61, 0xef, 0xc0, 0xaf, 0x04, 0x27, 0xc1, /* .a....'. */
  0x46, 0x04, 0x5b, 0xc0, 0x3e, 0x40, 0xef, 0xc1, /* F.[.>@.. */
  0xaf, 0x01, 0x2e, 0x8d, 0x8b, 0xd0, 0x41, 0x07, /* ......A. */
  0x5b, 0x02, 0x2e, 0x9e, 0x8c, 0xf2, 0x5b, 0xae, /* [.....[. */
  0x87, 0xdd, 0x97, 0x1f, 0x2d, 0x06, 0x5b, 0x03, /* ....-.[. */
  0x50, 0x01, 0x0c, 0x02, 0xcf, 0xc0, 0x1c, 0x01, /* P....... */
  0x55, 0x02, 0x2e, 0x81, 0x8c, 0xc0, 0x23, 0x20, /* U.....#  */
  0xef, 0xc1, 0xaf, 0x00, 0x2e, 0xd8, 0x8d, 0x05, /* ........ */
  0x27, 0x9b, 0x22, 0x37, 0x45, 0x05, 0x5a, 0x41, /* '."7E.ZA */
  0x2e, 0x52, 0xdf, 0xdf, 0x2d, 0x04, 0x5b, 0x00, /* .R..-.[. */
  0x28, 0x02, 0x50, 0x3b, 0x43, 0x00, 0x5b, 0x00, /* (.P;C.[. */
  0x28, 0xd3, 0xdf, 0xc6, 0x2e, 0x15, 0xbf, 0xd3, /* (....... */
  0x3c, 0x40, 0xef, 0x09, 0x80, 0xdd, 0x97, 0xff, /* <@...... */
  0xaf, 0x00, 0x2e, 0xa7, 0x8a, 0xc6, 0x2e, 0x15, /* ........ */
  0xdf, 0x10, 0x1d, 0xd3, 0xcf, 0x3f, 0x02, 0xd3, /* .....?.. */
  0xaf, 0x05, 0x27, 0xed, 0x83, 0x07, 0x45, 0x05, /* ..'...E. */
  0x27, 0x3d, 0x23, 0x07, 0x3d, 0x02, 0xcf, 0xdf, /* '=#.=... */
  0x96, 0x02, 0x2e, 0xee, 0x8c, 0x02, 0x2e, 0xd7, /* ........ */
  0x8c, 0xfe, 0xef, 0x16, 0x80                  /* ..... */
};

static const unsigned char pkt251[] = {
  0x42, 0xca, 0x00,
  0xab, 0x2c, 0x05, 0xef, 0xc0, /* B...,... */
  0xaf, 0x04, 0x27, 0xd0, 0x41, 0x03, 0x5b, 0x02, /* ..'.A.[. */
  0x2e, 0x9e, 0x8c, 0xf9, 0x5b, 0xbb, 0x50, 0x27, /* ....[.P' */
  0x82, 0x04, 0xef, 0xc1, 0xaf, 0x9b, 0x22, 0x02, /* ......". */
  0x2e, 0x12, 0x8d, 0xdd, 0x97, 0xdf, 0x22, 0x02, /* ......". */
  0x55, 0x58, 0x40, 0x02, 0x5a, 0x0b, 0x50, 0x58, /* UX@.Z.PX */
  0x40, 0x09, 0x5a, 0xdd, 0x97, 0x1f, 0x2d, 0xb3, /* @.Z...-. */
  0x5a, 0x02, 0x2e, 0x5d, 0x8d, 0xb2, 0x5b, 0x9b, /* Z..]..[. */
  0x22, 0x02, 0x2e, 0x8d, 0x8c, 0xae, 0x50, 0x02, /* ".....P. */
  0x2e, 0x5d, 0x8d, 0xd5, 0x5b, 0x02, 0x2e, 0x81, /* .]..[... */
  0x8c, 0x58, 0x41, 0x02, 0x5b, 0x01, 0x0c, 0x02, /* .XA.[... */
  0x2e, 0x7b, 0x7c, 0xcd, 0x50, 0x41, 0x2e, 0x52, /* .{|.PA.R */
  0xdf, 0xed, 0x1c, 0x04, 0x18, 0x41, 0x2e, 0x52, /* .....A.R */
  0xbf, 0x05, 0x27, 0x1a, 0x27, 0x56, 0xdf, 0x59, /* ..'.'V.Y */
  0x80, 0xdf, 0x22, 0x00, 0x54, 0x01, 0x16, 0x3f, /* ..".T..? */
  0x1e, 0x00, 0x5f, 0x3f, 0xef, 0x1f, 0x22, 0x1f, /* .._?..". */
  0x22, 0xaf, 0xdf, 0x03, 0x1d, 0x5f, 0x02, 0xaf, /* "...._.. */
  0xbf, 0x05, 0x27, 0x00, 0x2e, 0xa1, 0x8d, 0xae, /* ..'..... */
  0xdf, 0xe0, 0x19, 0xae, 0xbf, 0x57, 0x40, 0x00, /* .....W@. */
  0x5b, 0xa1, 0x32, 0x57, 0x43, 0x00, 0x5b, 0xae, /* [.2WC.[. */
  0x3f, 0xed, 0x80, 0x05, 0x27, 0x41, 0x2e, 0x52, /* ?...'A.R */
  0xdf, 0x03, 0x18, 0x41, 0x2e, 0x52, 0xbf, 0x05, /* ...A.R.. */
  0x27, 0xd3, 0xdf, 0xff, 0x96, 0x9b, 0x22, 0x02, /* '.....". */
  0x2e, 0x12, 0x8d, 0xd3, 0x3c, 0x07, 0x80, 0xdd, /* ....<... */
  0x97, 0xff, 0xaf, 0x00, 0x2e, 0xa7, 0x8a, 0xdd, /* ........ */
  0x9f, 0x10, 0x1d, 0xd3, 0xcf                  /* ..... */
};

static const unsigned char pkt253[] = {
  0x42, 0xb4, 0x00,
  0x0f, 0x2d, 0x3f, 0x02, 0xd3, /* B...-?.. */
  0xaf, 0x05, 0x27, 0x37, 0x45, 0x10, 0x5a, 0x02, /* ..'7E.Z. */
  0x2e, 0x5c, 0x8d, 0x0d, 0x5a, 0x02, 0x2e, 0xee, /* .\..Z... */
  0x8c, 0x00, 0x2e, 0xbf, 0x8d, 0x00, 0x2e, 0xd3, /* ........ */
  0x8d, 0x41, 0x2e, 0x52, 0xdf, 0xdf, 0x2d, 0x00, /* .A.R..-. */
  0x5b, 0x00, 0x28, 0x00, 0x2e, 0xd8, 0x8d, 0x05, /* [.(..... */
  0x27, 0xf9, 0xcf, 0xdf, 0x96, 0xf9, 0x3d, 0x01, /* '.....=. */
  0x2e, 0x06, 0x82, 0x2d, 0xff, 0xa7, 0xef, 0xe1, /* ...-.... */
  0xbf, 0xe0, 0xaf, 0x09, 0xef, 0x10, 0xaf, 0x01, /* ........ */
  0x2e, 0x94, 0x86, 0xed, 0x80, 0x01, 0x2e, 0x18, /* ........ */
  0x87, 0x01, 0x2e, 0x1a, 0x87, 0x00, 0x2e, 0x16, /* ........ */
  0x82, 0x5b, 0x82, 0x00, 0x2e, 0x2d, 0x82, 0x26, /* .[...-.& */
  0x37, 0x00, 0x5a, 0x0c, 0x50, 0xff, 0x96, 0xdf, /* 7.Z.P... */
  0x96, 0x79, 0xcf, 0x7a, 0xdf, 0xff, 0x96, 0xdf, /* .y.z.... */
  0x96, 0xc2, 0x97, 0xc3, 0x9f, 0x9d, 0x01, 0x5d, /* .......] */
  0x01, 0x07, 0x53, 0xdd, 0x97, 0xdd, 0x9f, 0x79, /* ..S....y */
  0xaf, 0x7a, 0xbf, 0x3b, 0x3b, 0xdd, 0x97, 0xf9, /* .z.;;... */
  0xaf, 0x05, 0x27, 0xff, 0x22, 0x01, 0x55, 0x32, /* ..'.".U2 */
  0x1e, 0xf3, 0x5f, 0x02, 0x0c, 0x02, 0x2e, 0x4f, /* .._....O */
  0x8d, 0x3b, 0x33, 0x00, 0x28, 0x05, 0x27, 0x0d, /* .;3.(.'. */
  0x27, 0x3c, 0x26, 0x07, 0x54, 0x3c, 0xcf, 0x3d, /* '<&.T<.= */
  0xdf, 0x3f, 0x20, 0x3d, 0xbf, 0xdf, 0x03, 0x01, /* .? =.... */
  0x52, 0x3d, 0x23, 0x0c, 0x27, 0x0d, 0x27      /* R=#.'.' */
};

static const unsigned char pkt255[] = {
  0x42, 0x1a, 0x00,
  0x68, 0x2d, 0x09, 0x43, 0x04, /* B..h-.C. */
  0x5b, 0x00, 0xef, 0xf8, 0xaf, 0x32, 0xef, 0xfa, /* [....2.. */
  0xaf, 0x03, 0x50, 0x01, 0xef, 0xf8, 0xaf, 0x42, /* ..P....B */
  0xef, 0xfa, 0xaf, 0x05, 0x27                  /* ....' */
};

static const unsigned char pkt257[] = {
  0x42, 0x26, 0x00,
  0x74, 0x2d, 0xd3, 0x45, 0xfd, /* B&.t-.E. */
  0x5a, 0x0f, 0x27, 0xd3, 0x45, 0xfa, 0x5a, 0x0f, /* Z.'.E.Z. */
  0x27, 0xd3, 0x45, 0xf7, 0x5a, 0x0f, 0x27, 0x41, /* '.E.Z.'A */
  0x2e, 0xa1, 0xdf, 0x9f, 0x28, 0x41, 0x2e, 0xa1, /* ....(A.. */
  0xbf, 0x02, 0x2e, 0x58, 0x80, 0x00, 0x2e, 0xcb, /* ...X.... */
  0x7a                                          /* z */
};

static const unsigned char pkt259[] = {
  0x42, 0x16, 0x00,
  0x86, 0x2d, 0xd3, 0x45, 0x02, /* B...-.E. */
  0x5a, 0xd3, 0x45, 0x00, 0x5a, 0x05, 0x27, 0xd0, /* Z.E.Z.'. */
  0x41, 0xf8, 0x5b, 0xed, 0x83, 0xf6, 0x50, 0x05, /* A.[...P. */
  0x27                                          /* ' */
};

static const unsigned char pkt261[] = {
  0x42, 0x4e, 0x00,
  0x90, 0x2d, 0x07, 0x3d, 0x02, /* BN..-.=. */
  0x5b, 0xae, 0x87, 0xc2, 0x23, 0xd3, 0x3c, 0x02, /* [...#.<. */
  0x44, 0xd4, 0x2f, 0xf8, 0x5a, 0x02, 0x45, 0xd5, /* D./.Z.E. */
  0x2f, 0x59, 0x5b, 0x58, 0x23, 0x3c, 0x26, 0x02, /* /Y[X#<&. */
  0x55, 0x02, 0x2e, 0x4f, 0x8c, 0x00, 0x50, 0xff, /* U..O..P. */
  0x8e, 0x07, 0x45, 0xeb, 0x5a, 0x41, 0x2e, 0x53, /* ..E.ZA.S */
  0xdf, 0xc1, 0x2e, 0x79, 0xdf, 0xff, 0x96, 0xdf, /* ...y.... */
  0x96, 0x41, 0x2e, 0x55, 0xdf, 0xc1, 0x2e, 0x78, /* .A.U...x */
  0xdf, 0x9d, 0x01, 0x5d, 0x01, 0xff, 0x22, 0x00, /* ...]..". */
  0x54, 0xff, 0xef, 0x2e, 0xaf, 0xd5, 0x2f, 0x3e, /* T...../> */
  0x50                                          /* P */
};

static const unsigned char pkt263[] = {
  0x42, 0xca, 0x00,
  0xb6, 0x2d, 0xfa, 0xcf, 0xdf, /* B...-... */
  0x96, 0x37, 0xcf, 0xdf, 0x96, 0x08, 0xcf, 0xdf, /* .7...... */
  0x96, 0x3d, 0xcf, 0xdf, 0x96, 0x77, 0xcf, 0xdf, /* .=...w.. */
  0x96, 0x26, 0xcf, 0xdf, 0x96, 0x26, 0x23, 0x01, /* .&...&#. */
  0x2e, 0x08, 0x82, 0xdd, 0x97, 0x26, 0xaf, 0xdd, /* .....&.. */
  0x97, 0x77, 0xaf, 0xdd, 0x97, 0x3d, 0xaf, 0xdd, /* .w...=.. */
  0x97, 0x08, 0xaf, 0xdd, 0x97, 0x37, 0xaf, 0xdd, /* .....7.. */
  0x97, 0xfa, 0xaf, 0x9f, 0x22, 0x6d, 0x2e, 0x6d, /* ...."m.m */
  0xbf, 0x26, 0x40, 0x3a, 0x5b, 0x73, 0x43, 0x08, /* .&@:[sC. */
  0x5b, 0x73, 0xcf, 0x09, 0x27, 0x0f, 0x1c, 0x10, /* [s..'... */
  0xaf, 0x82, 0xcf, 0x01, 0x2e, 0xf9, 0x81, 0xed, /* ........ */
  0x80, 0x05, 0x50, 0x81, 0xcf, 0x10, 0xaf, 0x82, /* ..P..... */
  0xcf, 0x01, 0x2e, 0xf9, 0x81, 0xed, 0x80, 0xc8, /* ........ */
  0xef, 0x0f, 0xaf, 0x08, 0x31, 0x31, 0xef, 0x0a, /* ....11.. */
  0xaf, 0x9f, 0x22, 0x0d, 0xaf, 0x81, 0xcf, 0x10, /* .."..... */
  0xaf, 0x01, 0x2e, 0x94, 0x86, 0x82, 0xcf, 0x04, /* ........ */
  0x16, 0x01, 0x2e, 0xf9, 0x81, 0xed, 0x80, 0x10, /* ........ */
  0xcf, 0xb0, 0xff, 0x3f, 0x00, 0x00, 0xff, 0x00, /* ...?.... */
  0x13, 0x1a, 0x27, 0xd6, 0x97, 0x82, 0xdf, 0xbf, /* ..'..... */
  0x01, 0x6d, 0x2e, 0x6d, 0xbf, 0x74, 0xcf, 0x02, /* .m.m.t.. */
  0x1e, 0x04, 0x55, 0xa0, 0xcf, 0xcf, 0x1c, 0x10, /* ..U..... */
  0x18, 0xa0, 0xaf, 0x05, 0x50, 0x01, 0x1e, 0x04, /* ....P... */
  0x55, 0xa0, 0xcf, 0xcf, 0x1c, 0x20, 0x18, 0xa0, /* U.... .. */
  0xaf, 0xed, 0x80, 0xfb, 0x39, 0x08, 0x45, 0x00, /* ....9.E. */
  0x5a, 0xfb, 0x31, 0x0a, 0x42, 0x00, 0x5b, 0xa2, /* Z.1.B.[. */
  0x39, 0x02, 0xcf, 0x5c, 0xaf                  /* 9..\. */
};

static const unsigned char pkt265[] = {
  0x42, 0x96, 0x00,
  0x1a, 0x2e, 0xe1, 0xef, 0x95, /* B....... */
  0xaf, 0x9f, 0x22, 0x61, 0xaf, 0x62, 0xaf, 0x69, /* .."a.b.i */
  0xaf, 0x6a, 0xaf, 0x16, 0xaf, 0x96, 0xaf, 0x29, /* .j.....) */
  0xaf, 0x28, 0xaf, 0xc1, 0x2e, 0x49, 0xdf, 0x3f, /* .(...I.? */
  0x2d, 0x03, 0x5b, 0xa0, 0xcf, 0x8f, 0x1c, 0x20, /* -.[....  */
  0x18, 0xa0, 0xaf, 0xa1, 0xcf, 0xdf, 0x1c, 0xa1, /* ........ */
  0xaf, 0xed, 0x80, 0x02, 0x47, 0x06, 0x5a, 0x0a, /* ....G.Z. */
  0x41, 0x04, 0x5a, 0x0a, 0xdf, 0x30, 0x1d, 0x30, /* A.Z..0.0 */
  0x1f, 0x00, 0x54, 0x01, 0xef, 0x44, 0xaf, 0x09, /* ..T..D.. */
  0xef, 0x60, 0xaf, 0xa7, 0xef, 0x40, 0xaf, 0x2d, /* .`...@.- */
  0xef, 0x41, 0xaf, 0x0a, 0x40, 0x06, 0x5a, 0x10, /* .A..@.Z. */
  0xcf, 0x27, 0xaf, 0xe2, 0x23, 0x6e, 0x23, 0x08, /* .'..#n#. */
  0xef, 0xe3, 0xaf, 0x6f, 0xaf, 0xe2, 0x23, 0x08, /* ...o..#. */
  0xef, 0xe3, 0xaf, 0x79, 0x82, 0x4d, 0xaf, 0xdf, /* ...y.M.. */
  0x96, 0x64, 0xdf, 0x8b, 0x82, 0x93, 0xbf, 0xdd, /* .d...... */
  0x97, 0x77, 0xff, 0x8b, 0x82, 0x94, 0xbf, 0x9e, /* .w...... */
  0x23, 0xe4, 0x23, 0x9f, 0x22, 0x6d, 0x2e, 0x6b, /* #.#."m.k */
  0xbf, 0x6d, 0x2e, 0x6a, 0xbf, 0x6d, 0x2e, 0x6c, /* .m.j.m.l */
  0xbf, 0xa5, 0xcf, 0xf3, 0x1c, 0xa5, 0xaf, 0x05, /* ........ */
  0x27                                          /* ' */
};

static const unsigned char pkt267[] = {
  0x42, 0x2c, 0x00,
  0x64, 0x2e, 0xf9, 0x32, 0xf9, /* B,.d..2. */
  0x3a, 0xaf, 0xef, 0x00, 0xff, 0x1b, 0x27, 0xf6, /* :.....'. */
  0x97, 0xcf, 0xaf, 0x0f, 0xef, 0x06, 0xaf, 0xce, /* ........ */
  0x30, 0x0f, 0x27, 0x0f, 0x27, 0xfa, 0x97, 0xcf, /* 0.'.'... */
  0xaf, 0xce, 0x32, 0x06, 0x25, 0xf8, 0x55, 0xce, /* ..2.%.U. */
  0x43, 0xfd, 0x5b, 0xce, 0x31, 0x05, 0x27      /* C.[.1.' */
};

static const unsigned char pkt269[] = {
  0x42, 0x26, 0x00,
  0x79, 0x2e, 0xbf, 0x22, 0xff, /* B&.y..". */
  0xef, 0x1b, 0x27, 0xdd, 0x97, 0xdd, 0x9f, 0x06, /* ..'..... */
  0xaf, 0x4e, 0xbf, 0xff, 0x22, 0x04, 0x54, 0xbd, /* .N..".T. */
  0x9d, 0x06, 0x25, 0xfc, 0x53, 0x4e, 0x25, 0xfa, /* ..%.SN%. */
  0x55, 0xbd, 0x9d, 0x06, 0x25, 0xfc, 0x55, 0x05, /* U...%.U. */
  0x27                                          /* ' */
};

static const unsigned char pkt271[] = {
  0x42, 0x3c, 0x00,
  0x8b, 0x2e, 0xbf, 0x22, 0xff, /* B<....". */
  0xef, 0x1b, 0x27, 0xdc, 0x99, 0x28, 0x28, 0x28, /* ..'..((( */
  0x2a, 0x68, 0x2c, 0xfd, 0x5b, 0xa9, 0x9d, 0x6a, /* *h,.[..j */
  0x28, 0x06, 0x25, 0xf6, 0x53, 0x4e, 0x25, 0xf4, /* (.%.SN%. */
  0x55, 0xdc, 0x99, 0x28, 0x28, 0x28, 0x2a, 0x68, /* U..(((*h */
  0x2c, 0xfd, 0x5b, 0xa9, 0x9d, 0x6a, 0x28, 0x06, /* ,.[..j(. */
  0x25, 0xf6, 0x55, 0x01, 0x2e, 0xe4, 0x85, 0xbf, /* %.U..... */
  0x22, 0xbf, 0x99, 0x7f, 0x99, 0x05, 0x27      /* ".....' */
};

static const unsigned char pkt273[] = {
  0x67, 0x05, 0x00, 0x00, 0x02, 0x2e, 0x00, 0x70, /* g......p */
  0x67, 0x05, 0x00, 0x02, 0x02, 0x2e, 0x88, 0x70, /* g......p */
  0x67, 0x05, 0x00, 0x01, 0x02, 0x2e, 0x3c, 0x71, /* g.....<q */
  0x67, 0x05, 0x00, 0x03, 0x02, 0x2e, 0xcf, 0x71, /* g......q */
  0x5a, 0x05, 0x00, 0x00, 0x94, 0x16, 0xef, 0x22, /* Z......" */
  0x67, 0x05, 0x00, 0x0b, 0x02, 0x2e, 0x31, 0x73, /* g.....1s */
  0x67, 0x05, 0x00, 0x05, 0x02, 0x2e, 0x6e, 0x74, /* g.....nt */
  0x67, 0x05, 0x00, 0x0c, 0x02, 0x2e, 0x03, 0x78, /* g......x */
  0x67, 0x05, 0x00, 0x0d, 0x02, 0x2e, 0x00, 0x7c, /* g......| */
  0x67, 0x05, 0x00, 0x0e, 0x02, 0x2e, 0x32, 0x7c, /* g.....2| */
  0x67, 0x05, 0x00, 0x07, 0x02, 0x2e, 0x47, 0x7c, /* g.....G| */
  0x5a, 0x05, 0x00, 0x01, 0x55, 0x01, 0x68, 0x2d, /* Z...U.h- */
  0x5a, 0x05, 0x00, 0x02, 0xc4, 0x0a, 0x74, 0x2d, /* Z.....t- */
  0x5a, 0x05, 0x00, 0x03, 0xcc, 0x0a, 0x86, 0x2d, /* Z......- */
  0x5a, 0x05, 0x00, 0x04, 0xd9, 0x02, 0x90, 0x2d, /* Z......- */
  0x67, 0x05, 0x00, 0x08, 0x02, 0x2e, 0xb6, 0x7d, /* g......} */
  0x5a, 0x05, 0x00, 0x05, 0xed, 0x00, 0x64, 0x2e, /* Z.....d. */
  0x5a, 0x05, 0x00, 0x06, 0xa9, 0x15, 0x79, 0x2e, /* Z.....y. */
  0x5a, 0x05, 0x00, 0x07, 0x8c, 0x15, 0x8b, 0x2e, /* Z....... */
  0x0d,
  0x42, 0x04, 0x00,
  0x8b, 0x2e, 0xbf, 0x22 /* .B....." */
};

/* INIT2 is over, 0x07 cmd returns
 * { 0x07, 0x05, 0x00, 0x91, 0x26, 0x21, 0x00, 0x7a }
 */
struct aesX660_cmd aes2660_init_2[] = {
  { .cmd = pkt187, .len = sizeof (pkt187) },
  { .cmd = pkt189, .len = sizeof (pkt189) },
  { .cmd = pkt191, .len = sizeof (pkt191) },
  { .cmd = pkt193, .len = sizeof (pkt193) },
  { .cmd = pkt195, .len = sizeof (pkt195) },
  { .cmd = pkt197, .len = sizeof (pkt197) },
  { .cmd = pkt199, .len = sizeof (pkt199) },
  { .cmd = pkt201, .len = sizeof (pkt201) },
  { .cmd = pkt203, .len = sizeof (pkt203) },
  { .cmd = pkt205, .len = sizeof (pkt205) },
  { .cmd = pkt207, .len = sizeof (pkt207) },
  { .cmd = pkt209, .len = sizeof (pkt209) },
  { .cmd = pkt211, .len = sizeof (pkt211) },
  { .cmd = pkt213, .len = sizeof (pkt213) },
  { .cmd = pkt215, .len = sizeof (pkt215) },
  { .cmd = pkt217, .len = sizeof (pkt217) },
  { .cmd = pkt219, .len = sizeof (pkt219) },
  { .cmd = pkt221, .len = sizeof (pkt221) },
  { .cmd = pkt223, .len = sizeof (pkt223) },
  { .cmd = pkt225, .len = sizeof (pkt225) },
  { .cmd = pkt227, .len = sizeof (pkt227) },
  { .cmd = pkt229, .len = sizeof (pkt229) },
  { .cmd = pkt231, .len = sizeof (pkt231) },
  { .cmd = pkt233, .len = sizeof (pkt233) },
  { .cmd = pkt235, .len = sizeof (pkt235) },
  { .cmd = pkt237, .len = sizeof (pkt237) },
  { .cmd = pkt239, .len = sizeof (pkt239) },
  { .cmd = pkt241, .len = sizeof (pkt241) },
  { .cmd = pkt243, .len = sizeof (pkt243) },
  { .cmd = pkt245, .len = sizeof (pkt245) },
  { .cmd = pkt247, .len = sizeof (pkt247) },
  { .cmd = pkt249, .len = sizeof (pkt249) },
  { .cmd = pkt251, .len = sizeof (pkt251) },
  { .cmd = pkt253, .len = sizeof (pkt253) },
  { .cmd = pkt255, .len = sizeof (pkt255) },
  { .cmd = pkt257, .len = sizeof (pkt257) },
  { .cmd = pkt259, .len = sizeof (pkt259) },
  { .cmd = pkt261, .len = sizeof (pkt261) },
  { .cmd = pkt263, .len = sizeof (pkt263) },
  { .cmd = pkt265, .len = sizeof (pkt265) },
  { .cmd = pkt267, .len = sizeof (pkt267) },
  { .cmd = pkt269, .len = sizeof (pkt269) },
  { .cmd = pkt271, .len = sizeof (pkt271) },
  { .cmd = pkt273, .len = sizeof (pkt273) },
};

/* This command sequence starts imaging */
static const unsigned char aes2660_start_imaging_cmd[] = {
  0x13,
  0x20,
  0x4c, 0x01, 0x00, 0x00,
  0x4b, 0x04, 0x00, 0x78, 0x56, 0x34, 0x12,
  0x55, 0x07, 0x00, 0x80, 0x42, 0x00, 0xbf, 0x00, 0x00, 0x18,
  0x49, 0x03, 0x00, 0x20, 0x08, 0xc8
};
