"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getMessage = exports.quote = exports.getElementText = exports.detectExpectType = void 0;
const ExpectTypePage = "Page";
const ExpectTypeElementHandle = "ElementHandle";
const detectExpectType = (value) => {
    const className = value.constructor.name;
    switch (className) {
        case "Page":
            return ExpectTypePage;
        case "ElementHandle":
            return ExpectTypeElementHandle;
        default:
            throw new Error(`could not recognize type: ${className}`);
    }
};
exports.detectExpectType = detectExpectType;
const lastElementHasType = (args, type) => typeof args[args.length - 1] === type;
const getSelectorOptions = (args) => {
    let selectorOptions = undefined;
    if (args.length === 3 && lastElementHasType(args, "object")) {
        selectorOptions = args[2];
    }
    if (args.length === 4 && lastElementHasType(args, "object")) {
        selectorOptions = args[3];
    }
    return selectorOptions;
};
const getElementText = async (...args) => {
    if (args.length > 1) {
        const type = exports.detectExpectType(args[0]);
        /**
         * Handle the following cases:
         * - expect(page).foo("bar")
         * - expect(element).foo("bar")
         */
        if (args.length === 2) {
            if (type === ExpectTypeElementHandle) {
                const iframe = await args[0].contentFrame();
                const elem = iframe ? await iframe.$("body") : args[0];
                return {
                    elementHandle: elem,
                    expectedValue: args[1],
                };
            }
            const page = args[0];
            return {
                elementHandle: (await page.$("body")),
                expectedValue: args[1],
            };
        }
        /**
         * Handle the following case:
         * - expect(page).foo("#foo", "bar")
         */
        const selector = args[1];
        if (type === ExpectTypePage) {
            const page = args[0];
            const selectorOptions = getSelectorOptions(args);
            try {
                await page.waitForSelector(selector, selectorOptions);
            }
            catch (err) {
                throw new Error(`Timeout exceed for element ${exports.quote(selector)}`);
            }
            return {
                elementHandle: (await page.$(selector)),
                expectedValue: args[2],
            };
        }
        if (type === ExpectTypeElementHandle) {
            const iframe = await args[0].contentFrame();
            const elem = iframe ? await iframe.$("body") : args[0];
            const selectorOptions = getSelectorOptions(args);
            try {
                await elem.waitForSelector(selector, selectorOptions);
            }
            catch (err) {
                throw new Error(`Timeout exceed for element ${exports.quote(selector)}`);
            }
            return {
                elementHandle: (await elem.$(selector)),
                expectedValue: args[2],
            };
        }
    }
    throw new Error(`Invalid input length: ${args.length}`);
};
exports.getElementText = getElementText;
const quote = (val) => (val === null ? "" : `'${val}'`);
exports.quote = quote;
const getMessage = ({ isNot, promise, utils }, matcher, expected, received) => {
    const message = isNot
        ? `Expected: not ${utils.printExpected(expected)}`
        : `Expected: ${utils.printExpected(expected)}\n` +
            `Received: ${utils.printReceived(received)}`;
    return (utils.matcherHint(matcher, undefined, undefined, { isNot, promise }) +
        "\n\n" +
        message);
};
exports.getMessage = getMessage;
