"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.removeLockFile = exports.printGrafanaTutorialsDetails = exports.formatPluginDetails = exports.removeGitFiles = exports.prepareJsonFiles = exports.fetchTemplate = exports.promptPluginDetails = exports.promptPluginType = exports.verifyGitExists = exports.getPluginKeywords = exports.getPluginId = exports.getPluginIdFromName = exports.getGitUsername = void 0;
const tslib_1 = require("tslib");
const chalk_1 = tslib_1.__importDefault(require("chalk"));
const command_exists_1 = tslib_1.__importDefault(require("command-exists"));
const fs_1 = require("fs");
const inquirer_1 = require("inquirer");
const lodash_1 = require("lodash");
const path_1 = tslib_1.__importDefault(require("path"));
const simple_git_1 = tslib_1.__importDefault(require("simple-git"));
const prompt_1 = require("../../utils/prompt");
const rmdir_1 = require("../../utils/rmdir");
const useSpinner_1 = require("../../utils/useSpinner");
const simpleGit = (0, simple_git_1.default)(process.cwd());
const PluginNames = {
    'panel-plugin': 'Grafana Panel Plugin',
    'datasource-plugin': 'Grafana Data Source Plugin',
    'backend-datasource-plugin': 'Grafana Backend Datasource Plugin',
};
const RepositoriesPaths = {
    'panel-plugin': 'https://github.com/grafana/simple-react-panel.git',
    'datasource-plugin': 'https://github.com/grafana/simple-datasource.git',
    'backend-datasource-plugin': 'https://github.com/grafana/simple-datasource-backend.git',
};
const TutorialPaths = {
    'panel-plugin': 'https://grafana.com/tutorials/build-a-panel-plugin',
    'datasource-plugin': 'https://grafana.com/tutorials/build-a-data-source-plugin',
    'backend-datasource-plugin': 'TODO',
};
const getGitUsername = () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const name = yield simpleGit.raw(['config', '--global', 'user.name']);
    return name || '';
});
exports.getGitUsername = getGitUsername;
const getPluginIdFromName = (name) => (0, lodash_1.kebabCase)(name);
exports.getPluginIdFromName = getPluginIdFromName;
const getPluginId = (pluginDetails) => `${(0, lodash_1.kebabCase)(pluginDetails.org)}-${(0, exports.getPluginIdFromName)(pluginDetails.name)}`;
exports.getPluginId = getPluginId;
const getPluginKeywords = (pluginDetails) => pluginDetails.keywords
    .split(',')
    .map((k) => k.trim())
    .filter((k) => k !== '');
exports.getPluginKeywords = getPluginKeywords;
const verifyGitExists = () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    return new Promise((resolve, reject) => {
        (0, command_exists_1.default)('git', (err, exists) => {
            if (exists) {
                resolve(true);
            }
            reject(new Error('git is not installed'));
        });
    });
});
exports.verifyGitExists = verifyGitExists;
const promptPluginType = () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    return (0, inquirer_1.prompt)([
        {
            type: 'list',
            message: 'Select plugin type',
            name: 'type',
            choices: [
                { name: 'Panel Plugin', value: 'panel-plugin' },
                { name: 'Datasource Plugin', value: 'datasource-plugin' },
                { name: 'Backend Datasource Plugin', value: 'backend-datasource-plugin' },
            ],
        },
    ]);
});
exports.promptPluginType = promptPluginType;
const promptPluginDetails = (name) => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const username = (yield (0, exports.getGitUsername)()).trim();
    const responses = yield (0, inquirer_1.prompt)([
        (0, prompt_1.promptInput)('name', 'Plugin name', true, name),
        (0, prompt_1.promptInput)('org', 'Organization (used as part of plugin ID)', true),
        (0, prompt_1.promptInput)('description', 'Description'),
        (0, prompt_1.promptInput)('keywords', 'Keywords (separated by comma)'),
        // Try using git specified username
        (0, prompt_1.promptConfirm)('author', `Author (${username})`, username, username !== ''),
        // Prompt for manual author entry if no git user.name specified
        (0, prompt_1.promptInput)('author', `Author`, true, undefined, (answers) => !answers.author || username === ''),
        (0, prompt_1.promptInput)('url', 'Your URL (i.e. organisation url)'),
    ]);
    return Object.assign(Object.assign({}, responses), { author: responses.author === true ? username : responses.author });
});
exports.promptPluginDetails = promptPluginDetails;
const fetchTemplate = ({ type, dest }) => (0, useSpinner_1.useSpinner)('Fetching plugin template...', () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const url = RepositoriesPaths[type];
    if (!url) {
        throw new Error('Unknown plugin type');
    }
    yield simpleGit.clone(url, dest);
}));
exports.fetchTemplate = fetchTemplate;
const prepareJsonFiles = ({ type, pluginDetails, pluginPath, }) => (0, useSpinner_1.useSpinner)('Saving package.json and plugin.json files', () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    const packageJsonPath = path_1.default.resolve(pluginPath, 'package.json');
    const pluginJsonPath = path_1.default.resolve(pluginPath, 'src/plugin.json');
    const packageJson = JSON.parse((0, fs_1.readFileSync)(packageJsonPath, 'utf8'));
    const pluginJson = JSON.parse((0, fs_1.readFileSync)(pluginJsonPath, 'utf8'));
    const pluginId = `${(0, lodash_1.kebabCase)(pluginDetails.org)}-${(0, exports.getPluginIdFromName)(pluginDetails.name)}`;
    packageJson.name = pluginId;
    packageJson.author = pluginDetails.author;
    packageJson.description = pluginDetails.description;
    pluginJson.name = pluginDetails.name;
    pluginJson.id = pluginId;
    if (type === 'backend-datasource-plugin') {
        pluginJson.backend = true;
        pluginJson.executable = 'gpx_' + pluginDetails.name;
    }
    pluginJson.info = Object.assign(Object.assign({}, pluginJson.info), { description: pluginDetails.description, author: {
            name: pluginDetails.author,
            url: pluginDetails.url,
        }, keywords: (0, exports.getPluginKeywords)(pluginDetails) });
    yield Promise.all([packageJson, pluginJson].map((f, i) => {
        const filePath = i === 0 ? packageJsonPath : pluginJsonPath;
        return fs_1.promises.writeFile(filePath, JSON.stringify(f, null, 2));
    }));
}));
exports.prepareJsonFiles = prepareJsonFiles;
const removeGitFiles = (pluginPath) => (0, useSpinner_1.useSpinner)('Cleaning', () => tslib_1.__awaiter(void 0, void 0, void 0, function* () { return (0, rmdir_1.rmdir)(`${path_1.default.resolve(pluginPath, '.git')}`); }));
exports.removeGitFiles = removeGitFiles;
/* eslint-disable no-console */
const formatPluginDetails = (details) => {
    console.group();
    console.log();
    console.log(chalk_1.default.bold.yellow('Your plugin details'));
    console.log('---');
    console.log(chalk_1.default.bold('Name: '), details.name);
    console.log(chalk_1.default.bold('ID: '), (0, exports.getPluginId)(details));
    console.log(chalk_1.default.bold('Description: '), details.description);
    console.log(chalk_1.default.bold('Keywords: '), (0, exports.getPluginKeywords)(details));
    console.log(chalk_1.default.bold('Author: '), details.author);
    console.log(chalk_1.default.bold('Organisation: '), details.org);
    console.log(chalk_1.default.bold('Website: '), details.url);
    console.log();
    console.groupEnd();
};
exports.formatPluginDetails = formatPluginDetails;
const printGrafanaTutorialsDetails = (type) => {
    console.group();
    console.log();
    console.log(chalk_1.default.bold.yellow(`Congrats! You have just created ${PluginNames[type]}.`));
    console.log('Please run `yarn install` to install frontend dependencies.');
    console.log();
    if (type !== 'backend-datasource-plugin') {
        console.log(`${PluginNames[type]} tutorial: ${TutorialPaths[type]}`);
    }
    console.log('Learn more about Grafana Plugins at https://grafana.com/docs/grafana/latest/plugins/developing/development/');
    console.log();
    console.groupEnd();
};
exports.printGrafanaTutorialsDetails = printGrafanaTutorialsDetails;
/* eslint-enable no-console */
const removeLockFile = ({ pluginPath }) => {
    const lockFilePath = path_1.default.resolve(pluginPath, 'yarn.lock');
    if ((0, fs_1.existsSync)(lockFilePath)) {
        (0, fs_1.unlinkSync)(lockFilePath);
    }
};
exports.removeLockFile = removeLockFile;
//# sourceMappingURL=create.js.map