"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MeterSharedState = void 0;
const InstrumentDescriptor_1 = require("../InstrumentDescriptor");
const Meter_1 = require("../Meter");
const utils_1 = require("../utils");
const AsyncMetricStorage_1 = require("./AsyncMetricStorage");
const MetricStorageRegistry_1 = require("./MetricStorageRegistry");
const MultiWritableMetricStorage_1 = require("./MultiWritableMetricStorage");
const ObservableRegistry_1 = require("./ObservableRegistry");
const SyncMetricStorage_1 = require("./SyncMetricStorage");
/**
 * An internal record for shared meter provider states.
 */
class MeterSharedState {
    constructor(_meterProviderSharedState, _instrumentationScope) {
        this._meterProviderSharedState = _meterProviderSharedState;
        this._instrumentationScope = _instrumentationScope;
        this._metricStorageRegistry = new MetricStorageRegistry_1.MetricStorageRegistry();
        this.observableRegistry = new ObservableRegistry_1.ObservableRegistry();
        this.meter = new Meter_1.Meter(this);
    }
    registerMetricStorage(descriptor) {
        const views = this._meterProviderSharedState.viewRegistry.findViews(descriptor, this._instrumentationScope);
        const storages = views
            .map(view => {
            const viewDescriptor = (0, InstrumentDescriptor_1.createInstrumentDescriptorWithView)(view, descriptor);
            const aggregator = view.aggregation.createAggregator(viewDescriptor);
            const storage = new SyncMetricStorage_1.SyncMetricStorage(viewDescriptor, aggregator, view.attributesProcessor);
            return this._metricStorageRegistry.register(storage);
        })
            .filter(utils_1.isNotNullish);
        if (storages.length === 1) {
            return storages[0];
        }
        return new MultiWritableMetricStorage_1.MultiMetricStorage(storages);
    }
    registerAsyncMetricStorage(descriptor) {
        const views = this._meterProviderSharedState.viewRegistry.findViews(descriptor, this._instrumentationScope);
        const storages = views
            .map(view => {
            const viewDescriptor = (0, InstrumentDescriptor_1.createInstrumentDescriptorWithView)(view, descriptor);
            const aggregator = view.aggregation.createAggregator(viewDescriptor);
            const viewStorage = new AsyncMetricStorage_1.AsyncMetricStorage(viewDescriptor, aggregator, view.attributesProcessor);
            return this._metricStorageRegistry.register(viewStorage);
        })
            .filter(utils_1.isNotNullish);
        return storages;
    }
    /**
     * @param collector opaque handle of {@link MetricCollector} which initiated the collection.
     * @param collectionTime the HrTime at which the collection was initiated.
     * @returns the list of metric data collected.
     */
    async collect(collector, collectionTime, options) {
        /**
         * 1. Call all observable callbacks first.
         * 2. Collect metric result for the collector.
         */
        const errors = await this.observableRegistry.observe(options === null || options === void 0 ? void 0 : options.timeoutMillis);
        const metricDataList = Array.from(this._metricStorageRegistry.getStorages())
            .map(metricStorage => {
            return metricStorage.collect(collector, this._meterProviderSharedState.metricCollectors, this._meterProviderSharedState.sdkStartTime, collectionTime);
        })
            .filter(utils_1.isNotNullish);
        return {
            scopeMetrics: {
                scope: this._instrumentationScope,
                metrics: metricDataList.filter(utils_1.isNotNullish),
            },
            errors,
        };
    }
}
exports.MeterSharedState = MeterSharedState;
//# sourceMappingURL=MeterSharedState.js.map