"use strict";
/*
 * Copyright The OpenTelemetry Authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.hostDetector = void 0;
const semantic_conventions_1 = require("@opentelemetry/semantic-conventions");
const Resource_1 = require("../../Resource");
const os_1 = require("os");
/**
 * HostDetector detects the resources related to the host current process is
 * running on. Currently only non-cloud-based attributes are included.
 */
class HostDetector {
    async detect(_config) {
        const attributes = {
            [semantic_conventions_1.SemanticResourceAttributes.HOST_NAME]: (0, os_1.hostname)(),
            [semantic_conventions_1.SemanticResourceAttributes.HOST_ARCH]: this._normalizeArch((0, os_1.arch)()),
        };
        return new Resource_1.Resource(attributes);
    }
    _normalizeArch(nodeArchString) {
        // Maps from https://nodejs.org/api/os.html#osarch to arch values in spec:
        // https://github.com/open-telemetry/opentelemetry-specification/blob/main/specification/resource/semantic_conventions/host.md
        switch (nodeArchString) {
            case 'arm':
                return 'arm32';
            case 'ppc':
                return 'ppc32';
            case 'x64':
                return 'amd64';
            default:
                return nodeArchString;
        }
    }
}
exports.hostDetector = new HostDetector();
//# sourceMappingURL=HostDetector.js.map