# -*- coding: utf-8 -*-

#
# Copyright (c) 2012 Red Hat, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#           http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import redhat_support_lib.utils.confighelper as confighelper
import redhat_support_lib.utils.reporthelper as reporthelper
import tempfile
import gzip
import logging
import os.path
import shutil
from pexpect import spawn, TIMEOUT, EOF
from urlparse import urlparse
from redhat_support_lib.web.connection import Connection
from redhat_support_lib.infrastructure.errors import RequestError, \
    ConnectionError
from redhat_support_lib.infrastructure.errors import SftpUploadWarnings, DontWaitForCompletion, SftpError
import json
import socket
import time
import sys

logger = logging.getLogger("redhat_support_lib.utils.ftphelper")

__author__ = 'Spenser Shumaker sshumake@redhat.com'
__author__ = 'Keith Robertson kroberts@redhat.com'
__author__ = 'Jake Hunsaker jhunsake@redhat.com'
__author__ = 'Pranita Ghole pghole@redhat.com'

config = confighelper.get_config_helper()


def form_connect_str():
    """
    Form the connection string for forming connection to the SFTP.

    :return: Connection String

    """
    scmd = "sftp"
    if config.sftp_port:
        try:
            # ensure the specified port is an integer
            int(config.sftp_port)
            scmd += " -P %s" % int(config.sftp_port)
        except Exception:
            msg = ("Invalid port '%s' specified. Please check your "
                   "configuration" % config.sftp_port)
            raise Exception(msg)
    if config.proxy_url != None:
        http_proxy_url = urlparse(config.proxy_url)
        hostname = http_proxy_url.netloc
        if config.proxy_user and config.proxy_pass:
            connect_str = '{} -o "ProxyCommand nc --proxy {} --proxy-auth {}:{} --proxy-type {} %h %p" {}@{}'.format(
                scmd, hostname, config.proxy_user, config.proxy_pass,
                http_proxy_url.scheme, config.username, config.sftp_host)
        else:
            connect_str = '{} -o "ProxyCommand nc --proxy {} --proxy-type {} %h %p" {}@{}'.format(
                scmd, hostname, http_proxy_url.scheme, config.username,
                config.sftp_host)
    else:
        connect_str = '{} {}@{}'.format(scmd, config.username, config.sftp_host)

    return connect_str


def ftp_attachment(fileName=None, caseNumber=None, fileChunk=None, is_internal_user=None):
    """
    Upload a file to Red Hat Secure FTP
    :return: True for successful uploads.

    """
    sftp_config = {'sftp_host': config.sftp_host, 'sftp_port': config.sftp_port, 'sftp_timeout': config.sftp_timeout}
    if not fileName:
        raise Exception('sftp_file(%s) cannot be empty.' % fileName)

    logger.debug("Creating connection to SFTP server %s" % config.sftp_host)

    if not caseNumber:
        caseNumber = 'RHST-upload'
    try:
        if fileChunk:
            print("The split uploads '-s' is not supported by Red Hat Secure FTP")
            return
        valid_sftp_name = make_sftp_filename(fileName=fileName, is_internal_user=is_internal_user, caseNumber=caseNumber)
        if not valid_sftp_name:
            return
        connect_str = form_connect_str()
        logger.debug("Sending file %s over SFTP" % fileName)
        logger.debug("SFTP config - {} ".format(sftp_config))
        resp = upload_to_sftp(connect_str, fileName=fileName, caseNumber=caseNumber, sftp_filename=valid_sftp_name,
                              is_internal_user=is_internal_user)
    except Exception, e:
        # BZ 1882771
        raise Exception("Could not upload file to SFTP: {}".format(e))
    if caseNumber and caseNumber != 'RHST-upload':
        watch_hydra_for_upload(sftp_filename=valid_sftp_name)
    return resp


def get_response_from_hydra(uri, body=None, method='GET'):
    """
    Get response from Hydra API and return the JSON response.

    :return: JSON response from Hydra APIs.
    :rtype:
    """
    conn = Connection(url=config.url,
                      manager=None,
                      key_file=config.key_file,
                      cert_file=config.cert_file,
                      timeout=config.timeout,
                      username=config.username,
                      password=config.password,
                      proxy_url=config.proxy_url,
                      proxy_user=config.proxy_user,
                      proxy_pass=config.proxy_pass,
                      debug=config.http_debug,
                      noverify=config.no_verify_ssl,
                      ssl_ca=config.ssl_ca,
                      auth=config.auth
                      )
    try:
        try:
            response = conn.doRequest(method=method, url=uri, headers=config.userAgent, body=body)
            if response.status < 400:
                res = response.read()
                doc = None
                if res is not None and res is not '' and res.strip() is not '':
                    doc = json.loads(res)
                return doc
            else:
                logger.debug("HTTP status(%s) HTTP reason(%s) HTTP response(%s)" % (response.status, response.reason,
                                                                                    response.read()))
                raise RequestError(response.status, response.reason, response.read())
        except socket.error, e:
            raise ConnectionError(str(e))
    finally:
        conn.close()


def fetch_sftp_token():
    '''
    Request Hydra for the SFTP token.

   :return: Token for the configured user to login to the SFTP server.
   '''

    data = {"isOneTime": True}
    res = get_response_from_hydra(method='POST', uri='/support/v2/sftp/token', body=json.dumps(data))
    if res:
        return res.get('token')


def upload_to_sftp(connect_str=None, fileName=None, caseNumber=None, sftp_filename=None, is_internal_user=None):
    '''

    :param connect_str: The connection string for the SFTP
    :param fileName: The file to be uploaded
    :return:
    '''

    passwd = fetch_sftp_token()
    remote_path = sftp_filename
    if is_internal_user:
        remote_path = "/users/{}/{}".format(config.username, sftp_filename)
    fileName = fileName.replace(" ", "\\ ")
    remote_path = remote_path.replace(" ", "\\ ")
    spwan_string = 'echo "put {} {}" | {}'.format(
                fileName, remote_path, connect_str)
    p = spawn('/bin/bash', ['-c', spwan_string])
    if config.http_debug:
        p.logfile_read = sys.stdout

    try:
        sftp_expects = [
            u'sftp>',
            u'password:',
            u'Connection refused',
            u'Are you sure you want to continue connecting (yes/no)?',
            u'Host key verification failed.',
            TIMEOUT,
            EOF
        ]

        sftp_out = p.expect(sftp_expects, timeout=60)

        if sftp_out == 3:
            connect_y_n = raw_input("Server SSH identity is not locally known. "
                                    "Are you sure you want to continue connecting (yes/no)?")
            if not (connect_y_n.strip() == 'yes' or connect_y_n.strip() == 'no'):
                while connect_y_n.strip() != 'yes' or connect_y_n.strip() != 'no':
                    connect_y_n = raw_input("Please enter yes/no only ")
                    if connect_y_n.strip() == 'yes' or connect_y_n.strip() == 'no':
                        break

            p.sendline(connect_y_n)

            sftp_out = p.expect(sftp_expects, timeout=60)

        if sftp_out == 1:
            logger.debug('Sending password token for SFTP login')
            p.sendline(passwd)
            sftp_prompt = p.expect([u'sftp>', u'Permission denied', TIMEOUT], timeout=60)

            if sftp_prompt == 0:
                sftp_out = 0
            elif sftp_prompt == 1:
                p.kill(0)
                raise Exception("Permission Denied - Incorrect or invalid login credentials")
            else:
                p.kill(0)
                raise Exception("Timeout exceeded while authenticating to the SFTP server")

        if sftp_out == 0:
            upload_out = p.expect([u'100%', u'Permission denied', TIMEOUT, EOF], timeout=int(config.sftp_timeout))
            if upload_out == 0:
                print(p.after.decode())
                p.sendline('bye')  # Quits SFTP
                p.isalive()
                p.close()
                logger.info('Upload was successful and SFTP connection has been closed')
                return True
            elif upload_out == 1:
                logger.debug("Permission denied while trying to upload the file, possible filename collision")
                p.kill(0)
                raise Exception("Permission denied while uploading file to SFTP server")
            elif upload_out == 2:
                p.kill(0)
                raise Exception('Timeout exceeded while uploading file to SFTP server.'
                                ' However, the file may have been partially uploaded.')
            else:
                p.kill(0)
                raise Exception('Unexpected response during SFTP upload, {}'.format(p.before))

        elif sftp_out == 2:
            p.kill(0)
            raise Exception('Connection refused by SFTP host.')

        elif sftp_out == 4:
            p.kill(0)
            raise Exception('Host key verification failed.')

        elif sftp_out == 5:
            raise Exception('Timeout exceeded while connecting to SFTP server {}'.format(p.before))

        elif sftp_out == 6:
            raise Exception('Unexpected response while connecting to SFTP server - {}'.format(p.before))
    except Exception, err:
        logger.debug(err)
        raise


def make_sftp_filename(fileName, is_internal_user, caseNumber=None,):
    """

    Attempt to create a unique filename for sftp upload using the case
    number.

    :param fileName:    The name of the file
    :param caseNumber:  Case number for RH Customer Portal
    """

    def _filename_exists(fileName):

        """Check hydra to see if the fileName already exists
        """

        if is_internal_user:
            uri = '/support/v1/sftp/attachments?path=users/{}'.format(config.username)
        else:
            uri = '/support/v1/sftp/attachments'

        try:
            resp = get_response_from_hydra(uri=uri)
            logger.debug("JSON response for {} - {}".format(uri, resp))
            if not resp:
                return False
            files = resp.get('files')
            if fileName in files:
                return True
            return False
        except Exception as err:
            logger.debug("\nUnable to verify filename for upload. "
                         "Upload may be denied or may overwrite existing file(s)")
            logger.error(err)
            return False
    basename = os.path.basename(fileName)
    if caseNumber:
        fileName = "{}_{}".format(caseNumber, basename)
    if _filename_exists(fileName):
        print("File {} already exists in Red Hat Secure FTP.".format(fileName))
        logger.error("The file named {} already exists in your SFTP folder. Please change the filename"
                     " and try uploading again.".format(fileName))
        return
    return fileName


def watch_hydra_for_upload(sftp_filename):
    max_attempts = 30  # wait for at most 5 minutes
    attempts = 0
    print("The file has been uploaded to the Red Hat Secure FTP."
          " It may take a few minutes to attach the file to the case.")
    print("You may safely use Ctrl+C to skip or interrupt waiting for this.")
    try:
        while not poll_upload_completion(sftp_filename):
            attempts += 1
            if attempts >= max_attempts:
                msg = "Automatic file attachment is taking longer than expected." \
                      " Please check your case attachments after a few minutes."
                print(msg)
                logger.warning(msg)
                raise KeyboardInterrupt  # exit from the command
                # break
            time.sleep(10)
    except KeyboardInterrupt:
        raise DontWaitForCompletion()
    except SftpUploadWarnings, e:
        raise
    except Exception, err:
        msg = "Problem encountered whilst obtaining the attachment status."
        logger.error("{} {}. The file {} may or may not be attached to the case.".format(msg, err, sftp_filename))
        raise SftpError(msg)


def poll_upload_completion(sftp_filename):

    """

    Continually poll the hydra endpoint for the status of the upload for a
    given file, so that the user can be sure that the file is properly attached
    to their case.

    """
    data = {
        "username": config.username,
        "fileName": sftp_filename
    }

    res = get_response_from_hydra(uri='/support/v1/sftp/attachments/status',
                                  method='POST', body=json.dumps(data))
    logger.debug("JSON response for /support/v1/sftp/attachments/status - {}".format(res))

    try:
        res = res.get('sftpAttachments')[0]
    except Exception:
        raise Exception("Malformed response: %s" % res)
    if res['status'] == 'COMPLETED':
        return True
    elif res['status'] == 'UPLOADED_WITH_WARNINGS':
        logger.error("File %s is uploaded with warnings. Hence, not attached to your case: %s" % (sftp_filename,
                                                                                                  res['error']))
        raise SftpUploadWarnings(res['error'])
    return False


def compress_attachment(fileName):
    try:
        try:
            tmp_dir = tempfile.mkdtemp()
            gzipName = "%s/%s.gz" % (tmp_dir, os.path.basename(fileName))
            gzf = gzip.open(gzipName, 'w+')
            f = open(fileName, 'rb')
            gzf.writelines(f)
        except Exception, e:
            err = ("Failed.\nERROR: unable to compress attachment.  Reason: %s" % e)
            print err
            logger.log(logging.ERROR, err)
            if os.path.exists(tmp_dir):
                shutil.rmtree(tmp_dir)
            return None
        return gzipName
    finally:
        f.close()
        gzf.close()


def is_compressed_file(fileName):
    file_type = reporthelper.get_file_type(fileName)
    for compressed_type in ['zip', 'x-xz', 'x-rar']:
        if compressed_type in file_type:
            return True
    return False

