/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation.allocator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.stream.StreamSupport;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.IntroSorter;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.RoutingNodes;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.routing.UnassignedInfo;
import org.elasticsearch.cluster.routing.allocation.AllocateUnassignedDecision;
import org.elasticsearch.cluster.routing.allocation.AllocationDecision;
import org.elasticsearch.cluster.routing.allocation.MoveDecision;
import org.elasticsearch.cluster.routing.allocation.NodeAllocationResult;
import org.elasticsearch.cluster.routing.allocation.RoutingAllocation;
import org.elasticsearch.cluster.routing.allocation.ShardAllocationDecision;
import org.elasticsearch.cluster.routing.allocation.allocator.ShardsAllocator;
import org.elasticsearch.cluster.routing.allocation.decider.AllocationDeciders;
import org.elasticsearch.cluster.routing.allocation.decider.Decision;
import org.elasticsearch.cluster.routing.allocation.decider.DiskThresholdDecider;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.gateway.PriorityComparator;

public class BalancedShardsAllocator
implements ShardsAllocator {
    private static final Logger logger = LogManager.getLogger(BalancedShardsAllocator.class);
    public static final Setting<Float> INDEX_BALANCE_FACTOR_SETTING = Setting.floatSetting("cluster.routing.allocation.balance.index", 0.55f, 0.0f, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Float> SHARD_BALANCE_FACTOR_SETTING = Setting.floatSetting("cluster.routing.allocation.balance.shard", 0.45f, 0.0f, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Float> THRESHOLD_SETTING = Setting.floatSetting("cluster.routing.allocation.balance.threshold", 1.0f, 0.0f, Setting.Property.Dynamic, Setting.Property.NodeScope);
    private volatile WeightFunction weightFunction;
    private volatile float threshold;

    public BalancedShardsAllocator(Settings settings) {
        this(settings, new ClusterSettings(settings, ClusterSettings.BUILT_IN_CLUSTER_SETTINGS));
    }

    @Inject
    public BalancedShardsAllocator(Settings settings, ClusterSettings clusterSettings) {
        this.setWeightFunction(INDEX_BALANCE_FACTOR_SETTING.get(settings).floatValue(), SHARD_BALANCE_FACTOR_SETTING.get(settings).floatValue());
        this.setThreshold(THRESHOLD_SETTING.get(settings).floatValue());
        clusterSettings.addSettingsUpdateConsumer(INDEX_BALANCE_FACTOR_SETTING, SHARD_BALANCE_FACTOR_SETTING, this::setWeightFunction);
        clusterSettings.addSettingsUpdateConsumer(THRESHOLD_SETTING, this::setThreshold);
    }

    private void setWeightFunction(float indexBalance, float shardBalanceFactor) {
        this.weightFunction = new WeightFunction(indexBalance, shardBalanceFactor);
    }

    private void setThreshold(float threshold) {
        this.threshold = threshold;
    }

    @Override
    public void allocate(RoutingAllocation allocation) {
        if (allocation.routingNodes().size() == 0) {
            this.failAllocationOfNewPrimaries(allocation);
            return;
        }
        Balancer balancer = new Balancer(logger, allocation, this.weightFunction, this.threshold);
        balancer.allocateUnassigned();
        balancer.moveShards();
        balancer.balance();
    }

    @Override
    public ShardAllocationDecision decideShardAllocation(ShardRouting shard, RoutingAllocation allocation) {
        Balancer balancer = new Balancer(logger, allocation, this.weightFunction, this.threshold);
        AllocateUnassignedDecision allocateUnassignedDecision = AllocateUnassignedDecision.NOT_TAKEN;
        MoveDecision moveDecision = MoveDecision.NOT_TAKEN;
        if (shard.unassigned()) {
            allocateUnassignedDecision = balancer.decideAllocateUnassigned(shard);
        } else {
            moveDecision = balancer.decideMove(shard);
            if (moveDecision.isDecisionTaken() && moveDecision.canRemain()) {
                MoveDecision rebalanceDecision = balancer.decideRebalance(shard);
                moveDecision = rebalanceDecision.withRemainDecision(moveDecision.getCanRemainDecision());
            }
        }
        return new ShardAllocationDecision(allocateUnassignedDecision, moveDecision);
    }

    private void failAllocationOfNewPrimaries(RoutingAllocation allocation) {
        RoutingNodes routingNodes = allocation.routingNodes();
        assert (routingNodes.size() == 0) : routingNodes;
        RoutingNodes.UnassignedShards.UnassignedIterator unassignedIterator = routingNodes.unassigned().iterator();
        while (unassignedIterator.hasNext()) {
            ShardRouting shardRouting = unassignedIterator.next();
            UnassignedInfo unassignedInfo = shardRouting.unassignedInfo();
            if (!shardRouting.primary() || unassignedInfo.getLastAllocationStatus() != UnassignedInfo.AllocationStatus.NO_ATTEMPT) continue;
            unassignedIterator.updateUnassigned(new UnassignedInfo(unassignedInfo.getReason(), unassignedInfo.getMessage(), unassignedInfo.getFailure(), unassignedInfo.getNumFailedAllocations(), unassignedInfo.getUnassignedTimeInNanos(), unassignedInfo.getUnassignedTimeInMillis(), unassignedInfo.isDelayed(), UnassignedInfo.AllocationStatus.DECIDERS_NO, unassignedInfo.getFailedNodeIds()), shardRouting.recoverySource(), allocation.changes());
        }
    }

    public float getThreshold() {
        return this.threshold;
    }

    public float getIndexBalance() {
        return this.weightFunction.indexBalance;
    }

    public float getShardBalance() {
        return this.weightFunction.shardBalance;
    }

    private static class WeightFunction {
        private final float indexBalance;
        private final float shardBalance;
        private final float theta0;
        private final float theta1;

        WeightFunction(float indexBalance, float shardBalance) {
            float sum = indexBalance + shardBalance;
            if (sum <= 0.0f) {
                throw new IllegalArgumentException("Balance factors must sum to a value > 0 but was: " + sum);
            }
            this.theta0 = shardBalance / sum;
            this.theta1 = indexBalance / sum;
            this.indexBalance = indexBalance;
            this.shardBalance = shardBalance;
        }

        float weight(Balancer balancer, ModelNode node, String index) {
            float weightShard = (float)node.numShards() - balancer.avgShardsPerNode();
            float weightIndex = (float)node.numShards(index) - balancer.avgShardsPerNode(index);
            return this.theta0 * weightShard + this.theta1 * weightIndex;
        }
    }

    public static class Balancer {
        private final Logger logger;
        private final Map<String, ModelNode> nodes;
        private final RoutingAllocation allocation;
        private final RoutingNodes routingNodes;
        private final WeightFunction weight;
        private final float threshold;
        private final Metadata metadata;
        private final float avgShardsPerNode;
        private final NodeSorter sorter;
        private static final Comparator<ShardRouting> BY_DESCENDING_SHARD_ID = Comparator.comparing(ShardRouting::shardId).reversed();

        public Balancer(Logger logger, RoutingAllocation allocation, WeightFunction weight, float threshold) {
            this.logger = logger;
            this.allocation = allocation;
            this.weight = weight;
            this.threshold = threshold;
            this.routingNodes = allocation.routingNodes();
            this.metadata = allocation.metadata();
            this.avgShardsPerNode = (float)this.metadata.getTotalNumberOfShards() / (float)this.routingNodes.size();
            this.nodes = Collections.unmodifiableMap(this.buildModelFromAssigned());
            this.sorter = this.newNodeSorter();
        }

        private ModelNode[] nodesArray() {
            return this.nodes.values().toArray(new ModelNode[this.nodes.size()]);
        }

        public float avgShardsPerNode(String index) {
            return (float)this.metadata.index(index).getTotalNumberOfShards() / (float)this.nodes.size();
        }

        public float avgShardsPerNode() {
            return this.avgShardsPerNode;
        }

        private NodeSorter newNodeSorter() {
            return new NodeSorter(this.nodesArray(), this.weight, this);
        }

        private static float absDelta(float lower, float higher) {
            assert (higher >= lower) : higher + " lt " + lower + " but was expected to be gte";
            return Math.abs(higher - lower);
        }

        private static boolean lessThan(float delta, float threshold) {
            return delta <= threshold + 0.001f;
        }

        private void balance() {
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Start balancing cluster");
            }
            if (this.allocation.hasPendingAsyncFetch()) {
                this.logger.debug("skipping rebalance due to in-flight shard/store fetches");
                return;
            }
            if (this.allocation.deciders().canRebalance(this.allocation).type() != Decision.Type.YES) {
                this.logger.trace("skipping rebalance as it is disabled");
                return;
            }
            if (this.nodes.size() < 2) {
                this.logger.trace("skipping rebalance as single node only");
                return;
            }
            this.balanceByWeights();
        }

        private MoveDecision decideRebalance(ShardRouting shard) {
            if (!shard.started()) {
                return MoveDecision.NOT_TAKEN;
            }
            Decision canRebalance = this.allocation.deciders().canRebalance(shard, this.allocation);
            this.sorter.reset(shard.getIndexName());
            ModelNode[] modelNodes = this.sorter.modelNodes;
            String currentNodeId = shard.currentNodeId();
            ModelNode currentNode = null;
            for (ModelNode node : modelNodes) {
                if (!node.getNodeId().equals(currentNodeId)) continue;
                currentNode = node;
                break;
            }
            assert (currentNode != null) : "currently assigned node could not be found";
            String idxName = shard.getIndexName();
            float currentWeight = this.weight.weight(this, currentNode, idxName);
            AllocationDeciders deciders = this.allocation.deciders();
            Decision.Type rebalanceDecisionType = Decision.Type.NO;
            ModelNode assignedNode = null;
            ArrayList<Tuple<ModelNode, Decision>> betterBalanceNodes = new ArrayList<Tuple<ModelNode, Decision>>();
            ArrayList<Tuple<ModelNode, Decision>> sameBalanceNodes = new ArrayList<Tuple<ModelNode, Decision>>();
            ArrayList<Tuple<ModelNode, Decision>> worseBalanceNodes = new ArrayList<Tuple<ModelNode, Decision>>();
            for (ModelNode modelNode : modelNodes) {
                if (modelNode == currentNode) continue;
                Decision decision = deciders.canAllocate(shard, modelNode.getRoutingNode(), this.allocation);
                float nodeWeight = this.weight.weight(this, modelNode, idxName);
                boolean betterWeightThanCurrent = nodeWeight <= currentWeight;
                boolean rebalanceConditionsMet = false;
                if (betterWeightThanCurrent) {
                    float currentDelta = Balancer.absDelta(nodeWeight, currentWeight);
                    boolean deltaAboveThreshold = !Balancer.lessThan(currentDelta, this.threshold);
                    boolean betterWeightWithShardAdded = nodeWeight + 1.0f < currentWeight;
                    boolean bl = rebalanceConditionsMet = deltaAboveThreshold && betterWeightWithShardAdded;
                    if (rebalanceConditionsMet && decision.type().higherThan(rebalanceDecisionType)) {
                        rebalanceDecisionType = decision.type();
                        assignedNode = modelNode;
                    }
                }
                Tuple<ModelNode, Decision> nodeResult = Tuple.tuple(modelNode, decision);
                if (rebalanceConditionsMet) {
                    betterBalanceNodes.add(nodeResult);
                    continue;
                }
                if (betterWeightThanCurrent) {
                    sameBalanceNodes.add(nodeResult);
                    continue;
                }
                worseBalanceNodes.add(nodeResult);
            }
            int weightRanking = 0;
            ArrayList<NodeAllocationResult> nodeDecisions = new ArrayList<NodeAllocationResult>(modelNodes.length - 1);
            for (Tuple tuple : betterBalanceNodes) {
                nodeDecisions.add(new NodeAllocationResult(((ModelNode)tuple.v1()).routingNode.node(), AllocationDecision.fromDecisionType(((Decision)tuple.v2()).type()), (Decision)tuple.v2(), ++weightRanking));
            }
            int currentNodeWeightRanking = ++weightRanking;
            for (Tuple tuple : sameBalanceNodes) {
                AllocationDecision nodeDecision = ((Decision)tuple.v2()).type() == Decision.Type.NO ? AllocationDecision.NO : AllocationDecision.WORSE_BALANCE;
                nodeDecisions.add(new NodeAllocationResult(((ModelNode)tuple.v1()).routingNode.node(), nodeDecision, (Decision)tuple.v2(), currentNodeWeightRanking));
            }
            for (Tuple tuple : worseBalanceNodes) {
                AllocationDecision nodeDecision = ((Decision)tuple.v2()).type() == Decision.Type.NO ? AllocationDecision.NO : AllocationDecision.WORSE_BALANCE;
                nodeDecisions.add(new NodeAllocationResult(((ModelNode)tuple.v1()).routingNode.node(), nodeDecision, (Decision)tuple.v2(), ++weightRanking));
            }
            if (canRebalance.type() != Decision.Type.YES || this.allocation.hasPendingAsyncFetch()) {
                AllocationDecision allocationDecision = this.allocation.hasPendingAsyncFetch() ? AllocationDecision.AWAITING_INFO : AllocationDecision.fromDecisionType(canRebalance.type());
                return MoveDecision.cannotRebalance(canRebalance, allocationDecision, currentNodeWeightRanking, nodeDecisions);
            }
            return MoveDecision.rebalance(canRebalance, AllocationDecision.fromDecisionType(rebalanceDecisionType), assignedNode != null ? assignedNode.routingNode.node() : null, currentNodeWeightRanking, nodeDecisions);
        }

        private void balanceByWeights() {
            AllocationDeciders deciders = this.allocation.deciders();
            ModelNode[] modelNodes = this.sorter.modelNodes;
            float[] weights = this.sorter.weights;
            block0: for (String index : this.buildWeightOrderedIndices()) {
                IndexMetadata indexMetadata = this.metadata.index(index);
                int relevantNodes = 0;
                for (int i = 0; i < modelNodes.length; ++i) {
                    ModelNode modelNode = modelNodes[i];
                    if (modelNode.getIndex(index) == null && deciders.canAllocate(indexMetadata, modelNode.getRoutingNode(), this.allocation).type() == Decision.Type.NO) continue;
                    modelNodes[i] = modelNodes[relevantNodes];
                    modelNodes[relevantNodes] = modelNode;
                    ++relevantNodes;
                }
                if (relevantNodes < 2) continue;
                this.sorter.reset(index, 0, relevantNodes);
                int lowIdx = 0;
                int highIdx = relevantNodes - 1;
                while (true) {
                    ModelNode minNode = modelNodes[lowIdx];
                    ModelNode maxNode = modelNodes[highIdx];
                    if (maxNode.numShards(index) > 0) {
                        float delta = Balancer.absDelta(weights[lowIdx], weights[highIdx]);
                        if (Balancer.lessThan(delta, this.threshold)) {
                            if (lowIdx <= 0 || highIdx - 1 <= 0 || !(Balancer.absDelta(weights[0], weights[highIdx - 1]) > this.threshold)) {
                                if (!this.logger.isTraceEnabled()) continue block0;
                                this.logger.trace("Stop balancing index [{}]  min_node [{}] weight: [{}]  max_node [{}] weight: [{}]  delta: [{}]", (Object)index, (Object)maxNode.getNodeId(), (Object)Float.valueOf(weights[highIdx]), (Object)minNode.getNodeId(), (Object)Float.valueOf(weights[lowIdx]), (Object)Float.valueOf(delta));
                                continue block0;
                            }
                        } else {
                            if (this.logger.isTraceEnabled()) {
                                this.logger.trace("Balancing from node [{}] weight: [{}] to node [{}] weight: [{}]  delta: [{}]", (Object)maxNode.getNodeId(), (Object)Float.valueOf(weights[highIdx]), (Object)minNode.getNodeId(), (Object)Float.valueOf(weights[lowIdx]), (Object)Float.valueOf(delta));
                            }
                            if (delta <= 1.0f) {
                                this.logger.trace("Couldn't find shard to relocate from node [{}] to node [{}]", (Object)maxNode.getNodeId(), (Object)minNode.getNodeId());
                            } else if (this.tryRelocateShard(minNode, maxNode, index)) {
                                weights[lowIdx] = this.sorter.weight(modelNodes[lowIdx]);
                                weights[highIdx] = this.sorter.weight(modelNodes[highIdx]);
                                this.sorter.sort(0, relevantNodes);
                                lowIdx = 0;
                                highIdx = relevantNodes - 1;
                                continue;
                            }
                        }
                    }
                    if (lowIdx < highIdx - 1) {
                        ++lowIdx;
                        continue;
                    }
                    if (lowIdx <= 0) continue block0;
                    lowIdx = 0;
                    --highIdx;
                }
            }
        }

        private String[] buildWeightOrderedIndices() {
            final String[] indices = (String[])this.allocation.routingTable().indicesRouting().keys().toArray(String.class);
            final float[] deltas = new float[indices.length];
            for (int i = 0; i < deltas.length; ++i) {
                this.sorter.reset(indices[i]);
                deltas[i] = this.sorter.delta();
            }
            new IntroSorter(){
                float pivotWeight;

                @Override
                protected void swap(int i, int j) {
                    String tmpIdx = indices[i];
                    indices[i] = indices[j];
                    indices[j] = tmpIdx;
                    float tmpDelta = deltas[i];
                    deltas[i] = deltas[j];
                    deltas[j] = tmpDelta;
                }

                @Override
                protected int compare(int i, int j) {
                    return Float.compare(deltas[j], deltas[i]);
                }

                @Override
                protected void setPivot(int i) {
                    this.pivotWeight = deltas[i];
                }

                @Override
                protected int comparePivot(int j) {
                    return Float.compare(deltas[j], this.pivotWeight);
                }
            }.sort(0, deltas.length);
            return indices;
        }

        public void moveShards() {
            Iterator<ShardRouting> it = this.allocation.routingNodes().nodeInterleavedShardIterator();
            while (it.hasNext()) {
                ShardRouting shardRouting = it.next();
                MoveDecision moveDecision = this.decideMove(shardRouting);
                if (moveDecision.isDecisionTaken() && moveDecision.forceMove()) {
                    ModelNode sourceNode = this.nodes.get(shardRouting.currentNodeId());
                    ModelNode targetNode = this.nodes.get(moveDecision.getTargetNode().getId());
                    sourceNode.removeShard(shardRouting);
                    Tuple<ShardRouting, ShardRouting> relocatingShards = this.routingNodes.relocateShard(shardRouting, targetNode.getNodeId(), this.allocation.clusterInfo().getShardSize(shardRouting, -1L), this.allocation.changes());
                    targetNode.addShard(relocatingShards.v2());
                    if (!this.logger.isTraceEnabled()) continue;
                    this.logger.trace("Moved shard [{}] to node [{}]", (Object)shardRouting, (Object)targetNode.getRoutingNode());
                    continue;
                }
                if (!moveDecision.isDecisionTaken() || moveDecision.canRemain()) continue;
                this.logger.trace("[{}][{}] can't move", (Object)shardRouting.index(), (Object)shardRouting.id());
            }
        }

        public MoveDecision decideMove(ShardRouting shardRouting) {
            if (!shardRouting.started()) {
                return MoveDecision.NOT_TAKEN;
            }
            boolean explain = this.allocation.debugDecision();
            ModelNode sourceNode = this.nodes.get(shardRouting.currentNodeId());
            assert (sourceNode != null && sourceNode.containsShard(shardRouting));
            RoutingNode routingNode = sourceNode.getRoutingNode();
            Decision canRemain = this.allocation.deciders().canRemain(shardRouting, routingNode, this.allocation);
            if (canRemain.type() != Decision.Type.NO) {
                return MoveDecision.stay(canRemain);
            }
            this.sorter.reset(shardRouting.getIndexName());
            Decision.Type bestDecision = Decision.Type.NO;
            RoutingNode targetNode = null;
            ArrayList<NodeAllocationResult> nodeExplanationMap = explain ? new ArrayList<NodeAllocationResult>() : null;
            int weightRanking = 0;
            for (ModelNode currentNode : this.sorter.modelNodes) {
                if (currentNode == sourceNode) continue;
                RoutingNode target = currentNode.getRoutingNode();
                Decision allocationDecision = this.allocation.deciders().canAllocate(shardRouting, target, this.allocation);
                if (explain) {
                    nodeExplanationMap.add(new NodeAllocationResult(currentNode.getRoutingNode().node(), allocationDecision, ++weightRanking));
                }
                if (!allocationDecision.type().higherThan(bestDecision) || (bestDecision = allocationDecision.type()) != Decision.Type.YES) continue;
                targetNode = target;
                if (!explain) break;
            }
            return MoveDecision.cannotRemain(canRemain, AllocationDecision.fromDecisionType(bestDecision), targetNode != null ? targetNode.node() : null, nodeExplanationMap);
        }

        private Map<String, ModelNode> buildModelFromAssigned() {
            HashMap<String, ModelNode> nodes = new HashMap<String, ModelNode>();
            for (RoutingNode rn : this.routingNodes) {
                ModelNode node = new ModelNode(rn);
                nodes.put(rn.nodeId(), node);
                for (ShardRouting shard : rn) {
                    assert (rn.nodeId().equals(shard.currentNodeId()));
                    if (shard.state() == ShardRoutingState.RELOCATING) continue;
                    node.addShard(shard);
                    if (!this.logger.isTraceEnabled()) continue;
                    this.logger.trace("Assigned shard [{}] to node [{}]", (Object)shard, (Object)node.getNodeId());
                }
            }
            return nodes;
        }

        private void allocateUnassigned() {
            RoutingNodes.UnassignedShards unassigned = this.routingNodes.unassigned();
            assert (!this.nodes.isEmpty());
            if (this.logger.isTraceEnabled()) {
                this.logger.trace("Start allocating unassigned shards");
            }
            if (unassigned.isEmpty()) {
                return;
            }
            PriorityComparator secondaryComparator = PriorityComparator.getAllocationComparator(this.allocation);
            Comparator comparator = (o1, o2) -> {
                if (o1.primary() ^ o2.primary()) {
                    return o1.primary() ? -1 : 1;
                }
                if (o1.getIndexName().compareTo(o2.getIndexName()) == 0) {
                    return o1.getId() - o2.getId();
                }
                int secondary = secondaryComparator.compare((ShardRouting)o1, (ShardRouting)o2);
                assert (secondary != 0) : "Index names are equal, should be returned early.";
                return secondary;
            };
            ShardRouting[] primary = unassigned.drain();
            ShardRouting[] secondary = new ShardRouting[primary.length];
            int secondaryLength = 0;
            int primaryLength = primary.length;
            ArrayUtil.timSort(primary, comparator);
            do {
                for (int i = 0; i < primaryLength; ++i) {
                    long shardSize;
                    ModelNode minNode;
                    ShardRouting shard = primary[i];
                    AllocateUnassignedDecision allocationDecision = this.decideAllocateUnassigned(shard);
                    String assignedNodeId = allocationDecision.getTargetNode() != null ? allocationDecision.getTargetNode().getId() : null;
                    ModelNode modelNode = minNode = assignedNodeId != null ? this.nodes.get(assignedNodeId) : null;
                    if (allocationDecision.getAllocationDecision() == AllocationDecision.YES) {
                        if (this.logger.isTraceEnabled()) {
                            this.logger.trace("Assigned shard [{}] to [{}]", (Object)shard, (Object)minNode.getNodeId());
                        }
                        shardSize = DiskThresholdDecider.getExpectedShardSize(shard, -1L, this.allocation.clusterInfo(), this.allocation.snapshotShardSizeInfo(), this.allocation.metadata(), this.allocation.routingTable());
                        shard = this.routingNodes.initializeShard(shard, minNode.getNodeId(), null, shardSize, this.allocation.changes());
                        minNode.addShard(shard);
                        if (shard.primary()) continue;
                        while (i < primaryLength - 1 && comparator.compare(primary[i], primary[i + 1]) == 0) {
                            secondary[secondaryLength++] = primary[++i];
                        }
                        continue;
                    }
                    if (this.logger.isTraceEnabled()) {
                        this.logger.trace("No eligible node found to assign shard [{}] allocation_status [{}]", (Object)shard, (Object)allocationDecision.getAllocationStatus());
                    }
                    if (minNode != null) {
                        assert (allocationDecision.getAllocationStatus() == UnassignedInfo.AllocationStatus.DECIDERS_THROTTLED);
                        shardSize = DiskThresholdDecider.getExpectedShardSize(shard, -1L, this.allocation.clusterInfo(), this.allocation.snapshotShardSizeInfo(), this.allocation.metadata(), this.allocation.routingTable());
                        minNode.addShard(shard.initialize(minNode.getNodeId(), null, shardSize));
                    } else if (this.logger.isTraceEnabled()) {
                        this.logger.trace("No Node found to assign shard [{}]", (Object)shard);
                    }
                    unassigned.ignoreShard(shard, allocationDecision.getAllocationStatus(), this.allocation.changes());
                    if (shard.primary()) continue;
                    while (i < primaryLength - 1 && comparator.compare(primary[i], primary[i + 1]) == 0) {
                        unassigned.ignoreShard(primary[++i], allocationDecision.getAllocationStatus(), this.allocation.changes());
                    }
                }
                primaryLength = secondaryLength;
                ShardRouting[] tmp = primary;
                primary = secondary;
                secondary = tmp;
                secondaryLength = 0;
            } while (primaryLength > 0);
        }

        private AllocateUnassignedDecision decideAllocateUnassigned(ShardRouting shard) {
            if (shard.assignedToNode()) {
                return AllocateUnassignedDecision.NOT_TAKEN;
            }
            boolean explain = this.allocation.debugDecision();
            Decision shardLevelDecision = this.allocation.deciders().canAllocate(shard, this.allocation);
            if (shardLevelDecision.type() == Decision.Type.NO && !explain) {
                return AllocateUnassignedDecision.no(UnassignedInfo.AllocationStatus.DECIDERS_NO, null);
            }
            float minWeight = Float.POSITIVE_INFINITY;
            ModelNode minNode = null;
            Decision decision = null;
            HashMap<String, NodeAllocationResult> nodeExplanationMap = explain ? new HashMap<String, NodeAllocationResult>() : null;
            ArrayList<Tuple<String, Float>> nodeWeights = explain ? new ArrayList<Tuple<String, Float>>() : null;
            for (ModelNode node : this.nodes.values()) {
                boolean updateMinNode;
                float currentWeight;
                if (node.containsShard(shard) && !explain || (currentWeight = this.weight.weight(this, node, shard.getIndexName())) > minWeight && !explain) continue;
                Decision decision2 = this.allocation.deciders().canAllocate(shard, node.getRoutingNode(), this.allocation);
                if (explain) {
                    nodeExplanationMap.put(node.getNodeId(), new NodeAllocationResult(node.getRoutingNode().node(), decision2, 0));
                    nodeWeights.add(Tuple.tuple(node.getNodeId(), Float.valueOf(currentWeight)));
                }
                if (decision2.type() != Decision.Type.YES && decision2.type() != Decision.Type.THROTTLE) continue;
                if (currentWeight == minWeight) {
                    if (decision2.type() == decision.type()) {
                        int repId = shard.id();
                        int nodeHigh = node.highestPrimary(shard.index().getName());
                        int minNodeHigh = minNode.highestPrimary(shard.getIndexName());
                        updateMinNode = (nodeHigh > repId && minNodeHigh > repId || nodeHigh < repId && minNodeHigh < repId) && nodeHigh < minNodeHigh || nodeHigh > repId && minNodeHigh < repId;
                    } else {
                        updateMinNode = decision2.type() == Decision.Type.YES;
                    }
                } else {
                    updateMinNode = true;
                }
                if (!updateMinNode) continue;
                minNode = node;
                minWeight = currentWeight;
                decision = decision2;
            }
            if (decision == null) {
                decision = Decision.NO;
            }
            ArrayList<NodeAllocationResult> nodeDecisions = null;
            if (explain) {
                nodeDecisions = new ArrayList<NodeAllocationResult>();
                nodeWeights.sort((nodeWeight1, nodeWeight2) -> Float.compare(((Float)nodeWeight1.v2()).floatValue(), ((Float)nodeWeight2.v2()).floatValue()));
                int weightRanking = 0;
                for (Tuple tuple : nodeWeights) {
                    NodeAllocationResult current = (NodeAllocationResult)nodeExplanationMap.get(tuple.v1());
                    nodeDecisions.add(new NodeAllocationResult(current.getNode(), current.getCanAllocateDecision(), ++weightRanking));
                }
            }
            return AllocateUnassignedDecision.fromDecision(decision, minNode != null ? minNode.routingNode.node() : null, nodeDecisions);
        }

        private boolean tryRelocateShard(ModelNode minNode, ModelNode maxNode, String idx) {
            ModelIndex index = maxNode.getIndex(idx);
            if (index != null) {
                this.logger.trace("Try relocating shard of [{}] from [{}] to [{}]", (Object)idx, (Object)maxNode.getNodeId(), (Object)minNode.getNodeId());
                Iterable shardRoutings = StreamSupport.stream(index.spliterator(), false).filter(ShardRouting::started).filter(maxNode::containsShard).sorted(BY_DESCENDING_SHARD_ID)::iterator;
                AllocationDeciders deciders = this.allocation.deciders();
                for (ShardRouting shard : shardRoutings) {
                    Decision allocationDecision;
                    Decision rebalanceDecision = deciders.canRebalance(shard, this.allocation);
                    if (rebalanceDecision.type() == Decision.Type.NO || (allocationDecision = deciders.canAllocate(shard, minNode.getRoutingNode(), this.allocation)).type() == Decision.Type.NO) continue;
                    Decision.Multi decision = new Decision.Multi().add(allocationDecision).add(rebalanceDecision);
                    maxNode.removeShard(shard);
                    long shardSize = this.allocation.clusterInfo().getShardSize(shard, -1L);
                    if (((Decision)decision).type() == Decision.Type.YES) {
                        this.logger.debug("Relocate [{}] from [{}] to [{}]", (Object)shard, (Object)maxNode.getNodeId(), (Object)minNode.getNodeId());
                        minNode.addShard(this.routingNodes.relocateShard(shard, minNode.getNodeId(), shardSize, this.allocation.changes()).v1());
                        return true;
                    }
                    this.logger.debug("Simulate relocation of [{}] from [{}] to [{}]", (Object)shard, (Object)maxNode.getNodeId(), (Object)minNode.getNodeId());
                    assert (((Decision)decision).type() == Decision.Type.THROTTLE);
                    minNode.addShard(shard.relocate(minNode.getNodeId(), shardSize));
                    return false;
                }
            }
            this.logger.trace("No shards of [{}] can relocate from [{}] to [{}]", (Object)idx, (Object)maxNode.getNodeId(), (Object)minNode.getNodeId());
            return false;
        }
    }

    static final class NodeSorter
    extends IntroSorter {
        final ModelNode[] modelNodes;
        final float[] weights;
        private final WeightFunction function;
        private String index;
        private final Balancer balancer;
        private float pivotWeight;

        NodeSorter(ModelNode[] modelNodes, WeightFunction function, Balancer balancer) {
            this.function = function;
            this.balancer = balancer;
            this.modelNodes = modelNodes;
            this.weights = new float[modelNodes.length];
        }

        public void reset(String index, int from, int to) {
            this.index = index;
            for (int i = from; i < to; ++i) {
                this.weights[i] = this.weight(this.modelNodes[i]);
            }
            this.sort(from, to);
        }

        public void reset(String index) {
            this.reset(index, 0, this.modelNodes.length);
        }

        public float weight(ModelNode node) {
            return this.function.weight(this.balancer, node, this.index);
        }

        @Override
        protected void swap(int i, int j) {
            ModelNode tmpNode = this.modelNodes[i];
            this.modelNodes[i] = this.modelNodes[j];
            this.modelNodes[j] = tmpNode;
            float tmpWeight = this.weights[i];
            this.weights[i] = this.weights[j];
            this.weights[j] = tmpWeight;
        }

        @Override
        protected int compare(int i, int j) {
            return Float.compare(this.weights[i], this.weights[j]);
        }

        @Override
        protected void setPivot(int i) {
            this.pivotWeight = this.weights[i];
        }

        @Override
        protected int comparePivot(int j) {
            return Float.compare(this.pivotWeight, this.weights[j]);
        }

        public float delta() {
            return this.weights[this.weights.length - 1] - this.weights[0];
        }
    }

    static final class ModelIndex
    implements Iterable<ShardRouting> {
        private final String id;
        private final Set<ShardRouting> shards = new HashSet<ShardRouting>(4);
        private int highestPrimary = -1;

        ModelIndex(String id) {
            this.id = id;
        }

        public int highestPrimary() {
            if (this.highestPrimary == -1) {
                int maxId = -1;
                for (ShardRouting shard : this.shards) {
                    if (!shard.primary()) continue;
                    maxId = Math.max(maxId, shard.id());
                }
                this.highestPrimary = maxId;
                return this.highestPrimary;
            }
            return this.highestPrimary;
        }

        public String getIndexId() {
            return this.id;
        }

        public int numShards() {
            return this.shards.size();
        }

        @Override
        public Iterator<ShardRouting> iterator() {
            return this.shards.iterator();
        }

        public void removeShard(ShardRouting shard) {
            this.highestPrimary = -1;
            assert (this.shards.contains(shard)) : "Shard not allocated on current node: " + shard;
            this.shards.remove(shard);
        }

        public void addShard(ShardRouting shard) {
            this.highestPrimary = -1;
            assert (!this.shards.contains(shard)) : "Shard already allocated on current node: " + shard;
            this.shards.add(shard);
        }

        public boolean containsShard(ShardRouting shard) {
            return this.shards.contains(shard);
        }
    }

    static class ModelNode
    implements Iterable<ModelIndex> {
        private final Map<String, ModelIndex> indices = new HashMap<String, ModelIndex>();
        private int numShards = 0;
        private final RoutingNode routingNode;

        ModelNode(RoutingNode routingNode) {
            this.routingNode = routingNode;
        }

        public ModelIndex getIndex(String indexId) {
            return this.indices.get(indexId);
        }

        public String getNodeId() {
            return this.routingNode.nodeId();
        }

        public RoutingNode getRoutingNode() {
            return this.routingNode;
        }

        public int numShards() {
            return this.numShards;
        }

        public int numShards(String idx) {
            ModelIndex index = this.indices.get(idx);
            return index == null ? 0 : index.numShards();
        }

        public int highestPrimary(String index) {
            ModelIndex idx = this.indices.get(index);
            if (idx != null) {
                return idx.highestPrimary();
            }
            return -1;
        }

        public void addShard(ShardRouting shard) {
            ModelIndex index = this.indices.get(shard.getIndexName());
            if (index == null) {
                index = new ModelIndex(shard.getIndexName());
                this.indices.put(index.getIndexId(), index);
            }
            index.addShard(shard);
            ++this.numShards;
        }

        public void removeShard(ShardRouting shard) {
            ModelIndex index = this.indices.get(shard.getIndexName());
            if (index != null) {
                index.removeShard(shard);
                if (index.numShards() == 0) {
                    this.indices.remove(shard.getIndexName());
                }
            }
            --this.numShards;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("Node(").append(this.routingNode.nodeId()).append(")");
            return sb.toString();
        }

        @Override
        public Iterator<ModelIndex> iterator() {
            return this.indices.values().iterator();
        }

        public boolean containsShard(ShardRouting shard) {
            ModelIndex index = this.getIndex(shard.getIndexName());
            return index == null ? false : index.containsShard(shard);
        }
    }
}

