// SPDX-License-Identifier: (BSD-3-Clause OR GPL-2.0-only)
/* Copyright(c) 2017 - 2018, 2020 - 2021 Intel Corporation */

#include <adf_accel_devices.h>
#include <adf_common_drv.h>
#include <adf_cfg.h>
#include "adf_c4xxx_hw_data.h"
#include "adf_c4xxx_inline.h"
#include "adf_c4xxx_accel_units.h"

/* Congestion management profile definitions */
struct adf_congest_mngt {
	u32 ic_bb_fchthresh[ADF_C4XXX_NUM_CONGEST_DOMAINS];
	u32 ic_bb_fclthresh[ADF_C4XXX_NUM_CONGEST_DOMAINS];
	u32 ic_bb_behthresh;
	u32 ic_bb_belthresh;
	u32 ic_bewip_thresh;
	u32 ic_ctpb_thresh;
	u32 ic_cirq;
	u32 ic_q2memap[ADF_C4XXX_NUM_CONGEST_DOMAINS];
};

/* Congestion management profile initialisation for ingress direction.
 * q2memap registers are initialised in function adf_set_q2memap_registers()
 * function depending on the ingress/egress split
 */
static struct adf_congest_mngt adf_c4xxx_congest_mngt_profiles_ingress[] = {
	/* Profile CIRQ Cfg1 (Committed Information Rate Queue) */
	{
		/* ic_bb_fchthresh[0...7] */
		{0x00007800, 0x00007800, 0x00007800, 0x00007800,
		 0x00007800, 0x00007800, 0x00007800, 0x00007800},
		/* ic_bb_fclthresh[0...7] */
		{0x00005000, 0x00005000, 0x00005000, 0x00005000,
		 0x00005000, 0x00005000, 0x00005000, 0x00005000},
		/* ic_bb_behthresh */
		0x00028000,
		/* ic_bb_belthresh */
		0x00005000,
		/* ic_bewip_thresh */
		0x0001d000,
		/* ic_ctpb_thresh */
		0x0002d000,
		/* ic_cirq */
		0x00000000,
		/* ic_q2memap */
		{0x000001ff, 0x000001ff, 0x000001ff, 0x000001ff,
		 0x000001ff, 0x000001ff, 0x000001ff, 0x000001ff}
	},

	/* Profile CIRQ Cfg2 */
	{
		/* ic_bb_fchthresh[0...7] */
		{0x00007800, 0x00007800, 0x00007800, 0x00007800,
		 0x00007800, 0x00007800, 0x00007800, 0x00007800},
		/* ic_bb_fclthresh[0...7] */
		{0x00005000, 0x00005000, 0x00005000, 0x00005000,
		 0x00005000, 0x00005000, 0x00005000, 0x00005000},
		/* ic_bb_behthresh */
		0x00028000,
		/* ic_bb_belthresh */
		0x00005000,
		/* ic_bewip_thresh */
		0x0001d000,
		/* ic_ctpb_thresh */
		0x0002d000,
		/* ic_cirq */
		0x00000000,
		/* ic_q2memap */
		{0x00000fff, 0x00000fff, 0x00000fff, 0x00000fff,
		 0x00000fff, 0x00000fff, 0x00000fff, 0x00000fff}
	},

	/* Profile CIRQ Cfg3 */
	{
		/* ic_bb_fchthresh[0...7] */
		{0x00007800, 0x00007800, 0x00007800, 0x00007800,
		 0x00007800, 0x00007800, 0x00007800, 0x00007800},
		/* ic_bb_fclthresh[0...7] */
		{0x00005000, 0x00005000, 0x00005000, 0x00005000,
		 0x00005000, 0x00005000, 0x00005000, 0x00005000},
		/* ic_bb_behthresh */
		0x00028000,
		/* ic_bb_belthresh */
		0x00005000,
		/* ic_bewip_thresh */
		0x0001d000,
		/* ic_ctpb_thresh */
		0x0002d000,
		/* ic_cirq */
		0x00000000,
		/* ic_q2memap */
		{0x00000fff, 0x00000fff, 0x00000fff, 0x00000fff,
		 0x00000fff, 0x00000fff, 0x00000fff, 0x00000fff}
	},

	/* Profile Best Effort Single Queue */
	{
		/* ic_bb_fchthresh[0...7] */
		{0x00029000, 0x00000400, 0x00000400, 0x00000400,
		 0x00000400, 0x00000400, 0x00000400, 0x00000400},
		/* ic_bb_fclthresh[0...7] */
		{0x00005000, 0x00000200, 0x00000200, 0x00000200,
		 0x00000200, 0x00000200, 0x00000200, 0x00000200},
		/* ic_bb_behthresh */
		0x00029000,
		/* ic_bb_belthresh */
		0x00005000,
		/* ic_bewip_thresh */
		0x00020000,
		/* ic_ctpb_thresh */
		0x0002d000,
		/* ic_cirq */
		0x00000000,
		/* ic_q2memap */
		{0x00000000, 0x00000000, 0x00000000, 0x00000000,
		 0x00000000, 0x00000000, 0x00000000, 0x00000000}
	},

	/* Profile Best Effort 8 Queues */
	{
		/* ic_bb_fchthresh[0...7] */
		{0x00007800, 0x00007800, 0x00007800, 0x00007800,
		 0x00007800, 0x00007800, 0x00007800, 0x00007800},
		/* ic_bb_fclthresh[0...7] */
		{0x00005000, 0x00005000, 0x00005000, 0x00005000,
		 0x00005000, 0x00005000, 0x00005000, 0x00005000},
		/* ic_bb_behthresh */
		0x0001A000,
		/* ic_bb_belthresh */
		0x00010000,
		/* ic_bewip_thresh */
		0x00020000,
		/* ic_ctpb_thresh */
		0x0002d000,
		/* ic_cirq */
		0x00000000,
		/* ic_q2memap */
		{0x00000000, 0x00000000, 0x00000000, 0x00000000,
		 0x00000000, 0x00000000, 0x00000000, 0x00000000}
	},
};

/* Congestion management profile initialisation for egress direction. */
static struct adf_congest_mngt adf_c4xxx_congest_mngt_profiles_egress[] = {
	/* Profile CIRQ Cfg1 (Committed Information Rate Queue) */
	{
		/* ic_bb_fchthresh[0...7] */
		{0x00007800, 0x00007800, 0x00007800, 0x00007800,
		 0x00007800, 0x00007800, 0x00007800, 0x00007800},
		/* ic_bb_fclthresh[0...7] */
		{0x00005000, 0x00005000, 0x00005000, 0x00005000,
		 0x00005000, 0x00005000, 0x00005000, 0x00005000},
		/* ic_bb_behthresh */
		0x0001a000,
		/* ic_bb_belthresh */
		0x00005000,
		/* ic_bewip_thresh */
		0x0001c000,
		/* ic_ctpb_thresh */
		0x0002d000,
		/* ic_cirq */
		0x00000017,
		/* ic_q2memap */
		{0x0000fe00, 0x0000fe00, 0x0000fe00, 0x0000fe00,
		 0x0000fe00, 0x0000fe00, 0x0000fe00, 0x0000fe00}
	},

	/* Profile CIRQ Cfg2 */
	{
		/* ic_bb_fchthresh[0...7] */
		{0x00007800, 0x00007800, 0x00007800, 0x00007800,
		 0x00007800, 0x00007800, 0x00007800, 0x00007800},
		/* ic_bb_fclthresh[0...7] */
		{0x00005000, 0x00005000, 0x00005000, 0x00005000,
		 0x00005000, 0x00005000, 0x00005000, 0x00005000},
		/* ic_bb_behthresh */
		0x0001a000,
		/* ic_bb_belthresh */
		0x00005000,
		/* ic_bewip_thresh */
		0x0001c000,
		/* ic_ctpb_thresh */
		0x0002d000,
		/* ic_cirq */
		0x00000017,
		/* ic_q2memap */
		{0x0ffff000, 0x0ffff000, 0x0ffff000, 0x0ffff000,
		 0x0ffff000, 0x0ffff000, 0x0ffff000, 0x0ffff000}
	},

	/* Profile CIRQ Cfg3 */
	{
		/* ic_bb_fchthresh[0...7] */
		{0x00007800, 0x00007800, 0x00007800, 0x00007800,
		 0x00007800, 0x00007800, 0x00007800, 0x00007800},
		/* ic_bb_fclthresh[0...7] */
		{0x00005000, 0x00005000, 0x00005000, 0x00005000,
		 0x00005000, 0x00005000, 0x00005000, 0x00005000},
		/* ic_bb_behthresh */
		0x0001a000,
		/* ic_bb_belthresh */
		0x00005000,
		/* ic_bewip_thresh */
		0x0001c000,
		/* ic_ctpb_thresh */
		0x0002d000,
		/* ic_cirq */
		0x00000017,
		/* ic_q2memap */
		{0x0ffff000, 0x0ffff000, 0x0ffff000, 0x0ffff000,
		 0x0ffff000, 0x0ffff000, 0x0ffff000, 0x0ffff000}
	},

	/* Profile Best Effort Single Queue */
	{
		/* ic_bb_fchthresh[0...7] */
		{0x00029000, 0x00000400, 0x00000400, 0x00000400,
		 0x00000400, 0x00000400, 0x00000400, 0x00000400},
		/* ic_bb_fclthresh[0...7] */
		{0x00005000, 0x00000200, 0x00000200, 0x00000200,
		 0x00000200, 0x00000200, 0x00000200, 0x00000200},
		/* ic_bb_behthresh */
		0x00029000,
		/* ic_bb_belthresh */
		0x00005000,
		/* ic_bewip_thresh */
		0x00020000,
		/* ic_ctpb_thresh */
		0x0002d000,
		/* ic_cirq */
		0x00000000,
		/* ic_q2memap */
		{0x00000000, 0x00000000, 0x00000000, 0x00000000,
		 0x00000000, 0x00000000, 0x00000000, 0x00000000}
	},

	/* Profile Best Effort 8 Queues */
	{
		/* ic_bb_fchthresh[0...7] */
		{0x00007800, 0x00007800, 0x00007800, 0x00007800,
		 0x00007800, 0x00007800, 0x00007800, 0x00007800},
		/* ic_bb_fclthresh[0...7] */
		{0x00005000, 0x00005000, 0x00005000, 0x00005000,
		 0x00005000, 0x00005000, 0x00005000, 0x00005000},
		/* ic_bb_behthresh */
		0x0001a000,
		/* ic_bb_belthresh */
		0x00010000,
		/* ic_bewip_thresh */
		0x00020000,
		/* ic_ctpb_thresh */
		0x0002d000,
		/* ic_cirq */
		0x00000000,
		/* ic_q2memap */
		{0x00000000, 0x00000000, 0x00000000, 0x00000000,
		 0x00000000, 0x00000000, 0x00000000, 0x00000000}
	},
};

int get_congestion_management_profile(struct adf_accel_dev *accel_dev,
				      u8 *profile)
{
	char key[ADF_CFG_MAX_KEY_LEN_IN_BYTES];
	char val[ADF_CFG_MAX_VAL_LEN_IN_BYTES];
	u8 max_num_profiles;
	u8 user_profile;

	/* Get the Inline congestion management profile */
	snprintf(key, sizeof(key), ADF_INLINE_CONGEST_MNGT_PROFILE);
	if (adf_cfg_get_param_value(accel_dev, ADF_INLINE_SEC, key, val))
		return -EFAULT;
	if (kstrtou8(val, 10, &user_profile))
		return -EFAULT;

	/* Verify if profile is valid */
	max_num_profiles = ARRAY_SIZE(adf_c4xxx_congest_mngt_profiles_egress);

	if (user_profile >= max_num_profiles) {
		dev_err(&GET_DEV(accel_dev),
			"Invalid congestion management profile! Value must range between 0 and %d\n",
			 max_num_profiles - 1);
		return -EINVAL;
	}
	*profile = user_profile;

	return 0;
}

static void adf_display_congest_mngt_profile(struct adf_accel_dev *accel_dev,
					     const u8 profile)
{
	switch (profile) {
	case CIRQ_CFG_1:
		dev_info(&GET_DEV(accel_dev),
			 "Loading CIRQ Configuration 1 congestion management profile\n");
		break;
	case CIRQ_CFG_2:
		dev_info(&GET_DEV(accel_dev),
			 "Loading CIRQ Configuration 2 congestion management profile\n");
		break;
	case CIRQ_CFG_3:
		dev_info(&GET_DEV(accel_dev),
			 "Loading CIRQ Configuration 3 congestion management profile\n");
		break;
	case BEST_EFFORT_SINGLE_QUEUE:
		dev_info(&GET_DEV(accel_dev),
			 "Loading Best Effort Single Queue congestion management profile\n");
		break;
	case BEST_EFFORT_8_QUEUES:
		dev_info(&GET_DEV(accel_dev),
			 "Loading Best Effort 8 Queues congestion management profile\n");
		break;
	default:
		break;
	}
}

static void adf_set_q2memap_registers(const u8 profile, u32 *q2memap_regs,
				      const u32 mask)
{
	u8 i;

	switch (profile) {
	/* The following profiles require only one q2Memap
	 * register to be initialised
	 */
	case BEST_EFFORT_SINGLE_QUEUE:
		memset(q2memap_regs, 0, sizeof(u32) *
			  ADF_C4XXX_NUM_CONGEST_DOMAINS);
		*(q2memap_regs) = mask;
		break;
	/* The following profiles require all the q2Memap
	 * registers to be initialised
	 */
	case CIRQ_CFG_1:
	case CIRQ_CFG_2:
	case CIRQ_CFG_3:
	case BEST_EFFORT_8_QUEUES:
		for (i = 0; i < ADF_C4XXX_NUM_CONGEST_DOMAINS; i++)
			*(q2memap_regs + i) = mask;
		break;
	default:
		break;
	}
}

static void adf_disable_inline_asym_threads(struct adf_accel_dev *accel_dev,
					    const u8 profile)
{
	struct adf_accel_unit_info *au_info = accel_dev->au_info;
	u32 inline_ae_msk;

	switch (profile) {
	/* All the CIR profiles must have the arbiter
	 * configured to not use inline AEs for PKE
	 */
	case CIRQ_CFG_1:
	case CIRQ_CFG_2:
	case CIRQ_CFG_3:
		/* Retrieve inline AEs */
		inline_ae_msk = au_info->inline_egress_msk |
					au_info->inline_ingress_msk;
		/* Only keep lookaside AEs */
		au_info->asym_ae_msk &= ~inline_ae_msk;
		break;
	default:
		break;
	}
}

int adf_init_congestion_management_c4xxx(struct adf_accel_dev *accel_dev)
{
	void __iomem *aram_csr_base;
	struct adf_congest_mngt *ingress_cfg;
	struct adf_congest_mngt *egress_cfg;
	u8 profile = 0;
	u32 i;

	if (get_congestion_management_profile(accel_dev, &profile))
		return -EFAULT;

	adf_display_congest_mngt_profile(accel_dev,
					 (enum congest_mngt_profile_info)profile
					 );

	/* Retrieve ingress and egress masks depending on user setting
	 * in config file
	 */
	ingress_cfg = &adf_c4xxx_congest_mngt_profiles_ingress[profile];
	egress_cfg = &adf_c4xxx_congest_mngt_profiles_egress[profile];
	aram_csr_base = (&GET_BARS(accel_dev)[ADF_C4XXX_SRAM_BAR])->virt_addr;

	/* Set ingress q2Memap registers depending on profile */
	adf_set_q2memap_registers(profile,
				  ingress_cfg->ic_q2memap,
				  accel_dev->au_info->inline_ingress_msk);

	/* Set egress q2Memap registers depending on profile */
	adf_set_q2memap_registers(profile,
				  egress_cfg->ic_q2memap,
				  accel_dev->au_info->inline_egress_msk);

	adf_disable_inline_asym_threads(accel_dev, profile);

	for (i = 0; i < ADF_C4XXX_NUM_CONGEST_DOMAINS; i++) {
		/* High threshold in the ingress space */
		ADF_C4XXX_WR_ICI_BB_FCHTHRESH(aram_csr_base,
					      i,
					      ingress_cfg->ic_bb_fchthresh[i]);
		/* Low threshold in the ingress space */
		ADF_C4XXX_WR_ICI_BB_FCLTHRESH(aram_csr_base,
					      i,
					      ingress_cfg->ic_bb_fclthresh[i]);
		/* IC_Q2MEMAP registers in ingress space */
		ADF_C4XXX_WR_CSR_ICI_Q2MEMAP(aram_csr_base,
					     i,
					     ingress_cfg->ic_q2memap[i]);

		/* High threshold in the egress space */
		ADF_C4XXX_WR_ICE_BB_FCHTHRESH(aram_csr_base,
					      i,
					      egress_cfg->ic_bb_fchthresh[i]);
		/* Low threshold in the egress space */
		ADF_C4XXX_WR_ICE_BB_FCLTHRESH(aram_csr_base,
					      i,
					      egress_cfg->ic_bb_fclthresh[i]);
		/* IC_Q2MEMAP registers in egress space */
		ADF_C4XXX_WR_CSR_ICE_Q2MEMAP(aram_csr_base,
					     i,
					     egress_cfg->ic_q2memap[i]);
	}

	/* IC_BB_BEHTHRESH register both in Ingress and Egress space */
	ADF_CSR_WR(aram_csr_base,
		   ADF_C4XXX_ICI_BB_BEHTHRESH_OFFSET,
		   ingress_cfg->ic_bb_behthresh);
	ADF_CSR_WR(aram_csr_base,
		   ADF_C4XXX_ICE_BB_BEHTHRESH_OFFSET,
		   egress_cfg->ic_bb_behthresh);

	/* IC_BB_BELTHRESH register both in Ingress and Egress space */
	ADF_CSR_WR(aram_csr_base,
		   ADF_C4XXX_ICI_BB_BELTHRESH_OFFSET,
		   ingress_cfg->ic_bb_belthresh);
	ADF_CSR_WR(aram_csr_base,
		   ADF_C4XXX_ICE_BB_BELTHRESH_OFFSET,
		   egress_cfg->ic_bb_belthresh);

	/* IC_BEWIP_THRESH register both in Ingress and Egress space */
	ADF_CSR_WR(aram_csr_base,
		   ADF_C4XXX_ICI_BEWIP_THRESH_OFFSET,
		   ingress_cfg->ic_bewip_thresh);
	ADF_CSR_WR(aram_csr_base,
		   ADF_C4XXX_ICE_BEWIP_THRESH_OFFSET,
		   egress_cfg->ic_bewip_thresh);

	/* IC_CTPB_THRESH register both in Ingress and Egress space */
	ADF_CSR_WR(aram_csr_base,
		   ADF_C4XXX_ICI_CTPB_THRESH_OFFSET,
		   ingress_cfg->ic_ctpb_thresh);
	ADF_CSR_WR(aram_csr_base,
		   ADF_C4XXX_ICE_CTPB_THRESH_OFFSET,
		   egress_cfg->ic_ctpb_thresh);

	/* IC_CTPB_THRESH register both in Ingress and Egress space */
	ADF_CSR_WR(aram_csr_base,
		   ADF_C4XXX_ICI_CIRQ_OFFSET,
		   ingress_cfg->ic_cirq);
	ADF_CSR_WR(aram_csr_base,
		   ADF_C4XXX_ICE_CIRQ_OFFSET,
		   egress_cfg->ic_cirq);

	return 0;
}
