﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

#nullable disable

using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Linq;
using System.Text;
using Microsoft.CodeAnalysis.CSharp.Symbols;
using Microsoft.CodeAnalysis.CSharp.Symbols.Metadata.PE;
using Microsoft.CodeAnalysis.CSharp.Syntax;
using Microsoft.CodeAnalysis.CSharp.Test.Utilities;
using Microsoft.CodeAnalysis.Test.Utilities;
using Roslyn.Test.Utilities;
using Roslyn.Utilities;
using Xunit;
using static Microsoft.CodeAnalysis.CSharp.Symbols.FlowAnalysisAnnotations;

namespace Microsoft.CodeAnalysis.CSharp.UnitTests.Semantics
{
    [CompilerTrait(CompilerFeature.NullableReferenceTypes)]
    public class NullableReferenceTypesTests : CSharpTestBase
    {
        private const string Tuple2NonNullable =
                    @"
namespace System
{
#nullable enable
    // struct with two values
    public struct ValueTuple<T1, T2>
        where T1 : notnull
        where T2 : notnull
    {
        public T1 Item1;
        public T2 Item2;

        public ValueTuple(T1 item1, T2 item2)
        {
            this.Item1 = item1;
            this.Item2 = item2;
        }

        public override string ToString()
        {
            return """";
        }
    }
}";

        private const string TupleRestNonNullable =
                    @"
namespace System
{
#nullable enable
    public struct ValueTuple<T1, T2, T3, T4, T5, T6, T7, TRest>
        where T1 : notnull
        where T2 : notnull
        where T3 : notnull
        where T4 : notnull
        where T5 : notnull
        where T6 : notnull
        where T7 : notnull
    {
        public T1 Item1;
        public T2 Item2;
        public T3 Item3;
        public T4 Item4;
        public T5 Item5;
        public T6 Item6;
        public T7 Item7;
        public TRest Rest;

        public ValueTuple(T1 item1, T2 item2, T3 item3, T4 item4, T5 item5, T6 item6, T7 item7, TRest rest)
        {
            Item1 = item1;
            Item2 = item2;
            Item3 = item3;
            Item4 = item4;
            Item5 = item5;
            Item6 = item6;
            Item7 = item7;
            Rest = rest;
        }

        public override string ToString()
        {
            return base.ToString();
        }
    }
}";

        private CSharpCompilation CreateNullableCompilation(CSharpTestSource source,
            IEnumerable<MetadataReference> references = null,
            CSharpParseOptions parseOptions = null)
        {
            return CreateCompilation(source, options: WithNullableEnable(), references: references, parseOptions: parseOptions);
        }

        private static bool IsNullableAnalysisEnabled(CSharpCompilation compilation, string methodName)
        {
            var method = compilation.GetMember<MethodSymbol>(methodName);
            return method.IsNullableAnalysisEnabled();
        }

        [Fact]
        public void DefaultLiteralInConditional()
        {
            var comp = CreateNullableCompilation(@"
class C
{
    public void M<T>(bool condition, T t)
    {
        t = default;
        _ = condition ? t : default;
        _ = condition ? default : t;
    }
}");

            comp.VerifyDiagnostics(
                // (6,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         t = default;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "default").WithLocation(6, 13));
        }

        [Fact, WorkItem(46461, "https://github.com/dotnet/roslyn/issues/46461")]
        public void DefaultLiteralInConditional_02()
        {
            var comp = CreateNullableCompilation(@"
using System;

public class C
{
    public void M()
    {
        M1(true, b => b ? """" : default);
        M1<bool, string?>(true, b => b ? """" : default);
        M1<bool, string>(true, b => b ? """" : default); // 1

        M1(true, b => b ? """" : null);
        M1<bool, string?>(true, b => b ? """" : null);
        M1<bool, string>(true, b => b ? """" : null); // 2
    }

    public void M1<T,U>(T t, Func<T, U> fn)
    {
        fn(t);
    }
}");

            comp.VerifyDiagnostics(
                // (10,37): warning CS8603: Possible null reference return.
                //         M1<bool, string>(true, b => b ? "" : default); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReturn, @"b ? """" : default", isSuppressed: false).WithLocation(10, 37),
                // (14,37): warning CS8603: Possible null reference return.
                //         M1<bool, string>(true, b => b ? "" : null); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReturn, @"b ? """" : null", isSuppressed: false).WithLocation(14, 37));
        }

        [Fact, WorkItem(33982, "https://github.com/dotnet/roslyn/issues/33982")]
        public void AssigningNullToRefLocalIsSafetyWarning()
        {
            var comp = CreateCompilation(@"
class C
{
    void M(ref string s1, ref string s2)
    {
        s1 = null; // 1
        s1.ToString(); // 2

        ref string s3 = ref s2;
        s3 = null; // 3
        s3.ToString(); // 4

        ref string s4 = ref s2;
        s4 ??= null; // 5
        s4.ToString(); // 6

        ref string s5 = ref s2;
        M2(out s5); // 7
        s5.ToString(); // 8
    }
    void M2(out string? x) => throw null!;
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (6,14): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         s1 = null; // 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(6, 14),
                // (7,9): warning CS8602: Possible dereference of a null reference.
                //         s1.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s1").WithLocation(7, 9),
                // (10,14): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         s3 = null; // 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 14),
                // (11,9): warning CS8602: Possible dereference of a null reference.
                //         s3.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s3").WithLocation(11, 9),
                // (14,16): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         s4 ??= null; // 5
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(14, 16),
                // (15,9): warning CS8602: Possible dereference of a null reference.
                //         s4.ToString(); // 6
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s4").WithLocation(15, 9),
                // (18,16): warning CS8601: Possible null reference assignment.
                //         M2(out s5); // 7
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "s5").WithLocation(18, 16),
                // (19,9): warning CS8602: Possible dereference of a null reference.
                //         s5.ToString(); // 8
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s5").WithLocation(19, 9)
                );
        }

        [Fact, WorkItem(33365, "https://github.com/dotnet/roslyn/issues/33365")]
        public void AssigningNullToConditionalRefLocal()
        {
            var comp = CreateCompilation(@"
class C
{
    void M(ref string s1, ref string s2, bool b)
    {
        (b ? ref s1 : ref s1) = null; // 1
        s1.ToString();

        ref string s3 = ref s2;
        (b ? ref s3 : ref s3) = null; // 2
        s3.ToString();

        ref string s4 = ref s2;
        (b ? ref s4 : ref s4) ??= null; // 3
        s4.ToString();

        ref string s5 = ref s2;
        M2(out (b ? ref s5 : ref s5)); // 4
        s5.ToString();
    }
    void M2(out string? x) => throw null!;
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (6,33): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         (b ? ref s1 : ref s1) = null; // 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(6, 33),
                // (10,33): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         (b ? ref s3 : ref s3) = null; // 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 33),
                // (14,35): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         (b ? ref s4 : ref s4) ??= null; // 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(14, 35),
                // (18,17): warning CS8601: Possible null reference assignment.
                //         M2(out (b ? ref s5 : ref s5)); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "b ? ref s5 : ref s5").WithLocation(18, 17)
                );
        }

        [Fact, WorkItem(33365, "https://github.com/dotnet/roslyn/issues/33365")]
        public void AssigningNullToConditionalRefLocal_NullableLocals()
        {
            var comp = CreateCompilation(@"
class C
{
    void M(ref string? s1, ref string? s2, bool b)
    {
        (b ? ref s1 : ref s1) = null;
        s1.ToString(); // 1

        ref string? s3 = ref s2;
        (b ? ref s3 : ref s3) = null;
        s3.ToString(); // 2

        ref string? s4 = ref s2;
        (b ? ref s4 : ref s4) ??= null;
        s4.ToString(); // 3

        ref string? s5 = ref s2;
        M2(out (b ? ref s5 : ref s5));
        s5.ToString(); // 4
    }
    void M2(out string? x) => throw null!;
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (7,9): warning CS8602: Possible dereference of a null reference.
                //         s1.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s1").WithLocation(7, 9),
                // (11,9): warning CS8602: Possible dereference of a null reference.
                //         s3.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s3").WithLocation(11, 9),
                // (15,9): warning CS8602: Possible dereference of a null reference.
                //         s4.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s4").WithLocation(15, 9),
                // (19,9): warning CS8602: Possible dereference of a null reference.
                //         s5.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s5").WithLocation(19, 9)
                );
        }

        [Fact, WorkItem(33365, "https://github.com/dotnet/roslyn/issues/33365")]
        public void AssigningNullToConditionalRefLocal_NullableLocals_NonNullValues()
        {
            var comp = CreateCompilation(@"
class C
{
    void M(ref string? s1, ref string? s2, bool b)
    {
        (b ? ref s1 : ref s1) = """";
        s1.ToString(); // 1

        ref string? s3 = ref s2;
        (b ? ref s3 : ref s3) = """";
        s3.ToString(); // 2

        ref string? s4 = ref s2;
        (b ? ref s4 : ref s4) ??= """";
        s4.ToString(); // 3

        ref string? s5 = ref s2;
        M2(out (b ? ref s5 : ref s5));
        s5.ToString(); // 4
    }
    void M2(out string x) => throw null!;
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (7,9): warning CS8602: Possible dereference of a null reference.
                //         s1.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s1").WithLocation(7, 9),
                // (11,9): warning CS8602: Possible dereference of a null reference.
                //         s3.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s3").WithLocation(11, 9),
                // (15,9): warning CS8602: Possible dereference of a null reference.
                //         s4.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s4").WithLocation(15, 9),
                // (19,9): warning CS8602: Possible dereference of a null reference.
                //         s5.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s5").WithLocation(19, 9)
                );
        }

        [Fact, WorkItem(32495, "https://github.com/dotnet/roslyn/issues/32495")]
        public void CheckImplicitObjectInitializerReceiver()
        {
            var comp = CreateCompilation(@"
public class B
{
    public B? f2;
    public B? f3;
}
public class C
{
    public B? f;
    static void Main()
    {
        new C() { f = { f2 = null, f3 = null }}; // 1
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (12,23): warning CS8670: Object or collection initializer implicitly dereferences possibly null member 'C.f'.
                //         new C() { f = { f2 = null, f3 = null }}; // 1
                Diagnostic(ErrorCode.WRN_NullReferenceInitializer, "{ f2 = null, f3 = null }").WithArguments("C.f").WithLocation(12, 23));
        }

        [Fact, WorkItem(32495, "https://github.com/dotnet/roslyn/issues/32495")]
        public void CheckImplicitObjectInitializerReceiver_Generic()
        {
            var comp = CreateCompilation(@"
public interface IB
{
    object? f2 { get; set; }
    object? f3 { get; set; }
}

public struct StructB : IB
{
    public object? f2 { get; set; }
    public object? f3 { get; set; }
}

public class ClassB : IB
{
    public object? f2 { get; set; }
    public object? f3 { get; set; }
}

public class C<T> where T : IB?
{
    public T f = default!;
    static void Main()
    {
        new C<T>() { f = { f2 = null, f3 = null }}; // 1
        new C<ClassB>() { f = { f2 = null, f3 = null }};
        new C<ClassB?>() { f = { f2 = null, f3 = null }}; // 2
        new C<StructB>() { f = { f2 = null, f3 = null }};
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (25,26): warning CS8670: Object or collection initializer implicitly dereferences possibly null member 'C<T>.f'.
                //         new C<T>() { f = { f2 = null, f3 = null }}; // 1
                Diagnostic(ErrorCode.WRN_NullReferenceInitializer, "{ f2 = null, f3 = null }").WithArguments("C<T>.f").WithLocation(25, 26),
                // (27,32): warning CS8670: Object or collection initializer implicitly dereferences possibly null member 'C<ClassB?>.f'.
                //         new C<ClassB?>() { f = { f2 = null, f3 = null }}; // 2
                Diagnostic(ErrorCode.WRN_NullReferenceInitializer, "{ f2 = null, f3 = null }").WithArguments("C<ClassB?>.f").WithLocation(27, 32));
        }

        [Fact, WorkItem(38060, "https://github.com/dotnet/roslyn/issues/38060")]
        public void CheckImplicitObjectInitializerReceiver_EmptyInitializers()
        {
            var comp = CreateCompilation(@"
public class B { }
public class C
{
    public B? f;
    static void Main()
    {
        new C() { f = { }};
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(32495, "https://github.com/dotnet/roslyn/issues/32495")]
        public void CheckImplicitObjectInitializerReceiver_CollectionInitializer()
        {
            var comp = CreateCompilation(@"
using System.Collections;

public class B : IEnumerable
{
    public void Add(object o) { }
    public IEnumerator GetEnumerator() => null!;
}

public class C
{
    public B? f;
    static void Main()
    {
        new C() { f = { new object(), new object() }}; // 1
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (15,23): warning CS8670: Object or collection initializer implicitly dereferences nullable member 'C.f'.
                //         new C() { f = { new object(), new object() }}; // 1
                Diagnostic(ErrorCode.WRN_NullReferenceInitializer, "{ new object(), new object() }").WithArguments("C.f").WithLocation(15, 23));
        }

        [Fact, WorkItem(32495, "https://github.com/dotnet/roslyn/issues/32495")]
        public void CheckImplicitObjectInitializerReceiver_CollectionInitializer_Generic()
        {
            var comp = CreateCompilation(@"
using System.Collections;

public interface IAddable : IEnumerable
{
    void Add(object o);
}

public class ClassAddable : IAddable
{
    public void Add(object o) { }
    public IEnumerator GetEnumerator() => null!;
}

public struct StructAddable : IAddable
{
    public void Add(object o) { }
    public IEnumerator GetEnumerator() => null!;
}


public class C<T> where T : IAddable?
{
    public T f = default!;
    static void Main()
    {
        new C<T>() { f = { new object(), new object() }}; // 1
        new C<ClassAddable>() { f = { new object(), new object() }};
        new C<ClassAddable?>() { f = { new object(), new object() }}; // 2
        new C<StructAddable>() { f = { new object(), new object() }};
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (27,26): warning CS8670: Object or collection initializer implicitly dereferences possibly null member 'C<T>.f'.
                //         new C<T>() { f = { new object(), new object() }}; // 1
                Diagnostic(ErrorCode.WRN_NullReferenceInitializer, "{ new object(), new object() }").WithArguments("C<T>.f").WithLocation(27, 26),
                // (29,38): warning CS8670: Object or collection initializer implicitly dereferences possibly null member 'C<ClassAddable?>.f'.
                //         new C<ClassAddable?>() { f = { new object(), new object() }}; // 2
                Diagnostic(ErrorCode.WRN_NullReferenceInitializer, "{ new object(), new object() }").WithArguments("C<ClassAddable?>.f").WithLocation(29, 38));
        }

        [Fact, WorkItem(38060, "https://github.com/dotnet/roslyn/issues/38060")]
        public void CheckImplicitObjectInitializerReceiver_EmptyCollectionInitializer()
        {
            var comp = CreateCompilation(@"
public class B
{
    void Add(object o) { }
}
public class C
{
    public B? f;
    static void Main()
    {
        new C() { f = { }};
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(32495, "https://github.com/dotnet/roslyn/issues/32495")]
        public void CheckImplicitObjectInitializerReceiver_ReceiverNotNullable()
        {
            var comp = CreateCompilation(@"
public class B
{
    public B? f2;
}
public class C
{
    public B f;
    static void Main()
    {
        new C() { f = { f2 = null }};
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (8,14): warning CS8618: Non-nullable field 'f' is uninitialized. Consider declaring the field as nullable.
                //     public B f;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "f").WithArguments("field", "f").WithLocation(8, 14)
                );
        }

        [Fact, WorkItem(32495, "https://github.com/dotnet/roslyn/issues/32495")]
        public void CheckImplicitObjectInitializerReceiver_ReceiverOblivious()
        {
            var comp = CreateCompilation(@"
public class B
{
    public B? f2;
}
public class C
{
#nullable disable
    public B f;
#nullable enable
    static void Main()
    {
        new C() { f = { f2 = null }};
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(32495, "https://github.com/dotnet/roslyn/issues/32495")]
        public void CheckImplicitObjectInitializerReceiver_ReceiverNullableIndexer()
        {
            var comp = CreateCompilation(@"
public class B
{
    public B? f2;
}
public class C
{
    static void Main()
    {
        new C() { [0] = { f2 = null }};
    }
    public B? this[int i] { get => throw null!; set => throw null!; }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (10,25): warning CS8670: Object or collection initializer implicitly dereferences possibly null member 'C.this[int]'.
                //         new C() { [0] = { f2 = null }};
                Diagnostic(ErrorCode.WRN_NullReferenceInitializer, "{ f2 = null }").WithArguments("C.this[int]").WithLocation(10, 25));
        }

        [Fact, WorkItem(32495, "https://github.com/dotnet/roslyn/issues/32495")]
        public void CheckImplicitObjectInitializerReceiver_Nested()
        {
            var comp = CreateCompilation(@"
public class B
{
    public B? f2;
}
public class C
{
    public C? fc;
    public B? fb;
    static void Main()
    {
        new C() { fc = { fb = { f2 = null} }};
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (12,24): warning CS8670: Object or collection initializer implicitly dereferences possibly null member 'C.fc'.
                //         new C() { fc = { fb = { f2 = null} }};
                Diagnostic(ErrorCode.WRN_NullReferenceInitializer, "{ fb = { f2 = null} }").WithArguments("C.fc").WithLocation(12, 24),
                // (12,31): warning CS8670: Object or collection initializer implicitly dereferences possibly null member 'C.fb'.
                //         new C() { fc = { fb = { f2 = null} }};
                Diagnostic(ErrorCode.WRN_NullReferenceInitializer, "{ f2 = null}").WithArguments("C.fb").WithLocation(12, 31));
        }

        [Fact, WorkItem(32495, "https://github.com/dotnet/roslyn/issues/32495")]
        public void CheckImplicitObjectInitializerReceiver_Index()
        {
            var comp = CreateCompilation(@"
public class B
{
    public B? this[int i] { get => throw null!; set => throw null!; }
}
public class C
{
    public B? f;
    static void Main()
    {
        new C() { f = { [0] = null }};
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (11,23): warning CS8670: Object or collection initializer implicitly dereferences possibly null member 'C.f'.
                //         new C() { f = { [0] = null }};
                Diagnostic(ErrorCode.WRN_NullReferenceInitializer, "{ [0] = null }").WithArguments("C.f").WithLocation(11, 23));
        }

        [Fact, WorkItem(32495, "https://github.com/dotnet/roslyn/issues/32495")]
        public void CheckImplicitCollectionInitializerReceiver()
        {
            var comp = CreateCompilation(@"
public class B : System.Collections.Generic.IEnumerable<int>
{
    public void Add(int i) { }
    System.Collections.Generic.IEnumerator<int> System.Collections.Generic.IEnumerable<int>.GetEnumerator() => throw null!;
    System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator() => throw null!;
}
public class C
{
    public B? f;
    static void Main()
    {
        new C() { f = { 1 }};
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (13,23): warning CS8670: Object or collection initializer implicitly dereferences possibly null member 'C.f'.
                //         new C() { f = { 1 }};
                Diagnostic(ErrorCode.WRN_NullReferenceInitializer, "{ 1 }").WithArguments("C.f").WithLocation(13, 23));
        }

        [Fact, WorkItem(29964, "https://github.com/dotnet/roslyn/issues/29964")]
        public void IndexerUpdatedBasedOnReceiver_ReturnType()
        {
            var comp = CreateCompilation(@"
using System.Collections.Generic;
class C
{
    static void M(object? o, object o2)
    {
        L(o)[0].ToString(); // 1
        foreach (var x in L(o))
        {
            x.ToString(); // 2
        }

        L(o2)[0].ToString();
    }

    static List<T> L<T>(T t) => null!;
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (7,9): warning CS8602: Dereference of a possibly null reference.
                //         L(o)[0].ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "L(o)[0]").WithLocation(7, 9),
                // (10,13): warning CS8602: Dereference of a possibly null reference.
                //             x.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x").WithLocation(10, 13));
        }

        [Fact, WorkItem(29964, "https://github.com/dotnet/roslyn/issues/29964")]
        public void IndexerUpdatedBasedOnReceiver_ReturnType_Inferred()
        {
            var comp = CreateCompilation(@"
public class C<T>
{
    public T Field = default!;
    public C<T> this[T index] { get => throw null!; set => throw null!; }
}
public class Main
{
    static void M(object? o, object o2)
    {
        if (o is null) return;
        L(o)[0].Field.ToString();

        o2 = null;
        L(o2)[0].Field.ToString(); // 1
    }

    static C<U> L<U>(U u) => null!;
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (14,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         o2 = null;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(14, 14),
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         L(o2)[0].Field.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "L(o2)[0].Field").WithLocation(15, 9)
                );
        }

        [Fact, WorkItem(29964, "https://github.com/dotnet/roslyn/issues/29964")]
        public void IndexerUpdatedBasedOnReceiver_ReturnType_NestedNullability_Inferred()
        {
            var comp = CreateCompilation(@"
public class C<T>
{
    public T this[int index] { get => throw null!; set => throw null!; }
}
public class Program
{
    static void M(object? x)
    {
        var y = L(new[] { x });
        y[0][0].ToString(); // warning

        if (x == null) return;
        var z = L(new[] { x });
        z[0][0].ToString(); // ok
    }
    static C<U> L<U>(U u) => null!;
}
", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (11,9): warning CS8602: Dereference of a possibly null reference.
                //         y[0][0].ToString(); // warning
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "y[0][0]").WithLocation(11, 9)
                );
        }

        [Fact, WorkItem(29964, "https://github.com/dotnet/roslyn/issues/29964")]
        public void IndexerUpdatedBasedOnReceiver_SetterArguments()
        {
            var comp = CreateCompilation(@"
public class C<T>
{
    public int this[T index] { get => throw null!; set => throw null!; }
}
class Program
{
    static void M(object? o, object o2)
    {
        L(o)[o] = 1;
        L(o)[o2] = 2;

        L(o2)[o] = 3; // warn
        L(o2)[o2] = 4;
    }

    static C<U> L<U>(U u) => null!;
}
", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (13,15): warning CS8604: Possible null reference argument for parameter 'index' in 'int C<object>.this[object index]'.
                //         L(o2)[o] = 3; // warn
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "o").WithArguments("index", "int C<object>.this[object index]").WithLocation(13, 15)
                );
        }

        [Fact, WorkItem(29964, "https://github.com/dotnet/roslyn/issues/29964")]
        public void IndexerUpdatedBasedOnReceiver_SetterArguments_Inferred()
        {
            var comp = CreateCompilation(@"
public class C<T>
{
    public int this[T index] { get => throw null!; set => throw null!; }
}
class Program
{
    static void M(object? o, object o2, object? input, object input2)
    {
        if (o is null) return;
        L(o)[input] = 1; // 1
        L(o)[input2] = 2;

        o2 = null;
        L(o2)[input] = 3;
        L(o2)[input2] = 4;
    }

    static C<U> L<U>(U u) => null!;
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (11,14): warning CS8604: Possible null reference argument for parameter 'index' in 'int C<object>.this[object index]'.
                //         L(o)[input] = 1; // 1
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "input").WithArguments("index", "int C<object>.this[object index]").WithLocation(11, 14),
                // (14,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         o2 = null;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(14, 14)
                );
        }

        [Fact, WorkItem(29964, "https://github.com/dotnet/roslyn/issues/29964")]
        public void IndexerUpdatedBasedOnReceiver_GetterArguments()
        {
            var comp = CreateCompilation(@"
public class C<T>
{
    public int this[T index] { get => throw null!; set => throw null!; }
}
class Program
{
    static void M(object? o, object o2)
    {
        _ = L(o)[o];
        _ = L(o)[o2];

        _ = L(o2)[o]; // warn
        _ = L(o2)[o2];
    }

    static C<U> L<U>(U u) => null!;
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (13,19): warning CS8604: Possible null reference argument for parameter 'index' in 'int C<object>.this[object index]'.
                //         _ = L(o2)[o]; // warn
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "o").WithArguments("index", "int C<object>.this[object index]").WithLocation(13, 19)
                );
        }

        [Fact, WorkItem(29964, "https://github.com/dotnet/roslyn/issues/29964")]
        public void IndexerUpdatedBasedOnReceiver_SetterArguments_NestedNullability()
        {
            var comp = CreateCompilation(@"
public class C<T>
{
    public int this[C<T> index] { get => throw null!; set => throw null!; }
}
class Program
{
    static void M(object? o, object o2)
    {
        L(o)[L(o)] = 1;
        L(o)[L(o2)] = 2; // 1

        L(o2)[L(o)] = 3; // 2
        L(o2)[L(o2)] = 4;
    }

    static C<U> L<U>(U u) => null!;
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (11,14): warning CS8620: Argument of type 'C<object>' cannot be used for parameter 'index' of type 'C<object?>' in 'int C<object?>.this[C<object?> index]' due to differences in the nullability of reference types.
                //         L(o)[L(o2)] = 2; // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "L(o2)").WithArguments("C<object>", "C<object?>", "index", "int C<object?>.this[C<object?> index]").WithLocation(11, 14),
                // (13,15): warning CS8620: Argument of type 'C<object?>' cannot be used for parameter 'index' of type 'C<object>' in 'int C<object>.this[C<object> index]' due to differences in the nullability of reference types.
                //         L(o2)[L(o)] = 3; // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "L(o)").WithArguments("C<object?>", "C<object>", "index", "int C<object>.this[C<object> index]").WithLocation(13, 15)
                );
        }

        [Fact, WorkItem(33289, "https://github.com/dotnet/roslyn/issues/33289")]
        public void ConditionalReceiver()
        {
            var comp = CreateCompilation(@"
public class Container<T>
{
    public T Field = default!;
}

class C
{
    static void M(string? s)
    {
        var x = Create(s);
        _ = x /*T:Container<string?>?*/;
        x?.Field.ToString(); // 1
    }

    public static Container<U>? Create<U>(U u) => new Container<U>();
}", options: WithNullableEnable());
            comp.VerifyTypes();
            comp.VerifyDiagnostics(
                // (13,11): warning CS8602: Dereference of a possibly null reference.
                //         x?.Field.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, ".Field").WithLocation(13, 11)
                );
        }

        [Fact, WorkItem(28792, "https://github.com/dotnet/roslyn/issues/28792")]
        public void ConditionalReceiver_Verify28792()
        {
            var comp = CreateNullableCompilation(@"
class C
{
    void M<T>(T t) => t?.ToString();
}");
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(33289, "https://github.com/dotnet/roslyn/issues/33289")]
        public void ConditionalReceiver_Chained()
        {
            var comp = CreateCompilation(@"
public class Container<T>
{
    public T Field = default!;
}

class C
{
    static void M(string? s)
    {
        var x = Create(s);
        if (x is null) return;
        _ = x /*T:Container<string?>!*/;
        x?.Field.ToString(); // 1

        x = Create(s);
        if (x is null) return;
        x.Field?.ToString();
    }

    public static Container<U>? Create<U>(U u) => new Container<U>();
}", options: WithNullableEnable());
            comp.VerifyTypes();
            comp.VerifyDiagnostics(
                // (14,11): warning CS8602: Dereference of a possibly null reference.
                //         x?.Field.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, ".Field").WithLocation(14, 11)
                );
        }

        [Fact, WorkItem(33289, "https://github.com/dotnet/roslyn/issues/33289")]
        public void ConditionalReceiver_Chained_Inversed()
        {
            var comp = CreateCompilation(@"
public class Container<T>
{
    public T Field = default!;
}

class C
{
    static void M(string s)
    {
        var x = Create(s);
        _ = x /*T:Container<string!>?*/;
        x?.Field.ToString();

        x = Create(s);
        x.Field?.ToString(); // 1
    }

    public static Container<U>? Create<U>(U u) => new Container<U>();
}", options: WithNullableEnable());
            comp.VerifyTypes();
            comp.VerifyDiagnostics(
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         x.Field?.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x").WithLocation(16, 9)
                );
        }

        [Fact, WorkItem(33289, "https://github.com/dotnet/roslyn/issues/33289")]
        public void ConditionalReceiver_Nested()
        {
            var comp = CreateCompilation(@"
public class Container<T>
{
    public T Field = default!;
}

class C
{
    static void M(string? s1, string s2)
    {
        var x = Create(s1);
        var y = Create(s2);
        x?.Field /*1*/
            .Extension(y?.Field.ToString());
    }

    public static Container<U>? Create<U>(U u) => new Container<U>();
}
public static class Extensions
{
    public static void Extension(this string s, object? o) => throw null!;
}", options: WithNullableEnable());
            comp.VerifyTypes();
            comp.VerifyDiagnostics(
                // (13,11): warning CS8604: Possible null reference argument for parameter 's' in 'void Extensions.Extension(string s, object? o)'.
                //         x?.Field /*1*/
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, ".Field").WithArguments("s", "void Extensions.Extension(string s, object? o)").WithLocation(13, 11)
                );
        }

        [Fact, WorkItem(33289, "https://github.com/dotnet/roslyn/issues/33289")]
        public void ConditionalReceiver_MiscTypes()
        {
            var comp = CreateCompilation(@"
public class Container<T>
{
    public T M() => default!;
}
class C
{
    static void M<T>(int i, Missing m, string s, T t)
    {
        Create(i)?.M().ToString();
        Create(m)?.M().ToString();
        Create(s)?.M().ToString();
        Create(t)?.M().ToString(); // 1
    }

    public static Container<U>? Create<U>(U u) => new Container<U>();
}", options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (8,29): error CS0246: The type or namespace name 'Missing' could not be found (are you missing a using directive or an assembly reference?)
                //     static void M<T>(int i, Missing m, string s, T t)
                Diagnostic(ErrorCode.ERR_SingleTypeNameNotFound, "Missing").WithArguments("Missing").WithLocation(8, 29),
                // (13,19): warning CS8602: Dereference of a possibly null reference.
                //         Create(t)?.M().ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, ".M()").WithLocation(13, 19)
                );
        }

        [Fact, WorkItem(26810, "https://github.com/dotnet/roslyn/issues/26810")]
        public void LockStatement()
        {
            var comp = CreateCompilation(@"
class C
{
    void F(object? maybeNull, object nonNull, Missing? annotatedMissing, Missing unannotatedMissing)
    {
        lock (maybeNull) { }
        lock (nonNull) { }
        lock (annotatedMissing) { }
        lock (unannotatedMissing) { }
    }
#nullable disable
    void F(object oblivious, Missing obliviousMissing)
#nullable enable
    {
        lock (oblivious) { }
        lock (obliviousMissing) { }
    }
}", options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (4,47): error CS0246: The type or namespace name 'Missing' could not be found (are you missing a using directive or an assembly reference?)
                //     void F(object? maybeNull, object nonNull, Missing? annotatedMissing, Missing unannotatedMissing)
                Diagnostic(ErrorCode.ERR_SingleTypeNameNotFound, "Missing").WithArguments("Missing").WithLocation(4, 47),
                // (4,74): error CS0246: The type or namespace name 'Missing' could not be found (are you missing a using directive or an assembly reference?)
                //     void F(object? maybeNull, object nonNull, Missing? annotatedMissing, Missing unannotatedMissing)
                Diagnostic(ErrorCode.ERR_SingleTypeNameNotFound, "Missing").WithArguments("Missing").WithLocation(4, 74),
                // (6,15): warning CS8602: Dereference of a possibly null reference.
                //         lock (maybeNull) { }
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "maybeNull").WithLocation(6, 15),
                // (8,15): error CS0185: 'Missing?' is not a reference type as required by the lock statement
                //         lock (annotatedMissing) { }
                Diagnostic(ErrorCode.ERR_LockNeedsReference, "annotatedMissing").WithArguments("Missing?").WithLocation(8, 15),
                // (12,30): error CS0246: The type or namespace name 'Missing' could not be found (are you missing a using directive or an assembly reference?)
                //     void F(object oblivious, Missing obliviousMissing)
                Diagnostic(ErrorCode.ERR_SingleTypeNameNotFound, "Missing").WithArguments("Missing").WithLocation(12, 30)
                );
        }

        [Fact, WorkItem(33537, "https://github.com/dotnet/roslyn/issues/33537")]
        public void SuppressOnNullLiteralInAs()
        {
            var comp = CreateCompilation(@"
class C
{
    public static void Main()
    {
        var x = null! as object;
    }
}");
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(26654, "https://github.com/dotnet/roslyn/issues/26654")]
        [WorkItem(27522, "https://github.com/dotnet/roslyn/issues/27522")]
        public void SuppressNullableWarning_DeclarationExpression()
        {
            var source = @"
public class C
{
    public void M(out string? x) => throw null!;
    public void M2()
    {
        M(out string y!);
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            // We don't allow suppressions on out-variable-declarations at this point (LDM 2/13/2019)
            comp.VerifyDiagnostics(
                // (7,23): error CS1003: Syntax error, ',' expected
                //         M(out string y!);
                Diagnostic(ErrorCode.ERR_SyntaxError, "!").WithArguments(",").WithLocation(7, 23),
                // (7,24): error CS1525: Invalid expression term ')'
                //         M(out string y!);
                Diagnostic(ErrorCode.ERR_InvalidExprTerm, ")").WithArguments(")").WithLocation(7, 24)
                );
        }

        [Fact, WorkItem(26654, "https://github.com/dotnet/roslyn/issues/26654")]
        public void SuppressNullableWarning_LocalDeclarationAndAssignmentTarget()
        {
            var source = @"
public class C
{
    public void M2()
    {
        object y! = null;
        object y2;
        y2! = null;
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            // We don't allow suppressions in those positions at this point (LDM 2/13/2019)
            comp.VerifyDiagnostics(
                // (6,16): warning CS0168: The variable 'y' is declared but never used
                //         object y! = null;
                Diagnostic(ErrorCode.WRN_UnreferencedVar, "y").WithArguments("y").WithLocation(6, 16),
                // (6,17): error CS1002: ; expected
                //         object y! = null;
                Diagnostic(ErrorCode.ERR_SemicolonExpected, "!").WithLocation(6, 17),
                // (6,19): error CS1525: Invalid expression term '='
                //         object y! = null;
                Diagnostic(ErrorCode.ERR_InvalidExprTerm, "=").WithArguments("=").WithLocation(6, 19),
                // (7,16): warning CS0219: The variable 'y2' is assigned but its value is never used
                //         object y2;
                Diagnostic(ErrorCode.WRN_UnreferencedVarAssg, "y2").WithArguments("y2").WithLocation(7, 16),
                // (8,9): error CS8598: The suppression operator is not allowed in this context
                //         y2! = null;
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "y2").WithLocation(8, 9),
                // (8,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         y2! = null;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(8, 15)
                );
        }

        [Fact]
        [WorkItem(788968, "https://devdiv.visualstudio.com/DevDiv/_workitems/edit/788968")]
        public void MissingMethodOnTupleLiteral()
        {
            var source = @"
#nullable enable
class C
{
    void M()
    {
        (0, (string?)null).Missing();
        _ = (0, (string?)null).Missing;
    }
}
";
            var compilation = CreateCompilation(source);
            compilation.VerifyDiagnostics(
                // (7,28): error CS1061: '(int, string)' does not contain a definition for 'Missing' and no accessible extension method 'Missing' accepting a first argument of type '(int, string)' could be found (are you missing a using directive or an assembly reference?)
                //         (0, (string?)null).Missing();
                Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "Missing").WithArguments("(int, string)", "Missing").WithLocation(7, 28),
                // (8,32): error CS1061: '(int, string)' does not contain a definition for 'Missing' and no accessible extension method 'Missing' accepting a first argument of type '(int, string)' could be found (are you missing a using directive or an assembly reference?)
                //         _ = (0, (string?)null).Missing;
                Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "Missing").WithArguments("(int, string)", "Missing").WithLocation(8, 32)
                );
        }

        [Fact, WorkItem(41520, "https://github.com/dotnet/roslyn/issues/41520")]
        public void WarnInsideTupleLiteral()
        {
            var source = @"
class C
{
    (int, int) M(string? s)
    {
        return (s.Length, 1);
    }
}
";
            var compilation = CreateNullableCompilation(source);
            compilation.VerifyDiagnostics(
                // (6,17): warning CS8602: Dereference of a possibly null reference.
                //         return (s.Length, 1);
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(6, 17)
                );
        }

        [Theory]
        [InlineData(LanguageVersion.CSharp10)]
        [InlineData(LanguageVersion.CSharp11)]
        public void SuppressNullableWarning_RefReturn(LanguageVersion langVersion)
        {
            var source =
@"#nullable enable
ref struct R<T>
{
    public R(ref T t) { }
}
class C
{
    static ref R<string?> M(bool b)
    {
        string s = string.Empty;
        R<string> r = new R<string>(ref s);
        if (b)
            return ref r; // 1
        else
            return ref r!; // 2
    }
}";
            var comp = CreateCompilation(source, parseOptions: TestOptions.Regular.WithLanguageVersion(langVersion));
            comp.VerifyDiagnostics(
                // (13,24): error CS8168: Cannot return local 'r' by reference because it is not a ref local
                //             return ref r; // 1
                Diagnostic(ErrorCode.ERR_RefReturnLocal, "r").WithArguments("r").WithLocation(13, 24),
                // (13,24): warning CS8619: Nullability of reference types in value of type 'R<string>' doesn't match target type 'R<string?>'.
                //             return ref r; // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "r").WithArguments("R<string>", "R<string?>").WithLocation(13, 24),
                // (15,24): error CS8168: Cannot return local 'r' by reference because it is not a ref local
                //             return ref r!; // 2
                Diagnostic(ErrorCode.ERR_RefReturnLocal, "r").WithArguments("r").WithLocation(15, 24),
                // (15,24): warning CS8619: Nullability of reference types in value of type 'R<string>' doesn't match target type 'R<string?>'.
                //             return ref r!; // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "r").WithArguments("R<string>", "R<string?>").WithLocation(15, 24));
        }

        [WorkItem(31297, "https://github.com/dotnet/roslyn/issues/31297")]
        [Theory]
        [InlineData(LanguageVersion.CSharp10)]
        [InlineData(LanguageVersion.CSharp11)]
        public void SuppressNullableWarning_RefSpanReturn(LanguageVersion languageVersion)
        {
            var source =
@"#nullable enable
using System;
class C
{
    ref Span<byte> M(bool b)
    {
        Span<byte> x = stackalloc byte[10];
        ref Span<byte> y = ref x!;
        if (b)
            return ref y; // 1
        else
        return ref y!; // 2
    }
    ref Span<string> M2(ref Span<string?> x, bool b)
    {
        if (b)
            return ref x; // 3
        else
            return ref x!;
    }
}";

            var comp = CreateCompilationWithMscorlibAndSpan(source, parseOptions: TestOptions.Regular.WithLanguageVersion(languageVersion), options: TestOptions.ReleaseDll);
            comp.VerifyDiagnostics(
                // (10,24): error CS8157: Cannot return 'y' by reference because it was initialized to a value that cannot be returned by reference
                //             return ref y; // 1
                Diagnostic(ErrorCode.ERR_RefReturnNonreturnableLocal, "y").WithArguments("y").WithLocation(10, 24),
                // (12,20): error CS8157: Cannot return 'y' by reference because it was initialized to a value that cannot be returned by reference
                //         return ref y!; // 2
                Diagnostic(ErrorCode.ERR_RefReturnNonreturnableLocal, "y").WithArguments("y").WithLocation(12, 20),
                // (17,24): warning CS8619: Nullability of reference types in value of type 'System.Span<string?>' doesn't match target type 'System.Span<string>'.
                //             return ref x; // 3
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("System.Span<string?>", "System.Span<string>").WithLocation(17, 24)
                );
        }

        [Fact, WorkItem(31297, "https://github.com/dotnet/roslyn/issues/31297")]
        public void SuppressNullableWarning_RefReassignment()
        {
            var comp = CreateCompilation(@"
class C
{
    void M()
    {
        ref string x = ref NullableRef(); // 1
        ref string x2 = ref x; // 2
        ref string x3 = ref x!;

        ref string y = ref NullableRef()!;
        ref string y2 = ref y;
        ref string y3 = ref y!;
    }

    ref string? NullableRef() => throw null!;
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (6,28): warning CS8619: Nullability of reference types in value of type 'string?' doesn't match target type 'string'.
                //         ref string x = ref NullableRef(); // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "NullableRef()").WithArguments("string?", "string").WithLocation(6, 28),
                // (7,29): warning CS8601: Possible null reference assignment.
                //         ref string x2 = ref x; // 2
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "x").WithLocation(7, 29)
                );
        }

        [Fact]
        public void SuppressNullableWarning_This()
        {
            var comp = CreateCompilation(@"
class C
{
    void M()
    {
        this!.M();
    }
}", options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void SuppressNullableWarning_UnaryOperator()
        {
            var comp = CreateCompilation(@"
class C
{
    void M(C? y)
    {
        y++;
        y!++;
        y--;
    }
    public static C operator++(C x) => throw null!;
    public static C operator--(C? x) => throw null!;
}", options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,9): warning CS8604: Possible null reference argument for parameter 'x' in 'C C.operator ++(C x)'.
                //         y++;
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "y").WithArguments("x", "C C.operator ++(C x)").WithLocation(6, 9)
                );
        }

        [Fact, WorkItem(30151, "https://github.com/dotnet/roslyn/issues/30151")]
        public void SuppressNullableWarning_WholeArrayInitializer()
        {
            var comp = CreateCompilationWithMscorlibAndSpan(@"
class C
{
    unsafe void M()
    {
        string[] s = new[] { null, string.Empty }; // expecting a warning
        string[] s2 = (new[] { null, string.Empty })!;
        int* s3 = (stackalloc[] { 1 })!;
        System.Span<string> s4 = (stackalloc[] { null, string.Empty })!;
    }
}", options: TestOptions.UnsafeDebugDll);

            // Missing warning
            // Need to confirm whether this suppression should be allowed or be effective
            // If so, we need to verify the semantic model
            // Tracked by https://github.com/dotnet/roslyn/issues/30151
            comp.VerifyDiagnostics(
                // (8,20): error CS8346: Conversion of a stackalloc expression of type 'int' to type 'int*' is not possible.
                //         int* s3 = (stackalloc[] { 1 })!;
                Diagnostic(ErrorCode.ERR_StackAllocConversionNotPossible, "stackalloc[] { 1 }").WithArguments("int", "int*").WithLocation(8, 20),
                // (9,35): error CS0208: Cannot take the address of, get the size of, or declare a pointer to a managed type ('string')
                //         System.Span<string> s4 = (stackalloc[] { null, string.Empty })!;
                Diagnostic(ErrorCode.ERR_ManagedAddr, "stackalloc[] { null, string.Empty }").WithArguments("string").WithLocation(9, 35));
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_NullCoalescingAssignment()
        {
            var comp = CreateCompilation(@"
class C
{
    void M(int? i, int i2, dynamic d)
    {
        i! ??= i2; // 1
        i ??= i2!;
        i! ??= d; // 2
        i ??= d!;
    }
    void M(string? s, string s2, dynamic d)
    {
        s! ??= s2; // 3
        s ??= s2!;
        s! ??= d; // 4
        s ??= d!;
    }
}", options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,9): error CS8598: The suppression operator is not allowed in this context
                //         i! ??= i2; // 1
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "i").WithLocation(6, 9),
                // (8,9): error CS8598: The suppression operator is not allowed in this context
                //         i! ??= d; // 2
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "i").WithLocation(8, 9),
                // (13,9): error CS8598: The suppression operator is not allowed in this context
                //         s! ??= s2; // 3
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "s").WithLocation(13, 9),
                // (15,9): error CS8598: The suppression operator is not allowed in this context
                //         s! ??= d; // 4
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "s").WithLocation(15, 9)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_ExpressionTree()
        {
            var comp = CreateCompilation(@"
using System;
using System.Linq.Expressions;
class C
{
    void M()
    {
        string x = null;
        Expression<Func<string>> e = () => x!;
        Expression<Func<string>> e2 = (() => x)!;
        Expression<Func<Func<string>>> e3 = () => M2!;
    }
    string M2() => throw null!;
}");
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_QueryReceiver()
        {
            string source = @"
using System;
namespace NS { }
static class C
{
    static void Main()
    {
        _ = from x in null! select x;
        _ = from x in default! select x;
        _ = from x in (y => y)! select x;
        _ = from x in NS! select x;
        _ = from x in Main! select x;
    }

    static object Select(this object x, Func<int, int> y)
    {
        return null;
    }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (8,29): error CS0186: Use of null is not valid in this context
                //         _ = from x in null! select x;
                Diagnostic(ErrorCode.ERR_NullNotValid, "select x").WithLocation(8, 29),
                // (9,23): error CS8716: There is no target type for the default literal.
                //         _ = from x in default! select x;
                Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(9, 23),
                // (10,33): error CS1936: Could not find an implementation of the query pattern for source type 'anonymous method'.  'Select' not found.
                //         _ = from x in (y => y)! select x;
                Diagnostic(ErrorCode.ERR_QueryNoProvider, "select x").WithArguments("anonymous method", "Select").WithLocation(10, 33),
                // (11,23): error CS8598: The suppression operator is not allowed in this context
                //         _ = from x in NS! select x;
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "NS").WithLocation(11, 23),
                // (11,23): error CS0119: 'NS' is a namespace, which is not valid in the given context
                //         _ = from x in NS! select x;
                Diagnostic(ErrorCode.ERR_BadSKunknown, "NS").WithArguments("NS", "namespace").WithLocation(11, 23),
                // (12,23): error CS0119: 'C.Main()' is a method, which is not valid in the given context
                //         _ = from x in Main! select x;
                Diagnostic(ErrorCode.ERR_BadSKunknown, "Main").WithArguments("C.Main()", "method").WithLocation(12, 23)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_Query()
        {
            string source = @"
using System.Linq;
class C
{
    static void M(System.Collections.Generic.IEnumerable<int> c)
    {
        var q = (from x in c select x)!;
    }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_EventInCompoundAssignment()
        {
            var comp = CreateCompilation(@"
class C
{
    public event System.Action E;
    void M()
    {
        E! += () => {};
        E();
    }
}");
            comp.VerifyDiagnostics(
                // (7,9): error CS8598: The suppression operator is not allowed in this context
                //         E! += () => {};
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "E").WithLocation(7, 9)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void OperationsInNonDeclaringType()
        {
            var text = @"
class C
{
    public event System.Action E;
}

class D
{
    void Method(ref System.Action a, C c)
    {
        c.E! = a; //CS0070
        c.E! += a;
        a = c.E!; //CS0070
        Method(ref c.E!, c); //CS0070
        c.E!.Invoke(); //CS0070
        bool b1 = c.E! is System.Action; //CS0070
        c.E!++; //CS0070
        c.E! |= true; //CS0070
    }
}
";
            CreateCompilation(text).VerifyDiagnostics(
                // (11,11): error CS0070: The event 'C.E' can only appear on the left hand side of += or -= (except when used from within the type 'C')
                //         c.E! = a; //CS0070
                Diagnostic(ErrorCode.ERR_BadEventUsage, "E").WithArguments("C.E", "C").WithLocation(11, 11),
                // (12,9): error CS8598: The suppression operator is not allowed in this context
                //         c.E! += a;
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "c.E").WithLocation(12, 9),
                // (13,15): error CS0070: The event 'C.E' can only appear on the left hand side of += or -= (except when used from within the type 'C')
                //         a = c.E!; //CS0070
                Diagnostic(ErrorCode.ERR_BadEventUsage, "E").WithArguments("C.E", "C").WithLocation(13, 15),
                // (14,22): error CS0070: The event 'C.E' can only appear on the left hand side of += or -= (except when used from within the type 'C')
                //         Method(ref c.E!, c); //CS0070
                Diagnostic(ErrorCode.ERR_BadEventUsage, "E").WithArguments("C.E", "C").WithLocation(14, 22),
                // (15,11): error CS0070: The event 'C.E' can only appear on the left hand side of += or -= (except when used from within the type 'C')
                //         c.E!.Invoke(); //CS0070
                Diagnostic(ErrorCode.ERR_BadEventUsage, "E").WithArguments("C.E", "C").WithLocation(15, 11),
                // (16,21): error CS0070: The event 'C.E' can only appear on the left hand side of += or -= (except when used from within the type 'C')
                //         bool b1 = c.E! is System.Action; //CS0070
                Diagnostic(ErrorCode.ERR_BadEventUsage, "E").WithArguments("C.E", "C").WithLocation(16, 21),
                // (17,11): error CS0070: The event 'C.E' can only appear on the left hand side of += or -= (except when used from within the type 'C')
                //         c.E!++; //CS0070
                Diagnostic(ErrorCode.ERR_BadEventUsage, "E").WithArguments("C.E", "C").WithLocation(17, 11),
                // (18,9): error CS8598: The suppression operator is not allowed in this context
                //         c.E! |= true; //CS0070
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "c.E").WithLocation(18, 9),
                // (18,11): error CS0070: The event 'C.E' can only appear on the left hand side of += or -= (except when used from within the type 'C')
                //         c.E! |= true; //CS0070
                Diagnostic(ErrorCode.ERR_BadEventUsage, "E").WithArguments("C.E", "C").WithLocation(18, 11)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_Fixed()
        {
            var text = @"
public class MyClass
{
    int field = 0;
    unsafe public void Main()
    {
        int i = 45;
        fixed (int *j = &(i!)) { }
        fixed (int *k = &(this!.field)) { }

        int[] a = new int[] {1,2,3};
        fixed (int *b = a!)
        {
            fixed (int *c = b!) { }
        }
    }
}
";
            CreateCompilation(text, options: TestOptions.UnsafeReleaseDll).VerifyDiagnostics(
                // (8,25): error CS0213: You cannot use the fixed statement to take the address of an already fixed expression
                //         fixed (int *j = &(i!)) { }
                Diagnostic(ErrorCode.ERR_FixedNotNeeded, "&(i!)").WithLocation(8, 25),
                // (8,27): error CS8598: The suppression operator is not allowed in this context
                //         fixed (int *j = &(i!)) { }
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "i").WithLocation(8, 27),
                // (14,29): error CS8385: The given expression cannot be used in a fixed statement
                //             fixed (int *c = b!) { }
                Diagnostic(ErrorCode.ERR_ExprCannotBeFixed, "b").WithLocation(14, 29)
                );
        }

        [Fact, WorkItem(31748, "https://github.com/dotnet/roslyn/issues/31748")]
        public void NullableRangeIndexer()
        {
            var text = @"
#nullable enable
class Program
{
    void M(int[] x, string m)
    {
        var y = x[1..3];
        var z = m[1..3];
    }
}" + TestSources.GetSubArray;
            CreateCompilationWithIndexAndRange(text).VerifyDiagnostics();
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_MemberAccess()
        {
            var comp = CreateCompilation(@"
namespace NS
{
    public static class C
    {
        public static void M()
        {
            _ = null!.field; // 1
            _ = default!.field; // 2
            _ = NS!.C.field; // 3
            _ = NS.C!.field; // 4
            _ = nameof(C!.M); // 5
            _ = nameof(C.M!); // 6
            _ = nameof(missing!); // 7
        }
    }
    public class Base { public virtual void M() { } }
    public class D : Base
    {
        public override void M()
        {
            _ = this!.ToString(); // 8
            _ = base!.ToString(); // 9
        }
    }
}");
            // Like cast, suppressions are allowed on `this`, but not on `base`

            comp.VerifyDiagnostics(
                // (8,17): error CS0023: Operator '.' cannot be applied to operand of type '<null>'
                //             _ = null!.field; // 1
                Diagnostic(ErrorCode.ERR_BadUnaryOp, "null!.field").WithArguments(".", "<null>").WithLocation(8, 17),
                // (9,17): error CS8716: There is no target type for the default literal.
                //             _ = default!.field; // 2
                Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(9, 17),
                // (10,17): error CS8598: The suppression operator is not allowed in this context
                //             _ = NS!.C.field; // 3
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "NS").WithLocation(10, 17),
                // (10,23): error CS0117: 'C' does not contain a definition for 'field'
                //             _ = NS!.C.field; // 3
                Diagnostic(ErrorCode.ERR_NoSuchMember, "field").WithArguments("NS.C", "field").WithLocation(10, 23),
                // (11,17): error CS8598: The suppression operator is not allowed in this context
                //             _ = NS.C!.field; // 4
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "NS.C").WithLocation(11, 17),
                // (11,23): error CS0117: 'C' does not contain a definition for 'field'
                //             _ = NS.C!.field; // 4
                Diagnostic(ErrorCode.ERR_NoSuchMember, "field").WithArguments("NS.C", "field").WithLocation(11, 23),
                // (12,24): error CS8598: The suppression operator is not allowed in this context
                //             _ = nameof(C!.M); // 5
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "C").WithLocation(12, 24),
                // (12,24): error CS8082: Sub-expression cannot be used in an argument to nameof.
                //             _ = nameof(C!.M); // 5
                Diagnostic(ErrorCode.ERR_SubexpressionNotInNameof, "C!").WithLocation(12, 24),
                // (13,24): error CS8081: Expression does not have a name.
                //             _ = nameof(C.M!); // 6
                Diagnostic(ErrorCode.ERR_ExpressionHasNoName, "C.M!").WithLocation(13, 24),
                // (14,24): error CS0103: The name 'missing' does not exist in the current context
                //             _ = nameof(missing!); // 7
                Diagnostic(ErrorCode.ERR_NameNotInContext, "missing").WithArguments("missing").WithLocation(14, 24),
                // (23,17): error CS0175: Use of keyword 'base' is not valid in this context
                //             _ = base!.ToString(); // 9
                Diagnostic(ErrorCode.ERR_BaseIllegal, "base").WithLocation(23, 17)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SymbolInfoForMethodGroup03()
        {
            var source = @"
public class A
{
    public static void M(A a)
    {
        _ = nameof(a.Extension!);
    }
}
public static class X1
{
    public static string Extension(this A a) { return null; }
}";
            var compilation = CreateCompilationWithMscorlib40AndSystemCore(source);
            compilation.VerifyDiagnostics(
                // (6,20): error CS8081: Expression does not have a name.
                //         _ = nameof(a.Extension!);
                Diagnostic(ErrorCode.ERR_ExpressionHasNoName, "a.Extension!").WithLocation(6, 20)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_StringInterpolation()
        {
            var source = @"
public class C
{
    public static void Main()
    {
        M(""world"", null);
    }
    public static void M(string x, string? y)
    {
        System.IFormattable z = $""hello ""!;
        System.IFormattable z2 = $""{x} {y} {y!}""!;
        System.Console.Write(z);
        System.Console.Write(z2);
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics();
            CompileAndVerify(comp, expectedOutput: "hello world");

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);

            var suppression = tree.GetRoot().DescendantNodes().OfType<PostfixUnaryExpressionSyntax>().First();
            Assert.Equal(@"$""hello ""!", suppression.ToString());
            VerifyTypeInfo(model, suppression, "System.String", "System.IFormattable");

            var interpolated = suppression.Operand;
            Assert.Equal(@"$""hello """, interpolated.ToString());
            VerifyTypeInfo(model, interpolated, "System.String", "System.IFormattable");
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_StringInterpolation_ExplicitCast()
        {
            var source = @"
public class C
{
    public static void Main()
    {
        M(""world"", null);
    }
    public static void M(string x, string? y)
    {
        var z = (System.IFormattable)($""hello ""!);
        var z2 = (System.IFormattable)($""{x} {y} {y!}""!);
        System.Console.Write(z);
        System.Console.Write(z2);
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics();
            CompileAndVerify(comp, expectedOutput: "hello world");

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);

            var suppression = tree.GetRoot().DescendantNodes().OfType<PostfixUnaryExpressionSyntax>().First();

            Assert.Equal(@"$""hello ""!", suppression.ToString());
            VerifyTypeInfo(model, suppression, "System.String", "System.String");

            var interpolated = suppression.Operand;
            Assert.Equal(@"$""hello """, interpolated.ToString());
            VerifyTypeInfo(model, interpolated, "System.String", "System.String");
        }

        private static void VerifyTypeInfo(SemanticModel model, ExpressionSyntax expression, string expectedType, string expectedConvertedType)
        {
            var type = model.GetTypeInfoAndVerifyIOperation(expression);
            if (expectedType is null)
            {
                Assert.Null(type.Type);
            }
            else
            {
                var actualType = type.Type.ToTestDisplayString();
                Assert.True(expectedType == actualType, $"Unexpected TypeInfo.Type '{actualType}'");
            }

            if (expectedConvertedType is null)
            {
                Assert.Null(type.ConvertedType);
            }
            else
            {
                var actualConvertedType = type.ConvertedType.ToTestDisplayString();
                Assert.True(expectedConvertedType == actualConvertedType, $"Unexpected TypeInfo.ConvertedType '{actualConvertedType}'");
            }
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_Default()
        {
            var source =
@"class C
{
    static void M()
    {
        string s = default!;
        s.ToString();
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics();
            CompileAndVerify(comp);

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);

            var suppression = tree.GetRoot().DescendantNodes().OfType<PostfixUnaryExpressionSyntax>().Single();
            VerifyTypeInfo(model, suppression, "System.String", "System.String");

            var literal = suppression.Operand;
            VerifyTypeInfo(model, literal, "System.String", "System.String");
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_Constant()
        {
            var source =
@"class C
{
    static void M()
    {
        const int i = 3!;
        _ = i;
        const string s = null!;
        _ = s;
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics();

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);
            var suppressions = tree.GetRoot().DescendantNodes().OfType<PostfixUnaryExpressionSyntax>().ToArray();

            Assert.Equal(3, model.GetConstantValue(suppressions[0]).Value);
            Assert.Null(model.GetConstantValue(suppressions[1]).Value);
        }

        [Fact]
        public void SuppressNullableWarning_GetTypeInfo()
        {
            var source =
@"class C<T>
{
    void M(string s, string? s2, C<string> c, C<string?> c2)
    {
        _ = s!;
        _ = s2!;
        _ = c!;
        _ = c2!;
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics();

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);
            var suppressions = tree.GetRoot().DescendantNodes().OfType<PostfixUnaryExpressionSyntax>().ToArray();

            var s = model.GetTypeInfoAndVerifyIOperation(suppressions[0]);
            Assert.Equal("System.String", s.Type.ToTestDisplayString());
            Assert.Equal("System.String", s.ConvertedType.ToTestDisplayString());
            Assert.Equal("System.String s", model.GetSymbolInfo(suppressions[0]).Symbol.ToTestDisplayString());

            var s2 = model.GetTypeInfoAndVerifyIOperation(suppressions[1]);
            Assert.Equal("System.String", s2.Type.ToTestDisplayString());
            Assert.Equal("System.String", s2.ConvertedType.ToTestDisplayString());
            Assert.Equal("System.String? s2", model.GetSymbolInfo(suppressions[1]).Symbol.ToTestDisplayString());

            var c = model.GetTypeInfoAndVerifyIOperation(suppressions[2]);
            Assert.Equal("C<System.String>", c.Type.ToTestDisplayString());
            Assert.Equal("C<System.String>", c.ConvertedType.ToTestDisplayString());
            Assert.Equal("C<System.String> c", model.GetSymbolInfo(suppressions[2]).Symbol.ToTestDisplayString());

            var c2 = model.GetTypeInfoAndVerifyIOperation(suppressions[3]);
            Assert.Equal("C<System.String?>", c2.Type.ToTestDisplayString());
            Assert.Equal("C<System.String?>", c2.ConvertedType.ToTestDisplayString());
            Assert.Equal("C<System.String?> c2", model.GetSymbolInfo(suppressions[3]).Symbol.ToTestDisplayString());
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_MethodGroupInNameof()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static void M<T>()
    {
        _ = nameof(C.M<T>!);
    }
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (6,24): error CS0119: 'T' is a type, which is not valid in the given context
                //         _ = nameof(C.M<T>!);
                Diagnostic(ErrorCode.ERR_BadSKunknown, "T").WithArguments("T", "type").WithLocation(6, 24),
                // (6,27): error CS1525: Invalid expression term ')'
                //         _ = nameof(C.M<T>!);
                Diagnostic(ErrorCode.ERR_InvalidExprTerm, ")").WithArguments(")").WithLocation(6, 27)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_MethodGroup()
        {
            var source =
@"class C
{
    delegate string Copier(string s);
    static void Main()
    {
        Copier c = M2;
        Copier c2 = M2!;
    }
    static string? M2(string? s) => throw null!;
}";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics(
                // (6,20): warning CS8621: Nullability of reference types in return type of 'string? C.M2(string? s)' doesn't match the target delegate 'C.Copier'.
                //         Copier c = M2;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOfTargetDelegate, "M2").WithArguments("string? C.M2(string? s)", "C.Copier").WithLocation(6, 20)
                );
            CompileAndVerify(comp);

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);

            var suppression = tree.GetRoot().DescendantNodes().OfType<PostfixUnaryExpressionSyntax>().First();
            Assert.Equal("M2!", suppression.ToString());
            VerifyTypeInfo(model, suppression, null, "C.Copier");

            var methodGroup = suppression.Operand;
            VerifyTypeInfo(model, methodGroup, null, "C.Copier");
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void InvalidUseOfMethodGroup()
        {
            var source =
@"class A
{
    internal object E() { return null; }
    private object F() { return null; }
    static void M(A a)
    {
        object o;
        a.E! += a.E!; // 1
        if (a.E! != null) // 2
        {
            M(a.E!); // 3
            a.E!.ToString(); // 4
            a.P!.ToString(); // 5
            o = !(a.E!); // 6
            o = a.E! ?? a.F!; // 7
        }
        a.F! += a.F!; // 8
        if (a.F! != null) // 9
        {
            M(a.F!); // 10
            a.F!.ToString(); // 11
            o = !(a.F!); // 12
            o = (o != null) ? a.E! : a.F!; // 13
        }
    }
}";
            CreateCompilation(source, parseOptions: TestOptions.Regular9).VerifyDiagnostics(
                // (8,9): error CS1656: Cannot assign to 'E' because it is a 'method group'
                //         a.E! += a.E!; // 1
                Diagnostic(ErrorCode.ERR_AssgReadonlyLocalCause, "a.E").WithArguments("E", "method group").WithLocation(8, 9),
                // (9,13): error CS0019: Operator '!=' cannot be applied to operands of type 'method group' and '<null>'
                //         if (a.E! != null) // 2
                Diagnostic(ErrorCode.ERR_BadBinaryOps, "a.E! != null").WithArguments("!=", "method group", "<null>").WithLocation(9, 13),
                // (11,15): error CS1503: Argument 1: cannot convert from 'method group' to 'A'
                //             M(a.E!); // 3
                Diagnostic(ErrorCode.ERR_BadArgType, "a.E").WithArguments("1", "method group", "A").WithLocation(11, 15),
                // (12,15): error CS0119: 'A.E()' is a method, which is not valid in the given context
                //             a.E!.ToString(); // 4
                Diagnostic(ErrorCode.ERR_BadSKunknown, "E").WithArguments("A.E()", "method").WithLocation(12, 15),
                // (13,15): error CS1061: 'A' does not contain a definition for 'P' and no accessible extension method 'P' accepting a first argument of type 'A' could be found (are you missing a using directive or an assembly reference?)
                //             a.P!.ToString(); // 5
                Diagnostic(ErrorCode.ERR_NoSuchMemberOrExtension, "P").WithArguments("A", "P").WithLocation(13, 15),
                // (14,17): error CS0023: Operator '!' cannot be applied to operand of type 'method group'
                //             o = !(a.E!); // 6
                Diagnostic(ErrorCode.ERR_BadUnaryOp, "!(a.E!)").WithArguments("!", "method group").WithLocation(14, 17),
                // (15,17): error CS0019: Operator '??' cannot be applied to operands of type 'method group' and 'method group'
                //             o = a.E! ?? a.F!; // 7
                Diagnostic(ErrorCode.ERR_BadBinaryOps, "a.E! ?? a.F!").WithArguments("??", "method group", "method group").WithLocation(15, 17),
                // (17,9): error CS1656: Cannot assign to 'F' because it is a 'method group'
                //         a.F! += a.F!; // 8
                Diagnostic(ErrorCode.ERR_AssgReadonlyLocalCause, "a.F").WithArguments("F", "method group").WithLocation(17, 9),
                // (18,13): error CS0019: Operator '!=' cannot be applied to operands of type 'method group' and '<null>'
                //         if (a.F! != null) // 9
                Diagnostic(ErrorCode.ERR_BadBinaryOps, "a.F! != null").WithArguments("!=", "method group", "<null>").WithLocation(18, 13),
                // (20,15): error CS1503: Argument 1: cannot convert from 'method group' to 'A'
                //             M(a.F!); // 10
                Diagnostic(ErrorCode.ERR_BadArgType, "a.F").WithArguments("1", "method group", "A").WithLocation(20, 15),
                // (21,15): error CS0119: 'A.F()' is a method, which is not valid in the given context
                //             a.F!.ToString(); // 11
                Diagnostic(ErrorCode.ERR_BadSKunknown, "F").WithArguments("A.F()", "method").WithLocation(21, 15),
                // (22,17): error CS0023: Operator '!' cannot be applied to operand of type 'method group'
                //             o = !(a.F!); // 12
                Diagnostic(ErrorCode.ERR_BadUnaryOp, "!(a.F!)").WithArguments("!", "method group").WithLocation(22, 17),
                // (23,33): error CS0428: Cannot convert method group 'E' to non-delegate type 'object'. Did you intend to invoke the method?
                //             o = (o != null) ? a.E! : a.F!; // 13
                Diagnostic(ErrorCode.ERR_MethGrpToNonDel, "E").WithArguments("E", "object").WithLocation(23, 33),
                // (23,40): error CS0428: Cannot convert method group 'F' to non-delegate type 'object'. Did you intend to invoke the method?
                //             o = (o != null) ? a.E! : a.F!; // 13
                Diagnostic(ErrorCode.ERR_MethGrpToNonDel, "F").WithArguments("F", "object").WithLocation(23, 40)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_AccessPropertyWithoutArguments()
        {
            string source1 = @"
Imports System
Imports System.Runtime.InteropServices
<Assembly: PrimaryInteropAssembly(0, 0)>
<Assembly: Guid(""165F752D-E9C4-4F7E-B0D0-CDFD7A36E210"")>
<ComImport()>
<Guid(""165F752D-E9C4-4F7E-B0D0-CDFD7A36E211"")>
Public Interface IB
    Property Value(Optional index As Object = Nothing) As Object
End Interface
";

            var reference = BasicCompilationUtils.CompileToMetadata(source1);

            string source2 = @"
class CIB : IB
{
    public dynamic get_Value(object index = null)
    {
        return ""Test"";
    }

    public void set_Value(object index = null, object Value = null)
    {
    }
}

class Test
{
    static void Main()
    {
        IB x = new CIB();
        System.Console.WriteLine(x.Value!.Length);
    }
}
";

            var compilation2 = CreateCompilation(source2, new[] { reference.WithEmbedInteropTypes(true), CSharpRef }, options: TestOptions.ReleaseExe);

            CompileAndVerify(compilation2, expectedOutput: @"4");
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_CollectionInitializerProperty()
        {
            var source = @"
public class C
{
    public string P { get; set; } = null!;
    void M()
    {
        _ = new C() { P! = null };
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (7,21): error CS1922: Cannot initialize type 'C' with a collection initializer because it does not implement 'System.Collections.IEnumerable'
                //         _ = new C() { P! = null };
                Diagnostic(ErrorCode.ERR_CollectionInitRequiresIEnumerable, "{ P! = null }").WithArguments("C").WithLocation(7, 21),
                // (7,23): error CS0747: Invalid initializer member declarator
                //         _ = new C() { P! = null };
                Diagnostic(ErrorCode.ERR_InvalidInitializerElementInitializer, "P! = null").WithLocation(7, 23),
                // (7,23): error CS8598: The suppression operator is not allowed in this context
                //         _ = new C() { P! = null };
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "P").WithLocation(7, 23),
                // (7,28): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         _ = new C() { P! = null };
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(7, 28)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_InInvocation()
        {
            var source = @"
public class C
{
    public System.Action? field = null;
    void M()
    {
        this.M!();
        nameof!(M);
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (7,9): error CS8598: The suppression operator is not allowed in this context
                //         this.M!();
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "this.M").WithLocation(7, 9),
                // (8,9): error CS0103: The name 'nameof' does not exist in the current context
                //         nameof!(M);
                Diagnostic(ErrorCode.ERR_NameNotInContext, "nameof").WithArguments("nameof").WithLocation(8, 9)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_InInvocation2()
        {
            var source = @"
public class C
{
    public System.Action? field = null;
    void M()
    {
        this!.field!();
    }
}";
            var comp = CreateCompilationWithCSharp(source, options: WithNullableEnable());
            comp.VerifyDiagnostics();
            CompileAndVerify(comp);
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_InInvocationAndDynamic()
        {
            var source = @"
public class C
{
    void M()
    {
        dynamic? d = new object();
        d.M!(); // 1
        int z = d.y.z;

        d = null;
        d!.M();

        d = null;
        int y = d.y; // 2

        d = null;
        d.M!(); // 3

        d += null;
        d += null!;
    }
}";
            // What warnings should we produce on dynamic?
            // Should `!` be allowed on members/invocations on dynamic?
            // See https://github.com/dotnet/roslyn/issues/32364

            var comp = CreateCompilationWithCSharp(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (7,9): error CS8598: The suppression operator is not allowed in this context
                //         d.M!(); // 1
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "d.M").WithLocation(7, 9),
                // (14,17): warning CS8602: Dereference of a possibly null reference.
                //         int y = d.y; // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "d").WithLocation(14, 17),
                // (17,9): error CS8598: The suppression operator is not allowed in this context
                //         d.M!(); // 3
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "d.M").WithLocation(17, 9),
                // (17,9): warning CS8602: Dereference of a possibly null reference.
                //         d.M!(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "d").WithLocation(17, 9)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_Stackalloc()
        {
            var comp = CreateCompilationWithMscorlibAndSpan(@"
class Test
{
    void M()
    {
        System.Span<int> a3 = stackalloc[] { 1, 2, 3 }!;
        var x3 = true ? stackalloc[] { 1, 2, 3 }! : a3;
    }
}", TestOptions.UnsafeReleaseDll);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_MethodGroup2()
        {
            var source =
@"class C
{
    delegate string? Copier(string? s);
    static void Main()
    {
        Copier c = M2;
        Copier c2 = M2!;
    }
    static string M2(string s) => throw null!;
}";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics(
                // (6,20): warning CS8622: Nullability of reference types in type of parameter 's' of 'string C.M2(string s)' doesn't match the target delegate 'C.Copier'.
                //         Copier c = M2;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "M2").WithArguments("s", "string C.M2(string s)", "C.Copier").WithLocation(6, 20)
                );
            CompileAndVerify(comp);
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_Lambda()
        {
            var source =
@"class C
{
    delegate string Copier(string s);
    static void Main()
    {
        Copier c = (string? x) => { return null; }!;
        Copier c2 = (string? x) => { return null; };
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics(
                // (6,44): warning CS8603: Possible null reference return.
                //         Copier c = (string? x) => { return null; }!;
                Diagnostic(ErrorCode.WRN_NullReferenceReturn, "null").WithLocation(6, 44),
                // (7,21): warning CS8622: Nullability of reference types in type of parameter 'x' of 'lambda expression' doesn't match the target delegate 'C.Copier' (possibly because of nullability attributes).
                //         Copier c2 = (string? x) => { return null; };
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "(string? x) =>").WithArguments("x", "lambda expression", "C.Copier").WithLocation(7, 21),
                // (7,45): warning CS8603: Possible null reference return.
                //         Copier c2 = (string? x) => { return null; };
                Diagnostic(ErrorCode.WRN_NullReferenceReturn, "null").WithLocation(7, 45)
                );
            CompileAndVerify(comp);

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);

            var suppression = tree.GetRoot().DescendantNodes().OfType<PostfixUnaryExpressionSyntax>().Single();
            Assert.Equal("(string? x) => { return null; }!", suppression.ToString());
            VerifyTypeInfo(model, suppression, null, "C.Copier");

            var lambda = suppression.Operand;
            Assert.Equal("(string? x) => { return null; }", lambda.ToString());
            VerifyTypeInfo(model, lambda, null, "C.Copier");
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_Lambda_ExplicitCast()
        {
            var source =
@"class C
{
    delegate string Copier(string s);
    static void M()
    {
        var c = (Copier)((string? x) => { return null; }!);
        var c2 = (Copier)((string? x) => { return null; });
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,50): warning CS8603: Possible null reference return.
                //         var c = (Copier)((string? x) => { return null; }!);
                Diagnostic(ErrorCode.WRN_NullReferenceReturn, "null").WithLocation(6, 50),
                // (7,27): warning CS8622: Nullability of reference types in type of parameter 'x' of 'lambda expression' doesn't match the target delegate 'C.Copier' (possibly because of nullability attributes).
                //         var c2 = (Copier)((string? x) => { return null; });
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "(string? x) =>").WithArguments("x", "lambda expression", "C.Copier").WithLocation(7, 27),
                // (7,51): warning CS8603: Possible null reference return.
                //         var c2 = (Copier)((string? x) => { return null; });
                Diagnostic(ErrorCode.WRN_NullReferenceReturn, "null").WithLocation(7, 51)
                );
            CompileAndVerify(comp);

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);

            var suppression = tree.GetRoot().DescendantNodes().OfType<PostfixUnaryExpressionSyntax>().Single();
            Assert.Equal("(string? x) => { return null; }!", suppression.ToString());
            VerifyTypeInfo(model, suppression, null, "C.Copier");

            var lambda = suppression.Operand;
            Assert.Equal("(string? x) => { return null; }", lambda.ToString());
            VerifyTypeInfo(model, lambda, null, "C.Copier");
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_Lambda2()
        {
            var source =
@"class C
{
    delegate string? Copier(string? s);
    static void Main()
    {
        Copier c = (string x) => { return string.Empty; }!;
        Copier c2 = (string x) => { return string.Empty; };
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics(
                // (7,21): warning CS8622: Nullability of reference types in type of parameter 'x' of 'lambda expression' doesn't match the target delegate 'C.Copier' (possibly because of nullability attributes).
                //         Copier c2 = (string x) => { return string.Empty; };
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "(string x) =>").WithArguments("x", "lambda expression", "C.Copier").WithLocation(7, 21)
                );
            CompileAndVerify(comp);

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);
            var suppression = tree.GetRoot().DescendantNodes().OfType<PostfixUnaryExpressionSyntax>().Single();
            Assert.Equal("(string x) => { return string.Empty; }!", suppression.ToString());

            VerifyTypeInfo(model, suppression, null, "C.Copier");

            var lambda = suppression.Operand;
            Assert.Equal("(string x) => { return string.Empty; }", lambda.ToString());
            VerifyTypeInfo(model, lambda, null, "C.Copier");
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_Lambda2_ExplicitCast()
        {
            var source =
@"class C
{
    delegate string? Copier(string? s);
    static void Main()
    {
        var c = (Copier)((string x) => { return string.Empty; }!);
        var c2 = (Copier)((string x) => { return string.Empty; });
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics(
                // (7,27): warning CS8622: Nullability of reference types in type of parameter 'x' of 'lambda expression' doesn't match the target delegate 'C.Copier' (possibly because of nullability attributes).
                //         var c2 = (Copier)((string x) => { return string.Empty; });
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "(string x) =>").WithArguments("x", "lambda expression", "C.Copier").WithLocation(7, 27)
                );
            CompileAndVerify(comp);

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);

            var suppression = tree.GetRoot().DescendantNodes().OfType<PostfixUnaryExpressionSyntax>().Single();
            Assert.Equal("(string x) => { return string.Empty; }!", suppression.ToString());
            VerifyTypeInfo(model, suppression, null, "C.Copier");

            var lambda = suppression.Operand;
            Assert.Equal("(string x) => { return string.Empty; }", lambda.ToString());
            VerifyTypeInfo(model, lambda, null, "C.Copier");
        }

        [Fact, WorkItem(32697, "https://github.com/dotnet/roslyn/issues/32697")]
        public void SuppressNullableWarning_LambdaInOverloadResolution()
        {
            var source =
@"class C
{
    static void Main(string? x)
    {
        var s = M(() => { return x; });
        s /*T:string?*/ .ToString(); // 1

        var s2 = M(() => { return x; }!); // suppressed
        s2 /*T:string?*/ .ToString(); // 2

        var s3 = M(M2);
        s3 /*T:string?*/ .ToString(); // 3

        var s4 = M(M2!); // suppressed
        s4 /*T:string?*/ .ToString(); // 4
    }
    static T M<T>(System.Func<T> x) => throw null!;
    static string? M2() => throw null!;
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyTypes();

            comp.VerifyDiagnostics(
                // (6,9): warning CS8602: Dereference of a possibly null reference.
                //         s /*T:string?*/ .ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(6, 9),
                // (9,9): warning CS8602: Dereference of a possibly null reference.
                //         s2 /*T:string?*/ .ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s2").WithLocation(9, 9),
                // (12,9): warning CS8602: Dereference of a possibly null reference.
                //         s3 /*T:string?*/ .ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s3").WithLocation(12, 9),
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         s4 /*T:string?*/ .ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s4").WithLocation(15, 9));
            CompileAndVerify(comp);
        }

        [Fact, WorkItem(32698, "https://github.com/dotnet/roslyn/issues/32698")]
        public void SuppressNullableWarning_DelegateCreation()
        {
            var source =
@"class C
{
    static void Main()
    {
        _ = new System.Func<string, string>((string? x) => { return null; }!);
        _ = new System.Func<string?, string?>((string x) => { return string.Empty; }!);
        _ = new System.Func<string, string>(M1!);
        _ = new System.Func<string?, string?>(M2!);

        // without suppression
        _ = new System.Func<string, string>((string? x) => { return null; }); // 1
        _ = new System.Func<string?, string?>((string x) => { return string.Empty; }); // 2
        _ = new System.Func<string, string>(M1); // 3
        _ = new System.Func<string?, string?>(M2); // 4
    }
    static string? M1(string? x) => throw null!;
    static string M2(string x) => throw null!;
}";

            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics(
                // (5,69): warning CS8603: Possible null reference return.
                //         _ = new System.Func<string, string>((string? x) => { return null; }!);
                Diagnostic(ErrorCode.WRN_NullReferenceReturn, "null").WithLocation(5, 69),
                // (11,45): warning CS8622: Nullability of reference types in type of parameter 'x' of 'lambda expression' doesn't match the target delegate 'Func<string, string>' (possibly because of nullability attributes).
                //         _ = new System.Func<string, string>((string? x) => { return null; }); // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "(string? x) =>").WithArguments("x", "lambda expression", "System.Func<string, string>").WithLocation(11, 45),
                // (11,69): warning CS8603: Possible null reference return.
                //         _ = new System.Func<string, string>((string? x) => { return null; }); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReturn, "null").WithLocation(11, 69),
                // (12,47): warning CS8622: Nullability of reference types in type of parameter 'x' of 'lambda expression' doesn't match the target delegate 'Func<string?, string?>' (possibly because of nullability attributes).
                //         _ = new System.Func<string?, string?>((string x) => { return string.Empty; }); // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "(string x) =>").WithArguments("x", "lambda expression", "System.Func<string?, string?>").WithLocation(12, 47),
                // (13,45): warning CS8621: Nullability of reference types in return type of 'string? C.M1(string? x)' doesn't match the target delegate 'Func<string, string>' (possibly because of nullability attributes).
                //         _ = new System.Func<string, string>(M1); // 3
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOfTargetDelegate, "M1").WithArguments("string? C.M1(string? x)", "System.Func<string, string>").WithLocation(13, 45),
                // (14,47): warning CS8622: Nullability of reference types in type of parameter 'x' of 'string C.M2(string x)' doesn't match the target delegate 'Func<string?, string?>' (possibly because of nullability attributes).
                //         _ = new System.Func<string?, string?>(M2); // 4
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "M2").WithArguments("x", "string C.M2(string x)", "System.Func<string?, string?>").WithLocation(14, 47));
            CompileAndVerify(comp);

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree);
            var suppression = tree.GetRoot().DescendantNodes().OfType<PostfixUnaryExpressionSyntax>().First();
            Assert.Equal(@"(string? x) => { return null; }!", suppression.ToString());
            VerifyTypeInfo(model, suppression, null, "System.Func<System.String, System.String>");
            VerifyTypeInfo(model, suppression.Operand, null, "System.Func<System.String, System.String>");
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_MethodGroupInOverloadResolution_NoReceiver()
        {
            var source =
@"using System;
using System.Collections.Generic;
class A
{
    class B
    {
        void F()
        {
            IEnumerable<string?> c = null!;
            c.S(G);
            c.S(G!);

            IEnumerable<string> c2 = null!;
            c2.S(G);
            c2.S(G2);
        }
    }
    object G(string s) => throw null!;
    object G2(string? s) => throw null!;
}
static class E
{
    internal static IEnumerable<U> S<T, U>(this IEnumerable<T> c, Func<T, U> f)
    {
        throw new NotImplementedException();
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (10,17): error CS0120: An object reference is required for the non-static field, method, or property 'A.G(string)'
                //             c.S(G);
                Diagnostic(ErrorCode.ERR_ObjectRequired, "G").WithArguments("A.G(string)").WithLocation(10, 17),
                // (11,17): error CS0120: An object reference is required for the non-static field, method, or property 'A.G(string)'
                //             c.S(G!);
                Diagnostic(ErrorCode.ERR_ObjectRequired, "G").WithArguments("A.G(string)").WithLocation(11, 17),
                // (14,18): error CS0120: An object reference is required for the non-static field, method, or property 'A.G(string)'
                //             c2.S(G);
                Diagnostic(ErrorCode.ERR_ObjectRequired, "G").WithArguments("A.G(string)").WithLocation(14, 18),
                // (15,18): error CS0120: An object reference is required for the non-static field, method, or property 'A.G2(string?)'
                //             c2.S(G2);
                Diagnostic(ErrorCode.ERR_ObjectRequired, "G2").WithArguments("A.G2(string?)").WithLocation(15, 18)
                );
        }

        [Fact]
        [WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        [WorkItem(33635, "https://github.com/dotnet/roslyn/issues/33635")]
        public void SuppressNullableWarning_MethodGroupInOverloadResolution()
        {
            var source =
@"using System;
using System.Collections.Generic;
class A
{
    void M()
    {
        IEnumerable<string?> c = null!;
        c.S(G)/*T:System.Collections.Generic.IEnumerable<object!>!*/; // 1
        c.S(G!)/*T:System.Collections.Generic.IEnumerable<object!>!*/;

        IEnumerable<string> c2 = null!;
        c2.S(G)/*T:System.Collections.Generic.IEnumerable<object!>!*/;
        c2.S(G2)/*T:System.Collections.Generic.IEnumerable<object!>!*/;
    }
    static object G(string s) => throw null!;
    static object G2(string? s) => throw null!;
}
static class E
{
    internal static IEnumerable<U> S<T, U>(this IEnumerable<T> c, Func<T, U> f)
    {
        throw new NotImplementedException();
    }
}";

            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (8,13): warning CS8622: Nullability of reference types in type of parameter 's' of 'object A.G(string s)' doesn't match the target delegate 'Func<string?, object>'.
                //         c.S(G)/*T:IEnumerable<object!>!*/; // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "G").WithArguments("s", "object A.G(string s)", "System.Func<string?, object>").WithLocation(8, 13)
                );
            comp.VerifyTypes();
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void ErrorInLambdaArgumentList()
        {
            var source = @"
class Program
{
    public Program(string x) : this((() => x)!) { }
}";

            CreateCompilation(source).VerifyDiagnostics(
                // (4,38): error CS1660: Cannot convert lambda expression to type 'string' because it is not a delegate type
                //     public Program(string x) : this((() => x)!) { }
                Diagnostic(ErrorCode.ERR_AnonMethToNonDel, "() => x").WithArguments("lambda expression", "string").WithLocation(4, 38)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void CS1113ERR_ValueTypeExtDelegate01()
        {
            var source =
@"class C
{
    public void M() { }
}
interface I
{
    void M();
}
enum E
{
}
struct S
{
    public void M() { }
}
static class SC
{
    static void Test(C c, I i, E e, S s, double d)
    {
        System.Action cm = c.M!;   // OK -- instance method
        System.Action cm1 = c.M1!; // OK -- extension method on ref type
        System.Action im = i.M!;   // OK -- instance method
        System.Action im2 = i.M2!; // OK -- extension method on ref type
        System.Action em3 = e.M3!; // BAD -- extension method on value type
        System.Action sm = s.M!;   // OK -- instance method
        System.Action sm4 = s.M4!; // BAD -- extension method on value type
        System.Action dm5 = d.M5!; // BAD -- extension method on value type
    }

    static void M1(this C c) { }
    static void M2(this I i) { }
    static void M3(this E e) { }
    static void M4(this S s) { }
    static void M5(this double d) { }
}";
            CreateCompilationWithMscorlib40(source, references: new[] { TestMetadata.Net40.SystemCore }).VerifyDiagnostics(
                // (24,29): error CS1113: Extension methods 'SC.M3(E)' defined on value type 'E' cannot be used to create delegates
                Diagnostic(ErrorCode.ERR_ValueTypeExtDelegate, "e.M3").WithArguments("SC.M3(E)", "E").WithLocation(24, 29),
                // (26,29): error CS1113: Extension methods 'SC.M4(S)' defined on value type 'S' cannot be used to create delegates
                Diagnostic(ErrorCode.ERR_ValueTypeExtDelegate, "s.M4").WithArguments("SC.M4(S)", "S").WithLocation(26, 29),
                // (27,29): error CS1113: Extension methods 'SC.M5(double)' defined on value type 'double' cannot be used to create delegates
                Diagnostic(ErrorCode.ERR_ValueTypeExtDelegate, "d.M5").WithArguments("SC.M5(double)", "double").WithLocation(27, 29));
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void BugCodePlex_30_01()
        {
            string source1 = @"
using System;
class C
{
    static void Main()
    {
        Goo(() => { return () => 0; ; }!);
        Goo(() => { return () => 0; }!);
    }
    static void Goo(Func<Func<short>> x) { Console.Write(1); }
    static void Goo(Func<Func<int>> x) { Console.Write(2); }
}
";
            CompileAndVerify(source1, expectedOutput: @"22");
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_AddressOfWithLambdaOrMethodGroup()
        {
            var source = @"
unsafe class C
{
    static void M()
    {
        _ = &(() => {}!);
        _ = &(M!);
    }
}";
            var comp = CreateCompilation(source, options: TestOptions.UnsafeReleaseDll);
            comp.VerifyDiagnostics(
                // (6,15): error CS0211: Cannot take the address of the given expression
                //         _ = &(() => {}!);
                Diagnostic(ErrorCode.ERR_InvalidAddrOp, "() => {}").WithLocation(6, 15),
                // (7,9): error CS8183: Cannot infer the type of implicitly-typed discard.
                //         _ = &(M!);
                Diagnostic(ErrorCode.ERR_DiscardTypeInferenceFailed, "_").WithLocation(7, 9),
                // (7,15): error CS8598: The suppression operator is not allowed in this context
                //         _ = &(M!);
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "M").WithLocation(7, 15)
                );
        }

        [Fact, WorkItem(31297, "https://github.com/dotnet/roslyn/issues/31297")]
        public void SuppressNullableWarning_AddressOf()
        {
            var source = @"
unsafe class C<T>
{
    static void M(C<string> x)
    {
        C<string?>* y1 = &x;
        C<string?>* y2 = &x!;
    }
}";
            var comp = CreateCompilation(source, options: WithNullable(TestOptions.UnsafeReleaseDll, NullableContextOptions.Enable));
            comp.VerifyDiagnostics(
                // (6,9): warning CS8500: This takes the address of, gets the size of, or declares a pointer to a managed type ('C<string?>')
                //         C<string?>* y1 = &x;
                Diagnostic(ErrorCode.WRN_ManagedAddr, "C<string?>*").WithArguments("C<string?>").WithLocation(6, 9),
                // (6,26): warning CS8500: This takes the address of, gets the size of, or declares a pointer to a managed type ('C<string>')
                //         C<string?>* y1 = &x;
                Diagnostic(ErrorCode.WRN_ManagedAddr, "&x").WithArguments("C<string>").WithLocation(6, 26),
                // (6,26): warning CS8619: Nullability of reference types in value of type 'C<string>*' doesn't match target type 'C<string?>*'.
                //         C<string?>* y1 = &x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "&x").WithArguments("C<string>*", "C<string?>*").WithLocation(6, 26),
                // (7,9): warning CS8500: This takes the address of, gets the size of, or declares a pointer to a managed type ('C<string?>')
                //         C<string?>* y2 = &x!;
                Diagnostic(ErrorCode.WRN_ManagedAddr, "C<string?>*").WithArguments("C<string?>").WithLocation(7, 9),
                // (7,26): warning CS8500: This takes the address of, gets the size of, or declares a pointer to a managed type ('C<string>')
                //         C<string?>* y2 = &x!;
                Diagnostic(ErrorCode.WRN_ManagedAddr, "&x!").WithArguments("C<string>").WithLocation(7, 26),
                // (7,26): warning CS8619: Nullability of reference types in value of type 'C<string>*' doesn't match target type 'C<string?>*'.
                //         C<string?>* y2 = &x!;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "&x!").WithArguments("C<string>*", "C<string?>*").WithLocation(7, 26),
                // (7,27): error CS8598: The suppression operator is not allowed in this context
                //         C<string?>* y2 = &x!;
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "x").WithLocation(7, 27)
                );
        }

        [Fact]
        public void SuppressNullableWarning_Invocation()
        {
            var source = @"
class C
{
    void M()
    {
        _ = nameof!(M);
        _ = typeof!(C);
        this.M!();
        dynamic d = null!;
        d.M2!();
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,13): error CS0103: The name 'nameof' does not exist in the current context
                //         _ = nameof!(M);
                Diagnostic(ErrorCode.ERR_NameNotInContext, "nameof").WithArguments("nameof").WithLocation(6, 13),
                // (7,19): error CS1003: Syntax error, '(' expected
                //         _ = typeof!(C);
                Diagnostic(ErrorCode.ERR_SyntaxError, "!").WithArguments("(").WithLocation(7, 19),
                // (7,19): error CS1031: Type expected
                //         _ = typeof!(C);
                Diagnostic(ErrorCode.ERR_TypeExpected, "!").WithLocation(7, 19),
                // (7,19): error CS1026: ) expected
                //         _ = typeof!(C);
                Diagnostic(ErrorCode.ERR_CloseParenExpected, "!").WithLocation(7, 19),
                // (7,21): error CS0119: 'C' is a type, which is not valid in the given context
                //         _ = typeof!(C);
                Diagnostic(ErrorCode.ERR_BadSKunknown, "C").WithArguments("C", "type").WithLocation(7, 21),
                // (8,9): error CS8598: The suppression operator is not allowed in this context
                //         this.M!();
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "this.M").WithLocation(8, 9),
                // (10,9): error CS8598: The suppression operator is not allowed in this context
                //         d.M2!();
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "d.M2").WithLocation(10, 9)
                );
        }

        [Fact, WorkItem(31584, "https://github.com/dotnet/roslyn/issues/31584")]
        public void Verify31584()
        {
            var comp = CreateCompilation(@"
using System;
using System.Linq;
class C
{
    void M<T>(Func<T, bool>? predicate)
    {
        var items = Enumerable.Empty<T>();
        if (predicate != null)
            items = items.Where(x => predicate(x));
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(32463, "https://github.com/dotnet/roslyn/issues/32463")]
        public void Verify32463()
        {
            var comp = CreateCompilation(@"
using System.Linq;
class C
{
    public void F(string? param)
    {
        if (param != null)
            _ = new[] { 0 }.Select(_ => param.Length);

        string? local = """";
        if (local != null)
            _ = new[] { 0 }.Select(_ => local.Length);
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(32701, "https://github.com/dotnet/roslyn/issues/32701")]
        public void Verify32701()
        {
            var source = @"
class C
{
    static void M<T>(ref T t, dynamic? d)
    {
        t = d; // 1
    }
    static void M2<T>(T t, dynamic? d)
    {
        t = d; // 2
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);
            comp.VerifyDiagnostics(
                // (6,13): warning CS8601: Possible null reference assignment.
                //         t = d; // 1
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "d").WithLocation(6, 13));

            comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,13): warning CS8601: Possible null reference assignment.
                //         t = d; // 1
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "d").WithLocation(6, 13),
                // (10,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         t = d; // 2
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "d").WithLocation(10, 13));

            source = @"
class C
{
    static void M<T>(ref T? t, dynamic? d)
    {
        t = d;
    }
    static void M2<T>(T? t, dynamic? d)
    {
        t = d;
    }
}";
            comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(26696, "https://github.com/dotnet/roslyn/issues/26696")]
        public void Verify26696()
        {
            var source = @"
interface I
{
    object this[int i] { get; }
}
class C<T> : I
{
    T this[int i] => throw null!;
    object I.this[int i] => this[i]!;
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(31297, "https://github.com/dotnet/roslyn/issues/31297")]
        public void RefReturn()
        {
            var source = @"
struct S<T>
{
    ref S<string?> M(ref S<string> x)
    {
        return ref x; // 1
    }
    ref S<string?> M2(ref S<string> x)
    {
        return ref x!;
    }
}
class C<T>
{
    ref C<string>? M3(ref C<string> x)
    {
        return ref x; // 2
    }
    ref C<string> M4(ref C<string>? x)
    {
        return ref x; // 3
    }
    ref C<string>? M5(ref C<string> x)
    {
        return ref x!;
    }
    ref C<string> M6(ref C<string>? x)
    {
        return ref x!;
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,20): warning CS8619: Nullability of reference types in value of type 'S<string>' doesn't match target type 'S<string?>'.
                //         return ref x; // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("S<string>", "S<string?>").WithLocation(6, 20),
                // (17,20): warning CS8619: Nullability of reference types in value of type 'C<string>' doesn't match target type 'C<string>?'.
                //         return ref x; // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("C<string>", "C<string>?").WithLocation(17, 20),
                // (21,20): warning CS8619: Nullability of reference types in value of type 'C<string>?' doesn't match target type 'C<string>'.
                //         return ref x; // 3
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("C<string>?", "C<string>").WithLocation(21, 20)
                );
        }

        [Fact, WorkItem(33982, "https://github.com/dotnet/roslyn/issues/33982")]
        public void RefReturn_Lambda()
        {
            var comp = CreateCompilation(@"
delegate ref V D<T, U, V>(ref T t, ref U u);
interface I<T> { }
interface IIn<in T> { }
interface IOut<out T> { }
class C
{
    static V F<T, U, V>(D<T, U, V> d) => throw null!;
    void M(bool b)
    {
        F((ref object? x, ref object y) =>
            { if (b) return ref x; return ref y; }); // 1
        F((ref object? x, ref object y) =>
            { if (b) return ref y; return ref x; }); // 2

        F((ref I<object?> x, ref I<object> y) =>
            { if (b) return ref x; return ref y; }); // 3
        F((ref I<object?> x, ref I<object> y) =>
            { if (b) return ref y; return ref x; }); // 4

        F((ref IOut<object?> x, ref IOut<object> y) =>
            { if (b) return ref x; return ref y; }); // 5
        F((ref IOut<object?> x, ref IOut<object> y) =>
            { if (b) return ref y; return ref x; }); // 6

        F((ref IIn<object?> x, ref IIn<object> y) =>
            { if (b) return ref x; return ref y; }); // 7
        F((ref IIn<object?> x, ref IIn<object> y) =>
            { if (b) return ref y; return ref x; }); // 8
    }
}", options: WithNullableEnable());

            comp.VerifyDiagnostics(
                // (12,47): warning CS8619: Nullability of reference types in value of type 'object' doesn't match target type 'object?'.
                //             { if (b) return ref x; return ref y; }); // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "y").WithArguments("object", "object?").WithLocation(12, 47),
                // (14,33): warning CS8619: Nullability of reference types in value of type 'object' doesn't match target type 'object?'.
                //             { if (b) return ref y; return ref x; }); // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "y").WithArguments("object", "object?").WithLocation(14, 33),
                // (17,33): warning CS8619: Nullability of reference types in value of type 'I<object?>' doesn't match target type 'I<object>'.
                //             { if (b) return ref x; return ref y; }); // 3
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("I<object?>", "I<object>").WithLocation(17, 33),
                // (19,47): warning CS8619: Nullability of reference types in value of type 'I<object?>' doesn't match target type 'I<object>'.
                //             { if (b) return ref y; return ref x; }); // 4
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("I<object?>", "I<object>").WithLocation(19, 47),
                // (22,47): warning CS8619: Nullability of reference types in value of type 'IOut<object>' doesn't match target type 'IOut<object?>'.
                //             { if (b) return ref x; return ref y; }); // 5
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "y").WithArguments("IOut<object>", "IOut<object?>").WithLocation(22, 47),
                // (24,33): warning CS8619: Nullability of reference types in value of type 'IOut<object>' doesn't match target type 'IOut<object?>'.
                //             { if (b) return ref y; return ref x; }); // 6
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "y").WithArguments("IOut<object>", "IOut<object?>").WithLocation(24, 33),
                // (27,33): warning CS8619: Nullability of reference types in value of type 'IIn<object?>' doesn't match target type 'IIn<object>'.
                //             { if (b) return ref x; return ref y; }); // 7
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("IIn<object?>", "IIn<object>").WithLocation(27, 33),
                // (29,47): warning CS8619: Nullability of reference types in value of type 'IIn<object?>' doesn't match target type 'IIn<object>'.
                //             { if (b) return ref y; return ref x; }); // 8
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("IIn<object?>", "IIn<object>").WithLocation(29, 47)
                );
        }

        [Fact, WorkItem(31297, "https://github.com/dotnet/roslyn/issues/31297")]
        public void RefReturn_State()
        {
            var source = @"
class C
{
    ref C M1(ref C? w)
    {
        return ref w; // 1
    }
    ref C? M2(ref C w)
    {
        return ref w; // 2
    }
    ref C M3(ref C w)
    {
        return ref w;
    }
    ref C? M4(ref C? w)
    {
        return ref w;
    }
}";

            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,20): warning CS8619: Nullability of reference types in value of type 'C?' doesn't match target type 'C'.
                //         return ref w; // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "w").WithArguments("C?", "C").WithLocation(6, 20),
                // (10,20): warning CS8619: Nullability of reference types in value of type 'C' doesn't match target type 'C?'.
                //         return ref w; // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "w").WithArguments("C", "C?").WithLocation(10, 20)
                );
        }

        [Fact, WorkItem(31297, "https://github.com/dotnet/roslyn/issues/31297")]
        public void RefAssignment()
        {
            var source = @"
class C
{
    void M1(C? x)
    {
        ref C y = ref x; // 1
        y.ToString(); // 2
    }
    void M2(C x)
    {
        ref C? y = ref x; // 3
        y.ToString();
    }
    void M3(C? x, C nonNull)
    {
        x = nonNull;
        ref C y = ref x; // 4
        y.ToString();
        y = null; // 5
    }
    void M4(C x, C? nullable)
    {
        x = nullable; // 6
        ref C? y = ref x; // 7
        y.ToString(); // 8
    }
}";

            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,23): warning CS8619: Nullability of reference types in value of type 'C?' doesn't match target type 'C'.
                //         ref C y = ref x; // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("C?", "C").WithLocation(6, 23),
                // (7,9): warning CS8602: Dereference of a possibly null reference.
                //         y.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "y").WithLocation(7, 9),
                // (11,24): warning CS8619: Nullability of reference types in value of type 'C' doesn't match target type 'C?'.
                //         ref C? y = ref x; // 3
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("C", "C?").WithLocation(11, 24),
                // (17,23): warning CS8619: Nullability of reference types in value of type 'C?' doesn't match target type 'C'.
                //         ref C y = ref x; // 4
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("C?", "C").WithLocation(17, 23),
                // (19,13): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         y = null; // 5
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(19, 13),
                // (23,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         x = nullable; // 6
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "nullable").WithLocation(23, 13),
                // (24,24): warning CS8619: Nullability of reference types in value of type 'C' doesn't match target type 'C?'.
                //         ref C? y = ref x; // 7
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("C", "C?").WithLocation(24, 24),
                // (25,9): warning CS8602: Dereference of a possibly null reference.
                //         y.ToString(); // 8
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "y").WithLocation(25, 9)
                );
        }

        [Fact, WorkItem(31297, "https://github.com/dotnet/roslyn/issues/31297")]
        public void RefAssignment_WithSuppression()
        {
            var source = @"
class C
{
    void M1(C? x)
    {
        ref C y = ref x!;
        y.ToString();
    }
    void M2(C x)
    {
        ref C? y = ref x!;
        y.ToString();
    }
    void M3(C? x, C nonNull)
    {
        x = nonNull;
        ref C y = ref x!;
        y.ToString();
    }
    void M4(C x, C? nullable)
    {
        x = nullable; // 1
        ref C? y = ref x!;
        y.ToString();
    }
}";

            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (22,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         x = nullable; // 1
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "nullable").WithLocation(22, 13)
                );
        }

        [Fact, WorkItem(31297, "https://github.com/dotnet/roslyn/issues/31297")]
        public void RefAssignment_Nested()
        {
            var source = @"
struct S<T>
{
    void M(ref S<string> x)
    {
        S<string?> y = default;
        ref S<string> y2 = ref y; // 1
        y2 = ref y; // 2
        y2 = ref y!;
    }
}";

            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (7,32): warning CS8619: Nullability of reference types in value of type 'S<string?>' doesn't match target type 'S<string>'.
                //         ref S<string> y2 = ref y; // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "y").WithArguments("S<string?>", "S<string>").WithLocation(7, 32),
                // (8,18): warning CS8619: Nullability of reference types in value of type 'S<string?>' doesn't match target type 'S<string>'.
                //         y2 = ref y; // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "y").WithArguments("S<string?>", "S<string>").WithLocation(8, 18)
                );
        }

        [Fact, WorkItem(31297, "https://github.com/dotnet/roslyn/issues/31297")]
        public void RefAssignment_Foreach()
        {
            verify(variableType: "string?", itemType: "string",
                // (6,18): warning CS8619: Nullability of reference types in value of type 'string' doesn't match target type 'string?'.
                //         foreach (ref string? item in collection)
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "ref string?").WithArguments("string", "string?").WithLocation(6, 18));

            verify("string", "string?",
                // (6,18): warning CS8619: Nullability of reference types in value of type 'string?' doesn't match target type 'string'.
                //         foreach (ref string item in collection)
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "ref string").WithArguments("string?", "string").WithLocation(6, 18),
                // (8,13): warning CS8602: Dereference of a possibly null reference.
                //             item.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "item").WithLocation(8, 13));

            verify("string", "string");
            verify("string?", "string?",
                // (8,13): warning CS8602: Dereference of a possibly null reference.
                //             item.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "item").WithLocation(8, 13));

            verify("C<string?>", "C<string>",
                // (6,18): warning CS8619: Nullability of reference types in value of type 'C<string>' doesn't match target type 'C<string?>'.
                //         foreach (ref C<string?> item in collection)
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "ref C<string?>").WithArguments("C<string>", "C<string?>").WithLocation(6, 18));

            verify("C<string>", "C<string?>",
                // (6,18): warning CS8619: Nullability of reference types in value of type 'C<string?>' doesn't match target type 'C<string>'.
                //         foreach (ref C<string> item in collection)
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "ref C<string>").WithArguments("C<string?>", "C<string>").WithLocation(6, 18));

            verify("C<object?>", "C<dynamic>?",
                // (6,18): warning CS8619: Nullability of reference types in value of type 'C<dynamic>?' doesn't match target type 'C<object?>'.
                //         foreach (ref C<object?> item in collection)
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "ref C<object?>").WithArguments("C<dynamic>?", "C<object?>").WithLocation(6, 18),
                // (8,13): warning CS8602: Dereference of a possibly null reference.
                //             item.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "item").WithLocation(8, 13));

            verify("C<object>", "C<dynamic>");
            verify("var", "string");

            verify("var", "string?",
                // (8,13): warning CS8602: Dereference of a possibly null reference.
                //             item.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "item").WithLocation(8, 13));

            verify("T", "T",
                // (8,13): warning CS8602: Dereference of a possibly null reference.
                //             item.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "item").WithLocation(8, 13));

            void verify(string variableType, string itemType, params DiagnosticDescription[] expected)
            {
                var source = $$"""

class C<T>
{
    void M(RefEnumerable collection)
    {
        foreach (ref {{variableType}} item in collection)
        {
            item.ToString();
        }
    }

    class RefEnumerable
    {
        public StructEnum GetEnumerator() => throw null!;
        public struct StructEnum
        {
            public ref {{itemType}} Current => throw null!;
            public bool MoveNext() => throw null!;
        }
    }
}
""";

                var comp = CreateCompilation(source, options: WithNullableEnable());
                comp.VerifyDiagnostics(expected);
            }
        }

        [Fact, WorkItem(31297, "https://github.com/dotnet/roslyn/issues/31297")]
        public void RefAssignment_Foreach_Nested()
        {
            verify(fieldType: "string?",
                // (9,13): warning CS8602: Dereference of a possibly null reference.
                //             item.Field.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "item.Field").WithLocation(9, 13));

            verify(fieldType: "string",
                // (4,19): warning CS8618: Non-nullable field 'Field' is uninitialized. Consider declaring the field as nullable.
                //     public string Field;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Field").WithArguments("field", "Field").WithLocation(4, 19));

            void verify(string fieldType, params DiagnosticDescription[] expected)
            {
                var source = @"
class C
{
    public FIELDTYPE Field;
    void M(RefEnumerable collection)
    {
        foreach (ref C item in collection)
        {
            item.Field.ToString();
        }
    }

    class RefEnumerable
    {
        public StructEnum GetEnumerator() => throw null!;
        public struct StructEnum
        {
            public ref C Current => throw null!;
            public bool MoveNext() => throw null!;
        }
    }
}";
                var comp = CreateCompilation(source.Replace("FIELDTYPE", fieldType), options: WithNullableEnable());
                comp.VerifyDiagnostics(expected);
            }
        }
        [Fact]
        public void RefAssignment_Inferred()
        {
            var source = @"
class C
{
    void M1(C x)
    {
        ref var y = ref x;
        y = null;
        x.ToString();
        y.ToString(); // 1
    }

    void M2(C? x)
    {
        ref var y = ref x;
        y = null;
        x.ToString(); // 2
        y.ToString(); // 3
    }

    void M3(C? x)
    {
        ref var y = ref x;
        x.ToString(); // 4
        y.ToString(); // 5
    }
}";

            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (9,9): warning CS8602: Dereference of a possibly null reference.
                //         y.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "y").WithLocation(9, 9),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         x.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x").WithLocation(16, 9),
                // (17,9): warning CS8602: Dereference of a possibly null reference.
                //         y.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "y").WithLocation(17, 9),
                // (23,9): warning CS8602: Dereference of a possibly null reference.
                //         x.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x").WithLocation(23, 9),
                // (24,9): warning CS8602: Dereference of a possibly null reference.
                //         y.ToString(); // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "y").WithLocation(24, 9)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void TestLambdaWithError19()
        {
            var source =
@"using System;
using System.Linq.Expressions;

class Program
{
    static void Main(string[] args)
    {
        Ma(string.Empty, ((x, y) => x.ToString())!);
        Mb(string.Empty, ((x, y) => x.ToString())!);
        Mc(string.Empty, ((x, y) => x.ToString())!);
    }
    static void Ma<T>(T t, Expression<Action<T, T, int>> action) { }
    static void Mb<T>(T t, Expression<Action<T, T, int>> action) { }
    static void Mb<T>(T t, Action<T, T, int> action) { }
    static void Mc<T>(T t, Expression<Action<T, T, int>> action) { }
    static void Mc() { }
}
";
            var compilation = CreateCompilationWithMscorlib40AndSystemCore(source);
            var tree = compilation.SyntaxTrees[0];
            var sm = compilation.GetSemanticModel(tree);
            foreach (var lambda in tree.GetRoot().DescendantNodes().OfType<LambdaExpressionSyntax>())
            {
                var reference = lambda.Body.DescendantNodesAndSelf().OfType<IdentifierNameSyntax>().First();
                Assert.Equal("x", reference.ToString());
                var typeInfo = sm.GetTypeInfoAndVerifyIOperation(reference);
                Assert.Equal(TypeKind.Class, typeInfo.Type.TypeKind);
                Assert.Equal("String", typeInfo.Type.Name);
                Assert.NotEmpty(typeInfo.Type.GetMembers("Replace"));
            }
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_DelegateComparison()
        {
            var source = @"
class C
{
    static void M()
    {
        System.Func<int> x = null;
        _ = x == (() => 1)!;
    }
}";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (7,13): error CS0019: Operator '==' cannot be applied to operands of type 'Func<int>' and 'lambda expression'
                //         _ = x == (() => 1)!;
                Diagnostic(ErrorCode.ERR_BadBinaryOps, "x == (() => 1)!").WithArguments("==", "System.Func<int>", "lambda expression").WithLocation(7, 13)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_ArgList()
        {
            var source = @"
class C
{
    static void M()
    {
        _ = __arglist!;
        M(__arglist(__arglist()!));
    }
}";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (6,13): error CS0190: The __arglist construct is valid only within a variable argument method
                //         _ = __arglist!;
                Diagnostic(ErrorCode.ERR_ArgsInvalid, "__arglist").WithLocation(6, 13),
                // (7,21): error CS0226: An __arglist expression may only appear inside of a call or new expression
                //         M(__arglist(__arglist()!));
                Diagnostic(ErrorCode.ERR_IllegalArglist, "__arglist()").WithLocation(7, 21)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void BadInvocationInLambda()
        {
            var src = @"
using System;
using System.Linq.Expressions;

class C
{
    Expression<Action<dynamic>> e = x => new object[](x);
    Expression<Action<dynamic>> e2 = x => new object[](x)!;
}";
            // Suppressed expression cannot be used as a statement
            var comp = CreateCompilationWithMscorlib40AndSystemCore(src);
            comp.VerifyDiagnostics(
                // (7,52): error CS1586: Array creation must have array size or array initializer
                //     Expression<Action<dynamic>> e = x => new object[](x);
                Diagnostic(ErrorCode.ERR_MissingArraySize, "[]").WithLocation(7, 52),
                // (8,43): error CS0201: Only assignment, call, increment, decrement, await, and new object expressions can be used as a statement
                //     Expression<Action<dynamic>> e2 = x => new object[](x)!;
                Diagnostic(ErrorCode.ERR_IllegalStatement, "new object[](x)!").WithLocation(8, 43),
                // (8,53): error CS1586: Array creation must have array size or array initializer
                //     Expression<Action<dynamic>> e2 = x => new object[](x)!;
                Diagnostic(ErrorCode.ERR_MissingArraySize, "[]").WithLocation(8, 53)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_AsStatement()
        {
            var src = @"
class C
{
    void M2(string x)
    {
        x!;
        M2(x)!;
    }
    string M(string x)
    {
        x!;
        M(x)!;
        throw null!;
    }
}";
            // Suppressed expression cannot be used as a statement
            var comp = CreateCompilation(src);
            comp.VerifyDiagnostics(
                // (6,9): error CS8598: The suppression operator is not allowed in this context
                //         x!;
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "x").WithLocation(6, 9),
                // (6,9): error CS0201: Only assignment, call, increment, decrement, await, and new object expressions can be used as a statement
                //         x!;
                Diagnostic(ErrorCode.ERR_IllegalStatement, "x!").WithLocation(6, 9),
                // (7,9): error CS8598: The suppression operator is not allowed in this context
                //         M2(x)!;
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "M2(x)").WithLocation(7, 9),
                // (7,9): error CS0201: Only assignment, call, increment, decrement, await, and new object expressions can be used as a statement
                //         M2(x)!;
                Diagnostic(ErrorCode.ERR_IllegalStatement, "M2(x)!").WithLocation(7, 9),
                // (11,9): error CS8598: The suppression operator is not allowed in this context
                //         x!;
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "x").WithLocation(11, 9),
                // (11,9): error CS0201: Only assignment, call, increment, decrement, await, and new object expressions can be used as a statement
                //         x!;
                Diagnostic(ErrorCode.ERR_IllegalStatement, "x!").WithLocation(11, 9),
                // (12,9): error CS8598: The suppression operator is not allowed in this context
                //         M(x)!;
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "M(x)").WithLocation(12, 9),
                // (12,9): error CS0201: Only assignment, call, increment, decrement, await, and new object expressions can be used as a statement
                //         M(x)!;
                Diagnostic(ErrorCode.ERR_IllegalStatement, "M(x)!").WithLocation(12, 9)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_VoidInvocation()
        {
            var src = @"
class C
{
    void M()
    {
        _ = M()!;
    }
}";
            var comp = CreateCompilation(src);
            comp.VerifyDiagnostics(
                // (6,9): error CS8209: A value of type 'void' may not be assigned.
                //         _ = M()!;
                Diagnostic(ErrorCode.ERR_VoidAssignment, "_").WithLocation(6, 9)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void CS0023ERR_BadUnaryOp_lambdaExpression()
        {
            var text = @"
class X
{
    static void Main()
    {
        System.Func<int, int> f = (arg => { arg = 2; return arg; } !).ToString();
        
        var x = (delegate { } !).ToString();
    }
}
";
            CreateCompilation(text).VerifyDiagnostics(
                // (6,35): error CS0023: Operator '.' cannot be applied to operand of type 'lambda expression'
                //         System.Func<int, int> f = (arg => { arg = 2; return arg; } !).ToString();
                Diagnostic(ErrorCode.ERR_BadUnaryOp, "(arg => { arg = 2; return arg; } !).ToString").WithArguments(".", "lambda expression").WithLocation(6, 35),
                // (8,17): error CS0023: Operator '.' cannot be applied to operand of type 'anonymous method'
                //         var x = (delegate { } !).ToString();
                Diagnostic(ErrorCode.ERR_BadUnaryOp, "(delegate { } !).ToString").WithArguments(".", "anonymous method").WithLocation(8, 17)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void ConditionalMemberAccess001()
        {
            var text = @"
class Program
{
    static void Main(string[] args)
    {
        var x4 = (()=> { return 1; } !) ?.ToString();
    }
}
";

            CreateCompilationWithMscorlib45(text).VerifyDiagnostics(
                // (6,18): error CS0023: Operator '?' cannot be applied to operand of type 'lambda expression'
                //         var x4 = (()=> { return 1; } !) ?.ToString();
                Diagnostic(ErrorCode.ERR_BadUnaryOp, "(()=> { return 1; } !) ?.ToString()").WithArguments("?", "lambda expression").WithLocation(6, 18),
                // (6,18): error CS0023: Operator '?' cannot be applied to operand of type 'lambda expression'
                //         var x4 = (()=> { return 1; } !) ?.ToString();
                Diagnostic(ErrorCode.ERR_BadUnaryOp, "(()=> { return 1; } !) ?.ToString()").WithArguments("?", "lambda expression").WithLocation(6, 18)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void DynamicCollectionInitializer_Errors()
        {
            string source = @"
using System;

unsafe class C
{
    public dynamic X;
    public static int* ptr = null;

    static void M()
    {
        var c = new C
        {
            X =
            {
                M!,
                ptr!,
                () => {}!,
                default(TypedReference)!,
                M()!,
                __arglist
            }
        };
    }
}
";
            // Should `!` be disallowed on arguments to dynamic?
            // See https://github.com/dotnet/roslyn/issues/32364

            CreateCompilationWithMscorlib40AndSystemCore(source, options: TestOptions.UnsafeReleaseDll).VerifyDiagnostics(
                // (15,17): error CS1976: Cannot use a method group as an argument to a dynamically dispatched operation. Did you intend to invoke the method?
                //                 M!,
                Diagnostic(ErrorCode.ERR_BadDynamicMethodArgMemgrp, "M").WithLocation(15, 17),
                // (16,17): error CS1978: Cannot use an expression of type 'int*' as an argument to a dynamically dispatched operation.
                //                 ptr!,
                Diagnostic(ErrorCode.ERR_BadDynamicMethodArg, "ptr").WithArguments("int*").WithLocation(16, 17),
                // (17,17): error CS1977: Cannot use a lambda expression as an argument to a dynamically dispatched operation without first casting it to a delegate or expression tree type.
                //                 () => {}!,
                Diagnostic(ErrorCode.ERR_BadDynamicMethodArgLambda, "() => {}").WithLocation(17, 17),
                // (18,17): error CS1978: Cannot use an expression of type 'TypedReference' as an argument to a dynamically dispatched operation.
                //                 default(TypedReference)!,
                Diagnostic(ErrorCode.ERR_BadDynamicMethodArg, "default(TypedReference)").WithArguments("System.TypedReference").WithLocation(18, 17),
                // (19,17): error CS1978: Cannot use an expression of type 'void' as an argument to a dynamically dispatched operation.
                //                 M()!,
                Diagnostic(ErrorCode.ERR_BadDynamicMethodArg, "M()").WithArguments("void").WithLocation(19, 17),
                // (20,17): error CS0190: The __arglist construct is valid only within a variable argument method
                //                 __arglist
                Diagnostic(ErrorCode.ERR_ArgsInvalid, "__arglist").WithLocation(20, 17),
                // (20,17): error CS1978: Cannot use an expression of type 'RuntimeArgumentHandle' as an argument to a dynamically dispatched operation.
                //                 __arglist
                Diagnostic(ErrorCode.ERR_BadDynamicMethodArg, "__arglist").WithArguments("System.RuntimeArgumentHandle").WithLocation(20, 17)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void TestNullCoalesceWithMethodGroup()
        {
            var source = @"
using System;

class Program
{
    static void Main()
    {
        Action a = Main! ?? Main;
        Action a2 = Main ?? Main!;
    }
}
";
            CreateCompilation(source).VerifyDiagnostics(
                // (8,20): error CS0019: Operator '??' cannot be applied to operands of type 'method group' and 'method group'
                //         Action a = Main! ?? Main;
                Diagnostic(ErrorCode.ERR_BadBinaryOps, "Main! ?? Main").WithArguments("??", "method group", "method group").WithLocation(8, 20),
                // (9,21): error CS0019: Operator '??' cannot be applied to operands of type 'method group' and 'method group'
                //         Action a2 = Main ?? Main!;
                Diagnostic(ErrorCode.ERR_BadBinaryOps, "Main ?? Main!").WithArguments("??", "method group", "method group").WithLocation(9, 21)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_IsAsOperatorWithBadSuppressedExpression()
        {
            var source = @"
class C
{
    void M()
    {
        _ = (() => {}!) is null; // 1
        _ = (M!) is null; // 2
        _ = (null, null)! is object; // 3
        _ = null! is object; // 4
        _ = default! is object; // 5

        _ = (() => {}!) as object; // 6
        _ = (M!) as object; // 7
        _ = (null, null)! as object; // 8
        _ = null! as object; // ok
        _ = default! as string; // 10
    }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (6,13): error CS0837: The first operand of an 'is' or 'as' operator may not be a lambda expression, anonymous method, or method group.
                //         _ = (() => {}!) is null; // 1
                Diagnostic(ErrorCode.ERR_LambdaInIsAs, "(() => {}!) is null").WithLocation(6, 13),
                // (7,13): error CS0837: The first operand of an 'is' or 'as' operator may not be a lambda expression, anonymous method, or method group.
                //         _ = (M!) is null; // 2
                Diagnostic(ErrorCode.ERR_LambdaInIsAs, "(M!) is null").WithLocation(7, 13),
                // (8,13): error CS0023: Operator 'is' cannot be applied to operand of type '(<null>, <null>)'
                //         _ = (null, null)! is object; // 3
                Diagnostic(ErrorCode.ERR_BadUnaryOp, "(null, null)! is object").WithArguments("is", "(<null>, <null>)").WithLocation(8, 13),
                // (9,13): warning CS0184: The given expression is never of the provided ('object') type
                //         _ = null! is object; // 4
                Diagnostic(ErrorCode.WRN_IsAlwaysFalse, "null! is object").WithArguments("object").WithLocation(9, 13),
                // (10,13): error CS8716: There is no target type for the default literal.
                //         _ = default! is object; // 5
                Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(10, 13),
                // (12,13): error CS0837: The first operand of an 'is' or 'as' operator may not be a lambda expression, anonymous method, or method group.
                //         _ = (() => {}!) as object; // 6
                Diagnostic(ErrorCode.ERR_LambdaInIsAs, "(() => {}!) as object").WithLocation(12, 13),
                // (13,13): error CS0837: The first operand of an 'is' or 'as' operator may not be a lambda expression, anonymous method, or method group.
                //         _ = (M!) as object; // 7
                Diagnostic(ErrorCode.ERR_LambdaInIsAs, "(M!) as object").WithLocation(13, 13),
                // (14,13): error CS8307: The first operand of an 'as' operator may not be a tuple literal without a natural type.
                //         _ = (null, null)! as object; // 8
                Diagnostic(ErrorCode.ERR_TypelessTupleInAs, "(null, null)! as object").WithLocation(14, 13),
                // (16,13): error CS8716: There is no target type for the default literal.
                //         _ = default! as string; // 10
                Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(16, 13)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void ImplicitDelegateCreationWithIncompleteLambda()
        {
            var source =
@"
using System;
class C
{
    public void F()
    {
        Action<int> x = (i => i.)!
    }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (7,33): error CS1001: Identifier expected
                //         Action<int> x = (i => i.)!
                Diagnostic(ErrorCode.ERR_IdentifierExpected, ")").WithLocation(7, 33),
                // (7,35): error CS1002: ; expected
                //         Action<int> x = (i => i.)!
                Diagnostic(ErrorCode.ERR_SemicolonExpected, "").WithLocation(7, 35),
                // (7,31): error CS0201: Only assignment, call, increment, decrement, await, and new object expressions can be used as a statement
                //         Action<int> x = (i => i.)!
                Diagnostic(ErrorCode.ERR_IllegalStatement, "i.").WithLocation(7, 31)
                );

            var tree = comp.SyntaxTrees[0];
            var model = comp.GetSemanticModel(tree);
            var lambda = tree.GetRoot().DescendantNodes().Where(n => n.IsKind(SyntaxKind.SimpleLambdaExpression)).Single();

            var param = lambda.ChildNodes().Where(n => n.IsKind(SyntaxKind.Parameter)).Single();
            var symbol1 = model.GetDeclaredSymbol(param);
            Assert.Equal("System.Int32 i", symbol1.ToTestDisplayString());

            var id = lambda.DescendantNodes().First(n => n.IsKind(SyntaxKind.IdentifierName));
            var symbol2 = model.GetSymbolInfo(id).Symbol;
            Assert.Equal("System.Int32 i", symbol2.ToTestDisplayString());

            Assert.Same(symbol1, symbol2);
        }

        [Fact, WorkItem(32179, "https://github.com/dotnet/roslyn/issues/32179")]
        public void SuppressNullableWarning_DefaultStruct()
        {
            var source =
@"public struct S
{
    public string field;
}
public class C
{
    public S field;
    void M()
    {
        S s = default; // assigns null to S.field
        _ = s;
        _ = new C(); // assigns null to C.S.field
    }
}";
            // We should probably warn for such scenarios (either in the definition of S, or in the usage of S)
            // Tracked by https://github.com/dotnet/roslyn/issues/32179
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_ThrowNull()
        {
            var source =
@"class C
{
    void M()
    {
        throw null!;
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_ThrowExpression()
        {
            var source =
@"class C
{
    void M()
    {
        (throw null!)!;
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (5,9): error CS0201: Only assignment, call, increment, decrement, await, and new object expressions can be used as a statement
                //         (throw null!)!;
                Diagnostic(ErrorCode.ERR_IllegalStatement, "(throw null!)!").WithLocation(5, 9),
                // (5,10): error CS8115: A throw expression is not allowed in this context.
                //         (throw null!)!;
                Diagnostic(ErrorCode.ERR_ThrowMisplaced, "throw").WithLocation(5, 10)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_IsNull()
        {
            var source =
@"class C
{
    bool M(object o)
    {
        return o is null!;
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (5,21): error CS8598: The suppression operator is not allowed in this context
                //         return o is null!;
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "null!").WithLocation(5, 21));
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_MiscNull()
        {
            var source = @"
using System.Linq.Expressions;
class C
{
    void M<T>(object o, int? i)
    {
        _ = null is object; // 1
        _ = null! is object; // 2

        int i2 = null!; // 3
        var i3 = (int)null!; // 4
        T t = null!; // 5
        var t2 = (T)null!; // 6
        _ = null == null!;
        _ = (null!, null) == (null, null!);
        (null)++; // 9
        (null!)++; // 10
        _ = !null; // 11
        _ = !(null!); // 12
        Expression<System.Func<object>> testExpr = () => null! ?? ""hello""; // 13
        _ = o == null;
        _ = o == null!; // 14
        _ = null ?? o;
        _ = null! ?? o;
        _ = i == null;
        _ = i == null!;
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (7,13): warning CS0184: The given expression is never of the provided ('object') type
                //         _ = null is object; // 1
                Diagnostic(ErrorCode.WRN_IsAlwaysFalse, "null is object").WithArguments("object").WithLocation(7, 13),
                // (8,13): warning CS0184: The given expression is never of the provided ('object') type
                //         _ = null! is object; // 2
                Diagnostic(ErrorCode.WRN_IsAlwaysFalse, "null! is object").WithArguments("object").WithLocation(8, 13),
                // (10,18): error CS0037: Cannot convert null to 'int' because it is a non-nullable value type
                //         int i2 = null!; // 3
                Diagnostic(ErrorCode.ERR_ValueCantBeNull, "null").WithArguments("int").WithLocation(10, 18),
                // (11,18): error CS0037: Cannot convert null to 'int' because it is a non-nullable value type
                //         var i3 = (int)null!; // 4
                Diagnostic(ErrorCode.ERR_ValueCantBeNull, "(int)null!").WithArguments("int").WithLocation(11, 18),
                // (12,15): error CS0403: Cannot convert null to type parameter 'T' because it could be a non-nullable value type. Consider using 'default(T)' instead.
                //         T t = null!; // 5
                Diagnostic(ErrorCode.ERR_TypeVarCantBeNull, "null").WithArguments("T").WithLocation(12, 15),
                // (13,18): error CS0037: Cannot convert null to 'T' because it is a non-nullable value type
                //         var t2 = (T)null!; // 6
                Diagnostic(ErrorCode.ERR_ValueCantBeNull, "(T)null!").WithArguments("T").WithLocation(13, 18),
                // (16,10): error CS1059: The operand of an increment or decrement operator must be a variable, property or indexer
                //         (null)++; // 9
                Diagnostic(ErrorCode.ERR_IncrementLvalueExpected, "null").WithLocation(16, 10),
                // (17,10): error CS1059: The operand of an increment or decrement operator must be a variable, property or indexer
                //         (null!)++; // 10
                Diagnostic(ErrorCode.ERR_IncrementLvalueExpected, "null").WithLocation(17, 10),
                // (18,13): error CS8310: Operator '!' cannot be applied to operand '<null>'
                //         _ = !null; // 11
                Diagnostic(ErrorCode.ERR_BadOpOnNullOrDefaultOrNew, "!null").WithArguments("!", "<null>").WithLocation(18, 13),
                // (19,13): error CS8310: Operator '!' cannot be applied to operand '<null>'
                //         _ = !(null!); // 12
                Diagnostic(ErrorCode.ERR_BadOpOnNullOrDefaultOrNew, "!(null!)").WithArguments("!", "<null>").WithLocation(19, 13),
                // (20,58): error CS0845: An expression tree lambda may not contain a coalescing operator with a null or default literal left-hand side
                //         Expression<System.Func<object>> testExpr = () => null! ?? "hello"; // 13
                Diagnostic(ErrorCode.ERR_ExpressionTreeContainsBadCoalesce, "null").WithLocation(20, 58)
                );
        }

        [Fact, WorkItem(31370, "https://github.com/dotnet/roslyn/issues/31370")]
        public void SuppressNullableWarning_MiscDefault()
        {
            var source =
@"class C
{
    void M(dynamic d, int? i)
    {
        d.M(null!, default!, null, default); // 1
        _ = default == default!; // 2
        _ = default! == default!; // 3
        _ = 1 + default!; // 4
        _ = default ?? d; // 5
        _ = default! ?? d; // 6
        _ = i ?? default;
        _ = i ?? default!;
    }
    void M2(object o)
    {
        _ = o == default;
        _ = o == default!;
    }
}";
            // Should `!` be disallowed on arguments to dynamic (line // 1) ?
            // See https://github.com/dotnet/roslyn/issues/32364

            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (5,20): error CS8716: There is no target type for the default literal.
                //         d.M(null!, default!, null, default); // 1
                Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(5, 20),
                // (5,36): error CS8716: There is no target type for the default literal.
                //         d.M(null!, default!, null, default); // 1
                Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(5, 36),
                // (6,13): error CS8315: Operator '==' is ambiguous on operands 'default' and 'default'
                //         _ = default == default!; // 2
                Diagnostic(ErrorCode.ERR_AmbigBinaryOpsOnDefault, "default == default!").WithArguments("==", "default", "default").WithLocation(6, 13),
                // (7,13): error CS8315: Operator '==' is ambiguous on operands 'default' and 'default'
                //         _ = default! == default!; // 3
                Diagnostic(ErrorCode.ERR_AmbigBinaryOpsOnDefault, "default! == default!").WithArguments("==", "default", "default").WithLocation(7, 13),
                // (8,13): error CS8310: Operator '+' cannot be applied to operand 'default'
                //         _ = 1 + default!; // 4
                Diagnostic(ErrorCode.ERR_BadOpOnNullOrDefaultOrNew, "1 + default!").WithArguments("+", "default").WithLocation(8, 13),
                // (9,13): error CS8716: There is no target type for the default literal.
                //         _ = default ?? d; // 5
                Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(9, 13),
                // (10,13): error CS8716: There is no target type for the default literal.
                //         _ = default! ?? d; // 6
                Diagnostic(ErrorCode.ERR_DefaultLiteralNoTargetType, "default").WithLocation(10, 13)
            );
        }

        [Fact, WorkItem(32879, "https://github.com/dotnet/roslyn/issues/32879")]
        public void ThrowExpressionInNullCoalescingOperator()
        {
            var source = @"
class C
{
    string Field = string.Empty;
    void M(C? c)
    {
        _ = c ?? throw new System.ArgumentNullException(nameof(c));
        c.ToString();
    }
    void M2(C? c)
    {
        _ = c?.Field ?? throw new System.ArgumentNullException(nameof(c));
        c.ToString();
        c.Field.ToString();
    }
    void M3(C? c)
    {
        _ = c ?? throw new System.ArgumentNullException(c.ToString()); // 1
    }
    void M4(string? s)
    {
        _ = s ?? s.ToString(); // 2
        s.ToString();
    }
    void M5(string s)
    {
        _ = s ?? s.ToString(); // 3
    }
#nullable disable
    void M6(string s)
    {
#nullable enable
        _ = s ?? s.ToString(); // 4
    }
}
";
            CreateCompilation(source, options: WithNullableEnable()).VerifyDiagnostics(
                // (18,57): warning CS8602: Dereference of a possibly null reference.
                //         _ = c ?? throw new System.ArgumentNullException(c.ToString()); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c").WithLocation(18, 57),
                // (22,18): warning CS8602: Dereference of a possibly null reference.
                //         _ = s ?? s.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(22, 18),
                // (27,18): warning CS8602: Dereference of a possibly null reference.
                //         _ = s ?? s.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(27, 18),
                // (33,18): warning CS8602: Dereference of a possibly null reference.
                //         _ = s ?? s.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(33, 18));
        }

        [Fact, WorkItem(32877, "https://github.com/dotnet/roslyn/issues/32877")]
        public void CheckReceiverOfThrow()
        {
            var source = @"
class C
{
    void M(System.Exception? e, C? c)
    {
        _ = c ?? throw e; // 1
        _ = c ?? throw null; // 2
        throw null; // 3
    }
    void M2(System.Exception? e, bool b)
    {
        if (b)
            throw e; // 4
        else
            throw e!;
    }
    void M3()
    {
        throw this; // 5
    }
    public static implicit operator System.Exception?(C c) => throw null!;
    void M4<TException>(TException? e) where TException : System.Exception
    {
        throw e; // 6
    }
    void M5<TException>(TException e) where TException : System.Exception?
    {
        throw e; // 7
    }
    void M6<TException>(TException? e) where TException : Interface
    {
        throw e; // 8
    }
    void M7<TException>(TException e) where TException : Interface?
    {
        throw e; // 9
    }
    void M8<TException>(TException e) where TException : Interface
    {
        throw e; // 10
    }
    void M9<T>(T e)
    {
        throw e; // 11
    }
    void M10()
    {
        try
        {
        }
        catch
        {
            throw;
        }
    }
    interface Interface { }
}";
            CreateCompilation(source, options: WithNullableEnable(), parseOptions: TestOptions.Regular8).VerifyDiagnostics(
                    // (6,24): warning CS8597: Thrown value may be null.
                    //         _ = c ?? throw e; // 1
                    Diagnostic(ErrorCode.WRN_ThrowPossibleNull, "e").WithLocation(6, 24),
                    // (7,24): warning CS8597: Thrown value may be null.
                    //         _ = c ?? throw null; // 2
                    Diagnostic(ErrorCode.WRN_ThrowPossibleNull, "null").WithLocation(7, 24),
                    // (8,15): warning CS8597: Thrown value may be null.
                    //         throw null; // 3
                    Diagnostic(ErrorCode.WRN_ThrowPossibleNull, "null").WithLocation(8, 15),
                    // (13,19): warning CS8597: Thrown value may be null.
                    //             throw e; // 4
                    Diagnostic(ErrorCode.WRN_ThrowPossibleNull, "e").WithLocation(13, 19),
                    // (19,15): warning CS8597: Thrown value may be null.
                    //         throw this; // 5
                    Diagnostic(ErrorCode.WRN_ThrowPossibleNull, "this").WithLocation(19, 15),
                    // (24,15): warning CS8597: Thrown value may be null.
                    //         throw e; // 6
                    Diagnostic(ErrorCode.WRN_ThrowPossibleNull, "e").WithLocation(24, 15),
                    // (28,15): warning CS8597: Thrown value may be null.
                    //         throw e; // 7
                    Diagnostic(ErrorCode.WRN_ThrowPossibleNull, "e").WithLocation(28, 15),
                    // (30,25): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                    //     void M6<TException>(TException? e) where TException : Interface
                    Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "TException?").WithArguments("9.0").WithLocation(30, 25),
                    // (32,15): error CS0029: Cannot implicitly convert type 'TException' to 'System.Exception'
                    //         throw e; // 8
                    Diagnostic(ErrorCode.ERR_NoImplicitConv, "e").WithArguments("TException", "System.Exception").WithLocation(32, 15),
                    // (36,15): error CS0029: Cannot implicitly convert type 'TException' to 'System.Exception'
                    //         throw e; // 9
                    Diagnostic(ErrorCode.ERR_NoImplicitConv, "e").WithArguments("TException", "System.Exception").WithLocation(36, 15),
                    // (40,15): error CS0029: Cannot implicitly convert type 'TException' to 'System.Exception'
                    //         throw e; // 10
                    Diagnostic(ErrorCode.ERR_NoImplicitConv, "e").WithArguments("TException", "System.Exception").WithLocation(40, 15),
                    // (44,15): error CS0029: Cannot implicitly convert type 'T' to 'System.Exception'
                    //         throw e; // 11
                    Diagnostic(ErrorCode.ERR_NoImplicitConv, "e").WithArguments("T", "System.Exception").WithLocation(44, 15)
                );
        }

        [Fact, WorkItem(32877, "https://github.com/dotnet/roslyn/issues/32877")]
        public void CatchClause()
        {
            var source = @"
class C
{
    void M()
    {
        try { }
        catch (System.Exception? e) { throw e; }
    }
    void M2()
    {
        try { }
        catch (System.Exception? e) { throw; }
    }
    void M3()
    {
        try { }
        catch (System.Exception? e)
        {
            e = null;
            throw e; // 1
        }
    }
    void M4()
    {
        try { }
        catch (System.Exception e)
        {
            e = null; // 2
            throw e; // 3
        }
    }
    void M5()
    {
        try { }
        catch (System.Exception e)
        {
            e = null; // 4
            throw; // this rethrows the original exception, not an NRE
        }
    }
}";
            CreateCompilation(source, options: WithNullableEnable()).VerifyDiagnostics(
                // (12,34): warning CS0168: The variable 'e' is declared but never used
                //         catch (System.Exception? e) { throw; }
                Diagnostic(ErrorCode.WRN_UnreferencedVar, "e").WithArguments("e").WithLocation(12, 34),
                // (20,19): error CS8597: Thrown value may be null.
                //             throw e; // 1
                Diagnostic(ErrorCode.WRN_ThrowPossibleNull, "e").WithLocation(20, 19),
                // (28,17): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //             e = null; // 2
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(28, 17),
                // (29,19): error CS8597: Thrown value may be null.
                //             throw e; // 3
                Diagnostic(ErrorCode.WRN_ThrowPossibleNull, "e").WithLocation(29, 19),
                // (35,33): warning CS0168: The variable 'e' is declared but never used
                //         catch (System.Exception e)
                Diagnostic(ErrorCode.WRN_UnreferencedVar, "e").WithArguments("e").WithLocation(35, 33),
                // (37,17): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //             e = null; // 4
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(37, 17)
                );
        }

        [Fact]
        public void Test0()
        {
            var source = @"
class C
{
    static void Main()
    {
        string? x = null;
    }
}
";
            var c = CreateCompilation(source, parseOptions: TestOptions.Regular7);
            c.VerifyDiagnostics(
                // (6,15): error CS8652: The feature 'nullable reference types' is not available in C# 7.0. Please use language version 8.0 or greater.
                //         string? x = null;
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion7, "?").WithArguments("nullable reference types", "8.0").WithLocation(6, 15),
                // (6,17): warning CS0219: The variable 'x' is assigned but its value is never used
                //         string? x = null;
                Diagnostic(ErrorCode.WRN_UnreferencedVarAssg, "x").WithArguments("x").WithLocation(6, 17)
                );

            var c2 = CreateCompilation(source, parseOptions: TestOptions.Regular8);
            c2.VerifyDiagnostics(
                // (6,15): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //         string? x = null;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(6, 15),
                // (6,17): warning CS0219: The variable 'x' is assigned but its value is never used
                //         string? x = null;
                Diagnostic(ErrorCode.WRN_UnreferencedVarAssg, "x").WithArguments("x").WithLocation(6, 17)
                );
        }

        [Fact]
        public void SpeakableInference_MethodTypeInference()
        {
            var source =
@"class Program
{
    void M<T>(T t)
    {
        if (t == null) throw null!;
        t.ToString();
        var t2 = Copy(t);
        t2.ToString(); // warn
    }
    static T Copy<T>(T t) => throw null!;
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (8,9): warning CS8602: Dereference of a possibly null reference.
                //         t2.ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "t2").WithLocation(8, 9)
                );
        }

        [Fact]
        public void SpeakableInference_MethodTypeInference_WithTuple()
        {
            var source =
@"class Program
{
    void M<T>(T t)
    {
        if (t == null) throw null!;
        var tuple = (t, t);
        tuple.Item1.ToString();
        tuple.Item2.ToString();

        var tuple2 = Copy(tuple);
        tuple2.Item1.ToString(); // warn
        tuple2.Item2.ToString(); // warn

        var tuple3 = Copy<T, T>(tuple);
        tuple3.Item1.ToString(); // warn
        tuple3.Item2.ToString(); // warn
    }
    static (T, U) Copy<T, U>((T, U) t) => throw null!;
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (11,9): warning CS8602: Dereference of a possibly null reference.
                //         tuple2.Item1.ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "tuple2.Item1").WithLocation(11, 9),
                // (12,9): warning CS8602: Dereference of a possibly null reference.
                //         tuple2.Item2.ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "tuple2.Item2").WithLocation(12, 9),
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         tuple3.Item1.ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "tuple3.Item1").WithLocation(15, 9),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         tuple3.Item2.ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "tuple3.Item2").WithLocation(16, 9)
                );
        }

        [Fact]
        public void SpeakableInference_MethodTypeInference_WithNull()
        {
            var source =
@"class Program
{
    void M<T>(T t) where T : class?
    {
        if (t == null) throw null!;
        t.ToString();
        var t2 = Copy(t, null);
        t2.ToString(); // warn
    }
    static T Copy<T, U>(T t, U t) => throw null!;
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (7,18): error CS0411: The type arguments for method 'Program.Copy<T, U>(T, U)' cannot be inferred from the usage. Try specifying the type arguments explicitly.
                //         var t2 = Copy(t, null);
                Diagnostic(ErrorCode.ERR_CantInferMethTypeArgs, "Copy").WithArguments("Program.Copy<T, U>(T, U)").WithLocation(7, 18),
                // (10,32): error CS0100: The parameter name 't' is a duplicate
                //     static T Copy<T, U>(T t, U t) => throw null!;
                Diagnostic(ErrorCode.ERR_DuplicateParamName, "t").WithArguments("t").WithLocation(10, 32)
                );
        }

        [Fact]
        public void SpeakableInference_MethodTypeInference_NullAssigned()
        {
            var source =
@"class Program
{
    void M<T>(T t) where T : class
    {
        t = null;
        var t2 = Copy(t);
        t2 /*T:T?*/ .ToString(); // warn
    }
    static T Copy<T>(T t) => throw null!;
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyTypes();
            comp.VerifyDiagnostics(
                // (5,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         t = null;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(5, 13),
                // (7,9): warning CS8602: Dereference of a possibly null reference.
                //         t2 /*T:T?*/ .ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "t2").WithLocation(7, 9)
                );
        }

        [Fact]
        public void SpeakableInference_MethodTypeInference_NullableValueType()
        {
            var source =
@"class Program
{
    void M(int? t)
    {
        if (t == null) throw null!;
        t.Value.ToString();
        var t2 = Copy(t);
        t2.Value.ToString(); // warn
    }
    void M2<T>(T? t) where T : struct
    {
        if (t == null) throw null!;
        t.Value.ToString();
        var t2 = Copy(t);
        t2.Value.ToString(); // warn
    }
     static T Copy<T>(T t) => throw null!;
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (8,9): warning CS8629: Nullable value type may be null.
                //         t2.Value.ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullableValueTypeMayBeNull, "t2").WithLocation(8, 9),
                // (15,9): warning CS8629: Nullable value type may be null.
                //         t2.Value.ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullableValueTypeMayBeNull, "t2").WithLocation(15, 9)
                );
        }

        [Fact]
        public void SpeakableInference_ArrayTypeInference()
        {
            var source =
@"class Program
{
    void M<T>(T t)
    {
        if (t == null) throw null!;
        t.ToString();
        var t2 = new[] { t };
        t2[0].ToString(); // warn
    }
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (8,9): warning CS8602: Dereference of a possibly null reference.
                //         t2[0].ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "t2[0]").WithLocation(8, 9)
                );
        }

        [Fact]
        public void SpeakableInference_ArrayTypeInference_WithTuple()
        {
            var source =
@"class Program
{
    void M<T>(T t)
    {
        if (t == null) throw null!;
        var a = new[] { (t, t) };
        a[0].Item1.ToString();
        a[0].Item2.ToString();

        var b = new (T, T)[] { (t, t) };
        b[0].Item1.ToString();
        b[0].Item2.ToString();
    }
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (7,9): warning CS8602: Dereference of a possibly null reference.
                //         a[0].Item1.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "a[0].Item1").WithLocation(7, 9),
                // (8,9): warning CS8602: Dereference of a possibly null reference.
                //         a[0].Item2.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "a[0].Item2").WithLocation(8, 9),
                // (11,9): warning CS8602: Dereference of a possibly null reference.
                //         b[0].Item1.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "b[0].Item1").WithLocation(11, 9),
                // (12,9): warning CS8602: Dereference of a possibly null reference.
                //         b[0].Item2.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "b[0].Item2").WithLocation(12, 9)
                );
        }

        [Fact]
        public void SpeakableInference_ArrayTypeInference_WithNull()
        {
            var source =
@"class Program
{
    void M<T>(T t) where T : class?
    {
        if (t == null) throw null!;
        t.ToString();
        var t2 = new[] { t, null };
        t2[0].ToString(); // 1
    }
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (8,9): warning CS8602: Dereference of a possibly null reference.
                //         t2[0].ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "t2[0]").WithLocation(8, 9)
                );
        }

        [Fact, WorkItem(30941, "https://github.com/dotnet/roslyn/issues/30941")]
        public void Verify30941()
        {
            var source = @"
class Outer : Base
{
    void M1(Base? x1)
    {
        Outer y = x1;
    }
}
class Base { }
";
            CreateCompilation(source, options: WithNullableEnable()).VerifyDiagnostics(
                // (6,19): error CS0266: Cannot implicitly convert type 'Base' to 'Outer'. An explicit conversion exists (are you missing a cast?)
                //         Outer y = x1;
                Diagnostic(ErrorCode.ERR_NoImplicitConvCast, "x1").WithArguments("Base", "Outer").WithLocation(6, 19),
                // (6,19): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         Outer y = x1;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x1").WithLocation(6, 19)
                );
        }

        [Fact, WorkItem(31958, "https://github.com/dotnet/roslyn/issues/31958")]
        public void Verify31958()
        {
            var source = @"
class C
{
    string? M1(string s)
    {
        var d = (D)M1; // 1
        d(null).ToString();

        return null;
    }

    string M2(string s)
    {
        var d = (D)M2; // 2
        d(null).ToString();

        return s;
    }

    delegate string D(string? s);
}
";
            CreateCompilation(source, options: WithNullableEnable()).VerifyDiagnostics(
                // (6,17): warning CS8621: Nullability of reference types in return type of 'string? C.M1(string s)' doesn't match the target delegate 'C.D' (possibly because of nullability attributes).
                //         var d = (D)M1; // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOfTargetDelegate, "(D)M1").WithArguments("string? C.M1(string s)", "C.D").WithLocation(6, 17),
                // (14,17): warning CS8622: Nullability of reference types in type of parameter 's' of 'string C.M2(string s)' doesn't match the target delegate 'C.D' (possibly because of nullability attributes).
                //         var d = (D)M2; // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "(D)M2").WithArguments("s", "string C.M2(string s)", "C.D").WithLocation(14, 17)
                );
        }

        [Fact, WorkItem(28377, "https://github.com/dotnet/roslyn/issues/28377")]
        public void Verify28377()
        {
            var source = @"
class C
{
    void M()
    {
        object x;
        x! = null;
    }
}
";
            CreateCompilation(source, options: WithNullableEnable()).VerifyDiagnostics(
                // (6,16): warning CS0219: The variable 'x' is assigned but its value is never used
                //         object x;
                Diagnostic(ErrorCode.WRN_UnreferencedVarAssg, "x").WithArguments("x").WithLocation(6, 16),
                // (7,9): error CS8598: The suppression operator is not allowed in this context
                //         x! = null;
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "x").WithLocation(7, 9),
                // (7,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         x! = null;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(7, 14)
                );
        }

        [Fact, WorkItem(31295, "https://github.com/dotnet/roslyn/issues/31295")]
        public void Verify31295()
        {
            var source = @"
class C
{
    void M()
    {
        for (var x = 0; ; x!++) { }
    }
}";
            CreateCompilation(source, options: WithNullableEnable()).VerifyDiagnostics();
        }

        [Fact, WorkItem(26654, "https://github.com/dotnet/roslyn/issues/26654")]
        public void Verify26654()
        {
            var source = @"
public class C
{
    public void M(out string? x) => throw null!;
    public void M2()
    {
        string y;
        M(out y!);
    }
}";
            CreateCompilation(source, options: WithNullableEnable()).VerifyDiagnostics();
        }

        [Fact, WorkItem(33782, "https://github.com/dotnet/roslyn/issues/33782")]
        public void AnnotationInXmlDoc_TwoDeclarations()
        {
            var source = @"
/// <summary />
public class C
{
    void M<T>(T? t) where T : struct { }
    void M<T>(T t) where T : class { }

    /// <summary>
    /// See <see cref=""M{T}(T?)""/>
    /// </summary>
    void M2() { }

    /// <summary>
    /// See <see cref=""M{T}(T)""/>
    /// </summary>
    void M3() { }
}";
            // In cref, `T?` always binds to `Nullable<T>`
            var comp = CreateCompilation(source, parseOptions: TestOptions.RegularWithDocumentationComments, options: WithNullableEnable());
            comp.VerifyDiagnostics();

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree, ignoreAccessibility: false);
            var firstCref = tree.GetRoot().DescendantNodes(descendIntoTrivia: true).OfType<NameMemberCrefSyntax>().First();
            var firstCrefSymbol = model.GetSymbolInfo(firstCref).Symbol;
            Assert.Equal("void C.M<T>(T? t)", firstCrefSymbol.ToTestDisplayString());

            var lastCref = tree.GetRoot().DescendantNodes(descendIntoTrivia: true).OfType<NameMemberCrefSyntax>().Last();
            var lastCrefSymbol = model.GetSymbolInfo(lastCref).Symbol;
            Assert.Equal("void C.M<T>(T t)", lastCrefSymbol.ToTestDisplayString());
        }

        [Fact, WorkItem(33782, "https://github.com/dotnet/roslyn/issues/33782")]
        public void AnnotationInXmlDoc_DeclaredAsNonNullableReferenceType()
        {
            var source = @"
/// <summary />
public class C
{
    void M<T>(T t) where T : class { }

    /// <summary>
    /// <see cref=""M{T}(T?)""/> warn 1
    /// </summary>
    void M2() { }

    /// <summary>
    /// <see cref=""M{T}(T)""/>
    /// </summary>
    void M3() { }
}";
            var comp = CreateCompilation(source, parseOptions: TestOptions.RegularWithDocumentationComments, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (8,20): warning CS1574: XML comment has cref attribute 'M{T}(T?)' that could not be resolved
                //     /// <see cref="M{T}(T?)"/> warn 1
                Diagnostic(ErrorCode.WRN_BadXMLRef, "M{T}(T?)").WithArguments("M{T}(T?)").WithLocation(8, 20));

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree, ignoreAccessibility: false);
            var lastCref = tree.GetRoot().DescendantNodes(descendIntoTrivia: true).OfType<NameMemberCrefSyntax>().Last();
            var lastCrefSymbol = model.GetSymbolInfo(lastCref).Symbol;
            Assert.Equal("void C.M<T>(T t)", lastCrefSymbol.ToTestDisplayString());
        }

        [Fact, WorkItem(33782, "https://github.com/dotnet/roslyn/issues/33782")]
        public void AnnotationInXmlDoc_DeclaredAsNullableReferenceType()
        {
            var source = @"
/// <summary />
public class C
{
    void M<T>(T? t) where T : class { }

    /// <summary>
    /// <see cref=""M{T}(T?)""/> warn 1
    /// </summary>
    void M2() { }

    /// <summary>
    /// <see cref=""M{T}(T)""/>
    /// </summary>
    void M3() { }
}";
            var comp = CreateCompilation(source, parseOptions: TestOptions.RegularWithDocumentationComments, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (8,20): warning CS1574: XML comment has cref attribute 'M{T}(T?)' that could not be resolved
                //     /// <see cref="M{T}(T?)"/> warn 1
                Diagnostic(ErrorCode.WRN_BadXMLRef, "M{T}(T?)").WithArguments("M{T}(T?)").WithLocation(8, 20));

            var tree = comp.SyntaxTrees.Single();
            var model = comp.GetSemanticModel(tree, ignoreAccessibility: false);
            var lastCref = tree.GetRoot().DescendantNodes(descendIntoTrivia: true).OfType<NameMemberCrefSyntax>().Last();
            var lastCrefSymbol = model.GetSymbolInfo(lastCref).Symbol;
            Assert.Equal("void C.M<T>(T? t)", lastCrefSymbol.ToTestDisplayString());
        }

        [Fact, WorkItem(30955, "https://github.com/dotnet/roslyn/issues/30955")]
        public void ArrayTypeInference_Verify30955()
        {
            var source = @"
class Outer
{
    void M0(object x0, object? y0)
    {
        var a = new[] { x0, 1 };
        a[0] = y0;
        var b = new[] { x0, x0 };
        b[0] = y0;
    }
}
";

            CreateCompilation(source, options: WithNullableEnable()).VerifyDiagnostics(
                // (7,16): warning CS8601: Possible null reference assignment.
                //         a[0] = y0;
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "y0").WithLocation(7, 16),
                // (9,16): warning CS8601: Possible null reference assignment.
                //         b[0] = y0;
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "y0").WithLocation(9, 16)
                );
        }

        [Fact, WorkItem(30598, "https://github.com/dotnet/roslyn/issues/30598")]
        public void Verify30598()
        {
            var source = @"
class Program
{
    static object F(object[]? x)
    {
        return x[          // warning: possibly null
            x.Length - 1]; // no warning
    }
}
";

            CreateCompilation(source, options: WithNullableEnable()).VerifyDiagnostics(
                // (6,16): warning CS8602: Dereference of a possibly null reference.
                //         return x[          // warning: possibly null
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x").WithLocation(6, 16)
                );
        }

        [Fact, WorkItem(30925, "https://github.com/dotnet/roslyn/issues/30925")]
        public void Verify30925()
        {
            var source = @"
class Outer
{
    void M1<T>(T x1, object? y1) where T : class?
    {
        x1 = (T)y1;
    }

    void M2<T>(T x2, object? y2) where T : class?
    {
        x2 = y2;
    }

    void M3(string x3, object? y3)
    {
        x3 = (string)y3;
    }

    void M4(string x4, object? y4)
    {
        x4 = y4;
    }
}
";

            CreateCompilation(source, options: WithNullableEnable(), parseOptions: TestOptions.Regular8).VerifyDiagnostics(
                // (11,14): error CS0266: Cannot implicitly convert type 'object' to 'T'. An explicit conversion exists (are you missing a cast?)
                //         x2 = y2;
                Diagnostic(ErrorCode.ERR_NoImplicitConvCast, "y2").WithArguments("object", "T").WithLocation(11, 14),
                // (16,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         x3 = (string)y3;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "(string)y3").WithLocation(16, 14),
                // (21,14): error CS0266: Cannot implicitly convert type 'object' to 'string'. An explicit conversion exists (are you missing a cast?)
                //         x4 = y4;
                Diagnostic(ErrorCode.ERR_NoImplicitConvCast, "y4").WithArguments("object", "string").WithLocation(21, 14),
                // (21,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         x4 = y4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y4").WithLocation(21, 14));
        }

        [Fact]
        public void SpeakableInference_ArrayTypeInference_NullableValueType()
        {
            var source =
@"class Program
{
    void M(int? t)
    {
        if (t == null) throw null!;
        t.Value.ToString();
        var t2 = new[] { t };
        t2[0].Value.ToString(); // warn
    }
    void M2<T>(T? t) where T : struct
    {
        if (t == null) throw null!;
        t.Value.ToString();
        var t2 = new[] { t };
        t2[0].Value.ToString(); // warn
    }
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (8,9): warning CS8629: Nullable value type may be null.
                //         t2[0].Value.ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullableValueTypeMayBeNull, "t2[0]").WithLocation(8, 9),
                // (15,9): warning CS8629: Nullable value type may be null.
                //         t2[0].Value.ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullableValueTypeMayBeNull, "t2[0]").WithLocation(15, 9)
                );
        }

        [Fact]
        public void SpeakableInference_ArrayTypeInference_ConversionWithNullableOutput_WithNestedMismatch()
        {
            var source =
@"class A<T>
{
    public static implicit operator C<T>?(A<T> a) => throw null!;
}
class B<T> : A<T>
{
}
class C<T>
{
    void M(B<string> x, C<string?> y)
    {
        var a = new[] { x, y };
        a[0].ToString();

        var b = new[] { y, x };
        b[0].ToString();
    }
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (12,25): warning CS8619: Nullability of reference types in value of type 'C<string>' doesn't match target type 'C<string?>'.
                //         var a = new[] { x, y };
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("C<string>", "C<string?>").WithLocation(12, 25),
                // (13,9): warning CS8602: Dereference of a possibly null reference.
                //         a[0].ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "a[0]").WithLocation(13, 9),
                // (15,28): warning CS8619: Nullability of reference types in value of type 'C<string>' doesn't match target type 'C<string?>'.
                //         var b = new[] { y, x };
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("C<string>", "C<string?>").WithLocation(15, 28),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         b[0].ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "b[0]").WithLocation(16, 9)
                );
        }

        [Fact]
        public void SpeakableInference_LambdaReturnTypeInference()
        {
            var source =
@"class Program
{
    void M<T>(T t)
    {
        var x1 = F(() =>
        {
            if (t == null) throw null!;
            bool b = true;
            if (b) return t;
            return t;
        });
        x1.ToString();

        var x2 = F<T>(() =>
        {
            if (t == null) throw null!;
            bool b = true;
            if (b) return t;
            return t;
        });
        x2.ToString();
    }
    T F<T>(System.Func<T> f) => throw null!;
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (12,9): warning CS8602: Dereference of a possibly null reference.
                //         x1.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x1").WithLocation(12, 9),
                // (21,9): warning CS8602: Dereference of a possibly null reference.
                //         x2.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x2").WithLocation(21, 9)
                );
        }

        [Fact]
        public void SpeakableInference_LambdaReturnTypeInference2()
        {
            var source =
@"class Program
{
    void M<T>(T t, T t2)
    {
        var x1 = F(() =>
        {
            if (t == null) throw null!;
            bool b = true;
            if (b) return t;
            return t2;
        });
        x1.ToString();

        var x2 = F<T>(() =>
        {
            if (t == null) throw null!;
            bool b = true;
            if (b) return t;
            return t2;
        });
        x2.ToString();
    }
    T F<T>(System.Func<T> f) => throw null!;
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (12,9): warning CS8602: Dereference of a possibly null reference.
                //         x1.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x1").WithLocation(12, 9),
                // (21,9): warning CS8602: Dereference of a possibly null reference.
                //         x2.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x2").WithLocation(21, 9)
                );
        }

        [Fact]
        public void SpeakableInference_LambdaReturnTypeInference_WithSingleReturn()
        {
            var source =
@"class Program
{
    void M<T>()
    {
        var x1 = Copy(() => """");
        x1.ToString();
    }
    void M2<T>(T t)
    {
        if (t == null) throw null!;
        var x1 = Copy(() => t);
        x1.ToString(); // 1
    }
    T Copy<T>(System.Func<T> f) => throw null!;
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (12,9): warning CS8602: Dereference of a possibly null reference.
                //         x1.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x1").WithLocation(12, 9)
                );
        }

        [Fact]
        public void SpeakableInference_LambdaReturnTypeInference_WithTuple()
        {
            var source =
@"class Program
{
    void M<T>(T t)
    {
        var x1 = Copy(() =>
        {
            if (t == null) throw null!;
            bool b = true;
            if (b) return (t, t);
            return (t, t);
        });
        x1.Item1.ToString();
        x1.Item2.ToString();
    }
    T Copy<T>(System.Func<T> f) => throw null!;
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (12,9): warning CS8602: Dereference of a possibly null reference.
                //         x1.Item1.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x1.Item1").WithLocation(12, 9),
                // (13,9): warning CS8602: Dereference of a possibly null reference.
                //         x1.Item2.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x1.Item2").WithLocation(13, 9)
                );
        }

        [Fact]
        public void SpeakableInference_LambdaReturnTypeInference_ConversionWithNullableOutput()
        {
            var source =
@"class A
{
    public static implicit operator C?(A a) => new C();
}
class B : A
{
}
class C
{
    void M(B x, C y)
    {
        var x1 = F(() =>
        {
            bool b = true;
            if (b) return x;
            return y;
        });
        x1.ToString();

        var x2 = F(() =>
        {
            bool b = true;
            if (b) return y;
            return x;
        });
        x2.ToString();
    }
    T F<T>(System.Func<T> f) => throw null!;
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (18,9): warning CS8602: Dereference of a possibly null reference.
                //         x1.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x1").WithLocation(18, 9),
                // (26,9): warning CS8602: Dereference of a possibly null reference.
                //         x2.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x2").WithLocation(26, 9)
                );
        }

        [Fact]
        public void SpeakableInference_LambdaReturnTypeInference_ConversionWithNullableOutput2()
        {
            var source =
@"class A<T>
{
    public static implicit operator C<T>?(A<T> a) => throw null!;
}
class B<T> : A<T>
{
}
class C<T>
{
    void M(B<string?> x, C<string?> y)
    {
        var x1 = F(() =>
        {
            bool b = true;
            if (b) return x;
            return y;
        });
        x1.ToString();

        var x2 = F(() =>
        {
            bool b = true;
            if (b) return y;
            return x;
        });
        x2.ToString();
    }
    U F<U>(System.Func<U> f) => throw null!;
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (18,9): warning CS8602: Dereference of a possibly null reference.
                //         x1.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x1").WithLocation(18, 9),
                // (26,9): warning CS8602: Dereference of a possibly null reference.
                //         x2.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x2").WithLocation(26, 9)
                );
        }

        [Fact]
        public void SpeakableInference_LambdaReturnTypeInference_ConversionWithNullableOutput_WithNestedMismatch()
        {
            var source =
@"class A<T>
{
    public static implicit operator C<T>?(A<T> a) => throw null!;
}
class B<T> : A<T>
{
}
class C<T>
{
    void M(B<string> x, C<string?> y)
    {
        var x1 = F(() =>
        {
            bool b = true;
            if (b) return x;
            return y;
        });
        x1.ToString();

        var x2 = F(() =>
        {
            bool b = true;
            if (b) return y;
            return x;
        });
        x2.ToString();
    }
    U F<U>(System.Func<U> f) => throw null!;
}";

            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (15,27): warning CS8619: Nullability of reference types in value of type 'C<string>' doesn't match target type 'C<string?>'.
                //             if (b) return x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("C<string>", "C<string?>").WithLocation(15, 27),
                // (18,9): warning CS8602: Dereference of a possibly null reference.
                //         x1.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x1").WithLocation(18, 9),
                // (24,20): warning CS8619: Nullability of reference types in value of type 'C<string>' doesn't match target type 'C<string?>'.
                //             return x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("C<string>", "C<string?>").WithLocation(24, 20),
                // (26,9): warning CS8602: Dereference of a possibly null reference.
                //         x2.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x2").WithLocation(26, 9)
                );
        }

        [Fact]
        public void SpeakableInference_LambdaReturnTypeInference_ConversionWithNullableInput()
        {
            var source =
@"class A
{
    public static implicit operator C(A? a) => null; // warn
}
class B : A
{
}
class C
{
    void M(B? x, C y)
    {
        var x1 = F(() =>
        {
            bool b = true;
            if (b) return x;
            return y;
        });
        x1.ToString();

        var x2 = F(() =>
        {
            bool b = true;
            if (b) return y;
            return x;
        });
        x2.ToString();
    }
    T F<T>(System.Func<T> f) => throw null!;
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (3,48): warning CS8603: Possible null reference return.
                //     public static implicit operator C(A? a) => null; // warn
                Diagnostic(ErrorCode.WRN_NullReferenceReturn, "null").WithLocation(3, 48)
                );
        }

        [Fact]
        public void SpeakableInference_LambdaReturnTypeInference_WithNullabilityMismatch()
        {
            var source = @"
class C<T>
{
    void M(C<object>? x, C<object?> y)
    {
        var x1 = F(() =>
        {
            bool b = true;
            if (b) return x;
            return y;
        });
        _ = x1 /*T:C<object!>?*/;
        x1.ToString();

        var x2 = F(() =>
        {
            bool b = true;
            if (b) return y;
            return x;
        });
        _ = x2 /*T:C<object!>?*/;
        x2.ToString();
    }
    U F<U>(System.Func<U> f) => throw null!;
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyTypes();
            comp.VerifyDiagnostics(
                // (10,20): warning CS8619: Nullability of reference types in value of type 'C<object?>' doesn't match target type 'C<object>'.
                //             return y;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "y").WithArguments("C<object?>", "C<object>").WithLocation(10, 20),
                // (13,9): warning CS8602: Dereference of a possibly null reference.
                //         x1.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x1").WithLocation(13, 9),
                // (18,27): warning CS8619: Nullability of reference types in value of type 'C<object?>' doesn't match target type 'C<object>'.
                //             if (b) return y;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "y").WithArguments("C<object?>", "C<object>").WithLocation(18, 27),
                // (22,9): warning CS8602: Dereference of a possibly null reference.
                //         x2.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x2").WithLocation(22, 9)
                );
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/32006")]
        [WorkItem(32006, "https://github.com/dotnet/roslyn/issues/32006")]
        public void SpeakableInference_LambdaReturnTypeInference_NonNullableTypelessOuptut()
        {
            // See https://github.com/dotnet/roslyn/issues/32006
            // need to relax assertion in GetImplicitTupleLiteralConversion
            var source =
@"
class C
{
    void M(C? c)
    {
        var x1 = F(() =>
        {
            bool b = true;
            if (b) return (c, c);
            return (null, null);
        });
        x1.ToString();
        x1.Item1.ToString();
    }
    T F<T>(System.Func<T> f) => throw null!;
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (13,9): warning CS8602: Dereference of a possibly null reference.
                //         x1.Item1.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x1.Item1").WithLocation(13, 9)
                );
        }

        [Fact]
        public void SpeakableInference_VarInference()
        {
            var source =
@"class Program
{
    void M<T>(T t)
    {
        if (t == null) throw null!;
        var t2 = t;
        t2.ToString();
    }
}";
            var comp = CreateCompilationWithIndexAndRange(source, options: WithNullableEnable());
            comp.VerifyDiagnostics();
            var syntaxTree = comp.SyntaxTrees[0];
            var declaration = syntaxTree.GetRoot().DescendantNodes().OfType<VariableDeclaratorSyntax>().Single();
            var model = comp.GetSemanticModel(syntaxTree);
            var local = (ILocalSymbol)model.GetDeclaredSymbol(declaration);
            Assert.Equal("T? t2", local.ToTestDisplayString());
            Assert.Equal(CodeAnalysis.NullableAnnotation.Annotated, local.NullableAnnotation);
            Assert.Equal(CodeAnalysis.NullableAnnotation.Annotated, local.Type.NullableAnnotation);
            Assert.Equal("T?", local.Type.ToTestDisplayString(includeNonNullable: true));
        }

        [Fact]
        public void Directive_Qualifiers()
        {
            var source =
@"#nullable
#nullable enable
#nullable disable
#nullable restore
#nullable safeonly
#nullable yes
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics(
                // (1,10): error CS8637: Expected 'enable', 'disable', or 'restore'
                // #nullable
                Diagnostic(ErrorCode.ERR_NullableDirectiveQualifierExpected, "").WithLocation(1, 10),
                // (5,11): error CS8637: Expected 'enable', 'disable', or 'restore'
                // #nullable safeonly
                Diagnostic(ErrorCode.ERR_NullableDirectiveQualifierExpected, "safeonly").WithLocation(5, 11),
                // (6,11): error CS8637: Expected 'enable', 'disable', or 'restore'
                // #nullable yes
                Diagnostic(ErrorCode.ERR_NullableDirectiveQualifierExpected, "yes").WithLocation(6, 11)
                );
            comp.VerifyTypes();
        }

        [Fact]
        public void Directive_NullableDefault()
        {
            var source =
@"#pragma warning disable 0649, 8618
class A<T, U> { }
class B
{
    static A<object, string?> F1;
    static void G1()
    {
        object o1;
        o1 = F1/*T:A<object, string?>!*/;
        o1 = F2/*T:A<object, string?>!*/;
        o1 = F3/*T:A<object!, string?>!*/;
    }
#nullable disable
    static A<object, string?> F2;
    static void G2()
    {
        object o2;
        o2 = F1/*T:A<object, string?>!*/;
        o2 = F2/*T:A<object, string?>!*/;
        o2 = F3/*T:A<object!, string?>!*/;
    }
#nullable enable
    static A<object, string?> F3;
    static void G3()
    {
        object o3;
        o3 = F1/*T:A<object, string?>!*/;
        o3 = F2/*T:A<object, string?>!*/;
        o3 = F3/*T:A<object!, string?>!*/;
    }
}";
            var comp = CreateCompilation(source, parseOptions: TestOptions.Regular.WithFeature("run-nullable-analysis", "always"), options: TestOptions.DebugDll);
            comp.VerifyDiagnostics(
                // (5,28): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     static A<object, string?> F1;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 28),
                // (14,28): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     static A<object, string?> F2;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(14, 28));
            comp.VerifyTypes();
        }

        [Fact]
        public void Directive_NullableFalse()
        {
            var source =
@"#pragma warning disable 0649, 8618
class A<T, U> { }
class B
{
    static A<object, string?> F1;
    static void G1()
    {
        object o1;
        o1 = F1/*T:A<object, string?>!*/;
        o1 = F2/*T:A<object, string?>!*/;
        o1 = F3/*T:A<object!, string?>!*/;
    }
#nullable disable
    static A<object, string?> F2;
    static void G2()
    {
        object o2;
        o2 = F1/*T:A<object, string?>!*/;
        o2 = F2/*T:A<object, string?>!*/;
        o2 = F3/*T:A<object!, string?>!*/;
    }
#nullable enable
    static A<object, string?> F3;
    static void G3()
    {
        object o3;
        o3 = F1/*T:A<object, string?>!*/;
        o3 = F2/*T:A<object, string?>!*/;
        o3 = F3/*T:A<object!, string?>!*/;
    }
}";
            var comp = CreateCompilation(source, parseOptions: TestOptions.Regular.WithFeature("run-nullable-analysis", "always"), options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Disable));
            comp.VerifyDiagnostics(
                // (5,28): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     static A<object, string?> F1;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 28),
                // (14,28): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     static A<object, string?> F2;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(14, 28));
            comp.VerifyTypes();
        }

        [Fact]
        public void Directive_NullableTrue()
        {
            var source =
@"#pragma warning disable 0649, 8618
class A<T, U> { }
class B
{
    static A<object, string?> F1;
    static void G1()
    {
        object o1;
        o1 = F1/*T:A<object!, string?>!*/;
        o1 = F2/*T:A<object, string?>!*/;
        o1 = F3/*T:A<object!, string?>!*/;
    }
#nullable disable
    static A<object, string?> F2;
    static void G2()
    {
        object o2;
        o2 = F1/*T:A<object!, string?>!*/;
        o2 = F2/*T:A<object, string?>!*/;
        o2 = F3/*T:A<object!, string?>!*/;
    }
#nullable enable
    static A<object, string?> F3;
    static void G3()
    {
        object o3;
        o3 = F1/*T:A<object!, string?>!*/;
        o3 = F2/*T:A<object, string?>!*/;
        o3 = F3/*T:A<object!, string?>!*/;
    }
}";
            var comp = CreateCompilation(source, parseOptions: TestOptions.Regular.WithFeature("run-nullable-analysis", "always"), options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Enable));
            comp.VerifyDiagnostics(
                // (14,28): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     static A<object, string?> F2;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(14, 28));
            comp.VerifyTypes();
        }

        [Fact]
        public void Directive_PartialClasses()
        {
            var source0 =
@"class Base<T> { }
class Program
{
#nullable enable
    static void F(Base<object?> b)
    {
    }
    static void Main()
    {
        F(new C1());
        F(new C2());
        F(new C3());
        F(new C4());
        F(new C5());
        F(new C6());
        F(new C7());
        F(new C8());
        F(new C9());
    }
}";
            var source1 =
@"#pragma warning disable 8632
partial class C1 : Base<object> { }
partial class C2 { }
partial class C3 : Base<object> { }
#nullable disable
partial class C4 { }
partial class C5 : Base<object> { }
partial class C6 { }
#nullable enable
partial class C7 : Base<object> { }
partial class C8 { }
partial class C9 : Base<object> { }
";
            var source2 =
@"#pragma warning disable 8632
partial class C1 { }
partial class C4 : Base<object> { }
partial class C7 { }
#nullable disable
partial class C2 : Base<object> { }
partial class C5 { }
partial class C8 : Base<object> { }
#nullable enable
partial class C3 { }
partial class C6 : Base<object> { }
partial class C9 { }
";

            // -nullable (default):
            var comp = CreateCompilation(new[] { source0, source1, source2 }, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics(
                // (15,11): warning CS8620: Nullability of reference types in argument of type 'C6' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C6());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C6()").WithArguments("C6", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(15, 11),
                // (16,11): warning CS8620: Nullability of reference types in argument of type 'C7' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C7());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C7()").WithArguments("C7", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(16, 11),
                // (18,11): warning CS8620: Nullability of reference types in argument of type 'C9' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C9());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C9()").WithArguments("C9", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(18, 11));

            // -nullable-:
            comp = CreateCompilation(new[] { source0, source1, source2 }, options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Disable));
            comp.VerifyDiagnostics(
                // (15,11): warning CS8620: Nullability of reference types in argument of type 'C6' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C6());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C6()").WithArguments("C6", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(15, 11),
                // (16,11): warning CS8620: Nullability of reference types in argument of type 'C7' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C7());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C7()").WithArguments("C7", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(16, 11),
                // (18,11): warning CS8620: Nullability of reference types in argument of type 'C9' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C9());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C9()").WithArguments("C9", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(18, 11));

            // -nullable+:
            comp = CreateCompilation(new[] { source0, source1, source2 }, options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Enable));
            comp.VerifyDiagnostics(
                // (10,11): warning CS8620: Nullability of reference types in argument of type 'C1' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C1());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C1()").WithArguments("C1", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(10, 11),
                // (12,11): warning CS8620: Nullability of reference types in argument of type 'C3' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C3());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C3()").WithArguments("C3", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(12, 11),
                // (13,11): warning CS8620: Nullability of reference types in argument of type 'C4' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C4());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C4()").WithArguments("C4", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(13, 11),
                // (15,11): warning CS8620: Nullability of reference types in argument of type 'C6' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C6());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C6()").WithArguments("C6", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(15, 11),
                // (16,11): warning CS8620: Nullability of reference types in argument of type 'C7' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C7());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C7()").WithArguments("C7", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(16, 11),
                // (18,11): warning CS8620: Nullability of reference types in argument of type 'C9' doesn't match target type 'Base<object?>' for parameter 'b' in 'void Program.F(Base<object?> b)'.
                //         F(new C9());
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new C9()").WithArguments("C9", "Base<object?>", "b", "void Program.F(Base<object?> b)").WithLocation(18, 11));
        }

        [WorkItem(30840, "https://github.com/dotnet/roslyn/issues/30840")]
        [Fact]
        public void Directive_EnableAndDisable_01()
        {
            var source =
@"#nullable enable
class Program
{
    static void F(object x)
    {
        object? y = null;
        F(y); // warning
    }
#nullable disable
    static void G()
    {
        F(null);
    }
}";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (7,11): warning CS8604: Possible null reference argument for parameter 'x' in 'void Program.F(object x)'.
                //         F(y); // warning
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "y").WithArguments("x", "void Program.F(object x)").WithLocation(7, 11));
        }

        [WorkItem(30840, "https://github.com/dotnet/roslyn/issues/30840")]
        [Fact]
        public void Directive_EnableAndDisable_02()
        {
            var source =
@"class Program
{
#nullable disable
    static void G()
    {
        F(null);
    }
#nullable enable
    static void F(object x)
    {
        object? y = null;
        F(y); // warning
    }
}";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (12,11): warning CS8604: Possible null reference argument for parameter 'x' in 'void Program.F(object x)'.
                //         F(y); // warning
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "y").WithArguments("x", "void Program.F(object x)").WithLocation(12, 11));
        }

        [WorkItem(30840, "https://github.com/dotnet/roslyn/issues/30840")]
        [Fact]
        public void Directive_EnableAndDisable_03()
        {
            var source =
@"class Program
{
    static void F(object x)
    {
        object? y = null;
        F(y); // warning
    }
#nullable disable
    static void G()
    {
        F(null);
    }
}";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,11): warning CS8604: Possible null reference argument for parameter 'x' in 'void Program.F(object x)'.
                //         F(y); // warning
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "y").WithArguments("x", "void Program.F(object x)").WithLocation(6, 11));
        }

        [WorkItem(30840, "https://github.com/dotnet/roslyn/issues/30840")]
        [Fact]
        public void Directive_EnableAndDisable_04()
        {
            var source =
@"class Program
{
    static void G()
    {
        F(null);
    }
#nullable enable
    static void F(object x)
    {
        object? y = null;
        F(y); // warning
    }
}";
            var comp = CreateCompilation(source, options: WithNullableDisable());
            comp.VerifyDiagnostics(
                // (11,11): warning CS8604: Possible null reference argument for parameter 'x' in 'void Program.F(object x)'.
                //         F(y); // warning
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "y").WithArguments("x", "void Program.F(object x)").WithLocation(11, 11));
        }

        [Fact]
        public void Directive_GloballyEnabled_GeneratedCode_DisabledByDefault()
        {
            var source =
@"
// <autogenerated />
class Program
{
    static void F(object x)
    {
        x = null; // no warning, generated file
    }
}";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Enable));
            comp.VerifyDiagnostics();
        }

        [Theory]
        [InlineData("")]
        [InlineData("#nullable enable warnings")]
        [InlineData("#nullable disable")]
        public void Directive_GloballyEnabled_GeneratedCode_Annotations_Disabled(string nullableDirective)
        {
            var source =
$@"
// <autogenerated />
{nullableDirective}
class Program
{{
    static void F(string? s)
    {{
    }}
}}";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Enable));
            comp.VerifyDiagnostics(
                    // (6,25): error CS8669: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context. Auto-generated code requires an explicit '#nullable enable' directive in source.
                    //     static void F(string? s)
                    Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotationInGeneratedCode, "?").WithLocation(6, 25));
        }

        [Theory]
        [InlineData("#nullable enable")]
        [InlineData("#nullable enable annotations")]
        public void Directive_GloballyEnabled_GeneratedCode_Annotations_Enabled(string nullableDirective)
        {
            var source =
$@"
// <autogenerated />
{nullableDirective}
class Program
{{
    static void F(string? s)
    {{
    }}
}}";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Enable));
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Directive_GloballyEnabled_GeneratedCode_Enabled()
        {
            var source =
@"
// <autogenerated />
#nullable enable
class Program
{
    static void F(object x)
    {
        x = null; // warning
    }
}";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Enable));
            comp.VerifyDiagnostics(
                 // (8,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                 //         x = null; // warning
                 Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(8, 13)
                );
        }

        [Fact]
        public void Directive_GloballyEnabled_GeneratedCode_RestoreToProjectDefault()
        {
            var source =
@"
// <autogenerated />
partial class Program
{
#nullable restore
    static void F(object x)
    {
        x = null; // warning, restored to project default
    }
}";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Enable));
            comp.VerifyDiagnostics(
                 // (8,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                 //         x = null; // warning
                 Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(8, 13)
                );
        }

        [Fact]
        public void Directive_GloballyEnabled_GeneratedCode_WarningsAreDisabledByDefault()
        {
            var source =
@"
// <autogenerated />
partial class Program
{
    static void G(object x)
    {
        x = null; // no warning
        F = null; // no warning
    
#nullable enable warnings
        x = null; // no warning - declared out of nullable context
        F = null; // warning - declared in a nullable context
    }

#nullable enable
    static object F = new object();
}";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Enable));
            comp.VerifyDiagnostics(
                 // (12,13): warning CS8625: Cannot convert null literal to non-nullable reference type.
                 //         F = null; // warning - declared in a nullable context
                 Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(12, 13)
                );
        }

        [Fact]
        public void Directive_GloballyEnabled_GeneratedCode_PartialClasses()
        {
            var source1 =
@"
// <autogenerated />
partial class Program
{
    static void G(object x)
    {
        x = null; // no warning
        F = null; // no warning
    }
}";

            var source2 =
@"
partial class Program
{
    static object F = new object();

    static void H()
    {
        F = null; // warning
    }
}";
            var comp = CreateCompilation(new[] { source1, source2 }, options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Enable));
            comp.VerifyDiagnostics(
                // (8,13): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         F = null; // warning
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(8, 13)
                );
        }

        [Fact]
        public void Directive_GloballyEnabled_GeneratedCode_PartialClasses2()
        {
            var source1 =
@"
// <autogenerated />
partial class Program
{
#nullable enable warnings
    static void G(object x)
    {
        x = null; // no warning
        F = null; // warning
    }
}";

            var source2 =
@"
partial class Program
{
    static object F = new object();

    static void H()
    {
        F = null; // warning
    }
}";
            var comp = CreateCompilation(new[] { source1, source2 }, options: TestOptions.DebugDll.WithNullableContextOptions(NullableContextOptions.Enable));
            comp.VerifyDiagnostics(
                // (8,13): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         F = null; // warning
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(8, 13),
                // (9,13): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         F = null; // warning
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(9, 13)
                );
        }

        [Fact]
        [WorkItem(3705, "https://github.com/dotnet/roslyn/issues/3705")]
        public void GeneratedSyntaxTrees_Nullable()
        {
#pragma warning disable CS0618 // This test is intentionally calling the obsolete method to assert it's isGeneratedCode input is now ignored

            var source1 = CSharpSyntaxTree.ParseText(@"
partial class Program
{
    static void G(object x)
    {
        x = null; // no warning
        F = null; // no warning
    }
}", options: null, path: "", encoding: null, diagnosticOptions: null, isGeneratedCode: true, cancellationToken: default);

            Assert.False(((CSharpSyntaxTree)source1).IsGeneratedCode(null, cancellationToken: default));

            var source2 = CSharpSyntaxTree.ParseText(@"
partial class Program
{
    static object F = new object();

    static void H(object x)
    {
        x = null; // warning 1
        F = null; // warning 2
    }
}", options: null, path: "", encoding: null, diagnosticOptions: null, isGeneratedCode: false, cancellationToken: default);

            Assert.False(((CSharpSyntaxTree)source2).IsGeneratedCode(null, cancellationToken: default));

            var source3 = CSharpSyntaxTree.ParseText(
@"
partial class Program
{
    static void I(object x)
    {
        x = null; // warning 3
        F = null; // warning 4
    }
}", options: null, path: "", encoding: null, diagnosticOptions: null, isGeneratedCode: null /* use heuristic */, cancellationToken: default);

            Assert.False(((CSharpSyntaxTree)source3).IsGeneratedCode(null, cancellationToken: default));

            var source4 = SyntaxFactory.ParseSyntaxTree(@"
partial class Program
{
    static void J(object x)
    {
        x = null; // no warning
        F = null; // no warning
    }
}", options: null, path: "", encoding: null, diagnosticOptions: null, isGeneratedCode: true, cancellationToken: default);

            Assert.False(((CSharpSyntaxTree)source4).IsGeneratedCode(null, cancellationToken: default));

#pragma warning restore CS0618

            var syntaxOptions = new TestSyntaxTreeOptionsProvider(
                (source1, GeneratedKind.MarkedGenerated),
                (source2, GeneratedKind.NotGenerated),
                (source3, GeneratedKind.Unknown),
                (source4, GeneratedKind.MarkedGenerated)
            );

            var comp = CreateCompilation(new[] { source1, source2, source3, source4 },
                options: TestOptions.DebugDll
                    .WithNullableContextOptions(NullableContextOptions.Enable)
                    .WithSyntaxTreeOptionsProvider(syntaxOptions));
            comp.VerifyDiagnostics(
                // (6,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         x = null; // warning 3
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(6, 13),
                // (7,13): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         F = null; // warning 4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(7, 13),
                // (8,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         x = null; // warning 1
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(8, 13),
                // (9,13): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         F = null; // warning 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(9, 13)
                );
        }

        [WorkItem(30862, "https://github.com/dotnet/roslyn/issues/30862")]
        [Fact]
        public void DirectiveDisableWarningEnable()
        {
            var source =
@"#nullable enable
class Program
{
    static void F(object x)
    {
    }
#nullable disable
    static void F1(object? y, object? z)
    {
        F(y);
#pragma warning restore 8604
        F(z); // 1
    }
    static void F2(object? w)
    {
        F(w); // 2
    }
}";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (8,26): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     static void F1(object? y, object? z)
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(8, 26),
                // (8,37): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     static void F1(object? y, object? z)
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(8, 37),
                // (14,26): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     static void F2(object? w)
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(14, 26));
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void PragmaNullable_NoEffect()
        {
            var source = @"
#pragma warning disable nullable
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                    // (7,13): warning CS8602: Dereference of a possibly null reference.
                    //         _ = s.ToString();
                    Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(7, 13));
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Annotations_Enable_01()
        {
            var source = @"
#nullable enable annotations
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Annotations_Enable_02()
        {
            var source = @"
#nullable enable
#nullable disable warnings
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Annotations_Enable_03()
        {
            var source = @"
#nullable enable
#nullable restore warnings
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Annotations_Enable_04()
        {
            var source = @"
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source, options: WithNullable(NullableContextOptions.Annotations));
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Annotations_Enable_05()
        {
            var source = @"
#nullable enable
#nullable restore warnings
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source, options: WithNullableDisable());
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Annotations_Enable_06()
        {
            var source = @"
#nullable disable
#nullable restore warnings
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                    // (6,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                    //     static void M(string? s)
                    Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(6, 25),
                    // (8,13): warning CS8602: Dereference of a possibly null reference.
                    //         _ = s.ToString();
                    Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(8, 13));
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Annotations_NonFlowAnalysisWarning_01()
        {
            var source = @"
#nullable enable annotations
public partial class C
{
    partial void M(string? s);
}

public partial class C
{
    partial void M(string s) { }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Annotations_NonFlowAnalysisWarning_02()
        {
            var source = @"
using System.Collections.Generic;
#nullable enable annotations
class Base
{
    internal virtual void M(List<string> list) { }
}

class Derived : Base
{
    internal override void M(List<string?> list) { }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Annotations_NonFlowAnalysisWarning_03()
        {
            var source = @"
using System.Collections.Generic;
class Base
{
    internal virtual void M(List<string> list) { }
}

#nullable enable annotations
class Derived : Base
{
    internal override void M(List<string?> list) { }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Annotations_NonFlowAnalysisWarning_04()
        {
            var source = @"
#nullable enable annotations
public interface I
{
    void M(string? s);
}

public class C : I
{
    public void M(string s) => throw null!;
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Annotations_NonFlowAnalysisWarning_05()
        {
            var source = @"
#nullable enable annotations
public interface I
{
    string M();
}

public class C : I
{
    public string? M() => throw null!;
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Enable_NonFlowAnalysisWarning_01()
        {
            var source = @"
#nullable enable
public partial class C
{
    partial void M(string? s);
}

public partial class C
{
    partial void M(string s) { }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                    // (10,18): warning CS8611: Nullability of reference types in type of parameter 's' doesn't match partial method declaration.
                    //     partial void M(string s) { }
                    Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnPartial, "M").WithArguments("s").WithLocation(10, 18));
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Enable_NonFlowAnalysisWarning_02()
        {
            var source = @"
using System.Collections.Generic;
#nullable enable
class Base
{
    internal virtual void M(List<string> list) { }
}

class Derived : Base
{
    internal override void M(List<string?> list) { }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                    // (11,28): warning CS8610: Nullability of reference types in type of parameter 'list' doesn't match overridden member.
                    //     internal override void M(List<string?> list) { }
                    Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "M").WithArguments("list").WithLocation(11, 28));
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Enable_NonFlowAnalysisWarning_03()
        {
            var source = @"
using System.Collections.Generic;
class Base
{
    internal virtual void M(List<string> list) { }
}

#nullable enable
class Derived : Base
{
    // No warning because the base method's parameter type is oblivious
    internal override void M(List<string?> list) { }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Enable_NonFlowAnalysisWarning_04()
        {
            var source = @"
#nullable enable
public interface I
{
    void M(string? s);
}

public class C : I
{
    public void M(string s) => throw null!;
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (10,17): warning CS8767: Nullability of reference types in type of parameter 's' of 'void C.M(string s)' doesn't match implicitly implemented member 'void I.M(string? s)' (possibly because of nullability attributes).
                //     public void M(string s) => throw null!;
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "M").WithArguments("s", "void C.M(string s)", "void I.M(string? s)").WithLocation(10, 17)
                );
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Enable_NonFlowAnalysisWarning_05()
        {
            var source = @"
#nullable enable
public interface I
{
    string M();
}

public class C : I
{
    public string? M() => throw null!;
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (10,20): warning CS8766: Nullability of reference types in return type of 'string? C.M()' doesn't match implicitly implemented member 'string I.M()' (possibly because of nullability attributes).
                //     public string? M() => throw null!;
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "M").WithArguments("string? C.M()", "string I.M()").WithLocation(10, 20)
                );
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Warnings_Enable_01()
        {
            var source = @"
#nullable enable warnings
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                    // (5,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                    //     static void M(string? s)
                    Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 25),
                    // (7,13): warning CS8602: Dereference of a possibly null reference.
                    //         _ = s.ToString();
                    Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(7, 13));
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Warnings_Enable_02()
        {
            var source = @"
#nullable enable
#nullable disable annotations
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                    // (6,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                    //     static void M(string? s)
                    Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(6, 25),
                    // (8,13): warning CS8602: Dereference of a possibly null reference.
                    //         _ = s.ToString();
                    Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(8, 13));
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Warnings_Enable_03()
        {
            var source = @"
#nullable enable
#nullable restore annotations
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                    // (6,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                    //     static void M(string? s)
                    Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(6, 25),
                    // (8,13): warning CS8602: Dereference of a possibly null reference.
                    //         _ = s.ToString();
                    Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(8, 13));
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Warnings_Enable_04()
        {
            var source = @"
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source, options: WithNullable(NullableContextOptions.Warnings));
            comp.VerifyDiagnostics(
                    // (4,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                    //     static void M(string? s)
                    Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(4, 25),
                    // (6,13): warning CS8602: Dereference of a possibly null reference.
                    //         _ = s.ToString();
                    Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(6, 13));
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Warnings_Enable_05()
        {
            var source = @"
#nullable enable
#nullable restore annotations
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source, options: WithNullableDisable());
            comp.VerifyDiagnostics(
                    // (6,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                    //     static void M(string? s)
                    Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(6, 25),
                    // (8,13): warning CS8602: Dereference of a possibly null reference.
                    //         _ = s.ToString();
                    Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(8, 13));
        }

        [Fact, WorkItem(35748, "https://github.com/dotnet/roslyn/issues/35748")]
        public void Directive_Warnings_Enable_06()
        {
            var source = @"
#nullable disable
#nullable restore annotations
class Program
{
    static void M(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(35730, "https://github.com/dotnet/roslyn/issues/35730")]
        public void Directive_ProjectNoWarn()
        {
            var source = @"
#nullable enable
class Program
{
    void M1(string? s)
    {
        _ = s.ToString();
    }
}
";
            var comp1 = CreateCompilation(source);
            comp1.VerifyDiagnostics(
                    // (7,13): warning CS8602: Dereference of a possibly null reference.
                    //         _ = s.ToString();
                    Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(7, 13));

            var id = MessageProvider.Instance.GetIdForErrorCode((int)ErrorCode.WRN_NullReferenceReceiver);
            var comp2 = CreateCompilation(source, options: TestOptions.DebugDll.WithSpecificDiagnosticOptions(id, ReportDiagnostic.Suppress));
            comp2.VerifyDiagnostics();
        }

        [Fact, WorkItem(31740, "https://github.com/dotnet/roslyn/issues/31740")]
        public void Attribute_ArrayWithDifferentNullability()
        {
            var source =
@"
public class MyAttribute : System.Attribute
{
    public MyAttribute(string[] x) { }
}
#nullable enable
[My(new string[] { ""hello"" })]
class C { }
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(31740, "https://github.com/dotnet/roslyn/issues/31740")]
        public void Attribute_ArrayWithDifferentNullability2()
        {
            var source =
@"
public class MyAttribute : System.Attribute
{
    public MyAttribute(string[] x) { }
}
#nullable enable
[My(new string[] { null })]
class C { }
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics(
                // (7,20): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [My(new string[] { null })]
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(7, 20)
                );
        }

        [Fact, WorkItem(31740, "https://github.com/dotnet/roslyn/issues/31740")]
        public void Attribute_ArrayWithDifferentNullability_DynamicAndTupleNames()
        {
            var source = @"
public class MyAttribute : System.Attribute
{
    public MyAttribute((string alice, string)[] x, dynamic[] y, object[] z) { }
}
#nullable enable
[My(new (string, string bob)[] { }, new object[] { }, new dynamic[] { })]
class C { }
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics(
                // (7,2): error CS0181: Attribute constructor parameter 'x' has type '(string alice, string)[]', which is not a valid attribute parameter type
                // [My(new (string, string bob)[] { }, new object[] { }, new dynamic[] { })]
                Diagnostic(ErrorCode.ERR_BadAttributeParamType, "My").WithArguments("x", "(string alice, string)[]").WithLocation(7, 2),
                // (7,2): error CS0181: Attribute constructor parameter 'y' has type 'dynamic[]', which is not a valid attribute parameter type
                // [My(new (string, string bob)[] { }, new object[] { }, new dynamic[] { })]
                Diagnostic(ErrorCode.ERR_BadAttributeParamType, "My").WithArguments("y", "dynamic[]").WithLocation(7, 2)
                );
        }

        [Fact, WorkItem(31740, "https://github.com/dotnet/roslyn/issues/31740")]
        public void Attribute_ArrayWithDifferentNullability_Dynamic()
        {
            var source =
@"
public class MyAttribute : System.Attribute
{
    public MyAttribute(dynamic[] y) { }
}
#nullable enable
[My(new object[] { })]
class C { }
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics(
                // (7,2): error CS0181: Attribute constructor parameter 'y' has type 'dynamic[]', which is not a valid attribute parameter type
                // [My(new object[] { })]
                Diagnostic(ErrorCode.ERR_BadAttributeParamType, "My").WithArguments("y", "dynamic[]").WithLocation(7, 2)
                );
        }

        [Fact, WorkItem(36974, "https://github.com/dotnet/roslyn/issues/36974")]
        public void Attribute_Params()
        {
            var source = @"
using System;

[My(new object[] { new string[] { ""a"" } })]
public class C { }

#nullable enable
public class MyAttribute : Attribute
{
    public MyAttribute(params object[] o) { }
}
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics();
            var c = comp.GetTypeByMetadataName("C");
            var attribute = c.GetAttributes().Single();
            Assert.Equal(@"{{""a""}}", attribute.CommonConstructorArguments.Single().ToCSharpString());
        }

        [Fact, WorkItem(36974, "https://github.com/dotnet/roslyn/issues/36974")]
        public void Attribute_Params_Disabled()
        {
            var source = @"
using System;

[My(new object[] { new string[] { ""a"" } })]
public class C { }

#nullable disable
public class MyAttribute : Attribute
{
    public MyAttribute(params object[] o) { }
}
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics();
            var c = comp.GetTypeByMetadataName("C");
            var attribute = c.GetAttributes().Single();
            Assert.Equal(@"{{""a""}}", attribute.CommonConstructorArguments.Single().ToCSharpString());
        }

        [Fact, WorkItem(37155, "https://github.com/dotnet/roslyn/issues/37155")]
        public void Attribute_Params_DifferentNullability()
        {
            var source = @"
using System;
#nullable enable

[My(new object?[] { null })]
public class C { }

public class MyAttribute : Attribute
{
    public MyAttribute(params object[] o) { }
}
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics(
                // (5,5): warning CS8620: Argument of type 'object?[]' cannot be used for parameter 'o' of type 'object[]' in 'MyAttribute.MyAttribute(params object[] o)' due to differences in the nullability of reference types.
                // [My(new object?[] { null })]
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new object?[] { null }").WithArguments("object?[]", "object[]", "o", "MyAttribute.MyAttribute(params object[] o)").WithLocation(5, 5)
                );
        }

        [Fact, WorkItem(37155, "https://github.com/dotnet/roslyn/issues/37155")]
        public void Attribute_DifferentNullability()
        {
            var source = @"
using System;
#nullable enable

[My(new object?[] { null })]
public class C { }

public class MyAttribute : Attribute
{
    public MyAttribute(object[] o) { }
}
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics(
                // (5,5): warning CS8620: Argument of type 'object?[]' cannot be used for parameter 'o' of type 'object[]' in 'MyAttribute.MyAttribute(object[] o)' due to differences in the nullability of reference types.
                // [My(new object?[] { null })]
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new object?[] { null }").WithArguments("object?[]", "object[]", "o", "MyAttribute.MyAttribute(object[] o)").WithLocation(5, 5)
                );
        }

        [Fact, WorkItem(36974, "https://github.com/dotnet/roslyn/issues/36974")]
        public void Attribute_Params_TupleNames()
        {
            var source = @"
using System;

[My(new (int a, int b)[] { (1, 2) })]
public class C { }

public class MyAttribute : Attribute
{
    public MyAttribute(params (int c, int)[] o) { }
}
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics(
                // (4,2): error CS0181: Attribute constructor parameter 'o' has type '(int c, int)[]', which is not a valid attribute parameter type
                // [My(new (int a, int b)[] { (1, 2) })]
                Diagnostic(ErrorCode.ERR_BadAttributeParamType, "My").WithArguments("o", "(int c, int)[]").WithLocation(4, 2)
                );
        }

        [Fact, WorkItem(36974, "https://github.com/dotnet/roslyn/issues/36974")]
        public void Attribute_Params_Dynamic()
        {
            var source = @"
using System;

[My(new object[] { 1 })]
public class C { }

public class MyAttribute : Attribute
{
    public MyAttribute(params dynamic[] o) { }
}
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics(
                // (4,2): error CS0181: Attribute constructor parameter 'o' has type 'dynamic[]', which is not a valid attribute parameter type
                // [My(new object[] { 1 })]
                Diagnostic(ErrorCode.ERR_BadAttributeParamType, "My").WithArguments("o", "dynamic[]").WithLocation(4, 2)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_NullLiteral()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string s) { }
}

[MyAttribute(null)] //1
class C { }

[MyAttribute(""str"")]
class D { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (8,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null)] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(8, 14)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_NullLiteral_Suppressed()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string s) { }
}

[MyAttribute(null!)] 
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void AttributeArgument_Constructor_NullLiteral_CSharp7_3()
        {
            var source =
@"
class MyAttribute : System.Attribute
{
    public MyAttribute(string s) { }
}

[MyAttribute(null)] 
class C { }

[MyAttribute(""str"")]
class D { }
";
            var comp = CreateCompilation(source, parseOptions: TestOptions.Regular7_3);
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void AttributeArgument_Constructor_NullLiteral_WithDefaultArgument()
        {
            var source =
@"#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string s, string s2 = ""str"") { }
}

[MyAttribute(null)] //1
class C { }

[MyAttribute(null, null)] // 2, 3
class D { }

[MyAttribute(null, ""str"")] // 4
class E { }

";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (7,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null)] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(7, 14),
                // (10,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null, null)] // 2, 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 14),
                // (10,20): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null, null)] // 2, 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 20),
                // (13,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null, "str")] // 4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(13, 14)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_NullLiteral_WithNullDefaultArgument()
        {
            var source =
@"#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string s, string s2 = null) { } // 1
}

[MyAttribute(""str"")]
class C { }

[MyAttribute(""str"", null)] // 2
class D { }

[MyAttribute(""str"", ""str"")]
class E { }

";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (4,46): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //     public MyAttribute(string s, string s2 = null) { } // 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(4, 46),
                // (10,21): warning CS8625: Cannot convert null literal to non-nullable reference type.
                // [MyAttribute("str", null)] // 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 21)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_NullLiteral_WithNamedArguments()
        {
            var source =
@"#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string s, string s2 = ""str"", string? s3 = ""str"") { }
}

[MyAttribute(""str"", s2: null, s3: null)] //1
class C { }

[MyAttribute(s3: null, s2: null, s: ""str"")] // 2
class D { }

[MyAttribute(s3: null, s2: ""str"", s: ""str"")]
class E { }

";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (7,25): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute("str", s2: null, s3: null)] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(7, 25),
                // (10,28): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(s3: null, s2: null, s: "str")] // 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 28)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_NullLiteral_DisabledEnabled()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string s, string s2) { }
}

[MyAttribute(null, //1
#nullable disable
null
#nullable enable
)] 
class C { }

[MyAttribute(
#nullable disable
null, 
#nullable enable
null //2
)] 
class D { }

[MyAttribute(null, //3
s2: 
#nullable disable
null
#nullable enable
)] 
class E { }

[MyAttribute(
#nullable disable
null,
s2: 
#nullable enable
null //4
)] 
class F { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (8,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null, //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(8, 14),
                // (19,1): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // null //2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(19, 1),
                // (23,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null, //3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(23, 14),
                // (36,1): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // null //4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(36, 1)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_NullLiteral_WarningDisabledEnabled()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string s, string s2) { }
}
#nullable disable

#nullable enable warnings
[MyAttribute(null, //1
#nullable disable warnings
null
#nullable enable warnings
)] 
class C { }

[MyAttribute(
#nullable disable warnings
null, 
#nullable enable warnings
null //2
)] 
class D { }

[MyAttribute(null, //3
s2: 
#nullable disable warnings
null
#nullable enable warnings
)] 
class E { }

[MyAttribute(
#nullable disable warnings
null,
s2: 
#nullable enable warnings
null //4
)] 
class F { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (10,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null, //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 14),
                // (21,1): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // null //2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(21, 1),
                // (25,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null, //3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(25, 14),
                // (38,1): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // null //4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(38, 1)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_Array_LiteralNull()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string[] s) { }
}

[MyAttribute(null)] //1
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (8,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null)] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(8, 14)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_Array_LiteralNull_Suppressed()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string[] s) { }
}

[MyAttribute(null!)]
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void AttributeArgument_Constructor_Array_ArrayOfNullable()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string[] s) { }
}

[MyAttribute(new string?[]{ null })] //1
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (8,14): warning CS8620: Argument of type 'string?[]' cannot be used as an input of type 'string[]' for parameter 's' in 'MyAttribute.MyAttribute(string[] s)' due to differences in the nullability of reference types.
                // [MyAttribute(new string?[]{ null })] //1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new string?[]{ null }").WithArguments("string?[]", "string[]", "s", "MyAttribute.MyAttribute(string[] s)").WithLocation(8, 14)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_Array_ArrayOfNullable_Suppressed()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string[] s) { }
}

[MyAttribute(new string?[]{ null }!)]
class C { }

[MyAttribute(new string[]{ null! })]
class D { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void AttributeArgument_Constructor_Array_ArrayOfNullable_ImplicitType()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string[] s) { }
}

[MyAttribute(new []{ ""str"", null })] //1
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (8,14): warning CS8620: Argument of type 'string?[]' cannot be used as an input of type 'string[]' for parameter 's' in 'MyAttribute.MyAttribute(string[] s)' due to differences in the nullability of reference types.
                // [MyAttribute(new []{ "str", null })] //1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, @"new []{ ""str"", null }").WithArguments("string?[]", "string[]", "s", "MyAttribute.MyAttribute(string[] s)").WithLocation(8, 14)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_Array_NullValueInInitializer()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string[] s) { }
}

[MyAttribute(new string[]{ ""str"", null, ""str"" })] //1
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (8,35): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(new string[]{ "str", null, "str" })] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(8, 35)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_Array_NullValueInNestedInitializer()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(object[] s) { }
}

[MyAttribute(new object[]
{ 
    new string[] { ""str"", null }, //1
    new string[] { null }, //2
    new string?[] { null }
})]
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (10,27): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                //     new string[] { "str", null }, //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 27),
                // (11,20): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                //     new string[] { null }, //2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(11, 20)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_ParamsArrayOfNullable_NullLiteral()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(params object?[] s) { }
}

[MyAttribute(null)] //1
class C { }

[MyAttribute(null, null)] 
class D { }

[MyAttribute((object?)null)] 
class E { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (8,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null)] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(8, 14)
                );
        }

        [Fact]
        public void AttributeArgument_Constructor_ParamsArray_NullItem()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute(string s1, params object[] s) { }
}

[MyAttribute(""str"", null, ""str"", ""str"")] //1
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (8,21): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute("str", null, "str", "str")] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(8, 21)
                );
        }

        [Fact]
        public void AttributeArgument_PropertyAssignment_NullLiteral()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute() { }
    
    public string MyValue { get; set; } = ""str"";
}

[MyAttribute(MyValue = null)] //1
class C { }

[MyAttribute(MyValue = ""str"")]
class D { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (10,24): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(MyValue = null)] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 24)
                );
        }

        [Fact]
        public void AttributeArgument_PropertyAssignment_Array_NullLiteral()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute() { }
    
    public string[] PropertyArray { get; set; } = new string[] { };

    public string[]? NullablePropertyArray { get; set; } = null;

}

[MyAttribute(PropertyArray = null)] //1
class C { }

[MyAttribute(NullablePropertyArray = null)] 
class D { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (13,30): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(PropertyArray = null)] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(13, 30)
                );
        }

        [Fact]
        public void AttributeArgument_PropertyAssignment_Array_ArrayOfNullable()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute() { }

    public string[] PropertyArray { get; set; } = new string[] { ""str"" };

    public string[]? PropertyNullableArray { get; set; } = new string[] { ""str"" };
}

[MyAttribute(PropertyArray = new string?[]{ null })] //1
class C { }

[MyAttribute(PropertyNullableArray = null)]
class D { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (12,30): warning CS8619: Nullability of reference types in value of type 'string?[]' doesn't match target type 'string[]'.
                // [MyAttribute(PropertyArray = new string?[]{ null })] //1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "new string?[]{ null }").WithArguments("string?[]", "string[]").WithLocation(12, 30)
                );
        }

        [Fact]
        public void AttributeArgument_FieldAssignment_NullLiteral()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute() { }

    public string myValue = ""str"";
}

[MyAttribute(myValue = null)] //1
class C { }

[MyAttribute(myValue = ""str"")]
class D { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (10,24): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(myValue = null)] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 24)
                );
        }

        [Fact]
        public void AttributeArgument_FieldAssignment_Array_NullLiteral()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute() { }

    public string[] fieldArray = new string[] { };

    public string[]? nullableFieldArray = null;
}

[MyAttribute(fieldArray = null)] //1
class C { }

[MyAttribute(nullableFieldArray = null)]
class D { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (12,27): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(fieldArray = null)] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(12, 27)
                );
        }

        [Fact]
        public void AttributeArgument_FieldAssignment_Array_ArrayOfNullable()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public MyAttribute() { }

    public string[] fieldArray = new string[] { };

    public string?[] fieldArrayOfNullable = new string?[] { };
}

[MyAttribute(fieldArray = new string?[]{ null })] //1
class C { }

[MyAttribute(fieldArrayOfNullable = new string?[]{ null })] 
class D { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (12,27): warning CS8619: Nullability of reference types in value of type 'string?[]' doesn't match target type 'string[]'.
                // [MyAttribute(fieldArray = new string?[]{ null })] //1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "new string?[]{ null }").WithArguments("string?[]", "string[]").WithLocation(12, 27)
                );
        }

        [Fact]
        public void AttributeArgument_NoMatchingConstructor_NullLiteral()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute 
{
}

[MyAttribute(null)] //1
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (7,2): error CS1729: 'MyAttribute' does not contain a constructor that takes 1 arguments
                // [MyAttribute(null)] //1
                Diagnostic(ErrorCode.ERR_BadCtorArgCount, "MyAttribute(null)").WithArguments("MyAttribute", "1").WithLocation(7, 2)
                );
        }

        [Fact]
        public void AttributeArgument_NoMatchingConstructor_Array_NullValueInInitializer()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
}

[MyAttribute(new string[] { null })] //1
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (7,2): error CS1729: 'MyAttribute' does not contain a constructor that takes 1 arguments
                // [MyAttribute(new string[] { null })] //1
                Diagnostic(ErrorCode.ERR_BadCtorArgCount, "MyAttribute(new string[] { null })").WithArguments("MyAttribute", "1").WithLocation(7, 2),
                // (7,29): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(new string[] { null })] //1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(7, 29)
                );
        }

        [Fact]
        public void AttributeArgument_NoMatchingConstructor_PropertyAssignment_NullLiteral()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public string[] PropertyArray { get; set; } = new string[] { ""str"" };

    public string[]? PropertyNullableArray { get; set; } = new string[] { ""str"" };
}

[MyAttribute(  // 1
    new string[] { null }, // 2
    PropertyArray = null, // 3
    PropertyNullableArray = new string[] { null } // 4
)]
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (10,2): error CS1729: 'MyAttribute' does not contain a constructor that takes 1 arguments
                // [MyAttribute(  // 1
                Diagnostic(ErrorCode.ERR_BadCtorArgCount, @"MyAttribute(  // 1
    new string[] { null }, // 2
    PropertyArray = null, // 3
    PropertyNullableArray = new string[] { null } // 4
)").WithArguments("MyAttribute", "1").WithLocation(10, 2),
                // (11,20): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                //     new string[] { null }, // 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(11, 20),
                // (12,21): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                //     PropertyArray = null, // 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(12, 21),
                // (13,44): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                //     PropertyNullableArray = new string[] { null } // 4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(13, 44)
                );
        }

        [Fact]
        public void AttributeArgument_NoMatchingConstructor_FieldAssignment_NullLiteral()
        {
            var source =
@"
#nullable enable
class MyAttribute : System.Attribute
{
    public string[] fieldArray = new string[] { };

    public string?[] fieldArrayOfNullable = new string?[] { };
}

[MyAttribute( // 1
    new string[] { null }, // 2
    fieldArray = null, // 3
    fieldArrayOfNullable = new string[] { null } // 4
)] 
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (10,2): error CS1729: 'MyAttribute' does not contain a constructor that takes 1 arguments
                // [MyAttribute( // 1
                Diagnostic(ErrorCode.ERR_BadCtorArgCount, @"MyAttribute( // 1
    new string[] { null }, // 2
    fieldArray = null, // 3
    fieldArrayOfNullable = new string[] { null } // 4
)").WithArguments("MyAttribute", "1").WithLocation(10, 2),
                // (11,20): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                //     new string[] { null }, // 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(11, 20),
                // (12,18): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                //     fieldArray = null, // 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(12, 18),
                // (13,43): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                //     fieldArrayOfNullable = new string[] { null } // 4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(13, 43)
                );
        }

        [Fact]
        public void AttributeArgument_ComplexAssignment()
        {
            var source =
@"
#nullable enable
[System.AttributeUsage(System.AttributeTargets.All, AllowMultiple = true)]
class MyAttribute : System.Attribute
{
    public MyAttribute(string s, string s2 = ""str"", string s3 = ""str"") { }

    public string[] fieldArray = new string[] { };

    public string?[] fieldArrayOfNullable = new string[] { };

    public string[]? nullableFieldArray = null;

    public string[] PropertyArray { get; set; } = new string[] { };

    public string?[] PropertyArrayOfNullable { get; set; } = new string[] { };

    public string[]? NullablePropertyArray { get; set; } = null;

}

[MyAttribute(""s1"")]
[MyAttribute(""s1"", s3: ""s3"", fieldArray = new string[]{})]
[MyAttribute(""s1"", s2: ""s2"", fieldArray = new string[]{}, PropertyArray = new string[]{})]
[MyAttribute(""s1"", fieldArrayOfNullable = new string?[]{ null }, NullablePropertyArray = null)]
[MyAttribute(null)] // 1
[MyAttribute(""s1"", s3: null, fieldArray = new string[]{})] // 2
[MyAttribute(""s1"", s2: ""s2"", fieldArray = new string?[]{ null }, PropertyArray = new string[]{})] // 3
[MyAttribute(""s1"", PropertyArrayOfNullable = null)] // 4
[MyAttribute(""s1"", NullablePropertyArray = new string?[]{ null })] // 5
[MyAttribute(""s1"", fieldArrayOfNullable = null)] // 6
[MyAttribute(""s1"", nullableFieldArray = new string[]{ null })] // 7
[MyAttribute(null, //8
            s2: null, //9
            fieldArrayOfNullable = null, //10
            NullablePropertyArray = new string?[]{ null })] // 11
[MyAttribute(null, // 12
#nullable disable
            s2: null,
#nullable enable
            fieldArrayOfNullable = null, //13
#nullable disable warnings
            NullablePropertyArray = new string?[]{ null },
#nullable enable warnings
            nullableFieldArray = new string?[]{ null })] //14
class C { }
";
            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (26,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null)] // 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(26, 14),
                // (27,24): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute("s1", s3: null, fieldArray = new string[]{})] // 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(27, 24),
                // (28,43): warning CS8619: Nullability of reference types in value of type 'string?[]' doesn't match target type 'string[]'.
                // [MyAttribute("s1", s2: "s2", fieldArray = new string?[]{ null }, PropertyArray = new string[]{})] // 3
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "new string?[]{ null }").WithArguments("string?[]", "string[]").WithLocation(28, 43),
                // (29,46): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute("s1", PropertyArrayOfNullable = null)] // 4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(29, 46),
                // (30,44): warning CS8619: Nullability of reference types in value of type 'string?[]' doesn't match target type 'string[]'.
                // [MyAttribute("s1", NullablePropertyArray = new string?[]{ null })] // 5
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "new string?[]{ null }").WithArguments("string?[]", "string[]").WithLocation(30, 44),
                // (31,43): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute("s1", fieldArrayOfNullable = null)] // 6
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(31, 43),
                // (32,55): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute("s1", nullableFieldArray = new string[]{ null })] // 7
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(32, 55),
                // (33,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null, //8
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(33, 14),
                // (34,17): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                //             s2: null, //9
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(34, 17),
                // (35,36): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                //             fieldArrayOfNullable = null, //10
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(35, 36),
                // (36,37): warning CS8619: Nullability of reference types in value of type 'string?[]' doesn't match target type 'string[]'.
                //             NullablePropertyArray = new string?[]{ null })] // 11
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "new string?[]{ null }").WithArguments("string?[]", "string[]").WithLocation(36, 37),
                // (37,14): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                // [MyAttribute(null, // 12
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(37, 14),
                // (41,36): warning CS8625: Cannot convert null literal to non-nullable reference or unconstrained type parameter.
                //             fieldArrayOfNullable = null, //13
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(41, 36),
                // (45,34): warning CS8619: Nullability of reference types in value of type 'string?[]' doesn't match target type 'string[]'.
                //             nullableFieldArray = new string?[]{ null })] //14
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "new string?[]{ null }").WithArguments("string?[]", "string[]").WithLocation(45, 34)
                );
        }

        [Fact, WorkItem(40136, "https://github.com/dotnet/roslyn/issues/40136")]
        public void SelfReferencingAttribute()
        {
            var source = @"
using System;

[AttributeUsage(AttributeTargets.All)]
[ExplicitCrossPackageInternal(ExplicitCrossPackageInternalAttribute.s)]
internal sealed class ExplicitCrossPackageInternalAttribute : Attribute
{
    internal const string s = """";

    [ExplicitCrossPackageInternal(s)]
    internal ExplicitCrossPackageInternalAttribute([ExplicitCrossPackageInternal(s)] string prop)
    {
    }

    [return: ExplicitCrossPackageInternal(s)]
    [ExplicitCrossPackageInternal(s)]
    internal void Method() { }
}";
            var comp = CreateNullableCompilation(source);
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void NullableAndConditionalOperators()
        {
            var source =
@"class Program
{
    static void F1(object x)
    {
        _ = x is string? 1 : 2;
        _ = x is string? ? 1 : 2;  // error 1: is a nullable reference type
        _ = x is string ? ? 1 : 2; // error 2: is a nullable reference type
        _ = x as string?? x;
        _ = x as string ? ?? x;    // error 3: as a nullable reference type
    }
    static void F2(object y)
    {
        _ = y is object[]? 1 : 2;
        _ = y is object[]? ? 1 : 2;   // error 4
        _ = y is object[] ? ? 1 : 2;  // error 5
        _ = y as object[]?? y;
        _ = y as object[] ? ?? y;     // error 6
    }
    static void F3<T>(object z)
    {
        _ = z is T[][]? 1 : 2;
        _ = z is T[]?[] ? 1 : 2;
        _ = z is T[] ? [] ? 1 : 2;
        _ = z as T[][]?? z;
        _ = z as T[] ? [] ?? z;
    }
}";

            var comp = CreateCompilation(source, parseOptions: TestOptions.Regular7);
            comp.VerifyDiagnostics(
                // (6,18): error CS8650: It is not legal to use nullable reference type 'string?' in an is-type expression; use the underlying type 'string' instead.
                //         _ = x is string? ? 1 : 2;  // error 1: is a nullable reference type
                Diagnostic(ErrorCode.ERR_IsNullableType, "string?").WithArguments("string").WithLocation(6, 18),
                // (6,24): error CS8652: The feature 'nullable reference types' is not available in C# 7.0. Please use language version 8.0 or greater.
                //         _ = x is string? ? 1 : 2;  // error 1: is a nullable reference type
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion7, "?").WithArguments("nullable reference types", "8.0").WithLocation(6, 24),
                // (7,18): error CS8650: It is not legal to use nullable reference type 'string?' in an is-type expression; use the underlying type 'string' instead.
                //         _ = x is string ? ? 1 : 2; // error 2: is a nullable reference type
                Diagnostic(ErrorCode.ERR_IsNullableType, "string ?").WithArguments("string").WithLocation(7, 18),
                // (7,25): error CS8652: The feature 'nullable reference types' is not available in C# 7.0. Please use language version 8.0 or greater.
                //         _ = x is string ? ? 1 : 2; // error 2: is a nullable reference type
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion7, "?").WithArguments("nullable reference types", "8.0").WithLocation(7, 25),
                // (9,18): error CS8651: It is not legal to use nullable reference type 'string?' in an as expression; use the underlying type 'string' instead.
                //         _ = x as string ? ?? x;    // error 3: as a nullable reference type
                Diagnostic(ErrorCode.ERR_AsNullableType, "string ?").WithArguments("string").WithLocation(9, 18),
                // (9,25): error CS8652: The feature 'nullable reference types' is not available in C# 7.0. Please use language version 8.0 or greater.
                //         _ = x as string ? ?? x;    // error 3: as a nullable reference type
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion7, "?").WithArguments("nullable reference types", "8.0").WithLocation(9, 25),
                // (14,18): error CS8650: It is not legal to use nullable reference type 'object[]?' in an is-type expression; use the underlying type 'object[]' instead.
                //         _ = y is object[]? ? 1 : 2;   // error 4
                Diagnostic(ErrorCode.ERR_IsNullableType, "object[]?").WithArguments("object[]").WithLocation(14, 18),
                // (14,26): error CS8652: The feature 'nullable reference types' is not available in C# 7.0. Please use language version 8.0 or greater.
                //         _ = y is object[]? ? 1 : 2;   // error 4
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion7, "?").WithArguments("nullable reference types", "8.0").WithLocation(14, 26),
                // (15,18): error CS8650: It is not legal to use nullable reference type 'object[]?' in an is-type expression; use the underlying type 'object[]' instead.
                //         _ = y is object[] ? ? 1 : 2;  // error 5
                Diagnostic(ErrorCode.ERR_IsNullableType, "object[] ?").WithArguments("object[]").WithLocation(15, 18),
                // (15,27): error CS8652: The feature 'nullable reference types' is not available in C# 7.0. Please use language version 8.0 or greater.
                //         _ = y is object[] ? ? 1 : 2;  // error 5
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion7, "?").WithArguments("nullable reference types", "8.0").WithLocation(15, 27),
                // (17,18): error CS8651: It is not legal to use nullable reference type 'object[]?' in an as expression; use the underlying type 'object[]' instead.
                //         _ = y as object[] ? ?? y;     // error 6
                Diagnostic(ErrorCode.ERR_AsNullableType, "object[] ?").WithArguments("object[]").WithLocation(17, 18),
                // (17,27): error CS8652: The feature 'nullable reference types' is not available in C# 7.0. Please use language version 8.0 or greater.
                //         _ = y as object[] ? ?? y;     // error 6
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion7, "?").WithArguments("nullable reference types", "8.0").WithLocation(17, 27),
                // (22,21): error CS8652: The feature 'nullable reference types' is not available in C# 7.0. Please use language version 8.0 or greater.
                //         _ = z is T[]?[] ? 1 : 2;
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion7, "?").WithArguments("nullable reference types", "8.0").WithLocation(22, 21),
                // (23,22): error CS8652: The feature 'nullable reference types' is not available in C# 7.0. Please use language version 8.0 or greater.
                //         _ = z is T[] ? [] ? 1 : 2;
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion7, "?").WithArguments("nullable reference types", "8.0").WithLocation(23, 22),
                // (25,22): error CS8652: The feature 'nullable reference types' is not available in C# 7.0. Please use language version 8.0 or greater.
                //         _ = z as T[] ? [] ?? z;
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion7, "?").WithArguments("nullable reference types", "8.0").WithLocation(25, 22)
                );

            comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,18): error CS8650: It is not legal to use nullable reference type 'string?' in an is-type expression; use the underlying type 'string' instead.
                //         _ = x is string? ? 1 : 2;  // error 1: is a nullable reference type
                Diagnostic(ErrorCode.ERR_IsNullableType, "string?").WithArguments("string").WithLocation(6, 18),
                // (7,18): error CS8650: It is not legal to use nullable reference type 'string?' in an is-type expression; use the underlying type 'string' instead.
                //         _ = x is string ? ? 1 : 2; // error 2: is a nullable reference type
                Diagnostic(ErrorCode.ERR_IsNullableType, "string ?").WithArguments("string").WithLocation(7, 18),
                // (9,18): error CS8651: It is not legal to use nullable reference type 'string?' in an as expression; use the underlying type 'string' instead.
                //         _ = x as string ? ?? x;    // error 3: as a nullable reference type
                Diagnostic(ErrorCode.ERR_AsNullableType, "string ?").WithArguments("string").WithLocation(9, 18),
                // (14,18): error CS8650: It is not legal to use nullable reference type 'object[]?' in an is-type expression; use the underlying type 'object[]' instead.
                //         _ = y is object[]? ? 1 : 2;   // error 4
                Diagnostic(ErrorCode.ERR_IsNullableType, "object[]?").WithArguments("object[]").WithLocation(14, 18),
                // (15,18): error CS8650: It is not legal to use nullable reference type 'object[]?' in an is-type expression; use the underlying type 'object[]' instead.
                //         _ = y is object[] ? ? 1 : 2;  // error 5
                Diagnostic(ErrorCode.ERR_IsNullableType, "object[] ?").WithArguments("object[]").WithLocation(15, 18),
                // (17,18): error CS8651: It is not legal to use nullable reference type 'object[]?' in an as expression; use the underlying type 'object[]' instead.
                //         _ = y as object[] ? ?? y;     // error 6
                Diagnostic(ErrorCode.ERR_AsNullableType, "object[] ?").WithArguments("object[]").WithLocation(17, 18)
                );
        }

        [Fact, WorkItem(29318, "https://github.com/dotnet/roslyn/issues/29318")]
        public void IsOperatorOnNonNullExpression()
        {
            var source = @"
class C
{
    void M(object o)
    {
        if (o is string)
        {
            o.ToString();
        }
        else
        {
            o.ToString();
        }
    }
}
";

            var c = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);
            c.VerifyDiagnostics();
        }

        [Fact, WorkItem(29318, "https://github.com/dotnet/roslyn/issues/29318")]
        public void IsOperatorOnMaybeNullExpression()
        {
            var source = @"
class C
{
    static void Main(object? o)
    {
        if (o is string)
        {
            o.ToString();
        }
        else
        {
            o.ToString(); // 1
        }
    }
}
";

            var c = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);
            c.VerifyDiagnostics(
                // (12,13): warning CS8602: Dereference of a possibly null reference.
                //             o.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "o").WithLocation(12, 13)
                );
        }

        [Fact, WorkItem(29318, "https://github.com/dotnet/roslyn/issues/29318")]
        public void IsOperatorOnUnconstrainedType()
        {
            var source = @"
class C
{
    static void M<T>(T t)
    {
        if (t is string)
        {
            t.ToString();
        }
        else
        {
            t.ToString(); // 1
        }
    }
}
";

            var c = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);
            c.VerifyDiagnostics(
                // (12,13): warning CS8602: Dereference of a possibly null reference.
                //             t.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "t").WithLocation(12, 13)
                );
        }

        [Fact]
        public void IsOperator_AffectsNullConditionalOperator()
        {
            var source = @"
class C
{
    public object? field = null;
    static void M(C? c)
    {
        if (c?.field is string)
        {
            c.ToString();
            c.field.ToString();
        }
        else
        {
            c.ToString(); // 1
        }
    }
}
";

            var c = CreateCompilation(new[] { source }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (14,13): warning CS8602: Dereference of a possibly null reference.
                //             c.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c").WithLocation(14, 13)
                );
        }

        [Fact]
        public void OmittedCall()
        {
            var source = @"
partial class C
{
    void M(string? x)
    {
        OmittedMethod(x);
    }
    partial void OmittedMethod(string x);
}
";

            var c = CreateCompilation(new[] { source }, parseOptions: TestOptions.Regular8, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (6,23): warning CS8604: Possible null reference argument for parameter 'x' in 'void C.OmittedMethod(string x)'.
                //         OmittedMethod(x);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x").WithArguments("x", "void C.OmittedMethod(string x)").WithLocation(6, 23)
                );
        }

        [Fact]
        public void OmittedInitializerCall()
        {
            var source = @"
using System.Collections;
partial class Collection : IEnumerable
{
    void M(string? x)
    {
        _ = new Collection() { x };
    }
    IEnumerator IEnumerable.GetEnumerator() => throw null!;
    partial void Add(string x);
}
";

            var c = CreateCompilation(new[] { source }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (7,32): warning CS8604: Possible null reference argument for parameter 'x' in 'void Collection.Add(string x)'.
                //         _ = new Collection() { x };
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x").WithArguments("x", "void Collection.Add(string x)").WithLocation(7, 32)
                );
        }

        [Fact]
        public void UpdateArrayRankSpecifier()
        {
            var source = @"
class C
{
    static void Main()
    {
        object[]? x = null;
    }
}
";
            var tree = Parse(source);
            var specifier = tree.GetRoot().DescendantNodes().OfType<ArrayRankSpecifierSyntax>().Single();
            Assert.Equal("[]", specifier.ToString());

            var newSpecifier = specifier.Update(
                specifier.OpenBracketToken,
                SyntaxFactory.SeparatedList<ExpressionSyntax>(
                    new[] { SyntaxFactory.LiteralExpression(SyntaxKind.NumericLiteralExpression, SyntaxFactory.Literal(3)) }),
                specifier.CloseBracketToken);
            Assert.Equal("[3]", newSpecifier.ToString());
        }

        [Fact]
        public void TestUnaryNegation()
        {
            // This test verifies that we no longer crash hitting an assertion
            var source = @"
public class C<T>
{
    C(C<object> c) => throw null!;
    void M(bool b)
    {
        _ = new C<object>(!b);
    }
    public static implicit operator C<T>(T x) => throw null!;
}
";

            var c = CreateCompilation(source, parseOptions: TestOptions.Regular8);
            c.VerifyDiagnostics();
        }

        [Fact]
        public void UnconstrainedAndErrorNullableFields()
        {
            var source = @"
public class C<T>
{
    public T? field;
    public Unknown? field2;
}
";

            var c = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);
            c.VerifyDiagnostics(
                // (5,12): error CS0246: The type or namespace name 'Unknown' could not be found (are you missing a using directive or an assembly reference?)
                //     public Unknown? field2;
                Diagnostic(ErrorCode.ERR_SingleTypeNameNotFound, "Unknown").WithArguments("Unknown").WithLocation(5, 12),
                // (4,12): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     public T? field;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(4, 12));
        }

        [Fact]
        public void NoNullableAnalysisWithoutNonNullTypes()
        {
            var source = @"
class C
{
    void M(string z)
    {
        z = null;
        z.ToString();
    }
}
#nullable enable
class C2
{
    void M(string z)
    {
        z = null; // 1
        z.ToString(); // 2
    }
}
";
            var expected = new[]
            {
                // (15,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         z = null; // 1
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(15, 13),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         z.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "z").WithLocation(16, 9)
            };

            var c = CreateCompilation(new[] { source });
            c.VerifyDiagnostics(expected);

            c = CreateCompilation(source, parseOptions: TestOptions.Regular8);
            c.VerifyDiagnostics(expected);

            expected = new[]
            {
                // (10,2): error CS8652: The feature 'nullable reference types' is not available in C# 7.3. Please use language version 8.0 or greater.
                // #nullable enable
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion7_3, "nullable").WithArguments("nullable reference types", "8.0").WithLocation(10, 2)
            };

            var c2 = CreateCompilation(new[] { source }, parseOptions: TestOptions.Regular7_3, skipUsesIsNullable: true);
            c2.VerifyDiagnostics(expected);
        }

        [Fact]
        public void NonNullTypesOnPartialSymbol()
        {
            var source = @"
#nullable enable
partial class C
{
#nullable disable
    partial void M();
}
#nullable enable
partial class C
{
#nullable disable
    partial void M() { }
}
";
            var c = CreateCompilation(new[] { source });
            c.VerifyDiagnostics(
                );
        }

        [Fact]
        public void SuppressionAsLValue()
        {
            var source = @"
class C
{
    void M(string? x)
    {
        ref string y = ref x;
        ref string y2 = ref x;
        (y2! = ref y) = ref y;
    }
}
";
            var c = CreateCompilation(new[] { source }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (6,28): warning CS8619: Nullability of reference types in value of type 'string?' doesn't match target type 'string'.
                //         ref string y = ref x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("string?", "string").WithLocation(6, 28),
                // (7,29): warning CS8619: Nullability of reference types in value of type 'string?' doesn't match target type 'string'.
                //         ref string y2 = ref x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("string?", "string").WithLocation(7, 29),
                // (8,10): error CS8598: The suppression operator is not allowed in this context
                //         (y2! = ref y) = ref y;
                Diagnostic(ErrorCode.ERR_IllegalSuppression, "y2").WithLocation(8, 10),
                // (8,20): warning CS8601: Possible null reference assignment.
                //         (y2! = ref y) = ref y;
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "y").WithLocation(8, 20),
                // (8,29): warning CS8601: Possible null reference assignment.
                //         (y2! = ref y) = ref y;
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "y").WithLocation(8, 29)
                );
        }

        [Fact]
        [WorkItem(31732, "https://github.com/dotnet/roslyn/issues/31732")]
        public void SuppressionOnUnconstrainedTypeParameter()
        {
            var source = @"
class C
{
    void M<T>(T t)
    {
        t!.ToString();
        t.ToString();
    }
}
";
            var c = CreateCompilation(new[] { source });
            c.VerifyDiagnostics();

            c = CreateCompilation(new[] { source }, options: WithNullableEnable());
            c.VerifyDiagnostics();
        }

        [Fact]
        [WorkItem(31732, "https://github.com/dotnet/roslyn/issues/31732")]
        public void SuppressionOnNullableValueType()
        {
            var source = @"
class C
{
    void M(int? i)
    {
        i!.Value.ToString();
        i.Value.ToString();
    }
}
";
            var c = CreateCompilation(new[] { source });
            c.VerifyDiagnostics();

            c = CreateCompilation(new[] { source }, options: WithNullableEnable());
            c.VerifyDiagnostics();
        }

        [Fact]
        [WorkItem(31732, "https://github.com/dotnet/roslyn/issues/31732")]
        public void SuppressionOnNullableValueType_AppliedOnField()
        {
            var source = @"
public struct S
{
    public string? field;
}
class C
{
    void M(S? s)
    {
        s.Value.field!.ToString();
    }
}
";
            var c = CreateCompilation(new[] { source }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (10,9): warning CS8629: Nullable value type may be null.
                //         s.Value.field!.ToString();
                Diagnostic(ErrorCode.WRN_NullableValueTypeMayBeNull, "s").WithLocation(10, 9)
                );
        }

        [Fact]
        [WorkItem(31732, "https://github.com/dotnet/roslyn/issues/31732")]
        public void SuppressionOnNullableReferenceType_AppliedOnField()
        {
            var source = @"
public class C
{
    public string? field;
    void M(C? c)
    {
        c.field!.ToString();
    }
}
";
            var c = CreateCompilation(new[] { source }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (7,9): warning CS8602: Dereference of a possibly null reference.
                //         c.field!.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c").WithLocation(7, 9)
                );
        }

        [Fact]
        [WorkItem(31732, "https://github.com/dotnet/roslyn/issues/31732")]
        public void SuppressionOnNullableReferenceType_AppliedOnField2()
        {
            var source = @"
public class C
{
    public string? field;
    void M1(C? c)
    {
        c?.field!.ToString();
        c.ToString(); // 1
    }

    void M2(C? c)
    {
        c!?.field!.ToString();
        c.ToString(); // 2
    }

    void M3(C? c)
    {
        _ = c?.field!.ToString()!;
        c.ToString(); // 3
    }

    void M4(C? c)
    {
        (c?.field!.ToString()!).ToString();
        c.ToString(); // no warning because 'c' was part of a call receiver in previous line
    }
}
";
            var c = CreateCompilation(new[] { source }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (8,9): warning CS8602: Dereference of a possibly null reference.
                //         c.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c", isSuppressed: false).WithLocation(8, 9),
                // (14,9): warning CS8602: Dereference of a possibly null reference.
                //         c.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c", isSuppressed: false).WithLocation(14, 9),
                // (20,9): warning CS8602: Dereference of a possibly null reference.
                //         c.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c", isSuppressed: false).WithLocation(20, 9));
        }

        [Fact]
        [WorkItem(43659, "https://github.com/dotnet/roslyn/issues/43659")]
        public void SuppressionOnNullableReferenceType_AppliedOnField3()
        {
            var source = @"
public class C
{
    public string[]? F1;
    public System.Func<object>? F2;
    static void M1(C? c)
    {
        c?.F1![0].ToString();
        c.ToString(); // 1
    }
    static void M2(C? c)
    {
        c?.F2!().ToString();
        c.ToString(); // 2
    }
}
";
            var c = CreateNullableCompilation(source);
            c.VerifyDiagnostics(
                // (9,9): warning CS8602: Dereference of a possibly null reference.
                //         c.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c", isSuppressed: false).WithLocation(9, 9),
                // (14,9): warning CS8602: Dereference of a possibly null reference.
                //         c.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c", isSuppressed: false).WithLocation(14, 9));
        }

        [Fact]
        [WorkItem(31732, "https://github.com/dotnet/roslyn/issues/31732")]
        public void SuppressionOnArgument()
        {
            var source = @"
class C
{
    void M(string? s)
    {
        NonNull(s!);
        s.ToString(); // warn
    }
    void NonNull(string s) => throw null!;
}
";
            var c = CreateCompilation(new[] { source }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (7,9): warning CS8602: Dereference of a possibly null reference.
                //         s.ToString(); // warn
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "s").WithLocation(7, 9)
                );
        }

        [Fact]
        public void SuppressionWithoutNonNullTypes()
        {
            var source = @"
[System.Obsolete("""", true!)] // 1, 2
class C
{
    string x = null!; // 3, 4
    static void Main(string z = null!) // 5
    {
        string y = null!; // 6, 7
    }
}
";
            var c = CreateCompilation(source);
            c.VerifyEmitDiagnostics(
                // (8,16): warning CS0219: The variable 'y' is assigned but its value is never used
                //         string y = null!; // 6, 7
                Diagnostic(ErrorCode.WRN_UnreferencedVarAssg, "y").WithArguments("y").WithLocation(8, 16),
                // (5,12): warning CS0414: The field 'C.x' is assigned but its value is never used
                //     string x = null!; // 3, 4
                Diagnostic(ErrorCode.WRN_UnreferencedFieldAssg, "x").WithArguments("C.x").WithLocation(5, 12)
                );
        }

        [Fact, WorkItem(26812, "https://github.com/dotnet/roslyn/issues/26812")]
        public void DoubleAssignment()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
using static System.Console;
class C
{
    static void Main()
    {
        string? x;
        x = x = """";
        WriteLine(x.Length);
        string? y;
        x = y = """";
        WriteLine(x.Length);
        WriteLine(y.Length);
    }
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics();
        }

        [Fact, WorkItem(26746, "https://github.com/dotnet/roslyn/issues/26746")]
        public void TernaryWithConversionFromExpression()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
public class C 
{
    public void M() 
    {
        uint a = 0;
        uint x = true ? a : 1;
        uint y = true ? 1 : a;
    }
}
" }, options: WithNullableEnable());
            c.VerifyDiagnostics();
        }

        [Fact, WorkItem(26746, "https://github.com/dotnet/roslyn/issues/26746")]
        public void TernaryWithImplicitUsedDefinedConversion_ConstantTrue()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
public class C
{
    public void M()
    {
        C c = new C();
        C x = true ? c : 1;
        C y = true ? 1 : c;
    }
    public static implicit operator C?(int i) => throw null!;
}
" }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (8,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         C y = true ? 1 : c;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "true ? 1 : c").WithLocation(8, 15)
                );
        }

        [Fact, WorkItem(26746, "https://github.com/dotnet/roslyn/issues/26746")]
        public void TernaryWithImplicitUsedDefinedConversion_ConstantFalse()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
public class C
{
    public void M()
    {
        C c = new C();
        C x = false ? c : 1;
        C y = false ? 1 : c;
    }
    public static implicit operator C?(int i) => throw null!;
}
" }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (7,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         C x = false ? c : 1;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "false ? c : 1").WithLocation(7, 15)
                );
        }

        [Fact, WorkItem(26746, "https://github.com/dotnet/roslyn/issues/26746")]
        public void TernaryWithImplicitUsedDefinedConversion_NotConstant()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
public class C
{
    public void M(bool b)
    {
        C c = new C();
        C x = b ? c : 1;
        C y = b ? 1 : c;
    }
    public static implicit operator C?(int i) => throw null!;
}
" }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (7,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         C x = b ? c : 1;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "b ? c : 1").WithLocation(7, 15),
                // (8,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         C y = b ? 1 : c;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "b ? 1 : c").WithLocation(8, 15)
                );
        }

        [Fact, WorkItem(26746, "https://github.com/dotnet/roslyn/issues/26746")]
        public void TernaryWithImplicitUsedDefinedConversion2()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
public class C
{
    public void M()
    {
        C c = null!;
        int x = true ? c : 1;
        int y = true ? 1 : c;

        C? c2 = null;
        int x2 = true ? c2 : 1;
        int y2 = true ? 1 : c2;
    }
    public static implicit operator int(C i) => throw null!;
}
" }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (11,25): warning CS8604: Possible null reference argument for parameter 'i' in 'C.implicit operator int(C i)'.
                //         int x2 = true ? c2 : 1;
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "c2").WithArguments("i", "C.implicit operator int(C i)").WithLocation(11, 25)
                );
        }

        [Fact]
        public void AnnotationWithoutNonNullTypes()
        {
            var source = @"
class C<T> where T : class
{
    static string? field = M2(out string? x1); // warn 1 and 2
    static string? P // warn 3
    {
        get
        {
            string? x2 = null; // warn 4
            return x2;
        }
    }
    static string? MethodWithLocalFunction() // warn 5
    {
        string? x3 = local(null); // warn 6
        return x3;

        string? local(C<string?>? x) // warn 7, 8 and 9
        {
            string? x4 = null; // warn 10
            return x4;
        }
    }
    static string? Lambda() // warn 11
    {
        System.Func<string?, string?> x5 = (string? x) =>  // warn 12, 13 and 14
        {
            string? x6 = null; // warn 15
            return x6;
        };
        return x5(null);
    }
    static string M2(out string? x4) => throw null!; // warn 16
    static string M3(C<string?> x, C<string> y) => throw null!; // warn 17
    delegate string? MyDelegate(C<string?> x); // warn 18 and 19
    event MyDelegate? Event; // warn 20
    void M4() { Event(new C<string?>()); } // warn 21
    class D<T2> where T2 : T? { } // warn 22
    class D2 : C<string?> { } // warn 23
    public static C<T?> operator +(C<T> x, C<T?> y) => throw null!; // warn 24 and 25
    class D3
    {
        D3(C<T?> x) => throw null!; // warn 26
    }
    public string? this[C<string?> x] { get => throw null!; } // warn 27 and 28
}
";
            var expectedDiagnostics = new[] {
                // (36,21): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     event MyDelegate? Event; // warn 20
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(36, 21),
                // (5,18): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     static string? P // warn 3
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 18),
                // (13,18): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     static string? MethodWithLocalFunction() // warn 5
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(13, 18),
                // (24,18): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     static string? Lambda() // warn 11
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(24, 18),
                // (33,32): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     static string M2(out string? x4) => throw null!; // warn 16
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(33, 32),
                // (34,30): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     static string M3(C<string?> x, C<string> y) => throw null!; // warn 17
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(34, 30),
                // (40,47): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public static C<T?> operator +(C<T> x, C<T?> y) => throw null!; // warn 24 and 25
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(40, 47),
                // (40,46): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     public static C<T?> operator +(C<T> x, C<T?> y) => throw null!; // warn 24 and 25
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(40, 46),
                // (40,22): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public static C<T?> operator +(C<T> x, C<T?> y) => throw null!; // warn 24 and 25
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(40, 22),
                // (40,21): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     public static C<T?> operator +(C<T> x, C<T?> y) => throw null!; // warn 24 and 25
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(40, 21),
                // (45,33): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public string? this[C<string?> x] { get => throw null!; } // warn 27 and 28
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(45, 33),
                // (45,18): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public string? this[C<string?> x] { get => throw null!; } // warn 27 and 28
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(45, 18),
                // (43,15): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         D3(C<T?> x) => throw null!; // warn 26
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(43, 15),
                // (43,14): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //         D3(C<T?> x) => throw null!; // warn 26
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(43, 14),
                // (35,20): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     delegate string? MyDelegate(C<string?> x); // warn 18 and 19
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(35, 20),
                // (35,41): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     delegate string? MyDelegate(C<string?> x); // warn 18 and 19
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(35, 41),
                // (38,29): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     class D<T2> where T2 : T? { } // warn 22
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(38, 29),
                // (38,28): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     class D<T2> where T2 : T? { } // warn 22
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(38, 28),
                // (39,24): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     class D2 : C<string?> { } // warn 23
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(39, 24),
                // (4,18): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     static string? field = M2(out string? x1); // warn 1 and 2
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(4, 18),
                // (4,41): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     static string? field = M2(out string? x1); // warn 1 and 2
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(4, 41),
                // (9,19): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //             string? x2 = null; // warn 4
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(9, 19),
                // (15,15): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         string? x3 = local(null); // warn 6
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(15, 15),
                // (20,19): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //             string? x4 = null; // warn 10
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(20, 19),
                // (18,31): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         string? local(C<string?>? x) // warn 7, 8 and 9
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(18, 31),
                // (18,33): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         string? local(C<string?>? x) // warn 7, 8 and 9
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(18, 33),
                // (18,15): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         string? local(C<string?>? x) // warn 7, 8 and 9
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(18, 15),
                // (26,27): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         System.Func<string?, string?> x5 = (string? x) =>  // warn 12, 13 and 14
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(26, 27),
                // (26,36): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         System.Func<string?, string?> x5 = (string? x) =>  // warn 12, 13 and 14
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(26, 36),
                // (26,51): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         System.Func<string?, string?> x5 = (string? x) =>  // warn 12, 13 and 14
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(26, 51),
                // (28,19): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //             string? x6 = null; // warn 15
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(28, 19),
                // (37,35): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     void M4() { Event(new C<string?>()); } // warn 21
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(37, 35)
            };

            var c = CreateCompilation(source, parseOptions: TestOptions.Regular8);
            c.VerifyDiagnostics(expectedDiagnostics);

            var c2 = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);
            c2.VerifyDiagnostics(
                // (18,25): warning CS8634: The type 'string?' cannot be used as type parameter 'T' in the generic type or method 'C<T>'. Nullability of type argument 'string?' doesn't match 'class' constraint.
                //         string? local(C<string?>? x) // warn 7, 8 and 9
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeParameterReferenceTypeConstraint, "string?").WithArguments("C<T>", "T", "string?").WithLocation(18, 25),
                // (34,33): warning CS8634: The type 'string?' cannot be used as type parameter 'T' in the generic type or method 'C<T>'. Nullability of type argument 'string?' doesn't match 'class' constraint.
                //     static string M3(C<string?> x, C<string> y) => throw null!; // warn 17
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeParameterReferenceTypeConstraint, "x").WithArguments("C<T>", "T", "string?").WithLocation(34, 33),
                // (35,35): warning CS8634: The type 'string?' cannot be used as type parameter 'T' in the generic type or method 'C<T>'. Nullability of type argument 'string?' doesn't match 'class' constraint.
                //     delegate string? MyDelegate(C<string?> x); // warn 18 and 19
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeParameterReferenceTypeConstraint, "string?").WithArguments("C<T>", "T", "string?").WithLocation(35, 35),
                // (37,17): warning CS8602: Dereference of a possibly null reference.
                //     void M4() { Event(new C<string?>()); } // warn 21
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "Event").WithLocation(37, 17),
                // (37,29): warning CS8634: The type 'string?' cannot be used as type parameter 'T' in the generic type or method 'C<T>'. Nullability of type argument 'string?' doesn't match 'class' constraint.
                //     void M4() { Event(new C<string?>()); } // warn 21
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeParameterReferenceTypeConstraint, "string?").WithArguments("C<T>", "T", "string?").WithLocation(37, 29),
                // (39,11): warning CS8634: The type 'string?' cannot be used as type parameter 'T' in the generic type or method 'C<T>'. Nullability of type argument 'string?' doesn't match 'class' constraint.
                //     class D2 : C<string?> { } // warn 23
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeParameterReferenceTypeConstraint, "D2").WithArguments("C<T>", "T", "string?").WithLocation(39, 11),
                // (40,34): warning CS8634: The type 'T?' cannot be used as type parameter 'T' in the generic type or method 'C<T>'. Nullability of type argument 'T?' doesn't match 'class' constraint.
                //     public static C<T?> operator +(C<T> x, C<T?> y) => throw null!; // warn 24 and 25
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeParameterReferenceTypeConstraint, "+").WithArguments("C<T>", "T", "T?").WithLocation(40, 34),
                // (40,50): warning CS8634: The type 'T?' cannot be used as type parameter 'T' in the generic type or method 'C<T>'. Nullability of type argument 'T?' doesn't match 'class' constraint.
                //     public static C<T?> operator +(C<T> x, C<T?> y) => throw null!; // warn 24 and 25
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeParameterReferenceTypeConstraint, "y").WithArguments("C<T>", "T", "T?").WithLocation(40, 50),
                // (43,18): warning CS8634: The type 'T?' cannot be used as type parameter 'T' in the generic type or method 'C<T>'. Nullability of type argument 'T?' doesn't match 'class' constraint.
                //         D3(C<T?> x) => throw null!; // warn 26
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeParameterReferenceTypeConstraint, "x").WithArguments("C<T>", "T", "T?").WithLocation(43, 18),
                // (45,36): warning CS8634: The type 'string?' cannot be used as type parameter 'T' in the generic type or method 'C<T>'. Nullability of type argument 'string?' doesn't match 'class' constraint.
                //     public string? this[C<string?> x] { get => throw null!; } // warn 27 and 28
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeParameterReferenceTypeConstraint, "x").WithArguments("C<T>", "T", "string?").WithLocation(45, 36)
                );

            var c3 = CreateCompilation(new[] { source }, options: WithNullableDisable(), parseOptions: TestOptions.Regular8);
            c3.VerifyDiagnostics(expectedDiagnostics);
        }

        [Fact]
        public void AnnotationWithoutNonNullTypes_GenericType()
        {
            var source = @"
public class C<T> where T : class
{
    public T? M(T? x1) // warn 1 and 2
    {
        T? y1 = x1; // warn 3
        return y1;
    }
}
public class E<T> where T : struct
{
    public T? M(T? x2)
    {
        T? y2 = x2;
        return y2;
    }
}
";
            CSharpCompilation c = CreateCompilation(source, parseOptions: TestOptions.Regular8);
            c.VerifyDiagnostics(
                // (4,18): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public T? M(T? x1) // warn 1 and 2
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(4, 18),
                // (4,17): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     public T? M(T? x1) // warn 1 and 2
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(4, 17),
                // (4,13): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public T? M(T? x1) // warn 1 and 2
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(4, 13),
                // (4,12): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     public T? M(T? x1) // warn 1 and 2
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(4, 12),
                // (6,10): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         T? y1 = x1; // warn 3
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(6, 10),
                // (6,9): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //         T? y1 = x1; // warn 3
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(6, 9)
                );

            var client = @"
class Client
{
    void M(C<string> c)
    {
        c.M("""").ToString();
    }
}
";
            var comp2 = CreateCompilation(new[] { client }, options: WithNullableEnable(), references: new[] { c.ToMetadataReference() });
            comp2.VerifyDiagnostics(
                // (6,9): warning CS8602: Dereference of a possibly null reference.
                //         c.M("").ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, @"c.M("""")").WithLocation(6, 9)
                );

            c = CreateCompilation(new[] { source }, options: WithNullableEnable());
            c.VerifyDiagnostics();

            comp2 = CreateCompilation(new[] { client }, options: WithNullableEnable(), references: new[] { c.EmitToImageReference() });
            comp2.VerifyDiagnostics(
                // (6,9): warning CS8602: Dereference of a possibly null reference.
                //         c.M("").ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, @"c.M("""")").WithLocation(6, 9)
                );

            comp2 = CreateCompilation(new[] { client }, options: WithNullableEnable(), references: new[] { c.ToMetadataReference() });
            comp2.VerifyDiagnostics(
                // (6,9): warning CS8602: Dereference of a possibly null reference.
                //         c.M("").ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, @"c.M("""")").WithLocation(6, 9)
                );
        }

        [Fact]
        public void AnnotationWithoutNonNullTypes_AttributeArgument()
        {
            var source =
@"class AAttribute : System.Attribute
{
    internal AAttribute(object o) { }
}
class B<T> { }
[A(typeof(object?))] // 1
class C1 { }
[A(typeof(int?))]
class C2 { }
[A(typeof(B<object?>))] // 2
class C3 { }
[A(typeof(B<int?>))]
class C4 { }";

            var comp = CreateCompilation(source);
            comp.VerifyDiagnostics(
                // (6,4): error CS8639: The typeof operator cannot be used on a nullable reference type
                // [A(typeof(object?))] // 1
                Diagnostic(ErrorCode.ERR_BadNullableTypeof, "typeof(object?)").WithLocation(6, 4),
                // (6,17): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                // [A(typeof(object?))] // 1
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(6, 17),
                // (10,19): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                // [A(typeof(B<object?>))] // 2
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(10, 19));

            comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,4): error CS8639: The typeof operator cannot be used on a nullable reference type
                // [A(typeof(object?))] // 1
                Diagnostic(ErrorCode.ERR_BadNullableTypeof, "typeof(object?)").WithLocation(6, 4));
        }

        [Fact]
        public void Nullable_False_InCSharp7()
        {
            var comp = CreateCompilation("", options: WithNullableDisable(), parseOptions: TestOptions.Regular7);
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void NullableOption()
        {
            var comp = CreateCompilation("", options: WithNullable(NullableContextOptions.Enable), parseOptions: TestOptions.Regular7_3);
            comp.VerifyDiagnostics(
                // error CS8630: Invalid 'NullableContextOptions' value: 'Enable' for C# 7.3. Please use language version '8.0' or greater.
                Diagnostic(ErrorCode.ERR_NullableOptionNotAvailable).WithArguments("NullableContextOptions", "Enable", "7.3", "8.0").WithLocation(1, 1)
                );

            comp = CreateCompilation("", options: WithNullable(NullableContextOptions.Warnings), parseOptions: TestOptions.Regular7_3);
            comp.VerifyDiagnostics(
                // error CS8630: Invalid 'NullableContextOptions' value: 'Warnings' for C# 7.3. Please use language version '8.0' or greater.
                Diagnostic(ErrorCode.ERR_NullableOptionNotAvailable).WithArguments("NullableContextOptions", "Warnings", "7.3", "8.0").WithLocation(1, 1)
                );

            comp = CreateCompilation("", options: WithNullable(NullableContextOptions.Disable), parseOptions: TestOptions.Regular7_3);
            comp.VerifyDiagnostics();

            comp = CreateCompilation("", options: WithNullable(NullableContextOptions.Enable), parseOptions: TestOptions.Regular8);
            comp.VerifyDiagnostics();

            comp = CreateCompilation("", options: WithNullable(NullableContextOptions.Warnings), parseOptions: TestOptions.Regular8);
            comp.VerifyDiagnostics();

            comp = CreateCompilation("", options: WithNullable(NullableContextOptions.Disable), parseOptions: TestOptions.Regular8);
            comp.VerifyDiagnostics();

            comp = CreateCompilation(new string[] { }, options: WithNullable(NullableContextOptions.Enable), parseOptions: TestOptions.Regular7_3);
            comp.VerifyDiagnostics();

            comp = CreateCompilation(new string[] { }, options: WithNullable(NullableContextOptions.Enable), parseOptions: TestOptions.Regular8);
            comp.VerifyDiagnostics();

            comp = CreateCompilation(new string[] { }, options: WithNullable(NullableContextOptions.Warnings), parseOptions: TestOptions.Regular7_3);
            comp.VerifyDiagnostics();

            comp = CreateCompilation(new string[] { }, options: WithNullable(NullableContextOptions.Warnings), parseOptions: TestOptions.Regular8);
            comp.VerifyDiagnostics();

            comp = CreateCompilation(new string[] { }, options: WithNullable(NullableContextOptions.Disable), parseOptions: TestOptions.Regular7_3);
            comp.VerifyDiagnostics();

            comp = CreateCompilation(new string[] { }, options: WithNullable(NullableContextOptions.Disable), parseOptions: TestOptions.Regular8);
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void NullableAttribute_NotRequiredCSharp7_01()
        {
            var source =
@"using System.Threading.Tasks;
class C
{
    static async Task<string> F()
    {
        return await Task.FromResult(default(string));
    }
}";
            var comp = CreateCompilationWithMscorlib45(source, parseOptions: TestOptions.Regular7);
            comp.VerifyEmitDiagnostics();
        }

        [Fact]
        public void NullableAttribute_NotRequiredCSharp7_02()
        {
            var source =
@"using System;
using System.Threading.Tasks;
class C
{
    static async Task F<T>(Func<Task> f)
    {
        await G(async () =>
        {
            await f();
            return default(object);
        });
    }
    static async Task<TResult> G<TResult>(Func<Task<TResult>> f)
    {
        throw new NotImplementedException();
    }
}";
            var comp = CreateCompilationWithMscorlib45(source, parseOptions: TestOptions.Regular7);
            comp.VerifyEmitDiagnostics(
                // (13,32): warning CS1998: This async method lacks 'await' operators and will run synchronously. Consider using the 'await' operator to await non-blocking API calls, or 'await Task.Run(...)' to do CPU-bound work on a background thread.
                //     static async Task<TResult> G<TResult>(Func<Task<TResult>> f)
                Diagnostic(ErrorCode.WRN_AsyncLacksAwaits, "G").WithLocation(13, 32));
        }

        [Fact, WorkItem(26739, "https://github.com/dotnet/roslyn/issues/26618")]
        public void SuppressionOnNullConvertedToConstrainedTypeParameterType()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
public class C
{
    public T M<T>() where T : C
    {
        return null!;
    }
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics();
        }

        [Fact]
        public void MissingInt()
        {
            var source0 =
@"namespace System
{
    public class Object { }
    public abstract class ValueType { }
    public struct Void { }
    public struct Boolean { }
    public struct Enum { }
    public class Attribute { }
}";
            var comp0 = CreateEmptyCompilation(source0, parseOptions: TestOptions.Regular7);
            comp0.VerifyDiagnostics();
            var ref0 = comp0.EmitToImageReference();

            var source =
@"enum E { A }
class C
{
    int F() => (int)E.A;
}";
            var comp = CreateEmptyCompilation(
                source,
                references: new[] { ref0 },
                parseOptions: TestOptions.Regular8);
            comp.VerifyEmitDiagnostics(
                // (1,6): error CS0518: Predefined type 'System.Int32' is not defined or imported
                // enum E { A }
                Diagnostic(ErrorCode.ERR_PredefinedTypeNotFound, "E").WithArguments("System.Int32").WithLocation(1, 6),
                // (4,5): error CS0518: Predefined type 'System.Int32' is not defined or imported
                //     int F() => (int)E.A;
                Diagnostic(ErrorCode.ERR_PredefinedTypeNotFound, "int").WithArguments("System.Int32").WithLocation(4, 5),
                // (4,17): error CS0518: Predefined type 'System.Int32' is not defined or imported
                //     int F() => (int)E.A;
                Diagnostic(ErrorCode.ERR_PredefinedTypeNotFound, "int").WithArguments("System.Int32").WithLocation(4, 17));
        }

        [Fact]
        public void MissingNullable()
        {
            var source = @"
namespace System
{
    public class Object { }
    public abstract class ValueType { }
    public struct Void { }
    public struct Boolean { }
    public struct Enum { }
    public class Attribute { }
}";
            var source2 = @"
class C<T> where T : struct
{
    void M()
    {
        T? local = null;
        _ = local;
    }
}
";
            var comp = CreateEmptyCompilation(new[] { source, source2 });
            comp.VerifyDiagnostics(
                // (6,9): error CS0518: Predefined type 'System.Nullable`1' is not defined or imported
                //         T? local = null;
                Diagnostic(ErrorCode.ERR_PredefinedTypeNotFound, "T?").WithArguments("System.Nullable`1").WithLocation(6, 9)
                );

            var source3 = @"
class C<T> where T : struct
{
    void M(T? nullable) { }
}
";
            var comp2 = CreateEmptyCompilation(new[] { source, source3 });
            comp2.VerifyDiagnostics(
                // (4,12): error CS0518: Predefined type 'System.Nullable`1' is not defined or imported
                //     void M(T? nullable) { }
                Diagnostic(ErrorCode.ERR_PredefinedTypeNotFound, "T?").WithArguments("System.Nullable`1").WithLocation(4, 12)
                );

            var source4 = @"
class C<T> where T : struct
{
    void M<U>() where U : T? { }
}
";
            var comp3 = CreateEmptyCompilation(new[] { source, source4 });
            comp3.VerifyDiagnostics(
                // (4,27): error CS0518: Predefined type 'System.Nullable`1' is not defined or imported
                //     void M<U>() where U : T? { }
                Diagnostic(ErrorCode.ERR_PredefinedTypeNotFound, "T?").WithArguments("System.Nullable`1").WithLocation(4, 27),
                // (4,12): error CS0518: Predefined type 'System.Nullable`1' is not defined or imported
                //     void M<U>() where U : T? { }
                Diagnostic(ErrorCode.ERR_PredefinedTypeNotFound, "U").WithArguments("System.Nullable`1").WithLocation(4, 12)
                );
        }

        [Fact]
        public void UnannotatedAssemblies_01()
        {
            var source0 =
@"public class A
{
    public static void F(string s) { }
}";
            var source1 =
@"class B
{
    static void Main()
    {
        A.F(string.Empty);
        A.F(null);
    }
}";
            TypeWithAnnotations getParameterType(CSharpCompilation c) => c.GetMember<MethodSymbol>("A.F").Parameters[0].TypeWithAnnotations;

            // 7.0 library
            var comp0 = CreateCompilation(source0, parseOptions: TestOptions.Regular7);
            comp0.VerifyDiagnostics();
            var compRefs0 = new MetadataReference[] { new CSharpCompilationReference(comp0) };
            var metadataRefs0 = new[] { comp0.EmitToImageReference() };
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp0).NullableAnnotation);

            // ... used in 7.0.
            var comp1 = CreateCompilation(source1, references: compRefs0, parseOptions: TestOptions.Regular7);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp1).NullableAnnotation);
            comp1 = CreateCompilation(source1, references: metadataRefs0, parseOptions: TestOptions.Regular7);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp1).NullableAnnotation);

            // ... used in 8.0.
            comp1 = CreateCompilation(source1, references: compRefs0);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp1).NullableAnnotation);
            comp1 = CreateCompilation(source1, references: metadataRefs0);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp1).NullableAnnotation);

            // 8.0 library
            comp0 = CreateCompilation(source0);
            comp0.VerifyDiagnostics();
            compRefs0 = new MetadataReference[] { new CSharpCompilationReference(comp0) };
            metadataRefs0 = new[] { comp0.EmitToImageReference() };
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp0).NullableAnnotation);

            // ... used in 7.0.
            comp1 = CreateCompilation(source1, references: compRefs0, parseOptions: TestOptions.Regular7);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp1).NullableAnnotation);
            comp1 = CreateCompilation(source1, references: metadataRefs0, parseOptions: TestOptions.Regular7);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp1).NullableAnnotation);

            // ... used in 8.0.
            comp1 = CreateCompilation(source1, references: compRefs0);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp1).NullableAnnotation);
            comp1 = CreateCompilation(source1, references: metadataRefs0);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp1).NullableAnnotation);

            comp1 = CreateCompilation(new[] { source1 }, options: WithNullableEnable(), references: compRefs0);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp1).NullableAnnotation);
            comp1 = CreateCompilation(new[] { source1 }, options: WithNullableEnable(), references: metadataRefs0);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.Oblivious, getParameterType(comp1).NullableAnnotation);

            // 8.0 library
            comp0 = CreateCompilation(new[] { source0 }, options: WithNullableEnable());
            comp0.VerifyDiagnostics();
            compRefs0 = new MetadataReference[] { new CSharpCompilationReference(comp0) };
            metadataRefs0 = new[] { comp0.EmitToImageReference() };
            Assert.Equal(NullableAnnotation.NotAnnotated, getParameterType(comp0).NullableAnnotation);

            // ... used in 7.0.
            comp1 = CreateCompilation(source1, references: compRefs0, parseOptions: TestOptions.Regular7);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.NotAnnotated, getParameterType(comp1).NullableAnnotation);
            comp1 = CreateCompilation(source1, references: metadataRefs0, parseOptions: TestOptions.Regular7);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.NotAnnotated, getParameterType(comp1).NullableAnnotation);

            // ... used in 8.0.
            comp1 = CreateCompilation(source1, references: compRefs0);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.NotAnnotated, getParameterType(comp1).NullableAnnotation);
            comp1 = CreateCompilation(source1, references: metadataRefs0);
            comp1.VerifyDiagnostics();
            Assert.Equal(NullableAnnotation.NotAnnotated, getParameterType(comp1).NullableAnnotation);

            comp1 = CreateCompilation(new[] { source1 }, options: WithNullableEnable(), references: compRefs0);
            comp1.VerifyDiagnostics(
                // (6,13): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         A.F(null);
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(6, 13));
            Assert.Equal(NullableAnnotation.NotAnnotated, getParameterType(comp1).NullableAnnotation);
            comp1 = CreateCompilation(new[] { source1 }, options: WithNullableEnable(), references: metadataRefs0);
            comp1.VerifyDiagnostics(
                // (6,13): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         A.F(null);
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(6, 13));
            Assert.Equal(NullableAnnotation.NotAnnotated, getParameterType(comp1).NullableAnnotation);
        }

        [Fact]
        public void UnannotatedAssemblies_02()
        {
            var source0 =
@"#pragma warning disable 67
public delegate void D();
public class C
{
    public object F;
    public event D E;
    public object P => null;
    public object this[object o] => null;
    public object M(object o) => null;
}";
            var source1 =
@"class P
{
    static void F(C c)
    {
        object o;
        o = c.F;
        c.E += null;
        o = c.P;
        o = c[null];
        o = c.M(null);
    }
}";

            var comp0 = CreateCompilation(source0, parseOptions: TestOptions.Regular7);
            comp0.VerifyDiagnostics();

            void verify(CSharpCompilation c)
            {
                c.VerifyDiagnostics();
                Assert.Equal(NullableAnnotation.Oblivious, c.GetMember<FieldSymbol>("C.F").TypeWithAnnotations.NullableAnnotation);
                Assert.Equal(NullableAnnotation.Oblivious, c.GetMember<EventSymbol>("C.E").TypeWithAnnotations.NullableAnnotation);
                Assert.Equal(NullableAnnotation.Oblivious, c.GetMember<PropertySymbol>("C.P").TypeWithAnnotations.NullableAnnotation);
                var indexer = c.GetMember<PropertySymbol>("C.this[]");
                Assert.Equal(NullableAnnotation.Oblivious, indexer.TypeWithAnnotations.NullableAnnotation);
                Assert.Equal(NullableAnnotation.Oblivious, indexer.Parameters[0].TypeWithAnnotations.NullableAnnotation);
                var method = c.GetMember<MethodSymbol>("C.M");
                Assert.Equal(NullableAnnotation.Oblivious, method.ReturnTypeWithAnnotations.NullableAnnotation);
                Assert.Equal(NullableAnnotation.Oblivious, method.Parameters[0].TypeWithAnnotations.NullableAnnotation);
            }

            var comp1A = CreateCompilation(source1, references: new MetadataReference[] { new CSharpCompilationReference(comp0) });
            verify(comp1A);

            var comp1B = CreateCompilation(source1, references: new[] { comp0.EmitToImageReference() });
            verify(comp1B);
        }

        [Fact]
        public void UnannotatedAssemblies_03()
        {
            var source0 =
@"#pragma warning disable 67
public class C
{
    public (object, object) F;
    public (object, object) P => (null, null);
    public (object, object) M((object, object) o) => o;
}";
            var source1 =
@"class P
{
    static void F(C c)
    {
        (object, object) t;
        t = c.F;
        t = c.P;
        t = c.M((null, null));
    }
}";

            var comp0 = CreateCompilation(source0, parseOptions: TestOptions.Regular7);
            comp0.VerifyDiagnostics();

            void verifyTuple(TypeWithAnnotations type)
            {
                var tuple = (NamedTypeSymbol)type.Type;
                Assert.Equal(NullableAnnotation.Oblivious, tuple.TupleElements[0].TypeWithAnnotations.NullableAnnotation);
                Assert.Equal(NullableAnnotation.Oblivious, tuple.TupleElements[1].TypeWithAnnotations.NullableAnnotation);
            }

            void verify(CSharpCompilation c)
            {
                c.VerifyDiagnostics();
                verifyTuple(c.GetMember<FieldSymbol>("C.F").TypeWithAnnotations);
                verifyTuple(c.GetMember<PropertySymbol>("C.P").TypeWithAnnotations);
                var method = c.GetMember<MethodSymbol>("C.M");
                verifyTuple(method.ReturnTypeWithAnnotations);
                verifyTuple(method.Parameters[0].TypeWithAnnotations);
            }

            var comp1A = CreateCompilation(source1, references: new[] { new CSharpCompilationReference(comp0) });
            verify(comp1A);

            var comp1B = CreateCompilation(source1, references: new[] { comp0.EmitToImageReference() });
            verify(comp1B);
        }

        [Fact]
        public void UnannotatedAssemblies_04()
        {
            var source =
@"class A { }
class B : A { }
interface I<T> where T : A { }
abstract class C<T> where T : A
{
    internal abstract void M<U>() where U : T;
}
class D : C<B>, I<B>
{
    internal override void M<T>() { }
}";
            var comp = CreateCompilation(new[] { source }, parseOptions: TestOptions.Regular7);
            comp.VerifyDiagnostics();
            var derivedType = comp.GetMember<NamedTypeSymbol>("D");
            var baseType = derivedType.BaseTypeNoUseSiteDiagnostics;
            var constraintType = baseType.TypeParameters.Single().ConstraintTypesNoUseSiteDiagnostics.Single();
            Assert.Equal(NullableAnnotation.Oblivious, constraintType.NullableAnnotation);
            var interfaceType = derivedType.Interfaces().Single();
            constraintType = interfaceType.TypeParameters.Single().ConstraintTypesNoUseSiteDiagnostics.Single();
            Assert.Equal(NullableAnnotation.Oblivious, constraintType.NullableAnnotation);
            var method = baseType.GetMember<MethodSymbol>("M");
            constraintType = method.TypeParameters.Single().ConstraintTypesNoUseSiteDiagnostics.Single();
            Assert.Equal(NullableAnnotation.Oblivious, constraintType.NullableAnnotation);
        }

        [Fact]
        public void UnannotatedAssemblies_05()
        {
            var source =
@"interface I<T>
{
    I<object[]> F(I<T> t);
}
class C : I<string>
{
    I<object[]> I<string>.F(I<string> s) => null;
}";
            var comp = CreateCompilation(new[] { source }, parseOptions: TestOptions.Regular7);
            comp.VerifyDiagnostics();
            var type = comp.GetMember<NamedTypeSymbol>("C");
            var interfaceType = type.Interfaces().Single();
            var typeArg = interfaceType.TypeArgumentsWithAnnotationsNoUseSiteDiagnostics.Single();
            Assert.Equal(NullableAnnotation.Oblivious, typeArg.NullableAnnotation);
            var method = type.GetMember<MethodSymbol>("I<System.String>.F");
            Assert.Equal(NullableAnnotation.Oblivious, method.ReturnTypeWithAnnotations.NullableAnnotation);
            typeArg = ((NamedTypeSymbol)method.ReturnType).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics.Single();
            Assert.Equal(NullableAnnotation.Oblivious, typeArg.NullableAnnotation);
            var parameter = method.Parameters.Single();
            Assert.Equal(NullableAnnotation.Oblivious, parameter.TypeWithAnnotations.NullableAnnotation);
            typeArg = ((NamedTypeSymbol)parameter.Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics.Single();
            Assert.Equal(NullableAnnotation.Oblivious, typeArg.NullableAnnotation);
        }

        [Fact]
        public void UnannotatedAssemblies_06()
        {
            var source0 =
@"public class C<T>
{
    public T F;
}
public class C
{
    public static C<T> Create<T>(T t) => new C<T>();
}";
            var source1 =
@"class P
{
    static void F(object x, object? y)
    {
        object z;
        z = C.Create(x).F;
        z = C.Create(y).F;
    }
}";
            var comp0 = CreateCompilation(source0, parseOptions: TestOptions.Regular7);
            comp0.VerifyDiagnostics();
            var comp1 = CreateCompilation(new[] { source1 }, options: WithNullableEnable(),
                references: new[] { comp0.EmitToImageReference() });
            comp1.VerifyDiagnostics(
                // (7,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         z = C.Create(y).F;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "C.Create(y).F").WithLocation(7, 13));
        }

        [Fact]
        public void UnannotatedAssemblies_07()
        {
            var source0 =
@"public interface I
{
    object F(object o);
}";
            var source1 =
@"class A1 : I
{
    object I.F(object? o) => new object();
}
class A2 : I
{
    object? I.F(object o) => o;
}
class B1 : I
{
    public object F(object? o) => new object();
}
class B2 : I
{
    public object? F(object o) => o;
}
class C1
{
    public object F(object? o) => new object();
}
class C2
{
    public object? F(object o) => o;
}
class D1 : C1, I
{
}
class D2 : C2, I
{
}
class P
{
    static void F(object? x, A1 a1, A2 a2)
    {
        object y;
        y = ((I)a1).F(x);
        y = ((I)a2).F(x);
    }
    static void F(object? x, B1 b1, B2 b2)
    {
        object y;
        y = b1.F(x);
        y = b2.F(x);
        y = ((I)b1).F(x);
        y = ((I)b2).F(x);
    }
    static void F(object? x, D1 d1, D2 d2)
    {
        object y;
        y = d1.F(x);
        y = d2.F(x);
        y = ((I)d1).F(x);
        y = ((I)d2).F(x);
    }
}";
            var comp0 = CreateCompilation(source0, parseOptions: TestOptions.Regular7);
            comp0.VerifyDiagnostics();

            var comp1 = CreateCompilation(new[] { source1 }, options: WithNullableEnable(),
                references: new MetadataReference[] { new CSharpCompilationReference(comp0) });
            comp1.VerifyDiagnostics(
                // (43,18): warning CS8604: Possible null reference argument for parameter 'o' in 'object? B2.F(object o)'.
                //         y = b2.F(x);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x").WithArguments("o", "object? B2.F(object o)").WithLocation(43, 18),
                // (43,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         y = b2.F(x);
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "b2.F(x)").WithLocation(43, 13),
                // (51,18): warning CS8604: Possible null reference argument for parameter 'o' in 'object? C2.F(object o)'.
                //         y = d2.F(x);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x").WithArguments("o", "object? C2.F(object o)").WithLocation(51, 18),
                // (51,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         y = d2.F(x);
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "d2.F(x)").WithLocation(51, 13));

            comp1 = CreateCompilation(new[] { source1 }, options: WithNullableEnable(),
                references: new[] { comp0.EmitToImageReference() });
            comp1.VerifyDiagnostics(
                // (43,18): warning CS8604: Possible null reference argument for parameter 'o' in 'object? B2.F(object o)'.
                //         y = b2.F(x);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x").WithArguments("o", "object? B2.F(object o)").WithLocation(43, 18),
                // (43,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         y = b2.F(x);
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "b2.F(x)").WithLocation(43, 13),
                // (51,18): warning CS8604: Possible null reference argument for parameter 'o' in 'object? C2.F(object o)'.
                //         y = d2.F(x);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x").WithArguments("o", "object? C2.F(object o)").WithLocation(51, 18),
                // (51,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         y = d2.F(x);
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "d2.F(x)").WithLocation(51, 13));
        }

        [Fact]
        public void UnannotatedAssemblies_08()
        {
            var source0 =
@"public interface I
{
    object? F(object? o);
    object G(object o);
}";
            var source1 =
@"public class A : I
{
    object I.F(object o) => null;
    object I.G(object o) => null;
}
public class B : I
{
    public object F(object o) => null;
    public object G(object o) => null;
}
public class C
{
    public object F(object o) => null;
    public object G(object o) => null;
}
public class D : C
{
}";
            var source2 =
@"class P
{
    static void F(object o, A a)
    {
        ((I)a).F(o).ToString();
        ((I)a).G(null).ToString();
    }
    static void F(object o, B b)
    {
        b.F(o).ToString();
        b.G(null).ToString();
        ((I)b).F(o).ToString();
        ((I)b).G(null).ToString();
    }
    static void F(object o, D d)
    {
        d.F(o).ToString();
        d.G(null).ToString();
        ((I)d).F(o).ToString();
        ((I)d).G(null).ToString();
    }
}";
            var comp0 = CreateCompilation(new[] { source0 }, options: WithNullableEnable());
            comp0.VerifyDiagnostics();
            var ref0 = comp0.EmitToImageReference();

            var comp1 = CreateCompilation(source1, references: new[] { ref0 }, parseOptions: TestOptions.Regular7);
            comp1.VerifyDiagnostics();
            var ref1 = comp1.EmitToImageReference();

            var comp2A = CreateCompilation(source2, references: new[] { ref0, ref1 }, parseOptions: TestOptions.Regular7);
            comp2A.VerifyDiagnostics();

            var comp2B = CreateCompilation(source2, references: new[] { ref0, ref1 });
            comp2B.VerifyDiagnostics();

            var comp2C = CreateCompilation(new[] { source2 }, options: WithNullableEnable(), references: new[] { ref0, ref1 });
            comp2C.VerifyDiagnostics(
                // (5,9): warning CS8602: Dereference of a possibly null reference.
                //         ((I)a).F(o).ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "((I)a).F(o)").WithLocation(5, 9),
                // (6,18): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         ((I)a).G(null).ToString();
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(6, 18),
                // (12,9): warning CS8602: Dereference of a possibly null reference.
                //         ((I)b).F(o).ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "((I)b).F(o)").WithLocation(12, 9),
                // (13,18): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         ((I)b).G(null).ToString();
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(13, 18),
                // (19,9): warning CS8602: Dereference of a possibly null reference.
                //         ((I)d).F(o).ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "((I)d).F(o)").WithLocation(19, 9),
                // (20,18): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         ((I)d).G(null).ToString();
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(20, 18));

            var comp2D = CreateCompilation(new[] { source2 }, options: WithNullableDisable(), references: new[] { ref0, ref1 });
            comp2D.VerifyDiagnostics();
        }

        [Fact]
        public void UnannotatedAssemblies_09()
        {
            var source0 =
@"public abstract class A
{
    public abstract object? F(object x, object? y);
}";
            var source1 =
@"public abstract class B : A
{
    public abstract override object F(object x, object y);
    public abstract object G(object x, object y);
}";
            var source2 =
@"class C1 : B
{
    public override object F(object x, object y) => x;
    public override object G(object x, object y) => x;
}
class C2 : B
{
    public override object? F(object? x, object? y) => x;
    public override object? G(object? x, object? y) => x;
}
class P
{
    static void F(bool b, object? x, object y, C1 c)
    {
        if (b) c.F(x, y).ToString();
        if (b) c.G(x, y).ToString();
        ((B)c).F(x, y).ToString();
        ((B)c).G(x, y).ToString();
        ((A)c).F(x, y).ToString();
    }
    static void F(object? x, object y, C2 c)
    {
        c.F(x, y).ToString();
        c.G(x, y).ToString();
        ((B)c).F(x, y).ToString();
        ((B)c).G(x, y).ToString();
        ((A)c).F(x, y).ToString();
    }
}";
            var comp0 = CreateCompilation(source0);
            comp0.VerifyDiagnostics(
                // (3,47): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public abstract object? F(object x, object? y);
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(3, 47),
                // (3,27): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public abstract object? F(object x, object? y);
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(3, 27)
                );
            var ref0 = comp0.EmitToImageReference();

            var comp1 = CreateCompilation(source1, references: new[] { ref0 }, parseOptions: TestOptions.Regular7);
            comp1.VerifyDiagnostics();
            var ref1 = comp1.EmitToImageReference();

            var comp2 = CreateCompilation(source2, references: new[] { ref0, ref1 });
            comp2.VerifyDiagnostics(
                // (9,37): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public override object? G(object? x, object? y) => x;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(9, 37),
                // (9,48): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public override object? G(object? x, object? y) => x;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(9, 48),
                // (9,27): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public override object? G(object? x, object? y) => x;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(9, 27),
                // (21,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     static void F(object? x, object y, C2 c)
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(21, 25),
                // (13,33): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     static void F(bool b, object? x, object y, C1 c)
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(13, 33),
                // (8,37): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public override object? F(object? x, object? y) => x;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(8, 37),
                // (8,48): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public override object? F(object? x, object? y) => x;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(8, 48),
                // (8,27): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public override object? F(object? x, object? y) => x;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(8, 27)
                );

            comp0 = CreateCompilation(new[] { source0 }, options: WithNullableEnable());
            comp0.VerifyDiagnostics();
            ref0 = comp0.EmitToImageReference();

            comp1 = CreateCompilation(source1, references: new[] { ref0 }, parseOptions: TestOptions.Regular7);
            comp1.VerifyDiagnostics();
            ref1 = comp1.EmitToImageReference();

            comp2 = CreateCompilation(new[] { source2 }, options: WithNullableEnable(), references: new[] { ref0, ref1 });
            comp2.VerifyDiagnostics(
                // (15,20): warning CS8604: Possible null reference argument for parameter 'x' in 'object C1.F(object x, object y)'.
                //         if (b) c.F(x, y).ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x").WithArguments("x", "object C1.F(object x, object y)").WithLocation(15, 20),
                // (16,20): warning CS8604: Possible null reference argument for parameter 'x' in 'object C1.G(object x, object y)'.
                //         if (b) c.G(x, y).ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x").WithArguments("x", "object C1.G(object x, object y)").WithLocation(16, 20),
                // (19,18): warning CS8604: Possible null reference argument for parameter 'x' in 'object? A.F(object x, object? y)'.
                //         ((A)c).F(x, y).ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x").WithArguments("x", "object? A.F(object x, object? y)").WithLocation(19, 18),
                // (19,9): warning CS8602: Dereference of a possibly null reference.
                //         ((A)c).F(x, y).ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "((A)c).F(x, y)").WithLocation(19, 9),
                // (23,9): warning CS8602: Dereference of a possibly null reference.
                //         c.F(x, y).ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c.F(x, y)").WithLocation(23, 9),
                // (24,9): warning CS8602: Dereference of a possibly null reference.
                //         c.G(x, y).ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c.G(x, y)").WithLocation(24, 9),
                // (27,18): warning CS8604: Possible null reference argument for parameter 'x' in 'object? A.F(object x, object? y)'.
                //         ((A)c).F(x, y).ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x").WithArguments("x", "object? A.F(object x, object? y)").WithLocation(27, 18),
                // (27,9): warning CS8602: Dereference of a possibly null reference.
                //         ((A)c).F(x, y).ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "((A)c).F(x, y)").WithLocation(27, 9));
        }

        [Fact]
        public void UnannotatedAssemblies_10()
        {
            var source0 =
@"public abstract class A<T>
{
    public T F;
}
public sealed class B : A<object>
{
}";
            var source1 =
@"class C
{
    static void Main()
    {
        B b = new B();
        b.F = null;
    }
}";
            var comp0 = CreateCompilation(source0, parseOptions: TestOptions.Regular7);
            comp0.VerifyDiagnostics();

            var comp1 = CreateCompilation(source1, references: new MetadataReference[] { new CSharpCompilationReference(comp0) });
            comp1.VerifyDiagnostics();

            comp1 = CreateCompilation(source1, references: new[] { comp0.EmitToImageReference() });
            comp1.VerifyDiagnostics();
        }

        [Fact]
        public void Embedded_WithObsolete()
        {
            string source = @"
namespace Microsoft.CodeAnalysis
{
    [Embedded]
    [System.Obsolete(""obsolete"")]
    class EmbeddedAttribute : System.Attribute
    {
        public EmbeddedAttribute() { }
    }
}
";
            var comp = CreateCompilation(new[] { source });
            comp.VerifyDiagnostics();
            Assert.False(comp.GetMember("Microsoft.CodeAnalysis.EmbeddedAttribute").IsImplicitlyDeclared);
        }

        [Fact]
        public void NonNullTypes_Cycle5()
        {
            string source = @"
namespace System.Runtime.CompilerServices
{
    [AttributeUsage(AttributeTargets.Class, AllowMultiple = false)]
    class SomeAttribute : Attribute
    {
        public SomeAttribute() { }
        public int Property { get; set; }
    }
}
";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void NonNullTypes_Cycle12()
        {
            string source = @"
[System.Flags]
enum E { }
";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void NonNullTypes_Cycle13()
        {
            string source = @"
interface I { }

[System.Obsolete(nameof(I2))]
interface I2 : I { }

";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void NonNullTypes_Cycle15()
        {
            string lib_cs = "public class Base { }";
            var lib = CreateCompilation(lib_cs, assemblyName: "lib");

            string lib2_cs = "public class C : Base { }";
            var lib2 = CreateCompilation(lib2_cs, references: new[] { lib.EmitToImageReference() }, assemblyName: "lib2");

            string source_cs = @"
[D]
class DAttribute : C { }
";
            var comp = CreateCompilation(source_cs, references: new[] { lib2.EmitToImageReference() });
            comp.VerifyDiagnostics(
                // (3,20): error CS0012: The type 'Base' is defined in an assembly that is not referenced. You must add a reference to assembly 'lib, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
                // class DAttribute : C { }
                Diagnostic(ErrorCode.ERR_NoTypeDef, "C").WithArguments("Base", "lib, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(3, 20),
                // (2,2): error CS0012: The type 'Base' is defined in an assembly that is not referenced. You must add a reference to assembly 'lib, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
                // [D]
                Diagnostic(ErrorCode.ERR_NoTypeDef, "D").WithArguments("Base", "lib, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(2, 2),
                // (2,2): error CS0012: The type 'Base' is defined in an assembly that is not referenced. You must add a reference to assembly 'lib, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null'.
                // [D]
                Diagnostic(ErrorCode.ERR_NoTypeDef, "D").WithArguments("Base", "lib, Version=0.0.0.0, Culture=neutral, PublicKeyToken=null").WithLocation(2, 2)
                );
        }

        [Fact]
        public void NonNullTypes_Cycle16()
        {
            string source = @"
using System;
[AttributeUsage(AttributeTargets.Property)]
class AttributeWithProperty : System.ComponentModel.DisplayNameAttribute
{
    public override string DisplayName { get => throw null!; }
}
";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void NonNullTypes_OnFields()
        {
            var obliviousLib = @"
public class Oblivious
{
    public static string s;
}
";

            var obliviousComp = CreateCompilation(obliviousLib, parseOptions: TestOptions.Regular7);

            var lib = @"
#nullable enable
#pragma warning disable 8618
public class External
{
    public static string s;
    public static string? ns;

#nullable disable
    public static string fs;

#nullable disable
    public static string? fns;
}
";

            var libComp = CreateCompilation(new[] { lib }, options: WithNullableEnable());
            libComp.VerifyDiagnostics(
                // (13,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public static string? fns;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(13, 25)
                );

            var source = @"


#nullable disable
public class OuterA
{
#nullable enable
    public class A
    {
        public static string s;
        public static string? ns;
    }
}

// NonNullTypes(true) by default
public class B
{
    public static string s;
    public static string? ns;
}

#nullable disable
public class C
{
#nullable enable
    public static string s;

#nullable enable
    public static string? ns;
}

#nullable disable
public class OuterD
{
    public class D
    {
#nullable enable
        public static string s;
#nullable enable
        public static string? ns;
    }
}

public class Oblivious2
{
#nullable disable
    public static string s;
#nullable disable
    public static string? ns;
}
#nullable enable
class E
{
    public void M()
    {
        Oblivious.s /*T:string!*/ = null;

        External.s /*T:string!*/ = null; // warn 1
        External.ns /*T:string?*/ = null;

        External.fs /*T:string!*/ = null;
        External.fns /*T:string?*/ = null;

        OuterA.A.s /*T:string!*/ = null; // warn 2
        OuterA.A.ns /*T:string?*/ = null;

        B.s /*T:string!*/ = null; // warn 3
        B.ns /*T:string?*/ = null;

        C.s /*T:string!*/ = null; // warn 4
        C.ns /*T:string?*/ = null;

        OuterD.D.s /*T:string!*/ = null; // warn 5
        OuterD.D.ns /*T:string?*/ = null;

        Oblivious2.s /*T:string!*/ = null;
        Oblivious2.ns /*T:string?*/ = null;
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(),
                parseOptions: TestOptions.Regular8, references: new[] { obliviousComp.EmitToImageReference(), libComp.EmitToImageReference() });

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (10,30): warning CS8618: Non-nullable field 's' is uninitialized. Consider declaring the field as nullable.
                //         public static string s;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("field", "s").WithLocation(10, 30),
                // (18,26): warning CS8618: Non-nullable field 's' is uninitialized. Consider declaring the field as nullable.
                //     public static string s;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("field", "s").WithLocation(18, 26),
                // (26,26): warning CS8618: Non-nullable field 's' is uninitialized. Consider declaring the field as nullable.
                //     public static string s;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("field", "s").WithLocation(26, 26),
                // (38,30): warning CS8618: Non-nullable field 's' is uninitialized. Consider declaring the field as nullable.
                //         public static string s;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("field", "s").WithLocation(38, 30),
                // (49,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public static string? ns;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(49, 25),
                // (58,36): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         External.s /*T:string!*/ = null; // warn 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(58, 36),
                // (64,36): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterA.A.s /*T:string!*/ = null; // warn 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(64, 36),
                // (67,29): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         B.s /*T:string!*/ = null; // warn 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(67, 29),
                // (70,29): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         C.s /*T:string!*/ = null; // warn 4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(70, 29),
                // (73,36): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterD.D.s /*T:string!*/ = null; // warn 5
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(73, 36)
                );
        }

        [Fact]
        public void SuppressedNullConvertedToUnconstrainedT()
        {
            var source = @"
public class List2<T> { public T Item { get; set; } = null!; }
";

            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (2,55): error CS0403: Cannot convert null to type parameter 'T' because it could be a non-nullable value type. Consider using 'default(T)' instead.
                // public class List2<T> { public T Item { get; set; } = null!; }
                Diagnostic(ErrorCode.ERR_TypeVarCantBeNull, "null").WithArguments("T").WithLocation(2, 55)
                );
        }

        [Fact]
        public void NonNullTypes_OnFields_Nested()
        {
            var obliviousLib = @"
public class List1<T> { public T Item { get; set; } = default(T); }
public class Oblivious
{
    public static List1<string> s;
}
";

            var obliviousComp = CreateCompilation(obliviousLib, parseOptions: TestOptions.Regular7);

            var lib = @"
#pragma warning disable 8618
using System.Diagnostics.CodeAnalysis;
public class List2<T> { public T Item { get; set; } = default!; }
public class External
{
    public static List2<string> s;
    public static List2<string?> ns;

#nullable disable
    public static List2<string> fs;
#nullable disable
    public static List2<string?> fns;
}
";

            var libComp = CreateCompilation(new[] { lib }, options: WithNullableEnable());

            var source = @"

public class List3<T> { public T Item { get; set; } = default!; }

#nullable disable
public class OuterA
{
#nullable enable
    public class A
    {
        public static List3<string> s;
        public static List3<string?> ns;
    }
}

// NonNullTypes(true) by default
public class B
{
    public static List3<string> s;
    public static List3<string?> ns;
}

#nullable disable
public class OuterD
{
    public class D
    {
#nullable enable
        public static List3<string> s;
#nullable enable
        public static List3<string?> ns;
    }
}

#nullable disable
public class Oblivious2
{
    public static List3<string> s;
    public static List3<string?> ns;
}
#nullable enable
class E
{
    public void M()
    {
        Oblivious.s.Item /*T:string!*/ = null;

        External.s.Item /*T:string!*/ = null; // warn 1
        External.ns.Item /*T:string?*/ = null;

        External.fs.Item /*T:string!*/ = null;
        External.fns.Item /*T:string?*/ = null;

        OuterA.A.s.Item /*T:string!*/ = null; // warn 2
        OuterA.A.ns.Item /*T:string?*/ = null;

        B.s.Item /*T:string!*/ = null; // warn 3
        B.ns.Item /*T:string?*/ = null;

        OuterD.D.s.Item /*T:string!*/ = null; // warn 4
        OuterD.D.ns.Item /*T:string?*/ = null;

        Oblivious2.s.Item /*T:string!*/ = null;
        Oblivious2.ns.Item /*T:string?*/ = null;
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(),
                parseOptions: TestOptions.Regular8, references: new[] { obliviousComp.EmitToImageReference(), libComp.EmitToImageReference() });

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (11,37): warning CS8618: Non-nullable field 's' is uninitialized. Consider declaring the field as nullable.
                //         public static List3<string> s;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("field", "s").WithLocation(11, 37),
                // (12,38): warning CS8618: Non-nullable field 'ns' is uninitialized. Consider declaring the field as nullable.
                //         public static List3<string?> ns;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "ns").WithArguments("field", "ns").WithLocation(12, 38),
                // (19,33): warning CS8618: Non-nullable field 's' is uninitialized. Consider declaring the field as nullable.
                //     public static List3<string> s;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("field", "s").WithLocation(19, 33),
                // (20,34): warning CS8618: Non-nullable field 'ns' is uninitialized. Consider declaring the field as nullable.
                //     public static List3<string?> ns;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "ns").WithArguments("field", "ns").WithLocation(20, 34),
                // (29,37): warning CS8618: Non-nullable field 's' is uninitialized. Consider declaring the field as nullable.
                //         public static List3<string> s;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("field", "s").WithLocation(29, 37),
                // (31,38): warning CS8618: Non-nullable field 'ns' is uninitialized. Consider declaring the field as nullable.
                //         public static List3<string?> ns;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "ns").WithArguments("field", "ns").WithLocation(31, 38),
                // (39,31): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public static List3<string?> ns;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(39, 31),
                // (48,41): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         External.s.Item /*T:string!*/ = null; // warn 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(48, 41),
                // (54,41): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterA.A.s.Item /*T:string!*/ = null; // warn 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(54, 41),
                // (57,34): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         B.s.Item /*T:string!*/ = null; // warn 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(57, 34),
                // (60,41): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterD.D.s.Item /*T:string!*/ = null; // warn 4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(60, 41));
        }

        [Fact]
        public void NonNullTypes_OnFields_Tuples()
        {
            var obliviousLib = @"
public class Oblivious
{
    public static (string s, string s2) t;
}
";

            var obliviousComp = CreateCompilation(obliviousLib, parseOptions: TestOptions.Regular7);

            var lib = @"
public class External
{
    public static (string s, string? ns) t;
}
";

            var libComp = CreateCompilation(new[] { lib }, options: WithNullableEnable());

            var source = @"


#nullable disable
public class OuterA
{
#nullable enable
    public class A
    {
        public static (string s, string? ns) t;
    }
}

// NonNullTypes(true) by default
public class B
{
    public static (string s, string? ns) t;
}

#nullable disable
public class OuterD
{
    public class D
    {
#nullable enable
        public static (string s, string? ns) t;
    }
}

#nullable disable
public class Oblivious2
{
    public static (string s, string? ns) t;
}
#nullable enable
class E
{
    public void M()
    {
        Oblivious.t.s /*T:string!*/ = null;

        External.t.s /*T:string!*/ = null; // warn 1
        External.t.ns /*T:string?*/ = null;

        OuterA.A.t.s /*T:string!*/ = null; // warn 2
        OuterA.A.t.ns /*T:string?*/ = null;

        B.t.s /*T:string!*/ = null; // warn 3
        B.t.ns /*T:string?*/ = null;

        OuterD.D.t.s /*T:string!*/ = null; // warn 4
        OuterD.D.t.ns /*T:string?*/ = null;

        Oblivious2.t.s /*T:string!*/ = null;
        Oblivious2.t.ns /*T:string?*/ = null;
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(),
                parseOptions: TestOptions.Regular8, references: new[] { obliviousComp.EmitToImageReference(), libComp.EmitToImageReference() });

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (33,36): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public static (string s, string? ns) t;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(33, 36),
                // (42,38): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         External.t.s /*T:string!*/ = null; // warn 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(42, 38),
                // (45,38): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterA.A.t.s /*T:string!*/ = null; // warn 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(45, 38),
                // (48,31): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         B.t.s /*T:string!*/ = null; // warn 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(48, 31),
                // (51,38): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterD.D.t.s /*T:string!*/ = null; // warn 4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(51, 38)
                );
        }

        [Fact]
        public void NonNullTypes_OnFields_Arrays()
        {
            var obliviousLib = @"
public class Oblivious
{
    public static string[] s;
}
";

            var obliviousComp = CreateCompilation(obliviousLib, parseOptions: TestOptions.Regular7);

            var lib = @"
#pragma warning disable 8618
public class External
{
    public static string[] s;
    public static string?[] ns;
}
";

            var libComp = CreateCompilation(new[] { lib }, options: WithNullableEnable());

            var source = @"


#nullable disable
public class OuterA
{
#nullable enable
    public class A
    {
        public static string[] s;
        public static string?[] ns;
    }
}

// NonNullTypes(true) by default
public class B
{
    public static string[] s;
    public static string?[] ns;
}

#nullable disable
public class OuterD
{
    public class D
    {
#nullable enable
        public static string[] s;
#nullable enable
        public static string?[] ns;
    }
}

#nullable disable
public class Oblivious2
{
    public static string[] s;
    public static string?[] ns;
}
#nullable enable
class E
{
    public void M()
    {
        Oblivious.s[0] /*T:string!*/ = null;

        External.s[0] /*T:string!*/ = null; // warn 1
        External.ns[0] /*T:string?*/ = null;

        OuterA.A.s[0] /*T:string!*/ = null; // warn 2
        OuterA.A.ns[0] /*T:string?*/ = null;

        B.s[0] /*T:string!*/ = null; // warn 3
        B.ns[0] /*T:string?*/ = null;

        OuterD.D.s[0] /*T:string!*/ = null; // warn 4
        OuterD.D.ns[0] /*T:string?*/ = null;

        Oblivious2.s[0] /*T:string!*/ = null;
        Oblivious2.ns[0] /*T:string?*/ = null;
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(),
                parseOptions: TestOptions.Regular8, references: new[] { obliviousComp.EmitToImageReference(), libComp.EmitToImageReference() });

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (10,32): warning CS8618: Non-nullable field 's' is uninitialized. Consider declaring the field as nullable.
                //         public static string[] s;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("field", "s").WithLocation(10, 32),
                // (11,33): warning CS8618: Non-nullable field 'ns' is uninitialized. Consider declaring the field as nullable.
                //         public static string?[] ns;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "ns").WithArguments("field", "ns").WithLocation(11, 33),
                // (18,28): warning CS8618: Non-nullable field 's' is uninitialized. Consider declaring the field as nullable.
                //     public static string[] s;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("field", "s").WithLocation(18, 28),
                // (19,29): warning CS8618: Non-nullable field 'ns' is uninitialized. Consider declaring the field as nullable.
                //     public static string?[] ns;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "ns").WithArguments("field", "ns").WithLocation(19, 29),
                // (28,32): warning CS8618: Non-nullable field 's' is uninitialized. Consider declaring the field as nullable.
                //         public static string[] s;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("field", "s").WithLocation(28, 32),
                // (30,33): warning CS8618: Non-nullable field 'ns' is uninitialized. Consider declaring the field as nullable.
                //         public static string?[] ns;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "ns").WithArguments("field", "ns").WithLocation(30, 33),
                // (38,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public static string?[] ns;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(38, 25),
                // (47,39): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         External.s[0] /*T:string!*/ = null; // warn 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(47, 39),
                // (50,39): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterA.A.s[0] /*T:string!*/ = null; // warn 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(50, 39),
                // (53,32): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         B.s[0] /*T:string!*/ = null; // warn 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(53, 32),
                // (56,39): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterD.D.s[0] /*T:string!*/ = null; // warn 4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(56, 39)
                );
        }

        [Fact]
        public void NonNullTypes_OnProperties()
        {
            var obliviousLib = @"
public class Oblivious
{
    public static string s { get; set; }
}
";

            var obliviousComp = CreateCompilation(obliviousLib, parseOptions: TestOptions.Regular7);
            obliviousComp.VerifyDiagnostics();

            var lib = @"
#pragma warning disable 8618
public class External
{
    public static string s { get; set; }
    public static string? ns { get; set; }
}
";

            var libComp = CreateCompilation(new[] { lib }, options: WithNullableEnable());

            var source = @"


#nullable disable
public class OuterA
{
    public class A
    {
#nullable enable
        public static string s { get; set; }
#nullable enable
        public static string? ns { get; set; }
    }
}

// NonNullTypes(true) by default
public class B
{
    public static string s { get; set; }
    public static string? ns { get; set; }
}

#nullable disable
public class OuterD
{
#nullable enable
    public class D
    {
        public static string s { get; set; }
        public static string? ns { get; set; }
    }
}

public class Oblivious2
{
#nullable disable
    public static string s { get; set; }
#nullable disable
    public static string? ns { get; set; }
}
#nullable enable
class E
{
    public void M()
    {
        Oblivious.s /*T:string!*/ = null;

        External.s /*T:string!*/ = null; // warn 1
        External.ns /*T:string?*/ = null;

        OuterA.A.s /*T:string!*/ = null; // warn 2
        OuterA.A.ns /*T:string?*/ = null;

        B.s /*T:string!*/ = null; // warn 3
        B.ns /*T:string?*/ = null;

        OuterD.D.s /*T:string!*/ = null; // warn 4
        OuterD.D.ns /*T:string?*/ = null;

        Oblivious2.s /*T:string!*/ = null;
        Oblivious2.ns /*T:string?*/ = null;
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(),
                parseOptions: TestOptions.Regular8, references: new[] { obliviousComp.EmitToImageReference(), libComp.EmitToImageReference() });

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (10,30): warning CS8618: Non-nullable property 's' is uninitialized. Consider declaring the property as nullable.
                //         public static string s { get; set; }
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("property", "s").WithLocation(10, 30),
                // (19,26): warning CS8618: Non-nullable property 's' is uninitialized. Consider declaring the property as nullable.
                //     public static string s { get; set; }
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("property", "s").WithLocation(19, 26),
                // (29,30): warning CS8618: Non-nullable property 's' is uninitialized. Consider declaring the property as nullable.
                //         public static string s { get; set; }
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "s").WithArguments("property", "s").WithLocation(29, 30),
                // (39,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public static string? ns { get; set; }
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(39, 25),
                // (48,36): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         External.s /*T:string!*/ = null; // warn 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(48, 36),
                // (51,36): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterA.A.s /*T:string!*/ = null; // warn 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(51, 36),
                // (54,29): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         B.s /*T:string!*/ = null; // warn 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(54, 29),
                // (57,36): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterD.D.s /*T:string!*/ = null; // warn 4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(57, 36)
                );
        }

        [Fact]
        public void NonNullTypes_OnMethods()
        {
            var obliviousLib = @"
public class Oblivious
{
    public static string Method(string s) => throw null;
}
";

            var obliviousComp = CreateCompilation(obliviousLib, parseOptions: TestOptions.Regular7);

            var lib = @"
public class External
{
    public static string Method(string s) => throw null!;
    public static string? NMethod(string? ns) => throw null!;
}
";

            var libComp = CreateCompilation(new[] { lib }, options: WithNullableEnable());

            var source = @"


#nullable disable
public class OuterA
{
#nullable enable
    public class A
    {
        public static string Method(string s) => throw null!;
        public static string? NMethod(string? ns) => throw null!;
    }
}

// NonNullTypes(true) by default
public class B
{
    public static string Method(string s) => throw null!;
    public static string? NMethod(string? ns) => throw null!;
}

#nullable disable
public class OuterD
{
    public class D
    {
#nullable enable
        public static string Method(string s) => throw null!;
#nullable enable
        public static string? NMethod(string? ns) => throw null!;
    }
}

#nullable disable
public class Oblivious2
{
    public static string Method(string s) => throw null!;
    public static string? NMethod(string? ns) => throw null!;
}
#nullable enable
class E
{
    public void M()
    {
        Oblivious.Method(null) /*T:string!*/;

        External.Method(null) /*T:string!*/; // warn 1
        External.NMethod(null) /*T:string?*/;

        OuterA.A.Method(null) /*T:string!*/; // warn 2
        OuterA.A.NMethod(null) /*T:string?*/;

        B.Method(null) /*T:string!*/; // warn 3
        B.NMethod(null) /*T:string?*/;

        OuterD.D.Method(null) /*T:string!*/; // warn 4
        OuterD.D.NMethod(null) /*T:string?*/;

        Oblivious2.Method(null) /*T:string!*/;
        Oblivious2.NMethod(null) /*T:string?*/;
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(),
                parseOptions: TestOptions.Regular8, references: new[] { obliviousComp.EmitToImageReference(), libComp.EmitToImageReference() });

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (38,41): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public static string? NMethod(string? ns) => throw null!;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(38, 41),
                // (38,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public static string? NMethod(string? ns) => throw null!;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(38, 25),
                // (47,25): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         External.Method(null) /*T:string!*/; // warn 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(47, 25),
                // (50,25): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterA.A.Method(null) /*T:string!*/; // warn 2
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(50, 25),
                // (53,18): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         B.Method(null) /*T:string!*/; // warn 3
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(53, 18),
                // (56,25): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         OuterD.D.Method(null) /*T:string!*/; // warn 4
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(56, 25)
                );
        }

        [Fact]
        public void NonNullTypes_OnModule()
        {
            var obliviousLib =
@"#nullable disable
public class Oblivious { }
";

            var obliviousComp = CreateCompilation(new[] { obliviousLib });
            obliviousComp.VerifyDiagnostics();

            var compilation = CreateCompilation("", options: TestOptions.ReleaseDll,
                parseOptions: TestOptions.Regular8, references: new[] { obliviousComp.EmitToImageReference() });
            compilation.VerifyDiagnostics();
        }

        [Fact]
        public void NonNullTypes_ValueTypeArgument()
        {
            var source =
@"#nullable disable
class A<T> { }
class B
{
    A<byte> P { get; }
}";
            var comp = CreateCompilation(new[] { source });
            comp.VerifyDiagnostics();
        }

        [WorkItem(28324, "https://github.com/dotnet/roslyn/issues/28324")]
        [Fact]
        public void NonNullTypes_GenericOverriddenMethod_ValueType()
        {
            var source =
@"#nullable disable
class C<T> { }
abstract class A
{
    internal abstract C<T> F<T>() where T : struct;
}
class B : A
{
    internal override C<T> F<T>() => throw null!;
}";
            var comp = CreateCompilation(new[] { source });
            comp.VerifyDiagnostics();

            var method = comp.GetMember<MethodSymbol>("A.F");
            var typeArg = ((NamedTypeSymbol)method.ReturnType).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0];
            Assert.True(typeArg.Type.IsValueType);
            Assert.Equal(NullableAnnotation.Oblivious, typeArg.NullableAnnotation);

            method = comp.GetMember<MethodSymbol>("B.F");
            typeArg = ((NamedTypeSymbol)method.ReturnType).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0];
            Assert.True(typeArg.Type.IsValueType);
            Assert.Equal(NullableAnnotation.Oblivious, typeArg.NullableAnnotation);

            // https://github.com/dotnet/roslyn/issues/29843: Test all combinations of base and derived
            // including explicit Nullable<T>.
        }

        // BoundExpression.Type for Task.FromResult(_f[0]) is Task<T!>
        // but the inferred type is Task<T~>.
        [Fact]
        public void CompareUnannotatedAndNonNullableTypeParameter()
        {
            var source =
@"#pragma warning disable 0649
using System.Threading.Tasks;
class C<T>
{
    T[] _f;
    Task<T> F() => Task.FromResult(_f[0]);
}";
            var comp = CreateCompilation(new[] { source }, parseOptions: TestOptions.Regular7);
            comp.VerifyDiagnostics();

            comp = CreateCompilation(new[] { source });
            comp.VerifyDiagnostics();

            comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (5,9): warning CS8618: Non-nullable field '_f' is uninitialized. Consider declaring the field as nullable.
                //     T[] _f;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "_f").WithArguments("field", "_f").WithLocation(5, 9)
                );
        }

        [Fact]
        public void CircularConstraints()
        {
            var source =
@"class A<T> where T : B<T>.I
{
    internal interface I { }
}
class B<T> : A<T> where T : A<T>.I
{
}";
            var comp = CreateCompilation(new[] { source }, parseOptions: TestOptions.Regular7);
            comp.VerifyDiagnostics();

            comp = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular7, skipUsesIsNullable: true);
            comp.VerifyDiagnostics(
                // error CS8630: Invalid 'NullableContextOptions' value: 'Enable' for C# 7.0. Please use language version '8.0' or greater.
                Diagnostic(ErrorCode.ERR_NullableOptionNotAvailable).WithArguments("NullableContextOptions", "Enable", "7.0", "8.0").WithLocation(1, 1)
                );

            comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact, WorkItem(27686, "https://github.com/dotnet/roslyn/issues/27686")]
        public void AssignObliviousIntoLocals()
        {
            var obliviousLib = @"
public class Oblivious
{
    public static string f;
}
";

            var obliviousComp = CreateCompilation(obliviousLib, parseOptions: TestOptions.Regular7);

            var source = @"
class C
{
    void M()
    {
        string s = Oblivious.f;
        s /*T:string!*/ .ToString();
        string ns = Oblivious.f;
        ns /*T:string!*/ .ToString();
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(),
                parseOptions: TestOptions.Regular8, references: new[] { obliviousComp.EmitToImageReference() });

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics();
        }

        [Fact, WorkItem(37309, "https://github.com/dotnet/roslyn/issues/37309")]
        public void NonNullTypesTrue_Foreach()
        {
            var source = @"
class C
{
#nullable enable
    public void M2()
    {
        foreach (string s in Collection())
        {
            s /*T:string!*/ .ToString();
        }

        foreach (string? ns in NCollection())
        {
            ns /*T:string?*/ .ToString(); // 1
        }

        foreach (var s1 in Collection())
        {
            s1 /*T:string!*/ .ToString();
        }

        foreach (var ns1 in NCollection())
        {
            ns1 /*T:string?*/ .ToString(); // 2
        }

        foreach (string s in FalseCollection())
        {
            s /*T:string!*/ .ToString();
        }

        foreach (string? ns in FalseNCollection())
        {
            ns /*T:string?*/ .ToString(); // 3
        }

        foreach (var s1 in FalseCollection())
        {
            s1 /*T:string!*/ .ToString();
        }

        foreach (var ns1 in FalseNCollection())
        {
            ns1 /*T:string?*/ .ToString(); // 4
        }
    }

#nullable enable
    string[] Collection() => throw null!;

#nullable enable
    string?[] NCollection() => throw null!;

#nullable disable
    string[] FalseCollection() => throw null!;

#nullable disable
    string?[] FalseNCollection() => throw null!; // 5
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (58,11): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     string?[] FalseNCollection() => throw null!; // 5
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(58, 11),
                // (14,13): warning CS8602: Dereference of a possibly null reference.
                //             ns /*T:string?*/ .ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns").WithLocation(14, 13),
                // (24,13): warning CS8602: Dereference of a possibly null reference.
                //             ns1 /*T:string?*/ .ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns1").WithLocation(24, 13),
                // (34,13): warning CS8602: Dereference of a possibly null reference.
                //             ns /*T:string?*/ .ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns").WithLocation(34, 13),
                // (44,13): warning CS8602: Dereference of a possibly null reference.
                //             ns1 /*T:string?*/ .ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns1").WithLocation(44, 13)
                );

            var tree = compilation.SyntaxTrees.Single();
            var model = compilation.GetSemanticModel(tree);
            var foreachSyntax = tree.GetRoot().DescendantNodes().OfType<ForEachStatementSyntax>().ToArray();

            var type1 = foreachSyntax[0].Type;
            Assert.Equal("string", type1.ToString());
            Assert.Equal("System.String!", model.GetTypeInfo(type1).Type.ToTestDisplayString(includeNonNullable: true));

            var type2 = foreachSyntax[1].Type;
            Assert.Equal("string?", type2.ToString());
            Assert.Equal("System.String?", model.GetTypeInfo(type2).Type.ToTestDisplayString(includeNonNullable: true));
        }

        [WorkItem(30840, "https://github.com/dotnet/roslyn/issues/30840")]
        [Fact]
        public void NonNullTypesFalse_Foreach()
        {
            var source = @"


class C
{
#nullable disable
    public void M2()
    {
        foreach (string s in Collection())
        {
            s /*T:string!*/ .ToString();
        }

        foreach (string? ns in NCollection()) // 1
        {
            ns /*T:string?*/ .ToString();
        }

        foreach (var s1 in Collection())
        {
            s1 /*T:string!*/ .ToString();
        }

        foreach (var ns1 in NCollection())
        {
            ns1 /*T:string?*/ .ToString();
        }

        foreach (string s in FalseCollection())
        {
            s /*T:string!*/ .ToString();
        }

        foreach (string? ns in FalseNCollection()) // 2
        {
            ns /*T:string?*/ .ToString();
        }

        foreach (var s1 in FalseCollection())
        {
            s1 /*T:string!*/ .ToString();
        }

        foreach (var ns1 in FalseNCollection())
        {
            ns1 /*T:string?*/ .ToString();
        }
    }

#nullable enable
    string[] Collection() => throw null!;

#nullable enable
    string?[] NCollection() => throw null!;

#nullable disable
    string[] FalseCollection() => throw null!;

#nullable disable
    string?[] FalseNCollection() => throw null!; // 3
}
";
            var compilation = CreateCompilation(source, parseOptions: TestOptions.Regular.WithFeature("run-nullable-analysis", "always"), options: WithNullableEnable());

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (60,11): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     string?[] FalseNCollection() => throw null!; // 3
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(60, 11),
                // (14,24): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //         foreach (string? ns in NCollection()) // 1
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(14, 24),
                // (34,24): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //         foreach (string? ns in FalseNCollection()) // 2
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(34, 24)
                );
        }

        [Fact]
        public void NonNullTypesTrue_OutVars()
        {
            var source = @"


class C
{
#nullable enable
    public void M()
    {
        Out(out string s2);
        s2 /*T:string!*/ .ToString();
        s2 = null; // 1

        NOut(out string? ns2);
        ns2 /*T:string?*/ .ToString(); // 2
        ns2 = null;

        FalseOut(out string s3);
        s3 /*T:string!*/ .ToString();
        s3 = null; // 3

        FalseNOut(out string? ns3);
        ns3 /*T:string?*/ .ToString(); // 4
        ns3 = null;

        Out(out var s4);
        s4 /*T:string!*/ .ToString();
        s4 = null;

        NOut(out var ns4);
        ns4 /*T:string?*/ .ToString(); // 5
        ns4 = null;

        FalseOut(out var s5);
        s5 /*T:string!*/ .ToString();
        s5 = null;

        FalseNOut(out var ns5);
        ns5 /*T:string?*/ .ToString(); // 6
        ns5 = null;
    }

#nullable enable
    void Out(out string s) => throw null!;

#nullable enable
    void NOut(out string? ns) => throw null!;

#nullable disable
    void FalseOut(out string s) => throw null!;

#nullable disable
    void FalseNOut(out string? ns) => throw null!; // 7
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (52,30): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     void FalseNOut(out string? ns) => throw null!; // 7
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(52, 30),
                // (11,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         s2 = null; // 1
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(11, 14),
                // (14,9): warning CS8602: Dereference of a possibly null reference.
                //         ns2 /*T:string?*/ .ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns2").WithLocation(14, 9),
                // (19,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         s3 = null; // 3
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(19, 14),
                // (22,9): warning CS8602: Dereference of a possibly null reference.
                //         ns3 /*T:string?*/ .ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns3").WithLocation(22, 9),
                // (30,9): warning CS8602: Dereference of a possibly null reference.
                //         ns4 /*T:string?*/ .ToString(); // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns4").WithLocation(30, 9),
                // (38,9): warning CS8602: Dereference of a possibly null reference.
                //         ns5 /*T:string?*/ .ToString(); // 6
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns5").WithLocation(38, 9)
                );
        }

        [WorkItem(30840, "https://github.com/dotnet/roslyn/issues/30840")]
        [Fact]
        public void NonNullTypesFalse_OutVars()
        {
            var source = @"


class C
{
#nullable disable
    public void M()
    {
        Out(out string s2);
        s2 /*T:string!*/ .ToString();
        s2 = null;

        NOut(out string? ns2); // 1
        ns2 /*T:string?*/ .ToString(); // 2
        ns2 = null;

        FalseOut(out string s3);
        s3 /*T:string!*/ .ToString();
        s3 = null;

        FalseNOut(out string? ns3); // 3
        ns3 /*T:string?*/ .ToString(); // 4
        ns3 = null;

        Out(out var s4);
        s4 /*T:string!*/ .ToString();
        s4 = null; // 5

        NOut(out var ns4);
        ns4 /*T:string?*/ .ToString(); // 6
        ns4 = null;

        FalseOut(out var s5);
        s5 /*T:string!*/ .ToString();
        s5 = null;

        FalseNOut(out var ns5);
        ns5 /*T:string?*/ .ToString(); // 7
        ns5 = null;
    }

#nullable enable
    void Out(out string s) => throw null!;

#nullable enable
    void NOut(out string? ns) => throw null!;

#nullable disable
    void FalseOut(out string s) => throw null!;

#nullable disable
    void FalseNOut(out string? ns) => throw null!; // 8
}
";
            var compilation = CreateCompilation(source, parseOptions: TestOptions.Regular.WithFeature("run-nullable-analysis", "always"), options: WithNullableEnable());

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (52,30): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     void FalseNOut(out string? ns) => throw null!; // 8
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(52, 30),
                // (13,24): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //         NOut(out string? ns2); // 1
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(13, 24),
                // (21,29): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //         FalseNOut(out string? ns3); // 3
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(21, 29)
                );
        }

        [Fact]
        public void NonNullTypesTrue_LocalDeclarations()
        {
            var source = @"


#nullable enable
public class C : Base
{
    public void M()
    {
        string s2 = Method();
        s2 /*T:string!*/ .ToString();
        s2 = null; // warn 1

        string? ns2 = NMethod();
        ns2 /*T:string?*/ .ToString(); // warn 2
        ns2 = null;

        string s3 = FalseMethod();
        s3 /*T:string!*/ .ToString();
        s3 = null; // warn 3

        string? ns3 = FalseNMethod();
        ns3 /*T:string?*/ .ToString(); // warn 4
        ns3 = null;

        var s4 = Method();
        s4 /*T:string!*/ .ToString();
        s4 = null;

        var ns4 = NMethod();
        ns4 /*T:string?*/ .ToString(); // warn 5
        ns4 = null;

        var s5 = FalseMethod();
        s5 /*T:string!*/ .ToString();
        s5 = null;

        var ns5 = FalseNMethod();
        ns5 /*T:string?*/ .ToString(); // warn 6
        ns5 = null;
    }
}
public class Base
{
#nullable enable
    public string Method() => throw null!;

#nullable enable
    public string? NMethod() => throw null!;

#nullable disable
    public string FalseMethod() => throw null!;

#nullable disable
    public string? FalseNMethod() => throw null!; // warn 7
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (54,18): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public string? FalseNMethod() => throw null!; // warn 7
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(54, 18),
                // (11,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         s2 = null; // warn 1
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(11, 14),
                // (14,9): warning CS8602: Dereference of a possibly null reference.
                //         ns2 /*T:string?*/ .ToString(); // warn 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns2").WithLocation(14, 9),
                // (19,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         s3 = null; // warn 3
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(19, 14),
                // (22,9): warning CS8602: Dereference of a possibly null reference.
                //         ns3 /*T:string?*/ .ToString(); // warn 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns3").WithLocation(22, 9),
                // (30,9): warning CS8602: Dereference of a possibly null reference.
                //         ns4 /*T:string?*/ .ToString(); // warn 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns4").WithLocation(30, 9),
                // (38,9): warning CS8602: Dereference of a possibly null reference.
                //         ns5 /*T:string?*/ .ToString(); // warn 6
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "ns5").WithLocation(38, 9)
                );
        }

        [WorkItem(30840, "https://github.com/dotnet/roslyn/issues/30840")]
        [Fact]
        public void NonNullTypesFalse_LocalDeclarations()
        {
            var source = @"


#nullable disable
public class C : Base
{
    public void M()
    {
        string s2 = Method();
        s2 /*T:string!*/ .ToString();
        s2 = null;

        string? ns2 = NMethod(); // 1
        ns2 /*T:string?*/ .ToString();
        ns2 = null;

        string s3 = FalseMethod();
        s3 /*T:string!*/ .ToString();
        s3 = null;

        string? ns3 = FalseNMethod(); // 2
        ns3 /*T:string?*/ .ToString();
        ns3 = null;

        var s4 = Method();
        s4 /*T:string!*/ .ToString();
        s4 = null;

        var ns4 = NMethod();
        ns4 /*T:string?*/ .ToString();
        ns4 = null;

        var s5 = FalseMethod();
        s5 /*T:string!*/ .ToString();
        s5 = null;

        var ns5 = FalseNMethod();
        ns5 /*T:string?*/ .ToString();
        ns5 = null;
    }
}
public class Base
{
#nullable enable
    public string Method() => throw null!;

#nullable enable
    public string? NMethod() => throw null!;

#nullable disable
    public string FalseMethod() => throw null!;

#nullable disable
    public string? FalseNMethod() => throw null!; // 3
}
";
            var compilation = CreateCompilation(source, parseOptions: TestOptions.Regular.WithFeature("run-nullable-analysis", "always"), options: WithNullableEnable());

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (54,18): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public string? FalseNMethod() => throw null!; // 3
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(54, 18),
                // (13,15): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //         string? ns2 = NMethod(); // 1
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(13, 15),
                // (21,15): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //         string? ns3 = FalseNMethod(); // 2
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(21, 15)
                );
        }

        [Fact]
        public void NonNullTypes_Constraint()
        {
            var source = @"

public class S { }

#nullable enable
public struct C<T> where T : S
{
    public void M(T t)
    {
        t.ToString();
        t = null; // warn
    }
}

#nullable disable
public struct D<T> where T : S
{
    public void M(T t)
    {
        t.ToString();
        t = null;
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (11,13): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         t = null; // warn
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(11, 13)
                );
        }

        [Fact]
        public void NonNullTypes_Delegate()
        {
            var source = @"


#nullable enable
public delegate string[] MyDelegate(string[] x);

#nullable disable
public delegate string[] MyFalseDelegate(string[] x);
#nullable enable
public delegate string[]? MyNullableDelegate(string[]? x);

class C
{
    void M()
    {
        MyDelegate x1 = Method;
        MyDelegate x2 = FalseMethod;
        MyDelegate x4 = NullableReturnMethod; // warn 1
        MyDelegate x5 = NullableParameterMethod;
        MyFalseDelegate y1 = Method;
        MyFalseDelegate y2 = FalseMethod;
        MyFalseDelegate y4 = NullableReturnMethod;
        MyFalseDelegate y5 = NullableParameterMethod;
        MyNullableDelegate z1 = Method; // warn 2
        MyNullableDelegate z2 = FalseMethod;
        MyNullableDelegate z4 = NullableReturnMethod; // warn 3
        MyNullableDelegate z5 = NullableParameterMethod;
     }

#nullable enable
    public string[] Method(string[] x) => throw null!;

#nullable disable
    public string[] FalseMethod(string[] x) => throw null!;
#nullable enable
    public string[]? NullableReturnMethod(string[] x) => throw null!;
    public string[] NullableParameterMethod(string[]? x) => throw null!;
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (18,25): warning CS8621: Nullability of reference types in return type of 'string[]? C.NullableReturnMethod(string[] x)' doesn't match the target delegate 'MyDelegate'.
                //         MyDelegate x4 = NullableReturnMethod; // warn 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOfTargetDelegate, "NullableReturnMethod").WithArguments("string[]? C.NullableReturnMethod(string[] x)", "MyDelegate").WithLocation(18, 25),
                // (24,33): warning CS8622: Nullability of reference types in type of parameter 'x' of 'string[] C.Method(string[] x)' doesn't match the target delegate 'MyNullableDelegate'.
                //         MyNullableDelegate z1 = Method; // warn 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "Method").WithArguments("x", "string[] C.Method(string[] x)", "MyNullableDelegate").WithLocation(24, 33),
                // (26,33): warning CS8622: Nullability of reference types in type of parameter 'x' of 'string[]? C.NullableReturnMethod(string[] x)' doesn't match the target delegate 'MyNullableDelegate'.
                //         MyNullableDelegate z4 = NullableReturnMethod; // warn 3
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOfTargetDelegate, "NullableReturnMethod").WithArguments("x", "string[]? C.NullableReturnMethod(string[] x)", "MyNullableDelegate").WithLocation(26, 33)
                );
        }

        [Fact]
        public void NonNullTypes_Constructor()
        {
            var source = @"


public class C
{
#nullable enable
    public C(string[] x) => throw null!;
}
public class D
{
#nullable disable
    public D(string[] x) => throw null!;
}
#nullable enable
public class E
{
    public string[] field = null!;
#nullable disable
    public string[] obliviousField;
#nullable enable
    public string[]? nullableField;

    void M()
    {
        new C(field);
        new C(obliviousField);
        new C(nullableField); // warn
        new D(field);
        new D(obliviousField);
        new D(nullableField);
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (27,15): warning CS8604: Possible null reference argument for parameter 'x' in 'C.C(string[] x)'.
                //         new C(nullableField); // warn
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "nullableField").WithArguments("x", "C.C(string[] x)").WithLocation(27, 15)
                );
        }

        [Fact]
        public void NonNullTypes_Constraint_Nested()
        {
            var source = @"

public class S { }
public class List<T> { public T Item { get; set; } = default!; }

#nullable enable
public struct C<T, NT>
    where T : List<S>
    where NT : List<S?>
{
    public void M(T t, NT nt)
    {
        t.Item /*T:S!*/ .ToString();
        t.Item = null; // warn 1
        nt.Item /*T:S?*/ .ToString(); // warn 2
        nt.Item = null;
    }
}

#nullable disable
public struct D<T, NT>
    where T : List<S>
    where NT : List<S?> // warn 3
{
    public void M(T t, NT nt)
    {
        t.Item /*T:S!*/ .ToString();
        t.Item = null;
        nt.Item /*T:S?*/ .ToString();
        nt.Item = null;
    }
}
";
            var compilation = CreateCompilation(source, parseOptions: TestOptions.Regular.WithFeature("run-nullable-analysis", "always"), options: WithNullableEnable());

            compilation.VerifyTypes();
            compilation.VerifyDiagnostics(
                // (14,18): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         t.Item = null; // warn 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(14, 18),
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         nt.Item /*T:S?*/ .ToString(); // warn 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "nt.Item").WithLocation(15, 9),
                // (23,22): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     where NT : List<S?> // warn 3
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(23, 22));
        }

        [Fact]
        public void IsAnnotated_01()
        {
            var source =
@"using System;

class C1
{
    string F1() => throw null!;
    string? F2() => throw null!;
    int F3() => throw null!;
    int? F4() => throw null!;
    Nullable<int> F5() => throw null!;
}
#nullable disable
class C2
{
    string F1() => throw null!;
    string? F2() => throw null!;
    int F3() => throw null!;
    int? F4() => throw null!;
    Nullable<int> F5() => throw null!;
}
#nullable enable
class C3
{
    string F1() => throw null!;
    string? F2() => throw null!;
    int F3() => throw null!;
    int? F4() => throw null!;
    Nullable<int> F5() => throw null!;
}";
            var comp = CreateCompilation(new[] { source });
            comp.VerifyDiagnostics(
                // (6,11): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     string? F2() => throw null!;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(6, 11),
                // (15,11): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     string? F2() => throw null!;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(15, 11)
                );

            verify("C1.F1", "System.String", NullableAnnotation.Oblivious);
            verify("C1.F2", "System.String?", NullableAnnotation.Annotated);
            verify("C1.F3", "System.Int32", NullableAnnotation.Oblivious);
            verify("C1.F4", "System.Int32?", NullableAnnotation.Annotated);
            verify("C1.F5", "System.Int32?", NullableAnnotation.Annotated);
            verify("C2.F1", "System.String", NullableAnnotation.Oblivious);
            verify("C2.F2", "System.String?", NullableAnnotation.Annotated);
            verify("C2.F3", "System.Int32", NullableAnnotation.Oblivious);
            verify("C2.F4", "System.Int32?", NullableAnnotation.Annotated);
            verify("C2.F5", "System.Int32?", NullableAnnotation.Annotated);
            verify("C3.F1", "System.String!", NullableAnnotation.NotAnnotated);
            verify("C3.F2", "System.String?", NullableAnnotation.Annotated);
            verify("C3.F3", "System.Int32", NullableAnnotation.NotAnnotated);
            verify("C3.F4", "System.Int32?", NullableAnnotation.Annotated);
            verify("C3.F5", "System.Int32?", NullableAnnotation.Annotated);

            // https://github.com/dotnet/roslyn/issues/29845: Test nested nullability.

            void verify(string methodName, string displayName, NullableAnnotation nullableAnnotation)
            {
                var method = comp.GetMember<MethodSymbol>(methodName);
                var type = method.ReturnTypeWithAnnotations;
                Assert.Equal(displayName, type.ToTestDisplayString(true));
                Assert.Equal(nullableAnnotation, type.NullableAnnotation);
            }
        }

        [Fact]
        public void IsAnnotated_02()
        {
            var source =
@"using System;

class C1
{
    T F1<T>() => throw null!;
    T? F2<T>() => throw null!;
    T F3<T>() where T : class => throw null!;
    T? F4<T>() where T : class => throw null!;
    T F5<T>() where T : struct => throw null!;
    T? F6<T>() where T : struct => throw null!;
    Nullable<T> F7<T>() where T : struct => throw null!;
}
#nullable disable
class C2
{
    T F1<T>() => throw null!;
    T? F2<T>() => throw null!;
    T F3<T>() where T : class => throw null!;
    T? F4<T>() where T : class => throw null!;
    T F5<T>() where T : struct => throw null!;
    T? F6<T>() where T : struct => throw null!;
    Nullable<T> F7<T>() where T : struct => throw null!;
}
#nullable enable
class C3
{
    T F1<T>() => throw null!;
    T? F2<T>() => throw null!;
    T F3<T>() where T : class => throw null!;
    T? F4<T>() where T : class => throw null!;
    T F5<T>() where T : struct => throw null!;
    T? F6<T>() where T : struct => throw null!;
    Nullable<T> F7<T>() where T : struct => throw null!;
}";
            var comp = CreateCompilation(new[] { source }, parseOptions: TestOptions.Regular8);
            comp.VerifyDiagnostics(
                // (28,5): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     T? F2<T>() => throw null!;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(28, 5),
                // (17,6): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     T? F2<T>() => throw null!;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(17, 6),
                // (17,5): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     T? F2<T>() => throw null!;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(17, 5),
                // (6,6): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     T? F2<T>() => throw null!;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(6, 6),
                // (6,5): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     T? F2<T>() => throw null!;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(6, 5),
                // (19,6): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     T? F4<T>() where T : class => throw null!;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(19, 6),
                // (19,5): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     T? F4<T>() where T : class => throw null!;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(19, 5),
                // (8,6): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     T? F4<T>() where T : class => throw null!;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(8, 6),
                // (8,5): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     T? F4<T>() where T : class => throw null!;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(8, 5)
            );

            verify("C1.F1", "T", NullableAnnotation.Oblivious);
            verify("C1.F2", "T?", NullableAnnotation.Annotated);
            verify("C1.F3", "T", NullableAnnotation.Oblivious);
            verify("C1.F4", "T?", NullableAnnotation.Annotated);
            verify("C1.F5", "T", NullableAnnotation.Oblivious);
            verify("C1.F6", "T?", NullableAnnotation.Annotated);
            verify("C1.F7", "T?", NullableAnnotation.Annotated);
            verify("C2.F1", "T", NullableAnnotation.Oblivious);
            verify("C2.F2", "T?", NullableAnnotation.Annotated);
            verify("C2.F3", "T", NullableAnnotation.Oblivious);
            verify("C2.F4", "T?", NullableAnnotation.Annotated);
            verify("C2.F5", "T", NullableAnnotation.Oblivious);
            verify("C2.F6", "T?", NullableAnnotation.Annotated);
            verify("C2.F7", "T?", NullableAnnotation.Annotated);
            verify("C3.F1", "T", NullableAnnotation.NotAnnotated);
            verify("C3.F2", "T?", NullableAnnotation.Annotated);
            verify("C3.F3", "T!", NullableAnnotation.NotAnnotated);
            verify("C3.F4", "T?", NullableAnnotation.Annotated);
            verify("C3.F5", "T", NullableAnnotation.NotAnnotated);
            verify("C3.F6", "T?", NullableAnnotation.Annotated);
            verify("C3.F7", "T?", NullableAnnotation.Annotated);

            // https://github.com/dotnet/roslyn/issues/29845: Test nested nullability.
            // https://github.com/dotnet/roslyn/issues/29845: Test all combinations of overrides.

            void verify(string methodName, string displayName, NullableAnnotation nullableAnnotation)
            {
                var method = comp.GetMember<MethodSymbol>(methodName);
                var type = method.ReturnTypeWithAnnotations;
                Assert.Equal(displayName, type.ToTestDisplayString(true));
                Assert.Equal(nullableAnnotation, type.NullableAnnotation);
            }
        }

        [Fact]
        public void InheritedValueConstraintForNullable1_01()
        {
            var source = @"
class A
{
    public virtual T? Goo<T>() where T : struct 
    { 
        return null; 
    }
}

class B : A
{
    public override T? Goo<T>()
    {
        return null;
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            //var a = compilation.GetTypeByMetadataName("A");
            //var aGoo = a.GetMember<MethodSymbol>("Goo");
            //Assert.Equal("T? A.Goo<T>()", aGoo.ToTestDisplayString());

            //var b = compilation.GetTypeByMetadataName("B");
            //var bGoo = b.GetMember<MethodSymbol>("Goo");
            //Assert.Equal("T? A.Goo<T>()", bGoo.OverriddenMethod.ToTestDisplayString());

            compilation.VerifyDiagnostics();
        }

        [Fact]
        public void InheritedValueConstraintForNullable1_02()
        {
            var source = @"
class A
{
    public virtual void Goo<T>(T? x) where T : struct 
    { 
    }
}

class B : A
{
    public override void Goo<T>(T? x)
    {
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics();
        }

        [Fact]
        public void InheritedValueConstraintForNullable1_03()
        {
            var source = @"
class A
{
    public virtual System.Nullable<T> Goo<T>() where T : struct 
    { 
        return null; 
    }
}

class B : A
{
    public override T? Goo<T>()
    {
        return null;
    }
} 
";
            CreateCompilation(new[] { source }, options: WithNullableEnable()).VerifyDiagnostics();
        }

        [Fact]
        public void InheritedValueConstraintForNullable1_04()
        {
            var source = @"
class A
{
    public virtual void Goo<T>(System.Nullable<T> x) where T : struct 
    { 
    }
}

class B : A
{
    public override void Goo<T>(T? x)
    {
    }
} 
";
            CreateCompilation(new[] { source }, options: WithNullableEnable()).VerifyDiagnostics();
        }

        [Fact]
        public void InheritedValueConstraintForNullable1_05()
        {
            var source = @"
class A
{
    public virtual T? Goo<T>() where T : struct 
    { 
        return null; 
    }
}

class B : A
{
    public override System.Nullable<T> Goo<T>()
    {
        return null;
    }
} 
";
            CreateCompilation(new[] { source }, options: WithNullableEnable()).VerifyDiagnostics();
        }

        [Fact]
        public void InheritedValueConstraintForNullable1_06()
        {
            var source = @"
class A
{
    public virtual void M1<T>(T? x) where T : struct 
    { 
    }
}

class B : A
{
    public override void M1<T>(System.Nullable<T> x)
    {
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics();

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.True(m1.Parameters[0].Type.IsNullableType());
            Assert.True(m1.Parameters[0].Type.IsValueType);
            Assert.True(m1.OverriddenMethod.Parameters[0].Type.IsNullableType());
        }

        [Fact]
        public void Overriding_03()
        {
            var source = @"
class A
{
    public virtual void M1<T>(T? x) where T : class 
    { 
    }

    public virtual T? M2<T>() where T : class 
    { 
        return null;
    }
}

class B : A
{
    public override void M1<T>(T? x)
    {
    }

    public override T? M2<T>()
    { 
        return null;
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (16,26): error CS0115: 'B.M1<T>(T?)': no suitable method found to override
                //     public override void M1<T>(T? x)
                Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M1").WithArguments("B.M1<T>(T?)").WithLocation(16, 26),
                // (16,35): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override void M1<T>(T? x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(16, 35),
                // (20,24): error CS0508: 'B.M2<T>()': return type must be 'T' to match overridden member 'A.M2<T>()'
                //     public override T? M2<T>()
                Diagnostic(ErrorCode.ERR_CantChangeReturnTypeOnOverride, "M2").WithArguments("B.M2<T>()", "A.M2<T>()", "T").WithLocation(20, 24),
                // (20,24): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override T? M2<T>()
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M2").WithArguments("System.Nullable<T>", "T", "T").WithLocation(20, 24));

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.True(m1.Parameters[0].Type.IsNullableType());
            Assert.False(m1.Parameters[0].Type.IsReferenceType);
            Assert.Null(m1.OverriddenMethod);

            var m2 = b.GetMember<MethodSymbol>("M2");
            Assert.True(m2.ReturnType.IsNullableType());
            Assert.False(m2.ReturnType.IsReferenceType);
            Assert.False(m2.OverriddenMethod.ReturnType.IsNullableType());
        }

        [Fact]
        [WorkItem(29846, "https://github.com/dotnet/roslyn/issues/29846")]
        public void Overriding_04()
        {
            var source = @"
class A
{
    public virtual void M1<T>(T? x) where T : struct 
    { 
    }

    public virtual void M1<T>(T x) 
    { 
    }

    public virtual void M2<T>(T? x) where T : struct 
    { 
    }

    public virtual void M2<T>(T x) 
    { 
    }

    public virtual void M3<T>(T x) 
    { 
    }

    public virtual void M3<T>(T? x) where T : struct 
    { 
    }
}

class B : A
{
    public override void M1<T>(T? x)
    {
    }

    public override void M2<T>(T x)
    {
    }

    public override void M3<T>(T? x)
    {
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics(
                );

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.True(m1.Parameters[0].Type.IsNullableType());
            Assert.True(m1.OverriddenMethod.Parameters[0].Type.IsNullableType());

            var m2 = b.GetMember<MethodSymbol>("M2");
            Assert.False(m2.Parameters[0].Type.IsNullableType());
            Assert.False(m2.OverriddenMethod.Parameters[0].Type.IsNullableType());

            var m3 = b.GetMember<MethodSymbol>("M3");
            Assert.True(m3.Parameters[0].Type.IsNullableType());
            Assert.True(m3.OverriddenMethod.Parameters[0].Type.IsNullableType());
        }

        [Fact]
        [WorkItem(29847, "https://github.com/dotnet/roslyn/issues/29847")]
        public void Overriding_05()
        {
            var source = @"
class A
{
    public virtual void M1<T>(T? x) where T : struct 
    { 
    }

    public virtual void M1<T>(T? x) where T : class 
    { 
    }
}

class B : A
{
    public override void M1<T>(T? x)
    {
    }
} 

class C
{
    public virtual void M2<T>(T? x) where T : class 
    { 
    }

    public virtual void M2<T>(T? x) where T : struct 
    { 
    }
}

class D : C
{
    public override void M2<T>(T? x)
    {
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics();

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.True(m1.Parameters[0].Type.IsNullableType());
            Assert.True(m1.OverriddenMethod.Parameters[0].Type.IsNullableType());

            var d = compilation.GetTypeByMetadataName("D");
            var m2 = d.GetMember<MethodSymbol>("M2");
            Assert.True(m2.Parameters[0].Type.IsNullableType());
            Assert.True(m2.OverriddenMethod.Parameters[0].Type.IsNullableType());
        }

        [Fact]
        public void Overriding_06()
        {
            var source = @"
class A
{
    public virtual void M1<T>(System.Nullable<T> x) where T : struct
    {
    }

    public virtual void M2<T>(T? x) where T : struct
    {
    }

    public virtual void M3<T>(C<T?> x) where T : struct
    {
    }

    public virtual void M4<T>(C<System.Nullable<T>> x) where T : struct
    {
    }

    public virtual void M5<T>(C<T?> x) where T : class
    {
    }
}

class B : A
{
    public override void M1<T>(T? x)
    {
    }

    public override void M2<T>(T? x)
    {
    }

    public override void M3<T>(C<T?> x)
    {
    }

    public override void M4<T>(C<T?> x)
    {
    }

    public override void M5<T>(C<T?> x)
    {
    }
}

class C<T> {}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics(
                // (43,26): error CS0115: 'B.M5<T>(C<T?>)': no suitable method found to override
                //     public override void M5<T>(C<T?> x) where T : class
                Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M5").WithArguments("B.M5<T>(C<T?>)").WithLocation(43, 26),
                // (43,38): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override void M5<T>(C<T?> x) where T : class
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(43, 38));

            var b = compilation.GetTypeByMetadataName("B");
            var m3 = b.GetMember<MethodSymbol>("M3");
            var m4 = b.GetMember<MethodSymbol>("M4");
            var m5 = b.GetMember<MethodSymbol>("M5");
            Assert.True(((NamedTypeSymbol)m3.Parameters[0].Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
            Assert.True(((NamedTypeSymbol)m3.OverriddenMethod.Parameters[0].Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
            Assert.True(((NamedTypeSymbol)m4.Parameters[0].Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
            Assert.True(((NamedTypeSymbol)m4.OverriddenMethod.Parameters[0].Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
            Assert.True(((NamedTypeSymbol)m5.Parameters[0].Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
            Assert.Null(m5.OverriddenMethod);
        }

        [Fact]
        public void Overriding_07()
        {
            var source = @"
class A
{
    public void M1<T>(T x) 
    {
    }
}

class B : A
{
    public void M1<T>(T? x) where T : struct
    {
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics();

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.True(m1.Parameters[0].Type.IsNullableType());
            Assert.True(m1.Parameters[0].Type.StrippedType().IsValueType);
            Assert.Null(m1.OverriddenMethod);
        }

        [Fact]
        public void Overriding_08()
        {
            var source = @"
class A
{
    public void M1<T>(T x) 
    {
    }
}

class B : A
{
    public override void M1<T>(T? x) where T : struct
    {
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics(
                // (11,26): error CS0115: 'B.M1<T>(T?)': no suitable method found to override
                //     public override void M1<T>(T? x) where T : struct
                Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M1").WithArguments("B.M1<T>(T?)").WithLocation(11, 26),
                // (11,35): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override void M1<T>(T? x) where T : struct
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(11, 35)
                );

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.True(m1.Parameters[0].Type.IsNullableType());
            Assert.False(m1.Parameters[0].Type.StrippedType().IsValueType);
            Assert.False(m1.Parameters[0].Type.StrippedType().IsReferenceType);
            Assert.Null(m1.OverriddenMethod);
        }

        [Fact]
        public void Overriding_09()
        {
            var source = @"
class A
{
    public void M1<T>(T x) 
    {
    }

    public void M2<T>(T? x) 
    {
    }

    public void M3<T>(T? x) where T : class
    {
    }

    public void M4<T>(T? x) where T : struct
    {
    }
}

class B : A
{
    public override void M1<T>(T? x)
    {
    }

    public override void M2<T>(T? x)
    {
    }

    public override void M3<T>(T? x)
    {
    }

    public override void M4<T>(T? x)
    {
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);
            compilation.VerifyDiagnostics(
                // (8,23): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     public void M2<T>(T? x) 
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(8, 23),
                // (27,26): error CS0115: 'B.M2<T>(T?)': no suitable method found to override
                //     public override void M2<T>(T? x)
                Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M2").WithArguments("B.M2<T>(T?)").WithLocation(27, 26),
                // (31,26): error CS0115: 'B.M3<T>(T?)': no suitable method found to override
                //     public override void M3<T>(T? x)
                Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M3").WithArguments("B.M3<T>(T?)").WithLocation(31, 26),
                // (35,26): error CS0506: 'B.M4<T>(T?)': cannot override inherited member 'A.M4<T>(T?)' because it is not marked virtual, abstract, or override
                //     public override void M4<T>(T? x)
                Diagnostic(ErrorCode.ERR_CantOverrideNonVirtual, "M4").WithArguments("B.M4<T>(T?)", "A.M4<T>(T?)").WithLocation(35, 26),
                // (23,26): error CS0115: 'B.M1<T>(T?)': no suitable method found to override
                //     public override void M1<T>(T? x)
                Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M1").WithArguments("B.M1<T>(T?)").WithLocation(23, 26),
                // (27,35): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override void M2<T>(T? x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(27, 35),
                // (31,35): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override void M3<T>(T? x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(31, 35),
                // (35,35): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override void M4<T>(T? x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(35, 35),
                // (23,35): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override void M1<T>(T? x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(23, 35)
                );

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            var m2 = b.GetMember<MethodSymbol>("M2");
            var m3 = b.GetMember<MethodSymbol>("M3");
            var m4 = b.GetMember<MethodSymbol>("M4");
            Assert.True(m1.Parameters[0].Type.IsNullableType());
            Assert.True(m2.Parameters[0].Type.IsNullableType());
            Assert.True(m3.Parameters[0].Type.IsNullableType());
            Assert.True(m4.Parameters[0].Type.IsNullableType());

            Assert.Null(m1.OverriddenMethod);
            Assert.Null(m2.OverriddenMethod);
            Assert.Null(m3.OverriddenMethod);
            Assert.Null(m4.OverriddenMethod);
        }

        [Fact]
        public void Overriding_10()
        {
            var source = @"
class A
{
    public virtual void M1<T>(System.Nullable<T> x) where T : class
    { 
    }
}

class B : A
{
    public override void M1<T>(T? x)
    {
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics(
                // (4,50): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public virtual void M1<T>(System.Nullable<T> x) where T : class
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(4, 50),
                // (11,35): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override void M1<T>(T? x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(11, 35)
                );

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.True(m1.Parameters[0].Type.IsNullableType());
            Assert.NotNull(m1.OverriddenMethod);
        }

        [Fact]
        public void Overriding_11()
        {
            var source = @"
class A
{
    public virtual C<System.Nullable<T>> M1<T>() where T : class
    { 
        throw new System.NotImplementedException();
    }
}

class B : A
{
    public override C<T?> M1<T>()
    {
        throw new System.NotImplementedException();
    }
} 

class C<T> {}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics(
                // (4,42): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public virtual C<System.Nullable<T>> M1<T>() where T : class
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M1").WithArguments("System.Nullable<T>", "T", "T").WithLocation(4, 42),
                // (12,27): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override C<T?> M1<T>()
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M1").WithArguments("System.Nullable<T>", "T", "T").WithLocation(12, 27)
                );

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.True(((NamedTypeSymbol)m1.ReturnType).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
            Assert.True(((NamedTypeSymbol)m1.OverriddenMethod.ReturnType).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
        }

        [Fact]
        public void Overriding_12()
        {
            var source = @"
class A
{
    public virtual string M1()
    { 
        throw new System.NotImplementedException();
    }

    public virtual string? M2()
    { 
        throw new System.NotImplementedException();
    }

    public virtual string? M3()
    { 
        throw new System.NotImplementedException();
    }

    public virtual System.Nullable<string> M4()
    { 
        throw new System.NotImplementedException();
    }

    public System.Nullable<string> M5()
    { 
        throw new System.NotImplementedException();
    }
}

class B : A
{
    public override string? M1()
    {
        throw new System.NotImplementedException();
    }

    public override string? M2()
    {
        throw new System.NotImplementedException();
    }

    public override string M3()
    {
        throw new System.NotImplementedException();
    }

    public override string? M4()
    {
        throw new System.NotImplementedException();
    }

    public override string? M5()
    {
        throw new System.NotImplementedException();
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics(
                // (47,29): error CS0508: 'B.M4()': return type must be 'string?' to match overridden member 'A.M4()'
                //     public override string? M4()
                Diagnostic(ErrorCode.ERR_CantChangeReturnTypeOnOverride, "M4").WithArguments("B.M4()", "A.M4()", "string?").WithLocation(47, 29),
                // (52,29): error CS0506: 'B.M5()': cannot override inherited member 'A.M5()' because it is not marked virtual, abstract, or override
                //     public override string? M5()
                Diagnostic(ErrorCode.ERR_CantOverrideNonVirtual, "M5").WithArguments("B.M5()", "A.M5()").WithLocation(52, 29),
                // (32,29): warning CS8764: Nullability of return type doesn't match overridden member (possibly because of nullability attributes).
                //     public override string? M1()
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "M1").WithLocation(32, 29),
                // (19,44): error CS0453: The type 'string' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public virtual System.Nullable<string> M4()
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M4").WithArguments("System.Nullable<T>", "T", "string").WithLocation(19, 44),
                // (24,36): error CS0453: The type 'string' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public System.Nullable<string> M5()
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M5").WithArguments("System.Nullable<T>", "T", "string").WithLocation(24, 36)
                );

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.False(m1.ReturnType.IsNullableType());
            Assert.False(m1.OverriddenMethod.ReturnType.IsNullableType());

            var m4 = b.GetMember<MethodSymbol>("M4");
            Assert.False(m4.ReturnType.IsNullableType());
            Assert.True(m4.OverriddenMethod.ReturnType.IsNullableType());

            var m5 = b.GetMember<MethodSymbol>("M4");
            Assert.False(m5.ReturnType.IsNullableType());
        }

        [Fact]
        public void Overriding_13()
        {
            var source = @"
class A
{
    public virtual void M1(string x)
    { 
    }

    public virtual void M2(string? x)
    { 
    }

    public virtual void M3(string? x)
    { 
    }

    public virtual void M4(System.Nullable<string> x)
    { 
    }

    public void M5(System.Nullable<string> x)
    { 
    }
}

class B : A
{
    public override void M1(string? x)
    {
    }

    public override void M2(string? x)
    {
    }

    public override void M3(string x)
    {
    }

    public override void M4(string? x)
    {
    }

    public override void M5(string? x)
    {
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (16,52): error CS0453: The type 'string' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public virtual void M4(System.Nullable<string> x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "string").WithLocation(16, 52),
                // (20,44): error CS0453: The type 'string' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public void M5(System.Nullable<string> x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "string").WithLocation(20, 44),
                // (35,26): warning CS8765: Type of parameter 'x' doesn't match overridden member because of nullability attributes.
                //     public override void M3(string x)
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M3").WithArguments("x").WithLocation(35, 26),
                // (39,26): error CS0115: 'B.M4(string?)': no suitable method found to override
                //     public override void M4(string? x)
                Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M4").WithArguments("B.M4(string?)").WithLocation(39, 26),
                // (43,26): error CS0115: 'B.M5(string?)': no suitable method found to override
                //     public override void M5(string? x)
                Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M5").WithArguments("B.M5(string?)").WithLocation(43, 26)
                );

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.False(m1.Parameters[0].Type.IsNullableType());
            Assert.Equal(NullableAnnotation.Annotated, m1.Parameters[0].TypeWithAnnotations.NullableAnnotation);
            Assert.True(m1.Parameters[0].Type.IsReferenceType);
            Assert.False(m1.OverriddenMethod.Parameters[0].Type.IsNullableType());

            var m4 = b.GetMember<MethodSymbol>("M4");
            Assert.False(m4.Parameters[0].Type.IsNullableType());
            Assert.Null(m4.OverriddenMethod);

            var m5 = b.GetMember<MethodSymbol>("M4");
            Assert.False(m5.Parameters[0].Type.IsNullableType());
        }

        [Fact]
        public void Overriding_14()
        {
            var source = @"
class A
{
    public virtual int M1()
    { 
        throw new System.NotImplementedException();
    }

    public virtual int? M2()
    { 
        throw new System.NotImplementedException();
    }

    public virtual int? M3()
    { 
        throw new System.NotImplementedException();
    }

    public virtual System.Nullable<int> M4()
    { 
        throw new System.NotImplementedException();
    }

    public System.Nullable<int> M5()
    { 
        throw new System.NotImplementedException();
    }
}

class B : A
{
    public override int? M1()
    {
        throw new System.NotImplementedException();
    }

    public override int? M2()
    {
        throw new System.NotImplementedException();
    }

    public override int M3()
    {
        throw new System.NotImplementedException();
    }

    public override int? M4()
    {
        throw new System.NotImplementedException();
    }

    public override int? M5()
    {
        throw new System.NotImplementedException();
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics(
                 // (42,25): error CS0508: 'B.M3()': return type must be 'int?' to match overridden member 'A.M3()'
                 //     public override int M3()
                 Diagnostic(ErrorCode.ERR_CantChangeReturnTypeOnOverride, "M3").WithArguments("B.M3()", "A.M3()", "int?").WithLocation(42, 25),
                 // (52,26): error CS0506: 'B.M5()': cannot override inherited member 'A.M5()' because it is not marked virtual, abstract, or override
                 //     public override int? M5()
                 Diagnostic(ErrorCode.ERR_CantOverrideNonVirtual, "M5").WithArguments("B.M5()", "A.M5()").WithLocation(52, 26),
                 // (32,26): error CS0508: 'B.M1()': return type must be 'int' to match overridden member 'A.M1()'
                 //     public override int? M1()
                 Diagnostic(ErrorCode.ERR_CantChangeReturnTypeOnOverride, "M1").WithArguments("B.M1()", "A.M1()", "int").WithLocation(32, 26)
                );

            var b = compilation.GetTypeByMetadataName("B");
            Assert.True(b.GetMember<MethodSymbol>("M1").ReturnType.IsNullableType());
            Assert.True(b.GetMember<MethodSymbol>("M2").ReturnType.IsNullableType());
            Assert.False(b.GetMember<MethodSymbol>("M3").ReturnType.IsNullableType());
            Assert.True(b.GetMember<MethodSymbol>("M4").ReturnType.IsNullableType());
            Assert.True(b.GetMember<MethodSymbol>("M5").ReturnType.IsNullableType());
        }

        [Fact]
        public void Overriding_15()
        {
            var source = @"
class A
{
    public virtual void M1(int x)
    { 
    }

    public virtual void M2(int? x)
    { 
    }

    public virtual void M3(int? x)
    { 
    }

    public virtual void M4(System.Nullable<int> x)
    { 
    }

    public void M5(System.Nullable<int> x)
    { 
    }
}

class B : A
{
    public override void M1(int? x)
    {
    }

    public override void M2(int? x)
    {
    }

    public override void M3(int x)
    {
    }

    public override void M4(int? x)
    {
    }

    public override void M5(int? x)
    {
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                 // (35,26): error CS0115: 'B.M3(int)': no suitable method found to override
                 //     public override void M3(int x)
                 Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M3").WithArguments("B.M3(int)").WithLocation(35, 26),
                 // (43,26): error CS0506: 'B.M5(int?)': cannot override inherited member 'A.M5(int?)' because it is not marked virtual, abstract, or override
                 //     public override void M5(int? x)
                 Diagnostic(ErrorCode.ERR_CantOverrideNonVirtual, "M5").WithArguments("B.M5(int?)", "A.M5(int?)").WithLocation(43, 26),
                 // (27,26): error CS0115: 'B.M1(int?)': no suitable method found to override
                 //     public override void M1(int? x)
                 Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M1").WithArguments("B.M1(int?)").WithLocation(27, 26)
                );

            var b = compilation.GetTypeByMetadataName("B");
            Assert.True(b.GetMember<MethodSymbol>("M1").Parameters[0].Type.IsNullableType());
            Assert.True(b.GetMember<MethodSymbol>("M2").Parameters[0].Type.IsNullableType());
            Assert.False(b.GetMember<MethodSymbol>("M3").Parameters[0].Type.IsNullableType());
            Assert.True(b.GetMember<MethodSymbol>("M4").Parameters[0].Type.IsNullableType());
            Assert.True(b.GetMember<MethodSymbol>("M5").Parameters[0].Type.IsNullableType());
        }

        [Fact]
        public void Overriding_16()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

abstract class A
{
    public abstract event System.Action<string> E1; 
    public abstract event System.Action<string>? E2; 
    public abstract event System.Action<string?>? E3; 
}

class B1 : A
{
    public override event System.Action<string?> E1 {add {} remove{}}
    public override event System.Action<string> E2 {add {} remove{}}
    public override event System.Action<string?>? E3 {add {} remove{}}
}

class B2 : A
{
    public override event System.Action<string?> E1; // 2
    public override event System.Action<string> E2; // 2
    public override event System.Action<string?>? E3; // 2

    void Dummy()
    {
        var e1 = E1;
        var e2 = E2;
        var e3 = E3;
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (18,50): warning CS8608: Nullability of reference types in type doesn't match overridden member.
                //     public override event System.Action<string?> E1 {add {} remove{}}
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnOverride, "E1").WithLocation(18, 50),
                // (19,49): warning CS8608: Nullability of reference types in type doesn't match overridden member.
                //     public override event System.Action<string> E2 {add {} remove{}}
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnOverride, "E2").WithLocation(19, 49),
                // (25,50): warning CS8608: Nullability of reference types in type doesn't match overridden member.
                //     public override event System.Action<string?> E1; // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnOverride, "E1").WithLocation(25, 50),
                // (25,50): warning CS8618: Non-nullable event 'E1' is uninitialized. Consider declaring the event as nullable.
                //     public override event System.Action<string?> E1; // 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "E1").WithArguments("event", "E1").WithLocation(25, 50),
                // (26,49): warning CS8608: Nullability of reference types in type doesn't match overridden member.
                //     public override event System.Action<string> E2; // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnOverride, "E2").WithLocation(26, 49),
                // (26,49): warning CS8618: Non-nullable event 'E2' is uninitialized. Consider declaring the event as nullable.
                //     public override event System.Action<string> E2; // 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "E2").WithArguments("event", "E2").WithLocation(26, 49)
                );

            foreach (string typeName in new[] { "B1", "B2" })
            {
                var type = compilation.GetTypeByMetadataName(typeName);

                foreach (string memberName in new[] { "E1", "E2" })
                {
                    var member = type.GetMember<EventSymbol>(memberName);
                    Assert.False(member.TypeWithAnnotations.Equals(member.OverriddenEvent.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
                }

                var e3 = type.GetMember<EventSymbol>("E3");
                Assert.True(e3.TypeWithAnnotations.Equals(e3.OverriddenEvent.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (string typeName in new[] { "A", "B1", "B2" })
            {
                var type = compilation.GetTypeByMetadataName(typeName);

                foreach (var ev in type.GetMembers().OfType<EventSymbol>())
                {
                    Assert.True(ev.TypeWithAnnotations.Equals(ev.AddMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                    Assert.True(ev.TypeWithAnnotations.Equals(ev.RemoveMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                }
            }
        }

        [Fact]
        [WorkItem(29851, "https://github.com/dotnet/roslyn/issues/29851")]
        public void Overriding_Methods()
        {
            var source = @"

public abstract class A
{
#nullable disable
    public abstract System.Action<string> Oblivious1(System.Action<string> x);
#nullable enable

    public abstract System.Action<string> Oblivious2(System.Action<string> x);
    public abstract System.Action<string> M3(System.Action<string> x);
    public abstract System.Action<string> M4(System.Action<string> x);
    public abstract System.Action<string>? M5(System.Action<string>? x);
}

public class B1 : A
{
    public override System.Action<string?> Oblivious1(System.Action<string?> x) => throw null!;
    public override System.Action<string?> Oblivious2(System.Action<string?> x) => throw null!; // warn 3 and 4 // https://github.com/dotnet/roslyn/issues/29851: Should not warn
    public override System.Action<string?> M3(System.Action<string?> x) => throw null!; // warn 5 and 6
    public override System.Action<string?> M4(System.Action<string?> x) => throw null!; // warn 7 and 8
    public override System.Action<string?> M5(System.Action<string?> x) => throw null!; // warn 9 and 10
}

public class B2 : A
{

    public override System.Action<string> Oblivious1(System.Action<string> x) => throw null!;
    public override System.Action<string> Oblivious2(System.Action<string> x) => throw null!;
#nullable disable
    public override System.Action<string> M3(System.Action<string> x) => throw null!;
#nullable enable

    public override System.Action<string> M4(System.Action<string> x) => throw null!;
#nullable disable
    public override System.Action<string> M5(System.Action<string> x) => throw null!;
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (18,44): warning CS8610: Nullability of reference types in type of parameter 'x' doesn't match overridden member.
                //     public override System.Action<string?> Oblivious2(System.Action<string?> x) => throw null!; // warn 3 and 4 // https://github.com/dotnet/roslyn/issues/29851: Should not warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "Oblivious2").WithArguments("x").WithLocation(18, 44),
                // (19,44): warning CS8610: Nullability of reference types in type of parameter 'x' doesn't match overridden member.
                //     public override System.Action<string?> M3(System.Action<string?> x) => throw null!; // warn 5 and 6
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "M3").WithArguments("x").WithLocation(19, 44),
                // (20,44): warning CS8610: Nullability of reference types in type of parameter 'x' doesn't match overridden member.
                //     public override System.Action<string?> M4(System.Action<string?> x) => throw null!; // warn 7 and 8
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "M4").WithArguments("x").WithLocation(20, 44),
                // (21,44): warning CS8610: Nullability of reference types in type of parameter 'x' doesn't match overridden member.
                //     public override System.Action<string?> M5(System.Action<string?> x) => throw null!; // warn 9 and 10
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "M5").WithArguments("x").WithLocation(21, 44)
                );

            var b1 = compilation.GetTypeByMetadataName("B1");
            verifyMethodMatchesOverridden(expectMatch: false, b1, "Oblivious1");
            verifyMethodMatchesOverridden(expectMatch: false, b1, "Oblivious2");
            verifyMethodMatchesOverridden(expectMatch: false, b1, "M3");
            verifyMethodMatchesOverridden(expectMatch: false, b1, "M4");
            verifyMethodMatchesOverridden(expectMatch: false, b1, "M5");

            var b2 = compilation.GetTypeByMetadataName("B2");
            verifyMethodMatchesOverridden(expectMatch: false, b2, "Oblivious1"); // https://github.com/dotnet/roslyn/issues/29851: They should match
            verifyMethodMatchesOverridden(expectMatch: true, b2, "Oblivious2"); // https://github.com/dotnet/roslyn/issues/29851: They should not match
            verifyMethodMatchesOverridden(expectMatch: false, b2, "M3");
            verifyMethodMatchesOverridden(expectMatch: true, b2, "M4"); // https://github.com/dotnet/roslyn/issues/29851: They should not match
            verifyMethodMatchesOverridden(expectMatch: false, b2, "M5");

            void verifyMethodMatchesOverridden(bool expectMatch, NamedTypeSymbol type, string methodName)
            {
                var member = type.GetMember<MethodSymbol>(methodName);
                Assert.Equal(expectMatch, member.ReturnTypeWithAnnotations.Equals(member.OverriddenMethod.ReturnTypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                Assert.Equal(expectMatch, member.Parameters.Single().TypeWithAnnotations.Equals(member.OverriddenMethod.Parameters.Single().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
            }
        }

        [Fact]
        public void Overriding_Properties_WithNullableTypeArgument()
        {
            var source = @"

#nullable enable
public class List<T> { }
public class Base<T>
{
    public virtual List<T?> P { get; set; } = default;
}
public class Class<T> : Base<T>
{
#nullable disable
    public override List<T?> P { get; set; } = default;
}
";
            var comp = CreateCompilation(new[] { source }, parseOptions: TestOptions.Regular8);
            comp.VerifyDiagnostics(
                // (7,25): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     public virtual List<T?> P { get; set; } = default;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(7, 25),
                // (7,47): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //     public virtual List<T?> P { get; set; } = default;
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "default").WithLocation(7, 47),
                // (12,26): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     public override List<T?> P { get; set; } = default;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(12, 26),
                // (12,27): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public override List<T?> P { get; set; } = default;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(12, 27)
                );
        }

        [Fact]
        public void Overriding_Properties_WithNullableTypeArgument_WithClassConstraint()
        {
            var source = @"

#nullable enable
public class List<T> { }
public class Base<T> where T : class
{
    public virtual List<T?> P { get; set; } = default;
}
public class Class<T> : Base<T> where T : class
{
#nullable disable
    public override List<T?> P { get; set; } = default;
}
";
            var comp = CreateCompilation(new[] { source });
            comp.VerifyDiagnostics(
                // (7,47): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //     public virtual List<T?> P { get; set; } = default;
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "default").WithLocation(7, 47),
                // (12,27): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override List<T?> P { get; set; } = default;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(12, 27)
                );
        }

        [Fact]
        public void Overriding_Properties_WithNullableTypeArgument_WithStructConstraint()
        {
            var source = @"

public class List<T> { }
public class Base<T> where T : struct
{
    public virtual List<T?> P { get; set; } = default;
}
public class Class<T> : Base<T> where T : struct
{
#nullable disable
    public override List<T?> P { get; set; } = default;
}
";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,47): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //     public virtual List<T?> P { get; set; } = default;
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "default").WithLocation(6, 47));
        }

        [Fact]
        public void Overriding_Indexer()
        {
            var source = @"
public class List<T> { }
public class Base
{
    public virtual List<string[]> this[List<string[]> x] { get => throw null!; set => throw null!; }
}
public class Class : Base
{
#nullable disable
    public override List<string[]> this[List<string[]> x] { get => throw null!; set => throw null!; }
}
public class Class2 : Base {
#nullable enable
    public override List<string[]> this[List<string[]> x] { get => throw null!; set => throw null!; }
}
";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Overriding_Indexer2()
        {
            var source = @"

#nullable enable
public class List<T> { }
public class Oblivious
{
#nullable disable
    public virtual List<string[]> this[List<string[]> x] { get => throw null!; set => throw null!; }
}
#nullable enable
public class Class : Oblivious
{
    public override List<string[]> this[List<string[]> x] { get => throw null!; set => throw null!; }
}
";
            var comp = CreateCompilation(new[] { source });
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Overriding_21()
        {
            var source = @"

class C
{
    public static void Main()
    { 
    }
}

abstract class A
{
    public abstract event System.Action<string> E1; 
    public abstract event System.Action<string>? E2; 
}

class B1 : A
{
#nullable disable annotations
    public override event System.Action<string?> E1 {add {} remove{}} // 1
#nullable disable annotations
    public override event System.Action<string> E2 {add {} remove{}}
}
#nullable enable
class B2 : A
{
#nullable disable annotations
    public override event System.Action<string?> E1; // 3
#nullable disable annotations
    public override event System.Action<string> E2;
#nullable enable
    void Dummy()
    {
        var e1 = E1;
        var e2 = E2;
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (19,47): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override event System.Action<string?> E1 {add {} remove{}} // 1
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(19, 47),
                // (19,50): warning CS8608: Nullability of reference types in type doesn't match overridden member.
                //     public override event System.Action<string?> E1 {add {} remove{}} // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnOverride, "E1").WithLocation(19, 50),
                // (27,47): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override event System.Action<string?> E1; // 3
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(27, 47),
                // (27,50): warning CS8608: Nullability of reference types in type doesn't match overridden member.
                //     public override event System.Action<string?> E1; // 3
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnOverride, "E1").WithLocation(27, 50)
                );
        }

        [Fact]
        public void Implementing_01()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

interface IA
{
    event System.Action<string> E1; 
    event System.Action<string>? E2; 
    event System.Action<string?>? E3; 
}

class B1 : IA
{
    public event System.Action<string?> E1 {add {} remove{}}
    public event System.Action<string> E2 {add {} remove{}}
    public event System.Action<string?>? E3 {add {} remove{}}
}

class B2 : IA
{
    public event System.Action<string?> E1; // 2
    public event System.Action<string> E2; // 2
    public event System.Action<string?>? E3; // 2

    void Dummy()
    {
        var e1 = E1;
        var e2 = E2;
        var e3 = E3;
    }
}

";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (26,40): warning CS8612: Nullability of reference types in type of 'event Action<string> B2.E2' doesn't match implicitly implemented member 'event Action<string>? IA.E2'.
                //     public event System.Action<string> E2; // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnImplicitImplementation, "E2").WithArguments("event Action<string> B2.E2", "event Action<string>? IA.E2").WithLocation(26, 40),
                // (25,41): warning CS8612: Nullability of reference types in type of 'event Action<string?> B2.E1' doesn't match implicitly implemented member 'event Action<string> IA.E1'.
                //     public event System.Action<string?> E1; // 2
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnImplicitImplementation, "E1").WithArguments("event Action<string?> B2.E1", "event Action<string> IA.E1").WithLocation(25, 41),
                // (19,40): warning CS8612: Nullability of reference types in type of 'event Action<string> B1.E2' doesn't match implicitly implemented member 'event Action<string>? IA.E2'.
                //     public event System.Action<string> E2 {add {} remove{}}
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnImplicitImplementation, "E2").WithArguments("event Action<string> B1.E2", "event Action<string>? IA.E2").WithLocation(19, 40),
                // (18,41): warning CS8612: Nullability of reference types in type of 'event Action<string?> B1.E1' doesn't match implicitly implemented member 'event Action<string> IA.E1'.
                //     public event System.Action<string?> E1 {add {} remove{}}
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnImplicitImplementation, "E1").WithArguments("event Action<string?> B1.E1", "event Action<string> IA.E1").WithLocation(18, 41),
                // (25,41): warning CS8618: Non-nullable event 'E1' is uninitialized. Consider declaring the event as nullable.
                //     public event System.Action<string?> E1; // 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "E1").WithArguments("event", "E1").WithLocation(25, 41),
                // (26,40): warning CS8618: Non-nullable event 'E2' is uninitialized. Consider declaring the event as nullable.
                //     public event System.Action<string> E2; // 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "E2").WithArguments("event", "E2").WithLocation(26, 40)
                );

            var ia = compilation.GetTypeByMetadataName("IA");

            foreach (string memberName in new[] { "E1", "E2" })
            {
                var member = ia.GetMember<EventSymbol>(memberName);

                foreach (string typeName in new[] { "B1", "B2" })
                {
                    var type = compilation.GetTypeByMetadataName(typeName);

                    var impl = (EventSymbol)type.FindImplementationForInterfaceMember(member);
                    Assert.False(impl.TypeWithAnnotations.Equals(member.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
                }
            }

            var e3 = ia.GetMember<EventSymbol>("E3");

            foreach (string typeName in new[] { "B1", "B2" })
            {
                var type = compilation.GetTypeByMetadataName(typeName);

                var impl = (EventSymbol)type.FindImplementationForInterfaceMember(e3);
                Assert.True(impl.TypeWithAnnotations.Equals(e3.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (string typeName in new[] { "IA", "B1", "B2" })
            {
                var type = compilation.GetTypeByMetadataName(typeName);

                foreach (var ev in type.GetMembers().OfType<EventSymbol>())
                {
                    Assert.True(ev.TypeWithAnnotations.Equals(ev.AddMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                    Assert.True(ev.TypeWithAnnotations.Equals(ev.RemoveMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                }
            }
        }

        [Fact]
        public void Implementing_02()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

interface IA
{
    event System.Action<string> E1; 
    event System.Action<string>? E2; 
    event System.Action<string?>? E3; 
}

class B1 : IA
{
    event System.Action<string?> IA.E1 {add {} remove{}}
    event System.Action<string> IA.E2 {add {} remove{}}
    event System.Action<string?>? IA.E3 {add {} remove{}}
}

interface IB
{
    //event System.Action<string> E1; 
    //event System.Action<string>? E2; 
    event System.Action<string?>? E3; 
}

class B2 : IB
{
    //event System.Action<string?> IB.E1; // 2
    //event System.Action<string> IB.E2; // 2
    event System.Action<string?>? IB.E3; // 2
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (34,38): error CS0071: An explicit interface implementation of an event must use event accessor syntax
                //     event System.Action<string?>? IB.E3; // 2
                Diagnostic(ErrorCode.ERR_ExplicitEventFieldImpl, "E3").WithLocation(34, 38),
                // (30,12): error CS0535: 'B2' does not implement interface member 'IB.E3.remove'
                // class B2 : IB
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "IB").WithArguments("B2", "IB.E3.remove").WithLocation(30, 12),
                // (30,12): error CS0535: 'B2' does not implement interface member 'IB.E3.add'
                // class B2 : IB
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "IB").WithArguments("B2", "IB.E3.add").WithLocation(30, 12),
                // (19,36): warning CS8615: Nullability of reference types in type doesn't match implemented member 'event Action<string>? IA.E2'.
                //     event System.Action<string> IA.E2 {add {} remove{}}
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnExplicitImplementation, "E2").WithArguments("event Action<string>? IA.E2").WithLocation(19, 36),
                // (18,37): warning CS8615: Nullability of reference types in type doesn't match implemented member 'event Action<string> IA.E1'.
                //     event System.Action<string?> IA.E1 {add {} remove{}}
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnExplicitImplementation, "E1").WithArguments("event Action<string> IA.E1").WithLocation(18, 37)
                );

            var ia = compilation.GetTypeByMetadataName("IA");
            var b1 = compilation.GetTypeByMetadataName("B1");

            foreach (string memberName in new[] { "E1", "E2" })
            {
                var member = ia.GetMember<EventSymbol>(memberName);

                var impl = (EventSymbol)b1.FindImplementationForInterfaceMember(member);
                Assert.False(impl.TypeWithAnnotations.Equals(member.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            var e3 = ia.GetMember<EventSymbol>("E3");
            {
                var impl = (EventSymbol)b1.FindImplementationForInterfaceMember(e3);
                Assert.True(impl.TypeWithAnnotations.Equals(e3.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (string typeName in new[] { "IA", "B1" })
            {
                var type = compilation.GetTypeByMetadataName(typeName);

                foreach (var ev in type.GetMembers().OfType<EventSymbol>())
                {
                    Assert.True(ev.TypeWithAnnotations.Equals(ev.AddMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                    Assert.True(ev.TypeWithAnnotations.Equals(ev.RemoveMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                }
            }
        }

        [Fact]
        public void Overriding_17()
        {
            var source =
@"#pragma warning disable 8618
class C
{
    public static void Main()
    { 
    }
}

abstract class A1
{
    public abstract string?[] P1 {get; set;} 
    public abstract string[] P2 {get; set;} 

    public abstract string?[] this[int x] {get; set;} 
    public abstract string[] this[short x] {get; set;} 
}

abstract class A2
{
    public abstract string?[]? P3 {get; set;} 

    public abstract string?[]? this[long x] {get; set;} 
}

class B1 : A1
{
    public override string[] P1 {get; set;} 
    public override string[]? P2 {get; set;} 
    
    public override string[] this[int x] // 1
    {
        get {throw new System.NotImplementedException();}
        set {}
    } 
    
    public override string[]? this[short x] // 2
    {
        get {throw new System.NotImplementedException();}
        set {}
    } 
}

class B2 : A2
{
    public override string?[]? P3 {get; set;} 
    
    public override string?[]? this[long x] // 3
    {
        get {throw new System.NotImplementedException();}
        set {}
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (27,39): warning CS8610: Nullability of reference types in type of parameter 'value' doesn't match overridden member.
                //     public override string[] P1 {get; set;} 
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("value").WithLocation(27, 39),
                // (28,35): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override string[]? P2 {get; set;} 
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "get").WithLocation(28, 35),
                // (33,9): warning CS8610: Nullability of reference types in type of parameter 'value' doesn't match overridden member.
                //         set {}
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("value").WithLocation(33, 9),
                // (38,9): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //         get {throw new System.NotImplementedException();}
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "get").WithLocation(38, 9)
                );

            foreach (var member in compilation.GetTypeByMetadataName("B1").GetMembers().OfType<PropertySymbol>())
            {
                Assert.False(member.TypeWithAnnotations.Equals(member.OverriddenProperty.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (var member in compilation.GetTypeByMetadataName("B2").GetMembers().OfType<PropertySymbol>())
            {
                Assert.True(member.TypeWithAnnotations.Equals(member.OverriddenProperty.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (string typeName in new[] { "A1", "B1", "A2", "B2" })
            {
                var type = compilation.GetTypeByMetadataName(typeName);

                foreach (var property in type.GetMembers().OfType<PropertySymbol>())
                {
                    Assert.True(property.TypeWithAnnotations.Equals(property.GetMethod.ReturnTypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                    Assert.True(property.TypeWithAnnotations.Equals(property.SetMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                }
            }
        }

        [Fact]
        public void Overriding_22()
        {
            var source =
@"#pragma warning disable 8618
class C
{
    public static void Main()
    { 
    }
}

abstract class A1
{
    public abstract string?[] P1 {get; set;} // 1
    public abstract string[] P2 {get; set;} 

    public abstract string?[] this[int x] {get; set;} // 2
    public abstract string[] this[short x] {get; set;} 
}

class B1 : A1
{
#nullable disable
    public override string[] P1 {get; set;}
#nullable disable
    public override string[]? P2 {get; set;} // 3

#nullable disable
    public override string[] this[int x]
    {
        get {throw new System.NotImplementedException();}
        set {}
    }

#nullable disable
    public override string[]? this[short x] // 4
    {
        get {throw new System.NotImplementedException();}
        set {}
    }
}
";
            var compilation = CreateCompilation(new[] { source });

            compilation.VerifyDiagnostics(
                // (14,27): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public abstract string?[] this[int x] {get; set;} // 2
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(14, 27),
                // (11,27): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public abstract string?[] P1 {get; set;} // 1
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(11, 27),
                // (23,29): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override string[]? P2 {get; set;} // 3
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(23, 29),
                // (33,29): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override string[]? this[short x] // 4
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(33, 29)
                );
        }

        [Fact]
        public void Implementing_03()
        {
            var source =
@"#pragma warning disable 8618
class C
{
    public static void Main()
    { 
    }
}
interface IA
{
    string?[] P1 {get; set;} 
    string[] P2 {get; set;} 
    string?[] this[int x] {get; set;} 
    string[] this[short x] {get; set;} 
}
interface IA2
{
    string?[]? P3 {get; set;} 
    string?[]? this[long x] {get; set;} 
}
class B : IA, IA2
{
    public string[] P1 {get; set;} 
    public string[]? P2 {get; set;} 
    public string?[]? P3 {get; set;} 

    public string[] this[int x]
    {
        get {throw new System.NotImplementedException();}
        set {} // 1
    } 

    public string[]? this[short x]
    {
        get {throw new System.NotImplementedException();} // 2
        set {}
    } 

    public string?[]? this[long x]
    {
        get {throw new System.NotImplementedException();}
        set {}
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (23,26): warning CS8766: Nullability of reference types in return type of 'string[]? B.P2.get' doesn't match implicitly implemented member 'string[] IA.P2.get' (possibly because of nullability attributes).
                //     public string[]? P2 {get; set;} 
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "get").WithArguments("string[]? B.P2.get", "string[] IA.P2.get").WithLocation(23, 26),
                // (29,9): warning CS8614: Nullability of reference types in type of parameter 'value' of 'void B.this[int x].set' doesn't match implicitly implemented member 'void IA.this[int x].set'.
                //         set {} // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnImplicitImplementation, "set").WithArguments("value", "void B.this[int x].set", "void IA.this[int x].set").WithLocation(29, 9),
                // (34,9): warning CS8766: Nullability of reference types in return type of 'string[]? B.this[short x].get' doesn't match implicitly implemented member 'string[] IA.this[short x].get' (possibly because of nullability attributes).
                //         get {throw new System.NotImplementedException();} // 2
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "get").WithArguments("string[]? B.this[short x].get", "string[] IA.this[short x].get").WithLocation(34, 9),
                // (22,30): warning CS8614: Nullability of reference types in type of parameter 'value' of 'void B.P1.set' doesn't match implicitly implemented member 'void IA.P1.set'.
                //     public string[] P1 {get; set;} 
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnImplicitImplementation, "set").WithArguments("value", "void B.P1.set", "void IA.P1.set").WithLocation(22, 30)
                );

            var b = compilation.GetTypeByMetadataName("B");

            foreach (var member in compilation.GetTypeByMetadataName("IA").GetMembers().OfType<PropertySymbol>())
            {
                var impl = (PropertySymbol)b.FindImplementationForInterfaceMember(member);
                Assert.False(impl.TypeWithAnnotations.Equals(member.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (var member in compilation.GetTypeByMetadataName("IA2").GetMembers().OfType<PropertySymbol>())
            {
                var impl = (PropertySymbol)b.FindImplementationForInterfaceMember(member);
                Assert.True(impl.TypeWithAnnotations.Equals(member.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (string typeName in new[] { "IA", "IA2", "B" })
            {
                var type = compilation.GetTypeByMetadataName(typeName);

                foreach (var property in type.GetMembers().OfType<PropertySymbol>())
                {
                    Assert.True(property.TypeWithAnnotations.Equals(property.GetMethod.ReturnTypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                    Assert.True(property.TypeWithAnnotations.Equals(property.SetMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                }
            }
        }

        [Fact]
        public void Implementing_04()
        {
            var source =
@"#pragma warning disable 8618
class C
{
    public static void Main()
    { 
    }
}
interface IA
{
    string?[] P1 {get; set;} 
    string[] P2 {get; set;} 
    string?[] this[int x] {get; set;} 
    string[] this[short x] {get; set;} 
}
interface IA2
{
    string?[]? P3 {get; set;} 
    string?[]? this[long x] {get; set;} 
}
class B : IA, IA2
{
    string[] IA.P1 {get; set;} 
    string[]? IA.P2 {get; set;} 
    string?[]? IA2.P3 {get; set;} 

    string[] IA.this[int x]
    {
        get {throw new System.NotImplementedException();}
        set {} // 1
    } 

    string[]? IA.this[short x]
    {
        get {throw new System.NotImplementedException();} // 2
        set {}
    } 

    string?[]? IA2.this[long x]
    {
        get {throw new System.NotImplementedException();}
        set {}
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (22,26): warning CS8617: Nullability of reference types in type of parameter 'value' doesn't match implemented member 'void IA.P1.set'.
                //     string[] IA.P1 {get; set;} 
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnExplicitImplementation, "set").WithArguments("value", "void IA.P1.set").WithLocation(22, 26),
                // (23,22): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'string[] IA.P2.get' (possibly because of nullability attributes).
                //     string[]? IA.P2 {get; set;} 
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "get").WithArguments("string[] IA.P2.get").WithLocation(23, 22),
                // (29,9): warning CS8617: Nullability of reference types in type of parameter 'value' doesn't match implemented member 'void IA.this[int x].set'.
                //         set {} // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnExplicitImplementation, "set").WithArguments("value", "void IA.this[int x].set").WithLocation(29, 9),
                // (34,9): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'string[] IA.this[short x].get' (possibly because of nullability attributes).
                //         get {throw new System.NotImplementedException();} // 2
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "get").WithArguments("string[] IA.this[short x].get").WithLocation(34, 9)
                );

            var b = compilation.GetTypeByMetadataName("B");

            foreach (var member in compilation.GetTypeByMetadataName("IA").GetMembers().OfType<PropertySymbol>())
            {
                var impl = (PropertySymbol)b.FindImplementationForInterfaceMember(member);
                Assert.False(impl.TypeWithAnnotations.Equals(member.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (var member in compilation.GetTypeByMetadataName("IA2").GetMembers().OfType<PropertySymbol>())
            {
                var impl = (PropertySymbol)b.FindImplementationForInterfaceMember(member);
                Assert.True(impl.TypeWithAnnotations.Equals(member.TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (string typeName in new[] { "IA", "IA2", "B" })
            {
                var type = compilation.GetTypeByMetadataName(typeName);

                foreach (var property in type.GetMembers().OfType<PropertySymbol>())
                {
                    Assert.True(property.TypeWithAnnotations.Equals(property.GetMethod.ReturnTypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                    Assert.True(property.TypeWithAnnotations.Equals(property.SetMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                }
            }
        }

        [Fact]
        public void Overriding_18()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

abstract class A
{
    public abstract string[] M1(); 
    public abstract T[] M2<T>() where T : class; 
    public abstract T?[]? M3<T>() where T : class; 
}

class B : A
{
    public override string?[] M1()
    {
        return new string?[] {};
    } 

    public override S?[] M2<S>()
    {
        return new S?[] {};
    } 

    public override S?[]? M3<S>()
    {
        return new S?[] {};
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (23,26): error CS0508: 'B.M2<S>()': return type must be 'S[]' to match overridden member 'A.M2<T>()'
                //     public override S?[] M2<S>()
                Diagnostic(ErrorCode.ERR_CantChangeReturnTypeOnOverride, "M2").WithArguments("B.M2<S>()", "A.M2<T>()", "S[]").WithLocation(23, 26),
                // (28,27): error CS0508: 'B.M3<S>()': return type must be 'S?[]' to match overridden member 'A.M3<T>()'
                //     public override S?[]? M3<S>()
                Diagnostic(ErrorCode.ERR_CantChangeReturnTypeOnOverride, "M3").WithArguments("B.M3<S>()", "A.M3<T>()", "S?[]").WithLocation(28, 27),
                // (18,31): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                //     public override string?[] M1()
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "M1").WithLocation(18, 31),
                // (23,26): error CS0453: The type 'S' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override S?[] M2<S>()
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M2").WithArguments("System.Nullable<T>", "T", "S").WithLocation(23, 26),
                // (28,27): error CS0453: The type 'S' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override S?[]? M3<S>()
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M3").WithArguments("System.Nullable<T>", "T", "S").WithLocation(28, 27));

            var b = compilation.GetTypeByMetadataName("B");

            foreach (string memberName in new[] { "M1", "M2", "M3" })
            {
                var member = b.GetMember<MethodSymbol>(memberName);
                Assert.False(member.ReturnTypeWithAnnotations.Equals(member.OverriddenMethod.ConstructIfGeneric(member.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t))).ReturnTypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }
        }

        [Fact]
        [WorkItem(28684, "https://github.com/dotnet/roslyn/issues/28684")]
        public void Overriding_23()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

abstract class A
{
    public abstract string[] M1(); 
    public abstract T[] M2<T>() where T : class; 
}

class B : A
{
#nullable disable annotations
    public override string?[] M1()
    {
        return new string?[] {};
    }

#nullable disable annotations
    public override S?[] M2<S>()
    {
        return new S?[] {};
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics(
                // (24,22): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override S?[] M2<S>()
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(24, 22),
                // (18,27): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override string?[] M1()
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(18, 27),
                // (24,26): error CS0508: 'B.M2<S>()': return type must be 'S[]' to match overridden member 'A.M2<T>()'
                //     public override S?[] M2<S>()
                Diagnostic(ErrorCode.ERR_CantChangeReturnTypeOnOverride, "M2").WithArguments("B.M2<S>()", "A.M2<T>()", "S[]").WithLocation(24, 26),
                // (18,31): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                //     public override string?[] M1()
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "M1").WithLocation(18, 31),
                // (24,26): error CS0453: The type 'S' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override S?[] M2<S>()
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M2").WithArguments("System.Nullable<T>", "T", "S").WithLocation(24, 26),
                // (20,26): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //         return new string?[] {};
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(20, 26),
                // (26,21): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //         return new S?[] {};
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(26, 21)
                );
        }

        [Fact]
        public void Implementing_05()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

interface IA
{
    string[] M1(); 
    T[] M2<T>() where T : class; 
    T?[]? M3<T>() where T : class; 
}

class B : IA
{
    public string?[] M1()
    {
        return new string?[] {};
    } 

    public S?[] M2<S>() where S : class
    {
        return new S?[] {};
    } 

    public S?[]? M3<S>() where S : class
    {
        return new S?[] {};
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (23,17): warning CS8613: Nullability of reference types in return type of 'S?[] B.M2<S>()' doesn't match implicitly implemented member 'T[] IA.M2<T>()'.
                //     public S?[] M2<S>() where S : class
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnImplicitImplementation, "M2").WithArguments("S?[] B.M2<S>()", "T[] IA.M2<T>()").WithLocation(23, 17),
                // (18,22): warning CS8613: Nullability of reference types in return type of 'string?[] B.M1()' doesn't match implicitly implemented member 'string[] IA.M1()'.
                //     public string?[] M1()
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnImplicitImplementation, "M1").WithArguments("string?[] B.M1()", "string[] IA.M1()").WithLocation(18, 22)
                );

            var ia = compilation.GetTypeByMetadataName("IA");
            var b = compilation.GetTypeByMetadataName("B");

            foreach (var memberName in new[] { "M1", "M2" })
            {
                var member = ia.GetMember<MethodSymbol>(memberName);
                var implementing = (MethodSymbol)b.FindImplementationForInterfaceMember(member);
                var implemented = member.ConstructIfGeneric(implementing.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t)));
                Assert.False(implementing.ReturnTypeWithAnnotations.Equals(implemented.ReturnTypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            {
                var member = ia.GetMember<MethodSymbol>("M3");
                var implementing = (MethodSymbol)b.FindImplementationForInterfaceMember(member);
                var implemented = member.ConstructIfGeneric(implementing.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t)));
                Assert.True(implementing.ReturnTypeWithAnnotations.Equals(implemented.ReturnTypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }
        }

        [Fact]
        public void Implementing_06()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

interface IA
{
    string[] M1(); 
    T[] M2<T>() where T : class; 
    T?[]? M3<T>() where T : class; 
}

class B : IA
{
    string?[] IA.M1()
    {
        return new string?[] {};
    } 

    S?[] IA.M2<S>()
    {
        return new S?[] {};
    } 

    S?[]? IA.M3<S>()
    {
        return new S?[] {};
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);

            compilation.VerifyDiagnostics(
                // (23,13): error CS0539: 'B.M2<S>()' in explicit interface declaration is not found among members of the interface that can be implemented
                //     S?[] IA.M2<S>()
                Diagnostic(ErrorCode.ERR_InterfaceMemberNotFound, "M2").WithArguments("B.M2<S>()").WithLocation(23, 13),
                // (28,14): error CS0539: 'B.M3<S>()' in explicit interface declaration is not found among members of the interface that can be implemented
                //     S?[]? IA.M3<S>()
                Diagnostic(ErrorCode.ERR_InterfaceMemberNotFound, "M3").WithArguments("B.M3<S>()").WithLocation(28, 14),
                // (18,18): warning CS8616: Nullability of reference types in return type doesn't match implemented member 'string[] IA.M1()'.
                //     string?[] IA.M1()
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "M1").WithArguments("string[] IA.M1()").WithLocation(18, 18),
                // (16,11): error CS0535: 'B' does not implement interface member 'IA.M3<T>()'
                // class B : IA
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "IA").WithArguments("B", "IA.M3<T>()").WithLocation(16, 11),
                // (16,11): error CS0535: 'B' does not implement interface member 'IA.M2<T>()'
                // class B : IA
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "IA").WithArguments("B", "IA.M2<T>()").WithLocation(16, 11),
                // (23,13): error CS0453: The type 'S' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     S?[] IA.M2<S>()
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M2").WithArguments("System.Nullable<T>", "T", "S").WithLocation(23, 13),
                // (28,14): error CS0453: The type 'S' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     S?[]? IA.M3<S>()
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M3").WithArguments("System.Nullable<T>", "T", "S").WithLocation(28, 14),
                // (25,20): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //         return new S?[] {};
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "S?").WithArguments("9.0").WithLocation(25, 20),
                // (30,20): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //         return new S?[] {};
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "S?").WithArguments("9.0").WithLocation(30, 20)
                );

            var ia = compilation.GetTypeByMetadataName("IA");
            var b = compilation.GetTypeByMetadataName("B");

            var member = ia.GetMember<MethodSymbol>("M1");
            var implementing = (MethodSymbol)b.FindImplementationForInterfaceMember(member);
            var implemented = member.ConstructIfGeneric(implementing.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t)));
            Assert.False(implementing.ReturnTypeWithAnnotations.Equals(implemented.ReturnTypeWithAnnotations,
                TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));

            member = ia.GetMember<MethodSymbol>("M2");
            Assert.Null(b.FindImplementationForInterfaceMember(member));

            member = ia.GetMember<MethodSymbol>("M3");
            Assert.Null(b.FindImplementationForInterfaceMember(member));
        }

        [Fact]
        public void ImplementingNonNullWithNullable_01()
        {
            var source = @"
interface IA
{
    string[] M1();
    T[] M2<T>() where T : class;
}

class B : IA
{
#nullable disable annotations
    string?[] IA.M1()
    {
        return new string?[] {};
    }

#nullable disable annotations
    S?[] IA.M2<S>()
    {
        return new S?[] {};
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);
            compilation.VerifyDiagnostics(
                // (17,6): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     S?[] IA.M2<S>()
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(17, 6),
                // (17,13): error CS0539: 'B.M2<S>()' in explicit interface declaration is not found among members of the interface that can be implemented
                //     S?[] IA.M2<S>()
                Diagnostic(ErrorCode.ERR_InterfaceMemberNotFound, "M2").WithArguments("B.M2<S>()").WithLocation(17, 13),
                // (11,11): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     string?[] IA.M1()
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(11, 11),
                // (11,18): warning CS8616: Nullability of reference types in return type doesn't match implemented member 'string[] IA.M1()'.
                //     string?[] IA.M1()
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "M1").WithArguments("string[] IA.M1()").WithLocation(11, 18),
                // (8,11): error CS0535: 'B' does not implement interface member 'IA.M2<T>()'
                // class B : IA
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "IA").WithArguments("B", "IA.M2<T>()").WithLocation(8, 11),
                // (17,13): error CS0453: The type 'S' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     S?[] IA.M2<S>()
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "M2").WithArguments("System.Nullable<T>", "T", "S").WithLocation(17, 13),
                // (13,26): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         return new string?[] {};
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(13, 26),
                // (19,21): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         return new S?[] {};
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(19, 21),
                // (19,20): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //         return new S?[] {};
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "S?").WithArguments("9.0").WithLocation(19, 20)
                );
        }

        [Fact]
        [WorkItem(28684, "https://github.com/dotnet/roslyn/issues/28684")]
        public void ImplementingNonNullWithNullable_02()
        {
            var source = @"
interface IA
{
    string[] M1();
    T[] M2<T>() where T : class;
}

class B : IA
{
#nullable disable annotations
    string?[] IA.M1()
    {
        return new string?[] {};
    }

#nullable disable annotations
    S?[] IA.M2<S>() where S : class
    {
        return new S?[] {};
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            var tree = compilation.SyntaxTrees[0];
            var model = compilation.GetSemanticModel(tree);
            var returnStatement = tree.GetRoot().DescendantNodes().OfType<ReturnStatementSyntax>().Skip(1).Single();
            AssertEx.Equal("S?[]", model.GetTypeInfo(returnStatement.Expression).Type.ToTestDisplayString());

            compilation.VerifyDiagnostics(
                // (17,6): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     S?[] IA.M2<S>() where S : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(17, 6),
                // (17,13): warning CS8616: Nullability of reference types in return type doesn't match implemented member 'T[] IA.M2<T>()'.
                //     S?[] IA.M2<S>() where S : class
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "M2").WithArguments("T[] IA.M2<T>()").WithLocation(17, 13),
                // (11,11): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     string?[] IA.M1()
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(11, 11),
                // (11,18): warning CS8616: Nullability of reference types in return type doesn't match implemented member 'string[] IA.M1()'.
                //     string?[] IA.M1()
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "M1").WithArguments("string[] IA.M1()").WithLocation(11, 18),
                // (13,26): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         return new string?[] {};
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(13, 26),
                // (19,21): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //         return new S?[] {};
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(19, 21)
                );
        }

        [Fact]
        public void ImplementingNullableWithNonNull_ReturnType()
        {
            var source = @"
interface IA
{
#nullable disable
    string?[] M1();
#nullable disable
    T?[] M2<T>() where T : class;
}
#nullable enable
class B : IA
{
    string[] IA.M1() => throw null!;
    S[] IA.M2<S>() => throw null!;
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);
            compilation.VerifyDiagnostics(
                // (7,6): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     T?[] M2<T>() where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(7, 6),
                // (7,5): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     T?[] M2<T>() where T : class;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(7, 5),
                // (5,11): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     string?[] M1();
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 11)
                );
        }

        [Fact]
        public void ImplementingNullableWithNonNull_Parameter()
        {
            var source = @"
interface IA
{
#nullable disable
    void M1(string?[] x);
#nullable disable
    void M2<T>(T?[] x) where T : class;
}
#nullable enable
class B : IA
{
    void IA.M1(string[] x) => throw null!;
    void IA.M2<S>(S[] x)
        => throw null!;
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);
            compilation.VerifyDiagnostics(
                // (5,19): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     void M1(string?[] x);
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 19),
                // (7,16): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     void M2<T>(T?[] x) where T : class;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(7, 16),
                // (7,17): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     void M2<T>(T?[] x) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(7, 17),
                // (12,13): warning CS8617: Nullability of reference types in type of parameter 'x' doesn't match implemented member 'void IA.M1(string?[] x)'.
                //     void IA.M1(string[] x) => throw null!;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnExplicitImplementation, "M1").WithArguments("x", "void IA.M1(string?[] x)").WithLocation(12, 13),
                // (13,13): warning CS8617: Nullability of reference types in type of parameter 'x' doesn't match implemented member 'void IA.M2<T>(T?[] x)'.
                //     void IA.M2<S>(S[] x)
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnExplicitImplementation, "M2").WithArguments("x", "void IA.M2<T>(T?[] x)").WithLocation(13, 13)
                );
        }

        [Fact]
        public void ImplementingObliviousWithNonNull()
        {
            var source = @"
interface IA
{
#nullable disable
    string[] M1();
#nullable disable
    T[] M2<T>() where T : class;
}
#nullable enable
class B : IA
{
    string[] IA.M1() => throw null!;
    S[] IA.M2<S>() => throw null!;
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics();
        }

        [Fact]
        public void Overriding_19()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

abstract class A
{
    public abstract void M1(string[] x); 
    public abstract void M2<T>(T[] x) where T : class; 
    public abstract void M3<T>(T?[]? x) where T : class; 
}

class B : A
{
    public override void M1(string?[] x)
    {
    } 

    public override void M2<T>(T?[] x)
    {
    } 

    public override void M3<T>(T?[]? x)
    {
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (22,26): error CS0115: 'B.M2<T>(T?[])': no suitable method found to override
                //     public override void M2<T>(T?[] x)
                Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M2").WithArguments("B.M2<T>(T?[])").WithLocation(22, 26),
                // (26,26): error CS0115: 'B.M3<T>(T?[]?)': no suitable method found to override
                //     public override void M3<T>(T?[]? x)
                Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M3").WithArguments("B.M3<T>(T?[]?)").WithLocation(26, 26),
                // (16,7): error CS0534: 'B' does not implement inherited abstract member 'A.M3<T>(T?[]?)'
                // class B : A
                Diagnostic(ErrorCode.ERR_UnimplementedAbstractMethod, "B").WithArguments("B", "A.M3<T>(T?[]?)").WithLocation(16, 7),
                // (16,7): error CS0534: 'B' does not implement inherited abstract member 'A.M2<T>(T[])'
                // class B : A
                Diagnostic(ErrorCode.ERR_UnimplementedAbstractMethod, "B").WithArguments("B", "A.M2<T>(T[])").WithLocation(16, 7),
                // (22,37): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override void M2<T>(T?[] x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(22, 37),
                // (26,38): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override void M3<T>(T?[]? x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(26, 38)
                );

            var b = compilation.GetTypeByMetadataName("B");

            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.False(m1.Parameters[0].TypeWithAnnotations.Equals(m1.OverriddenMethod.ConstructIfGeneric(m1.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t))).Parameters[0].TypeWithAnnotations,
                TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));

            var m2 = b.GetMember<MethodSymbol>("M2");
            Assert.Null(m2.OverriddenMethod);

            var m3 = b.GetMember<MethodSymbol>("M3");
            Assert.Null(m3.OverriddenMethod);
        }

        [Fact]
        [WorkItem(28684, "https://github.com/dotnet/roslyn/issues/28684")]
        public void Overriding_24()
        {
            var source = @"
#nullable enable

abstract class A
{
    public abstract void M1(string[] x);
    public abstract void M2<T>(T[] x) where T : class;
}

class B : A
{
#nullable disable
    public override void M1(string?[] x)
    {
    }

#nullable disable
    public override void M2<T>(T?[] x)
    {
    }
}
";
            var compilation = CreateCompilation(new[] { source });

            compilation.VerifyDiagnostics(
                // (18,33): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override void M2<T>(T?[] x)
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(18, 33),
                // (13,35): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override void M1(string?[] x)
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(13, 35),
                // (18,26): error CS0115: 'B.M2<T>(T?[])': no suitable method found to override
                //     public override void M2<T>(T?[] x)
                Diagnostic(ErrorCode.ERR_OverrideNotExpected, "M2").WithArguments("B.M2<T>(T?[])").WithLocation(18, 26),
                // (10,7): error CS0534: 'B' does not implement inherited abstract member 'A.M2<T>(T[])'
                // class B : A
                Diagnostic(ErrorCode.ERR_UnimplementedAbstractMethod, "B").WithArguments("B", "A.M2<T>(T[])").WithLocation(10, 7),
                // (18,37): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     public override void M2<T>(T?[] x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(18, 37)
                );
        }

        [Fact]
        public void Overriding_25()
        {
            var ilSource = @"
.assembly extern mscorlib
{
  .publickeytoken = (B7 7A 5C 56 19 34 E0 89 )                         // .z\V.4..
  .ver 4:0:0:0
}

.assembly '<<GeneratedFileName>>'
{
}

.module '<<GeneratedFileName>>.dll'

.class public auto ansi beforefieldinit C`2<T,S>
       extends [mscorlib]System.Object
{
  .method public hidebysig specialname rtspecialname 
          instance void  .ctor() cil managed
  {
    // Code size       8 (0x8)
    .maxstack  8
    IL_0000:  ldarg.0
    IL_0001:  call       instance void [mscorlib]System.Object::.ctor()
    IL_0006:  nop
    IL_0007:  ret
  } // end of method C`2::.ctor

} // end of class C`2

.class public abstract auto ansi beforefieldinit A
       extends [mscorlib]System.Object
{
  .method public hidebysig newslot abstract virtual 
          instance class C`2<string modopt([mscorlib]System.Runtime.CompilerServices.IsConst),string> 
          M1() cil managed
  {
    .param [0]
    .custom instance void System.Runtime.CompilerServices.NullableAttribute::.ctor(uint8[]) = ( 01 00 03 00 00 00 01 02 01 00 00 ) 
  } // end of method A::M1

  .method family hidebysig specialname rtspecialname 
          instance void  .ctor() cil managed
  {
    // Code size       8 (0x8)
    .maxstack  8
    IL_0000:  ldarg.0
    IL_0001:  call       instance void [mscorlib]System.Object::.ctor()
    IL_0006:  nop
    IL_0007:  ret
  } // end of method A::.ctor

} // end of class A

.class public auto ansi beforefieldinit System.Runtime.CompilerServices.NullableAttribute
       extends [mscorlib]System.Attribute
{
  .custom instance void [mscorlib]System.AttributeUsageAttribute::.ctor(valuetype [mscorlib]System.AttributeTargets) = ( 01 00 86 6B 00 00 01 00 54 02 0D 41 6C 6C 6F 77   // ...k....T..Allow
                                                                                                                         4D 75 6C 74 69 70 6C 65 00 )              // Multiple.
  .method public hidebysig specialname rtspecialname 
          instance void  .ctor(uint8 transformFlag) cil managed
  {
    // Code size       9 (0x9)
    .maxstack  8
    IL_0000:  ldarg.0
    IL_0001:  call       instance void [mscorlib]System.Attribute::.ctor()
    IL_0006:  nop
    IL_0007:  nop
    IL_0008:  ret
  } // end of method NullableAttribute::.ctor

  .method public hidebysig specialname rtspecialname 
          instance void  .ctor(uint8[] transformFlags) cil managed
  {
    // Code size       9 (0x9)
    .maxstack  8
    IL_0000:  ldarg.0
    IL_0001:  call       instance void [mscorlib]System.Attribute::.ctor()
    IL_0006:  nop
    IL_0007:  nop
    IL_0008:  ret
  } // end of method NullableAttribute::.ctor

} // end of class System.Runtime.CompilerServices.NullableAttribute
";

            var source = @"
class C
{
    public static void Main()
    { 
    }
}

class B : A
{
    public override C<string, string?> M1()
    {
        return new C<string, string?>();
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, new[] { CompileIL(ilSource, prependDefaultHeader: false) },
                                                            options: WithNullableEnable(),
                                                            parseOptions: TestOptions.Regular8);

            var m1 = compilation.GetTypeByMetadataName("B").GetMember<MethodSymbol>("M1");
            Assert.Equal("C<System.String? modopt(System.Runtime.CompilerServices.IsConst), System.String>", m1.OverriddenMethod.ReturnTypeWithAnnotations.ToTestDisplayString());
            Assert.Equal("C<System.String modopt(System.Runtime.CompilerServices.IsConst), System.String?>", m1.ReturnTypeWithAnnotations.ToTestDisplayString());

            compilation.VerifyDiagnostics(
                 // (11,40): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                 //     public override C<string, string?> M1()
                 Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "M1").WithLocation(11, 40)
                );
        }

        [Fact]
        public void Overriding_26()
        {
            var source = @"
class A
{
    public virtual void M1<T>(T? x) where T : class 
    { 
    }

    public virtual T? M2<T>() where T : class 
    { 
        return null;
    }
}

class B : A
{
    public override void M1<T>(T? x) where T : class
    {
    }

    public override T? M2<T>() where T : class
    { 
        return null;
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics();

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.False(m1.Parameters[0].Type.IsNullableType());
            Assert.Equal(NullableAnnotation.Annotated, m1.Parameters[0].TypeWithAnnotations.NullableAnnotation);
            Assert.True(m1.Parameters[0].Type.IsReferenceType);
            Assert.False(m1.OverriddenMethod.Parameters[0].Type.IsNullableType());

            var m2 = b.GetMember<MethodSymbol>("M2");
            Assert.False(m2.ReturnType.IsNullableType());
            Assert.Equal(NullableAnnotation.Annotated, m2.ReturnTypeWithAnnotations.NullableAnnotation);
            Assert.True(m2.ReturnType.IsReferenceType);
            Assert.False(m2.OverriddenMethod.ReturnType.IsNullableType());
        }

        [Fact]
        public void Overriding_27()
        {
            var source = @"
class A
{
    public virtual void M1<T>(System.Nullable<T> x) where T : struct
    {
    }

    public virtual void M2<T>(T? x) where T : struct
    {
    }

    public virtual void M3<T>(C<T?> x) where T : struct
    {
    }

    public virtual void M4<T>(C<System.Nullable<T>> x) where T : struct
    {
    }

    public virtual void M5<T>(C<T?> x) where T : class
    {
    }
}

class B : A
{
    public override void M1<T>(T? x)
    {
    }

    public override void M2<T>(T? x)
    {
    }

    public override void M3<T>(C<T?> x)
    {
    }

    public override void M4<T>(C<T?> x)
    {
    }

    public override void M5<T>(C<T?> x) where T : class
    {
    }
}

class C<T> {}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics();

            var b = compilation.GetTypeByMetadataName("B");
            var m3 = b.GetMember<MethodSymbol>("M3");
            var m4 = b.GetMember<MethodSymbol>("M4");
            var m5 = b.GetMember<MethodSymbol>("M5");
            Assert.True(((NamedTypeSymbol)m3.Parameters[0].Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
            Assert.True(((NamedTypeSymbol)m3.OverriddenMethod.Parameters[0].Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
            Assert.True(((NamedTypeSymbol)m4.Parameters[0].Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
            Assert.True(((NamedTypeSymbol)m4.OverriddenMethod.Parameters[0].Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
            Assert.False(((NamedTypeSymbol)m5.Parameters[0].Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
            Assert.False(((NamedTypeSymbol)m5.OverriddenMethod.Parameters[0].Type).TypeArgumentsWithAnnotationsNoUseSiteDiagnostics[0].IsNullableType());
        }

        [Fact]
        public void Overriding_28()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

abstract class A
{
    public abstract string[] M1(); 
    public abstract T[] M2<T>() where T : class; 
    public abstract T?[]? M3<T>() where T : class; 
}

class B : A
{
    public override string?[] M1()
    {
        return new string?[] {};
    } 

    public override S?[] M2<S>() where S : class
    {
        return new S?[] {};
    } 

    public override S?[]? M3<S>() where S : class
    {
        return new S?[] {};
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                 // (23,26): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                 //     public override S?[] M2<S>()
                 Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "M2").WithLocation(23, 26),
                 // (18,31): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                 //     public override string?[] M1()
                 Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "M1").WithLocation(18, 31)
                );

            var b = compilation.GetTypeByMetadataName("B");
            foreach (string memberName in new[] { "M1", "M2" })
            {
                var member = b.GetMember<MethodSymbol>(memberName);
                Assert.False(member.ReturnTypeWithAnnotations.Equals(member.OverriddenMethod.ConstructIfGeneric(member.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t))).ReturnTypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            var m3 = b.GetMember<MethodSymbol>("M3");
            Assert.True(m3.ReturnTypeWithAnnotations.Equals(m3.OverriddenMethod.ConstructIfGeneric(m3.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t))).ReturnTypeWithAnnotations,
                TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
        }

        [Fact]
        [WorkItem(28684, "https://github.com/dotnet/roslyn/issues/28684")]
        public void Overriding_29()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

abstract class A
{
    public abstract string[] M1(); 
    public abstract T[] M2<T>() where T : class; 
}

class B : A
{
#nullable disable annotations
    public override string?[] M1()
    {
        return new string?[] {};
    }

#nullable disable annotations
    public override S?[] M2<S>() where S : class
    {
        return new S?[] {};
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics(
                // (18,31): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                //     public override string?[] M1()
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "M1").WithLocation(18, 31),
                // (24,22): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override S?[] M2<S>()
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(24, 22),
                // (24,26): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                //     public override S?[] M2<S>()
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "M2").WithLocation(24, 26),
                // (18,27): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override string?[] M1()
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(18, 27),
                // (20,26): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //         return new string?[] {};
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(20, 26),
                // (26,21): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //         return new S?[] {};
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(26, 21)
                );
        }

        [Fact]
        public void Overriding_30()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

abstract class A
{
    public abstract void M1(string[] x); 
    public abstract void M2<T>(T[] x) where T : class; 
    public abstract void M3<T>(T?[]? x) where T : class; 
}

class B : A
{
    public override void M1(string?[] x)
    {
    } 

    public override void M2<T>(T?[] x) where T : class
    {
    } 

    public override void M3<T>(T?[]? x) where T : class
    {
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                );

            var b = compilation.GetTypeByMetadataName("B");
            foreach (string memberName in new[] { "M1", "M2" })
            {
                var member = b.GetMember<MethodSymbol>(memberName);
                Assert.False(member.Parameters[0].TypeWithAnnotations.Equals(member.OverriddenMethod.ConstructIfGeneric(member.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t))).Parameters[0].TypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            var m3 = b.GetMember<MethodSymbol>("M3");
            Assert.True(m3.Parameters[0].TypeWithAnnotations.Equals(m3.OverriddenMethod.ConstructIfGeneric(m3.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t))).Parameters[0].TypeWithAnnotations,
                TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
        }

        [Fact]
        [WorkItem(28684, "https://github.com/dotnet/roslyn/issues/28684")]
        public void Overriding_31()
        {
            var source = @"
#nullable enable

abstract class A
{
    public abstract void M1(string[] x);
    public abstract void M2<T>(T[] x) where T : class;
}

class B : A
{
#nullable disable
    public override void M1(string?[] x)
    {
    }

#nullable disable
    public override void M2<T>(T?[] x) where T : class
    {
    }
}
";
            var compilation = CreateCompilation(new[] { source });

            compilation.VerifyDiagnostics(
                // (18,33): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override void M2<T>(T?[] x)
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(18, 33),
                // (13,35): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' context.
                //     public override void M1(string?[] x)
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(13, 35)
                );
        }

        [Fact]
        [WorkItem(29847, "https://github.com/dotnet/roslyn/issues/29847")]
        public void Overriding_32()
        {
            var source = @"
class A
{
    public virtual void M1<T>(T? x) where T : struct 
    { 
    }

    public virtual void M1<T>(T? x) where T : class 
    { 
    }
}

class B : A
{
    public override void M1<T>(T? x) where T : class
    {
    }
} 

class C
{
    public virtual void M2<T>(T? x) where T : class 
    { 
    }

    public virtual void M2<T>(T? x) where T : struct 
    { 
    }
}

class D : C
{
    public override void M2<T>(T? x) where T : class
    {
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics();

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.False(m1.Parameters[0].Type.IsNullableType());
            Assert.False(m1.OverriddenMethod.Parameters[0].Type.IsNullableType());

            var d = compilation.GetTypeByMetadataName("D");
            var m2 = d.GetMember<MethodSymbol>("M2");
            Assert.False(m2.Parameters[0].Type.IsNullableType());
            Assert.False(m2.OverriddenMethod.Parameters[0].Type.IsNullableType());
        }

        [Fact]
        [WorkItem(29847, "https://github.com/dotnet/roslyn/issues/29847")]
        public void Overriding_33()
        {
            var source = @"
class A
{
    public virtual void M1<T>(T? x) where T : struct 
    { 
    }

    public virtual void M1<T>(T? x) where T : class 
    { 
    }
}

class B : A
{
    public override void M1<T>(T? x) where T : struct
    {
    }
} 

class C
{
    public virtual void M2<T>(T? x) where T : class 
    { 
    }

    public virtual void M2<T>(T? x) where T : struct 
    { 
    }
}

class D : C
{
    public override void M2<T>(T? x) where T : struct
    {
    }
} 
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics();

            var b = compilation.GetTypeByMetadataName("B");
            var m1 = b.GetMember<MethodSymbol>("M1");
            Assert.True(m1.Parameters[0].Type.IsNullableType());
            Assert.True(m1.OverriddenMethod.Parameters[0].Type.IsNullableType());

            var d = compilation.GetTypeByMetadataName("D");
            var m2 = d.GetMember<MethodSymbol>("M2");
            Assert.True(m2.Parameters[0].Type.IsNullableType());
            Assert.True(m2.OverriddenMethod.Parameters[0].Type.IsNullableType());
        }

        [Fact]
        [WorkItem(33276, "https://github.com/dotnet/roslyn/issues/33276")]
        public void Overriding_34()
        {
            var source1 =
@"
public class MyEntity
{
}

public abstract class BaseController<T>
    where T : MyEntity
{
    public abstract void SomeMethod<R>(R? lite)
        where R : MyEntity;
}
";

            var source2 =
@"
class DerivedController<T> : BaseController<T>
    where T : MyEntity
{
    Table<T> table = null!;
    public override void SomeMethod<R>(R? lite) where R : class
    {
        table.OtherMethod(lite);
    }
}

class Table<T>
    where T : MyEntity
{
    public void OtherMethod<R>(R? lite) where R : MyEntity
    {
        lite?.ToString();
    }
}
";
            var compilation1 = CreateCompilation(new[] { source1 }, options: WithNullableEnable());
            compilation1.VerifyDiagnostics();

            foreach (var reference in new[] { compilation1.ToMetadataReference(), compilation1.EmitToImageReference() })
            {
                var compilation2 = CreateCompilation(new[] { source2 }, options: WithNullableEnable(),
                                                     references: new[] { reference });
                compilation2.VerifyDiagnostics();
                CompileAndVerify(compilation2);
            }
        }

        [Fact]
        [WorkItem(31676, "https://github.com/dotnet/roslyn/issues/31676")]
        public void Overriding_35()
        {
            var source1 =
@"
using System;

namespace Microsoft.EntityFrameworkCore.TestUtilities
{
    public abstract class QueryAsserterBase
    {
        public abstract void AssertQueryScalar<TItem1, TResult>(
            Func<IQueryable<TItem1>, IQueryable<Nullable<TResult>>> actualQuery)
            where TResult : struct;
    }
}

public interface IQueryable<out T>
{
}
";

            var source2 =
@"
using System;

namespace Microsoft.EntityFrameworkCore.TestUtilities
{
    public class QueryAsserter<TContext> : QueryAsserterBase
    {
        public override void AssertQueryScalar<TItem1, TResult>(
            Func<IQueryable<TItem1>, IQueryable<TResult?>> actualQuery)
        {          
        }
    }
}
";
            foreach (var options1 in new[] { WithNullableEnable(), WithNullableDisable() })
            {
                var compilation1 = CreateCompilation(new[] { source1 }, options: options1);
                compilation1.VerifyDiagnostics();

                foreach (var reference in new[] { compilation1.ToMetadataReference(), compilation1.EmitToImageReference() })
                {
                    foreach (var options2 in new[] { WithNullableEnable(), WithNullableDisable() })
                    {
                        var compilation2 = CreateCompilation(new[] { source2 }, options: options2,
                                                         references: new[] { reference });
                        compilation2.VerifyDiagnostics();
                        CompileAndVerify(compilation2);
                    }
                }

                var compilation3 = CreateCompilation(new[] { source1, source2 }, options: options1);
                compilation3.VerifyDiagnostics();
                CompileAndVerify(compilation3);

                var compilation4 = CreateCompilation(new[] { source2, source1 }, options: options1);
                compilation4.VerifyDiagnostics();
                CompileAndVerify(compilation4);
            }
        }

        [Fact]
        [WorkItem(38403, "https://github.com/dotnet/roslyn/issues/38403")]
        public void Overriding_36()
        {
            var source =
@"
#nullable enable
using System;
using System.Collections.Generic;
using System.Text;

public abstract class QueryAsserterBase
{
    public abstract void AssertQueryScalar<TItem1, TResult>(
        Func<IQueryable<TItem1>, IQueryable<Nullable<TResult>>> actualQuery)
        where TResult : struct;
}

public class QueryAsserter<TContext> : QueryAsserterBase
{
    public override void AssertQueryScalar<TItem1, TResult>(
        Func<IQueryable<TItem1>, IQueryable<TResult?>> actualQuery)
    {          
    }
}
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            comp.VerifyDiagnostics(
                // (4,1): hidden CS8019: Unnecessary using directive.
                // using System.Collections.Generic;
                Diagnostic(ErrorCode.HDN_UnusedUsingDirective, "using System.Collections.Generic;").WithLocation(4, 1),
                // (5,1): hidden CS8019: Unnecessary using directive.
                // using System.Text;
                Diagnostic(ErrorCode.HDN_UnusedUsingDirective, "using System.Text;").WithLocation(5, 1),
                // (10,14): error CS0246: The type or namespace name 'IQueryable<>' could not be found (are you missing a using directive or an assembly reference?)
                //         Func<IQueryable<TItem1>, IQueryable<Nullable<TResult>>> actualQuery)
                Diagnostic(ErrorCode.ERR_SingleTypeNameNotFound, "IQueryable<TItem1>").WithArguments("IQueryable<>").WithLocation(10, 14),
                // (10,34): error CS0246: The type or namespace name 'IQueryable<>' could not be found (are you missing a using directive or an assembly reference?)
                //         Func<IQueryable<TItem1>, IQueryable<Nullable<TResult>>> actualQuery)
                Diagnostic(ErrorCode.ERR_SingleTypeNameNotFound, "IQueryable<Nullable<TResult>>").WithArguments("IQueryable<>").WithLocation(10, 34),
                // (17,14): error CS0246: The type or namespace name 'IQueryable<>' could not be found (are you missing a using directive or an assembly reference?)
                //         Func<IQueryable<TItem1>, IQueryable<TResult?>> actualQuery)
                Diagnostic(ErrorCode.ERR_SingleTypeNameNotFound, "IQueryable<TItem1>").WithArguments("IQueryable<>").WithLocation(17, 14),
                // (17,34): error CS0246: The type or namespace name 'IQueryable<>' could not be found (are you missing a using directive or an assembly reference?)
                //         Func<IQueryable<TItem1>, IQueryable<TResult?>> actualQuery)
                Diagnostic(ErrorCode.ERR_SingleTypeNameNotFound, "IQueryable<TResult?>").WithArguments("IQueryable<>").WithLocation(17, 34)
                );
        }

        [Fact]
        [WorkItem(38403, "https://github.com/dotnet/roslyn/issues/38403")]
        public void Overriding_37()
        {
            var source =
@"
#nullable enable
using System;
using System.Collections.Generic;
using System.Text;
using System.Linq;

public abstract class QueryAsserterBase
{
    public abstract void AssertQueryScalar<TItem1, TResult>(
        Func<IQueryable<TItem1>, IQueryable<Nullable<TResult>>> actualQuery)
        where TResult : struct;
}

public class QueryAsserter<TContext> : QueryAsserterBase
{
    public override void AssertQueryScalar<TItem1, TResult>(
        Func<IQueryable<TItem1>, IQueryable<TResult?>> actualQuery)
    {          
    }
}
";
            var comp = CreateCompilation(source, options: TestOptions.DebugDll);
            CompileAndVerify(comp);
        }

        [Fact]
        public void Override_NullabilityCovariance_01()
        {
            var src = @"
using System;

interface I<out T> {}
class A
{
    public virtual object? M() => null;
    public virtual (object?, object?) M2() => throw null!;
    public virtual I<object?> M3() => throw null!;
    public virtual ref object? M4() => throw null!;
    public virtual ref readonly object? M5() => throw null!;
    public virtual Func<object>? P1 { get; set; } = null!;
    public virtual Func<object?> P2 { get; set; } = null!;
    public virtual object? P3 => null!;
    public virtual event Func<object>? E1 { add {} remove {} }
    public virtual event Func<object?> E2 { add {} remove {} }
}
class B : A
{
    public override object M() => null!;
    public override (object, object) M2() => throw null!;
    public override I<object> M3() => throw null!;
    public override ref object M4() => throw null!; // warn
    public override ref readonly object M5() => throw null!;
    public override Func<object> P1 { get; set; } = null!; // warn
    public override Func<object> P2 { get; set; } = null!; // warn
    public override object P3 => null!; // ok
    public override event Func<object> E1 { add {} remove {} } // warn
    public override event Func<object> E2 { add {} remove {} } // warn
}
class C : B
{
    public override object? M() => null; // warn
    public override (object?, object?) M2() => throw null!; // warn
    public override I<object?> M3() => throw null!; // warn
    public override ref object? M4() => throw null!; // warn
    public override ref readonly object? M5() => throw null!; // warn
    public override Func<object>? P1 { get; set; } = null!; // warn
    public override Func<object?> P2 { get; set; } = null!; // warn
    public override object? P3 => null!; // warn
    public override event Func<object>? E1 { add {} remove {} } // ok
    public override event Func<object?> E2 { add {} remove {} } // ok
}
class D : C
{
    public override object M() => null!;
    public override (object, object) M2() => throw null!;
    public override I<object> M3() => throw null!;
    public override ref object M4() => throw null!; // warn
    public override ref readonly object M5() => throw null!;
    public override Func<object> P1 { get; set; } = null!; // warn
    public override Func<object> P2 { get; set; } = null!; // warn
    public override object P3 => null!; // ok
    public override event Func<object> E1 { add {} remove {} } // ok
    public override event Func<object> E2 { add {} remove {} } // ok
}
class E : D
{
    public override object? M() => null; // warn
    public override (object?, object?) M2() => throw null!; // warn
    public override I<object?> M3() => throw null!; // warn
    public override ref object? M4() => throw null!; // warn
    public override ref readonly object? M5() => throw null!; // warn
    public override Func<object>? P1 { get; set; } = null!; // warn
    public override Func<object?> P2 { get; set; } = null!; // warn
    public override object? P3 => null!; // warn
    public override event Func<object>? E1 { add {} remove {} } // warn
    public override event Func<object?> E2 { add {} remove {} } // warn
}";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (23,32): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override ref object M4() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "M4").WithLocation(23, 32),
                // (25,44): warning CS8765: Type of parameter 'value' doesn't match overridden member because of nullability attributes.
                //     public override Func<object> P1 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("value").WithLocation(25, 44),
                // (26,44): warning CS8610: Nullability of reference types in type of parameter 'value' doesn't match overridden member.
                //     public override Func<object> P2 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("value").WithLocation(26, 44),
                // (28,40): warning CS8608: Nullability of reference types in type doesn't match overridden member.
                //     public override event Func<object> E1 { add {} remove {} } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnOverride, "E1").WithLocation(28, 40),
                // (29,40): warning CS8608: Nullability of reference types in type doesn't match overridden member.
                //     public override event Func<object> E2 { add {} remove {} } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnOverride, "E2").WithLocation(29, 40),
                // (33,29): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override object? M() => null; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "M").WithLocation(33, 29),
                // (34,40): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                //     public override (object?, object?) M2() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "M2").WithLocation(34, 40),
                // (35,32): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                //     public override I<object?> M3() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "M3").WithLocation(35, 32),
                // (36,33): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override ref object? M4() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "M4").WithLocation(36, 33),
                // (37,42): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override ref readonly object? M5() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "M5").WithLocation(37, 42),
                // (38,40): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override Func<object>? P1 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "get").WithLocation(38, 40),
                // (39,40): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                //     public override Func<object?> P2 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "get").WithLocation(39, 40),
                // (40,35): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override object? P3 => null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "null!").WithLocation(40, 35),
                // (49,32): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override ref object M4() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "M4").WithLocation(49, 32),
                // (51,44): warning CS8765: Type of parameter 'value' doesn't match overridden member because of nullability attributes.
                //     public override Func<object> P1 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("value").WithLocation(51, 44),
                // (52,44): warning CS8610: Nullability of reference types in type of parameter 'value' doesn't match overridden member.
                //     public override Func<object> P2 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("value").WithLocation(52, 44),
                // (54,40): warning CS8608: Nullability of reference types in type doesn't match overridden member.
                //     public override event Func<object> E1 { add {} remove {} } // ok
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnOverride, "E1").WithLocation(54, 40),
                // (55,40): warning CS8608: Nullability of reference types in type doesn't match overridden member.
                //     public override event Func<object> E2 { add {} remove {} } // ok
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnOverride, "E2").WithLocation(55, 40),
                // (59,29): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override object? M() => null; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "M").WithLocation(59, 29),
                // (60,40): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                //     public override (object?, object?) M2() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "M2").WithLocation(60, 40),
                // (61,32): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                //     public override I<object?> M3() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "M3").WithLocation(61, 32),
                // (62,33): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override ref object? M4() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "M4").WithLocation(62, 33),
                // (63,42): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override ref readonly object? M5() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "M5").WithLocation(63, 42),
                // (64,40): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override Func<object>? P1 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "get").WithLocation(64, 40),
                // (65,40): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                //     public override Func<object?> P2 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "get").WithLocation(65, 40),
                // (66,35): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override object? P3 => null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "null!").WithLocation(66, 35)
                );
        }

        [Fact]
        public void Override_NullabilityCovariance_02()
        {
            var src = @"
using System;

class A
{
    public virtual Func<object>? P1 { get; set; } = null!;
}
class B : A
{
    public override Func<object> P1 { get => null!; }
}
class C : B
{
    public override Func<object> P1 { get; set; } = null!; // warn
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (14,44): warning CS8765: Type of parameter 'value' doesn't match overridden member because of nullability attributes.
                //     public override Func<object> P1 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("value").WithLocation(14, 44)
                );
        }

        [Fact]
        public void Override_NullabilityCovariance_03()
        {
            var src = @"
using System;

class B
{
    public virtual Func<object> P1 { get; set; } = null!;
}
class C : B
{
    public override Func<object>? P1 { set {} }
}
class D : C
{
    public override Func<object>? P1 { get; set; } = null!; // warn
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (14,40): warning CS8764: Nullability of return type doesn't match overridden member (possibly because of nullability attributes).
                //     public override Func<object>? P1 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "get").WithLocation(14, 40)
                );
        }

        [Fact]
        public void Override_NullabilityContravariance_01()
        {
            var src = @"
interface I<out T> {}
class A
{
    public virtual void M(object o) { }
    public virtual void M2((object, object) t) { }
    public virtual void M3(I<object> i) { }
    public virtual void M4(ref object o) { }
    public virtual void M5(out object o) => throw null!;
    public virtual void M6(in object o) { }
    public virtual object this[object o] { get => null!; set { } }
    public virtual string this[string s] => null!;
    public virtual string this[int[] a] { set { } }
}
class B : A
{
    public override void M(object? o) { }
    public override void M2((object?, object?) t) { }
    public override void M3(I<object?> i) { }
    public override void M4(ref object? o) { } // warn
    public override void M5(out object? o) => throw null!; // warn
    public override void M6(in object? o) { }
    public override object? this[object? o] { get => null!; set { } }
    public override string this[string? s] => null!;
    public override string this[int[]? a] { set { } }
}
class C : B
{
    public override void M(object o) { } // warn
    public override void M2((object, object) t) { } // warn
    public override void M3(I<object> i) { } // warn
    public override void M4(ref object o) { } // warn
    public override void M5(out object o) => throw null!;
    public override void M6(in object o) { } // warn
    public override object this[object o] { get => null!; set { } }
    public override string this[string s] => null!;
    public override string this[int[] a] { set { } }
}
class D : C
{
    public override void M(object? o) { }
    public override void M2((object?, object?) t) { }
    public override void M3(I<object?> i) { }
    public override void M4(ref object? o) { } // warn
    public override void M5(out object? o) => throw null!; // warn
    public override void M6(in object? o) { }
    public override object? this[object? o] { get => null!; set { } }
    public override string this[string? s] => null!;
    public override string this[int[]? a] { set { } }
}
class E : D
{
    public override void M(object o) { } // warn
    public override void M2((object, object) t) { } // warn
    public override void M3(I<object> i) { } // warn
    public override void M4(ref object o) { } // warn
    public override void M5(out object o) => throw null!;
    public override void M6(in object o) { } // warn
    public override object this[object o] { get => null!; set { } }
    public override string this[string s] => null!;
    public override string this[int[] a] { set { } }
}";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (20,26): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override void M4(ref object? o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M4").WithArguments("o").WithLocation(20, 26),
                // (21,26): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override void M5(out object? o) => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M5").WithArguments("o").WithLocation(21, 26),
                // (23,47): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override object? this[object? o] { get => null!; set { } }
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "get").WithLocation(23, 47),
                // (29,26): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override void M(object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M").WithArguments("o").WithLocation(29, 26),
                // (30,26): warning CS8610: Nullability of reference types in type of parameter 't' doesn't match overridden member.
                //     public override void M2((object, object) t) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "M2").WithArguments("t").WithLocation(30, 26),
                // (31,26): warning CS8610: Nullability of reference types in type of parameter 'i' doesn't match overridden member.
                //     public override void M3(I<object> i) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "M3").WithArguments("i").WithLocation(31, 26),
                // (32,26): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override void M4(ref object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M4").WithArguments("o").WithLocation(32, 26),
                // (34,26): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override void M6(in object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M6").WithArguments("o").WithLocation(34, 26),
                // (35,59): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override object this[object o] { get => null!; set { } }
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("o").WithLocation(35, 59),
                // (35,59): warning CS8765: Type of parameter 'value' doesn't match overridden member because of nullability attributes.
                //     public override object this[object o] { get => null!; set { } }
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("value").WithLocation(35, 59),
                // (36,46): warning CS8765: Type of parameter 's' doesn't match overridden member because of nullability attributes.
                //     public override string this[string s] => null!;
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "null!").WithArguments("s").WithLocation(36, 46),
                // (37,44): warning CS8765: Type of parameter 'a' doesn't match overridden member because of nullability attributes.
                //     public override string this[int[] a] { set { } }
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("a").WithLocation(37, 44),
                // (44,26): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override void M4(ref object? o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M4").WithArguments("o").WithLocation(44, 26),
                // (45,26): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override void M5(out object? o) => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M5").WithArguments("o").WithLocation(45, 26),
                // (47,47): warning CS8764: Return type doesn't match overridden member because of nullability attributes.
                //     public override object? this[object? o] { get => null!; set { } }
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "get").WithLocation(47, 47),
                // (53,26): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override void M(object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M").WithArguments("o").WithLocation(53, 26),
                // (54,26): warning CS8610: Nullability of reference types in type of parameter 't' doesn't match overridden member.
                //     public override void M2((object, object) t) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "M2").WithArguments("t").WithLocation(54, 26),
                // (55,26): warning CS8610: Nullability of reference types in type of parameter 'i' doesn't match overridden member.
                //     public override void M3(I<object> i) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "M3").WithArguments("i").WithLocation(55, 26),
                // (56,26): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override void M4(ref object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M4").WithArguments("o").WithLocation(56, 26),
                // (58,26): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override void M6(in object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M6").WithArguments("o").WithLocation(58, 26),
                // (59,59): warning CS8765: Type of parameter 'o' doesn't match overridden member because of nullability attributes.
                //     public override object this[object o] { get => null!; set { } }
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("o").WithLocation(59, 59),
                // (59,59): warning CS8765: Type of parameter 'value' doesn't match overridden member because of nullability attributes.
                //     public override object this[object o] { get => null!; set { } }
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("value").WithLocation(59, 59),
                // (60,46): warning CS8765: Type of parameter 's' doesn't match overridden member because of nullability attributes.
                //     public override string this[string s] => null!;
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "null!").WithArguments("s").WithLocation(60, 46),
                // (61,44): warning CS8765: Type of parameter 'a' doesn't match overridden member because of nullability attributes.
                //     public override string this[int[] a] { set { } }
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("a").WithLocation(61, 44)
                );
        }

        [Fact]
        public void Override_NullabilityContravariance_02()
        {
            var src = @"
class B
{
    public virtual object? this[object? o] { get => null!; set { } }
}
class C : B
{
#nullable disable
    public override object this[object o] { set { } } 
#nullable enable
}
class D : C
{
    public override object this[object? o] { get => null!; }
}
class E : D
{
    public override object this[object o] { get => null!; set { } } // warn
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (18,45): warning CS8765: Nullability of type of parameter 'o' doesn't match overridden member (possibly because of nullability attributes).
                //     public override object this[object o] { get => null!; set { } } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "get").WithArguments("o").WithLocation(18, 45)
                );
        }

        [Fact]
        public void Override_NullabilityContravariance_03()
        {
            var src = @"
class B
{
    public virtual object? this[object? o] { get => null!; set { } }
}
class C : B
{
    public override object this[object? o] { get => null!; }
}
class D : C
{
#nullable disable
    public override object this[object o] { set { } } 
#nullable enable
}
class E : D
{
    public override object this[object o] { get => null!; set { } } // warn
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (18,45): warning CS8765: Nullability of type of parameter 'o' doesn't match overridden member (possibly because of nullability attributes).
                //     public override object this[object o] { get => null!; set { } } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "get").WithArguments("o").WithLocation(18, 45)
                );
        }

        [Fact]
        public void OverrideConstraintVariance()
        {
            var src = @"
using System.Collections.Generic;
class A
{
    public virtual List<T>? M<T>(List<T>? t) => null!;
}
class B : A
{
    public override List<T> M<T>(List<T> t) => null!;
}
class C : B
{
    public override List<T>? M<T>(List<T>? t) => null!;
}
class D : C
{
    public override List<T> M<T>(List<T> t) => null!;
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (9,29): warning CS8765: Nullability of type of parameter 't' doesn't match overridden member (possibly because of nullability attributes).
                //     public override List<T> M<T>(List<T> t) => null!;
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M").WithArguments("t").WithLocation(9, 29),
                // (13,30): warning CS8764: Nullability of return type doesn't match overridden member (possibly because of nullability attributes).
                //     public override List<T>? M<T>(List<T>? t) => null!;
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "M").WithLocation(13, 30),
                // (17,29): warning CS8765: Nullability of type of parameter 't' doesn't match overridden member (possibly because of nullability attributes).
                //     public override List<T> M<T>(List<T> t) => null!;
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M").WithArguments("t").WithLocation(17, 29));
        }

        [Fact]
        public void OverrideVarianceGenericBase()
        {
            var comp = CreateCompilation(@"
class A<T>
{
    public virtual T M(T t) => default!;
}
#nullable enable
class B : A<object>
{
    public override object? M(object? o) => null!; // warn
}
class C : A<object?>
{
    public override object M(object o) => null!; // warn
}
#nullable disable
class D : A<object>
{
#nullable enable
    public override object? M(object? o) => null!;
}
class E : A<object>
{
#nullable disable
    public override object M(object o) => null!;
}
#nullable enable
class F : A<object?>
{
#nullable disable
    public override object M(object o) => null;
}
");
            comp.VerifyDiagnostics(
                // (9,29): warning CS8764: Nullability of return type doesn't match overridden member (possibly because of nullability attributes).
                //     public override object? M(object? o) => null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "M").WithLocation(9, 29),
                // (13,28): warning CS8765: Nullability of type of parameter 'o' doesn't match overridden member (possibly because of nullability attributes).
                //     public override object M(object o) => null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "M").WithArguments("o").WithLocation(13, 28));
        }

        [Fact]
        public void NullableVarianceConsumer()
        {
            var comp = CreateCompilation(@"
class A
{
    public virtual object? M() => null;
}
class B : A
{
    public override object M() => null; // warn
}
class C
{
    void M()
    {
        var b = new B();
        b.M().ToString();
        A a = b;
        a.M().ToString(); // warn
    }
}", options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (8,35): warning CS8603: Possible null reference return.
                //     public override object M() => null;
                Diagnostic(ErrorCode.WRN_NullReferenceReturn, "null").WithLocation(8, 35),
                // (17,9): warning CS8602: Dereference of a possibly null reference.
                //         a.M().ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "a.M()").WithLocation(17, 9));
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_01()
        {
            var src = @"
using System;

interface I<out T> {}
interface A
{
    object? M();
    (object?, object?) M2();
    I<object?> M3();
    ref object? M4();
    ref readonly object? M5();
    Func<object>? P1 { get; set; }
    Func<object?> P2 { get; set; }
    object? P3 {get;}
    event Func<object>? E1;
    event Func<object?> E2;
}
class B : A
{
    public object M() => null!;
    public (object, object) M2() => throw null!;
    public I<object> M3() => throw null!;
    public ref object M4() => throw null!; // warn
    public ref readonly object M5() => throw null!;
    public Func<object> P1 { get; set; } = null!; // warn
    public Func<object> P2 { get; set; } = null!; // warn
    public object P3 => null!; // ok
    public event Func<object> E1 { add {} remove {} } // warn
    public event Func<object> E2 { add {} remove {} } // warn
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (23,23): warning CS8766: Nullability of reference types in return type of 'ref object B.M4()' doesn't match implicitly implemented member 'ref object? A.M4()' (possibly because of nullability attributes).
                //     public ref object M4() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "M4").WithArguments("ref object B.M4()", "ref object? A.M4()").WithLocation(23, 23),
                // (25,35): warning CS8767: Nullability of reference types in type of parameter 'value' of 'void B.P1.set' doesn't match implicitly implemented member 'void A.P1.set' (possibly because of nullability attributes).
                //     public Func<object> P1 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "set").WithArguments("value", "void B.P1.set", "void A.P1.set").WithLocation(25, 35),
                // (26,35): warning CS8614: Nullability of reference types in type of parameter 'value' of 'void B.P2.set' doesn't match implicitly implemented member 'void A.P2.set'.
                //     public Func<object> P2 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnImplicitImplementation, "set").WithArguments("value", "void B.P2.set", "void A.P2.set").WithLocation(26, 35),
                // (28,31): warning CS8612: Nullability of reference types in type of 'event Func<object> B.E1' doesn't match implicitly implemented member 'event Func<object>? A.E1'.
                //     public event Func<object> E1 { add {} remove {} } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnImplicitImplementation, "E1").WithArguments("event Func<object> B.E1", "event Func<object>? A.E1").WithLocation(28, 31),
                // (29,31): warning CS8612: Nullability of reference types in type of 'event Func<object> B.E2' doesn't match implicitly implemented member 'event Func<object?> A.E2'.
                //     public event Func<object> E2 { add {} remove {} } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnImplicitImplementation, "E2").WithArguments("event Func<object> B.E2", "event Func<object?> A.E2").WithLocation(29, 31)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_02()
        {
            var src = @"
using System;

interface I<out T> {}
interface B
{
    object M();
    (object, object) M2();
    I<object> M3();
    ref object M4();
    ref readonly object M5();
    Func<object> P1 { get; set; }
    Func<object> P2 { get; set; }
    object P3 { get; }
    event Func<object> E1;
    event Func<object> E2;
}
class C : B
{
    public object? M() => null; // warn
    public (object?, object?) M2() => throw null!; // warn
    public I<object?> M3() => throw null!; // warn
    public ref object? M4() => throw null!; // warn
    public ref readonly object? M5() => throw null!; // warn
    public Func<object>? P1 { get; set; } = null!; // warn
    public Func<object?> P2 { get; set; } = null!; // warn
    public object? P3 => null!; // warn
    public event Func<object>? E1 { add {} remove {} } // ok
    public event Func<object?> E2 { add {} remove {} } // ok
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (20,20): warning CS8766: Nullability of reference types in return type of 'object? C.M()' doesn't match implicitly implemented member 'object B.M()' (possibly because of nullability attributes).
                //     public object? M() => null; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "M").WithArguments("object? C.M()", "object B.M()").WithLocation(20, 20),
                // (21,31): warning CS8613: Nullability of reference types in return type of '(object?, object?) C.M2()' doesn't match implicitly implemented member '(object, object) B.M2()'.
                //     public (object?, object?) M2() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnImplicitImplementation, "M2").WithArguments("(object?, object?) C.M2()", "(object, object) B.M2()").WithLocation(21, 31),
                // (22,23): warning CS8613: Nullability of reference types in return type of 'I<object?> C.M3()' doesn't match implicitly implemented member 'I<object> B.M3()'.
                //     public I<object?> M3() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnImplicitImplementation, "M3").WithArguments("I<object?> C.M3()", "I<object> B.M3()").WithLocation(22, 23),
                // (23,24): warning CS8766: Nullability of reference types in return type of 'ref object? C.M4()' doesn't match implicitly implemented member 'ref object B.M4()' (possibly because of nullability attributes).
                //     public ref object? M4() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "M4").WithArguments("ref object? C.M4()", "ref object B.M4()").WithLocation(23, 24),
                // (24,33): warning CS8766: Nullability of reference types in return type of 'ref readonly object? C.M5()' doesn't match implicitly implemented member 'ref readonly object B.M5()' (possibly because of nullability attributes).
                //     public ref readonly object? M5() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "M5").WithArguments("ref readonly object? C.M5()", "ref readonly object B.M5()").WithLocation(24, 33),
                // (25,31): warning CS8766: Nullability of reference types in return type of 'Func<object>? C.P1.get' doesn't match implicitly implemented member 'Func<object> B.P1.get' (possibly because of nullability attributes).
                //     public Func<object>? P1 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "get").WithArguments("Func<object>? C.P1.get", "Func<object> B.P1.get").WithLocation(25, 31),
                // (26,31): warning CS8613: Nullability of reference types in return type of 'Func<object?> C.P2.get' doesn't match implicitly implemented member 'Func<object> B.P2.get'.
                //     public Func<object?> P2 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnImplicitImplementation, "get").WithArguments("Func<object?> C.P2.get", "Func<object> B.P2.get").WithLocation(26, 31),
                // (27,26): warning CS8766: Nullability of reference types in return type of 'object? C.P3.get' doesn't match implicitly implemented member 'object B.P3.get' (possibly because of nullability attributes).
                //     public object? P3 => null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "null!").WithArguments("object? C.P3.get", "object B.P3.get").WithLocation(27, 26)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_05()
        {
            var src = @"
using System;

interface I<out T> {}
interface A
{
    object? M();
    (object?, object?) M2();
    I<object?> M3();
    ref object? M4();
    ref readonly object? M5();
    Func<object>? P1 { get; set; }
    Func<object?> P2 { get; set; }
    object? P3 {get;}
    event Func<object>? E1;
    event Func<object?> E2;
}
class B
{
    public object M() => null!;
    public (object, object) M2() => throw null!;
    public I<object> M3() => throw null!;
    public ref object M4() => throw null!; // warn
    public ref readonly object M5() => throw null!;
    public Func<object> P1 { get; set; } = null!; // warn
    public Func<object> P2 { get; set; } = null!; // warn
    public object P3 => null!; // ok
    public event Func<object> E1 { add {} remove {} } // warn
    public event Func<object> E2 { add {} remove {} } // warn
}
class C : B, A
{}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (31,14): warning CS8766: Nullability of reference types in return type of 'ref object B.M4()' doesn't match implicitly implemented member 'ref object? A.M4()' (possibly because of nullability attributes).
                // class C : B, A
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "A").WithArguments("ref object B.M4()", "ref object? A.M4()").WithLocation(31, 14),
                // (31,14): warning CS8767: Nullability of reference types in type of parameter 'value' of 'void B.P1.set' doesn't match implicitly implemented member 'void A.P1.set' (possibly because of nullability attributes).
                // class C : B, A
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "A").WithArguments("value", "void B.P1.set", "void A.P1.set").WithLocation(31, 14),
                // (31,14): warning CS8614: Nullability of reference types in type of parameter 'value' of 'void B.P2.set' doesn't match implicitly implemented member 'void A.P2.set'.
                // class C : B, A
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnImplicitImplementation, "A").WithArguments("value", "void B.P2.set", "void A.P2.set").WithLocation(31, 14),
                // (31,14): warning CS8612: Nullability of reference types in type of 'event Func<object> B.E1' doesn't match implicitly implemented member 'event Func<object>? A.E1'.
                // class C : B, A
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnImplicitImplementation, "A").WithArguments("event Func<object> B.E1", "event Func<object>? A.E1").WithLocation(31, 14),
                // (31,14): warning CS8612: Nullability of reference types in type of 'event Func<object> B.E2' doesn't match implicitly implemented member 'event Func<object?> A.E2'.
                // class C : B, A
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnImplicitImplementation, "A").WithArguments("event Func<object> B.E2", "event Func<object?> A.E2").WithLocation(31, 14)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_06()
        {
            var src = @"
using System;

interface A
{
    Func<object>? P1 { get; set; }
    Func<object?> P2 { get; set; }
}
class B
{
    public virtual Func<object> P1 { get; set; } = null!; // warn
    public virtual Func<object> P2 { get; set; } = null!; // warn
}
class C : B, A
{
    public override Func<object> P1 { get => null!;}
    public override Func<object> P2 => null!;
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (14,14): warning CS8614: Nullability of reference types in type of parameter 'value' of 'void B.P2.set' doesn't match implicitly implemented member 'void A.P2.set'.
                // class C : B, A
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnImplicitImplementation, "A").WithArguments("value", "void B.P2.set", "void A.P2.set").WithLocation(14, 14),
                // (14,14): warning CS8767: Nullability of reference types in type of parameter 'value' of 'void B.P1.set' doesn't match implicitly implemented member 'void A.P1.set' (possibly because of nullability attributes).
                // class C : B, A
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "A").WithArguments("value", "void B.P1.set", "void A.P1.set").WithLocation(14, 14)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_07()
        {
            var src = @"
using System;

interface A
{
    Func<object>? P1 { get; set; }
    Func<object?> P2 { get; set; }
}
class B
{
    public virtual Func<object>? P1 { get; set; } = null!;
    public virtual Func<object?> P2 { get; set; } = null!;
}
class C : B, A
{
    public override Func<object> P1 { get => null!;}
    public override Func<object> P2 => null!;
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_08()
        {
            var src = @"
using System;
interface A
{
    Func<object>? P1 { get; set; }
    Func<object?> P2 { get; set; }
}
class B
{
    public virtual Func<object>? P1 { get; set; } = null!;
    public virtual Func<object?> P2 { get; set; } = null!;
}
class C : B, A
{
    public override Func<object> P1 { set {} } // warn
    public override Func<object> P2 { set; } // warn
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (15,39): warning CS8765: Nullability of type of parameter 'value' doesn't match overridden member (possibly because of nullability attributes).
                //     public override Func<object> P1 { set {} } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("value").WithLocation(15, 39),
                // (15,39): warning CS8767: Nullability of reference types in type of parameter 'value' of 'void C.P1.set' doesn't match implicitly implemented member 'void A.P1.set' (possibly because of nullability attributes).
                //     public override Func<object> P1 { set {} } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "set").WithArguments("value", "void C.P1.set", "void A.P1.set").WithLocation(15, 39),
                // (16,39): error CS8051: Auto-implemented properties must have get accessors.
                //     public override Func<object> P2 { set; } // warn
                Diagnostic(ErrorCode.ERR_AutoPropertyMustHaveGetAccessor, "set").WithLocation(16, 39),
                // (16,39): warning CS8610: Nullability of reference types in type of parameter 'value' doesn't match overridden member.
                //     public override Func<object> P2 { set; } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("value").WithLocation(16, 39),
                // (16,39): warning CS8614: Nullability of reference types in type of parameter 'value' of 'void C.P2.set' doesn't match implicitly implemented member 'void A.P2.set'.
                //     public override Func<object> P2 { set; } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnImplicitImplementation, "set").WithArguments("value", "void C.P2.set", "void A.P2.set").WithLocation(16, 39)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_09()
        {
            var src = @"
using System;

interface A
{
    Func<object>? P1 { get; set; }
    Func<object?> P2 { get; set; }
}
class B
{
    public virtual Func<object> P1 { get; set; } = null!;
    public virtual Func<object> P2 { get; set; } = null!;
}
class C : B, A
{
    public override Func<object>? P1 { set {} }
    public override Func<object?> P2 { set {} }
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_10()
        {
            var src = @"
using System;

interface A
{
    Func<object>? P1 { get => null!; private set {} }
    Func<object?> P2 { get => null!; private set {} }
}
class B : A
{
    public Func<object> P1 { get; set; } = null!;
    public Func<object> P2 { get; set; } = null!;
}
";
            var comp = CreateCompilation(src, targetFramework: TargetFramework.NetCoreApp, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_11()
        {
            var src = @"
using System;

interface A
{
    Func<object>? P1 { get => null!; private set {} }
    Func<object?> P2 { get => null!; private set {} }
}
class B : A
{
    public Func<object> P1 { get; } = null!;
    public Func<object> P2 { get; } = null!;
}
";
            var comp = CreateCompilation(src, targetFramework: TargetFramework.NetCoreApp, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_12()
        {
            var src = @"
using System;

interface A
{
    Func<object>? P1 { get; }
    Func<object?> P2 { get; }
}
class B : A
{
    public Func<object> P1 { get; set; } = null!;
    public Func<object> P2 { get; set; } = null!;
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_13()
        {
            var src = @"
using System;

interface A
{
    Func<object>? P1 { get; set; }
    Func<object?> P2 { get; set; }
}
class B : A
{
    public Func<object> P1 { get; } = null!;
    public Func<object> P2 { get; } = null!;
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (9,11): error CS0535: 'B' does not implement interface member 'A.P2.set'
                // class B : A
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "A").WithArguments("B", "A.P2.set").WithLocation(9, 11),
                // (9,11): error CS0535: 'B' does not implement interface member 'A.P1.set'
                // class B : A
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "A").WithArguments("B", "A.P1.set").WithLocation(9, 11)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_14()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { get; set; }
    Func<object> P2 { get; set; }
}
class C
{
    public virtual Func<object>? P1 { get; set; } = null!; // warn
    public virtual Func<object?> P2 { get; set; } = null!; // warn
}
class D : C, B
{
    public override Func<object>? P1 { set {} }
    public override Func<object?> P2 { set {} }
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (14,14): warning CS8613: Nullability of reference types in return type of 'Func<object?> C.P2.get' doesn't match implicitly implemented member 'Func<object> B.P2.get'.
                // class D : C, B
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnImplicitImplementation, "B").WithArguments("Func<object?> C.P2.get", "Func<object> B.P2.get").WithLocation(14, 14),
                // (14,14): warning CS8766: Nullability of reference types in return type of 'Func<object>? C.P1.get' doesn't match implicitly implemented member 'Func<object> B.P1.get' (possibly because of nullability attributes).
                // class D : C, B
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "B").WithArguments("Func<object>? C.P1.get", "Func<object> B.P1.get").WithLocation(14, 14)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_15()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { get; set; }
    Func<object> P2 { get; set; }
}
class C
{
    public virtual Func<object> P1 { get; set; } = null!;
    public virtual Func<object> P2 { get; set; } = null!;
}
class D : C, B
{
    public override Func<object>? P1 { set {} }
    public override Func<object?> P2 { set {} }
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_16()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { get; set; }
    Func<object> P2 { get; set; }
}
class C
{
    public virtual Func<object> P1 { get; set; } = null!;
    public virtual Func<object> P2 { get; set; } = null!;
}
class D : C, B
{
    public override Func<object>? P1 { get; } = null!; // warn
    public override Func<object?> P2 { get => null!; } // warn
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (16,35): error CS8080: Auto-implemented properties must override all accessors of the overridden property.
                //     public override Func<object>? P1 { get; } = null!; // warn
                Diagnostic(ErrorCode.ERR_AutoPropertyMustOverrideSet, "P1").WithLocation(16, 35),
                // (16,40): warning CS8764: Nullability of return type doesn't match overridden member (possibly because of nullability attributes).
                //     public override Func<object>? P1 { get; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnOverride, "get").WithLocation(16, 40),
                // (16,40): warning CS8766: Nullability of reference types in return type of 'Func<object>? D.P1.get' doesn't match implicitly implemented member 'Func<object> B.P1.get' (possibly because of nullability attributes).
                //     public override Func<object>? P1 { get; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "get").WithArguments("Func<object>? D.P1.get", "Func<object> B.P1.get").WithLocation(16, 40),
                // (17,40): warning CS8609: Nullability of reference types in return type doesn't match overridden member.
                //     public override Func<object?> P2 { get => null!; } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnOverride, "get").WithLocation(17, 40),
                // (17,40): warning CS8613: Nullability of reference types in return type of 'Func<object?> D.P2.get' doesn't match implicitly implemented member 'Func<object> B.P2.get'.
                //     public override Func<object?> P2 { get => null!; } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnImplicitImplementation, "get").WithArguments("Func<object?> D.P2.get", "Func<object> B.P2.get").WithLocation(17, 40)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_17()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { get; set; }
    Func<object> P2 { get; set; }
}
class C
{
    public virtual Func<object>? P1 { get; set; } = null!;
    public virtual Func<object?> P2 { get; set; } = null!;
}
class D : C, B
{
    public override Func<object> P1 { get => null!; }
    public override Func<object> P2 { get => null!; }
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_18()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { private get => null!; set{} }
    Func<object> P2 { private get => null!; set{} }
}
class C : B
{
    public Func<object>? P1 { get; set; } = null!;
    public Func<object?> P2 { get; set; } = null!;
}
";
            var comp = CreateCompilation(src, targetFramework: TargetFramework.NetCoreApp, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_19()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { private get => null!; set{} }
    Func<object> P2 { private get => null!; set{} }
}
class C : B
{
    public Func<object>? P1 { set{} }
    public Func<object?> P2 { set{} }
}
";
            var comp = CreateCompilation(src, targetFramework: TargetFramework.NetCoreApp, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_20()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { set; }
    Func<object> P2 { set; }
}
class C : B
{
    public Func<object>? P1 { get; set; }
    public Func<object?> P2 { get; set; } = null!;
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Implicit_21()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { get; set; }
    Func<object> P2 { get; set; }
}
class C : B
{
    public Func<object>? P1 { set {} }
    public Func<object?> P2 { set {} }
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (9,11): error CS0535: 'C' does not implement interface member 'B.P1.get'
                // class C : B
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "B").WithArguments("C", "B.P1.get").WithLocation(9, 11),
                // (9,11): error CS0535: 'C' does not implement interface member 'B.P2.get'
                // class C : B
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "B").WithArguments("C", "B.P2.get").WithLocation(9, 11)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_01()
        {
            var src = @"
using System;

interface I<out T> {}
interface A
{
    object? M();
    (object?, object?) M2();
    I<object?> M3();
    ref object? M4();
    ref readonly object? M5();
    Func<object>? P1 { get; set; }
    Func<object?> P2 { get; set; }
    object? P3 {get;}
    event Func<object>? E1;
    event Func<object?> E2;
}
class B : A
{
    object A.M() => null!;
    (object, object) A.M2() => throw null!;
    I<object> A.M3() => throw null!;
    ref object A.M4() => throw null!; // warn
    ref readonly object A.M5() => throw null!;
    Func<object> A.P1 { get; set; } = null!; // warn
    Func<object> A.P2 { get; set; } = null!; // warn
    object A.P3 => null!; // ok
    event Func<object> A.E1 { add {} remove {} } // warn
    event Func<object> A.E2 { add {} remove {} } // warn
}

class C : B, A
{}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (23,18): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'ref object? A.M4()' (possibly because of nullability attributes).
                //     ref object A.M4() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "M4").WithArguments("ref object? A.M4()").WithLocation(23, 18),
                // (25,30): warning CS8769: Nullability of reference types in type of parameter 'value' doesn't match implemented member 'void A.P1.set' (possibly because of nullability attributes).
                //     Func<object> A.P1 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnExplicitImplementation, "set").WithArguments("value", "void A.P1.set").WithLocation(25, 30),
                // (26,30): warning CS8617: Nullability of reference types in type of parameter 'value' doesn't match implemented member 'void A.P2.set'.
                //     Func<object> A.P2 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnExplicitImplementation, "set").WithArguments("value", "void A.P2.set").WithLocation(26, 30),
                // (28,26): warning CS8615: Nullability of reference types in type doesn't match implemented member 'event Func<object>? A.E1'.
                //     event Func<object> A.E1 { add {} remove {} } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnExplicitImplementation, "E1").WithArguments("event Func<object>? A.E1").WithLocation(28, 26),
                // (29,26): warning CS8615: Nullability of reference types in type doesn't match implemented member 'event Func<object?> A.E2'.
                //     event Func<object> A.E2 { add {} remove {} } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnExplicitImplementation, "E2").WithArguments("event Func<object?> A.E2").WithLocation(29, 26)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_02()
        {
            var src = @"
using System;

interface I<out T> {}
interface B
{
    object M();
    (object, object) M2();
    I<object> M3();
    ref object M4();
    ref readonly object M5();
    Func<object> P1 { get; set; }
    Func<object> P2 { get; set; }
    object P3 { get; }
    event Func<object> E1;
    event Func<object> E2;
}
class C : B
{
    object? B.M() => null; // warn
    (object?, object?) B.M2() => throw null!; // warn
    I<object?> B.M3() => throw null!; // warn
    ref object? B.M4() => throw null!; // warn
    ref readonly object? B.M5() => throw null!; // warn
    Func<object>? B.P1 { get; set; } = null!; // warn
    Func<object?> B.P2 { get; set; } = null!; // warn
    object? B.P3 => null!; // warn
    event Func<object>? B.E1 { add {} remove {} } // ok
    event Func<object?> B.E2 { add {} remove {} } // ok
}

class D : C, B
{}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (20,15): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'object B.M()' (possibly because of nullability attributes).
                //     object? B.M() => null; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "M").WithArguments("object B.M()").WithLocation(20, 15),
                // (21,26): warning CS8616: Nullability of reference types in return type doesn't match implemented member '(object, object) B.M2()'.
                //     (object?, object?) B.M2() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "M2").WithArguments("(object, object) B.M2()").WithLocation(21, 26),
                // (22,18): warning CS8616: Nullability of reference types in return type doesn't match implemented member 'I<object> B.M3()'.
                //     I<object?> B.M3() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "M3").WithArguments("I<object> B.M3()").WithLocation(22, 18),
                // (23,19): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'ref object B.M4()' (possibly because of nullability attributes).
                //     ref object? B.M4() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "M4").WithArguments("ref object B.M4()").WithLocation(23, 19),
                // (24,28): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'ref readonly object B.M5()' (possibly because of nullability attributes).
                //     ref readonly object? B.M5() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "M5").WithArguments("ref readonly object B.M5()").WithLocation(24, 28),
                // (25,26): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'Func<object> B.P1.get' (possibly because of nullability attributes).
                //     Func<object>? B.P1 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "get").WithArguments("Func<object> B.P1.get").WithLocation(25, 26),
                // (26,26): warning CS8616: Nullability of reference types in return type doesn't match implemented member 'Func<object> B.P2.get'.
                //     Func<object?> B.P2 { get; set; } = null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "get").WithArguments("Func<object> B.P2.get").WithLocation(26, 26),
                // (27,21): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'object B.P3.get' (possibly because of nullability attributes).
                //     object? B.P3 => null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "null!").WithArguments("object B.P3.get").WithLocation(27, 21)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_03()
        {
            var src = @"
using System;

interface I<out T> {}
interface A
{
    object? M();
    (object?, object?) M2();
    I<object?> M3();
    ref object? M4();
    ref readonly object? M5();
    Func<object>? P1 { get; set; }
    Func<object?> P2 { get; set; }
    object? P3 {get;}
    event Func<object>? E1;
    event Func<object?> E2;
}
interface B : A
{
    object A.M() => null!;
    (object, object) A.M2() => throw null!;
    I<object> A.M3() => throw null!;
    ref object A.M4() => throw null!; // warn
    ref readonly object A.M5() => throw null!;
    Func<object> A.P1 { get => null!; set {} } // warn
    Func<object> A.P2 { get => null!; set {} } // warn
    object A.P3 => null!; // ok
    event Func<object> A.E1 { add {} remove {} } // warn
    event Func<object> A.E2 { add {} remove {} } // warn
}

class C : B, A
{}
interface D : B, A
{}
";
            var comp = CreateCompilation(src, targetFramework: TargetFramework.NetCoreApp, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (23,18): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'ref object? A.M4()' (possibly because of nullability attributes).
                //     ref object A.M4() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "M4").WithArguments("ref object? A.M4()").WithLocation(23, 18),
                // (25,39): warning CS8769: Nullability of reference types in type of parameter 'value' doesn't match implemented member 'void A.P1.set' (possibly because of nullability attributes).
                //     Func<object> A.P1 { get => null!; set {} } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnExplicitImplementation, "set").WithArguments("value", "void A.P1.set").WithLocation(25, 39),
                // (26,39): warning CS8617: Nullability of reference types in type of parameter 'value' doesn't match implemented member 'void A.P2.set'.
                //     Func<object> A.P2 { get => null!; set {} } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnExplicitImplementation, "set").WithArguments("value", "void A.P2.set").WithLocation(26, 39),
                // (28,26): warning CS8615: Nullability of reference types in type doesn't match implemented member 'event Func<object>? A.E1'.
                //     event Func<object> A.E1 { add {} remove {} } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnExplicitImplementation, "E1").WithArguments("event Func<object>? A.E1").WithLocation(28, 26),
                // (29,26): warning CS8615: Nullability of reference types in type doesn't match implemented member 'event Func<object?> A.E2'.
                //     event Func<object> A.E2 { add {} remove {} } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInTypeOnExplicitImplementation, "E2").WithArguments("event Func<object?> A.E2").WithLocation(29, 26)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_04()
        {
            var src = @"
using System;

interface I<out T> {}
interface B
{
    object M();
    (object, object) M2();
    I<object> M3();
    ref object M4();
    ref readonly object M5();
    Func<object> P1 { get; set; }
    Func<object> P2 { get; set; }
    object P3 { get; }
    event Func<object> E1;
    event Func<object> E2;
}
interface C : B
{
    object? B.M() => null; // warn
    (object?, object?) B.M2() => throw null!; // warn
    I<object?> B.M3() => throw null!; // warn
    ref object? B.M4() => throw null!; // warn
    ref readonly object? B.M5() => throw null!; // warn
    Func<object>? B.P1 { get => null!; set {} } // warn
    Func<object?> B.P2 { get => null!; set {} } // warn
    object? B.P3 => null!; // warn
    event Func<object>? B.E1 { add {} remove {} } // ok
    event Func<object?> B.E2 { add {} remove {} } // ok
}

class D : C, B
{}
interface E : C, B
{}
";
            var comp = CreateCompilation(src, targetFramework: TargetFramework.NetCoreApp, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (20,15): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'object B.M()' (possibly because of nullability attributes).
                //     object? B.M() => null; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "M").WithArguments("object B.M()").WithLocation(20, 15),
                // (21,26): warning CS8616: Nullability of reference types in return type doesn't match implemented member '(object, object) B.M2()'.
                //     (object?, object?) B.M2() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "M2").WithArguments("(object, object) B.M2()").WithLocation(21, 26),
                // (22,18): warning CS8616: Nullability of reference types in return type doesn't match implemented member 'I<object> B.M3()'.
                //     I<object?> B.M3() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "M3").WithArguments("I<object> B.M3()").WithLocation(22, 18),
                // (23,19): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'ref object B.M4()' (possibly because of nullability attributes).
                //     ref object? B.M4() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "M4").WithArguments("ref object B.M4()").WithLocation(23, 19),
                // (24,28): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'ref readonly object B.M5()' (possibly because of nullability attributes).
                //     ref readonly object? B.M5() => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "M5").WithArguments("ref readonly object B.M5()").WithLocation(24, 28),
                // (25,26): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'Func<object> B.P1.get' (possibly because of nullability attributes).
                //     Func<object>? B.P1 { get => null!; set {} } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "get").WithArguments("Func<object> B.P1.get").WithLocation(25, 26),
                // (26,26): warning CS8616: Nullability of reference types in return type doesn't match implemented member 'Func<object> B.P2.get'.
                //     Func<object?> B.P2 { get => null!; set {} } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "get").WithArguments("Func<object> B.P2.get").WithLocation(26, 26),
                // (27,21): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'object B.P3.get' (possibly because of nullability attributes).
                //     object? B.P3 => null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "null!").WithArguments("object B.P3.get").WithLocation(27, 21)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_05()
        {
            var src = @"
using System;

interface A
{
    Func<object>? P1 { get => null!; private set {} }
    Func<object?> P2 { get => null!; private set {} }
}
class B : A
{
    Func<object> A.P1 { get; set; } = null!;
    Func<object> A.P2 { get; set; } = null!;
}
";
            var comp = CreateCompilation(src, targetFramework: TargetFramework.NetCoreApp, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (11,30): error CS0550: 'B.A.P1.set' adds an accessor not found in interface member 'A.P1'
                //     Func<object> A.P1 { get; set; } = null!;
                Diagnostic(ErrorCode.ERR_ExplicitPropertyAddingAccessor, "set").WithArguments("B.A.P1.set", "A.P1").WithLocation(11, 30),
                // (12,30): error CS0550: 'B.A.P2.set' adds an accessor not found in interface member 'A.P2'
                //     Func<object> A.P2 { get; set; } = null!;
                Diagnostic(ErrorCode.ERR_ExplicitPropertyAddingAccessor, "set").WithArguments("B.A.P2.set", "A.P2").WithLocation(12, 30)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_06()
        {
            var src = @"
using System;

interface A
{
    Func<object>? P1 { get => null!; private set {} }
    Func<object?> P2 { get => null!; private set {} }
}
class B : A
{
    Func<object> A.P1 { get; } = null!;
    Func<object> A.P2 { get; } = null!;
}
";
            var comp = CreateCompilation(src, targetFramework: TargetFramework.NetCoreApp, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_07()
        {
            var src = @"
using System;

interface A
{
    Func<object>? P1 { get; set; }
    Func<object?> P2 { get; set; }
}
class B : A
{
    Func<object> A.P1 { get; } = null!;
    Func<object> A.P2 { get; } = null!;
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (9,11): error CS0535: 'B' does not implement interface member 'A.P2.set'
                // class B : A
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "A").WithArguments("B", "A.P2.set").WithLocation(9, 11),
                // (9,11): error CS0535: 'B' does not implement interface member 'A.P1.set'
                // class B : A
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "A").WithArguments("B", "A.P1.set").WithLocation(9, 11),
                // (11,20): error CS0551: Explicit interface implementation 'B.A.P1' is missing accessor 'A.P1.set'
                //     Func<object> A.P1 { get; } = null!;
                Diagnostic(ErrorCode.ERR_ExplicitPropertyMissingAccessor, "P1").WithArguments("B.A.P1", "A.P1.set").WithLocation(11, 20),
                // (12,20): error CS0551: Explicit interface implementation 'B.A.P2' is missing accessor 'A.P2.set'
                //     Func<object> A.P2 { get; } = null!;
                Diagnostic(ErrorCode.ERR_ExplicitPropertyMissingAccessor, "P2").WithArguments("B.A.P2", "A.P2.set").WithLocation(12, 20)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_08()
        {
            var src = @"
using System;

interface A
{
    Func<object>? P1 { get; }
    Func<object?> P2 { get; }
}
class B : A
{
    Func<object> A.P1 { get; set; } = null!;
    Func<object> A.P2 { get; set; } = null!;
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (11,30): error CS0550: 'B.A.P1.set' adds an accessor not found in interface member 'A.P1'
                //     Func<object> A.P1 { get; set; } = null!;
                Diagnostic(ErrorCode.ERR_ExplicitPropertyAddingAccessor, "set").WithArguments("B.A.P1.set", "A.P1").WithLocation(11, 30),
                // (12,30): error CS0550: 'B.A.P2.set' adds an accessor not found in interface member 'A.P2'
                //     Func<object> A.P2 { get; set; } = null!;
                Diagnostic(ErrorCode.ERR_ExplicitPropertyAddingAccessor, "set").WithArguments("B.A.P2.set", "A.P2").WithLocation(12, 30)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_09()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { private get => null!; set{} }
    Func<object> P2 { private get => null!; set{} }
}
class C : B
{
    Func<object>? B.P1 { get; set; } = null!;
    Func<object?> B.P2 { get; set; } = null!;
}
";
            var comp = CreateCompilation(src, targetFramework: TargetFramework.NetCoreApp, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (11,26): error CS0550: 'C.B.P1.get' adds an accessor not found in interface member 'B.P1'
                //     Func<object>? B.P1 { get; set; } = null!;
                Diagnostic(ErrorCode.ERR_ExplicitPropertyAddingAccessor, "get").WithArguments("C.B.P1.get", "B.P1").WithLocation(11, 26),
                // (12,26): error CS0550: 'C.B.P2.get' adds an accessor not found in interface member 'B.P2'
                //     Func<object?> B.P2 { get; set; } = null!;
                Diagnostic(ErrorCode.ERR_ExplicitPropertyAddingAccessor, "get").WithArguments("C.B.P2.get", "B.P2").WithLocation(12, 26)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_10()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { private get => null!; set{} }
    Func<object> P2 { private get => null!; set{} }
}
class C : B
{
    Func<object>? B.P1 { set{} }
    Func<object?> B.P2 { set{} }
}
";
            var comp = CreateCompilation(src, targetFramework: TargetFramework.NetCoreApp, options: WithNullableEnable());
            comp.VerifyDiagnostics();
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_11()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { set; }
    Func<object> P2 { set; }
}
class C : B
{
    Func<object>? B.P1 { get; set; }
    Func<object?> B.P2 { get; set; } = null!;
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (11,26): error CS0550: 'C.B.P1.get' adds an accessor not found in interface member 'B.P1'
                //     Func<object>? B.P1 { get; set; }
                Diagnostic(ErrorCode.ERR_ExplicitPropertyAddingAccessor, "get").WithArguments("C.B.P1.get", "B.P1").WithLocation(11, 26),
                // (12,26): error CS0550: 'C.B.P2.get' adds an accessor not found in interface member 'B.P2'
                //     Func<object?> B.P2 { get; set; } = null!;
                Diagnostic(ErrorCode.ERR_ExplicitPropertyAddingAccessor, "get").WithArguments("C.B.P2.get", "B.P2").WithLocation(12, 26)
                );
        }

        [Fact]
        public void Implement_NullabilityCovariance_Explicit_12()
        {
            var src = @"
using System;

interface B
{
    Func<object> P1 { get; set; }
    Func<object> P2 { get; set; }
}
class C : B
{
    Func<object>? B.P1 { set {} }
    Func<object?> B.P2 { set {} }
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (9,11): error CS0535: 'C' does not implement interface member 'B.P1.get'
                // class C : B
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "B").WithArguments("C", "B.P1.get").WithLocation(9, 11),
                // (9,11): error CS0535: 'C' does not implement interface member 'B.P2.get'
                // class C : B
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "B").WithArguments("C", "B.P2.get").WithLocation(9, 11),
                // (11,21): error CS0551: Explicit interface implementation 'C.B.P1' is missing accessor 'B.P1.get'
                //     Func<object>? B.P1 { set {} }
                Diagnostic(ErrorCode.ERR_ExplicitPropertyMissingAccessor, "P1").WithArguments("C.B.P1", "B.P1.get").WithLocation(11, 21),
                // (12,21): error CS0551: Explicit interface implementation 'C.B.P2' is missing accessor 'B.P2.get'
                //     Func<object?> B.P2 { set {} }
                Diagnostic(ErrorCode.ERR_ExplicitPropertyMissingAccessor, "P2").WithArguments("C.B.P2", "B.P2.get").WithLocation(12, 21)
                );
        }

        [Fact]
        public void Implement_NullabilityContravariance_Implicit_01()
        {
            var src = @"
interface I<out T> {}
interface A
{
    void M(object o);
    void M2((object, object) t);
    void M3(I<object> i);
    void M4(ref object o);
    void M5(out object o);
    void M6(in object o);
    object this[object o] { get; set; }
    string this[string s] { get; }
    string this[int[] a] { set; }
}
class B : A
{
    public void M(object? o) { }
    public void M2((object?, object?) t) { }
    public void M3(I<object?> i) { }
    public void M4(ref object? o) { } // warn
    public void M5(out object? o) => throw null!; // warn
    public void M6(in object? o) { }
    public object? this[object? o] { get => null!; set { } } // warn
    public string this[string? s] => null!;
    public string this[int[]? a] { set { } }
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (20,17): warning CS8767: Nullability of reference types in type of parameter 'o' of 'void B.M4(ref object? o)' doesn't match implicitly implemented member 'void A.M4(ref object o)' (possibly because of nullability attributes).
                //     public void M4(ref object? o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "M4").WithArguments("o", "void B.M4(ref object? o)", "void A.M4(ref object o)").WithLocation(20, 17),
                // (21,17): warning CS8767: Nullability of reference types in type of parameter 'o' of 'void B.M5(out object? o)' doesn't match implicitly implemented member 'void A.M5(out object o)' (possibly because of nullability attributes).
                //     public void M5(out object? o) => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "M5").WithArguments("o", "void B.M5(out object? o)", "void A.M5(out object o)").WithLocation(21, 17),
                // (23,38): warning CS8766: Nullability of reference types in return type of 'object? B.this[object? o].get' doesn't match implicitly implemented member 'object A.this[object o].get' (possibly because of nullability attributes).
                //     public object? this[object? o] { get => null!; set { } } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnImplicitImplementation, "get").WithArguments("object? B.this[object? o].get", "object A.this[object o].get").WithLocation(23, 38)
                );
        }

        [Fact]
        public void Implement_NullabilityContravariance_Implicit_02()
        {
            var src = @"
interface I<out T> {}
interface B
{
    void M(object? o);
    void M2((object?, object?) t);
    void M3(I<object?> i);
    void M4(ref object? o);
    void M5(out object? o);
    void M6(in object? o);
    object? this[object? o] { get; set; }
    string this[string? s] { get; }
    string this[int[]? a] { set; }
}
class C : B
{
    public void M(object o) { } // warn
    public void M2((object, object) t) { } // warn
    public void M3(I<object> i) { } // warn
    public void M4(ref object o) { } // warn
    public void M5(out object o) => throw null!;
    public void M6(in object o) { } // warn
    public object this[object o] { get => null!; set { } } // warn
    public string this[string s] => null!; // warn
    public string this[int[] a] { set { } } // warn
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (17,17): warning CS8767: Nullability of reference types in type of parameter 'o' of 'void C.M(object o)' doesn't match implicitly implemented member 'void B.M(object? o)' (possibly because of nullability attributes).
                //     public void M(object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "M").WithArguments("o", "void C.M(object o)", "void B.M(object? o)").WithLocation(17, 17),
                // (18,17): warning CS8614: Nullability of reference types in type of parameter 't' of 'void C.M2((object, object) t)' doesn't match implicitly implemented member 'void B.M2((object?, object?) t)'.
                //     public void M2((object, object) t) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnImplicitImplementation, "M2").WithArguments("t", "void C.M2((object, object) t)", "void B.M2((object?, object?) t)").WithLocation(18, 17),
                // (19,17): warning CS8614: Nullability of reference types in type of parameter 'i' of 'void C.M3(I<object> i)' doesn't match implicitly implemented member 'void B.M3(I<object?> i)'.
                //     public void M3(I<object> i) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnImplicitImplementation, "M3").WithArguments("i", "void C.M3(I<object> i)", "void B.M3(I<object?> i)").WithLocation(19, 17),
                // (20,17): warning CS8767: Nullability of reference types in type of parameter 'o' of 'void C.M4(ref object o)' doesn't match implicitly implemented member 'void B.M4(ref object? o)' (possibly because of nullability attributes).
                //     public void M4(ref object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "M4").WithArguments("o", "void C.M4(ref object o)", "void B.M4(ref object? o)").WithLocation(20, 17),
                // (22,17): warning CS8767: Nullability of reference types in type of parameter 'o' of 'void C.M6(in object o)' doesn't match implicitly implemented member 'void B.M6(in object? o)' (possibly because of nullability attributes).
                //     public void M6(in object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "M6").WithArguments("o", "void C.M6(in object o)", "void B.M6(in object? o)").WithLocation(22, 17),
                // (23,50): warning CS8767: Nullability of reference types in type of parameter 'o' of 'void C.this[object o].set' doesn't match implicitly implemented member 'void B.this[object? o].set' (possibly because of nullability attributes).
                //     public object this[object o] { get => null!; set { } } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "set").WithArguments("o", "void C.this[object o].set", "void B.this[object? o].set").WithLocation(23, 50),
                // (23,50): warning CS8767: Nullability of reference types in type of parameter 'value' of 'void C.this[object o].set' doesn't match implicitly implemented member 'void B.this[object? o].set' (possibly because of nullability attributes).
                //     public object this[object o] { get => null!; set { } } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "set").WithArguments("value", "void C.this[object o].set", "void B.this[object? o].set").WithLocation(23, 50),
                // (24,37): warning CS8767: Nullability of reference types in type of parameter 's' of 'string C.this[string s].get' doesn't match implicitly implemented member 'string B.this[string? s].get' (possibly because of nullability attributes).
                //     public string this[string s] => null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "null!").WithArguments("s", "string C.this[string s].get", "string B.this[string? s].get").WithLocation(24, 37),
                // (25,35): warning CS8767: Nullability of reference types in type of parameter 'a' of 'void C.this[int[] a].set' doesn't match implicitly implemented member 'void B.this[int[]? a].set' (possibly because of nullability attributes).
                //     public string this[int[] a] { set { } } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "set").WithArguments("a", "void C.this[int[] a].set", "void B.this[int[]? a].set").WithLocation(25, 35)
                );
        }

        [Fact]
        public void Implement_NullabilityContravariance_Implicit_03()
        {
            var src = @"
interface B
{
    object? this[object? o] { get; set; }
}
class C
{
#nullable disable
    public virtual object this[object o] { get => null!; set { } }
#nullable enable
}
class D : C, B
{
    public override object this[object o] { get => null!; } // warn
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (14,45): warning CS8767: Nullability of reference types in type of parameter 'o' of 'object D.this[object o].get' doesn't match implicitly implemented member 'object? B.this[object? o].get' (possibly because of nullability attributes).
                //     public override object this[object o] { get => null!; } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "get").WithArguments("o", "object D.this[object o].get", "object? B.this[object? o].get").WithLocation(14, 45)
                );
        }

        [Fact]
        public void Implement_NullabilityContravariance_Implicit_04()
        {
            var src = @"
interface B
{
    object? this[object? o] { get; set; }
}
class C
{
    public virtual object this[object o] { get => null!; set { } }
}
class D : C, B
{
#nullable disable
    public override object this[object o
#nullable enable
                                        ] { set {} }
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (10,14): warning CS8767: Nullability of reference types in type of parameter 'o' of 'object C.this[object o].get' doesn't match implicitly implemented member 'object? B.this[object? o].get' (possibly because of nullability attributes).
                // class D : C, B
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnImplicitImplementation, "B").WithArguments("o", "object C.this[object o].get", "object? B.this[object? o].get").WithLocation(10, 14)
                );
        }

        [Fact]
        public void Implement_NullabilityContravariance_Explicit_01()
        {
            var src = @"
interface I<out T> {}
interface A
{
    void M(object o);
    void M2((object, object) t);
    void M3(I<object> i);
    void M4(ref object o);
    void M5(out object o);
    void M6(in object o);
    object this[object o] { get; set; }
    string this[string s] { get; }
    string this[int[] a] { set; }
}
class B : A
{
    void A.M(object? o) { }
    void A.M2((object?, object?) t) { }
    void A.M3(I<object?> i) { }
    void A.M4(ref object? o) { } // warn
    void A.M5(out object? o) => throw null!; // warn
    void A.M6(in object? o) { }
    object? A.this[object? o] { get => null!; set { } } // warn
    string A.this[string? s] => null!;
    string A.this[int[]? a] { set { } }
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (20,12): warning CS8769: Nullability of reference types in type of parameter 'o' doesn't match implemented member 'void A.M4(ref object o)' (possibly because of nullability attributes).
                //     void A.M4(ref object? o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnExplicitImplementation, "M4").WithArguments("o", "void A.M4(ref object o)").WithLocation(20, 12),
                // (21,12): warning CS8769: Nullability of reference types in type of parameter 'o' doesn't match implemented member 'void A.M5(out object o)' (possibly because of nullability attributes).
                //     void A.M5(out object? o) => throw null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnExplicitImplementation, "M5").WithArguments("o", "void A.M5(out object o)").WithLocation(21, 12),
                // (23,33): warning CS8768: Nullability of reference types in return type doesn't match implemented member 'object A.this[object o].get' (possibly because of nullability attributes).
                //     object? A.this[object? o] { get => null!; set { } } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInReturnTypeOnExplicitImplementation, "get").WithArguments("object A.this[object o].get").WithLocation(23, 33)
                );
        }

        [Fact]
        public void Implement_NullabilityContravariance_Explicit_02()
        {
            var src = @"
interface I<out T> {}
interface B
{
    void M(object? o);
    void M2((object?, object?) t);
    void M3(I<object?> i);
    void M4(ref object? o);
    void M5(out object? o);
    void M6(in object? o);
    object? this[object? o] { get; set; }
    string this[string? s] { get; }
    string this[int[]? a] { set; }
}
class C : B
{
    void B.M(object o) { } // warn
    void B.M2((object, object) t) { } // warn
    void B.M3(I<object> i) { } // warn
    void B.M4(ref object o) { } // warn
    void B.M5(out object o) => throw null!;
    void B.M6(in object o) { } // warn
    object B.this[object o] { get => null!; set { } } // warn
    string B.this[string s] => null!; // warn
    string B.this[int[] a] { set { } } // warn
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (17,12): warning CS8769: Nullability of reference types in type of parameter 'o' doesn't match implemented member 'void B.M(object? o)' (possibly because of nullability attributes).
                //     void B.M(object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnExplicitImplementation, "M").WithArguments("o", "void B.M(object? o)").WithLocation(17, 12),
                // (18,12): warning CS8617: Nullability of reference types in type of parameter 't' doesn't match implemented member 'void B.M2((object?, object?) t)'.
                //     void B.M2((object, object) t) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnExplicitImplementation, "M2").WithArguments("t", "void B.M2((object?, object?) t)").WithLocation(18, 12),
                // (19,12): warning CS8617: Nullability of reference types in type of parameter 'i' doesn't match implemented member 'void B.M3(I<object?> i)'.
                //     void B.M3(I<object> i) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnExplicitImplementation, "M3").WithArguments("i", "void B.M3(I<object?> i)").WithLocation(19, 12),
                // (20,12): warning CS8769: Nullability of reference types in type of parameter 'o' doesn't match implemented member 'void B.M4(ref object? o)' (possibly because of nullability attributes).
                //     void B.M4(ref object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnExplicitImplementation, "M4").WithArguments("o", "void B.M4(ref object? o)").WithLocation(20, 12),
                // (22,12): warning CS8769: Nullability of reference types in type of parameter 'o' doesn't match implemented member 'void B.M6(in object? o)' (possibly because of nullability attributes).
                //     void B.M6(in object o) { } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnExplicitImplementation, "M6").WithArguments("o", "void B.M6(in object? o)").WithLocation(22, 12),
                // (23,45): warning CS8769: Nullability of reference types in type of parameter 'o' doesn't match implemented member 'void B.this[object? o].set' (possibly because of nullability attributes).
                //     object B.this[object o] { get => null!; set { } } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnExplicitImplementation, "set").WithArguments("o", "void B.this[object? o].set").WithLocation(23, 45),
                // (23,45): warning CS8769: Nullability of reference types in type of parameter 'value' doesn't match implemented member 'void B.this[object? o].set' (possibly because of nullability attributes).
                //     object B.this[object o] { get => null!; set { } } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnExplicitImplementation, "set").WithArguments("value", "void B.this[object? o].set").WithLocation(23, 45),
                // (24,32): warning CS8769: Nullability of reference types in type of parameter 's' doesn't match implemented member 'string B.this[string? s].get' (possibly because of nullability attributes).
                //     string B.this[string s] => null!; // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnExplicitImplementation, "null!").WithArguments("s", "string B.this[string? s].get").WithLocation(24, 32),
                // (25,30): warning CS8769: Nullability of reference types in type of parameter 'a' doesn't match implemented member 'void B.this[int[]? a].set' (possibly because of nullability attributes).
                //     string B.this[int[] a] { set { } } // warn
                Diagnostic(ErrorCode.WRN_TopLevelNullabilityMismatchInParameterTypeOnExplicitImplementation, "set").WithArguments("a", "void B.this[int[]? a].set").WithLocation(25, 30)
                );
        }

        [Fact]
        public void Partial_NullabilityContravariance_01()
        {
            var src = @"
interface I<out T> {}
partial class A
{
    partial void M(object o);
    partial void M2((object, object) t);
    partial void M3(I<object> i);
    partial void M4(ref object o);
    partial void M6(in object o);
}
partial class A
{
    partial void M(object? o) { }
    partial void M2((object?, object?) t) { }
    partial void M3(I<object?> i) { }
    partial void M4(ref object? o) { } // warn
    partial void M6(in object? o) { }
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (13,18): warning CS8826: Partial method declarations 'void A.M(object o)' and 'void A.M(object? o)' have signature differences.
                //     partial void M(object? o) { }
                Diagnostic(ErrorCode.WRN_PartialMethodTypeDifference, "M").WithArguments("void A.M(object o)", "void A.M(object? o)").WithLocation(13, 18),
                // (14,18): warning CS8826: Partial method declarations 'void A.M2((object, object) t)' and 'void A.M2((object?, object?) t)' have signature differences.
                //     partial void M2((object?, object?) t) { }
                Diagnostic(ErrorCode.WRN_PartialMethodTypeDifference, "M2").WithArguments("void A.M2((object, object) t)", "void A.M2((object?, object?) t)").WithLocation(14, 18),
                // (15,18): warning CS8826: Partial method declarations 'void A.M3(I<object> i)' and 'void A.M3(I<object?> i)' have signature differences.
                //     partial void M3(I<object?> i) { }
                Diagnostic(ErrorCode.WRN_PartialMethodTypeDifference, "M3").WithArguments("void A.M3(I<object> i)", "void A.M3(I<object?> i)").WithLocation(15, 18),
                // (16,18): warning CS8611: Nullability of reference types in type of parameter 'o' doesn't match partial method declaration.
                //     partial void M4(ref object? o) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnPartial, "M4").WithArguments("o").WithLocation(16, 18),
                // (17,18): warning CS8826: Partial method declarations 'void A.M6(in object o)' and 'void A.M6(in object? o)' have signature differences.
                //     partial void M6(in object? o) { }
                Diagnostic(ErrorCode.WRN_PartialMethodTypeDifference, "M6").WithArguments("void A.M6(in object o)", "void A.M6(in object? o)").WithLocation(17, 18)
                );
        }

        [Fact]
        public void Partial_NullabilityContravariance_02()
        {
            var src = @"
interface I<out T> {}
partial class B
{
    partial void M(object? o);
    partial void M2((object?, object?) t);
    partial void M3(I<object?> i);
    partial void M4(ref object? o);
    partial void M6(in object? o);
}
partial class B
{
    partial void M(object o) { } // warn
    partial void M2((object, object) t) { } // warn
    partial void M3(I<object> i) { } // warn
    partial void M4(ref object o) { } // warn
    partial void M6(in object o) { } // warn
}
";
            var comp = CreateCompilation(src, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (13,18): warning CS8611: Nullability of reference types in type of parameter 'o' doesn't match partial method declaration.
                //     partial void M(object o) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnPartial, "M").WithArguments("o").WithLocation(13, 18),
                // (14,18): warning CS8611: Nullability of reference types in type of parameter 't' doesn't match partial method declaration.
                //     partial void M2((object, object) t) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnPartial, "M2").WithArguments("t").WithLocation(14, 18),
                // (15,18): warning CS8611: Nullability of reference types in type of parameter 'i' doesn't match partial method declaration.
                //     partial void M3(I<object> i) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnPartial, "M3").WithArguments("i").WithLocation(15, 18),
                // (16,18): warning CS8611: Nullability of reference types in type of parameter 'o' doesn't match partial method declaration.
                //     partial void M4(ref object o) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnPartial, "M4").WithArguments("o").WithLocation(16, 18),
                // (17,18): warning CS8611: Nullability of reference types in type of parameter 'o' doesn't match partial method declaration.
                //     partial void M6(in object o) { } // warn
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnPartial, "M6").WithArguments("o").WithLocation(17, 18)
                );
        }

        [Fact]
        public void Implementing_07()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}
interface IA
{
    void M1(string[] x); 
    void M2<T>(T[] x) where T : class; 
    void M3<T>(T?[]? x) where T : class; 
}
class B : IA
{
    public void M1(string?[] x)
    {
    } 

    public void M2<T>(T?[] x)  where T : class
    {
    } 

    public void M3<T>(T?[]? x)  where T : class
    {
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics();

            var ia = compilation.GetTypeByMetadataName("IA");
            var b = compilation.GetTypeByMetadataName("B");

            foreach (var memberName in new[] { "M1", "M2" })
            {
                var member = ia.GetMember<MethodSymbol>(memberName);
                var implementing = (MethodSymbol)b.FindImplementationForInterfaceMember(member);
                var implemented = member.ConstructIfGeneric(implementing.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t)));
                Assert.False(implementing.Parameters[0].TypeWithAnnotations.Equals(implemented.Parameters[0].TypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            {
                var member = ia.GetMember<MethodSymbol>("M3");
                var implementing = (MethodSymbol)b.FindImplementationForInterfaceMember(member);
                var implemented = member.ConstructIfGeneric(implementing.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t)));
                Assert.True(implementing.Parameters[0].TypeWithAnnotations.Equals(implemented.Parameters[0].TypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }
        }

        [Fact]
        public void Implementing_08()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}
interface IA
{
    void M1(string[] x); 
    void M2<T>(T[] x) where T : class; 
    void M3<T>(T?[]? x) where T : class; 
}
class B : IA
{
    void IA.M1(string?[] x)
    {
    } 

    void IA.M2<T>(T?[] x)  
    {
    } 

    void IA.M3<T>(T?[]? x)  
    {
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (20,13): error CS0539: 'B.M2<T>(T?[])' in explicit interface declaration is not found among members of the interface that can be implemented
                //     void IA.M2<T>(T?[] x)
                Diagnostic(ErrorCode.ERR_InterfaceMemberNotFound, "M2").WithArguments("B.M2<T>(T?[])").WithLocation(20, 13),
                // (24,13): error CS0539: 'B.M3<T>(T?[]?)' in explicit interface declaration is not found among members of the interface that can be implemented
                //     void IA.M3<T>(T?[]? x)
                Diagnostic(ErrorCode.ERR_InterfaceMemberNotFound, "M3").WithArguments("B.M3<T>(T?[]?)").WithLocation(24, 13),
                // (14,11): error CS0535: 'B' does not implement interface member 'IA.M3<T>(T?[]?)'
                // class B : IA
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "IA").WithArguments("B", "IA.M3<T>(T?[]?)").WithLocation(14, 11),
                // (14,11): error CS0535: 'B' does not implement interface member 'IA.M2<T>(T[])'
                // class B : IA
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "IA").WithArguments("B", "IA.M2<T>(T[])").WithLocation(14, 11),
                // (20,24): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     void IA.M2<T>(T?[] x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(20, 24),
                // (24,25): error CS0453: The type 'T' must be a non-nullable value type in order to use it as parameter 'T' in the generic type or method 'Nullable<T>'
                //     void IA.M3<T>(T?[]? x)
                Diagnostic(ErrorCode.ERR_ValConstraintNotSatisfied, "x").WithArguments("System.Nullable<T>", "T", "T").WithLocation(24, 25));
        }

        [Fact]
        public void Overriding_20()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

abstract class A1
{
    public abstract int this[string?[] x] {get; set;} 
}
abstract class A2
{
    public abstract int this[string[] x] {get; set;} 
}
abstract class A3
{
    public abstract int this[string?[]? x] {get; set;} 
}

class B1 : A1
{
    public override int this[string[] x]
    {
        get {throw new System.NotImplementedException();}
        set {} // 1
    } 
}
class B2 : A2
{
    public override int this[string[]? x]
    {
        get {throw new System.NotImplementedException();}
        set {}
    } 
}
class B3 : A3
{
    public override int this[string?[]? x]
    {
        get {throw new System.NotImplementedException();}
        set {}
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (27,9): warning CS8610: Nullability of reference types in type of parameter 'x' doesn't match overridden member.
                //         set {} // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnOverride, "set").WithArguments("x").WithLocation(27, 9)
                );

            foreach (string typeName in new[] { "B1", "B2" })
            {
                foreach (var member in compilation.GetTypeByMetadataName(typeName).GetMembers().OfType<PropertySymbol>())
                {
                    Assert.False(member.Parameters[0].TypeWithAnnotations.Equals(member.OverriddenProperty.Parameters[0].TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
                }
            }

            foreach (var member in compilation.GetTypeByMetadataName("B3").GetMembers().OfType<PropertySymbol>())
            {
                Assert.True(member.Parameters[0].TypeWithAnnotations.Equals(member.OverriddenProperty.Parameters[0].TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (string typeName in new[] { "A1", "A2", "A3", "B1", "B2", "B3" })
            {
                var type = compilation.GetTypeByMetadataName(typeName);

                foreach (var property in type.GetMembers().OfType<PropertySymbol>())
                {
                    Assert.True(property.TypeWithAnnotations.Equals(property.GetMethod.ReturnTypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                    Assert.True(property.TypeWithAnnotations.Equals(property.SetMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                }
            }
        }

        [Fact]
        public void Implementing_09()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

interface IA1
{
    int this[string?[] x] {get; set;} 
}
interface IA2
{
    int this[string[] x] {get; set;} 
}
interface IA3
{
    int this[string?[]? x] {get; set;} 
}

class B1 : IA1
{
    public int this[string[] x]
    {
        get {throw new System.NotImplementedException();}
        set {} // 1
    } 
}
class B2 : IA2
{
    public int this[string[]? x] // 2
    {
        get {throw new System.NotImplementedException();}
        set {}
    } 
}
class B3 : IA3
{
    public int this[string?[]? x] // 3
    {
        get {throw new System.NotImplementedException();}
        set {}
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (27,9): warning CS8614: Nullability of reference types in type of parameter 'x' of 'void B1.this[string[] x].set' doesn't match implicitly implemented member 'void IA1.this[string?[] x].set'.
                //         set {} // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnImplicitImplementation, "set").WithArguments("x", "void B1.this[string[] x].set", "void IA1.this[string?[] x].set").WithLocation(27, 9)
                );

            foreach (string[] typeName in new[] { new[] { "IA1", "B1" }, new[] { "IA2", "B2" } })
            {
                var implemented = compilation.GetTypeByMetadataName(typeName[0]).GetMembers().OfType<PropertySymbol>().Single();
                var implementing = (PropertySymbol)compilation.GetTypeByMetadataName(typeName[1]).FindImplementationForInterfaceMember(implemented);
                Assert.False(implementing.Parameters[0].TypeWithAnnotations.Equals(implemented.Parameters[0].TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            {
                var implemented = compilation.GetTypeByMetadataName("IA3").GetMembers().OfType<PropertySymbol>().Single();
                var implementing = (PropertySymbol)compilation.GetTypeByMetadataName("B3").FindImplementationForInterfaceMember(implemented);
                Assert.True(implementing.Parameters[0].TypeWithAnnotations.Equals(implemented.Parameters[0].TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (string typeName in new[] { "IA1", "IA2", "IA3", "B1", "B2", "B3" })
            {
                var type = compilation.GetTypeByMetadataName(typeName);

                foreach (var property in type.GetMembers().OfType<PropertySymbol>())
                {
                    Assert.True(property.TypeWithAnnotations.Equals(property.GetMethod.ReturnTypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                    Assert.True(property.TypeWithAnnotations.Equals(property.SetMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                }
            }
        }

        [Fact]
        public void Implementing_10()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

interface IA1
{
    int this[string?[] x] {get; set;} 
}
interface IA2
{
    int this[string[] x] {get; set;} 
}
interface IA3
{
    int this[string?[]? x] {get; set;} 
}

class B1 : IA1
{
    int IA1.this[string[] x]
    {
        get {throw new System.NotImplementedException();}
        set {} // 1
    } 
}
class B2 : IA2
{
    int IA2.this[string[]? x] // 2
    {
        get {throw new System.NotImplementedException();}
        set {}
    } 
}
class B3 : IA3
{
    int IA3.this[string?[]? x] // 3
    {
        get {throw new System.NotImplementedException();}
        set {}
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (27,9): warning CS8617: Nullability of reference types in type of parameter 'x' doesn't match implemented member 'void IA1.this[string?[] x].set'.
                //         set {} // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnExplicitImplementation, "set").WithArguments("x", "void IA1.this[string?[] x].set").WithLocation(27, 9)
                );

            foreach (string[] typeName in new[] { new[] { "IA1", "B1" }, new[] { "IA2", "B2" } })
            {
                var implemented = compilation.GetTypeByMetadataName(typeName[0]).GetMembers().OfType<PropertySymbol>().Single();
                var implementing = (PropertySymbol)compilation.GetTypeByMetadataName(typeName[1]).FindImplementationForInterfaceMember(implemented);
                Assert.False(implementing.Parameters[0].TypeWithAnnotations.Equals(implemented.Parameters[0].TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            {
                var implemented = compilation.GetTypeByMetadataName("IA3").GetMembers().OfType<PropertySymbol>().Single();
                var implementing = (PropertySymbol)compilation.GetTypeByMetadataName("B3").FindImplementationForInterfaceMember(implemented);
                Assert.True(implementing.Parameters[0].TypeWithAnnotations.Equals(implemented.Parameters[0].TypeWithAnnotations, TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            foreach (string typeName in new[] { "IA1", "IA2", "IA3", "B1", "B2", "B3" })
            {
                var type = compilation.GetTypeByMetadataName(typeName);

                foreach (var property in type.GetMembers().OfType<PropertySymbol>())
                {
                    Assert.True(property.TypeWithAnnotations.Equals(property.GetMethod.ReturnTypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                    Assert.True(property.TypeWithAnnotations.Equals(property.SetMethod.Parameters.Last().TypeWithAnnotations, TypeCompareKind.ConsiderEverything));
                }
            }
        }

        [Fact]
        public void Implementing_11()
        {
            var source = @"
public interface I1<T>
{
    void M();
}

public class A {}
public interface I2 : I1<A?> { }
public interface I3 : I1<A> { }

public class C1 : I2, I1<A>
{
    void I1<A?>.M(){}
    void I1<A>.M(){}
}

public class C2 : I1<A>, I2
{
    void I1<A?>.M(){}
    void I1<A>.M(){}
}

public class C3 : I1<A>, I1<A?>
{
    void I1<A?>.M(){}
    void I1<A>.M(){}
}

public class C4 : I2, I3
{
    void I1<A?>.M(){}
    void I1<A>.M(){}
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (11,14): warning CS8645: 'I1<A>' is already listed in the interface list on type 'C1' with different nullability of reference types.
                // public class C1 : I2, I1<A>
                Diagnostic(ErrorCode.WRN_DuplicateInterfaceWithNullabilityMismatchInBaseList, "C1").WithArguments("I1<A>", "C1").WithLocation(11, 14),
                // (11,14): error CS8646: 'I1<A?>.M()' is explicitly implemented more than once.
                // public class C1 : I2, I1<A>
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C1").WithArguments("I1<A?>.M()").WithLocation(11, 14),
                // (11,14): error CS8646: 'I1<A>.M()' is explicitly implemented more than once.
                // public class C1 : I2, I1<A>
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C1").WithArguments("I1<A>.M()").WithLocation(11, 14),
                // (17,14): warning CS8645: 'I1<A?>' is already listed in the interface list on type 'C2' with different nullability of reference types.
                // public class C2 : I1<A>, I2
                Diagnostic(ErrorCode.WRN_DuplicateInterfaceWithNullabilityMismatchInBaseList, "C2").WithArguments("I1<A?>", "C2").WithLocation(17, 14),
                // (17,14): error CS8646: 'I1<A>.M()' is explicitly implemented more than once.
                // public class C2 : I1<A>, I2
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C2").WithArguments("I1<A>.M()").WithLocation(17, 14),
                // (17,14): error CS8646: 'I1<A?>.M()' is explicitly implemented more than once.
                // public class C2 : I1<A>, I2
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C2").WithArguments("I1<A?>.M()").WithLocation(17, 14),
                // (23,14): warning CS8645: 'I1<A?>' is already listed in the interface list on type 'C3' with different nullability of reference types.
                // public class C3 : I1<A>, I1<A?>
                Diagnostic(ErrorCode.WRN_DuplicateInterfaceWithNullabilityMismatchInBaseList, "C3").WithArguments("I1<A?>", "C3").WithLocation(23, 14),
                // (23,14): error CS8646: 'I1<A>.M()' is explicitly implemented more than once.
                // public class C3 : I1<A>, I1<A?>
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C3").WithArguments("I1<A>.M()").WithLocation(23, 14),
                // (23,14): error CS8646: 'I1<A?>.M()' is explicitly implemented more than once.
                // public class C3 : I1<A>, I1<A?>
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C3").WithArguments("I1<A?>.M()").WithLocation(23, 14),
                // (29,14): warning CS8645: 'I1<A>' is already listed in the interface list on type 'C4' with different nullability of reference types.
                // public class C4 : I2, I3
                Diagnostic(ErrorCode.WRN_DuplicateInterfaceWithNullabilityMismatchInBaseList, "C4").WithArguments("I1<A>", "C4").WithLocation(29, 14),
                // (29,14): error CS8646: 'I1<A?>.M()' is explicitly implemented more than once.
                // public class C4 : I2, I3
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C4").WithArguments("I1<A?>.M()").WithLocation(29, 14),
                // (29,14): error CS8646: 'I1<A>.M()' is explicitly implemented more than once.
                // public class C4 : I2, I3
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C4").WithArguments("I1<A>.M()").WithLocation(29, 14)
                );

            var c1 = (NamedTypeSymbol)comp.GetTypeByMetadataName("C1");
            var c1Interfaces = c1.Interfaces();
            var c1AllInterfaces = c1.AllInterfaces();
            Assert.Equal(2, c1Interfaces.Length);
            Assert.Equal(3, c1AllInterfaces.Length);
            Assert.Equal("I2", c1Interfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<A!>", c1Interfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I2", c1AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<A?>", c1AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<A!>", c1AllInterfaces[2].ToTestDisplayString(includeNonNullable: true));
            assertExplicitInterfaceImplementations(c1);

            var c2 = (NamedTypeSymbol)comp.GetTypeByMetadataName("C2");
            var c2Interfaces = c2.Interfaces();
            var c2AllInterfaces = c2.AllInterfaces();
            Assert.Equal(2, c2Interfaces.Length);
            Assert.Equal(3, c2AllInterfaces.Length);
            Assert.Equal("I1<A!>", c2Interfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I2", c2Interfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<A!>", c2AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I2", c2AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<A?>", c2AllInterfaces[2].ToTestDisplayString(includeNonNullable: true));
            assertExplicitInterfaceImplementations(c2);

            var c3 = (NamedTypeSymbol)comp.GetTypeByMetadataName("C3");
            var c3Interfaces = c3.Interfaces();
            var c3AllInterfaces = c3.AllInterfaces();
            Assert.Equal(2, c3Interfaces.Length);
            Assert.Equal(2, c3AllInterfaces.Length);
            Assert.Equal("I1<A!>", c3Interfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<A?>", c3Interfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<A!>", c3AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<A?>", c3AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
            assertExplicitInterfaceImplementations(c3);

            var c4 = (NamedTypeSymbol)comp.GetTypeByMetadataName("C4");
            var c4Interfaces = c4.Interfaces();
            var c4AllInterfaces = c4.AllInterfaces();
            Assert.Equal(2, c4Interfaces.Length);
            Assert.Equal(4, c4AllInterfaces.Length);
            Assert.Equal("I2", c4Interfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I3", c4Interfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I2", c4AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<A?>", c4AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I3", c4AllInterfaces[2].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<A!>", c4AllInterfaces[3].ToTestDisplayString(includeNonNullable: true));
            assertExplicitInterfaceImplementations(c4);

            void assertExplicitInterfaceImplementations(NamedTypeSymbol c)
            {
                var members = c.GetMembers("I1<A>.M");
                Assert.Equal(2, members.Length);
                var cMabImplementations = ((MethodSymbol)members[0]).ExplicitInterfaceImplementations;
                Assert.Equal(1, cMabImplementations.Length);
                Assert.Equal("void I1<A?>.M()", cMabImplementations[0].ToTestDisplayString(includeNonNullable: true));
                var cMcdImplementations = ((MethodSymbol)members[1]).ExplicitInterfaceImplementations;
                Assert.Equal(1, cMcdImplementations.Length);
                Assert.Equal("void I1<A!>.M()", cMcdImplementations[0].ToTestDisplayString(includeNonNullable: true));
            }
        }

        [Fact]
        public void Implementing_12()
        {
            var source = @"
public interface I1<T> 
{
    void M1();
    void M2();
}

public class A {}

public class C1 : I1<A?> 
{ 
    public void M1() => System.Console.Write(""C1.M1 "");
    void I1<A?>.M2() => System.Console.Write(""C1.M2 "");
}

public class C2 : C1, I1<A>
{
    new public void M1() => System.Console.Write(""C2.M1 "");
    void I1<A>.M2() => System.Console.Write(""C2.M2 "");

    static void Main()
    {
        var x = (C1)new C2();
        var y = (I1<A?>)x;
        y.M1();
        y.M2();
    }
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics();

            Action<ModuleSymbol> validate = (m) =>
            {
                bool isMetadata = m is PEModuleSymbol;

                var c1 = (NamedTypeSymbol)m.GlobalNamespace.GetTypeMember("C1");
                var c1Interfaces = c1.Interfaces();
                var c1AllInterfaces = c1.AllInterfaces();
                Assert.Equal(1, c1Interfaces.Length);
                Assert.Equal(1, c1AllInterfaces.Length);
                Assert.Equal("I1<A?>", c1Interfaces[0].ToTestDisplayString(includeNonNullable: true));
                Assert.Equal("I1<A?>", c1AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));

                var c2 = (NamedTypeSymbol)m.GlobalNamespace.GetTypeMember("C2");
                var c2Interfaces = c2.Interfaces();
                var c2AllInterfaces = c2.AllInterfaces();
                Assert.Equal(1, c2Interfaces.Length);
                Assert.Equal(2, c2AllInterfaces.Length);
                Assert.Equal("I1<A!>", c2Interfaces[0].ToTestDisplayString(includeNonNullable: true));
                Assert.Equal("I1<A?>", c2AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
                Assert.Equal("I1<A!>", c2AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));

                Assert.Equal("void C2.M1()", c2.FindImplementationForInterfaceMember(((TypeSymbol)c2Interfaces[0]).GetMember("M1")).ToTestDisplayString(includeNonNullable: true));
                Assert.Equal("void C2.M1()",
                             c2.FindImplementationForInterfaceMember(((TypeSymbol)c1Interfaces[0]).GetMember("M1")).ToTestDisplayString(includeNonNullable: true));

                var m2 = (MethodSymbol)((TypeSymbol)c2).GetMember("I1<A>.M2");
                var m2Implementations = m2.ExplicitInterfaceImplementations;
                Assert.Equal(1, m2Implementations.Length);
                Assert.Equal(isMetadata ?
                                 "void I1<A>.M2()" :
                                 "void I1<A!>.M2()",
                             m2Implementations[0].ToTestDisplayString(includeNonNullable: true));

                Assert.Same(m2,
                            c2.FindImplementationForInterfaceMember(((TypeSymbol)c2Interfaces[0]).GetMember("M2")));
                Assert.Same(m2,
                            c2.FindImplementationForInterfaceMember(((TypeSymbol)c1Interfaces[0]).GetMember("M2")));
            };

            CompileAndVerify(comp, sourceSymbolValidator: validate, symbolValidator: validate, expectedOutput: @"C2.M1 C2.M2");
        }

        [Fact]
        public void Implementing_13()
        {
            var source = @"
public interface I1<T> 
{
    void M1();
    void M2();
}

public class A {}

public class C1 : I1<A?> 
{ 
    public void M1() => System.Console.Write(""C1.M1 "");
    void I1<A?>.M2() => System.Console.Write(""C1.M2 "");
    public virtual void M2() {}
}

public class C2 : C1, I1<A>
{
    static void Main()
    {
        var x = (C1)new C2();
        var y = (I1<A>)x;
        y.M1();
        y.M2();
    }
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics(
                // (17,23): warning CS8644: 'C2' does not implement interface member 'I1<A>.M2()'. Nullability of reference types in interface implemented by the base type doesn't match.
                // public class C2 : C1, I1<A>
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInInterfaceImplementedByBase, "I1<A>").WithArguments("C2", "I1<A>.M2()").WithLocation(17, 23)
                );

            Action<ModuleSymbol> validate = (m) =>
            {
                bool isMetadata = m is PEModuleSymbol;

                var c1 = (NamedTypeSymbol)m.GlobalNamespace.GetTypeMember("C1");
                var c1Interfaces = c1.Interfaces();
                var c1AllInterfaces = c1.AllInterfaces();
                Assert.Equal(1, c1Interfaces.Length);
                Assert.Equal(1, c1AllInterfaces.Length);
                Assert.Equal("I1<A?>", c1Interfaces[0].ToTestDisplayString(includeNonNullable: true));
                Assert.Equal("I1<A?>", c1AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));

                var c2 = (NamedTypeSymbol)m.GlobalNamespace.GetTypeMember("C2");
                var c2Interfaces = c2.Interfaces();
                var c2AllInterfaces = c2.AllInterfaces();
                Assert.Equal(1, c2Interfaces.Length);
                Assert.Equal(2, c2AllInterfaces.Length);
                Assert.Equal("I1<A!>", c2Interfaces[0].ToTestDisplayString(includeNonNullable: true));
                Assert.Equal("I1<A?>", c2AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
                Assert.Equal("I1<A!>", c2AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));

                Assert.Equal("void C1.M1()", c2.FindImplementationForInterfaceMember(((TypeSymbol)c2Interfaces[0]).GetMember("M1")).ToTestDisplayString(includeNonNullable: true));
                Assert.Equal("void C1.M1()",
                             c2.FindImplementationForInterfaceMember(((TypeSymbol)c1Interfaces[0]).GetMember("M1")).ToTestDisplayString(includeNonNullable: true));

                var m2 = (MethodSymbol)((TypeSymbol)c1).GetMember("I1<A>.M2");
                var m2Implementations = m2.ExplicitInterfaceImplementations;
                Assert.Equal(1, m2Implementations.Length);
                Assert.Equal(isMetadata ?
                                 "void I1<A>.M2()" :
                                 "void I1<A?>.M2()",
                             m2Implementations[0].ToTestDisplayString(includeNonNullable: true));

                Assert.Same(m2,
                            c2.FindImplementationForInterfaceMember(((TypeSymbol)c2Interfaces[0]).GetMember("M2")));
                Assert.Same(m2,
                            c2.FindImplementationForInterfaceMember(((TypeSymbol)c1Interfaces[0]).GetMember("M2")));
            };

            CompileAndVerify(comp, sourceSymbolValidator: validate, symbolValidator: validate, expectedOutput: @"C1.M1 C1.M2");
        }

        [Fact]
        public void Implementing_14()
        {
            var source = @"
public interface I1<T> 
{
    void M1();
}

public class A {}

public class C1 : I1<A?> 
{ 
}

public class C2 : C1, I1<A> 
{ 
}

public class C3 : C1, I1<A?> 
{ 
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (9,19): error CS0535: 'C1' does not implement interface member 'I1<A?>.M1()'
                // public class C1 : I1<A?>
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "I1<A?>").WithArguments("C1", "I1<A?>.M1()").WithLocation(9, 19),
                // (13,23): warning CS8644: 'C2' does not implement interface member 'I1<A>.M1()'. Nullability of reference types in interface implemented by the base type doesn't match.
                // public class C2 : C1, I1<A>
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInInterfaceImplementedByBase, "I1<A>").WithArguments("C2", "I1<A>.M1()").WithLocation(13, 23)
                );
        }

        [Fact]
        public void Implementing_15()
        {
            var source1 = @"
public interface I1<T> 
{
    void M1();
}

public class A {}

public class C1 : I1<A?> 
{ 
}
";
            var comp1 = CreateCompilation(source1, options: WithNullableEnable());
            comp1.VerifyDiagnostics(
                // (9,19): error CS0535: 'C1' does not implement interface member 'I1<A?>.M1()'
                // public class C1 : I1<A?>
                Diagnostic(ErrorCode.ERR_UnimplementedInterfaceMember, "I1<A?>").WithArguments("C1", "I1<A?>.M1()").WithLocation(9, 19)
                );

            var source2 = @"
public class C2 : C1, I1<A> 
{ 
}

public class C3 : C1, I1<A?> 
{ 
}
";
            var comp2 = CreateCompilation(source2, references: new[] { comp1.ToMetadataReference() }, options: WithNullableEnable());
            comp2.VerifyDiagnostics(
                // (2,23): warning CS8644: 'C2' does not implement interface member 'I1<A>.M1()'. Nullability of reference types in interface implemented by the base type doesn't match.
                // public class C2 : C1, I1<A>
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInInterfaceImplementedByBase, "I1<A>").WithArguments("C2", "I1<A>.M1()").WithLocation(2, 23)
                );
        }

        [Fact]
        public void Implementing_16()
        {
            var source = @"
public interface I1<T> 
{
    void M();
}

public interface I2<I2T> : I1<I2T?> where I2T : class { }
public interface I3<I3T> : I1<I3T> where I3T : class { }

public class C1<T> : I2<T>, I1<T> where T : class 
{ 
    void I1<T?>.M(){}
    void I1<T>.M(){}
}

public class C2<T> : I1<T>, I2<T> where T : class  
{ 
    void I1<T?>.M(){}
    void I1<T>.M(){}
}

public class C3<T> : I1<T>, I1<T?> where T : class 
{
    void I1<T?>.M(){}
    void I1<T>.M(){}
}

public class C4<T> : I2<T>, I3<T> where T : class 
{
    void I1<T?>.M(){}
    void I1<T>.M(){}
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (10,14): warning CS8645: 'I1<T>' is already listed in the interface list on type 'C1<T>' with different nullability of reference types.
                // public class C1<T> : I2<T>, I1<T> where T : class 
                Diagnostic(ErrorCode.WRN_DuplicateInterfaceWithNullabilityMismatchInBaseList, "C1").WithArguments("I1<T>", "C1<T>").WithLocation(10, 14),
                // (10,14): error CS8646: 'I1<T?>.M()' is explicitly implemented more than once.
                // public class C1<T> : I2<T>, I1<T> where T : class 
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C1").WithArguments("I1<T?>.M()").WithLocation(10, 14),
                // (10,14): error CS8646: 'I1<T>.M()' is explicitly implemented more than once.
                // public class C1<T> : I2<T>, I1<T> where T : class 
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C1").WithArguments("I1<T>.M()").WithLocation(10, 14),
                // (16,14): warning CS8645: 'I1<T?>' is already listed in the interface list on type 'C2<T>' with different nullability of reference types.
                // public class C2<T> : I1<T>, I2<T> where T : class  
                Diagnostic(ErrorCode.WRN_DuplicateInterfaceWithNullabilityMismatchInBaseList, "C2").WithArguments("I1<T?>", "C2<T>").WithLocation(16, 14),
                // (16,14): error CS8646: 'I1<T>.M()' is explicitly implemented more than once.
                // public class C2<T> : I1<T>, I2<T> where T : class  
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C2").WithArguments("I1<T>.M()").WithLocation(16, 14),
                // (16,14): error CS8646: 'I1<T?>.M()' is explicitly implemented more than once.
                // public class C2<T> : I1<T>, I2<T> where T : class  
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C2").WithArguments("I1<T?>.M()").WithLocation(16, 14),
                // (22,14): warning CS8645: 'I1<T?>' is already listed in the interface list on type 'C3<T>' with different nullability of reference types.
                // public class C3<T> : I1<T>, I1<T?> where T : class 
                Diagnostic(ErrorCode.WRN_DuplicateInterfaceWithNullabilityMismatchInBaseList, "C3").WithArguments("I1<T?>", "C3<T>").WithLocation(22, 14),
                // (22,14): error CS8646: 'I1<T>.M()' is explicitly implemented more than once.
                // public class C3<T> : I1<T>, I1<T?> where T : class 
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C3").WithArguments("I1<T>.M()").WithLocation(22, 14),
                // (22,14): error CS8646: 'I1<T?>.M()' is explicitly implemented more than once.
                // public class C3<T> : I1<T>, I1<T?> where T : class 
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C3").WithArguments("I1<T?>.M()").WithLocation(22, 14),
                // (28,14): warning CS8645: 'I1<T>' is already listed in the interface list on type 'C4<T>' with different nullability of reference types.
                // public class C4<T> : I2<T>, I3<T> where T : class 
                Diagnostic(ErrorCode.WRN_DuplicateInterfaceWithNullabilityMismatchInBaseList, "C4").WithArguments("I1<T>", "C4<T>").WithLocation(28, 14),
                // (28,14): error CS8646: 'I1<T?>.M()' is explicitly implemented more than once.
                // public class C4<T> : I2<T>, I3<T> where T : class 
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C4").WithArguments("I1<T?>.M()").WithLocation(28, 14),
                // (28,14): error CS8646: 'I1<T>.M()' is explicitly implemented more than once.
                // public class C4<T> : I2<T>, I3<T> where T : class 
                Diagnostic(ErrorCode.ERR_DuplicateExplicitImpl, "C4").WithArguments("I1<T>.M()").WithLocation(28, 14)
                );

            var c1 = (NamedTypeSymbol)comp.GetTypeByMetadataName("C1`1");
            var c1Interfaces = c1.Interfaces();
            var c1AllInterfaces = c1.AllInterfaces();
            Assert.Equal(2, c1Interfaces.Length);
            Assert.Equal(3, c1AllInterfaces.Length);
            Assert.Equal("I2<T!>", c1Interfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<T!>", c1Interfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I2<T!>", c1AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<T?>", c1AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<T!>", c1AllInterfaces[2].ToTestDisplayString(includeNonNullable: true));
            assertExplicitInterfaceImplementations(c1);

            var c2 = (NamedTypeSymbol)comp.GetTypeByMetadataName("C2`1");
            var c2Interfaces = c2.Interfaces();
            var c2AllInterfaces = c2.AllInterfaces();
            Assert.Equal(2, c2Interfaces.Length);
            Assert.Equal(3, c2AllInterfaces.Length);
            Assert.Equal("I1<T!>", c2Interfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I2<T!>", c2Interfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<T!>", c2AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I2<T!>", c2AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<T?>", c2AllInterfaces[2].ToTestDisplayString(includeNonNullable: true));
            assertExplicitInterfaceImplementations(c2);

            var c3 = (NamedTypeSymbol)comp.GetTypeByMetadataName("C3`1");
            var c3Interfaces = c3.Interfaces();
            var c3AllInterfaces = c3.AllInterfaces();
            Assert.Equal(2, c3Interfaces.Length);
            Assert.Equal(2, c3AllInterfaces.Length);
            Assert.Equal("I1<T!>", c3Interfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<T?>", c3Interfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<T!>", c3AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<T?>", c3AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
            assertExplicitInterfaceImplementations(c3);

            var c4 = (NamedTypeSymbol)comp.GetTypeByMetadataName("C4`1");
            var c4Interfaces = c4.Interfaces();
            var c4AllInterfaces = c4.AllInterfaces();
            Assert.Equal(2, c4Interfaces.Length);
            Assert.Equal(4, c4AllInterfaces.Length);
            Assert.Equal("I2<T!>", c4Interfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I3<T!>", c4Interfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I2<T!>", c4AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<T?>", c4AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I3<T!>", c4AllInterfaces[2].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<T!>", c4AllInterfaces[3].ToTestDisplayString(includeNonNullable: true));
            assertExplicitInterfaceImplementations(c4);

            void assertExplicitInterfaceImplementations(NamedTypeSymbol c)
            {
                var members = c.GetMembers("I1<T>.M");
                Assert.Equal(2, members.Length);
                var cMabImplementations = ((MethodSymbol)members[0]).ExplicitInterfaceImplementations;
                Assert.Equal(1, cMabImplementations.Length);
                Assert.Equal("void I1<T?>.M()", cMabImplementations[0].ToTestDisplayString(includeNonNullable: true));
                var cMcdImplementations = ((MethodSymbol)members[1]).ExplicitInterfaceImplementations;
                Assert.Equal(1, cMcdImplementations.Length);
                Assert.Equal("void I1<T!>.M()", cMcdImplementations[0].ToTestDisplayString(includeNonNullable: true));
            }
        }

        [Fact]
        public void Implementing_17()
        {
            var source = @"
public interface I1<T> 
{
    void M();
}

public class C3<T, U> : I1<T>, I1<U?> where T : class where U : class 
{
    void I1<U?>.M(){}
    void I1<T>.M(){}
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (7,14): error CS0695: 'C3<T, U>' cannot implement both 'I1<T>' and 'I1<U?>' because they may unify for some type parameter substitutions
                // public class C3<T, U> : I1<T>, I1<U?> where T : class where U : class
                Diagnostic(ErrorCode.ERR_UnifyingInterfaceInstantiations, "C3").WithArguments("C3<T, U>", "I1<T>", "I1<U?>").WithLocation(7, 14)
                );

            var c3 = (NamedTypeSymbol)comp.GetTypeByMetadataName("C3`2");
            var c3Interfaces = c3.Interfaces();
            var c3AllInterfaces = c3.AllInterfaces();
            Assert.Equal(2, c3Interfaces.Length);
            Assert.Equal(2, c3AllInterfaces.Length);
            Assert.Equal("I1<T!>", c3Interfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<U?>", c3Interfaces[1].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<T!>", c3AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
            Assert.Equal("I1<U?>", c3AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
            assertExplicitInterfaceImplementations(c3);

            void assertExplicitInterfaceImplementations(NamedTypeSymbol c)
            {
                var cMabImplementations = ((MethodSymbol)((TypeSymbol)c).GetMember("I1<T>.M")).ExplicitInterfaceImplementations;
                Assert.Equal(1, cMabImplementations.Length);
                Assert.Equal("void I1<T!>.M()", cMabImplementations[0].ToTestDisplayString(includeNonNullable: true));
                var cMcdImplementations = ((MethodSymbol)((TypeSymbol)c).GetMember("I1<U>.M")).ExplicitInterfaceImplementations;
                Assert.Equal(1, cMcdImplementations.Length);
                Assert.Equal("void I1<U?>.M()", cMcdImplementations[0].ToTestDisplayString(includeNonNullable: true));
            }
        }

        [Fact]
        public void Implementing_18()
        {
            var source = @"
public interface I1<T> 
{
    void M();
}

public class A {}

public class C3 : I1<A>
{
    void I1<A?>.M() 
    {
        System.Console.Write(""C3.M "");
    }

    static void Main()
    {
        var x = new C3();
        ((I1<A>)x).M();
        ((I1<A?>)x).M();
    }
}

public class C4 : I1<A?>
{
    void I1<A?>.M(){}
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics(
                // (11,10): warning CS8643: Nullability of reference types in explicit interface specifier doesn't match interface implemented by the type.
                //     void I1<A?>.M()
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInExplicitlyImplementedInterface, "I1<A?>").WithLocation(11, 10)
                );

            Action<ModuleSymbol> validate = (ModuleSymbol m) =>
            {
                bool isMetadata = m is PEModuleSymbol;

                var c3 = (NamedTypeSymbol)m.GlobalNamespace.GetTypeMember("C3");
                var c3Interfaces = c3.Interfaces();
                var c3AllInterfaces = c3.AllInterfaces();
                Assert.Equal(1, c3Interfaces.Length);
                Assert.Equal(1, c3AllInterfaces.Length);
                Assert.Equal("I1<A!>", c3Interfaces[0].ToTestDisplayString(includeNonNullable: true));
                Assert.Equal("I1<A!>", c3AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));

                var method = (MethodSymbol)((TypeSymbol)c3).GetMember("I1<A>.M");
                Assert.Equal("I1<A>.M", method.Name);

                var mImplementations = method.ExplicitInterfaceImplementations;
                Assert.Equal(1, mImplementations.Length);
                Assert.Equal(isMetadata ?
                                 "void I1<A>.M()" :
                                 "void I1<A?>.M()",
                             mImplementations[0].ToTestDisplayString(includeNonNullable: true));

                Assert.Same(method,
                            c3.FindImplementationForInterfaceMember(((TypeSymbol)c3Interfaces[0]).GetMember("M")));

                Assert.Same(method,
                            c3.FindImplementationForInterfaceMember(mImplementations[0]));

                Assert.Same(method,
                            c3.FindImplementationForInterfaceMember(m.GlobalNamespace.GetTypeMember("C4").InterfacesNoUseSiteDiagnostics()[0].GetMember("M")));
            };

            CompileAndVerify(comp, sourceSymbolValidator: validate, symbolValidator: validate, expectedOutput: @"C3.M C3.M");
        }

        [Fact]
        public void Implementing_19()
        {
            var source = @"
public interface I1<T> 
{
    void M();
}

public class A {}

public class C3 : I1<A>, I1<A?>
{
    void I1<A?>.M() 
    {
        System.Console.Write(""C3.M "");
    }

    static void Main()
    {
        var x = new C3();
        ((I1<A>)x).M();
        ((I1<A?>)x).M();
    }
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics(
                // (9,14): warning CS8645: 'I1<A?>' is already listed in the interface list on type 'C3' with different nullability of reference types.
                // public class C3 : I1<A>, I1<A?>
                Diagnostic(ErrorCode.WRN_DuplicateInterfaceWithNullabilityMismatchInBaseList, "C3").WithArguments("I1<A?>", "C3").WithLocation(9, 14)
                );

            Action<ModuleSymbol> validate = (ModuleSymbol m) =>
            {
                bool isMetadata = m is PEModuleSymbol;

                var c3 = (NamedTypeSymbol)m.GlobalNamespace.GetTypeMember("C3");
                var c3Interfaces = c3.Interfaces();
                var c3AllInterfaces = c3.AllInterfaces();

                if (isMetadata)
                {
                    Assert.Equal(1, c3Interfaces.Length);
                    Assert.Equal(1, c3AllInterfaces.Length);
                    Assert.Equal("I1<A!>", c3Interfaces[0].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A!>", c3AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
                }
                else
                {
                    Assert.Equal(2, c3Interfaces.Length);
                    Assert.Equal(2, c3AllInterfaces.Length);
                    Assert.Equal("I1<A!>", c3Interfaces[0].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A?>", c3Interfaces[1].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A!>", c3AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A?>", c3AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
                }

                var method = (MethodSymbol)((TypeSymbol)c3).GetMember("I1<A>.M");
                Assert.Equal("I1<A>.M", method.Name);

                var mImplementations = method.ExplicitInterfaceImplementations;
                Assert.Equal(1, mImplementations.Length);
                Assert.Equal(isMetadata ?
                                 "void I1<A>.M()" :
                                 "void I1<A?>.M()",
                             mImplementations[0].ToTestDisplayString(includeNonNullable: true));

                Assert.Same(method,
                            c3.FindImplementationForInterfaceMember(((TypeSymbol)c3Interfaces[0]).GetMember("M")));

                Assert.Same(method,
                            c3.FindImplementationForInterfaceMember(mImplementations[0]));
            };

            CompileAndVerify(comp, sourceSymbolValidator: validate, symbolValidator: validate, expectedOutput: @"C3.M C3.M");
        }

        [Fact]
        public void Implementing_20()
        {
            var source = @"
public interface I1<T> 
{
    void M();
}

public class A {}

public interface I2 : I1<A?>
{}

public class C3 : I2, I1<A>
{
    void I1<A?>.M() 
    {
        System.Console.Write(""C3.M "");
    }

    static void Main()
    {
        var x = new C3();
        ((I1<A>)x).M();
        ((I1<A?>)x).M();
    }
}
";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics(
                // (12,14): warning CS8645: 'I1<A>' is already listed in the interface list on type 'C3' with different nullability of reference types.
                // public class C3 : I2, I1<A>
                Diagnostic(ErrorCode.WRN_DuplicateInterfaceWithNullabilityMismatchInBaseList, "C3").WithArguments("I1<A>", "C3").WithLocation(12, 14)
                );

            Action<ModuleSymbol> validate = (ModuleSymbol m) =>
            {
                bool isMetadata = m is PEModuleSymbol;

                var c3 = (NamedTypeSymbol)m.GlobalNamespace.GetTypeMember("C3");
                var c3Interfaces = c3.Interfaces();
                var c3AllInterfaces = c3.AllInterfaces();

                if (isMetadata)
                {
                    Assert.Equal(2, c3Interfaces.Length);
                    Assert.Equal(2, c3AllInterfaces.Length);
                    Assert.Equal("I2", c3Interfaces[0].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A?>", c3Interfaces[1].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I2", c3AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A?>", c3AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
                }
                else
                {
                    Assert.Equal(2, c3Interfaces.Length);
                    Assert.Equal(3, c3AllInterfaces.Length);
                    Assert.Equal("I2", c3Interfaces[0].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A!>", c3Interfaces[1].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I2", c3AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A?>", c3AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A!>", c3AllInterfaces[2].ToTestDisplayString(includeNonNullable: true));
                }

                var method = (MethodSymbol)((TypeSymbol)c3).GetMember("I1<A>.M");
                Assert.Equal("I1<A>.M", method.Name);

                var mImplementations = method.ExplicitInterfaceImplementations;
                Assert.Equal(1, mImplementations.Length);
                Assert.Equal(isMetadata ?
                                 "void I1<A>.M()" :
                                 "void I1<A?>.M()",
                             mImplementations[0].ToTestDisplayString(includeNonNullable: true));

                Assert.Same(method,
                            c3.FindImplementationForInterfaceMember(((TypeSymbol)c3Interfaces[1]).GetMember("M")));

                Assert.Same(method,
                            c3.FindImplementationForInterfaceMember(mImplementations[0]));
            };

            CompileAndVerify(comp, sourceSymbolValidator: validate, symbolValidator: validate, expectedOutput: @"C3.M C3.M");
        }

        [Fact]
        public void Implementing_21()
        {
            var source = @"
public interface I1<T> 
{
    void M();
}

public class A {}

public partial class C3 : I1<A>
{
    void I1<A?>.M() 
    {
        System.Console.Write(""C3.M "");
    }

    static void Main()
    {
        var x = new C3();
        ((I1<A>)x).M();
        ((I1<A?>)x).M();
    }
}

public partial class C3 : I1<A?> {}

";
            var comp = CreateCompilation(source, options: WithNullableEnable(TestOptions.DebugExe));
            comp.VerifyDiagnostics(
                // (9,22): warning CS8645: 'I1<A?>' is already listed in the interface list on type 'C3' with different nullability of reference types.
                // public partial class C3 : I1<A>
                Diagnostic(ErrorCode.WRN_DuplicateInterfaceWithNullabilityMismatchInBaseList, "C3").WithArguments("I1<A?>", "C3").WithLocation(9, 22)
                );

            Action<ModuleSymbol> validate = (ModuleSymbol m) =>
            {
                bool isMetadata = m is PEModuleSymbol;

                var c3 = (NamedTypeSymbol)m.GlobalNamespace.GetTypeMember("C3");
                var c3Interfaces = c3.Interfaces();
                var c3AllInterfaces = c3.AllInterfaces();

                if (isMetadata)
                {
                    Assert.Equal(1, c3Interfaces.Length);
                    Assert.Equal(1, c3AllInterfaces.Length);
                    Assert.Equal("I1<A!>", c3Interfaces[0].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A!>", c3AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
                }
                else
                {
                    Assert.Equal(2, c3Interfaces.Length);
                    Assert.Equal(2, c3AllInterfaces.Length);
                    Assert.Equal("I1<A!>", c3Interfaces[0].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A?>", c3Interfaces[1].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A!>", c3AllInterfaces[0].ToTestDisplayString(includeNonNullable: true));
                    Assert.Equal("I1<A?>", c3AllInterfaces[1].ToTestDisplayString(includeNonNullable: true));
                }

                var method = (MethodSymbol)((TypeSymbol)c3).GetMember("I1<A>.M");
                Assert.Equal("I1<A>.M", method.Name);

                var mImplementations = method.ExplicitInterfaceImplementations;
                Assert.Equal(1, mImplementations.Length);
                Assert.Equal(isMetadata ?
                                 "void I1<A>.M()" :
                                 "void I1<A?>.M()",
                             mImplementations[0].ToTestDisplayString(includeNonNullable: true));

                Assert.Same(method,
                            c3.FindImplementationForInterfaceMember(((TypeSymbol)c3Interfaces[0]).GetMember("M")));

                Assert.Same(method,
                            c3.FindImplementationForInterfaceMember(mImplementations[0]));
            };

            CompileAndVerify(comp, sourceSymbolValidator: validate, symbolValidator: validate, expectedOutput: @"C3.M C3.M");
        }

        [Fact]
        public void Implementing_23()
        {

            var source = @"
class C
{
    public static void Main()
    { 
    }
}
interface IA
{
    void M1(string[] x); 
    void M2<T>(T[] x) where T : class; 
    void M3<T>(T?[]? x) where T : class; 
}
class B : IA
{
    void IA.M1(string?[] x)
    {
    } 

    void IA.M2<T>(T?[] x)  where T : class
    {
    } 

    void IA.M3<T>(T?[]? x)  where T : class
    {
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics();

            var ia = compilation.GetTypeByMetadataName("IA");
            var b = compilation.GetTypeByMetadataName("B");

            foreach (var memberName in new[] { "M1", "M2" })
            {
                var member = ia.GetMember<MethodSymbol>(memberName);
                var implementing = (MethodSymbol)b.FindImplementationForInterfaceMember(member);
                var implemented = member.ConstructIfGeneric(implementing.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t)));
                Assert.False(implementing.Parameters[0].TypeWithAnnotations.Equals(implemented.Parameters[0].TypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            {
                var member = ia.GetMember<MethodSymbol>("M3");
                var implementing = (MethodSymbol)b.FindImplementationForInterfaceMember(member);
                var implemented = member.ConstructIfGeneric(implementing.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t)));
                Assert.True(implementing.Parameters[0].TypeWithAnnotations.Equals(implemented.Parameters[0].TypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }
        }

        [Fact]
        public void Implementing_24()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

interface IA
{
    string[] M1(); 
    T[] M2<T>() where T : class; 
    T?[]? M3<T>() where T : class; 
}

class B : IA
{
    string?[] IA.M1()
    {
        return new string?[] {};
    } 

    S?[] IA.M2<S>() where S : class
    {
        return new S?[] {};
    } 

    S?[]? IA.M3<S>() where S : class
    {
        return new S?[] {};
    } 
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());

            compilation.VerifyDiagnostics(
                // (23,13): warning CS8616: Nullability of reference types in return type doesn't match implemented member 'T[] IA.M2<T>()'.
                //     S?[] IA.M2<S>() where S : class
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "M2").WithArguments("T[] IA.M2<T>()").WithLocation(23, 13),
                // (18,18): warning CS8616: Nullability of reference types in return type doesn't match implemented member 'string[] IA.M1()'.
                //     string?[] IA.M1()
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInReturnTypeOnExplicitImplementation, "M1").WithArguments("string[] IA.M1()").WithLocation(18, 18)
                );

            var ia = compilation.GetTypeByMetadataName("IA");
            var b = compilation.GetTypeByMetadataName("B");

            foreach (var memberName in new[] { "M1", "M2" })
            {
                var member = ia.GetMember<MethodSymbol>(memberName);
                var implementing = (MethodSymbol)b.FindImplementationForInterfaceMember(member);
                var implemented = member.ConstructIfGeneric(implementing.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t)));
                Assert.False(implementing.ReturnTypeWithAnnotations.Equals(implemented.ReturnTypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            {
                var member = ia.GetMember<MethodSymbol>("M3");
                var implementing = (MethodSymbol)b.FindImplementationForInterfaceMember(member);
                var implemented = member.ConstructIfGeneric(implementing.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t)));
                Assert.True(implementing.ReturnTypeWithAnnotations.Equals(implemented.ReturnTypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }
        }

        [Fact]
        public void Implementing_25()
        {
            var source = @"
#nullable enable
    interface I
    {
        void M<T>(T value) where T : class;
    }

    class C : I
    {
        void I.M<T>(T value)
        {
            T? x = value;
        }
    }
";
            var compilation = CreateCompilation(source).VerifyDiagnostics();

            var c = compilation.GetTypeByMetadataName("C");

            var member = c.GetMember<MethodSymbol>("I.M");
            var tp = member.GetMemberTypeParameters()[0];
            Assert.True(tp.IsReferenceType);
            Assert.False(tp.IsNullableType());
        }

        [Fact]
        public void PartialMethods_01()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

partial class C1
{
    partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
}

partial class C1
{
    partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
    { }
}";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics(
                 // (16,18): warning CS8611: Nullability of reference types in type of parameter 'y' doesn't match partial method declaration.
                 //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                 Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnPartial, "M1").WithArguments("y").WithLocation(16, 18),
                 // (16,18): warning CS8611: Nullability of reference types in type of parameter 'z' doesn't match partial method declaration.
                 //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                 Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnPartial, "M1").WithArguments("z").WithLocation(16, 18)
                );

            var c1 = compilation.GetTypeByMetadataName("C1");
            var m1 = c1.GetMember<MethodSymbol>("M1");
            var m1Impl = m1.PartialImplementationPart;
            var m1Def = m1.ConstructIfGeneric(m1Impl.TypeParameters.SelectAsArray(t => TypeWithAnnotations.Create(t)));

            for (int i = 0; i < 3; i++)
            {
                Assert.False(m1Impl.Parameters[i].TypeWithAnnotations.Equals(m1Def.Parameters[i].TypeWithAnnotations,
                    TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));
            }

            Assert.True(m1Impl.Parameters[3].TypeWithAnnotations.Equals(m1Def.Parameters[3].TypeWithAnnotations,
                TypeCompareKind.AllIgnoreOptions & ~TypeCompareKind.AllNullableIgnoreOptions));

            compilation = CreateCompilation("", references: new[] { compilation.EmitToImageReference() }, options: TestOptions.ReleaseDll.WithMetadataImportOptions(MetadataImportOptions.All));
            c1 = compilation.GetTypeByMetadataName("C1");
            m1 = c1.GetMember<MethodSymbol>("M1");
            Assert.Equal("void C1.M1<T>(T! x, T?[]! y, System.Action<T!>! z, System.Action<T?[]?>?[]? u)", m1.ToTestDisplayString(includeNonNullable: true));
        }

        [Fact]
        public void PartialMethods_02_01()
        {
            var source = @"
partial class C1
{
#nullable disable
    partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
}

partial class C1
{
    partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
    { }
}";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);

            compilation.VerifyDiagnostics(
                // (5,30): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 30),
                // (5,29): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(5, 29),
                // (5,72): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 72),
                // (5,71): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(5, 71),
                // (5,75): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 75),
                // (5,77): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 77),
                // (5,80): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 80),
                // (10,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(10, 25),
                // (10,24): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(10, 24),
                // (10,33): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(10, 33),
                // (10,53): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(10, 53),
                // (10,52): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(10, 52),
                // (10,74): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(10, 74),
                // (10,73): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(10, 73),
                // (10,77): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(10, 77),
                // (10,79): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(10, 79),
                // (10,82): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(10, 82)
                );
        }

        [Fact]
        [WorkItem(30229, "https://github.com/dotnet/roslyn/issues/30229")]
        public void PartialMethods_02_02()
        {
            var source = @"
partial class C1
{
#nullable disable
    partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
}
#nullable enable
partial class C1
{
    partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
    { }
}";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable(), parseOptions: TestOptions.Regular8);

            compilation.VerifyDiagnostics(
                // (5,30): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 30),
                // (5,29): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(5, 29),
                // (5,72): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 72),
                // (5,71): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(5, 71),
                // (5,75): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 75),
                // (5,77): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 77),
                // (5,80): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 80),
                // (10,18): warning CS8611: Nullability of reference types in type of parameter 'y' doesn't match partial method declaration.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInParameterTypeOnPartial, "M1").WithArguments("y").WithLocation(10, 18)
                );
        }

        [Fact]
        public void PartialMethods_03()
        {
            var source = @"
class C
{
    public static void Main()
    { 
    }
}

partial class C1
{
    partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
}

partial class C1
{
#nullable disable
    partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
    { }
}";
            var compilation = CreateCompilation(new[] { source }, parseOptions: TestOptions.Regular8);
            compilation.VerifyDiagnostics(
                // (11,30): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(11, 30),
                // (11,29): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(11, 29),
                // (11,72): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(11, 72),
                // (11,71): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(11, 71),
                // (11,75): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(11, 75),
                // (11,77): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(11, 77),
                // (11,80): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T x, T?[] y, System.Action<T> z, System.Action<T?[]?>?[]? u) where T : class;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(11, 80),
                // (17,25): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(17, 25),
                // (17,24): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(17, 24),
                // (17,33): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(17, 33),
                // (17,53): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(17, 53),
                // (17,52): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(17, 52),
                // (17,74): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(17, 74),
                // (17,73): error CS8627: A nullable type parameter must be known to be a value type or non-nullable reference type unless language version '9.0' or greater is used. Consider changing the language version or adding a 'class', 'struct', or type constraint.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.ERR_NullableUnconstrainedTypeParameter, "T?").WithArguments("9.0").WithLocation(17, 73),
                // (17,77): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(17, 77),
                // (17,79): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(17, 79),
                // (17,82): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     partial void M1<T>(T? x, T[]? y, System.Action<T?> z, System.Action<T?[]?>?[]? u) where T : class
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(17, 82)
                );
        }

        [Fact]
        public void Overloading_01()
        {
            var source = @"
class A
{
    void Test1(string? x1) {}
    void Test1(string x2) {}

    string Test2(string y1) { return y1; }
    string? Test2(string y2) { return y2; }
}
";
            CreateCompilation(new[] { source }, options: WithNullableEnable()).
                VerifyDiagnostics(
                // (5,10): error CS0111: Type 'A' already defines a member called 'Test1' with the same parameter types
                //     void Test1(string x2) {}
                Diagnostic(ErrorCode.ERR_MemberAlreadyExists, "Test1").WithArguments("Test1", "A").WithLocation(5, 10),
                // (8,13): error CS0111: Type 'A' already defines a member called 'Test2' with the same parameter types
                //     string? Test2(string y2) { return y2; }
                Diagnostic(ErrorCode.ERR_MemberAlreadyExists, "Test2").WithArguments("Test2", "A").WithLocation(8, 13)
                );
        }

        [Fact]
        public void Overloading_02()
        {
            var source = @"
class A
{
    public void M1<T>(T? x) where T : struct 
    { 
    }

    public void M1<T>(T? x) where T : class 
    { 
    }
}
";
            var compilation = CreateCompilation(new[] { source }, options: WithNullableEnable());
            compilation.VerifyDiagnostics();
        }

        [Fact]
        public void Test1()
        {
            CSharpCompilation c = CreateCompilation(
new[] { @"#pragma warning disable 8618
class C
{
    static void Main()
    {
    }

    void Test1()
    {
        string? x1 = null;
        string? y1 = x1;
        string z1 = x1;
    }

    void Test2()
    {
        string? x2 = """";
        string z2 = x2;
    }

    void Test3()
    {
        string? x3;
        string z3 = x3;
    }

    void Test4()
    {
        string x4;
        string z4 = x4;
    }

    void Test5()
    {
        string? x5 = """";
        x5 = null;
        string? y5;
        y5 = x5;
        string z5;
        z5 = x5;
    }

    void Test6()
    {
        string? x6 = """";
        string z6;
        z6 = x6;
    }

    void Test7()
    {
        CL1? x7 = null;
        CL1 y7 = x7.P1;
        CL1 z7 = x7?.P1;
        x7 = new CL1();
        CL1 u7 = x7.P1;
    }

    void Test8()
    {
        CL1? x8 = new CL1();
        CL1 y8 = x8.M1();
        x8 = null;
        CL1 u8 = x8.M1();
        CL1 z8 = x8?.M1();
    }

    void Test9(CL1? x9, CL1 y9)
    {
        CL1 u9;
        u9 = x9;
        u9 = y9;
        x9 = y9;
        CL1 v9;
        v9 = x9;
        y9 = null;
    }

    void Test10(CL1 x10)
    {
        CL1 u10;
        u10 = x10.P1;
        u10 = x10.P2;
        u10 = x10.M1();
        u10 = x10.M2();
        CL1? v10;
        v10 = x10.P2;
        v10 = x10.M2();
    }

    void Test11(CL1 x11, CL1? y11)
    {
        CL1 u11;
        u11 = x11.F1;
        u11 = x11.F2;
        CL1? v11;
        v11 = x11.F2;
        x11.F2 = x11.F1;
        u11 = x11.F2;

        v11 = y11.F1;
    }

    void Test12(CL1 x12)
    {
        S1 y12;
        CL1 u12;
        u12 = y12.F3;
        u12 = y12.F4;
    }

    void Test13(CL1 x13)
    {
        S1 y13;
        CL1? u13;
        u13 = y13.F3;
        u13 = y13.F4;
    }

    void Test14(CL1 x14)
    {
        S1 y14;
        y14.F3 = null;
        y14.F4 = null;
        y14.F3 = x14;
        y14.F4 = x14;
    }

    void Test15(CL1 x15)
    {
        S1 y15;
        CL1 u15;
        y15.F3 = null;
        y15.F4 = null;
        u15 = y15.F3;
        u15 = y15.F4;

        CL1? v15;
        v15 = y15.F4;
        y15.F4 = x15;
        u15 = y15.F4;
    }

    void Test16()
    {
        S1 y16;
        CL1 u16;
        y16 = new S1();
        u16 = y16.F3;
        u16 = y16.F4;
    }

    void Test17(CL1 z17)
    {
        S1 x17;
        x17.F4 = z17;
        S1 y17 = new S1();
        CL1 u17;
        u17 = y17.F4;

        y17 = x17;
        CL1 v17;
        v17 = y17.F4;
    }

    void Test18(CL1 z18)
    {
        S1 x18;
        x18.F4 = z18;
        S1 y18 = x18;
        CL1 u18;
        u18 = y18.F4;
    }

    void Test19(S1 x19, CL1 z19)
    {
        S1 y19;
        y19.F4 = null;
        CL1 u19;
        u19 = y19.F4;

        x19.F4 = z19;
        y19 = x19;
        CL1 v19;
        v19 = y19.F4;
    }

    void Test20(S1 x20, CL1 z20)
    {
        S1 y20;
        y20.F4 = z20;
        CL1 u20;
        u20 = y20.F4;

        y20 = x20;
        CL1 v20;
        v20 = y20.F4;
    }

    S1 GetS1()
    {
        return new S1();
    }
    void Test21(CL1 z21)
    {
        S1 y21;
        y21.F4 = z21;
        CL1 u21;
        u21 = y21.F4;

        y21 = GetS1();
        CL1 v21;
        v21 = y21.F4;
    }

    void Test22()
    {
        S1 y22;
        CL1 u22;
        u22 = y22.F4;

        y22 = GetS1();
        CL1 v22;
        v22 = y22.F4;
    }

    void Test23(CL1 z23)
    {
        S2 y23;
        y23.F5.F4 = z23;
        CL1 u23;
        u23 = y23.F5.F4;

        y23 = GetS2();
        CL1 v23;
        v23 = y23.F5.F4;
    }

    S2 GetS2()
    {
        return new S2();
    }

    void Test24()
    {
        S2 y24;
        CL1 u24;
        u24 = y24.F5.F4; // 1
        u24 = y24.F5.F4; // 2

        y24 = GetS2();
        CL1 v24;
        v24 = y24.F5.F4;
    }

    void Test25(CL1 z25)
    {
        S2 y25;
        S2 x25 = GetS2();
        x25.F5.F4 = z25;
        y25 = x25;
        CL1 v25;
        v25 = y25.F5.F4;
    }

    void Test26(CL1 x26, CL1? y26, CL1 z26)
    {
        x26.P1 = y26;
        x26.P1 = z26;
    }

    void Test27(CL1 x27, CL1? y27, CL1 z27)
    {
        x27[x27] = y27;
        x27[x27] = z27;
    }

    void Test28(CL1 x28, CL1? y28, CL1 z28)
    {
        x28[y28] = z28;
    }

    void Test29(CL1 x29, CL1 y29, CL1 z29)
    {
        z29 = x29[y29];
        z29 = x29[1];
    }

    void Test30(CL1? x30, CL1 y30, CL1 z30)
    {
        z30 = x30[y30];
    }

    void Test31(CL1 x31)
    {
        x31 = default(CL1);
    }

    void Test32(CL1 x32)
    {
        var y32 = new CL1() ?? x32;
    }

    void Test33(object x33)
    {
        var y33 = new { p = (object)null } ?? x33;
    }
}

class CL1
{
    public CL1()
    {
        F1 = this;
    }

    public CL1 F1;
    public CL1? F2;

    public CL1 P1 { get; set; }
    public CL1? P2 { get; set; }

    public CL1 M1() { return new CL1(); }
    public CL1? M2() { return null; }

    public CL1 this[CL1 x]
    {
        get { return x; }
        set { }
    }

    public CL1? this[int x]
    {
        get { return null; }
        set { }
    }
}

struct S1
{
    public CL1 F3;
    public CL1? F4;
}

struct S2
{
    public S1 F5;
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (12,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         string z1 = x1;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x1").WithLocation(12, 21),
                // (24,21): error CS0165: Use of unassigned local variable 'x3'
                //         string z3 = x3;
                Diagnostic(ErrorCode.ERR_UseDefViolation, "x3").WithArguments("x3").WithLocation(24, 21),
                // (24,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         string z3 = x3;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x3").WithLocation(24, 21),
                // (30,21): error CS0165: Use of unassigned local variable 'x4'
                //         string z4 = x4;
                Diagnostic(ErrorCode.ERR_UseDefViolation, "x4").WithArguments("x4").WithLocation(30, 21),
                // (40,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         z5 = x5;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x5").WithLocation(40, 14),
                // (53,18): warning CS8602: Dereference of a possibly null reference.
                //         CL1 y7 = x7.P1;
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x7").WithLocation(53, 18),
                // (54,18): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         CL1 z7 = x7?.P1;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x7?.P1").WithLocation(54, 18),
                // (64,18): warning CS8602: Dereference of a possibly null reference.
                //         CL1 u8 = x8.M1();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x8").WithLocation(64, 18),
                // (65,18): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         CL1 z8 = x8?.M1();
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x8?.M1()").WithLocation(65, 18),
                // (71,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u9 = x9;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x9").WithLocation(71, 14),
                // (76,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         y9 = null;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(76, 14),
                // (83,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u10 = x10.P2;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x10.P2").WithLocation(83, 15),
                // (85,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u10 = x10.M2();
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x10.M2()").WithLocation(85, 15),
                // (95,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u11 = x11.F2;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x11.F2").WithLocation(95, 15),
                // (101,15): warning CS8602: Dereference of a possibly null reference.
                //         v11 = y11.F1;
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "y11").WithLocation(101, 15),
                // (108,15): error CS0170: Use of possibly unassigned field 'F3'
                //         u12 = y12.F3;
                Diagnostic(ErrorCode.ERR_UseDefViolationField, "y12.F3").WithArguments("F3").WithLocation(108, 15),
                // (109,15): error CS0170: Use of possibly unassigned field 'F4'
                //         u12 = y12.F4;
                Diagnostic(ErrorCode.ERR_UseDefViolationField, "y12.F4").WithArguments("F4").WithLocation(109, 15),
                // (109,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u12 = y12.F4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y12.F4").WithLocation(109, 15),
                // (116,15): error CS0170: Use of possibly unassigned field 'F3'
                //         u13 = y13.F3;
                Diagnostic(ErrorCode.ERR_UseDefViolationField, "y13.F3").WithArguments("F3").WithLocation(116, 15),
                // (117,15): error CS0170: Use of possibly unassigned field 'F4'
                //         u13 = y13.F4;
                Diagnostic(ErrorCode.ERR_UseDefViolationField, "y13.F4").WithArguments("F4").WithLocation(117, 15),
                // (123,18): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         y14.F3 = null;
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(123, 18),
                // (133,18): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         y15.F3 = null;
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(133, 18),
                // (135,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u15 = y15.F3;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y15.F3").WithLocation(135, 15),
                // (136,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u15 = y15.F4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y15.F4").WithLocation(136, 15),
                // (149,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u16 = y16.F3;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y16.F3").WithLocation(149, 15),
                // (150,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u16 = y16.F4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y16.F4").WithLocation(150, 15),
                // (161,15): error CS0165: Use of unassigned local variable 'x17'
                //         y17 = x17;
                Diagnostic(ErrorCode.ERR_UseDefViolation, "x17").WithArguments("x17").WithLocation(161, 15),
                // (159,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u17 = y17.F4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y17.F4").WithLocation(159, 15),
                // (170,18): error CS0165: Use of unassigned local variable 'x18'
                //         S1 y18 = x18;
                Diagnostic(ErrorCode.ERR_UseDefViolation, "x18").WithArguments("x18").WithLocation(170, 18),
                // (180,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u19 = y19.F4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y19.F4").WithLocation(180, 15),
                // (197,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         v20 = y20.F4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y20.F4").WithLocation(197, 15),
                // (213,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         v21 = y21.F4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y21.F4").WithLocation(213, 15),
                // (220,15): error CS0170: Use of possibly unassigned field 'F4'
                //         u22 = y22.F4;
                Diagnostic(ErrorCode.ERR_UseDefViolationField, "y22.F4").WithArguments("F4").WithLocation(220, 15),
                // (220,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u22 = y22.F4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y22.F4").WithLocation(220, 15),
                // (224,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         v22 = y22.F4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y22.F4").WithLocation(224, 15),
                // (236,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         v23 = y23.F5.F4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y23.F5.F4").WithLocation(236, 15),
                // (248,15): error CS0170: Use of possibly unassigned field 'F4'
                //         u24 = y24.F5.F4; // 1
                Diagnostic(ErrorCode.ERR_UseDefViolationField, "y24.F5.F4").WithArguments("F4").WithLocation(248, 15),
                // (248,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u24 = y24.F5.F4; // 1
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y24.F5.F4").WithLocation(248, 15),
                // (249,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         u24 = y24.F5.F4; // 2
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y24.F5.F4").WithLocation(249, 15),
                // (253,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         v24 = y24.F5.F4;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y24.F5.F4").WithLocation(253, 15),
                // (268,18): warning CS8601: Possible null reference assignment.
                //         x26.P1 = y26;
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "y26").WithLocation(268, 18),
                // (274,20): warning CS8601: Possible null reference assignment.
                //         x27[x27] = y27;
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "y27").WithLocation(274, 20),
                // (280,13): warning CS8604: Possible null reference argument for parameter 'x' in 'CL1 CL1.this[CL1 x]'.
                //         x28[y28] = z28;
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "y28").WithArguments("x", "CL1 CL1.this[CL1 x]").WithLocation(280, 13),
                // (286,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         z29 = x29[1];
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x29[1]").WithLocation(286, 15),
                // (291,15): warning CS8602: Dereference of a possibly null reference.
                //         z30 = x30[y30];
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x30").WithLocation(291, 15),
                // (296,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         x31 = default(CL1);
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "default(CL1)").WithLocation(296, 15),
                // (306,29): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         var y33 = new { p = (object)null } ?? x33;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "(object)null").WithLocation(306, 29)
                );
        }

        [Fact]
        public void PassingParameters_01()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static void Main()
    {
    }

    void M1(CL1 p) {}

    void Test1(CL1? x1, CL1 y1)
    {
        M1(x1);
        M1(y1);
    }

    void Test2()
    {
        CL1? x2;
        M1(x2);
    }

    void M2(ref CL1? p) {}

    void Test3()
    {
        CL1 x3;
        M2(ref x3);
    }

    void Test4(CL1 x4)
    {
        M2(ref x4);
    }

    void M3(out CL1? p) { p = null; }

    void Test5()
    {
        CL1 x5;
        M3(out x5);
    }

    void M4(ref CL1 p) {}

    void Test6()
    {
        CL1? x6 = null;
        M4(ref x6);
    }

    void M5(out CL1 p) { p = new CL1(); }

    void Test7()
    {
        CL1? x7 = null;
        CL1 u7 = x7;
        M5(out x7);
        CL1 v7 = x7;
    }

    void M6(CL1 p1, CL1? p2) {}

    void Test8(CL1? x8, CL1? y8)
    {
        M6(p2: x8, p1: y8);
    }

    void M7(params CL1[] p1) {}

    void Test9(CL1 x9, CL1? y9)
    {
        M7(x9, y9);
    }

    void Test10(CL1? x10, CL1 y10)
    {
        M7(x10, y10);
    }

    void M8(CL1 p1, params CL1[] p2) {}

    void Test11(CL1? x11, CL1 y11, CL1? z11)
    {
        M8(x11, y11, z11);
    }

    void Test12(CL1? x12, CL1 y12)
    {
        M8(p2: x12, p1: y12);
    }
}

class CL1
{
}
" }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (12,12): warning CS8604: Possible null reference argument for parameter 'p' in 'void C.M1(CL1 p)'.
                //         M1(x1);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x1").WithArguments("p", "void C.M1(CL1 p)").WithLocation(12, 12),
                // (19,12): error CS0165: Use of unassigned local variable 'x2'
                //         M1(x2);
                Diagnostic(ErrorCode.ERR_UseDefViolation, "x2").WithArguments("x2").WithLocation(19, 12),
                // (19,12): warning CS8604: Possible null reference argument for parameter 'p' in 'void C.M1(CL1 p)'.
                //         M1(x2);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x2").WithArguments("p", "void C.M1(CL1 p)").WithLocation(19, 12),
                // (27,16): error CS0165: Use of unassigned local variable 'x3'
                //         M2(ref x3);
                Diagnostic(ErrorCode.ERR_UseDefViolation, "x3").WithArguments("x3").WithLocation(27, 16),
                // (27,16): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         M2(ref x3);
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x3").WithLocation(27, 16),
                // (32,16): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         M2(ref x4);
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x4").WithLocation(32, 16),
                // (40,16): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         M3(out x5);
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x5").WithLocation(40, 16),
                // (48,16): warning CS8601: Possible null reference assignment.
                //         M4(ref x6);
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "x6").WithLocation(48, 16),
                // (56,18): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         CL1 u7 = x7;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x7").WithLocation(56, 18),
                // (65,24): warning CS8604: Possible null reference argument for parameter 'p1' in 'void C.M6(CL1 p1, CL1? p2)'.
                //         M6(p2: x8, p1: y8);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "y8").WithArguments("p1", "void C.M6(CL1 p1, CL1? p2)").WithLocation(65, 24),
                // (72,16): warning CS8604: Possible null reference argument for parameter 'p1' in 'void C.M7(params CL1[] p1)'.
                //         M7(x9, y9);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "y9").WithArguments("p1", "void C.M7(params CL1[] p1)").WithLocation(72, 16),
                // (77,12): warning CS8604: Possible null reference argument for parameter 'p1' in 'void C.M7(params CL1[] p1)'.
                //         M7(x10, y10);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x10").WithArguments("p1", "void C.M7(params CL1[] p1)").WithLocation(77, 12),
                // (84,12): warning CS8604: Possible null reference argument for parameter 'p1' in 'void C.M8(CL1 p1, params CL1[] p2)'.
                //         M8(x11, y11, z11);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x11").WithArguments("p1", "void C.M8(CL1 p1, params CL1[] p2)").WithLocation(84, 12),
                // (84,22): warning CS8604: Possible null reference argument for parameter 'p2' in 'void C.M8(CL1 p1, params CL1[] p2)'.
                //         M8(x11, y11, z11);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "z11").WithArguments("p2", "void C.M8(CL1 p1, params CL1[] p2)").WithLocation(84, 22),
                // (89,16): warning CS8604: Possible null reference argument for parameter 'p2' in 'void C.M8(CL1 p1, params CL1[] p2)'.
                //         M8(p2: x12, p1: y12);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x12").WithArguments("p2", "void C.M8(CL1 p1, params CL1[] p2)").WithLocation(89, 16)
                );
        }

        [Fact]
        public void PassingParameters_02()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static void Main()
    {
    }

    void Test1(CL0 x1)
    {
        var y1 = new CL0() { [null] = x1 };
    }
}

class CL0
{
    public CL0 this[CL0 x]
    {
        get { return x; }
        set { }
    }
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (10,31): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         var y1 = new CL0() { [null] = x1 };
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 31));
        }

        [Fact]
        public void PassingParameters_03()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static void Main()
    {
    }

    void Test1(CL0 x1)
    {
        var y1 = new CL0() { null };
    }
}

class CL0 : System.Collections.IEnumerable 
{
    public void Add(CL0 x)
    {
    }

    System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
    {
        throw new System.NotImplementedException();
    }
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (10,30): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         var y1 = new CL0() { null };
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(10, 30));
        }

        [Fact]
        public void PassingParameters_04()
        {
            var source =
@"interface I<T> { }
class C
{
    static void F(I<object> x, I<object?> y, I<object>? z, I<object?>? w, I<object?>[]? a)
    {
        G(x);
        G(y); // 1
        G(x, x, x); // 2, 3
        G(x, y, y);
        G(x, x, y, z, w); // 4, 5, 6, 7
        G(y: x, x: y); // 8, 9
        G(y: y, x: x);
        G(x, a); // 10
        G(x, new I<object?>[0]);
        G(x, new[] { x, x }); // 11
        G(x, new[] { y, y });
        // note that the array type below is reinferred to 'I<object>[]'
        // due to previous usage of the variables as call arguments.
        G(x, new[] { x, y, z }); // 12, 13
        G(y: new[] { x, x }, x: y); // 14, 15
        G(y: new[] { y, y }, x: x);
    }
    static void G(I<object> x, params I<object?>[] y)
    {
    }
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (7,11): warning CS8620: Argument of type 'I<object?>' cannot be used for parameter 'x' of type 'I<object>' in 'void C.G(I<object> x, params I<object?>[] y)' due to differences in the nullability of reference types.
                //         G(y); // 1
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "y").WithArguments("I<object?>", "I<object>", "x", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(7, 11),
                // (8,14): warning CS8620: Argument of type 'I<object>' cannot be used for parameter 'y' of type 'I<object?>' in 'void C.G(I<object> x, params I<object?>[] y)' due to differences in the nullability of reference types.
                //         G(x, x, x); // 2, 3
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "x").WithArguments("I<object>", "I<object?>", "y", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(8, 14),
                // (8,17): warning CS8620: Argument of type 'I<object>' cannot be used for parameter 'y' of type 'I<object?>' in 'void C.G(I<object> x, params I<object?>[] y)' due to differences in the nullability of reference types.
                //         G(x, x, x); // 2, 3
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "x").WithArguments("I<object>", "I<object?>", "y", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(8, 17),
                // (10,14): warning CS8620: Argument of type 'I<object>' cannot be used for parameter 'y' of type 'I<object?>' in 'void C.G(I<object> x, params I<object?>[] y)' due to differences in the nullability of reference types.
                //         G(x, x, y, z, w); // 4, 5, 6, 7
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "x").WithArguments("I<object>", "I<object?>", "y", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(10, 14),
                // (10,20): warning CS8604: Possible null reference argument for parameter 'y' in 'void C.G(I<object> x, params I<object?>[] y)'.
                //         G(x, x, y, z, w); // 4, 5, 6, 7
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "z").WithArguments("y", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(10, 20),
                // (10,20): warning CS8620: Argument of type 'I<object>' cannot be used for parameter 'y' of type 'I<object?>' in 'void C.G(I<object> x, params I<object?>[] y)' due to differences in the nullability of reference types.
                //         G(x, x, y, z, w); // 4, 5, 6, 7
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "z").WithArguments("I<object>", "I<object?>", "y", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(10, 20),
                // (10,23): warning CS8604: Possible null reference argument for parameter 'y' in 'void C.G(I<object> x, params I<object?>[] y)'.
                //         G(x, x, y, z, w); // 4, 5, 6, 7
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "w").WithArguments("y", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(10, 23),
                // (11,14): warning CS8620: Argument of type 'I<object>' cannot be used for parameter 'y' of type 'I<object?>' in 'void C.G(I<object> x, params I<object?>[] y)' due to differences in the nullability of reference types.
                //         G(y: x, x: y); // 8, 9
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "x").WithArguments("I<object>", "I<object?>", "y", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(11, 14),
                // (11,20): warning CS8620: Argument of type 'I<object?>' cannot be used for parameter 'x' of type 'I<object>' in 'void C.G(I<object> x, params I<object?>[] y)' due to differences in the nullability of reference types.
                //         G(y: x, x: y); // 8, 9
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "y").WithArguments("I<object?>", "I<object>", "x", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(11, 20),
                // (13,14): warning CS8604: Possible null reference argument for parameter 'y' in 'void C.G(I<object> x, params I<object?>[] y)'.
                //         G(x, a); // 10
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "a").WithArguments("y", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(13, 14),
                // (15,14): warning CS8620: Argument of type 'I<object>[]' cannot be used for parameter 'y' of type 'I<object?>[]' in 'void C.G(I<object> x, params I<object?>[] y)' due to differences in the nullability of reference types.
                //         G(x, new[] { x, x }); // 11
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new[] { x, x }").WithArguments("I<object>[]", "I<object?>[]", "y", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(15, 14),
                // (19,14): warning CS8620: Argument of type 'I<object>[]' cannot be used for parameter 'y' of type 'I<object?>[]' in 'void C.G(I<object> x, params I<object?>[] y)' due to differences in the nullability of reference types.
                //         G(x, new[] { x, y, z }); // 12, 13
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new[] { x, y, z }").WithArguments("I<object>[]", "I<object?>[]", "y", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(19, 14),
                // (19,25): warning CS8619: Nullability of reference types in value of type 'I<object?>' doesn't match target type 'I<object>'.
                //         G(x, new[] { x, y, z }); // 12, 13
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "y").WithArguments("I<object?>", "I<object>").WithLocation(19, 25),
                // (20,14): warning CS8620: Argument of type 'I<object>[]' cannot be used for parameter 'y' of type 'I<object?>[]' in 'void C.G(I<object> x, params I<object?>[] y)' due to differences in the nullability of reference types.
                //         G(y: new[] { x, x }, x: y); // 14, 15
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "new[] { x, x }").WithArguments("I<object>[]", "I<object?>[]", "y", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(20, 14),
                // (20,33): warning CS8620: Argument of type 'I<object?>' cannot be used for parameter 'x' of type 'I<object>' in 'void C.G(I<object> x, params I<object?>[] y)' due to differences in the nullability of reference types.
                //         G(y: new[] { x, x }, x: y); // 14, 15
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "y").WithArguments("I<object?>", "I<object>", "x", "void C.G(I<object> x, params I<object?>[] y)").WithLocation(20, 33)
                );
        }

        [Fact]
        public void PassingParameters_DifferentRefKinds()
        {
            var source = @"
class C
{
    void M(string xNone, ref string xRef, out string xOut)
    {
        xNone = null;
        xRef = null;
        xOut = null;
    }
}
";
            var c = CreateCompilation(new[] { source }, options: WithNullableEnable());
            c.VerifyDiagnostics(
                // (6,17): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         xNone = null;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(6, 17),
                // (7,16): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         xRef = null;
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(7, 16),
                // (8,16): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         xOut = null;
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(8, 16)
                );

            var source2 = @"
class C
{
    void M(in string xIn)
    {
        xIn = null;
    }
}
";
            var c2 = CreateCompilation(new[] { source2 }, options: WithNullableEnable());
            c2.VerifyDiagnostics(
                // (6,9): error CS8331: Cannot assign to variable 'in string' or use it as the right hand side of a ref assignment because it is a readonly variable
                //         xIn = null;
                Diagnostic(ErrorCode.ERR_AssignReadonlyNotField, "xIn").WithArguments("variable", "in string").WithLocation(6, 9));
        }

        [Fact]
        [WorkItem(29916, "https://github.com/dotnet/roslyn/issues/29916")]
        public void PassingParameters_UnknownMethod()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static int F(object x)
    {
        Missing(F(null)); // 1
        Missing(F(x = null)); // 2
        x.ToString();

        Missing(F(x = this)); // 3
        x.ToString();
        return 0;
    }
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (6,9): error CS0103: The name 'Missing' does not exist in the current context
                //         Missing(F(null)); // 1
                Diagnostic(ErrorCode.ERR_NameNotInContext, "Missing").WithArguments("Missing").WithLocation(6, 9),
                // (6,19): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         Missing(F(null)); // 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(6, 19),
                // (7,9): error CS0103: The name 'Missing' does not exist in the current context
                //         Missing(F(x = null)); // 2
                Diagnostic(ErrorCode.ERR_NameNotInContext, "Missing").WithArguments("Missing").WithLocation(7, 9),
                // (7,19): warning CS8604: Possible null reference argument for parameter 'x' in 'int C.F(object x)'.
                //         Missing(F(x = null)); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x = null").WithArguments("x", "int C.F(object x)").WithLocation(7, 19),
                // (7,23): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         Missing(F(x = null)); // 2
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(7, 23),
                // (10,9): error CS0103: The name 'Missing' does not exist in the current context
                //         Missing(F(x = this)); // 3
                Diagnostic(ErrorCode.ERR_NameNotInContext, "Missing").WithArguments("Missing").WithLocation(10, 9),
                // (10,23): error CS0026: Keyword 'this' is not valid in a static property, static method, or static field initializer
                //         Missing(F(x = this)); // 3
                Diagnostic(ErrorCode.ERR_ThisInStaticMeth, "this").WithLocation(10, 23)
                );
        }

        [Fact]
        [WorkItem(29916, "https://github.com/dotnet/roslyn/issues/29916")]
        public void PassingParameters_UnknownMethod_UnknownReceiver()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static int F(object x)
    {
        bad.Missing(F(null)); // 1
        bad.Missing(F(x = null)); // 2
        x.ToString();

        bad.Missing(F(x = this)); // 3
        x.ToString();
        return 0;
    }
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (6,9): error CS0103: The name 'bad' does not exist in the current context
                //         bad.Missing(F(null)); // 1
                Diagnostic(ErrorCode.ERR_NameNotInContext, "bad").WithArguments("bad").WithLocation(6, 9),
                // (6,23): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //         bad.Missing(F(null)); // 1
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(6, 23),
                // (7,9): error CS0103: The name 'bad' does not exist in the current context
                //         bad.Missing(F(x = null)); // 2
                Diagnostic(ErrorCode.ERR_NameNotInContext, "bad").WithArguments("bad").WithLocation(7, 9),
                // (7,23): warning CS8604: Possible null reference argument for parameter 'x' in 'int C.F(object x)'.
                //         bad.Missing(F(x = null)); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x = null").WithArguments("x", "int C.F(object x)").WithLocation(7, 23),
                // (7,27): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         bad.Missing(F(x = null)); // 2
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(7, 27),
                // (10,9): error CS0103: The name 'bad' does not exist in the current context
                //         bad.Missing(F(x = this)); // 3
                Diagnostic(ErrorCode.ERR_NameNotInContext, "bad").WithArguments("bad").WithLocation(10, 9),
                // (10,27): error CS0026: Keyword 'this' is not valid in a static property, static method, or static field initializer
                //         bad.Missing(F(x = this)); // 3
                Diagnostic(ErrorCode.ERR_ThisInStaticMeth, "this").WithLocation(10, 27)
                );
        }

        [Fact]
        [WorkItem(29916, "https://github.com/dotnet/roslyn/issues/29916")]
        public void PassingParameters_UnknownMethod_AffectingState()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    object F(object x)
    {
        Missing(
            F(x = null) /*warn*/,
            x.ToString(),
            F(x = this),
            x.ToString());
        return x;
    }
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (6,9): error CS0103: The name 'Missing' does not exist in the current context
                //         Missing(
                Diagnostic(ErrorCode.ERR_NameNotInContext, "Missing").WithArguments("Missing").WithLocation(6, 9),
                // (7,15): warning CS8604: Possible null reference argument for parameter 'x' in 'object C.F(object x)'.
                //             F(x = null) /*warn*/,
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x = null").WithArguments("x", "object C.F(object x)").WithLocation(7, 15),
                // (7,19): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //             F(x = null) /*warn*/,
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(7, 19)
                );
        }

        [Fact]
        [WorkItem(29916, "https://github.com/dotnet/roslyn/issues/29916")]
        public void PassingParameters_UnknownMethod_AffectingConditionalState()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    int F(object x)
    {
        if (G(F(x = null)))
        {
            x.ToString();
        }
        else
        {
            x.ToString();
        }
        return 0;
    }
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (6,13): error CS0103: The name 'G' does not exist in the current context
                //         if (G(F(x = null)))
                Diagnostic(ErrorCode.ERR_NameNotInContext, "G").WithArguments("G").WithLocation(6, 13),
                // (6,17): warning CS8604: Possible null reference argument for parameter 'x' in 'int C.F(object x)'.
                //         if (G(F(x = null)))
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x = null").WithArguments("x", "int C.F(object x)").WithLocation(6, 17),
                // (6,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         if (G(F(x = null)))
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(6, 21)
                );
        }

        [Fact]
        [WorkItem(29916, "https://github.com/dotnet/roslyn/issues/29916")]
        public void PassingParameters_UnknownMethod_AffectingConditionalState2()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    void F(object x)
    {
        if (Missing(x) && Missing(x = null))
        {
            x.ToString(); // 1
        }
        else
        {
            x.ToString(); // 2
        }
    }
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (6,13): error CS0103: The name 'Missing' does not exist in the current context
                //         if (Missing(x) && Missing(x = null))
                Diagnostic(ErrorCode.ERR_NameNotInContext, "Missing").WithArguments("Missing").WithLocation(6, 13),
                // (6,27): error CS0103: The name 'Missing' does not exist in the current context
                //         if (Missing(x) && Missing(x = null))
                Diagnostic(ErrorCode.ERR_NameNotInContext, "Missing").WithArguments("Missing").WithLocation(6, 27),
                // (6,39): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         if (Missing(x) && Missing(x = null))
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "null").WithLocation(6, 39),
                // (8,13): warning CS8602: Dereference of a possibly null reference.
                //             x.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x").WithLocation(8, 13),
                // (12,13): warning CS8602: Dereference of a possibly null reference.
                //             x.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x").WithLocation(12, 13)
                );
        }

        [Fact]
        public void DuplicateArguments()
        {
            var source =
@"class C
{
    static void F(object x, object? y)
    {
        // Duplicate x
        G(x: x, x: y);
        G(x: y, x: x);
        G(x: x, x: y, y: y);
        G(x: y, x: x, y: y);
        G(y: y, x: x, x: y);
        G(y: y, x: y, x: x);
        // Duplicate y
        G(y: x, y: y);
        G(y: y, y: x);
        G(x, y: x, y: y);
        G(x, y: y, y: x);
        G(y: x, y: y, x: x);
        G(y: y, y: x, x: x);
    }
    static void G(object x, params object?[] y)
    {
    }
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,17): error CS1740: Named argument 'x' cannot be specified multiple times
                //         G(x: x, x: y);
                Diagnostic(ErrorCode.ERR_DuplicateNamedArgument, "x").WithArguments("x").WithLocation(6, 17),
                // (7,17): error CS1740: Named argument 'x' cannot be specified multiple times
                //         G(x: y, x: x);
                Diagnostic(ErrorCode.ERR_DuplicateNamedArgument, "x").WithArguments("x").WithLocation(7, 17),
                // (8,17): error CS1740: Named argument 'x' cannot be specified multiple times
                //         G(x: x, x: y, y: y);
                Diagnostic(ErrorCode.ERR_DuplicateNamedArgument, "x").WithArguments("x").WithLocation(8, 17),
                // (9,17): error CS1740: Named argument 'x' cannot be specified multiple times
                //         G(x: y, x: x, y: y);
                Diagnostic(ErrorCode.ERR_DuplicateNamedArgument, "x").WithArguments("x").WithLocation(9, 17),
                // (10,23): error CS1740: Named argument 'x' cannot be specified multiple times
                //         G(y: y, x: x, x: y);
                Diagnostic(ErrorCode.ERR_DuplicateNamedArgument, "x").WithArguments("x").WithLocation(10, 23),
                // (11,23): error CS1740: Named argument 'x' cannot be specified multiple times
                //         G(y: y, x: y, x: x);
                Diagnostic(ErrorCode.ERR_DuplicateNamedArgument, "x").WithArguments("x").WithLocation(11, 23),
                // (13,9): error CS7036: There is no argument given that corresponds to the required parameter 'x' of 'C.G(object, params object?[])'
                //         G(y: x, y: y);
                Diagnostic(ErrorCode.ERR_NoCorrespondingArgument, "G").WithArguments("x", "C.G(object, params object?[])").WithLocation(13, 9),
                // (14,9): error CS7036: There is no argument given that corresponds to the required parameter 'x' of 'C.G(object, params object?[])'
                //         G(y: y, y: x);
                Diagnostic(ErrorCode.ERR_NoCorrespondingArgument, "G").WithArguments("x", "C.G(object, params object?[])").WithLocation(14, 9),
                // (15,20): error CS1740: Named argument 'y' cannot be specified multiple times
                //         G(x, y: x, y: y);
                Diagnostic(ErrorCode.ERR_DuplicateNamedArgument, "y").WithArguments("y").WithLocation(15, 20),
                // (16,20): error CS1740: Named argument 'y' cannot be specified multiple times
                //         G(x, y: y, y: x);
                Diagnostic(ErrorCode.ERR_DuplicateNamedArgument, "y").WithArguments("y").WithLocation(16, 20),
                // (17,17): error CS1740: Named argument 'y' cannot be specified multiple times
                //         G(y: x, y: y, x: x);
                Diagnostic(ErrorCode.ERR_DuplicateNamedArgument, "y").WithArguments("y").WithLocation(17, 17),
                // (18,17): error CS1740: Named argument 'y' cannot be specified multiple times
                //         G(y: y, y: x, x: x);
                Diagnostic(ErrorCode.ERR_DuplicateNamedArgument, "y").WithArguments("y").WithLocation(18, 17));
        }

        [Fact]
        public void MissingArguments()
        {
            var source =
@"class C
{
    static void F(object? x)
    {
        G(y: x);
    }
    static void G(object? x = null, object y)
    {
    }
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (7,45): error CS1737: Optional parameters must appear after all required parameters
                //     static void G(object? x = null, object y)
                Diagnostic(ErrorCode.ERR_DefaultValueBeforeRequiredValue, ")").WithLocation(7, 45),
                // (5,14): warning CS8604: Possible null reference argument for parameter 'y' in 'void C.G(object? x = null, object y)'.
                //         G(y: x);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "x").WithArguments("y", "void C.G(object? x = null, object y)").WithLocation(5, 14));
        }

        [Fact]
        public void ParamsArgument_NotLast()
        {
            var source =
@"class C
{
    static void F(object[]? a, object? b)
    {
        G(a, b, a);
    }
    static void G(params object[] x, params object[] y)
    {
    }
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (7,19): error CS0231: A params parameter must be the last parameter in a parameter list
                //     static void G(params object[] x, params object[] y)
                Diagnostic(ErrorCode.ERR_ParamsLast, "params object[] x").WithLocation(7, 19),
                // (5,11): warning CS8604: Possible null reference argument for parameter 'x' in 'void C.G(params object[] x, params object[] y)'.
                //         G(a, b, a);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "a").WithArguments("x", "void C.G(params object[] x, params object[] y)").WithLocation(5, 11),
                // (5,14): warning CS8604: Possible null reference argument for parameter 'y' in 'void C.G(params object[] x, params object[] y)'.
                //         G(a, b, a);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "b").WithArguments("y", "void C.G(params object[] x, params object[] y)").WithLocation(5, 14),
                // (5,17): warning CS8604: Possible null reference argument for parameter 'y' in 'void C.G(params object[] x, params object[] y)'.
                //         G(a, b, a);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "a").WithArguments("y", "void C.G(params object[] x, params object[] y)").WithLocation(5, 17));
        }

        [Fact]
        public void ParamsArgument_NotArray()
        {
            var source =
@"class C
{
    static void F1(bool b, object x, object? y)
    {
        if (b) F2(y);
        if (b) F2(x, y);
        if (b) F3(y, x);
        if (b) F3(x, y, x);
    }
    static void F2(params object x)
    {
    }
    static void F3(params object x, params object[] y)
    {
    }
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (10,20): error CS0225: The params parameter must be a single dimensional array
                //     static void F2(params object x)
                Diagnostic(ErrorCode.ERR_ParamsMustBeArray, "params").WithLocation(10, 20),
                // (13,20): error CS0225: The params parameter must be a single dimensional array
                //     static void F3(params object x, params object[] y)
                Diagnostic(ErrorCode.ERR_ParamsMustBeArray, "params").WithLocation(13, 20),
                // (13,20): error CS0231: A params parameter must be the last parameter in a parameter list
                //     static void F3(params object x, params object[] y)
                Diagnostic(ErrorCode.ERR_ParamsLast, "params object x").WithLocation(13, 20),
                // (6,16): error CS1501: No overload for method 'F2' takes 2 arguments
                //         if (b) F2(x, y);
                Diagnostic(ErrorCode.ERR_BadArgCount, "F2").WithArguments("F2", "2").WithLocation(6, 16),
                // (5,19): warning CS8604: Possible null reference argument for parameter 'x' in 'void C.F2(params object x)'.
                //         if (b) F2(y);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "y").WithArguments("x", "void C.F2(params object x)").WithLocation(5, 19),
                // (7,19): warning CS8604: Possible null reference argument for parameter 'x' in 'void C.F3(params object x, params object[] y)'.
                //         if (b) F3(y, x);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "y").WithArguments("x", "void C.F3(params object x, params object[] y)").WithLocation(7, 19),
                // (8,22): warning CS8604: Possible null reference argument for parameter 'y' in 'void C.F3(params object x, params object[] y)'.
                //         if (b) F3(x, y, x);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "y").WithArguments("y", "void C.F3(params object x, params object[] y)").WithLocation(8, 22));
        }

        [Fact]
        public void ParamsArgument_BinaryOperator()
        {
            var source =
@"class C
{
    public static object operator+(C x, params object?[] y) => x;
    static object F(C x, object[] y) => x + y;
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (3,41): error CS1670: params is not valid in this context
                //     public static object operator+(C x, params object?[] y) => x;
                Diagnostic(ErrorCode.ERR_IllegalParams, "params").WithLocation(3, 41));
        }

        [Fact]
        public void RefOutParameters_01()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static void Main()
    {
    }

    void Test1(ref CL1 x1, CL1 y1)
    {
        y1 = x1;
    }

    void Test2(ref CL1? x2, CL1 y2)
    {
        y2 = x2;
    }

    void Test3(ref CL1? x3, CL1 y3)
    {
        x3 = y3;
        y3 = x3;
    }

    void Test4(out CL1 x4, CL1 y4)
    {
        y4 = x4;
        x4 = y4;
    }

    void Test5(out CL1? x5, CL1 y5)
    {
        y5 = x5;
        x5 = y5;
    }

    void Test6(out CL1? x6, CL1 y6)
    {
        x6 = y6;
        y6 = x6;
    }
}

class CL1
{
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (15,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         y2 = x2;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x2").WithLocation(15, 14),
                // (26,14): error CS0269: Use of unassigned out parameter 'x4'
                //         y4 = x4;
                Diagnostic(ErrorCode.ERR_UseDefViolationOut, "x4").WithArguments("x4").WithLocation(26, 14),
                // (32,14): error CS0269: Use of unassigned out parameter 'x5'
                //         y5 = x5;
                Diagnostic(ErrorCode.ERR_UseDefViolationOut, "x5").WithArguments("x5").WithLocation(32, 14),
                // (32,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         y5 = x5;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x5").WithLocation(32, 14));
        }

        [Fact]
        public void RefOutParameters_02()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static void Main()
    {
    }

    void Test1(ref S1 x1, CL1 y1)
    {
        y1 = x1.F1;
    }

    void Test2(ref S1 x2, CL1 y2)
    {
        y2 = x2.F2;
    }

    void Test3(ref S1 x3, CL1 y3)
    {
        x3.F2 = y3;
        y3 = x3.F2;
    }

    void Test4(out S1 x4, CL1 y4)
    {
        y4 = x4.F1;
        x4.F1 = y4;
        x4.F2 = y4;
    }

    void Test5(out S1 x5, CL1 y5)
    {
        y5 = x5.F2;
        x5.F1 = y5;
        x5.F2 = y5;
    }

    void Test6(out S1 x6, CL1 y6)
    {
        x6.F1 = y6;
        x6.F2 = y6;
        y6 = x6.F2;
    }
}

class CL1
{
}

struct S1
{
    public CL1 F1;
    public CL1? F2;
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (15,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         y2 = x2.F2;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x2.F2").WithLocation(15, 14),
                // (26,14): error CS0170: Use of possibly unassigned field 'F1'
                //         y4 = x4.F1;
                Diagnostic(ErrorCode.ERR_UseDefViolationField, "x4.F1").WithArguments("F1").WithLocation(26, 14),
                // (33,14): error CS0170: Use of possibly unassigned field 'F2'
                //         y5 = x5.F2;
                Diagnostic(ErrorCode.ERR_UseDefViolationField, "x5.F2").WithArguments("F2").WithLocation(33, 14),
                // (33,14): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         y5 = x5.F2;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x5.F2").WithLocation(33, 14),
                // (34,17): warning CS8601: Possible null reference assignment.
                //         x5.F1 = y5;
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "y5").WithLocation(34, 17));
        }

        [Fact]
        public void RefOutParameters_03()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static void Main()
    {
    }

    void Test3(ref S1 x3, CL1 y3)
    {
        S1 z3;
        z3.F1 = y3;
        z3.F2 = y3;
        x3 = z3;
        y3 = x3.F2;
    }

    void Test6(out S1 x6, CL1 y6)
    {
        S1 z6;
        z6.F1 = y6;
        z6.F2 = y6;
        x6 = z6;
        y6 = x6.F2;
    }
}

class CL1
{
}

struct S1
{
    public CL1 F1;
    public CL1? F2;
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics();
        }

        [Fact]
        public void RefOutParameters_04()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static void Main()
    {
    }

    void M1(ref CL0<string> x) {}

    void Test1(CL0<string?> x1)
    {
        M1(ref x1);
    }

    void M2(out CL0<string?> x) { throw new System.NotImplementedException(); }

    void Test2(CL0<string> x2)
    {
        M2(out x2);
    }

    void M3(CL0<string> x) {}

    void Test3(CL0<string?> x3)
    {
        M3(x3);
    }
}

class CL0<T>
{
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (12,16): warning CS8620: Argument of type 'CL0<string?>' cannot be used as an input of type 'CL0<string>' for parameter 'x' in 'void C.M1(ref CL0<string> x)' due to differences in the nullability of reference types.
                //         M1(ref x1);
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "x1").WithArguments("CL0<string?>", "CL0<string>", "x", "void C.M1(ref CL0<string> x)").WithLocation(12, 16),
                // (19,16): warning CS8624: Argument of type 'CL0<string>' cannot be used as an output of type 'CL0<string?>' for parameter 'x' in 'void C.M2(out CL0<string?> x)' due to differences in the nullability of reference types.
                //         M2(out x2);
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgumentForOutput, "x2").WithArguments("CL0<string>", "CL0<string?>", "x", "void C.M2(out CL0<string?> x)").WithLocation(19, 16),
                // (26,12): warning CS8620: Argument of type 'CL0<string?>' cannot be used as an input of type 'CL0<string>' for parameter 'x' in 'void C.M3(CL0<string> x)' due to differences in the nullability of reference types.
                //         M3(x3);
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInArgument, "x3").WithArguments("CL0<string?>", "CL0<string>", "x", "void C.M3(CL0<string> x)").WithLocation(26, 12));
        }

        [Fact]
        public void RefOutParameters_05()
        {
            var source =
@"class C
{
    static void F(object? x, object? y, object? z)
    {
        G(out x, ref y, in z);
        x.ToString();
        y.ToString();
        z.ToString();
    }
    static void G(out object x, ref object y, in object z)
    {
        x = new object();
    }
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (5,22): warning CS8601: Possible null reference assignment.
                //         G(out x, ref y, in z);
                Diagnostic(ErrorCode.WRN_NullReferenceAssignment, "y").WithLocation(5, 22),
                // (5,28): warning CS8604: Possible null reference argument for parameter 'z' in 'void C.G(out object x, ref object y, in object z)'.
                //         G(out x, ref y, in z);
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "z").WithArguments("z", "void C.G(out object x, ref object y, in object z)").WithLocation(5, 28)
                );
        }

        [Fact]
        public void RefOutParameters_06()
        {
            var source =
@"class C
{
    static void F(object x, object y, object z)
    {
        G(out x, ref y, in z);
        x.ToString();
        y.ToString();
        z.ToString();
    }
    static void G(out object? x, ref object? y, in object? z)
    {
        x = new object();
    }
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (5,15): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         G(out x, ref y, in z);
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x").WithLocation(5, 15),
                // (5,22): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         G(out x, ref y, in z);
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "y").WithLocation(5, 22),
                // (6,9): warning CS8602: Dereference of a possibly null reference.
                //         x.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "x").WithLocation(6, 9),
                // (7,9): warning CS8602: Dereference of a possibly null reference.
                //         y.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "y").WithLocation(7, 9));
        }

        [Fact]
        public void TargetingUnannotatedAPIs_01()
        {
            CSharpCompilation c0 = CreateCompilation(@"
public class CL0 
{
    public object F1;

    public object P1 { get; set;}

    public object this[object x]
    {
        get { return null; }
        set { }
    }

    public S1 M1() { return new S1(); }
}

public struct S1
{
    public CL0 F1;
}
", options: TestOptions.DebugDll, parseOptions: TestOptions.Regular7);

            CSharpCompilation c = CreateCompilation(new[] { @"
class C 
{
    static void Main()
    {
    }

    bool Test1(string? x1, string y1)
    {
        return string.Equals(x1, y1);
    }

    object Test2(ref object? x2, object? y2)
    {
        System.Threading.Interlocked.Exchange(ref x2, y2);
        return x2 ?? new object(); 
    }    

    object Test3(ref object? x3, object? y3)
    {
        return System.Threading.Interlocked.Exchange(ref x3, y3) ?? new object(); 
    }    

    object Test4(System.Delegate x4)
    {
        return x4.Target ?? new object(); 
    }    

    object Test5(CL0 x5)
    {
        return x5.F1 ?? new object(); 
    }    

    void Test6(CL0 x6, object? y6)
    {
        x6.F1 = y6;
    }    

    void Test7(CL0 x7, object? y7)
    {
        x7.P1 = y7;
    }    

    void Test8(CL0 x8, object? y8, object? z8)
    {
        x8[y8] = z8;
    }    

    object Test9(CL0 x9)
    {
        return x9[1] ?? new object(); 
    }    

    object Test10(CL0 x10)
    {
        return x10.M1().F1 ?? new object(); 
    }    

    object Test11(CL0 x11, CL0? z11)
    {
        S1 y11 = x11.M1();
        y11.F1 = z11;
        return y11.F1; 
    }    

    object Test12(CL0 x12)
    {
        S1 y12 = x12.M1();
        y12.F1 = x12;
        return y12.F1 ?? new object(); 
    }    

    void Test13(CL0 x13, object? y13)
    {
        y13 = x13.F1;
        object z13 = y13;
        z13 = y13 ?? new object();
    }    

    void Test14(CL0 x14)
    {
        object? y14 = x14.F1;
        object z14 = y14;
        z14 = y14 ?? new object();
    }    

    void Test15(CL0 x15)
    {
        S2 y15;
        y15.F2 = x15.F1;
        object z15 = y15.F2;
        z15 = y15.F2 ?? new object();
    }    

    struct Test16
    {
        object? y16 {get;}

        public Test16(CL0 x16)
        {
            y16 = x16.F1;
            object z16 = y16;
            z16 = y16 ?? new object();
        }    
    }

    void Test17(CL0 x17)
    {
        var y17 = new { F2 = x17.F1 };
        object z17 = y17.F2;
        z17 = y17.F2 ?? new object();
    }    
}

public struct S2
{
    public object? F2;
}
" }, options: WithNullableEnable(), references: new[] { c0.EmitToImageReference() });

            c.VerifyDiagnostics(
                // (63,16): warning CS8603: Possible null reference return.
                //         return y11.F1;
                Diagnostic(ErrorCode.WRN_NullReferenceReturn, "y11.F1").WithLocation(63, 16));
        }

        [Fact]
        public void TargetingUnannotatedAPIs_02()
        {
            CSharpCompilation c0 = CreateCompilation(@"
public class CL0 
{
    public static object M1() { return new object(); }
}
", options: TestOptions.DebugDll, parseOptions: TestOptions.Regular7);

            CSharpCompilation c = CreateCompilation(new[] { @"
class C 
{
    static void Main()
    {
    }

    public static object? M2() { return null; }
    public static object M3() { return new object(); }

    void Test1()
    {
        object? x1 = CL0.M1() ?? M2();
        object y1 = x1;
        object z1 = x1 ?? new object();
    }

    void Test2()
    {
        object? x2 = CL0.M1() ?? M3();
        object z2 = x2 ?? new object();
    }

    void Test3()
    {
        object? x3 = M3() ?? M2();
        object z3 = x3 ?? new object();
    }

    void Test4()
    {
        object? x4 = CL0.M1() ?? CL0.M1();
        object y4 = x4;
        object z4 = x4 ?? new object();
    }

    void Test5()
    {
        object x5 = M2() ?? M2();
    }

    void Test6()
    {
        object? x6 = M3() ?? M3();
        object z6 = x6 ?? new object();
    }
}
" }, options: WithNullableEnable(), references: new[] { c0.EmitToImageReference() });

            c.VerifyDiagnostics(
                // (14,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         object y1 = x1;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x1").WithLocation(14, 21),
                // (39,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         object x5 = M2() ?? M2();
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "M2() ?? M2()").WithLocation(39, 21));
        }

        [Fact]
        public void TargetingUnannotatedAPIs_03()
        {
            CSharpCompilation c0 = CreateCompilation(@"
public class CL0 
{
    public static object M1() { return new object(); }
}
", options: TestOptions.DebugDll, parseOptions: TestOptions.Regular7);

            CSharpCompilation c = CreateCompilation(new[] { @"
class C 
{
    static void Main()
    {
    }

    public static object? M2() { return null; }
    public static object M3() { return new object(); }

    void Test1()
    {
        object? x1 = M2() ?? CL0.M1();
        object y1 = x1;
        object z1 = x1 ?? new object();
    }

    void Test2()
    {
        object? x2 = M3() ?? CL0.M1();
        object z2 = x2 ?? new object();
    }

    void Test3()
    {
        object? x3 = M2() ?? M3();
        object z3 = x3 ?? new object();
    }
}
" }, options: WithNullableEnable(), references: new[] { c0.EmitToImageReference() });

            c.VerifyDiagnostics(
                );
        }

        [Fact]
        public void TargetingUnannotatedAPIs_04()
        {
            CSharpCompilation c0 = CreateCompilation(@"
public class CL0 
{
    public static object M1() { return new object(); }
}
", options: TestOptions.DebugDll, parseOptions: TestOptions.Regular7);

            CSharpCompilation c = CreateCompilation(new[] { @"
class C 
{
    static void Main()
    {
    }

    public static object? M2() { return null; }
    public static object M3() { return new object(); }
    public static bool M4() {return false;}

    void Test1()
    {
        object x1 = M4() ? CL0.M1() : M2();
    }

    void Test2()
    {
        object? x2 = M4() ? CL0.M1() : M3();
        object y2 = x2;
        object z2 = x2 ?? new object();
    }

    void Test3()
    {
        object x3 =  M4() ? M3() : M2();
    }

    void Test4()
    {
        object? x4 =  M4() ? CL0.M1() : CL0.M1();
        object y4 = x4;
        object z4 = x4 ?? new object();
    }

    void Test5()
    {
        object x5 =  M4() ? M2() : M2();
    }

    void Test6()
    {
        object? x6 =  M4() ? M3() : M3();
        object z6 = x6 ?? new object();
    }
}
" }, options: WithNullableEnable(), references: new[] { c0.EmitToImageReference() });

            c.VerifyDiagnostics(
                // (14,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         object x1 = M4() ? CL0.M1() : M2();
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "M4() ? CL0.M1() : M2()").WithLocation(14, 21),
                // (26,22): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         object x3 =  M4() ? M3() : M2();
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "M4() ? M3() : M2()").WithLocation(26, 22),
                // (38,22): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         object x5 =  M4() ? M2() : M2();
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "M4() ? M2() : M2()").WithLocation(38, 22)
                );
        }

        [Fact]
        public void TargetingUnannotatedAPIs_05()
        {
            CSharpCompilation c0 = CreateCompilation(@"
public class CL0 
{
    public static object M1() { return new object(); }
}
", options: TestOptions.DebugDll, parseOptions: TestOptions.Regular7);

            CSharpCompilation c = CreateCompilation(new[] { @"
class C 
{
    static void Main()
    {
    }

    public static object? M2() { return null; }
    public static object M3() { return new object(); }
    public static bool M4() {return false;}

    void Test1()
    {
        object x1 = M4() ? M2() : CL0.M1();
    }

    void Test2()
    {
        object? x2 = M4() ? M3() : CL0.M1();
        object y2 = x2;
        object z2 = x2 ?? new object();
    }

    void Test3()
    {
        object x3 =  M4() ? M2() : M3();
    }
}
" }, options: WithNullableEnable(), references: new[] { c0.EmitToImageReference() });

            c.VerifyDiagnostics(
                 // (14,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                 //         object x1 = M4() ? M2() : CL0.M1();
                 Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "M4() ? M2() : CL0.M1()").WithLocation(14, 21),
                 // (26,22): warning CS8600: Converting null literal or possible null value to non-nullable type.
                 //         object x3 =  M4() ? M2() : M3();
                 Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "M4() ? M2() : M3()").WithLocation(26, 22)
                );
        }

        [Fact]
        public void TargetingUnannotatedAPIs_06()
        {
            CSharpCompilation c0 = CreateCompilation(@"
public class CL0 
{
    public static object M1() { return new object(); }
}
", options: TestOptions.DebugDll, parseOptions: TestOptions.Regular7);

            CSharpCompilation c = CreateCompilation(new[] { @"
class C 
{
    static void Main()
    {
    }

    public static object? M2() { return null; }
    public static object M3() { return new object(); }
    public static bool M4() {return false;}

    void Test1()
    {
        object? x1;
        if (M4()) x1 = CL0.M1(); else x1 = M2();
        object y1 = x1;
    }

    void Test2()
    {
        object? x2;
        if (M4()) x2 = CL0.M1(); else x2 = M3();
        object y2 = x2;
        object z2 = x2 ?? new object();
    }

    void Test3()
    {
        object? x3;
        if (M4()) x3 = M3(); else x3 = M2();
        object y3 = x3;
    }

    void Test4()
    {
        object? x4;
        if (M4()) x4 = CL0.M1(); else x4 = CL0.M1();
        object y4 = x4;
        object z4 = x4 ?? new object();
    }

    void Test5()
    {
        object? x5;
        if (M4()) x5 = M2(); else x5 = M2();
        object y5 = x5;
    }

    void Test6()
    {
        object? x6;
        if (M4()) x6 = M3(); else x6 = M3();
        object z6 = x6 ?? new object();
    }
}
" }, options: WithNullableEnable(), references: new[] { c0.EmitToImageReference() });

            c.VerifyDiagnostics(
                // (16,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         object y1 = x1;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x1").WithLocation(16, 21),
                // (31,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         object y3 = x3;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x3").WithLocation(31, 21),
                // (46,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         object y5 = x5;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x5").WithLocation(46, 21)
                );
        }

        [Fact]
        public void TargetingUnannotatedAPIs_07()
        {
            CSharpCompilation c0 = CreateCompilation(@"
public class CL0 
{
    public static object M1() { return new object(); }
}
", options: TestOptions.DebugDll, parseOptions: TestOptions.Regular7);

            CSharpCompilation c = CreateCompilation(new[] { @"
class C 
{
    static void Main()
    {
    }

    public static object? M2() { return null; }
    public static object M3() { return new object(); }
    public static bool M4() {return false;}

    void Test1()
    {
        object? x1;
        if (M4()) x1 = M2(); else x1 = CL0.M1();
        object y1 = x1;
    }

    void Test2()
    {
        object? x2;
        if (M4()) x2 = M3(); else x2 = CL0.M1();
        object y2 = x2;
        object z2 = x2 ?? new object();
    }

    void Test3()
    {
        object? x3;
        if (M4()) x3 = M2(); else x3 = M3();
        object y3 = x3;
    }
}
" }, options: WithNullableEnable(), references: new[] { c0.EmitToImageReference() });

            c.VerifyDiagnostics(
                // (16,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         object y1 = x1;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x1").WithLocation(16, 21),
                // (31,21): warning CS8600: Converting null literal or possible null value to non-nullable type.
                //         object y3 = x3;
                Diagnostic(ErrorCode.WRN_ConvertingNullableToNonNullable, "x3").WithLocation(31, 21)
                );
        }

        [Fact]
        public void TargetingUnannotatedAPIs_08()
        {
            CSharpCompilation c0 = CreateCompilation(@"
public abstract class A1
{
    public abstract event System.Action E1;
    public abstract string P2 { get; set; }
    public abstract string M3(string x);
    public abstract event System.Action E4;
    public abstract string this[string x] { get; set; }
}

public interface IA2
{
    event System.Action E5;
    string P6 { get; set; }
    string M7(string x);
    event System.Action E8;
    string this[string x] { get; set; }
}
", options: TestOptions.DebugDll, parseOptions: TestOptions.Regular7);

            CSharpCompilation c = CreateCompilation(new[] { @"
class B1 : A1
{
    static void Main()
    {
    }

    public override string? P2 { get; set; }
    public override event System.Action? E1;
    public override string? M3(string? x)
    {
        var dummy = E1;
        throw new System.NotImplementedException();
    }
    public override event System.Action? E4
    {
        add { }
        remove { }
    }

    public override string? this[string? x]
    {
        get
        {
            throw new System.NotImplementedException();
        }

        set
        {
            throw new System.NotImplementedException();
        }
    }
}

class B2 : IA2
{
    public string? P6 { get; set; }
    public event System.Action? E5;
    public event System.Action? E8
    {
        add { }
        remove { }
    }

    public string? M7(string? x)
    {
        var dummy = E5;
        throw new System.NotImplementedException();
    }

    public string? this[string? x]
    {
        get
        {
            throw new System.NotImplementedException();
        }

        set
        {
            throw new System.NotImplementedException();
        }
    }
}

class B3 : IA2
{
    string? IA2.P6 { get; set; }

    event System.Action? IA2.E5
    {
        add { }
        remove { }
    }

    event System.Action? IA2.E8
    {
        add { }
        remove { }
    }

    string? IA2.M7(string? x)
    {
        throw new System.NotImplementedException();
    }
    
    string? IA2.this[string? x]
    {
        get
        {
            throw new System.NotImplementedException();
        }

        set
        {
            throw new System.NotImplementedException();
        }
    }

}
" }, options: WithNullableEnable(), references: new[] { c0.EmitToImageReference() });

            c.VerifyDiagnostics();
        }

        [Fact]
        public void ReturningValues_01()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static void Main()
    {
    }

    CL1 Test1(CL1? x1)
    {
        return x1;
    }

    CL1? Test2(CL1? x2)
    {
        return x2;
    }

    CL1? Test3(CL1 x3)
    {
        return x3;
    }
}

class CL1
{
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                 // (10,16): warning CS8603: Possible null reference return.
                 //         return x1;
                 Diagnostic(ErrorCode.WRN_NullReferenceReturn, "x1").WithLocation(10, 16)
                );
        }

        [Fact]
        public void ReturningValues_02()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    static void Main()
    {
    }

    CL1<string?> Test1(CL1<string> x1)
    {
        return x1;
    }

    CL1<string> Test2(CL1<string?> x2)
    {
        return x2;
    }
}

class CL1<T>
{
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (10,16): warning CS8619: Nullability of reference types in value of type 'CL1<string>' doesn't match target type 'CL1<string?>'.
                //         return x1;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x1").WithArguments("CL1<string>", "CL1<string?>").WithLocation(10, 16),
                // (15,16): warning CS8619: Nullability of reference types in value of type 'CL1<string?>' doesn't match target type 'CL1<string>'.
                //         return x2;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x2").WithArguments("CL1<string?>", "CL1<string>").WithLocation(15, 16)
                );
        }

        [Fact]
        public void ReturningValues_BadValue()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
class C
{
    string M()
    {
        return bad;
    }
}
" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (6,16): error CS0103: The name 'bad' does not exist in the current context
                //         return bad;
                Diagnostic(ErrorCode.ERR_NameNotInContext, "bad").WithArguments("bad").WithLocation(6, 16)
                );
        }

        [Fact]
        public void IdentityConversion_Return_01()
        {
            var source =
@"interface I<T> { }
interface IIn<in T> { }
interface IOut<out T> { }
class C
{
    static I<object?> F(I<object> x) => x;
    static IIn<object?> F(IIn<object> x) => x;
    static IOut<object?> F(IOut<object> x) => x;
    static I<object> G(I<object?> x) => x;
    static IIn<object> G(IIn<object?> x) => x;
    static IOut<object> G(IOut<object?> x) => x;
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (6,41): warning CS8619: Nullability of reference types in value of type 'I<object>' doesn't match target type 'I<object?>'.
                //     static I<object?> F(I<object> x) => x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("I<object>", "I<object?>").WithLocation(6, 41),
                // (7,45): warning CS8619: Nullability of reference types in value of type 'IIn<object>' doesn't match target type 'IIn<object?>'.
                //     static IIn<object?> F(IIn<object> x) => x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("IIn<object>", "IIn<object?>").WithLocation(7, 45),
                // (9,41): warning CS8619: Nullability of reference types in value of type 'I<object?>' doesn't match target type 'I<object>'.
                //     static I<object> G(I<object?> x) => x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("I<object?>", "I<object>").WithLocation(9, 41),
                // (11,47): warning CS8619: Nullability of reference types in value of type 'IOut<object?>' doesn't match target type 'IOut<object>'.
                //     static IOut<object> G(IOut<object?> x) => x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("IOut<object?>", "IOut<object>").WithLocation(11, 47));
        }

        [Fact]
        public void IdentityConversion_Return_02()
        {
            var source =
@"#pragma warning disable 1998
using System.Threading.Tasks;
interface I<T> { }
interface IIn<in T> { }
interface IOut<out T> { }
class C
{
    static async Task<I<object?>> F(I<object> x) => x;
    static async Task<IIn<object?>> F(IIn<object> x) => x;
    static async Task<IOut<object?>> F(IOut<object> x) => x;
    static async Task<I<object>> G(I<object?> x) => x;
    static async Task<IIn<object>> G(IIn<object?> x) => x;
    static async Task<IOut<object>> G(IOut<object?> x) => x;
}";
            var comp = CreateCompilation(new[] { source }, options: WithNullableEnable());
            comp.VerifyDiagnostics(
                // (8,53): warning CS8619: Nullability of reference types in value of type 'I<object>' doesn't match target type 'I<object?>'.
                //     static async Task<I<object?>> F(I<object> x) => x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("I<object>", "I<object?>").WithLocation(8, 53),
                // (9,57): warning CS8619: Nullability of reference types in value of type 'IIn<object>' doesn't match target type 'IIn<object?>'.
                //     static async Task<IIn<object?>> F(IIn<object> x) => x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("IIn<object>", "IIn<object?>").WithLocation(9, 57),
                // (11,53): warning CS8619: Nullability of reference types in value of type 'I<object?>' doesn't match target type 'I<object>'.
                //     static async Task<I<object>> G(I<object?> x) => x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("I<object?>", "I<object>").WithLocation(11, 53),
                // (13,59): warning CS8619: Nullability of reference types in value of type 'IOut<object?>' doesn't match target type 'IOut<object>'.
                //     static async Task<IOut<object>> G(IOut<object?> x) => x;
                Diagnostic(ErrorCode.WRN_NullabilityMismatchInAssignment, "x").WithArguments("IOut<object?>", "IOut<object>").WithLocation(13, 59));
        }

        [Fact]
        public void MakeMethodKeyForWhereMethod()
        {
            // this test verifies that a bad method symbol doesn't crash when generating a key for external annotations
            CSharpCompilation c = CreateCompilation(new[] { @"
class Test
{
    public void SimpleWhere()
    {
        int[] numbers = { 1, 2, 3 };
        var lowNums = from n in numbers
                      where n < 5
                      select n;
    }
}" }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (7,33): error CS1935: Could not find an implementation of the query pattern for source type 'int[]'.  'Where' not found.  Are you missing required assembly references or a using directive for 'System.Linq'?
                //         var lowNums = from n in numbers
                Diagnostic(ErrorCode.ERR_QueryNoProviderStandard, "numbers").WithArguments("int[]", "Where").WithLocation(7, 33)
                );
        }

        [Fact]
        public void MemberNotNull_Simple()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        Init();
        field1.ToString();
        field2.ToString();
        field3.ToString(); // 1
        field4.ToString(); // 2
    }

    [MemberNotNull(nameof(field1), nameof(field2))]
    void Init() => throw null!;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(15, 9),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(16, 9)
                );
        }

        [Fact, WorkItem(32671, "https://github.com/dotnet/roslyn/issues/32671")]
        public void NotNullWhenTrue_UserDefinedOperator_Implicit()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
class C
{
  void M(C? c)
  {
    if (c)
    {
      c.ToString(); // ok
    }
    else
    {
      c.ToString(); // 1
    }
  }

  void M2(C? c)
  {
    bool b = c;
    if (b)
    {
      c.ToString(); // 2 (not tracked through assignment)
    }
  }

  public static implicit operator bool([NotNullWhen(true)] C? c) { return true; }
}
", NotNullWhenAttributeDefinition }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (13,7): warning CS8602: Possible dereference of a null reference.
                //       c.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c").WithLocation(13, 7),
                // (22,7): warning CS8602: Possible dereference of a null reference.
                //       c.ToString(); // 2 (not tracked through assignment)
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c").WithLocation(22, 7)
                );

            VerifyAnnotationsAndMetadata(c, "C.op_Implicit", NotNullWhenTrue);
        }

        [Fact, WorkItem(32671, "https://github.com/dotnet/roslyn/issues/32671")]
        public void NotNullWhenTrue_UserDefinedOperator_Explicit()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
class C
{
  void M(C? c)
  {
    if ((bool)c)
    {
      c.ToString(); // ok
    }
    else
    {
      c.ToString(); // 1
    }
  }

  void M2(C? c)
  {
    bool b = (bool)c;
    if (b)
    {
      c.ToString(); // 2 (not tracked through assignment)
    }
  }

  public static explicit operator bool([NotNullWhen(true)] C? c) { return true; }
}
", NotNullWhenAttributeDefinition }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (13,7): warning CS8602: Possible dereference of a null reference.
                //       c.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c").WithLocation(13, 7),
                // (22,7): warning CS8602: Possible dereference of a null reference.
                //       c.ToString(); // 2 (not tracked through assignment)
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c").WithLocation(22, 7)
                );

            VerifyAnnotationsAndMetadata(c, "C.op_Explicit", NotNullWhenTrue);
        }

        [Fact, WorkItem(32671, "https://github.com/dotnet/roslyn/issues/32671")]
        public void NotNullWhenTrue_UserDefinedOperator_Derived()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
class C : B
{
  void M(C? c)
  {
    if (c)
    {
      c.ToString(); // ok
    }
    else
    {
      c.ToString(); // 1
    }
  }
}

public class B
{
  public static implicit operator bool([NotNullWhen(true)] B? b) { return true; }
}
", NotNullWhenAttributeDefinition }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (13,7): warning CS8602: Possible dereference of a null reference.
                //       c.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c").WithLocation(13, 7)
                );

            VerifyAnnotationsAndMetadata(c, "B.op_Implicit", NotNullWhenTrue);
        }

        [Fact, WorkItem(32671, "https://github.com/dotnet/roslyn/issues/32671")]
        public void NotNull_UserDefinedOperator1()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
#pragma warning disable CS0660, CS0661 // no equals, hashcode 
using System.Diagnostics.CodeAnalysis;
public class C
{
    public void Main(C? c, C? c2)
    {
        if(c == c2)
        {
         	c.ToString();
            c2.ToString(); // 1
        }
        else
        {
            c.ToString();
            c2.ToString(); // 2
        }
    }

    public static bool operator ==([NotNull] C? c1, C? c2)
    {
        throw null!;
    }

    public static bool operator !=(C? c1, C? c2)
    {
        return true;
    }
}
", NotNullAttributeDefinition }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (11,13): warning CS8602: Dereference of a possibly null reference.
                //             c2.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c2").WithLocation(11, 13),
                // (16,13): warning CS8602: Dereference of a possibly null reference.
                //             c2.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c2").WithLocation(16, 13)
                );

            VerifyAnnotationsAndMetadata(c, "C.op_Equality", NotNull, None);
        }

        [Fact, WorkItem(32671, "https://github.com/dotnet/roslyn/issues/32671")]
        public void NotNull_UserDefinedOperator2()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
#pragma warning disable CS0660, CS0661 // no equals, hashcode 
using System.Diagnostics.CodeAnalysis;
public class C
{
    public void Main(C? c, C? c2)
    {
        if(c == c2)
        {
         	c.ToString();  // 1
            c2.ToString();
        }
        else
        {
            c.ToString(); // 2
            c2.ToString();
        }
    }

    public static bool operator ==(C? c1, [NotNull]C? c2)
    {
        throw null!;
    }

    public static bool operator !=(C? c1, C? c2)
    {
        return true;
    }
}
", NotNullAttributeDefinition }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (10,11): warning CS8602: Dereference of a possibly null reference.
                //          	c.ToString();  // 1 
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c").WithLocation(10, 11),
                // (15,13): warning CS8602: Dereference of a possibly null reference.
                //             c.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c").WithLocation(15, 13)
                );

            VerifyAnnotationsAndMetadata(c, "C.op_Equality", None, NotNull);
        }

        [Fact, WorkItem(32671, "https://github.com/dotnet/roslyn/issues/32671")]
        public void NotNullWhen_UserDefinedConversion_Completeness()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;

public class C
{
    public bool Field;

    public static bool M0(bool b = new C()) // 1
    {
        _ = new C() { Field = new C() };
        _ = new bool[] { new C(), true };
        _ = new [] { new C(), true };

        return new C();
        return true; // 2
    }

    public static IEnumerable<bool> Iterator()
    {
        yield return new C();
        yield return true;
    }

    public static implicit operator bool([NotNullWhen(true)] C? c) => c != null;
}
", NotNullWhenAttributeDefinition }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (9,36): error CS1736: Default parameter value for 'b' must be a compile-time constant
                //     public static bool M0(bool b = new C()) // 1
                Diagnostic(ErrorCode.ERR_DefaultValueMustBeConstant, "new C()").WithArguments("b").WithLocation(9, 36),
                // (16,9): warning CS0162: Unreachable code detected
                //         return true; // 2
                Diagnostic(ErrorCode.WRN_UnreachableCode, "return").WithLocation(16, 9)
                );
        }

        [Fact, WorkItem(32671, "https://github.com/dotnet/roslyn/issues/32671")]
        public void DisallowNull_UserDefinedOperator_01()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
#pragma warning disable CS0660, CS0661 // no equals, hashcode 
using System.Diagnostics.CodeAnalysis;

public class C
{
    public static void M0(C? c)
    {
        bool b = c; // 1
        b = new C();
    }

    public static void M0(C? c1, C? c2)
    {
        bool b = c1 == c2; // 2
        b = c2 != c1; // 3
        b = new C() == new C();
    }

    public static implicit operator bool([DisallowNull] C? c) => c != null;

    public static bool operator ==([DisallowNull] C? c1, C? c2) => (object?)c1 == c2;
    public static bool operator !=([DisallowNull] C? c1, C? c2) => (object?)c1 != c2;
}
", DisallowNullAttributeDefinition }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (9,18): warning CS8604: Possible null reference argument for parameter 'c' in 'C.implicit operator bool(C? c)'.
                //         bool b = c; // 1
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "c").WithArguments("c", "C.implicit operator bool(C? c)").WithLocation(9, 18),
                // (15,18): warning CS8604: Possible null reference argument for parameter 'c1' in 'bool C.operator ==(C? c1, C? c2)'.
                //         bool b = c1 == c2; // 2
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "c1").WithArguments("c1", "bool C.operator ==(C? c1, C? c2)").WithLocation(15, 18),
                // (16,13): warning CS8604: Possible null reference argument for parameter 'c1' in 'bool C.operator !=(C? c1, C? c2)'.
                //         b = c2 != c1; // 3
                Diagnostic(ErrorCode.WRN_NullReferenceArgument, "c2").WithArguments("c1", "bool C.operator !=(C? c1, C? c2)").WithLocation(16, 13)
                );
        }

        [Fact, WorkItem(32671, "https://github.com/dotnet/roslyn/issues/32671")]
        public void DisallowNull_UserDefinedOperator_02()
        {
            CSharpCompilation c = CreateCompilation(new[] { @"
#pragma warning disable CS0660, CS0661 // no equals, hashcode 
using System.Diagnostics.CodeAnalysis;

public struct S
{
    public static void M0(S? s)
    {
        bool? b = s; // 1
        b = new S();
    }

    public static void M0(S? s1, S? s2)
    {
        bool b = s1 == s2; // 2
        b = s2 != s1; // 3
        b = new S() == new S();
    }

    public static implicit operator bool([DisallowNull] S? s) => s != null;

    public static bool operator ==([DisallowNull] S? s1, S? s2) => throw null!;
    public static bool operator !=([DisallowNull] S? s1, S? s2) => throw null!;
}
", DisallowNullAttributeDefinition }, options: WithNullableEnable());

            c.VerifyDiagnostics(
                // (9,19): warning CS8607: A possible null value may not be used for a type marked with [NotNull] or [DisallowNull]
                //         bool? b = s; // 1
                Diagnostic(ErrorCode.WRN_DisallowNullAttributeForbidsMaybeNullAssignment, "s").WithLocation(9, 19),
                // (15,18): warning CS8607: A possible null value may not be used for a type marked with [NotNull] or [DisallowNull]
                //         bool b = s1 == s2; // 2
                Diagnostic(ErrorCode.WRN_DisallowNullAttributeForbidsMaybeNullAssignment, "s1").WithLocation(15, 18),
                // (16,13): warning CS8607: A possible null value may not be used for a type marked with [NotNull] or [DisallowNull]
                //         b = s2 != s1; // 3
                Diagnostic(ErrorCode.WRN_DisallowNullAttributeForbidsMaybeNullAssignment, "s2").WithLocation(16, 13)
                );
        }

        [Fact, WorkItem(58598, "https://github.com/dotnet/roslyn/issues/58598")]
        public void MemberNotNull_NullConstant()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    [MemberNotNull((string[]?)null)]
    public void Init() => throw null!;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            CompileAndVerify(c, symbolValidator: module =>
            {
                var c = module.ContainingAssembly.GetTypeByMetadataName("C");
                var init = c.GetMember<MethodSymbol>("Init");
                Assert.Empty(init.NotNullMembers);
                Assert.Empty(init.NotNullWhenFalseMembers);
                Assert.Empty(init.NotNullWhenTrueMembers);
            });
        }

        [Fact, WorkItem(58598, "https://github.com/dotnet/roslyn/issues/58598")]
        public void MemberNotNullWhen_NullConstant()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    [MemberNotNullWhen(true, (string[]?)null)]
    public bool Init() => throw null!;
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            CompileAndVerify(c, symbolValidator: module =>
            {
                var c = module.ContainingAssembly.GetTypeByMetadataName("C");
                var init = c.GetMember<MethodSymbol>("Init");
                Assert.Empty(init.NotNullMembers);
                Assert.Empty(init.NotNullWhenFalseMembers);
                Assert.Empty(init.NotNullWhenTrueMembers);
            });
        }

        [Fact]
        public void MemberNotNull_NullValue()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        Init();
        field1.ToString(); // 1
        field2.ToString(); // 2
        field3.ToString(); // 3
        field4.ToString(); // 4
    }

    [MemberNotNull(null!, null!)]
    [MemberNotNull(members: null!)]
    [MemberNotNull(member: null!)]
    void Init() => throw null!;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (13,9): warning CS8602: Dereference of a possibly null reference.
                //         field1.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field1").WithLocation(13, 9),
                // (14,9): warning CS8602: Dereference of a possibly null reference.
                //         field2.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(14, 9),
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(15, 9),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(16, 9)
                );
        }

        [Fact]
        public void MemberNotNull_LocalFunction()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        init();
        field1.ToString(); // 1
        field2.ToString(); // 2
        field3.ToString(); // 3
        field4.ToString(); // 4

        [MemberNotNull(nameof(field1), nameof(field2))]
        void init() => throw null!;
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            // Note: the local function is not invoked on this or base
            c.VerifyDiagnostics(
                // (13,9): warning CS8602: Dereference of a possibly null reference.
                //         field1.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field1").WithLocation(13, 9),
                // (14,9): warning CS8602: Dereference of a possibly null reference.
                //         field2.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(14, 9),
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(15, 9),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(16, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Interfaces()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public interface I
{
    string Property1 { get; }
    string? Property2 { get; }
    string Property3 { get; }
    string? Property4 { get; }

    [MemberNotNull(nameof(Property1), nameof(Property2))] // Note: attribute ineffective
    void Init();
}
public class C : I
{
    public string Property1 { get; set; }
    public string? Property2 { get; set; }
    public string Property3 { get; set; }
    public string? Property4 { get; set; }

    public void M()
    {
        Init();
        Property1.ToString();
        Property2.ToString(); // 1
        Property3.ToString();
        Property4.ToString(); // 2
    }

    public void Init() => throw null!;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (15,19): warning CS8618: Non-nullable property 'Property1' is uninitialized. Consider declaring the property as nullable.
                //     public string Property1 { get; set; }
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Property1").WithArguments("property", "Property1").WithLocation(15, 19),
                // (17,19): warning CS8618: Non-nullable property 'Property3' is uninitialized. Consider declaring the property as nullable.
                //     public string Property3 { get; set; }
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Property3").WithArguments("property", "Property3").WithLocation(17, 19),
                // (24,9): warning CS8602: Dereference of a possibly null reference.
                //         Property2.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "Property2").WithLocation(24, 9),
                // (26,9): warning CS8602: Dereference of a possibly null reference.
                //         Property4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "Property4").WithLocation(26, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Inheritance()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string? field0;
    public string field1 = null!;
    public string? field2;
    public string field3 = null!;
    public string? field4;

    [MemberNotNull(nameof(field0), nameof(field1), nameof(field2))]
    public virtual void Init() => throw null!;
}
public class Derived : Base
{
    public new string? field0;
    public string field5 = null!;
    public string? field6;
    public string field7 = null!;
    public string? field8;

    [MemberNotNull(nameof(field5), nameof(field6))]
    public override void Init()
    {
        field5 = """";
        field6 = """";
    } // 1, 2, 3

    Derived()
    {
        Init();
        field0.ToString(); // 4
        field1.ToString();
        field2.ToString();
        field3.ToString();
        field4.ToString(); // 5
        field5.ToString();
        field6.ToString();
        field7.ToString();
        field8.ToString(); // 6
    }
}
public class Derived2 : Base
{
    public new string? field0;
    public string field5 = null!;
    public string? field6;
    public string field7 = null!;
    public string? field8;

    [MemberNotNull(nameof(field5), nameof(field6))]
    public override void Init()
    {
        base.Init();
        field5 = """";
        field6 = """";
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (27,5): warning CS8771: Member 'field0' must have a non-null value when exiting.
                //     } // 1, 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field0").WithLocation(27, 5),
                // (27,5): warning CS8771: Member 'field1' must have a non-null value when exiting.
                //     } // 1, 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(27, 5),
                // (27,5): warning CS8771: Member 'field2' must have a non-null value when exiting.
                //     } // 1, 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field2").WithLocation(27, 5),
                // (32,9): warning CS8602: Dereference of a possibly null reference.
                //         field0.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field0").WithLocation(32, 9),
                // (36,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(36, 9),
                // (40,9): warning CS8602: Dereference of a possibly null reference.
                //         field8.ToString(); // 6
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(40, 9)
                );
        }

        [Fact]
        public void MemberNotNull_OnInstance()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string? field0;
    public string field1 = null!;
    public string? field2;

    [MemberNotNull(nameof(field1), nameof(field2))]
    public virtual void Init() => throw null!;
}
public class Derived : Base
{
    public new string? field0;
    public string field3 = null!;
    public string? field4;

    [MemberNotNull(nameof(field0), nameof(field3), nameof(field4))]
    public override void Init() => throw null!;
}
public class C 
{
    void M(Derived d, Derived d2)
    {
        d.field0.ToString(); // 1
        d.field1.ToString();
        d.field2.ToString(); // 2
        d.field3.ToString();
        d.field4.ToString(); // 3

        d2.Init();
        d2.field0.ToString();
        d2.field1.ToString();
        d2.field2.ToString();
        d2.field3.ToString();
        d2.field4.ToString();
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (25,9): warning CS8602: Dereference of a possibly null reference.
                //         d.field0.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "d.field0").WithLocation(25, 9),
                // (27,9): warning CS8602: Dereference of a possibly null reference.
                //         d.field2.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "d.field2").WithLocation(27, 9),
                // (29,9): warning CS8602: Dereference of a possibly null reference.
                //         d.field4.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "d.field4").WithLocation(29, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Property_OnInstance()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string? field0;
    public string field1 = null!;
    public string? field2;

    [MemberNotNull(nameof(field1), nameof(field2))]
    public virtual int Init => throw null!;
}
public class Derived : Base
{
    public new string? field0;
    public string field3 = null!;
    public string? field4;

    [MemberNotNull(nameof(field0), nameof(field3), nameof(field4))]
    public override int Init => throw null!;
}
public class C 
{
    void M(Derived d, Derived d2)
    {
        d.field0.ToString(); // 1
        d.field1.ToString();
        d.field2.ToString(); // 2
        d.field3.ToString();
        d.field4.ToString(); // 3

        _ = d2.Init;
        d2.field0.ToString();
        d2.field1.ToString();
        d2.field2.ToString();
        d2.field3.ToString();
        d2.field4.ToString();
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (25,9): warning CS8602: Dereference of a possibly null reference.
                //         d.field0.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "d.field0").WithLocation(25, 9),
                // (27,9): warning CS8602: Dereference of a possibly null reference.
                //         d.field2.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "d.field2").WithLocation(27, 9),
                // (29,9): warning CS8602: Dereference of a possibly null reference.
                //         d.field4.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "d.field4").WithLocation(29, 9)
                );
        }

        [Fact]
        public void MemberNotNull_OnMethodWithConditionalAttribute()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1 = null!;
    public string? field2;
    public string field3 = null!;
    public string? field4;

    [MemberNotNull(nameof(field1), nameof(field2))]
    public bool Init([NotNullWhen(true)] string p) // NotNullWhen splits the state before we analyze MemberNotNull
    {
        field1 = """";
        field2 = """";
        return false;
    }

    C()
    {
        if (Init(""""))
        {
            field1.ToString();
            field2.ToString();
            field3.ToString();
            field4.ToString(); // 1
        }
        else
        {
            field1.ToString();
            field2.ToString();
            field3.ToString();
            field4.ToString(); // 2
        }
    }
}
", MemberNotNullAttributeDefinition, NotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (25,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(25, 13),
                // (32,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(32, 13)
                );
        }

        [Fact]
        public void MemberNotNull_Inheritance_PE()
        {
            var libComp = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string? field0;
    public string field1 = null!;
    public string? field2;
    public string? field2b;
    public string field3 = null!;
    public string? field4;

    [MemberNotNull(nameof(field0))]
    [MemberNotNull(nameof(field1), nameof(field2))]
    [MemberNotNull(nameof(field2b))]
    public virtual void Init() => throw null!;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            var c = CreateNullableCompilation(@"
using System.Diagnostics.CodeAnalysis;
public class Derived : Base
{
    public new string? field0;
    public string field5 = null!;
    public string? field6;
    public string field7 = null!;
    public string? field8;

    [MemberNotNull(nameof(field5), nameof(field6))]
    public override void Init()
    {
        field5 = """";
        field6 = """";
    } // 1, 2, 3, 4

    Derived()
    {
        Init();
        field0.ToString(); // 5
        field1.ToString();
        field2.ToString();
        field2b.ToString();
        field3.ToString();
        field4.ToString(); // 6
        field5.ToString();
        field6.ToString();
        field7.ToString();
        field8.ToString(); // 7
    }
}
public class Derived2 : Base
{
    public new string? field0;
    public string field5 = null!;
    public string? field6;
    public string field7 = null!;
    public string? field8;

    [MemberNotNull(nameof(field5), nameof(field6))]
    public override void Init()
    {
        base.Init();
        field5 = """";
        field6 = """";
    }
}
", references: new[] { libComp.EmitToImageReference() }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (16,5): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //     } // 1, 2, 3, 4
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(16, 5),
                // (16,5): warning CS8774: Member 'field2' must have a non-null value when exiting.
                //     } // 1, 2, 3, 4
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field2").WithLocation(16, 5),
                // (16,5): warning CS8774: Member 'field0' must have a non-null value when exiting.
                //     } // 1, 2, 3, 4
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field0").WithLocation(16, 5),
                // (16,5): warning CS8774: Member 'field2b' must have a non-null value when exiting.
                //     } // 1, 2, 3, 4
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field2b").WithLocation(16, 5),
                // (21,9): warning CS8602: Dereference of a possibly null reference.
                //         field0.ToString(); // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field0").WithLocation(21, 9),
                // (26,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 6
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(26, 9),
                // (30,9): warning CS8602: Dereference of a possibly null reference.
                //         field8.ToString(); // 7
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(30, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Inheritance_Property()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string? field0;
    public string field1 = null!;
    public string? field2;
    public string field3 = null!;
    public string? field4;

    [MemberNotNull(nameof(field0), nameof(field1), nameof(field2))]
    public virtual int Init { get => throw null!; set => throw null!; }
}
public class Derived : Base
{
    public new string? field0;
    public string field5 = null!;
    public string? field6;
    public string field7 = null!;
    public string? field8;

    [MemberNotNull(nameof(field5), nameof(field6))]
    public override int Init
    {
        get
        {
            field5 = """";
            field6 = """";
            return 0; // 1, 2, 3
        }
        set 
        { 
            field5 = """";
            field6 = """";
        } // 4, 5, 6 
    }

    Derived()
    {
        _ = Init;
        field0.ToString(); // 7
        field1.ToString();
        field2.ToString();
        field3.ToString();
        field4.ToString(); // 8
        field5.ToString();
        field6.ToString();
        field7.ToString();
        field8.ToString(); // 9
    }

    Derived(int unused)
    {
        Init = 42;
        field0.ToString(); // 10
        field1.ToString();
        field2.ToString();
        field3.ToString();
        field4.ToString(); // 11
        field5.ToString();
        field6.ToString();
        field7.ToString();
        field8.ToString(); // 12
    }
}
public class Derived2 : Base
{
    public new string? field0;
    public string field5 = null!;
    public string? field6;
    public string field7 = null!;
    public string? field8;

    [MemberNotNull(nameof(field5), nameof(field6))]
    public override int Init
    {
        get
        {
            _ = base.Init;
            field5 = """";
            field6 = """";
            return 0;
        }
        set
        {
            base.Init = value;
            field5 = """";
            field6 = """";
        }
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (29,13): warning CS8774: Member 'field0' must have a non-null value when exiting.
                //             return 0; // 1, 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return 0;").WithArguments("field0").WithLocation(29, 13),
                // (29,13): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //             return 0; // 1, 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return 0;").WithArguments("field1").WithLocation(29, 13),
                // (29,13): warning CS8774: Member 'field2' must have a non-null value when exiting.
                //             return 0; // 1, 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return 0;").WithArguments("field2").WithLocation(29, 13),
                // (35,9): warning CS8774: Member 'field0' must have a non-null value when exiting.
                //         } // 4, 5, 6 
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field0").WithLocation(35, 9),
                // (35,9): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //         } // 4, 5, 6 
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(35, 9),
                // (35,9): warning CS8774: Member 'field2' must have a non-null value when exiting.
                //         } // 4, 5, 6 
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field2").WithLocation(35, 9),
                // (41,9): warning CS8602: Dereference of a possibly null reference.
                //         field0.ToString(); // 7
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field0").WithLocation(41, 9),
                // (45,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 8
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(45, 9),
                // (49,9): warning CS8602: Dereference of a possibly null reference.
                //         field8.ToString(); // 9
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(49, 9),
                // (55,9): warning CS8602: Dereference of a possibly null reference.
                //         field0.ToString(); // 10
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field0").WithLocation(55, 9),
                // (59,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 11
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(59, 9),
                // (63,9): warning CS8602: Dereference of a possibly null reference.
                //         field8.ToString(); // 12
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(63, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Inheritance_Property_PE()
        {
            var libComp = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string? field0;
    public string field1 = null!;
    public string? field2;
    public string field3 = null!;
    public string? field4;

    [MemberNotNull(nameof(field0), nameof(field1), nameof(field2))]
    public virtual int Init { get => throw null!; set => throw null!; }
}", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            var c = CreateNullableCompilation(@"
using System.Diagnostics.CodeAnalysis;
public class Derived : Base
{
    public new string? field0;
    public string field5 = null!;
    public string? field6;
    public string field7 = null!;
    public string? field8;

    [MemberNotNull(nameof(field5), nameof(field6))]
    public override int Init
    {
        get
        {
            field5 = """";
            field6 = """";
            return 0; // 1, 2, 3
        }
        set 
        {
            field5 = """";
            field6 = """";
        } // 4, 5, 6 
    }

    Derived()
    {
        _ = Init;
        field0.ToString(); // 7
        field1.ToString();
        field2.ToString();
        field3.ToString();
        field4.ToString(); // 8
        field5.ToString();
        field6.ToString();
        field7.ToString();
        field8.ToString(); // 9
    }

    Derived(int unused)
    {
        Init = 42;
        field0.ToString(); // 10
        field1.ToString();
        field2.ToString();
        field3.ToString();
        field4.ToString(); // 11
        field5.ToString();
        field6.ToString();
        field7.ToString();
        field8.ToString(); // 12
    }
}
", new[] { libComp.EmitToImageReference() }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (18,13): warning CS8774: Member 'field0' must have a non-null value when exiting.
                //             return 0; // 1, 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return 0;").WithArguments("field0").WithLocation(18, 13),
                // (18,13): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //             return 0; // 1, 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return 0;").WithArguments("field1").WithLocation(18, 13),
                // (18,13): warning CS8774: Member 'field2' must have a non-null value when exiting.
                //             return 0; // 1, 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return 0;").WithArguments("field2").WithLocation(18, 13),
                // (24,9): warning CS8774: Member 'field0' must have a non-null value when exiting.
                //         } // 4, 5, 6 
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field0").WithLocation(24, 9),
                // (24,9): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //         } // 4, 5, 6 
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(24, 9),
                // (24,9): warning CS8774: Member 'field2' must have a non-null value when exiting.
                //         } // 4, 5, 6 
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field2").WithLocation(24, 9),
                // (30,9): warning CS8602: Dereference of a possibly null reference.
                //         field0.ToString(); // 7
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field0").WithLocation(30, 9),
                // (34,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 8
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(34, 9),
                // (38,9): warning CS8602: Dereference of a possibly null reference.
                //         field8.ToString(); // 9
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(38, 9),
                // (44,9): warning CS8602: Dereference of a possibly null reference.
                //         field0.ToString(); // 10
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field0").WithLocation(44, 9),
                // (48,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 11
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(48, 9),
                // (52,9): warning CS8602: Dereference of a possibly null reference.
                //         field8.ToString(); // 12
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(52, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Inheritance_SpecifiedOnDerived()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string? field0;
    public string field1 = null!;
    public string? field2;
    public string field3 = null!;
    public string? field4;

    public virtual void Init() => throw null!;
}
public class Derived : Base
{
    public new string? field0;
    public string field5 = null!;
    public string? field6;
    public string field7 = null!;
    public string? field8;

    [MemberNotNull(nameof(field0), nameof(field1), nameof(field2), nameof(field5), nameof(field6))] // 1, 2
    public override void Init() => throw null!;

    Derived()
    {
        Init();
        field0.ToString();
        field1.ToString();
        field2.ToString(); // 3
        field3.ToString();
        field4.ToString(); // 4
        field5.ToString();
        field6.ToString();
        field7.ToString();
        field8.ToString(); // 5
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (21,6): error CS8776: Member 'field1' cannot be used in this attribute.
                //     [MemberNotNull(nameof(field0), nameof(field1), nameof(field2), nameof(field5), nameof(field6))] // 1, 2
                Diagnostic(ErrorCode.WRN_MemberNotNullBadMember, "MemberNotNull(nameof(field0), nameof(field1), nameof(field2), nameof(field5), nameof(field6))").WithArguments("field1").WithLocation(21, 6),
                // (21,6): error CS8776: Member 'field2' cannot be used in this attribute.
                //     [MemberNotNull(nameof(field0), nameof(field1), nameof(field2), nameof(field5), nameof(field6))] // 1, 2
                Diagnostic(ErrorCode.WRN_MemberNotNullBadMember, "MemberNotNull(nameof(field0), nameof(field1), nameof(field2), nameof(field5), nameof(field6))").WithArguments("field2").WithLocation(21, 6),
                // (29,9): warning CS8602: Dereference of a possibly null reference.
                //         field2.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(29, 9),
                // (31,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(31, 9),
                // (35,9): warning CS8602: Dereference of a possibly null reference.
                //         field8.ToString(); // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(35, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Inheritance_Property_SpecifiedOnDerived_BadMember()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string? field;
    public virtual int Init => throw null!;
}
public class Derived : Base
{
    [MemberNotNull(nameof(field))]
    public override int Init => 0;
}
public class Derived2 : Base
{
    [MemberNotNull(nameof(field))]
    public override int Init
    {
        get
        {
            field = """";
            return 0;
        }
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,6): error CS8776: Member 'field' cannot be used in this attribute.
                //     [MemberNotNull(nameof(field))]
                Diagnostic(ErrorCode.WRN_MemberNotNullBadMember, "MemberNotNull(nameof(field))").WithArguments("field").WithLocation(10, 6),
                // (15,6): error CS8776: Member 'field' cannot be used in this attribute.
                //     [MemberNotNull(nameof(field))]
                Diagnostic(ErrorCode.WRN_MemberNotNullBadMember, "MemberNotNull(nameof(field))").WithArguments("field").WithLocation(15, 6)
                );
        }

        [Fact]
        public void MemberNotNull_BadMember()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C 
{
    [MemberNotNull(nameof(missing))]
    public int Init => 0;

    [MemberNotNull(nameof(missing))]
    public int Init2() => 0;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (5,27): error CS0103: The name 'missing' does not exist in the current context
                //     [MemberNotNull(nameof(missing))]
                Diagnostic(ErrorCode.ERR_NameNotInContext, "missing").WithArguments("missing").WithLocation(5, 27),
                // (8,27): error CS0103: The name 'missing' does not exist in the current context
                //     [MemberNotNull(nameof(missing))]
                Diagnostic(ErrorCode.ERR_NameNotInContext, "missing").WithArguments("missing").WithLocation(8, 27)
                );
        }

        [Fact]
        public void MemberNotNull_BadMember_2()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    [MemberNotNull(nameof(C))]
    public int Init => 0;

    [MemberNotNull(nameof(M))]
    public int Init2() => 0;

    public void M() { }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (5,6): warning CS8776: Member 'C' cannot be used in this attribute.
                //     [MemberNotNull(nameof(C))]
                Diagnostic(ErrorCode.WRN_MemberNotNullBadMember, "MemberNotNull(nameof(C))").WithArguments("C").WithLocation(5, 6),
                // (8,6): warning CS8776: Member 'M' cannot be used in this attribute.
                //     [MemberNotNull(nameof(M))]
                Diagnostic(ErrorCode.WRN_MemberNotNullBadMember, "MemberNotNull(nameof(M))").WithArguments("M").WithLocation(8, 6)
                );
        }

        [Fact]
        public void MemberNotNull_AppliedInCSharp8()
        {
            var libComp = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string? field;

    [MemberNotNull(nameof(field))]
    public int Init() => throw null!;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);
            libComp.VerifyDiagnostics();

            var c = CreateNullableCompilation(new[] { @"
public class D
{
    void M(C c, C c2)
    {
        c.field.ToString();

        c2.Init();
        c2.field.ToString();
    }
}
" }, references: new[] { libComp.EmitToImageReference() }, parseOptions: TestOptions.Regular8);

            // Note: attribute honored in C# 8 caller
            c.VerifyDiagnostics(
                // (6,9): warning CS8602: Dereference of a possibly null reference.
                //         c.field.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c.field").WithLocation(6, 9)
                );
        }

        [Fact]
        public void MemberNotNullWhen_Inheritance_SpecifiedOnDerived_BadMember()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string? field;
    public string? Property;
    public virtual bool Init => throw null!;
    public virtual bool Init2() => throw null!;
}
public class Derived : Base
{
    [MemberNotNullWhen(true, nameof(field), nameof(Property))]
    public override bool Init => throw null!;

    [MemberNotNullWhen(true, nameof(field), nameof(Property))]
    public override bool Init2() => throw null!;
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (12,6): error CS8776: Member 'field' cannot be used in this attribute.
                //     [MemberNotNullWhen(true, nameof(field), nameof(Property))]
                Diagnostic(ErrorCode.WRN_MemberNotNullBadMember, "MemberNotNullWhen(true, nameof(field), nameof(Property))").WithArguments("field").WithLocation(12, 6),
                // (12,6): error CS8776: Member 'Property' cannot be used in this attribute.
                //     [MemberNotNullWhen(true, nameof(field), nameof(Property))]
                Diagnostic(ErrorCode.WRN_MemberNotNullBadMember, "MemberNotNullWhen(true, nameof(field), nameof(Property))").WithArguments("Property").WithLocation(12, 6),
                // (15,6): error CS8776: Member 'field' cannot be used in this attribute.
                //     [MemberNotNullWhen(true, nameof(field), nameof(Property))]
                Diagnostic(ErrorCode.WRN_MemberNotNullBadMember, "MemberNotNullWhen(true, nameof(field), nameof(Property))").WithArguments("field").WithLocation(15, 6),
                // (15,6): error CS8776: Member 'Property' cannot be used in this attribute.
                //     [MemberNotNullWhen(true, nameof(field), nameof(Property))]
                Diagnostic(ErrorCode.WRN_MemberNotNullBadMember, "MemberNotNullWhen(true, nameof(field), nameof(Property))").WithArguments("Property").WithLocation(15, 6)
                );
        }

        [Fact]
        public void MemberNotNull_EnforcementInTryFinally()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;
    public string field5;
    public string? field6;

    public C() // 1
    {
        Init();
    }

    [MemberNotNull(nameof(field1), nameof(field2), nameof(field3), nameof(field4))]
    void Init()
    {
        try
        {
            bool b = true;
            if (b)
            {
                return; // 2, 3
            }
            else
            {
                field3 = """";
                field4 = """";
                return;
            }
        }
        finally
        {
            field1 = """";
            field2 = """";
        }
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (12,12): warning CS8618: Non-nullable field 'field5' is uninitialized. Consider declaring the field as nullable.
                //     public C() // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "C").WithArguments("field", "field5").WithLocation(12, 12),
                // (25,17): warning CS8771: Member 'field3' must have a non-null value when exiting.
                //                 return; // 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return;").WithArguments("field3").WithLocation(25, 17),
                // (25,17): warning CS8771: Member 'field4' must have a non-null value when exiting.
                //                 return; // 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return;").WithArguments("field4").WithLocation(25, 17)
                );
        }

        [Fact]
        public void MemberNotNull_LangVersion()
        {
            var source = @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string? field1;
    [MemberNotNull(nameof(field1))]
    void Init() => throw null!;

    [MemberNotNullWhen(true, nameof(field1))]
    bool Init2() => throw null!;

    [MemberNotNull(nameof(field1))]
    bool IsInit { get { throw null!; } }

    [MemberNotNullWhen(true, nameof(field1))]
    bool IsInit2 { get { throw null!; } }
}
";
            var c = CreateCompilation(new[] { source, MemberNotNullAttributeDefinition, MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular8);
            c.VerifyDiagnostics(
                // (5,18): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public string? field1;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 18),
                // (6,6): error CS8400: Feature 'MemberNotNull attribute' is not available in C# 8.0. Please use language version 9.0 or greater.
                //     [MemberNotNull(nameof(field1))]
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion8, "MemberNotNull(nameof(field1))").WithArguments("MemberNotNull attribute", "9.0").WithLocation(6, 6),
                // (9,6): error CS8400: Feature 'MemberNotNull attribute' is not available in C# 8.0. Please use language version 9.0 or greater.
                //     [MemberNotNullWhen(true, nameof(field1))]
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion8, "MemberNotNullWhen(true, nameof(field1))").WithArguments("MemberNotNull attribute", "9.0").WithLocation(9, 6),
                // (12,6): error CS8400: Feature 'MemberNotNull attribute' is not available in C# 8.0. Please use language version 9.0 or greater.
                //     [MemberNotNull(nameof(field1))]
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion8, "MemberNotNull(nameof(field1))").WithArguments("MemberNotNull attribute", "9.0").WithLocation(12, 6),
                // (15,6): error CS8400: Feature 'MemberNotNull attribute' is not available in C# 8.0. Please use language version 9.0 or greater.
                //     [MemberNotNullWhen(true, nameof(field1))]
                Diagnostic(ErrorCode.ERR_FeatureNotAvailableInVersion8, "MemberNotNullWhen(true, nameof(field1))").WithArguments("MemberNotNull attribute", "9.0").WithLocation(15, 6)
                );

            var c2 = CreateCompilation(new[] { source, MemberNotNullAttributeDefinition, MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);
            c2.VerifyDiagnostics(
                // (5,18): warning CS8632: The annotation for nullable reference types should only be used in code within a '#nullable' annotations context.
                //     public string? field1;
                Diagnostic(ErrorCode.WRN_MissingNonNullTypesContextForAnnotation, "?").WithLocation(5, 18)
                );
        }

        [Fact]
        public void MemberNotNull_BoolReturning()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        if (Init())
        {
            field1.ToString();
            field2.ToString();
            field3.ToString(); // 1
            field4.ToString(); // 2
        }
        else
        {
            field1.ToString();
            field2.ToString();
            field3.ToString(); // 3
            field4.ToString(); // 4
        }
    }

    [MemberNotNull(nameof(field1), nameof(field2))]
    bool Init() => true; // 5, 6
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (16,13): warning CS8602: Dereference of a possibly null reference.
                //             field3.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(16, 13),
                // (17,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(17, 13),
                // (23,13): warning CS8602: Dereference of a possibly null reference.
                //             field3.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(23, 13),
                // (24,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(24, 13),
                // (29,20): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //     bool Init() => true; // 5, 6
                Diagnostic(ErrorCode.WRN_MemberNotNull, "true").WithArguments("field1").WithLocation(29, 20),
                // (29,20): warning CS8774: Member 'field2' must have a non-null value when exiting.
                //     bool Init() => true; // 5, 6
                Diagnostic(ErrorCode.WRN_MemberNotNull, "true").WithArguments("field2").WithLocation(29, 20)
                );
        }

        [Fact]
        public void MemberNotNull_OtherType()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public void M()
    {
        var d = new D();
        d.Init();
        d.field1.ToString();
        d.field2.ToString();
        d.field3.ToString();
        d.field4.ToString(); // 1
    }
}
public class D
{
    public string field1; // 2
    public string? field2;
    public string field3; // 3
    public string? field4;

    [MemberNotNull(nameof(field1), nameof(field2))]
    public void Init() => throw null!;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (12,9): warning CS8602: Dereference of a possibly null reference.
                //         d.field4.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "d.field4").WithLocation(12, 9),
                // (17,19): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public string field1; // 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field1").WithArguments("field", "field1").WithLocation(17, 19),
                // (19,19): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public string field3; // 3
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field3").WithArguments("field", "field3").WithLocation(19, 19)
                );
        }

        [Fact]
        public void MemberNotNull_OtherType_ExtensionMethod()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public void M()
    {
        var d = new D();
        d.Init();
        d.field1.ToString();
        d.field2.ToString();
        d.field3.ToString();
        d.field4.ToString();
    }
}
public class D
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;
}
public static class Extension
{
    [MemberNotNull(nameof(field1), nameof(field2))]
    public static void Init(this D d) => throw null!;
}
", MemberNotNullAttributeDefinition });

            c.VerifyDiagnostics(
                // (10,9): warning CS8602: Dereference of a possibly null reference.
                //         d.field2.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "d.field2").WithLocation(10, 9),
                // (12,9): warning CS8602: Dereference of a possibly null reference.
                //         d.field4.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "d.field4").WithLocation(12, 9),
                // (17,19): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public string field1;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field1").WithArguments("field", "field1").WithLocation(17, 19),
                // (19,19): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public string field3;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field3").WithArguments("field", "field3").WithLocation(19, 19),
                // (24,27): error CS0103: The name 'field1' does not exist in the current context
                //     [MemberNotNull(nameof(field1), nameof(field2))]
                Diagnostic(ErrorCode.ERR_NameNotInContext, "field1").WithArguments("field1").WithLocation(24, 27),
                // (24,43): error CS0103: The name 'field2' does not exist in the current context
                //     [MemberNotNull(nameof(field1), nameof(field2))]
                Diagnostic(ErrorCode.ERR_NameNotInContext, "field2").WithArguments("field2").WithLocation(24, 43)
                );
        }

        [Fact]
        public void MemberNotNull_Property_Getter()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        _ = Count;
        field1.ToString();
        field2.ToString();
        field3.ToString(); // 1
        field4.ToString(); // 2
    }

    public C(int unused)
    {
        Count = 42;
        field1.ToString(); // 3
        field2.ToString(); // 4
        field3.ToString(); // 5
        field4.ToString(); // 6
    }

    int Count
    {
        [MemberNotNull(nameof(field1), nameof(field2))]
        get => throw null!;
        set { }
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(15, 9),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(16, 9),
                // (22,9): warning CS8602: Dereference of a possibly null reference.
                //         field1.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field1").WithLocation(22, 9),
                // (23,9): warning CS8602: Dereference of a possibly null reference.
                //         field2.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(23, 9),
                // (24,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(24, 9),
                // (25,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 6
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(25, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Property_Setter()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        _ = Count;
        field1.ToString(); // 1
        field2.ToString(); // 2
        field3.ToString(); // 3
        field4.ToString(); // 4
    }

    public C(int unused)
    {
        Count = 1;
        field1.ToString();
        field2.ToString();
        field3.ToString(); // 5
        field4.ToString(); // 6
    }

    int Count
    {
        get
        {
            field1 = """";
            return 0;
        }
        [MemberNotNull(nameof(field1), nameof(field2))]
        set
        {
            field2 = """";
        } // 7
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (13,9): warning CS8602: Dereference of a possibly null reference.
                //         field1.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field1").WithLocation(13, 9),
                // (14,9): warning CS8602: Dereference of a possibly null reference.
                //         field2.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(14, 9),
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(15, 9),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(16, 9),
                // (24,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(24, 9),
                // (25,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 6
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(25, 9),
                // (39,9): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //         } // 7
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(39, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Property_Getter_Static()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public static class C
{
    public static string field1;
    public static string? field2;
    public static string field3;
    public static string? field4;

    static C()
    {
        _ = Count;
        field1.ToString();
        field2.ToString();
        field3.ToString(); // 1
        field4.ToString(); // 2
    }

    static int Count
    {
        [MemberNotNull(nameof(field1), nameof(field2))]
        get => throw null!;
        set { }
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(15, 9),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(16, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Property_Getter_Static_2()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public static class C
{
    public static string field1;
    public static string? field2;
    public static string field3;
    public static string? field4;

    static C()
    {
        Count = 42;
        field1.ToString(); // 1
        field2.ToString(); // 2
        field3.ToString(); // 3
        field4.ToString(); // 4
    }

    static int Count
    {
        [MemberNotNull(nameof(field1), nameof(field2))]
        get => throw null!;
        set { }
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (13,9): warning CS8602: Dereference of a possibly null reference.
                //         field1.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field1").WithLocation(13, 9),
                // (14,9): warning CS8602: Dereference of a possibly null reference.
                //         field2.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(14, 9),
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(15, 9),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(16, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Property_Setter_Static()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public static class C
{
    public static string field1;
    public static string? field2;
    public static string field3;
    public static string? field4;

    static C()
    {
        _ = Count;
        field1.ToString(); // 1
        field2.ToString(); // 2
        field3.ToString(); // 3
        field4.ToString(); // 4
    }

    static int Count
    {
        get => throw null!;

        [MemberNotNull(nameof(field1), nameof(field2))]
        set => throw null!;
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (13,9): warning CS8602: Dereference of a possibly null reference.
                //         field1.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field1").WithLocation(13, 9),
                // (14,9): warning CS8602: Dereference of a possibly null reference.
                //         field2.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(14, 9),
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(15, 9),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(16, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Property_Setter_Static_2()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public static class C
{
    public static string field1;
    public static string? field2;
    public static string field3;
    public static string? field4;

    static C()
    {
        Count = 1;
        field1.ToString();
        field2.ToString();
        field3.ToString(); // 1
        field4.ToString(); // 2
    }

    static int Count
    {
        get => throw null!;

        [MemberNotNull(nameof(field1), nameof(field2))]
        set => throw null!;
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(15, 9),
                // (16,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(16, 9)
                );
        }

        [Fact]
        public void MemberNotNull_Branches()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string field2;
    public string field3;
    public string field4;

    public C()
    {
        bool b = true;
        if (b)
            Init();
        else
            Init2();
    }

    [MemberNotNull(nameof(field1), nameof(field2))]
    void Init() => throw null!;

    [MemberNotNull(nameof(field2), nameof(field3))]
    void Init2() => throw null!;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,12): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public C()
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "C").WithArguments("field", "field3").WithLocation(10, 12),
                // (10,12): warning CS8618: Non-nullable field 'field4' is uninitialized. Consider declaring the field as nullable.
                //     public C()
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "C").WithArguments("field", "field4").WithLocation(10, 12),
                // (10,12): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public C()
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "C").WithArguments("field", "field1").WithLocation(10, 12)
                );
        }

        [Fact]
        public void MemberNotNull_Static()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public static string field1;
    public static string? field2;
    public static string field3;
    public static string? field4;

    static C()
    {
        Init();
        field1.ToString();
        field2.ToString();
        field3.ToString(); // 1
        field4.ToString();
    }

    [MemberNotNull(nameof(field1), nameof(field2), nameof(field4))]
    static void Init()
    {
        field2 = """";
    } // 2, 3
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (15,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(15, 9),
                // (23,5): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //     } // 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(23, 5),
                // (23,5): warning CS8774: Member 'field4' must have a non-null value when exiting.
                //     } // 2, 3
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field4").WithLocation(23, 5)
                );
        }

        [Fact]
        public void MemberNotNull_Multiple()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string field2;
    public string field3;
    public string field4;

    public C()
    {
        Init();
    }

    [MemberNotNull(nameof(field1))]
    [MemberNotNull(nameof(field2), nameof(field3))]
    void Init() => throw null!;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,12): warning CS8618: Non-nullable field 'field4' is uninitialized. Consider declaring the field as nullable.
                //     public C()
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "C").WithArguments("field", "field4").WithLocation(10, 12)
                );
        }

        [Fact]
        public void MemberNotNull_Multiple_ParamsConstructor()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string field2;
    public string field3;
    public string field4;

    public C()
    {
        Init();
        field1.ToString();
        field2.ToString();
        field3.ToString();
        field4.ToString();
    }

    [MemberNotNull(nameof(field1), nameof(field2))]
    [MemberNotNull(nameof(field3), nameof(field4))]
    void Init()
    {
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (23,5): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //     }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(23, 5),
                // (23,5): warning CS8774: Member 'field2' must have a non-null value when exiting.
                //     }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field2").WithLocation(23, 5),
                // (23,5): warning CS8774: Member 'field3' must have a non-null value when exiting.
                //     }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field3").WithLocation(23, 5),
                // (23,5): warning CS8774: Member 'field4' must have a non-null value when exiting.
                //     }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field4").WithLocation(23, 5)
                );
        }

        [Fact]
        public void MemberNotNull_Multiple_Property()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string field2;
    public string field3;
    public string field4;

    public C()
    {
        _ = Init;
    }

    [MemberNotNull(nameof(field1))]
    [MemberNotNull(nameof(field2), nameof(field3))]
    int Init
    {
        get => 1;
        set { }
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);
            c.VerifyDiagnostics(
                // (10,12): warning CS8618: Non-nullable field 'field4' is uninitialized. Consider declaring the field as nullable.
                //     public C()
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "C").WithArguments("field", "field4").WithLocation(10, 12),
                // (19,16): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //         get => 1;
                Diagnostic(ErrorCode.WRN_MemberNotNull, "1").WithArguments("field1").WithLocation(19, 16),
                // (19,16): warning CS8774: Member 'field2' must have a non-null value when exiting.
                //         get => 1;
                Diagnostic(ErrorCode.WRN_MemberNotNull, "1").WithArguments("field2").WithLocation(19, 16),
                // (19,16): warning CS8774: Member 'field3' must have a non-null value when exiting.
                //         get => 1;
                Diagnostic(ErrorCode.WRN_MemberNotNull, "1").WithArguments("field3").WithLocation(19, 16),
                // (20,15): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //         set { }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(20, 15),
                // (20,15): warning CS8774: Member 'field2' must have a non-null value when exiting.
                //         set { }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field2").WithLocation(20, 15),
                // (20,15): warning CS8774: Member 'field3' must have a non-null value when exiting.
                //         set { }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field3").WithLocation(20, 15)
                );
        }

        [Fact]
        public void MemberNotNull_NotFound()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public C()
    {
        Init();
    }

    [MemberNotNull(nameof(field))]
    void Init() => throw null!;
}
", MemberNotNullAttributeDefinition });

            c.VerifyDiagnostics(
                // (10,27): error CS0103: The name 'field' does not exist in the current context
                //     [MemberNotNull(nameof(field))]
                Diagnostic(ErrorCode.ERR_NameNotInContext, "field").WithArguments("field").WithLocation(10, 27)
                );
        }

        [Fact]
        public void MemberNotNull_NotFound_PrivateInBase()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    private string? field;
    public string? P { get { return field; } set { field = value; } }
}
public class C : Base
{
    public C()
    {
        Init();
    }

    [MemberNotNull(nameof(field))]
    void Init() => throw null!;
}
", MemberNotNullAttributeDefinition });

            c.VerifyDiagnostics(
                // (15,27): error CS0122: 'Base.field' is inaccessible due to its protection level
                //     [MemberNotNull(nameof(field))]
                Diagnostic(ErrorCode.ERR_BadAccess, "field").WithArguments("Base.field").WithLocation(15, 27)
                );
        }

        [Fact]
        public void MemberNotNull_Null()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    [MemberNotNull(members: null)]
    void Init() => throw null!;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (5,29): warning CS8625: Cannot convert null literal to non-nullable reference type.
                //     [MemberNotNull(members: null)]
                Diagnostic(ErrorCode.WRN_NullAsNonNullable, "null").WithLocation(5, 29)
                );
        }

        [Fact]
        public void MemberNotNull_EnforcedInMethodBody()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    C()
    {
        Init();
    }

    [MemberNotNull(nameof(field1), nameof(field2))]
    void Init() { }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,5): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     C()
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "C").WithArguments("field", "field3").WithLocation(10, 5),
                // (16,19): warning CS8771: Member 'field1' must have a non-null value when exiting.
                //     void Init() { }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(16, 19),
                // (16,19): warning CS8771: Member 'field2' must have a non-null value when exiting.
                //     void Init() { }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field2").WithLocation(16, 19)
                );
        }

        [Fact]
        public void MemberNotNull_EnforcedInMethodBody_AccessedInBody()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    C()
    {
        Init();
    }

    [MemberNotNull(nameof(field1), nameof(field2))]
    void Init()
    {
        field1.ToString();
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,5): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     C()
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "C").WithArguments("field", "field3").WithLocation(10, 5),
                // (18,9): warning CS8602: Dereference of a possibly null reference.
                //         field1.ToString();
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field1").WithLocation(18, 9),
                // (19,5): warning CS8771: Member 'field2' must have a non-null value when exiting.
                //     }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field2").WithLocation(19, 5)
                );
        }

        [Fact]
        public void MemberNotNull_EnforcedInMethodBody_AccessedInBody_OnlyThisField()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    C()
    {
        Init("""", new C());
    }

    [MemberNotNull(nameof(field1), nameof(field2))]
    void Init(string field1, C c)
    {
        field1.ToString();
        c.field1.ToString();
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,5): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     C()
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "C").WithArguments("field", "field3").WithLocation(10, 5),
                // (20,5): warning CS8771: Member 'field1' must have a non-null value when exiting.
                //     }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(20, 5),
                // (20,5): warning CS8771: Member 'field2' must have a non-null value when exiting.
                //     }
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field2").WithLocation(20, 5)
                );
        }

        [Fact]
        public void MemberNotNull_EnforcedInMethodBody_Throw()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    [MemberNotNull(nameof(field1), nameof(field2))]
    void Init() => throw null!; 
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (5,19): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public string field1;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field1").WithArguments("field", "field1").WithLocation(5, 19),
                // (7,19): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public string field3;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field3").WithArguments("field", "field3").WithLocation(7, 19)
                );
        }

        [Fact]
        public void MemberNotNull_EnforcedInMethodBody_BranchWithReturn()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1; // 1
    public string? field2;
    public string field3; // 2
    public string? field4;

    [MemberNotNull(nameof(field1), nameof(field2))]
    void Init()
    {
        bool b =  true;
        if (b)
        {
            return; // 3, 4
        }
        field2 = """";
    } // 5
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (5,19): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public string field1; // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field1").WithArguments("field", "field1").WithLocation(5, 19),
                // (7,19): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public string field3; // 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field3").WithArguments("field", "field3").WithLocation(7, 19),
                // (16,13): warning CS8771: Member 'field1' must have a non-null value when exiting.
                //             return; // 3, 4
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return;").WithArguments("field1").WithLocation(16, 13),
                // (16,13): warning CS8771: Member 'field2' must have a non-null value when exiting.
                //             return; // 3, 4
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return;").WithArguments("field2").WithLocation(16, 13),
                // (19,5): warning CS8771: Member 'field1' must have a non-null value when exiting.
                //     } // 5
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(19, 5)
                );
        }

        [Fact]
        public void MemberNotNull_EnforcedInMethodBody_BranchWithReturn_WithValue()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1; // 1
    public string? field2;
    public string field3; // 2
    public string? field4;

    [MemberNotNull(nameof(field1), nameof(field2))]
    int Init()
    {
        bool b =  true;
        if (b)
        {
            return 0; // 3, 4
        }
        field2 = """";
        return 0; // 5
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (5,19): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public string field1; // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field1").WithArguments("field", "field1").WithLocation(5, 19),
                // (7,19): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public string field3; // 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field3").WithArguments("field", "field3").WithLocation(7, 19),
                // (16,13): warning CS8771: Member 'field1' must have a non-null value when exiting.
                //             return 0; // 3, 4
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return 0;").WithArguments("field1").WithLocation(16, 13),
                // (16,13): warning CS8771: Member 'field2' must have a non-null value when exiting.
                //             return 0; // 3, 4
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return 0;").WithArguments("field2").WithLocation(16, 13),
                // (19,9): warning CS8771: Member 'field1' must have a non-null value when exiting.
                //         return 0; // 5
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return 0;").WithArguments("field1").WithLocation(19, 9)
                );
        }

        [Fact]
        public void MemberNotNull_EnforcedInProperty()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    C() // 1
    {
        _ = Init;
    }

    C(int ignored) // 2
    {
        Init = 42;
    }

    [MemberNotNull(nameof(field1), nameof(field2))]
    int Init
    {
        get { return 42; } // 3, 4
        set { } // 5, 6
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,5): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     C() // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "C").WithArguments("field", "field3").WithLocation(10, 5),
                // (15,5): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     C(int ignored) // 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "C").WithArguments("field", "field3").WithLocation(15, 5),
                // (23,15): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //         get { return 42; } // 3, 4
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return 42;").WithArguments("field1").WithLocation(23, 15),
                // (23,15): warning CS8774: Member 'field2' must have a non-null value when exiting.
                //         get { return 42; } // 3, 4
                Diagnostic(ErrorCode.WRN_MemberNotNull, "return 42;").WithArguments("field2").WithLocation(23, 15),
                // (24,15): warning CS8774: Member 'field1' must have a non-null value when exiting.
                //         set { } // 5, 6
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field1").WithLocation(24, 15),
                // (24,15): warning CS8774: Member 'field2' must have a non-null value when exiting.
                //         set { } // 5, 6
                Diagnostic(ErrorCode.WRN_MemberNotNull, "}").WithArguments("field2").WithLocation(24, 15)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_Simple()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        if (Init())
        {
            field1.ToString();
            field2.ToString();
            field3.ToString(); // 1
            field4.ToString(); // 2
        }
        else
        {
            throw null!;
        }
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2))]
    bool Init() => throw null!;
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (16,13): warning CS8602: Dereference of a possibly null reference.
                //             field3.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(16, 13),
                // (17,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(17, 13)
                );
        }

        [Fact, WorkItem(47667, "https://github.com/dotnet/roslyn/issues/47667")]
        public void MemberNotNullWhenTrue_Generic_01()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C<T>
{
    public T field1;
    public T? field2;
    public T field3;
    public T? field4;

    public C()
    {
        if (Init())
        {
            field1.ToString();
            field2.ToString();
            field3.ToString(); // 1
            field4.ToString(); // 2
        }
        else
        {
            throw null!;
        }
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2))]
    bool Init() => throw null!;
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (16,13): warning CS8602: Dereference of a possibly null reference.
                //             field3.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(16, 13),
                // (17,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(17, 13)
                );
        }

        [Fact, WorkItem(47667, "https://github.com/dotnet/roslyn/issues/47667")]
        public void MemberNotNullWhenTrue_Generic_02()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
using System.Collections.Generic;

class GetResult<T>
{
    [MemberNotNullWhen(true, ""Value"")]
    public bool OK { get; set; }
    public T? Value { get; init; }
}

record Manager(int Age);

class Archive
{
    readonly Dictionary<string, Manager> Dict = new Dictionary<string, Manager>();

    public GetResult<Manager> this[string key]
        => Dict.TryGetValue(key, out var value)
            ? new GetResult<Manager> { OK = true, Value = value }
            : new GetResult<Manager> { OK = false, Value = null };
}

public class C
{
    public void M()
    {
        Archive archive = new Archive();

        var result = archive[""John""];
        int age1 = result.OK
            ? result.Value.Age
            : result.Value.Age; // 1
    }
}
", MemberNotNullWhenAttributeDefinition, IsExternalInitTypeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (33,15): warning CS8602: Dereference of a possibly null reference.
                //             : result.Value.Age; // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "result.Value", isSuppressed: false).WithLocation(33, 15)
                );
        }

        [Fact, WorkItem(47667, "https://github.com/dotnet/roslyn/issues/47667")]
        public void MemberNotNullWhenFalse_Generic_01()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;

public class C<T> where T : new()
{
    [MemberNotNullWhen(false, ""Value"")]
    public bool IsBad { get; set; }
    public T? Value { get; set; }
}

public class Program
{
    public void M(C<object> c)
    {
        _ = c.IsBad
            ? c.Value.ToString() // 1
            : c.Value.ToString();
    }
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (16,15): warning CS8602: Dereference of a possibly null reference.
                //             ? c.Value.ToString() // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c.Value", isSuppressed: false).WithLocation(16, 15)
                );
        }

        [Fact, WorkItem(47667, "https://github.com/dotnet/roslyn/issues/47667")]
        public void MemberNotNull_Generic_01()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;

public class C<T> where T : new()
{
    [MemberNotNull(""Value"")]
    public void Init() { Value = new T(); }
    public T? Value { get; set; }
}

public class Program
{
    public void M(bool b, C<object> c)
    {
        if (b) c.Value.ToString(); // 1
        c.Init();
        c.Value.ToString();
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (15,16): warning CS8602: Dereference of a possibly null reference.
                //         if (b) c.Value.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c.Value", isSuppressed: false).WithLocation(15, 16)
                );
        }

        [Fact, WorkItem(47667, "https://github.com/dotnet/roslyn/issues/47667")]
        public void MemberNotNull_Extension_01()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;

public class C
{
    public string? _field;
}

public static class Ext
{
    public static string? _field;

    [MemberNotNull(""_field"")]
    public static void AssertFieldNotNull(this C c) { if (_field == null) throw null!; }
}

class Program
{
    void M1(C c)
    {
        c.AssertFieldNotNull();
        Ext._field.ToString();
        c._field.ToString(); // 1
    }

    void M2(C c)
    {
        Ext.AssertFieldNotNull(c);
        Ext._field.ToString();
        c._field.ToString(); // 2
    }
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (23,9): warning CS8602: Dereference of a possibly null reference.
                //         c._field.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c._field", isSuppressed: false).WithLocation(23, 9),
                // (30,9): warning CS8602: Dereference of a possibly null reference.
                //         c._field.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c._field", isSuppressed: false).WithLocation(30, 9)
                );
        }

        [Fact, WorkItem(47667, "https://github.com/dotnet/roslyn/issues/47667")]
        public void MemberNotNullWhen_Extension_01()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;

public class C
{
    public string? _field;
}

public static class Ext
{
    public static string? _field;

    [MemberNotNullWhen(true, ""_field"")]
    public static bool IsFieldPresent(this C c) { return _field != null; }

    [MemberNotNullWhen(false, ""_field"")]
    public static bool IsFieldAbsent(this C c) { return _field == null; }
}

class Program
{
    void M1(C c)
    {
        _ = c.IsFieldPresent()
            ? Ext._field.ToString()
            : Ext._field.ToString(); // 1

        _ = c.IsFieldPresent()
            ? c._field.ToString() // 2
            : c._field.ToString(); // 3
    }

    void M2(C c)
    {
        _ = c.IsFieldAbsent()
            ? Ext._field.ToString() // 4
            : Ext._field.ToString();

        _ = c.IsFieldAbsent()
            ? c._field.ToString() // 5
            : c._field.ToString(); // 6
    }
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (26,15): warning CS8602: Dereference of a possibly null reference.
                //             : Ext._field.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "Ext._field", isSuppressed: false).WithLocation(26, 15),
                // (29,15): warning CS8602: Dereference of a possibly null reference.
                //             ? c._field.ToString() // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c._field", isSuppressed: false).WithLocation(29, 15),
                // (30,15): warning CS8602: Dereference of a possibly null reference.
                //             : c._field.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c._field", isSuppressed: false).WithLocation(30, 15),
                // (36,15): warning CS8602: Dereference of a possibly null reference.
                //             ? Ext._field.ToString() // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "Ext._field", isSuppressed: false).WithLocation(36, 15),
                // (40,15): warning CS8602: Dereference of a possibly null reference.
                //             ? c._field.ToString() // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c._field", isSuppressed: false).WithLocation(40, 15),
                // (41,15): warning CS8602: Dereference of a possibly null reference.
                //             : c._field.ToString(); // 6
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "c._field", isSuppressed: false).WithLocation(41, 15)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_NonConstantBool()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1; // 1

    [MemberNotNullWhen(true, nameof(field1))]
    bool Init()
    {
        bool b = true;
        return b;
    }
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (5,19): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public string field1; // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field1").WithArguments("field", "field1").WithLocation(5, 19)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_NullValues()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        if (Init())
        {
            field1.ToString(); // 1
            field2.ToString(); // 2
            field3.ToString(); // 3
            field4.ToString(); // 4
        }
        else
        {
            throw null!;
        }
    }

    [MemberNotNullWhen(true, null!, null!)]
    [MemberNotNullWhen(true, members: null!)]
    [MemberNotNullWhen(true, member: null!)]
    bool Init() => throw null!;
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (14,13): warning CS8602: Dereference of a possibly null reference.
                //             field1.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field1").WithLocation(14, 13),
                // (15,13): warning CS8602: Dereference of a possibly null reference.
                //             field2.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(15, 13),
                // (16,13): warning CS8602: Dereference of a possibly null reference.
                //             field3.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(16, 13),
                // (17,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(17, 13)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_EnforcementInTryFinally()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        if (!Init())
        {
            throw null!;
        }
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2), nameof(field3), nameof(field4))]
    bool Init()
    {
        try
        {
            bool b = true;
            if (b)
            {
                field3 = """";
                field4 = """";
                return true; // 1, 2 
            }
        }
        finally
        {
            field1 = """";
            field2 = """";
        }

        return false;
    }
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics();
        }

        [Fact]
        public void MemberNotNullWhenTrue_EnforcementInTryFinally_2()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;

    [MemberNotNullWhen(true, nameof(field1))]
    bool Init()
    {
        try
        {
            return true;
        }
        finally
        {
            field1 = string.Empty;
        }
    }
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (5,19): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public string field1;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field1").WithArguments("field", "field1").WithLocation(5, 19)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_EnforcementInTryFinally_2_Reversed()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;

    [MemberNotNullWhen(true, nameof(field1))]
    bool Init()
    {
        try
        {
            return false;
        }
        finally
        {
            field1 = string.Empty;
        }
    }
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (5,19): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public string field1;
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field1").WithArguments("field", "field1").WithLocation(5, 19)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_EnforcementInTryFinally_Reversed()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        if (!Init())
        {
            throw null!;
        }
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2), nameof(field3), nameof(field4))]
    bool Init()
    {
        try
        {
            bool b = true;
            if (b)
            {
                field3 = """";
                field4 = """";
                return false;
            }
        }
        finally
        {
            field1 = """";
            field2 = """";
        }

        return true; // 1, 2
    }
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (37,9): warning CS8772: Member 'field3' must have a non-null value when exiting with 'true'.
                //         return true; // 1, 2
                Diagnostic(ErrorCode.WRN_MemberNotNullWhen, "return true;").WithArguments("field3", "true").WithLocation(37, 9),
                // (37,9): warning CS8772: Member 'field4' must have a non-null value when exiting with 'true'.
                //         return true; // 1, 2
                Diagnostic(ErrorCode.WRN_MemberNotNullWhen, "return true;").WithArguments("field4", "true").WithLocation(37, 9)
                );
        }

        [Fact]
        public void MemberNotNullWhenFalse_EnforcementInTryFinally()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        if (NotInit())
        {
            throw null!;
        }
    }

    [MemberNotNullWhen(false, nameof(field1), nameof(field2), nameof(field3), nameof(field4))]
    bool NotInit()
    {
        try
        {
            bool b = true;
            if (b)
            {
                field3 = """";
                field4 = """";
                return false;
            }
        }
        finally
        {
            field1 = """";
            field2 = """";
        }

        return true;
    }
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics();
        }

        [Fact]
        public void MemberNotNullWhenTrue_Inheritance()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public Base() // 1
    {
        if (!Init())
            throw null!;
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2))]
    public virtual bool Init() => throw null!;
}
public class Derived : Base
{
    public string field5;
    public string? field6;
    public string field7;
    public string? field8;

    public Derived() : base() // 2
    {
        if (!Init())
            throw null!;
    }

    void M()
    {
        if (Init())
        {
            field1.ToString();
            field2.ToString();
            field3.ToString();
            field4.ToString(); // 3
            field5.ToString();
            field6.ToString();
            field7.ToString();
            field8.ToString(); // 4
        }
    }

    [MemberNotNullWhen(true, nameof(field6), nameof(field7))]
    public override bool Init() => throw null!;
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,12): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public Base() // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Base").WithArguments("field", "field3").WithLocation(10, 12),
                // (26,12): warning CS8618: Non-nullable field 'field5' is uninitialized. Consider declaring the field as nullable.
                //     public Derived() : base() // 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Derived").WithArguments("field", "field5").WithLocation(26, 12),
                // (39,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(39, 13),
                // (43,13): warning CS8602: Dereference of a possibly null reference.
                //             field8.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(43, 13)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_Inheritance_PE()
        {
            var libComp = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string field1;
    public string? field2;
    public string? field2b;
    public string field3;
    public string? field4;

    public Base()
    {
        if (!Init())
            throw null!;
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2))]
    [MemberNotNullWhen(true, nameof(field2b))]
    public virtual bool Init() => throw null!;
}", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            var c = CreateNullableCompilation(@"
using System.Diagnostics.CodeAnalysis;
public class Derived : Base
{
    public string field5;
    public string? field6;
    public string field7;
    public string? field8;

    public Derived() : base() // 1
    {
        if (!Init())
            throw null!;
    }

    void M()
    {
        if (Init())
        {
            field1.ToString();
            field2.ToString();
            field2b.ToString();
            field3.ToString();
            field4.ToString(); // 2
            field5.ToString();
            field6.ToString();
            field7.ToString();
            field8.ToString(); // 3
        }
    }

    [MemberNotNullWhen(true, nameof(field6), nameof(field7))]
    public override bool Init() => throw null!;
}
", references: new[] { libComp.EmitToImageReference() }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,12): warning CS8618: Non-nullable field 'field5' is uninitialized. Consider declaring the field as nullable.
                //     public Derived() : base() // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Derived").WithArguments("field", "field5").WithLocation(10, 12),
                // (24,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(24, 13),
                // (28,13): warning CS8602: Dereference of a possibly null reference.
                //             field8.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(28, 13)
                );
        }

        [Fact]
        public void MemberNotNullWhenFalse_Inheritance_PE()
        {
            var libComp = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public Base()
    {
        if (!NotInit())
            throw null!;
    }

    [MemberNotNullWhen(false, nameof(field1), nameof(field2))]
    public virtual bool NotInit() => throw null!;
}", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            var c = CreateNullableCompilation(@"
using System.Diagnostics.CodeAnalysis;
public class Derived : Base
{
    public string field5;
    public string? field6;
    public string field7;
    public string? field8;

    public Derived() : base() // 1
    {
        if (NotInit())
            throw null!;
    }

    void M()
    {
        if (!NotInit())
        {
            field1.ToString();
            field2.ToString();
            field3.ToString();
            field4.ToString(); // 2
            field5.ToString();
            field6.ToString();
            field7.ToString();
            field8.ToString(); // 3
        }
    }

    [MemberNotNullWhen(false, nameof(field6), nameof(field7))]
    public override bool NotInit() => throw null!;
}
", references: new[] { libComp.EmitToImageReference() }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,12): warning CS8618: Non-nullable field 'field5' is uninitialized. Consider declaring the field as nullable.
                //     public Derived() : base() // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Derived").WithArguments("field", "field5").WithLocation(10, 12),
                // (23,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(23, 13),
                // (27,13): warning CS8602: Dereference of a possibly null reference.
                //             field8.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(27, 13)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_Inheritance_Property()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public Base() // 1
    {
        if (!Init)
            throw null!;
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2))]
    public virtual bool Init => throw null!;
}
public class Derived : Base
{
    public string field5;
    public string? field6;
    public string field7;
    public string? field8;

    public Derived() : base() // 3
    {
        if (!Init)
            throw null!;
    }

    void M()
    {
        if (Init)
        {
            field1.ToString();
            field2.ToString();
            field3.ToString();
            field4.ToString(); // 4
            field5.ToString();
            field6.ToString();
            field7.ToString();
            field8.ToString(); // 5
        }
    }

    [MemberNotNullWhen(true, nameof(field6), nameof(field7))]
    public override bool Init => throw null!;
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,12): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public Base() // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Base").WithArguments("field", "field3").WithLocation(10, 12),
                // (26,12): warning CS8618: Non-nullable field 'field5' is uninitialized. Consider declaring the field as nullable.
                //     public Derived() : base() // 3
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Derived").WithArguments("field", "field5").WithLocation(26, 12),
                // (39,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(39, 13),
                // (43,13): warning CS8602: Dereference of a possibly null reference.
                //             field8.ToString(); // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(43, 13)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_Inheritance_Property_PE()
        {
            var libComp = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public Base()
    {
        if (!Init)
            throw null!;
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2))]
    public virtual bool Init => throw null!;
}", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            var c = CreateNullableCompilation(@"
using System.Diagnostics.CodeAnalysis;
public class Derived : Base
{
    public string field5;
    public string? field6;
    public string field7;
    public string? field8;

    public Derived() : base() // 1
    {
        if (!Init)
            throw null!;
    }

    void M()
    {
        if (Init)
        {
            field1.ToString();
            field2.ToString();
            field3.ToString();
            field4.ToString(); // 2
            field5.ToString();
            field6.ToString();
            field7.ToString();
            field8.ToString(); // 3
        }
    }

    [MemberNotNullWhen(true, nameof(field6), nameof(field7))]
    public override bool Init => throw null!;
}
", references: new[] { libComp.EmitToImageReference() }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,12): warning CS8618: Non-nullable field 'field5' is uninitialized. Consider declaring the field as nullable.
                //     public Derived() : base() // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Derived").WithArguments("field", "field5").WithLocation(10, 12),
                // (23,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(23, 13),
                // (27,13): warning CS8602: Dereference of a possibly null reference.
                //             field8.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field8").WithLocation(27, 13)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_Inheritance_New()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class Base
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public Base() // 1, 2
    {
        Init();
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2))]
    public virtual bool Init() => throw null!;
}
public class Derived : Base
{
    public new string field1;
    public new string? field2;
    public new string field3;
    public new string? field4;

    public Derived() : base() // 3, 4
    {
        Init();
    }

    void M()
    {
        Init();
        field1.ToString();
        field2.ToString(); // 5
        field3.ToString();
        field4.ToString(); // 6
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2))]
    public override bool Init() => throw null!;
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (10,12): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public Base() // 1, 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Base").WithArguments("field", "field3").WithLocation(10, 12),
                // (10,12): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public Base() // 1, 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Base").WithArguments("field", "field1").WithLocation(10, 12),
                // (25,12): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public Derived() : base() // 3, 4
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Derived").WithArguments("field", "field3").WithLocation(25, 12),
                // (25,12): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public Derived() : base() // 3, 4
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "Derived").WithArguments("field", "field1").WithLocation(25, 12),
                // (34,9): warning CS8602: Dereference of a possibly null reference.
                //         field2.ToString(); // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(34, 9),
                // (36,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 6
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(36, 9)
                );
        }

        [Fact]
        public void MemberNotNull_InstanceMethodValidatingStaticFields()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public static string field1; // 1
    public static string? field2;
    public static string field3; // 2
    public static string? field4;

    public void M()
    {
        if (Init())
        {
            field1.ToString();
            field2.ToString();
            field3.ToString();
            field4.ToString(); // 3
        }
        else
        {
            field1.ToString();
            field2.ToString();
            field3.ToString();
            field4.ToString(); // 4
        }
    }

    [MemberNotNull(nameof(field1), nameof(field2))]
    bool Init() => throw null!;
}
", MemberNotNullAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (5,26): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public static string field1; // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field1").WithArguments("field", "field1").WithLocation(5, 26),
                // (7,26): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public static string field3; // 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field3").WithArguments("field", "field3").WithLocation(7, 26),
                // (17,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(17, 13),
                // (24,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(24, 13)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_InstanceMethodValidatingStaticFields()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public static string field1; // 1
    public static string? field2;
    public static string field3; // 2
    public static string? field4;

    public void M()
    {
        if (Init())
        {
            field1.ToString();
            field2.ToString();
            field3.ToString();
            field4.ToString(); // 3
        }
        else
        {
            field1.ToString();
            field2.ToString(); // 4
            field3.ToString();
            field4.ToString(); // 5
        }
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2))]
    bool Init() => throw null!;
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (5,26): warning CS8618: Non-nullable field 'field1' is uninitialized. Consider declaring the field as nullable.
                //     public static string field1; // 1
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field1").WithArguments("field", "field1").WithLocation(5, 26),
                // (7,26): warning CS8618: Non-nullable field 'field3' is uninitialized. Consider declaring the field as nullable.
                //     public static string field3; // 2
                Diagnostic(ErrorCode.WRN_UninitializedNonNullableField, "field3").WithArguments("field", "field3").WithLocation(7, 26),
                // (17,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(17, 13),
                // (22,13): warning CS8602: Dereference of a possibly null reference.
                //             field2.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(22, 13),
                // (24,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(24, 13)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_Property()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        if (IsInit)
        {
            field1.ToString();
            field2.ToString();
            field3.ToString(); // 1
            field4.ToString(); // 2
        }
        else
        {
            field1.ToString(); // 3
            field2.ToString(); // 4
            field3.ToString(); // 5
            field4.ToString(); // 6
            throw null!;
        }
    }

    public C(bool b)
    {
        IsInit = b;
        field1.ToString(); // 7
        field2.ToString(); // 8
        field3.ToString(); // 9
        field4.ToString(); // 10
    }

    [MemberNotNullWhen(true, nameof(field1), nameof(field2))]
    bool IsInit { get => throw null!; set => throw null!; }
}
", MemberNotNullWhenAttributeDefinition }, parseOptions: TestOptions.Regular9);

            c.VerifyDiagnostics(
                // (16,13): warning CS8602: Dereference of a possibly null reference.
                //             field3.ToString(); // 1
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(16, 13),
                // (17,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 2
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(17, 13),
                // (21,13): warning CS8602: Dereference of a possibly null reference.
                //             field1.ToString(); // 3
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field1").WithLocation(21, 13),
                // (22,13): warning CS8602: Dereference of a possibly null reference.
                //             field2.ToString(); // 4
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(22, 13),
                // (23,13): warning CS8602: Dereference of a possibly null reference.
                //             field3.ToString(); // 5
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(23, 13),
                // (24,13): warning CS8602: Dereference of a possibly null reference.
                //             field4.ToString(); // 6
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(24, 13),
                // (32,9): warning CS8602: Dereference of a possibly null reference.
                //         field1.ToString(); // 7
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field1").WithLocation(32, 9),
                // (33,9): warning CS8602: Dereference of a possibly null reference.
                //         field2.ToString(); // 8
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field2").WithLocation(33, 9),
                // (34,9): warning CS8602: Dereference of a possibly null reference.
                //         field3.ToString(); // 9
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field3").WithLocation(34, 9),
                // (35,9): warning CS8602: Dereference of a possibly null reference.
                //         field4.ToString(); // 10
                Diagnostic(ErrorCode.WRN_NullReferenceReceiver, "field4").WithLocation(35, 9)
                );
        }

        [Fact]
        public void MemberNotNullWhenTrue_Property_AppliedOnGetter()
        {
            var c = CreateNullableCompilation(new[] { @"
using System.Diagnostics.CodeAnalysis;
public class C
{
    public string field1;
    public string? field2;
    public string field3;
    public string? field4;

    public C()
    {
        if (IsInit)
        {
            field1.ToString();
            field2.ToString();
            field3.ToString(); // 1
            field4.ToString(); // 2
        }
        else