// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using Xunit;

namespace Microsoft.Extensions.Configuration.Binder.Test
{
    public class ConfigurationCollectionBinding
    {
        [Fact]
        public void GetList()
        {
            var input = new Dictionary<string, string>
            {
                {"StringList:0", "val0"},
                {"StringList:1", "val1"},
                {"StringList:2", "val2"},
                {"StringList:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();
            var list = new List<string>();
            config.GetSection("StringList").Bind(list);

            Assert.Equal(4, list.Count);

            Assert.Equal("val0", list[0]);
            Assert.Equal("val1", list[1]);
            Assert.Equal("val2", list[2]);
            Assert.Equal("valx", list[3]);
        }

        [Fact]
        public void GetListNullValues()
        {
            var input = new Dictionary<string, string>
            {
                {"StringList:0", null},
                {"StringList:1", null},
                {"StringList:2", null},
                {"StringList:x", null}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();
            var list = new List<string>();
            config.GetSection("StringList").Bind(list);

            Assert.Empty(list);
        }

        [Fact]
        public void GetListInvalidValues()
        {
            var input = new Dictionary<string, string>
            {
                {"InvalidList:0", "true"},
                {"InvalidList:1", "invalid"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();
            var list = new List<bool>();
            config.GetSection("InvalidList").Bind(list);

            Assert.Single(list);
            Assert.True(list[0]);
        }

        [Fact]
        public void GetDictionaryInvalidValues()
        {
            var input = new Dictionary<string, string>
            {
                {"InvalidDictionary:0", "true"},
                {"InvalidDictionary:1", "invalid"},
            };
            var config = new ConfigurationBuilder().AddInMemoryCollection(input).Build();
            var dict = new Dictionary<string, bool>();

            config.Bind("InvalidDictionary", dict);

            Assert.Single(dict);
            Assert.True(dict["0"]);
        }

        [Fact]
        public void BindList()
        {
            var input = new Dictionary<string, string>
            {
                {"StringList:0", "val0"},
                {"StringList:1", "val1"},
                {"StringList:2", "val2"},
                {"StringList:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var list = new List<string>();
            config.GetSection("StringList").Bind(list);

            Assert.Equal(4, list.Count);

            Assert.Equal("val0", list[0]);
            Assert.Equal("val1", list[1]);
            Assert.Equal("val2", list[2]);
            Assert.Equal("valx", list[3]);
        }

        [Fact]
        public void GetObjectList()
        {
            var input = new Dictionary<string, string>
            {
                {"ObjectList:0:Integer", "30"},
                {"ObjectList:1:Integer", "31"},
                {"ObjectList:2:Integer", "32"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new List<NestedOptions>();
            config.GetSection("ObjectList").Bind(options);

            Assert.Equal(3, options.Count);

            Assert.Equal(30, options[0].Integer);
            Assert.Equal(31, options[1].Integer);
            Assert.Equal(32, options[2].Integer);
        }

        [Fact]
        public void GetStringDictionary()
        {
            var input = new Dictionary<string, string>
            {
                {"StringDictionary:abc", "val_1"},
                {"StringDictionary:def", "val_2"},
                {"StringDictionary:ghi", "val_3"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new Dictionary<string, string>();
            config.GetSection("StringDictionary").Bind(options);

            Assert.Equal(3, options.Count);

            Assert.Equal("val_1", options["abc"]);
            Assert.Equal("val_2", options["def"]);
            Assert.Equal("val_3", options["ghi"]);
        }

        [Fact]
        public void GetEnumDictionary()
        {
            var input = new Dictionary<string, string>
            {
                {"EnumDictionary:abc", "val_1"},
                {"EnumDictionary:def", "val_2"},
                {"EnumDictionary:ghi", "val_3"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new Dictionary<KeyEnum, string>();
            config.GetSection("EnumDictionary").Bind(options);

            Assert.Equal(3, options.Count);

            Assert.Equal("val_1", options[KeyEnum.abc]);
            Assert.Equal("val_2", options[KeyEnum.def]);
            Assert.Equal("val_3", options[KeyEnum.ghi]);
        }

        [Fact]
        public void GetUintEnumDictionary()
        {
            var input = new Dictionary<string, string>
            {
                {"EnumDictionary:abc", "val_1"},
                {"EnumDictionary:def", "val_2"},
                {"EnumDictionary:ghi", "val_3"}
            };
            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();
            var options = new Dictionary<KeyUintEnum, string>();
            config.GetSection("EnumDictionary").Bind(options);
            Assert.Equal(3, options.Count);
            Assert.Equal("val_1", options[KeyUintEnum.abc]);
            Assert.Equal("val_2", options[KeyUintEnum.def]);
            Assert.Equal("val_3", options[KeyUintEnum.ghi]);
        }

        [Fact]
        public void GetSByteDictionary()
        {
            GetIntDictionaryT<sbyte>(0, 1, 2);
        }

        [Fact]
        public void GetByteDictionary()
        {
            GetIntDictionaryT<byte>(0, 1, 2);
        }

        [Fact]
        public void GetShortDictionary()
        {
            GetIntDictionaryT<short>(0, 1, 2);
        }

        [Fact]
        public void GetUShortDictionary()
        {
            GetIntDictionaryT<ushort>(0, 1, 2);
        }

        [Fact]
        public void GetIntDictionary()
        {
            GetIntDictionaryT<int>(0, 1, 2);
        }

        [Fact]
        public void GetUIntDictionary()
        {
            GetIntDictionaryT<uint>(0, 1, 2);
        }

        [Fact]
        public void GetLongDictionary()
        {
            GetIntDictionaryT<long>(0, 1, 2);
        }

        [Fact]
        public void GetULongDictionary()
        {
            GetIntDictionaryT<ulong>(0, 1, 2);
        }

        private void GetIntDictionaryT<T>(T k1, T k2, T k3)
        {
            var input = new Dictionary<string, string>
            {
                {"IntegerKeyDictionary:0", "val_0"},
                {"IntegerKeyDictionary:1", "val_1"},
                {"IntegerKeyDictionary:2", "val_2"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new Dictionary<T, string>();
            config.GetSection("IntegerKeyDictionary").Bind(options);

            Assert.Equal(3, options.Count);

            Assert.Equal("val_0", options[k1]);
            Assert.Equal("val_1", options[k2]);
            Assert.Equal("val_2", options[k3]);
        }

        [Fact]
        public void BindStringList()
        {
            var input = new Dictionary<string, string>
            {
                {"StringList:0", "val0"},
                {"StringList:1", "val1"},
                {"StringList:2", "val2"},
                {"StringList:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();
            var options = new OptionsWithLists();
            config.Bind(options);

            var list = options.StringList;

            Assert.Equal(4, list.Count);

            Assert.Equal("val0", list[0]);
            Assert.Equal("val1", list[1]);
            Assert.Equal("val2", list[2]);
            Assert.Equal("valx", list[3]);
        }

        [Fact]
        public void BindIntList()
        {
            var input = new Dictionary<string, string>
            {
                {"IntList:0", "42"},
                {"IntList:1", "43"},
                {"IntList:2", "44"},
                {"IntList:x", "45"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithLists();
            config.Bind(options);

            var list = options.IntList;

            Assert.Equal(4, list.Count);

            Assert.Equal(42, list[0]);
            Assert.Equal(43, list[1]);
            Assert.Equal(44, list[2]);
            Assert.Equal(45, list[3]);
        }

        [Fact]
        public void AlreadyInitializedListBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"AlreadyInitializedList:0", "val0"},
                {"AlreadyInitializedList:1", "val1"},
                {"AlreadyInitializedList:2", "val2"},
                {"AlreadyInitializedList:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithLists();
            config.Bind(options);

            var list = options.AlreadyInitializedList;

            Assert.Equal(5, list.Count);

            Assert.Equal("This was here before", list[0]);
            Assert.Equal("val0", list[1]);
            Assert.Equal("val1", list[2]);
            Assert.Equal("val2", list[3]);
            Assert.Equal("valx", list[4]);
        }

        [Fact]
        public void AlreadyInitializedListInterfaceBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"AlreadyInitializedListInterface:0", "val0"},
                {"AlreadyInitializedListInterface:1", "val1"},
                {"AlreadyInitializedListInterface:2", "val2"},
                {"AlreadyInitializedListInterface:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithLists();
            config.Bind(options);

            var list = options.AlreadyInitializedListInterface;

            Assert.Equal(5, list.Count);

            Assert.Equal("This was here too", list[0]);
            Assert.Equal("val0", list[1]);
            Assert.Equal("val1", list[2]);
            Assert.Equal("val2", list[3]);
            Assert.Equal("valx", list[4]);

            // Ensure expandability of the returned list
            options.AlreadyInitializedListInterface.Add("ExtraItem");
            Assert.Equal(6, options.AlreadyInitializedListInterface.Count);
            Assert.Equal("ExtraItem", options.AlreadyInitializedListInterface[5]);
        }

        [Fact]
        public void CustomListBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"CustomList:0", "val0"},
                {"CustomList:1", "val1"},
                {"CustomList:2", "val2"},
                {"CustomList:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithLists();
            config.Bind(options);

            var list = options.CustomList;

            Assert.Equal(4, list.Count);

            Assert.Equal("val0", list[0]);
            Assert.Equal("val1", list[1]);
            Assert.Equal("val2", list[2]);
            Assert.Equal("valx", list[3]);
        }

        [Fact]
        public void ObjectListBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"ObjectList:0:Integer", "30"},
                {"ObjectList:1:Integer", "31"},
                {"ObjectList:2:Integer", "32"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithLists();
            config.Bind(options);

            Assert.Equal(3, options.ObjectList.Count);

            Assert.Equal(30, options.ObjectList[0].Integer);
            Assert.Equal(31, options.ObjectList[1].Integer);
            Assert.Equal(32, options.ObjectList[2].Integer);
        }

        [Fact]
        public void NestedListsBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"NestedLists:0:0", "val00"},
                {"NestedLists:0:1", "val01"},
                {"NestedLists:1:0", "val10"},
                {"NestedLists:1:1", "val11"},
                {"NestedLists:1:2", "val12"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithLists();
            config.Bind(options);

            Assert.Equal(2, options.NestedLists.Count);
            Assert.Equal(2, options.NestedLists[0].Count);
            Assert.Equal(3, options.NestedLists[1].Count);

            Assert.Equal("val00", options.NestedLists[0][0]);
            Assert.Equal("val01", options.NestedLists[0][1]);
            Assert.Equal("val10", options.NestedLists[1][0]);
            Assert.Equal("val11", options.NestedLists[1][1]);
            Assert.Equal("val12", options.NestedLists[1][2]);
        }

        [Fact]
        public void StringDictionaryBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"StringDictionary:abc", "val_1"},
                {"StringDictionary:def", "val_2"},
                {"StringDictionary:ghi", "val_3"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithDictionary();
            config.Bind(options);

            Assert.Equal(3, options.StringDictionary.Count);

            Assert.Equal("val_1", options.StringDictionary["abc"]);
            Assert.Equal("val_2", options.StringDictionary["def"]);
            Assert.Equal("val_3", options.StringDictionary["ghi"]);
        }

        [Fact]
        public void ShouldPreserveExistingKeysInDictionary()
        {
            var input = new Dictionary<string, string> { { "ascii:b", "98" } };
            var config = new ConfigurationBuilder().AddInMemoryCollection(input).Build();
            var origin = new Dictionary<string, int> { ["a"] = 97 };

            config.Bind("ascii", origin);

            Assert.Equal(2, origin.Count);
            Assert.Equal(97, origin["a"]);
            Assert.Equal(98, origin["b"]);
        }

        [Fact]
        public void ShouldPreserveExistingKeysInNestedDictionary()
        {
            var input = new Dictionary<string, string> { ["ascii:b"] = "98" };
            var config = new ConfigurationBuilder().AddInMemoryCollection(input).Build();
            var origin = new Dictionary<string, IDictionary<string, int>>
            {
                ["ascii"] = new Dictionary<string, int> { ["a"] = 97 }
            };

            config.Bind(origin);

            Assert.Equal(2, origin["ascii"].Count);
            Assert.Equal(97, origin["ascii"]["a"]);
            Assert.Equal(98, origin["ascii"]["b"]);
        }

        [Fact]
        public void ShouldPreserveExistingKeysInDictionaryWithEnumAsKeyType()
        {
            var input = new Dictionary<string, string>
            {
                ["abc:def"] = "val_2",
                ["abc:ghi"] = "val_3"
            };
            var config = new ConfigurationBuilder().AddInMemoryCollection(input).Build();
            var origin = new Dictionary<KeyEnum, IDictionary<KeyUintEnum, string>>
            {
                [KeyEnum.abc] = new Dictionary<KeyUintEnum, string> {  [KeyUintEnum.abc] = "val_1" }
            };

            config.Bind(origin);

            Assert.Equal(3, origin[KeyEnum.abc].Count);
            Assert.Equal("val_1", origin[KeyEnum.abc][KeyUintEnum.abc]);
            Assert.Equal("val_2", origin[KeyEnum.abc][KeyUintEnum.def]);
            Assert.Equal("val_3", origin[KeyEnum.abc][KeyUintEnum.ghi]);
        }

        [Fact]
        public void ShouldPreserveExistingValuesInArrayWhenItIsDictionaryElement()
        {
            var input = new Dictionary<string, string>
            {
                ["ascii:b"] = "98",
            };
            var config = new ConfigurationBuilder().AddInMemoryCollection(input).Build();
            var origin = new Dictionary<string, int[]>
            {
                ["ascii"] = new int[] { 97 }
            };

            config.Bind(origin);

            Assert.Equal(new int[] { 97, 98 }, origin["ascii"]);
        }

        [Fact]
        public void AlreadyInitializedStringDictionaryBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"AlreadyInitializedStringDictionaryInterface:abc", "val_1"},
                {"AlreadyInitializedStringDictionaryInterface:def", "val_2"},
                {"AlreadyInitializedStringDictionaryInterface:ghi", "val_3"},

                {"IDictionaryNoSetter:Key1", "Value1"},
                {"IDictionaryNoSetter:Key2", "Value2"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithDictionary();
            config.Bind(options);

            Assert.NotNull(options.AlreadyInitializedStringDictionaryInterface);
            Assert.Equal(4, options.AlreadyInitializedStringDictionaryInterface.Count);

            Assert.Equal("This was already here", options.AlreadyInitializedStringDictionaryInterface["123"]);
            Assert.Equal("val_1", options.AlreadyInitializedStringDictionaryInterface["abc"]);
            Assert.Equal("val_2", options.AlreadyInitializedStringDictionaryInterface["def"]);
            Assert.Equal("val_3", options.AlreadyInitializedStringDictionaryInterface["ghi"]);

            Assert.Equal(2, options.IDictionaryNoSetter.Count);
            Assert.Equal("Value1", options.IDictionaryNoSetter["Key1"]);
            Assert.Equal("Value2", options.IDictionaryNoSetter["Key2"]);
        }

        [Fact]
        public void AlreadyInitializedHashSetDictionaryBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"AlreadyInitializedHashSetDictionary:123:0", "val_1"},
                {"AlreadyInitializedHashSetDictionary:123:1", "val_2"},
                {"AlreadyInitializedHashSetDictionary:123:2", "val_3"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithDictionary();
            config.Bind(options);

            Assert.NotNull(options.AlreadyInitializedHashSetDictionary);
            Assert.Equal(1, options.AlreadyInitializedHashSetDictionary.Count);

            Assert.Equal("This was already here", options.AlreadyInitializedHashSetDictionary["123"].ElementAt(0));
            Assert.Equal("val_1", options.AlreadyInitializedHashSetDictionary["123"].ElementAt(1));
            Assert.Equal("val_2", options.AlreadyInitializedHashSetDictionary["123"].ElementAt(2));
            Assert.Equal("val_3", options.AlreadyInitializedHashSetDictionary["123"].ElementAt(3));
        }

        [Fact]
        public void CanOverrideExistingDictionaryKey()
        {
            var input = new Dictionary<string, string>
            {
                {"abc", "override"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new Dictionary<string, string>
            {
                {"abc", "default"}
            };

            config.Bind(options);

            var optionsCount = options.Count;

            Assert.Equal(1, optionsCount);
            Assert.Equal("override", options["abc"]);
        }

        [Fact]
        public void IntDictionaryBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"IntDictionary:abc", "42"},
                {"IntDictionary:def", "43"},
                {"IntDictionary:ghi", "44"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithDictionary();
            config.Bind(options);

            Assert.Equal(3, options.IntDictionary.Count);

            Assert.Equal(42, options.IntDictionary["abc"]);
            Assert.Equal(43, options.IntDictionary["def"]);
            Assert.Equal(44, options.IntDictionary["ghi"]);
        }

        [Fact]
        public void ObjectDictionary()
        {
            var input = new Dictionary<string, string>
            {
                {"ObjectDictionary:abc:Integer", "1"},
                {"ObjectDictionary:def:Integer", "2"},
                {"ObjectDictionary:ghi:Integer", "3"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithDictionary();
            config.Bind(options);

            Assert.Equal(3, options.ObjectDictionary.Count);

            Assert.Equal(1, options.ObjectDictionary["abc"].Integer);
            Assert.Equal(2, options.ObjectDictionary["def"].Integer);
            Assert.Equal(3, options.ObjectDictionary["ghi"].Integer);
        }

        [Fact]
        public void ListDictionary()
        {
            var input = new Dictionary<string, string>
            {
                {"ListDictionary:abc:0", "abc_0"},
                {"ListDictionary:abc:1", "abc_1"},
                {"ListDictionary:def:0", "def_0"},
                {"ListDictionary:def:1", "def_1"},
                {"ListDictionary:def:2", "def_2"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithDictionary();
            config.Bind(options);

            Assert.Equal(2, options.ListDictionary.Count);
            Assert.Equal(2, options.ListDictionary["abc"].Count);
            Assert.Equal(3, options.ListDictionary["def"].Count);

            Assert.Equal("abc_0", options.ListDictionary["abc"][0]);
            Assert.Equal("abc_1", options.ListDictionary["abc"][1]);
            Assert.Equal("def_0", options.ListDictionary["def"][0]);
            Assert.Equal("def_1", options.ListDictionary["def"][1]);
            Assert.Equal("def_2", options.ListDictionary["def"][2]);
        }

        [Fact]
        public void ISetDictionary()
        {
            var input = new Dictionary<string, string>
            {
                {"ISetDictionary:abc:0", "abc_0"},
                {"ISetDictionary:abc:1", "abc_1"},
                {"ISetDictionary:def:0", "def_0"},
                {"ISetDictionary:def:1", "def_1"},
                {"ISetDictionary:def:2", "def_2"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithDictionary();
            config.Bind(options);

            Assert.Equal(2, options.ISetDictionary.Count);
            Assert.Equal(2, options.ISetDictionary["abc"].Count);
            Assert.Equal(3, options.ISetDictionary["def"].Count);

            Assert.Equal("abc_0", options.ISetDictionary["abc"].ElementAt(0));
            Assert.Equal("abc_1", options.ISetDictionary["abc"].ElementAt(1));
            Assert.Equal("def_0", options.ISetDictionary["def"].ElementAt(0));
            Assert.Equal("def_1", options.ISetDictionary["def"].ElementAt(1));
            Assert.Equal("def_2", options.ISetDictionary["def"].ElementAt(2));
        }

        [Fact]
        public void ListInNestedOptionBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"ObjectList:0:ListInNestedOption:0", "00"},
                {"ObjectList:0:ListInNestedOption:1", "01"},
                {"ObjectList:1:ListInNestedOption:0", "10"},
                {"ObjectList:1:ListInNestedOption:1", "11"},
                {"ObjectList:1:ListInNestedOption:2", "12"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithLists();
            config.Bind(options);

            Assert.Equal(2, options.ObjectList.Count);
            Assert.Equal(2, options.ObjectList[0].ListInNestedOption.Count);
            Assert.Equal(3, options.ObjectList[1].ListInNestedOption.Count);

            Assert.Equal("00", options.ObjectList[0].ListInNestedOption[0]);
            Assert.Equal("01", options.ObjectList[0].ListInNestedOption[1]);
            Assert.Equal("10", options.ObjectList[1].ListInNestedOption[0]);
            Assert.Equal("11", options.ObjectList[1].ListInNestedOption[1]);
            Assert.Equal("12", options.ObjectList[1].ListInNestedOption[2]);
        }

        [Fact]
        public void NonStringKeyDictionaryBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"NonStringKeyDictionary:abc", "val_1"},
                {"NonStringKeyDictionary:def", "val_2"},
                {"NonStringKeyDictionary:ghi", "val_3"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithDictionary();
            config.Bind(options);

            Assert.Empty(options.NonStringKeyDictionary);
        }

        [Fact]
        public void GetStringArray()
        {
            var input = new Dictionary<string, string>
            {
                {"StringArray:0", "val0"},
                {"StringArray:1", "val1"},
                {"StringArray:2", "val2"},
                {"StringArray:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithArrays();
            config.Bind(options);

            var array = options.StringArray;

            Assert.Equal(4, array.Length);

            Assert.Equal("val0", array[0]);
            Assert.Equal("val1", array[1]);
            Assert.Equal("val2", array[2]);
            Assert.Equal("valx", array[3]);
        }


        [Fact]
        public void BindStringArray()
        {
            var input = new Dictionary<string, string>
            {
                {"StringArray:0", "val0"},
                {"StringArray:1", "val1"},
                {"StringArray:2", "val2"},
                {"StringArray:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var instance = new OptionsWithArrays();
            config.Bind(instance);

            var array = instance.StringArray;

            Assert.Equal(4, array.Length);

            Assert.Equal("val0", array[0]);
            Assert.Equal("val1", array[1]);
            Assert.Equal("val2", array[2]);
            Assert.Equal("valx", array[3]);
        }

        [Fact]
        public void GetAlreadyInitializedArray()
        {
            var input = new Dictionary<string, string>
            {
                {"AlreadyInitializedArray:0", "val0"},
                {"AlreadyInitializedArray:1", "val1"},
                {"AlreadyInitializedArray:2", "val2"},
                {"AlreadyInitializedArray:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithArrays();
            config.Bind(options);
            var array = options.AlreadyInitializedArray;

            Assert.Equal(7, array.Length);

            Assert.Equal(OptionsWithArrays.InitialValue, array[0]);
            Assert.Null(array[1]);
            Assert.Null(array[2]);
            Assert.Equal("val0", array[3]);
            Assert.Equal("val1", array[4]);
            Assert.Equal("val2", array[5]);
            Assert.Equal("valx", array[6]);
        }

        [Fact]
        public void BindAlreadyInitializedArray()
        {
            var input = new Dictionary<string, string>
            {
                {"AlreadyInitializedArray:0", "val0"},
                {"AlreadyInitializedArray:1", "val1"},
                {"AlreadyInitializedArray:2", "val2"},
                {"AlreadyInitializedArray:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithArrays();
            config.Bind(options);

            var array = options.AlreadyInitializedArray;

            Assert.Equal(7, array.Length);

            Assert.Equal(OptionsWithArrays.InitialValue, array[0]);
            Assert.Null(array[1]);
            Assert.Null(array[2]);
            Assert.Equal("val0", array[3]);
            Assert.Equal("val1", array[4]);
            Assert.Equal("val2", array[5]);
            Assert.Equal("valx", array[6]);
        }

        [Fact]
        public void ArrayInNestedOptionBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"ObjectArray:0:ArrayInNestedOption:0", "0"},
                {"ObjectArray:0:ArrayInNestedOption:1", "1"},
                {"ObjectArray:1:ArrayInNestedOption:0", "10"},
                {"ObjectArray:1:ArrayInNestedOption:1", "11"},
                {"ObjectArray:1:ArrayInNestedOption:2", "12"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();
            var options = new OptionsWithArrays();
            config.Bind(options);

            Assert.Equal(2, options.ObjectArray.Length);
            Assert.Equal(2, options.ObjectArray[0].ArrayInNestedOption.Length);
            Assert.Equal(3, options.ObjectArray[1].ArrayInNestedOption.Length);

            Assert.Equal(0, options.ObjectArray[0].ArrayInNestedOption[0]);
            Assert.Equal(1, options.ObjectArray[0].ArrayInNestedOption[1]);
            Assert.Equal(10, options.ObjectArray[1].ArrayInNestedOption[0]);
            Assert.Equal(11, options.ObjectArray[1].ArrayInNestedOption[1]);
            Assert.Equal(12, options.ObjectArray[1].ArrayInNestedOption[2]);
        }

        [Fact]
        public void UnsupportedMultidimensionalArrays()
        {
            var input = new Dictionary<string, string>
            {
                {"DimensionalArray:0:0", "a"},
                {"DimensionalArray:0:1", "b"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();
            var options = new OptionsWithArrays();

            var exception = Assert.Throws<InvalidOperationException>(
                () => config.Bind(options));
            Assert.Equal(
                SR.Format(SR.Error_UnsupportedMultidimensionalArray, typeof(string[,])),
                exception.Message);
        }

        [Fact]
        public void JaggedArrayBinding()
        {
            var input = new Dictionary<string, string>
            {
                {"JaggedArray:0:0", "00"},
                {"JaggedArray:0:1", "01"},
                {"JaggedArray:1:0", "10"},
                {"JaggedArray:1:1", "11"},
                {"JaggedArray:1:2", "12"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();
            var options = new OptionsWithArrays();
            config.Bind(options);

            Assert.Equal(2, options.JaggedArray.Length);
            Assert.Equal(2, options.JaggedArray[0].Length);
            Assert.Equal(3, options.JaggedArray[1].Length);

            Assert.Equal("00", options.JaggedArray[0][0]);
            Assert.Equal("01", options.JaggedArray[0][1]);
            Assert.Equal("10", options.JaggedArray[1][0]);
            Assert.Equal("11", options.JaggedArray[1][1]);
            Assert.Equal("12", options.JaggedArray[1][2]);
        }

        [Fact]
        public void ReadOnlyArrayIsIgnored()
        {
            var input = new Dictionary<string, string>
            {
                {"ReadOnlyArray:0", "10"},
                {"ReadOnlyArray:1", "20"},
                {"ReadOnlyArray:2", "30"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();
            var options = new OptionsWithArrays();
            config.Bind(options);

            Assert.Equal(new OptionsWithArrays().ReadOnlyArray, options.ReadOnlyArray);
        }

        [Fact]
        public void CanBindUninitializedIEnumerable()
        {
            var input = new Dictionary<string, string>
            {
                {"IEnumerable:0", "val0"},
                {"IEnumerable:1", "val1"},
                {"IEnumerable:2", "val2"},
                {"IEnumerable:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new UninitializedCollectionsOptions();
            config.Bind(options);

            var array = options.IEnumerable.ToArray();

            Assert.Equal(4, array.Length);

            Assert.Equal("val0", array[0]);
            Assert.Equal("val1", array[1]);
            Assert.Equal("val2", array[2]);
            Assert.Equal("valx", array[3]);
        }

        [Fact]
        public void CanBindInitializedIEnumerableAndTheOriginalItemsAreNotMutated()
        {
            var input = new Dictionary<string, string>
            {
                {"AlreadyInitializedIEnumerableInterface:0", "val0"},
                {"AlreadyInitializedIEnumerableInterface:1", "val1"},
                {"AlreadyInitializedIEnumerableInterface:2", "val2"},
                {"AlreadyInitializedIEnumerableInterface:x", "valx"},

                {"ICollectionNoSetter:0", "val0"},
                {"ICollectionNoSetter:1", "val1"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new InitializedCollectionsOptions();
            config.Bind(options);

            var array = options.AlreadyInitializedIEnumerableInterface.ToArray();

            Assert.Equal(6, array.Length);

            Assert.Equal("This was here too", array[0]);
            Assert.Equal("Don't touch me!", array[1]);
            Assert.Equal("val0", array[2]);
            Assert.Equal("val1", array[3]);
            Assert.Equal("val2", array[4]);
            Assert.Equal("valx", array[5]);

            // the original list hasn't been touched
            Assert.Equal(2, options.ListUsedInIEnumerableFieldAndShouldNotBeTouched.Count);
            Assert.Equal("This was here too", options.ListUsedInIEnumerableFieldAndShouldNotBeTouched.ElementAt(0));
            Assert.Equal("Don't touch me!", options.ListUsedInIEnumerableFieldAndShouldNotBeTouched.ElementAt(1));

            Assert.Equal(2, options.ICollectionNoSetter.Count);
            Assert.Equal("val0", options.ICollectionNoSetter.ElementAt(0));
            Assert.Equal("val1", options.ICollectionNoSetter.ElementAt(1));

            // Ensure expandability of the returned collection
            options.ICollectionNoSetter.Add("ExtraItem");
            Assert.Equal(3, options.ICollectionNoSetter.Count);
            Assert.Equal("ExtraItem", options.ICollectionNoSetter.ElementAt(2));
        }

        [Fact]
        public void CanBindInitializedCustomIEnumerableBasedList()
        {
            // A field declared as IEnumerable<T> that is instantiated with a class
            // that directly implements IEnumerable<T> is still bound, but with
            // a new List<T> with the original values copied over.

            var input = new Dictionary<string, string>
            {
                {"AlreadyInitializedCustomListDerivedFromIEnumerable:0", "val0"},
                {"AlreadyInitializedCustomListDerivedFromIEnumerable:1", "val1"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new InitializedCollectionsOptions();
            config.Bind(options);

            var array = options.AlreadyInitializedCustomListDerivedFromIEnumerable.ToArray();

            Assert.Equal(4, array.Length);

            Assert.Equal("Item1", array[0]);
            Assert.Equal("Item2", array[1]);
            Assert.Equal("val0", array[2]);
            Assert.Equal("val1", array[3]);
        }

        [Fact]
        public void CanBindInitializedCustomIndirectlyDerivedIEnumerableList()
        {
            // A field declared as IEnumerable<T> that is instantiated with a class
            // that indirectly implements IEnumerable<T> is still bound, but with
            // a new List<T> with the original values copied over.

            var input = new Dictionary<string, string>
            {
                {"AlreadyInitializedCustomListIndirectlyDerivedFromIEnumerable:0", "val0"},
                {"AlreadyInitializedCustomListIndirectlyDerivedFromIEnumerable:1", "val1"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new InitializedCollectionsOptions();
            config.Bind(options);

            var array = options.AlreadyInitializedCustomListIndirectlyDerivedFromIEnumerable.ToArray();

            Assert.Equal(4, array.Length);

            Assert.Equal("Item1", array[0]);
            Assert.Equal("Item2", array[1]);
            Assert.Equal("val0", array[2]);
            Assert.Equal("val1", array[3]);
        }

        [Fact]
        public void CanBindInitializedIReadOnlyDictionaryAndDoesNotModifyTheOriginal()
        {
            // A field declared as IEnumerable<T> that is instantiated with a class
            // that indirectly implements IEnumerable<T> is still bound, but with
            // a new List<T> with the original values copied over.

            var input = new Dictionary<string, string>
            {
                {"AlreadyInitializedDictionary:existing_key_1", "overridden!"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new InitializedCollectionsOptions();
            config.Bind(options);

            var array = options.AlreadyInitializedDictionary.ToArray();

            Assert.Equal(2, array.Length);

            Assert.Equal("overridden!", options.AlreadyInitializedDictionary["existing_key_1"]);
            Assert.Equal("val_2", options.AlreadyInitializedDictionary["existing_key_2"]);

            Assert.NotEqual(options.AlreadyInitializedDictionary, InitializedCollectionsOptions.ExistingDictionary);

            Assert.Equal("val_1", InitializedCollectionsOptions.ExistingDictionary["existing_key_1"]);
            Assert.Equal("val_2", InitializedCollectionsOptions.ExistingDictionary["existing_key_2"]);
        }

        [Fact]
        public void CanBindUninitializedICollection()
        {
            var input = new Dictionary<string, string>
            {
                {"ICollection:0", "val0"},
                {"ICollection:1", "val1"},
                {"ICollection:2", "val2"},
                {"ICollection:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new UninitializedCollectionsOptions();
            config.Bind(options);

            var array = options.ICollection.ToArray();

            Assert.Equal(4, array.Length);

            Assert.Equal("val0", array[0]);
            Assert.Equal("val1", array[1]);
            Assert.Equal("val2", array[2]);
            Assert.Equal("valx", array[3]);

            // Ensure expandability of the returned collection
            options.ICollection.Add("ExtraItem");
            Assert.Equal(5, options.ICollection.Count);
            Assert.Equal("ExtraItem", options.ICollection.ElementAt(4));
        }

        [Fact]
        public void CanBindUninitializedIList()
        {
            var input = new Dictionary<string, string>
            {
                {"IList:0", "val0"},
                {"IList:1", "val1"},
                {"IList:2", "val2"},
                {"IList:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new UninitializedCollectionsOptions();
            config.Bind(options);

            IList<string> list = options.IList;

            Assert.Equal(4, list.Count);

            Assert.Equal("val0", list[0]);
            Assert.Equal("val1", list[1]);
            Assert.Equal("val2", list[2]);
            Assert.Equal("valx", list[3]);

            // Ensure expandability of the returned list
            options.IList.Add("ExtraItem");
            Assert.Equal(5, options.IList.Count);
            Assert.Equal("ExtraItem", options.IList[4]);
        }

        [Fact]
        public void CanBindUninitializedIReadOnlyCollection()
        {
            var input = new Dictionary<string, string>
            {
                {"IReadOnlyCollection:0", "val0"},
                {"IReadOnlyCollection:1", "val1"},
                {"IReadOnlyCollection:2", "val2"},
                {"IReadOnlyCollection:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new UninitializedCollectionsOptions();
            config.Bind(options);

            var array = options.IReadOnlyCollection.ToArray();

            Assert.Equal(4, array.Length);

            Assert.Equal("val0", array[0]);
            Assert.Equal("val1", array[1]);
            Assert.Equal("val2", array[2]);
            Assert.Equal("valx", array[3]);
        }

        [Fact]
        public void CanBindUninitializedIReadOnlyList()
        {
            var input = new Dictionary<string, string>
            {
                {"IReadOnlyList:0", "val0"},
                {"IReadOnlyList:1", "val1"},
                {"IReadOnlyList:2", "val2"},
                {"IReadOnlyList:x", "valx"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new UninitializedCollectionsOptions();
            config.Bind(options);

            var array = options.IReadOnlyList.ToArray();

            Assert.Equal(4, array.Length);

            Assert.Equal("val0", array[0]);
            Assert.Equal("val1", array[1]);
            Assert.Equal("val2", array[2]);
            Assert.Equal("valx", array[3]);
        }

        [Fact]
        public void CanBindUninitializedIDictionary()
        {
            var input = new Dictionary<string, string>
            {
                {"IDictionary:abc", "val_1"},
                {"IDictionary:def", "val_2"},
                {"IDictionary:ghi", "val_3"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new UninitializedCollectionsOptions();
            config.Bind(options);

            Assert.Equal(3, options.IDictionary.Count);

            Assert.Equal("val_1", options.IDictionary["abc"]);
            Assert.Equal("val_2", options.IDictionary["def"]);
            Assert.Equal("val_3", options.IDictionary["ghi"]);
        }

        [Fact]
        public void CanBindUninitializedIReadOnlyDictionary()
        {
            var input = new Dictionary<string, string>
            {
                {"IReadOnlyDictionary:abc", "val_1"},
                {"IReadOnlyDictionary:def", "val_2"},
                {"IReadOnlyDictionary:ghi", "val_3"}
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new UninitializedCollectionsOptions();
            config.Bind(options);

            Assert.Equal(3, options.IReadOnlyDictionary.Count);

            Assert.Equal("val_1", options.IReadOnlyDictionary["abc"]);
            Assert.Equal("val_2", options.IReadOnlyDictionary["def"]);
            Assert.Equal("val_3", options.IReadOnlyDictionary["ghi"]);
        }

        /// <summary>
        /// Replicates scenario from https://github.com/dotnet/runtime/issues/65710
        /// </summary>
        [Fact]
        public void CanBindWithInterdependentProperties()
        {
            var input = new Dictionary<string, string>
            {
                {"ConfigValues:0", "5"},
                {"ConfigValues:1", "50"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithInterdependentProperties();
            config.Bind(options);

            Assert.Equal(new[] { 5, 50 }, options.ConfigValues);
            Assert.Equal(new[] { 50 }, options.FilteredConfigValues);
        }

        /// <summary>
        /// Replicates scenario from https://github.com/dotnet/runtime/issues/63479
        /// </summary>
        [Fact]
        public void TestCanBindListPropertyWithoutSetter()
        {
            var input = new Dictionary<string, string>
            {
                {"ListPropertyWithoutSetter:0", "a"},
                {"ListPropertyWithoutSetter:1", "b"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithLists();
            config.Bind(options);

            Assert.Equal(new[] { "a", "b" }, options.ListPropertyWithoutSetter);
        }

        private class UninitializedCollectionsOptions
        {
            public IEnumerable<string> IEnumerable { get; set; }
            public IDictionary<string, string> IDictionary { get; set; }
            public ICollection<string> ICollection { get; set; }
            public IList<string> IList { get; set; }
            public IReadOnlyCollection<string> IReadOnlyCollection { get; set; }
            public IReadOnlyList<string> IReadOnlyList { get; set; }
            public IReadOnlyDictionary<string, string> IReadOnlyDictionary { get; set; }
        }

        private class InitializedCollectionsOptions
        {
            public InitializedCollectionsOptions()
            {
                AlreadyInitializedIEnumerableInterface = ListUsedInIEnumerableFieldAndShouldNotBeTouched;
                AlreadyInitializedDictionary = ExistingDictionary;
            }

            public List<string> ListUsedInIEnumerableFieldAndShouldNotBeTouched = new()
            {
                "This was here too",
                "Don't touch me!"
            };

            public static ReadOnlyDictionary<string, string> ExistingDictionary = new(
                new Dictionary<string, string>
                {
                    {"existing_key_1", "val_1"},
                    {"existing_key_2", "val_2"}
                });

            public IEnumerable<string> AlreadyInitializedIEnumerableInterface { get; set; }

            public IEnumerable<string> AlreadyInitializedCustomListDerivedFromIEnumerable { get; set; } =
                new CustomListDerivedFromIEnumerable();

            public IEnumerable<string> AlreadyInitializedCustomListIndirectlyDerivedFromIEnumerable { get; set; } =
                new CustomListIndirectlyDerivedFromIEnumerable();

            public IReadOnlyDictionary<string, string> AlreadyInitializedDictionary { get; set; }

            public ICollection<string> ICollectionNoSetter { get; } = new List<string>();
        }

        private class CustomList : List<string>
        {
            // Add an overload, just to make sure binding picks the right Add method
            public void Add(string a, string b)
            {
            }
        }

        private class CustomListDerivedFromIEnumerable : IEnumerable<string>
        {
            private readonly List<string> _items = new List<string> { "Item1", "Item2" };

            public IEnumerator<string> GetEnumerator() => _items.GetEnumerator();

            IEnumerator IEnumerable.GetEnumerator() => GetEnumerator();
        }

        internal interface IDerivedOne : IDerivedTwo
        {
        }

        internal interface IDerivedTwo : IEnumerable<string>
        {
        }

        private class CustomListIndirectlyDerivedFromIEnumerable : IDerivedOne
        {
            private readonly List<string> _items = new List<string> { "Item1", "Item2" };

            public IEnumerator<string> GetEnumerator() => _items.GetEnumerator();

            IEnumerator IEnumerable.GetEnumerator() => GetEnumerator();
        }

        private class CustomDictionary<T> : Dictionary<string, T>
        {
        }

        private class NestedOptions
        {
            public int Integer { get; set; }

            public List<string> ListInNestedOption { get; set; }

            public int[] ArrayInNestedOption { get; set; }
        }

        private enum KeyEnum
        {
            abc,
            def,
            ghi
        }

        private enum KeyUintEnum : uint
        {
            abc,
            def,
            ghi
        }

        private class OptionsWithArrays
        {
            public const string InitialValue = "This was here before";

            public OptionsWithArrays()
            {
                AlreadyInitializedArray = new string[] { InitialValue, null, null };
            }

            public string[] AlreadyInitializedArray { get; set; }

            public string[] StringArray { get; set; }

            // this should throw because we do not support multidimensional arrays
            public string[,] DimensionalArray { get; set; }

            public string[][] JaggedArray { get; set; }

            public NestedOptions[] ObjectArray { get; set; }

            public int[] ReadOnlyArray { get; } = new[] { 1, 2 };
        }

        private class OptionsWithLists
        {
            public OptionsWithLists()
            {
                AlreadyInitializedList = new List<string>
                {
                    "This was here before"
                };
                AlreadyInitializedListInterface = new List<string>
                {
                    "This was here too"
                };
            }

            public CustomList CustomList { get; set; }

            public List<string> StringList { get; set; }

            public List<int> IntList { get; set; }

            // This cannot be initialized because we cannot
            // activate an interface
            public IList<string> StringListInterface { get; set; }

            public List<List<string>> NestedLists { get; set; }

            public List<string> AlreadyInitializedList { get; set; }

            public List<NestedOptions> ObjectList { get; set; }

            public IList<string> AlreadyInitializedListInterface { get; set; }

            public List<string> ListPropertyWithoutSetter { get; } = new();
        }

        private class OptionsWithDictionary
        {
            public OptionsWithDictionary()
            {
                AlreadyInitializedStringDictionaryInterface = new Dictionary<string, string>
                {
                    ["123"] = "This was already here"
                };

                AlreadyInitializedHashSetDictionary = new Dictionary<string, HashSet<string>>
                {
                    ["123"] = new HashSet<string>(new[] {"This was already here"})
                };
            }

            public Dictionary<string, int> IntDictionary { get; set; }

            public Dictionary<string, string> StringDictionary { get; set; }

            public IDictionary<string, string> IDictionaryNoSetter { get; } = new Dictionary<string, string>();

            public Dictionary<string, NestedOptions> ObjectDictionary { get; set; }

            public Dictionary<string, ISet<string>> ISetDictionary { get; set; }
            public Dictionary<string, List<string>> ListDictionary { get; set; }

            public Dictionary<NestedOptions, string> NonStringKeyDictionary { get; set; }

            // This cannot be initialized because we cannot
            // activate an interface
            public IDictionary<string, string> StringDictionaryInterface { get; set; }

            public IDictionary<string, string> AlreadyInitializedStringDictionaryInterface { get; set; }
            public IDictionary<string, HashSet<string>> AlreadyInitializedHashSetDictionary { get; set; }
        }

        private class OptionsWithInterdependentProperties
        {
            public IEnumerable<int> FilteredConfigValues => ConfigValues.Where(p => p > 10);
            public IEnumerable<int> ConfigValues { get; set; }
        }

        [Fact]
        public void DifferentDictionaryBindingCasesTest()
        {
            var dic = new Dictionary<string, string>() { { "key", "value" } };
            var config = new ConfigurationBuilder()
                .AddInMemoryCollection(dic)
                .Build();

            Assert.Single(config.Get<Dictionary<string, string>>());
            Assert.Single(config.Get<IDictionary<string, string>>());
            Assert.Single(config.Get<ExtendedDictionary<string, string>>());
            Assert.Single(config.Get<ImplementerOfIDictionaryClass<string, string>>());
        }

        public class ImplementerOfIDictionaryClass<TKey, TValue> : IDictionary<TKey, TValue>
        {
            private Dictionary<TKey, TValue> _dict = new();

            public TValue this[TKey key] { get => _dict[key]; set => _dict[key] = value; }

            public ICollection<TKey> Keys => _dict.Keys;

            public ICollection<TValue> Values => _dict.Values;

            public int Count => _dict.Count;

            public bool IsReadOnly => false;

            public void Add(TKey key, TValue value) => _dict.Add(key, value);

            public void Add(KeyValuePair<TKey, TValue> item) => _dict.Add(item.Key, item.Value);

            public void Clear() => _dict.Clear();

            public bool Contains(KeyValuePair<TKey, TValue> item) => _dict.Contains(item);

            public bool ContainsKey(TKey key) => _dict.ContainsKey(key);

            public void CopyTo(KeyValuePair<TKey, TValue>[] array, int arrayIndex) => throw new NotImplementedException();

            public IEnumerator<KeyValuePair<TKey, TValue>> GetEnumerator() => _dict.GetEnumerator();

            public bool Remove(TKey key) => _dict.Remove(key);

            public bool Remove(KeyValuePair<TKey, TValue> item) => _dict.Remove(item.Key);

            public bool TryGetValue(TKey key, out TValue value) => _dict.TryGetValue(key, out value);

            System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator() => _dict.GetEnumerator();

            // The following are members which have the same names as the IDictionary<,> members.
            // The following members test that there's no System.Reflection.AmbiguousMatchException when binding to the dictionary.
            private string? v;
            public string? this[string key] { get => v; set => v = value; }
            public bool TryGetValue() { return true; }

        }

        public class ExtendedDictionary<TKey, TValue> : Dictionary<TKey, TValue>
        {

        }

        private class OptionsWithDifferentCollectionInterfaces
        {
            private static IEnumerable<string> s_instantiatedIEnumerable = new List<string> { "value1", "value2" };
            public bool IsSameInstantiatedIEnumerable() => object.ReferenceEquals(s_instantiatedIEnumerable, InstantiatedIEnumerable);
            public IEnumerable<string> InstantiatedIEnumerable { get; set; } = s_instantiatedIEnumerable;

            private static IList<string> s_instantiatedIList = new List<string> { "value1", "value2" };
            public bool IsSameInstantiatedIList() => object.ReferenceEquals(s_instantiatedIList, InstantiatedIList);
            public IList<string> InstantiatedIList { get; set; } = s_instantiatedIList;

            private static IReadOnlyList<string> s_instantiatedIReadOnlyList = new List<string> { "value1", "value2" };
            public bool IsSameInstantiatedIReadOnlyList() => object.ReferenceEquals(s_instantiatedIReadOnlyList, InstantiatedIReadOnlyList);
            public IReadOnlyList<string> InstantiatedIReadOnlyList { get; set; } = s_instantiatedIReadOnlyList;

            private static IDictionary<string, string> s_instantiatedIDictionary = new Dictionary<string, string> { ["Key1"] = "value1", ["Key2"] = "value2" };
            public IDictionary<string, string> InstantiatedIDictionary { get; set; } = s_instantiatedIDictionary;
            public bool IsSameInstantiatedIDictionary() => object.ReferenceEquals(s_instantiatedIDictionary, InstantiatedIDictionary);

            private static IReadOnlyDictionary<string, string> s_instantiatedIReadOnlyDictionary = new Dictionary<string, string> { ["Key1"] = "value1", ["Key2"] = "value2" };
            public IReadOnlyDictionary<string, string> InstantiatedIReadOnlyDictionary { get; set; } = s_instantiatedIReadOnlyDictionary;
            public bool IsSameInstantiatedIReadOnlyDictionary() => object.ReferenceEquals(s_instantiatedIReadOnlyDictionary, InstantiatedIReadOnlyDictionary);

            private static ISet<string> s_instantiatedISet = new HashSet<string>(StringComparer.OrdinalIgnoreCase) { "a", "A", "b" };
            public ISet<string> InstantiatedISet { get; set; } = s_instantiatedISet;
            public bool IsSameInstantiatedISet() => object.ReferenceEquals(s_instantiatedISet, InstantiatedISet);

#if NETCOREAPP
            private static IReadOnlySet<string> s_instantiatedIReadOnlySet = new HashSet<string>(StringComparer.OrdinalIgnoreCase) { "a", "A", "b" };
            public IReadOnlySet<string> InstantiatedIReadOnlySet { get; set; } = s_instantiatedIReadOnlySet;
            public bool IsSameInstantiatedIReadOnlySet() => object.ReferenceEquals(s_instantiatedIReadOnlySet, InstantiatedIReadOnlySet);

            public IReadOnlySet<string> UnInstantiatedIReadOnlySet { get; set; }
#endif
            private static ICollection<string> s_instantiatedICollection = new List<string> { "a", "b", "c" };
            public ICollection<string> InstantiatedICollection { get; set; } = s_instantiatedICollection;
            public bool IsSameInstantiatedICollection() => object.ReferenceEquals(s_instantiatedICollection, InstantiatedICollection);

            private static IReadOnlyCollection<string> s_instantiatedIReadOnlyCollection = new List<string> { "a", "b", "c" };
            public IReadOnlyCollection<string> InstantiatedIReadOnlyCollection { get; set; } = s_instantiatedIReadOnlyCollection;
            public bool IsSameInstantiatedIReadOnlyCollection() => object.ReferenceEquals(s_instantiatedIReadOnlyCollection, InstantiatedIReadOnlyCollection);

            public IReadOnlyCollection<string> UnInstantiatedIReadOnlyCollection { get; set; }
            public ICollection<string> UnInstantiatedICollection { get; set; }
            public ISet<string> UnInstantiatedISet { get; set; }
            public IReadOnlyDictionary<string, string> UnInstantiatedIReadOnlyDictionary { get; set; }
            public IEnumerable<string> UnInstantiatedIEnumerable { get; set; }
            public IList<string> UnInstantiatedIList { get; set; }
            public IReadOnlyList<string> UnInstantiatedIReadOnlyList { get; set; }
        }
        [Fact]
        public void TestOptionsWithDifferentCollectionInterfaces()
        {
            var input = new Dictionary<string, string>
            {
                {"InstantiatedIEnumerable:0", "value3"},
                {"UnInstantiatedIEnumerable:0", "value1"},
                {"InstantiatedIList:0", "value3"},
                {"InstantiatedIReadOnlyList:0", "value3"},
                {"UnInstantiatedIReadOnlyList:0", "value"},
                {"UnInstantiatedIList:0", "value"},
                {"InstantiatedIDictionary:Key3", "value3"},
                {"InstantiatedIReadOnlyDictionary:Key3", "value3"},
                {"UnInstantiatedIReadOnlyDictionary:Key", "value"},
                {"InstantiatedISet:0", "B"},
                {"InstantiatedISet:1", "C"},
                {"UnInstantiatedISet:0", "a"},
                {"UnInstantiatedISet:1", "A"},
                {"UnInstantiatedISet:2", "B"},
                {"InstantiatedIReadOnlySet:0", "Z"},
                {"UnInstantiatedIReadOnlySet:0", "y"},
                {"UnInstantiatedIReadOnlySet:1", "z"},
                {"InstantiatedICollection:0", "d"},
                {"UnInstantiatedICollection:0", "t"},
                {"UnInstantiatedICollection:1", "a"},
                {"InstantiatedIReadOnlyCollection:0", "d"},
                {"UnInstantiatedIReadOnlyCollection:0", "r"},
                {"UnInstantiatedIReadOnlyCollection:1", "e"},
            };

            var configurationBuilder = new ConfigurationBuilder();
            configurationBuilder.AddInMemoryCollection(input);
            var config = configurationBuilder.Build();

            var options = new OptionsWithDifferentCollectionInterfaces();
            config.Bind(options);

            Assert.True(3 == options.InstantiatedIEnumerable.Count(), $"InstantiatedIEnumerable count is {options.InstantiatedIEnumerable.Count()} .. {options.InstantiatedIEnumerable.ElementAt(options.InstantiatedIEnumerable.Count() - 1)}");
            Assert.Equal("value1", options.InstantiatedIEnumerable.ElementAt(0));
            Assert.Equal("value2", options.InstantiatedIEnumerable.ElementAt(1));
            Assert.Equal("value3", options.InstantiatedIEnumerable.ElementAt(2));
            Assert.False(options.IsSameInstantiatedIEnumerable());

            Assert.Equal(1, options.UnInstantiatedIEnumerable.Count());
            Assert.Equal("value1", options.UnInstantiatedIEnumerable.ElementAt(0));

            Assert.True(3 == options.InstantiatedIList.Count(), $"InstantiatedIList count is {options.InstantiatedIList.Count()} .. {options.InstantiatedIList[options.InstantiatedIList.Count() - 1]}");
            Assert.Equal("value1", options.InstantiatedIList[0]);
            Assert.Equal("value2", options.InstantiatedIList[1]);
            Assert.Equal("value3", options.InstantiatedIList[2]);
            Assert.True(options.IsSameInstantiatedIList());

            Assert.Equal(1, options.UnInstantiatedIList.Count());
            Assert.Equal("value", options.UnInstantiatedIList[0]);

            Assert.True(3 == options.InstantiatedIReadOnlyList.Count(), $"InstantiatedIReadOnlyList count is {options.InstantiatedIReadOnlyList.Count()} .. {options.InstantiatedIReadOnlyList[options.InstantiatedIReadOnlyList.Count() - 1]}");
            Assert.Equal("value1", options.InstantiatedIReadOnlyList[0]);
            Assert.Equal("value2", options.InstantiatedIReadOnlyList[1]);
            Assert.Equal("value3", options.InstantiatedIReadOnlyList[2]);
            Assert.False(options.IsSameInstantiatedIReadOnlyList());

            Assert.Equal(1, options.UnInstantiatedIReadOnlyList.Count());
            Assert.Equal("value", options.UnInstantiatedIReadOnlyList[0]);

            Assert.True(3 == options.InstantiatedIReadOnlyList.Count(), $"InstantiatedIReadOnlyList count is {options.InstantiatedIReadOnlyList.Count()} .. {options.InstantiatedIReadOnlyList[options.InstantiatedIReadOnlyList.Count() - 1]}");
            Assert.Equal(new string[] { "Key1", "Key2", "Key3" }, options.InstantiatedIDictionary.Keys);
            Assert.Equal(new string[] { "value1", "value2", "value3" }, options.InstantiatedIDictionary.Values);
            Assert.True(options.IsSameInstantiatedIDictionary());

            Assert.True(3 == options.InstantiatedIReadOnlyDictionary.Count(), $"InstantiatedIReadOnlyDictionary count is {options.InstantiatedIReadOnlyDictionary.Count()} .. {options.InstantiatedIReadOnlyDictionary.ElementAt(options.InstantiatedIReadOnlyDictionary.Count() - 1)}");
            Assert.Equal(new string[] { "Key1", "Key2", "Key3" }, options.InstantiatedIReadOnlyDictionary.Keys);
            Assert.Equal(new string[] { "value1", "value2", "value3" }, options.InstantiatedIReadOnlyDictionary.Values);
            Assert.False(options.IsSameInstantiatedIReadOnlyDictionary());

            Assert.Equal(1, options.UnInstantiatedIReadOnlyDictionary.Count());
            Assert.Equal(new string[] { "Key" }, options.UnInstantiatedIReadOnlyDictionary.Keys);
            Assert.Equal(new string[] { "value" }, options.UnInstantiatedIReadOnlyDictionary.Values);

            Assert.True(3 == options.InstantiatedISet.Count(), $"InstantiatedISet count is {options.InstantiatedISet.Count()} .. {string.Join(", ", options.InstantiatedISet)} .. {options.IsSameInstantiatedISet()}");
            Assert.Equal(new string[] { "a", "b", "C" }, options.InstantiatedISet);
            Assert.True(options.IsSameInstantiatedISet());

            Assert.True(3 == options.UnInstantiatedISet.Count(), $"UnInstantiatedISet count is {options.UnInstantiatedISet.Count()} .. {options.UnInstantiatedISet.ElementAt(options.UnInstantiatedISet.Count() - 1)}");
            Assert.Equal(new string[] { "a", "A", "B" }, options.UnInstantiatedISet);

#if NETCOREAPP
            Assert.True(3 == options.InstantiatedIReadOnlySet.Count(), $"InstantiatedIReadOnlySet count is {options.InstantiatedIReadOnlySet.Count()} .. {options.InstantiatedIReadOnlySet.ElementAt(options.InstantiatedIReadOnlySet.Count() - 1)}");
            Assert.Equal(new string[] { "a", "b", "Z" }, options.InstantiatedIReadOnlySet);
            Assert.False(options.IsSameInstantiatedIReadOnlySet());

            Assert.Equal(2, options.UnInstantiatedIReadOnlySet.Count());
            Assert.Equal(new string[] { "y", "z" }, options.UnInstantiatedIReadOnlySet);
#endif
            Assert.Equal(4, options.InstantiatedICollection.Count());
            Assert.Equal(new string[] { "a", "b", "c", "d" }, options.InstantiatedICollection);
            Assert.True(options.IsSameInstantiatedICollection());

            Assert.Equal(2, options.UnInstantiatedICollection.Count());
            Assert.Equal(new string[] { "t", "a" }, options.UnInstantiatedICollection);

            Assert.Equal(4, options.InstantiatedIReadOnlyCollection.Count());
            Assert.Equal(new string[] { "a", "b", "c", "d" }, options.InstantiatedIReadOnlyCollection);
            Assert.False(options.IsSameInstantiatedIReadOnlyCollection());

            Assert.Equal(2, options.UnInstantiatedIReadOnlyCollection.Count());
            Assert.Equal(new string[] { "r", "e" }, options.UnInstantiatedIReadOnlyCollection);
        }

        [Fact]
        public void TestMutatingDictionaryValues()
        {
            IConfiguration config = new ConfigurationBuilder()
                .AddInMemoryCollection()
                .Build();

            config["Key:0"] = "NewValue";
            var dict = new Dictionary<string, string[]>() { { "Key", new[] { "InitialValue" } } };

            Assert.Equal(1, dict["Key"].Length);
            Assert.Equal("InitialValue", dict["Key"][0]);

            // Binding will accumulate to the values inside the dictionary.
            config.Bind(dict);
            Assert.Equal(2, dict["Key"].Length);
            Assert.Equal("InitialValue", dict["Key"][0]);
            Assert.Equal("NewValue", dict["Key"][1]);
        }
    }
}
