use super::HashAlgo;
use crate::errors::*;
use crate::tee::claims::Claims;

use ciborium::Value;
use serde::{Deserialize, Deserializer, Serialize, Serializer};

/// Intel TEE quote, including all SGX (both EPID and ECDSA) and TDX (ECDSA) quote types
#[allow(dead_code)]
pub const OCBR_TAG_EVIDENCE_INTEL_TEE_QUOTE: u64 = 60000;
/// Intel TEE report (TDX report or SGX report type 2)
#[allow(dead_code)]
pub const OCBR_TAG_EVIDENCE_INTEL_TEE_REPORT: u64 = 60001;
/// SGX report (legacy, generated by EREPORT)
#[allow(dead_code)]
pub const OCBR_TAG_EVIDENCE_INTEL_SGX_LEGACY_REPORT: u64 = 60002;
// TODO: register the following CBOR tags at IANA: https://www.iana.org/assignments/cbor-tags/cbor-tags.xhtml
#[allow(dead_code)]
pub const OCBR_TAG_EVIDENCE_SEV_SNP: u64 = 0x1a7504;
#[allow(dead_code)]
pub const OCBR_TAG_EVIDENCE_SEV: u64 = 0x1a7505;
#[allow(dead_code)]
pub const OCBR_TAG_EVIDENCE_CSV: u64 = 0x1a7506;
/// The evidence generated by CoCo attestation-agent
#[allow(dead_code)]
pub const OCBR_TAG_EVIDENCE_COCO_EVIDENCE: u64 = 0xC0C000;
/// The token generated by CoCo attestation-agent
#[allow(dead_code)]
pub const OCBR_TAG_EVIDENCE_COCO_TOKEN: u64 = 0xC0C001;

pub type HashAlgoIanaId = u8;

impl TryFrom<HashAlgoIanaId> for HashAlgo {
    type Error = Error;

    fn try_from(value: HashAlgoIanaId) -> Result<Self> {
        Ok(match value {
            1 => Self::Sha256,
            7 => Self::Sha384,
            8 => Self::Sha512,
            _ => Err(Error::kind_with_msg(
                ErrorKind::UnsupportedHashAlgo,
                format!("unsupported hash-alg-id: {}", value),
            ))?,
        })
    }
}
impl Into<HashAlgoIanaId> for HashAlgo {
    fn into(self) -> HashAlgoIanaId {
        match self {
            HashAlgo::Sha256 => 1,
            HashAlgo::Sha384 => 7,
            HashAlgo::Sha512 => 8,
        }
    }
}

/// An definite-length encoded tagged CBOR array with two entries.
///  For SGX / TDX ECDSA evidence, it is: definite-length encoded tagged CBOR array with two entries, holding TEE quote and serialized custom claims:
///     60000([ TEE_ECDSA_quote(claims-buffer-hash), claims-buffer ])
#[derive(PartialEq, Debug)]
struct TaggedEvidenceBufferHelperRef<'a> {
    tag: u64,                /* a tag for distinguishing evidence type */
    evidence: &'a [u8],      /* shall be serialized to cbor byte string */
    claims_buffer: &'a [u8], /* shall be serialized to cbor byte string */
}

impl<'a> Serialize for TaggedEvidenceBufferHelperRef<'a> {
    fn serialize<S: Serializer>(&self, s: S) -> std::result::Result<S::Ok, S::Error> {
        Value::Tag(
            self.tag,
            Box::new(Value::Array(vec![
                Value::Bytes(self.evidence.into()), // TODO: this can be optimized with serde_bytes::Bytes, since there is no need to construct a Vec
                Value::Bytes(self.claims_buffer.into()),
            ])),
        )
        .serialize(s)
    }
}

/// See TaggedEvidenceBufferHelperRef
#[derive(PartialEq, Debug)]
struct TaggedEvidenceBufferHelperOwned {
    tag: u64,
    evidence: Vec<u8>,
    claims_buffer: Vec<u8>,
}

impl<'a> Deserialize<'a> for TaggedEvidenceBufferHelperOwned {
    fn deserialize<D: Deserializer<'a>>(deserializer: D) -> std::result::Result<Self, D::Error> {
        let tagged = Value::deserialize(deserializer)?;
        match tagged {
            Value::Tag(tag, any) => match *any {
                Value::Array(a) => {
                    if a.len() == 2 {
                        let mut it = a.into_iter();
                        let (a_0, a_1) = (it.next().unwrap(), it.next().unwrap());
                        let evidence = match a_0 {
                            Value::Bytes(bs) => bs,
                            _ => Err(serde::de::Error::invalid_type(
                                (&a_0).into(),
                                &"bytes (raw_evidence)",
                            ))?,
                        };

                        let claims_buffer = match a_1 {
                            Value::Bytes(bs) => bs,
                            _ => Err(serde::de::Error::invalid_type(
                                (&a_1).into(),
                                &"bytes (claims_buffer)",
                            ))?,
                        };
                        Ok(Self {
                            tag,
                            evidence,
                            claims_buffer,
                        })
                    } else {
                        Err(serde::de::Error::invalid_type(
                            (&Value::Array(a)).into(),
                            &"an array with length 2",
                        ))
                    }
                }
                _ => Err(serde::de::Error::invalid_type((&*any).into(), &"array")),
            },
            _ => Err(serde::de::Error::invalid_type((&tagged).into(), &"tag")),
        }
    }
}

pub fn generate_evidence_buffer_with_tag(
    tag: u64,
    raw_evidence: &[u8],
    claims_buffer: &[u8],
) -> Result<Vec<u8>> {
    let helper = TaggedEvidenceBufferHelperRef {
        tag,
        evidence: raw_evidence,
        claims_buffer,
    };
    let mut res = vec![];
    ciborium::into_writer(&helper, &mut res)?;
    Ok(res)
}

pub fn parse_evidence_buffer_with_tag(
    evidence_buffer: &[u8],
) -> Result<(
    /* tag */ u64,
    /* raw_evidence */ Vec<u8>,
    /* claims_buffer */ Vec<u8>,
)> {
    let helper: TaggedEvidenceBufferHelperOwned = ciborium::from_reader(evidence_buffer)?;
    Ok((helper.tag, helper.evidence, helper.claims_buffer))
}

// TODO: endorsement buffer

/// The helper is used to serialize claims and forces Serde to treat values as bytes instead of slice of `u8`.
struct ClaimsHelper<'a>(&'a Claims);

impl<'a> Serialize for ClaimsHelper<'a> {
    fn serialize<S: Serializer>(&self, s: S) -> std::result::Result<S::Ok, S::Error> {
        let v = self
            .0
            .iter()
            .map(|claim| (Value::Text(claim.0.clone()), Value::Bytes(claim.1.clone())))
            .collect::<Vec<_>>();
        Value::Map(v).serialize(s)
    }
}

/// The claims-buffer is a byte string of definite-length encoded CBOR map of one or two custom claims, with each claim name in text string format, and its value in byte string format.
pub fn generate_claims_buffer(claims: &Claims) -> Result<Vec<u8>> {
    let mut res = vec![];
    ciborium::into_writer(&ClaimsHelper(claims), &mut res)?;
    Ok(res)
}

pub fn parse_claims_buffer(claims_buffer: &[u8]) -> Result<Claims> {
    let claims: Claims = ciborium::from_reader(claims_buffer)?;
    Ok(claims)
}

pub fn generate_pubkey_hash_value_buffer(hash_algo: HashAlgo, hash: &[u8]) -> Result<Vec<u8>> {
    let mut res = vec![];
    ciborium::into_writer(
        &(
            Into::<HashAlgoIanaId>::into(hash_algo),
            serde_bytes::Bytes::new(hash),
        ),
        &mut res,
    )?;
    Ok(res)
}

pub fn parse_pubkey_hash_value_buffer(
    pubkey_hash_value_buffer: &[u8],
) -> Result<(/* hash_algo */ HashAlgo, /* hash */ Vec<u8>)> {
    let (hash_algo_id, hash): (HashAlgoIanaId, Vec<u8>) =
        ciborium::from_reader(pubkey_hash_value_buffer)?;
    Ok((hash_algo_id.try_into()?, hash))
}

#[cfg(test)]
pub mod tests {

    use super::*;

    #[test]
    fn test_generate_evidence_buffer_with_tag() -> Result<()> {
        let expected_tag = OCBR_TAG_EVIDENCE_INTEL_TEE_QUOTE;
        let expected_raw_evidence = b"\x01\x02\x03\x04";
        let expected_claims_buffer = b"\x05\x06\x07\x08";

        let evidence_buffer = generate_evidence_buffer_with_tag(
            expected_tag,
            expected_raw_evidence,
            expected_claims_buffer,
        )?;
        println!(
            "evidence_buffer with tag in cbor: {}",
            hex::encode(&evidence_buffer)
        );

        assert_eq!(
            hex::encode(&evidence_buffer),
            "d9ea608244010203044405060708" /* 60000_1([h'01020304', h'05060708']) */
        );

        let res = parse_evidence_buffer_with_tag(&evidence_buffer)?;
        assert_eq!(expected_tag, res.0);
        assert_eq!(&expected_raw_evidence[..], &res.1);
        assert_eq!(&expected_claims_buffer[..], &res.2);
        Ok(())
    }

    #[test]
    fn test_generate_claims_buffer() -> Result<()> {
        let mut expected = Claims::new();
        expected.insert("key1".into(), "value1".into());
        expected.insert("key2".into(), "value2".into());

        let claims_buffer = generate_claims_buffer(&expected)?;
        println!("claims buffer in cbor: {}", hex::encode(&claims_buffer));

        assert_eq!(
            hex::encode(&claims_buffer),
            "a2646b6579314676616c756531646b6579324676616c756532" /* {"key1": h'76616c756531', "key2": h'76616c756532'} */
        );

        let deserialized = parse_claims_buffer(&claims_buffer)?;
        assert_eq!(expected, deserialized);
        Ok(())
    }

    #[test]
    fn test_generate_pubkey_hash_value_buffer() -> Result<()> {
        let expected_hash_algo = HashAlgo::Sha256;
        let expected_hash = b"\x00\x01\x02\x03";
        let pubkey_hash_value_buffer =
            generate_pubkey_hash_value_buffer(expected_hash_algo, expected_hash)?;
        println!(
            "claims pubkey_hash_value_buffer in cbor: {}",
            hex::encode(&pubkey_hash_value_buffer)
        );
        assert_eq!(pubkey_hash_value_buffer, hex::decode("82014400010203")?);

        let deserialized = parse_pubkey_hash_value_buffer(&pubkey_hash_value_buffer)?;
        assert_eq!(expected_hash_algo, deserialized.0);
        assert_eq!(&expected_hash[..], deserialized.1);
        Ok(())
    }
}
