/**
 * @version 1.0.0.0
 * @author Microsoft
 * @copyright © Microsoft Corporation. All rights reserved.
 * @compiler Bridge.NET 17.10.1
 */
Bridge.assembly("Kusto.JavaScript.Client", function ($asm, globals) {
    "use strict";

    Bridge.define("Kusto.Charting.AnomalyDataHelper2", {
        statics: {
            props: {
                TimestampColumnName: null,
                ValueColumnName: null,
                SeriesColumnName: null,
                AnomalySerieSuffix: null
            },
            ctors: {
                init: function () {
                    this.TimestampColumnName = "Timestamp";
                    this.ValueColumnName = "Value";
                    this.SeriesColumnName = "Series";
                    this.AnomalySerieSuffix = "(anomaly)";
                }
            },
            methods: {
                EnchanceDataWithAnomalyDataFromServiceAsync: function (kustoContext, data) {
                    var $step = 0,
                        $task1, 
                        $taskResult1, 
                        $jumpFromFinally, 
                        $tcs = new System.Threading.Tasks.TaskCompletionSource(), 
                        $returnValue, 
                        res, 
                        dataAsCsl, 
                        query, 
                        results, 
                        $t, 
                        $async_e, 
                        $asyncBody = Bridge.fn.bind(this, function () {
                            try {
                                for (;;) {
                                    $step = System.Array.min([0,1], $step);
                                    switch ($step) {
                                        case 0: {
                                            res = new Kusto.Charting.AnomalyDataHelper2.AnomalyDataFromServiceResult();

                                            if (kustoContext == null) {
                                                res.Data = data;
                                                res.HasErrors = true;
                                                res.Message = "Failed sending the request for anomalies to the service";
                                                $tcs.setResult(res);
                                                return;
                                            }

                                            dataAsCsl = Kusto.Charting.AnomalyDataHelper2.DataItemsToDataTableLiteral(data);
                                            query = (dataAsCsl || "") + (System.String.format("| summarize dt = makelist({0}, 100000), y=makelist({1}, 100000) by {2}\r\n| extend (anomalies, scores, baseline)=series_decompose_anomalies(y)\r\n| project dt, y, anomalies, Series\r\n| mvexpand dt to typeof(datetime), y to typeof(double), anomalies to typeof(double) limit 1000000\r\n| where anomalies != 0\r\n| project dt, anomaly_value = (anomalies * anomalies)*y, Series", Kusto.Charting.AnomalyDataHelper2.TimestampColumnName, Kusto.Charting.AnomalyDataHelper2.ValueColumnName, Kusto.Charting.AnomalyDataHelper2.SeriesColumnName) || "");

                                            if (System.Text.Encoding.UTF8.GetByteCount$2(query) > 2000000) {
                                                res.Data = data;
                                                res.HasErrors = true;
                                                res.Message = "The anomalies request to the service exceeds the 2MB limit, consider adding more filters";
                                                $tcs.setResult(res);
                                                return;
                                            }

                                            $task1 = kustoContext.Kusto$Charting$IKustoClientContext$ExecuteQueryAsync(Kusto.Charting.AnomalyDataHelper2.AnomalyInputOutputRecord, query);
                                            $step = 1;
                                            if ($task1.isCompleted()) {
                                                continue;
                                            }
                                            $task1.continue($asyncBody);
                                            return;
                                        }
                                        case 1: {
                                            $taskResult1 = $task1.getAwaitedResult();
                                            results = $taskResult1;
                                            results = ($t = Kusto.Charting.AnomalyDataHelper2.AnomalyInputOutputRecord, System.Linq.Enumerable.from(results, $t).ToArray($t));
                                            if (results == null || !System.Linq.Enumerable.from(results, Kusto.Charting.AnomalyDataHelper2.AnomalyInputOutputRecord).any()) {
                                                res.Data = data;
                                                res.HasErrors = true;
                                                res.Message = "Failed retrieving the anomalies result from the service";
                                                $tcs.setResult(res);
                                                return;
                                            }

                                            // Translate anomaly results back to the data points
                                            data.AddRange(System.Linq.Enumerable.from(results, Kusto.Charting.AnomalyDataHelper2.AnomalyInputOutputRecord).select($asm.$.Kusto.Charting.AnomalyDataHelper2.f1));

                                            res.Data = data;
                                            res.HasErrors = false;
                                            res.Message = "";
                                            $tcs.setResult(res);
                                            return;
                                        }
                                        default: {
                                            $tcs.setResult(null);
                                            return;
                                        }
                                    }
                                }
                            } catch($async_e1) {
                                $async_e = System.Exception.create($async_e1);
                                $tcs.setException($async_e);
                            }
                        }, arguments);

                    $asyncBody();
                    return $tcs.task;
                },
                DataItemsToDataTableLiteral: function (items) {
                    var $t;
                    var result = new System.Text.StringBuilder();
                    result.append("datatable");
                    result.appendLine(System.String.format("({0}:datetime,{1}:double,{2}:string)", Kusto.Charting.AnomalyDataHelper2.TimestampColumnName, Kusto.Charting.AnomalyDataHelper2.ValueColumnName, Kusto.Charting.AnomalyDataHelper2.SeriesColumnName));
                    result.append("[");
                    $t = Bridge.getEnumerator(items, Kusto.Charting.DataItem);
                    try {
                        while ($t.moveNext()) {
                            var item = $t.Current;
                            result.append("datetime(");
                            result.append(System.DateTime.format(item.ArgumentDateTime, "o"));
                            result.append("),");
                            result.append("double("); //needed for negative values
                            result.append(System.Double.format(item.ValueData));
                            result.append("),");
                            var seriesName = Kusto.Cloud.Platform.Text.StringLiteral.GetLiteral(item.SeriesName);
                            result.append(seriesName);
                            result.appendLine(",");
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }
                    result.appendLine("]");
                    return result.toString();
                },
                EnchanceDataWithAnomalyDataFromColumns: function (data, anomalyColumns, anomalySeriesMap) {
                    var $t, $t1;
                    anomalySeriesMap.v = new (System.Collections.Generic.Dictionary$2(System.String,System.String)).ctor();

                    var result = new (System.Collections.Generic.List$1(Kusto.Charting.DataItem)).ctor();

                    var anomalyColumnsHashset = new (System.Collections.Generic.HashSet$1(System.String)).$ctor1(anomalyColumns);
                    // Choose first column that is not listed as anomaly column
                    var valueColumn = System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).where(function (d) {
                            return !anomalyColumnsHashset.contains(d.ValueName);
                        }).select($asm.$.Kusto.Charting.AnomalyDataHelper2.f2).firstOrDefault(null, null);
                    if (System.String.isNullOrEmpty(valueColumn)) {
                        return data;
                    }

                    // Copy all data except anomalies
                    result.AddRange(System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).where(function (d) {
                            return !anomalyColumnsHashset.contains(d.ValueName);
                        }));

                    // Divide data points into groups of data and anomalies.
                    var valueData = System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).where(function (d) {
                            return System.String.equals(d.ValueName, valueColumn);
                        }).ToArray(Kusto.Charting.DataItem);

                    $t = Bridge.getEnumerator(anomalyColumns);
                    try {
                        while ($t.moveNext()) {
                            var anomalyColumn = { v : $t.Current };
                            var anomalyData = System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).where((function ($me, anomalyColumn) {
                                    return function (d) {
                                        return System.String.equals(d.ValueName, anomalyColumn.v);
                                    };
                                })(this, anomalyColumn)).ToArray(Kusto.Charting.DataItem);
                            if (valueData.length !== anomalyData.length) {
                                // Streams of data are not of the same size - return
                                return data;
                            }

                            var anomalyPointsCount = 0;
                            for (var i = 0; i < valueData.length; i = (i + 1) | 0) {
                                var valuePoint = valueData[System.Array.index(i, valueData)];
                                var anomalyPoint = anomalyData[System.Array.index(i, anomalyData)];
                                // This is an anomaly
                                if (anomalyPoint.ValueData !== 0) {
                                    anomalyPointsCount = (anomalyPointsCount + 1) | 0;
                                    var seriesName = (anomalyPoint.SeriesName || "") + (Kusto.Charting.AnomalyDataHelper2.AnomalySerieSuffix || "");
                                    var a = ($t1 = new Kusto.Charting.DataItem.ctor(), $t1.ArgumentDateTime = anomalyPoint.ArgumentDateTime, $t1.ArgumentData = anomalyPoint.ArgumentData, $t1.SeriesName = seriesName, $t1.ValueData = valuePoint.ValueData, $t1.PrefferredChartKind = Kusto.Charting.ChartKind.Point, $t1);
                                    result.add(a);
                                    if (!anomalySeriesMap.v.containsKey(seriesName)) {
                                        anomalySeriesMap.v.add(seriesName, valuePoint.SeriesName);
                                    }
                                }
                            }

                            //adding dummy point in order to display anomaly series in Legend, in case if there are no points to render
                            if (anomalyPointsCount === 0) {
                                var a1 = ($t1 = new Kusto.Charting.DataItem.ctor(), $t1.ArgumentDateTime = anomalyData[System.Array.index(0, anomalyData)].ArgumentDateTime, $t1.ArgumentData = null, $t1.SeriesName = (anomalyData[System.Array.index(0, anomalyData)].SeriesName || "") + (Kusto.Charting.AnomalyDataHelper2.AnomalySerieSuffix || ""), $t1.ValueData = Number.NaN, $t1.PrefferredChartKind = Kusto.Charting.ChartKind.Point, $t1);
                                result.add(a1);
                            }
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return result;
                }
            }
        }
    });

    Bridge.ns("Kusto.Charting.AnomalyDataHelper2", $asm.$);

    Bridge.apply($asm.$.Kusto.Charting.AnomalyDataHelper2, {
        f1: function (r) {
        var $t1;
        return ($t1 = new Kusto.Charting.DataItem.ctor(), $t1.ArgumentDateTime = r.Timestamp, $t1.ArgumentData = System.DateTime.format(r.Timestamp, "o"), $t1.SeriesName = (r.Series || "") + (Kusto.Charting.AnomalyDataHelper2.AnomalySerieSuffix || ""), $t1.ValueData = r.Value, $t1.PrefferredChartKind = Kusto.Charting.ChartKind.Point, $t1);
    },
        f2: function (d) {
            return d.ValueName;
        }
    });

    Bridge.define("Kusto.Charting.AnomalyDataHelper2.AnomalyDataFromServiceResult", {
        $kind: "nested class",
        fields: {
            Data: null,
            HasErrors: false,
            Message: null
        }
    });

    Bridge.define("Kusto.Charting.AnomalyDataHelper2.AnomalyInputOutputRecord", {
        $kind: "nested class",
        fields: {
            Timestamp: null,
            Value: 0,
            Series: null
        },
        ctors: {
            init: function () {
                this.Timestamp = System.DateTime.getDefaultValue();
            }
        }
    });

    Bridge.define("Kusto.Charting.ArgumentColumnType", {
        $kind: "enum",
        statics: {
            fields: {
                None: 0,
                Numeric: 2,
                DateTime: 4,
                TimeSpan: 8,
                String: 16,
                Object: 32,
                Geospatial: 64,
                DateTimeOrTimeSpan: 12,
                StringOrDateTimeOrTimeSpan: 28,
                NumericOrDateTimeOrTimeSpan: 14,
                StringOrObject: 48,
                AllExceptGeospatial: 62
            }
        },
        $flags: true
    });

    Bridge.define("Kusto.Charting.ArgumentRestrictions", {
        $kind: "enum",
        statics: {
            fields: {
                /**
                 * No restrictiosn apply
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Charting.ArgumentRestrictions
                 * @constant
                 * @default 0
                 * @type Kusto.Charting.ArgumentRestrictions
                 */
                None: 0,
                /**
                 * Must find a valid argument
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Charting.ArgumentRestrictions
                 * @constant
                 * @default 1
                 * @type Kusto.Charting.ArgumentRestrictions
                 */
                MustHave: 1,
                /**
                 * Argument column used cannot appear also in series
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Charting.ArgumentRestrictions
                 * @constant
                 * @default 2
                 * @type Kusto.Charting.ArgumentRestrictions
                 */
                NotIncludedInSeries: 2,
                /**
                 * Try to locate an argument that allows geo-spatial types to appear in series
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Charting.ArgumentRestrictions
                 * @constant
                 * @default 4
                 * @type Kusto.Charting.ArgumentRestrictions
                 */
                GeospatialAsSeries: 4,
                /**
                 * Prefer picking last column that matches arguments restrictions
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Charting.ArgumentRestrictions
                 * @constant
                 * @default 8
                 * @type Kusto.Charting.ArgumentRestrictions
                 */
                PreferLast: 8,
                /**
                 * Try to locate an argument that allows numerics to appear in series
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Charting.ArgumentRestrictions
                 * @constant
                 * @default 16
                 * @type Kusto.Charting.ArgumentRestrictions
                 */
                NumericAsSeries: 16
            }
        },
        $flags: true
    });

    Bridge.define("Kusto.Charting.ChartKind", {
        $kind: "enum",
        statics: {
            fields: {
                Unspecified: 0,
                Line: 1,
                Point: 2,
                Bar: 3
            }
        }
    });

    Bridge.define("Kusto.Charting.ChartLimitsPolicy", {
        props: {
            /**
             * The maximal amount of points allowed for visualization on the chart
             *
             * @instance
             * @public
             * @memberof Kusto.Charting.ChartLimitsPolicy
             * @function MaxPointsPerChartError
             * @type number
             */
            MaxPointsPerChartError: 0,
            /**
             * Amount of points above which required user's approvement to visualize chart
             *
             * @instance
             * @public
             * @memberof Kusto.Charting.ChartLimitsPolicy
             * @function MaxPointsPerChartWarning
             * @type number
             */
            MaxPointsPerChartWarning: 0,
            /**
             * The maximal amount of series allowed for visualization on the chart
             *
             * @instance
             * @public
             * @memberof Kusto.Charting.ChartLimitsPolicy
             * @function MaxSeriesPerChartError
             * @type number
             */
            MaxSeriesPerChartError: 0,
            /**
             * Amount of series above which required user's approvement to visualize chart
             *
             * @instance
             * @public
             * @memberof Kusto.Charting.ChartLimitsPolicy
             * @function MaxSeriesPerChartWarning
             * @type number
             */
            MaxSeriesPerChartWarning: 0,
            /**
             * The maximal interval of DateTime argument allowed for visualization on the chart
             *
             * @instance
             * @public
             * @memberof Kusto.Charting.ChartLimitsPolicy
             * @function MaxDatetimePeriodError
             * @type System.TimeSpan
             */
            MaxDatetimePeriodError: null
        },
        ctors: {
            init: function () {
                this.MaxDatetimePeriodError = new System.TimeSpan();
            },
            ctor: function (maxPointsPerChartError, maxPointsPerChartWarning, maxSeriesPerChartError, maxSeriesPerChartWarning, maxDatetimePeriodError) {
                this.$initialize();
                this.MaxPointsPerChartError = maxPointsPerChartError;
                this.MaxPointsPerChartWarning = maxPointsPerChartWarning;
                this.MaxSeriesPerChartError = maxSeriesPerChartError;
                this.MaxSeriesPerChartWarning = maxSeriesPerChartWarning;
                this.MaxDatetimePeriodError = maxDatetimePeriodError;
            }
        }
    });

    Bridge.define("Kusto.Charting.DataChartsHelper", {
        statics: {
            fields: {
                c_minFractionOfIntervalsToDetectDistribution: 0,
                c_minElementsAmountToFillGapsWithNans: 0,
                c_decimalRoundingPrecision: 0,
                c_doubleAcceptableApproximation: 0,
                c_defaultArgumentNumeric: 0,
                c_defaultArgumentString: null,
                c_defaultArgumentDatetime: null
            },
            ctors: {
                init: function () {
                    this.c_defaultArgumentDatetime = System.DateTime.getDefaultValue();
                    this.c_minFractionOfIntervalsToDetectDistribution = 0.8;
                    this.c_minElementsAmountToFillGapsWithNans = 4;
                    this.c_decimalRoundingPrecision = 8;
                    this.c_doubleAcceptableApproximation = 1E-08;
                    this.c_defaultArgumentNumeric = Number.NaN;
                    this.c_defaultArgumentString = "<empty>";
                    this.c_defaultArgumentDatetime = Bridge.getDefaultValue(System.DateTime);
                }
            },
            methods: {
                /**
                 * Generates set of a DataItem objects from provided structure,
                 based on required argument parameters(column type and restrictions).
                 Takes in account x-columns, y-columns and series if provided.
                 *
                 * @static
                 * @public
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {Kusto.Charting.IChartingDataSource}          table                   Data source, which has to implement IChartingDataSource.
                 * @param   {Kusto.Charting.ArgumentColumnType}           argumentColumnType      Required column type
                 * @param   {Kusto.Charting.ArgumentRestrictions}         argumentRestrictions    Argument restrictions
                 * @param   {System.Collections.Generic.IEnumerable$1}    seriesColumns           Optional. Names of columns to be considered as series.
                 * @param   {boolean}                                     accumulateResults       Optional. Defines the necessity of accumulation in each DataItem values from the previous ones. Default - false.
                 * @param   {string}                                      xColumn                 Optional. Name of column to be considered as an argument.
                 * @param   {System.Collections.Generic.IEnumerable$1}    yColumns                Optional. Names of columns to be considered as a function.
                 * @return  {System.Collections.Generic.IEnumerable$1}                            Collection of DataItem objects.
                 */
                GetData: function (table, argumentColumnType, argumentRestrictions, seriesColumns, accumulateResults, xColumn, yColumns) {
                    if (argumentColumnType === void 0) { argumentColumnType = 16; }
                    if (argumentRestrictions === void 0) { argumentRestrictions = 0; }
                    if (seriesColumns === void 0) { seriesColumns = null; }
                    if (accumulateResults === void 0) { accumulateResults = false; }
                    if (xColumn === void 0) { xColumn = null; }
                    if (yColumns === void 0) { yColumns = null; }
                    var chartProps = Kusto.Charting.DataChartsHelper.GetMetaData(table, argumentColumnType, argumentRestrictions, seriesColumns, xColumn, yColumns);
                    return Kusto.Charting.DataChartsHelper.GetData$1(table, chartProps, accumulateResults);
                },
                /**
                 * Generates set of a DataItem objects from provided structure,
                 based on chart meta data
                 *
                 * @static
                 * @public
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {Kusto.Charting.IChartingDataSource}          table                Data source, which has to implement IChartingDataSource.
                 * @param   {Kusto.Charting.IChartMetaData}               metaData             chart meta data (argument, series and data columns) for data generation
                 * @param   {boolean}                                     accumulateResults    Optional. Defines the necessity of accumulation in each DataItem values from the previous ones. Default - false.
                 * @return  {System.Collections.Generic.IEnumerable$1}                         Collection of DataItem objects.
                 */
                GetData$1: function (table, metaData, accumulateResults) {
                    if (accumulateResults === void 0) { accumulateResults = false; }
                    if (table == null || metaData == null) {
                        return System.Linq.Enumerable.empty();
                    }

                    var tableSchema = table.Kusto$Charting$IChartingDataSource$GetSchema();

                    // IN JS Enumeration are slow
                    // Special Contains and ElementAt
                    var allColumns = System.Linq.Enumerable.from(tableSchema, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).select($asm.$.Kusto.Charting.DataChartsHelper.f1).toList(Kusto.Charting.DataChartsHelper.ColumnDesc);
                    var seriesList = System.Linq.Enumerable.from(metaData.Kusto$Charting$IChartMetaData$SeriesIndexes, System.Int32).any() ? System.Linq.Enumerable.from(allColumns, Kusto.Charting.DataChartsHelper.ColumnDesc).where(function (col) {
                            return System.Linq.Enumerable.from(metaData.Kusto$Charting$IChartMetaData$SeriesIndexes, System.Int32).contains(col.Index);
                        }).ToArray(Kusto.Charting.DataChartsHelper.ColumnDesc) : null;
                    var dataColumns = System.Linq.Enumerable.from(allColumns, Kusto.Charting.DataChartsHelper.ColumnDesc).where(function (col) {
                            return System.Linq.Enumerable.from(metaData.Kusto$Charting$IChartMetaData$DataIndexes, System.Int32).contains(col.Index);
                        }).ToArray(Kusto.Charting.DataChartsHelper.ColumnDesc);
                    var unusedColumns = System.Linq.Enumerable.from(allColumns, Kusto.Charting.DataChartsHelper.ColumnDesc).where(function (col) {
                            return System.Linq.Enumerable.from(metaData.Kusto$Charting$IChartMetaData$UnusedIndexes, System.Int32).contains(col.Index);
                        }).ToArray(Kusto.Charting.DataChartsHelper.ColumnDesc);

                    // pre building the list
                    var argumentData = new Kusto.Charting.DataChartsHelper.ArgumentData(metaData.Kusto$Charting$IChartMetaData$ArgumentDataColumnIndex, metaData.Kusto$Charting$IChartMetaData$ArgumentDataColumnIndex >= 0 ? allColumns.getItem(metaData.Kusto$Charting$IChartMetaData$ArgumentDataColumnIndex).Name : "", metaData.Kusto$Charting$IChartMetaData$ArgumentDataColumnIndex >= 0 ? System.Linq.Enumerable.from(tableSchema, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).elementAt(metaData.Kusto$Charting$IChartMetaData$ArgumentDataColumnIndex).Item2 : Kusto.Charting.ArgumentColumnType.None, metaData.Kusto$Charting$IChartMetaData$ArgumentColumnType, metaData.Kusto$Charting$IChartMetaData$GeospatialColumnIndexes); // geospatial column indexes

                    var result = new (System.Collections.Generic.List$1(Kusto.Charting.DataItem)).ctor();
                    var lastValues = new (System.Collections.Generic.Dictionary$2(System.String,System.Double)).ctor();
                    for (var i = 0; i < table.Kusto$Charting$IChartingDataSource$RowsCount; i = (i + 1) | 0) {
                        var baseSeriesName = Kusto.Charting.DataChartsHelper.GetBaseSeriesName(table, seriesList, i);

                        if (!metaData.Kusto$Charting$IChartMetaData$IsDataFormedAsSeries) {
                            argumentData.ResolveArgumentFromRow(table, i);

                            Kusto.Charting.DataChartsHelper.ResolveDataItemsFromDataRow(result, table, dataColumns, unusedColumns, i, baseSeriesName, argumentData, lastValues, accumulateResults);
                        } else {
                            Kusto.Charting.DataChartsHelper.ResolveDataSeriesFromDataRow(result, table, dataColumns, unusedColumns, i, baseSeriesName, metaData.Kusto$Charting$IChartMetaData$ArgumentColumnType, accumulateResults, metaData.Kusto$Charting$IChartMetaData$ArgumentDataColumnIndex);
                        }
                    }

                    return result;
                },
                /**
                 * Figure out the chart meta data, which that will be used to generate the chart data 
                 based on required argument parameters(column type and restrictions).
                 Takes in account x-columns, y-columns and series if provided.
                 *
                 * @static
                 * @public
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {Kusto.Charting.IChartingDataSource}          table                   Data source, which has to implement IChartingDataSource.
                 * @param   {Kusto.Charting.ArgumentColumnType}           argumentColumnType      Required column type
                 * @param   {Kusto.Charting.ArgumentRestrictions}         argumentRestrictions    Argument restrictions
                 * @param   {System.Collections.Generic.IEnumerable$1}    seriesColumns           Optional. Names of columns to be considered as series.
                 * @param   {string}                                      xColumn                 Optional. Name of column to be considered as an argument.
                 * @param   {System.Collections.Generic.IEnumerable$1}    yColumns                Optional. Names of columns to be considered as a function.
                 * @return  {Kusto.Charting.IChartMetaData}                                       return the actual argument, series and data columns or null if fails
                 */
                GetMetaData: function (table, argumentColumnType, argumentRestrictions, seriesColumns, xColumn, yColumns) {
                    if (argumentColumnType === void 0) { argumentColumnType = 16; }
                    if (argumentRestrictions === void 0) { argumentRestrictions = 0; }
                    if (seriesColumns === void 0) { seriesColumns = null; }
                    if (xColumn === void 0) { xColumn = null; }
                    if (yColumns === void 0) { yColumns = null; }
                    var tableSchema = table.Kusto$Charting$IChartingDataSource$GetSchema();
                    if (tableSchema == null || !System.Linq.Enumerable.from(tableSchema, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).any()) {
                        return null;
                    }

                    if (seriesColumns == null) {
                        seriesColumns = new (System.Collections.Generic.List$1(System.String)).ctor();
                    }

                    if (yColumns == null) {
                        yColumns = new (System.Collections.Generic.List$1(System.String)).ctor();
                    }

                    var metaData = { v : new Kusto.Charting.DataChartsHelper.ChartMetaData(argumentColumnType) };
                    Kusto.Charting.DataChartsHelper.ResolvePredefinedColumnsIndexes(table, seriesColumns, yColumns, xColumn, argumentRestrictions, metaData);

                    var isTableValidForCharting = false;
                    if (!metaData.v.IsDataFormedAsSeries) {
                        isTableValidForCharting = Kusto.Charting.DataChartsHelper.DetectChartDimensionsUsingColumnTypesAndData(tableSchema, table, argumentRestrictions, metaData);
                        metaData.v.IsDataFormedAsSeries = !isTableValidForCharting;
                    }

                    if (metaData.v.IsDataFormedAsSeries) {
                        isTableValidForCharting = Kusto.Charting.DataChartsHelper.DetectChartDimensionsUsingData(tableSchema, table, seriesColumns, argumentRestrictions, metaData);
                    }

                    if (!isTableValidForCharting) {
                        return null;
                    }

                    // Explode & Filter Data indexes 
                    var tempDataIndex = metaData.v.DataIndexes;
                    metaData.v.DataIndexesList = System.Linq.Enumerable.from(tableSchema, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).select(function (col, index) {
                            return (index !== metaData.v.ArgumentDataColumnIndex && !metaData.v.GeospatiaColumnlIndexesList.contains(index) && (!System.Linq.Enumerable.from(tempDataIndex, System.Int32).any() || System.Linq.Enumerable.from(tempDataIndex, System.Int32).contains(index)) && !System.Linq.Enumerable.from(metaData.v.SeriesIndexes, System.Int32).contains(index) && ((!metaData.v.IsDataFormedAsSeries && System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.NumericOrDateTimeOrTimeSpan, Bridge.box(col.Item2, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType)))) || (metaData.v.IsDataFormedAsSeries && System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.StringOrObject, Bridge.box(col.Item2, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))))) ? index : -1);
                        }).where($asm.$.Kusto.Charting.DataChartsHelper.f2).toList(System.Int32);

                    if (metaData.v.ArgumentColumnType === Kusto.Charting.ArgumentColumnType.Geospatial) {
                        if (metaData.v.DataIndexesList.Count > 1) {
                            // in geospatial case, we want just the 1st data index, we can't afford to have more than one geo point just because the input rows has additional data index columns.
                            var firstDataIndex = System.Linq.Enumerable.from(metaData.v.DataIndexesList, System.Int32).first();
                            metaData.v.DataIndexesList.clear();
                            metaData.v.DataIndexesList.add(firstDataIndex);
                        }

                        if (argumentRestrictions === Kusto.Charting.ArgumentRestrictions.GeospatialAsSeries && (metaData.v.ArgumentDataColumnIndex === -1 || !System.Linq.Enumerable.from(metaData.v.DataIndexesList, System.Int32).any())) {
                            // not sufficient data for presenting map pie chart. Both ArgumentData and it's value are required, in addition to geo coordinates.
                            return null;
                        }
                    }

                    // Update unused indexes
                    metaData.v.UnusedIndexes = System.Linq.Enumerable.range(0, System.Linq.Enumerable.from(tableSchema, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).count()).except(metaData.v.DataIndexes).except(metaData.v.GeospatialColumnIndexes).except(metaData.v.SeriesIndexes).except(System.Array.init([metaData.v.ArgumentDataColumnIndex], System.Int32)).toList(System.Int32);

                    return metaData.v;
                },
                /**
                 * Detects if provided type is numeric.
                 *
                 * @static
                 * @public
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {System.Type}    type                                    Type to be analyzed.
                 * @param   {boolean}        considerDateTimeAndTimeSpanAsNumeric    Optional. Defines if DateTime and TimeSpan should be considered as numeric.
                 Default - true.
                 * @return  {boolean}                                                True, if provided type is numreic, false - if not.
                 */
                IsNumericType: function (type, considerDateTimeAndTimeSpanAsNumeric) {
                    if (considerDateTimeAndTimeSpanAsNumeric === void 0) { considerDateTimeAndTimeSpanAsNumeric = true; }
                    if (type == null) {
                        return false;
                    } else if (Bridge.referenceEquals(type, System.TimeSpan) || Bridge.referenceEquals(type, System.DateTime)) {
                        return considerDateTimeAndTimeSpanAsNumeric;
                    } else if (Bridge.referenceEquals(type, System.Byte) || Bridge.referenceEquals(type, System.Decimal) || Bridge.referenceEquals(type, System.Double) || Bridge.referenceEquals(type, System.Int16) || Bridge.referenceEquals(type, System.Int32) || Bridge.referenceEquals(type, System.Int64) || Bridge.referenceEquals(type, System.SByte) || Bridge.referenceEquals(type, System.UInt16) || Bridge.referenceEquals(type, System.UInt32) || Bridge.referenceEquals(type, System.UInt64)) {
                        return true;
                    } else if (Bridge.referenceEquals(type, System.Object)) {
                        if (Bridge.Reflection.isGenericType(type) && Bridge.referenceEquals(Bridge.Reflection.getGenericTypeDefinition(type), System.Nullable$1)) {
                            return Kusto.Charting.DataChartsHelper.IsNumericType(System.Nullable.getUnderlyingType(type), considerDateTimeAndTimeSpanAsNumeric);
                        }
                    }

                    return false;
                },
                /**
                 * Returns names of columns, from provided data source, which which may be considered as an argument.
                 *
                 * @static
                 * @public
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {Kusto.Charting.IChartingDataSource}          table               Data source, which has to implement IChartingDataSource.
                 * @param   {System.Collections.Generic.IEnumerable$1}    columnsToExclude    Optional. Columns to be excluded from detection.
                 * @return  {System.Collections.Generic.IEnumerable$1}                        Collection of columns' names.
                 */
                GetAllArgumentColumns: function (table, columnsToExclude) {
                    if (columnsToExclude === void 0) { columnsToExclude = null; }
                    var result = new (System.Collections.Generic.List$1(System.String)).ctor();
                    var schema = table.Kusto$Charting$IChartingDataSource$GetSchema();
                    var n = System.Linq.Enumerable.from(table.Kusto$Charting$IChartingDataSource$GetSchema(), System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).count();
                    var columnsToExcludeDefined = columnsToExclude != null && System.Linq.Enumerable.from(columnsToExclude, System.String).any();
                    if (n > 1) {
                        for (var i = 0; i < n; i = (i + 1) | 0) {
                            var columnName = System.Linq.Enumerable.from(schema, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).elementAt(i).Item1;
                            if (columnsToExcludeDefined && System.Linq.Enumerable.from(columnsToExclude, System.String).contains(columnName)) {
                                continue;
                            }

                            result.add(columnName);
                        }
                    }

                    return result;
                },
                /**
                 * Detects the first column of type string.
                 *
                 * @static
                 * @public
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {Kusto.Charting.IChartingDataSource}    table           Data source, which has to implement IChartingDataSource.
                 * @param   {number}                                amountToSkip    Amount of columns type string to be skipped
                 * @return  {string}                                                Name of first column of type string.
                 */
                GetFirstStringColumnName: function (table, amountToSkip) {
                    if (amountToSkip === void 0) { amountToSkip = 0; }
                    var schema = table.Kusto$Charting$IChartingDataSource$GetSchema();
                    var n = System.Linq.Enumerable.from(table.Kusto$Charting$IChartingDataSource$GetSchema(), System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).count();
                    for (var i = 0; i < n; i = (i + 1) | 0) {
                        if (System.Linq.Enumerable.from(schema, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).elementAt(i).Item2 === Kusto.Charting.ArgumentColumnType.String) {
                            if (amountToSkip === 0) {
                                return System.Linq.Enumerable.from(schema, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).elementAt(i).Item1;
                            } else {
                                amountToSkip = (amountToSkip - 1) | 0;
                            }
                        }
                    }

                    return null;
                },
                /**
                 * Method fills gaps in arguments sequence in collection of DataItems,
                 pasting new items with NaN values
                 *
                 * @static
                 * @public
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {System.Collections.Generic.List$1}    data       Input collection of DataItems. For consistent result, data should be sorted.
                 * @param   {Kusto.Charting.ArgumentColumnType}    argType    Type of argument
                 * @return  {System.Collections.Generic.List$1}
                 */
                FillGapsWithNaNs: function (data, argType) {
                    var $t;
                    if (data == null || !System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).any() || data.Count < Kusto.Charting.DataChartsHelper.c_minElementsAmountToFillGapsWithNans) {
                        return data;
                    }

                    var dataGroups = System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).groupBy($asm.$.Kusto.Charting.DataChartsHelper.f3);
                    var result = new (System.Collections.Generic.List$1(Kusto.Charting.DataItem)).$ctor2(data.Count);
                    //filling gaps for each serie
                    $t = Bridge.getEnumerator(dataGroups);
                    try {
                        while ($t.moveNext()) {
                            var dg = $t.Current;
                            var serieDataItems = Kusto.Charting.DataChartsHelper.FillGapsWithNaNs$1(dg.select($asm.$.Kusto.Charting.DataChartsHelper.f4), argType);
                            result.AddRange(serieDataItems);
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return result;
                },
                FillGapsWithNaNs$1: function (data, argType) {
                    var $t;
                    if (System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).count() < Kusto.Charting.DataChartsHelper.c_minElementsAmountToFillGapsWithNans) {
                        return data;
                    }

                    var argData;
                    //resolving sequence of argument values
                    switch (argType) {
                        case Kusto.Charting.ArgumentColumnType.TimeSpan: 
                        case Kusto.Charting.ArgumentColumnType.DateTime: 
                            data = System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).orderBy($asm.$.Kusto.Charting.DataChartsHelper.f5);
                            argData = System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).select($asm.$.Kusto.Charting.DataChartsHelper.f6).ToArray(System.Double);
                            break;
                        case Kusto.Charting.ArgumentColumnType.Numeric: 
                            data = System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).orderBy($asm.$.Kusto.Charting.DataChartsHelper.f7);
                            argData = System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).select($asm.$.Kusto.Charting.DataChartsHelper.f7).ToArray(System.Double);
                            break;
                        default: 
                            return data;
                    }

                    if (argData == null || !System.Linq.Enumerable.from(argData, System.Double).any()) {
                        return data;
                    }

                    //resolving collection of distances between values of argument
                    var distances = Kusto.Charting.DataChartsHelper.SelectSuccessivePairs(argData);
                    //getting the distance which appears most often and the number of appearances
                    var maxGroup = Kusto.Charting.DataChartsHelper.GetTopGroupByCount(distances, argType);
                    var initialDistance = maxGroup.Item1;
                    if (maxGroup == null || System.Convert.toDouble(Bridge.box(maxGroup.Item2, System.Int32)) / System.Linq.Enumerable.from(distances, System.Double).count() < Kusto.Charting.DataChartsHelper.c_minFractionOfIntervalsToDetectDistribution) {
                        return data;
                    }

                    var result = new (System.Collections.Generic.List$1(Kusto.Charting.DataItem)).$ctor2(System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).count());
                    var prevItem = null;
                    $t = Bridge.getEnumerator(data, Kusto.Charting.DataItem);
                    try {
                        while ($t.moveNext()) {
                            var item = $t.Current;
                            if (prevItem == null) {
                                result.add(item);
                                prevItem = item;
                                continue;
                            }

                            var currentDistance = Kusto.Charting.DataChartsHelper.GetDistance(prevItem, item, argType);
                            //if currentDistance is larger than initialDistance - we assume a gap
                            if (currentDistance - initialDistance > Kusto.Charting.DataChartsHelper.c_doubleAcceptableApproximation) {
                                var gapLength = 0;
                                //amount of initialDistances in the gap
                                try {
                                    gapLength = System.Convert.toInt32(Bridge.box(Bridge.Math.round(currentDistance / initialDistance, 0, 6), System.Double, System.Double.format, System.Double.getHashCode));
                                } catch ($e1) {
                                    $e1 = System.Exception.create($e1);
                                    return data;
                                }

                                // regular modulo (%) does not work well with doubles, so using next formula
                                var mod = currentDistance - gapLength * initialDistance;
                                // if amount of initialDistances in currentDistance is not integer, gaps can't be filled
                                if (mod > Kusto.Charting.DataChartsHelper.c_doubleAcceptableApproximation) {
                                    return data;
                                }

                                //filling the gap with DataItems in which ValueData is set to NaN
                                for (var j = 1; j < gapLength; j = (j + 1) | 0) {
                                    var newItem = prevItem.Clone();
                                    newItem.ValueData = Number.NaN;
                                    switch (argType) {
                                        case Kusto.Charting.ArgumentColumnType.Numeric: 
                                            newItem.ArgumentNumeric += initialDistance * j;
                                            newItem.ArgumentData = System.Double.format(newItem.ArgumentNumeric);
                                            break;
                                        case Kusto.Charting.ArgumentColumnType.DateTime: 
                                            newItem.ArgumentDateTime = System.DateTime.addTicks(newItem.ArgumentDateTime, System.Convert.toInt64(Bridge.box(initialDistance, System.Double, System.Double.format, System.Double.getHashCode)).mul(System.Int64(j)));
                                            newItem.ArgumentData = System.DateTime.format(newItem.ArgumentDateTime);
                                            break;
                                        case Kusto.Charting.ArgumentColumnType.TimeSpan: 
                                            newItem.ArgumentDateTime = System.DateTime.addTicks(newItem.ArgumentDateTime, System.Convert.toInt64(Bridge.box(initialDistance, System.Double, System.Double.format, System.Double.getHashCode)).mul(System.Int64(j)));
                                            newItem.ArgumentData = Bridge.toString(System.TimeSpan.fromTicks(System.DateTime.getTicks(newItem.ArgumentDateTime)));
                                            break;
                                    }

                                    result.add(newItem);
                                }
                            }

                            result.add(item);
                            prevItem = item;
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return result;
                },
                /**
                 * Method resolves probable type of argument based on generated data items
                 *
                 * @static
                 * @public
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {System.Collections.Generic.IEnumerable$1}    data    List of DataItems, which used as data source for visualization
                 * @return  {Kusto.Charting.ArgumentColumnType}                   Type of data
                 */
                ResolveArgumentType: function (data) {
                    if (data == null || !System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).any()) {
                        return Kusto.Charting.ArgumentColumnType.None;
                    }

                    var ts = { };
                    if (System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).all(function (d) {
                            return !System.String.isNullOrEmpty(d.ArgumentData) && System.String.indexOf(d.ArgumentData, ":") > 0 && System.TimeSpan.tryParse(d.ArgumentData, null, ts);
                        })) {
                        return Kusto.Charting.ArgumentColumnType.TimeSpan;
                    }

                    if (System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).all($asm.$.Kusto.Charting.DataChartsHelper.f8)) {
                        return Kusto.Charting.ArgumentColumnType.DateTime;
                    }

                    if (System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).any($asm.$.Kusto.Charting.DataChartsHelper.f9)) {
                        return Kusto.Charting.ArgumentColumnType.Numeric;
                    }

                    return Kusto.Charting.ArgumentColumnType.String;
                },
                /**
                 * The method checks whether the data is suitable for visualization based on provided limits
                 *
                 * @static
                 * @public
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {System.Collections.Generic.IEnumerable$1}    data       Data co check
                 * @param   {Kusto.Charting.ArgumentColumnType}           argType    Type of argument
                 * @param   {Kusto.Charting.ChartLimitsPolicy}            limits     Data limitations
                 * @param   {System.String}                               error      Description of found overlimits
                 * @return  {Kusto.Charting.ValidationStatus}                        State of processed validation
                 */
                ValidateData: function (data, argType, limits, error) {
                    var $t;
                    error.v = "";

                    if (data == null || !System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).any()) {
                        error.v = "Data was not provided";
                        return Kusto.Charting.ValidationStatus.PolicyViolationError;
                    }

                    // Check that all values are finite
                    if (System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).where($asm.$.Kusto.Charting.DataChartsHelper.f10).any()) {
                        error.v = "Data includes non-finite values";
                        return Kusto.Charting.ValidationStatus.PolicyViolationError;
                    }

                    // Check validity of Geo coordinates
                    if (argType === Kusto.Charting.ArgumentColumnType.Geospatial && System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).any($asm.$.Kusto.Charting.DataChartsHelper.f11)) {
                        error.v = "Data includes invalid geospatial coordinates. Longitude values must be in range [-180,180]. Latitude values must be in range [-90,90].";
                        return Kusto.Charting.ValidationStatus.PolicyViolationError;
                    }

                    if (argType === Kusto.Charting.ArgumentColumnType.DateTime && System.TimeSpan.neq(limits.MaxDatetimePeriodError, Bridge.getDefaultValue(System.TimeSpan))) {
                        var minValue = System.DateTime.getMaxValue();
                        var maxValue = System.DateTime.getMinValue();
                        $t = Bridge.getEnumerator(data, Kusto.Charting.DataItem);
                        try {
                            while ($t.moveNext()) {
                                var item = $t.Current;
                                if (System.DateTime.lt(maxValue, item.ArgumentDateTime)) {
                                    maxValue = item.ArgumentDateTime;
                                }

                                if (System.DateTime.gt(minValue, item.ArgumentDateTime)) {
                                    minValue = item.ArgumentDateTime;
                                }
                            }
                        } finally {
                            if (Bridge.is($t, System.IDisposable)) {
                                $t.System$IDisposable$Dispose();
                            }
                        }
                        if (System.DateTime.gt(minValue, maxValue) || System.TimeSpan.gt((System.DateTime.subdd(maxValue, minValue)), limits.MaxDatetimePeriodError)) {
                            error.v = "Input time range is too wide to fit into chart";
                            return Kusto.Charting.ValidationStatus.PolicyViolationError;
                        }
                    }

                    var seriesCount = System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).select($asm.$.Kusto.Charting.DataChartsHelper.f3).distinct().count();
                    var dataPoints = System.Linq.Enumerable.from(data, Kusto.Charting.DataItem).count();

                    if (seriesCount > limits.MaxSeriesPerChartError) {
                        error.v = "Too many series" + ("\n" || "") + (System.String.format("Provided data contains {0} series which exceeds allowed amount for the chart: {1}", Bridge.box(seriesCount, System.Int32), Bridge.box(limits.MaxSeriesPerChartError, System.Int32)) || "");
                        return Kusto.Charting.ValidationStatus.PolicyViolationError;
                    }

                    if (dataPoints > limits.MaxPointsPerChartError) {
                        error.v = "Too many points" + ("\n" || "") + (System.String.format("Provided data contains {0} points which exceeds allowed amount for the chart: {1}", Bridge.box(dataPoints, System.Int32), Bridge.box(limits.MaxPointsPerChartError, System.Int32)) || "");
                        return Kusto.Charting.ValidationStatus.PolicyViolationError;
                    }

                    if (seriesCount > limits.MaxSeriesPerChartWarning || dataPoints > limits.MaxPointsPerChartWarning) {
                        error.v = "Too many series or data points" + ("\n" || "") + "Chart can perform badly when large amount of data is used." + ("\n" || "") + (System.String.format("You are about to plot {0} series with {1} points.", Bridge.box(seriesCount, System.Int32), Bridge.box(dataPoints, System.Int32)) || "");
                        return Kusto.Charting.ValidationStatus.PolicyViolationWarning;
                    }

                    return Kusto.Charting.ValidationStatus.Valid;
                },
                /**
                 * Function resolves the type of data from JSON array represented as a string
                 *
                 * @static
                 * @public
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {string}                               value    Input string. Expected to be JSON array
                 * @return  {Kusto.Charting.ArgumentColumnType}             Type of data
                 */
                ResolveJsonArrayType: function (value) {
                    var arr;
                    try {
                        arr = Newtonsoft.Json.JsonConvert.DeserializeObject(value, System.Array.type(System.String));
                    } catch ($e1) {
                        $e1 = System.Exception.create($e1);
                        return Kusto.Charting.ArgumentColumnType.None;
                    }

                    if (arr == null || System.Linq.Enumerable.from(arr, System.String).count() === 0) {
                        return Kusto.Charting.ArgumentColumnType.None;
                    }

                    if (Kusto.Charting.DataChartsHelper.ArrayIsDouble(arr)) {
                        return Kusto.Charting.ArgumentColumnType.Numeric;
                    }

                    if (Kusto.Charting.DataChartsHelper.ArrayIsTimespan(arr)) {
                        return Kusto.Charting.ArgumentColumnType.TimeSpan;
                    }

                    if (Kusto.Charting.DataChartsHelper.ArrayIsDatetime(arr)) {
                        return Kusto.Charting.ArgumentColumnType.DateTime;
                    }

                    return Kusto.Charting.ArgumentColumnType.String;
                },
                /**
                 * Calculates intervals between values in collection
                 *
                 * @static
                 * @private
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {Array.<number>}    collection
                 * @return  {Array.<number>}                  Array of doubles
                 */
                SelectSuccessivePairs: function (collection) {
                    var $t;
                    if (collection == null || System.Linq.Enumerable.from(collection, System.Double).count() <= 1) {
                        return null;
                    }

                    var result = System.Array.init(((System.Linq.Enumerable.from(collection, System.Double).count() - 1) | 0), 0, System.Double);
                    var isFirstElement = true;
                    var prev = Bridge.getDefaultValue(System.Double);
                    var idx = 0;
                    $t = Bridge.getEnumerator(collection);
                    try {
                        while ($t.moveNext()) {
                            var item = $t.Current;
                            if (isFirstElement) {
                                isFirstElement = false;
                                prev = item;
                                continue;
                            }

                            result[System.Array.index(idx, result)] = Math.abs(item - prev);
                            idx = (idx + 1) | 0;
                            prev = item;
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return result;
                },
                GetTopGroupByCount: function (data, sequenceType) {
                    var $t;
                    var sorted = System.Linq.Enumerable.from(data, System.Double).orderBy($asm.$.Kusto.Charting.DataChartsHelper.f4);
                    var currentGroup = new (System.Collections.Generic.List$1(System.Double)).$ctor2(System.Linq.Enumerable.from(data, System.Double).count());
                    var maxGroupCount = 0;
                    var maxGroupValue = 0.0;
                    var prevNum = null;
                    $t = Bridge.getEnumerator(sorted);
                    try {
                        while ($t.moveNext()) {
                            var num = $t.Current;
                            if (!System.Nullable.hasValue(prevNum)) {
                                prevNum = num;
                                currentGroup.add(num);
                                continue;
                            }

                            var diff = num - System.Nullable.getValue(prevNum);
                            //if the difference is larger than acceptable for error of double type, we assume the start of a new group
                            if (diff > Kusto.Charting.DataChartsHelper.c_doubleAcceptableApproximation) {
                                if (System.Linq.Enumerable.from(currentGroup, System.Double).count() > maxGroupCount) {
                                    maxGroupCount = System.Linq.Enumerable.from(currentGroup, System.Double).count();
                                    maxGroupValue = Bridge.Math.round(System.Linq.Enumerable.from(currentGroup, System.Double).average(), Kusto.Charting.DataChartsHelper.c_decimalRoundingPrecision, 6);
                                }

                                currentGroup.clear();
                            }

                            currentGroup.add(num);
                            prevNum = num;
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return { Item1: maxGroupValue, Item2: maxGroupCount };
                },
                GetDistance: function (start, end, sequenceType) {
                    if (start == null || end == null) {
                        return -1;
                    }

                    var distance = -1;
                    switch (sequenceType) {
                        case Kusto.Charting.ArgumentColumnType.TimeSpan: 
                        case Kusto.Charting.ArgumentColumnType.DateTime: 
                            distance = System.Int64.toNumber(System.DateTime.getTicks(end.ArgumentDateTime).sub(System.DateTime.getTicks(start.ArgumentDateTime)));
                            break;
                        case Kusto.Charting.ArgumentColumnType.Numeric: 
                            distance = end.ArgumentNumeric - start.ArgumentNumeric;
                            break;
                    }

                    return Bridge.Math.round(Math.abs(distance), Kusto.Charting.DataChartsHelper.c_decimalRoundingPrecision, 6);
                },
                ResolvePredefinedColumnsIndexes: function (data, seriesColumns, yColumns, xColumn, argumentRestrictions, metaData) {
                    var dataIsSeries = null;
                    var columns = data.Kusto$Charting$IChartingDataSource$GetSchema();

                    if (argumentRestrictions === Kusto.Charting.ArgumentRestrictions.GeospatialAsSeries && System.Linq.Enumerable.from(seriesColumns, System.String).any() && (System.Linq.Enumerable.from(yColumns, System.String).any() || !System.String.isNullOrWhiteSpace(xColumn))) {
                        // in GeospatialAsSeries case it's not possible to set both series and x\y columns,
                        // because both have exactly the same meaning, so they can't point to a different columns.
                        throw new Kusto.Charting.SeriesCreationException("GeospatialAsSeries: it's not possible to set both series and x/y columns.");
                    }

                    for (var i = 0; i < System.Linq.Enumerable.from(columns, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).count(); i = (i + 1) | 0) {
                        var column = System.Linq.Enumerable.from(columns, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).elementAt(i);
                        var columnName = column.Item1;
                        if (System.Linq.Enumerable.from(seriesColumns, System.String).any() && System.Linq.Enumerable.from(seriesColumns, System.String).contains(columnName)) {
                            metaData.v.SeriesIndexesList.add(i);

                            if (argumentRestrictions === Kusto.Charting.ArgumentRestrictions.GeospatialAsSeries) {
                                // geo coordinates is the series (map pie chart), therefore predefining series means setting geo coordinates
                                metaData.v.GeospatiaColumnlIndexesList.add(i);
                            }

                            continue;
                        }

                        var isY = System.Linq.Enumerable.from(yColumns, System.String).any() && System.Linq.Enumerable.from(yColumns, System.String).contains(columnName);
                        var isX = !System.String.isNullOrWhiteSpace(xColumn) && Bridge.referenceEquals(xColumn, columnName);
                        if (isY || isX) {
                            var columnType = column.Item2;
                            var isSeries = false;
                            if (System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.StringOrObject, Bridge.box(columnType, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) && metaData.v.ArgumentColumnType !== Kusto.Charting.ArgumentColumnType.Geospatial) {
                                //checking if data formed as series
                                var value = Bridge.toString(data.Kusto$Charting$IChartingDataSource$GetValue(0, i));
                                var type = Kusto.Charting.DataChartsHelper.ResolveJsonArrayType(value);
                                if (type !== Kusto.Charting.ArgumentColumnType.None) {
                                    columnType = type;
                                    isSeries = true;
                                }
                            }

                            if (System.Nullable.hasValue(dataIsSeries)) {
                                if (System.Nullable.getValue(dataIsSeries) !== isSeries) {
                                    throw new Kusto.Charting.SeriesCreationException("Y-Axes and X-Axis both should be defined as scalars or as series");
                                }
                            } else {
                                dataIsSeries = isSeries;
                            }

                            if (metaData.v.ArgumentColumnType === Kusto.Charting.ArgumentColumnType.Geospatial) {
                                if (metaData.v.GeospatiaColumnlIndexesList.Count > 0 && isX) {
                                    // fixing order to [longitude, latitude]
                                    metaData.v.GeospatiaColumnlIndexesList.insert(0, i);
                                } else {
                                    metaData.v.GeospatiaColumnlIndexesList.add(i);
                                }

                                if (argumentRestrictions === Kusto.Charting.ArgumentRestrictions.GeospatialAsSeries) {
                                    // in GeospatialAsSeries case, geo column is a series so we fill series index as well
                                    if (metaData.v.SeriesIndexesList.Count > 0 && isX) {
                                        // fixing order to [longitude, latitude]
                                        metaData.v.SeriesIndexesList.insert(0, i);
                                    } else {
                                        metaData.v.SeriesIndexesList.add(i);
                                    }
                                }
                            } else if (isX) {
                                if (System.Enum.hasFlag(metaData.v.ArgumentColumnType, Bridge.box(columnType, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType)))) {
                                    metaData.v.ArgumentDataColumnIndex = i;
                                } else {
                                    throw new Kusto.Charting.SeriesCreationException(System.String.format("Type of column {0}, provided as X-Axis, does not match required by chart type", [columnName]));
                                }
                            } else if (isY) {
                                if (System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.NumericOrDateTimeOrTimeSpan, Bridge.box(columnType, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType)))) {
                                    metaData.v.DataIndexesList.add(i);
                                } else {
                                    throw new Kusto.Charting.SeriesCreationException(System.String.format("Column {0}, provided as Y-Axis, sould be one of types: Numeric, DateTime, Timespan", [columnName]));
                                }
                            }
                        }
                    }

                    metaData.v.IsDataFormedAsSeries = System.Nullable.hasValue(dataIsSeries) ? System.Nullable.getValue(dataIsSeries) : false;

                    if (!System.Linq.Enumerable.from(metaData.v.DataIndexesList, System.Int32).any() && System.Linq.Enumerable.from(yColumns, System.String).any() && metaData.v.ArgumentColumnType !== Kusto.Charting.ArgumentColumnType.Geospatial) {
                        throw new Kusto.Charting.SeriesCreationException("Any of columns defined as Y-Axes were not found in data, not of an appropriate type or used as argument or series");
                    }

                    if (!System.String.isNullOrWhiteSpace(xColumn) && metaData.v.ArgumentDataColumnIndex < 0 && metaData.v.ArgumentColumnType !== Kusto.Charting.ArgumentColumnType.Geospatial) {
                        throw new Kusto.Charting.SeriesCreationException(System.String.format("Column {0}, provided as X-Axis, was not found in data", [xColumn]));
                    }

                    if (System.Linq.Enumerable.from(seriesColumns, System.String).any() && !System.Linq.Enumerable.from(metaData.v.SeriesIndexesList, System.Int32).any()) {
                        throw new Kusto.Charting.SeriesCreationException("Any of columns, provided as Series, were not found in data");
                    }
                },
                ResolveDataSeriesFromDataRow: function (result, table, columns, propertyColumns, rowIdx, baseSeriesName, argumentColumnType, accumulate, argumentDataColumnIndex) {
                    var $t, $t1, $t2;

                    var argumentValue = table.Kusto$Charting$IChartingDataSource$GetValue(rowIdx, argumentDataColumnIndex);
                    var argumentActualType = Kusto.Charting.DataChartsHelper.ResolveJsonArrayType(Bridge.toString(argumentValue));
                    // In case the first column to be series, and the last one before numeric one - to be the ArgumentData
                    $t = Bridge.getEnumerator(columns, Kusto.Charting.DataChartsHelper.ColumnDesc);
                    try {
                        while ($t.moveNext()) {
                            var column = $t.Current;
                            var cellValue = table.Kusto$Charting$IChartingDataSource$GetValue(rowIdx, column.Index);
                            var value = ($t1 = (Bridge.as(cellValue, System.String)), $t1 != null ? $t1 : (Bridge.toString(cellValue)));
                            var type = Kusto.Charting.DataChartsHelper.ResolveJsonArrayType(value);
                            if (type === Kusto.Charting.ArgumentColumnType.None) {
                                continue;
                            }

                            if (System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.NumericOrDateTimeOrTimeSpan, Bridge.box(type, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType)))) {

                                var values = Kusto.Charting.DataChartsHelper.ParseJsonArrayAsDouble(value, true);
                                if (values == null || System.Linq.Enumerable.from(value, System.Char).count() === 0) {
                                    continue;
                                }

                                var seriesName = (System.String.isNullOrEmpty(baseSeriesName)) ? column.Name : (baseSeriesName || "") + ":" + (column.Name || "");
                                var lastValue = 0;
                                var argumentDateTime = Kusto.Charting.DataChartsHelper.GetArgumentDateTimeArray(argumentValue, argumentActualType, argumentDataColumnIndex, System.Linq.Enumerable.from(values, System.Double).count());
                                var argumentNumeric = Kusto.Charting.DataChartsHelper.GetArgumentNumericArray(argumentValue, argumentActualType, argumentDataColumnIndex, System.Linq.Enumerable.from(values, System.Double).count());
                                var argumentString = Kusto.Charting.DataChartsHelper.GetArgumentStringArray(argumentValue, argumentActualType, argumentDataColumnIndex, System.Linq.Enumerable.from(values, System.Double).count());
                                var argumentProperties = System.Linq.Enumerable.from(propertyColumns, Kusto.Charting.DataChartsHelper.ColumnDesc).select(function (p) {
                                        return table.Kusto$Charting$IChartingDataSource$GetValue(rowIdx, p.Index);
                                    }).select($asm.$.Kusto.Charting.DataChartsHelper.f12).ToArray(System.Object);

                                for (var j = 0; j < values.length; j = (j + 1) | 0) {
                                    var argumentData = "";
                                    if (System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.DateTimeOrTimeSpan, Bridge.box(argumentColumnType, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) && System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.DateTimeOrTimeSpan, Bridge.box(argumentActualType, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType)))) {
                                        argumentData = System.DateTime.format(argumentDateTime[System.Array.index(j, argumentDateTime)]);
                                    } else if (System.Enum.hasFlag(argumentColumnType, Bridge.box(Kusto.Charting.ArgumentColumnType.Numeric, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) && argumentActualType === Kusto.Charting.ArgumentColumnType.Numeric) {
                                        argumentData = System.Double.format(argumentNumeric[System.Array.index(j, argumentNumeric)]);
                                    } else if (System.Enum.hasFlag(argumentColumnType, Bridge.box(Kusto.Charting.ArgumentColumnType.String, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType)))) {
                                        argumentData = argumentString[System.Array.index(j, argumentString)];
                                        if (System.String.isNullOrEmpty(argumentData)) {
                                            argumentData = Bridge.as(argumentValue, System.String);
                                        }
                                    }

                                    var dataItem = ($t2 = new Kusto.Charting.DataItem.ctor(), $t2.ArgumentData = argumentData, $t2.ArgumentDateTime = argumentDateTime[System.Array.index(j, argumentDateTime)], $t2.ArgumentNumeric = argumentNumeric[System.Array.index(j, argumentNumeric)], $t2.ValueData = accumulate ? values[System.Array.index(j, values)] + lastValue : values[System.Array.index(j, values)], $t2.ValueName = column.Name, $t2.SeriesName = seriesName, $t2.Properties = Kusto.Charting.DataChartsHelper.ResolvePropertiesFromMultiValue(table, argumentProperties, propertyColumns, j), $t2);

                                    // Correction for charts - ArgumentData cannot be empty for some charts
                                    if (System.String.isNullOrEmpty(dataItem.ArgumentData)) {
                                        dataItem.ArgumentData = Kusto.Charting.DataChartsHelper.c_defaultArgumentString;
                                    }
                                    result.add(dataItem);
                                    lastValue = dataItem.ValueData;
                                }
                            }
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }
                },
                ResolveProperties: function (table, rowIdx, propertyColumns) {
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(Kusto.Charting.DataChartsHelper.ColumnDesc, propertyColumns)) {
                        return "";
                    }

                    return Bridge.toArray(System.Linq.Enumerable.from(propertyColumns, Kusto.Charting.DataChartsHelper.ColumnDesc).select(function (column) {
                                var cellValue = table.Kusto$Charting$IChartingDataSource$GetValue(rowIdx, column.Index);
                                var value = Kusto.Charting.DataChartsHelper.ObjectToString(cellValue);
                                return System.String.format("{0}:{1}", column.Name, value);
                            })).join(", ");
                },
                ResolvePropertiesFromMultiValue: function (table, properties, propertyColumns, j) {
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(Kusto.Charting.DataChartsHelper.ColumnDesc, propertyColumns)) {
                        return "";
                    }

                    return Bridge.toArray(System.Linq.Enumerable.from(propertyColumns, Kusto.Charting.DataChartsHelper.ColumnDesc).zip(properties, $asm.$.Kusto.Charting.DataChartsHelper.f13).select(function (kvp) {
                            var value;
                            var arr;
                            if (((arr = Bridge.as(kvp.Item2, System.Array.type(System.String)))) != null) {
                                value = arr[System.Array.index(j, arr)];
                            } else {
                                value = Kusto.Charting.DataChartsHelper.ObjectToString(kvp.Item2);
                            }
                            return System.String.format("{0}:{1}", kvp.Item1.Name, value);
                        })).join(", ");
                },
                ObjectToString: function (o) {
                    var s;
                    if (((s = Bridge.as(o, System.String))) != null) {
                        return s;
                    } else {
                        var d;
                        if (!Bridge.equals(((d = Bridge.is(o, System.DateTime) ? System.Nullable.getValue(Bridge.cast(Bridge.unbox(o, System.DateTime), System.DateTime)) : null)), null)) {
                            return Kusto.Cloud.Platform.Utils.ExtendedDateTime.ToUtcString(d);
                        } else {
                            return Bridge.toString(o);
                        }
                    }
                },
                GetArgumentStringArray: function (value, argumentColumnType, argumentDataColumnIndex, count) {
                    if (!System.Enum.hasFlag(argumentColumnType, Bridge.box(Kusto.Charting.ArgumentColumnType.String, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) || argumentDataColumnIndex < 0) {
                        return System.Array.init(count, null, System.String);
                    }

                    var result = Kusto.Charting.DataChartsHelper.ParseJsonArrayAsString(Bridge.toString(value));
                    if (result == null) {
                        return System.Array.init(count, null, System.String);
                    }

                    return result;
                },
                GetArgumentNumericArray: function (value, argumentColumnType, argumentDataColumnIndex, count) {
                    if (argumentColumnType !== Kusto.Charting.ArgumentColumnType.Numeric || argumentDataColumnIndex < 0) {
                        return System.Linq.Enumerable.repeat(Kusto.Charting.DataChartsHelper.c_defaultArgumentNumeric, count).ToArray(System.Double);
                    }

                    var result = Kusto.Charting.DataChartsHelper.ParseJsonArrayAsDouble(Bridge.toString(value));
                    if (result == null) {
                        return System.Linq.Enumerable.repeat(Kusto.Charting.DataChartsHelper.c_defaultArgumentNumeric, count).ToArray(System.Double);
                    }

                    return result;
                },
                GetArgumentDateTimeArray: function (value, argumentColumnType, argumentDataColumnIndex, count) {
                    if (!System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.DateTimeOrTimeSpan, Bridge.box(argumentColumnType, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) || argumentDataColumnIndex < 0) {
                        return System.Linq.Enumerable.repeat(Kusto.Charting.DataChartsHelper.c_defaultArgumentDatetime, count).ToArray(System.DateTime);
                    }

                    var result = Kusto.Charting.DataChartsHelper.ParseJsonArrayAsDateTime(Bridge.toString(value), argumentColumnType);
                    if (result == null) {
                        return System.Linq.Enumerable.repeat(Kusto.Charting.DataChartsHelper.c_defaultArgumentDatetime, count).ToArray(System.DateTime);
                    }

                    return result;
                },
                ResolveDataItemsFromDataRow: function (result, table, columns, propertyColumns, rowIdx, baseSeriesName, argumentData, lastValues, accumulate) {
                    var $t, $t1;
                    if (!System.Linq.Enumerable.from(columns, Kusto.Charting.DataChartsHelper.ColumnDesc).any() && argumentData.GeospatialArgumentDataType === Kusto.Charting.ArgumentColumnType.Geospatial) {
                        // in map chart, argumentData is data in itself, therefore even if there are no data columns but only geo point data, we are creating data item
                        result.add(($t = new Kusto.Charting.DataItem.ctor(), $t.GeoCoordinates = argumentData.GeoCoordinates, $t.SeriesName = System.String.isNullOrWhiteSpace(baseSeriesName) ? "" : baseSeriesName, $t.Properties = Kusto.Charting.DataChartsHelper.ResolveProperties(table, rowIdx, propertyColumns), $t));
                    }

                    // In case the first column to be series, and the last one before numeric one - to be the ArgumentData
                    $t = Bridge.getEnumerator(columns, Kusto.Charting.DataChartsHelper.ColumnDesc);
                    try {
                        while ($t.moveNext()) {
                            var column = $t.Current;
                            var cellValue = table.Kusto$Charting$IChartingDataSource$GetValue(rowIdx, column.Index);
                            var value = Kusto.Charting.DataChartsHelper.TryConvertToDouble(cellValue, column.Type);

                            var lastValue = { };
                            var dataItem = ($t1 = new Kusto.Charting.DataItem.ctor(), $t1.ArgumentData = argumentData.Value, $t1.ArgumentDateTime = argumentData.DateTime, $t1.ArgumentNumeric = argumentData.NumericValue, $t1.ValueName = column.Name, $t1.Properties = Kusto.Charting.DataChartsHelper.ResolveProperties(table, rowIdx, propertyColumns), $t1);

                            if (argumentData.GeospatialArgumentDataType === Kusto.Charting.ArgumentColumnType.Geospatial) {
                                // in geospatial case data column name is not part of series name
                                dataItem.SeriesName = System.String.isNullOrWhiteSpace(baseSeriesName) ? "" : baseSeriesName;
                                dataItem.GeoCoordinates = argumentData.GeoCoordinates;
                            } else {
                                dataItem.SeriesName = System.String.isNullOrEmpty(baseSeriesName) ? column.Name : (baseSeriesName || "") + ":" + (column.Name || "");
                            }

                            dataItem.ValueData = accumulate && lastValues.tryGetValue(dataItem.SeriesName, lastValue) ? value + lastValue.v : value;

                            result.add(dataItem);
                            lastValues.setItem(dataItem.SeriesName, dataItem.ValueData);
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }
                },
                GetBaseSeriesName: function (table, seriesColumns, rowIdx) {
                    var $t;
                    var baseSeriesName = "";
                    if (seriesColumns != null) {
                        var baseSeriesNameBuilder = new System.Text.StringBuilder();
                        $t = Bridge.getEnumerator(seriesColumns, Kusto.Charting.DataChartsHelper.ColumnDesc);
                        try {
                            while ($t.moveNext()) {
                                var column = $t.Current;
                                if (baseSeriesNameBuilder.getLength() > 0) {
                                    baseSeriesNameBuilder.append(", ");
                                }

                                var columnValue = Bridge.toString(table.Kusto$Charting$IChartingDataSource$GetValue(rowIdx, column.Index));
                                columnValue = System.String.isNullOrWhiteSpace(columnValue) ? Kusto.Charting.DataChartsHelper.c_defaultArgumentString : columnValue;
                                baseSeriesNameBuilder.appendFormat("{0}:{1}", column.Name, columnValue);
                            }
                        } finally {
                            if (Bridge.is($t, System.IDisposable)) {
                                $t.System$IDisposable$Dispose();
                            }
                        }

                        baseSeriesName = baseSeriesNameBuilder.toString();
                    }

                    return baseSeriesName;
                },
                TrySetGeoJSONPoint: function (value, point) {
                    point.v = null;

                    try {
                        point.v = Newtonsoft.Json.JsonConvert.DeserializeObject(value, Kusto.Charting.DataChartsHelper.GeoJSONPoint);
                    } catch ($e1) {
                        $e1 = System.Exception.create($e1);
                        if (Bridge.is($e1, Newtonsoft.Json.JsonException)) {
                            // invalid coordinate data
                            return false;
                        } else {
                            throw $e1;
                        }
                    }

                    if ((point.v != null ? point.v.coordinates : null) == null) {
                        // invalid coordinate data
                        return false;
                    } else if (point.v.coordinates.length !== 2) {
                        // invalid coordinate data
                        return false;
                    }

                    return true;
                },
                GetArgumentDateTime: function (value, argumentColumnType) {
                    if (!System.Enum.hasFlag(argumentColumnType, Bridge.box(Kusto.Charting.ArgumentColumnType.DateTime, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) && !System.Enum.hasFlag(argumentColumnType, Bridge.box(Kusto.Charting.ArgumentColumnType.TimeSpan, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType)))) {
                        return Kusto.Charting.DataChartsHelper.c_defaultArgumentDatetime;
                    }

                    if (Bridge.is(value, System.DateTime)) {
                        return System.Nullable.getValue(Bridge.cast(Bridge.unbox(value, System.DateTime), System.DateTime));
                    } else if (Bridge.is(value, System.TimeSpan)) {
                        return Kusto.Charting.DataChartsHelper.TimeSpanToDateTime(System.Nullable.getValue(Bridge.cast(Bridge.unbox(value, System.TimeSpan), System.TimeSpan)));
                    }

                    return Kusto.Charting.DataChartsHelper.c_defaultArgumentDatetime;
                },
                ConvertToDouble: function (obj, type) {
                    var value = Kusto.Charting.DataChartsHelper.c_defaultArgumentNumeric;
                    if (obj == null) {
                        return value;
                    }

                    if (type === Kusto.Charting.ArgumentColumnType.DateTime) {
                        value = Kusto.Charting.DataChartsHelper.DateTimeToTotalSeconds(System.Nullable.getValue(Bridge.cast(Bridge.unbox(obj, System.DateTime), System.DateTime)));
                    } else if (type === Kusto.Charting.ArgumentColumnType.TimeSpan) {
                        value = Kusto.Charting.DataChartsHelper.TimeSpanToTotalSeconds(System.Nullable.getValue(Bridge.cast(Bridge.unbox(obj, System.TimeSpan), System.TimeSpan)));
                    } else {
                        try {
                            value = System.Convert.toDouble(obj);
                        } catch ($e1) {
                            $e1 = System.Exception.create($e1);
                            value = Kusto.Charting.DataChartsHelper.c_defaultArgumentNumeric;
                        }
                    }

                    return value;
                },
                TryConvertToDouble: function (value, type) {
                    if (value == null || Kusto.Charting.DataChartsHelper.IsEmptyValue(value)) {
                        return Kusto.Charting.DataChartsHelper.c_defaultArgumentNumeric;
                    }

                    return Kusto.Charting.DataChartsHelper.ConvertToDouble(value, type);
                },
                /**
                 * from the table columns and reuirements, deduce the x, y, and series axis.
                 *
                 * @static
                 * @private
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {System.Collections.Generic.IEnumerable$1}         columns                 
                 * @param   {Kusto.Charting.IChartingDataSource}               table                   
                 * @param   {System.Collections.Generic.IEnumerable$1}         seriesColumns           
                 * @param   {Kusto.Charting.ArgumentRestrictions}              argumentRestrictions    
                 * @param   {Kusto.Charting.DataChartsHelper.ChartMetaData}    metaData
                 * @return  {boolean}
                 */
                DetectChartDimensionsUsingData: function (columns, table, seriesColumns, argumentRestrictions, metaData) {
                    var $t;
                    var resolvedColumnTypes = System.Array.init(System.Linq.Enumerable.from(columns, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).count(), 0, Kusto.Charting.ArgumentColumnType);

                    if (table.Kusto$Charting$IChartingDataSource$RowsCount === 0) {
                        return false;
                    }

                    var firstNumericColumnIndex = -1;
                    for (var i = 0; i < System.Linq.Enumerable.from(columns, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).count(); i = (i + 1) | 0) {
                        var column = System.Linq.Enumerable.from(columns, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).elementAt(i);
                        resolvedColumnTypes[System.Array.index(i, resolvedColumnTypes)] = column.Item2;
                        if (metaData.v.DataIndexesList.contains(i)) {
                            continue;
                        }

                        if (column.Item2 === Kusto.Charting.ArgumentColumnType.String || column.Item2 === Kusto.Charting.ArgumentColumnType.Object) {
                            var item = table.Kusto$Charting$IChartingDataSource$GetValue(0, i);
                            var value = ($t = (Bridge.as(item, System.String)), $t != null ? $t : (Bridge.toString(item)));
                            var type = Kusto.Charting.DataChartsHelper.ResolveJsonArrayType(value);
                            if (type === Kusto.Charting.ArgumentColumnType.None) {
                                if (!System.Linq.Enumerable.from(seriesColumns, System.String).any() && metaData.v.ArgumentDataColumnIndex < 0) {
                                    // Add column to series mapping
                                    metaData.v.SeriesIndexesList.add(i);
                                }
                                continue;
                            }

                            resolvedColumnTypes[System.Array.index(i, resolvedColumnTypes)] = type;
                            if (System.Enum.hasFlag(metaData.v.ArgumentColumnType, Bridge.box(type, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) && (metaData.v.ArgumentDataColumnIndex < 0 || System.Enum.hasFlag(argumentRestrictions, Bridge.box(Kusto.Charting.ArgumentRestrictions.PreferLast, Kusto.Charting.ArgumentRestrictions, System.Enum.toStringFn(Kusto.Charting.ArgumentRestrictions))))) {
                                metaData.v.ArgumentDataColumnIndex = i;
                            } else if (type === Kusto.Charting.ArgumentColumnType.Numeric && firstNumericColumnIndex < 0) {
                                firstNumericColumnIndex = i;
                            }
                        }
                    }

                    return Kusto.Charting.DataChartsHelper.CompleteChartDimentionsDetection(columns, argumentRestrictions, firstNumericColumnIndex, resolvedColumnTypes, metaData);
                },
                /**
                 * from the table columns and requirements, deduce the x, y, and series axis.
                 *
                 * @static
                 * @private
                 * @this Kusto.Charting.DataChartsHelper
                 * @memberof Kusto.Charting.DataChartsHelper
                 * @param   {System.Collections.Generic.IEnumerable$1}         columns                 
                 * @param   {Kusto.Charting.IChartingDataSource}               table                   
                 * @param   {Kusto.Charting.ArgumentRestrictions}              argumentRestrictions    
                 * @param   {Kusto.Charting.DataChartsHelper.ChartMetaData}    metaData
                 * @return  {boolean}
                 */
                DetectChartDimensionsUsingColumnTypesAndData: function (columns, table, argumentRestrictions, metaData) {
                    var $t;
                    var firstNumericColumnIndex = -1;
                    var geoJSONPointCandidateColumns = new (System.Collections.Generic.List$1(System.Int32)).ctor(); // candidates for GeoJSON column
                    var geoColumnsPredefined = System.Linq.Enumerable.from(metaData.v.GeospatiaColumnlIndexesList, System.Int32).any();

                    for (var i = 0; i < System.Linq.Enumerable.from(columns, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).count(); i = (i + 1) | 0) {
                        if (metaData.v.DataIndexesList.contains(i) || metaData.v.SeriesIndexesList.contains(i)) {
                            continue;
                        }

                        var column = System.Linq.Enumerable.from(columns, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).elementAt(i);

                        if (metaData.v.ArgumentColumnType === Kusto.Charting.ArgumentColumnType.Geospatial && System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.StringOrObject, Bridge.box(column.Item2, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType)))) {
                            geoJSONPointCandidateColumns.add(i);
                        }

                        if (metaData.v.ArgumentColumnType === Kusto.Charting.ArgumentColumnType.Geospatial && System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.Numeric, Bridge.box(column.Item2, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) && System.Linq.Enumerable.from(metaData.v.GeospatiaColumnlIndexesList, System.Int32).count() < 2 && !geoColumnsPredefined) {
                            // first two numeric columns will be considered as Lng,Lat
                            metaData.v.GeospatiaColumnlIndexesList.add(i);
                        } else if (System.Enum.hasFlag(metaData.v.ArgumentColumnType, Bridge.box(column.Item2, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) && (metaData.v.ArgumentDataColumnIndex < 0 || System.Enum.hasFlag(argumentRestrictions, Bridge.box(Kusto.Charting.ArgumentRestrictions.PreferLast, Kusto.Charting.ArgumentRestrictions, System.Enum.toStringFn(Kusto.Charting.ArgumentRestrictions))))) {
                            metaData.v.ArgumentDataColumnIndex = i;
                        } else if (column.Item2 === Kusto.Charting.ArgumentColumnType.Numeric) {
                            // Give a chance for argument to find itself
                            if (metaData.v.ArgumentDataColumnIndex !== -1 && System.Enum.hasFlag(argumentRestrictions, Bridge.box(Kusto.Charting.ArgumentRestrictions.NumericAsSeries, Kusto.Charting.ArgumentRestrictions, System.Enum.toStringFn(Kusto.Charting.ArgumentRestrictions)))) {
                                firstNumericColumnIndex = i;
                                break;
                            }
                        }
                    }

                    if (metaData.v.ArgumentColumnType === Kusto.Charting.ArgumentColumnType.Geospatial && System.Linq.Enumerable.from(metaData.v.GeospatiaColumnlIndexesList, System.Int32).count() < 2 && !geoColumnsPredefined) {
                        // reset geospatial column index
                        metaData.v.GeospatiaColumnlIndexesList.clear();

                        // Trying to detect GeoJSON Point by looking into candidates in the 1st row
                        $t = Bridge.getEnumerator(geoJSONPointCandidateColumns);
                        try {
                            while ($t.moveNext()) {
                                var columnId = $t.Current;
                                var columnFirstValue = Bridge.toString(table.Kusto$Charting$IChartingDataSource$GetValue(0, columnId));
                                var geojsonPoint = { };
                                if (Kusto.Charting.DataChartsHelper.TrySetGeoJSONPoint(columnFirstValue, geojsonPoint)) {
                                    metaData.v.GeospatiaColumnlIndexesList.add(columnId);
                                    break;
                                }
                            }
                        } finally {
                            if (Bridge.is($t, System.IDisposable)) {
                                $t.System$IDisposable$Dispose();
                            }
                        }
                    }

                    var resolvedColumnTypes = System.Linq.Enumerable.from(columns, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).select($asm.$.Kusto.Charting.DataChartsHelper.f14).ToArray(Kusto.Charting.ArgumentColumnType);
                    return Kusto.Charting.DataChartsHelper.CompleteChartDimentionsDetection(columns, argumentRestrictions, firstNumericColumnIndex, resolvedColumnTypes, metaData);
                },
                CompleteChartDimentionsDetection: function (columns, argumentRestrictions, firstNumericColumnIndex, resolvedColumnTypes, metaData) {
                    var $t;
                    if (metaData.v.ArgumentDataColumnIndex >= 0 && System.Linq.Enumerable.from(metaData.v.SeriesIndexesList, System.Int32).any()) {
                        return true;
                    }
                    // if required argument is of numeric type, there should be at least one numeric column
                    if (metaData.v.ArgumentDataColumnIndex < 0 && firstNumericColumnIndex < 0 && System.Enum.hasFlag(metaData.v.ArgumentColumnType, Bridge.box(Kusto.Charting.ArgumentColumnType.Numeric, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType)))) {
                        return false;
                    } else if (metaData.v.ArgumentColumnType === Kusto.Charting.ArgumentColumnType.Geospatial && !System.Linq.Enumerable.from(metaData.v.GeospatiaColumnlIndexesList, System.Int32).any()) {
                        // failed to detect geospatial column indexes, GeospatiaColumnlIndexesList should have been set at this stage.
                        return false;
                    }

                    if (metaData.v.ArgumentDataColumnIndex < 0 && metaData.v.ArgumentColumnType !== Kusto.Charting.ArgumentColumnType.Geospatial) {
                        if (System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.DateTimeOrTimeSpan, Bridge.box(metaData.v.ArgumentColumnType, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType)))) {
                            // If the argument requested type is string DateTimeOrTimeSpan 
                            // we already know that it was not found
                            return false;
                        }

                        if (System.Enum.hasFlag(metaData.v.ArgumentColumnType, Bridge.box(Kusto.Charting.ArgumentColumnType.Numeric, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType)))) {
                            if (System.Linq.Enumerable.from(columns, System.Tuple$2(System.String,Kusto.Charting.ArgumentColumnType)).count() > 1) {
                                metaData.v.ArgumentDataColumnIndex = firstNumericColumnIndex;
                            }
                        } else {
                            if (System.Enum.hasFlag(argumentRestrictions, Bridge.box(Kusto.Charting.ArgumentRestrictions.NotIncludedInSeries, Kusto.Charting.ArgumentRestrictions, System.Enum.toStringFn(Kusto.Charting.ArgumentRestrictions)))) {
                                metaData.v.ArgumentDataColumnIndex = Kusto.Charting.DataChartsHelper.GoBackwardsAndFindColumnNotInList(firstNumericColumnIndex, metaData.v.SeriesIndexes, metaData.v.DataIndexes);
                            } else {
                                metaData.v.ArgumentDataColumnIndex = (firstNumericColumnIndex - 1) | 0;
                            }
                        }
                    }

                    if (metaData.v.ArgumentDataColumnIndex < 0 && System.Enum.hasFlag(argumentRestrictions, Bridge.box(Kusto.Charting.ArgumentRestrictions.MustHave, Kusto.Charting.ArgumentRestrictions, System.Enum.toStringFn(Kusto.Charting.ArgumentRestrictions)))) {
                        metaData.v.ArgumentDataColumnIndex = 0;
                    }

                    if (metaData.v.ArgumentColumnType === Kusto.Charting.ArgumentColumnType.Geospatial) {
                        var seriesOrArgumentDataIndex = Kusto.Charting.DataChartsHelper.GetFirstStringAvailableColumnIndexOrNonStringIfAbsent(resolvedColumnTypes, metaData.v.GeospatialColumnIndexes);

                        if (argumentRestrictions === Kusto.Charting.ArgumentRestrictions.GeospatialAsSeries) {
                            metaData.v.ArgumentDataColumnIndex = seriesOrArgumentDataIndex;

                            if (!System.Linq.Enumerable.from(metaData.v.SeriesIndexesList, System.Int32).any()) {
                                // geo coordinates is the series (map pie chart)
                                $t = Bridge.getEnumerator(metaData.v.GeospatiaColumnlIndexesList);
                                try {
                                    while ($t.moveNext()) {
                                        var geoColumnIndex = $t.Current;
                                        metaData.v.SeriesIndexesList.add(geoColumnIndex);
                                    }
                                } finally {
                                    if (Bridge.is($t, System.IDisposable)) {
                                        $t.System$IDisposable$Dispose();
                                    }
                                }
                            }
                        } else if (seriesOrArgumentDataIndex !== -1 && !System.Linq.Enumerable.from(metaData.v.SeriesIndexesList, System.Int32).any()) {
                            // set geospatial series column index (map scatter chart series)
                            metaData.v.SeriesIndexesList.add(seriesOrArgumentDataIndex);
                        }
                    } else if (!System.Linq.Enumerable.from(metaData.v.SeriesIndexesList, System.Int32).any() && metaData.v.ArgumentDataColumnIndex >= 0) {
                        var seriesDefaultIndex = metaData.v.ArgumentDataColumnIndex;
                        if (resolvedColumnTypes[System.Array.index(metaData.v.ArgumentDataColumnIndex, resolvedColumnTypes)] !== Kusto.Charting.ArgumentColumnType.String) {
                            seriesDefaultIndex = Kusto.Charting.DataChartsHelper.GetFirstStringColumnIndex(resolvedColumnTypes);
                        } else if (System.Enum.hasFlag(argumentRestrictions, Bridge.box(Kusto.Charting.ArgumentRestrictions.NotIncludedInSeries, Kusto.Charting.ArgumentRestrictions, System.Enum.toStringFn(Kusto.Charting.ArgumentRestrictions)))) {
                            seriesDefaultIndex = (metaData.v.ArgumentDataColumnIndex - 1) | 0; // In case argument shouldn't be included to series - move one before
                        }

                        if (!metaData.v.IsDataFormedAsSeries) {
                            if (seriesDefaultIndex === -1 && System.Enum.hasFlag(argumentRestrictions, Bridge.box(Kusto.Charting.ArgumentRestrictions.NumericAsSeries, Kusto.Charting.ArgumentRestrictions, System.Enum.toStringFn(Kusto.Charting.ArgumentRestrictions)))) {
                                seriesDefaultIndex = Kusto.Charting.DataChartsHelper.GetFirstStringAvailableColumnIndexOrNonStringIfAbsent(resolvedColumnTypes, null);
                                // Revert series index in case argument and series index are the same - and it is not allowed
                                if (System.Enum.hasFlag(argumentRestrictions, Bridge.box(Kusto.Charting.ArgumentRestrictions.NotIncludedInSeries, Kusto.Charting.ArgumentRestrictions, System.Enum.toStringFn(Kusto.Charting.ArgumentRestrictions))) && seriesDefaultIndex === metaData.v.ArgumentDataColumnIndex) {
                                    seriesDefaultIndex = -1;
                                }
                            }
                        }

                        if (seriesDefaultIndex >= 0 && !metaData.v.DataIndexesList.contains(seriesDefaultIndex)) {
                            metaData.v.SeriesIndexesList.add(seriesDefaultIndex);
                        }
                    }

                    return true;
                },
                GoBackwardsAndFindColumnNotInList: function (startIndex, seriesIndices, yIndexes) {
                    for (var i = (startIndex - 1) | 0; i >= 0; i = (i - 1) | 0) {
                        var isNotInSeries = (seriesIndices == null) ? true : !System.Linq.Enumerable.from(seriesIndices, System.Int32).contains(i);
                        var isNotInYs = (yIndexes == null) ? true : !System.Linq.Enumerable.from(yIndexes, System.Int32).contains(i);
                        if (isNotInSeries && isNotInYs) {
                            return i;
                        }
                    }
                    return -1;
                },
                GetFirstStringColumnIndex: function (columns) {
                    for (var i = 0; i < System.Linq.Enumerable.from(columns, Kusto.Charting.ArgumentColumnType).count(); i = (i + 1) | 0) {
                        if (System.Linq.Enumerable.from(columns, Kusto.Charting.ArgumentColumnType).elementAt(i) === Kusto.Charting.ArgumentColumnType.String) {
                            return i;
                        }
                    }
                    return -1;
                },
                GetFirstStringAvailableColumnIndexOrNonStringIfAbsent: function (columns, indexesToExclude) {
                    var $t, $t1;
                    var nonStringIndex = -1;

                    for (var i = 0; i < System.Linq.Enumerable.from(columns, Kusto.Charting.ArgumentColumnType).count(); i = (i + 1) | 0) {
                        if (columns[System.Array.index(i, columns)] === Kusto.Charting.ArgumentColumnType.String && (($t = System.Nullable.not((indexesToExclude != null ? System.Linq.Enumerable.from(indexesToExclude, System.Int32).contains(i) : null)), $t != null ? $t : true))) {
                            return i;
                        } else if (nonStringIndex === -1 && (($t1 = System.Nullable.not((indexesToExclude != null ? System.Linq.Enumerable.from(indexesToExclude, System.Int32).contains(i) : null)), $t1 != null ? $t1 : true))) {
                            nonStringIndex = i;
                        }
                    }

                    return nonStringIndex;
                },
                TimeSpanToDateTime: function (ts) {
                    if (ts.getTicks().lte(System.Int64(0))) {
                        return System.DateTime.getMinValue();
                    }
                    return System.DateTime.create$2(ts.getTicks(), 1);
                },
                TimeSpanToTotalSeconds: function (ts) {
                    return ts.getTotalSeconds();
                },
                DateTimeToTotalSeconds: function (dt) {
                    return System.TimeSpan.fromTicks(System.DateTime.getTicks(dt)).getTotalSeconds();
                },
                ParseJsonArrayAsDateTime: function (value, columnType) {
                    if (System.String.isNullOrWhiteSpace(value) || !System.String.startsWith(value.trim(), "[", 4)) {
                        return null;
                    }

                    try {
                        if (columnType === Kusto.Charting.ArgumentColumnType.DateTime) {
                            return System.Linq.Enumerable.from(Newtonsoft.Json.JsonConvert.DeserializeObject(value, System.Array.type(System.DateTime)), System.DateTime).select($asm.$.Kusto.Charting.DataChartsHelper.f15).ToArray(System.DateTime);
                        } else if (columnType === Kusto.Charting.ArgumentColumnType.TimeSpan) {
                            var resultAsTimeSpan = Newtonsoft.Json.JsonConvert.DeserializeObject(value, System.Array.type(System.TimeSpan));
                            return System.Linq.Enumerable.from(resultAsTimeSpan, System.TimeSpan).select($asm.$.Kusto.Charting.DataChartsHelper.f16).ToArray(System.DateTime);
                        }
                    } catch ($e1) {
                        $e1 = System.Exception.create($e1);

                    }

                    return null;
                },
                ParseJsonArrayAsDouble: function (value, considerDateTimeAndTimeSpanAsDouble) {
                    if (considerDateTimeAndTimeSpanAsDouble === void 0) { considerDateTimeAndTimeSpanAsDouble = false; }
                    if (System.String.isNullOrWhiteSpace(value) || !System.String.startsWith(value.trim(), "[", 4)) {
                        return null;
                    }

                    var result;
                    var valueAsArr;
                    try {
                        valueAsArr = Newtonsoft.Json.JsonConvert.DeserializeObject(value, System.Array.type(System.String));
                        result = System.Array.init(System.Linq.Enumerable.from(valueAsArr, System.String).count(), 0, System.Double);
                    } catch ($e1) {
                        $e1 = System.Exception.create($e1);
                        return null;
                    }

                    var resolvedValueType = Kusto.Charting.ArgumentColumnType.None;
                    var d = { };
                    for (var i = 0; i < valueAsArr.length; i = (i + 1) | 0) {
                        var val = valueAsArr[System.Array.index(i, valueAsArr)];
                        if (val == null) {
                            result[System.Array.index(i, result)] = Number.NaN;
                            continue;
                        } else if (System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.Numeric, Bridge.box(resolvedValueType, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) && System.Double.tryParse(val, null, d)) {
                            if (resolvedValueType === Kusto.Charting.ArgumentColumnType.None) {
                                resolvedValueType = Kusto.Charting.ArgumentColumnType.Numeric;
                            }

                            result[System.Array.index(i, result)] = d.v;
                            continue;
                        } else if (considerDateTimeAndTimeSpanAsDouble) {
                            var ts = { };
                            if (System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.TimeSpan, Bridge.box(resolvedValueType, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) && System.TimeSpan.tryParse(val, null, ts)) {
                                if (resolvedValueType === Kusto.Charting.ArgumentColumnType.None) {
                                    resolvedValueType = Kusto.Charting.ArgumentColumnType.TimeSpan;
                                }

                                result[System.Array.index(i, result)] = ts.v.getTotalSeconds();
                                continue;
                            }

                            var dt = { };
                            if (System.Enum.hasFlag(Kusto.Charting.ArgumentColumnType.DateTime, Bridge.box(resolvedValueType, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) && System.DateTime.tryParse(val, null, dt)) {
                                if (resolvedValueType === Kusto.Charting.ArgumentColumnType.None) {
                                    resolvedValueType = Kusto.Charting.ArgumentColumnType.DateTime;
                                }

                                result[System.Array.index(i, result)] = System.TimeSpan.fromTicks(System.DateTime.getTicks(dt.v)).getTotalSeconds();
                                continue;
                            }
                        }

                        return null;
                    }

                    if (resolvedValueType !== Kusto.Charting.ArgumentColumnType.None) {
                        return result;
                    }

                    return null;
                },
                ParseJsonArrayAsString: function (value) {
                    if (System.String.isNullOrWhiteSpace(value) || !System.String.startsWith(value.trim(), "[", 4)) {
                        return null;
                    }

                    try {
                        return Newtonsoft.Json.JsonConvert.DeserializeObject(value, System.Array.type(System.String));
                    } catch ($e1) {
                        $e1 = System.Exception.create($e1);

                    }

                    return null;
                },
                ArrayIsTimespan: function (arr) {
                    var $t;
                    var hasTimeSpanValues = false;
                    var ts = { };
                    $t = Bridge.getEnumerator(arr);
                    try {
                        while ($t.moveNext()) {
                            var val = $t.Current;
                            if (val == null) {
                                continue;
                            } else if (System.TimeSpan.tryParse(val, null, ts)) {
                                hasTimeSpanValues = true;
                                continue;
                            }

                            return false;
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return hasTimeSpanValues;
                },
                ArrayIsDatetime: function (arr) {
                    var $t;
                    var hasDateTimeValues = false;
                    var d = { };
                    $t = Bridge.getEnumerator(arr);
                    try {
                        while ($t.moveNext()) {
                            var val = $t.Current;
                            if (val == null) {
                                continue;
                            } else if (System.DateTime.tryParse(val, null, d)) {
                                hasDateTimeValues = true;
                                continue;
                            }

                            return false;
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return hasDateTimeValues;
                },
                ArrayIsDouble: function (arr) {
                    var $t;
                    var hasNumericValues = false;
                    var d = { };
                    $t = Bridge.getEnumerator(arr);
                    try {
                        while ($t.moveNext()) {
                            var val = $t.Current;
                            if (val == null) {
                                continue;
                            } else if (System.Double.tryParse(val, null, d)) {
                                hasNumericValues = true;
                                continue;
                            }

                            return false;
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return hasNumericValues;
                },
                IsEmptyValue: function (value) {
                    if (value == null) {
                        return true;
                    }

                    if (Bridge.referenceEquals(value, System.DBNull.Value)) {
                        return true;
                    }

                    if (Bridge.is(value, System.String) && System.String.isNullOrEmpty(Bridge.cast(value, System.String))) {
                        return true;
                    }

                    return false;
                }
            }
        }
    });

    Bridge.ns("Kusto.Charting.DataChartsHelper", $asm.$);

    Bridge.apply($asm.$.Kusto.Charting.DataChartsHelper, {
        f1: function (col, index) {
            return new Kusto.Charting.DataChartsHelper.ColumnDesc(col.Item1, col.Item2, index);
        },
        f2: function (colIndex) {
            return colIndex >= 0;
        },
        f3: function (d) {
            return d.SeriesName;
        },
        f4: function (d) {
            return d;
        },
        f5: function (d) {
            return System.DateTime.getTicks(d.ArgumentDateTime);
        },
        f6: function (d) {
            return System.Convert.toDouble(System.DateTime.getTicks(d.ArgumentDateTime));
        },
        f7: function (d) {
            return d.ArgumentNumeric;
        },
        f8: function (d) {
            return !Bridge.equals(d.ArgumentDateTime, Kusto.Charting.DataChartsHelper.c_defaultArgumentDatetime);
        },
        f9: function (d) {
            return !isNaN(d.ArgumentNumeric);
        },
        f10: function (d) {
            return (Math.abs(d.ValueData) === Number.POSITIVE_INFINITY);
        },
        f11: function (d) {
            return d.GeoCoordinates != null && (d.GeoCoordinates.Longitude > 180 || d.GeoCoordinates.Longitude < -180 || d.GeoCoordinates.Latitude > 90 || d.GeoCoordinates.Latitude < -90);
        },
        f12: function (o) {
            return Kusto.Charting.DataChartsHelper.ParseJsonArrayAsString(Bridge.as(o, System.String)) || o;
        },
        f13: function (a, b) {
            return { Item1: a, Item2: b };
        },
        f14: function (c) {
            return c.Item2;
        },
        f15: function (dt) {
            return Kusto.Cloud.Platform.Utils.ExtendedDateTime.ToUtc(dt);
        },
        f16: function (ts) {
            return Kusto.Charting.DataChartsHelper.TimeSpanToDateTime(ts);
        }
    });

    /** @namespace Kusto.Charting */

    /**
     * The class will pre-calculate the argument value and convert it to different types.
     *
     * @private
     * @class Kusto.Charting.DataChartsHelper.ArgumentData
     */
    Bridge.define("Kusto.Charting.DataChartsHelper.ArgumentData", {
        $kind: "nested class",
        fields: {
            m_requestedType: 0,
            m_colIndex: 0,
            m_colName: null,
            m_colType: 0,
            m_geospatialIndexes: null
        },
        props: {
            GeospatialArgumentDataType: 0,
            Value: null,
            DateTime: null,
            NumericValue: 0,
            GeoCoordinates: null
        },
        ctors: {
            init: function () {
                this.DateTime = System.DateTime.getDefaultValue();
            },
            ctor: function (colIndex, colName, colType, requestedType, geospatialColumnIndexes) {
                if (geospatialColumnIndexes === void 0) { geospatialColumnIndexes = null; }
                var $t;

                this.$initialize();
                this.m_colIndex = colIndex;
                this.m_colName = colName;
                this.m_colType = colType;
                this.m_requestedType = requestedType;

                if (geospatialColumnIndexes != null && System.Linq.Enumerable.from(geospatialColumnIndexes, System.Int32).any()) {
                    this.m_geospatialIndexes = ($t = System.Int32, System.Linq.Enumerable.from(geospatialColumnIndexes, $t).toList($t));
                    this.GeospatialArgumentDataType = Kusto.Charting.ArgumentColumnType.Geospatial;
                } else {
                    this.GeospatialArgumentDataType = Kusto.Charting.ArgumentColumnType.None;
                }
            }
        },
        methods: {
            ResolveArgumentFromRow: function (table, rowIndex) {
                var $t, $t1, $t2;
                if (this.m_requestedType !== Kusto.Charting.ArgumentColumnType.Geospatial) {
                    this.ResolveArgument(table, rowIndex);
                    return;
                }

                // resolving case: m_type == ArgumentColumnType.Geospatial
                if (this.m_geospatialIndexes.Count === 2) {
                    // argument data represents lng,lat geo coordinates
                    var longitudeValue = table.Kusto$Charting$IChartingDataSource$GetValue(rowIndex, this.m_geospatialIndexes.getItem(0));
                    var latitudeValue = table.Kusto$Charting$IChartingDataSource$GetValue(rowIndex, this.m_geospatialIndexes.getItem(1));
                    this.GeoCoordinates = ($t = new Kusto.Charting.GeospatialCoordinates(), $t.Longitude = Kusto.Charting.DataChartsHelper.TryConvertToDouble(longitudeValue, Kusto.Charting.ArgumentColumnType.Numeric), $t.Latitude = Kusto.Charting.DataChartsHelper.TryConvertToDouble(latitudeValue, Kusto.Charting.ArgumentColumnType.Numeric), $t);
                } else {
                    var geojsonPoint = { };
                    if (this.m_geospatialIndexes.Count === 1 && Kusto.Charting.DataChartsHelper.TrySetGeoJSONPoint(Bridge.toString(table.Kusto$Charting$IChartingDataSource$GetValue(rowIndex, this.m_geospatialIndexes.getItem(0))), geojsonPoint)) {
                        // argument data represents GeoJSON Point
                        this.GeoCoordinates = ($t = new Kusto.Charting.GeospatialCoordinates(), $t.Longitude = ($t1 = geojsonPoint.v.coordinates)[System.Array.index(0, $t1)], $t.Latitude = ($t2 = geojsonPoint.v.coordinates)[System.Array.index(1, $t2)], $t);
                    } else {
                        // failed to set Geo coordinates
                        this.GeoCoordinates = ($t = new Kusto.Charting.GeospatialCoordinates(), $t.Longitude = Kusto.Charting.DataChartsHelper.c_defaultArgumentNumeric, $t.Latitude = Kusto.Charting.DataChartsHelper.c_defaultArgumentNumeric, $t);
                    }
                }
                if (this.m_colIndex !== -1) {
                    // ArgumentData represents Geospatial coordinates and argumentData (in case of pie map, argumentData is a weight of segment\slice of pie)
                    this.ResolveArgument(table, rowIndex, true);
                }
            },
            ResolveArgument: function (table, rowIndex, includeColName) {
                if (includeColName === void 0) { includeColName = false; }
                var argumentValue = table.Kusto$Charting$IChartingDataSource$GetValue(rowIndex, this.m_colIndex);
                var argumentActualType = (argumentValue == null) ? this.m_requestedType : this.m_colType;

                this.DateTime = Kusto.Charting.DataChartsHelper.GetArgumentDateTime(argumentValue, argumentActualType);
                var value = (this.m_colIndex >= 0) ? Bridge.toString(argumentValue) : "";
                if (System.String.isNullOrWhiteSpace(value)) {
                    value = Kusto.Charting.DataChartsHelper.c_defaultArgumentString;
                }

                this.Value = !includeColName ? value : System.String.format("{0}:{1}", this.m_colName, value);

                if (!System.Enum.hasFlag(argumentActualType, Bridge.box(Kusto.Charting.ArgumentColumnType.Numeric, Kusto.Charting.ArgumentColumnType, System.Enum.toStringFn(Kusto.Charting.ArgumentColumnType))) || Kusto.Charting.DataChartsHelper.IsEmptyValue(argumentValue)) {
                    this.NumericValue = Kusto.Charting.DataChartsHelper.c_defaultArgumentNumeric;
                } else {
                    this.NumericValue = Kusto.Charting.DataChartsHelper.TryConvertToDouble(argumentValue, argumentActualType);
                }
            }
        }
    });

    /**
     * Chart meta data
     Describe the columns that will be used for the chart
     *
     * @abstract
     * @public
     * @class Kusto.Charting.IChartMetaData
     */
    Bridge.define("Kusto.Charting.IChartMetaData", {
        $kind: "interface"
    });

    Bridge.define("Kusto.Charting.DataChartsHelper.ColumnDesc", {
        $kind: "nested class",
        props: {
            Name: null,
            Type: 0,
            Index: 0
        },
        ctors: {
            ctor: function (name, type, index) {
                this.$initialize();
                this.Name = name;
                this.Type = type;
                this.Index = index;
            }
        }
    });

    Bridge.define("Kusto.Charting.DataChartsHelper.GeoJSON", {
        $kind: "nested enum",
        statics: {
            fields: {
                Point: 0
            }
        }
    });

    Bridge.define("Kusto.Charting.DataChartsHelper.GeoJSONPoint", {
        $kind: "nested class",
        props: {
            type: 0,
            coordinates: null
        }
    });

    Bridge.define("Kusto.Charting.DataItem", {
        props: {
            SeriesName: null,
            ArgumentData: null,
            ValueData: 0,
            ValueName: null,
            ArgumentDateTime: null,
            ArgumentNumeric: 0,
            GeoCoordinates: null,
            SecondaryAxisYName: null,
            PrefferredChartKind: 0,
            Properties: null,
            /**
             * Truncated value of the property - used for tooltip presentation.
             *
             * @instance
             * @public
             * @readonly
             * @memberof Kusto.Charting.DataItem
             * @function PropertiesTruncated
             * @type string
             */
            PropertiesTruncated: {
                get: function () {
                    if (System.String.isNullOrEmpty(this.Properties)) {
                        return "";
                    }

                    var maxLen = 150;

                    if (this.Properties.length < maxLen) {
                        return this.Properties;
                    }
                    return (this.Properties.substr(0, maxLen) || "") + "...";
                }
            }
        },
        ctors: {
            init: function () {
                this.ArgumentDateTime = System.DateTime.getDefaultValue();
            },
            ctor: function () {
                this.$initialize();
            },
            $ctor1: function (other) {
                var $t;
                this.$initialize();
                this.SeriesName = other.SeriesName;
                this.ArgumentData = other.ArgumentData;
                this.ValueData = other.ValueData;
                this.ValueName = other.ValueName;
                this.ArgumentDateTime = other.ArgumentDateTime;
                this.ArgumentNumeric = other.ArgumentNumeric;
                this.SecondaryAxisYName = other.SecondaryAxisYName;
                this.PrefferredChartKind = other.PrefferredChartKind;
                this.Properties = other.Properties;
                this.GeoCoordinates = other.GeoCoordinates == null ? null : ($t = new Kusto.Charting.GeospatialCoordinates(), $t.Longitude = other.GeoCoordinates.Longitude, $t.Latitude = other.GeoCoordinates.Latitude, $t);
            }
        },
        methods: {
            Clone: function () {
                return new Kusto.Charting.DataItem.$ctor1(this);
            }
        }
    });

    Bridge.define("Kusto.Charting.ExtendedDouble", {
        statics: {
            methods: {
                IsFinite: function (d) {
                    return !(Math.abs(d) === Number.POSITIVE_INFINITY) && !isNaN(d);
                }
            }
        }
    });

    Bridge.define("Kusto.Charting.GeospatialCoordinates", {
        props: {
            Longitude: 0,
            Latitude: 0
        }
    });

    Bridge.define("Kusto.Charting.IChartingDataSource", {
        $kind: "interface"
    });

    Bridge.define("Kusto.Charting.IKustoClientContext", {
        $kind: "interface"
    });

    Bridge.define("Kusto.Charting.SeriesCreationException", {
        inherits: [System.Exception],
        ctors: {
            ctor: function (error) {
                this.$initialize();
                System.Exception.ctor.call(this, error);
            }
        }
    });

    Bridge.define("Kusto.Charting.ValidationStatus", {
        $kind: "enum",
        statics: {
            fields: {
                Valid: 0,
                PolicyViolationError: 1,
                PolicyViolationWarning: 2
            }
        }
    });

    Bridge.define("Kusto.Cloud.Platform.Text.StringLiteral", {
        statics: {
            fields: {
                MultiLineStringQuote: null,
                AlternateMultiLineStringQuote: null,
                categoryForLatin1: null
            },
            ctors: {
                init: function () {
                    this.MultiLineStringQuote = "```";
                    this.AlternateMultiLineStringQuote = "~~~";
                    this.categoryForLatin1 = System.Array.init([
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        11, 
                        24, 
                        24, 
                        24, 
                        26, 
                        24, 
                        24, 
                        24, 
                        20, 
                        21, 
                        24, 
                        25, 
                        24, 
                        19, 
                        24, 
                        24, 
                        8, 
                        8, 
                        8, 
                        8, 
                        8, 
                        8, 
                        8, 
                        8, 
                        8, 
                        8, 
                        24, 
                        24, 
                        25, 
                        25, 
                        25, 
                        24, 
                        24, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        20, 
                        24, 
                        21, 
                        27, 
                        18, 
                        27, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        20, 
                        25, 
                        21, 
                        25, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        14, 
                        11, 
                        24, 
                        26, 
                        26, 
                        26, 
                        26, 
                        28, 
                        28, 
                        27, 
                        28, 
                        1, 
                        22, 
                        25, 
                        19, 
                        28, 
                        27, 
                        28, 
                        25, 
                        10, 
                        10, 
                        27, 
                        1, 
                        28, 
                        24, 
                        27, 
                        10, 
                        1, 
                        23, 
                        10, 
                        10, 
                        10, 
                        24, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        25, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        0, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        25, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1, 
                        1
                    ], System.Byte);
                }
            },
            methods: {
                /**
                 * Parses a CSL string literal. There are five kinds of string literals:
 
                 Each kind of string literal may be preceeded by an 'h' to mark it as hidden - i.e. will <b>not</b> be saved in our traces.
                 For example: h"This is a hidden string"/
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Text.StringLiteral
                 * @memberof Kusto.Cloud.Platform.Text.StringLiteral
                 * @param   {string}    literal
                 * @return  {string}
                 */
                ParseStringLiteral: function (literal) {
                    var _discard1 = { };
                    return Kusto.Cloud.Platform.Text.StringLiteral.ParseStringLiteral$1(literal, _discard1);
                },
                /**
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Text.StringLiteral
                 * @memberof Kusto.Cloud.Platform.Text.StringLiteral
                 * @param   {string}            literal     
                 * @param   {System.Boolean}    isHidden
                 * @return  {string}
                 */
                ParseStringLiteral$1: function (literal, isHidden) {
                    var result = { };
                    if (Kusto.Cloud.Platform.Text.StringLiteral.TryParseStringLiteral$1(literal, result, isHidden)) {
                        return result.v;
                    }

                    // TODO: Replace with an exception that traces
                    throw new System.ArgumentException.$ctor1(System.String.format("Malformed string literal: '{0}'", [literal]));
                },
                TryParseStringLiteral: function (str, literal) {
                    var _discard2 = { };
                    return Kusto.Cloud.Platform.Text.StringLiteral.TryParseStringLiteral$1(str, literal, _discard2);
                },
                TryParseStringLiteral$1: function (str, literal, isHidden) {
                    literal.v = str;
                    isHidden.v = false;

                    // We don't generally expect this to be true:
                    if (System.String.isNullOrWhiteSpace(literal.v)) {
                        return false;
                    }

                    // String literals starting with an 'h' or 'H' denote literals
                    // that should be hidden, but here we don't care.
                    if (literal.v.charCodeAt(0) === 104 || literal.v.charCodeAt(0) === 72) {
                        isHidden.v = true;
                        literal.v = literal.v.substr(1);
                    }
                    var multiLineLiteral = { };

                    if (Kusto.Cloud.Platform.Text.StringLiteral.TryParseMultiLineStringLiteral(literal.v, multiLineLiteral)) {
                        literal.v = multiLineLiteral.v;
                        return true;
                    }

                    if (System.String.startsWith(literal.v, "\"", 4)) {
                        if (literal.v.length >= 2 && System.String.endsWith(literal.v, "\"")) {
                            literal.v = literal.v.substr(1, ((literal.v.length - 2) | 0));
                            if (!Kusto.Cloud.Platform.Utils.ExtendedRegex.TryUnescape(literal.v, literal)) {
                                return false;
                            }
                            return true;
                        }
                    } else if (System.String.startsWith(literal.v, "'", 4)) {
                        if (literal.v.length >= 2 && System.String.endsWith(literal.v, "'", 4)) {
                            literal.v = literal.v.substr(1, ((literal.v.length - 2) | 0));
                            if (!Kusto.Cloud.Platform.Utils.ExtendedRegex.TryUnescape(literal.v, literal)) {
                                return false;
                            }
                            return true;
                        }
                    } else if (System.String.startsWith(literal.v, "@\"", 4)) {
                        if (literal.v.length >= 3 && System.String.endsWith(literal.v, "\"", 4)) {
                            var ret = literal.v.substr(2, ((literal.v.length - 3) | 0));
                            literal.v = System.String.replaceAll(ret, "\"\"", "\""); // "" -> "
                            return true;
                        }
                    } else if (System.String.startsWith(literal.v, "@'", 4)) {
                        if (literal.v.length >= 3 && System.String.endsWith(literal.v, "'", 4)) {
                            var ret1 = literal.v.substr(2, ((literal.v.length - 3) | 0));
                            literal.v = System.String.replaceAll(ret1, "''", "'"); // '' -> '
                            return true;
                        }
                    }

                    return false;
                },
                TryParseMultiLineStringLiteral: function (str, literal) {
                    return Kusto.Cloud.Platform.Text.StringLiteral.TryParseSimpleStringLiteral(str, Kusto.Cloud.Platform.Text.StringLiteral.MultiLineStringQuote, literal) || Kusto.Cloud.Platform.Text.StringLiteral.TryParseSimpleStringLiteral(str, Kusto.Cloud.Platform.Text.StringLiteral.AlternateMultiLineStringQuote, literal);
                },
                TryParseSimpleStringLiteral: function (str, quote, literal) {
                    // check for multi-line string
                    if (System.String.startsWith(str, quote, 4)) {
                        var twiceQuoteLen = quote.length << 1;
                        if (str.length >= twiceQuoteLen && System.String.endsWith(str, quote, 4)) {
                            literal.v = str.substr(quote.length, ((str.length - twiceQuoteLen) | 0));
                            return true;
                        } else {
                            literal.v = str.substr(quote.length);
                            return true;
                        }
                    }

                    literal.v = null;
                    return false;
                },
                Equals: function (str1, str2) {
                    if (str1 == null && str2 == null) {
                        return true;
                    }

                    if (str1 == null || str2 == null) {
                        return false;
                    }

                    return System.String.equals(str1, str2, 4);
                },
                TrimSingleQuotes: function (name) {
                    if (System.String.isNullOrWhiteSpace(name)) {
                        return name;
                    }

                    if (System.String.startsWith(name, "'", 4) && System.String.endsWith(name, "'", 4) && name.length >= 2) {
                        name = name.substr(1, ((name.length - 2) | 0));
                    }

                    return name;
                },
                TrimBrackets: function (input) {
                    if (System.String.startsWith(input, "[", 4) && System.String.endsWith(input, "]", 4) && input.length >= 2) {
                        input = input.substr(1, ((input.length - 2) | 0));
                    }

                    return input;
                },
                InitArray: function (T, arr, value) {
                    if (arr != null) {
                        for (var i = 0; i < arr.length; i = (i + 1) | 0) {
                            arr[System.Array.index(i, arr)] = value;
                        }
                    }
                },
                /**
                 * Generate string that will be parsed as a string literal.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Text.StringLiteral
                 * @memberof Kusto.Cloud.Platform.Text.StringLiteral
                 * @param   {string}    value
                 * @return  {string}
                 */
                GetLiteral: function (value) {
                    return Kusto.Cloud.Platform.Text.StringLiteral.GetLiteral$1(value, false);
                },
                /**
                 * Generate string that will be parsed as a string literal.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Text.StringLiteral
                 * @memberof Kusto.Cloud.Platform.Text.StringLiteral
                 * @param   {string}     value     
                 * @param   {boolean}    hidden
                 * @return  {string}
                 */
                GetLiteral$1: function (value, hidden) {
                    var $t;
                    if (value == null) {
                        return null;
                    }

                    var literal = new System.Text.StringBuilder();
                    if (hidden) {
                        literal.append("h");
                    }
                    literal.append("\"");
                    $t = Bridge.getEnumerator(value);
                    try {
                        while ($t.moveNext()) {
                            var c = $t.Current;
                            switch (c) {
                                case 39: 
                                    literal.append("\\'");
                                    break;
                                case 34: 
                                    literal.append("\\\"");
                                    break;
                                case 92: 
                                    literal.append("\\\\");
                                    break;
                                case 0: 
                                    literal.append("\\0");
                                    break;
                                case 7: 
                                    literal.append("\\a");
                                    break;
                                case 8: 
                                    literal.append("\\b");
                                    break;
                                case 12: 
                                    literal.append("\\f");
                                    break;
                                case 10: 
                                    literal.append("\\n");
                                    break;
                                case 13: 
                                    literal.append("\\r");
                                    break;
                                case 9: 
                                    literal.append("\\t");
                                    break;
                                case 11: 
                                    literal.append("\\v");
                                    break;
                                default: 
                                    if (!Kusto.Cloud.Platform.Text.StringLiteral.ShouldBeEscaped(c)) {
                                        literal.append(String.fromCharCode(c));
                                    } else {
                                        literal.append("\\u");
                                        literal.append(System.UInt16.format(c, "x4"));
                                    }
                                    break;
                            }
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }
                    literal.append("\"");

                    var result = literal.toString();
                    return result;
                },
                /**
                 * Generate string that will be parsed as a string literal.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Text.StringLiteral
                 * @memberof Kusto.Cloud.Platform.Text.StringLiteral
                 * @param   {string}    value
                 * @return  {string}
                 */
                GetLiteralAsHiddenString: function (value) {
                    return Kusto.Cloud.Platform.Text.StringLiteral.GetLiteral$1(value, true);
                },
                ShouldBeEscaped: function (c) {
                    if (Kusto.Cloud.Platform.Text.StringLiteral.IsLatin1(c)) {
                        return (Kusto.Cloud.Platform.Text.StringLiteral.GetLatin1UnicodeCategory(c)) === 14;
                    }

                    // Any non-latin character will be escaped (for simplicity)
                    return true;
                },
                GetLatin1UnicodeCategory: function (ch) {
                    return Kusto.Cloud.Platform.Text.StringLiteral.categoryForLatin1[System.Array.index(ch, Kusto.Cloud.Platform.Text.StringLiteral.categoryForLatin1)];
                },
                IsLatin1: function (ch) {
                    return (ch <= 255);
                },
                IsAscii: function (ch) {
                    return (ch <= 127);
                }
            }
        }
    });

    /** @namespace Kusto.Cloud.Platform.Utils */

    /**
     * CS freshman.
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Cloud.Platform.Utils.BitTwiddling
     */
    Bridge.define("Kusto.Cloud.Platform.Utils.BitTwiddling", {
        statics: {
            methods: {
                /**
                 * Count the number of bits set in the input (int).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.BitTwiddling
                 * @memberof Kusto.Cloud.Platform.Utils.BitTwiddling
                 * @param   {number}    n
                 * @return  {number}
                 */
                CountBits: function (n) {
                    var ret = 0;

                    while (n !== 0) {
                        n = n & (((n - 1) | 0));
                        ret = (ret + 1) | 0;
                    }
                    return ret;
                },
                /**
                 * Count the number of bits set in the input (long).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.BitTwiddling
                 * @memberof Kusto.Cloud.Platform.Utils.BitTwiddling
                 * @param   {System.Int64}    n
                 * @return  {number}
                 */
                CountBits$1: function (n) {
                    var ret = 0;

                    while (n.ne(System.Int64(0))) {
                        n = n.and((n.sub(System.Int64(1))));
                        ret = (ret + 1) | 0;
                    }
                    return ret;
                }
            }
        }
    });

    /**
     * Options to apply when chopping strings.
     *
     * @public
     * @class Kusto.Cloud.Platform.Utils.ChopOptions
     */
    Bridge.define("Kusto.Cloud.Platform.Utils.ChopOptions", {
        $kind: "enum",
        statics: {
            fields: {
                ChopFromBeginning: 1,
                ChopFromEnd: 2,
                ChopFromMiddle: 4,
                ThreeDotsCharacterNotSupported: 8
            }
        }
    });

    Bridge.define("Kusto.Cloud.Platform.Utils.CountConstants", {
        statics: {
            fields: {
                _1K: 0,
                _64K: 0,
                _1M: 0
            },
            ctors: {
                init: function () {
                    this._1K = 1000;
                    this._64K = 64000;
                    this._1M = 1000000;
                }
            }
        }
    });

    /**
     * Represents a range between 2 dates
     *
     * @public
     * @class Kusto.Cloud.Platform.Utils.DateTimeRange
     * @implements  System.IEquatable$1
     */
    Bridge.define("Kusto.Cloud.Platform.Utils.DateTimeRange", {
        inherits: function () { return [System.IEquatable$1(Kusto.Cloud.Platform.Utils.DateTimeRange)]; },
        $kind: "struct",
        statics: {
            fields: {
                /**
                 * Get a range representing all of eternity
                 *
                 * @static
                 * @public
                 * @readonly
                 * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
                 * @type Kusto.Cloud.Platform.Utils.DateTimeRange
                 */
                All: null
            },
            ctors: {
                init: function () {
                    this.All = new Kusto.Cloud.Platform.Utils.DateTimeRange();
                    this.All = new Kusto.Cloud.Platform.Utils.DateTimeRange.$ctor1(Kusto.Cloud.Platform.Utils.ExtendedDateTime.MinValueUtc, Kusto.Cloud.Platform.Utils.ExtendedDateTime.MaxValueUtc);
                }
            },
            methods: {
                /**
                 * Return a range that "covers" two ranges, and potentially
                 values that are not in any of the ranges (if the inputs
                 do not overlap).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.DateTimeRange
                 * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
                 * @param   {Kusto.Cloud.Platform.Utils.DateTimeRange}    a    
                 * @param   {Kusto.Cloud.Platform.Utils.DateTimeRange}    b
                 * @return  {Kusto.Cloud.Platform.Utils.DateTimeRange}
                 */
                Aggregate: function (a, b) {
                    return new Kusto.Cloud.Platform.Utils.DateTimeRange.$ctor1((System.DateTime.gt(a.Begin, b.Begin) ? b.Begin : a.Begin), (System.DateTime.gt(a.End, b.End) ? a.End : b.End));
                },
                /**
                 * Merge any number of ranges and return the minimal set of
                 ranges that covers them all precisely.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.DateTimeRange
                 * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
                 * @param   {System.Collections.Generic.IEnumerable$1}    ranges
                 * @return  {System.Collections.Generic.HashSet$1}
                 */
                Merge: function (ranges) {
                    var $t;
                    var count = Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastCountUpTo(Kusto.Cloud.Platform.Utils.DateTimeRange, ranges, System.Int64(2));
                    if (count.equals(System.Int64(0))) {
                        return new (System.Collections.Generic.HashSet$1(Kusto.Cloud.Platform.Utils.DateTimeRange)).ctor();
                    } else if (count.equals(System.Int64(1))) {
                        return new (System.Collections.Generic.HashSet$1(Kusto.Cloud.Platform.Utils.DateTimeRange)).$ctor1(ranges);
                    }

                    // We have more than 2 items, so we need to merge them
                    var result = new (System.Collections.Generic.HashSet$1(Kusto.Cloud.Platform.Utils.DateTimeRange)).ctor();
                    var orderedRanges = System.Linq.Enumerable.from(ranges, Kusto.Cloud.Platform.Utils.DateTimeRange).orderBy($asm.$.Kusto.Cloud.Platform.Utils.DateTimeRange.f1);
                    var firstItem = true;
                    var begin = null;
                    var end = null;

                    $t = Bridge.getEnumerator(orderedRanges);
                    try {
                        while ($t.moveNext()) {
                            var range = $t.Current;
                            if (firstItem) {
                                begin = range.Begin;
                                end = range.End;
                                firstItem = false;
                            }

                            // Range can be merged if:
                            //   [----------]      : (begin,end)
                            //          [-------]  : range
                            if (System.DateTime.lte(range.Begin, end)) {
                                if (System.DateTime.lt(end, range.End)) {
                                    end = range.End;
                                }
                            } else {
                                result.add(new Kusto.Cloud.Platform.Utils.DateTimeRange.$ctor1(System.Nullable.getValue(begin), System.Nullable.getValue(end)));
                                begin = range.Begin;
                                end = range.End;
                            }
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    result.add(new Kusto.Cloud.Platform.Utils.DateTimeRange.$ctor1(System.Nullable.getValue(begin), System.Nullable.getValue(end)));
                    return result;
                },
                /**
                 * Returns every intersection of two ranges from each group.
                 For example ([0, 10], [20, max]) and ([8, 30]) will produce ([8, 10], [20 - 30])
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.DateTimeRange
                 * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
                 * @param   {System.Collections.Generic.IEnumerable$1}    rangeGroupA    
                 * @param   {System.Collections.Generic.IEnumerable$1}    rangeGroupB
                 * @return  {System.Collections.Generic.HashSet$1}
                 */
                Intersect: function (rangeGroupA, rangeGroupB) {
                    var $t, $t1;
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(Kusto.Cloud.Platform.Utils.DateTimeRange, rangeGroupA) || Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(Kusto.Cloud.Platform.Utils.DateTimeRange, rangeGroupB)) {
                        return new (System.Collections.Generic.HashSet$1(Kusto.Cloud.Platform.Utils.DateTimeRange)).ctor();
                    }

                    var intersectedTimeRanges = new (System.Collections.Generic.HashSet$1(Kusto.Cloud.Platform.Utils.DateTimeRange)).ctor();

                    $t = Bridge.getEnumerator(rangeGroupA, Kusto.Cloud.Platform.Utils.DateTimeRange);
                    try {
                        while ($t.moveNext()) {
                            var rangeFromA = $t.Current;
                            $t1 = Bridge.getEnumerator(rangeGroupB, Kusto.Cloud.Platform.Utils.DateTimeRange);
                            try {
                                while ($t1.moveNext()) {
                                    var rangeFromB = $t1.Current;
                                    // The start of the intersection is the later start
                                    var intersectedStart = System.DateTime.create$2(System.Int64.max(System.DateTime.getTicks(rangeFromA.Begin), System.DateTime.getTicks(rangeFromB.Begin)), 1);

                                    // The end of the intersection is the earlier end
                                    var intersectedEnd = System.DateTime.create$2(System.Int64.min(System.DateTime.getTicks(rangeFromA.End), System.DateTime.getTicks(rangeFromB.End)), 1);

                                    // If we got a valid time range then it is the intersection of the two ranges.
                                    if (System.DateTime.lte(intersectedStart, intersectedEnd)) {
                                        intersectedTimeRanges.add(new Kusto.Cloud.Platform.Utils.DateTimeRange.$ctor1(intersectedStart, intersectedEnd));
                                    }
                                }
                            } finally {
                                if (Bridge.is($t1, System.IDisposable)) {
                                    $t1.System$IDisposable$Dispose();
                                }
                            }
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return intersectedTimeRanges;
                },
                getDefaultValue: function () { return new Kusto.Cloud.Platform.Utils.DateTimeRange(); }
            }
        },
        fields: {
            m_begin: null,
            m_end: null
        },
        props: {
            /**
             * The beginning date
             *
             * @instance
             * @public
             * @readonly
             * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
             * @function Begin
             * @type System.DateTime
             */
            Begin: {
                get: function () {
                    return this.m_begin;
                }
            },
            /**
             * The end date
             *
             * @instance
             * @public
             * @readonly
             * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
             * @function End
             * @type System.DateTime
             */
            End: {
                get: function () {
                    return this.m_end;
                }
            },
            /**
             * The span between the dates
             *
             * @instance
             * @public
             * @readonly
             * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
             * @function Span
             * @type System.TimeSpan
             */
            Span: {
                get: function () {
                    return System.DateTime.subdd(this.End, this.Begin);
                }
            },
            /**
             * If the range is open.
             *
             * @instance
             * @public
             * @readonly
             * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
             * @function IsOpenRange
             * @type boolean
             */
            IsOpenRange: {
                get: function () {
                    return Bridge.equalsT(this.Begin, Kusto.Cloud.Platform.Utils.ExtendedDateTime.MinValueUtc) || Bridge.equalsT(this.End, Kusto.Cloud.Platform.Utils.ExtendedDateTime.MaxValueUtc);
                }
            }
        },
        alias: ["equalsT", "System$IEquatable$1$Kusto$Cloud$Platform$Utils$DateTimeRange$equalsT"],
        ctors: {
            init: function () {
                this.m_begin = System.DateTime.getDefaultValue();
                this.m_end = System.DateTime.getDefaultValue();
            },
            /**
             * Construction
             *
             * @instance
             * @public
             * @this Kusto.Cloud.Platform.Utils.DateTimeRange
             * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
             * @param   {System.DateTime}    begin    
             * @param   {System.DateTime}    end
             * @return  {void}
             */
            $ctor1: function (begin, end) {
                this.$initialize();

                this.m_begin = begin;
                this.m_end = end;
                throw new System.ArgumentException.$ctor1("DateTimeRange construction: End parameter should be later than Begin parameter");
            },
            ctor: function () {
                this.$initialize();
            }
        },
        methods: {
            IsInRangeInclusive: function (value) {
                return System.DateTime.lte(this.Begin, value) && System.DateTime.lte(value, this.End);
            },
            IsInRangeExclusive: function (value) {
                return System.DateTime.lt(this.Begin, value) && System.DateTime.lt(value, this.End);
            },
            /**
             * Implement IEquatable{ReportingHandsetStatRecord}
             *
             * @instance
             * @public
             * @this Kusto.Cloud.Platform.Utils.DateTimeRange
             * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
             * @param   {Kusto.Cloud.Platform.Utils.DateTimeRange}    other
             * @return  {boolean}
             */
            equalsT: function (other) {
                return Bridge.equalsT(this.Begin, other.Begin) && Bridge.equalsT(this.End, other.End);
            },
            /**
             * Override {@link }
             *
             * @instance
             * @public
             * @override
             * @this Kusto.Cloud.Platform.Utils.DateTimeRange
             * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
             * @param   {System.Object}    other
             * @return  {boolean}
             */
            equals: function (other) {
                if (other == null) {
                    return false;
                }
                if (Bridge.is(other, Kusto.Cloud.Platform.Utils.DateTimeRange)) {
                    return this.equalsT(System.Nullable.getValue(Bridge.cast(Bridge.unbox(other, Kusto.Cloud.Platform.Utils.DateTimeRange), Kusto.Cloud.Platform.Utils.DateTimeRange)));
                }
                return false;
            },
            /**
             * Override {@link }
             *
             * @instance
             * @public
             * @override
             * @this Kusto.Cloud.Platform.Utils.DateTimeRange
             * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
             * @return  {number}
             */
            getHashCode: function () {
                return (Bridge.getHashCode(this.Begin) ^ Bridge.getHashCode(this.End));
            },
            /**
             * Override {@link }
             *
             * @instance
             * @public
             * @override
             * @this Kusto.Cloud.Platform.Utils.DateTimeRange
             * @memberof Kusto.Cloud.Platform.Utils.DateTimeRange
             * @return  {string}
             */
            toString: function () {
                return System.String.formatProvider(System.Globalization.CultureInfo.invariantCulture, "[Begin={0}, End={1}]", System.DateTime.format(this.Begin, "R"), System.DateTime.format(this.End, "R"));
            },
            $clone: function (to) {
                var s = to || new Kusto.Cloud.Platform.Utils.DateTimeRange();
                s.m_begin = this.m_begin;
                s.m_end = this.m_end;
                return s;
            }
        }
    });

    Bridge.ns("Kusto.Cloud.Platform.Utils.DateTimeRange", $asm.$);

    Bridge.apply($asm.$.Kusto.Cloud.Platform.Utils.DateTimeRange, {
        f1: function (r) {
            return r.Begin;
        }
    });

    /** @namespace System */

    /**
     * @memberof System
     * @callback System.Action
     * @return  {void}
     */

    /**
     * A wrapper class for {@link }.
     Provides time simulation capabilities.
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Cloud.Platform.Utils.ExtendedDateTime
     */
    Bridge.define("Kusto.Cloud.Platform.Utils.ExtendedDateTime", {
        statics: {
            fields: {
                MaxUnixTimeMilliseconds: System.Int64(0),
                MinUnixTimeMilliseconds: System.Int64(0),
                MaxUnixTimeMicroseconds: System.Int64(0),
                MinUnixTimeMicroseconds: System.Int64(0),
                MicrosecondsPerMillisecond: System.Int64(0),
                TicksPerMillisecond: System.Int64(0),
                TicksPerSecond: System.Int64(0),
                TicksPerMinute: System.Int64(0),
                TicksPerHour: System.Int64(0),
                TicksPerDay: System.Int64(0),
                MillisPerSecond: 0,
                MillisPerMinute: 0,
                MillisPerHour: 0,
                MillisPerDay: 0,
                DaysPerYear: 0,
                DaysPer4Years: 0,
                DaysPer100Years: 0,
                DaysPer400Years: 0,
                DaysTo1601: 0,
                DaysTo1899: 0,
                DaysTo1970: 0,
                DaysTo10000: 0,
                MinTicks: System.Int64(0),
                MaxTicks: System.Int64(0),
                MaxMillis: System.Int64(0),
                FileTimeOffset: System.Int64(0),
                DoubleDateOffset: System.Int64(0),
                OADateMinAsTicks: System.Int64(0),
                OADateMinAsDouble: 0,
                OADateMaxAsDouble: 0,
                DatePartYear: 0,
                DatePartDayOfYear: 0,
                DatePartMonth: 0,
                DatePartDay: 0,
                /**
                 * The min value of a DateTime, in UTC.
                 *
                 * @static
                 * @public
                 * @readonly
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @type System.DateTime
                 */
                MinValueUtc: null,
                /**
                 * The max value of a DateTime, in UTC.
                 *
                 * @static
                 * @public
                 * @readonly
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @type System.DateTime
                 */
                MaxValueUtc: null,
                /**
                 * A list of datetime formats which we support but aren't supported by the default IFormatProvider
                 which we use for DateTime.Parse.
                 *
                 * @static
                 * @public
                 * @readonly
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @type System.Collections.Generic.Dictionary$2
                 */
                SupportedNonStandardFormats: null,
                /**
                 * Jan 1 1970 ("epoch")
                 *
                 * @static
                 * @public
                 * @readonly
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @type System.DateTime
                 */
                EpochStart: null,
                /**
                 * If set, time is being simulated.
                 *
                 * @static
                 * @private
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @type boolean
                 */
                s_simulating: false,
                s_numCharactersInIso8601: 0,
                /**
                 * The last value of UtcNow when UtcNowAsString.get was invoked.
                 *
                 * @static
                 * @private
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @type System.DateTime
                 */
                ts_lastUtcNow: null,
                /**
                 * The last value of UtcNow.ToString("O") when UtcNowAsString.get was invoked.
                 *
                 * @static
                 * @private
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @type string
                 */
                ts_lastUtcNowAsString: null,
                DaysToMonth365: null,
                DaysToMonth366: null
            },
            events: {
                /**
                 * An event invoked when {@link } is invoked
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @function addOnStopSimulatingTime
                 * @param   {System.Action}    value
                 * @return  {void}
                 */
                /**
                 * An event invoked when {@link } is invoked
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @function removeOnStopSimulatingTime
                 * @param   {System.Action}    value
                 * @return  {void}
                 */
                OnStopSimulatingTime: null
            },
            props: {
                /**
                 * Is time currently being simulated?
                 *
                 * @static
                 * @public
                 * @readonly
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @function IsSimulatingTime
                 * @type boolean
                 */
                IsSimulatingTime: {
                    get: function () {
                        return Kusto.Cloud.Platform.Utils.ExtendedDateTime.s_simulating;
                    }
                },
                /**
                 * Wraps {@link }
                 *
                 * @static
                 * @public
                 * @readonly
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @function UtcNow
                 * @type System.DateTime
                 */
                UtcNow: {
                    get: function () {
                        return System.DateTime.getUtcNow();
                    }
                }
            },
            ctors: {
                init: function () {
                    this.MinValueUtc = System.DateTime.getDefaultValue();
                    this.MaxValueUtc = System.DateTime.getDefaultValue();
                    this.EpochStart = System.DateTime.getDefaultValue();
                    this.ts_lastUtcNow = System.DateTime.getDefaultValue();
                    this.MaxUnixTimeMilliseconds = System.Int64([-769664001,58999]);
                    this.MinUnixTimeMilliseconds = System.Int64([-304928768,-14468]);
                    this.MaxUnixTimeMicroseconds = System.Int64([-864855016,58999820]);
                    this.MinUnixTimeMicroseconds = System.Int64([13910016,-14467071]);
                    this.MicrosecondsPerMillisecond = System.Int64(1000);
                    this.TicksPerMillisecond = System.Int64(10000);
                    this.TicksPerSecond = System.Int64(10000000);
                    this.TicksPerMinute = System.Int64(600000000);
                    this.TicksPerHour = System.Int64([1640261632,8]);
                    this.TicksPerDay = System.Int64([711573504,201]);
                    this.MillisPerSecond = 1000;
                    this.MillisPerMinute = 60000;
                    this.MillisPerHour = 3600000;
                    this.MillisPerDay = 86400000;
                    this.DaysPerYear = 365;
                    this.DaysPer4Years = 1461;
                    this.DaysPer100Years = 36524;
                    this.DaysPer400Years = 146097;
                    this.DaysTo1601 = 584388;
                    this.DaysTo1899 = 693593;
                    this.DaysTo1970 = 719162;
                    this.DaysTo10000 = 3652059;
                    this.MinTicks = System.Int64(0);
                    this.MaxTicks = System.Int64([-197705729,734668917]);
                    this.MaxMillis = System.Int64([-464735232,73466]);
                    this.FileTimeOffset = System.Int64([578224128,117558807]);
                    this.DoubleDateOffset = System.Int64([-880558080,139527104]);
                    this.OADateMinAsTicks = System.Int64([-1362739200,7273949]);
                    this.OADateMinAsDouble = -657435;
                    this.OADateMaxAsDouble = 2958466.0;
                    this.DatePartYear = 0;
                    this.DatePartDayOfYear = 1;
                    this.DatePartMonth = 2;
                    this.DatePartDay = 3;
                    this.MinValueUtc = System.DateTime.create$2(System.DateTime.getTicks(System.DateTime.getMinValue()), 1);
                    this.MaxValueUtc = System.DateTime.create$2(System.DateTime.getTicks(System.DateTime.getMaxValue()), 1);
                    this.SupportedNonStandardFormats = $asm.$.Kusto.Cloud.Platform.Utils.ExtendedDateTime.f1(new (System.Collections.Generic.Dictionary$2(System.Int32,System.Array.type(System.String))).ctor());
                    this.EpochStart = System.DateTime.create(1970, 1, 1, 0, 0, 0, 0, 1);
                    this.s_numCharactersInIso8601 = System.DateTime.format(Kusto.Cloud.Platform.Utils.ExtendedDateTime.MinValueUtc, "O").length;
                    this.ts_lastUtcNow = Kusto.Cloud.Platform.Utils.ExtendedDateTime.MinValueUtc;
                    this.ts_lastUtcNowAsString = System.DateTime.format(Kusto.Cloud.Platform.Utils.ExtendedDateTime.ts_lastUtcNow, "O");
                    this.DaysToMonth365 = System.Array.init([
                        0, 
                        31, 
                        59, 
                        90, 
                        120, 
                        151, 
                        181, 
                        212, 
                        243, 
                        273, 
                        304, 
                        334, 
                        365
                    ], System.Int32);
                    this.DaysToMonth366 = System.Array.init([
                        0, 
                        31, 
                        60, 
                        91, 
                        121, 
                        152, 
                        182, 
                        213, 
                        244, 
                        274, 
                        305, 
                        335, 
                        366
                    ], System.Int32);
                }
            },
            methods: {
                /**
                 * Stops time simulation.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @return  {void}
                 */
                StopSimulatingTime: function () {
                    Kusto.Cloud.Platform.Utils.ExtendedDateTime.s_simulating = false;
                    if (!Bridge.staticEquals(Kusto.Cloud.Platform.Utils.ExtendedDateTime.OnStopSimulatingTime, null)) {
                        Kusto.Cloud.Platform.Utils.ExtendedDateTime.OnStopSimulatingTime();
                    }
                },
                ToUtcString: function (value) {
                    return Kusto.Cloud.Platform.Utils.ExtendedDateTime.ToUtcString$1(value, System.Globalization.CultureInfo.invariantCulture);
                },
                ToUtcString$1: function (value, culture) {
                    var inputTime = value;
                    var utcTime = inputTime;
                    if (System.DateTime.getKind(inputTime) === 0) {
                        utcTime = System.DateTime.create$2(System.DateTime.getTicks(value), 1);
                    } else if (System.DateTime.getKind(inputTime) === 2) {
                        utcTime = System.DateTime.toUniversalTime(inputTime);
                    }
                    return System.String.formatProvider(culture, "{0:yyyy-MM-dd HH:mm:ss.fffffff}", [Bridge.box(utcTime, System.DateTime, System.DateTime.format)]);
                },
                /**
                 * Returns the greater of 2 dates
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @param   {System.DateTime}    t1    The t1.
                 * @param   {System.DateTime}    t2    The t2.
                 * @return  {System.DateTime}
                 */
                Max: function (t1, t2) {
                    return (System.DateTime.gt(t1, t2) ? t1 : t2);
                },
                /**
                 * Returns the greater of 2 dates
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @param   {?System.DateTime}    t1    The t1.
                 * @param   {?System.DateTime}    t2    The t2.
                 * @return  {?System.DateTime}
                 */
                Max$1: function (t1, t2) {
                    if (Bridge.equals(t2, null)) {
                        return t1;
                    }
                    if (Bridge.equals(t1, null)) {
                        return t2;
                    }

                    return Kusto.Cloud.Platform.Utils.ExtendedDateTime.Max(System.Nullable.getValue(t1), System.Nullable.getValue(t2));
                },
                /**
                 * Returns the lesser of 2 dates
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @param   {System.DateTime}    t1    The t1.
                 * @param   {System.DateTime}    t2    The t2.
                 * @return  {System.DateTime}
                 */
                Min: function (t1, t2) {
                    return (System.DateTime.lt(t1, t2) ? t1 : t2);
                },
                /**
                 * Returns the lesser of 2 dates
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @param   {?System.DateTime}    t1    The t1.
                 * @param   {?System.DateTime}    t2    The t2.
                 * @return  {?System.DateTime}
                 */
                Min$1: function (t1, t2) {
                    if (Bridge.equals(t2, null)) {
                        return t1;
                    }
                    if (Bridge.equals(t1, null)) {
                        return t2;
                    }

                    return Kusto.Cloud.Platform.Utils.ExtendedDateTime.Min(System.Nullable.getValue(t1), System.Nullable.getValue(t2));
                },
                /**
                 * Checks if there is an intersection between two date ranges.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @param   {System.DateTime}    firstStart     Start datetime of the first range
                 * @param   {System.DateTime}    firstEnd       End datetime of the first range
                 * @param   {System.DateTime}    secondStart    Start datetime of the second range
                 * @param   {System.DateTime}    secondEnd      End datetime of the second range
                 * @return  {boolean}
                 */
                RangesIntersect: function (firstStart, firstEnd, secondStart, secondEnd) {
                    // Check if any of the point in the first interval are included in
                    // the second interval, or any of the points of the second interval
                    // are included in the first interval.
                    return (System.DateTime.gte(firstStart, secondStart) && System.DateTime.lte(firstStart, secondEnd)) || (System.DateTime.gte(firstEnd, secondStart) && System.DateTime.lte(firstEnd, secondEnd)) || (System.DateTime.gte(secondStart, firstStart) && System.DateTime.lte(secondStart, firstEnd)) || (System.DateTime.gte(secondEnd, firstStart) && System.DateTime.lte(secondEnd, firstEnd));
                },
                /**
                 * Returns the time in UNIX format (the diff time between the date and Jan 1 1970).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @param   {System.DateTime}    date    The date.
                 * @return  {System.TimeSpan}            The diff time since Jan 1 1970
                 */
                ToUnixTimeSpan: function (date) {
                    return System.DateTime.subdd(System.DateTime.toUniversalTime(date), Kusto.Cloud.Platform.Utils.ExtendedDateTime.EpochStart);
                },
                /**
                 * Returns a {@link } that has the same numeric value in ticks
                 as the time span.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @param   {System.TimeSpan}    timeSpan
                 * @return  {System.DateTime}
                 */
                FromTimeSpan: function (timeSpan) {
                    return System.DateTime.create$2(timeSpan.getTicks());
                },
                /**
                 * Returns ISO 8601 Datetime string with timezone
                 Example: 2020-10-28 13:48:10 +03:00
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @param   {System.DateTime}    dateTime
                 * @return  {string}
                 */
                ToZonalTimeDisplayString: function (dateTime) {
                    return System.DateTime.format(dateTime, "yyyy-MM-dd HH:mm:ss zzz");
                },
                /**
                 * Returns a {@link } that is the value in UTC of the date/time.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedDateTime
                 * @param   {System.DateTime}    value
                 * @return  {System.DateTime}
                 */
                ToUtc: function (value) {
                    switch (System.DateTime.getKind(value)) {
                        case 1: 
                            return value;
                        case 0: 
                            // There's really no way of knowing, so we assume UTC
                            return System.DateTime.create$2(System.DateTime.getTicks(value), 1);
                        case 2: 
                            return System.DateTime.toUniversalTime(value);
                        default: 
                            throw new System.ArgumentException.$ctor1(System.String.format("DateTimeKind: {0} is not supported", [Bridge.box(System.DateTime.getKind(value), System.DateTimeKind, System.Enum.toStringFn(System.DateTimeKind))]));
                    }
                },
                FastAppendFormattedInt7: function (sb, value) {
                    var g = System.Int64.clipu16(System.Int64(48).add(value.mod(System.Int64(10))));
                    value = value.div(System.Int64(10));
                    var f = System.Int64.clipu16(System.Int64(48).add(value.mod(System.Int64(10))));
                    value = value.div(System.Int64(10));
                    var e = System.Int64.clipu16(System.Int64(48).add(value.mod(System.Int64(10))));
                    value = value.div(System.Int64(10));
                    var d = System.Int64.clipu16(System.Int64(48).add(value.mod(System.Int64(10))));
                    value = value.div(System.Int64(10));
                    var c = System.Int64.clipu16(System.Int64(48).add(value.mod(System.Int64(10))));
                    value = value.div(System.Int64(10));
                    var b = System.Int64.clipu16(System.Int64(48).add(value.mod(System.Int64(10))));
                    value = value.div(System.Int64(10));
                    var a = System.Int64.clipu16(System.Int64(48).add(value.mod(System.Int64(10))));
                    sb.append(String.fromCharCode(a));
                    sb.append(String.fromCharCode(b));
                    sb.append(String.fromCharCode(c));
                    sb.append(String.fromCharCode(d));
                    sb.append(String.fromCharCode(e));
                    sb.append(String.fromCharCode(f));
                    sb.append(String.fromCharCode(g));
                },
                FastAppendFormattedInt4: function (sb, value) {
                    var d = (((48 + value % 10) | 0)) & 65535;
                    value = (Bridge.Int.div(value, 10)) | 0;
                    var c = (((48 + value % 10) | 0)) & 65535;
                    value = (Bridge.Int.div(value, 10)) | 0;
                    var b = (((48 + value % 10) | 0)) & 65535;
                    value = (Bridge.Int.div(value, 10)) | 0;
                    var a = (((48 + value % 10) | 0)) & 65535;
                    sb.append(String.fromCharCode(a));
                    sb.append(String.fromCharCode(b));
                    sb.append(String.fromCharCode(c));
                    sb.append(String.fromCharCode(d));
                },
                FastAppendFormattedInt2: function (sb, value) {
                    var b = (((48 + value % 10) | 0)) & 65535;
                    value = (Bridge.Int.div(value, 10)) | 0;
                    var a = (((48 + value % 10) | 0)) & 65535;
                    sb.append(String.fromCharCode(a));
                    sb.append(String.fromCharCode(b));
                },
                FastGetParts: function (value, year, month, day, hour, minute, second, fraction) {
                    var ticks = System.DateTime.getTicks(value);
                    // n = number of days since 1/1/0001
                    var n = System.Int64.clip32(ticks.div(Kusto.Cloud.Platform.Utils.ExtendedDateTime.TicksPerDay));
                    // y400 = number of whole 400-year periods since 1/1/0001
                    var y400 = (Bridge.Int.div(n, Kusto.Cloud.Platform.Utils.ExtendedDateTime.DaysPer400Years)) | 0;
                    // n = day number within 400-year period
                    n = (n - (Bridge.Int.mul(y400, Kusto.Cloud.Platform.Utils.ExtendedDateTime.DaysPer400Years))) | 0;
                    // y100 = number of whole 100-year periods within 400-year period
                    var y100 = (Bridge.Int.div(n, Kusto.Cloud.Platform.Utils.ExtendedDateTime.DaysPer100Years)) | 0;
                    // Last 100-year period has an extra day, so decrement result if 4
                    if (y100 === 4) {
                        y100 = 3;
                    }
                    // n = day number within 100-year period
                    n = (n - (Bridge.Int.mul(y100, Kusto.Cloud.Platform.Utils.ExtendedDateTime.DaysPer100Years))) | 0;
                    // y4 = number of whole 4-year periods within 100-year period
                    var y4 = (Bridge.Int.div(n, Kusto.Cloud.Platform.Utils.ExtendedDateTime.DaysPer4Years)) | 0;
                    // n = day number within 4-year period
                    n = (n - (Bridge.Int.mul(y4, Kusto.Cloud.Platform.Utils.ExtendedDateTime.DaysPer4Years))) | 0;
                    // y1 = number of whole years within 4-year period
                    var y1 = (Bridge.Int.div(n, Kusto.Cloud.Platform.Utils.ExtendedDateTime.DaysPerYear)) | 0;
                    // Last year has an extra day, so decrement result if 4
                    if (y1 === 4) {
                        y1 = 3;
                    }

                    year.v = (((((((Bridge.Int.mul(y400, 400) + Bridge.Int.mul(y100, 100)) | 0) + Bridge.Int.mul(y4, 4)) | 0) + y1) | 0) + 1) | 0;

                    // n = day number within year
                    n = (n - (Bridge.Int.mul(y1, Kusto.Cloud.Platform.Utils.ExtendedDateTime.DaysPerYear))) | 0;

                    // If day-of-year was requested, return it
                    //if (part == DatePartDayOfYear) return n + 1;

                    // Leap year calculation looks different from IsLeapYear since y1, y4,
                    // and y100 are relative to year 1, not year 0
                    var leapYear = y1 === 3 && (y4 !== 24 || y100 === 3);
                    var days = leapYear ? Kusto.Cloud.Platform.Utils.ExtendedDateTime.DaysToMonth366 : Kusto.Cloud.Platform.Utils.ExtendedDateTime.DaysToMonth365;
                    // All months have less than 32 days, so n >> 5 is a good conservative
                    // estimate for the month
                    var m = n >> 6;
                    // m = 1-based month number
                    while (n >= days[System.Array.index(m, days)]) {
                        m = (m + 1) | 0;
                    }

                    month.v = m;

                    // 1-based day-of-month
                    day.v = (((n - days[System.Array.index(((m - 1) | 0), days)]) | 0) + 1) | 0;

                    hour.v = System.DateTime.getHour(value);

                    minute.v = System.DateTime.getMinute(value);

                    second.v = System.DateTime.getSecond(value);

                    fraction.v = ticks.mod(Kusto.Cloud.Platform.Utils.ExtendedDateTime.TicksPerSecond);
                },
                FromUnixTimeSeconds$1: function (seconds) {
                    return System.DateTimeOffset.FromUnixTimeSeconds(seconds).UtcDateTime;
                },
                FromUnixTimeSeconds: function (seconds) {
                    var secondsLong = Bridge.Int.clip64(seconds);
                    var additionTicks = Bridge.Int.clip64((seconds - System.Int64.toNumber(secondsLong)) * Kusto.Cloud.Platform.Utils.ExtendedDateTime.TicksPerSecond);
                    return System.DateTimeOffset.FromUnixTimeSeconds(secondsLong).AddTicks(additionTicks).UtcDateTime;
                },
                FromUnixTimeMilliseconds$1: function (milliseconds) {
                    return System.DateTimeOffset.FromUnixTimeMilliseconds(milliseconds).UtcDateTime;
                },
                FromUnixTimeMilliseconds: function (milliseconds) {
                    var millisecondsLong = Bridge.Int.clip64(milliseconds);
                    var additionTicks = Bridge.Int.clip64((milliseconds - System.Int64.toNumber(millisecondsLong)) * Kusto.Cloud.Platform.Utils.ExtendedDateTime.TicksPerMillisecond);
                    return System.DateTimeOffset.FromUnixTimeMilliseconds(millisecondsLong).AddTicks(additionTicks).UtcDateTime;
                },
                FromUnixTimeMicroseconds$1: function (microseconds) {
                    var TicksPerMicrosecond = System.Int64(10);
                    var MicrosecondsPerMillisecond = System.Int64(1000);
                    var milliseconds = microseconds.div(MicrosecondsPerMillisecond);
                    var additionTicks = (microseconds.sub(milliseconds.mul(MicrosecondsPerMillisecond))).mul(TicksPerMicrosecond);
                    var result = System.DateTimeOffset.FromUnixTimeMilliseconds(milliseconds).AddTicks(additionTicks).UtcDateTime;
                    return result;
                },
                FromUnixTimeMicroseconds: function (microseconds) {
                    var TicksPerMicrosecond = System.Int64(10);
                    var MicrosecondsPerMillisecond = System.Int64(1000);
                    var milliseconds = Bridge.Int.clip64(microseconds / MicrosecondsPerMillisecond);
                    var additionTicks = Bridge.Int.clip64((microseconds - System.Int64.toNumber(milliseconds.mul(MicrosecondsPerMillisecond))) * TicksPerMicrosecond);
                    var result = System.DateTimeOffset.FromUnixTimeMilliseconds(milliseconds).AddTicks(additionTicks).UtcDateTime;
                    return result;
                },
                FromUnixTimeNanoseconds: function (nanoseconds) {
                    var NanosecondsPerTick = System.Int64(100);
                    var NanosecondsPerMillisecond = System.Int64(1000000);
                    var milliseconds = nanoseconds.div(NanosecondsPerMillisecond);
                    var additionTicks = (nanoseconds.sub(milliseconds.mul(NanosecondsPerMillisecond))).div(NanosecondsPerTick);
                    var result = System.DateTimeOffset.FromUnixTimeMilliseconds(milliseconds).AddTicks(additionTicks).UtcDateTime;
                    return result;
                }
            }
        }
    });

    Bridge.ns("Kusto.Cloud.Platform.Utils.ExtendedDateTime", $asm.$);

    Bridge.apply($asm.$.Kusto.Cloud.Platform.Utils.ExtendedDateTime, {
        f1: function (_o1) {
            _o1.add(4, System.Array.init(["yyyy"], System.String));
            _o1.add(6, System.Array.init(["yyyyMM"], System.String));
            _o1.add(8, System.Array.init(["yyyyMMdd"], System.String));
            _o1.add(10, System.Array.init(["yyyyMMddHH"], System.String));
            _o1.add(12, System.Array.init(["yyyyMMddHHmm"], System.String));
            _o1.add(14, System.Array.init(["yyyyMMddHHmmss"], System.String));
            _o1.add(17, System.Array.init(["yyyyMMdd HH:mm:ss"], System.String));
            _o1.add(19, System.Array.init(["yyyyMMdd HH:mm:ss.f"], System.String));
            _o1.add(20, System.Array.init(["yyyyMMdd HH:mm:ss.ff"], System.String));
            _o1.add(21, System.Array.init(["yyyyMMdd HH:mm:ss.fff"], System.String));
            _o1.add(22, System.Array.init(["yyyyMMdd HH:mm:ss.ffff"], System.String));
            _o1.add(23, System.Array.init(["yyyyMMdd HH:mm:ss.fffff"], System.String));
            _o1.add(24, System.Array.init(["yyyyMMdd HH:mm:ss.ffffff"], System.String));
            _o1.add(25, System.Array.init(["yyyyMMdd HH:mm:ss.fffffff"], System.String));
            return _o1;
        }
    });

    /**
     * Date Time Round To options
     *
     * @public
     * @class number
     */
    Bridge.define("Kusto.Cloud.Platform.Utils.ExtendedDateTime.RoundTo", {
        $kind: "nested enum",
        statics: {
            fields: {
                /**
                 * Round to nearest second
                 *
                 * @static
                 * @public
                 * @memberof number
                 * @constant
                 * @default 0
                 * @type number
                 */
                Second: 0,
                /**
                 * Round to nearest minute
                 *
                 * @static
                 * @public
                 * @memberof number
                 * @constant
                 * @default 1
                 * @type number
                 */
                Minute: 1,
                /**
                 * Round to nearest hour
                 *
                 * @static
                 * @public
                 * @memberof number
                 * @constant
                 * @default 2
                 * @type number
                 */
                Hour: 2,
                /**
                 * Round to nearest day
                 *
                 * @static
                 * @public
                 * @memberof number
                 * @constant
                 * @default 3
                 * @type number
                 */
                Day: 3
            }
        }
    });

    /**
     * @memberof System
     * @callback System.Func
     * @param   {T}          arg1    
     * @param   {T}          arg2
     * @return  {boolean}
     */

    /**
     * Enumerable extension methods
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Cloud.Platform.Utils.ExtendedEnumerable
     */
    Bridge.define("Kusto.Cloud.Platform.Utils.ExtendedEnumerable", {
        statics: {
            methods: {
                /**
                 * Like Enumerable.LongCount, but stops counting at maxCount.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    TSource     
                 * @param   {System.Collections.Generic.IEnumerable$1}    source      
                 * @param   {System.Int64}                                maxCount
                 * @return  {System.Int64}
                 */
                LongCount: function (TSource, source, maxCount) {
                    if (maxCount.equals(System.Int64.MaxValue)) {
                        return System.Linq.Enumerable.from(source, TSource).count();
                    }

                    if (source == null) {
                        throw new System.ArgumentNullException.$ctor1("source");
                    }
                    if (maxCount.lt(System.Int64(0))) {
                        throw new System.ArgumentOutOfRangeException.$ctor1("maxCount");
                    }

                    var count = System.Int64(0);
                    var e = Bridge.getEnumerator(source, TSource);
                    try {
                        while (e.System$Collections$IEnumerator$moveNext()) {
                            count = count.inc(1);
                            if (count.equals(maxCount)) {
                                break;
                            }
                        }
                    }
                    finally {
                        if (Bridge.hasValue(e)) {
                            e.System$IDisposable$Dispose();
                        }
                    }

                    return count;
                },
                SafeFastDistinct: function (T, collection) {
                    if (collection == null) {
                        return null;
                    }
                    var asICollectionT;
                    if (((asICollectionT = Bridge.as(collection, System.Collections.Generic.ICollection$1(T)))) != null) {
                        if (System.Array.getCount(asICollectionT, T) <= 1) {
                            return collection;
                        }
                        // Otherwise fall back to using Distinct()
                    }
                    var asICollection;
                    if (((asICollection = Bridge.as(collection, System.Collections.ICollection))) != null) {
                        if (System.Array.getCount(asICollection) <= 1) {
                            return collection;
                        }
                        // Otherwise fall back to using Distinct()
                    }


                    return System.Linq.Enumerable.from(collection, T).distinct();
                },
                /**
                 * Counts how many elements are in the collection, but does it safely
                 (if the collection is null, returns 0) and efficiently (if the count
                 is more than maxCount, stops at maxCount).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T             
                 * @param   {System.Collections.Generic.IEnumerable$1}    collection    
                 * @param   {System.Int64}                                maxCount
                 * @return  {System.Int64}
                 */
                SafeFastCountUpTo: function (T, collection, maxCount) {
                    if (maxCount.lt(System.Int64(0))) {
                        throw new System.ArgumentOutOfRangeException.$ctor1("maxCount");
                    }

                    if (collection == null) {
                        return System.Int64(0);
                    }

                    { // Scope to ensure no cross-talk with next block
                        var asICollectionT;
                        if (((asICollectionT = Bridge.as(collection, System.Collections.Generic.ICollection$1(T)))) != null) {
                            return System.Int64.min(maxCount, System.Int64(System.Array.getCount(asICollectionT, T)));
                        }
                    }

                    {
                        var asICollection;
                        if (((asICollection = Bridge.as(collection, System.Collections.ICollection))) != null) {
                            return System.Int64.min(maxCount, System.Int64(System.Array.getCount(asICollection)));
                        }
                    }



                    return Kusto.Cloud.Platform.Utils.ExtendedEnumerable.LongCount(T, collection, maxCount);
                    //return collection.LongCount(maxCount);
                },
                /**
                 * Safely checks if 2 sequences are the same, by ordering them and then using an IEqualityComparer (or a default EqualityComparer in case of null).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                          T                   
                 * @param   {System.Collections.Generic.IEnumerable$1}          collection          
                 * @param   {System.Collections.Generic.IEnumerable$1}          other               
                 * @param   {System.Collections.Generic.IEqualityComparer$1}    equalityComparer
                 * @return  {boolean}
                 */
                SafeSequenceEquals: function (T, collection, other, equalityComparer) {
                    if (equalityComparer === void 0) { equalityComparer = null; }
                    var basicComparison = Kusto.Cloud.Platform.Utils.ExtendedEnumerable.BasicCompare(T, collection, other);

                    if (System.Nullable.hasValue(basicComparison)) {
                        return System.Nullable.getValue(basicComparison);
                    }

                    return System.Linq.Enumerable.from(System.Linq.Enumerable.from(collection, T).orderBy($asm.$.Kusto.Cloud.Platform.Utils.ExtendedEnumerable.f1), T).sequenceEqual(System.Linq.Enumerable.from(other, T).orderBy($asm.$.Kusto.Cloud.Platform.Utils.ExtendedEnumerable.f2), equalityComparer);
                },
                /**
                 * Safely checks if 2 already sorted sequences are the same,
                 returns true only if all elements and their corresponding element are equal with the given IEqualityComparer (or a default EqualityComparer in case of null).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                          T                    
                 * @param   {System.Collections.Generic.IEnumerable$1}          sortedCollection1    The sorted first collection.
                 * @param   {System.Collections.Generic.IEnumerable$1}          sortedCollection2    The sorted second collection2.
                 * @param   {System.Collections.Generic.IEqualityComparer$1}    equalityComparer     The equality comparer.
                 * @return  {boolean}
                 */
                SortedSequenceEquals: function (T, sortedCollection1, sortedCollection2, equalityComparer) {
                    var $t;
                    if (equalityComparer === void 0) { equalityComparer = null; }
                    var basicComparison = Kusto.Cloud.Platform.Utils.ExtendedEnumerable.BasicCompare(T, sortedCollection1, sortedCollection2);

                    if (System.Nullable.hasValue(basicComparison)) {
                        return System.Nullable.getValue(basicComparison);
                    }

                    return Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SequenceEqual(T, sortedCollection1, sortedCollection2, Bridge.fn.cacheBind($t = (equalityComparer || System.Collections.Generic.EqualityComparer$1(T).def), $t)[Bridge.geti($t, "System$Collections$Generic$IEqualityComparer$1$" + Bridge.getTypeAlias(T) + "$equals2", "System$Collections$Generic$IEqualityComparer$1$equals2")]));
                },
                /**
                 * Safely checks if 2 sequences are the same using a predicate. Does not order the sequences and compares their elements as is.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T             
                 * @param   {System.Collections.Generic.IEnumerable$1}    collection    
                 * @param   {System.Collections.Generic.IEnumerable$1}    other         
                 * @param   {System.Func}                                 predicate
                 * @return  {boolean}
                 */
                SequenceEqual: function (T, collection, other, predicate) {
                    var basicComparison = Kusto.Cloud.Platform.Utils.ExtendedEnumerable.BasicCompare(T, collection, other);

                    if (System.Nullable.hasValue(basicComparison)) {
                        return System.Nullable.getValue(basicComparison);
                    }
                    var enumerator1 = Bridge.getEnumerator(collection, T);
                    try {
                        var enumerator2 = Bridge.getEnumerator(other, T);
                        try {
                            while (enumerator1.System$Collections$IEnumerator$moveNext()) {
                                if (!enumerator2.System$Collections$IEnumerator$moveNext() || !predicate(enumerator1[Bridge.geti(enumerator1, "System$Collections$Generic$IEnumerator$1$" + Bridge.getTypeAlias(T) + "$Current$1", "System$Collections$Generic$IEnumerator$1$Current$1")], enumerator2[Bridge.geti(enumerator2, "System$Collections$Generic$IEnumerator$1$" + Bridge.getTypeAlias(T) + "$Current$1", "System$Collections$Generic$IEnumerator$1$Current$1")])) {
                                    return false;
                                }
                            }

                            if (enumerator2.System$Collections$IEnumerator$moveNext()) {
                                return false;
                            }
                        }
                        finally {
                            if (Bridge.hasValue(enumerator2)) {
                                enumerator2.System$IDisposable$Dispose();
                            }
                        }
                    }
                    finally {
                        if (Bridge.hasValue(enumerator1)) {
                            enumerator1.System$IDisposable$Dispose();
                        }
                    }

                    return true;
                },
                /**
                 * Safely compares two collections, ignoring duplications.
                 Returns true if they are equal.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T        
                 * @param   {System.Collections.Generic.IEnumerable$1}    that     
                 * @param   {System.Collections.Generic.IEnumerable$1}    other
                 * @return  {boolean}
                 */
                SafeSetEquals: function (T, that, other) {
                    if (Bridge.referenceEquals(that, other)) {
                        return true;
                    }

                    if (that == null || other == null) {
                        return false;
                    }

                    return new (System.Collections.Generic.HashSet$1(T)).$ctor1(that).setEquals(other);
                },
                BasicCompare: function (T, collection, other) {
                    if (Bridge.referenceEquals(collection, other)) {
                        return true;
                    }

                    if (collection == null || other == null) {
                        return false;
                    }

                    var length1 = Kusto.Cloud.Platform.Utils.ExtendedEnumerable.GetSequenceLength(T, collection);
                    var length2 = Kusto.Cloud.Platform.Utils.ExtendedEnumerable.GetSequenceLength(T, other);

                    if (System.Nullable.hasValue(length1) && System.Nullable.hasValue(length2)) {
                        if (System.Nullable.getValue(length1).ne(System.Nullable.getValue(length2))) {
                            return false;
                        }
                    }

                    return null;
                },
                GetSequenceLength: function (T, sequence) {
                    { // Scope to ensure no cross-talk with next block
                        var asArrayT;
                        if (((asArrayT = Bridge.as(sequence, System.Array.type(T)))) != null) {
                            return System.Int64.lift(asArrayT.length);
                        }
                    }

                    {
                        var asIReadOnlyCollectionT;
                        if (((asIReadOnlyCollectionT = Bridge.as(sequence, System.Collections.Generic.IReadOnlyCollection$1(T)))) != null) {
                            return System.Int64.lift(System.Array.getCount(asIReadOnlyCollectionT, T));
                        }
                    }

                    {
                        var asICollectionT;
                        if (((asICollectionT = Bridge.as(sequence, System.Collections.Generic.ICollection$1(T)))) != null) {
                            return System.Int64.lift(System.Array.getCount(asICollectionT, T));
                        }
                    }

                    {
                        var asICollection;
                        if (((asICollection = Bridge.as(sequence, System.Collections.ICollection))) != null) {
                            return System.Int64.lift(System.Array.getCount(asICollection));
                        }
                    }

                    return System.Int64.lift(null);
                },
                /**
                 * Creates a display string for collection
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T                      
                 * @param   {System.Collections.Generic.IEnumerable$1}    collection             The collection.
                 * @param   {string}                                      delimiter              The delimiter between each displayed element.
                 * @param   {?number}                                     maxTake                The maximum number of elements displayed.
                 * @param   {string}                                      nullString             The string representing null value.
                 * @param   {number}                                      estimatedStringSize    Estimated final string size, for improved performance.
                 * @param   {string}                                      elementPrefix          Prefix to element.
                 * @param   {string}                                      elementPostfix         Postfix to element.
                 * @return  {string}                                                             Formatted display string
                 */
                ToDisplayString: function (T, collection, delimiter, maxTake, nullString, estimatedStringSize, elementPrefix, elementPostfix) {
                    var $t, $t1, $t2, $t3;
                    if (delimiter === void 0) { delimiter = ", "; }
                    if (maxTake === void 0) { maxTake = null; }
                    if (nullString === void 0) { nullString = "null"; }
                    if (estimatedStringSize === void 0) { estimatedStringSize = 128; }
                    if (elementPrefix === void 0) { elementPrefix = ""; }
                    if (elementPostfix === void 0) { elementPostfix = ""; }
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(T, collection) || System.Nullable.lte(maxTake, 0)) {
                        return "";
                    }
                    delimiter = ($t = delimiter, $t != null ? $t : ", ");
                    nullString = ($t1 = nullString, $t1 != null ? $t1 : "null");

                    // Currently StringBuilder.DefaultCapacity = 16 which I find is too low. prefer using higher default of 128.
                    var displayStringBuilder = new System.Text.StringBuilder("", estimatedStringSize);
                    var itemsPassed = 0;
                    var enumerator = Bridge.getEnumerator(collection, T);
                    try {
                        if (enumerator.System$Collections$IEnumerator$moveNext()) {
                            var toStringRes = ($t2 = enumerator[Bridge.geti(enumerator, "System$Collections$Generic$IEnumerator$1$" + Bridge.getTypeAlias(T) + "$Current$1", "System$Collections$Generic$IEnumerator$1$Current$1")]) != null ? Bridge.toString($t2) : null;
                            displayStringBuilder.append(toStringRes != null ? (elementPrefix || "") + (toStringRes || "") + (elementPostfix || "") : nullString);
                            itemsPassed = (itemsPassed + 1) | 0;
                        }
                        while (enumerator.System$Collections$IEnumerator$moveNext() && !(System.Nullable.gte(itemsPassed, maxTake))) {
                            displayStringBuilder.append(delimiter);
                            var toStringRes1 = ($t3 = enumerator[Bridge.geti(enumerator, "System$Collections$Generic$IEnumerator$1$" + Bridge.getTypeAlias(T) + "$Current$1", "System$Collections$Generic$IEnumerator$1$Current$1")]) != null ? Bridge.toString($t3) : null;
                            displayStringBuilder.append(toStringRes1 != null ? (elementPrefix || "") + (toStringRes1 || "") + (elementPostfix || "") : nullString);
                            itemsPassed = (itemsPassed + 1) | 0;
                        }
                        if (enumerator.System$Collections$IEnumerator$moveNext()) {
                            displayStringBuilder = displayStringBuilder.append(delimiter).append("... more");
                        }
                    }
                    finally {
                        if (Bridge.hasValue(enumerator)) {
                            enumerator.System$IDisposable$Dispose();
                        }
                    }

                    return displayStringBuilder.toString();
                },
                /**
                 * Safely clones an existing collection in a shallow manner (it's the responsibility
                 of the element itself to be a value type of immutable).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T             
                 * @param   {System.Collections.Generic.IEnumerable$1}    collection
                 * @return  {System.Collections.Generic.IEnumerable$1}
                 */
                SafeShallowClone: function (T, collection) {
                    if (collection == null) {
                        return null;
                    }

                    return new (System.Collections.Generic.List$1(T)).$ctor1(collection);
                },
                /**
                 * Returns the 0-based index of the first item in the collection
                 for which the predicate returns true, or -1.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    TCollection    
                 * @param   {System.Collections.Generic.IEnumerable$1}    collection     
                 * @param   {System.Func}                                 predicate
                 * @return  {number}
                 */
                IndexOf: function (TCollection, collection, predicate) {
                    var $t;
                    // TODO: What about Javascript?
                    // Ensure.ArgIsNotNull(predicate, "predicate");
                    if (Bridge.staticEquals(predicate, null)) {
                        return -1;
                    }

                    if (collection == null) {
                        return -1;
                    }

                    var i = 0;
                    $t = Bridge.getEnumerator(collection, TCollection);
                    try {
                        while ($t.moveNext()) {
                            var item = $t.Current;
                            if (predicate(item)) {
                                return i;
                            }
                            i = (i + 1) | 0;
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return -1;
                },
                /**
                 * Returns a collection that is the concatenation of two other collections.
                 Any of the arguments may be null. Result might also be null.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T         
                 * @param   {System.Collections.Generic.IEnumerable$1}    first     
                 * @param   {System.Collections.Generic.IEnumerable$1}    second
                 * @return  {System.Collections.Generic.IEnumerable$1}
                 */
                SafeConcat: function (T, first, second) {
                    if (first == null) {
                        return second;
                    }

                    if (second == null) {
                        return first;
                    }

                    return System.Linq.Enumerable.from(first, T).concat(second);
                },
                /**
                 * Returns a collection that is the union of two other collections (duplicates are excluded).
                 Any of the arguments may be null. Result might also be null.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T         
                 * @param   {System.Collections.Generic.IEnumerable$1}    first     
                 * @param   {System.Collections.Generic.IEnumerable$1}    second
                 * @return  {System.Collections.Generic.IEnumerable$1}
                 */
                SafeUnion: function (T, first, second) {
                    if (first == null && second == null) {
                        return null;
                    }

                    if (first == null) {
                        return System.Linq.Enumerable.from(second, T).distinct();
                    }

                    if (second == null) {
                        return System.Linq.Enumerable.from(first, T).distinct();
                    }

                    return System.Linq.Enumerable.from(first, T).union(second);
                },
                /**
                 * Returns the collection unless it is null or holds no elements,
                 in which case return null.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}       TCollection    
                 * @param   {TCollection}    collection
                 * @return  {TCollection}
                 */
                SafeFastNullIfEmpty: function (TCollection, collection) {
                    if (collection == null || Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone(collection)) {
                        return null;
                    }

                    return collection;
                },
                SafeEmptyIEnumerableIfNull: function (T, collectionEnumerable) {
                    return collectionEnumerable || System.Linq.Enumerable.empty();
                },
                /**
                 * Similar to IEnumerable{T}.Any, only faster and works on null arguments.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T             
                 * @param   {System.Collections.Generic.IEnumerable$1}    collection
                 * @return  {boolean}
                 */
                SafeFastAny$2: function (T, collection) {
                    if (collection == null) {
                        return false;
                    }

                    { // Scope to ensure no cross-talk with next block
                        var asICollection;
                        if (((asICollection = Bridge.as(collection, System.Collections.ICollection))) != null) {
                            return System.Array.getCount(asICollection) !== 0;
                        }
                    }

                    {
                        var asIReadOnlyCollection;
                        if (((asIReadOnlyCollection = Bridge.as(collection, System.Collections.Generic.IReadOnlyCollection$1(T)))) != null) {
                            return System.Array.getCount(asIReadOnlyCollection, T) !== 0;
                        }
                    }


                    return System.Linq.Enumerable.from(collection, T).any();
                },
                SafeFastAny$4: function (T, list) {
                    return list != null && list.Count > 0;
                },
                SafeFastAny$1: function (T, collection) {
                    return collection != null && System.Array.getCount(collection, T) > 0;
                },
                /**
                 * Similar to IEnumerable.Any, only faster and works on null arguments.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {System.Collections.IEnumerable}    collection
                 * @return  {boolean}
                 */
                SafeFastAny: function (collection) {
                    var $t;
                    if (collection == null) {
                        return false;
                    }

                    { // Scope to ensure no cross-talk with next block
                        var asICollection;
                        if (((asICollection = Bridge.as(collection, System.Collections.ICollection))) != null) {
                            return System.Array.getCount(asICollection) !== 0;
                        }
                    }

                    var e = Bridge.getEnumerator(collection);
                    $t = Bridge.as(e, System.IDisposable);
                    try {
                        if (e.System$Collections$IEnumerator$moveNext()) {
                            return true;
                        }
                    }
                    finally {
                        if (Bridge.hasValue($t)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }
                    return false;
                },
                SafeFastAny$3: function (T, collection, predicate) {
                    if (!Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$2(T, collection)) {
                        return false;
                    }

                    return System.Linq.Enumerable.from(collection, T).any(predicate);
                },
                /**
                 * Similar to IEnumerable{T}.None, only faster and works on null arguments.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T             
                 * @param   {System.Collections.Generic.IEnumerable$1}    collection
                 * @return  {boolean}
                 */
                SafeFastNone$2: function (T, collection) {
                    return !Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$2(T, collection);
                },
                SafeFastNone$1: function (T, collection) {
                    return !Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$1(T, collection);
                },
                /**
                 * Similar to IEnumerable.None, only faster and works on null arguments.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {System.Collections.IEnumerable}    collection
                 * @return  {boolean}
                 */
                SafeFastNone: function (collection) {
                    return !Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny(collection);
                },
                SafeFastNone$3: function (T, collection, predicate) {
                    return !Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$3(T, collection, predicate);
                },
                /**
                 * Inverse of IEnumerable{T}.Any(predicate)
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T            Element type
                 * @param   {System.Collections.Generic.IEnumerable$1}    source       The source IEnumerable.
                 * @param   {System.Func}                                 predicate    The predicate.
                 * @return  {boolean}                                                  true if none of the specified collection items match the given predicate
                 */
                None$1: function (T, source, predicate) {
                    // TODO: Should we call here SafeFastAny to be aligned with the behavior of the method below?
                    return !System.Linq.Enumerable.from(source, T).any(predicate);
                },
                /**
                 * Overload. Inverse of IEnumerable{T}.Any()
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T         Element type
                 * @param   {System.Collections.Generic.IEnumerable$1}    source    The source IEnumerable.
                 * @return  {boolean}                                               true if the collection is empty
                 */
                None: function (T, source) {
                    return Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(T, source);
                },
                /**
                 * Similar to IEnumerable{T}.Contains, only faster and works on null arguments.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                          T             
                 * @param   {System.Collections.Generic.IEnumerable$1}          collection    
                 * @param   {T}                                                 item          
                 * @param   {System.Collections.Generic.IEqualityComparer$1}    comparer
                 * @return  {boolean}
                 */
                SafeFastContains: function (T, collection, item, comparer) {
                    if (comparer === void 0) { comparer = null; }
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(T, collection)) {
                        return false;
                    }

                    return System.Linq.Enumerable.from(collection, T).contains(item, comparer);
                },
                /**
                 * Returns a collection where each key appears only once
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T              Element type
                 * @param   {Function}                                    TKey           Key selector
                 * @param   {System.Collections.Generic.IEnumerable$1}    source         IEnumerable collection
                 * @param   {System.Func}                                 keySelector    Key selector function
                 * @return  {System.Collections.Generic.IEnumerable$1}                   Returns collection of items that are aggregated by unique key selector
                 */
                DistinctBy: function (T, TKey, source, keySelector) {
                    return System.Linq.Enumerable.from(source, T).groupBy(keySelector).select($asm.$.Kusto.Cloud.Platform.Utils.ExtendedEnumerable.f3);
                },
                /**
                 * Returns an intersection of the given collections
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T              Element type
                 * @param   {System.Collections.Generic.IEnumerable$1}    collections    The collection to intersect
                 * @return  {System.Collections.Generic.IEnumerable$1}                   Returns collection of items with the given collections intersection
                 */
                Intersect: function (T, collections) {
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(System.Collections.Generic.IEnumerable$1(T), collections)) {
                        return null;
                    }

                    var enumerator = Bridge.getEnumerator(collections, System.Collections.Generic.IEnumerable$1(T));
                    if (enumerator.System$Collections$IEnumerator$moveNext()) {
                        var result = new (System.Collections.Generic.HashSet$1(T)).$ctor1(enumerator[Bridge.geti(enumerator, "System$Collections$Generic$IEnumerator$1$System$Collections$Generic$IEnumerable$1$" + Bridge.getTypeAlias(T) + "$Current$1", "System$Collections$Generic$IEnumerator$1$Current$1")]);

                        while (enumerator.System$Collections$IEnumerator$moveNext()) {
                            result.intersectWith(enumerator[Bridge.geti(enumerator, "System$Collections$Generic$IEnumerator$1$System$Collections$Generic$IEnumerable$1$" + Bridge.getTypeAlias(T) + "$Current$1", "System$Collections$Generic$IEnumerator$1$Current$1")]);
                        }

                        return result;
                    }
                    return null;
                },
                /**
                 * Splits a collection to a Tuple of two lists.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedEnumerable
                 * @param   {Function}                                    T             Element type of the collection.
                 * @param   {Function}                                    TR            Element type of the results in the generated lists.
                 * @param   {System.Collections.Generic.IEnumerable$1}    collection    The collection to split
                 * @param   {System.Func}                                 isFirst       Function that runs on each item and determines where it should go.
                 - null will be filtered out
                 - true will go in the first array
                 - false will go in the second array
                 * @param   {System.Func}                                 transform     Function that can transform the returned items.
                 - Default will be the first item of the tuple.
                 * @return  {System.Tuple$2}                                            Returns Tuple of two lists
                 */
                SplitBy: function (T, TR, collection, isFirst, transform) {
                    var $t;
                    if (transform === void 0) { transform = null; }
                    var typeOfT = T;
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.IsTupleType(typeOfT) && Bridge.staticEquals(transform, null)) {
                        transform = $asm.$.Kusto.Cloud.Platform.Utils.ExtendedEnumerable.f4;
                    } else if (Bridge.referenceEquals(typeOfT, TR) && Bridge.staticEquals(transform, null)) {
                        transform = function (t) {
                            return Bridge.cast(Bridge.unbox(t, TR), TR);
                        };
                    }

                    if (Bridge.staticEquals(transform, null)) {
                        throw new System.NullReferenceException.ctor();
                    }

                    var list1 = new (System.Collections.Generic.List$1(TR)).ctor();
                    var list2 = new (System.Collections.Generic.List$1(TR)).ctor();

                    $t = Bridge.getEnumerator(collection, T);
                    try {
                        while ($t.moveNext()) {
                            var item = $t.Current;
                            var itemToAdd = transform(item);
                            var first = isFirst(item);
                            switch (first) {
                                case null: 
                                case undefined: 
                                    continue;
                                case true: 
                                    list1.add(itemToAdd);
                                    break;
                                default: 
                                    list2.add(itemToAdd);
                                    break;
                            }
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return { Item1: list1, Item2: list2 };
                },
                IntersectWith: function (T, collection, other) {
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(T, collection)) {
                        return System.Linq.Enumerable.empty();
                    }

                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(T, other)) {
                        return System.Linq.Enumerable.empty();
                    }

                    var result = new (System.Collections.Generic.HashSet$1(T)).$ctor1(collection);
                    result.intersectWith(other);
                    return result;
                },
                IsTupleType: function (type, checkBaseTypes) {
                    if (checkBaseTypes === void 0) { checkBaseTypes = false; }
                    if (type == null) {
                        throw new System.ArgumentNullException.$ctor1("type");
                    }

                    if (Bridge.referenceEquals(type, System.Tuple)) {
                        return true;
                    }

                    while (type != null) {
                        if (Bridge.Reflection.isGenericType(type)) {
                            var genType = Bridge.Reflection.getGenericTypeDefinition(type);
                            if (Bridge.referenceEquals(genType, System.Tuple$1) || Bridge.referenceEquals(genType, System.Tuple$2) || Bridge.referenceEquals(genType, System.Tuple$3) || Bridge.referenceEquals(genType, System.Tuple$4) || Bridge.referenceEquals(genType, System.Tuple$5) || Bridge.referenceEquals(genType, System.Tuple$6) || Bridge.referenceEquals(genType, System.Tuple$7) || Bridge.referenceEquals(genType, System.Tuple$8) || Bridge.referenceEquals(genType, System.Tuple$8)) {
                                return true;
                            }
                        }

                        if (!checkBaseTypes) {
                            break;
                        }

                        type = Bridge.Reflection.getBaseType(type);
                    }

                    return false;
                }
            }
        }
    });

    Bridge.ns("Kusto.Cloud.Platform.Utils.ExtendedEnumerable", $asm.$);

    Bridge.apply($asm.$.Kusto.Cloud.Platform.Utils.ExtendedEnumerable, {
        f1: function (item) {
            return item;
        },
        f2: function (items) {
            return items;
        },
        f3: function (x) {
            return x.first();
        },
        f4: function (t) {
            return t.Item1;
        }
    });

    Bridge.define("Kusto.Cloud.Platform.Utils.ExtendedRegex", {
        statics: {
            methods: {
                TryUnescape: function (str, result) {
                    result.v = "";
                    try {
                        result.v = System.Text.RegularExpressions.Regex.unescape(str);
                        return true;
                    } catch ($e1) {
                        $e1 = System.Exception.create($e1);
                        if (Bridge.is($e1, System.ArgumentException)) {
                            return false;
                        } else {
                            throw $e1;
                        }
                    }
                },
                TryTransformWildCardPatternToRegex: function (wildcardPattern) {
                    try {
                        // First, remove the [] brackets
                        var input = System.String.trim(wildcardPattern, [91, 93]);
                        var regexString = System.String.replaceAll(input, "*", ".*");
                        return new System.Text.RegularExpressions.Regex.ctor(regexString, 0);
                    } catch ($e1) {
                        $e1 = System.Exception.create($e1);
                        // Creation of regex failed
                        return null;
                    }
                },
                IsWildCardPattern: function (wildcardPattern) {
                    return System.String.indexOf(wildcardPattern, String.fromCharCode(42)) >= 0;
                },
                RegexEquals: function (regex, other) {
                    if (Bridge.referenceEquals(regex, other)) {
                        return true;
                    }
                    var otherAsRegex;
                    if (regex == null || other == null || !(((otherAsRegex = Bridge.as(other, System.Text.RegularExpressions.Regex))) != null)) {
                        return false;
                    }

                    return Kusto.Cloud.Platform.Utils.ExtendedRegex.RegexEquals$1(regex, otherAsRegex);
                },
                /**
                 * Compares Regex instances based on their Pattern, Options and MatchTimeout
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedRegex
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedRegex
                 * @param   {System.Text.RegularExpressions.Regex}    regex    
                 * @param   {System.Text.RegularExpressions.Regex}    other
                 * @return  {boolean}
                 */
                RegexEquals$1: function (regex, other) {
                    if (Bridge.referenceEquals(regex, other)) {
                        return true;
                    }

                    if (regex == null || other == null) {
                        return false;
                    }

                    return System.String.equals(Bridge.toString(regex), Bridge.toString(other)) && regex.getOptions() === other.getOptions() && System.TimeSpan.eq(regex.getMatchTimeout(), other.getMatchTimeout());
                },
                /**
                 * Compares Regex instances based on their Pattern and Options (disregards MatchTimeout)
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedRegex
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedRegex
                 * @param   {System.Text.RegularExpressions.Regex}    regex    
                 * @param   {System.Text.RegularExpressions.Regex}    other
                 * @return  {boolean}
                 */
                RegexEquivalent: function (regex, other) {
                    if (Bridge.referenceEquals(regex, other)) {
                        return true;
                    }

                    if (regex == null || other == null) {
                        return false;
                    }

                    return System.String.equals(Bridge.toString(regex), Bridge.toString(other)) && regex.getOptions() === other.getOptions();
                },
                /**
                 * Replaces all strings in the input string that match the regular expression with the replacement string, and count number of replacements done.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedRegex
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedRegex
                 * @param   {System.Text.RegularExpressions.Regex}    regex            
                 * @param   {string}                                  input            
                 * @param   {string}                                  replacement      
                 * @param   {System.Int32}                            replacesCount
                 * @return  {string}
                 */
                ReplaceAndCountReplacements: function (regex, input, replacement, replacesCount) {
                    var count = 0;
                    var result = regex.replace(input, function (match) {
                        count = (count + 1) | 0;
                        return replacement;
                    });

                    replacesCount.v = count;
                    return result;
                }
            }
        }
    });

    /**
     * Extensions for {@link }
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Cloud.Platform.Utils.ExtendedString
     */
    Bridge.define("Kusto.Cloud.Platform.Utils.ExtendedString", {
        statics: {
            fields: {
                c_postfix: null,
                c_wrap: null,
                c_newlineAsStringArray: null,
                c_nullGuids: null,
                /**
                 * A delegate that can be used to get an object's value safely
                 when it is unknown if the object will be null or not.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @type System.Func
                 */
                SafeToString: null,
                /**
                 * An empty (no elements) string array.
                 *
                 * @static
                 * @public
                 * @readonly
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @type Array.<string>
                 */
                EmptyArray: null,
                s_alternativeSpaceChars: null,
                s_alternativeNewlineChars: null,
                /**
                 * Normalize quotes coming from 'Word' into ASCII quotes.
                 https://unicode-table.com/en/sets/quotation-marks/
                 *
                 * @static
                 * @private
                 * @readonly
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @type System.Collections.Generic.HashSet$1
                 */
                s_alternativeQutoes: null
            },
            ctors: {
                init: function () {
                    this.c_postfix = "...";
                    this.c_wrap = "  ";
                    this.c_newlineAsStringArray = System.Array.init(["\n"], System.String);
                    this.c_nullGuids = System.Array.init([System.Guid.Empty.toString(), "{" + (System.Guid.Empty.toString() || "") + "}"], System.String);
                    this.SafeToString = $asm.$.Kusto.Cloud.Platform.Utils.ExtendedString.f1;
                    this.EmptyArray = System.Array.init(0, null, System.String);
                    this.s_alternativeSpaceChars = $asm.$.Kusto.Cloud.Platform.Utils.ExtendedString.f2(new (System.Collections.Generic.HashSet$1(System.Char)).ctor());
                    this.s_alternativeNewlineChars = $asm.$.Kusto.Cloud.Platform.Utils.ExtendedString.f3(new (System.Collections.Generic.HashSet$1(System.Char)).ctor());
                    this.s_alternativeQutoes = $asm.$.Kusto.Cloud.Platform.Utils.ExtendedString.f4(new (System.Collections.Generic.HashSet$1(System.Char)).ctor());
                }
            },
            methods: {
                /**
                 * A safe version (will not throw on null) of string.GetHashCode().
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    value
                 * @return  {number}
                 */
                SafeGetHashCode: function (value) {
                    if (value == null) {
                        return 20080512;
                    }
                    return Bridge.getHashCode(value);
                },
                /**
                 * This fast calculation is relying on the fact that the different characters in the 
                 Guid are already random (across a limited range of values though)".
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    value
                 * @return  {number}
                 */
                GuidSafeFastGetHashCode: function (value) {
                    if (value == null || value.length < 26) {
                        return Kusto.Cloud.Platform.Utils.ExtendedString.SafeGetHashCode(value);
                    }
                    return (value.charCodeAt(1) ^ value.charCodeAt(9) << 8 | value.charCodeAt(10)) ^ (value.charCodeAt(16) << 16 | value.charCodeAt(17)) ^ (value.charCodeAt(24) << 24 | value.charCodeAt(25));
                },
                /**
                 * Returns the value of the string, unless it is a null or empty string,
                 in which returns a specific text to print in a trace.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    value
                 * @return  {string}
                 */
                SafeToTraceString: function (value) {
                    if (value == null) {
                        return "[null]";
                    }
                    if (value.length === 0) {
                        return "[empty]";
                    }
                    return value;
                },
                /**
                 * Like {@link }, but extra-safe with nulls.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}                   format    
                 * @param   {Array.<System.Object>}    args
                 * @return  {string}
                 */
                SafeFormat: function (format, args) {
                    if (args === void 0) { args = []; }
                    if (format == null) {
                        return "[format:null]";
                    }

                    if (args == null || args.length === 0) {
                        return System.String.format.apply(System.String, [format].concat(args));
                    }

                    var args2 = System.Array.init(args.length, null, System.String);
                    for (var a = 0; a < args.length; a = (a + 1) | 0) {
                        var arg = args[System.Array.index(a, args)];
                        var arg2;
                        if (arg == null) {
                            arg2 = "[null]";
                        } else {
                            var argStr = Bridge.toString(arg); // NOTE: This loses the user-provided format string!
                            if (argStr == null) {
                                arg2 = "[ToStringNull]";
                            } else if (argStr.length === 0) {
                                arg2 = "[ToStringEmpty]";
                            } else if (System.String.isNullOrWhiteSpace(argStr)) {
                                arg2 = "[ToStringWhitespace]";
                            } else {
                                arg2 = argStr;
                            }
                        }
                        args2[System.Array.index(a, args2)] = arg2;
                    }

                    return System.String.format.apply(System.String, [format].concat(args2));
                },
                /**
                 * Replaces the format item in a specified string with the string representation
                 of a corresponding object in a specified array with the current culture.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @throws {System.ArgumentNullException} format or args is null.
                 * @throws {System.FormatException} format is invalid.-or- The index of a format item is less than zero, or greater
                 than or equal to the length of the args array.
                 * @param   {string}                   format    A composite format string.
                 * @param   {Array.<System.Object>}    args      An object array that contains zero or more objects to format.
                 * @return  {string}                             A copy of format in which the format items have been replaced by the string
                 representation of the corresponding objects in args.
                 */
                FormatWithCurrentCulture: function (format, args) {
                    if (args === void 0) { args = []; }
                    return Kusto.Cloud.Platform.Utils.ExtendedString.FormatWith(format, System.Globalization.CultureInfo.getCurrentCulture(), args);
                },
                /**
                 * Replaces the format item in a specified string with the string representation
                 of a corresponding object in a specified array with an invariant culture.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @throws {System.ArgumentNullException} format or args is null.
                 * @throws {System.FormatException} format is invalid.-or- The index of a format item is less than zero, or greater
                 than or equal to the length of the args array.
                 * @param   {string}                   format    A composite format string.
                 * @param   {Array.<System.Object>}    args      An object array that contains zero or more objects to format.
                 * @return  {string}                             A copy of format in which the format items have been replaced by the string
                 representation of the corresponding objects in args.
                 */
                FormatWithInvariantCulture: function (format, args) {
                    if (args === void 0) { args = []; }
                    return Kusto.Cloud.Platform.Utils.ExtendedString.FormatWith(format, System.Globalization.CultureInfo.invariantCulture, args);
                },
                /**
                 * Replaces the format item in a specified string with the string representation
                 of a corresponding object in a specified array. A specified parameter supplies
                 culture-specific formatting information.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @throws {System.ArgumentNullException} format or args is null.
                 * @throws {System.FormatException} format is invalid.-or- The index of a format item is less than zero, or greater
                 than or equal to the length of the args array.
                 * @param   {string}                    format            A composite format string.
                 * @param   {System.IFormatProvider}    formatProvider    An object that supplies culture-specific formatting information.
                 * @param   {Array.<System.Object>}     args              An object array that contains zero or more objects to format.
                 * @return  {string}                                      A copy of format in which the format items have been replaced by the string
                 representation of the corresponding objects in args.
                 */
                FormatWith: function (format, formatProvider, args) {
                    if (args === void 0) { args = []; }
                    return System.String.formatProvider.apply(System.String, [formatProvider, format].concat(args));
                },
                /**
                 * Returns true if the string is null, empty, whitespace, or
                 indicates a null Guid (all zeros).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}     value
                 * @return  {boolean}
                 */
                SafeIsNullGuid: function (value) {
                    return System.String.isNullOrWhiteSpace(value) || Bridge.referenceEquals(value, Kusto.Cloud.Platform.Utils.ExtendedString.c_nullGuids[System.Array.index(0, Kusto.Cloud.Platform.Utils.ExtendedString.c_nullGuids)]) || Bridge.referenceEquals(value, Kusto.Cloud.Platform.Utils.ExtendedString.c_nullGuids[System.Array.index(1, Kusto.Cloud.Platform.Utils.ExtendedString.c_nullGuids)]);
                },
                /**
                 * Returns true if the string is a valid, non-null GUID
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}     value
                 * @return  {boolean}
                 */
                SafeIsValidGuid: function (value) {
                    var _discard1 = { v : new System.Guid() };
                    return !Kusto.Cloud.Platform.Utils.ExtendedString.SafeIsNullGuid(value) && System.Guid.TryParse(value, _discard1);
                },
                /**
                 * Gets back a string that describes the difference between two strings.
                 Useful mainly for tests, in case one wants to easily highlight test failures
                 due to expected/actual mismatch in the values of two strings.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    lhs    
                 * @param   {string}    rhs
                 * @return  {string}
                 */
                GetDiff: function (lhs, rhs) {
                    if (Bridge.referenceEquals(lhs, rhs)) {
                        if (lhs == null) {
                            return "Strings are the same (both null)";
                        } else if (Bridge.referenceEquals(lhs, "")) {
                            return "Strings are the same (both empty)";
                        }
                        return "Strings are the same";
                    }
                    if (lhs == null) {
                        return "LHS is null, RHS is not";
                    }
                    if (rhs == null) {
                        return "RHS is null, LHS is not";
                    }

                    var ret;
                    for (var i = 0; i < Math.min(lhs.length, rhs.length); i = (i + 1) | 0) {
                        if (lhs.charCodeAt(i) !== rhs.charCodeAt(i)) {
                            ret = Kusto.Cloud.Platform.Utils.ExtendedString.GetDiffFromPosition(lhs, rhs, i);
                            return ret;
                        }
                    }
                    // If we reached this point, one string is a proper prefix of the other
                    ret = Kusto.Cloud.Platform.Utils.ExtendedString.GetDiffFromPosition(lhs, rhs, Math.min(lhs.length, rhs.length));
                    return ret;
                },
                GetDiffFromPosition: function (lhs, rhs, position) {
                    var before = 20;
                    var after = 20;
                    var start = Math.max(0, ((position - before) | 0));
                    var count = (before + after) | 0;
                    var ret = "Strings differ starting from position " + (Bridge.toString(position) || "") + " " + ("\n" || "") + "LHS=...'" + (Kusto.Cloud.Platform.Utils.ExtendedString.SafeSubstring(lhs, start, count) || "") + "'..." + ("\n" || "") + "RHS=...'" + (Kusto.Cloud.Platform.Utils.ExtendedString.SafeSubstring(rhs, start, count) || "") + "'..." + ("\n" || "") + (System.String.fromCharCount(32, (((((start === 0 ? position : before) + 1) | 0) + ("LHS...'").length) | 0)) || "") + "^";
                    return ret;
                },
                /**
                 * Like {@link }, but ensures that it doesn't throw
                 on invalid strings or invalid boundary conditions.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    value         
                 * @param   {number}    startIndex    
                 * @param   {number}    count
                 * @return  {string}
                 */
                SafeSubstring: function (value, startIndex, count) {
                    if (value == null) {
                        return null;
                    }

                    startIndex = Math.max(0, startIndex);
                    if (startIndex >= value.length) {
                        return "";
                    }

                    if (count <= 0) {
                        return "";
                    }

                    count = Math.min(count, ((value.length - startIndex) | 0));
                    return value.substr(startIndex, count);
                },
                /**
                 * Returns true if the string is a valid and non-empty
                 base-64 string.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}     value              
                 * @param   {boolean}    allowWhitespace
                 * @return  {boolean}
                 */
                SafeIsValidNonEmptyBase64: function (value, allowWhitespace) {
                    var $t;
                    if (allowWhitespace === void 0) { allowWhitespace = false; }
                    if (allowWhitespace && value != null) {
                        value = value.trim();
                    }

                    if (System.String.isNullOrEmpty(value)) {
                        return false;
                    }

                    if (value.length % 4 !== 0) {
                        return false;
                    }

                    var offsetFirstEquality = -1;
                    var i = -1;
                    $t = Bridge.getEnumerator(value);
                    try {
                        while ($t.moveNext()) {
                            var c = $t.Current;
                            i = (i + 1) | 0;
                            if (c >= 48 && c <= 57 || c >= 65 && c <= 90 || c >= 97 && c <= 122 || c === 43 || c === 47) {
                                continue;
                            }
                            if (c === 61) {
                                offsetFirstEquality = i;
                                break;
                            }

                            // Any other character immediately invalidates the string
                            return false;
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    if (offsetFirstEquality === -1) {
                        return value.length % 4 === 0;
                    }

                    // The '=' character is legal at the end of the base-64 string, and is used
                    // to pad the string to achieve a length that divides 4
                    if (((value.length - offsetFirstEquality) | 0) > 3) {
                        return false;
                    }

                    for (var j = offsetFirstEquality; j < value.length; j = (j + 1) | 0) {
                        if (value.charCodeAt(j) !== 61) {
                            return false;
                        }
                    }

                    return true;
                },
                /**
                 * Returns true if the string is nothing but a '//' comment block,
                 potentially with whitespace characters before each '//' and multiple
                 lines. If the string is empty or null we return true as well.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}     cmd
                 * @return  {boolean}
                 */
                SafeIsDoubleSlashCommentOrEmptyBlock: function (cmd) {
                    if (System.String.isNullOrWhiteSpace(cmd)) {
                        return true;
                    }

                    var slash = false;
                    var i = 0;
                    var c;
                    while (true) {
                        if (i >= cmd.length) {
                            // If slash, it means we saw a solitary slash and then the string
                            // ended, so this counts as "non-comment"
                            return !slash;
                        }

                        c = cmd.charCodeAt(i);

                        if (c === 47) {
                            if (!slash) {
                                slash = true;
                            } else {
                                // Move forward until CR or LF or end-of-string
                                while (((i = (i + 1) | 0)) < cmd.length) {
                                    c = cmd.charCodeAt(i);
                                    if (c === 13 || c === 10) {
                                        break;
                                    }
                                }
                                slash = false;
                            }
                            i = (i + 1) | 0;
                            continue;
                        }

                        if (slash) {
                            // We have a solitary slash, so there's *some* body
                            return false;
                        }
                        if (System.Char.isWhiteSpace(String.fromCharCode(c)) || c === 13 || c === 10) {
                            i = (i + 1) | 0;
                            continue;
                        }
                        return false;
                    }
                },
                /**
                 * Removes initial newlines, terminating newlines, and all mid-newlines that
                 indicate empty lines.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    text
                 * @return  {string}
                 */
                SafeFastTrimEmptyLines: function (text) {
                    if (System.String.isNullOrEmpty(text)) {
                        return text;
                    }

                    if (System.String.indexOf(text, "\n") === -1) {
                        // Fast path
                        return text;
                    }

                    var sb = Kusto.Cloud.Platform.Utils.UtilsStringBuilderCache.Acquire(Math.min(text.length, Kusto.Cloud.Platform.Utils.MemoryConstants.LargeObjectHeapChars));

                    var hasPrecedingText = false;
                    var i = 0;
                    while (i < text.length) {
                        var nextNewLine = System.String.indexOf(text, "\n", i);
                        if (nextNewLine === -1) {
                            // No newlines until the end of the string.
                            // Append the text (possibly with a preceding newline
                            // if there's previous text) and quit.
                            if (hasPrecedingText) {
                                sb.appendLine();
                            }
                            sb.append(text, i, ((text.length - i) | 0));
                            break;
                        }

                        if (nextNewLine !== i) {
                            if (hasPrecedingText) {
                                sb.appendLine();
                            }
                            sb.append(text, i, ((nextNewLine - i) | 0));
                            i = (nextNewLine + "\n".length) | 0;
                            hasPrecedingText = true;
                            continue;
                        }

                        // We're at an immediate newline. Skip it.
                        i = (i + "\n".length) | 0;
                    }

                    return Kusto.Cloud.Platform.Utils.UtilsStringBuilderCache.GetStringAndRelease(sb);
                },
                /**
                 * Removes all leading and trailing white-space characters from the given string
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    text
                 * @return  {string}
                 */
                SafeFastTrim: function (text) {
                    if (System.String.isNullOrEmpty(text)) {
                        return text;
                    }
                    return text.trim();
                },
                /**
                 * Trims a prefix from a string
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}                     text          The text to trim.
                 * @param   {string}                     prefix        Prefix to remove.
                 * @param   {System.StringComparison}    comparison    How to compare strings.
                 * @return  {string}                                   The trimmed string.
                 */
                TrimStart$1: function (text, prefix, comparison) {
                    if (comparison === void 0) { comparison = 4; }
                    if (System.String.startsWith(text, prefix, comparison)) {
                        return text.substr(prefix.length, ((text.length - prefix.length) | 0));
                    }
                    return text;
                },
                /**
                 * Trims a character from the beginning of a string.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}     text       The text to trim.
                 * @param   {number}     c          The character to remove from the beginning of <b>c</b>
                 if it exists.
                 * @param   {boolean}    justOne    If true, remove just one instance of the character.
                 * @return  {string}                The trimmed string.
                 */
                TrimStart: function (text, c, justOne) {
                    if (System.String.isNullOrEmpty(text)) {
                        return text;
                    }

                    if (justOne) {
                        return text.charCodeAt(0) === c ? text.substr(1) : text;
                    }

                    if (text.charCodeAt(0) !== c) {
                        return text;
                    }

                    var i = 0;
                    while (i < text.length && text.charCodeAt(i) === c) {
                        i = (i + 1) | 0;
                    }
                    if (i === text.length) {
                        return "";
                    }
                    return text.substr(i);
                },
                /**
                 * Trims a suffix from a string
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}                     text          The text to trim.
                 * @param   {string}                     suffix        Suffix to remove
                 * @param   {System.StringComparison}    comparison
                 * @return  {string}
                 */
                TrimEnd: function (text, suffix, comparison) {
                    if (comparison === void 0) { comparison = 4; }
                    if (System.String.endsWith(text, suffix, comparison)) {
                        return text.substr(0, ((text.length - suffix.length) | 0));
                    }
                    return text;
                },
                /**
                 * Removes a single pair of balanced [ and ] characters
                 that start and end the string, if they exist.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    text
                 * @return  {string}
                 */
                TrimBalancedSquareBrackets: function (text) {
                    if (text != null && text.length >= 2 && text.charCodeAt(0) === 91 && text.charCodeAt(((text.length - 1) | 0)) === 93) {
                        text = text.substr(1, ((text.length - 2) | 0));
                    }

                    return text;
                },
                /**
                 * Removes a single pair of balanced ( and ) characters
                 that start and end the string, if they exist.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    text
                 * @return  {string}
                 */
                TrimBalancedRoundBrackets: function (text) {
                    return Kusto.Cloud.Platform.Utils.ExtendedString.TrimBalanced(text, "(", ")");
                },
                /**
                 * Removes a single pair of balanced { and } characters
                 that start and end the string, if they exist.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    text
                 * @return  {string}
                 */
                TrimBalancedCurlyBrackets: function (text) {
                    return Kusto.Cloud.Platform.Utils.ExtendedString.TrimBalanced(text, "{", "}");
                },
                /**
                 * Removes a single pair of balanced [ and ] characters
                 that start and end the string, if they exist.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}                     text          
                 * @param   {string}                     openning      
                 * @param   {string}                     closing       
                 * @param   {System.StringComparison}    comparison
                 * @return  {string}
                 */
                TrimBalanced: function (text, openning, closing, comparison) {
                    if (comparison === void 0) { comparison = 4; }
                    if (text != null && System.String.startsWith(text, openning, comparison) && System.String.endsWith(text, closing, comparison)) {
                        text = text.substr(openning.length, ((text.length - (((openning.length + closing.length) | 0))) | 0));
                    }

                    return text;
                },
                /**
                 * Trims a string from single (') or double (") quotes surrounding it,
                 as long as they are balanced.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    text
                 * @return  {string}
                 */
                TrimBalancedSingleAndDoubleQuotes: function (text) {
                    if (System.String.isNullOrEmpty(text)) {
                        return text;
                    }

                    var start = 0;
                    var end = (text.length - 1) | 0;

                    while (start < end) {
                        if (text.charCodeAt(start) === 34 && text.charCodeAt(end) === 34) {
                            start = (start + 1) | 0;
                            end = (end - 1) | 0;
                            continue;
                        }
                        if (text.charCodeAt(start) === 39 && text.charCodeAt(end) === 39) {
                            start = (start + 1) | 0;
                            end = (end - 1) | 0;
                            continue;
                        }

                        break;
                    }

                    if (start > end) {
                        return "";
                    }

                    // TODO: Check if start == 0, and return text

                    return text.substr(start, ((((end - start) | 0) + 1) | 0));
                },
                /**
                 * Given a string, breaks the string into lines
                 and determines the length of the longest line.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    text
                 * @return  {number}
                 */
                DetermineMaxLineWidth: function (text) {
                    if (System.String.isNullOrEmpty(text)) {
                        return 0;
                    }

                    var ret = 0;
                    var currentPos = 0;
                    for (var i = 0; i < text.length; i = (i + 1) | 0) {
                        var c = text.charCodeAt(i);
                        if (c === 13 || c === 10) {
                            ret = Math.max(ret, ((i - currentPos) | 0));
                            currentPos = (i + 1) | 0;
                        } else if (i === ((text.length - 1) | 0)) {
                            ret = Math.max(ret, ((((i - currentPos) | 0) + 1) | 0));
                        }
                    }
                    return ret;
                },
                /**
                 * Counts the number of occurrences of the character <b />
                 within the string <b />. Does not fail on null  values.
                 Count stops at <b /> (if negative, counting stops
                 when the string is exhausted).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}          value       
                 * @param   {number}          c           
                 * @param   {System.Int64}    maxCount
                 * @return  {System.Int64}
                 */
                SafeCountCharacterUpTo: function (value, c, maxCount) {
                    if (value == null) {
                        return System.Int64(-1);
                    }

                    var ret = 0;
                    for (var i = 0; i < value.length; i = (i + 1) | 0) {
                        if (value.charCodeAt(i) === c) {
                            ret = (ret + 1) | 0;
                        }

                        if (System.Int64(ret).equals(maxCount)) {
                            break;
                        }
                    }
                    return System.Int64(ret);
                },
                /**
                 * Counts the number of occurrences of any of these characters <b />
                 within the string <b />. Does not fail on null  values.
                 Count stops at <b /> (if negative, counting stops
                 when the string is exhausted).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}            value       
                 * @param   {Array.<number>}    chars       
                 * @param   {System.Int64}      maxCount
                 * @return  {System.Int64}
                 */
                SafeCountAnyCharacterUpTo: function (value, chars, maxCount) {
                    if (value == null) {
                        return System.Int64(-1);
                    }

                    var ret = 0;
                    for (var i = 0; i < value.length; i = (i + 1) | 0) {
                        if (System.Array.contains(chars, value.charCodeAt(i), System.Char)) {
                            ret = (ret + 1) | 0;
                        }

                        if (System.Int64(ret).equals(maxCount)) {
                            break;
                        }
                    }
                    return System.Int64(ret);
                },
                
                FindFirstNonWhitespaceCharacter: function (value, index) {
                    if (index === void 0) { index = 0; }
                    if (value == null) {
                        return -1;
                    }

                    while (true) {
                        if (index >= value.length) {
                            return -1;
                        }

                        if (!System.Char.isWhiteSpace(String.fromCharCode(value.charCodeAt(index)))) {
                            return index;
                        }

                        index = (index + 1) | 0;
                    }
                },
                /**
                 * Returns the index of the first character that is not equal between two strings.
                 Returns -1 if they are identical.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    lhs    
                 * @param   {string}    rhs
                 * @return  {number}
                 */
                FirstFirstUnequalCharacter: function (lhs, rhs) {
                    if (Bridge.referenceEquals(lhs, rhs)) {
                        return -1;
                    }

                    // Since the two are not equal, we can safely assume that if one has no characters,
                    // it's the first character that is different
                    if (lhs == null || rhs == null || lhs.length === 0 || rhs.length === 0) {
                        return 0;
                    }

                    var i = 0;
                    for (; i < lhs.length && i < rhs.length; i = (i + 1) | 0) {
                        if (lhs.charCodeAt(i) !== rhs.charCodeAt(i)) {
                            return i;
                        }
                    }

                    return i;
                },
                /**
                 * Counts how many times a substring appears in the string, non-overlapping.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}                     text          The string we suspect may have one or more appearrences of the substring.
                 * @param   {string}                     substring     The substring to find and count. Non-null and non-empty.
                 * @param   {System.StringComparison}    comparison    How to compare strings.
                 * @return  {number}                                   The number of times <b /> appears in <b />
                 (non-overlapping: "aa" appears exactly twice in "aaaa", not three times).
                 */
                CountNonOverlappingSubstrings: function (text, substring, comparison) {
                    if (comparison === void 0) { comparison = 4; }
                    if (System.String.isNullOrEmpty(text) || System.String.isNullOrEmpty(substring)) {
                        return 0;
                    }

                    var count = 0;
                    var i = 0;
                    while (i < text.length) {
                        i = System.String.indexOf(text, substring, i, null, comparison);
                        if (i < 0) {
                            return count;
                        }

                        count = (count + 1) | 0;
                        i = (i + substring.length) | 0;
                    }

                    return count;
                },
                /**
                 * Returns true if <b />'s first character
                 is <b />, or false otherwise. Tolerant
                 to null strings.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}     what    
                 * @param   {number}     c
                 * @return  {boolean}
                 */
                SafeStartsWith: function (what, c) {
                    if (System.String.isNullOrEmpty(what)) {
                        return false;
                    }

                    return what.charCodeAt(0) === c;
                },
                /**
                 * Returns true if <b /> starts with <b />, or false otherwise. Tolerant
                 to null strings.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}                     what          
                 * @param   {string}                     value         
                 * @param   {System.StringComparison}    comparison
                 * @return  {boolean}
                 */
                SafeStartsWith$1: function (what, value, comparison) {
                    if (comparison === void 0) { comparison = 4; }
                    if (System.String.isNullOrEmpty(what)) {
                        return false;
                    }

                    return System.String.startsWith(what, value, comparison);
                },
                EqualsOrdinalIgnoreCase: function (me, other) {
                    return System.String.equals(me, other, 5);
                },
                IsEmpty: function (me) {
                    return Bridge.referenceEquals(me, "");
                },
                IsNotEmpty: function (me) {
                    return !Bridge.referenceEquals(me, "");
                },
                IsNullOrEmpty: function (me) {
                    return System.String.isNullOrEmpty(me);
                },
                IsNotNullOrEmpty: function (me) {
                    return !System.String.isNullOrEmpty(me);
                },
                /**
                 * Similar to {@link }, but (1) gets the number of characters
                 in the "tail" to return rather than the count of characters to skip, and (2) is safe
                 and does not throw.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    what      
                 * @param   {number}    length
                 * @return  {string}
                 */
                SafeGetTail: function (what, length) {
                    if (what == null) {
                        return null;
                    }

                    if (length <= 0) {
                        return "";
                    }

                    if (length >= what.length) {
                        return what;
                    }

                    return what.substr(((what.length - length) | 0));
                },
                /**
                 * Returns the index of the string <b /> inside string <b />
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    what     
                 * @param   {number}    value
                 * @return  {number}
                 */
                SafeIndexOf: function (what, value) {
                    if (System.String.isNullOrEmpty(what)) {
                        return -1;
                    }

                    return System.String.indexOf(what, String.fromCharCode(value));
                },
                /**
                 * Returns the index of the string <b /> inside string <b />
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}                     what              
                 * @param   {string}                     value             
                 * @param   {number}                     startIndex        
                 * @param   {System.StringComparison}    comparisonType
                 * @return  {number}
                 */
                SafeIndexOf$2: function (what, value, startIndex, comparisonType) {
                    if (System.String.isNullOrEmpty(what) || System.String.isNullOrEmpty(value) || startIndex > what.length) {
                        return -1;
                    }

                    return System.String.indexOf(what, value, startIndex, null, comparisonType);
                },
                /**
                 * Returns the index of the string <b /> inside string <b />
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}                     what              
                 * @param   {string}                     value             
                 * @param   {number}                     startIndex        
                 * @param   {number}                     count             
                 * @param   {System.StringComparison}    comparisonType
                 * @return  {number}
                 */
                SafeIndexOf$1: function (what, value, startIndex, count, comparisonType) {
                    if (System.String.isNullOrEmpty(what) || System.String.isNullOrEmpty(value) || ((startIndex + value.length) | 0) > what.length) {
                        return -1;
                    }

                    return System.String.indexOf(what, value, startIndex, count, comparisonType);
                },
                /**
                 * Returns the beginning of the string <b /> up to
                 but not including the first appearance of <b />.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}                     what          
                 * @param   {string}                     delimiter     
                 * @param   {System.StringComparison}    comparison
                 * @return  {string}
                 */
                SplitFirst$2: function (what, delimiter, comparison) {
                    if (comparison === void 0) { comparison = 4; }
                    if (what == null || System.String.isNullOrEmpty(delimiter)) {
                        return null;
                    }

                    var delimiterIndex = System.String.indexOf(what, delimiter, 0, null, comparison);
                    if (delimiterIndex < 0) {
                        return what;
                    }

                    var first = what.substr(0, delimiterIndex);
                    return first;
                },
                /**
                 * Returns the beginning of the string <b /> up to
                 but not including the first appearance of <b />.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    what         
                 * @param   {number}    delimiter
                 * @return  {string}
                 */
                SplitFirst: function (what, delimiter) {
                    if (what == null) {
                        return null;
                    }

                    var delimiterIndex = System.String.indexOf(what, String.fromCharCode(delimiter));
                    if (delimiterIndex < 0) {
                        return what;
                    }

                    var first = what.substr(0, delimiterIndex);
                    return first;
                },
                /**
                 * Returns the beginning of the string <b /> up to
                 but not including the first appearance of any of the chars in <b />.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}            what          
                 * @param   {Array.<number>}    delimiters
                 * @return  {string}
                 */
                SplitFirst$1: function (what, delimiters) {
                    if (what == null || delimiters == null || delimiters.length === 0) {
                        return null;
                    }

                    var delimitersIndex = System.String.indexOfAny(what, delimiters);
                    if (delimitersIndex < 0) {
                        return what;
                    }

                    var first = what.substr(0, delimitersIndex);
                    return first;
                },
                /**
                 * Rotates right the string (right shift, where the last character becomes the first).
                 (It might create an invalid string, for Some unicode characters that don't fit 16 bits and are represented by two UTF16 characters).
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    s
                 * @return  {string}
                 */
                Rotate: function (s) {
                    if (System.String.isNullOrEmpty(s) || s.length < 2) {
                        return s;
                    }

                    return (s.substr(1, ((s.length - 1) | 0)) || "") + (s.substr(0, 1) || "");
                },
                /**
                 * Changes the new line symbols '\n' and '\r' to Envorinment.NewLine
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}    text
                 * @return  {string}
                 */
                AdjustNewLineDelimiter: function (text) {
                    var resultBuilder = new System.Text.StringBuilder();
                    for (var i = 0; i < text.length; i = (i + 1) | 0) {
                        var c = text.charCodeAt(i);
                        switch (c) {
                            case 13: 
                                var isNextChar_slashN = i < ((text.length - 1) | 0) && text.charCodeAt(((i + 1) | 0)) === 10;
                                if (isNextChar_slashN) {
                                    i = (i + 1) | 0;
                                }
                                resultBuilder.append("\n");
                                break;
                            case 10: 
                                resultBuilder.append("\n");
                                break;
                            default: 
                                resultBuilder.append(String.fromCharCode(c));
                                break;
                        }
                    }

                    return resultBuilder.toString();
                },
                ReplaceInvisibleWhitespaces: function (value) {
                    var c_space = " ";
                    return Kusto.Cloud.Platform.Utils.ExtendedString.Replace$1(value, function (c, replacement) {
                        replacement.v = null;
                        if (Kusto.Cloud.Platform.Utils.ExtendedString.s_alternativeSpaceChars.contains(c)) {
                            replacement.v = c_space;
                        } else if (Kusto.Cloud.Platform.Utils.ExtendedString.s_alternativeNewlineChars.contains(c)) {
                            replacement.v = "\n";
                        }

                        return replacement.v != null;
                    });
                },
                /**
                 * Checks if the string has non-Ascii quotes
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.ExtendedString
                 * @memberof Kusto.Cloud.Platform.Utils.ExtendedString
                 * @param   {string}     value
                 * @return  {boolean}
                 */
                HasNonAsciiQuotes: function (value) {
                    if (System.String.isNullOrWhiteSpace(value)) {
                        return false;
                    }

                    return System.Linq.Enumerable.from(value, System.Char).any(Bridge.fn.cacheBind(Kusto.Cloud.Platform.Utils.ExtendedString.s_alternativeQutoes, Kusto.Cloud.Platform.Utils.ExtendedString.s_alternativeQutoes.contains));
                },
                ReplaceNonAsciiQuotes: function (value) {
                    var c_quotes = "\"";
                    return Kusto.Cloud.Platform.Utils.ExtendedString.Replace$1(value, function (c, replacement) {
                        replacement.v = c_quotes;
                        return Kusto.Cloud.Platform.Utils.ExtendedString.s_alternativeQutoes.contains(c);
                    });
                },
                Replace: function (value, replacements) {
                    if (!Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$1(System.Collections.Generic.KeyValuePair$2(System.Char,System.String), replacements)) {
                        return value;
                    }

                    return Kusto.Cloud.Platform.Utils.ExtendedString.Replace$1(value, Bridge.fn.cacheBind(replacements, replacements.System$Collections$Generic$IDictionary$2$System$Char$System$String$tryGetValue));
                },
                Replace$1: function (value, shouldReplace) {
                    if (System.String.isNullOrEmpty(value)) {
                        return value;
                    }

                    var sb = null;
                    for (var i = 0; i < value.length; i = (i + 1) | 0) {
                        var c = value.charCodeAt(i);
                        var replacement = { };
                        if (shouldReplace(c, replacement)) {
                            sb = Kusto.Cloud.Platform.Utils.ExtendedStringBuilder.InitializeStringBuildIfNeeded(value, sb, i);
                            sb.append(replacement.v);
                        } else if (sb != null) {
                            sb.append(String.fromCharCode(c));
                        }
                    }

                    if (sb == null) {
                        // No replacement was needed
                        return value;
                    }

                    return sb.toString();
                }
            }
        }
    });

    Bridge.ns("Kusto.Cloud.Platform.Utils.ExtendedString", $asm.$);

    Bridge.apply($asm.$.Kusto.Cloud.Platform.Utils.ExtendedString, {
        f1: function (obj) {
            var $t;
            return obj != null ? (($t = Bridge.toString(obj), $t != null ? $t : "[ToStringNull]")) : "[null]";
        },
        f2: function (_o1) {
            _o1.add(160);
            _o1.add(5760);
            _o1.add(6158);
            _o1.add(8192);
            _o1.add(8193);
            _o1.add(8194);
            _o1.add(8195);
            _o1.add(8196);
            _o1.add(8197);
            _o1.add(8198);
            _o1.add(8199);
            _o1.add(8200);
            _o1.add(8201);
            _o1.add(8202);
            _o1.add(8203);
            _o1.add(8239);
            _o1.add(8287);
            _o1.add(12288);
            _o1.add(65279);
            return _o1;
        },
        f3: function (_o2) {
            _o2.add(11);
            return _o2;
        },
        f4: function (_o3) {
            _o3.add(171);
            _o3.add(187);
            _o3.add(8216);
            _o3.add(8217);
            _o3.add(8218);
            _o3.add(8220);
            _o3.add(8221);
            _o3.add(8222);
            _o3.add(8249);
            _o3.add(8250);
            return _o3;
        }
    });

    Bridge.define("Kusto.Cloud.Platform.Utils.ExtendedStringBuilder", {
        statics: {
            methods: {
                InitializeStringBuildIfNeeded: function (value, sb, i) {
                    if (sb != null) {
                        return sb;
                    }

                    sb = new System.Text.StringBuilder("", value.length);
                    // Append all chars before this one
                    if (i > 0) {
                        sb.append(value.substr(0, i));
                    }
                    return sb;
                }
            }
        }
    });

    Bridge.define("Kusto.Cloud.Platform.Utils.MemoryConstants", {
        statics: {
            fields: {
                _1KB: 0,
                _2KB: 0,
                _4KB: 0,
                _8KB: 0,
                _16KB: 0,
                _32KB: 0,
                _64KB: 0,
                _128KB: 0,
                _256KB: 0,
                _512KB: 0,
                _1MB: 0,
                _2MB: 0,
                _32MB: 0,
                _64MB: 0,
                _512MB: 0,
                _1GB: 0,
                _1TB: System.Int64(0),
                _1PB: System.Int64(0),
                LargeObjectHeap: 0,
                LargeObjectHeapChars: 0,
                /**
                 * When using UTF8Encoding, StreamWriter multiplies the bufferSize supplied during construction by 3. Use this constant to avoid LOH
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Cloud.Platform.Utils.MemoryConstants
                 * @constant
                 * @default 24576
                 * @type number
                 */
                LargeObjectHeapCharsStreamReaderWriter: 0
            },
            ctors: {
                init: function () {
                    this._1KB = 1024;
                    this._2KB = 2048;
                    this._4KB = 4096;
                    this._8KB = 8192;
                    this._16KB = 16384;
                    this._32KB = 32768;
                    this._64KB = 65536;
                    this._128KB = 131072;
                    this._256KB = 262144;
                    this._512KB = 524288;
                    this._1MB = 1048576;
                    this._2MB = 2097152;
                    this._32MB = 33554432;
                    this._64MB = 67108864;
                    this._512MB = 536870912;
                    this._1GB = 1073741824;
                    this._1TB = System.Int64([0,256]);
                    this._1PB = System.Int64([0,262144]);
                    this.LargeObjectHeap = 65536;
                    this.LargeObjectHeapChars = 32768;
                    this.LargeObjectHeapCharsStreamReaderWriter = 24576;
                }
            }
        }
    });

    /**
     * A helper for creating a per-thread ([ThreadStatic]) StringBuilder cache.
     This code is stolen from the .NET Framework source code. The code requires
     the caller to declare a [ThreadStatic] field member of type StringBuilder,
     and provide a reference to that field with each operation. See
     {@link } for an example.
     Note that it's not advisable to share such objects if their lifetime
     overlaps (which is why {@link } is made
     internal -- to prevent people from making mistakes).
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Cloud.Platform.Utils.StringBuilderCache
     */
    Bridge.define("Kusto.Cloud.Platform.Utils.StringBuilderCache", {
        statics: {
            fields: {
                MAX_BUILDER_SIZE: 0,
                DEFAULT_CAPACITY: 0
            },
            ctors: {
                init: function () {
                    this.MAX_BUILDER_SIZE = 32768;
                    this.DEFAULT_CAPACITY = 16;
                }
            },
            methods: {
                /**
                 * Given a [ThreadStatic] field, returns a "clean" instance of {@link }.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.StringBuilderCache
                 * @memberof Kusto.Cloud.Platform.Utils.StringBuilderCache
                 * @param   {System.Text.StringBuilder}    threadStaticStringBuilder    [ThreadStatic] static System.Text.StringBuilder s_myStringBuilderCache
                 * @param   {number}                       capacity                     Capacity to ensure the returned object holds.
                 * @param   {number}                       maxBuilderSize               The maximum size to allow the string builder to grow to.
                 * @return  {System.Text.StringBuilder}
                 */
                Acquire: function (threadStaticStringBuilder, capacity, maxBuilderSize) {
                    if (capacity === void 0) { capacity = 16; }
                    if (maxBuilderSize === void 0) { maxBuilderSize = 32768; }
                    if (capacity <= maxBuilderSize) {
                        var sb = threadStaticStringBuilder.v;
                        if (sb != null) {
                            // Avoid stringbuilder block fragmentation by getting a new StringBuilder
                            // when the requested size is larger than the current capacity
                            if (capacity <= sb.getCapacity()) {
                                threadStaticStringBuilder.v = null;
                                sb.clear();
                                return sb;
                            }
                        }
                    }
                    return new System.Text.StringBuilder("", capacity);
                },
                /**
                 * Given a [ThreadStatic] field, returns an instance of {@link } with the given initial value.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.StringBuilderCache
                 * @memberof Kusto.Cloud.Platform.Utils.StringBuilderCache
                 * @param   {System.Text.StringBuilder}    threadStaticStringBuilder    [ThreadStatic] static System.Text.StringBuilder s_myStringBuilderCache
                 * @param   {string}                       value                        Initial value to assign the {@link } being returned.
                 * @return  {System.Text.StringBuilder}
                 */
                Acquire$1: function (threadStaticStringBuilder, value) {
                    var sb = Kusto.Cloud.Platform.Utils.StringBuilderCache.Acquire(threadStaticStringBuilder, Math.max(value.length, Kusto.Cloud.Platform.Utils.StringBuilderCache.DEFAULT_CAPACITY));
                    sb.append(value);
                    return sb;
                },
                /**
                 * Given a [ThreadStatic] field and an existing {@link } that was acquired from it,
                 release the acquired instance to make it available in the future to other functions.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.StringBuilderCache
                 * @memberof Kusto.Cloud.Platform.Utils.StringBuilderCache
                 * @param   {System.Text.StringBuilder}    threadStaticStringBuilder    [ThreadStatic] static System.Text.StringBuilder s_myStringBuilderCache
                 * @param   {System.Text.StringBuilder}    sb                           
                 * @param   {number}                       maxBuilderSize
                 * @return  {void}
                 */
                Release: function (threadStaticStringBuilder, sb, maxBuilderSize) {
                    if (maxBuilderSize === void 0) { maxBuilderSize = 32768; }
                    if (sb.getCapacity() <= maxBuilderSize) {
                        threadStaticStringBuilder.v = sb;
                    }
                },
                /**
                 * Given a [ThreadStatic] field and an existing {@link } that was acquired from it,
                 release the acquired instance to make it available in the future to other functions.
                 Returns the string held in the returned <b />.
                 *
                 * @static
                 * @public
                 * @this Kusto.Cloud.Platform.Utils.StringBuilderCache
                 * @memberof Kusto.Cloud.Platform.Utils.StringBuilderCache
                 * @param   {System.Text.StringBuilder}    threadStaticStringBuilder    [ThreadStatic] static System.Text.StringBuilder s_myStringBuilderCache
                 * @param   {System.Text.StringBuilder}    sb                           
                 * @param   {number}                       maxBuilderSize
                 * @return  {string}
                 */
                GetStringAndRelease: function (threadStaticStringBuilder, sb, maxBuilderSize) {
                    if (maxBuilderSize === void 0) { maxBuilderSize = 32768; }
                    var result = sb.toString();
                    Kusto.Cloud.Platform.Utils.StringBuilderCache.Release(threadStaticStringBuilder, sb, maxBuilderSize);
                    return result;
                },
                GetStringAndClear: function (threadStaticStringBuilder, sb) {
                    var result = sb.toString();
                    sb.clear();
                    return result;
                }
            }
        }
    });

    Bridge.define("Kusto.Cloud.Platform.Utils.TimeSpanConstants", {
        statics: {
            fields: {
                _2sec: null,
                _10sec: null,
                _20sec: null,
                _1min: null,
                _2min: null,
                _3min: null,
                _4min: null,
                _5min: null,
                _10min: null,
                _59min: null,
                _60min: null,
                _1hour: null,
                _2hours: null,
                _3hours: null,
                _4hours: null,
                _5hours: null,
                _6hours: null,
                _10hours: null
            },
            ctors: {
                init: function () {
                    this._2sec = new System.TimeSpan();
                    this._10sec = new System.TimeSpan();
                    this._20sec = new System.TimeSpan();
                    this._1min = new System.TimeSpan();
                    this._2min = new System.TimeSpan();
                    this._3min = new System.TimeSpan();
                    this._4min = new System.TimeSpan();
                    this._5min = new System.TimeSpan();
                    this._10min = new System.TimeSpan();
                    this._59min = new System.TimeSpan();
                    this._60min = new System.TimeSpan();
                    this._1hour = new System.TimeSpan();
                    this._2hours = new System.TimeSpan();
                    this._3hours = new System.TimeSpan();
                    this._4hours = new System.TimeSpan();
                    this._5hours = new System.TimeSpan();
                    this._6hours = new System.TimeSpan();
                    this._10hours = new System.TimeSpan();
                    this._2sec = System.TimeSpan.fromSeconds(2);
                    this._10sec = System.TimeSpan.fromSeconds(10);
                    this._20sec = System.TimeSpan.fromSeconds(20);
                    this._1min = System.TimeSpan.fromMinutes(1);
                    this._2min = System.TimeSpan.fromMinutes(2);
                    this._3min = System.TimeSpan.fromMinutes(3);
                    this._4min = System.TimeSpan.fromMinutes(4);
                    this._5min = System.TimeSpan.fromMinutes(5);
                    this._10min = System.TimeSpan.fromMinutes(10);
                    this._59min = System.TimeSpan.fromMinutes(59);
                    this._60min = System.TimeSpan.fromMinutes(60);
                    this._1hour = System.TimeSpan.fromHours(1);
                    this._2hours = System.TimeSpan.fromHours(2);
                    this._3hours = System.TimeSpan.fromHours(3);
                    this._4hours = System.TimeSpan.fromHours(4);
                    this._5hours = System.TimeSpan.fromHours(5);
                    this._6hours = System.TimeSpan.fromHours(6);
                    this._10hours = System.TimeSpan.fromHours(10);
                }
            }
        }
    });

    /**
     * A per-thread cache of up to one {@link } object.
     This code is stolen from the .NET Framework source code. It is explicitly
     internal to Kusto.Cloud.Platform -- so not reuse the cache in other assemblies.
     *
     * @static
     * @abstract
     * @class Kusto.Cloud.Platform.Utils.UtilsStringBuilderCache
     */
    Bridge.define("Kusto.Cloud.Platform.Utils.UtilsStringBuilderCache", {
        statics: {
            fields: {
                MAX_BUILDER_SIZE: 0,
                DEFAULT_CAPACITY: 0,
                t_cachedInstance: null
            },
            ctors: {
                init: function () {
                    this.MAX_BUILDER_SIZE = 24576;
                    this.DEFAULT_CAPACITY = 16;
                }
            },
            methods: {
                Acquire: function (capacity) {
                    if (capacity === void 0) { capacity = 16; }
                    return Kusto.Cloud.Platform.Utils.StringBuilderCache.Acquire(Bridge.ref(Kusto.Cloud.Platform.Utils.UtilsStringBuilderCache, "t_cachedInstance"), capacity);
                },
                Acquire$1: function (value) {
                    return Kusto.Cloud.Platform.Utils.StringBuilderCache.Acquire$1(Bridge.ref(Kusto.Cloud.Platform.Utils.UtilsStringBuilderCache, "t_cachedInstance"), value);
                },
                Release: function (sb) {
                    Kusto.Cloud.Platform.Utils.StringBuilderCache.Release(Bridge.ref(Kusto.Cloud.Platform.Utils.UtilsStringBuilderCache, "t_cachedInstance"), sb);
                },
                GetStringAndRelease: function (sb) {
                    return Kusto.Cloud.Platform.Utils.StringBuilderCache.GetStringAndRelease(Bridge.ref(Kusto.Cloud.Platform.Utils.UtilsStringBuilderCache, "t_cachedInstance"), sb);
                },
                GetStringAndClear: function (sb) {
                    return Kusto.Cloud.Platform.Utils.StringBuilderCache.GetStringAndClear(Bridge.ref(Kusto.Cloud.Platform.Utils.UtilsStringBuilderCache, "t_cachedInstance"), sb);
                }
            }
        }
    });

    Bridge.define("Kusto.Data.Common.CslQueryParser", {
        statics: {
            methods: {
                PrettifyQuery: function (query, appendText) {
                    var parser = new Kusto.Data.IntelliSense.CslCommandParser();
                    var commands = parser.Parse(null, query, Kusto.Data.IntelliSense.ParseMode.CommandTokensOnly);
                    if (System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).count() !== 1) {
                        // Skip any prettifying if there is more than one command
                        return query;
                    }

                    // Format query nicely
                    var queryFormatted = System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).first().FormatAsString("\n", 7);
                    if (System.String.isNullOrEmpty(queryFormatted)) {
                        queryFormatted = query;
                    }

                    if (!System.String.isNullOrEmpty(appendText)) {
                        queryFormatted = (queryFormatted || "") + (appendText || "");
                    }
                    return queryFormatted;
                }
            }
        }
    });

    /** @namespace Kusto.Data.Common */

    /**
     * Support for generating CSL (Kusto) queries and control commands
     using code-generation.
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Common.CslSyntaxGenerator
     */
    Bridge.define("Kusto.Data.Common.CslSyntaxGenerator", {
        statics: {
            fields: {
                s_keywords: null
            },
            ctors: {
                init: function () {
                    this.s_keywords = $asm.$.Kusto.Data.Common.CslSyntaxGenerator.f1(new (System.Collections.Generic.HashSet$1(System.String)).ctor());
                }
            },
            methods: {
                NormalizeDatabaseName$1: function (databaseName, forceNormalization) {
                    return Kusto.Data.Common.CslSyntaxGenerator.NormalizeName(databaseName, forceNormalization);
                },
                NormalizeDatabaseName: function (databaseName) {
                    return Kusto.Data.Common.CslSyntaxGenerator.NormalizeName(databaseName, false);
                },
                NormalizeVirutalClusterName: function (virtualClusterName, forceNormalization) {
                    if (forceNormalization === void 0) { forceNormalization = false; }
                    return Kusto.Data.Common.CslSyntaxGenerator.NormalizeName(virtualClusterName, forceNormalization);
                },
                NormalizeFunctionName: function (functionName, forceNormalizeFunctionName) {
                    if (forceNormalizeFunctionName === void 0) { forceNormalizeFunctionName = false; }
                    return Kusto.Data.Common.CslSyntaxGenerator.NormalizeName(functionName, forceNormalizeFunctionName);
                },
                NormalizeTableName: function (tableName, forceNormalizeTableName) {
                    if (forceNormalizeTableName === void 0) { forceNormalizeTableName = false; }
                    return Kusto.Data.Common.CslSyntaxGenerator.NormalizeName(tableName, forceNormalizeTableName);
                },
                NormalizeColumnName: function (columnName, forceNormalizeColumnName) {
                    if (forceNormalizeColumnName === void 0) { forceNormalizeColumnName = false; }

                    return Kusto.Data.Common.CslSyntaxGenerator.NormalizeName(columnName, forceNormalizeColumnName);
                },
                NormalizeRowStoreName: function (rowStoreName) {

                    return Kusto.Data.Common.CslSyntaxGenerator.NormalizeName(rowStoreName, false);
                },
                NormalizeWorkloadGroupName: function (workloadGroupName) {

                    return Kusto.Data.Common.CslSyntaxGenerator.NormalizeName(workloadGroupName, false);
                },
                IsIdentifier: function (token) {
                    if (System.String.isNullOrEmpty(token)) {
                        return false;
                    }

                    if (!System.Char.isLetter(token.charCodeAt(0)) && token.charCodeAt(0) !== 95) {
                        return false;
                    }

                    var ret = System.Linq.Enumerable.from(token, System.Char).all($asm.$.Kusto.Data.Common.CslSyntaxGenerator.f2);
                    return ret;
                },
                IsAscii: function (c) {
                    return (c <= 127);
                },
                NormalizeName: function (name, forceNormalization) {
                    if (System.String.isNullOrEmpty(name)) {
                        return name;
                    }

                    if (!forceNormalization && Kusto.Data.Common.CslSyntaxGenerator.IsIdentifier(name) && !Kusto.Data.Common.CslSyntaxGenerator.IsKeyword(name)) {
                        return name;
                    }

                    if (System.String.startsWith(name, "[", 4)) {
                        return name;
                    }

                    if (!System.String.contains(name,"'")) {
                        return "['" + (name || "") + "']";
                    }

                    return "[\"" + (name || "") + "\"]";
                },
                IsKeyword: function (token) {
                    return Kusto.Data.Common.CslSyntaxGenerator.s_keywords.contains(token);
                }
            }
        }
    });

    Bridge.ns("Kusto.Data.Common.CslSyntaxGenerator", $asm.$);

    Bridge.apply($asm.$.Kusto.Data.Common.CslSyntaxGenerator, {
        f1: function (_o1) {
            _o1.add("access");
            _o1.add("accounts");
            _o1.add("accumulate");
            _o1.add("add");
            _o1.add("adl");
            _o1.add("admin");
            _o1.add("admins");
            _o1.add("aggregations");
            _o1.add("alias");
            _o1.add("all");
            _o1.add("alldatabasesadmins");
            _o1.add("alldatabasesmonitors");
            _o1.add("alldatabasesviewers");
            _o1.add("alter");
            _o1.add("alter-merge");
            _o1.add("and");
            _o1.add("anomalycolumns");
            _o1.add("append");
            _o1.add("artifacts");
            _o1.add("as");
            _o1.add("asc");
            _o1.add("assert-schema");
            _o1.add("async");
            _o1.add("attach");
            _o1.add("autoUpdateSchema");
            _o1.add("auto_delete");
            _o1.add("avro");
            _o1.add("apacheavro");
            _o1.add("axes");
            _o1.add("base");
            _o1.add("based-on");
            _o1.add("basicauth");
            _o1.add("between");
            _o1.add("bin");
            _o1.add("blob");
            _o1.add("block");
            _o1.add("blocked");
            _o1.add("by");
            _o1.add("cache");
            _o1.add("caching");
            _o1.add("cachingpolicy");
            _o1.add("call-tree");
            _o1.add("callout");
            _o1.add("cancel");
            _o1.add("capacity");
            _o1.add("classification");
            _o1.add("clean");
            _o1.add("clear");
            _o1.add("cluster");
            _o1.add("column");
            _o1.add("columns");
            _o1.add("column-docstrings");
            _o1.add("commands");
            _o1.add("commands-and-queries");
            _o1.add("commconcurrency");
            _o1.add("commpools");
            _o1.add("compact");
            _o1.add("completed");
            _o1.add("compressed");
            _o1.add("consume");
            _o1.add("contains");
            _o1.add("continuous-export");
            _o1.add("continuous-exports");
            _o1.add("continuous-job");
            _o1.add("conservative");
            _o1.add("copy");
            _o1.add("corrupted");
            _o1.add("count");
            _o1.add("crash");
            _o1.add("create");
            _o1.add("create-or-alter");
            _o1.add("createdon");
            _o1.add("create-merge");
            _o1.add("csl");
            _o1.add("cslschema");
            _o1.add("csv");
            _o1.add("cursor");
            _o1.add("data");
            _o1.add("database");
            _o1.add("databasecreators");
            _o1.add("databases");
            _o1.add("dataexport");
            _o1.add("data-export");
            _o1.add("dataformat");
            _o1.add("datatable");
            _o1.add("declare");
            _o1.add("decryption-certificate-thumbprint");
            _o1.add("default");
            _o1.add("define");
            _o1.add("delete");
            _o1.add("delta");
            _o1.add("desc");
            _o1.add("detach");
            _o1.add("details");
            _o1.add("diagnostics");
            _o1.add("disable");
            _o1.add("distinct");
            _o1.add("dnshostname");
            _o1.add("docstring");
            _o1.add("drop");
            _o1.add("drop-pretend");
            _o1.add("dup-next-failed-ingest");
            _o1.add("dup-next-ingest");
            _o1.add("earliest");
            _o1.add("echo");
            _o1.add("effective");
            _o1.add("enable");
            _o1.add("encoding");
            _o1.add("encodingpolicy");
            _o1.add("entity");
            _o1.add("entity_group");
            _o1.add("entity_groups");
            _o1.add("ephemeral");
            _o1.add("evaluate");
            _o1.add("except");
            _o1.add("execute");
            _o1.add("export");
            _o1.add("extend");
            _o1.add("extent");
            _o1.add("extentcontainers");
            _o1.add("extents");
            _o1.add("extentsmerge");
            _o1.add("extents-merge");
            _o1.add("extents-partition");
            _o1.add("extent_tags_retention");
            _o1.add("external");
            _o1.add("externaldata");
            _o1.add("external_data");
            _o1.add("fabric");
            _o1.add("fabriccache");
            _o1.add("fabricclocks");
            _o1.add("fabriclocks");
            _o1.add("facet");
            _o1.add("failures");
            _o1.add("featureflags");
            _o1.add("filter");
            _o1.add("find");
            _o1.add("first");
            _o1.add("flags");
            _o1.add("folder");
            _o1.add("fork");
            _o1.add("freshness");
            _o1.add("from");
            _o1.add("function");
            _o1.add("functions");
            _o1.add("generic");
            _o1.add("get");
            _o1.add("getschema");
            _o1.add("granny-asc");
            _o1.add("granny-desc");
            _o1.add("granularity");
            _o1.add("groups");
            _o1.add("harddelete");
            _o1.add("hardretention");
            _o1.add("has");
            _o1.add("has_all");
            _o1.add("has_any");
            _o1.add("hash");
            _o1.add("hidden");
            _o1.add("hot");
            _o1.add("hotdata");
            _o1.add("hotindex");
            _o1.add("hours");
            _o1.add("id");
            _o1.add("if_later_than");
            _o1.add("ifexists");
            _o1.add("ifnotexists");
            _o1.add("in");
            _o1.add("in~");
            _o1.add("info");
            _o1.add("ingest");
            _o1.add("ingest-from-storage");
            _o1.add("ingest-inline");
            _o1.add("ingestion");
            _o1.add("ingestionbatching");
            _o1.add("ingestions");
            _o1.add("ingestiontime");
            _o1.add("ingestors");
            _o1.add("inline");
            _o1.add("internal");
            _o1.add("into");
            _o1.add("invoke");
            _o1.add("journal");
            _o1.add("json");
            _o1.add("jobs");
            _o1.add("key");
            _o1.add("keys");
            _o1.add("kind");
            _o1.add("last");
            _o1.add("latest");
            _o1.add("legend");
            _o1.add("let");
            _o1.add("limit");
            _o1.add("linear");
            _o1.add("list");
            _o1.add("load");
            _o1.add("local");
            _o1.add("location");
            _o1.add("lookback");
            _o1.add("lookup");
            _o1.add("log");
            _o1.add("macro-expand");
            _o1.add("maintenance_mode");
            _o1.add("make-series");
            _o1.add("management");
            _o1.add("map");
            _o1.add("mapping");
            _o1.add("mappings");
            _o1.add("managed_identity");
            _o1.add("materialize");
            _o1.add("materialized-view");
            _o1.add("materialized-views");
            _o1.add("materialized-view-combine");
            _o1.add("maxRecords");
            _o1.add("mdm");
            _o1.add("memory");
            _o1.add("mempools");
            _o1.add("merge");
            _o1.add("metadata");
            _o1.add("method");
            _o1.add("missing");
            _o1.add("monitoring");
            _o1.add("monitors");
            _o1.add("move");
            _o1.add("multidatabaseadmins");
            _o1.add("mv-apply");
            _o1.add("mv-expand");
            _o1.add("mvapply");
            _o1.add("mvexpand");
            _o1.add("nan");
            _o1.add("node");
            _o1.add("nodes");
            _o1.add("none");
            _o1.add("!between");
            _o1.add("!in");
            _o1.add("!in~");
            _o1.add("null");
            _o1.add("nulls");
            _o1.add("of");
            _o1.add("on");
            _o1.add("older");
            _o1.add("operation");
            _o1.add("operations");
            _o1.add("ops");
            _o1.add("optional");
            _o1.add("or");
            _o1.add("orc");
            _o1.add("order");
            _o1.add("others");
            _o1.add("output");
            _o1.add("over");
            _o1.add("pack");
            _o1.add("panels");
            _o1.add("parse");
            _o1.add("parse-where");
            _o1.add("parquet");
            _o1.add("partition");
            _o1.add("__partitionby");
            _o1.add("partitioning");
            _o1.add("password");
            _o1.add("patch");
            _o1.add("pathformat");
            _o1.add("pattern");
            _o1.add("pending");
            _o1.add("period");
            _o1.add("periodic-storage-artifacts-cleanup");
            _o1.add("persist");
            _o1.add("plugin");
            _o1.add("plugins");
            _o1.add("policies");
            _o1.add("policy");
            _o1.add("pretend");
            _o1.add("prettyname");
            _o1.add("principal");
            _o1.add("principals");
            _o1.add("print");
            _o1.add("project");
            _o1.add("project-away");
            _o1.add("project-keep");
            _o1.add("project-rename");
            _o1.add("project-reorder");
            _o1.add("project-smart");
            _o1.add("purge");
            _o1.add("purge-cleanup");
            _o1.add("purge-storage-artifacts-cleanup");
            _o1.add("queries");
            _o1.add("query");
            _o1.add("query_parameters");
            _o1.add("queryexecution");
            _o1.add("queryplan");
            _o1.add("query_weak_consistency");
            _o1.add("query_results");
            _o1.add("range");
            _o1.add("readonly");
            _o1.add("readwrite");
            _o1.add("reason");
            _o1.add("rebuild");
            _o1.add("record");
            _o1.add("records");
            _o1.add("recycle");
            _o1.add("reduce");
            _o1.add("remote-schema");
            _o1.add("rename");
            _o1.add("render");
            _o1.add("replace");
            _o1.add("resources");
            _o1.add("restore");
            _o1.add("restrict");
            _o1.add("restricted_view_access");
            _o1.add("resume-purges");
            _o1.add("retention");
            _o1.add("request_classification");
            _o1.add("request_support");
            _o1.add("roles");
            _o1.add("row_level_security");
            _o1.add("roworder");
            _o1.add("roworderpolicy");
            _o1.add("rowstore");
            _o1.add("rowstorepolicy");
            _o1.add("rowstore_references");
            _o1.add("rowstores");
            _o1.add("rowstore_sealinfo");
            _o1.add("run");
            _o1.add("running");
            _o1.add("sample");
            _o1.add("sample-distinct");
            _o1.add("sandbox");
            _o1.add("sandboxes");
            _o1.add("save");
            _o1.add("scalein");
            _o1.add("scan");
            _o1.add("schema");
            _o1.add("script");
            _o1.add("seal");
            _o1.add("seals");
            _o1.add("search");
            _o1.add("serialize");
            _o1.add("series");
            _o1.add("servicepoints");
            _o1.add("services");
            _o1.add("set");
            _o1.add("set-or-append");
            _o1.add("set-or-replace");
            _o1.add("sharding");
            _o1.add("shards");
            _o1.add("shard-groups");
            _o1.add("shards_grouping");
            _o1.add("show");
            _o1.add("softdelete");
            _o1.add("softretention");
            _o1.add("sort");
            _o1.add("sql");
            _o1.add("stacked");
            _o1.add("stacked100");
            _o1.add("state");
            _o1.add("stats");
            _o1.add("statistics");
            _o1.add("step");
            _o1.add("storedqueryresultcontainers");
            _o1.add("stream");
            _o1.add("streamingingestion");
            _o1.add("streaming-ingestion-post-processing");
            _o1.add("stored-query-results");
            _o1.add("summarize");
            _o1.add("sstream");
            _o1.add("storage");
            _o1.add("suspend");
            _o1.add("systemdb");
            _o1.add("table");
            _o1.add("tablepurge");
            _o1.add("table-purge");
            _o1.add("tables");
            _o1.add("tags");
            _o1.add("take");
            _o1.add("tcpconnections");
            _o1.add("tcpports");
            _o1.add("tempstorage");
            _o1.add("tenants");
            _o1.add("threadpools");
            _o1.add("threshold");
            _o1.add("throw");
            _o1.add("title");
            _o1.add("to");
            _o1.add("top");
            _o1.add("top-hitters");
            _o1.add("top-nested");
            _o1.add("toscalar");
            _o1.add("totable");
            _o1.add("trace");
            _o1.add("traceresults");
            _o1.add("tsv");
            _o1.add("type");
            _o1.add("typeof");
            _o1.add("undo");
            _o1.add("unstacked");
            _o1.add("unrestrictedviewers");
            _o1.add("update");
            _o1.add("usage");
            _o1.add("user");
            _o1.add("users");
            _o1.add("using");
            _o1.add("uuid");
            _o1.add("verbose");
            _o1.add("version");
            _o1.add("view");
            _o1.add("views");
            _o1.add("viewers");
            _o1.add("virtual_cluster");
            _o1.add("virtual_clusters");
            _o1.add("visible");
            _o1.add("volatile");
            _o1.add("v9");
            _o1.add("v10");
            _o1.add("w3clogfile");
            _o1.add("warm");
            _o1.add("warming");
            _o1.add("whatif");
            _o1.add("where");
            _o1.add("with");
            _o1.add("workload_group");
            _o1.add("workload_groups");
            _o1.add("writeaheadlog");
            _o1.add("xaxis");
            _o1.add("xcolumn");
            _o1.add("xmin");
            _o1.add("xmax");
            _o1.add("xtitle");
            _o1.add("yaxis");
            _o1.add("ycolumns");
            _o1.add("ymin");
            _o1.add("ymax");
            _o1.add("ysplit");
            _o1.add("ytitle");
            _o1.add("bool");
            _o1.add("boolean");
            _o1.add("int8");
            _o1.add("char");
            _o1.add("uint8");
            _o1.add("byte");
            _o1.add("int16");
            _o1.add("uint16");
            _o1.add("int");
            _o1.add("int32");
            _o1.add("uint");
            _o1.add("uint32");
            _o1.add("long");
            _o1.add("int64");
            _o1.add("ulong");
            _o1.add("uint64");
            _o1.add("float");
            _o1.add("real");
            _o1.add("double");
            _o1.add("string");
            _o1.add("time");
            _o1.add("timespan");
            _o1.add("date");
            _o1.add("datetime");
            _o1.add("guid");
            _o1.add("uniqueid");
            _o1.add("dynamic");
            _o1.add("decimal");
            _o1.add("GB");
            _o1.add("MB");
            _o1.add("application");
            _o1.add("blockedprincipals");
            _o1.add("bytes");
            _o1.add("callstacks");
            _o1.add("concurrency");
            _o1.add("configuration");
            _o1.add("container");
            _o1.add("containers");
            _o1.add("datasize");
            _o1.add("datastats");
            _o1.add("datetime_pattern");
            _o1.add("days");
            _o1.add("dimensions");
            _o1.add("disabled");
            _o1.add("dryrun");
            _o1.add("empty");
            _o1.add("enabled");
            _o1.add("exclude");
            _o1.add("expired_tables_cleanup");
            _o1.add("extentsize");
            _o1.add("follower");
            _o1.add("for");
            _o1.add("format_datetime");
            _o1.add("hot_window");
            _o1.add("identity");
            _o1.add("include");
            _o1.add("network");
            _o1.add("recoverability");
            _o1.add("startofday");
            _o1.add("startofmonth");
            _o1.add("startofweek");
            _o1.add("startofyear");
            _o1.add("stored_query_result");
            _o1.add("stored_query_results");
            _o1.add("transactions");
            _o1.add("trim");
            _o1.add("union");
            _o1.add("until");
            _o1.add("unused");
            _o1.add("utilization");
            _o1.add("violations");
            _o1.add("3Dchart");
            _o1.add("__contextual_datatable");
            _o1.add("__crossCluster");
            _o1.add("__crossDB");
            _o1.add("__executeAndCache");
            _o1.add("__id");
            _o1.add("__isFuzzy");
            _o1.add("__noWithSource");
            _o1.add("__packedColumn");
            _o1.add("__projectAway");
            _o1.add("__sourceColumnIndex");
            _o1.add("anomalychart");
            _o1.add("areachart");
            _o1.add("bagexpansion");
            _o1.add("barchart");
            _o1.add("bin_legacy");
            _o1.add("card");
            _o1.add("columnchart");
            _o1.add("datascope");
            _o1.add("decodeblocks");
            _o1.add("expandoutput");
            _o1.add("hotcache");
            _o1.add("isfuzzy");
            _o1.add("join");
            _o1.add("ladderchart");
            _o1.add("linechart");
            _o1.add("nooptimization");
            _o1.add("piechart");
            _o1.add("pivotchart");
            _o1.add("regex");
            _o1.add("relaxed");
            _o1.add("scatterchart");
            _o1.add("simple");
            _o1.add("stackedareachart");
            _o1.add("timechart");
            _o1.add("timeline");
            _o1.add("timepivot");
            _o1.add("treemap");
            _o1.add("with_itemindex");
            _o1.add("with_match_id");
            _o1.add("with_source");
            _o1.add("with_step_name");
            _o1.add("withsource");
            return _o1;
        },
        f2: function (c) {
            return ((System.Char.isDigit(c) || System.Char.isLetter(c)) && Kusto.Data.Common.CslSyntaxGenerator.IsAscii(c)) || c === 95;
        }
    });

    /**
     * Flavor of DatabaseShow command
     *
     * @public
     * @class Kusto.Data.Common.DatabaseShowFlavor
     */
    Bridge.define("Kusto.Data.Common.DatabaseShowFlavor", {
        $kind: "enum",
        statics: {
            fields: {
                Identity: 0,
                Verbose: 1,
                Policies: 2,
                DataStats: 3,
                Default: 4
            }
        }
    });

    /**
     * Errors from Data Connections
     *
     * @public
     * @class Kusto.Data.Common.DataConnectionError
     */
    Bridge.define("Kusto.Data.Common.DataConnectionError", {
        $kind: "enum",
        statics: {
            fields: {
                /**
                 * Managed Identity doesn't have sufficient authorizations. This likely indicates that the user didn't provide the Managed Identity with the necessary role or scope.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.DataConnectionError
                 * @constant
                 * @default 0
                 * @type Kusto.Data.Common.DataConnectionError
                 */
                ManagedIdentityUnauthorized: 0,
                /**
                 * Managed Identity doesn't exist in the cluster. This likely indicates the user didn't link their Managed Identity to this ADX cluster.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.DataConnectionError
                 * @constant
                 * @default 1
                 * @type Kusto.Data.Common.DataConnectionError
                 */
                ManagedIdentityNotFound: 1,
                /**
                 * Data Connection authenticates using a connection string instead of a managed identity, which is deprecated.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.DataConnectionError
                 * @constant
                 * @default 2
                 * @type Kusto.Data.Common.DataConnectionError
                 */
                UsingDeprecatedConnectionString: 2
            }
        }
    });

    /**
     * Recoverability of data after soft deletion
     *
     * @public
     * @class Kusto.Data.Common.DataRecoverability
     */
    Bridge.define("Kusto.Data.Common.DataRecoverability", {
        $kind: "enum",
        statics: {
            fields: {
                Unknown: 0,
                Enabled: 1,
                Disabled: 2
            }
        }
    });

    /**
     * Data compression types supported for Kusto ingestion.
     *
     * @public
     * @class Kusto.Data.Common.DataSourceCompressionType
     */
    Bridge.define("Kusto.Data.Common.DataSourceCompressionType", {
        $kind: "enum",
        statics: {
            fields: {
                /**
                 * No compression
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.DataSourceCompressionType
                 * @constant
                 * @default 0
                 * @type Kusto.Data.Common.DataSourceCompressionType
                 */
                None: 0,
                /**
                 * GZip compression
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.DataSourceCompressionType
                 * @constant
                 * @default 1
                 * @type Kusto.Data.Common.DataSourceCompressionType
                 */
                GZip: 1,
                /**
                 * Zip compression
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.DataSourceCompressionType
                 * @constant
                 * @default 2
                 * @type Kusto.Data.Common.DataSourceCompressionType
                 */
                Zip: 2
            }
        }
    });

    /**
     * Kusto Entity Types.
     *
     * @public
     * @class Kusto.Data.Common.EntityType
     */
    Bridge.define("Kusto.Data.Common.EntityType", {
        $kind: "enum",
        statics: {
            fields: {
                /**
                 * Unknown
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.EntityType
                 * @constant
                 * @default 0
                 * @type Kusto.Data.Common.EntityType
                 */
                Unknown: 0,
                /**
                 * Cluster entity type
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.EntityType
                 * @constant
                 * @default 1
                 * @type Kusto.Data.Common.EntityType
                 */
                Cluster: 1,
                /**
                 * Database entity type
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.EntityType
                 * @constant
                 * @default 2
                 * @type Kusto.Data.Common.EntityType
                 */
                Database: 2,
                /**
                 * Table entity type
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.EntityType
                 * @constant
                 * @default 3
                 * @type Kusto.Data.Common.EntityType
                 */
                Table: 3,
                /**
                 * Column entity type
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.EntityType
                 * @constant
                 * @default 4
                 * @type Kusto.Data.Common.EntityType
                 */
                Column: 4,
                /**
                 * Function entity type
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.EntityType
                 * @constant
                 * @default 5
                 * @type Kusto.Data.Common.EntityType
                 */
                Function: 5,
                /**
                 * External table entity type
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.EntityType
                 * @constant
                 * @default 6
                 * @type Kusto.Data.Common.EntityType
                 */
                ExternalTable: 6,
                /**
                 * Materialized view entity type
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.EntityType
                 * @constant
                 * @default 7
                 * @type Kusto.Data.Common.EntityType
                 */
                MaterializedView: 7
            }
        }
    });

    /**
     * Extensions for enum DatabaseShowFlavor
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Common.ExtendedDatabaseShowFlavor
     */
    Bridge.define("Kusto.Data.Common.ExtendedDatabaseShowFlavor", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.DatabaseShowFlavor.Identity: 
                            return "Identity";
                        case Kusto.Data.Common.DatabaseShowFlavor.Verbose: 
                            return "Verbose";
                        case Kusto.Data.Common.DatabaseShowFlavor.Policies: 
                            return "Policies";
                        case Kusto.Data.Common.DatabaseShowFlavor.DataStats: 
                            return "DataStats";
                        case Kusto.Data.Common.DatabaseShowFlavor.Default: 
                            return "Default";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.Common.DatabaseShowFlavor() };
                    return Kusto.Data.Common.ExtendedDatabaseShowFlavor.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.Common.DatabaseShowFlavor.Identity: 
                            return true;
                        case Kusto.Data.Common.DatabaseShowFlavor.Verbose: 
                            return true;
                        case Kusto.Data.Common.DatabaseShowFlavor.Policies: 
                            return true;
                        case Kusto.Data.Common.DatabaseShowFlavor.DataStats: 
                            return true;
                        case Kusto.Data.Common.DatabaseShowFlavor.Default: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.Common.DatabaseShowFlavor() };
                    if (Kusto.Data.Common.ExtendedDatabaseShowFlavor.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "0": 
                        case "Identity": 
                            ret.v = Kusto.Data.Common.DatabaseShowFlavor.Identity;
                            return true;
                        case "1": 
                        case "Verbose": 
                            ret.v = Kusto.Data.Common.DatabaseShowFlavor.Verbose;
                            return true;
                        case "2": 
                        case "Policies": 
                            ret.v = Kusto.Data.Common.DatabaseShowFlavor.Policies;
                            return true;
                        case "3": 
                        case "DataStats": 
                            ret.v = Kusto.Data.Common.DatabaseShowFlavor.DataStats;
                            return true;
                        case "4": 
                        case "Default": 
                            ret.v = Kusto.Data.Common.DatabaseShowFlavor.Default;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "identity": 
                                ret.v = Kusto.Data.Common.DatabaseShowFlavor.Identity;
                                return true;
                            case "verbose": 
                                ret.v = Kusto.Data.Common.DatabaseShowFlavor.Verbose;
                                return true;
                            case "policies": 
                                ret.v = Kusto.Data.Common.DatabaseShowFlavor.Policies;
                                return true;
                            case "datastats": 
                                ret.v = Kusto.Data.Common.DatabaseShowFlavor.DataStats;
                                return true;
                            case "default": 
                                ret.v = Kusto.Data.Common.DatabaseShowFlavor.Default;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.DatabaseShowFlavor.Identity: 
                            return "";
                        case Kusto.Data.Common.DatabaseShowFlavor.Verbose: 
                            return "";
                        case Kusto.Data.Common.DatabaseShowFlavor.Policies: 
                            return "";
                        case Kusto.Data.Common.DatabaseShowFlavor.DataStats: 
                            return "";
                        case Kusto.Data.Common.DatabaseShowFlavor.Default: 
                            return "";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * Extensions for enum DataConnectionError
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Common.ExtendedDataConnectionError
     */
    Bridge.define("Kusto.Data.Common.ExtendedDataConnectionError", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.DataConnectionError.ManagedIdentityUnauthorized: 
                            return "ManagedIdentityUnauthorized";
                        case Kusto.Data.Common.DataConnectionError.ManagedIdentityNotFound: 
                            return "ManagedIdentityNotFound";
                        case Kusto.Data.Common.DataConnectionError.UsingDeprecatedConnectionString: 
                            return "UsingDeprecatedConnectionString";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.Common.DataConnectionError() };
                    return Kusto.Data.Common.ExtendedDataConnectionError.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.Common.DataConnectionError.ManagedIdentityUnauthorized: 
                            return true;
                        case Kusto.Data.Common.DataConnectionError.ManagedIdentityNotFound: 
                            return true;
                        case Kusto.Data.Common.DataConnectionError.UsingDeprecatedConnectionString: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.Common.DataConnectionError() };
                    if (Kusto.Data.Common.ExtendedDataConnectionError.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "ManagedIdentityUnauthorized": 
                            ret.v = Kusto.Data.Common.DataConnectionError.ManagedIdentityUnauthorized;
                            return true;
                        case "ManagedIdentityNotFound": 
                            ret.v = Kusto.Data.Common.DataConnectionError.ManagedIdentityNotFound;
                            return true;
                        case "UsingDeprecatedConnectionString": 
                            ret.v = Kusto.Data.Common.DataConnectionError.UsingDeprecatedConnectionString;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "managedidentityunauthorized": 
                                ret.v = Kusto.Data.Common.DataConnectionError.ManagedIdentityUnauthorized;
                                return true;
                            case "managedidentitynotfound": 
                                ret.v = Kusto.Data.Common.DataConnectionError.ManagedIdentityNotFound;
                                return true;
                            case "usingdeprecatedconnectionstring": 
                                ret.v = Kusto.Data.Common.DataConnectionError.UsingDeprecatedConnectionString;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.DataConnectionError.ManagedIdentityUnauthorized: 
                            return "Managed Identity doesn't have sufficient authorizations. This likely indicates that the user didn't provide the Managed Identity with the necessary role or scope.";
                        case Kusto.Data.Common.DataConnectionError.ManagedIdentityNotFound: 
                            return "Managed Identity doesn't exist in the cluster. This likely indicates the user didn't link their Managed Identity to this ADX cluster.";
                        case Kusto.Data.Common.DataConnectionError.UsingDeprecatedConnectionString: 
                            return "Data Connection authenticates using a connection string instead of a managed identity, which is deprecated.";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * Extensions for enum DataRecoverability
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Common.ExtendedDataRecoverability
     */
    Bridge.define("Kusto.Data.Common.ExtendedDataRecoverability", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.DataRecoverability.Unknown: 
                            return "Unknown";
                        case Kusto.Data.Common.DataRecoverability.Enabled: 
                            return "Enabled";
                        case Kusto.Data.Common.DataRecoverability.Disabled: 
                            return "Disabled";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.Common.DataRecoverability() };
                    return Kusto.Data.Common.ExtendedDataRecoverability.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.Common.DataRecoverability.Unknown: 
                            return true;
                        case Kusto.Data.Common.DataRecoverability.Enabled: 
                            return true;
                        case Kusto.Data.Common.DataRecoverability.Disabled: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.Common.DataRecoverability() };
                    if (Kusto.Data.Common.ExtendedDataRecoverability.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "0": 
                        case "Unknown": 
                            ret.v = Kusto.Data.Common.DataRecoverability.Unknown;
                            return true;
                        case "1": 
                        case "Enabled": 
                            ret.v = Kusto.Data.Common.DataRecoverability.Enabled;
                            return true;
                        case "2": 
                        case "Disabled": 
                            ret.v = Kusto.Data.Common.DataRecoverability.Disabled;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "unknown": 
                                ret.v = Kusto.Data.Common.DataRecoverability.Unknown;
                                return true;
                            case "enabled": 
                                ret.v = Kusto.Data.Common.DataRecoverability.Enabled;
                                return true;
                            case "disabled": 
                                ret.v = Kusto.Data.Common.DataRecoverability.Disabled;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.DataRecoverability.Unknown: 
                            return "";
                        case Kusto.Data.Common.DataRecoverability.Enabled: 
                            return "";
                        case Kusto.Data.Common.DataRecoverability.Disabled: 
                            return "";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * Extensions for enum DataSourceCompressionType
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Common.ExtendedDataSourceCompressionType
     */
    Bridge.define("Kusto.Data.Common.ExtendedDataSourceCompressionType", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.DataSourceCompressionType.None: 
                            return "None";
                        case Kusto.Data.Common.DataSourceCompressionType.GZip: 
                            return "GZip";
                        case Kusto.Data.Common.DataSourceCompressionType.Zip: 
                            return "Zip";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.Common.DataSourceCompressionType() };
                    return Kusto.Data.Common.ExtendedDataSourceCompressionType.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.Common.DataSourceCompressionType.None: 
                            return true;
                        case Kusto.Data.Common.DataSourceCompressionType.GZip: 
                            return true;
                        case Kusto.Data.Common.DataSourceCompressionType.Zip: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.Common.DataSourceCompressionType() };
                    if (Kusto.Data.Common.ExtendedDataSourceCompressionType.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "0": 
                        case "None": 
                            ret.v = Kusto.Data.Common.DataSourceCompressionType.None;
                            return true;
                        case "1": 
                        case "GZip": 
                            ret.v = Kusto.Data.Common.DataSourceCompressionType.GZip;
                            return true;
                        case "2": 
                        case "Zip": 
                            ret.v = Kusto.Data.Common.DataSourceCompressionType.Zip;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "none": 
                                ret.v = Kusto.Data.Common.DataSourceCompressionType.None;
                                return true;
                            case "gzip": 
                                ret.v = Kusto.Data.Common.DataSourceCompressionType.GZip;
                                return true;
                            case "zip": 
                                ret.v = Kusto.Data.Common.DataSourceCompressionType.Zip;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.DataSourceCompressionType.None: 
                            return "No compression";
                        case Kusto.Data.Common.DataSourceCompressionType.GZip: 
                            return "GZip compression";
                        case Kusto.Data.Common.DataSourceCompressionType.Zip: 
                            return "Zip compression";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * Extensions for enum EntityType
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Common.ExtendedEntityType
     */
    Bridge.define("Kusto.Data.Common.ExtendedEntityType", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.EntityType.Unknown: 
                            return "Unknown";
                        case Kusto.Data.Common.EntityType.Cluster: 
                            return "Cluster";
                        case Kusto.Data.Common.EntityType.Database: 
                            return "Database";
                        case Kusto.Data.Common.EntityType.Table: 
                            return "Table";
                        case Kusto.Data.Common.EntityType.Column: 
                            return "Column";
                        case Kusto.Data.Common.EntityType.Function: 
                            return "Function";
                        case Kusto.Data.Common.EntityType.ExternalTable: 
                            return "ExternalTable";
                        case Kusto.Data.Common.EntityType.MaterializedView: 
                            return "MaterializedView";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.Common.EntityType() };
                    return Kusto.Data.Common.ExtendedEntityType.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.Common.EntityType.Unknown: 
                            return true;
                        case Kusto.Data.Common.EntityType.Cluster: 
                            return true;
                        case Kusto.Data.Common.EntityType.Database: 
                            return true;
                        case Kusto.Data.Common.EntityType.Table: 
                            return true;
                        case Kusto.Data.Common.EntityType.Column: 
                            return true;
                        case Kusto.Data.Common.EntityType.Function: 
                            return true;
                        case Kusto.Data.Common.EntityType.ExternalTable: 
                            return true;
                        case Kusto.Data.Common.EntityType.MaterializedView: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.Common.EntityType() };
                    if (Kusto.Data.Common.ExtendedEntityType.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "0": 
                        case "Unknown": 
                            ret.v = Kusto.Data.Common.EntityType.Unknown;
                            return true;
                        case "1": 
                        case "Cluster": 
                            ret.v = Kusto.Data.Common.EntityType.Cluster;
                            return true;
                        case "2": 
                        case "Database": 
                            ret.v = Kusto.Data.Common.EntityType.Database;
                            return true;
                        case "3": 
                        case "Table": 
                            ret.v = Kusto.Data.Common.EntityType.Table;
                            return true;
                        case "4": 
                        case "Column": 
                            ret.v = Kusto.Data.Common.EntityType.Column;
                            return true;
                        case "5": 
                        case "Function": 
                            ret.v = Kusto.Data.Common.EntityType.Function;
                            return true;
                        case "6": 
                        case "ExternalTable": 
                            ret.v = Kusto.Data.Common.EntityType.ExternalTable;
                            return true;
                        case "7": 
                        case "MaterializedView": 
                        case "materialized-view": 
                            ret.v = Kusto.Data.Common.EntityType.MaterializedView;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "unknown": 
                                ret.v = Kusto.Data.Common.EntityType.Unknown;
                                return true;
                            case "cluster": 
                                ret.v = Kusto.Data.Common.EntityType.Cluster;
                                return true;
                            case "database": 
                                ret.v = Kusto.Data.Common.EntityType.Database;
                                return true;
                            case "table": 
                                ret.v = Kusto.Data.Common.EntityType.Table;
                                return true;
                            case "column": 
                                ret.v = Kusto.Data.Common.EntityType.Column;
                                return true;
                            case "function": 
                                ret.v = Kusto.Data.Common.EntityType.Function;
                                return true;
                            case "externaltable": 
                                ret.v = Kusto.Data.Common.EntityType.ExternalTable;
                                return true;
                            case "materializedview": 
                            case "materialized-view": 
                                ret.v = Kusto.Data.Common.EntityType.MaterializedView;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.EntityType.Unknown: 
                            return "Unknown";
                        case Kusto.Data.Common.EntityType.Cluster: 
                            return "Cluster entity type";
                        case Kusto.Data.Common.EntityType.Database: 
                            return "Database entity type";
                        case Kusto.Data.Common.EntityType.Table: 
                            return "Table entity type";
                        case Kusto.Data.Common.EntityType.Column: 
                            return "Column entity type";
                        case Kusto.Data.Common.EntityType.Function: 
                            return "Function entity type";
                        case Kusto.Data.Common.EntityType.ExternalTable: 
                            return "External table entity type";
                        case Kusto.Data.Common.EntityType.MaterializedView: 
                            return "Materialized view entity type";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * Extensions for enum ExtentsMergeLookbackPeriodKind
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Common.ExtendedExtentsMergeLookbackPeriodKind
     */
    Bridge.define("Kusto.Data.Common.ExtendedExtentsMergeLookbackPeriodKind", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.Default: 
                            return "Default";
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.HotCache: 
                            return "HotCache";
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.All: 
                            return "All";
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.Custom: 
                            return "Custom";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.Common.ExtentsMergeLookbackPeriodKind() };
                    return Kusto.Data.Common.ExtendedExtentsMergeLookbackPeriodKind.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.Default: 
                            return true;
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.HotCache: 
                            return true;
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.All: 
                            return true;
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.Custom: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.Common.ExtentsMergeLookbackPeriodKind() };
                    if (Kusto.Data.Common.ExtendedExtentsMergeLookbackPeriodKind.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "0": 
                        case "Default": 
                            ret.v = Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.Default;
                            return true;
                        case "1": 
                        case "HotCache": 
                            ret.v = Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.HotCache;
                            return true;
                        case "2": 
                        case "All": 
                            ret.v = Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.All;
                            return true;
                        case "3": 
                        case "Custom": 
                            ret.v = Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.Custom;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "default": 
                                ret.v = Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.Default;
                                return true;
                            case "hotcache": 
                                ret.v = Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.HotCache;
                                return true;
                            case "all": 
                                ret.v = Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.All;
                                return true;
                            case "custom": 
                                ret.v = Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.Custom;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.Default: 
                            return "The default lookback period, as defined by the system";
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.HotCache: 
                            return "The hot cache period";
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.All: 
                            return "The entire period in which data exists";
                        case Kusto.Data.Common.ExtentsMergeLookbackPeriodKind.Custom: 
                            return "A custom period, provided by the admin of the entity";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * Extensions for enum IngestionErrorCategory
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Common.ExtendedIngestionErrorCategory
     */
    Bridge.define("Kusto.Data.Common.ExtendedIngestionErrorCategory", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.IngestionErrorCategory.Unknown: 
                            return "Unknown";
                        case Kusto.Data.Common.IngestionErrorCategory.InternalServiceError: 
                            return "InternalServiceError";
                        case Kusto.Data.Common.IngestionErrorCategory.BadFormat: 
                            return "BadFormat";
                        case Kusto.Data.Common.IngestionErrorCategory.BadRequest: 
                            return "BadRequest";
                        case Kusto.Data.Common.IngestionErrorCategory.UpdatePolicyFailure: 
                            return "UpdatePolicyFailure";
                        case Kusto.Data.Common.IngestionErrorCategory.EntityNotFound: 
                            return "EntityNotFound";
                        case Kusto.Data.Common.IngestionErrorCategory.UserAccessNotAuthorized: 
                            return "UserAccessNotAuthorized";
                        case Kusto.Data.Common.IngestionErrorCategory.DataAccessNotAuthorized: 
                            return "DataAccessNotAuthorized";
                        case Kusto.Data.Common.IngestionErrorCategory.FileTooLarge: 
                            return "FileTooLarge";
                        case Kusto.Data.Common.IngestionErrorCategory.ThrottledOnEngine: 
                            return "ThrottledOnEngine";
                        case Kusto.Data.Common.IngestionErrorCategory.DownloadFailed: 
                            return "DownloadFailed";
                        case Kusto.Data.Common.IngestionErrorCategory.RetryAttemptsExceeded: 
                            return "RetryAttemptsExceeded";
                        case Kusto.Data.Common.IngestionErrorCategory.BlobAlreadyReceived: 
                            return "BlobAlreadyReceived";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.Common.IngestionErrorCategory() };
                    return Kusto.Data.Common.ExtendedIngestionErrorCategory.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.Common.IngestionErrorCategory.Unknown: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.InternalServiceError: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.BadFormat: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.BadRequest: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.UpdatePolicyFailure: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.EntityNotFound: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.UserAccessNotAuthorized: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.DataAccessNotAuthorized: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.FileTooLarge: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.ThrottledOnEngine: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.DownloadFailed: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.RetryAttemptsExceeded: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCategory.BlobAlreadyReceived: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.Common.IngestionErrorCategory() };
                    if (Kusto.Data.Common.ExtendedIngestionErrorCategory.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "Unknown": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.Unknown;
                            return true;
                        case "InternalServiceError": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.InternalServiceError;
                            return true;
                        case "BadFormat": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.BadFormat;
                            return true;
                        case "BadRequest": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.BadRequest;
                            return true;
                        case "UpdatePolicyFailure": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.UpdatePolicyFailure;
                            return true;
                        case "EntityNotFound": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.EntityNotFound;
                            return true;
                        case "UserAccessNotAuthorized": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.UserAccessNotAuthorized;
                            return true;
                        case "DataAccessNotAuthorized": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.DataAccessNotAuthorized;
                            return true;
                        case "FileTooLarge": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.FileTooLarge;
                            return true;
                        case "ThrottledOnEngine": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.ThrottledOnEngine;
                            return true;
                        case "DownloadFailed": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.DownloadFailed;
                            return true;
                        case "RetryAttemptsExceeded": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.RetryAttemptsExceeded;
                            return true;
                        case "BlobAlreadyReceived": 
                            ret.v = Kusto.Data.Common.IngestionErrorCategory.BlobAlreadyReceived;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "unknown": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.Unknown;
                                return true;
                            case "internalserviceerror": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.InternalServiceError;
                                return true;
                            case "badformat": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.BadFormat;
                                return true;
                            case "badrequest": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.BadRequest;
                                return true;
                            case "updatepolicyfailure": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.UpdatePolicyFailure;
                                return true;
                            case "entitynotfound": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.EntityNotFound;
                                return true;
                            case "useraccessnotauthorized": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.UserAccessNotAuthorized;
                                return true;
                            case "dataaccessnotauthorized": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.DataAccessNotAuthorized;
                                return true;
                            case "filetoolarge": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.FileTooLarge;
                                return true;
                            case "throttledonengine": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.ThrottledOnEngine;
                                return true;
                            case "downloadfailed": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.DownloadFailed;
                                return true;
                            case "retryattemptsexceeded": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.RetryAttemptsExceeded;
                                return true;
                            case "blobalreadyreceived": 
                                ret.v = Kusto.Data.Common.IngestionErrorCategory.BlobAlreadyReceived;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.IngestionErrorCategory.Unknown: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.InternalServiceError: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.BadFormat: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.BadRequest: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.UpdatePolicyFailure: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.EntityNotFound: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.UserAccessNotAuthorized: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.DataAccessNotAuthorized: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.FileTooLarge: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.ThrottledOnEngine: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.DownloadFailed: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.RetryAttemptsExceeded: 
                            return "";
                        case Kusto.Data.Common.IngestionErrorCategory.BlobAlreadyReceived: 
                            return "";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * Extensions for enum IngestionErrorCode
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Common.ExtendedIngestionErrorCode
     */
    Bridge.define("Kusto.Data.Common.ExtendedIngestionErrorCode", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.IngestionErrorCode.Unknown: 
                            return "Unknown";
                        case Kusto.Data.Common.IngestionErrorCode.OutOfMemory: 
                            return "OutOfMemory";
                        case Kusto.Data.Common.IngestionErrorCode.Stream_WrongNumberOfFields: 
                            return "Stream_WrongNumberOfFields";
                        case Kusto.Data.Common.IngestionErrorCode.Stream_InputStreamTooLarge: 
                            return "Stream_InputStreamTooLarge";
                        case Kusto.Data.Common.IngestionErrorCode.Stream_NoDataToIngest: 
                            return "Stream_NoDataToIngest";
                        case Kusto.Data.Common.IngestionErrorCode.Stream_ClosingQuoteMissing: 
                            return "Stream_ClosingQuoteMissing";
                        case Kusto.Data.Common.IngestionErrorCode.Stream_DynamicPropertyBagTooLarge: 
                            return "Stream_DynamicPropertyBagTooLarge";
                        case Kusto.Data.Common.IngestionErrorCode.Download_SourceNotFound: 
                            return "Download_SourceNotFound";
                        case Kusto.Data.Common.IngestionErrorCode.Download_AccessConditionNotSatisfied: 
                            return "Download_AccessConditionNotSatisfied";
                        case Kusto.Data.Common.IngestionErrorCode.Download_Forbidden: 
                            return "Download_Forbidden";
                        case Kusto.Data.Common.IngestionErrorCode.Download_AccountNotFound: 
                            return "Download_AccountNotFound";
                        case Kusto.Data.Common.IngestionErrorCode.Download_BadRequest: 
                            return "Download_BadRequest";
                        case Kusto.Data.Common.IngestionErrorCode.Download_NotTransient: 
                            return "Download_NotTransient";
                        case Kusto.Data.Common.IngestionErrorCode.Download_UnknownError: 
                            return "Download_UnknownError";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_QuerySchemaDoesNotMatchTableSchema: 
                            return "UpdatePolicy_QuerySchemaDoesNotMatchTableSchema";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_FailedDescendantTransaction: 
                            return "UpdatePolicy_FailedDescendantTransaction";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_IngestionError: 
                            return "UpdatePolicy_IngestionError";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_UnknownError: 
                            return "UpdatePolicy_UnknownError";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_MissingMappingFailure: 
                            return "BadRequest_MissingMappingFailure";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidBlob: 
                            return "BadRequest_InvalidBlob";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyBlob: 
                            return "BadRequest_EmptyBlob";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_NoRecordsOrWrongFormat: 
                            return "BadRequest_NoRecordsOrWrongFormat";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyArchive: 
                            return "BadRequest_EmptyArchive";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidArchive: 
                            return "BadRequest_InvalidArchive";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidBlobUri: 
                            return "BadRequest_InvalidBlobUri";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyBlobUri: 
                            return "BadRequest_EmptyBlobUri";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DatabaseNotExist: 
                            return "BadRequest_DatabaseNotExist";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_TableNotExist: 
                            return "BadRequest_TableNotExist";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidKustoIdentityToken: 
                            return "BadRequest_InvalidKustoIdentityToken";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_FileTooLarge: 
                            return "BadRequest_FileTooLarge";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_TableAccessDenied: 
                            return "BadRequest_TableAccessDenied";
                        case Kusto.Data.Common.IngestionErrorCode.General_RetryAttemptsExceeded: 
                            return "General_RetryAttemptsExceeded";
                        case Kusto.Data.Common.IngestionErrorCode.General_BadRequest: 
                            return "General_BadRequest";
                        case Kusto.Data.Common.IngestionErrorCode.General_InternalServerError: 
                            return "General_InternalServerError";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_Cyclic_Update_Not_Allowed: 
                            return "UpdatePolicy_Cyclic_Update_Not_Allowed";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidMapping: 
                            return "BadRequest_InvalidMapping";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidMappingReference: 
                            return "BadRequest_InvalidMappingReference";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_MappingReferenceWasNotFound: 
                            return "BadRequest_MappingReferenceWasNotFound";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_FormatNotSupported: 
                            return "BadRequest_FormatNotSupported";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DuplicateMapping: 
                            return "BadRequest_DuplicateMapping";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_CorruptedMessage: 
                            return "BadRequest_CorruptedMessage";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InconsistentMapping: 
                            return "BadRequest_InconsistentMapping";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_SyntaxError: 
                            return "BadRequest_SyntaxError";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidOrEmptyTableName: 
                            return "BadRequest_InvalidOrEmptyTableName";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyDatabaseName: 
                            return "BadRequest_EmptyDatabaseName";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyTenantIdentifier: 
                            return "BadRequest_EmptyTenantIdentifier";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyMappingReference: 
                            return "BadRequest_EmptyMappingReference";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EntityNotFound: 
                            return "BadRequest_EntityNotFound";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EntityNameIsNotValid: 
                            return "BadRequest_EntityNameIsNotValid";
                        case Kusto.Data.Common.IngestionErrorCode.General_AbandonedIngestion: 
                            return "General_AbandonedIngestion";
                        case Kusto.Data.Common.IngestionErrorCode.General_ThrottledIngestion: 
                            return "General_ThrottledIngestion";
                        case Kusto.Data.Common.IngestionErrorCode.General_TransientSchemaMismatch: 
                            return "General_TransientSchemaMismatch";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DatabaseAccessDenied: 
                            return "BadRequest_DatabaseAccessDenied";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_UnexpectedCharacterInInputStream: 
                            return "BadRequest_UnexpectedCharacterInInputStream";
                        case Kusto.Data.Common.IngestionErrorCode.Forbidden: 
                            return "Forbidden";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_ZeroRetentionPolicyWithNoUpdatePolicy: 
                            return "BadRequest_ZeroRetentionPolicyWithNoUpdatePolicy";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_CreationTimeEarlierThanSoftDeletePeriod: 
                            return "BadRequest_CreationTimeEarlierThanSoftDeletePeriod";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_NotSupported: 
                            return "BadRequest_NotSupported";
                        case Kusto.Data.Common.IngestionErrorCode.Skipped_IngestByTagAlreadyExists: 
                            return "Skipped_IngestByTagAlreadyExists";
                        case Kusto.Data.Common.IngestionErrorCode.Timeout: 
                            return "Timeout";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_MalformedIngestionProperty: 
                            return "BadRequest_MalformedIngestionProperty";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_IngestionPropertyNotSupportedInThisContext: 
                            return "BadRequest_IngestionPropertyNotSupportedInThisContext";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidAuthentication: 
                            return "BadRequest_InvalidAuthentication";
                        case Kusto.Data.Common.IngestionErrorCode.BlobAlreadyReceived_BlobAlreadyFoundInBatch: 
                            return "BlobAlreadyReceived_BlobAlreadyFoundInBatch";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidManagedIdentity: 
                            return "BadRequest_InvalidManagedIdentity";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DataCapacityLimitReached: 
                            return "BadRequest_DataCapacityLimitReached";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_BadRequest: 
                            return "UpdatePolicy_BadRequest";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_NonMultiDatabaseDataConnection: 
                            return "BadRequest_NonMultiDatabaseDataConnection";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.UInt32)]);
                },
                FastIsDefined: function (value) {
                    var dummy = { v : new Kusto.Data.Common.IngestionErrorCode() };
                    return Kusto.Data.Common.ExtendedIngestionErrorCode.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined$1: function (value) {
                    switch (value) {
                        case Kusto.Data.Common.IngestionErrorCode.Unknown: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.OutOfMemory: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Stream_WrongNumberOfFields: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Stream_InputStreamTooLarge: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Stream_NoDataToIngest: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Stream_ClosingQuoteMissing: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Stream_DynamicPropertyBagTooLarge: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Download_SourceNotFound: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Download_AccessConditionNotSatisfied: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Download_Forbidden: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Download_AccountNotFound: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Download_BadRequest: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Download_NotTransient: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Download_UnknownError: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_QuerySchemaDoesNotMatchTableSchema: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_FailedDescendantTransaction: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_IngestionError: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_UnknownError: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_MissingMappingFailure: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidBlob: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyBlob: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_NoRecordsOrWrongFormat: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyArchive: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidArchive: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidBlobUri: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyBlobUri: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DatabaseNotExist: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_TableNotExist: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidKustoIdentityToken: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_FileTooLarge: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_TableAccessDenied: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.General_RetryAttemptsExceeded: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.General_BadRequest: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.General_InternalServerError: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_Cyclic_Update_Not_Allowed: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidMapping: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidMappingReference: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_MappingReferenceWasNotFound: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_FormatNotSupported: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DuplicateMapping: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_CorruptedMessage: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InconsistentMapping: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_SyntaxError: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidOrEmptyTableName: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyDatabaseName: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyTenantIdentifier: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyMappingReference: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EntityNotFound: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EntityNameIsNotValid: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.General_AbandonedIngestion: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.General_ThrottledIngestion: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.General_TransientSchemaMismatch: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DatabaseAccessDenied: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_UnexpectedCharacterInInputStream: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Forbidden: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_ZeroRetentionPolicyWithNoUpdatePolicy: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_CreationTimeEarlierThanSoftDeletePeriod: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_NotSupported: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Skipped_IngestByTagAlreadyExists: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.Timeout: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_MalformedIngestionProperty: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_IngestionPropertyNotSupportedInThisContext: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidAuthentication: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BlobAlreadyReceived_BlobAlreadyFoundInBatch: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidManagedIdentity: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DataCapacityLimitReached: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_BadRequest: 
                            return true;
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_NonMultiDatabaseDataConnection: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((((that & flag) >>> 0)) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((((that & flag) >>> 0)) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.Common.IngestionErrorCode() };
                    if (Kusto.Data.Common.ExtendedIngestionErrorCode.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "Unknown": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Unknown;
                            return true;
                        case "0x80DA0007": 
                        case "OutOfMemory": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.OutOfMemory;
                            return true;
                        case "0x80DA0008": 
                        case "Stream_WrongNumberOfFields": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Stream_WrongNumberOfFields;
                            return true;
                        case "0x80DA0009": 
                        case "Stream_InputStreamTooLarge": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Stream_InputStreamTooLarge;
                            return true;
                        case "Stream_NoDataToIngest": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Stream_NoDataToIngest;
                            return true;
                        case "0x80DA000b": 
                        case "Stream_ClosingQuoteMissing": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Stream_ClosingQuoteMissing;
                            return true;
                        case "0x80DA000E": 
                        case "Stream_DynamicPropertyBagTooLarge": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Stream_DynamicPropertyBagTooLarge;
                            return true;
                        case "Download_SourceNotFound": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Download_SourceNotFound;
                            return true;
                        case "Download_AccessConditionNotSatisfied": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Download_AccessConditionNotSatisfied;
                            return true;
                        case "Download_Forbidden": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Download_Forbidden;
                            return true;
                        case "Download_AccountNotFound": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Download_AccountNotFound;
                            return true;
                        case "Download_BadRequest": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Download_BadRequest;
                            return true;
                        case "Download_NotTransient": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Download_NotTransient;
                            return true;
                        case "Download_UnknownError": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Download_UnknownError;
                            return true;
                        case "UpdatePolicy_QuerySchemaDoesNotMatchTableSchema": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_QuerySchemaDoesNotMatchTableSchema;
                            return true;
                        case "UpdatePolicy_FailedDescendantTransaction": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_FailedDescendantTransaction;
                            return true;
                        case "UpdatePolicy_IngestionError": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_IngestionError;
                            return true;
                        case "UpdatePolicy_UnknownError": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_UnknownError;
                            return true;
                        case "BadRequest_MissingMappingFailure": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_MissingMappingFailure;
                            return true;
                        case "BadRequest_InvalidBlob": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidBlob;
                            return true;
                        case "BadRequest_EmptyBlob": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyBlob;
                            return true;
                        case "BadRequest_NoRecordsOrWrongFormat": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_NoRecordsOrWrongFormat;
                            return true;
                        case "BadRequest_EmptyArchive": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyArchive;
                            return true;
                        case "BadRequest_InvalidArchive": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidArchive;
                            return true;
                        case "BadRequest_InvalidBlobUri": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidBlobUri;
                            return true;
                        case "BadRequest_EmptyBlobUri": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyBlobUri;
                            return true;
                        case "BadRequest_DatabaseNotExist": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_DatabaseNotExist;
                            return true;
                        case "BadRequest_TableNotExist": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_TableNotExist;
                            return true;
                        case "BadRequest_InvalidKustoIdentityToken": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidKustoIdentityToken;
                            return true;
                        case "BadRequest_FileTooLarge": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_FileTooLarge;
                            return true;
                        case "BadRequest_TableAccessDenied": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_TableAccessDenied;
                            return true;
                        case "General_RetryAttemptsExceeded": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.General_RetryAttemptsExceeded;
                            return true;
                        case "General_BadRequest": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.General_BadRequest;
                            return true;
                        case "General_InternalServerError": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.General_InternalServerError;
                            return true;
                        case "UpdatePolicy_Cyclic_Update_Not_Allowed": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_Cyclic_Update_Not_Allowed;
                            return true;
                        case "BadRequest_InvalidMapping": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidMapping;
                            return true;
                        case "BadRequest_InvalidMappingReference": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidMappingReference;
                            return true;
                        case "BadRequest_MappingReferenceWasNotFound": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_MappingReferenceWasNotFound;
                            return true;
                        case "BadRequest_FormatNotSupported": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_FormatNotSupported;
                            return true;
                        case "BadRequest_DuplicateMapping": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_DuplicateMapping;
                            return true;
                        case "BadRequest_CorruptedMessage": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_CorruptedMessage;
                            return true;
                        case "BadRequest_InconsistentMapping": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InconsistentMapping;
                            return true;
                        case "BadRequest_SyntaxError": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_SyntaxError;
                            return true;
                        case "BadRequest_InvalidOrEmptyTableName": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidOrEmptyTableName;
                            return true;
                        case "BadRequest_EmptyDatabaseName": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyDatabaseName;
                            return true;
                        case "BadRequest_EmptyTenantIdentifier": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyTenantIdentifier;
                            return true;
                        case "BadRequest_EmptyMappingReference": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyMappingReference;
                            return true;
                        case "BadRequest_EntityNotFound": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EntityNotFound;
                            return true;
                        case "BadRequest_EntityNameIsNotValid": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EntityNameIsNotValid;
                            return true;
                        case "General_AbandonedIngestion": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.General_AbandonedIngestion;
                            return true;
                        case "General_ThrottledIngestion": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.General_ThrottledIngestion;
                            return true;
                        case "General_TransientSchemaMismatch": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.General_TransientSchemaMismatch;
                            return true;
                        case "BadRequest_DatabaseAccessDenied": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_DatabaseAccessDenied;
                            return true;
                        case "BadRequest_UnexpectedCharacterInInputStream": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_UnexpectedCharacterInInputStream;
                            return true;
                        case "Forbidden": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Forbidden;
                            return true;
                        case "BadRequest_ZeroRetentionPolicyWithNoUpdatePolicy": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_ZeroRetentionPolicyWithNoUpdatePolicy;
                            return true;
                        case "BadRequest_CreationTimeEarlierThanSoftDeletePeriod": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_CreationTimeEarlierThanSoftDeletePeriod;
                            return true;
                        case "BadRequest_NotSupported": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_NotSupported;
                            return true;
                        case "Skipped_IngestByTagAlreadyExists": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Skipped_IngestByTagAlreadyExists;
                            return true;
                        case "0x80131505": 
                        case "Timeout": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.Timeout;
                            return true;
                        case "BadRequest_MalformedIngestionProperty": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_MalformedIngestionProperty;
                            return true;
                        case "BadRequest_IngestionPropertyNotSupportedInThisContext": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_IngestionPropertyNotSupportedInThisContext;
                            return true;
                        case "BadRequest_InvalidAuthentication": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidAuthentication;
                            return true;
                        case "BlobAlreadyReceived_BlobAlreadyFoundInBatch": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BlobAlreadyReceived_BlobAlreadyFoundInBatch;
                            return true;
                        case "BadRequest_InvalidManagedIdentity": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidManagedIdentity;
                            return true;
                        case "BadRequest_DataCapacityLimitReached": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_DataCapacityLimitReached;
                            return true;
                        case "UpdatePolicy_BadRequest": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_BadRequest;
                            return true;
                        case "BadRequest_NonMultiDatabaseDataConnection": 
                            ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_NonMultiDatabaseDataConnection;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "unknown": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Unknown;
                                return true;
                            case "outofmemory": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.OutOfMemory;
                                return true;
                            case "stream_wrongnumberoffields": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Stream_WrongNumberOfFields;
                                return true;
                            case "stream_inputstreamtoolarge": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Stream_InputStreamTooLarge;
                                return true;
                            case "stream_nodatatoingest": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Stream_NoDataToIngest;
                                return true;
                            case "stream_closingquotemissing": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Stream_ClosingQuoteMissing;
                                return true;
                            case "stream_dynamicpropertybagtoolarge": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Stream_DynamicPropertyBagTooLarge;
                                return true;
                            case "download_sourcenotfound": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Download_SourceNotFound;
                                return true;
                            case "download_accessconditionnotsatisfied": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Download_AccessConditionNotSatisfied;
                                return true;
                            case "download_forbidden": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Download_Forbidden;
                                return true;
                            case "download_accountnotfound": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Download_AccountNotFound;
                                return true;
                            case "download_badrequest": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Download_BadRequest;
                                return true;
                            case "download_nottransient": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Download_NotTransient;
                                return true;
                            case "download_unknownerror": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Download_UnknownError;
                                return true;
                            case "updatepolicy_queryschemadoesnotmatchtableschema": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_QuerySchemaDoesNotMatchTableSchema;
                                return true;
                            case "updatepolicy_faileddescendanttransaction": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_FailedDescendantTransaction;
                                return true;
                            case "updatepolicy_ingestionerror": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_IngestionError;
                                return true;
                            case "updatepolicy_unknownerror": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_UnknownError;
                                return true;
                            case "badrequest_missingmappingfailure": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_MissingMappingFailure;
                                return true;
                            case "badrequest_invalidblob": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidBlob;
                                return true;
                            case "badrequest_emptyblob": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyBlob;
                                return true;
                            case "badrequest_norecordsorwrongformat": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_NoRecordsOrWrongFormat;
                                return true;
                            case "badrequest_emptyarchive": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyArchive;
                                return true;
                            case "badrequest_invalidarchive": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidArchive;
                                return true;
                            case "badrequest_invalidbloburi": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidBlobUri;
                                return true;
                            case "badrequest_emptybloburi": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyBlobUri;
                                return true;
                            case "badrequest_databasenotexist": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_DatabaseNotExist;
                                return true;
                            case "badrequest_tablenotexist": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_TableNotExist;
                                return true;
                            case "badrequest_invalidkustoidentitytoken": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidKustoIdentityToken;
                                return true;
                            case "badrequest_filetoolarge": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_FileTooLarge;
                                return true;
                            case "badrequest_tableaccessdenied": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_TableAccessDenied;
                                return true;
                            case "general_retryattemptsexceeded": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.General_RetryAttemptsExceeded;
                                return true;
                            case "general_badrequest": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.General_BadRequest;
                                return true;
                            case "general_internalservererror": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.General_InternalServerError;
                                return true;
                            case "updatepolicy_cyclic_update_not_allowed": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_Cyclic_Update_Not_Allowed;
                                return true;
                            case "badrequest_invalidmapping": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidMapping;
                                return true;
                            case "badrequest_invalidmappingreference": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidMappingReference;
                                return true;
                            case "badrequest_mappingreferencewasnotfound": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_MappingReferenceWasNotFound;
                                return true;
                            case "badrequest_formatnotsupported": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_FormatNotSupported;
                                return true;
                            case "badrequest_duplicatemapping": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_DuplicateMapping;
                                return true;
                            case "badrequest_corruptedmessage": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_CorruptedMessage;
                                return true;
                            case "badrequest_inconsistentmapping": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InconsistentMapping;
                                return true;
                            case "badrequest_syntaxerror": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_SyntaxError;
                                return true;
                            case "badrequest_invalidoremptytablename": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidOrEmptyTableName;
                                return true;
                            case "badrequest_emptydatabasename": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyDatabaseName;
                                return true;
                            case "badrequest_emptytenantidentifier": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyTenantIdentifier;
                                return true;
                            case "badrequest_emptymappingreference": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyMappingReference;
                                return true;
                            case "badrequest_entitynotfound": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EntityNotFound;
                                return true;
                            case "badrequest_entitynameisnotvalid": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_EntityNameIsNotValid;
                                return true;
                            case "general_abandonedingestion": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.General_AbandonedIngestion;
                                return true;
                            case "general_throttledingestion": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.General_ThrottledIngestion;
                                return true;
                            case "general_transientschemamismatch": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.General_TransientSchemaMismatch;
                                return true;
                            case "badrequest_databaseaccessdenied": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_DatabaseAccessDenied;
                                return true;
                            case "badrequest_unexpectedcharacterininputstream": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_UnexpectedCharacterInInputStream;
                                return true;
                            case "forbidden": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Forbidden;
                                return true;
                            case "badrequest_zeroretentionpolicywithnoupdatepolicy": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_ZeroRetentionPolicyWithNoUpdatePolicy;
                                return true;
                            case "badrequest_creationtimeearlierthansoftdeleteperiod": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_CreationTimeEarlierThanSoftDeletePeriod;
                                return true;
                            case "badrequest_notsupported": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_NotSupported;
                                return true;
                            case "skipped_ingestbytagalreadyexists": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Skipped_IngestByTagAlreadyExists;
                                return true;
                            case "timeout": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.Timeout;
                                return true;
                            case "badrequest_malformedingestionproperty": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_MalformedIngestionProperty;
                                return true;
                            case "badrequest_ingestionpropertynotsupportedinthiscontext": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_IngestionPropertyNotSupportedInThisContext;
                                return true;
                            case "badrequest_invalidauthentication": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidAuthentication;
                                return true;
                            case "blobalreadyreceived_blobalreadyfoundinbatch": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BlobAlreadyReceived_BlobAlreadyFoundInBatch;
                                return true;
                            case "badrequest_invalidmanagedidentity": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidManagedIdentity;
                                return true;
                            case "badrequest_datacapacitylimitreached": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_DataCapacityLimitReached;
                                return true;
                            case "updatepolicy_badrequest": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_BadRequest;
                                return true;
                            case "badrequest_nonmultidatabasedataconnection": 
                                ret.v = Kusto.Data.Common.IngestionErrorCode.BadRequest_NonMultiDatabaseDataConnection;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v >>> 0;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return (that | 0);
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.IngestionErrorCode.Unknown: 
                            return "Unknown error occurred";
                        case Kusto.Data.Common.IngestionErrorCode.OutOfMemory: 
                            return "Ingestion operation ran out of memory";
                        case Kusto.Data.Common.IngestionErrorCode.Stream_WrongNumberOfFields: 
                            return "Wrong number of fields.";
                        case Kusto.Data.Common.IngestionErrorCode.Stream_InputStreamTooLarge: 
                            return "Input stream/record/field too large.";
                        case Kusto.Data.Common.IngestionErrorCode.Stream_NoDataToIngest: 
                            return "No data streams to ingest";
                        case Kusto.Data.Common.IngestionErrorCode.Stream_ClosingQuoteMissing: 
                            return "Invalid csv format - closing quote missing.";
                        case Kusto.Data.Common.IngestionErrorCode.Stream_DynamicPropertyBagTooLarge: 
                            return "The combined size of the property bag's distinct keys is getting too large.";
                        case Kusto.Data.Common.IngestionErrorCode.Download_SourceNotFound: 
                            return "Failed to download source from Azure storage - source not found";
                        case Kusto.Data.Common.IngestionErrorCode.Download_AccessConditionNotSatisfied: 
                            return "Failed to download source from Azure storage - access condition not satisfied";
                        case Kusto.Data.Common.IngestionErrorCode.Download_Forbidden: 
                            return "Failed to download source from Azure storage - access forbidden";
                        case Kusto.Data.Common.IngestionErrorCode.Download_AccountNotFound: 
                            return "Failed to download source from Azure storage - account not found";
                        case Kusto.Data.Common.IngestionErrorCode.Download_BadRequest: 
                            return "Failed to download source from Azure storage - bad request";
                        case Kusto.Data.Common.IngestionErrorCode.Download_NotTransient: 
                            return "Failed to download source from Azure storage - not transient error";
                        case Kusto.Data.Common.IngestionErrorCode.Download_UnknownError: 
                            return "Failed to download source from Azure storage - unknown error";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_QuerySchemaDoesNotMatchTableSchema: 
                            return "Failed to invoke update policy. Query schema does not match table schema";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_FailedDescendantTransaction: 
                            return "Failed to invoke update policy. Failed descendant transactional update policy";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_IngestionError: 
                            return "Failed to invoke update policy. Ingestion Error occurred";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_UnknownError: 
                            return "Failed to invoke update policy. Unknown error occurred";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_MissingMappingFailure: 
                            return "A format that requires ingestion mapping was sent without the one. Identity ingestion mapping can be auto-created only when the ingestion format is specified";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidBlob: 
                            return "Blob is invalid";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyBlob: 
                            return "Blob is empty";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_NoRecordsOrWrongFormat: 
                            return "No records to ingest or using wrong data format";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyArchive: 
                            return "Archive is empty";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidArchive: 
                            return "Archive is invalid";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidBlobUri: 
                            return "Blob Uri is invalid";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyBlobUri: 
                            return "Blob Uri is empty";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DatabaseNotExist: 
                            return "Database does not exist";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_TableNotExist: 
                            return "Table does not exist";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidKustoIdentityToken: 
                            return "Invalid kusto identity token";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_FileTooLarge: 
                            return "File too large";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_TableAccessDenied: 
                            return "Access to table is denied";
                        case Kusto.Data.Common.IngestionErrorCode.General_RetryAttemptsExceeded: 
                            return "Operation has exceeded the retry attempts limit or timespan following a recurring transient error";
                        case Kusto.Data.Common.IngestionErrorCode.General_BadRequest: 
                            return "Bad request";
                        case Kusto.Data.Common.IngestionErrorCode.General_InternalServerError: 
                            return "Internal server error occurred";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_Cyclic_Update_Not_Allowed: 
                            return "Failed to invoke update policy. Cyclic update is not allowed";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidMapping: 
                            return "Failed to parse ingestion mapping.";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidMappingReference: 
                            return "Invalid mapping reference.";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_MappingReferenceWasNotFound: 
                            return "Mapping reference wasn't found.";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_FormatNotSupported: 
                            return "Format is not supported";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DuplicateMapping: 
                            return "Ingestion properties contains ingestion mapping and ingestion mapping reference.";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_CorruptedMessage: 
                            return "Message is corrupted";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InconsistentMapping: 
                            return "Inconsistent ingestion mapping";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_SyntaxError: 
                            return "Syntax error";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidOrEmptyTableName: 
                            return "Table name is empty or invalid";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyDatabaseName: 
                            return "Database name is empty";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyTenantIdentifier: 
                            return "TenantIdentifier is empty";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EmptyMappingReference: 
                            return "MappingReference is empty";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EntityNotFound: 
                            return "Failed to locate an entity";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_EntityNameIsNotValid: 
                            return "Entity name is not valid";
                        case Kusto.Data.Common.IngestionErrorCode.General_AbandonedIngestion: 
                            return "Abandoned ingestion.";
                        case Kusto.Data.Common.IngestionErrorCode.General_ThrottledIngestion: 
                            return "Throttled ingestion.";
                        case Kusto.Data.Common.IngestionErrorCode.General_TransientSchemaMismatch: 
                            return "Schema of target table at start time doesn't match the one at commit time.";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DatabaseAccessDenied: 
                            return "Access to database is denied";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_UnexpectedCharacterInInputStream: 
                            return "Unexpected character in the input stream";
                        case Kusto.Data.Common.IngestionErrorCode.Forbidden: 
                            return "Insufficient security permissions to execute request";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_ZeroRetentionPolicyWithNoUpdatePolicy: 
                            return "Table has zero rentention policy and is not the source table for any update policy";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_CreationTimeEarlierThanSoftDeletePeriod: 
                            return "Specified creation time is earlier than table's soft delete period";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_NotSupported: 
                            return "Request not supported";
                        case Kusto.Data.Common.IngestionErrorCode.Skipped_IngestByTagAlreadyExists: 
                            return "Another stream with the same ingest-by tag was already ingested";
                        case Kusto.Data.Common.IngestionErrorCode.Timeout: 
                            return "Ingestion operation timed out";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_MalformedIngestionProperty: 
                            return "Ingestion property is malformed";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_IngestionPropertyNotSupportedInThisContext: 
                            return "Ingestion property is not supported in this context";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidAuthentication: 
                            return "Authentication to data is not valid";
                        case Kusto.Data.Common.IngestionErrorCode.BlobAlreadyReceived_BlobAlreadyFoundInBatch: 
                            return "One copy of this blob has already been processed. The service will only attempt to ingest one copy of the blob";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_InvalidManagedIdentity: 
                            return "Managed Identity is not defined, not allowed by a policy, or doesn't have permissions on the resource";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_DataCapacityLimitReached: 
                            return "Raised to indicate that storage data capacity has reached its limit";
                        case Kusto.Data.Common.IngestionErrorCode.UpdatePolicy_BadRequest: 
                            return "Failed to invoke update policy as a result of a bad request, e.g. query semantic error";
                        case Kusto.Data.Common.IngestionErrorCode.BadRequest_NonMultiDatabaseDataConnection: 
                            return "Event contains the 'database' property directing the ingestion into a database different than the target database configured in the data connection's settings. Since this data connection is set to multiDatabase = false, dynamic database routing is forbidden.";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * Extensions for enum StorageEngineKind
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Common.ExtendedStorageEngineKind
     */
    Bridge.define("Kusto.Data.Common.ExtendedStorageEngineKind", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.StorageEngineKind.StorageV2: 
                            return "StorageV2";
                        case Kusto.Data.Common.StorageEngineKind.StorageV3: 
                            return "StorageV3";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.Common.StorageEngineKind() };
                    return Kusto.Data.Common.ExtendedStorageEngineKind.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.Common.StorageEngineKind.StorageV2: 
                            return true;
                        case Kusto.Data.Common.StorageEngineKind.StorageV3: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.Common.StorageEngineKind() };
                    if (Kusto.Data.Common.ExtendedStorageEngineKind.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "0": 
                        case "StorageV2": 
                            ret.v = Kusto.Data.Common.StorageEngineKind.StorageV2;
                            return true;
                        case "1": 
                        case "StorageV3": 
                            ret.v = Kusto.Data.Common.StorageEngineKind.StorageV3;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "storagev2": 
                                ret.v = Kusto.Data.Common.StorageEngineKind.StorageV2;
                                return true;
                            case "storagev3": 
                                ret.v = Kusto.Data.Common.StorageEngineKind.StorageV3;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.StorageEngineKind.StorageV2: 
                            return "V2 - extents data storage engine";
                        case Kusto.Data.Common.StorageEngineKind.StorageV3: 
                            return "V3 - shard data storage engine";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * Extensions for enum TransformationMethod
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Common.ExtendedTransformationMethod
     */
    Bridge.define("Kusto.Data.Common.ExtendedTransformationMethod", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.TransformationMethod.None: 
                            return "None";
                        case Kusto.Data.Common.TransformationMethod.PropertyBagArrayToDictionary: 
                            return "PropertyBagArrayToDictionary";
                        case Kusto.Data.Common.TransformationMethod.SourceLocation: 
                            return "SourceLocation";
                        case Kusto.Data.Common.TransformationMethod.SourceLineNumber: 
                            return "SourceLineNumber";
                        case Kusto.Data.Common.TransformationMethod.UnknownMethod1: 
                            return "UnknownMethod1";
                        case Kusto.Data.Common.TransformationMethod.UnknownMethod: 
                            return "UnknownMethod";
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixSeconds: 
                            return "DateTimeFromUnixSeconds";
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixMilliseconds: 
                            return "DateTimeFromUnixMilliseconds";
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixMicroseconds: 
                            return "DateTimeFromUnixMicroseconds";
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixNanoseconds: 
                            return "DateTimeFromUnixNanoseconds";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.Common.TransformationMethod() };
                    return Kusto.Data.Common.ExtendedTransformationMethod.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.Common.TransformationMethod.None: 
                            return true;
                        case Kusto.Data.Common.TransformationMethod.PropertyBagArrayToDictionary: 
                            return true;
                        case Kusto.Data.Common.TransformationMethod.SourceLocation: 
                            return true;
                        case Kusto.Data.Common.TransformationMethod.SourceLineNumber: 
                            return true;
                        case Kusto.Data.Common.TransformationMethod.UnknownMethod1: 
                            return true;
                        case Kusto.Data.Common.TransformationMethod.UnknownMethod: 
                            return true;
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixSeconds: 
                            return true;
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixMilliseconds: 
                            return true;
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixMicroseconds: 
                            return true;
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixNanoseconds: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.Common.TransformationMethod() };
                    if (Kusto.Data.Common.ExtendedTransformationMethod.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "0": 
                        case "None": 
                            ret.v = Kusto.Data.Common.TransformationMethod.None;
                            return true;
                        case "1": 
                        case "PropertyBagArrayToDictionary": 
                            ret.v = Kusto.Data.Common.TransformationMethod.PropertyBagArrayToDictionary;
                            return true;
                        case "2": 
                        case "SourceLocation": 
                            ret.v = Kusto.Data.Common.TransformationMethod.SourceLocation;
                            return true;
                        case "3": 
                        case "SourceLineNumber": 
                            ret.v = Kusto.Data.Common.TransformationMethod.SourceLineNumber;
                            return true;
                        case "4": 
                        case "UnknownMethod1": 
                            ret.v = Kusto.Data.Common.TransformationMethod.UnknownMethod1;
                            return true;
                        case "5": 
                        case "UnknownMethod": 
                            ret.v = Kusto.Data.Common.TransformationMethod.UnknownMethod;
                            return true;
                        case "6": 
                        case "DateTimeFromUnixSeconds": 
                            ret.v = Kusto.Data.Common.TransformationMethod.DateTimeFromUnixSeconds;
                            return true;
                        case "7": 
                        case "DateTimeFromUnixMilliseconds": 
                            ret.v = Kusto.Data.Common.TransformationMethod.DateTimeFromUnixMilliseconds;
                            return true;
                        case "8": 
                        case "DateTimeFromUnixMicroseconds": 
                            ret.v = Kusto.Data.Common.TransformationMethod.DateTimeFromUnixMicroseconds;
                            return true;
                        case "9": 
                        case "DateTimeFromUnixNanoseconds": 
                            ret.v = Kusto.Data.Common.TransformationMethod.DateTimeFromUnixNanoseconds;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "none": 
                                ret.v = Kusto.Data.Common.TransformationMethod.None;
                                return true;
                            case "propertybagarraytodictionary": 
                                ret.v = Kusto.Data.Common.TransformationMethod.PropertyBagArrayToDictionary;
                                return true;
                            case "sourcelocation": 
                                ret.v = Kusto.Data.Common.TransformationMethod.SourceLocation;
                                return true;
                            case "sourcelinenumber": 
                                ret.v = Kusto.Data.Common.TransformationMethod.SourceLineNumber;
                                return true;
                            case "unknownmethod1": 
                                ret.v = Kusto.Data.Common.TransformationMethod.UnknownMethod1;
                                return true;
                            case "unknownmethod": 
                                ret.v = Kusto.Data.Common.TransformationMethod.UnknownMethod;
                                return true;
                            case "datetimefromunixseconds": 
                                ret.v = Kusto.Data.Common.TransformationMethod.DateTimeFromUnixSeconds;
                                return true;
                            case "datetimefromunixmilliseconds": 
                                ret.v = Kusto.Data.Common.TransformationMethod.DateTimeFromUnixMilliseconds;
                                return true;
                            case "datetimefromunixmicroseconds": 
                                ret.v = Kusto.Data.Common.TransformationMethod.DateTimeFromUnixMicroseconds;
                                return true;
                            case "datetimefromunixnanoseconds": 
                                ret.v = Kusto.Data.Common.TransformationMethod.DateTimeFromUnixNanoseconds;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.Common.TransformationMethod.None: 
                            return "Comma-separated value.";
                        case Kusto.Data.Common.TransformationMethod.PropertyBagArrayToDictionary: 
                            return "Property bag array to dictionary.";
                        case Kusto.Data.Common.TransformationMethod.SourceLocation: 
                            return "Source location.";
                        case Kusto.Data.Common.TransformationMethod.SourceLineNumber: 
                            return "Source line number.";
                        case Kusto.Data.Common.TransformationMethod.UnknownMethod1: 
                            return "Was GetPathElement";
                        case Kusto.Data.Common.TransformationMethod.UnknownMethod: 
                            return "Unknown method.";
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixSeconds: 
                            return "Converts UNIX epoch (seconds) to UTC datetime.";
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixMilliseconds: 
                            return "Converts UNIX epoch (milliseconds) to UTC datetime.";
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixMicroseconds: 
                            return "Converts UNIX epoch (microseconds) to UTC datetime.";
                        case Kusto.Data.Common.TransformationMethod.DateTimeFromUnixNanoseconds: 
                            return "Converts UNIX epoch (nanoseconds) to UTC datetime.";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * The kind of lookback period to use when choosing which extents to merge/rebuild
     *
     * @public
     * @class Kusto.Data.Common.ExtentsMergeLookbackPeriodKind
     */
    Bridge.define("Kusto.Data.Common.ExtentsMergeLookbackPeriodKind", {
        $kind: "enum",
        statics: {
            fields: {
                /**
                 * The default lookback period, as defined by the system
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.ExtentsMergeLookbackPeriodKind
                 * @constant
                 * @default 0
                 * @type Kusto.Data.Common.ExtentsMergeLookbackPeriodKind
                 */
                Default: 0,
                /**
                 * The hot cache period
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.ExtentsMergeLookbackPeriodKind
                 * @constant
                 * @default 1
                 * @type Kusto.Data.Common.ExtentsMergeLookbackPeriodKind
                 */
                HotCache: 1,
                /**
                 * The entire period in which data exists
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.ExtentsMergeLookbackPeriodKind
                 * @constant
                 * @default 2
                 * @type Kusto.Data.Common.ExtentsMergeLookbackPeriodKind
                 */
                All: 2,
                /**
                 * A custom period, provided by the admin of the entity
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.ExtentsMergeLookbackPeriodKind
                 * @constant
                 * @default 3
                 * @type Kusto.Data.Common.ExtentsMergeLookbackPeriodKind
                 */
                Custom: 3
            }
        }
    });

    /**
     * The IngestionErrorCode Category.
     *
     * @public
     * @class Kusto.Data.Common.IngestionErrorCategory
     */
    Bridge.define("Kusto.Data.Common.IngestionErrorCategory", {
        $kind: "enum",
        statics: {
            fields: {
                Unknown: 0,
                InternalServiceError: 1,
                BadFormat: 2,
                BadRequest: 3,
                UpdatePolicyFailure: 4,
                EntityNotFound: 5,
                UserAccessNotAuthorized: 6,
                DataAccessNotAuthorized: 7,
                FileTooLarge: 8,
                ThrottledOnEngine: 9,
                DownloadFailed: 10,
                RetryAttemptsExceeded: 11,
                BlobAlreadyReceived: 12
            }
        }
    });

    /**
     * The ingestion error code returned by a Kusto service. Note that this list is documented in Kusto external docs, so when changing the ingestion error codes, the documentation should also be updated.
     *
     * @public
     * @class Kusto.Data.Common.IngestionErrorCode
     * @augments number
     */
    Bridge.define("Kusto.Data.Common.IngestionErrorCode", {
        $kind: "enum",
        statics: {
            fields: {
                /**
                 * Unknown error occurred
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 0
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Unknown: 0,
                /**
                 * Ingestion operation ran out of memory
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770503
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                OutOfMemory: 2161770503,
                /**
                 * Wrong number of fields.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770504
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Stream_WrongNumberOfFields: 2161770504,
                /**
                 * Input stream/record/field too large.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770505
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Stream_InputStreamTooLarge: 2161770505,
                /**
                 * No data streams to ingest
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770506
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Stream_NoDataToIngest: 2161770506,
                /**
                 * Invalid csv format - closing quote missing.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770507
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Stream_ClosingQuoteMissing: 2161770507,
                /**
                 * The combined size of the property bag's distinct keys is getting too large.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770510
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Stream_DynamicPropertyBagTooLarge: 2161770510,
                /**
                 * Failed to download source from Azure storage - source not found
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770511
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Download_SourceNotFound: 2161770511,
                /**
                 * Failed to download source from Azure storage - access condition not satisfied
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770512
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Download_AccessConditionNotSatisfied: 2161770512,
                /**
                 * Failed to download source from Azure storage - access forbidden
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770513
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Download_Forbidden: 2161770513,
                /**
                 * Failed to download source from Azure storage - account not found
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770514
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Download_AccountNotFound: 2161770514,
                /**
                 * Failed to download source from Azure storage - bad request
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770515
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Download_BadRequest: 2161770515,
                /**
                 * Failed to download source from Azure storage - not transient error
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770516
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Download_NotTransient: 2161770516,
                /**
                 * Failed to download source from Azure storage - unknown error
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770517
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Download_UnknownError: 2161770517,
                /**
                 * Failed to invoke update policy. Query schema does not match table schema
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770518
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                UpdatePolicy_QuerySchemaDoesNotMatchTableSchema: 2161770518,
                /**
                 * Failed to invoke update policy. Failed descendant transactional update policy
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770519
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                UpdatePolicy_FailedDescendantTransaction: 2161770519,
                /**
                 * Failed to invoke update policy. Ingestion Error occurred
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770520
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                UpdatePolicy_IngestionError: 2161770520,
                /**
                 * Failed to invoke update policy. Unknown error occurred
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770521
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                UpdatePolicy_UnknownError: 2161770521,
                /**
                 * A format that requires ingestion mapping was sent without the one. Identity ingestion mapping can be auto-created only when the ingestion format is specified
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770522
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_MissingMappingFailure: 2161770522,
                /**
                 * Blob is invalid
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770523
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_InvalidBlob: 2161770523,
                /**
                 * Blob is empty
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770524
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_EmptyBlob: 2161770524,
                /**
                 * No records to ingest or using wrong data format
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770525
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_NoRecordsOrWrongFormat: 2161770525,
                /**
                 * Archive is empty
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770526
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_EmptyArchive: 2161770526,
                /**
                 * Archive is invalid
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770527
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_InvalidArchive: 2161770527,
                /**
                 * Blob Uri is invalid
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770528
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_InvalidBlobUri: 2161770528,
                /**
                 * Blob Uri is empty
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770529
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_EmptyBlobUri: 2161770529,
                /**
                 * Database does not exist
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770530
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_DatabaseNotExist: 2161770530,
                /**
                 * Table does not exist
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770531
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_TableNotExist: 2161770531,
                /**
                 * Invalid kusto identity token
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770532
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_InvalidKustoIdentityToken: 2161770532,
                /**
                 * File too large
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770533
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_FileTooLarge: 2161770533,
                /**
                 * Access to table is denied
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770534
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_TableAccessDenied: 2161770534,
                /**
                 * Operation has exceeded the retry attempts limit or timespan following a recurring transient error
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770535
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                General_RetryAttemptsExceeded: 2161770535,
                /**
                 * Bad request
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770536
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                General_BadRequest: 2161770536,
                /**
                 * Internal server error occurred
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770537
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                General_InternalServerError: 2161770537,
                /**
                 * Failed to invoke update policy. Cyclic update is not allowed
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770538
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                UpdatePolicy_Cyclic_Update_Not_Allowed: 2161770538,
                /**
                 * Failed to parse ingestion mapping.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770539
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_InvalidMapping: 2161770539,
                /**
                 * Invalid mapping reference.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770540
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_InvalidMappingReference: 2161770540,
                /**
                 * Mapping reference wasn't found.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770541
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_MappingReferenceWasNotFound: 2161770541,
                /**
                 * Format is not supported
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770542
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_FormatNotSupported: 2161770542,
                /**
                 * Ingestion properties contains ingestion mapping and ingestion mapping reference.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770543
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_DuplicateMapping: 2161770543,
                /**
                 * Message is corrupted
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770544
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_CorruptedMessage: 2161770544,
                /**
                 * Inconsistent ingestion mapping
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770545
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_InconsistentMapping: 2161770545,
                /**
                 * Syntax error
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770546
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_SyntaxError: 2161770546,
                /**
                 * Table name is empty or invalid
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770547
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_InvalidOrEmptyTableName: 2161770547,
                /**
                 * Database name is empty
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770548
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_EmptyDatabaseName: 2161770548,
                /**
                 * TenantIdentifier is empty
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770549
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_EmptyTenantIdentifier: 2161770549,
                /**
                 * MappingReference is empty
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770550
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_EmptyMappingReference: 2161770550,
                /**
                 * Failed to locate an entity
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770551
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_EntityNotFound: 2161770551,
                /**
                 * Entity name is not valid
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770552
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_EntityNameIsNotValid: 2161770552,
                /**
                 * Abandoned ingestion.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770553
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                General_AbandonedIngestion: 2161770553,
                /**
                 * Throttled ingestion.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770554
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                General_ThrottledIngestion: 2161770554,
                /**
                 * Schema of target table at start time doesn't match the one at commit time.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770555
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                General_TransientSchemaMismatch: 2161770555,
                /**
                 * Access to database is denied
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770556
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_DatabaseAccessDenied: 2161770556,
                /**
                 * Unexpected character in the input stream
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770557
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_UnexpectedCharacterInInputStream: 2161770557,
                /**
                 * Insufficient security permissions to execute request
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770558
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Forbidden: 2161770558,
                /**
                 * Table has zero rentention policy and is not the source table for any update policy
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770559
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_ZeroRetentionPolicyWithNoUpdatePolicy: 2161770559,
                /**
                 * Specified creation time is earlier than table's soft delete period
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770560
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_CreationTimeEarlierThanSoftDeletePeriod: 2161770560,
                /**
                 * Request not supported
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770561
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_NotSupported: 2161770561,
                /**
                 * Another stream with the same ingest-by tag was already ingested
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2161770562
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Skipped_IngestByTagAlreadyExists: 2161770562,
                /**
                 * Ingestion operation timed out
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2148734213
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                Timeout: 2148734213,
                /**
                 * Ingestion property is malformed
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2148734214
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_MalformedIngestionProperty: 2148734214,
                /**
                 * Ingestion property is not supported in this context
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2148734215
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_IngestionPropertyNotSupportedInThisContext: 2148734215,
                /**
                 * Authentication to data is not valid
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2148734216
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_InvalidAuthentication: 2148734216,
                /**
                 * One copy of this blob has already been processed. The service will only attempt to ingest one copy of the blob
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2148734217
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BlobAlreadyReceived_BlobAlreadyFoundInBatch: 2148734217,
                /**
                 * Managed Identity is not defined, not allowed by a policy, or doesn't have permissions on the resource
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2148734218
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_InvalidManagedIdentity: 2148734218,
                /**
                 * Raised to indicate that storage data capacity has reached its limit
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2148734219
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_DataCapacityLimitReached: 2148734219,
                /**
                 * Failed to invoke update policy as a result of a bad request, e.g. query semantic error
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2148734220
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                UpdatePolicy_BadRequest: 2148734220,
                /**
                 * Event contains the 'database' property directing the ingestion into a database different than the target database configured in the data connection's settings. Since this data connection is set to multiDatabase = false, dynamic database routing is forbidden.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.IngestionErrorCode
                 * @constant
                 * @default 2148734221
                 * @type Kusto.Data.Common.IngestionErrorCode
                 */
                BadRequest_NonMultiDatabaseDataConnection: 2148734221
            }
        },
        $utype: System.UInt32
    });

    /**
     * The kind of storage engine used for ingestion flows
     *
     * @public
     * @class Kusto.Data.Common.StorageEngineKind
     */
    Bridge.define("Kusto.Data.Common.StorageEngineKind", {
        $kind: "enum",
        statics: {
            fields: {
                /**
                 * V2 - extents data storage engine
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.StorageEngineKind
                 * @constant
                 * @default 0
                 * @type Kusto.Data.Common.StorageEngineKind
                 */
                StorageV2: 0,
                /**
                 * V3 - shard data storage engine
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.StorageEngineKind
                 * @constant
                 * @default 1
                 * @type Kusto.Data.Common.StorageEngineKind
                 */
                StorageV3: 1
            }
        }
    });

    /**
     * Transformation methods (NOTE: This type is mirrored by the type CsvFromJsonStream_TransformationMethod (defined in UtilsEnums.tt). Make sure that the two are always mirrors of each other.)
     *
     * @public
     * @class Kusto.Data.Common.TransformationMethod
     */
    Bridge.define("Kusto.Data.Common.TransformationMethod", {
        $kind: "enum",
        statics: {
            fields: {
                /**
                 * Comma-separated value.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.TransformationMethod
                 * @constant
                 * @default 0
                 * @type Kusto.Data.Common.TransformationMethod
                 */
                None: 0,
                /**
                 * Property bag array to dictionary.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.TransformationMethod
                 * @constant
                 * @default 1
                 * @type Kusto.Data.Common.TransformationMethod
                 */
                PropertyBagArrayToDictionary: 1,
                /**
                 * Source location.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.TransformationMethod
                 * @constant
                 * @default 2
                 * @type Kusto.Data.Common.TransformationMethod
                 */
                SourceLocation: 2,
                /**
                 * Source line number.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.TransformationMethod
                 * @constant
                 * @default 3
                 * @type Kusto.Data.Common.TransformationMethod
                 */
                SourceLineNumber: 3,
                /**
                 * Was GetPathElement
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.TransformationMethod
                 * @constant
                 * @default 4
                 * @type Kusto.Data.Common.TransformationMethod
                 */
                UnknownMethod1: 4,
                /**
                 * Unknown method.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.TransformationMethod
                 * @constant
                 * @default 5
                 * @type Kusto.Data.Common.TransformationMethod
                 */
                UnknownMethod: 5,
                /**
                 * Converts UNIX epoch (seconds) to UTC datetime.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.TransformationMethod
                 * @constant
                 * @default 6
                 * @type Kusto.Data.Common.TransformationMethod
                 */
                DateTimeFromUnixSeconds: 6,
                /**
                 * Converts UNIX epoch (milliseconds) to UTC datetime.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.TransformationMethod
                 * @constant
                 * @default 7
                 * @type Kusto.Data.Common.TransformationMethod
                 */
                DateTimeFromUnixMilliseconds: 7,
                /**
                 * Converts UNIX epoch (microseconds) to UTC datetime.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.TransformationMethod
                 * @constant
                 * @default 8
                 * @type Kusto.Data.Common.TransformationMethod
                 */
                DateTimeFromUnixMicroseconds: 8,
                /**
                 * Converts UNIX epoch (nanoseconds) to UTC datetime.
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Common.TransformationMethod
                 * @constant
                 * @default 9
                 * @type Kusto.Data.Common.TransformationMethod
                 */
                DateTimeFromUnixNanoseconds: 9
            }
        }
    });

    /** @namespace Kusto.Data */

    /**
     * DM Settings Types
     *
     * @public
     * @class Kusto.Data.DmSettingsType
     */
    Bridge.define("Kusto.Data.DmSettingsType", {
        $kind: "enum",
        statics: {
            fields: {
                AdminSettings: 0,
                DataObtainerServiceSettings: 1,
                DataAggregatorSettings: 2,
                DataObtainerSettings: 3,
                CloudManagersSettings: 5,
                SchemaManagmentSettings: 6,
                CommonSettings: 7,
                DataImporterSettings: 8,
                StatusReporterSettings: 9,
                All: 10,
                VirtualDmsSettings: 11
            }
        }
    });

    /**
     * Extensions for enum DmSettingsType
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.ExtendedDmSettingsType
     */
    Bridge.define("Kusto.Data.ExtendedDmSettingsType", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.DmSettingsType.AdminSettings: 
                            return "AdminSettings";
                        case Kusto.Data.DmSettingsType.DataObtainerServiceSettings: 
                            return "DataObtainerServiceSettings";
                        case Kusto.Data.DmSettingsType.DataAggregatorSettings: 
                            return "DataAggregatorSettings";
                        case Kusto.Data.DmSettingsType.DataObtainerSettings: 
                            return "DataObtainerSettings";
                        case Kusto.Data.DmSettingsType.CloudManagersSettings: 
                            return "CloudManagersSettings";
                        case Kusto.Data.DmSettingsType.SchemaManagmentSettings: 
                            return "SchemaManagmentSettings";
                        case Kusto.Data.DmSettingsType.CommonSettings: 
                            return "CommonSettings";
                        case Kusto.Data.DmSettingsType.DataImporterSettings: 
                            return "DataImporterSettings";
                        case Kusto.Data.DmSettingsType.StatusReporterSettings: 
                            return "StatusReporterSettings";
                        case Kusto.Data.DmSettingsType.All: 
                            return "All";
                        case Kusto.Data.DmSettingsType.VirtualDmsSettings: 
                            return "VirtualDmsSettings";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.DmSettingsType() };
                    return Kusto.Data.ExtendedDmSettingsType.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.DmSettingsType.AdminSettings: 
                            return true;
                        case Kusto.Data.DmSettingsType.DataObtainerServiceSettings: 
                            return true;
                        case Kusto.Data.DmSettingsType.DataAggregatorSettings: 
                            return true;
                        case Kusto.Data.DmSettingsType.DataObtainerSettings: 
                            return true;
                        case Kusto.Data.DmSettingsType.CloudManagersSettings: 
                            return true;
                        case Kusto.Data.DmSettingsType.SchemaManagmentSettings: 
                            return true;
                        case Kusto.Data.DmSettingsType.CommonSettings: 
                            return true;
                        case Kusto.Data.DmSettingsType.DataImporterSettings: 
                            return true;
                        case Kusto.Data.DmSettingsType.StatusReporterSettings: 
                            return true;
                        case Kusto.Data.DmSettingsType.All: 
                            return true;
                        case Kusto.Data.DmSettingsType.VirtualDmsSettings: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.DmSettingsType() };
                    if (Kusto.Data.ExtendedDmSettingsType.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "0": 
                        case "AdminSettings": 
                            ret.v = Kusto.Data.DmSettingsType.AdminSettings;
                            return true;
                        case "1": 
                        case "DataObtainerServiceSettings": 
                            ret.v = Kusto.Data.DmSettingsType.DataObtainerServiceSettings;
                            return true;
                        case "2": 
                        case "DataAggregatorSettings": 
                            ret.v = Kusto.Data.DmSettingsType.DataAggregatorSettings;
                            return true;
                        case "3": 
                        case "DataObtainerSettings": 
                            ret.v = Kusto.Data.DmSettingsType.DataObtainerSettings;
                            return true;
                        case "5": 
                        case "CloudManagersSettings": 
                            ret.v = Kusto.Data.DmSettingsType.CloudManagersSettings;
                            return true;
                        case "6": 
                        case "SchemaManagmentSettings": 
                            ret.v = Kusto.Data.DmSettingsType.SchemaManagmentSettings;
                            return true;
                        case "7": 
                        case "CommonSettings": 
                            ret.v = Kusto.Data.DmSettingsType.CommonSettings;
                            return true;
                        case "8": 
                        case "DataImporterSettings": 
                            ret.v = Kusto.Data.DmSettingsType.DataImporterSettings;
                            return true;
                        case "9": 
                        case "StatusReporterSettings": 
                            ret.v = Kusto.Data.DmSettingsType.StatusReporterSettings;
                            return true;
                        case "10": 
                        case "All": 
                            ret.v = Kusto.Data.DmSettingsType.All;
                            return true;
                        case "11": 
                        case "VirtualDmsSettings": 
                            ret.v = Kusto.Data.DmSettingsType.VirtualDmsSettings;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "adminsettings": 
                                ret.v = Kusto.Data.DmSettingsType.AdminSettings;
                                return true;
                            case "dataobtainerservicesettings": 
                                ret.v = Kusto.Data.DmSettingsType.DataObtainerServiceSettings;
                                return true;
                            case "dataaggregatorsettings": 
                                ret.v = Kusto.Data.DmSettingsType.DataAggregatorSettings;
                                return true;
                            case "dataobtainersettings": 
                                ret.v = Kusto.Data.DmSettingsType.DataObtainerSettings;
                                return true;
                            case "cloudmanagerssettings": 
                                ret.v = Kusto.Data.DmSettingsType.CloudManagersSettings;
                                return true;
                            case "schemamanagmentsettings": 
                                ret.v = Kusto.Data.DmSettingsType.SchemaManagmentSettings;
                                return true;
                            case "commonsettings": 
                                ret.v = Kusto.Data.DmSettingsType.CommonSettings;
                                return true;
                            case "dataimportersettings": 
                                ret.v = Kusto.Data.DmSettingsType.DataImporterSettings;
                                return true;
                            case "statusreportersettings": 
                                ret.v = Kusto.Data.DmSettingsType.StatusReporterSettings;
                                return true;
                            case "all": 
                                ret.v = Kusto.Data.DmSettingsType.All;
                                return true;
                            case "virtualdmssettings": 
                                ret.v = Kusto.Data.DmSettingsType.VirtualDmsSettings;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.DmSettingsType.AdminSettings: 
                            return "";
                        case Kusto.Data.DmSettingsType.DataObtainerServiceSettings: 
                            return "";
                        case Kusto.Data.DmSettingsType.DataAggregatorSettings: 
                            return "";
                        case Kusto.Data.DmSettingsType.DataObtainerSettings: 
                            return "";
                        case Kusto.Data.DmSettingsType.CloudManagersSettings: 
                            return "";
                        case Kusto.Data.DmSettingsType.SchemaManagmentSettings: 
                            return "";
                        case Kusto.Data.DmSettingsType.CommonSettings: 
                            return "";
                        case Kusto.Data.DmSettingsType.DataImporterSettings: 
                            return "";
                        case Kusto.Data.DmSettingsType.StatusReporterSettings: 
                            return "";
                        case Kusto.Data.DmSettingsType.All: 
                            return "";
                        case Kusto.Data.DmSettingsType.VirtualDmsSettings: 
                            return "";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * Extensions for enum IngestionQueueType
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.ExtendedIngestionQueueType
     */
    Bridge.define("Kusto.Data.ExtendedIngestionQueueType", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.IngestionQueueType.All: 
                            return "All";
                        case Kusto.Data.IngestionQueueType.InputQueue: 
                            return "InputQueue";
                        case Kusto.Data.IngestionQueueType.InternalQueue: 
                            return "InternalQueue";
                        case Kusto.Data.IngestionQueueType.ReadyForAggregationQueue: 
                            return "ReadyForAggregationQueue";
                        case Kusto.Data.IngestionQueueType.SecuredReadyForAggregationQueue: 
                            return "SecuredReadyForAggregationQueue";
                        case Kusto.Data.IngestionQueueType.AggregatedReadyForImportQueue: 
                            return "AggregatedReadyForImportQueue";
                        case Kusto.Data.IngestionQueueType.FailedIngestionsQueue: 
                            return "FailedIngestionsQueue";
                        case Kusto.Data.IngestionQueueType.SuccessfulIngestionsQueue: 
                            return "SuccessfulIngestionsQueue";
                        case Kusto.Data.IngestionQueueType.AbandonedIngestionsQueue: 
                            return "AbandonedIngestionsQueue";
                        case Kusto.Data.IngestionQueueType.SecuredAbandonedIngestionsQueue: 
                            return "SecuredAbandonedIngestionsQueue";
                        case Kusto.Data.IngestionQueueType.InternalFailedIngestionsQueue: 
                            return "InternalFailedIngestionsQueue";
                        case Kusto.Data.IngestionQueueType.AbandonedInputQueues: 
                            return "AbandonedInputQueues";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.IngestionQueueType() };
                    return Kusto.Data.ExtendedIngestionQueueType.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.IngestionQueueType.All: 
                            return true;
                        case Kusto.Data.IngestionQueueType.InputQueue: 
                            return true;
                        case Kusto.Data.IngestionQueueType.InternalQueue: 
                            return true;
                        case Kusto.Data.IngestionQueueType.ReadyForAggregationQueue: 
                            return true;
                        case Kusto.Data.IngestionQueueType.SecuredReadyForAggregationQueue: 
                            return true;
                        case Kusto.Data.IngestionQueueType.AggregatedReadyForImportQueue: 
                            return true;
                        case Kusto.Data.IngestionQueueType.FailedIngestionsQueue: 
                            return true;
                        case Kusto.Data.IngestionQueueType.SuccessfulIngestionsQueue: 
                            return true;
                        case Kusto.Data.IngestionQueueType.AbandonedIngestionsQueue: 
                            return true;
                        case Kusto.Data.IngestionQueueType.SecuredAbandonedIngestionsQueue: 
                            return true;
                        case Kusto.Data.IngestionQueueType.InternalFailedIngestionsQueue: 
                            return true;
                        case Kusto.Data.IngestionQueueType.AbandonedInputQueues: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.IngestionQueueType() };
                    if (Kusto.Data.ExtendedIngestionQueueType.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "0": 
                        case "All": 
                            ret.v = Kusto.Data.IngestionQueueType.All;
                            return true;
                        case "1": 
                        case "InputQueue": 
                            ret.v = Kusto.Data.IngestionQueueType.InputQueue;
                            return true;
                        case "2": 
                        case "InternalQueue": 
                            ret.v = Kusto.Data.IngestionQueueType.InternalQueue;
                            return true;
                        case "3": 
                        case "ReadyForAggregationQueue": 
                            ret.v = Kusto.Data.IngestionQueueType.ReadyForAggregationQueue;
                            return true;
                        case "4": 
                        case "SecuredReadyForAggregationQueue": 
                            ret.v = Kusto.Data.IngestionQueueType.SecuredReadyForAggregationQueue;
                            return true;
                        case "5": 
                        case "AggregatedReadyForImportQueue": 
                            ret.v = Kusto.Data.IngestionQueueType.AggregatedReadyForImportQueue;
                            return true;
                        case "6": 
                        case "FailedIngestionsQueue": 
                            ret.v = Kusto.Data.IngestionQueueType.FailedIngestionsQueue;
                            return true;
                        case "7": 
                        case "SuccessfulIngestionsQueue": 
                            ret.v = Kusto.Data.IngestionQueueType.SuccessfulIngestionsQueue;
                            return true;
                        case "9": 
                        case "AbandonedIngestionsQueue": 
                            ret.v = Kusto.Data.IngestionQueueType.AbandonedIngestionsQueue;
                            return true;
                        case "10": 
                        case "SecuredAbandonedIngestionsQueue": 
                            ret.v = Kusto.Data.IngestionQueueType.SecuredAbandonedIngestionsQueue;
                            return true;
                        case "11": 
                        case "InternalFailedIngestionsQueue": 
                            ret.v = Kusto.Data.IngestionQueueType.InternalFailedIngestionsQueue;
                            return true;
                        case "12": 
                        case "AbandonedInputQueues": 
                            ret.v = Kusto.Data.IngestionQueueType.AbandonedInputQueues;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "all": 
                                ret.v = Kusto.Data.IngestionQueueType.All;
                                return true;
                            case "inputqueue": 
                                ret.v = Kusto.Data.IngestionQueueType.InputQueue;
                                return true;
                            case "internalqueue": 
                                ret.v = Kusto.Data.IngestionQueueType.InternalQueue;
                                return true;
                            case "readyforaggregationqueue": 
                                ret.v = Kusto.Data.IngestionQueueType.ReadyForAggregationQueue;
                                return true;
                            case "securedreadyforaggregationqueue": 
                                ret.v = Kusto.Data.IngestionQueueType.SecuredReadyForAggregationQueue;
                                return true;
                            case "aggregatedreadyforimportqueue": 
                                ret.v = Kusto.Data.IngestionQueueType.AggregatedReadyForImportQueue;
                                return true;
                            case "failedingestionsqueue": 
                                ret.v = Kusto.Data.IngestionQueueType.FailedIngestionsQueue;
                                return true;
                            case "successfulingestionsqueue": 
                                ret.v = Kusto.Data.IngestionQueueType.SuccessfulIngestionsQueue;
                                return true;
                            case "abandonedingestionsqueue": 
                                ret.v = Kusto.Data.IngestionQueueType.AbandonedIngestionsQueue;
                                return true;
                            case "securedabandonedingestionsqueue": 
                                ret.v = Kusto.Data.IngestionQueueType.SecuredAbandonedIngestionsQueue;
                                return true;
                            case "internalfailedingestionsqueue": 
                                ret.v = Kusto.Data.IngestionQueueType.InternalFailedIngestionsQueue;
                                return true;
                            case "abandonedinputqueues": 
                                ret.v = Kusto.Data.IngestionQueueType.AbandonedInputQueues;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.IngestionQueueType.All: 
                            return "";
                        case Kusto.Data.IngestionQueueType.InputQueue: 
                            return "";
                        case Kusto.Data.IngestionQueueType.InternalQueue: 
                            return "";
                        case Kusto.Data.IngestionQueueType.ReadyForAggregationQueue: 
                            return "";
                        case Kusto.Data.IngestionQueueType.SecuredReadyForAggregationQueue: 
                            return "";
                        case Kusto.Data.IngestionQueueType.AggregatedReadyForImportQueue: 
                            return "";
                        case Kusto.Data.IngestionQueueType.FailedIngestionsQueue: 
                            return "";
                        case Kusto.Data.IngestionQueueType.SuccessfulIngestionsQueue: 
                            return "";
                        case Kusto.Data.IngestionQueueType.AbandonedIngestionsQueue: 
                            return "";
                        case Kusto.Data.IngestionQueueType.SecuredAbandonedIngestionsQueue: 
                            return "";
                        case Kusto.Data.IngestionQueueType.InternalFailedIngestionsQueue: 
                            return "";
                        case Kusto.Data.IngestionQueueType.AbandonedInputQueues: 
                            return "";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /** @namespace Kusto.Data.Ingestion */

    /**
     * Extensions for enum IngestionMappingKind
     *
     * @static
     * @abstract
     * @public
     * @class Kusto.Data.Ingestion.ExtendedIngestionMappingKind
     */
    Bridge.define("Kusto.Data.Ingestion.ExtendedIngestionMappingKind", {
        statics: {
            methods: {
                FastToString: function (that) {
                    switch (that) {
                        case Kusto.Data.Ingestion.IngestionMappingKind.Unknown: 
                            return "Unknown";
                        case Kusto.Data.Ingestion.IngestionMappingKind.Csv: 
                            return "Csv";
                        case Kusto.Data.Ingestion.IngestionMappingKind.Json: 
                            return "Json";
                        case Kusto.Data.Ingestion.IngestionMappingKind.Avro: 
                            return "Avro";
                        case Kusto.Data.Ingestion.IngestionMappingKind.Parquet: 
                            return "Parquet";
                        case Kusto.Data.Ingestion.IngestionMappingKind.SStream: 
                            return "SStream";
                        case Kusto.Data.Ingestion.IngestionMappingKind.Orc: 
                            return "Orc";
                        case Kusto.Data.Ingestion.IngestionMappingKind.ApacheAvro: 
                            return "ApacheAvro";
                        case Kusto.Data.Ingestion.IngestionMappingKind.W3CLogFile: 
                            return "W3CLogFile";
                        default: 
                            break;
                    }

                    return System.String.format("[unknown: {0}]", [Bridge.box(that, System.Int32)]);
                },
                FastIsDefined$1: function (value) {
                    var dummy = { v : new Kusto.Data.Ingestion.IngestionMappingKind() };
                    return Kusto.Data.Ingestion.ExtendedIngestionMappingKind.FastTryParse(value, dummy, false, true);
                },
                FastIsDefined: function (value) {
                    switch (value) {
                        case Kusto.Data.Ingestion.IngestionMappingKind.Unknown: 
                            return true;
                        case Kusto.Data.Ingestion.IngestionMappingKind.Csv: 
                            return true;
                        case Kusto.Data.Ingestion.IngestionMappingKind.Json: 
                            return true;
                        case Kusto.Data.Ingestion.IngestionMappingKind.Avro: 
                            return true;
                        case Kusto.Data.Ingestion.IngestionMappingKind.Parquet: 
                            return true;
                        case Kusto.Data.Ingestion.IngestionMappingKind.SStream: 
                            return true;
                        case Kusto.Data.Ingestion.IngestionMappingKind.Orc: 
                            return true;
                        case Kusto.Data.Ingestion.IngestionMappingKind.ApacheAvro: 
                            return true;
                        case Kusto.Data.Ingestion.IngestionMappingKind.W3CLogFile: 
                            return true;
                        default: 
                            return false;
                    }
                },
                FastHasFlag$1: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastHasFlag: function (that, flag) {
                    return ((that & flag) === flag);
                },
                FastParse: function (value, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    var ret = { v : new Kusto.Data.Ingestion.IngestionMappingKind() };
                    if (Kusto.Data.Ingestion.ExtendedIngestionMappingKind.FastTryParse(value, ret, ignoreCase, ignoreInts)) {
                        return ret.v;
                    }
                    throw new System.Exception("Invalid enum value: " + (value || ""));
                },
                FastTryParse: function (value, ret, ignoreCase, ignoreInts) {
                    if (ignoreCase === void 0) { ignoreCase = false; }
                    if (ignoreInts === void 0) { ignoreInts = false; }
                    // If it is a simple value, just use that
                    switch (value) {
                        case "0": 
                        case "Unknown": 
                            ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Unknown;
                            return true;
                        case "1": 
                        case "Csv": 
                            ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Csv;
                            return true;
                        case "2": 
                        case "Json": 
                            ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Json;
                            return true;
                        case "4": 
                        case "Avro": 
                            ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Avro;
                            return true;
                        case "5": 
                        case "Parquet": 
                            ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Parquet;
                            return true;
                        case "6": 
                        case "SStream": 
                            ret.v = Kusto.Data.Ingestion.IngestionMappingKind.SStream;
                            return true;
                        case "7": 
                        case "Orc": 
                            ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Orc;
                            return true;
                        case "8": 
                        case "ApacheAvro": 
                            ret.v = Kusto.Data.Ingestion.IngestionMappingKind.ApacheAvro;
                            return true;
                        case "9": 
                        case "W3CLogFile": 
                            ret.v = Kusto.Data.Ingestion.IngestionMappingKind.W3CLogFile;
                            return true;
                        default: 
                            break;
                    }
                    if (ignoreCase && !System.String.isNullOrEmpty(value)) {
                        switch (value.toLowerCase()) {
                            case "unknown": 
                                ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Unknown;
                                return true;
                            case "csv": 
                                ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Csv;
                                return true;
                            case "json": 
                                ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Json;
                                return true;
                            case "avro": 
                                ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Avro;
                                return true;
                            case "parquet": 
                                ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Parquet;
                                return true;
                            case "sstream": 
                                ret.v = Kusto.Data.Ingestion.IngestionMappingKind.SStream;
                                return true;
                            case "orc": 
                                ret.v = Kusto.Data.Ingestion.IngestionMappingKind.Orc;
                                return true;
                            case "apacheavro": 
                                ret.v = Kusto.Data.Ingestion.IngestionMappingKind.ApacheAvro;
                                return true;
                            case "w3clogfile": 
                                ret.v = Kusto.Data.Ingestion.IngestionMappingKind.W3CLogFile;
                                return true;
                            default: 
                                break;
                        }
                    }

                    // If it can be interpreted as an integer, do so (use IsDefined if unwanted)
                    if (!ignoreInts) {
                        var reti = { v : 0 };
                        if (System.Int32.tryParse(value, reti)) {
                            ret.v = reti.v;
                            return true;
                        }
                    }

                    ret.v = 0;
                    return false;
                },
                FastGetHashCode: function (that) {
                    return that;
                },
                FastGetDescription: function (that) {
                    switch (that) {
                        case Kusto.Data.Ingestion.IngestionMappingKind.Unknown: 
                            return "Unknown";
                        case Kusto.Data.Ingestion.IngestionMappingKind.Csv: 
                            return "Csv";
                        case Kusto.Data.Ingestion.IngestionMappingKind.Json: 
                            return "Json";
                        case Kusto.Data.Ingestion.IngestionMappingKind.Avro: 
                            return "Avro";
                        case Kusto.Data.Ingestion.IngestionMappingKind.Parquet: 
                            return "Parquet";
                        case Kusto.Data.Ingestion.IngestionMappingKind.SStream: 
                            return "SStream";
                        case Kusto.Data.Ingestion.IngestionMappingKind.Orc: 
                            return "Orc";
                        case Kusto.Data.Ingestion.IngestionMappingKind.ApacheAvro: 
                            return "Apache Avro (through avro2json)";
                        case Kusto.Data.Ingestion.IngestionMappingKind.W3CLogFile: 
                            return "W3C Extended Log File format";
                        default: 
                            return null;
                    }
                }
            }
        }
    });

    /**
     * The ingestion mapping kind
     *
     * @public
     * @class Kusto.Data.Ingestion.IngestionMappingKind
     */
    Bridge.define("Kusto.Data.Ingestion.IngestionMappingKind", {
        $kind: "enum",
        statics: {
            fields: {
                /**
                 * Unknown
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Ingestion.IngestionMappingKind
                 * @constant
                 * @default 0
                 * @type Kusto.Data.Ingestion.IngestionMappingKind
                 */
                Unknown: 0,
                /**
                 * Csv
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Ingestion.IngestionMappingKind
                 * @constant
                 * @default 1
                 * @type Kusto.Data.Ingestion.IngestionMappingKind
                 */
                Csv: 1,
                /**
                 * Json
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Ingestion.IngestionMappingKind
                 * @constant
                 * @default 2
                 * @type Kusto.Data.Ingestion.IngestionMappingKind
                 */
                Json: 2,
                /**
                 * Avro
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Ingestion.IngestionMappingKind
                 * @constant
                 * @default 4
                 * @type Kusto.Data.Ingestion.IngestionMappingKind
                 */
                Avro: 4,
                /**
                 * Parquet
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Ingestion.IngestionMappingKind
                 * @constant
                 * @default 5
                 * @type Kusto.Data.Ingestion.IngestionMappingKind
                 */
                Parquet: 5,
                /**
                 * SStream
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Ingestion.IngestionMappingKind
                 * @constant
                 * @default 6
                 * @type Kusto.Data.Ingestion.IngestionMappingKind
                 */
                SStream: 6,
                /**
                 * Orc
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Ingestion.IngestionMappingKind
                 * @constant
                 * @default 7
                 * @type Kusto.Data.Ingestion.IngestionMappingKind
                 */
                Orc: 7,
                /**
                 * Apache Avro (through avro2json)
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Ingestion.IngestionMappingKind
                 * @constant
                 * @default 8
                 * @type Kusto.Data.Ingestion.IngestionMappingKind
                 */
                ApacheAvro: 8,
                /**
                 * W3C Extended Log File format
                 *
                 * @static
                 * @public
                 * @memberof Kusto.Data.Ingestion.IngestionMappingKind
                 * @constant
                 * @default 9
                 * @type Kusto.Data.Ingestion.IngestionMappingKind
                 */
                W3CLogFile: 9
            }
        }
    });

    /**
     * Ingestion Queue Types
     *
     * @public
     * @class Kusto.Data.IngestionQueueType
     */
    Bridge.define("Kusto.Data.IngestionQueueType", {
        $kind: "enum",
        statics: {
            fields: {
                All: 0,
                InputQueue: 1,
                InternalQueue: 2,
                ReadyForAggregationQueue: 3,
                SecuredReadyForAggregationQueue: 4,
                AggregatedReadyForImportQueue: 5,
                FailedIngestionsQueue: 6,
                SuccessfulIngestionsQueue: 7,
                AbandonedIngestionsQueue: 9,
                SecuredAbandonedIngestionsQueue: 10,
                InternalFailedIngestionsQueue: 11,
                AbandonedInputQueues: 12
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.AdminEngineRuleKind", {
        $kind: "enum",
        statics: {
            fields: {
                None: 0,
                Start: 0,
                YieldShowCommandOptions: 0,
                YieldTableNamesForAdminOptions: 1,
                YieldFunctionNamesForAdminOptions: 2,
                YieldDatabaseNames: 3,
                YieldShowBasicAuthOptions: 4,
                YieldShowDatabasePrincipalsPoliciesAndSchemaOptions: 5,
                YieldShowDatabasesOptions: 6,
                YieldShowClusterPrincipalsOptions: 7,
                YieldSetDatabaseAdminsUsersViewersPrettyNameOptions: 8,
                YieldSetClusterAdminsUsersViewersDatabaseCreatorsOptions: 9,
                YieldSetTableAdminsOptions: 10,
                YieldAddDatabaseAdminsUsersViewersOptions: 11,
                YieldDropDatabaseOptions: 12,
                YieldAddClusterAdminsUsersViewersDatabaseCreatorsBlockedPrincipalsOptions: 13,
                YieldDropClusterAdminsUsersViewersDatabaseCreatorsBlockedPrincipalsOptions: 14,
                YieldAddClusterBlockedPrincipalsOptions: 15,
                YieldAddTableAdminsOptions: 16,
                YieldDropTableEntitiesOptions: 17,
                YieldDeleteTableEntitiesOptions: 18,
                YieldCreateTableEntitiesOptions: 19,
                YieldAlterTableEntitiesOptions: 20,
                YieldDropTableColumnsSyntaxOptions: 21,
                YieldSetDatabaseUsersNoneOptions: 22,
                YieldSetDatabaseAdminsNoneOptions: 23,
                YieldSetDatabaseViewersNoneOptions: 24,
                YieldSetClusterUsersNoneOptions: 25,
                YieldSetClusterAdminsNoneOptions: 26,
                YieldSetClusterViewersNoneOptions: 27,
                YieldSetClusterDatabaseCreatorsNoneOptions: 28,
                YieldSetTableAdminsNoneOptions: 29,
                YieldShowTableEntitiesOptions: 30,
                YieldAlterExtentTagsOptions: 31,
                YieldAttachExtentsOptions: 32,
                YieldAttachExtentsIntoTableOptions: 33,
                YieldDropExtentTagsOptions: 34,
                YieldMoveExtentsOptions: 35,
                YieldMoveSpecifiedExtentsOptions: 36,
                YieldMoveExtentsToTableOptions: 37,
                YieldReplaceCommandOptions: 38,
                YieldReplaceExtentsOptions: 39,
                YieldRunCommandOptions: 40,
                YieldShowExtentsOptions: 41,
                YieldShowExtentsByEntityOptions: 42,
                YieldShowExtentsByEntityWithTagsFiltersOptions: 43,
                YieldSetCommandOptions: 44,
                YieldSetOrAppendCommandOptions: 45,
                YieldAddCommandOptions: 46,
                YieldAttachCommandOptions: 47,
                YieldDropCommandOptions: 48,
                YieldMoveCommandOptions: 49,
                YieldPoliciesOptions: 50,
                YieldClusterPoliciesOptions: 51,
                YieldDatabasePoliciesOptions: 52,
                YieldTablePoliciesOptions: 53,
                YieldColumnPoliciesOptions: 54,
                YieldAlterCommandOptions: 55,
                YieldAlterMergeCommandOptions: 56,
                YieldDeleteCommandOptions: 57,
                YieldCreateCommandOptions: 58,
                YieldMultiplePoliciesOptions: 59,
                YieldDatabaseCreatePersistencyOptions: 60,
                YieldCreateIfNotExistsOptions: 61,
                YieldAdminCommandsOptions: 62,
                YieldExportCommandOptions: 63,
                YieldExportCommandWithModifiersAndOptions: 64,
                YieldExportCommandNoModifiersAndOptions: 65,
                YieldAlterDatabaseCommandOptions: 66,
                YieldDeleteMultiplePoliciesOptions: 67,
                YieldIngestionDuplicationOptions: 68,
                YieldPurgeOptions: 69,
                YieldPurgeCleanupOptions: 70,
                YieldPurgeWhatIfOptions: 71,
                YieldPurgeWithPropertiesOptions: 72,
                YieldPurgeTableOptions: 73,
                YieldShowPrincipalCommandOptions: 74,
                YieldAlterFunctionEntitiesOptions: 75,
                YieldAlterMergePolicyRetentionOptions: 76,
                YieldAlterMergePolicyRetentionSoftDeleteDefinedOptions: 77,
                YieldAlterTimeSpanPolicyOptions: 78,
                YieldCreateOrAlterOptions: 79,
                YieldShowFabricOptions: 80,
                YieldRowStoreCreatePersistencyOptions: 81,
                YieldSetDatabaseIngestorsNoneOptions: 82,
                YieldSetTableIngestorsNoneOptions: 83,
                YieldSuspendNodeControlCommandOptions: 84
            },
            ctors: {
                init: function () {
                    this.None = Kusto.Data.IntelliSense.RuleKind.None;
                    this.Start = Kusto.Data.IntelliSense.RuleKind.Last;
                }
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.AnalyzedCommand", {
        fields: {
            Command: null,
            Context: null
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.ApiKind", {
        $kind: "enum",
        statics: {
            fields: {
                Query: 0,
                AdminSyncCommand: 1,
                AdminAsyncCommand: 2
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.ApplyPolicy", {
        statics: {
            fields: {
                NullApplyPolicy: null,
                AppendSpacePolicy: null,
                AppendPipePolicy: null,
                AppendStringLiteralArrayPolicy: null,
                StringComparisonPolicy: null,
                AppendCommaPolicy: null,
                AppendDoubleCommaPolicy: null,
                AppendSpaceStepBackPolicy: null,
                AppendSpaceAfterBracketPolicy: null,
                MoveAfterTheBracketPolicy: null,
                AppendAssignmentPolicy: null,
                AppendJoinClauseWithOpenningBracketPolicy: null,
                AppendJoinClauseWithoutOpenningBracketPolicy: null,
                AppendFindInClausePolicy: null,
                AppendSearchInClausePolicy: null,
                AppendFindInClauseWithoutOpenningBracketPolicy: null
            },
            ctors: {
                init: function () {
                    var $t;
                    this.NullApplyPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.Text = "", $t);
                    this.AppendSpacePolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.Text = " ", $t);
                    this.AppendPipePolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.Text = " | ", $t);
                    this.AppendStringLiteralArrayPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.Text = " (\"\", \"\")", $t.OffsetPosition = 2, $t.OffsetToken = "(", $t);
                    this.StringComparisonPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.Text = " \"\" ", $t.OffsetPosition = -2, $t);
                    this.AppendCommaPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.Text = ", ", $t.OffsetPosition = 0, $t);
                    this.AppendDoubleCommaPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.Text = ", , ", $t.OffsetPosition = -2, $t);
                    this.AppendSpaceStepBackPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.Text = " ", $t.OffsetPosition = -2, $t);
                    this.AppendSpaceAfterBracketPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.Text = " ", $t.OffsetToken = "(", $t.OffsetPosition = 1, $t);
                    this.MoveAfterTheBracketPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.Text = "", $t.OffsetToken = ")", $t.OffsetPosition = 1, $t);
                    this.AppendAssignmentPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.Text = " = ", $t);
                    this.AppendJoinClauseWithOpenningBracketPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.OffsetPosition = -13, $t.Text = " (\r\n    \r\n) on Key1 ", $t);
                    this.AppendJoinClauseWithoutOpenningBracketPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.OffsetPosition = -13, $t.Text = "\r\n    \r\n) on Key1 ", $t);
                    this.AppendFindInClausePolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.OffsetPosition = -8, $t.Text = " () where ", $t);
                    this.AppendSearchInClausePolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.OffsetPosition = -2, $t.Text = " () ", $t);
                    this.AppendFindInClauseWithoutOpenningBracketPolicy = ($t = new Kusto.Data.IntelliSense.ApplyPolicy(), $t.OffsetPosition = -8, $t.Text = ") where ", $t);
                }
            }
        },
        props: {
            Text: null,
            OffsetPosition: 0,
            OffsetToken: null
        },
        ctors: {
            ctor: function () {
                this.$initialize();
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase", {
        statics: {
            fields: {
                CommonRegexOptions: 0,
                DefaultRegexOptions: 0,
                s_isCommandRegex: null,
                s_firstWordAfterPipeRegex: null
            },
            ctors: {
                init: function () {
                    this.CommonRegexOptions = 16;
                    this.DefaultRegexOptions = 0;
                    this.s_isCommandRegex = new System.Text.RegularExpressions.Regex.ctor("^\\s*\\.", Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.CommonRegexOptions);
                    this.s_firstWordAfterPipeRegex = new System.Text.RegularExpressions.Regex.ctor("^\\s*(?<FirstWord>[\\w\\-]+)\\s+", Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.CommonRegexOptions);
                }
            },
            methods: {
                FindRules: function (rules, command, commandContext, lastPart, keywordFromCommand) {
                    var $t;
                    $t = Bridge.getEnumerator(rules);
                    try {
                        while ($t.moveNext()) {
                            var r = $t.Current;
                            if (r.RequiredKeywords != null && r.RequiredKeywords.Count > 0) {
                                var hasKeywordMatch = false;
                                if (r.RequiresFullCommand) {
                                    // Check if command has one of the required keywords
                                    hasKeywordMatch = System.Linq.Enumerable.from(r.RequiredKeywords, System.String).any(function (keyword) {
                                            return System.String.contains(command,keyword);
                                        });
                                } else {
                                    hasKeywordMatch = (!System.String.isNullOrEmpty(keywordFromCommand) && r.RequiredKeywords.contains(keywordFromCommand));
                                }

                                if (!hasKeywordMatch) {
                                    continue;
                                }
                            }
                            if (r.IsMatch(commandContext, (r.RequiresFullCommand) ? command : lastPart)) {
                                return r;
                            }
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }

                    return null;
                },
                FindLastStatement: function (statements) {
                    if (System.String.isNullOrEmpty(statements)) {
                        return "";
                    }

                    return System.Linq.Enumerable.from(Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.ParseAsStatements(statements, 59, false), System.String).lastOrDefault(null, null);
                },
                ParseAsStatements: function (input, delimeter, trimDelimeter) {
                    var result = new (System.Collections.Generic.List$1(System.String)).ctor();

                    if (System.String.isNullOrEmpty(input)) {
                        return result;
                    }

                    var currentStatementStart = 0;
                    var chars = System.String.toCharArray(input, 0, input.length);
                    for (var i = 0; i < chars.length; i = (i + 1) | 0) {
                        var ch = chars[System.Array.index(i, chars)];

                        if (!!(ch === 34 | ch === 39)) {
                            i = Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.SkipToBalancedChar(chars, ((i + 1) | 0), ch, System.Array.init([34, 39], System.Char));
                        }

                        var eofString = (i === ((chars.length - 1) | 0));
                        if (ch === delimeter || eofString) {
                            var len = (((i - currentStatementStart) | 0) + 1) | 0;
                            // Remove delimeter if was requested
                            if (trimDelimeter && ch === delimeter) {
                                len = (len - 1) | 0;
                            }

                            if (len > 0) {
                                result.add(input.substr(currentStatementStart, len));
                            }
                            currentStatementStart = (i + 1) | 0;
                        }
                    }

                    return result;
                },
                SkipToBalancedChar: function (chars, start, balancedChar, recursionChars) {
                    for (var i = start; i < chars.length; i = (i + 1) | 0) {
                        var ch = chars[System.Array.index(i, chars)];
                        if (ch === balancedChar) {
                            // Found char that we've searched for
                            return i;
                        }

                        if (System.Array.contains(recursionChars, ch, System.Char)) {
                            i = Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.SkipToBalancedChar(chars, ((i + 1) | 0), ch, recursionChars);
                        }
                    }

                    // if we got so far - the balanced char was not found
                    return ((chars.length - 1) | 0);
                },
                GetApproximateCommandLastPart: function (command, lastPart, firstWordAfterPipe) {
                    var parts = System.String.split(command, [124].map(function (i) {{ return String.fromCharCode(i); }}));
                    var last = System.Linq.Enumerable.from(parts, System.String).lastOrDefault(null, null);
                    lastPart.v = last;
                    if (parts.length > 1) {
                        lastPart.v = "|" + (last || "");
                    }

                    if (System.String.isNullOrEmpty(last)) {
                        firstWordAfterPipe.v = "";
                        return;
                    }

                    firstWordAfterPipe.v = Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.GetFirstWordAfterPipe(last);
                },
                GetFirstWordAfterPipe: function (input) {
                    var match = Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.s_firstWordAfterPipeRegex.match(input);
                    return match.getGroups().getByName("FirstWord").toString();
                }
            }
        },
        props: {
            Locker: null,
            GeneralRules: null,
            CommandRules: null,
            QueryParametersRules: null,
            DefaultRule: null,
            CommandToolTips: null,
            ContextConnection: null
        },
        ctors: {
            ctor: function () {
                this.$initialize();
                this.Locker = { };
            }
        },
        methods: {
            /**
             * Attempts to match any rule for input string
             *
             * @instance
             * @public
             * @this Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase
             * @memberof Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase
             * @param   {string}                                      command    
             * @param   {Kusto.Data.IntelliSense.IntelliSenseRule}    rule
             * @return  {boolean}
             */
            TryMatchAnyRule: function (command, rule) {
                var $t, $t1;
                var analyzedCommand = this.AnalyzeCommand$1(command, null);
                var commandContext = analyzedCommand.Context;
                var firstWordAfterPipe = { }, lastPart = { };
                Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.GetApproximateCommandLastPart(analyzedCommand.Command, lastPart, firstWordAfterPipe);
                var isCommand = Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.s_isCommandRegex.isMatch(command);

                rule.v = null;
                if (isCommand) {
                    Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.PrivateTracer.Tracer.TraceVerbose("TryMatchAnyRule: start matching rules for commands rules");
                    $t = Bridge.getEnumerator(this.CommandRules);
                    try {
                        while ($t.moveNext()) {
                            var r = $t.Current;
                            if (r.IsMatch(commandContext, (r.RequiresFullCommand) ? command : lastPart.v)) {
                                rule.v = r;
                                break;
                            }
                            //PrivateTracer.Tracer.TraceVerbose("TryMatchAnyRule: rule {0} didn't match for any of commands rules", r.Kind);
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }
                }

                if (rule.v == null) {
                    Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.PrivateTracer.Tracer.TraceVerbose("TryMatchAnyRule: start matching rules for general rules");
                    rule.v = Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.FindRules(this.GeneralRules, command, commandContext, lastPart.v, firstWordAfterPipe.v);
                }

                if (rule.v == null) {
                    Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.PrivateTracer.Tracer.TraceVerbose("TryMatchAnyRule: start matching rules for query parameters rules");
                    $t1 = Bridge.getEnumerator(this.QueryParametersRules);
                    try {
                        while ($t1.moveNext()) {
                            var r1 = $t1.Current;
                            if (r1.IsMatch(commandContext, (r1.RequiresFullCommand) ? command : lastPart.v)) {
                                rule.v = r1;
                                break;
                            }
                            //PrivateTracer.Tracer.TraceVerbose("TryMatchAnyRule: rule {0} didn't match for any of query parameters rules", r.Kind);
                        }
                    } finally {
                        if (Bridge.is($t1, System.IDisposable)) {
                            $t1.System$IDisposable$Dispose();
                        }
                    }
                }

                if (rule.v != null && rule.v.IsContextual) {
                    Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.PrivateTracer.Tracer.TraceVerbose("TryMatchAnyRule: rule {0} was found", [Bridge.box(rule.v.Kind, System.Int32)]);
                    // Update selection options per command context
                    this.UpdateProviderAvailableEntities(command, commandContext);
                    Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.PrivateTracer.Tracer.TraceVerbose("TryMatchAnyRule: Entities were updated", [Bridge.box(rule.v.Kind, System.Int32)]);
                } else {
                    Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.PrivateTracer.Tracer.TraceVerbose("TryMatchAnyRule: no rule was found");
                }
                return (rule.v != null);
            },
            TryMatchSpecificRule: function (command, commandContext, ruleKind, rule) {
                rule.v = null;
                var isCommand = Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.s_isCommandRegex.isMatch(command);

                // If it is a command - try to match any command rules
                if (isCommand) {
                    rule.v = System.Linq.Enumerable.from(this.CommandRules, Kusto.Data.IntelliSense.IntelliSenseRule).firstOrDefault(function (r) {
                            return r.Kind === ruleKind && r.IsMatch(commandContext, command);
                        }, null);
                }

                // If nothing found (or not command) - go for the rest of rules
                if (rule.v == null) {
                    rule.v = System.Linq.Enumerable.from(this.GeneralRules, Kusto.Data.IntelliSense.IntelliSenseRule).firstOrDefault(function (r) {
                            return r.Kind === ruleKind && r.IsMatch(commandContext, command);
                        }, null);
                }

                if (rule.v != null && rule.v.IsContextual) {
                    // Update selection options per command context
                    this.UpdateProviderAvailableEntities(command, commandContext);
                }
                return (rule.v != null);
            },
            SetQueryParametersRule: function (queryParameters) { },
            Initialize: function () {
                this.CommandRules = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.IntelliSenseRule)).ctor();
                this.GeneralRules = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.IntelliSenseRule)).ctor();
                this.CommandToolTips = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.IntelliSenseCommandTip)).ctor();
                this.QueryParametersRules = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.IntelliSenseRule)).ctor();
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CMRuleKind", {
        $kind: "enum",
        statics: {
            fields: {
                None: 0,
                Start: 0,
                YieldServiceCancelMaintenance: 0,
                YieldServiceMoveToMaintenance: 1,
                YieldShowRegionsByHosters: 2,
                YieldAdminCommandsOptions: 3,
                YieldPipeOptions: 4,
                YieldOperatorsAfterPipe: 5,
                YieldShowCommandOptions: 6,
                YieldShowFabricCommandOptions: 7,
                YieldShowAccountCommandOptions: 8,
                YieldShowAccountsKeywordOptions: 9,
                YieldMoveAccountCommandOptions: 10,
                YieldMoveAccountSubscriptionCommandOptions: 11,
                YieldMoveCommandOptions: 12,
                YieldAlterCommandOptions: 13,
                YieldAddCommandOptions: 14,
                YieldDropCommandOptions: 15,
                YieldRemoveCommandOptions: 16,
                YieldCreateCommandOptions: 17,
                YieldCreateExternalCommandOptions: 18,
                YieldDeleteCommandOptions: 19,
                YieldDetachCommandOptions: 20,
                YieldCheckCommandOptions: 21,
                YieldSetCommandOptions: 22,
                YieldRegisterCommandOptions: 23,
                YieldPutCommandOptions: 24,
                YieldListCommandOptions: 25,
                YieldListKustoPoolsServicesCommandPropertiesSuffix: 26,
                YieldSyncCommandOptions: 27,
                YieldServiceKeywordOptions: 28,
                YieldServiceOrClusterKeywordOptions: 29,
                YieldStorageAccountsKeywordOptions: 30,
                YieldIngestLogsIntoKeywordsOptions: 31,
                YieldInServiceKeywordOptions: 32,
                YieldShowClusterKeywordOptions: 33,
                YieldOperationsKeywordOptions: 34,
                YieldOrchestrationKeywordOptions: 35,
                YieldOrchestrationTaskHubKeywordOptions: 36,
                YieldPcCodeKeywordOptions: 37,
                YieldsAlterServiceConfigurationKeywordOptions: 38,
                YieldFromAccountKeywordOptions: 39,
                YieldShowAuditLogCommandOptions: 40,
                YieldShowAuditLogFromCommandOptions: 41,
                YieldShowCmServiceCommandOptions: 42,
                YieldShowClusterClientOptions: 43,
                YieldShowClusterClientSubscriptionCommandOptions: 44,
                YieldShowClusterClusterNamePrincipalCommandOptions: 45,
                YieldShowServicesCompletionKeywordsOptions: 46,
                YieldConfigurationPropertySetKeywordOptions: 47,
                YieldConfigurationPropertyDropKeywordOptions: 48,
                YieldToStringKeywordOptions: 49,
                YieldToKeywordOptions: 50,
                YieldConfigurationPropertyWithConsentKeywordOptions: 51,
                YieldAccountAlterOperationsKeywordOptions: 52,
                YieldAccountAddOperationsKeywordOptions: 53,
                YieldAccountRemoveOperationsKeywordOptions: 54,
                YieldAccountSetOperationsKeywordOptions: 55,
                YieldAccountRenameOperationsKeywordOptions: 56,
                YieldServiceAlterOperationsKeywordOptions: 57,
                YieldMetadataKeywordOptions: 58,
                YieldSummaryKeywordOptions: 59,
                YieldSummaryOrMetadataKeywordOptions: 60,
                YieldDatabaseOrAccountMetadataKeywordOptions: 61,
                YieldAccountNames: 62,
                YieldDmServiceAddKeywordOptions: 63,
                YieldDmServiceDropKeywordOptions: 64,
                YieldDmServiceAlterKeywordOptions: 65,
                YieldDmServiceAlterSettingsOptions: 66,
                YieldDmServiceShowKeywordOptions: 67,
                YieldServicesNames: 68,
                YieldServicesNamesCombinedWithOptions: 69,
                YieldTargetServiceOptions: 70,
                YieldWithPropertiesSuffix: 71,
                YieldWithEventHubPropertiesSuffix: 72,
                YieldInstallCommandOptions: 73,
                YieldReinstallCommandOptions: 74,
                YieldUninstallServiceCommandOptions: 75,
                YieldUninstallCommandOptions: 76,
                YieldUpdateCommandOptions: 77,
                YieldMigrateServiceCommandOptions: 78,
                YieldMigrateClusterCommandOptions: 79,
                YieldFromServiceKeywordOptions: 80,
                YieldDatabaseMoveFromServiceCommandOptions: 81,
                YieldSetClusterPrincipalsCommandOptions: 82,
                YieldAddClusterPrincipalsCommandOptions: 83,
                YieldDropClusterPrincipalsCommandOptions: 84,
                YieldAddClusterJitPrincipalsCommandOptions: 85,
                YieldDropClusterJitPrincipalsCommandOptions: 86,
                YieldSetClusterPrincipalsNoneOptions: 87,
                YieldDropAccountPrincipalsCommandOptions: 88,
                YieldDatabaseCreateCommandOptions: 89,
                YieldServiceCreateCommandOptions: 90,
                YieldShowPrincipalRolesCommandOptions: 91,
                YieldAddKeywordOptions: 92,
                YieldAlterServiceConfigurationPropertiesOptions: 93,
                YieldAlterServiceConfigurationCommandOptions: 94,
                YieldMoveServiceConfigurationCommandOptions: 95,
                YieldAddServiceServiceNameOptions: 96,
                YieldLoadBalancerKeywordOptions: 97,
                YieldOperationKeywordOptions: 98,
                YieldShowSubscriptionCommandOptions: 99,
                YieldShowServiceSecurityRulesCommandOptions: 100,
                YieldKeysCommandOptions: 101,
                YieldKeyTypesCommandOptions: 102,
                YieldAlterServiceStorageAccountsCommandOptions: 103,
                YieldCmNotifyCommandOptions: 104,
                YieldExecuteCommandOptions: 105,
                YieldExecuteJobCommandOptions: 106,
                YieldGenerateCommandOptions: 107,
                YieldPublishCommandOptions: 108,
                YieldSkuAvailabilityPrefix: 109,
                YieldSkuAvailabilityCommandValues: 110,
                YieldSkuAvailabilityPropertiesSuffix: 111,
                YieldServiceRegenerateSuffix: 112,
                YieldVersionKeywordOptions: 113,
                YieldExportCommandOptions: 114,
                YieldExportPackageForOptions: 115,
                YieldExportPackageForServiceTypeOptions: 116,
                YieldSharedIdentity: 117,
                YieldSetRegionVisibility: 118,
                YieldRegisterFeature: 119,
                YieldBillingServiceCreateResources: 120,
                YieldPublishServiceCommandOptions: 121,
                YieldUnPublishCommandOptions: 122,
                YieldUnPublishServiceCommandOptions: 123,
                YieldRotateCommandOptions: 124,
                YieldRotateServiceCommandOptions: 125,
                YieldCreateFollowerDatabaseInServiceCommandOptions: 126,
                YieldColumnNamesAfterPipeCommandOptions: 127,
                YieldPutClientSubscriptionsNotification: 128,
                YieldDatabaseKeywordOptions: 129,
                YieldSyncKustoPoolRbacCommandOptions: 130
            },
            ctors: {
                init: function () {
                    this.None = Kusto.Data.IntelliSense.RuleKind.None;
                    this.Start = Kusto.Data.IntelliSense.RuleKind.Last;
                }
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CompletionOption", {
        props: {
            Kind: 0,
            Value: null
        },
        ctors: {
            ctor: function (kind, value) {
                this.$initialize();
                this.Kind = kind;
                this.Value = value;
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CompletionOptionCollection", {
        fields: {
            m_materializedOptions: null
        },
        props: {
            Kind: 0,
            Values: null,
            Priority: 0
        },
        ctors: {
            ctor: function (kind, values, priority) {
                if (priority === void 0) { priority = 0; }

                this.$initialize();
                this.Kind = kind;
                this.Values = values;
                this.Priority = priority;
            }
        },
        methods: {
            GetCompletionOptions: function () {
                if (this.m_materializedOptions == null) {
                    this.m_materializedOptions = System.Linq.Enumerable.from(this.Values, System.String).select(Bridge.fn.bind(this, $asm.$.Kusto.Data.IntelliSense.CompletionOptionCollection.f1)).ToArray(Kusto.Data.IntelliSense.CompletionOption);
                }

                return this.m_materializedOptions;
            }
        }
    });

    Bridge.ns("Kusto.Data.IntelliSense.CompletionOptionCollection", $asm.$);

    Bridge.apply($asm.$.Kusto.Data.IntelliSense.CompletionOptionCollection, {
        f1: function (o) {
            return new Kusto.Data.IntelliSense.CompletionOption(this.Kind, o);
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.ContextOperation", {
        $kind: "enum",
        statics: {
            fields: {
                None: 0,
                Union: 1,
                Intersect: 2
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.IntelliSenseRule", {
        props: {
            Kind: 0,
            RequiredKeywords: null,
            AfterApplyPolicies: null,
            BeforeApplyPolicies: null,
            DefaultAfterApplyPolicy: null,
            DefaultBeforeApplyPolicy: null
        },
        methods: {
            GetAfterApplyInfo: function (selectedOption) {
                var policy = { v : null };
                if (this.AfterApplyPolicies != null) {
                    this.AfterApplyPolicies.tryGetValue(selectedOption, policy);
                }
                return policy.v || this.DefaultAfterApplyPolicy || Kusto.Data.IntelliSense.ApplyPolicy.NullApplyPolicy;
            },
            GetBeforeApplyInfo: function (selectedOption) {
                var policy = { v : null };
                if (this.BeforeApplyPolicies != null) {
                    this.BeforeApplyPolicies.tryGetValue(selectedOption, policy);
                }
                return policy.v || this.DefaultBeforeApplyPolicy || Kusto.Data.IntelliSense.ApplyPolicy.NullApplyPolicy;
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CslCommand", {
        statics: {
            fields: {
                c_tabulation: null,
                s_newLineRegex: null,
                s_removeWhiteSpacesAfterPipeRegex: null,
                s_joinStartRegex: null,
                s_joinEndRegex: null,
                s_makeSeriesOperatorRegex: null,
                s_operatorRegex: null,
                s_operatorsNotRequiringFullEntitiesResolve: null,
                s_nameOrListRegex: null,
                s_hasAssignmentOperationRegex: null,
                s_startsWithAlpha: null
            },
            ctors: {
                init: function () {
                    this.c_tabulation = "    ";
                    this.s_newLineRegex = new System.Text.RegularExpressions.Regex.ctor("[\\n\\r]", Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.CommonRegexOptions);
                    this.s_removeWhiteSpacesAfterPipeRegex = new System.Text.RegularExpressions.Regex.ctor("^\\|\\s+", Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.DefaultRegexOptions);
                    this.s_joinStartRegex = new System.Text.RegularExpressions.Regex.ctor("(?<JoinOpPart>^\\s*\\|\\s*join\\s+(kind\\s*=\\s*\\w+\\s*)?)(?<Bracket>\\()?(?<PostJoinPart>.+$)", Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.CommonRegexOptions);
                    this.s_joinEndRegex = new System.Text.RegularExpressions.Regex.ctor("(?!^.*\\bmake-series\\b.*$)((?<InnerJoinPart>^.*?)(?<Bracket>\\)?)\\s*\\b(?<JoinOnPart>on\\s+.+))", Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.CommonRegexOptions);
                    this.s_makeSeriesOperatorRegex = new System.Text.RegularExpressions.Regex.ctor("\\bmake-series\\b", Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.DefaultRegexOptions);
                    this.s_operatorRegex = new System.Text.RegularExpressions.Regex.ctor("\\|\\s*(?<Operator>[\\w-]+)", Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.DefaultRegexOptions);
                    this.s_operatorsNotRequiringFullEntitiesResolve = $asm.$.Kusto.Data.IntelliSense.CslCommand.f1(new (System.Collections.Generic.HashSet$1(System.String)).ctor());
                    this.s_nameOrListRegex = "(?:\\w+)|(?:\\((\\w+)(,\\s*\\w+)*\\))";
                    this.s_hasAssignmentOperationRegex = new System.Text.RegularExpressions.Regex.ctor("(^\\s*\\|\\s*(extend|parse|summarize|project|mvexpand|mv-expand|make-series|project-rename)\\s+" + (Kusto.Data.IntelliSense.CslCommand.s_nameOrListRegex || "") + ")|(^\\s*range)", Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.DefaultRegexOptions);
                    this.s_startsWithAlpha = new System.Text.RegularExpressions.Regex.ctor("^\\s*[a-z]", 1);
                }
            },
            methods: {
                MakeTrivialCommandFromText: function (text) {
                    var $t, $t1;
                    return ($t = new Kusto.Data.IntelliSense.CslCommand(), $t.m_commandParts = ($t1 = Kusto.Data.IntelliSense.CslCommandToken, System.Linq.Enumerable.from(System.Array.init([new Kusto.Data.IntelliSense.CslCommandToken(text, 0, Kusto.Data.IntelliSense.CslCommandToken.Kind.CslCommandToken)], Kusto.Data.IntelliSense.CslCommandToken), $t1).toList($t1)), $t.Text = text, $t.Length = text.length, $t.AbsolutePositionBias = 0, $t.RelativeStart = 0, $t.ParseMode = Kusto.Data.IntelliSense.ParseMode.CommandTokensOnly, $t.IsRunnable = !System.String.isNullOrWhiteSpace(text), $t);
                },
                UnionCommands: function (commands) {
                    var $t;
                    if (System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).count() < 2) {
                        return System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).firstOrDefault(null, null);
                    }

                    var firstCommand = System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).first();
                    // count is 2 or more
                    var result = ($t = new Kusto.Data.IntelliSense.CslCommand(), $t.m_tokens = System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).selectMany($asm.$.Kusto.Data.IntelliSense.CslCommand.f2).toList(Kusto.Data.IntelliSense.CslCommandToken), $t.m_commandParts = System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).selectMany($asm.$.Kusto.Data.IntelliSense.CslCommand.f3).toList(Kusto.Data.IntelliSense.CslCommandToken), $t.m_commentsParts = System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).selectMany($asm.$.Kusto.Data.IntelliSense.CslCommand.f4).toList(Kusto.Data.IntelliSense.CslCommandToken), $t.m_clientDirectivesParts = System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).selectMany($asm.$.Kusto.Data.IntelliSense.CslCommand.f5).toList(Kusto.Data.IntelliSense.CslCommandToken), $t.m_bracketParts = System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).selectMany($asm.$.Kusto.Data.IntelliSense.CslCommand.f6).toList(Kusto.Data.IntelliSense.CslCommandToken), $t.Text = Bridge.toArray(System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).selectMany($asm.$.Kusto.Data.IntelliSense.CslCommand.f7)).join(""), $t.IsRunnable = System.Linq.Enumerable.from(commands, Kusto.Data.IntelliSense.CslCommand).any($asm.$.Kusto.Data.IntelliSense.CslCommand.f8), $t.RelativeStart = firstCommand.RelativeStart, $t.AbsolutePositionBias = firstCommand.AbsolutePositionBias, $t.ParseMode = firstCommand.ParseMode, $t);
                    result.Length = result.Text.length;

                    return result;
                },
                NormalizeCommandPart: function (partValue) {
                    partValue = partValue.trim();
                    partValue = Kusto.Data.IntelliSense.CslCommand.s_removeWhiteSpacesAfterPipeRegex.replace(partValue, "| ");
                    return partValue;
                },
                AppendTabulatedText: function (sb, level, text) {
                    Kusto.Data.IntelliSense.CslCommand.AppendTabulations(sb, level);
                    sb.append(text);
                },
                AppendTabulations: function (sb, level) {
                    if (level <= 0) {
                        return;
                    }

                    for (var j = 0; j < level; j = (j + 1) | 0) {
                        sb.append(Kusto.Data.IntelliSense.CslCommand.c_tabulation);
                    }
                },
                ResolveOperatorContext: function (part) {
                    if (part == null) {
                        return "";
                    }

                    if (System.String.indexOf(part.Text, " on ") > 0 && !Kusto.Data.IntelliSense.CslCommand.s_makeSeriesOperatorRegex.isMatch(part.Text)) {
                        return "";
                    }

                    return Kusto.Data.IntelliSense.CslCommand.s_operatorRegex.match(part.Text).getGroups().getByName("Operator").toString();
                },
                GetKnownEntities: function (rulesProvider, mapOfKnownEntities, mapOfOriginallyKnownEntities, part, commandBuilder, context, operatorContext, knownEntities, originallyKnownEntities) {
                    knownEntities.v = new (System.Collections.Generic.List$1(System.String)).ctor();

                    if (mapOfOriginallyKnownEntities.containsKey(context)) {
                        originallyKnownEntities.v = mapOfOriginallyKnownEntities.getItem(context);
                    } else {
                        originallyKnownEntities.v = new (System.Collections.Generic.List$1(System.String)).ctor();
                    }

                    if (part == null) {
                        if (mapOfKnownEntities.containsKey(context)) {
                            knownEntities.v = mapOfKnownEntities.getItem(context);
                        }
                        return false;
                    }

                    // Check if entities can be pulled from the tables
                    if (Kusto.Data.IntelliSense.CslCommand.s_operatorsNotRequiringFullEntitiesResolve.contains(operatorContext)) {
                        if (mapOfKnownEntities.containsKey(context)) {
                            knownEntities.v = mapOfKnownEntities.getItem(context);
                        }
                        return false;
                    }

                    var command = commandBuilder.toString();
                    return rulesProvider.GetKnownEntities(command, context, mapOfOriginallyKnownEntities, knownEntities, originallyKnownEntities);
                },
                IsMatchingRegex: function (commandPart, opRegex) {
                    if (System.String.isNullOrWhiteSpace(commandPart)) {
                        return false;
                    }
                    return opRegex.isMatch(commandPart);
                },
                StartsWithAlpha: function (text) {
                    if (System.String.isNullOrWhiteSpace(text)) {
                        return false;
                    }
                    return Kusto.Data.IntelliSense.CslCommand.s_startsWithAlpha.isMatch(text);
                }
            }
        },
        fields: {
            m_tokens: null,
            m_commandParts: null,
            m_commentsParts: null,
            m_clientDirectivesParts: null,
            m_bracketParts: null,
            /**
             * The next list holds parse states for each of the command parts
             This allows reusing the state when the command is slowly mutating
             as happens when user changes last command part
             *
             * @instance
             * @private
             * @memberof Kusto.Data.IntelliSense.CslCommand
             * @type System.Collections.Generic.List$1
             */
            m_commandPartsParseStates: null
        },
        props: {
            CslExpressionStartPosition: {
                get: function () {
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$1(Kusto.Data.IntelliSense.CslCommandToken, this.m_commandParts)) {
                        return 0;
                    }

                    return this.m_commandParts.getItem(0).RelativeStart;
                }
            },
            CslExpressionLength: {
                get: function () {
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$1(Kusto.Data.IntelliSense.CslCommandToken, this.m_commandParts)) {
                        return 0;
                    }

                    return ((System.Linq.Enumerable.from(this.m_commandParts, Kusto.Data.IntelliSense.CslCommandToken).last().RelativeEnd - this.m_commandParts.getItem(0).RelativeStart) | 0);
                }
            },
            Tokens: {
                get: function () {
                    return this.m_tokens;
                }
            },
            CommandParts: {
                get: function () {
                    return this.m_commandParts;
                }
            },
            CommentParts: {
                get: function () {
                    return this.m_commentsParts;
                }
            },
            BracketParts: {
                get: function () {
                    return this.m_bracketParts;
                }
            },
            AllParts: {
                get: function () {
                    var linqParts = 0;
                    var result = null;
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$4(Kusto.Data.IntelliSense.CslCommandToken, this.m_commandParts)) {
                        linqParts = (linqParts + 1) | 0;
                        result = this.m_commandParts;
                    }

                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$4(Kusto.Data.IntelliSense.CslCommandToken, this.m_commentsParts)) {
                        linqParts = (linqParts + 1) | 0;
                        result = ((result != null) ? System.Linq.Enumerable.from(result, Kusto.Data.IntelliSense.CslCommandToken).union(this.m_commentsParts) : Bridge.cast(this.m_commentsParts, System.Collections.Generic.IEnumerable$1(Kusto.Data.IntelliSense.CslCommandToken)));
                    }

                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$4(Kusto.Data.IntelliSense.CslCommandToken, this.m_clientDirectivesParts)) {
                        linqParts = (linqParts + 1) | 0;
                        result = ((result != null) ? System.Linq.Enumerable.from(result, Kusto.Data.IntelliSense.CslCommandToken).union(this.m_clientDirectivesParts) : Bridge.cast(this.m_clientDirectivesParts, System.Collections.Generic.IEnumerable$1(Kusto.Data.IntelliSense.CslCommandToken)));
                    }

                    if (linqParts > 1) {
                        return System.Linq.Enumerable.from(result, Kusto.Data.IntelliSense.CslCommandToken).orderBy($asm.$.Kusto.Data.IntelliSense.CslCommand.f9);
                    }

                    return result;
                }
            },
            Text: null,
            RelativeStart: 0,
            Length: 0,
            RelativeEnd: {
                get: function () {
                    return ((((this.RelativeStart + this.Length) | 0) - 1) | 0);
                }
            },
            AbsoluteStart: {
                get: function () {
                    return ((this.AbsolutePositionBias + this.RelativeStart) | 0);
                }
            },
            AbsoluteEnd: {
                get: function () {
                    return ((this.AbsolutePositionBias + this.RelativeEnd) | 0);
                }
            },
            AbsolutePositionBias: 0,
            IsRunnable: false,
            ParseMode: 0,
            /**
             * Optimization: keep on the command its context cache (mapping between different command parts to the command context)
             *
             * @instance
             * @memberof Kusto.Data.IntelliSense.CslCommand
             * @function ContextCache
             * @type System.Collections.Generic.Dictionary$2
             */
            ContextCache: null
        },
        ctors: {
            ctor: function () {
                this.$initialize();
            }
        },
        methods: {
            FormatAsString: function (separator, traits) {
                var $t;
                if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$1(Kusto.Data.IntelliSense.CslCommandToken, this.m_commandParts)) {
                    return "";
                }

                var parts = this.m_commandParts;
                if (System.Enum.hasFlag(traits, Bridge.box(Kusto.Data.IntelliSense.CslCommand.FormatTraits.IncludeComments, Kusto.Data.IntelliSense.CslCommand.FormatTraits, System.Enum.toStringFn(Kusto.Data.IntelliSense.CslCommand.FormatTraits))) && Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$4(Kusto.Data.IntelliSense.CslCommandToken, this.m_commentsParts)) {
                    parts = System.Linq.Enumerable.from(parts, Kusto.Data.IntelliSense.CslCommandToken).union(this.m_commentsParts).union(this.m_clientDirectivesParts).orderBy($asm.$.Kusto.Data.IntelliSense.CslCommand.f10).toList(Kusto.Data.IntelliSense.CslCommandToken);
                }

                var sb = new System.Text.StringBuilder();
                var level = { v : 0 };
                var isFirstTime = true;
                $t = Bridge.getEnumerator(parts);
                try {
                    while ($t.moveNext()) {
                        var part = $t.Current;
                        var partValue = Kusto.Data.IntelliSense.CslCommand.s_newLineRegex.replace(part.Value, " ");

                        if (!isFirstTime) {
                            sb.append(separator);
                        }
                        isFirstTime = false;
                        // Append tabulations
                        Kusto.Data.IntelliSense.CslCommand.AppendTabulations(sb, level.v);

                        var wasHandled = false;
                        if (!wasHandled && System.Enum.hasFlag(traits, Bridge.box(Kusto.Data.IntelliSense.CslCommand.FormatTraits.IncludeComments, Kusto.Data.IntelliSense.CslCommand.FormatTraits, System.Enum.toStringFn(Kusto.Data.IntelliSense.CslCommand.FormatTraits)))) {
                            wasHandled = this.HandleCommentsAndClientDirectives(separator, sb, level, part, partValue);
                        }

                        if (!wasHandled && System.Enum.hasFlag(traits, Bridge.box(Kusto.Data.IntelliSense.CslCommand.FormatTraits.TabulateOnFunctionBoundaries, Kusto.Data.IntelliSense.CslCommand.FormatTraits, System.Enum.toStringFn(Kusto.Data.IntelliSense.CslCommand.FormatTraits)))) {
                            wasHandled = this.HandleFunctions(separator, sb, level, part);
                        }

                        if (!wasHandled && System.Enum.hasFlag(traits, Bridge.box(Kusto.Data.IntelliSense.CslCommand.FormatTraits.TabulateOnJoins, Kusto.Data.IntelliSense.CslCommand.FormatTraits, System.Enum.toStringFn(Kusto.Data.IntelliSense.CslCommand.FormatTraits)))) {
                            wasHandled = this.HandleJoins(separator, sb, level, part, partValue);
                        }

                        if (!wasHandled) {
                            sb.append(Kusto.Data.IntelliSense.CslCommand.NormalizeCommandPart(partValue));
                        }
                    }
                } finally {
                    if (Bridge.is($t, System.IDisposable)) {
                        $t.System$IDisposable$Dispose();
                    }
                }

                return sb.toString();
            },
            HandleCommentsAndClientDirectives: function (separator, sb, level, part, partValue) {
                if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.None$1(Kusto.Data.IntelliSense.CslCommandToken, System.Linq.Enumerable.from(this.m_commentsParts, Kusto.Data.IntelliSense.CslCommandToken).union(this.m_clientDirectivesParts), function (cp) {
                    return cp.AbsoluteStart === part.AbsoluteStart && cp.AbsoluteEnd === part.AbsoluteEnd;
                })) {
                    return false;
                }

                sb.append(partValue.trim());
                return true;
            },
            HandleFunctions: function (separator, sb, level, part) {
                var wasHandled = false;
                var processedStart = 0;
                var startRelativeIndex = System.String.indexOf(part.Value, String.fromCharCode(123));
                var startAbsoluteIndex = (part.AbsoluteStart + startRelativeIndex) | 0;

                if (startRelativeIndex >= 0 && Kusto.Cloud.Platform.Utils.ExtendedEnumerable.None$1(Kusto.Data.IntelliSense.CslCommandToken, this.m_tokens, function (t) {
                    return startAbsoluteIndex >= t.AbsoluteStart && startAbsoluteIndex <= t.AbsoluteEnd;
                })) {
                    var funcPrefix = part.Value.substr(0, startRelativeIndex).trim();
                    funcPrefix = Kusto.Data.IntelliSense.CslCommand.s_newLineRegex.replace(funcPrefix, " ");
                    sb.append(funcPrefix);
                    sb.append(separator);
                    sb.append("{");
                    sb.append(separator);
                    level.v = (level.v + 1) | 0;
                    wasHandled = true;
                    processedStart = (startRelativeIndex + 1) | 0;
                }


                var endRelativeIndex = System.String.indexOf(part.Value, String.fromCharCode(125));
                var endAbsoluteIndex = (part.AbsoluteStart + endRelativeIndex) | 0;
                if (endRelativeIndex >= 0 && Kusto.Cloud.Platform.Utils.ExtendedEnumerable.None$1(Kusto.Data.IntelliSense.CslCommandToken, this.m_tokens, function (t) {
                    return endAbsoluteIndex >= t.AbsoluteStart && endAbsoluteIndex <= t.AbsoluteEnd;
                }) && startRelativeIndex < endRelativeIndex) {
                    var boddy = part.Value.substr(processedStart, ((endRelativeIndex - processedStart) | 0)).trim();
                    var funcSuffix = part.Value.substr(((endRelativeIndex + 1) | 0)).trim();
                    funcSuffix = Kusto.Data.IntelliSense.CslCommand.s_newLineRegex.replace(funcSuffix, " ");
                    if (wasHandled) {
                        Kusto.Data.IntelliSense.CslCommand.AppendTabulations(sb, level.v);
                    }
                    sb.append(boddy);
                    sb.append(separator);
                    sb.append("}");
                    level.v = (level.v - 1) | 0;
                    if (level.v < 0) {
                        level.v = 0;
                    }
                    Kusto.Data.IntelliSense.CslCommand.AppendTabulatedText(sb, level.v, funcSuffix);
                    wasHandled = true;
                } else if (wasHandled) {
                    // This is a case where function was started in this cycle, and start of the function has to be tabulated
                    var boddy1 = part.Value.substr(processedStart).trim();
                    boddy1 = Kusto.Data.IntelliSense.CslCommand.s_newLineRegex.replace(boddy1, " ");
                    Kusto.Data.IntelliSense.CslCommand.AppendTabulatedText(sb, level.v, boddy1);
                }

                return wasHandled;
            },
            HandleJoins: function (separator, sb, level, part, partValue) {
                var assumeJoinStartedWithBracket = false;
                var wasHandled = false;
                var remainingText = partValue;
                var joinStartMatch = Kusto.Data.IntelliSense.CslCommand.s_joinStartRegex.match(remainingText);
                var relativePosition = 0;
                if (joinStartMatch.getSuccess()) {
                    var joinOpPart = joinStartMatch.getGroups().getByName("JoinOpPart").toString();
                    sb.append(Kusto.Data.IntelliSense.CslCommand.NormalizeCommandPart(joinOpPart));
                    sb.append(separator);
                    assumeJoinStartedWithBracket = !System.String.isNullOrEmpty(joinStartMatch.getGroups().getByName("Bracket").toString());
                    level.v = (level.v + 1) | 0;
                    remainingText = joinStartMatch.getGroups().getByName("PostJoinPart").toString();
                    relativePosition = joinStartMatch.getGroups().getByName("PostJoinPart").getIndex();
                    wasHandled = true;
                }

                var joinEndMatch = Kusto.Data.IntelliSense.CslCommand.s_joinEndRegex.match(remainingText);
                if (joinEndMatch.getSuccess() && System.Linq.Enumerable.from(this.m_tokens, Kusto.Data.IntelliSense.CslCommandToken).any(function (t) {
                        return t.TokenKind === Kusto.Data.IntelliSense.CslCommandToken.Kind.SubOperatorToken && Bridge.referenceEquals(t.Value, "on") && t.AbsoluteStart === ((((joinEndMatch.getGroups().getByName("JoinOnPart").getIndex() + part.AbsoluteStart) | 0) + relativePosition) | 0);
                    })) {
                    var joinInnerPart = Kusto.Data.IntelliSense.CslCommand.NormalizeCommandPart(joinEndMatch.getGroups().getByName("InnerJoinPart").toString());
                    var onJoinPart = Kusto.Data.IntelliSense.CslCommand.NormalizeCommandPart(joinEndMatch.getGroups().getByName("JoinOnPart").toString());

                    var hasBracket = !System.String.isNullOrEmpty(joinEndMatch.getGroups().getByName("Bracket").toString());
                    var shouldCloseBracket = hasBracket || assumeJoinStartedWithBracket;

                    var openBrackets = Kusto.Cloud.Platform.Utils.ExtendedString.CountNonOverlappingSubstrings(joinInnerPart, "(");
                    var closeBrackets = Kusto.Cloud.Platform.Utils.ExtendedString.CountNonOverlappingSubstrings(joinInnerPart, ")");
                    if (hasBracket) {
                        if (openBrackets > closeBrackets) {
                            shouldCloseBracket = false;
                            joinInnerPart = (joinInnerPart || "") + ")";
                        }
                    }

                    if (joinStartMatch.getSuccess()) {
                        if (assumeJoinStartedWithBracket) {
                            Kusto.Data.IntelliSense.CslCommand.AppendTabulatedText(sb, ((level.v - 1) | 0), "(");
                            sb.append(separator);
                        }
                        Kusto.Data.IntelliSense.CslCommand.AppendTabulations(sb, level.v);
                    }

                    level.v = (level.v - 1) | 0;
                    if (level.v < 0) {
                        level.v = 0;
                    }
                    if (!System.String.isNullOrEmpty(joinInnerPart)) {
                        sb.append(joinInnerPart);
                        sb.append(separator);
                        if (shouldCloseBracket) {
                            Kusto.Data.IntelliSense.CslCommand.AppendTabulatedText(sb, level.v, ")");
                            sb.append(separator);
                        }

                        Kusto.Data.IntelliSense.CslCommand.AppendTabulatedText(sb, level.v, Kusto.Data.IntelliSense.CslCommand.NormalizeCommandPart(onJoinPart));
                    } else {
                        //if joinInnerPart is not available, trimming trailing tabulation
                        var tabLength = Kusto.Data.IntelliSense.CslCommand.c_tabulation.length;
                        sb.replace(Kusto.Data.IntelliSense.CslCommand.c_tabulation, "", ((sb.getLength() - tabLength) | 0), tabLength);
                        var bracketClosed = false;
                        if (shouldCloseBracket) {
                            sb.append(")");
                            sb.append(separator);
                            bracketClosed = true;
                        }

                        if (bracketClosed) {
                            Kusto.Data.IntelliSense.CslCommand.AppendTabulatedText(sb, level.v, Kusto.Data.IntelliSense.CslCommand.NormalizeCommandPart(onJoinPart));
                        } else {
                            sb.append(onJoinPart);
                        }
                    }

                    wasHandled = true;
                } else if (joinStartMatch.getSuccess()) {
                    if (assumeJoinStartedWithBracket) {
                        Kusto.Data.IntelliSense.CslCommand.AppendTabulatedText(sb, ((level.v - 1) | 0), "(");
                        sb.append(separator);
                    }
                    Kusto.Data.IntelliSense.CslCommand.AppendTabulatedText(sb, level.v, Kusto.Data.IntelliSense.CslCommand.NormalizeCommandPart(remainingText));
                }

                return wasHandled;
            },
            /**
             * Optimization: instead of re-parsing tokens, copy them from the already parsed command that has the same text
             *
             * @instance
             * @this Kusto.Data.IntelliSense.CslCommand
             * @memberof Kusto.Data.IntelliSense.CslCommand
             * @param   {Kusto.Data.IntelliSense.CslCommand}    other
             * @return  {void}
             */
            AcquireTokens: function (other) {
                this.m_tokens = System.Linq.Enumerable.from(other.m_tokens, Kusto.Data.IntelliSense.CslCommandToken).select(Bridge.fn.bind(this, $asm.$.Kusto.Data.IntelliSense.CslCommand.f11)).toList(Kusto.Data.IntelliSense.CslCommandToken);

                this.m_commandParts = System.Linq.Enumerable.from(other.m_commandParts, Kusto.Data.IntelliSense.CslCommandToken).select(Bridge.fn.bind(this, $asm.$.Kusto.Data.IntelliSense.CslCommand.f11)).toList(Kusto.Data.IntelliSense.CslCommandToken);

                this.m_commentsParts = System.Linq.Enumerable.from(other.m_commentsParts, Kusto.Data.IntelliSense.CslCommandToken).select(Bridge.fn.bind(this, $asm.$.Kusto.Data.IntelliSense.CslCommand.f11)).toList(Kusto.Data.IntelliSense.CslCommandToken);

                this.m_clientDirectivesParts = System.Linq.Enumerable.from(other.m_clientDirectivesParts, Kusto.Data.IntelliSense.CslCommandToken).select(Bridge.fn.bind(this, $asm.$.Kusto.Data.IntelliSense.CslCommand.f11)).toList(Kusto.Data.IntelliSense.CslCommandToken);

                this.m_bracketParts = System.Linq.Enumerable.from(other.m_bracketParts, Kusto.Data.IntelliSense.CslCommandToken).select(Bridge.fn.bind(this, $asm.$.Kusto.Data.IntelliSense.CslCommand.f11)).toList(Kusto.Data.IntelliSense.CslCommandToken);
            },
            /**
             * Re-parse the command string
             *
             * @instance
             * @this Kusto.Data.IntelliSense.CslCommand
             * @memberof Kusto.Data.IntelliSense.CslCommand
             * @param   {Kusto.Data.IntelliSense.CslIntelliSenseRulesProvider}    rulesProvider        
             * @param   {Kusto.Data.IntelliSense.ParseMode}                       parseMode            
             * @param   {Kusto.Data.IntelliSense.CslCommand}                      previousCandidate
             * @return  {void}
             */
            ParseTokens: function (rulesProvider, parseMode, previousCandidate) {
                var tokens = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandToken)).ctor();
                if (System.String.isNullOrEmpty(this.Text)) {
                    this.m_tokens = tokens;
                    return;
                }

                if (rulesProvider != null) {
                    rulesProvider.ResetState();
                    if (previousCandidate != null && Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$1(System.Collections.Generic.KeyValuePair$2(System.Int32,Kusto.Data.IntelliSense.KustoCommandContext), previousCandidate.ContextCache)) {
                        rulesProvider.ContextCache = new (System.Collections.Generic.Dictionary$2(System.Int32,Kusto.Data.IntelliSense.KustoCommandContext)).$ctor1(previousCandidate.ContextCache);
                    }
                }

                // Create a text indexer that takes care of breaking text into simple tokens and 
                // basic constructs (comments, string literals, rest)
                var allowQueryParameters = (rulesProvider != null) ? rulesProvider.AllowQueryParameters : false;
                var indexer = new Kusto.Data.IntelliSense.CslCommandIndexer(allowQueryParameters);
                indexer.AntiTokenizers = new (System.Collections.Generic.HashSet$1(System.Char)).$ctor1(System.Array.init([45, 95, 40], System.Char));
                indexer.TokenStarters = System.Array.init([46], System.Char);
                indexer.TokenTerminators = new (System.Collections.Generic.HashSet$1(System.Char)).$ctor1(System.Array.init([40, 46], System.Char));
                indexer.IndexText(this.Text);

                var recognizedTokens = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)).ctor();
                this.m_commandParts = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandToken)).ctor();
                var commandParts = indexer.GetCommandPartsPositions();
                this.AddCategorizedTokens(this.m_commandParts, null, commandParts, Kusto.Data.IntelliSense.CslCommandToken.Kind.CommandPartToken);

                this.m_commentsParts = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandToken)).ctor();
                var commentsParts = indexer.GetCommentsPositions();
                this.AddCategorizedTokens(this.m_commentsParts, null, commentsParts, Kusto.Data.IntelliSense.CslCommandToken.Kind.CommentToken);

                this.m_clientDirectivesParts = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandToken)).ctor();
                var clientDirectivesPositions = indexer.GetClientDirectivesPositions();
                this.AddCategorizedTokens(this.m_clientDirectivesParts, null, clientDirectivesPositions, Kusto.Data.IntelliSense.CslCommandToken.Kind.ClientDirectiveToken);

                this.m_bracketParts = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandToken)).ctor();
                this.AddCategorizedTokens(this.m_bracketParts, null, indexer.GetBracketsPositions(), Kusto.Data.IntelliSense.CslCommandToken.Kind.BracketRangeToken);

                this.AddCategorizedTokens(tokens, recognizedTokens, commentsParts, Kusto.Data.IntelliSense.CslCommandToken.Kind.CommentToken);
                this.AddCategorizedTokens(tokens, recognizedTokens, clientDirectivesPositions, Kusto.Data.IntelliSense.CslCommandToken.Kind.ClientDirectiveToken);
                if (allowQueryParameters) {
                    this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetQueryParametersPositions(), Kusto.Data.IntelliSense.CslCommandToken.Kind.QueryParametersToken);
                }
                this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetStringLiteralsPositions(), Kusto.Data.IntelliSense.CslCommandToken.Kind.StringLiteralToken);
                this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetAllTokenPositions(Kusto.Data.IntelliSense.CslCommandParser.ControlCommandsTokens), Kusto.Data.IntelliSense.CslCommandToken.Kind.ControlCommandToken);
                this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetAllTokenPositions(Kusto.Data.IntelliSense.CslCommandParser.CslCommandsTokens), Kusto.Data.IntelliSense.CslCommandToken.Kind.CslCommandToken);
                this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetAllTokenPositions(Kusto.Data.IntelliSense.CslCommandParser.OperatorCommandTokens), Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken);
                this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetAllTokenPositions(Kusto.Data.IntelliSense.CslCommandParser.SubOperatorsTokens), Kusto.Data.IntelliSense.CslCommandToken.Kind.SubOperatorToken);
                this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetAllTokenPositions(Kusto.Data.IntelliSense.CslCommandParser.JoinKindTokens), Kusto.Data.IntelliSense.CslCommandToken.Kind.SubOperatorToken);
                this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetAllTokenPositions(Kusto.Data.IntelliSense.CslCommandParser.ReduceByKindTokens), Kusto.Data.IntelliSense.CslCommandToken.Kind.SubOperatorToken);
                this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetAllTokenPositions(Kusto.Data.IntelliSense.CslCommandParser.DataTypesTokens), Kusto.Data.IntelliSense.CslCommandToken.Kind.DataTypeToken);
                this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetAllTokenPositions(Kusto.Data.IntelliSense.CslCommandParser.FunctionsTokens, 40), Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken);
                this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetAllTokenPositions(Kusto.Data.IntelliSense.CslCommandParser.PluginTokens, 40), Kusto.Data.IntelliSense.CslCommandToken.Kind.PluginToken);
                if (rulesProvider != null) {
                    this.AddEntitiesTokens(rulesProvider, tokens, recognizedTokens, indexer, commandParts, previousCandidate);
                }

                tokens.Sort$2($asm.$.Kusto.Data.IntelliSense.CslCommand.f12); // In-place sorting
                // Find tokens that are exceeding end or overlapping
                this.ValidateTokensOutput(tokens, this.Text);

                if (parseMode === Kusto.Data.IntelliSense.ParseMode.TokenizeAllText) {
                    this.EnsureAllTextIsAnnotated(indexer, tokens, recognizedTokens);
                    tokens.Sort$2($asm.$.Kusto.Data.IntelliSense.CslCommand.f12); // In-place sorting
                }

                this.m_tokens = tokens;

                // Capture command context cache from rules provider and store it on the command
                if (rulesProvider != null) {
                    this.ContextCache = rulesProvider.ContextCache;
                }
            },
            ValidateTokensOutput: function (tokens, text) {
                var $t;
                var invalidTokens = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandToken)).ctor();
                var currentEnd = 0;
                var maxLen = text.length;

                // Look for overlapping tokens or those that exceed text length
                for (var i = 0; i < System.Linq.Enumerable.from(tokens, Kusto.Data.IntelliSense.CslCommandToken).count(); i = (i + 1) | 0) {
                    var t = tokens.getItem(i);
                    if (t.RelativeEnd < currentEnd || t.RelativeStart < currentEnd || t.RelativeEnd > maxLen) {
                        invalidTokens.add(t);
                    } else {
                        currentEnd = t.RelativeEnd;
                    }
                }

                $t = Bridge.getEnumerator(invalidTokens);
                try {
                    while ($t.moveNext()) {
                        var tokenToRemove = $t.Current;
                        tokens.remove(tokenToRemove);
                    }
                } finally {
                    if (Bridge.is($t, System.IDisposable)) {
                        $t.System$IDisposable$Dispose();
                    }
                }
            },
            /**
             * Adds tokens that consist of tables, columns and calculated columns entities
             *
             * @instance
             * @private
             * @this Kusto.Data.IntelliSense.CslCommand
             * @memberof Kusto.Data.IntelliSense.CslCommand
             * @param   {Kusto.Data.IntelliSense.CslIntelliSenseRulesProvider}    rulesProvider       
             * @param   {System.Collections.Generic.List$1}                       tokens              
             * @param   {System.Collections.Generic.List$1}                       recognizedTokens    
             * @param   {Kusto.Data.IntelliSense.CslCommandIndexer}               indexer             
             * @param   {System.Collections.Generic.IList$1}                      commandParts        
             * @param   {Kusto.Data.IntelliSense.CslCommand}                      refernceCommand
             * @return  {void}
             */
            AddEntitiesTokens: function (rulesProvider, tokens, recognizedTokens, indexer, commandParts, refernceCommand) {
                // Mark table names
                this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetAllTokenPositions(rulesProvider.TableNames), Kusto.Data.IntelliSense.CslCommandToken.Kind.TableToken);

                this.m_commandPartsParseStates = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommand.AddEntitiesTokensState)).ctor();

                // Now we have a list of tokens
                var commandBuilder = new System.Text.StringBuilder();

                var letVariables = new (System.Collections.Generic.HashSet$1(System.String)).$ctor1(rulesProvider.FunctionNames); // Consume all default functions

                var canUseReference = (refernceCommand != null);
                var state = null;
                var analyzedCommand = null;

                for (var i = 0; i < ((System.Array.getCount(commandParts, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition) + 1) | 0); i = (i + 1) | 0) {
                    var prevPart = (i > 0) ? System.Array.getItem(commandParts, ((i - 1) | 0), Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition) : null;
                    var part = { v : (i < System.Array.getCount(commandParts, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)) ? System.Array.getItem(commandParts, i, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition) : null };
                    state = (state != null) ? state.Clone() : new Kusto.Data.IntelliSense.CslCommand.AddEntitiesTokensState();

                    if (part.v != null) {
                        commandBuilder.append(part.v.Text);
                    }

                    // check if reference part is equal to the current part
                    // It's pity that it is impossible to use object?.property notation
                    // since bridge.net does not support this
                    canUseReference = !!(canUseReference & ((part.v != null && refernceCommand != null && Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$2(Kusto.Data.IntelliSense.CslCommandToken, refernceCommand.CommandParts)) && Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$4(Kusto.Data.IntelliSense.CslCommand.AddEntitiesTokensState, refernceCommand.m_commandPartsParseStates) && System.Linq.Enumerable.from(refernceCommand.m_commandPartsParseStates, Kusto.Data.IntelliSense.CslCommand.AddEntitiesTokensState).count() > i && System.Linq.Enumerable.from(refernceCommand.CommandParts, Kusto.Data.IntelliSense.CslCommandToken).count() > i && System.String.equals(part.v.Text, System.Linq.Enumerable.from(refernceCommand.CommandParts, Kusto.Data.IntelliSense.CslCommandToken).elementAt(i).Value)));

                    if (canUseReference) {
                        var referencePart = System.Linq.Enumerable.from(refernceCommand.CommandParts, Kusto.Data.IntelliSense.CslCommandToken).elementAt(i);
                        var offset = { v : (part.v.Start - referencePart.RelativeStart) | 0 };
                        var capturedTokens = System.Linq.Enumerable.from(refernceCommand.Tokens, Kusto.Data.IntelliSense.CslCommandToken).where((function ($me, part) {
                                return function (t) {
                                    return (t.TokenKind === Kusto.Data.IntelliSense.CslCommandToken.Kind.CalculatedColumnToken || t.TokenKind === Kusto.Data.IntelliSense.CslCommandToken.Kind.TableColumnToken || t.TokenKind === Kusto.Data.IntelliSense.CslCommandToken.Kind.TableToken || t.TokenKind === Kusto.Data.IntelliSense.CslCommandToken.Kind.LetVariablesToken) && (t.RelativeStart >= part.v.Start && t.RelativeEnd <= part.v.End);
                                };
                            })(this, part)).select((function ($me, offset) {
                            return function (t) {
                                // Shift by the offset relative to the old token command
                                var clonedToken = Bridge.as(t.clone(), Kusto.Data.IntelliSense.CslCommandToken);
                                clonedToken.RelativeStart = (clonedToken.RelativeStart + offset.v) | 0;
                                return clonedToken;
                            };
                        })(this, offset)).ToArray(Kusto.Data.IntelliSense.CslCommandToken);

                        tokens.AddRange(capturedTokens);
                        recognizedTokens.AddRange(indexer.GetTokenPositionsInRange(System.Linq.Enumerable.from(capturedTokens, Kusto.Data.IntelliSense.CslCommandToken).select($asm.$.Kusto.Data.IntelliSense.CslCommand.f13), part.v.Start, part.v.End));

                        this.AddLetStatementTokens(tokens, recognizedTokens, indexer, letVariables, part.v);

                        state = refernceCommand.m_commandPartsParseStates.getItem(i).Clone();
                        this.m_commandPartsParseStates.add(state);
                        continue; // Move to next part
                    }

                    if (part.v != null) {
                        this.AddLetStatementTokens(tokens, recognizedTokens, indexer, letVariables, part.v);
                    }

                    if (analyzedCommand == null) {
                        var command = commandBuilder.toString();
                        analyzedCommand = rulesProvider.AnalyzeCommand$1(command, refernceCommand);
                    } else if (part.v != null) {
                        analyzedCommand = rulesProvider.AnalyzeCommand(analyzedCommand, part.v.Text);
                    }

                    var tableContext = analyzedCommand.Context;
                    if (tableContext.IsEmpty()) {
                        continue;
                    }

                    var operatorContext = Kusto.Data.IntelliSense.CslCommand.ResolveOperatorContext(part.v);

                    var knownEntities = { };
                    var originallyKnownEntities = { };
                    var wasFullyResolved = Kusto.Data.IntelliSense.CslCommand.GetKnownEntities(rulesProvider, state.MapOfKnownEntities, state.MapOfOriginallyKnownEntities, part.v, commandBuilder, tableContext, operatorContext, knownEntities, originallyKnownEntities);

                    if (part.v != null) {
                        // The next updates list of remote tables
                        rulesProvider.ResolveKnownEntitiesFromContext(tableContext);

                        var calculatedColumns = System.Linq.Enumerable.from(knownEntities.v, System.String).except(originallyKnownEntities.v);
                        var availableTableColumns = System.Linq.Enumerable.from(originallyKnownEntities.v, System.String).intersect(knownEntities.v);
                        this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetTokenPositionsInRange(availableTableColumns, part.v.Start, part.v.End), Kusto.Data.IntelliSense.CslCommandToken.Kind.TableColumnToken);
                        this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetTokenPositionsInRange(calculatedColumns, part.v.Start, part.v.End), Kusto.Data.IntelliSense.CslCommandToken.Kind.CalculatedColumnToken);
                        this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetTokenPositionsInRange(rulesProvider.RemoteTableNames, part.v.Start, part.v.End), Kusto.Data.IntelliSense.CslCommandToken.Kind.TableToken);
                    }


                    if (!wasFullyResolved && part.v != null) {
                        var newCalculatedColumns = new (System.Collections.Generic.List$1(System.String)).ctor();
                        var result = rulesProvider.ResolveEntitiesFromCommand((part.v.Text || "") + " | ", newCalculatedColumns, knownEntities.v);
                        switch (result) {
                            case Kusto.Data.IntelliSense.CslQueryIntelliSenseRulesProvider.ResolveResult.ReplaceEntities: 
                                knownEntities.v = newCalculatedColumns;
                                break;
                            case Kusto.Data.IntelliSense.CslQueryIntelliSenseRulesProvider.ResolveResult.None: 
                                break;
                            case Kusto.Data.IntelliSense.CslQueryIntelliSenseRulesProvider.ResolveResult.AppendEntities: 
                                knownEntities.v = System.Linq.Enumerable.from(knownEntities.v, System.String).union(newCalculatedColumns).toList(System.String);
                                break;
                        }
                    }

                    state.MapOfKnownEntities.setItem(tableContext, knownEntities.v);

                    // If the operation has an assignment operation - handle it by having another iteration of the appended text
                    if (prevPart != null && Kusto.Data.IntelliSense.CslCommand.IsMatchingRegex(prevPart.Text, Kusto.Data.IntelliSense.CslCommand.s_hasAssignmentOperationRegex) && state.MapOfPreviousCalculatedEntities.containsKey(tableContext)) {
                        var prevCalculatedColumns = state.MapOfPreviousCalculatedEntities.getItem(tableContext);
                        if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$4(System.String, prevCalculatedColumns)) {
                            var calculatedColumnTokens = indexer.GetTokenPositionsInRange(prevCalculatedColumns, prevPart.Start, prevPart.End);
                            if (System.Linq.Enumerable.from(calculatedColumnTokens, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).any()) {
                                this.AddCategorizedTokens(tokens, recognizedTokens, calculatedColumnTokens, Kusto.Data.IntelliSense.CslCommandToken.Kind.CalculatedColumnToken);
                            }
                        }
                    }

                    state.MapOfPreviousCalculatedEntities.setItem(tableContext, System.Linq.Enumerable.from(knownEntities.v, System.String).except(originallyKnownEntities.v).toList(System.String));
                    this.m_commandPartsParseStates.add(state);

                }
            },
            AddLetStatementTokens: function (tokens, recognizedTokens, indexer, letVariables, part) {
                // Collect new let variables
                var letVariable = Kusto.Data.IntelliSense.CslQueryIntelliSenseRulesProvider.ResolveLetVariable(part.Text);
                if (!System.String.isNullOrEmpty(letVariable)) {
                    letVariables.add(letVariable);
                }
                if (letVariables.Count > 0) {
                    this.AddCategorizedTokens(tokens, recognizedTokens, indexer.GetTokenPositionsInRange(letVariables, part.Start, part.End), Kusto.Data.IntelliSense.CslCommandToken.Kind.LetVariablesToken);
                }
            },
            AddCategorizedTokens: function (target, recognizedTokenList, textTokens, tokenKind) {
                if (recognizedTokenList != null) {
                    recognizedTokenList.AddRange(textTokens);
                }
                target.AddRange(System.Linq.Enumerable.from(textTokens, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).select(Bridge.fn.bind(this, function (t) {
                        var $t;
                        return ($t = new Kusto.Data.IntelliSense.CslCommandToken(t.Text, this.RelativeStart, tokenKind), $t.Length = t.Length, $t.RelativeStart = t.Start, $t);
                    })));
            },
            /**
             * When requested - this function takes care that between all previous tokens that were
             recognized - we will add a 'Unrecognized' tokens (e.g. so those can have specific color in UI)
             *
             * @instance
             * @private
             * @this Kusto.Data.IntelliSense.CslCommand
             * @memberof Kusto.Data.IntelliSense.CslCommand
             * @param   {Kusto.Data.IntelliSense.CslCommandIndexer}    indexer             
             * @param   {System.Collections.Generic.List$1}            tokens              
             * @param   {System.Collections.Generic.List$1}            recognizedTokens
             * @return  {void}
             */
            EnsureAllTextIsAnnotated: function (indexer, tokens, recognizedTokens) {
                if (tokens == null) {
                    return;
                }

                this.AddUnrecognizedTokens(indexer, tokens, recognizedTokens);

                tokens.Sort$2($asm.$.Kusto.Data.IntelliSense.CslCommand.f12);

                var start = 0;
                // After all other tokens are present
                var tokensCount = tokens.Count; // cache this value = we are going to add more items to the list
                for (var i = 0; i < tokensCount; i = (i + 1) | 0) {
                    var token = tokens.getItem(i);
                    if (start < token.RelativeStart) {
                        var newTokenLen = (token.RelativeStart - start) | 0;
                        if (newTokenLen > 0) {
                            var tokenText = this.Text.substr(start, newTokenLen);
                            this.AddPlainOrUnrecognizedTokens(tokens, start, tokenText);
                        }
                    }
                    start = token.RelativeEnd;
                }

                // Add text tokens
                if (start < this.Length) {
                    var tokenText1 = this.Text.substr(start, ((this.Length - start) | 0));
                    this.AddPlainOrUnrecognizedTokens(tokens, start, tokenText1);
                }
            },
            AddUnrecognizedTokens: function (indexer, tokens, recognizedTokens) {
                var $t;
                var unrecognizedTokens = indexer.GetUnrecognizedTokenPositions(recognizedTokens);
                $t = Bridge.getEnumerator(unrecognizedTokens, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition);
                try {
                    while ($t.moveNext()) {
                        var t = $t.Current;
                        this.AddPlainOrUnrecognizedTokens(tokens, t.Start, t.Text);
                    }
                } finally {
                    if (Bridge.is($t, System.IDisposable)) {
                        $t.System$IDisposable$Dispose();
                    }
                }
            },
            AddPlainOrUnrecognizedTokens: function (tokens, tokenStart, text) {
                var $t;
                var chars = System.String.toCharArray(text, 0, text.length);
                var startTextIndex = -1;
                var endTextIndex = chars.length;
                for (var i = 0; i < chars.length; i = (i + 1) | 0) {
                    if (!System.Char.isWhiteSpace(String.fromCharCode(chars[System.Array.index(i, chars)]))) {
                        endTextIndex = (i + 1) | 0;
                        startTextIndex = (startTextIndex >= 0) ? startTextIndex : i;
                    }
                }

                if (startTextIndex > 0) {
                    var prefixText = text.substr(0, startTextIndex);
                    var prefixToken = ($t = new Kusto.Data.IntelliSense.CslCommandToken(prefixText, this.RelativeStart, Kusto.Data.IntelliSense.CslCommandToken.Kind.PlainTextToken), $t.Length = startTextIndex, $t.RelativeStart = tokenStart, $t);
                    tokens.add(prefixToken);
                } else {
                    startTextIndex = 0;
                }

                var tokenText = text.substr(startTextIndex, ((endTextIndex - startTextIndex) | 0));
                var tokenKind = Kusto.Data.IntelliSense.CslCommand.StartsWithAlpha(tokenText) ? Kusto.Data.IntelliSense.CslCommandToken.Kind.UnknownToken : Kusto.Data.IntelliSense.CslCommandToken.Kind.PlainTextToken;
                var token = ($t = new Kusto.Data.IntelliSense.CslCommandToken(tokenText, this.RelativeStart, tokenKind), $t.Length = tokenText.length, $t.RelativeStart = ((tokenStart + startTextIndex) | 0), $t);
                tokens.add(token);

                if (endTextIndex < chars.length) {
                    var suffixText = text.substr(endTextIndex, ((chars.length - endTextIndex) | 0));
                    var suffixToken = ($t = new Kusto.Data.IntelliSense.CslCommandToken(suffixText, this.RelativeStart, Kusto.Data.IntelliSense.CslCommandToken.Kind.PlainTextToken), $t.Length = suffixText.length, $t.RelativeStart = ((tokenStart + endTextIndex) | 0), $t);
                    tokens.add(suffixToken);
                }
            }
        }
    });

    Bridge.ns("Kusto.Data.IntelliSense.CslCommand", $asm.$);

    Bridge.apply($asm.$.Kusto.Data.IntelliSense.CslCommand, {
        f1: function (_o1) {
            _o1.add("distinct");
            _o1.add("extend");
            _o1.add("project");
            _o1.add("project-rename");
            _o1.add("mvexpand");
            _o1.add("mv-expand");
            _o1.add("where");
            _o1.add("filter");
            _o1.add("parse");
            _o1.add("limit");
            _o1.add("count");
            _o1.add("order");
            _o1.add("sort");
            _o1.add("reduce");
            _o1.add("take");
            _o1.add("summarize");
            _o1.add("top");
            _o1.add("top-nested");
            _o1.add("top-hitters");
            _o1.add("render");
            _o1.add("fork");
            _o1.add("find");
            _o1.add("sample");
            _o1.add("sample-distinct");
            _o1.add("search");
            _o1.add("make-series");
            return _o1;
        },
        f2: function (c) {
            return c.m_tokens;
        },
        f3: function (c) {
            return c.m_commandParts;
        },
        f4: function (c) {
            return c.m_commentsParts;
        },
        f5: function (c) {
            return c.m_clientDirectivesParts;
        },
        f6: function (c) {
            return c.m_bracketParts;
        },
        f7: function (c) {
            return c.Text;
        },
        f8: function (c) {
            return c.IsRunnable;
        },
        f9: function (p) {
            return p.AbsoluteStart;
        },
        f10: function (p) {
            return p.RelativeStart;
        },
        f11: function (t) {
            var token = Bridge.as(t.clone(), Kusto.Data.IntelliSense.CslCommandToken);
            token.AbsolutePositionBias = this.RelativeStart;
            return token;
        },
        f12: function (x, y) {
            return x.CompareTo(y);
        },
        f13: function (ct) {
            return ct.Value;
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CslCommand.AddEntitiesTokensState", {
        $kind: "nested class",
        fields: {
            MapOfKnownEntities: null,
            MapOfPreviousCalculatedEntities: null,
            MapOfOriginallyKnownEntities: null
        },
        ctors: {
            init: function () {
                this.MapOfKnownEntities = new (System.Collections.Generic.Dictionary$2(Kusto.Data.IntelliSense.KustoCommandContext,System.Collections.Generic.List$1(System.String))).ctor();
                this.MapOfPreviousCalculatedEntities = new (System.Collections.Generic.Dictionary$2(Kusto.Data.IntelliSense.KustoCommandContext,System.Collections.Generic.List$1(System.String))).ctor();
                this.MapOfOriginallyKnownEntities = new (System.Collections.Generic.Dictionary$2(Kusto.Data.IntelliSense.KustoCommandContext,System.Collections.Generic.List$1(System.String))).ctor();
            }
        },
        methods: {
            Clone: function () {
                var $t;
                return ($t = new Kusto.Data.IntelliSense.CslCommand.AddEntitiesTokensState(), $t.MapOfKnownEntities = new (System.Collections.Generic.Dictionary$2(Kusto.Data.IntelliSense.KustoCommandContext,System.Collections.Generic.List$1(System.String))).$ctor1(this.MapOfKnownEntities), $t.MapOfOriginallyKnownEntities = new (System.Collections.Generic.Dictionary$2(Kusto.Data.IntelliSense.KustoCommandContext,System.Collections.Generic.List$1(System.String))).$ctor1(this.MapOfOriginallyKnownEntities), $t.MapOfPreviousCalculatedEntities = new (System.Collections.Generic.Dictionary$2(Kusto.Data.IntelliSense.KustoCommandContext,System.Collections.Generic.List$1(System.String))).$ctor1(this.MapOfPreviousCalculatedEntities), $t);
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CslCommand.FormatTraits", {
        $kind: "nested enum",
        statics: {
            fields: {
                None: 0,
                IncludeComments: 1,
                TabulateOnJoins: 2,
                TabulateOnFunctionBoundaries: 4
            }
        },
        $flags: true
    });

    Bridge.define("Kusto.Data.IntelliSense.CslCommand.PrivateTracer", {
        $kind: "nested class",
        statics: {
            fields: {
                Tracer: null
            },
            ctors: {
                init: function () {
                    this.Tracer = new Kusto.Data.IntelliSense.CslCommand.PrivateTracer();
                }
            }
        },
        methods: {
            TraceVerbose: function (format, list) {
                if (list === void 0) { list = []; }
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CslCommandIndexer", {
        statics: {
            fields: {
                MultiLineRegexOptions: 0,
                SingleLineRegexOptions: 0,
                s_defaultCommentRegex: null,
                s_defaultClientDirectivesRegex: null,
                s_defaultStringLiteralsRegex: null,
                s_defaultQueryParametersRegex: null,
                s_matchingBrackets: null
            },
            ctors: {
                init: function () {
                    this.MultiLineRegexOptions = 2;
                    this.SingleLineRegexOptions = 16;
                    this.s_defaultCommentRegex = new System.Text.RegularExpressions.Regex.ctor("(^\\s*//.*?(\\n|$))", Kusto.Data.IntelliSense.CslCommandIndexer.MultiLineRegexOptions);
                    this.s_defaultClientDirectivesRegex = new System.Text.RegularExpressions.Regex.ctor("(^\\s*#.*?(\\n|$))", Kusto.Data.IntelliSense.CslCommandIndexer.MultiLineRegexOptions);
                    this.s_defaultStringLiteralsRegex = new System.Text.RegularExpressions.Regex.ctor("(@?'.*?\\'|@?\\\".*?\\\")", Kusto.Data.IntelliSense.CslCommandIndexer.MultiLineRegexOptions);
                    this.s_defaultQueryParametersRegex = new System.Text.RegularExpressions.Regex.ctor("\\{([a-zA-Z]\\w+?)\\}", Kusto.Data.IntelliSense.CslCommandIndexer.MultiLineRegexOptions);
                    this.s_matchingBrackets = $asm.$.Kusto.Data.IntelliSense.CslCommandIndexer.f1(new (System.Collections.Generic.Dictionary$2(System.Char,System.Char)).ctor());
                }
            },
            methods: {
                AddToCollection: function (text, chars, tokenStart, scanStart, target) {
                    var $t;
                    var endOfArray = { };
                    var terminators = System.Array.init([10, 13], System.Char);
                    var token = ($t = new Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition(), $t.Start = tokenStart, $t.End = Kusto.Data.IntelliSense.CslCommandIndexer.ReadTill(chars, ((scanStart + 1) | 0), terminators, null, endOfArray), $t);

                    if (token.End > token.Start) {
                        if (endOfArray.v) {
                            token.End = (token.End - 1) | 0;
                        }
                        var len = (((token.End - token.Start) | 0) + 1) | 0;
                        token.Text = text.substr(token.Start, len);
                        target.add(token);
                    }
                    return token.End;
                },
                ReadTill: function (chars, i, terminators, escaping, endOfArray) {
                    endOfArray.v = true;
                    var foundTerminators = new (System.Collections.Generic.HashSet$1(System.Char)).ctor();
                    for (; i < chars.length; i = (i + 1) | 0) {
                        // If one terminator found, consume all others
                        if (System.Linq.Enumerable.from(foundTerminators, System.Char).any()) {
                            // Consume terminators (but only once)
                            if (System.Array.contains(terminators, chars[System.Array.index(i, chars)], System.Char) && !foundTerminators.contains(chars[System.Array.index(i, chars)])) {
                                foundTerminators.add(chars[System.Array.index(i, chars)]);
                                continue;
                            }

                            endOfArray.v = false;
                            i = (i - 1) | 0; // Move one back - to the previous terminator
                            break;
                        }

                        if (System.Array.contains(terminators, chars[System.Array.index(i, chars)], System.Char)) {
                            foundTerminators.add(chars[System.Array.index(i, chars)]);
                            continue;
                        }

                        if (escaping != null && System.Linq.Enumerable.from(escaping, System.Char).any(function (t) {
                                return chars[System.Array.index(i, chars)] === t;
                            })) {
                            // Move to next char
                            i = (i + 1) | 0;
                        }
                    }

                    if (i > chars.length) {
                        return chars.length;
                    }
                    return i;
                },
                ReadVerbatimTill: function (chars, i, terminators, endOfArray) {
                    endOfArray.v = true;
                    var escapeSequence = 0;
                    for (; i < chars.length; i = (i + 1) | 0) {
                        if (System.Array.contains(terminators, chars[System.Array.index(i, chars)], System.Char)) {
                            escapeSequence = (escapeSequence + 1) | 0;
                            if (escapeSequence % 2 !== 0 && i < ((chars.length - 1) | 0) && !System.Array.contains(terminators, chars[System.Array.index(((i + 1) | 0), chars)], System.Char)) {
                                endOfArray.v = false;
                                break;
                            }
                        }
                    }

                    return i;
                },
                CaptureTokensUsingRegex: function (text, regexList, tokenList) {
                    var $t, $t1, $t2, $t3;
                    if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$1(System.Text.RegularExpressions.Regex, regexList)) {
                        return;
                    }

                    $t = Bridge.getEnumerator(regexList);
                    try {
                        while ($t.moveNext()) {
                            var commentRegex = $t.Current;
                            var matches = commentRegex.matches(text);
                            $t1 = Bridge.getEnumerator(matches);
                            try {
                                while ($t1.moveNext()) {
                                    var m = Bridge.cast($t1.Current, System.Text.RegularExpressions.Match);
                                    $t2 = Bridge.getEnumerator(m.getCaptures());
                                    try {
                                        while ($t2.moveNext()) {
                                            var c = Bridge.cast($t2.Current, System.Text.RegularExpressions.Capture);
                                            var token = ($t3 = new Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition(), $t3.Start = c.getIndex(), $t3.End = ((((c.getIndex() + c.getLength()) | 0) - 1) | 0), $t3.Text = c.getValue(), $t3);
                                            tokenList.add(token);
                                        }
                                    } finally {
                                        if (Bridge.is($t2, System.IDisposable)) {
                                            $t2.System$IDisposable$Dispose();
                                        }
                                    }
                                }
                            } finally {
                                if (Bridge.is($t1, System.IDisposable)) {
                                    $t1.System$IDisposable$Dispose();
                                }
                            }
                        }
                    } finally {
                        if (Bridge.is($t, System.IDisposable)) {
                            $t.System$IDisposable$Dispose();
                        }
                    }
                }
            }
        },
        fields: {
            m_tokensAndPositions: null,
            m_commentTokenPositions: null,
            m_clientDirectivesTokenPositions: null,
            m_stringLiteralsPositions: null,
            m_queryParametersPositions: null,
            m_commandPartsPositions: null,
            m_bracketPartsPositions: null,
            m_sourceText: null,
            m_commentRegexCollection: null,
            m_clientDirectivesRegexCollection: null,
            m_stringLiteralsRegexCollection: null,
            m_queryParametersRegexCollection: null
        },
        props: {
            AntiTokenizers: null,
            TokenTerminators: null,
            TokenStarters: null
        },
        ctors: {
            init: function () {
                this.m_tokensAndPositions = new (System.Collections.Generic.Dictionary$2(System.String,System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition))).ctor();
                this.m_commentTokenPositions = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)).ctor();
                this.m_clientDirectivesTokenPositions = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)).ctor();
                this.m_stringLiteralsPositions = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)).ctor();
                this.m_queryParametersPositions = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)).ctor();
                this.m_commandPartsPositions = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)).ctor();
                this.m_bracketPartsPositions = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)).ctor();
                this.m_commentRegexCollection = new (System.Collections.Generic.List$1(System.Text.RegularExpressions.Regex)).ctor();
                this.m_clientDirectivesRegexCollection = new (System.Collections.Generic.List$1(System.Text.RegularExpressions.Regex)).ctor();
                this.m_stringLiteralsRegexCollection = new (System.Collections.Generic.List$1(System.Text.RegularExpressions.Regex)).ctor();
                this.m_queryParametersRegexCollection = new (System.Collections.Generic.List$1(System.Text.RegularExpressions.Regex)).ctor();
            },
            ctor: function (indexQueryParameters) {
                this.$initialize();
                this.m_commentRegexCollection.add(Kusto.Data.IntelliSense.CslCommandIndexer.s_defaultCommentRegex);
                this.m_clientDirectivesRegexCollection.add(Kusto.Data.IntelliSense.CslCommandIndexer.s_defaultClientDirectivesRegex);
                this.m_stringLiteralsRegexCollection.add(Kusto.Data.IntelliSense.CslCommandIndexer.s_defaultStringLiteralsRegex);

                if (indexQueryParameters) {
                    this.m_queryParametersRegexCollection.add(Kusto.Data.IntelliSense.CslCommandIndexer.s_defaultQueryParametersRegex);
                }
            }
        },
        methods: {
            IndexText: function (text) {
                var $t;
                this.m_tokensAndPositions.clear();
                this.m_sourceText = text;

                // Split text into words
                var chars = System.String.toCharArray(text, 0, text.length);

                // First, detect comments and string literals
                this.DetectCommentsAndStringLiterals(text);
                var skipRanges = this.GetTokenLookupSkipRanges(text.length);

                var tokenLookupState = Kusto.Data.IntelliSense.CslCommandIndexer.State.LookingForTokenStart;
                var tokenBuilder = null;
                var tokenPosition = null;
                var commandPartPosition = null;
                var bracketsStack = new (System.Collections.Generic.Stack$1(System.Tuple$2(System.Char,System.Int32))).ctor();
                for (var i = 0; i < chars.length; i = (i + 1) | 0) {
                    var isInsideTokenLookupSkipRange = skipRanges[System.Array.index(i, skipRanges)];
                    var isInsideCommentRange = (System.Linq.Enumerable.from(this.m_commentTokenPositions, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).where(function (r) {
                            return i >= r.Start && i <= r.End;
                        }).any());
                    var c = chars[System.Array.index(i, chars)];
                    var isAfterWhiteSpace = (i === 0) || System.Char.isWhiteSpace(String.fromCharCode(chars[System.Array.index(((i - 1) | 0), chars)]));

                    var canStartNewPart = true;
                    // Check if there is an existing part already
                    if (commandPartPosition != null && (((c === 124 || c === 59) && !isInsideTokenLookupSkipRange) || isInsideCommentRange || i === ((chars.length - 1) | 0))) {
                        if (i === ((chars.length - 1) | 0) || c === 59) {
                            commandPartPosition.End = i;
                            canStartNewPart = false;
                        } else {
                            commandPartPosition.End = (((i - 1) | 0));
                        }
                        var len = (((commandPartPosition.End - commandPartPosition.Start) | 0) + 1) | 0;
                        if (len > 1) {
                            var partText = text.substr(commandPartPosition.Start, len);
                            commandPartPosition.Text = partText;
                            // avoiding to consider a single newline symbol as an independent command part
                            if (!System.String.equals(partText, "\n") && !System.String.isNullOrWhiteSpace(partText)) {
                                this.m_commandPartsPositions.add(commandPartPosition);
                            }
                        }
                        commandPartPosition = null;
                    }
                    // Add new command part (separated by '|') if it was detected
                    if (commandPartPosition == null && !isInsideCommentRange && canStartNewPart) {
                        commandPartPosition = ($t = new Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition(), $t.Start = i, $t.End = i, $t);
                    }

                    // Brackets check
                    if (!isInsideTokenLookupSkipRange) {
                        if (Kusto.Data.IntelliSense.CslCommandIndexer.s_matchingBrackets.containsKey(c)) {
                            if (bracketsStack.Count > 0 && bracketsStack.Peek().Item1 === c) {
                                var bracketItem = bracketsStack.Pop();
                                var bracketToken = ($t = new Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition(), $t.Start = bracketItem.Item2, $t.End = i, $t);
                                bracketToken.Text = text.substr(bracketToken.Start, ((((bracketToken.End - bracketToken.Start) | 0) + 1) | 0));
                                this.m_bracketPartsPositions.add(bracketToken);
                            } else {
                                bracketsStack.Push({ Item1: Kusto.Data.IntelliSense.CslCommandIndexer.s_matchingBrackets.getItem(c), Item2: i });
                            }
                        }
                    }

                    switch (tokenLookupState) {
                        case Kusto.Data.IntelliSense.CslCommandIndexer.State.LookingForTokenStart: 
                            if (!isInsideTokenLookupSkipRange && ((this.IsPartOfTheToken(c) && !this.IsTokenTerminator(c)) || (isAfterWhiteSpace && this.IsTokenStarter(c)))) {
                                tokenBuilder = new System.Text.StringBuilder();
                                tokenPosition = ($t = new Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition(), $t.Start = i, $t.End = i, $t);
                                tokenBuilder.append(String.fromCharCode(c));
                                tokenLookupState = Kusto.Data.IntelliSense.CslCommandIndexer.State.LookingForTokenEnd;
                            }
                            break;
                        case Kusto.Data.IntelliSense.CslCommandIndexer.State.LookingForTokenEnd: 
                            var isTokenEnd = false;
                            if (!isInsideTokenLookupSkipRange && this.IsPartOfTheToken(c)) {
                                if (this.IsTokenTerminator(c)) {
                                    tokenPosition.TokenTerminator = c;
                                    isTokenEnd = true;
                                } else {
                                    tokenBuilder.append(String.fromCharCode(c));
                                    tokenPosition.End = i;
                                }
                            } else {
                                isTokenEnd = true;
                            }
                            if (isTokenEnd || i === ((chars.length - 1) | 0)) {
                                tokenPosition.Text = tokenBuilder.toString();
                                this.AddTokenPosition(tokenPosition);
                                // Token has ended
                                tokenLookupState = Kusto.Data.IntelliSense.CslCommandIndexer.State.LookingForTokenStart;
                            }
                            break;
                    }
                }
            },
            GetTokenLookupSkipRanges: function (size) {
                var ranges = new (System.Collections.Generic.List$1(System.Tuple$2(System.Int32,System.Int32))).ctor();
                ranges.AddRange(System.Linq.Enumerable.from(this.m_stringLiteralsPositions, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).select($asm.$.Kusto.Data.IntelliSense.CslCommandIndexer.f2));
                ranges.AddRange(System.Linq.Enumerable.from(this.m_commentTokenPositions, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).select($asm.$.Kusto.Data.IntelliSense.CslCommandIndexer.f2));
                ranges.AddRange(System.Linq.Enumerable.from(this.m_clientDirectivesTokenPositions, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).select($asm.$.Kusto.Data.IntelliSense.CslCommandIndexer.f2));
                ranges.AddRange(System.Linq.Enumerable.from(this.m_queryParametersPositions, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).select($asm.$.Kusto.Data.IntelliSense.CslCommandIndexer.f2));
                ranges.Sort$2($asm.$.Kusto.Data.IntelliSense.CslCommandIndexer.f3);

                var rIndex = 0;
                var currentSkipRange = System.Linq.Enumerable.from(ranges, System.Tuple$2(System.Int32,System.Int32)).firstOrDefault(null, null);
                var result = System.Array.init(size, false, System.Boolean);
                for (var i = 0; i < size && currentSkipRange != null; i = (i + 1) | 0) {
                    if (i >= currentSkipRange.Item1) {
                        result[System.Array.index(i, result)] = true;
                    }
                    if (currentSkipRange.Item2 === i) {
                        currentSkipRange = System.Linq.Enumerable.from(ranges, System.Tuple$2(System.Int32,System.Int32)).elementAtOrDefault(((rIndex = (rIndex + 1) | 0)), null);
                    }
                }

                return result;
            },
            GetCommandPartsPositions: function () {
                return this.m_commandPartsPositions;
            },
            GetCommentsPositions: function () {
                return this.m_commentTokenPositions;
            },
            GetClientDirectivesPositions: function () {
                return this.m_clientDirectivesTokenPositions;
            },
            GetStringLiteralsPositions: function () {
                return this.m_stringLiteralsPositions;
            },
            GetQueryParametersPositions: function () {
                return this.m_queryParametersPositions;
            },
            GetBracketsPositions: function () {
                return this.m_bracketPartsPositions;
            },
            GetUnrecognizedTokenPositions: function (recognizedTokens) {
                return System.Linq.Enumerable.from(this.m_tokensAndPositions.Values, System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)).selectMany($asm.$.Kusto.Data.IntelliSense.CslCommandIndexer.f4).except(recognizedTokens);
            },
            GetTokenPositionsInRange: function (searchForTokens, rangeStart, rangeEnd) {
                var $t;
                var result = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)).ctor();
                if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(System.String, searchForTokens)) {
                    return result;
                }

                $t = Bridge.getEnumerator(searchForTokens, System.String);
                try {
                    while ($t.moveNext()) {
                        var token = $t.Current;
                        if (System.String.isNullOrEmpty(token)) {
                            continue;
                        }
                        if (this.m_tokensAndPositions.containsKey(token)) {
                            var tokensInRange = System.Linq.Enumerable.from(this.m_tokensAndPositions.getItem(token), Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).where(function (t) {
                                    return t.Start >= rangeStart && t.End <= rangeEnd;
                                });
                            if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$2(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition, tokensInRange)) {
                                result.AddRange(tokensInRange);
                            }
                        }
                    }
                } finally {
                    if (Bridge.is($t, System.IDisposable)) {
                        $t.System$IDisposable$Dispose();
                    }
                }
                return result;
            },
            GetAllTokensSortedByPosition: function () {
                if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$1(System.Collections.Generic.KeyValuePair$2(System.String,System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)), this.m_tokensAndPositions)) {
                    return null;
                }

                return System.Linq.Enumerable.from(this.m_tokensAndPositions, System.Collections.Generic.KeyValuePair$2(System.String,System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition))).selectMany($asm.$.Kusto.Data.IntelliSense.CslCommandIndexer.f5).orderBy($asm.$.Kusto.Data.IntelliSense.CslCommandIndexer.f6);
            },
            GetAllTokenPositions: function (tokens, tokenSuffix) {
                var $t;
                if (tokenSuffix === void 0) { tokenSuffix = 0; }
                var result = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)).ctor();
                if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastNone$2(System.String, tokens)) {
                    return result;
                }

                $t = Bridge.getEnumerator(tokens, System.String);
                try {
                    while ($t.moveNext()) {
                        var token = $t.Current;
                        if (this.m_tokensAndPositions.containsKey(token)) {
                            result.AddRange(System.Linq.Enumerable.from(this.m_tokensAndPositions.getItem(token), Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).where(function (t) {
                                    return t.TokenTerminator === tokenSuffix;
                                }));
                        }
                    }
                } finally {
                    if (Bridge.is($t, System.IDisposable)) {
                        $t.System$IDisposable$Dispose();
                    }
                }
                return result;
            },
            IsPartOfTheToken: function (c) {
                return (System.Char.isDigit(c) || System.Char.isLetter(c)) || (this.AntiTokenizers != null && this.AntiTokenizers.contains(c));
            },
            IsTokenTerminator: function (c) {
                return (this.TokenTerminators != null && this.TokenTerminators.contains(c));
            },
            IsTokenStarter: function (c) {
                return (this.TokenStarters != null && System.Array.contains(this.TokenStarters, c, System.Char));
            },
            DetectCommentsAndStringLiterals: function (text) {
                Kusto.Data.IntelliSense.CslCommandIndexer.CaptureTokensUsingRegex(text, this.m_queryParametersRegexCollection, this.m_queryParametersPositions);
                if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.None(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition, this.m_queryParametersPositions)) {
                    // If there are no query parameters - 
                    // take more simple and more performant path:
                    this.DetectCommentsAndStringLiterals_Simple(text);
                } else {
                    // Query parameters exist in the command - therefore
                    // take complex and more expensive path: 
                    this.DetectCommentsAndStringLiterals_Complex(text);
                }
            },
            DetectCommentsAndStringLiterals_Simple: function (text) {
                var chars = System.String.toCharArray(text, 0, text.length);
                var state = 0;
                for (var i = 0; i < chars.length; i = (i + 1) | 0) {
                    var ch = chars[System.Array.index(i, chars)];
                    switch (state) {
                        case 0: 
                            if (ch === 64) {
                                state = 1;
                            } else if (ch === 39 || ch === 34) {
                                i = this.AddStringLiteralToken(text, chars, i, i, ch, false);
                            } else if (ch === 47) {
                                state = 2;
                            } else if (ch === 35) {
                                i = this.AddClientDirectiveToken(text, chars, i, i);
                                state = 0;
                            }
                            break;
                        case 1:  // Looking for string literals
                            if (ch === 39 || ch === 34) {
                                i = this.AddStringLiteralToken(text, chars, ((i - 1) | 0), i, ch, true);
                            }
                            state = 0;
                            break;
                        case 2:  // Looking for comment
                            if (ch === 47) {
                                i = this.AddCommentToken(text, chars, ((i - 1) | 0), i);
                            }
                            state = 0;
                            break;
                    }
                }
            },
            AddCommentToken: function (text, chars, tokenStart, scanStart) {
                return Kusto.Data.IntelliSense.CslCommandIndexer.AddToCollection(text, chars, tokenStart, scanStart, this.m_commentTokenPositions);
            },
            AddClientDirectiveToken: function (text, chars, tokenStart, scanStart) {
                return Kusto.Data.IntelliSense.CslCommandIndexer.AddToCollection(text, chars, tokenStart, scanStart, this.m_clientDirectivesTokenPositions);
            },
            AddStringLiteralToken: function (text, chars, start, position, ch, ifVerbatim) {
                var $t;
                var endOfArray = { };
                var token = ($t = new Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition(), $t.Start = start, $t.End = ifVerbatim ? Kusto.Data.IntelliSense.CslCommandIndexer.ReadVerbatimTill(chars, ((position + 1) | 0), System.Array.init([ch], System.Char), endOfArray) : Kusto.Data.IntelliSense.CslCommandIndexer.ReadTill(chars, ((position + 1) | 0), System.Array.init([ch], System.Char), System.Array.init([92], System.Char), endOfArray), $t);

                if (token.End > token.Start) {
                    if (endOfArray.v) {
                        token.End = (token.End - 1) | 0;
                    }
                    var len = (((token.End - token.Start) | 0) + 1) | 0;
                    token.Text = text.substr(token.Start, len);
                    this.m_stringLiteralsPositions.add(token);
                }
                return token.End;
            },
            DetectCommentsAndStringLiterals_Complex: function (text) {
                var $t;
                Kusto.Data.IntelliSense.CslCommandIndexer.CaptureTokensUsingRegex(text, this.m_commentRegexCollection, this.m_commentTokenPositions);
                Kusto.Data.IntelliSense.CslCommandIndexer.CaptureTokensUsingRegex(text, this.m_clientDirectivesRegexCollection, this.m_clientDirectivesTokenPositions);

                // Ignore tokens inside comments
                for (var i = (this.m_queryParametersPositions.Count - 1) | 0; i >= 0; i = (i - 1) | 0) {
                    var slp = { v : this.m_queryParametersPositions.getItem(i) };
                    if (System.Linq.Enumerable.from(this.m_commentTokenPositions, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).where((function ($me, slp) {
                            return function (comment) {
                                return comment.Start <= slp.v.Start && comment.End >= slp.v.End;
                            };
                        })(this, slp)).any()) {
                        this.m_queryParametersPositions.removeAt(i);
                    }
                }

                Kusto.Data.IntelliSense.CslCommandIndexer.CaptureTokensUsingRegex(text, this.m_stringLiteralsRegexCollection, this.m_stringLiteralsPositions);
                // Ignore tokens inside comments
                for (var i1 = (this.m_stringLiteralsPositions.Count - 1) | 0; i1 >= 0; i1 = (i1 - 1) | 0) {
                    var slp1 = { v : this.m_stringLiteralsPositions.getItem(i1) };
                    if (System.Linq.Enumerable.from(this.m_commentTokenPositions, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).where((function ($me, slp1) {
                            return function (comment) {
                                return comment.Start <= slp1.v.Start && comment.End >= slp1.v.End;
                            };
                        })(this, slp1)).any()) {
                        this.m_stringLiteralsPositions.removeAt(i1);
                    }
                }

                // Split string tokens if they contain query parameters
                for (var i2 = (this.m_queryParametersPositions.Count - 1) | 0; i2 >= 0; i2 = (i2 - 1) | 0) {
                    var qpp = { v : this.m_queryParametersPositions.getItem(i2) };
                    var slp2 = System.Linq.Enumerable.from(this.m_stringLiteralsPositions, Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition).where((function ($me, qpp) {
                            return function (s) {
                                return s.Start <= qpp.v.Start && s.End >= qpp.v.End;
                            };
                        })(this, qpp)).firstOrDefault(null, null);

                    if (slp2 != null) {
                        var newSlp = ($t = new Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition(), $t.Start = ((qpp.v.End + 1) | 0), $t.End = slp2.End, $t.Text = slp2.Text.substr(((((qpp.v.End - slp2.Start) | 0) + 1) | 0)), $t);

                        slp2.End = (qpp.v.Start - 1) | 0;
                        slp2.Text = slp2.Text.substr(0, slp2.Length);
                        this.m_stringLiteralsPositions.add(newSlp);
                    }
                }
            },
            AddTokenPosition: function (tokenPosition) {
                if (!this.m_tokensAndPositions.containsKey(tokenPosition.Text)) {
                    this.m_tokensAndPositions.add(tokenPosition.Text, new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition)).ctor());
                }

                this.m_tokensAndPositions.getItem(tokenPosition.Text).add(tokenPosition);
            }
        }
    });

    Bridge.ns("Kusto.Data.IntelliSense.CslCommandIndexer", $asm.$);

    Bridge.apply($asm.$.Kusto.Data.IntelliSense.CslCommandIndexer, {
        f1: function (_o1) {
            _o1.add(40, 41);
            _o1.add(41, 40);
            _o1.add(91, 93);
            _o1.add(93, 91);
            _o1.add(123, 125);
            _o1.add(125, 123);
            return _o1;
        },
        f2: function (p) {
            return { Item1: p.Start, Item2: p.End };
        },
        f3: function (r1, r2) {
            return Bridge.compare(r1.Item1, r2.Item1);
        },
        f4: function (t) {
            return t;
        },
        f5: function (kvp) {
            return kvp.value;
        },
        f6: function (t) {
            return t.Start;
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CslCommandIndexer.State", {
        $kind: "nested enum",
        statics: {
            fields: {
                LookingForTokenStart: 0,
                LookingForTokenEnd: 1,
                InsideComment: 2,
                InsideStringLiteral: 3
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CslCommandIndexer.TokenPosition", {
        $kind: "nested class",
        props: {
            Text: null,
            Start: 0,
            End: 0,
            TokenTerminator: 0,
            Length: {
                get: function () {
                    return ((((this.End - this.Start) | 0) + 1) | 0);
                }
            }
        },
        ctors: {
            ctor: function () {
                this.$initialize();
                // Explicitly set to '\0' to overcome Bridge.Net bug 3182:
                // http://forums.bridge.net/forum/bridge-net-pro/bugs/3182-comparing-0-to-unassigned-char-returns-false
                this.TokenTerminator = 0;
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CslCommandParser", {
        statics: {
            fields: {
                ControlCommandsTokens: null,
                CslCommandsTokens: null,
                ChartRenderTypesTokens: null,
                ChartRenderKindTokens: null,
                SubOperatorsTokens: null,
                JoinKindTokens: null,
                ReduceByKindTokens: null,
                DataTypesTokens: null,
                ScalarFunctionsDateTimeTokens: null,
                ScalarFunctionsNoDateTimeTokens: null,
                SingleParameterFunctionsDateTimeTokens: null,
                ZeroParameterFunctionsNoDateTimeTokens: null,
                SingleParameterFunctionsNoDateTimeTokens: null,
                IntrinsicFunctionTokens: null,
                TwoParameterFunctionsTokens: null,
                ThreeParameterFunctionsTokens: null,
                ManyParametersFunctionsTokens: null,
                PromotedOperatorCommandTokens: null,
                ClientDirectiveTokens: null,
                OperatorCommandTokens: null,
                DataOperatorTokens: null,
                SummarizeAggregationSingleParameterTokens: null,
                SummarizeAggregationTwoParametersTokens: null,
                SummarizeAggregationThreeParametersTokens: null,
                SummarizeAggregationManyParametersTokens: null,
                MakeSeriesAggregationTokens: null,
                PluginTokens: null,
                DatetimeFunctionsTokens: null,
                ScalarFunctionsTokens: null,
                SingleParameterFunctionsTokens: null,
                SummarizeAggregationTokens: null,
                SummarizeAggregationAliasesTokens: null,
                SortedSummarizeAggregators: null,
                SortedMakeSeriesAggregationTokens: null,
                SortedDatetimeFunctions: null,
                SortedExtendFunctions: null,
                FunctionsTokens: null,
                SortedEvaluateFunctions: null,
                s_asyncSeeker: null,
                s_isCommentLineRegex: null
            },
            ctors: {
                init: function () {
                    this.ControlCommandsTokens = System.Array.init([".add", ".alter", ".alter-merge", ".attach", ".append", ".create", ".create-merge", ".create-set", ".create-or-alter", ".define", ".detach", ".delete", ".drop", ".drop-pretend", ".dup-next-ingest", ".dup-next-failed-ingest", ".ingest", ".export", ".load", ".move", ".purge", ".purge-cleanup", ".remove", ".replace", ".save", ".set", ".set-or-append", ".set-or-replace", ".show", ".rename", ".run", "async", "data", "into", "ifnotexists", "whatif", "compressed", "monitoring", "metadata", "folder", "docstring", "details", "hot", "records", "until", "as", "csv", "tsv", "json", "sql", "policy", "encoding", "retention", "merge", "policies", "update", "ingestiontime", "caching", "sharding", "callout", "restricted_view_access", "ingestionbatching", "query_weak_consistency", "partitioning", "softdelete", "harddelete", "rowstore", "rowstores", "seal", "writeaheadlog", "streamingingestion", "rowstore_references", "follower"], System.String);
                    this.CslCommandsTokens = System.Array.init(["set", "let", "restrict", "access", "alias", "pattern", "declare", "query_parameters"], System.String);
                    this.ChartRenderTypesTokens = System.Linq.Enumerable.from(System.Array.init(["columnchart", "barchart", "piechart", "timechart", "anomalychart", "linechart", "ladderchart", "pivotchart", "areachart", "stackedareachart", "scatterchart", "timepivot", "timeline", "card", "table", "list", "treemap"], System.String), System.String).orderBy($asm.$.Kusto.Data.IntelliSense.CslCommandParser.f1).ToArray(System.String);
                    this.ChartRenderKindTokens = System.Array.init(["default", "stacked", "stacked100", "unstacked", "map"], System.String);
                    this.SubOperatorsTokens = System.Linq.Enumerable.from(System.Array.init(["like", "notlike", "contains", "notcontains", "!contains", "contains_cs", "!contains_cs", "startswith", "!startswith", "has", "!has", "has_cs", "!has_cs", "hasprefix", "!hasprefix", "hassuffix", "!hassuffix", "matches", "regex", "in", "!in", "endswith", "!endswith", "between", "!between", "extent", "database", "diagnostics", "jitmanagers", "admins", "basicauth", "cache", "capacity", "cluster", "continuous-export", "continuous-exports", "databases", "extents", "external", "journal", "memory", "extentcontainers", "viewers", "unrestrictedviewers", "tags", "prettyname", "blockedprincipals", "failed", "operations", "password", "principal", "principals", "settings", "schema", "table", "tables", "user", "users", "ingestors", "monitors", "version", "roles", "fabric", "locks", "service", "services", "nodes", "unallocated", "subscriptions", "nonregistered", "packages", "configuration", "commands", "commands-and-queries", "queries", "query", "function", "functions", "by", "on", "of", "true", "false", "and", "or", "asc", "desc", "nulls", "last", "first", "with", "withsource", "kind", "flags", "from", "to", "step", "ingestion", "failures", "mapping", "mappings", "geneva", "eventhub", "source", "sources", "types", "application", "period", "reason", "title"], System.String), System.String).union(Kusto.Data.IntelliSense.CslCommandParser.ChartRenderTypesTokens).union(Kusto.Data.IntelliSense.CslCommandParser.ChartRenderKindTokens).distinct().ToArray(System.String);
                    this.JoinKindTokens = System.Array.init(["anti", "inner", "innerunique", "fullouter", "leftanti", "leftantisemi", "leftouter", "leftsemi", "rightanti", "rightantisemi", "rightsemi", "rightouter"], System.String);
                    this.ReduceByKindTokens = System.Array.init(["mining"], System.String);
                    this.DataTypesTokens = System.Array.init(["timespan", "datetime", "int", "long", "real", "string", "bool", "double", "dynamic", "decimal", "guid"], System.String);
                    this.ScalarFunctionsDateTimeTokens = System.Array.init(["now", "ago", "datetime", "ingestion_time"], System.String);
                    this.ScalarFunctionsNoDateTimeTokens = System.Array.init(["time", "timespan", "dynamic", "decimal"], System.String);
                    this.SingleParameterFunctionsDateTimeTokens = System.Array.init(["todatetime", "between", "!between"], System.String);
                    this.ZeroParameterFunctionsNoDateTimeTokens = System.Array.init(["row_number", "extent_id", "extent_tags", "pi", "pack_all", "rowstore_ordinal_range"], System.String);
                    this.SingleParameterFunctionsNoDateTimeTokens = System.Array.init(["strlen", "tostring", "toupper", "tolower", "typeof", "reverse", "parsejson", "parse_json", "parse_xml", "parse_csv", "tobool", "toboolean", "todynamic", "toobject", "toint", "tolong", "toguid", "todouble", "toreal", "totimespan", "tohex", "todecimal", "isempty", "isnotempty", "isnull", "isnotnull", "isnan", "isinf", "isfinite", "dayofweek", "dayofmonth", "dayofyear", "week_of_year", "monthofyear", "sqrt", "rand", "log", "log10", "log2", "exp", "exp2", "exp10", "abs", "degrees", "radians", "sign", "sin", "cos", "tan", "asin", "acos", "atan", "cot", "getmonth", "getyear", "array_length", "gettype", "bag_keys", "cursor_after", "gamma", "loggamma", "dcount_hll", "parse_ipv4", "parse_url", "parse_path", "parse_version", "parse_urlquery", "url_encode", "url_decode", "binary_not", "not", "toscalar", "materialize", "series_stats", "series_fit_line", "series_fit_2lines", "series_stats_dynamic", "series_fit_line_dynamic", "series_fit_2lines_dynamic", "base64_encodestring", "base64_decodestring", "hash_sha256", "ceiling", "string_size", "isascii", "isutf8", "geo_geohash_to_central_point"], System.String);
                    this.IntrinsicFunctionTokens = System.Array.init(["cluster", "database", "table"], System.String);
                    this.TwoParameterFunctionsTokens = System.Array.init(["bin", "columnifexists", "floor", "countof", "hash", "round", "pow", "binary_and", "binary_or", "binary_xor", "binary_shift_left", "binary_shift_right", "datepart", "datetime_part", "repeat", "series_seasonal", "series_outliers", "array_split", "series_add", "series_subtract", "series_multiply", "series_divide", "series_pow", "series_pearson_correlation", "series_greater", "series_greater_equals", "series_less", "series_less_equals", "series_equals", "series_not_equals", "rank_tdigest", "percentrank_tdigest", "trim", "trim_start", "trim_end", "startofday", "startofweek", "startofmonth", "startofyear", "endofday", "endofweek", "endofmonth", "endofyear", "series_fill_backward", "series_fill_forward", "atan2", "format_datetime", "format_timespan", "strrep", "strcat_array", "parse_user_agent", "strcmp", "row_cumsum"], System.String);
                    this.ThreeParameterFunctionsTokens = System.Array.init(["iff", "iif", "range", "replace", "translate", "series_iir", "bin_at", "series_fill_const", "series_decompose", "series_decompose_forecast", "series_decompose_anomalies", "datetime_diff", "datetime_add", "geo_point_to_geohash", "geo_point_in_polygon"], System.String);
                    this.ManyParametersFunctionsTokens = System.Array.init(["extract", "extractjson", "extractall", "strcat", "strcat_delim", "substring", "indexof", "split", "case", "coalesce", "max_of", "min_of", "percentile_tdigest", "zip", "pack", "pack_dictionary", "pack_array", "array_concat", "array_slice", "welch_test", "row_window_session", "series_fir", "series_periods_detect", "prev", "next", "tdigest_merge", "hll_merge", "series_fill_linear", "series_periods_validate", "datatable", "make_datetime", "make_timespan", "make_string", "geo_distance_2points", "geo_point_in_circle"], System.String);
                    this.PromotedOperatorCommandTokens = System.Array.init(["where", "count", "extend", "join", "limit", "order", "project", "project-away", "project-rename", "project-reorder", "project-keep", "render", "sort", "summarize", "distinct", "take", "top", "top-nested", "top-hitters", "union", "mv-expand", "mv-apply", "reduce", "evaluate", "parse", "parse-where", "sample", "sample-distinct", "make-series", "getschema", "serialize", "invoke", "as", "scan"], System.String);
                    this.ClientDirectiveTokens = System.Array.init(["connect"], System.String);
                    this.OperatorCommandTokens = System.Linq.Enumerable.from(System.Array.init(["filter", "fork", "facet", "range", "consume", "find", "search", "print", "partition", "lookup"], System.String), System.String).union(Kusto.Data.IntelliSense.CslCommandParser.PromotedOperatorCommandTokens).ToArray(System.String);
                    this.DataOperatorTokens = System.Array.init(["datatable", "externaldata"], System.String);
                    this.SummarizeAggregationSingleParameterTokens = System.Array.init(["count", "countif", "dcount", "dcountif", "sum", "min", "max", "avg", "avgif", "any", "makelist", "makeset", "make_dictionary", "stdev", "stdevif", "varianceif", "variance", "buildschema", "hll", "hll_merge", "tdigest", "tdigest_merge"], System.String);
                    this.SummarizeAggregationTwoParametersTokens = System.Array.init(["percentile", "sumif"], System.String);
                    this.SummarizeAggregationThreeParametersTokens = System.Array.init(["percentilew"], System.String);
                    this.SummarizeAggregationManyParametersTokens = System.Array.init(["arg_min", "arg_max", "percentilesw_array", "percentilesw", "percentiles_array", "percentiles"], System.String);
                    this.MakeSeriesAggregationTokens = System.Array.init(["count", "countif", "dcount", "dcountif", "sum", "min", "max", "avg", "any", "stdev", "stdevp", "variance", "variancep", "sumif"], System.String);
                    this.PluginTokens = System.Array.init(["autocluster", "diffpatterns", "basket", "extractcolumns"], System.String);
                    this.DatetimeFunctionsTokens = System.Linq.Enumerable.from(Kusto.Data.IntelliSense.CslCommandParser.ScalarFunctionsDateTimeTokens, System.String).union(Kusto.Data.IntelliSense.CslCommandParser.SingleParameterFunctionsDateTimeTokens).ToArray(System.String);
                    this.ScalarFunctionsTokens = System.Linq.Enumerable.from(Kusto.Data.IntelliSense.CslCommandParser.ScalarFunctionsDateTimeTokens, System.String).union(Kusto.Data.IntelliSense.CslCommandParser.ScalarFunctionsNoDateTimeTokens).ToArray(System.String);
                    this.SingleParameterFunctionsTokens = System.Linq.Enumerable.from(Kusto.Data.IntelliSense.CslCommandParser.SingleParameterFunctionsDateTimeTokens, System.String).union(Kusto.Data.IntelliSense.CslCommandParser.SingleParameterFunctionsNoDateTimeTokens).ToArray(System.String);
                    this.SummarizeAggregationTokens = System.Linq.Enumerable.from(Kusto.Data.IntelliSense.CslCommandParser.SummarizeAggregationSingleParameterTokens, System.String).union(Kusto.Data.IntelliSense.CslCommandParser.SummarizeAggregationManyParametersTokens).union(Kusto.Data.IntelliSense.CslCommandParser.SummarizeAggregationThreeParametersTokens).union(Kusto.Data.IntelliSense.CslCommandParser.SummarizeAggregationTwoParametersTokens).ToArray(System.String);
                    this.SummarizeAggregationAliasesTokens = System.Array.init(["argmax", "argmin"], System.String);
                    this.SortedSummarizeAggregators = System.Linq.Enumerable.from(Kusto.Data.IntelliSense.CslCommandParser.SummarizeAggregationTokens, System.String).orderBy($asm.$.Kusto.Data.IntelliSense.CslCommandParser.f2).select($asm.$.Kusto.Data.IntelliSense.CslCommandParser.f3).ToArray(System.String);
                    this.SortedMakeSeriesAggregationTokens = System.Linq.Enumerable.from(Kusto.Data.IntelliSense.CslCommandParser.MakeSeriesAggregationTokens, System.String).orderBy($asm.$.Kusto.Data.IntelliSense.CslCommandParser.f2).select($asm.$.Kusto.Data.IntelliSense.CslCommandParser.f3).ToArray(System.String);
                    this.SortedDatetimeFunctions = System.Linq.Enumerable.from(Kusto.Data.IntelliSense.CslCommandParser.DatetimeFunctionsTokens, System.String).orderBy($asm.$.Kusto.Data.IntelliSense.CslCommandParser.f2).select($asm.$.Kusto.Data.IntelliSense.CslCommandParser.f3).ToArray(System.String);
                    this.SortedExtendFunctions = System.Linq.Enumerable.from(Kusto.Data.IntelliSense.CslCommandParser.ManyParametersFunctionsTokens, System.String).union(Kusto.Data.IntelliSense.CslCommandParser.ScalarFunctionsTokens).union(Kusto.Data.IntelliSense.CslCommandParser.ZeroParameterFunctionsNoDateTimeTokens).union(Kusto.Data.IntelliSense.CslCommandParser.SingleParameterFunctionsTokens).union(Kusto.Data.IntelliSense.CslCommandParser.TwoParameterFunctionsTokens).union(Kusto.Data.IntelliSense.CslCommandParser.ThreeParameterFunctionsTokens).orderBy($asm.$.Kusto.Data.IntelliSense.CslCommandParser.f2).select($asm.$.Kusto.Data.IntelliSense.CslCommandParser.f3).ToArray(System.String);
                    this.FunctionsTokens = System.Linq.Enumerable.from(Kusto.Data.IntelliSense.CslCommandParser.ManyParametersFunctionsTokens, System.String).union(Kusto.Data.IntelliSense.CslCommandParser.ScalarFunctionsTokens).union(Kusto.Data.IntelliSense.CslCommandParser.ZeroParameterFunctionsNoDateTimeTokens).union(Kusto.Data.IntelliSense.CslCommandParser.SingleParameterFunctionsTokens).union(Kusto.Data.IntelliSense.CslCommandParser.TwoParameterFunctionsTokens).union(Kusto.Data.IntelliSense.CslCommandParser.ThreeParameterFunctionsTokens).union(Kusto.Data.IntelliSense.CslCommandParser.SummarizeAggregationTokens).union(Kusto.Data.IntelliSense.CslCommandParser.IntrinsicFunctionTokens).ToArray(System.String);
                    this.SortedEvaluateFunctions = System.Linq.Enumerable.from(Kusto.Data.IntelliSense.CslCommandParser.PluginTokens, System.String).orderBy($asm.$.Kusto.Data.IntelliSense.CslCommandParser.f2).ToArray(System.String);
                    this.s_asyncSeeker = new System.Text.RegularExpressions.Regex.ctor("^\\.[\\w\\-]+\\s+async", Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.DefaultRegexOptions);
                    this.s_isCommentLineRegex = new System.Text.RegularExpressions.Regex.ctor("^\\s*//", Kusto.Data.IntelliSense.IntelliSenseRulesProviderBase.DefaultRegexOptions);
                }
            },
            methods: {
                IsAdminCommand$1: function (query, queryWithoutLeadingComments) {
                    return Kusto.Data.IntelliSense.CslCommandParser.IsStartingWithPrefix(query, ".", queryWithoutLeadingComments);
                },
                IsAdminCommand: function (command) {
                    var commandWithoutLeadingComments = { };
                    return Kusto.Data.IntelliSense.CslCommandParser.IsAdminCommand$1(command, commandWithoutLeadingComments);
                },
                ResolveRequestApiKind: function (requestContent) {
                    var requestContentWithoutLeadingComments = { };
                    var isAdminCommand = Kusto.Data.IntelliSense.CslCommandParser.IsStartingWithPrefix(requestContent, ".", requestContentWithoutLeadingComments);
                    if (!isAdminCommand) {
                        return Kusto.Data.IntelliSense.ApiKind.Query;
                    }
                    if (Kusto.Data.IntelliSense.CslCommandParser.s_asyncSeeker.isMatch(requestContentWithoutLeadingComments.v)) {
                        return Kusto.Data.IntelliSense.ApiKind.AdminAsyncCommand;
                    }
                    return Kusto.Data.IntelliSense.ApiKind.AdminSyncCommand;
                },
                IsStartingWithPrefix: function (query, prefix, queryWithoutLeadingComments) {
                    queryWithoutLeadingComments.v = query.trim();
                    var lines = System.String.split(query, System.Array.init([13, 10], System.Char).map(function (i) {{ return String.fromCharCode(i); }}), null, 1);
                    for (var i = 0; i < lines.length; i = (i + 1) | 0) {
                        var line = lines[System.Array.index(i, lines)];
                        if (System.String.startsWith(System.String.trimStart(line), prefix)) {
                            if (i > 0) {
                                queryWithoutLeadingComments.v = Bridge.toArray(System.Linq.Enumerable.from(lines, System.String).skip(i)).join("\n").trim();
                            }
                            return true;
                        }

                        if (!Kusto.Data.IntelliSense.CslCommandParser.s_isCommentLineRegex.isMatch(line)) {
                            if (i > 0) {
                                queryWithoutLeadingComments.v = Bridge.toArray(System.Linq.Enumerable.from(lines, System.String).skip(i)).join("\n").trim();
                            }
                            // If it is not a comment line - and it is not starting with dot - it is not a command
                            return false;
                        }
                    }

                    return false;
                },
                IsClientDirective: function (command, directiveWithoutLeadingComments) {
                    return Kusto.Data.IntelliSense.CslCommandParser.IsStartingWithPrefix(command, "#", directiveWithoutLeadingComments);
                }
            }
        },
        fields: {
            m_hashedCommands: null,
            m_rulesProvider: null
        },
        props: {
            Results: null
        },
        ctors: {
            ctor: function () {
                this.$initialize();
                this.Reset();
            }
        },
        methods: {
            Reset: function () {
                this.m_hashedCommands = new (System.Collections.Generic.Dictionary$2(System.String,Kusto.Data.IntelliSense.CslCommand)).ctor();
                this.Results = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommand)).ctor();
            },
            Parse: function (rulesProvider, text, parseMode) {
                var results = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommand)).ctor();

                // Check if previous parsing was done using other rules provider
                if (!Bridge.referenceEquals(this.m_rulesProvider, rulesProvider)) {
                    this.Reset();
                    this.m_rulesProvider = rulesProvider;
                }

                var parsedCommands = Kusto.Data.IntelliSense.CslCommandParser.CslCommandTokenizer.GetCommands(text);
                if (Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$4(Kusto.Data.IntelliSense.CslCommand, parsedCommands)) {
                    // Go over current commands and see if update needed
                    for (var i = 0; i < parsedCommands.Count; i = (i + 1) | 0) {
                        var newCommand = parsedCommands.getItem(i);
                        newCommand.ParseMode = parseMode;

                        var wasFound = this.FetchFromCache(parseMode, newCommand);
                        if (!wasFound) {
                            var previousCandidate = (this.Results != null) ? System.Linq.Enumerable.from(this.Results, Kusto.Data.IntelliSense.CslCommand).elementAtOrDefault(i, null) : null;
                            newCommand.ParseTokens(rulesProvider, parseMode, previousCandidate);
                        }
                        results.add(newCommand);
                    }
                }

                this.m_hashedCommands = System.Linq.Enumerable.from(Kusto.Cloud.Platform.Utils.ExtendedEnumerable.DistinctBy(Kusto.Data.IntelliSense.CslCommand, System.String, results, $asm.$.Kusto.Data.IntelliSense.CslCommandParser.f4), Kusto.Data.IntelliSense.CslCommand).toDictionary($asm.$.Kusto.Data.IntelliSense.CslCommandParser.f4, $asm.$.Kusto.Data.IntelliSense.CslCommandParser.f5, System.String, Kusto.Data.IntelliSense.CslCommand);
                this.Results = results;
                return results;
            },
            FetchFromCache: function (parseMode, newCommand) {
                var wasFound = false;
                var previouslyParsedCommand = { };
                if (this.m_hashedCommands.tryGetValue(newCommand.Text, previouslyParsedCommand)) {
                    if (previouslyParsedCommand.v.ParseMode === parseMode) {
                        // Command was already parsed and was not changed - use its parsed strings
                        newCommand.AcquireTokens(previouslyParsedCommand.v);
                        wasFound = true;
                    }
                }

                return wasFound;
            },
            Clone: function () {
                var $t;
                var parser = new Kusto.Data.IntelliSense.CslCommandParser();
                parser.m_hashedCommands = Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$1(System.Collections.Generic.KeyValuePair$2(System.String,Kusto.Data.IntelliSense.CslCommand), this.m_hashedCommands) ? new (System.Collections.Generic.Dictionary$2(System.String,Kusto.Data.IntelliSense.CslCommand)).$ctor1(this.m_hashedCommands) : new (System.Collections.Generic.Dictionary$2(System.String,Kusto.Data.IntelliSense.CslCommand)).ctor();
                parser.m_rulesProvider = this.m_rulesProvider;
                parser.Results = Kusto.Cloud.Platform.Utils.ExtendedEnumerable.SafeFastAny$2(Kusto.Data.IntelliSense.CslCommand, this.Results) ? ($t = Kusto.Data.IntelliSense.CslCommand, System.Linq.Enumerable.from(this.Results, $t).toList($t)) : new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommand)).ctor();
                return parser;
            }
        }
    });

    Bridge.ns("Kusto.Data.IntelliSense.CslCommandParser", $asm.$);

    Bridge.apply($asm.$.Kusto.Data.IntelliSense.CslCommandParser, {
        f1: function (t) {
            return t;
        },
        f2: function (f) {
            return f;
        },
        f3: function (f) {
            return (f || "") + "()";
        },
        f4: function (c) {
            return c.Text;
        },
        f5: function (c) {
            return c;
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CslCommandParser.CslCommandTokenizer", {
        $kind: "nested class",
        statics: {
            methods: {
                GetCommands: function (text) {
                    var $t, $t1;
                    var result = new (System.Collections.Generic.List$1(Kusto.Data.IntelliSense.CslCommand)).ctor();
                    if (System.String.isNullOrEmpty(text)) {
                        return result;
                    }

                    var command = null;
                    var mode = 0;
                    var endOfStream = false;
                    var expectedNewLine = 0;
                    var chars = ($t = System.Char, System.Linq.Enumerable.from(text, $t).ToArray($t));
                    for (var i = 0; i < chars.length; i = (i + 1) | 0) {
                        var ch = chars[System.Array.index(i, chars)];
                        endOfStream = (i === ((chars.length - 1) | 0));
                        if (endOfStream) {
                            if (mode === 0) {
                                command = ($t1 = new Kusto.Data.IntelliSense.CslCommand(), $t1.RelativeStart = i, $t1);
                            }
                            mode = 2; // Force completion
                        }

                        switch (mode) {
                            case 0:  // Starting new command
                                command = ($t1 = new Kusto.Data.IntelliSense.CslCommand(), $t1.RelativeStart = i, $t1);
                                mode = 1;
                                break;
                            case 1:  // Collecting command, looking for first newline
                                if (ch === 13 || ch === 10) {
                                    expectedNewLine = ch;
                                    mode = 2;
                                }
                                break;
                            case 2: 
                                if (ch === 10 || ch === 13) {
                                    // Check if the same new-line character is repeating new-line, patch ch to match expected new line
                                    if (chars[System.Array.index(((i - 1) | 0), chars)] === ch) {
                                        ch = expectedNewLine;
                                    }
                                }
                                if (ch === expectedNewLine || endOfStream) {
                                    if (command != null) {
                                        // Command termination
                                        while (!endOfStream) {
                                            // Peek at next symbol
                                            if (chars[System.Array.index(((i + 1) | 0), chars)] === 13 || chars[System.Array.index(((i + 1) | 0), chars)] === 10) {
                                                i = (i + 1) | 0;
                                            } else {
                                                break; // exit 'while' that appends empty lines
                                            }
                                            endOfStream = (i === ((chars.length - 1) | 0));
                                        }
                                        var commandLength = (((i - command.RelativeStart) | 0) + 1) | 0;
                                        command.Text = text.substr(command.RelativeStart, commandLength);
                                        command.Length = commandLength;
                                        command.IsRunnable = !System.String.isNullOrWhiteSpace(command.Text);
                                        result.add(command);
                                    }
                                    mode = 0; // Clean new line
                                    command = null;
                                } else if (System.Char.isWhiteSpace(String.fromCharCode(ch))) {
                                    // stay in this mode
                                } else {
                                    mode = 1;
                                }
                                break;
                        }
                    }

                    return result;
                }
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CslCommandToken", {
        inherits: [System.ICloneable],
        props: {
            AbsolutePositionBias: 0,
            Value: null,
            TokenKind: 0,
            AbsoluteStart: {
                get: function () {
                    return ((this.RelativeStart + this.AbsolutePositionBias) | 0);
                }
            },
            AbsoluteEnd: {
                get: function () {
                    return ((this.RelativeEnd + this.AbsolutePositionBias) | 0);
                }
            },
            RelativeStart: 0,
            Length: 0,
            RelativeEnd: {
                get: function () {
                    return ((this.RelativeStart + this.Length) | 0);
                }
            }
        },
        alias: ["clone", "System$ICloneable$clone"],
        ctors: {
            ctor: function (value, absolutePositionBias, tokenKind) {
                this.$initialize();
                this.Value = value;
                this.TokenKind = tokenKind;
                this.AbsolutePositionBias = absolutePositionBias;
            }
        },
        methods: {
            CompareTo: function (other) {
                if (other == null) {
                    return 1;
                }

                if (this.RelativeStart === other.RelativeStart) {
                    return 0;
                }

                return (this.RelativeStart < other.RelativeStart) ? -1 : 1;
            },
            clone: function () {
                var $t;
                return ($t = new Kusto.Data.IntelliSense.CslCommandToken(this.Value, this.AbsolutePositionBias, this.TokenKind), $t.RelativeStart = this.RelativeStart, $t.Length = this.Length, $t);
            },
            equals: function (obj) {
                var other = Bridge.as(obj, Kusto.Data.IntelliSense.CslCommandToken);
                if (other == null) {
                    return false;
                }

                return Bridge.referenceEquals(this.Value, other.Value) && this.TokenKind === other.TokenKind && this.RelativeStart === other.RelativeStart && this.RelativeEnd === other.RelativeEnd && this.AbsolutePositionBias === other.AbsolutePositionBias;
            },
            getHashCode: function () {
                return Bridge.getHashCode(this.Value) ^ Bridge.getHashCode(this.TokenKind) ^ this.RelativeStart ^ this.RelativeEnd ^ this.AbsolutePositionBias;
            }
        }
    });

    Bridge.define("Kusto.Data.IntelliSense.CslCommandToken.Kind", {
        $kind: "nested enum",
        statics: {
            fields: {
                TableToken: 2,
                TableColumnToken: 4,
                OperatorToken: 8,
                SubOperatorToken: 16,
                CalculatedColumnToken: 32,
                StringLiteralToken: 64,
                FunctionNameToken: 128,
                UnknownToken: 256,
                CommentToken: 512,
                PlainTextToken: 1024,
                DataTypeToken: 2048,
                ControlCommandToken: 4096,
                CommandPartToken: 8192,
                QueryParametersToken: 16384,
                CslCommandToken: 32768,
                LetVariablesToken: 65536,
                PluginToken: 131072,
                BracketRangeToken: 262144,
                ClientDirectiveToken: 524288,
                IsRecognizedEntity: 536870912,
                IsMismatchedTokenValue: 1073741824
            }
        },
        $flags: true
    });

    Bridge.define("Kusto.Data.IntelliSense.CslDocumentation", {
        statics: {
            fields: {
                s_instance: null
            },
            props: {
                Instance: {
                    get: function () {
                        if (Kusto.Data.IntelliSense.CslDocumentation.s_instance == null) {
                            Kusto.Data.IntelliSense.CslDocumentation.s_instance = new Kusto.Data.IntelliSense.CslDocumentation();
                            Kusto.Data.IntelliSense.CslDocumentation.s_instance.Initialize();
                        }
                        return Kusto.Data.IntelliSense.CslDocumentation.s_instance;
                    }
                }
            },
            methods: {
                TrimFunctionParameters: function (signature) {
                    var parenStart = System.String.indexOf(signature, String.fromCharCode(40));
                    if (parenStart > 0) {
                        return signature.substr(0, parenStart);
                    } else {
                        return signature;
                    }
                }
            }
        },
        fields: {
            m_documentation: null
        },
        props: {
            Topics: {
                get: function () {
                    if (this.m_documentation == null) {
                        return System.Linq.Enumerable.empty();
                    }

                    return System.Linq.Enumerable.from(this.m_documentation, System.Collections.Generic.KeyValuePair$2(System.Tuple$2(Kusto.Data.IntelliSense.CslCommandToken.Kind,System.String),Kusto.Data.IntelliSense.CslTopicDocumentation)).select($asm.$.Kusto.Data.IntelliSense.CslDocumentation.f1);
                }
            }
        },
        ctors: {
            init: function () {
                this.m_documentation = new (System.Collections.Generic.Dictionary$2(System.Tuple$2(Kusto.Data.IntelliSense.CslCommandToken.Kind,System.String),Kusto.Data.IntelliSense.CslTopicDocumentation)).ctor();
            }
        },
        methods: {
            AddTopic: function (topic) {
                var key = { Item1: topic.TokenKind, Item2: topic.Name.toLowerCase() };
                this.m_documentation.setItem(key, topic);
            },
            GetTopic: function (option) {
                if (option.Kind === Kusto.Data.IntelliSense.OptionKind.Operator) {
                    return this.GetTopic$1(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, option.Value);
                } else if (option.Kind === Kusto.Data.IntelliSense.OptionKind.FunctionAggregation || option.Kind === Kusto.Data.IntelliSense.OptionKind.FunctionScalar) {
                    return this.GetTopic$1(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, Kusto.Data.IntelliSense.CslDocumentation.TrimFunctionParameters(option.Value));
                }

                return null;
            },
            GetTopic$1: function (tokenKind, name) {
                var key = { Item1: tokenKind, Item2: name.toLowerCase() };
                var result = { };
                this.m_documentation.tryGetValue(key, result);
                return result.v;
            },
            GetTopicByMarkdownSource: function (markdownSource) {
                var $t;
                if (System.String.isNullOrWhiteSpace(markdownSource)) {
                    return null;
                }

                $t = Bridge.getEnumerator(this.m_documentation);
                try {
                    while ($t.moveNext()) {
                        var kvp = $t.Current;
                        if (Bridge.referenceEquals(kvp.value.MarkdownSource, markdownSource)) {
                            return kvp.value;
                        }
                    }
                } finally {
                    if (Bridge.is($t, System.IDisposable)) {
                        $t.System$IDisposable$Dispose();
                    }
                }

                return null;
            },
            GetTopicByName: function (name) {
                var $t;
                if (System.String.isNullOrWhiteSpace(name)) {
                    return null;
                }

                $t = Bridge.getEnumerator(this.m_documentation);
                try {
                    while ($t.moveNext()) {
                        var kvp = $t.Current;
                        if (Bridge.referenceEquals(kvp.key.Item2, name)) {
                            return kvp.value;
                        }
                    }
                } finally {
                    if (Bridge.is($t, System.IDisposable)) {
                        $t.System$IDisposable$Dispose();
                    }
                }

                return null;
            },
            equals: function (obj) {
                if (obj == null) {
                    return false;
                }

                var other = Bridge.as(obj, Kusto.Data.IntelliSense.CslDocumentation);
                if (other == null) {
                    return false;
                }

                if (this.m_documentation.Count !== other.m_documentation.Count) {
                    return false;
                }

                if (System.Linq.Enumerable.from(other.m_documentation, System.Collections.Generic.KeyValuePair$2(System.Tuple$2(Kusto.Data.IntelliSense.CslCommandToken.Kind,System.String),Kusto.Data.IntelliSense.CslTopicDocumentation)).except(this.m_documentation).any()) {
                    return false;
                }

                if (System.Linq.Enumerable.from(this.m_documentation, System.Collections.Generic.KeyValuePair$2(System.Tuple$2(Kusto.Data.IntelliSense.CslCommandToken.Kind,System.String),Kusto.Data.IntelliSense.CslTopicDocumentation)).except(other.m_documentation).any()) {
                    return false;
                }

                return true;
            },
            getHashCode: function () {
                return Bridge.getHashCode(this.m_documentation);
            },
            Initialize: function () {
                if (System.Linq.Enumerable.from(this.m_documentation, System.Collections.Generic.KeyValuePair$2(System.Tuple$2(Kusto.Data.IntelliSense.CslCommandToken.Kind,System.String),Kusto.Data.IntelliSense.CslTopicDocumentation)).any()) {
                    return;
                }
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!~", "Filters a record set for data that does not match a case-insensitive string.", "The following table provides a comparison of the `==` (equals) operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`==`](equals-cs-operator.md)|Equals |Yes|`\"aBc\" == \"aBc\"`|\r\n|[`!=`](not-equals-cs-operator.md)|Not equals |Yes |`\"abc\" != \"ABC\"`|\r\n|[`=~`](equals-operator.md) |Equals |No |`\"abc\" =~ \"ABC\"`|\r\n|[`!~`](not-equals-operator.md) |Not equals |No |`\"aBc\" !~ \"xyz\"`|\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\nCase-insensitive operators are currently supported only for ASCII-text. For non-ASCII comparison, use the [tolower()](tolowerfunction.md) function.\r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `==`, not `=~`.\r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. \r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `!~` `(`*expression*`)` \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where (State !~ \"texas\") and (event_count > 3000)\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|KANSAS|3,166|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-equals-operator", "not-equals-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!=", "Filters a record set for data that does not match a case-sensitive string.", "The following table provides a comparison of the `==` (equals) operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`==`](equals-cs-operator.md)|Equals |Yes|`\"aBc\" == \"aBc\"`|\r\n|[`!=`](not-equals-cs-operator.md)|Not equals |Yes |`\"abc\" != \"ABC\"`|\r\n|[`=~`](equals-operator.md) |Equals |No |`\"abc\" =~ \"ABC\"`|\r\n|[`!~`](not-equals-operator.md) |Not equals |No |`\"aBc\" !~ \"xyz\"`|\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `==`, not `=~`.\r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. \r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `!=` `(`*list of scalar expressions*`)`\r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where (State != \"FLORIDA\") and (event_count > 4000)\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|TEXAS|4,701|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-equals-cs-operator", "not-equals-cs-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!between", "Matches the input that is outside the inclusive range.", "```\r\nTable1 | where Num1 !between (1 .. 10)\r\nTable1 | where Time !between (datetime(2017-01-01) .. datetime(2017-01-01))\r\n```\r\n\r\n`!between` can operate on any numeric, datetime, or timespan expression.\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *expr* `!between` `(`*leftRange*` .. `*rightRange*`)`   \r\n\r\nIf *expr* expression is datetime - another syntactic sugar syntax is provided:\r\n\r\n*T* `|` `where` *expr* `!between` `(`*leftRangeDateTime*` .. `*rightRangeTimespan*`)`   \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be matched.\r\n* *expr* - the expression to filter.\r\n* *leftRange* - expression of the left range (inclusive).\r\n* *rightRange* - expression of the right range (inclusive).\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate of (*expr* < *leftRange* or *expr* > *rightRange*) evaluates to `true`.", "### Filter numeric values   \r\n\r\n```\r\nrange x from 1 to 10 step 1\r\n| where x !between (5 .. 9)\r\n```\r\n\r\n|x|\r\n|---|\r\n|1|\r\n|2|\r\n|3|\r\n|4|\r\n|10|\r\n\r\n### Filter datetime  \r\n\r\n```\r\nStormEvents\r\n| where StartTime !between (datetime(2007-07-27) .. datetime(2007-07-30))\r\n| count \r\n```\r\n\r\n|Count|\r\n|---|\r\n|58590|\r\n\r\n```\r\nStormEvents\r\n| where StartTime !between (datetime(2007-07-27) .. 3d)\r\n| count \r\n```\r\n\r\n|Count|\r\n|---|\r\n|58590|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/notbetweenoperator", "notbetweenoperator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!contains", "Filters a record set for data that does not include a case-sensitive string. `contains` searches for characters rather than [terms](datatypes-string-operators.md#what-is-a-term) of three or more characters. The query scans the values in the column, which is slower than looking up a term in a term index.", "The following table provides a comparison of the `contains` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`contains`](contains-operator.md) |RHS occurs as a subsequence of LHS |No |`\"FabriKam\" contains \"BRik\"`|\r\n|[`!contains`](not-contains-operator.md) |RHS doesn't occur in LHS |No |`\"Fabrikam\" !contains \"xyz\"`|\r\n|[`contains_cs`](contains-cs-operator.md) |RHS occurs as a subsequence of LHS |Yes |`\"FabriKam\" contains_cs \"Kam\"`|\r\n|[`!contains_cs`](not-contains-cs-operator.md)   |RHS doesn't occur in LHS |Yes |`\"Fabrikam\" !contains_cs \"Kam\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\nCase-insensitive operators are currently supported only for ASCII-text. For non-ASCII comparison, use the [tolower()](tolowerfunction.md) function.\r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `contains_cs`, not `contains`.\r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. Also, `has` works faster than `contains`, `startswith`, or `endswith`, however it is not as precise and could provide unwanted records.\r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n### Case insensitive syntax\r\n\r\n*T* `|` `where` *Column* `!contains` `(`*Expression*`)`   \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *Column* - The column to filter.\r\n* *Expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !contains \"kan\"\r\n    | where event_count > 3000\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|TEXAS|4701|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-contains-operator", "not-contains-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!contains_cs", "Filters a record set for data that does not include a case-sensitive string. `contains` searches for characters rather than [terms](datatypes-string-operators.md#what-is-a-term) of three or more characters. The query scans the values in the column, which is slower than looking up a term in a term index.", "The following table provides a comparison of the `contains` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`contains`](contains-operator.md) |RHS occurs as a subsequence of LHS |No |`\"FabriKam\" contains \"BRik\"`|\r\n|[`!contains`](not-contains-operator.md) |RHS doesn't occur in LHS |No |`\"Fabrikam\" !contains \"xyz\"`|\r\n|[`contains_cs`](contains-cs-operator.md) |RHS occurs as a subsequence of LHS |Yes |`\"FabriKam\" contains_cs \"Kam\"`|\r\n|[`!contains_cs`](not-contains-cs-operator.md)   |RHS doesn't occur in LHS |Yes |`\"Fabrikam\" !contains_cs \"Kam\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `contains_cs`, not `contains`.\r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. Also, `has` works faster than `contains`, `startswith`, or `endswith`, however it is not as precise and could provide unwanted records.\r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n### Case-sensitive syntax\r\n\r\n*T* `|` `where` *Column* `!contains_cs` `(`*Expression*`)` \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *Column* - The column to filter.\r\n* *Expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !contains_cs \"AS\"\r\n    | count\r\n```\r\n\r\n**Output**\r\n\r\n|Count|\r\n|-----|\r\n|59|\r\n\r\n```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !contains_cs \"TEX\"\r\n    | where event_count > 3000\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|KANSAS|3,166|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-contains-cs-operator", "not-contains-cs-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!endswith", "Filters a record set for data that excludes a case-insensitive ending string.", "The following table provides a comparison of the `endswith` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`endswith`](endswith-operator.md) |RHS is a closing subsequence of LHS |No |`\"Fabrikam\" endswith \"Kam\"`|\r\n|[`!endswith`](not-endswith-operator.md) |RHS isn't a closing subsequence of LHS |No |`\"Fabrikam\" !endswith \"brik\"`|\r\n|[`endswith_cs`](endswith-cs-operator.md) |RHS is a closing subsequence of LHS |Yes |`\"Fabrikam\" endswith_cs \"kam\"`|\r\n|[`!endswith_cs`](not-endswith-cs-operator.md) |RHS isn't a closing subsequence of LHS |Yes |`\"Fabrikam\" !endswith_cs \"brik\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor more information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\nCase-insensitive operators are currently supported only for ASCII-text. For non-ASCII comparison, use the [tolower()](tolowerfunction.md) function.\r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `endswith_cs`, not `endswith`. For best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `!endswith` `(`*expression*`)`   \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !endswith \"is\"\r\n    | where event_count > 2000\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|TEXAS|4701|\r\n|KANSAS|3166|\r\n|IOWA|2337|\r\n|MISSOURI|2016|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-endswith-operator", "not-endswith-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!endswith_cs", "Filters a record set for data that does not contain a case-insensitive ending string.", "The following table provides a comparison of the `endswith` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`endswith`](endswith-operator.md) |RHS is a closing subsequence of LHS |No |`\"Fabrikam\" endswith \"Kam\"`|\r\n|[`!endswith`](not-endswith-operator.md) |RHS isn't a closing subsequence of LHS |No |`\"Fabrikam\" !endswith \"brik\"`|\r\n|[`endswith_cs`](endswith-cs-operator.md) |RHS is a closing subsequence of LHS |Yes |`\"Fabrikam\" endswith_cs \"kam\"`|\r\n|[`!endswith_cs`](not-endswith-cs-operator.md) |RHS isn't a closing subsequence of LHS |Yes |`\"Fabrikam\" !endswith_cs \"brik\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `endswith_cs`, not `endswith`. For best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `!endswith_cs` `(`*expression*`)`  \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !endswith_cs \"AS\"\r\n    | count\r\n```\r\n\r\n**Output**\r\n\r\n|Count|\r\n|-----|\r\n|64|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-endswith-cs-operator", "not-endswith-cs-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!has", "Filters a record set for data that does not have a matching case-insensitive string. `has` searches for indexed terms, where a [term](datatypes-string-operators.md#what-is-a-term) is three or more characters. If your term is fewer than three characters, the query scans the values in the column, which is slower than looking up the term in the term index.", "The following table provides a comparison of the `has` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`has`](has-operator.md) |Right-hand-side (RHS) is a whole term in left-hand-side (LHS) |No |`\"North America\" has \"america\"`|\r\n|[`!has`](not-has-operator.md) |RHS isn't a full term in LHS |No |`\"North America\" !has \"amer\"`|\r\n|[`has_cs`](has-cs-operator.md) |RHS is a whole term in LHS |Yes |`\"North America\" has_cs \"America\"`|\r\n|[`!has_cs`](not-has-cs-operator.md) |RHS isn't a full term in LHS |Yes |`\"North America\" !has_cs \"amer\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\nCase-insensitive operators are currently supported only for ASCII-text. For non-ASCII comparison, use the [tolower()](tolowerfunction.md) function.\r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `has_cs`, not `has`.\r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. \r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *Column* `!has` `(`*Expression*`)`   \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *Column* - The column to filter.\r\n* *Expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !has \"NEW\"\r\n    | where event_count > 3000\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|TEXAS|4,701|\r\n|KANSAS|3,166|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-has-operator", "not-has-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!has_cs", "Filters a record set for data that does not have a matching case-sensitive string. `has` searches for indexed terms, where a [term](datatypes-string-operators.md#what-is-a-term) is three or more characters. If your term is fewer than three characters, the query scans the values in the column, which is slower than looking up the term in the term index.", "The following table provides a comparison of the `has` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`has`](has-operator.md) |Right-hand-side (RHS) is a whole term in left-hand-side (LHS) |No |`\"North America\" has \"america\"`|\r\n|[`!has`](not-has-operator.md) |RHS isn't a full term in LHS |No |`\"North America\" !has \"amer\"`|\r\n|[`has_cs`](has-cs-operator.md) |RHS is a whole term in LHS |Yes |`\"North America\" has_cs \"America\"`|\r\n|[`!has_cs`](not-has-cs-operator.md) |RHS isn't a full term in LHS |Yes |`\"North America\" !has_cs \"amer\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `has_cs`, not `has`.\r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. \r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *Column* `!has_cs` `(`*Expression*`)`  \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *Column* - The column to filter.\r\n* *Expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !has_cs \"new\"\r\n    | count\r\n```\r\n\r\n**Output**\r\n\r\n|Count|\r\n|-----|\r\n|67|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-has-cs-operator", "not-has-cs-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!hasprefix", "Filters a record set for data that does not include a case-insensitive starting string. `has` searches for indexed terms, where a [term](datatypes-string-operators.md#what-is-a-term) is three or more characters. If your term is fewer than three characters, the query scans the values in the column, which is slower than looking up the term in the term index.", "The following table provides a comparison of the `hasprefix` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`hasprefix`](hasprefix-operator.md) |RHS is a term prefix in LHS |No |`\"North America\" hasprefix \"ame\"`|\r\n|[`!hasprefix`](not-hasprefix-operator.md) |RHS isn't a term prefix in LHS |No |`\"North America\" !hasprefix \"mer\"`|\r\n|[`hasprefix_cs`](hasprefix-cs-operator.md) |RHS is a term prefix in LHS |Yes |`\"North America\" hasprefix_cs \"Ame\"`|\r\n|[`!hasprefix_cs`](not-hasprefix-cs-operator.md) |RHS isn't a term prefix in LHS |Yes |`\"North America\" !hasprefix_cs \"CA\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor more information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\nCase-insensitive operators are currently supported only for ASCII-text. For non-ASCII comparison, use the [tolower()](tolowerfunction.md) function.\r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `hasprefix_cs`, not `hasprefix`.\r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. \r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *Column* `!hasprefix` `(`*Expression*`)`   \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *Column* - The column to filter.\r\n* *Expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !hasprefix \"N\"\r\n    | where event_count > 2000\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|TEXAS|4701|\r\n|KANSAS|3166|\r\n|IOWA|2337|\r\n|ILLINOIS|2022|\r\n|MISSOURI|2016|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-hasprefix-operator", "not-hasprefix-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!hasprefix_cs", "Filters a record set for data that does not have a case-sensitive starting string. `has` searches for indexed terms, where a [term](datatypes-string-operators.md#what-is-a-term) is three or more characters. If your term is fewer than three characters, the query scans the values in the column, which is slower than looking up the term in the term index.", "|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`hasprefix`](hasprefix-operator.md) |RHS is a term prefix in LHS |No |`\"North America\" hasprefix \"ame\"`|\r\n|[`!hasprefix`](not-hasprefix-operator.md) |RHS isn't a term prefix in LHS |No |`\"North America\" !hasprefix \"mer\"`|\r\n|[`hasprefix_cs`](hasprefix-cs-operator.md) |RHS is a term prefix in LHS |Yes |`\"North America\" hasprefix_cs \"Ame\"`|\r\n|[`!hasprefix_cs`](not-hasprefix-cs-operator.md) |RHS isn't a term prefix in LHS |Yes |`\"North America\" !hasprefix_cs \"CA\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `hasprefix_cs`, not `hasprefix`. For best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *Column* `!hasprefix_cs` `(`*Expression*`)`  \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *Column* - The column to filter.\r\n* *Expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !hasprefix_cs \"P\"\r\n    | count\r\n```\r\n\r\n**Output**\r\n\r\n|Count|\r\n|-----|\r\n|64|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-hasprefix-cs-operator", "not-hasprefix-cs-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!hassuffix", "Filters a record set for data that does not have a case-insensitive ending string.`has` searches for indexed terms, where a [term](datatypes-string-operators.md#what-is-a-term) is three or more characters. If your term is fewer than three characters, the query scans the values in the column, which is slower than looking up the term in the term index.", "The following table provides a comparison of the `hassuffix` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`hassuffix`](hassuffix-operator.md) |RHS is a term suffix in LHS |No |`\"North America\" hassuffix \"ica\"`|\r\n|[`!hassuffix`](not-hassuffix-operator.md) |RHS isn't a term suffix in LHS |No |`\"North America\" !hassuffix \"americ\"`|\r\n|[`hassuffix_cs`](hassuffix-cs-operator.md)  |RHS is a term suffix in LHS |Yes |`\"North America\" hassuffix_cs \"ica\"`|\r\n|[`!hassuffix_cs`](not-hassuffix-cs-operator.md) |RHS isn't a term suffix in LHS |Yes |`\"North America\" !hassuffix_cs \"icA\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor more information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\nCase-insensitive operators are currently supported only for ASCII-text. For non-ASCII comparison, use the [tolower()](tolowerfunction.md) function.\r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `hassuffix_cs`, not `hassuffix`. \r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. \r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *Column* `!hassuffix` `(`*Expression*`)`\r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *Column* - The column to filter.\r\n* *Expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !hassuffix \"A\"\r\n    | where event_count > 2000\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|TEXAS|4701|\r\n|KANSAS|3166|\r\n|ILLINOIS|2022|\r\n|MISSOURI|2016|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-hassuffix-operator", "not-hassuffix-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!hassuffix_cs", "Filters a record set for data that does not have a case-sensitive ending string. `has` searches for indexed terms, where a [term](datatypes-string-operators.md#what-is-a-term) is three or more characters. If your term is fewer than three characters, the query scans the values in the column, which is slower than looking up the term in the term index.", "The following table provides a comparison of the `hassuffix` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`hassuffix`](hassuffix-operator.md) |RHS is a term suffix in LHS |No |`\"North America\" hassuffix \"ica\"`|\r\n|[`!hassuffix`](not-hassuffix-operator.md) |RHS isn't a term suffix in LHS |No |`\"North America\" !hassuffix \"americ\"`|\r\n|[`hassuffix_cs`](hassuffix-cs-operator.md)  |RHS is a term suffix in LHS |Yes |`\"North America\" hassuffix_cs \"ica\"`|\r\n|[`!hassuffix_cs`](not-hassuffix-cs-operator.md) |RHS isn't a term suffix in LHS |Yes |`\"North America\" !hassuffix_cs \"icA\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `hassuffix_cs`, not `hassuffix`. \r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. \r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *Column* `!hassuffix_cs` `(`*Expression*`)`  \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *Column* - The column to filter.\r\n* *Expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !hassuffix_cs \"AS\"\r\n    | where event_count > 2000\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|IOWA|2337|\r\n|ILLINOIS|2022|\r\n|MISSOURI|2016|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-hassuffix-cs-operator", "not-hassuffix-cs-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!in", "Filters a record set for data without a case-sensitive string.", "The following table provides a comparison of the `has` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`in`](in-cs-operator.md) |Equals to one of the elements |Yes |`\"abc\" in (\"123\", \"345\", \"abc\")`|\r\n|[`!in`](not-in-cs-operator.md) |Not equals to any of the elements |Yes | `\"bca\" !in (\"123\", \"345\", \"abc\")` |\r\n|[`in~`](inoperator.md) |Equals to any of the elements |No | `\"Abc\" in~ (\"123\", \"345\", \"abc\")` |\r\n|[`!in~`](not-in-operator.md) |Not equals to any of the elements |No | `\"bCa\" !in~ (\"123\", \"345\", \"ABC\")` |\r\n\r\n> [!NOTE]\r\n>\r\n> * In tabular expressions, the first column of the result set is selected.\r\n> * The expression list can produce up to `1,000,000` values.\r\n> * Nested arrays are flattened into a single list of values. For example, `x in (dynamic([1,[2,3]]))` becomes `x in (1,2,3)`.\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `in`, not `in~`. \r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. \r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `!in` `(`*list of scalar expressions*`)`\r\n*T* `|` `where` *col* `!in` `(`*tabular expression*`)`\r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *list of expressions* - A comma-separated list of tabular, scalar, or literal expressions.\r\n* *tabular expression* - A tabular expression that has a set of values. If the expression has multiple columns, the first column is used.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents \r\n| where State !in (\"FLORIDA\", \"GEORGIA\", \"NEW YORK\") \r\n| count\r\n```\r\n\r\n**Output**\r\n\r\n|Count|\r\n|---|\r\n|54291|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-in-cs-operator", "not-in-cs-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!in~", "Filters a record set for data without a case-insensitive string.", "The following table provides a comparison of the `has` operators:.\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`in`](in-cs-operator.md) |Equals to one of the elements |Yes |`\"abc\" in (\"123\", \"345\", \"abc\")`|\r\n|[`!in`](not-in-cs-operator.md) |Not equals to any of the elements |Yes | `\"bca\" !in (\"123\", \"345\", \"abc\")` |\r\n|[`in~`](inoperator.md) |Equals to any of the elements |No | `\"Abc\" in~ (\"123\", \"345\", \"abc\")` |\r\n|[`!in~`](not-in-operator.md) |Not equals to any of the elements |No | `\"bCa\" !in~ (\"123\", \"345\", \"ABC\")` |\r\n\r\n> [!NOTE]\r\n>\r\n> * In tabular expressions, the first column of the result set is selected.\r\n> * The expression list can produce up to `1,000,000` values.\r\n> * Nested arrays are flattened into a single list of values. For example, `x in (dynamic([1,[2,3]]))` becomes `x in (1,2,3)`.\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\nCase-insensitive operators are currently supported only for ASCII-text. For non-ASCII comparison, use the [tolower()](tolowerfunction.md) function.\r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `in`, not `in~`. \r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. \r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `!in~` `(`*list of scalar expressions*`)`  \r\n*T* `|` `where` *col* `!in~` `(`*tabular expression*`)`   \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *list of expressions* - A comma-separated list of tabular, scalar, or literal expressions.\r\n* *tabular expression* - A tabular expression that has a set of values. If the expression has multiple columns, the first column is used.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents \r\n| where State !in~ (\"Florida\", \"Georgia\", \"New York\") \r\n| count\r\n```\r\n\r\n**Output**\r\n\r\n|Count|\r\n|---|\r\n|54,291|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-in-operator", "not-in-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!startswith", "Filters a record set retrieving data that does not start a case-insensitive search string.", "The following table provides a comparison of the `startswith` operators:\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table below:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`startswith`](startswith-operator.md) |RHS is an initial subsequence of LHS |No |`\"Fabrikam\" startswith \"fab\"`|\r\n|[`!startswith`](not-startswith-operator.md) |RHS isn't an initial subsequence of LHS |No |`\"Fabrikam\" !startswith \"kam\"`|\r\n|[`startswith_cs`](startswith-cs-operator.md)  |RHS is an initial subsequence of LHS |Yes |`\"Fabrikam\" startswith_cs \"Fab\"`|\r\n|[`!startswith_cs`](not-startswith-cs-operator.md) |RHS isn't an initial subsequence of LHS |Yes |`\"Fabrikam\" !startswith_cs \"fab\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\nCase-insensitive operators are currently supported only for ASCII-text. For non-ASCII comparison, use the [tolower()](tolowerfunction.md) function.\r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `hassuffix_cs`, not `hassuffix`. \r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. \r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `!startswith` `(`*expression*`)` \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !startswith \"i\"\r\n    | where event_count > 2000\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|TEXAS|4701|\r\n|KANSAS|3166|\r\n|MISSOURI|2016|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-startswith-operator", "not-startswith-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "!startswith_cs", "Filters a record set for data that does not start with a case-sensitive search string.", "The following table provides a comparison of the `startswith` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`startswith`](startswith-operator.md) |RHS is an initial subsequence of LHS |No |`\"Fabrikam\" startswith \"fab\"`|\r\n|[`!startswith`](not-startswith-operator.md) |RHS isn't an initial subsequence of LHS |No |`\"Fabrikam\" !startswith \"kam\"`|\r\n|[`startswith_cs`](startswith-cs-operator.md)  |RHS is an initial subsequence of LHS |Yes |`\"Fabrikam\" startswith_cs \"Fab\"`|\r\n|[`!startswith_cs`](not-startswith-cs-operator.md) |RHS isn't an initial subsequence of LHS |Yes |`\"Fabrikam\" !startswith_cs \"fab\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `hassuffix_cs`, not `hassuffix`. \r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. \r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `!startswith_cs` `(`*expression*`)`  \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State !startswith_cs \"I\"\r\n    | where event_count > 2000\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|TEXAS|4701|\r\n|KANSAS|3166|\r\n|MISSOURI|2016|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/not-startswith-cs-operator", "not-startswith-cs-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "=~", "Filters a record set for data with a case-insensitive string.", "The following table provides a comparison of the `==` (equals) operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`==`](equals-cs-operator.md)|Equals |Yes|`\"aBc\" == \"aBc\"`|\r\n|[`!=`](not-equals-cs-operator.md)|Not equals |Yes |`\"abc\" != \"ABC\"`|\r\n|[`=~`](equals-operator.md) |Equals |No |`\"abc\" =~ \"ABC\"`|\r\n|[`!~`](not-equals-operator.md) |Not equals |No |`\"aBc\" !~ \"xyz\"`|\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\nCase-insensitive operators are currently supported only for ASCII-text. For non-ASCII comparison, use the [tolower()](tolowerfunction.md) function.\r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `==`, not `=~`. For best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax \r\n\r\n*T* `|` `where` *col* `=~` `(`*expression*`)`\r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | where State =~ \"kansas\"\r\n    | count \r\n```\r\n\r\n**Output**\r\n\r\n|Count|\r\n|---|\r\n|3,166|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/equals-operator", "equals-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "==", "Filters a record set for data matching a case-sensitive string.", "The following table provides a comparison of the `==` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`==`](equals-cs-operator.md)|Equals |Yes|`\"aBc\" == \"aBc\"`|\r\n|[`!=`](not-equals-cs-operator.md)|Not equals |Yes |`\"abc\" != \"ABC\"`|\r\n|[`=~`](equals-operator.md) |Equals |No |`\"abc\" =~ \"ABC\"`|\r\n|[`!~`](not-equals-operator.md) |Not equals |No |`\"aBc\" !~ \"xyz\"`|\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `==`, not `=~`. For best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `==` `(`*expressions`)`\r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | where State == \"kansas\"\r\n    | count \r\n```\r\n\r\n**Output**\r\n\r\n|Count|\r\n|---|\r\n|0|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/equals-cs-operator", "equals-cs-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "abs", "Calculates the absolute value of the input.", "## Syntax\r\n\r\n`abs(`*x*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: An integer or real number, or a timespan value.\r\n\r\n## Returns\r\n\r\n* Absolute value of x.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/abs-function", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "acos", "Returns the angle whose cosine is the specified number (the inverse operation of [`cos()`](cosfunction.md)).", "## Syntax\r\n\r\n`acos(`*x*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: A real number in range [-1, 1].\r\n\r\n## Returns\r\n\r\n* The value of the arc cosine of `x`\r\n* `null` if `x` < -1 or `x` > 1", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/acosfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "active_users_count", "Calculates distinct count of values, where each value has appeared in at least a minimum number of periods in a lookback period.", "Useful for calculating distinct counts of \"fans\" only,  while not including appearances of \"non-fans\". A user is counted as a \"fan\" only if it was active during the lookback period. The lookback period is only used to determine whether a user is considered `active` (\"fan\") or not. The aggregation itself doesn't include users from the lookback window. In comparison, the [sliding_window_counts](sliding-window-counts-plugin.md) aggregation is performed over a sliding window of the lookback period.\r\n\r\n```\r\nT | evaluate active_users_count(id, datetime_column, startofday(ago(30d)), startofday(now()), 7d, 1d, 2, 7d, dim1, dim2, dim3)\r\n```\r\n\r\n## Syntax\r\n\r\n*T* `| evaluate` `active_users_count(`*IdColumn*`,` *TimelineColumn*`,` *Start*`,` *End*`,` *LookbackWindow*`,` *Period*`,` *ActivePeriodsCount*`,` *Bin* `,` [*dim1*`,` *dim2*`,` ...]`)`\r\n\r\n## Arguments\r\n\r\n* *T*: The input tabular expression.\r\n* *IdColumn*: The name of the column with ID values that represent user activity.\r\n* *TimelineColumn*: The name of the column that represents timeline.\r\n* *Start*: (optional) Scalar with value of the analysis start period.\r\n* *End*: (optional) Scalar with value of the analysis end period.\r\n* *LookbackWindow*: A sliding time window defining a period where user appearance is checked. Lookback period starts at ([current appearance] - [lookback window]) and ends on ([current appearance]).\r\n* *Period*: Scalar constant timespan to count as single appearance (a user will be counted as active if it appears in at least distinct ActivePeriodsCount of this timespan.\r\n* *ActivePeriodsCount*: Minimal number of distinct active periods to decide if user is active. Active users are those users who appeared in at least (equal or greater than) active periods count.\r\n* *Bin*: Scalar constant value of the analysis step period. Can be a numeric/datetime/timestamp value, or a string that is `week`/`month`/`year`. All periods will be the corresponding [startofweek](startofweekfunction.md)/[startofmonth](startofmonthfunction.md)/[startofyear](startofyearfunction.md) functions.\r\n* *dim1*, *dim2*, ...: (optional) list of the dimensions columns that slice the activity metrics calculation.\r\n\r\n## Returns\r\n\r\nReturns a table that has distinct count values for IDs that have appeared in ActivePeriodCounts in the following periods: the lookback period, each timeline period, and each existing dimensions combination.\r\n\r\nOutput table schema is:\r\n\r\n|*TimelineColumn*|dim1|..|dim_n|dcount_values|\r\n|---|---|---|---|---|\r\n|type: as of *TimelineColumn*|..|..|..|long|", "Calculate weekly number of distinct users that appeared in at least three different days over a period of prior eight days. Period of analysis: July 2018.\r\n\r\n```\r\nlet Start = datetime(2018-07-01);\r\nlet End = datetime(2018-07-31);\r\nlet LookbackWindow = 8d;\r\nlet Period = 1d;\r\nlet ActivePeriods = 3;\r\nlet Bin = 7d; \r\nlet T =  datatable(User:string, Timestamp:datetime)\r\n[\r\n    \"B\",      datetime(2018-06-29),\r\n    \"B\",      datetime(2018-06-30),\r\n    \"A\",      datetime(2018-07-02),\r\n    \"B\",      datetime(2018-07-04),\r\n    \"B\",      datetime(2018-07-08),\r\n    \"A\",      datetime(2018-07-10),\r\n    \"A\",      datetime(2018-07-14),\r\n    \"A\",      datetime(2018-07-17),\r\n    \"A\",      datetime(2018-07-20),\r\n    \"B\",      datetime(2018-07-24)\r\n]; \r\nT | evaluate active_users_count(User, Timestamp, Start, End, LookbackWindow, Period, ActivePeriods, Bin)\r\n```\r\n\r\n|Timestamp|`dcount`|\r\n|---|---|\r\n|2018-07-01 00:00:00.0000000|1|\r\n|2018-07-15 00:00:00.0000000|1|\r\n\r\nA user is considered active if it fulfills both of the following criteria:\r\n\r\n* The user was seen in at least three distinct days (Period = 1d, ActivePeriods=3).\r\n* The user was seen in a lookback window of 8d before and including their current appearance.\r\n\r\n\r\nIn the illustration below, the only appearances that are active by this criteria are the following instances: User A on 7/20 and User B on 7/4 (see plugin results above).\r\nThe appearances of User B are included for the lookback window on 7/4, but not for the Start-End time range of 6/29-30.\r\n\r\n![Graph showing active users based on the loopback window and active period specified in the query.](images/queries/active-users-count.png)", "https://docs.microsoft.com/azure/data-explorer/kusto/query/active-users-count-plugin", "active-users-count-plugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "activity_counts_metrics", "Calculates useful activity metrics for each time window compared/aggregated to *all* previous time windows. Metrics include: total count values, distinct count values, distinct count of new values, and aggregated distinct count. Compare this plugin to [activity_metrics plugin](activity-metrics-plugin.md), in which every time window is compared to its previous time window only.", "```\r\nT | evaluate activity_counts_metrics(id, datetime_column, startofday(ago(30d)), startofday(now()), 1d, dim1, dim2, dim3)\r\n```\r\n\r\n## Syntax\r\n\r\n*T* `| evaluate` `activity_counts_metrics(`*IdColumn*`,` *TimelineColumn*`,` *Start*`,` *End*`,` *Window* [`,` *Cohort*] [`,` *dim1*`,` *dim2*`,` ...] [`,` *Lookback*] `)`\r\n\r\n## Arguments\r\n\r\n* *T*: The input tabular expression.\r\n* *IdColumn*: The name of the column with ID values that represent user activity.\r\n* *TimelineColumn*: The name of the column that represents the timeline.\r\n* *Start*: Scalar with value of the analysis start period.\r\n* *End*: Scalar with value of the analysis end period.\r\n* *Window*: Scalar with value of the analysis window period. Can be either a numeric/datetime/timestamp value, or a string that is one of `week`/`month`/`year`, in which case all periods will be [startofweek](startofweekfunction.md)/[startofmonth](startofmonthfunction.md) or [startofyear](startofyearfunction.md).\r\n* *dim1*, *dim2*, ...: (optional) list of the dimensions columns that slice the activity metrics calculation.\r\n\r\n## Returns\r\n\r\nReturns a table that has: total count values, distinct count values, distinct count of new values, and aggregated distinct count for each time window.\r\n\r\nOutput table schema is:\r\n\r\n|`TimelineColumn`|`dim1`|...|`dim_n`|`count`|`dcount`|`new_dcount`|`aggregated_dcount`|\r\n|---|---|---|---|---|---|---|---|\r\n|type: as of *`TimelineColumn`*|..|..|..|long|long|long|long|\r\n\r\n* *`TimelineColumn`*: The time window start time.\r\n* *`count`*: The total records count in the time window and *dim(s)*\r\n* *`dcount`*: The distinct ID values count in the time window and *dim(s)*\r\n* *`new_dcount`*: The distinct ID values in the time window and *dim(s)* compared to all previous time windows.\r\n* *`aggregated_dcount`*: The total aggregated distinct ID values of *dim(s)* from first-time window to current (inclusive).", "### Daily activity counts\r\n\r\nThe next query calculates daily activity counts for the provided input table\r\n\r\n```\r\nlet start=datetime(2017-08-01);\r\nlet end=datetime(2017-08-04);\r\nlet window=1d;\r\nlet T = datatable(UserId:string, Timestamp:datetime)\r\n[\r\n'A', datetime(2017-08-01),\r\n'D', datetime(2017-08-01), \r\n'J', datetime(2017-08-01),\r\n'B', datetime(2017-08-01),\r\n'C', datetime(2017-08-02),  \r\n'T', datetime(2017-08-02),\r\n'J', datetime(2017-08-02),\r\n'H', datetime(2017-08-03),\r\n'T', datetime(2017-08-03),\r\n'T', datetime(2017-08-03),\r\n'J', datetime(2017-08-03),\r\n'B', datetime(2017-08-03),\r\n'S', datetime(2017-08-03),\r\n'S', datetime(2017-08-04),\r\n];\r\n T \r\n | evaluate activity_counts_metrics(UserId, Timestamp, start, end, window)\r\n```\r\n\r\n|`Timestamp`|`count`|`dcount`|`new_dcount`|`aggregated_dcount`|\r\n|---|---|---|---|---|\r\n|2017-08-01 00:00:00.0000000|4|4|4|4|\r\n|2017-08-02 00:00:00.0000000|3|3|2|6|\r\n|2017-08-03 00:00:00.0000000|6|5|2|8|\r\n|2017-08-04 00:00:00.0000000|1|1|0|8|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/activity-counts-metrics-plugin", "activity-counts-metrics-plugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "activity_engagement", "Calculates activity engagement ratio based on ID column over a sliding timeline window.", "activity_engagement plugin can be used for calculating DAU/WAU/MAU (daily/weekly/monthly activities).\r\n\r\n```\r\nT | evaluate activity_engagement(id, datetime_column, 1d, 30d)\r\n```\r\n\r\n## Syntax\r\n\r\n*T* `| evaluate` `activity_engagement(`*IdColumn*`,` *TimelineColumn*`,` [*Start*`,` *End*`,`] *InnerActivityWindow*`,` *OuterActivityWindow* [`,` *dim1*`,` *dim2*`,` ...]`)`\r\n\r\n## Arguments\r\n\r\n* *T*: The input tabular expression.\r\n* *IdColumn*: The name of the column with ID values that represent user activity.\r\n* *TimelineColumn*: The name of the column that represents timeline.\r\n* *Start*: (optional) Scalar with value of the analysis start period.\r\n* *End*: (optional) Scalar with value of the analysis end period.\r\n* *InnerActivityWindow*: Scalar with value of the inner-scope analysis window period.\r\n* *OuterActivityWindow*: Scalar with value of the outer-scope analysis window period.\r\n* *dim1*, *dim2*, ...: (optional) list of the dimensions columns that slice the activity metrics calculation.\r\n\r\n## Returns\r\n\r\nReturns a table that has a distinct count of ID values inside an inner-scope window, a distinct count of ID values inside an outer-scope window, and the activity ratio for each inner-scope window period, and for each existing dimensions combination.\r\n\r\nOutput table schema is:\r\n\r\n|TimelineColumn|dcount_activities_inner|dcount_activities_outer|activity_ratio|dim1|..|dim_n|\r\n|---|---|---|---|--|--|--|--|--|--|\r\n|type: as of *TimelineColumn*|long|long|double|..|..|..|", "### DAU/WAU calculation\r\n\r\nThe following example calculates DAU/WAU (Daily Active Users / Weekly Active Users ratio) over a randomly generated data.\r\n\r\n```\r\n// Generate random data of user activities\r\nlet _start = datetime(2017-01-01);\r\nlet _end = datetime(2017-01-31);\r\nrange _day from _start to _end  step 1d\r\n| extend d = tolong((_day - _start)/1d)\r\n| extend r = rand()+1\r\n| extend _users=range(tolong(d*50*r), tolong(d*50*r+100*r-1), 1) \r\n| mv-expand id=_users to typeof(long) limit 1000000\r\n// Calculate DAU/WAU ratio\r\n| evaluate activity_engagement(['id'], _day, _start, _end, 1d, 7d)\r\n| project _day, Dau_Wau=activity_ratio*100 \r\n| render timechart \r\n```\r\n\r\n![Graph displaying the ratio of daily active users to weekly active users as specified in the query.](images/activity-engagement-plugin/activity-engagement-dau-wau.png)\r\n\r\n### DAU/MAU calculation\r\n\r\nThe following example calculates DAU/WAU (Daily Active Users / Weekly Active Users ratio) over a randomly generated data.\r\n\r\n```\r\n// Generate random data of user activities\r\nlet _start = datetime(2017-01-01);\r\nlet _end = datetime(2017-05-31);\r\nrange _day from _start to _end  step 1d\r\n| extend d = tolong((_day - _start)/1d)\r\n| extend r = rand()+1\r\n| extend _users=range(tolong(d*50*r), tolong(d*50*r+100*r-1), 1) \r\n| mv-expand id=_users to typeof(long) limit 1000000\r\n// Calculate DAU/MAU ratio\r\n| evaluate activity_engagement(['id'], _day, _start, _end, 1d, 30d)\r\n| project _day, Dau_Mau=activity_ratio*100 \r\n| render timechart \r\n```\r\n\r\n![Graph displaying the ratio of daily active users to monthly active users as specified in the query.](images/activity-engagement-plugin/activity-engagement-dau-mau.png)\r\n\r\n### DAU/MAU calculation with additional dimensions\r\n\r\nThe following example calculates DAU/WAU (Daily Active Users / Weekly Active Users ratio) over a randomly generated data with additional dimension (`mod3`).\r\n\r\n```\r\n// Generate random data of user activities\r\nlet _start = datetime(2017-01-01);\r\nlet _end = datetime(2017-05-31);\r\nrange _day from _start to _end  step 1d\r\n| extend d = tolong((_day - _start)/1d)\r\n| extend r = rand()+1\r\n| extend _users=range(tolong(d*50*r), tolong(d*50*r+100*r-1), 1) \r\n| mv-expand id=_users to typeof(long) limit 1000000\r\n| extend mod3 = strcat(\"mod3=\", id % 3)\r\n// Calculate DAU/MAU ratio\r\n| evaluate activity_engagement(['id'], _day, _start, _end, 1d, 30d, mod3)\r\n| project _day, Dau_Mau=activity_ratio*100, mod3 \r\n| render timechart \r\n```\r\n\r\n![Graph displaying the ratio of daily active users to monthly active users with modulo 3 as specified in the query.](images/activity-engagement-plugin/activity-engagement-dau-mau-mod3.png)", "https://docs.microsoft.com/azure/data-explorer/kusto/query/activity-engagement-plugin", "activity-engagement-plugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "activity_metrics", "Calculates useful activity metrics based on the current period window compared to the previous period window. The metrics include distinct count values, distinct count of new values, retention rate, and churn rate. This plugin is different from [activity_counts_metrics plugin](activity-counts-metrics-plugin.md) in which every time window is compared to *all* previous time windows.", "```\r\nT | evaluate activity_metrics(id, datetime_column, startofday(ago(30d)), startofday(now()), 1d, dim1, dim2, dim3)\r\n```\r\n\r\n## Syntax\r\n\r\n*T* `| evaluate` `activity_metrics(`*IdColumn*`,` *TimelineColumn*`,` [*Start*`,` *End*`,`] *Window* [`,` *dim1*`,` *dim2*`,` ...]`)`\r\n\r\n## Arguments\r\n\r\n* *T*: The input tabular expression.\r\n* *IdColumn*: The name of the column with ID values that represent user activity.\r\n* *TimelineColumn*: The name of the column that represents timeline.\r\n* *Start*: (optional) Scalar with value of the analysis start period.\r\n* *End*: (optional) Scalar with value of the analysis end period.\r\n* *Window*: Scalar with value of the analysis window period. Can be either a numeric, datetime, timestamp, or string value. Strings are either `week`, `month`, `year`, in which case all periods will be [startofweek](startofweekfunction.md), [startofmonth](startofmonthfunction.md), [startofyear](startofyearfunction.md) respectively.\r\n* *dim1*, *dim2*, ...: (optional) list of the dimensions columns that slice the activity metrics calculation.\r\n\r\n## Returns\r\n\r\nThe plugin returns a table with the distinct count values, distinct count of new values, retention rate, and churn rate for each timeline period, and for each existing dimensions combination.\r\n\r\nOutput table schema is:\r\n\r\n|*TimelineColumn*|dcount_values|dcount_newvalues|retention_rate|churn_rate|dim1|..|dim_n|\r\n|---|---|---|---|---|--|--|--|--|--|--|\r\n|type: as of *TimelineColumn*|long|long|double|double|..|..|..|\r\n\r\n### Notes\r\n\r\n***Retention Rate Definition***\r\n\r\n`Retention Rate` over a period is calculated as:\r\n\r\n> *number of customers returned during the period*\r\n> / (divided by)\r\n> *number customers at the beginning of the period*\r\n\r\nwhere the `# of customers returned during the period` is defined as:\r\n\r\n> *number of customers at end of period*\r\n> \\- (minus)\r\n> *number of new customers acquired during the period*\r\n\r\n`Retention Rate` can vary from 0.0 to 1.0\r\nThe higher score means the larger number of returning users.\r\n\r\n***Churn Rate Definition***\r\n\r\n`Churn Rate` over a period is calculated as:\r\n\r\n> *number of customers lost in the period*\r\n> / (divided by)\r\n> *number of customers at the beginning of the period*\r\n\r\nwhere the `# of customer lost in the period` is defined as:\r\n\r\n> *number of customers at the beginning of the period*\r\n> \\- (minus)\r\n> *number of customers at the end of the period*\r\n\r\n`Churn Rate` can vary from 0.0 to 1.0\r\nThe higher score means the larger number of users are NOT returning to the service.\r\n\r\n***Churn vs. Retention Rate***\r\n\r\nDerived from the definition of `Churn Rate` and `Retention Rate`, the following calculation is always true:\r\n\r\n> [`Retention Rate`] = 100.0% - [`Churn Rate`]", "### Weekly retention rate and churn rate\r\n\r\nThe next query calculates retention and churn rate for week-over-week window.\r\n\r\n```\r\n// Generate random data of user activities\r\nlet _start = datetime(2017-01-02);\r\nlet _end = datetime(2017-05-31);\r\nrange _day from _start to _end  step 1d\r\n| extend d = tolong((_day - _start)/1d)\r\n| extend r = rand()+1\r\n| extend _users=range(tolong(d*50*r), tolong(d*50*r+200*r-1), 1)\r\n| mv-expand id=_users to typeof(long) limit 1000000\r\n//\r\n| evaluate activity_metrics(['id'], _day, _start, _end, 7d)\r\n| project _day, retention_rate, churn_rate\r\n| render timechart\r\n```\r\n\r\n|_day|retention_rate|churn_rate|\r\n|---|---|---|\r\n|2017-01-02 00:00:00.0000000|NaN|NaN|\r\n|2017-01-09 00:00:00.0000000|0.179910044977511|0.820089955022489|\r\n|2017-01-16 00:00:00.0000000|0.744374437443744|0.255625562556256|\r\n|2017-01-23 00:00:00.0000000|0.612096774193548|0.387903225806452|\r\n|2017-01-30 00:00:00.0000000|0.681141439205955|0.318858560794045|\r\n|2017-02-06 00:00:00.0000000|0.278145695364238|0.721854304635762|\r\n|2017-02-13 00:00:00.0000000|0.223172628304821|0.776827371695179|\r\n|2017-02-20 00:00:00.0000000|0.38|0.62|\r\n|2017-02-27 00:00:00.0000000|0.295519001701645|0.704480998298355|\r\n|2017-03-06 00:00:00.0000000|0.280387770320656|0.719612229679344|\r\n|2017-03-13 00:00:00.0000000|0.360628154795289|0.639371845204711|\r\n|2017-03-20 00:00:00.0000000|0.288008028098344|0.711991971901656|\r\n|2017-03-27 00:00:00.0000000|0.306134969325153|0.693865030674847|\r\n|2017-04-03 00:00:00.0000000|0.356866537717602|0.643133462282398|\r\n|2017-04-10 00:00:00.0000000|0.495098039215686|0.504901960784314|\r\n|2017-04-17 00:00:00.0000000|0.198296836982968|0.801703163017032|\r\n|2017-04-24 00:00:00.0000000|0.0618811881188119|0.938118811881188|\r\n|2017-05-01 00:00:00.0000000|0.204657727593507|0.795342272406493|\r\n|2017-05-08 00:00:00.0000000|0.517391304347826|0.482608695652174|\r\n|2017-05-15 00:00:00.0000000|0.143667296786389|0.856332703213611|\r\n|2017-05-22 00:00:00.0000000|0.199122325836533|0.800877674163467|\r\n|2017-05-29 00:00:00.0000000|0.063468992248062|0.936531007751938|\r\n\r\n![Table showing the calculated retention and churn rates per 7 days as specified in the query.](images/activity-metrics-plugin/activity-metrics-churn-and-retention.png)\r\n\r\n### Distinct values and distinct 'new' values\r\n\r\nThe next query calculates distinct values and 'new' values (IDs that didn't appear in previous time window) for week-over-week window.\r\n\r\n```\r\n// Generate random data of user activities\r\nlet _start = datetime(2017-01-02);\r\nlet _end = datetime(2017-05-31);\r\nrange _day from _start to _end  step 1d\r\n| extend d = tolong((_day - _start)/1d)\r\n| extend r = rand()+1\r\n| extend _users=range(tolong(d*50*r), tolong(d*50*r+200*r-1), 1)\r\n| mv-expand id=_users to typeof(long) limit 1000000\r\n//\r\n| evaluate activity_metrics(['id'], _day, _start, _end, 7d)\r\n| project _day, dcount_values, dcount_newvalues\r\n| render timechart\r\n```\r\n\r\n| _day | dcount_values | dcount_newvalues |\r\n|--|--|--|\r\n| 2017-01-02 00:00:00.0000000 | 630 | 630 |\r\n| 2017-01-09 00:00:00.0000000 | 738 | 575 |\r\n| 2017-01-16 00:00:00.0000000 | 1187 | 841 |\r\n| 2017-01-23 00:00:00.0000000 | 1092 | 465 |\r\n| 2017-01-30 00:00:00.0000000 | 1261 | 647 |\r\n| 2017-02-06 00:00:00.0000000 | 1744 | 1043 |\r\n| 2017-02-13 00:00:00.0000000 | 1563 | 432 |\r\n| 2017-02-20 00:00:00.0000000 | 1406 | 818 |\r\n| 2017-02-27 00:00:00.0000000 | 1956 | 1429 |\r\n| 2017-03-06 00:00:00.0000000 | 1593 | 848 |\r\n| 2017-03-13 00:00:00.0000000 | 1801 | 1423 |\r\n| 2017-03-20 00:00:00.0000000 | 1710 | 1017 |\r\n| 2017-03-27 00:00:00.0000000 | 1796 | 1516 |\r\n| 2017-04-03 00:00:00.0000000 | 1381 | 1008 |\r\n| 2017-04-10 00:00:00.0000000 | 1756 | 1162 |\r\n| 2017-04-17 00:00:00.0000000 | 1831 | 1409 |\r\n| 2017-04-24 00:00:00.0000000 | 1823 | 1164 |\r\n| 2017-05-01 00:00:00.0000000 | 1811 | 1353 |\r\n| 2017-05-08 00:00:00.0000000 | 1691 | 1246 |\r\n| 2017-05-15 00:00:00.0000000 | 1812 | 1608 |\r\n| 2017-05-22 00:00:00.0000000 | 1740 | 1017 |\r\n| 2017-05-29 00:00:00.0000000 | 960 | 756 |\r\n\r\n![Table showing the count of distinct values (dcount_values) and of new distinct values (dcount_newvalues) that didn](images/activity-metrics-plugin/activity-metrics-dcount-and-dcount-newvalues.png)", "https://docs.microsoft.com/azure/data-explorer/kusto/query/activity-metrics-plugin", "activity-metrics-plugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "ago", "Subtracts the given timespan from the current UTC clock time.", "```\r\nago(1h)\r\nago(1d)\r\n```\r\n\r\nLike `now()`, this function can be used multiple times\r\nin a statement and the UTC clock time being referenced will be the same\r\nfor all instantiations.\r\n\r\n## Syntax\r\n\r\n`ago(`*a_timespan*`)`\r\n\r\n## Arguments\r\n\r\n* *[a_timespan](scalar-data-types/timespan.md)*: Interval to subtract from the current UTC clock time\r\n(`now()`).\r\n\r\n## Returns\r\n\r\n`now() - a_timespan`", "All rows with a timestamp in the past hour:\r\n\r\n```\r\nT | where Timestamp > ago(1h)\r\n```", "https://docs.microsoft.com/azure/data-explorer/kusto/query/agofunction", "agofunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "arg_max", "Finds a row in the group that maximizes *ExprToMaximize*, and returns the value of *ExprToReturn* (or `*` to return the entire row).", "* Can be used only in context of aggregation inside [summarize](summarizeoperator.md)\r\n\r\n## Syntax\r\n\r\n`arg_max` `(`*ExprToMaximize*`,` *\\** | *ExprToReturn*  [`,` ...]`)`\r\n\r\n## Arguments\r\n\r\n* *ExprToMaximize*: Expression that will be used for aggregation calculation.\r\n* *ExprToReturn*: Expression that will be used for returning the value when *ExprToMaximize* is\r\n  maximum. Expression to return may be a wildcard (*) to return all columns of the input table.\r\n\r\n## Returns\r\n\r\nFinds a row in the group that maximizes *ExprToMaximize*, and\r\nreturns the value of *ExprToReturn* (or `*` to return the entire row).", "See examples for [arg_min()](arg-min-aggfunction.md) aggregation function.", "https://docs.microsoft.com/azure/data-explorer/kusto/query/arg-max-aggfunction", "arg-max-aggfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "arg_min", "Finds a row in the group that minimizes *ExprToMinimize*, and returns the value of *ExprToReturn* (or `*` to return the entire row).", "* Can be used only in context of aggregation inside [summarize](summarizeoperator.md)\r\n\r\n## Syntax\r\n\r\n`arg_min` `(`*ExprToMinimize*`,` *\\** | *ExprToReturn*  [`,` ...]`)`\r\n\r\n## Arguments\r\n\r\n* *ExprToMinimize*: Expression that will be used for aggregation calculation.\r\n* *ExprToReturn*: Expression that will be used for returning the value when *ExprToMinimize* is\r\n  minimum. Expression to return may be a wildcard (*) to return all columns of the input table.\r\n  \r\n\r\n## Null handling\r\n\r\nWhen *ExprToMinimize* is null for all rows in a group, one row in the group is picked. Otherwise, rows where *ExprToMinimize* is null are ignored.\r\n\r\n## Returns\r\n\r\nFinds a row in the group that minimizes *ExprToMinimize*, and returns the value of *ExprToReturn* (or `*` to return the entire row).", "Show cheapest supplier of each product:\r\n\r\n```\r\nSupplies | summarize arg_min(Price, Supplier) by Product\r\n```\r\n\r\nShow all the details, not just the supplier name:\r\n\r\n```\r\nSupplies | summarize arg_min(Price, *) by Product\r\n```\r\n\r\nFind the southernmost city in each continent, with its country:\r\n\r\n```\r\nPageViewLog \r\n| summarize (latitude, min_lat_City, min_lat_country)=arg_min(latitude, City, country) \r\n    by continent\r\n```\r\n\r\n![Table showing the southernmost city with its country as calculated by the query.](images/arg-min-aggfunction/arg-min.png)\r\nNull handling example:\r\n\r\n```\r\ndatatable(Fruit: string, Color: string, Version: int) [\r\n    \"Apple\", \"Red\", 1,\r\n    \"Apple\", \"Green\", int(null),\r\n    \"Banana\", \"Yellow\", int(null),\r\n    \"Banana\", \"Green\", int(null),\r\n    \"Pear\", \"Brown\", 1,\r\n    \"Pear\", \"Green\", 2,\r\n]\r\n| summarize arg_min(Version, *) by Fruit\r\n```\r\n\r\n|Fruit|Version|Color|\r\n|---|---|---|\r\n|Apple|1|Red|\r\n|Banana||Yellow|\r\n|Pear|1|Brown|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/arg-min-aggfunction", "arg-min-aggfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "around", "Returns a `bool` value indicating if the first argument is within a range around the center value.", "## Syntax\r\n\r\n`around(`*value*`,`*center*`,`*delta*`)`\r\n\r\n## Arguments\r\n\r\n* *value*: A number, date, or [timespan](scalar-data-types/timespan.md) that is evaluated.\r\n* *center*: A number, date, or [timespan](scalar-data-types/timespan.md) representing the center of the range defined as [(`center`-`delta`) .. (`center` + `delta`)].\r\n* *delta*: A number, or [timespan](scalar-data-types/timespan.md) representing the delta value of the range defined as [(`center`-`delta`) .. (`center` + `delta`)].\r\n\r\n## Returns\r\n\r\nReturns `true` if the value is within the range, `false` if the value is outside the range.\r\nReturns `null` if any of the arguments is `null`.\r\n\r\n## Example: Filtering values around a specific timestamp\r\n\r\nThe following example filters rows around specific timestamp.\r\n\r\n```\r\nrange dt \r\n    from datetime(2021-01-01 01:00) \r\n    to datetime(2021-01-01 02:00) \r\n    step 1min\r\n| where around(dt, datetime(2021-01-01 01:30), 1min)\r\n```\r\n\r\n|dt|\r\n|---|\r\n|2021-01-01 01:29:00.0000000|\r\n|2021-01-01 01:30:00.0000000|\r\n|2021-01-01 01:31:00.0000000|", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/around-function", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_concat", "Concatenates many dynamic arrays to a single array.", "## Syntax\r\n\r\n`array_concat(`*arr1*`[`, `*arr2*, ...]`)`\r\n\r\n## Arguments\r\n\r\n* *arr1...arrN*: Input arrays to be concatenated into a dynamic array. All arguments must be dynamic arrays (see [pack_array](packarrayfunction.md)).\r\n\r\n## Returns\r\n\r\nDynamic array of arrays with arr1, arr2, ... , arrN.", "```\r\nrange x from 1 to 3 step 1\r\n| extend y = x * 2\r\n| extend z = y * 2\r\n| extend a1 = pack_array(x,y,z), a2 = pack_array(x, y)\r\n| project array_concat(a1, a2)\r\n```\r\n\r\n|Column1|\r\n|---|\r\n|[1,2,4,1,2]|\r\n|[2,4,8,2,4]|\r\n|[3,6,12,3,6]|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/arrayconcatfunction", "arrayconcatfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_iif", "Element-wise iif function on dynamic arrays.", "Another alias: array_iff().\r\n\r\n## Syntax\r\n\r\n`array_iif(`*ConditionArray*, *IfTrue*, *IfFalse*]`)`\r\n\r\n## Arguments\r\n\r\n* *conditionArray*: Input array of *boolean* or numeric values, must be dynamic array.\r\n* *ifTrue*: Input array of values or primitive value - the result value(s) when the corresponding value of *ConditionArray* is *true*.\r\n* *ifFalse*: Input array of values or primitive value - the result value(s) when the corresponding value of *ConditionArray* is *false*.\r\n\r\n### Notes\r\n\r\n* The result length is the length of *conditionArray*.\r\n* Numeric condition value is treated as *condition* != *0*.\r\n* Non-numeric/null condition value will have null in the corresponding index of the result.\r\n* Missing values (in shorter length arrays) are treated as null.\r\n\r\n## Returns\r\n\r\nDynamic array of the values taken either from the *IfTrue* or *IfFalse* [array] values, according to the corresponding value of the Condition array.", "```\r\nprint condition=dynamic([true,false,true]), l=dynamic([1,2,3]), r=dynamic([4,5,6]) \r\n| extend res=array_iif(condition, l, r)\r\n```\r\n\r\n|condition|l|r|res|\r\n|---|---|---|---|\r\n|[true, false, true]|[1, 2, 3]|[4, 5, 6]|[1, 5, 3]|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/arrayifffunction", "arrayifffunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_index_of", "Searches the array for the specified item, and returns its position.", "## Syntax\r\n\r\n`array_index_of(`*array*,*lookup*`)`\r\n\r\n## Arguments\r\n\r\n* *array*: Input array to search.\r\n* *lookup*: Value to lookup. The value should be of type long, integer, double, datetime, timespan, decimal, string, or guid.\r\n* *start_index*: Search start position. A negative value will offset the starting search value from the end of the array by this many steps: abs(start_index). Optional.\r\n* *length*: Number of values to examine. A value of -1 means unlimited length. Optional.\r\n* *occurrence*: The number of the occurrence. Default 1. Optional.\r\n\r\n## Returns\r\n\r\nZero-based index position of lookup.\r\nReturns -1 if the value isn't found in the array.\r\n\r\nFor irrelevant inputs (*occurrence* < 0 or  *length* < -1) - returns *null*.", "```\r\nlet arr=dynamic([\"this\", \"is\", \"an\", \"example\", \"an\", \"example\"]);\r\nprint\r\n idx1 = array_index_of(arr,\"an\")    // lookup found in input string\r\n , idx2 = array_index_of(arr,\"example\",1,3) // lookup found in researched range \r\n , idx3 = array_index_of(arr,\"example\",1,2) // search starts from index 1, but stops after 2 values, so lookup can't be found\r\n , idx4 = array_index_of(arr,\"is\",2,4) // search starts after occurrence of lookup\r\n , idx5 = array_index_of(arr,\"example\",2,-1)  // lookup found\r\n , idx6 = array_index_of(arr, \"an\", 1, -1, 2)   // second occurrence found in input range\r\n , idx7 = array_index_of(arr, \"an\", 1, -1, 3)   // no third occurrence in input array\r\n , idx8 = array_index_of(arr, \"an\", -3)   // negative start index will look at last 3 elements\r\n , idx9 = array_index_of(arr, \"is\", -4)   // negative start index will look at last 3 elements\r\n```\r\n\r\n|idx1|idx2|idx3|idx4|idx5|idx6|idx7|idx8|idx9|\r\n|----|----|----|----|----|----|----|----|----|\r\n|2   |3   |-1  |-1   |3   |4   |-1  |4  |-1  |\r\n\r\n## See also\r\n\r\nIf you only want to check whether a value exists in an array,\r\nbut you are not interested in its position, you can use\r\n[set_has_element(`arr`, `value`)](sethaselementfunction.md). This function will improve the readability of your query. Both functions have the same performance.", "https://docs.microsoft.com/azure/data-explorer/kusto/query/arrayindexoffunction", "arrayindexoffunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_length", "Calculates the number of elements in a dynamic array.", "## Syntax\r\n\r\n`array_length(`*array*`)`\r\n\r\n## Arguments\r\n\r\n* *array*: A `dynamic` value.\r\n\r\n## Returns\r\n\r\nThe number of elements in *array*, or `null` if *array* is not an array.", "```\r\nprint array_length(parse_json('[1, 2, 3, \"four\"]')) == 4\r\n\r\nprint array_length(parse_json('[8]')) == 1\r\n\r\nprint array_length(parse_json('[{}]')) == 1\r\n\r\nprint array_length(parse_json('[]')) == 0\r\n\r\nprint array_length(parse_json('{}')) == null\r\n\r\nprint array_length(parse_json('21')) == null\r\n```", "https://docs.microsoft.com/azure/data-explorer/kusto/query/arraylengthfunction", "arraylengthfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_reverse", "Reverses the order of the elements in a dynamic array.", "## Syntax\r\n\r\n`array_reverse(`*array*`)`\r\n\r\n## Arguments\r\n\r\n*array*: Input array to reverse.\r\n\r\n## Returns\r\n\r\nAn array that contains exactly the same elements as the input array, but in reverse order.", "```\r\nprint arr=dynamic([\"this\", \"is\", \"an\", \"example\"]) \r\n| project Result=array_reverse(arr)\r\n```\r\n\r\n|Result|\r\n|---|\r\n|[\"example\",\"an\",\"is\",\"this\"]|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/array-reverse-function", "array-reverse-function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_rotate_left", "Rotates values inside a `dynamic` array to the left.", "## Syntax\r\n\r\n`array_rotate_left(`*array*, *rotate_count*`)`\r\n\r\n## Arguments\r\n\r\n* *array*: Input array to rotate, must be dynamic array.\r\n* *rotate_count*: Integer specifying the number of positions that array elements will be rotated to the left. If the value is negative, the elements will be rotated to the right.\r\n\r\n## Returns\r\n\r\nDynamic array containing the same amount of the elements as in original array, where each element was rotated according to *rotate_count*.\r\n\r\n## See also\r\n\r\n* For rotating array to the right, see [array_rotate_right()](array_rotate_rightfunction.md).\r\n* For shifting array to the left, see [array_shift_left()](array_shift_leftfunction.md).\r\n* For shifting array to the right, see [array_shift_right()](array_shift_rightfunction.md).", "* Rotating to the left by two positions:\r\n\r\n    \r\n    ```\r\n    print arr=dynamic([1,2,3,4,5]) \r\n    | extend arr_rotated=array_rotate_left(arr, 2)\r\n    ```\r\n\r\n    |arr|arr_rotated|\r\n    |---|---|\r\n    |[1,2,3,4,5]|[3,4,5,1,2]|\r\n\r\n* Rotating to the right by two positions by using negative rotate_count value:\r\n\r\n    \r\n    ```\r\n    print arr=dynamic([1,2,3,4,5]) \r\n    | extend arr_rotated=array_rotate_left(arr, -2)\r\n    ```\r\n\r\n    |arr|arr_rotated|\r\n    |---|---|\r\n    |[1,2,3,4,5]|[4,5,1,2,3]|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/array_rotate_leftfunction", "array_rotate_leftfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_rotate_right", "Rotates values inside a `dynamic` array to the right.", "## Syntax\r\n\r\n`array_rotate_right(`*array*, *rotate_count*`)`\r\n\r\n## Arguments\r\n\r\n* *array*: Input array to rotate, must be dynamic array.\r\n* *rotate_count*: Integer specifying the number of positions that array elements will be rotated to the right. If the value is negative, the elements will be rotated to the left.\r\n\r\n## Returns\r\n\r\nDynamic array containing the same amount of the elements as in the original array, where each element was rotated according to *rotate_count*.\r\n\r\n## See also\r\n\r\n* For rotating array to the left, see [array_rotate_left()](array_rotate_leftfunction.md).\r\n* For shifting array to the left, see [array_shift_left()](array_shift_leftfunction.md).\r\n* For shifting array to the right, see [array_shift_right()](array_shift_rightfunction.md).", "* Rotating to the right by two positions:\r\n\r\n    \r\n    ```\r\n    print arr=dynamic([1,2,3,4,5]) \r\n    | extend arr_rotated=array_rotate_right(arr, 2)\r\n    ```\r\n\r\n    |arr|arr_rotated|\r\n    |---|---|\r\n    |[1,2,3,4,5]|[4,5,1,2,3]|\r\n\r\n* Rotating to the left by two positions by using negative rotate_count value:\r\n\r\n    \r\n    ```\r\n    print arr=dynamic([1,2,3,4,5]) \r\n    | extend arr_rotated=array_rotate_right(arr, -2)\r\n    ```\r\n\r\n    |arr|arr_rotated|\r\n    |---|---|\r\n    |[1,2,3,4,5]|[3,4,5,1,2]|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/array_rotate_rightfunction", "array_rotate_rightfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_shift_left", "Shifts the values inside a `dynamic` array to the left.", "## Syntax\r\n\r\n`array_shift_left(`*array*, *shift_count* `[,` *fill_value* ]`)`\r\n\r\n## Arguments\r\n\r\n* *array*: Input array to shift, must be dynamic array.\r\n* *shift_count*: Integer specifying the number of positions that array elements will be shifted to the left. If the value is negative, the elements will be shifted to the right.\r\n* *fill_value*: Scalar value that is used for inserting elements instead of the ones that were shifted and removed. Default: null value or empty string (depending on the *array* type).\r\n\r\n## Returns\r\n\r\nDynamic array containing the same number of elements as in the original array. Each element has been shifted according to *shift_count*. New elements that are added in place of removed elements will have a value of *fill_value*.\r\n\r\n## See also\r\n\r\n* For shifting array right, see [array_shift_right()](array_shift_rightfunction.md).\r\n* For rotating array right, see [array_rotate_right()](array_rotate_rightfunction.md).\r\n* For rotating array left, see [array_rotate_left()](array_rotate_leftfunction.md).", "* Shifting to the left by two positions:\r\n\r\n    \r\n    ```\r\n    print arr=dynamic([1,2,3,4,5]) \r\n    | extend arr_shift=array_shift_left(arr, 2)\r\n    ```\r\n\r\n    |`arr`|`arr_shift`|\r\n    |---|---|\r\n    |[1,2,3,4,5]|[3,4,5,null,null]|\r\n\r\n* Shifting to the left by two positions and adding default value:\r\n\r\n    \r\n    ```\r\n    print arr=dynamic([1,2,3,4,5]) \r\n    | extend arr_shift=array_shift_left(arr, 2, -1)\r\n    ```\r\n\r\n    |`arr`|`arr_shift`|\r\n    |---|---|\r\n    |[1,2,3,4,5]|[3,4,5,-1,-1]|\r\n\r\n* Shifting to the right by two positions by using negative *shift_count* value:\r\n\r\n    \r\n    ```\r\n    print arr=dynamic([1,2,3,4,5]) \r\n    | extend arr_shift=array_shift_left(arr, -2, -1)\r\n    ```\r\n\r\n    |`arr`|`arr_shift`|\r\n    |---|---|\r\n    |[1,2,3,4,5]|[-1,-1,1,2,3]|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/array_shift_leftfunction", "array_shift_leftfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_shift_right", "`array_shift_right()` shifts values inside an array to the right.", "## Syntax\r\n\r\n`array_shift_right(`*array*, *shift_count* [, *fill_value* ]`)`\r\n\r\n## Arguments\r\n\r\n* *array*: Input array to shift, must be dynamic array.\r\n* *shift_count*: Integer specifying the number of positions that array elements will be shifted to the right. If the value is negative, the elements will be shifted to the left.\r\n* *fill_value*: scalar value that is used for inserting elements instead of the ones that were shifted and removed. Default: null value or empty string (depending on the *array* type).\r\n\r\n## Returns\r\n\r\nDynamic array containing the same amount of the elements as in the original array. Each element has been shifted according to *shift_count*. New elements that are added instead of the removed elements will have a value of *fill_value*.\r\n\r\n## See also\r\n\r\n* For shifting array left, see [array_shift_left()](array_shift_leftfunction.md).\r\n* For rotating array right, see [array_rotate_right()](array_rotate_rightfunction.md).\r\n* For rotating array left, see [array_rotate_left()](array_rotate_leftfunction.md).", "* Shifting to the right by two positions:\r\n\r\n    \r\n    ```\r\n    print arr=dynamic([1,2,3,4,5]) \r\n    | extend arr_shift=array_shift_right(arr, 2)\r\n    ```\r\n\r\n    |arr|arr_shift|\r\n    |---|---|\r\n    |[1,2,3,4,5]|[null,null,1,2,3]|\r\n\r\n* Shifting to the right by two positions and adding a default value:\r\n\r\n    \r\n    ```\r\n    print arr=dynamic([1,2,3,4,5]) \r\n    | extend arr_shift=array_shift_right(arr, 2, -1)\r\n    ```\r\n\r\n    |arr|arr_shift|\r\n    |---|---|\r\n    |[1,2,3,4,5]|[-1,-1,1,2,3]|\r\n\r\n* Shifting to the left by two positions by using a negative shift_count value:\r\n\r\n    \r\n    ```\r\n    print arr=dynamic([1,2,3,4,5]) \r\n    | extend arr_shift=array_shift_right(arr, -2, -1)\r\n    ```\r\n\r\n    |arr|arr_shift|\r\n    |---|---|\r\n    |[1,2,3,4,5]|[3,4,5,-1,-1]|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/array_shift_rightfunction", "array_shift_rightfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_slice", "Extracts a slice of a dynamic array.", "## Syntax\r\n\r\n`array_slice`(*array*, *start*, *end*)\r\n\r\n## Arguments\r\n\r\n* *array*: Input array to extract the slice from must be dynamic array.\r\n* *start*: zero-based (inclusive) start index of the slice, negative values are converted to array_length+start.\r\n* *end*: zero-based (inclusive) end index of the slice, negative values are converted to array_length+end.\r\n\r\nNote: out of bounds indices are ignored.\r\n\r\n## Returns\r\n\r\nDynamic array of the values in the range [`start..end`] from `array`.", "```\r\nprint arr=dynamic([1,2,3]) \r\n| extend sliced=array_slice(arr, 1, 2)\r\n```\r\n\r\n|`arr`|`sliced`|\r\n|---|---|\r\n|[1,2,3]|[2,3]|\r\n\r\n```\r\nprint arr=dynamic([1,2,3,4,5]) \r\n| extend sliced=array_slice(arr, 2, -1)\r\n```\r\n\r\n|`arr`|sliced|\r\n|---|---|\r\n|[1,2,3,4,5]|[3,4,5]|\r\n\r\n```\r\nprint arr=dynamic([1,2,3,4,5]) \r\n| extend sliced=array_slice(arr, -3, -2)\r\n```\r\n\r\n|`arr`|sliced|\r\n|---|---|\r\n|[1,2,3,4,5]|[3,4]|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/arrayslicefunction", "arrayslicefunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_sort_asc", "Receives one or more arrays. Sorts the first array in ascending order. Orders the remaining arrays to match the reordered first array.", "## Syntax\r\n\r\n`array_sort_asc(`*array1*[, ..., *argumentN*]`)`\r\n\r\n`array_sort_asc(`*array1*[, ..., *argumentN*]`,`*nulls_last*`)`\r\n\r\nIf *nulls_last* isn't provided, a default value of `true` is used.\r\n\r\n## Arguments\r\n\r\n* *array1...arrayN*: Input arrays.\r\n* *nulls_last*: A bool indicating whether `null`s should be last\r\n\r\n## Returns\r\n\r\nReturns the same number of arrays as in the input, with the first array sorted in ascending order, and the remaining arrays ordered to match the reordered first array.\r\n\r\n`null` will be returned for every array that differs in length from the first one.\r\n\r\nIf an array contains elements of different types, it will be sorted in the following order:\r\n\r\n* Numeric, `datetime`, and `timespan` elements\r\n* String elements\r\n* Guid elements\r\n* All other elements\r\n\r\n## Example 1 - Sorting two arrays\r\n\r\n```\r\nlet array1 = dynamic([1,3,4,5,2]);\r\nlet array2 = dynamic([\"a\",\"b\",\"c\",\"d\",\"e\"]);\r\nprint array_sort_asc(array1,array2)\r\n```\r\n\r\n|`array1_sorted`|`array2_sorted`|\r\n|---|---|\r\n|[1,2,3,4,5]|[\"a\",\"e\",\"b\",\"c\",\"d\"]|\r\n\r\n> [!Note]\r\n> The output column names are generated automatically, based on the arguments to the function. To assign different names to the output columns, use the following syntax: `... | extend (out1, out2) = array_sort_asc(array1,array2)`\r\n\r\n## Example 2 - Sorting substrings\r\n\r\n```\r\nlet Names = \"John,Paul,George,Ringo\";\r\nlet SortedNames = strcat_array(array_sort_asc(split(Names, \",\")), \",\");\r\nprint result = SortedNames\r\n```\r\n\r\n|`result`|\r\n|---|\r\n|George,John,Paul,Ringo|\r\n\r\n## Example 3 - Combining summarize and array_sort_asc\r\n\r\n```\r\ndatatable(command:string, command_time:datetime, user_id:string)\r\n[\r\n    'chmod',   datetime(2019-07-15),   \"user1\",\r\n    'ls',      datetime(2019-07-02),   \"user1\",\r\n    'dir',     datetime(2019-07-22),   \"user1\",\r\n    'mkdir',   datetime(2019-07-14),   \"user1\",\r\n    'rm',      datetime(2019-07-27),   \"user1\",\r\n    'pwd',     datetime(2019-07-25),   \"user1\",\r\n    'rm',      datetime(2019-07-23),   \"user2\",\r\n    'pwd',     datetime(2019-07-25),   \"user2\",\r\n]\r\n| summarize timestamps = make_list(command_time), commands = make_list(command) by user_id\r\n| project user_id, commands_in_chronological_order = array_sort_asc(timestamps, commands)[1]\r\n```\r\n\r\n|`user_id`|`commands_in_chronological_order`|\r\n|---|---|\r\n|user1|[<br>  \"ls\",<br>  \"mkdir\",<br>  \"chmod\",<br>  \"dir\",<br>  \"pwd\",<br>  \"rm\"<br>]|\r\n|user2|[<br>  \"rm\",<br>  \"pwd\"<br>]|\r\n\r\n> [!Note]\r\n> If your data may contain `null` values, use [make_list_with_nulls](make-list-with-nulls-aggfunction.md) instead of [make_list](makelist-aggfunction.md).\r\n\r\n## Example 4 - Controlling location of `null` values\r\n\r\nBy default, `null` values are put last in the sorted array. However, you can control it explicitly by adding a `bool` value as the last argument to `array_sort_asc()`.\r\n\r\nExample with default behavior:\r\n\r\n```\r\nprint array_sort_asc(dynamic([null,\"blue\",\"yellow\",\"green\",null]))\r\n```\r\n\r\n|`print_0`|\r\n|---|\r\n|[\"blue\",\"green\",\"yellow\",null,null]|\r\n\r\nExample with non-default behavior:\r\n\r\n```\r\nprint array_sort_asc(dynamic([null,\"blue\",\"yellow\",\"green\",null]), false)\r\n```\r\n\r\n|`print_0`|\r\n|---|\r\n|[null,null,\"blue\",\"green\",\"yellow\"]|\r\n\r\n## See also\r\n\r\nTo sort the first array in descending order, use [array_sort_desc()](arraysortdescfunction.md).", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/arraysortascfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_sort_desc", "Receives one or more arrays. Sorts the first array in descending order. Orders the remaining arrays to match the reordered first array.", "## Syntax\r\n\r\n`array_sort_desc(`*array1*[, ..., *argumentN*]`)`\r\n\r\n`array_sort_desc(`*array1*[, ..., *argumentN*]`,`*nulls_last*`)`\r\n\r\nIf *nulls_last* isn't provided, a default value of `true` is used.\r\n\r\n## Arguments\r\n\r\n* *array1...arrayN*: Input arrays.\r\n* *nulls_last*: A bool indicating whether `null`s should be last\r\n\r\n## Returns\r\n\r\nReturns the same number of arrays as in the input, with the first array sorted in ascending order, and the remaining arrays ordered to match the reordered first array.\r\n\r\n`null` will be returned for every array that differs in length from the first one.\r\n\r\nIf an array contains elements of different types, it will be sorted in the following order:\r\n\r\n* Numeric, `datetime`, and `timespan` elements\r\n* String elements\r\n* Guid elements\r\n* All other elements\r\n\r\n## Example 1 - Sorting two arrays\r\n\r\n```\r\nlet array1 = dynamic([1,3,4,5,2]);\r\nlet array2 = dynamic([\"a\",\"b\",\"c\",\"d\",\"e\"]);\r\nprint array_sort_desc(array1,array2)\r\n```\r\n\r\n|`array1_sorted`|`array2_sorted`|\r\n|---|---|\r\n|[5,4,3,2,1]|[\"d\",\"c\",\"b\",\"e\",\"a\"]|\r\n\r\n> [!Note]\r\n> The output column names are generated automatically, based on the arguments to the function. To assign different names to the output columns, use the following syntax: `... | extend (out1, out2) = array_sort_desc(array1,array2)`\r\n\r\n## Example 2 - Sorting substrings\r\n\r\n```\r\nlet Names = \"John,Paul,George,Ringo\";\r\nlet SortedNames = strcat_array(array_sort_desc(split(Names, \",\")), \",\");\r\nprint result = SortedNames\r\n```\r\n\r\n|`result`|\r\n|---|\r\n|Ringo,Paul,John,George|\r\n\r\n## Example 3 - Combining summarize and array_sort_desc\r\n\r\n```\r\ndatatable(command:string, command_time:datetime, user_id:string)\r\n[\r\n    'chmod',   datetime(2019-07-15),   \"user1\",\r\n    'ls',      datetime(2019-07-02),   \"user1\",\r\n    'dir',     datetime(2019-07-22),   \"user1\",\r\n    'mkdir',   datetime(2019-07-14),   \"user1\",\r\n    'rm',      datetime(2019-07-27),   \"user1\",\r\n    'pwd',     datetime(2019-07-25),   \"user1\",\r\n    'rm',      datetime(2019-07-23),   \"user2\",\r\n    'pwd',     datetime(2019-07-25),   \"user2\",\r\n]\r\n| summarize timestamps = make_list(command_time), commands = make_list(command) by user_id\r\n| project user_id, commands_in_chronological_order = array_sort_desc(timestamps, commands)[1]\r\n```\r\n\r\n|`user_id`|`commands_in_chronological_order`|\r\n|---|---|\r\n|user1|[<br>  \"rm\",<br>  \"pwd\",<br>  \"dir\",<br>  \"chmod\",<br>  \"mkdir\",<br>  \"ls\"<br>]|\r\n|user2|[<br>  \"pwd\",<br>  \"rm\"<br>]|\r\n\r\n> [!Note]\r\n> If your data may contain `null` values, use [make_list_with_nulls](make-list-with-nulls-aggfunction.md) instead of [make_list](makelist-aggfunction.md).\r\n\r\n## Example 4 - Controlling location of `null` values\r\n\r\nBy default, `null` values are put last in the sorted array. However, you can control it explicitly by adding a `bool` value as the last argument to `array_sort_desc()`.\r\n\r\nExample with default behavior:\r\n\r\n```\r\nprint array_sort_desc(dynamic([null,\"blue\",\"yellow\",\"green\",null]))\r\n```\r\n\r\n|`print_0`|\r\n|---|\r\n|[\"yellow\",\"green\",\"blue\",null,null]|\r\n\r\nExample with non-default behavior:\r\n\r\n```\r\nprint array_sort_desc(dynamic([null,\"blue\",\"yellow\",\"green\",null]), false)\r\n```\r\n\r\n|`print_0`|\r\n|---|\r\n|[null,null,\"yellow\",\"green\",\"blue\"]|\r\n\r\n## See also\r\n\r\nTo sort the first array in ascending order, use [array_sort_asc()](arraysortascfunction.md).", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/arraysortdescfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_split", "Splits an array to multiple arrays according to the split indices and packs the generated array in a dynamic array.", "## Syntax\r\n\r\n`array_split`(*array*, *`indices`*)\r\n\r\n## Arguments\r\n\r\n* *array*: Input array to split, must be dynamic array.\r\n* *indices*: Integer or dynamic array of integers with the split indices (zero based), negative values are converted to array_length + value.\r\n\r\n## Returns\r\n\r\nDynamic array containing N+1 arrays with the values in the range `[0..i1), [i1..i2), ... [iN..array_length)` from `array`, where N is the number of input indices and `i1...iN` are the indices.", "```\r\nprint arr=dynamic([1,2,3,4,5]) \r\n| extend arr_split=array_split(arr, 2)\r\n```\r\n\r\n|`arr`|`arr_split`|\r\n|---|---|\r\n|[1,2,3,4,5]|[[1,2],[3,4,5]]|\r\n\r\n```\r\nprint arr=dynamic([1,2,3,4,5]) \r\n| extend arr_split=array_split(arr, dynamic([1,3]))\r\n```\r\n\r\n|`arr`|`arr_split`|\r\n|---|---|\r\n|[1,2,3,4,5]|[[1],[2,3],[4,5]]|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/arraysplitfunction", "arraysplitfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "array_sum", "Calculates the sum of elements in a dynamic array.", "## Syntax\r\n\r\n`array_sum`(*array*)\r\n\r\n## Arguments\r\n\r\n* *array*: Input array.\r\n\r\n## Returns\r\n\r\nDouble type value with the sum of the elements of the array.\r\n\r\n> [!NOTE]\r\n> If the array contains elements of non-numeric types, the result is `null`.", "```\r\nprint arr=dynamic([1,2,3,4]) \r\n| extend arr_sum=array_sum(arr)\r\n```\r\n\r\n|`arr`|`arr_sum`|\r\n|---|---|\r\n|[1,2,3,4]|10|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/array-sum-function", "array-sum-function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "as", "Binds a name to the operator's input tabular expression, thus allowing the query to reference the value of the tabular expression multiple times without breaking the query and binding a name through the [let statement](letstatement.md).", "## Syntax\r\n\r\n*T* `|` `as` [`hint.materialized` `=` `true`] *Name*\r\n\r\n## Arguments\r\n\r\n* *T*: A tabular expression.\r\n* *Name*: A temporary name for the tabular expression.\r\n* `hint.materialized`: If set to `true`, the value of the tabular expression will be\r\n  materialized as if it was wrapped by a [materialize()](./materializefunction.md) function\r\n  call.\r\n\r\n> [!NOTE]\r\n>\r\n> * The name given by `as` will be used in the `withsource=` column of [union](./unionoperator.md), the `source_` column of [find](./findoperator.md), and the `$table` column of [search](./searchoperator.md).\r\n> * The tabular expression named using the operator in a [join](./joinoperator.md)'s outer tabular input (`$left`) can also be used in the join's tabular inner input (`$right`).", "```\r\n// 1. In the following 2 example the union's generated TableName column will consist of 'T1' and 'T2'\r\nrange x from 1 to 10 step 1 \r\n| as T1 \r\n| union withsource=TableName (range x from 1 to 10 step 1 | as T2)\r\n\r\n// Alternatively, you can write the same example, as follows:\r\nunion withsource=TableName (range x from 1 to 10 step 1 | as T1), (range x from 1 to 10 step 1 | as T2)\r\n\r\n// 2. In the following example, the 'left side' of the join will be: \r\n//      MyLogTable filtered by type == \"Event\" and Name == \"Start\"\r\n//    and the 'right side' of the join will be: \r\n//      MyLogTable filtered by type == \"Event\" and Name == \"Stop\"\r\nMyLogTable  \r\n| where type == \"Event\"\r\n| as T\r\n| where Name == \"Start\"\r\n| join (\r\n    T\r\n    | where Name == \"Stop\"\r\n) on ActivityId\r\n```", "https://docs.microsoft.com/azure/data-explorer/kusto/query/asoperator", "asoperator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "asin", "Returns the angle whose sine is the specified number (the inverse operation of [`sin()`](sinfunction.md)) .", "## Syntax\r\n\r\n`asin(`*x*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: A real number in range [-1, 1].\r\n\r\n## Returns\r\n\r\n* The value of the arc sine of `x`\r\n* `null` if `x` < -1 or `x` > 1", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/asinfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "assert", "Checks for a condition. If the condition is false, outputs error messages and fails the query.", "## Syntax\r\n\r\n`assert(`*condition*`, `*message*`)`\r\n\r\n## Arguments\r\n\r\n* *condition*: The conditional expression to evaluate. If the condition is `false`, the specified message is used to report an error. If the condition is `true`, it returns `true` as an evaluation result. Condition must be evaluated to constant during the query analysis phase.\r\n* *message*: The message used if assertion is evaluated to `false`. The *message* must be a string literal.\r\n\r\n> [!NOTE]\r\n> `condition` must be evaluated to constant during the query analysis phase. In other words, it can be constructed from other expressions referencing constants, and can't be bound to row-context.\r\n\r\n## Returns\r\n\r\n* `true` - if the condition is `true`\r\n* Raises semantic error if the condition is evaluated to `false`.", "The following query defines a function `checkLength()` that checks input string length, and uses `assert` to validate input length parameter (checks that it is greater than zero).\r\n\r\n```\r\nlet checkLength = (len:long, s:string)\r\n{\r\n    assert(len > 0, \"Length must be greater than zero\") and \r\n    strlen(s) > len\r\n};\r\ndatatable(input:string)\r\n[\r\n    '123',\r\n    '4567'\r\n]\r\n| where checkLength(len=long(-1), input)\r\n```\r\n\r\nRunning this query yields an error:  \r\n`assert() has failed with message: 'Length must be greater than zero'`\r\n\r\nExample of running with valid `len` input:\r\n\r\n```\r\nlet checkLength = (len:long, s:string)\r\n{\r\n    assert(len > 0, \"Length must be greater than zero\") and strlen(s) > len\r\n};\r\ndatatable(input:string)\r\n[\r\n    '123',\r\n    '4567'\r\n]\r\n| where checkLength(len=3, input)\r\n```\r\n\r\n|input|\r\n|---|\r\n|4567|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/assert-function", "assert-function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "atan", "Returns the angle whose tangent is the specified number (the inverse operation of [`tan()`](tanfunction.md)) .", "## Syntax\r\n\r\n`atan(`*x*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: A real number.\r\n\r\n## Returns\r\n\r\n* The value of the arc tangent of `x`", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/atanfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "atan2", "Calculates the angle, in radians, between the positive x-axis and the ray from the origin to the point (y, x).", "## Syntax\r\n\r\n`atan2(`*y*`,`*x*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: X coordinate (a real number).\r\n* *y*: Y coordinate (a real number).\r\n\r\n## Returns\r\n\r\n* The angle, in radians, between the positive x-axis and the ray from the origin to the point (y, x).", "```\r\nprint atan2_0 = atan2(1,1) // Pi / 4 radians (45 degrees)\r\n| extend atan2_1 = atan2(0,-1) // Pi radians (180 degrees)\r\n| extend atan2_2 = atan2(-1,0) // - Pi / 2 radians (-90 degrees)\r\n```\r\n\r\n|atan2_0|atan2_1|atan2_2|\r\n|---|---|---|\r\n|0.785398163397448|3.14159265358979|-1.5707963267949|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/atan2function", "atan2function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "autocluster", "`autocluster` finds common patterns of discrete attributes (dimensions) in the data. It then reduces the results of the original query, whether it's 100 or 100k rows, to a small number of patterns. The plugin was developed to help analyze failures (such as exceptions or crashes) but can potentially work on any filtered data set. The plugin is invoked with the [`evaluate`](evaluateoperator.md) operator.", "> [!NOTE]\r\n> `autocluster` is largely based on the Seed-Expand algorithm from the following paper: [Algorithms for Telemetry Data Mining using Discrete Attributes](https://www.scitepress.org/DigitalLibrary/PublicationsDetail.aspx?ID=d5kcrO+cpEU=&t=1). \r\n\r\n## Syntax\r\n\r\n*Table* | `evaluate` `autocluster` `(`[*SizeWeight*, *WeightColumn*, *NumSeeds*, *CustomWildcard*, *CustomWildcard*, ...]`)`\r\n\r\n## Arguments \r\n\r\n> [!NOTE] \r\n> All arguments are optional.\r\n\r\nAll arguments are optional, but they must be ordered as above. To indicate that the default value should be used, put the string tilde value '~' (see the \"Example\" column in the table).\r\n\r\n|Argument  | Type, range, default  |Description | Example   |\r\n|----------------|-----------------------------------|---------------------------|------------------------------------------------|\r\n| SizeWeight     | 0 < *double* < 1 [default: 0.5]   | Gives you some control over the balance between generic (high coverage) and informative (many shared) values. If you increase the value, it usually reduces the number of patterns, and each pattern tends to cover a larger percentage coverage. If you decrease the value, it usually produces more specific patterns with more shared values, and a smaller percentage coverage. The under-the-hood formula is a weighted geometric mean, between the normalized generic score and the informative score with weights `SizeWeight` and `1-SizeWeight`    | `T | evaluate autocluster(0.8)`                |\r\n|WeightColumn    | *column_name*     | Considers each row in the input according to the specified weight (by default each row has a weight of '1'). The argument must be a name of a numeric column (such as int, long, real). A common usage of a weight column is to take into account sampling or bucketing/aggregation of the data that is already embedded into each row.   | `T | evaluate autocluster('~', sample_Count)` | \r\n| NumSeeds        | *int* [default: 25]  | The number of seeds determines the number of initial local search points of the algorithm. In some cases, depending on the structure of the data and if you increase the number of seeds, then the number (or quality) of the results increases through the expanded search space with a slower query tradeoff. The value has diminishing results in both directions, so if you decrease it to below five, it will achieve negligible performance improvements. If you increase to above 50, it will rarely generate additional patterns.  | `T | evaluate autocluster('~', '~', 15)`       |\r\n| CustomWildcard  | *\"any_value_per_type\"* | Sets the wildcard value for a specific type in the results table. It will indicate that the current pattern doesn't have a restriction on this column. The default is null, since the string default is an empty string. If the default is a good value in the data, a different wildcard value should be used (such as `*`).   | `T | evaluate autocluster('~', '~', '~', '*', int(-1), double(-1), long(0), datetime(1900-1-1))` |\r\n\r\n## Returns\r\n\r\nThe `autocluster` plugin returns a (usually small) set of patterns. The patterns capture portions of the data with shared common values across multiple discrete attributes. Each pattern in the results is represented by a row.\r\n\r\nThe first column is the segment ID. The next two columns are the count and percentage of rows out of the original query that are captured by the pattern. The remaining columns are from the original query. Their value is either a specific value from the column, or a wildcard value (which are by default null) meaning variable values.\r\n\r\nThe patterns aren't distinct, may be overlapping, and usually don't cover all the original rows. Some rows may not fall under any pattern.\r\n\r\n> [!TIP]\r\n> Use [where](./whereoperator.md) and [project](./projectoperator.md) in the input pipe to reduce the data to just what you're interested in.\r\n>\r\n> When you find an interesting row, you might want to drill into it further by adding its specific values to your `where` filter.", "### Using evaluate\r\n\r\n```\r\nT | evaluate autocluster()\r\n```\r\n\r\n### Using autocluster\r\n\r\n```\r\nStormEvents \r\n| where monthofyear(StartTime) == 5\r\n| extend Damage = iff(DamageCrops + DamageProperty > 0 , \"YES\" , \"NO\")\r\n| project State , EventType , Damage\r\n| evaluate autocluster(0.6)\r\n```\r\n\r\n|SegmentId|Count|Percent|State|EventType|Damage|\r\n|---|---|---|---|---|---|---|---|---|\r\n|0|2278|38.7||Hail|NO\r\n|1|512|8.7||Thunderstorm Wind|YES\r\n|2|898|15.3|TEXAS||\r\n\r\n### Using custom wildcards\r\n\r\n```\r\nStormEvents \r\n| where monthofyear(StartTime) == 5\r\n| extend Damage = iff(DamageCrops + DamageProperty > 0 , \"YES\" , \"NO\")\r\n| project State , EventType , Damage \r\n| evaluate autocluster(0.2, '~', '~', '*')\r\n```\r\n\r\n|SegmentId|Count|Percent|State|EventType|Damage|\r\n|---|---|---|---|---|---|---|---|---|\r\n|0|2278|38.7|\\*|Hail|NO\r\n|1|512|8.7|\\*|Thunderstorm Wind|YES\r\n|2|898|15.3|TEXAS|\\*|\\*\r\n\r\n## See also\r\n\r\n* [basket](./basketplugin.md)\r\n* [reduce](./reduceoperator.md)\r\n\r\n\r\n\r\n---\r\n\r\n> [!IMPORTANT]\r\n> The content below is Microsoft confidential(not available in the [public site](https://docs.microsoft.com/en-us/azure/data-explorer/))\r\n\r\n# autocluster plugin - Internal\r\n\r\nIf you have questions about autocluster or other ML related workflows in Kusto you can post them on the following DL: [KustoML](mailto:kustoML@microsoft.com).", "https://docs.microsoft.com/azure/data-explorer/kusto/query/autoclusterplugin", "autoclusterplugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "avg", "Calculates the average (arithmetic mean) of *Expr* across the group.", "* Can only be used in context of aggregation inside [summarize](summarizeoperator.md)\r\n\r\n## Syntax\r\n\r\n`avg` `(`*Expr*`)`\r\n\r\n## Arguments\r\n\r\n* *Expr*: Expression that will be used for aggregation calculation. Records with `null` values are ignored and not included in the calculation.\r\n\r\n## Returns\r\n\r\nThe average value of *Expr* across the group.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/avg-aggfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "avgif", "Calculates the [average](avg-aggfunction.md) of *Expr* across the group for which *Predicate* evaluates to `true`.", "* Can only be used in context of aggregation inside [summarize](summarizeoperator.md)\r\n\r\n## Syntax\r\n\r\n`avgif` `(`*Expr*`,` *Predicate*`)`\r\n\r\n## Arguments\r\n\r\n* *Expr*: Expression that will be used for aggregation calculation. Records with `null` values are ignored and not included in the calculation.\r\n* *Predicate*:  predicate that if true, the *Expr* calculated value will be added to the average.\r\n\r\n## Returns\r\n\r\nThe average value of *Expr* across the group where *Predicate* evaluates to `true`.", "```\r\nrange x from 1 to 100 step 1\r\n| summarize avgif(x, x%2 == 0)\r\n```\r\n\r\n|avgif_x|\r\n|---|\r\n|51|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/avgif-aggfunction", "avgif-aggfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "azure_digital_twins_query_request", "The `azure_digital_twins_query_request` plugin runs an Azure Digital Twins query as part of a Kusto Query Language query. The plugin is invoked with the [`evaluate`](evaluateoperator.md) operator.", "Using the plugin, you can query across data in both Azure Digital Twins and any data source accessible through the Kusto Query Language. For example, you can use the plugin to contextualize time series data in a Kusto query by joining it with knowledge graph data held in Azure Digital Twins.\r\n\r\n## Syntax\r\n\r\n`evaluate` `azure_digital_twins_query_request` `(` *AdtInstanceEndpoint* `,` *AdtQuery* `)`\r\n\r\n## Arguments\r\n\r\n* *AdtInstanceEndpoint*: A `string` literal indicating the Azure Digital Twins instance endpoint to be queried.\r\n* *AdtQuery*: A `string` literal indicating the query that is to be run against the Azure Digital Twins endpoint. This query is written in a custom SQL-like query language for Azure Digital Twins, referred to as the **Azure Digital Twins query language**. For more information, see [**Query language for Azure Digital Twins**](https://docs.microsoft.com/azure/digital-twins/concepts-query-language).\r\n\r\n## Authentication and authorization\r\n\r\nThe azure_digital_twins_query_request plugin uses the Azure AD account of the user running the query to authenticate. To run a query, a user must at least be granted the **Azure Digital Twins Data Reader** role. Information on how to assign this role can be found in [**Security for Azure Digital Twins solutions**](https://docs.microsoft.com/azure/digital-twins/concepts-security#authorization-azure-roles-for-azure-digital-twins).", "The following examples show how you can run various Azure Digital Twins queries, including queries that use additional Kusto expressions.\r\n\r\n### Retrieval of all twins within an Azure Digital Twins instance\r\n\r\nThe following example returns all digital twins within an Azure Digital Twins instance.\r\n\r\n```\r\nevaluate azure_digital_twins_query_request(\r\n  'https://contoso.api.wcus.digitaltwins.azure.net',\r\n  'SELECT T AS Twins FROM DIGITALTWINS T')\r\n```\r\n\r\n**Output**\r\n\r\n![Screenshot of the twins present in the Azure Digital Twins instance.](images/azure-digital-twins-query-request-plugin/adt-twins.png)\r\n\r\n### Projection of twin properties as columns along with additional Kusto expressions\r\n\r\nThe following example returns the result from the plugin as separate columns, and then performs additional operations using Kusto expressions.\r\n\r\n```\r\nevaluate azure_digital_twins_query_request(\r\n  'https://contoso.api.wcus.digitaltwins.azure.net',\r\n  'SELECT T.Temperature, T.Humidity FROM DIGITALTWINS T WHERE IS_PRIMITIVE(T.Temperature) AND IS_PRIMITIVE(T.Humidity)')\r\n| where Temperature > 20\r\n| project TemperatureInC = Temperature, Humidity\r\n```\r\n\r\n**Output**\r\n\r\n|TemperatureInC|Humidity|\r\n|---|---|\r\n|21|48|\r\n|49|34|\r\n|80|32|\r\n\r\n### Joining the plugin results with another data source\r\n\r\nThe following example shows how to perform complex analysis, such as anomaly detection, through a `join` operation between the plugin results and a table containing historical data in a Kusto table, based on the ID column (`$dtid`).\r\n\r\n```\r\nevaluate azure_digital_twins_query_request(\r\n  'https://contoso.api.wcus.digitaltwins.azure.net',\r\n  'SELECT T.$dtId AS tid, T.Temperature FROM DIGITALTWINS T WHERE IS_PRIMITIVE(T.$dtId) AND IS_PRIMITIVE(T.Temperature)')\r\n| project tostring(tid), todouble(Temperature)\r\n| join kind=inner (\r\n    ADT_Data_History\r\n) on $left.tid == $right.twinId\r\n| make-series num=avg(value) on timestamp from min_t to max_t step dt by tid\r\n| extend (anomalies, score , baseline) = \r\n          series_decompose_anomalies(num, 1.5, -1, 'linefit')\r\n| render anomalychart with(anomalycolumns=anomalies, title= 'Test, anomalies')\r\n```\r\n\r\nADT_Data_History is a table whose schema as follows:\r\n\r\n|timestamp|twinId|modelId|name|value|relationshipTarget|relationshipId|\r\n|---|---|---|---|---|---|---|\r\n|2021-02-01 17:24|contosoRoom|dtmi:com:contoso:Room;1|Temperature|24|...|..|\r\n\r\n**Output**\r\n\r\n![Screenshot of the Anomaly chart of the test expression. Highlighted point is the anomaly.](images/azure-digital-twins-query-request-plugin/adt-anomaly.png)\r\n\r\n---\r\n\r\n> [!IMPORTANT]\r\n> The content below is Microsoft confidential(not available in the [public site](https://docs.microsoft.com/en-us/azure/data-explorer/))\r\n\r\n# azure_digital_twins_query_request plugin (preview)\r\n\r\nRuns an Azure Digital Twins query as part of the KQL query.\r\n\r\nUsing the plugin, you can reason across data in both Azure Digital Twins and any data source accessible through the Kusto Query Language (KQL). For example, you can use the plugin to contextualize time series data in Kusto by joining it with knowledge graph data held in Azure Digital Twins.\r\n\r\n>[!NOTE]\r\n>* The Azure Digital Twins query plugin is in preview.\r\n\r\n## Syntax\r\n\r\n`evaluate` `azure_digital_twins_query_request` `(` *AdtInstanceEndpoint* `,` *AdtQuery* `)`\r\n\r\n## Arguments\r\n\r\n* *AdtInstanceEndpoint*: A `string` literal indicating the Azure Digital Twins instance endpoint to be queried.\r\n\r\n* *AdtQuery*: A `string` literal indicating the query that is to be executed against the Azure Digital Twins endpoint. This query is written in a custom SQL-like query language for Azure Digital Twins, referred to as the **Azure Digital Twins query language**. For more information on the query language, see [**Query language | Azure Digital Twins documentation**](https://docs.microsoft.com/azure/digital-twins/concepts-query-language).\r\n\r\n\r\n## Authentication and Authorization\r\n\r\nThe user of the azure_digital_twins_query_request plugin must be granted the **Azure Digital Twins Data Reader** role, as the user's Azure AD token is used to authenticate. Information on how to assign this role can be found in [**Security for Azure Digital Twins solutions | Azure Digital Twins documentation**](https://docs.microsoft.com/azure/digital-twins/concepts-security#authorization-azure-roles-for-azure-digital-twins).\r\n\r\n## Setup\r\n\r\nThis plugin is disabled by default, as it is in still in preview. To enable the plugin on your cluster, run the following command:\r\n`.enable plugin azure_digital_twins_query_request`. This command requires **All Databases admin** permission. \r\n\r\nFor more information on this command, see [**.enable plugin | Azure Data Explorer documentation**](../management/enable-plugin.md). For more information on role-based authorization in Kusto, see [**Role-based Authorization in Kusto | Azure Data Explorer documentation**](https://docs.microsoft.com/azure/data-explorer/kusto/management/access-control/role-based-authorization).\r\n\r\n## Examples\r\n\r\nThe following examples show how you can perform various queries, along with using additional Kusto expressions.\r\n\r\n### Retrieval of all twins within an Azure Digital Twins instance\r\n\r\nThis example returns all digital twins within an Azure Digital Twins instance.\r\n\r\n```kusto\r\nevaluate azure_digital_twins_query_request(\r\n  'https://contoso.api.wcus.digitaltwins.azure.net',\r\n  'SELECT T AS Twins FROM DIGITALTWINS T')\r\n```\r\n\r\nThe result looks like this:\r\n\r\n![The twins present in the Azure Digital Twins instance](images/adt-twins.png \"The twins present in the Azure Digital Twins instance\")\r\n\r\n### Projection of twin properties as columns along with additional Kusto expressions\r\n\r\nThis example returns the result from the plugin as separate columns, and then performs additional operations using Kusto expressions.\r\n\r\n```kusto\r\nevaluate azure_digital_twins_query_request(\r\n  'https://contoso.api.wcus.digitaltwins.azure.net',\r\n  'SELECT T.Temperature, T.Humidity FROM DIGITALTWINS T WHERE IS_PRIMITIVE(T.Temperature) AND IS_PRIMITIVE(T.Humidity)')\r\n| where Temperature > 20\r\n| project TemperatureInC = Temperature, Humidity\r\n```\r\n\r\nThe result looks like this:\r\n\r\n|TemperatureInC|Humidity|\r\n|---|---|\r\n|21|48|\r\n|49|34|\r\n|80|32|\r\n\r\n### Joining the plugin results with another data source\r\n\r\nThis example shows how to perform complex analysis, such as anomaly detection, through a `join` operation between the plugin results and a table containing historical data in a Kusto table, based on the ID column (`$dtid`).\r\n\r\n```kusto\r\nevaluate azure_digital_twins_query_request(\r\n  'https://contoso.api.wcus.digitaltwins.azure.net',\r\n  'SELECT T.$dtId AS tid, T.Temperature FROM DIGITALTWINS T WHERE IS_PRIMITIVE(T.$dtId) AND IS_PRIMITIVE(T.Temperature)')\r\n| project tostring(tid), todouble(Temperature)\r\n| join kind=inner (\r\n    ADT_Data_History\r\n) on $left.tid == $right.twinId\r\n| make-series num=avg(value) on timestamp from min_t to max_t step dt by tid\r\n| extend (anomalies, score , baseline) = \r\n          series_decompose_anomalies(num, 1.5, -1, 'linefit')\r\n| render anomalychart with(anomalycolumns=anomalies, title= 'Test, anomalies')\r\n```\r\n\r\nADT_Data_History is a table whose schema as follows:\r\n\r\n|timestamp|twinId|modelId|name|value|relationshipTarget|relationshipId|\r\n|---|---|---|---|---|---|---|\r\n|2021-02-01 17:24|contosoRoom|dtmi:com:contoso:Room;1|Temperature|24|...|..|\r\n\r\nThe output looks like this:\r\n\r\n![Anomaly chart of the above expression](images/adt-anomaly.png \"Highlighted point is the anomaly\")", "https://docs.microsoft.com/azure/data-explorer/kusto/query/azure-digital-twins-query-request-plugin", "azure-digital-twins-query-request-plugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "bag_has_key", "Checks whether a dynamic bag column contains a given key.", "## Syntax\r\n\r\n`bag_has_key(`*bag*`, `*key*`)`\r\n\r\n## Arguments\r\n\r\n| Name | Type | Required | Description |\r\n| -- | -- | -- | -- |\r\n| *bag* | dynamic | &check; | The property bag to search. |\r\n| *key* | string | &check; | The key to search for.  You can search for a nested key using the [JSONPath](jsonpath.md) notation. For an example, see [Search using a JSONPath key](#search-using-a-jsonpath-key). |\r\n\r\n## Returns\r\n\r\nTrue or false depending on if the key exists in the bag.", "```\r\ndatatable(input: dynamic)\r\n[\r\n    dynamic({'key1' : 123, 'key2': 'abc'}),\r\n    dynamic({'key1' : 123, 'key3': 'abc'}),\r\n]\r\n| extend result = bag_has_key(input, 'key2')\r\n```\r\n\r\n|input|result|\r\n|---|---|\r\n|{<br>  \"key1\": 123,<br>  \"key2\": \"abc\"<br>}|true<br>|\r\n|{<br>  \"key1\": 123,<br>  \"key3\": \"abc\"<br>}|false<br>|\r\n\r\n### Search using a JSONPath key\r\n\r\n```\r\ndatatable(input: dynamic)\r\n[\r\n    dynamic({'key1': 123, 'key2': {'prop1' : 'abc', 'prop2': 'xyz'}, 'key3': [100, 200]}),\r\n]\r\n| extend result = bag_has_key(input, '$.key2.prop1')\r\n```\r\n\r\n|input|result|\r\n|---|---|\r\n|{<br>  \"key1\": 123,<br>  \"key2\": {<br>    \"prop1\": \"abc\",<br>    \"prop2\": \"xyz\"<br>  },<br>  \"key3\": [<br>    100,<br>    200<br>  ]<br>}|true<br>|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/bag-has-key-function", "bag-has-key-function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "bag_keys", "Enumerates all the root keys in a dynamic property-bag object.", "## Syntax\r\n\r\n`bag_keys(`*dynamic object*`)`\r\n\r\n## Returns\r\n\r\nAn array of keys, order is undetermined.", "```\r\ndatatable(index:long, d:dynamic) [\r\n1, dynamic({'a':'b', 'c':123}), \r\n2, dynamic({'a':'b', 'c':{'d':123}}),\r\n3, dynamic({'a':'b', 'c':[{'d':123}]}),\r\n4, dynamic(null),\r\n5, dynamic({}),\r\n6, dynamic('a'),\r\n7, dynamic([])]\r\n| extend keys = bag_keys(d)\r\n```\r\n\r\n|index|d|keys|\r\n|---|---|---|\r\n|1|{<br>  \"a\": \"b\",<br>  \"c\": 123<br>}|[<br>  \"a\",<br>  \"c\"<br>]|\r\n|2|{<br>  \"a\": \"b\",<br>  \"c\": {<br>    \"d\": 123<br>  }<br>}|[<br>  \"a\",<br>  \"c\"<br>]|\r\n|3|{<br>  \"a\": \"b\",<br>  \"c\": [<br>    {<br>      \"d\": 123<br>    }<br>  ]<br>}|[<br>  \"a\",<br>  \"c\"<br>]|\r\n|4|||\r\n|5|{}|[]|\r\n|6|a||\r\n|7|[]||", "https://docs.microsoft.com/azure/data-explorer/kusto/query/bagkeysfunction", "bagkeysfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "bag_merge", "Merges `dynamic` property-bags into a `dynamic` property-bag with all properties merged.", "## Syntax\r\n\r\n`bag_merge(`*bag1*`, `*bag2*`[`,` *bag3*, ...])`\r\n\r\n## Arguments\r\n\r\n* *bag1...bagN*: Input `dynamic` property-bags. The function accepts between 2 to 64 arguments.\r\n\r\n## Returns\r\n\r\nReturns a `dynamic` property-bag. Results from merging all of the input property-bag objects. If a key appears in more than one input object, an arbitrary value (out of the possible values for this key) will be chosen.", "Expression:\r\n\r\n```\r\nprint result = bag_merge(\r\n   dynamic({'A1':12, 'B1':2, 'C1':3}),\r\n   dynamic({'A2':81, 'B2':82, 'A1':1}))\r\n```\r\n\r\n|result|\r\n|---|\r\n|{<br>  \"A1\": 12,<br>  \"B1\": 2,<br>  \"C1\": 3,<br>  \"A2\": 81,<br>  \"B2\": 82<br>}|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/bag-merge-function", "bag-merge-function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "bag_pack", "Creates a `dynamic` object (property bag) from a list of names and values.", "Alias to `pack_dictionary()` function.\r\n\r\n> [!NOTE]\r\n> The `bag_pack()` and `pack()` functions are interpreted equivalently.\r\n\r\n## Syntax\r\n\r\n`bag_pack(`*key1*`,` *value1*`,` *key2*`,` *value2*`,... )`\r\n\r\n`pack(`*key1*`,` *value1*`,` *key2*`,` *value2*`,... )`\r\n\r\n## Arguments\r\n\r\n* An alternating list of keys and values (the total length of the list must be even)\r\n* All keys must be non-empty constant strings", "The following example returns `{\"Level\":\"Information\",\"ProcessID\":1234,\"Data\":{\"url\":\"www.bing.com\"}}`:\r\n\r\n```\r\nbag_pack(\"Level\", \"Information\", \"ProcessID\", 1234, \"Data\", bag_pack(\"url\", \"www.bing.com\"))\r\n```\r\n\r\nLets take 2 tables, SmsMessages and MmsMessages:\r\n\r\nTable SmsMessages \r\n\r\n|SourceNumber |TargetNumber| CharsCount\r\n|---|---|---\r\n|555-555-1234 |555-555-1212 | 46 \r\n|555-555-1234 |555-555-1213 | 50 \r\n|555-555-1212 |555-555-1234 | 32 \r\n\r\nTable MmsMessages \r\n\r\n|SourceNumber |TargetNumber| AttachmentSize | AttachmentType | AttachmentName\r\n|---|---|---|---|---\r\n|555-555-1212 |555-555-1213 | 200 | jpeg | Pic1\r\n|555-555-1234 |555-555-1212 | 250 | jpeg | Pic2\r\n|555-555-1234 |555-555-1213 | 300 | png | Pic3\r\n\r\nThe following query:\r\n\r\n```\r\nSmsMessages \r\n| extend Packed=bag_pack(\"CharsCount\", CharsCount) \r\n| union withsource=TableName kind=inner \r\n( MmsMessages \r\n  | extend Packed=bag_pack(\"AttachmentSize\", AttachmentSize, \"AttachmentType\", AttachmentType, \"AttachmentName\", AttachmentName))\r\n| where SourceNumber == \"555-555-1234\"\r\n``` \r\n\r\nReturns:\r\n\r\n|TableName |SourceNumber |TargetNumber | Packed\r\n|---|---|---|---\r\n|SmsMessages|555-555-1234 |555-555-1212 | {\"CharsCount\": 46}\r\n|SmsMessages|555-555-1234 |555-555-1213 | {\"CharsCount\": 50}\r\n|MmsMessages|555-555-1234 |555-555-1212 | {\"AttachmentSize\": 250, \"AttachmentType\": \"jpeg\", \"AttachmentName\": \"Pic2\"}\r\n|MmsMessages|555-555-1234 |555-555-1213 | {\"AttachmentSize\": 300, \"AttachmentType\": \"png\", \"AttachmentName\": \"Pic3\"}", "https://docs.microsoft.com/azure/data-explorer/kusto/query/packfunction", "packfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "bag_remove_keys", "Removes keys and associated values from a `dynamic` property-bag.", "## Syntax\r\n\r\n`bag_remove_keys(`*bag*`, `*keys*`)`\r\n\r\n## Arguments\r\n\r\n* *bag*: `dynamic` property-bag input.\r\n* *keys*: `dynamic` array includes keys to be removed from the input. Keys refer to the first level of the property bag.\r\nYou can specify keys on the nested levels using [JSONPath](jsonpath.md) notation.\r\n\r\n## Returns\r\n\r\nReturns a `dynamic` property-bag without specified keys and their values.", "```\r\ndatatable(input:dynamic)\r\n[\r\n    dynamic({'key1' : 123,     'key2': 'abc'}),\r\n    dynamic({'key1' : 'value', 'key3': 42.0}),\r\n]\r\n| extend result=bag_remove_keys(input, dynamic(['key2', 'key4']))\r\n```\r\n\r\n|input|result|\r\n|---|---|\r\n|{<br>  \"key1\": 123,<br>  \"key2\": \"abc\"<br>}|{<br>  \"key1\": 123<br>}|\r\n|{<br>  \"key1\": \"value\",<br>  \"key3\": 42.0<br>}|{<br>  \"key1\": \"value\",<br>  \"key3\": 42.0<br>}|\r\n\r\n### Remove inner properties of dynamic values using JSONPath notation\r\n\r\n```\r\ndatatable(input:dynamic)\r\n[\r\n    dynamic({'key1': 123, 'key2': {'prop1' : 'abc', 'prop2': 'xyz'}, 'key3': [100, 200]}),\r\n]\r\n| extend result=bag_remove_keys(input, dynamic(['$.key2.prop1', 'key3']))\r\n```\r\n\r\n|input|result|\r\n|---|---|\r\n|{<br>  \"key1\": 123,<br>  \"key2\": {<br>    \"prop1\": \"abc\",<br>    \"prop2\": \"xyz\"<br>  },<br>  \"key3\": [<br>    100,<br>    200<br>  ]<br>}|{<br>  \"key1\": 123,<br>  \"key2\": {<br>    \"prop2\": \"xyz\"<br>  }<br>}|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/bag-remove-keys-function", "bag-remove-keys-function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "bag_unpack", "The `bag_unpack` plugin unpacks a single column of type `dynamic`, by treating each property bag top-level slot as a column. The plugin is invoked with the [`evaluate`](evaluateoperator.md) operator.", "## Syntax\r\n\r\n*T* `|` `evaluate` `bag_unpack(` *Column* [`,` *OutputColumnPrefix* ] [`,` *columnsConflict* ] [`,` *ignoredProperties* ] `)`\r\n\r\n## Arguments\r\n\r\n* *T*: The tabular input whose column *Column* is to be unpacked.\r\n* *Column*: The column of *T* to unpack. Must be of type `dynamic`.\r\n* *OutputColumnPrefix*: A common prefix to add to all columns produced by the plugin. This argument is optional.\r\n* *columnsConflict*: A direction for column conflict resolution. This argument is optional. When argument is provided, it's expected to be a string literal matching one of the following values:\r\n    - `error` - Query produces an error (default)\r\n    - `replace_source` - Source column is replaced\r\n    - `keep_source` - Source column is kept\r\n* *ignoredProperties*: Optional set of bag properties to be ignored. When argument is provided, it's expected to be a constant of `dynamic` array with one or more string literals.\r\n\r\n## Returns\r\n\r\nThe `bag_unpack` plugin returns a table with as many records as its tabular input (*T*). The schema of the table is the same as the schema of its tabular input with the following modifications:\r\n\r\n* The specified input column (*Column*) is removed.\r\n* The schema is extended with as many columns as there are distinct slots in\r\n  the top-level property bag values of *T*. The name of each column corresponds\r\n  to the name of each slot, optionally prefixed by *OutputColumnPrefix*. Its\r\n  type is either the type of the slot, if all values of the same slot have the\r\n  same type, or `dynamic`, if the values differ in type.\r\n\r\n> [!NOTE]\r\n> The plugin's output schema depends on the data values, making it as \"unpredictable\"\r\n> as the data itself. Multiple executions of the plugin, using different\r\n> data inputs, may produce different output schema.\r\n\r\n> [!NOTE]\r\n> The input data to the plugin must be such that the output schema follows all the rules for a tabular schema. In particular:\r\n>\r\n> * An output column name can't be the same as an existing column in the tabular\r\n    input *T*, unless it's the column to be unpacked (*Column*), since that will produce two columns with the same name.\r\n>\r\n> * All slot names, when prefixed by *OutputColumnPrefix*, must be valid\r\n    entity names and follow the [identifier naming rules](./schema-entities/entity-names.md#identifier-naming-rules).", "### Expand a bag\r\n\r\n```\r\ndatatable(d:dynamic)\r\n[\r\n    dynamic({\"Name\": \"John\", \"Age\":20}),\r\n    dynamic({\"Name\": \"Dave\", \"Age\":40}),\r\n    dynamic({\"Name\": \"Jasmine\", \"Age\":30}),\r\n]\r\n| evaluate bag_unpack(d)\r\n```\r\n\r\n**Output**\r\n\r\n|Age|Name   |\r\n|---|-------|\r\n|20 |John   |\r\n|40 |Dave   |\r\n|30 |Jasmine|\r\n\r\n### Expand a bag with OutputColumnPrefix\r\n\r\nExpand a bag and use the `OutputColumnPrefix` option to produce column names that begin with the prefix 'Property_'.\r\n\r\n```\r\ndatatable(d:dynamic)\r\n[\r\n    dynamic({\"Name\": \"John\", \"Age\":20}),\r\n    dynamic({\"Name\": \"Dave\", \"Age\":40}),\r\n    dynamic({\"Name\": \"Jasmine\", \"Age\":30}),\r\n]\r\n| evaluate bag_unpack(d, 'Property_')\r\n```\r\n\r\n**Output**\r\n\r\n|Property_Age|Property_Name|\r\n|------------|-------------|\r\n|20          |John         |\r\n|40          |Dave         |\r\n|30          |Jasmine      |\r\n\r\n### Expand a bag with columnsConflict\r\n\r\nExpand a bag and use the `columnsConflict` option to resolve conflicts between existing columns and columns produced by the `bag_unpack()` operator.\r\n\r\n```\r\ndatatable(Name:string, d:dynamic)\r\n[\r\n    'Old_name', dynamic({\"Name\": \"John\", \"Age\":20}),\r\n    'Old_name', dynamic({\"Name\": \"Dave\", \"Age\":40}),\r\n    'Old_name', dynamic({\"Name\": \"Jasmine\", \"Age\":30}),\r\n]\r\n| evaluate bag_unpack(d, columnsConflict='replace_source') // Use new name\r\n```\r\n\r\n**Output**\r\n\r\n|Age|Name   |\r\n|---|-------|\r\n|20 |John   |\r\n|40 |Dave   |\r\n|30 |Jasmine|\r\n\r\n```\r\ndatatable(Name:string, d:dynamic)\r\n[\r\n    'Old_name', dynamic({\"Name\": \"John\", \"Age\":20}),\r\n    'Old_name', dynamic({\"Name\": \"Dave\", \"Age\":40}),\r\n    'Old_name', dynamic({\"Name\": \"Jasmine\", \"Age\":30}),\r\n]\r\n| evaluate bag_unpack(d, columnsConflict='keep_source') // Keep old name\r\n```\r\n\r\n**Output**\r\n\r\n|Age|Name     |\r\n|---|---------|\r\n|20 |Old_name |\r\n|40 |Old_name |\r\n|30 |Old_name |\r\n\r\n### Expand a bag with ignoredProperties\r\n\r\nExpand a bag and use the `ignoredProperties` option to ignore certain properties in the property bag.\r\n\r\n```\r\ndatatable(d:dynamic)\r\n[\r\n    dynamic({\"Name\": \"John\", \"Age\":20, \"Address\": \"Address-1\" }),\r\n    dynamic({\"Name\": \"Dave\", \"Age\":40, \"Address\": \"Address-2\"}),\r\n    dynamic({\"Name\": \"Jasmine\", \"Age\":30, \"Address\": \"Address-3\"}),\r\n]\r\n// Ignore 'Age' and 'Address' properties\r\n| evaluate bag_unpack(d, ignoredProperties=dynamic(['Address', 'Age']))\r\n```\r\n\r\n**Output**\r\n\r\n|Name|\r\n|---|\r\n|John|\r\n|Dave|\r\n|Jasmine|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/bag-unpackplugin", "bag-unpackplugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "base64_decode_toarray", "Decodes a base64 string to an array of long values.", "## Syntax\r\n\r\n`base64_decode_toarray(`*String*`)`\r\n\r\n## Arguments\r\n\r\n* *String*: Input base64 string to be decoded from base64 to an array of long values.\r\n\r\n## Returns\r\n\r\nReturns an array of long values decoded from a base64 string.\r\n\r\n* To decode base64 strings to a UTF-8 string, see [base64_decode_tostring()](base64_decode_tostringfunction.md)\r\n* To encode strings to a base64 string, see [base64_encode_tostring()](base64_encode_tostringfunction.md)", "```\r\nprint Quine=base64_decode_toarray(\"S3VzdG8=\")  \r\n// 'K', 'u', 's', 't', 'o'\r\n```\r\n\r\n|Quine|\r\n|-----|\r\n|[75,117,115,116,111]|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/base64_decode_toarrayfunction", "base64_decode_toarrayfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "base64_decode_toguid", "Decodes a base64 string to a [GUID](./scalar-data-types/guid.md).", "## Syntax\r\n\r\n`base64_decode_toguid(`*String*`)`\r\n\r\n## Arguments\r\n\r\n* *String*: Input string to be decoded from base64 to a [GUID](./scalar-data-types/guid.md). \r\n\r\n## Returns\r\n\r\nReturns a [GUID](./scalar-data-types/guid.md) decoded from a base64 string.\r\n\r\n* To encode a [GUID](./scalar-data-types/guid.md) to a base64 string, see [base64_encode_fromguid()](base64-encode-fromguid-function.md)", "```\r\nprint Quine = base64_decode_toguid(\"JpbpECu8dUy7Pv5gbeJXAA==\")  \r\n```\r\n\r\n|Quine|\r\n|-----|\r\n|10e99626-bc2b-754c-bb3e-fe606de25700|\r\n\r\nIf you try to decode an invalid base64 string, \"null\" will be returned:\r\n\r\n```\r\nprint Empty = base64_decode_toarray(\"abcd1231\")\r\n```\r\n\r\n|Empty|\r\n|-----|\r\n||", "https://docs.microsoft.com/azure/data-explorer/kusto/query/base64-decode-toguid-function", "base64-decode-toguid-function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "base64_decode_tostring", "Decodes a base64 string to a UTF-8 string.", "## Syntax\r\n\r\n`base64_decode_tostring(`*String*`)`\r\n\r\n## Arguments\r\n\r\n* *String*: Input string to be decoded from base64 to UTF-8 string.\r\n\r\n## Returns\r\n\r\nReturns UTF-8 string decoded from base64 string.\r\n\r\n* To decode base64 strings to an array of long values, see [base64_decode_toarray()](base64_decode_toarrayfunction.md)\r\n* To encode strings to base64 string, see [base64_encode_tostring()](base64_encode_tostringfunction.md)", "```\r\nprint Quine=base64_decode_tostring(\"S3VzdG8=\")\r\n```\r\n\r\n|Quine|\r\n|-----|\r\n|Kusto|\r\n\r\nTrying to decode a base64 string that was generated from invalid UTF-8 encoding will return null:\r\n\r\n```\r\nprint Empty=base64_decode_tostring(\"U3RyaW5n0KHR0tGA0L7Rh9C60LA=\")\r\n```\r\n\r\n|Empty|\r\n|-----|\r\n||", "https://docs.microsoft.com/azure/data-explorer/kusto/query/base64_decode_tostringfunction", "base64_decode_tostringfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "base64_encode_fromarray", "Encodes a base64 string from a bytes array.", "## Syntax\r\n\r\n`base64_encode_fromarray(`*BytesArray*`)`\r\n\r\n## Arguments\r\n\r\n* *BytesArray*: Input bytes array to be encoded into base64 string.\r\n\r\n## Returns\r\n\r\nReturns the base64 string encoded from the bytes array.\r\n\r\n* For decoding base64 strings to a UTF-8 string see [base64_decode_tostring()](base64_decode_tostringfunction.md)\r\n* For encoding strings to base64 string see [base64_encode_tostring()](base64_encode_tostringfunction.md)\r\n* This function is the inverse of [base64_decode_toarray()](base64_decode_toarrayfunction.md)", "```\r\nlet bytes_array = toscalar(print base64_decode_toarray(\"S3VzdG8=\"));\r\nprint decoded_base64_string = base64_encode_fromarray(bytes_array)\r\n```\r\n\r\n|decoded_base64_string|\r\n|---|\r\n|S3VzdG8=|\r\n\r\nTrying to encode a base64 string from an invalid bytes array which was generated from invalid UTF-8 encoded string will return null:\r\n\r\n```\r\nlet empty_bytes_array = toscalar(print base64_decode_toarray(\"U3RyaW5n0KHR0tGA0L7Rh9C60LA\"));\r\nprint empty_string = base64_encode_fromarray(empty_bytes_array)\r\n```\r\n\r\n|empty_string|\r\n|---|\r\n||", "https://docs.microsoft.com/azure/data-explorer/kusto/query/base64_encode_fromarrayfunction", "base64_encode_fromarrayfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "base64_encode_fromguid", "Encodes a [GUID](./scalar-data-types/guid.md) to a base64 string.", "## Syntax\r\n\r\n`base64_encode_fromguid(`*GUID*`)`\r\n\r\n## Arguments\r\n\r\n* *GUID*: Input [GUID](./scalar-data-types/guid.md) to be encoded to a base64 string. \r\n\r\n## Returns\r\n\r\nReturns a base64 string encoded from a GUID.\r\n\r\n* To decode a base64 string to a [GUID](./scalar-data-types/guid.md), see [base64_decode_toguid()](base64-decode-toguid-function.md)\r\n* To create a [GUID](./scalar-data-types/guid.md) from a string, see [toguid()](toguidfunction.md)", "```\r\nprint Quine = base64_encode_fromguid(toguid(\"ae3133f2-6e22-49ae-b06a-16e6a9b212eb\"))  \r\n```\r\n\r\n|Quine|\r\n|-----|\r\n|8jMxriJurkmwahbmqbIS6w==|\r\n\r\nIf you try to encode anything that isn't a [GUID](./scalar-data-types/guid.md) as below, an error will be thrown:\r\n\r\n```\r\nprint Empty = base64_encode_fromguid(\"abcd1231\")\r\n```", "https://docs.microsoft.com/azure/data-explorer/kusto/query/base64-encode-fromguid-function", "base64-encode-fromguid-function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "base64_encode_tostring", "Encodes a string as base64 string.", "## Syntax\r\n\r\n`base64_encode_tostring(`*String*`)`\r\n\r\n## Arguments\r\n\r\n* *String*: Input string to be encoded as base64 string.\r\n\r\n## Returns\r\n\r\nReturns the string encoded as base64 string.\r\n\r\n* To decode base64 strings to UTF-8 strings, see [base64_decode_tostring()](base64_decode_tostringfunction.md)\r\n* To decode base64 strings to an array of long values, see [base64_decode_toarray()](base64_decode_toarrayfunction.md)", "```\r\nprint Quine=base64_encode_tostring(\"Kusto\")\r\n```\r\n\r\n|Quine   |\r\n|--------|\r\n|S3VzdG8=|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/base64_encode_tostringfunction", "base64_encode_tostringfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "basket", "The `basket` plugin finds frequent patterns of attributes in the data and returns the patterns that pass a frequency threshold in that data. A pattern represents a subset of the rows that have the same value across one or more columns. The `basket` plugin is based on the [Apriori algorithm](https://en.wikipedia.org/wiki/Association_rule_learning#Apriori_algorithm) originally developed for basket analysis data mining.", "## Syntax\r\n\r\n*T* | `evaluate` `basket` `(` [*Threshold*, *WeightColumn*, *MaxDimensions*, *CustomWildcard*, *CustomWildcard*, ...]`)`\r\n\r\n## Arguments\r\n\r\nAll arguments are optional, but they must be ordered as above. To indicate that the default value should be used, use the string tilde value - '~'. See examples below.\r\n\r\nAvailable arguments:\r\n\r\n* Threshold - 0.015 < *double* < 1 [default: 0.05]\r\n    Sets the minimal ratio of the rows to be considered frequent. Patterns with a smaller ratio won't be returned.\r\n    \r\n    Example: `T | evaluate basket(0.02)`\r\n* WeightColumn - *column_name*\r\n    Considers each row in the input according to the specified weight. By default, each row has a weight of '1'. The argument must be a name of a numeric column, such as int, long, real. A common use of a weight column, is to take into account sampling or bucketing/aggregation of the data that is already embedded into each row.\r\n    Example: `T | evaluate basket('~', sample_Count)`\r\n* MaxDimensions - 1 < *int* [default: 5]\r\n    Sets the maximal number of uncorrelated dimensions per basket, limited by default, to minimize the query runtime.\r\n    Example: `T | evaluate basket('~', '~', 3)`\r\n* CustomWildcard - *\"any_value_per_type\"*\r\n    Sets the wildcard value for a specific type in the result table that will indicate that the current pattern doesn't have a restriction on this column.\r\n    Default is null. The default for a string is an empty string. If the default is a good value in the data, a different wildcard value should be used, such as `*`.\r\n    For example:\r\n     `T | evaluate basket('~', '~', '~', '*', int(-1), double(-1), long(0), datetime(1900-1-1))`\r\n\r\n## Returns\r\n\r\nThe `basket` plugin returns frequent patterns that pass a ratio threshold. The default threshold is 0.05. \r\n\r\nEach pattern is represented by a row in the results. The first column is the segment ID. The next two columns are the *count* and *percentage of rows*, from the original query that match the pattern. The remaining columns relate to the original query, with either a specific value from the column or a wildcard value, which is by default null, meaning a variable value.\r\n\r\n**Notes**\r\nThe algorithm uses sampling to determine the initial frequent values. Consequently, the results could slightly differ between multiple runs for patterns whose frequency is close to the threshold.", "```\r\nStormEvents \r\n| where monthofyear(StartTime) == 5\r\n| extend Damage = iff(DamageCrops + DamageProperty > 0 , \"YES\" , \"NO\")\r\n| project State, EventType, Damage, DamageCrops\r\n| evaluate basket(0.2)\r\n```\r\n\r\n**Output**\r\n\r\n|SegmentId|Count|Percent|State|EventType|Damage|DamageCrops|\r\n|---|---|---|---|---|---|---|---|---|\r\n|0|4574|77.7|||NO|0\r\n|1|2278|38.7||Hail|NO|0\r\n|2|5675|96.4||||0\r\n|3|2371|40.3||Hail||0\r\n|4|1279|21.7||Thunderstorm Wind||0\r\n|5|2468|41.9||Hail||\r\n|6|1310|22.3|||YES|\r\n|7|1291|21.9||Thunderstorm Wind||\r\n\r\n**Example with custom wildcards**\r\n\r\n```\r\nStormEvents \r\n| where monthofyear(StartTime) == 5\r\n| extend Damage = iff(DamageCrops + DamageProperty > 0 , \"YES\" , \"NO\")\r\n| project State, EventType, Damage, DamageCrops\r\n| evaluate basket(0.2, '~', '~', '*', int(-1))\r\n```\r\n\r\n**Output**\r\n\r\n|SegmentId|Count|Percent|State|EventType|Damage|DamageCrops|\r\n|---|---|---|---|---|---|---|---|---|\r\n|0|4574|77.7|\\*|\\*|NO|0\r\n|1|2278|38.7|\\*|Hail|NO|0\r\n|2|5675|96.4|\\*|\\*|\\*|0\r\n|3|2371|40.3|\\*|Hail|\\*|0\r\n|4|1279|21.7|\\*|Thunderstorm Wind|\\*|0\r\n|5|2468|41.9|\\*|Hail|\\*|-1\r\n|6|1310|22.3|\\*|\\*|YES|-1\r\n|7|1291|21.9|\\*|Thunderstorm Wind|\\*|-1\r\n\r\n---\r\n\r\n> [!IMPORTANT]\r\n> The content below is Microsoft confidential(not available in the [public site](https://docs.microsoft.com/en-us/azure/data-explorer/))\r\n\r\n# Basket plugin - Internal\r\n\r\nIf you have questions about basket or other ML related workflows in Kusto you can post them on the following DL: [KustoML](mailto:kustoML@microsoft.com).", "https://docs.microsoft.com/azure/data-explorer/kusto/query/basketplugin", "basketplugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "beta_cdf", "Returns the standard cumulative beta distribution function.", "```\r\nbeta_cdf(0.2, 10.0, 50.0)\r\n```\r\n\r\nIf *probability* = `beta_cdf(`*x*,...`)`, then `beta_inv(`*probability*,...`)` = *x*.\r\n\r\nThe beta distribution is commonly used to study variation in the percentage of something across samples, such as the fraction of the day people spend watching television.\r\n\r\n## Syntax\r\n\r\n`beta_cdf(`*x*`, `*alpha*`, `*beta*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: A value at which to evaluate the function.\r\n* *alpha*: A parameter of the distribution.\r\n* *beta*: A parameter of the distribution.\r\n\r\n## Returns\r\n\r\n* The [cumulative beta distribution function](https://en.wikipedia.org/wiki/Beta_distribution#Cumulative_distribution_function).\r\n\r\n**Notes**\r\n\r\nIf any argument is nonnumeric, beta_cdf() returns null value.\r\n\r\nIf x < 0 or x > 1, beta_cdf() returns NaN value.\r\n\r\nIf alpha \u2264 0 or alpha > 10000, beta_cdf() returns the NaN value.\r\n\r\nIf beta \u2264 0 or beta > 10000, beta_cdf() returns the NaN value.", "```\r\ndatatable(x:double, alpha:double, beta:double, comment:string)\r\n[\r\n    0.9, 10.0, 20.0, \"Valid input\",\r\n    1.5, 10.0, 20.0, \"x > 1, yields NaN\",\r\n    double(-10), 10.0, 20.0, \"x < 0, yields NaN\",\r\n    0.1, double(-1.0), 20.0, \"alpha is < 0, yields NaN\"\r\n]\r\n| extend b = beta_cdf(x, alpha, beta)\r\n```\r\n\r\n|x|alpha|beta|comment|b|\r\n|---|---|---|---|---|\r\n|0.9|10|20|Valid input|0.999999999999959|\r\n|1.5|10|20|x > 1, yields NaN|NaN|\r\n|-10|10|20|x < 0, yields NaN|NaN|\r\n|0.1|-1|20|alpha is < 0, yields NaN|NaN|\r\n\r\n## See also\r\n\r\n* For computing the inverse of the beta cumulative probability density function, see [beta-inv()](./beta-invfunction.md).\r\n* For computing probability density function, see [beta-pdf()](./beta-pdffunction.md).", "https://docs.microsoft.com/azure/data-explorer/kusto/query/beta-cdffunction", "beta-cdffunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "beta_inv", "Returns the inverse of the beta cumulative probability beta density function.", "```\r\nbeta_inv(0.1, 10.0, 50.0)\r\n```\r\n\r\nIf *probability* = `beta_cdf(`*x*,...`)`, then `beta_inv(`*probability*,...`)` = *x*. \r\n\r\nThe beta distribution can be used in project planning to model probable completion times given an expected completion time and variability.\r\n\r\n## Syntax\r\n\r\n`beta_inv(`*probability*`, `*alpha*`, `*beta*`)`\r\n\r\n## Arguments\r\n\r\n* *probability*: A probability associated with the beta distribution.\r\n* *alpha*: A parameter of the distribution.\r\n* *beta*: A parameter of the distribution.\r\n\r\n## Returns\r\n\r\n* The inverse of the beta cumulative probability density function [beta_cdf()](./beta-cdffunction.md)\r\n\r\n**Notes**\r\n\r\nIf any argument is nonnumeric, beta_inv() returns null value.\r\n\r\nIf alpha \u2264 0 or beta \u2264 0, beta_inv() returns the null value.\r\n\r\nIf probability \u2264 0 or probability > 1, beta_inv() returns the NaN value.\r\n\r\nGiven a value for probability, beta_inv() seeks that value x such that beta_cdf(x, alpha, beta) = probability.", "```\r\ndatatable(p:double, alpha:double, beta:double, comment:string)\r\n[\r\n    0.1, 10.0, 20.0, \"Valid input\",\r\n    1.5, 10.0, 20.0, \"p > 1, yields null\",\r\n    0.1, double(-1.0), 20.0, \"alpha is < 0, yields NaN\"\r\n]\r\n| extend b = beta_inv(p, alpha, beta)\r\n```\r\n\r\n|p|alpha|beta|comment|b|\r\n|---|---|---|---|---|\r\n|0.1|10|20|Valid input|0.226415022388749|\r\n|1.5|10|20|p > 1, yields null||\r\n|0.1|-1|20|alpha is < 0, yields NaN|NaN|\r\n\r\n## See also\r\n\r\n* For computing cumulative beta distribution function, see [beta-cdf()](./beta-cdffunction.md).\r\n* For computing probability beta density function, see [beta-pdf()](./beta-pdffunction.md).", "https://docs.microsoft.com/azure/data-explorer/kusto/query/beta-invfunction", "beta-invfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "beta_pdf", "Returns the probability density beta function.", "```\r\nbeta_pdf(0.2, 10.0, 50.0)\r\n```\r\n\r\nThe beta distribution is commonly used to study variation in the percentage of something across samples, such as the fraction of the day people spend watching television.\r\n\r\n## Syntax\r\n\r\n`beta_pdf(`*x*`, `*alpha*`, `*beta*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: A value at which to evaluate the function.\r\n* *alpha*: A parameter of the distribution.\r\n* *beta*: A parameter of the distribution.\r\n\r\n## Returns\r\n\r\n* The [probability beta density function](https://en.wikipedia.org/wiki/Beta_distribution#Probability_density_function).\r\n\r\n**Notes**\r\n\r\nIf any argument is nonnumeric, beta_pdf() returns null value.\r\n\r\nIf x \u2264 0 or 1 \u2264 x, beta_pdf() returns NaN value.\r\n\r\nIf alpha \u2264 0 or beta \u2264 0, beta_pdf() returns the NaN value.", "```\r\ndatatable(x:double, alpha:double, beta:double, comment:string)\r\n[\r\n    0.5, 10.0, 20.0, \"Valid input\",\r\n    1.5, 10.0, 20.0, \"x > 1, yields NaN\",\r\n    double(-10), 10.0, 20.0, \"x < 0, yields NaN\",\r\n    0.1, double(-1.0), 20.0, \"alpha is < 0, yields NaN\"\r\n]\r\n| extend r = beta_pdf(x, alpha, beta)\r\n```\r\n\r\n|x|alpha|beta|comment|r|\r\n|---|---|---|---|---|\r\n|0.5|10|20|Valid input|0.746176019310951|\r\n|1.5|10|20|x > 1, yields NaN|NaN|\r\n|-10|10|20|x < 0, yields NaN|NaN|\r\n|0.1|-1|20|alpha is < 0, yields NaN|NaN|\r\n\r\n**References**\r\n\r\n* For computing the inverse of the beta cumulative probability density function, see [beta-inv()](./beta-invfunction.md).\r\n* For the standard cumulative beta distribution function, see [beta-cdf()](./beta-cdffunction.md).", "https://docs.microsoft.com/azure/data-explorer/kusto/query/beta-pdffunction", "beta-pdffunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "between", "Filters a record set for data matching the values in an inclusive range.", "`between` can operate on any numeric, datetime, or timespan expression.\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *expr* `between` `(`*leftRange*` .. `*rightRange*`)`   \r\n\r\nIf *expr* expression is datetime - another syntactic sugar syntax is provided:\r\n\r\n*T* `|` `where` *expr* `between` `(`*leftRangeDateTime*` .. `*rightRangeTimespan*`)`   \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be matched.\r\n* *expr* - the expression to filter.\r\n* *leftRange* - expression of the left range (inclusive).\r\n* *rightRange* - expression of the right range (inclusive).\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate of (*expr* >= *leftRange* and *expr* <= *rightRange*) evaluates to `true`.", "### Filter numeric values   \r\n\r\n```\r\nrange x from 1 to 100 step 1\r\n| where x between (50 .. 55)\r\n```\r\n\r\n|x|\r\n|---|\r\n|50|\r\n|51|\r\n|52|\r\n|53|\r\n|54|\r\n|55|\r\n\r\n### Filter datetime \r\n\r\n```\r\nStormEvents\r\n| where StartTime between (datetime(2007-07-27) .. datetime(2007-07-30))\r\n| count \r\n```\r\n\r\n|Count|\r\n|---|\r\n|476|\r\n\r\n```\r\nStormEvents\r\n| where StartTime between (datetime(2007-07-27) .. 3d)\r\n| count \r\n```\r\n\r\n|Count|\r\n|---|\r\n|476|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/betweenoperator", "betweenoperator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "bin", "Rounds values down to an integer multiple of a given bin size.", "Used frequently in combination with [`summarize by ...`](./summarizeoperator.md).\r\nIf you have a scattered set of values, they will be grouped into a smaller set of specific values.\r\n\r\nNull values, a null bin size, or a negative bin size will result in null. \r\n\r\nAlias to `floor()` function.\r\n\r\n## Syntax\r\n\r\n`bin(`*value*`,`*roundTo*`)`\r\n\r\n## Arguments\r\n\r\n* *value*: A number, date, or [timespan](scalar-data-types/timespan.md). \r\n* *roundTo*: The \"bin size\". A number or timespan that divides *value*. \r\n\r\n## Returns\r\n\r\nThe nearest multiple of *roundTo* below *value*.  \r\n\r\n```\r\n(toint((value/roundTo))) * roundTo`\r\n```", "Expression | Result\r\n---|---\r\n`bin(4.5, 1)` | `4.0`\r\n`bin(time(16d), 7d)` | `14d`\r\n`bin(datetime(1970-05-11 13:45:07), 1d)`|  `datetime(1970-05-11)`\r\n\r\nThe following expression calculates a histogram of durations,\r\nwith a bucket size of 1 second:\r\n\r\n```\r\nT | summarize Hits=count() by bin(Duration, 1s)\r\n```", "https://docs.microsoft.com/azure/data-explorer/kusto/query/binfunction", "binfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "bin_at", "Rounds values down to a fixed-size \"bin\", with control over the bin's starting point.\r\n(See also [`bin function`](./binfunction.md).)", "## Syntax\r\n\r\n`bin_at` `(`*Expression*`,` *BinSize*`, ` *FixedPoint*`)`\r\n\r\n## Arguments\r\n\r\n* *Expression*: A scalar expression of a numeric type (including `datetime` and `timespan`)\r\n  indicating the value to round.\r\n* *BinSize*: A scalar constant of a numeric type or `timespan` (for a `datetime` or `timespan` *Expression*) indicating\r\n  the size of each bin.\r\n* *FixedPoint*: A scalar constant of the same type as *Expression* indicating\r\n  one value of *Expression* which is a \"fixed point\" (that is, a value `fixed_point`\r\n  for which `bin_at(fixed_point, bin_size, fixed_point) == fixed_point`.)\r\n\r\n## Returns\r\n\r\nThe nearest multiple of *BinSize* below *Expression*, shifted so that *FixedPoint*\r\nwill be translated into itself.", "|Expression                                                                    |Result                           |Comments                   |\r\n|------------------------------------------------------------------------------|---------------------------------|---------------------------|\r\n|`bin_at(6.5, 2.5, 7)`                                                         |`4.5`                            ||\r\n|`bin_at(time(1h), 1d, 12h)`                                                   |`-12h`                           ||\r\n|`bin_at(datetime(2017-05-15 10:20:00.0), 1d, datetime(1970-01-01 12:00:00.0))`|`datetime(2017-05-14 12:00:00.0)`|All bins will be at noon   |\r\n|`bin_at(datetime(2017-05-17 10:20:00.0), 7d, datetime(2017-06-04 00:00:00.0))`|`datetime(2017-05-14 00:00:00.0)`|All bins will be on Sundays|\r\n\r\nIn the following example, notice that the `\"fixed point\"` arg is returned as one of the bins and the other bins are aligned to it based on the `bin_size`. Also note that each datetime bin represents the starting time of that bin:\r\n\r\n```\r\n\r\ndatatable(Date:datetime, Num:int)[\r\ndatetime(2018-02-24T15:14),3,\r\ndatetime(2018-02-23T16:14),4,\r\ndatetime(2018-02-26T15:14),5]\r\n| summarize sum(Num) by bin_at(Date, 1d, datetime(2018-02-24 15:14:00.0000000)) \r\n```\r\n\r\n|Date|sum_Num|\r\n|---|---|\r\n|2018-02-23 15:14:00.0000000|4|\r\n|2018-02-24 15:14:00.0000000|3|\r\n|2018-02-26 15:14:00.0000000|5|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/binatfunction", "binatfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "bin_auto", "Rounds values down to a fixed-size \"bin\", with control over the bin size and starting point provided by a query property.", "## Syntax\r\n\r\n`bin_auto` `(` *Expression* `)`\r\n\r\n## Arguments\r\n\r\n* *Expression*: A scalar expression of a numeric type indicating the value to round.\r\n\r\n**Client Request Properties**\r\n\r\n* `query_bin_auto_size`: A numeric literal indicating the size of each bin.\r\n* `query_bin_auto_at`: A numeric literal indicating one value of *Expression* which is a \"fixed point\" (that is, a value `fixed_point`\r\n  for which `bin_auto(fixed_point)` == `fixed_point`.)\r\n\r\n## Returns\r\n\r\nThe nearest multiple of `query_bin_auto_at` below *Expression*, shifted so that `query_bin_auto_at`\r\nwill be translated into itself.", "```\r\nset query_bin_auto_size=1h;\r\nset query_bin_auto_at=datetime(2017-01-01 00:05);\r\nrange Timestamp from datetime(2017-01-01 00:05) to datetime(2017-01-01 02:00) step 1m\r\n| summarize count() by bin_auto(Timestamp)\r\n```\r\n\r\n|Timestamp                    | count_|\r\n|-----------------------------|-------|\r\n|2017-01-01 00:05:00.0000000  | 60    |\r\n|2017-01-01 01:05:00.0000000  | 56    |", "https://docs.microsoft.com/azure/data-explorer/kusto/query/bin-autofunction", "bin-autofunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "binary_all_and", "Accumulates values using the binary `AND` operation per summarization group (or in total, if summarization is done without grouping).", "* Can be used only in context of aggregation inside [summarize](summarizeoperator.md)\r\n\r\n## Syntax\r\n\r\n`binary_all_and` `(`*Expr*`)`\r\n\r\n## Arguments\r\n\r\n* *Expr*: long number.\r\n\r\n## Returns\r\n\r\nReturns a value that is aggregated using the binary `AND` operation over records per summarization group (or in total, if summarization is done without grouping).", "Producing 'cafe-food' using binary `AND` operations:\r\n\r\n```\r\ndatatable(num:long)\r\n[\r\n  0xFFFFFFFF, \r\n  0xFFFFF00F,\r\n  0xCFFFFFFD,\r\n  0xFAFEFFFF,\r\n]\r\n| summarize result = toupper(tohex(binary_all_and(num)))\r\n```\r\n\r\n|result|\r\n|---|\r\n|CAFEF00D|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/binary-all-and-aggfunction", "binary-all-and-aggfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "binary_all_or", "Accumulates values using the binary `OR` operation per summarization group (or in total, if summarization is done without grouping).", "* Can be used only in context of aggregation inside [summarize](summarizeoperator.md)\r\n\r\n## Syntax\r\n\r\n`binary_all_or` `(`*Expr*`)`\r\n\r\n## Arguments\r\n\r\n* *Expr*: long number.\r\n\r\n## Returns\r\n\r\nReturns a value that is aggregated using the binary `OR` operation over records per summarization group (or in total, if summarization is done without grouping).", "Producing 'cafe-food' using binary `OR` operations:\r\n\r\n```\r\ndatatable(num:long)\r\n[\r\n  0x88888008,\r\n  0x42000000,\r\n  0x00767000,\r\n  0x00000005, \r\n]\r\n| summarize result = toupper(tohex(binary_all_or(num)))\r\n```\r\n\r\n|result|\r\n|---|\r\n|CAFEF00D|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/binary-all-or-aggfunction", "binary-all-or-aggfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "binary_all_xor", "Accumulates values using the binary `XOR` operation per summarization group (or in total, if summarization is done without grouping).", "* Can be used only in context of aggregation inside [summarize](summarizeoperator.md)\r\n\r\n## Syntax\r\n\r\n`binary_all_xor` `(`*Expr*`)`\r\n\r\n## Arguments\r\n\r\n* *Expr*: long number.\r\n\r\n## Returns\r\n\r\nReturns a value that is aggregated using the binary `XOR` operation over records per summarization group (or in total, if summarization is done without grouping).", "Producing 'cafe-food' using binary `XOR` operations:\r\n\r\n```\r\ndatatable(num:long)\r\n[\r\n  0x44404440,\r\n  0x1E1E1E1E,\r\n  0x90ABBA09,\r\n  0x000B105A,\r\n]\r\n| summarize result = toupper(tohex(binary_all_xor(num)))\r\n```\r\n\r\n|result|\r\n|---|\r\n|CAFEF00D|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/binary-all-xor-aggfunction", "binary-all-xor-aggfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "binary_and", "Returns a result of the bitwise `and` operation between two values.", "```\r\nbinary_and(x,y)\t\r\n```\r\n\r\n## Syntax\r\n\r\n`binary_and(`*num1*`,` *num2* `)`\r\n\r\n## Arguments\r\n\r\n* *num1*, *num2*: long numbers.\r\n\r\n## Returns\r\n\r\nReturns logical AND operation on a pair of numbers: num1 & num2.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/binary-andfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "binary_not", "Returns a bitwise negation of the input value.", "```\r\nbinary_not(x)\r\n```\r\n\r\n## Syntax\r\n\r\n`binary_not(`*num1*`)`\r\n\r\n## Arguments\r\n\r\n* *num1*: numeric \r\n\r\n## Returns\r\n\r\nReturns logical NOT operation on a number: num1.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/binary-notfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "binary_or", "Returns a result of the bitwise `or` operation of the two values.", "```\r\nbinary_or(x,y)\r\n```\r\n\r\n## Syntax\r\n\r\n`binary_or(`*num1*`,` *num2* `)`\r\n\r\n## Arguments\r\n\r\n* *num1*, *num2*: long numbers.\r\n\r\n## Returns\r\n\r\nReturns logical OR operation on a pair of numbers: num1 | num2.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/binary-orfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "binary_shift_left", "Returns binary shift left operation on a pair of numbers.", "```\r\nbinary_shift_left(x,y)\t\r\n```\r\n\r\n## Syntax\r\n\r\n`binary_shift_left(`*num1*`,` *num2* `)`\r\n\r\n## Arguments\r\n\r\n* *num1*, *num2*: int numbers.\r\n\r\n## Returns\r\n\r\nReturns binary shift left operation on a pair of numbers: num1 << (num2%64).\r\nIf n is negative a NULL value is returned.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/binary-shift-leftfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "binary_shift_right", "Returns binary shift right operation on a pair of numbers.", "```\r\nbinary_shift_right(x,y)\t\r\n```\r\n\r\n## Syntax\r\n\r\n`binary_shift_right(`*num1*`,` *num2* `)`\r\n\r\n## Arguments\r\n\r\n* *num1*, *num2*: long numbers.\r\n\r\n## Returns\r\n\r\nReturns binary shift right operation on a pair of numbers: num1 >> (num2%64).\r\nIf n is negative a NULL value is returned.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/binary-shift-rightfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "binary_xor", "Returns a result of the bitwise `xor` operation of the two values.", "```\r\nbinary_xor(x,y)\r\n```\r\n\r\n## Syntax\r\n\r\n`binary_xor(`*num1*`,` *num2* `)`\r\n\r\n## Arguments\r\n\r\n* *num1*, *num2*: long numbers.\r\n\r\n## Returns\r\n\r\nReturns logical XOR operation on a pair of numbers: num1 ^ num2.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/binary-xorfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "bitset_count_ones", "Returns the number of set bits in the binary representation of a number.", "```\r\nbitset_count_ones(42)\r\n```\r\n\r\n## Syntax\r\n\r\n`bitset_count_ones(`*num1*``)`\r\n\r\n## Arguments\r\n\r\n* *num1*: long or integer number.\r\n\r\n## Returns\r\n\r\nReturns the number of set bits in the binary representation of a number.", "```\r\n// 42 = 32+8+2 : b'00101010' == 3 bits set\r\nprint ones = bitset_count_ones(42) \r\n```\r\n\r\n|ones|\r\n|---|\r\n|3|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/bitset-count-onesfunction", "bitset-count-onesfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "buildschema", "Returns the minimal schema that admits all values of *DynamicExpr*.", "* Can be used only in context of aggregation, inside [summarize](summarizeoperator.md)\r\n\r\n## Syntax\r\n\r\n`buildschema` `(`*DynamicExpr*`)`\r\n\r\n## Arguments\r\n\r\n* *DynamicExpr*: Expression that is used for the aggregation calculation. The parameter column type must be `dynamic`. \r\n\r\n## Returns\r\n\r\nThe maximum value of *`Expr`* across the group.\r\n\r\n> [!TIP] \r\n> If `buildschema(json_column)` gives a syntax error:\r\n>\r\n> > *Is your `json_column` a string rather than a dynamic object?*\r\n>\r\n> then use `buildschema(parsejson(json_column))`.", "Assume the input column has three dynamic values.\r\n\r\n* `{\"x\":1, \"y\":3.5}`\r\n* `{\"x\":\"somevalue\", \"z\":[1, 2, 3]}`\r\n* `{\"y\":{\"w\":\"zzz\"}, \"t\":[\"aa\", \"bb\"], \"z\":[\"foo\"]}`\r\n\r\n\r\nThe resulting schema would be:\r\n\r\n```\r\n{ \r\n    \"x\":[\"int\", \"string\"],\r\n    \"y\":[\"double\", {\"w\": \"string\"}],\r\n    \"z\":{\"`indexer`\": [\"int\", \"string\"]},\r\n    \"t\":{\"`indexer`\": \"string\"}\r\n}\r\n```\r\n\r\nThe schema tells us that:\r\n\r\n* The root object is a container with four properties named x, y, z, and t.\r\n* The property called \"x\" that could be of type \"int\" or of type \"string\".\r\n* The property called \"y\" that could be of type \"double\", or another container with a property called \"w\" of type \"string\".\r\n* The ``indexer`` keyword indicates that \"z\" and \"t\" are arrays.\r\n* Each item in the array \"z\" is of type \"int\" or of type \"string\".\r\n* \"t\" is an array of strings.\r\n* Every property is implicitly optional, and any array may be empty.\r\n\r\n\r\n### Schema model\r\n\r\nThe syntax of the returned schema is:\r\n\r\n```output\r\nContainer ::= '{' Named-type* '}';\r\nNamed-type ::= (name | '\"`indexer`\"') ':' Type;\r\nType ::= Primitive-type | Union-type | Container;\r\nUnion-type ::= '[' Type* ']';\r\nPrimitive-type ::= \"int\" | \"string\" | ...;\r\n```\r\n\r\nThe values are equivalent to a subset of the TypeScript type annotations, encoded as a Kusto dynamic value. \r\nIn Typescript, the example schema would be:\r\n\r\n```typescript\r\nvar someobject: \r\n{\r\n    x?: (number | string),\r\n    y?: (number | { w?: string}),\r\n    z?: { [n:number] : (int | string)},\r\n    t?: { [n:number]: string }\r\n}\r\n```", "https://docs.microsoft.com/azure/data-explorer/kusto/query/buildschema-aggfunction", "buildschema-aggfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "case", "Evaluates a list of predicates and returns the first result expression whose predicate is satisfied.", "If neither of the predicates return `true`, the result of the last expression (the `else`) is returned.\r\nAll odd arguments (count starts at 1) must be expressions that evaluate to a  `boolean` value.\r\nAll even arguments (the `then`s) and the last argument (the `else`) must be of the same type.\r\n\r\n## Syntax\r\n\r\n`case(`*predicate_1*, *then_1*,\r\n       *predicate_2*, *then_2*,\r\n       *predicate_3*, *then_3*,\r\n       *else*`)`\r\n\r\n## Arguments\r\n\r\n* *predicate_i*: An expression that evaluates to a `boolean` value.\r\n* *then_i*: An expression that gets evaluated and its value is returned from the function if *predicate_i* is the first predicate that evaluates to `true`.\r\n* *else*: An expression that gets evaluated and its value is returned from the function if neither of the *predicate_i* evaluate to `true`.\r\n\r\n## Returns\r\n\r\nThe value of the first *then_i* whose *predicate_i* evaluates to `true`, or the value of *else* if neither of the predicates are satisfied.", "```\r\nrange Size from 1 to 15 step 2\r\n| extend bucket = case(Size <= 3, \"Small\", \r\n                       Size <= 10, \"Medium\", \r\n                       \"Large\")\r\n```\r\n\r\n|Size|bucket|\r\n|---|---|\r\n|1|Small|\r\n|3|Small|\r\n|5|Medium|\r\n|7|Medium|\r\n|9|Medium|\r\n|11|Large|\r\n|13|Large|\r\n|15|Large|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/casefunction", "casefunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "ceiling", "Calculates the smallest integer greater than, or equal to, the specified numeric expression.", "## Syntax\r\n\r\n`ceiling(`*x*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: A real number.\r\n\r\n## Returns\r\n\r\n* The smallest integer greater than, or equal to, the specified numeric expression.", "```\r\nprint c1 = ceiling(-1.1), c2 = ceiling(0), c3 = ceiling(0.9)\r\n```\r\n\r\n|c1|c2|c3|\r\n|---|---|---|\r\n|-1|0|1|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/ceilingfunction", "ceilingfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "cluster", "Changes the reference of the query to a remote cluster.", "```\r\ncluster('help').database('Sample').SomeTable\r\n```\r\n\r\n## Syntax\r\n\r\n`cluster(`*stringConstant*`)`\r\n\r\n## Arguments\r\n\r\n* *stringConstant*: Name of the cluster that is referenced. Cluster name can be either \r\na fully qualified DNS name, or a string that will be suffixed with `.kusto.windows.net`. Argument has to be _constant_ prior to the query's execution,\r\ni.e. cannot come from sub-query evaluation.\r\n\r\n**Notes**\r\n\r\n* For accessing database within the same cluster - use [database()](databasefunction.md) function.\r\n* More information about cross-cluster and cross-database queries available [here](cross-cluster-or-database-queries.md)", "### Use cluster() to access remote cluster \r\n\r\nThe next query can be run on any of the Kusto clusters.\r\n\r\n```\r\ncluster('help').database('Samples').StormEvents | count\r\n\r\ncluster('help.kusto.windows.net').database('Samples').StormEvents | count  \r\n```\r\n\r\n|Count|\r\n|---|\r\n|59066|\r\n\r\n### Use cluster() inside let statements \r\n\r\nThe same query as above can be rewritten to use inline function (let statement) that \r\nreceives a parameter `clusterName` - which is passed into the cluster() function.\r\n\r\n```\r\nlet foo = (clusterName:string)\r\n{\r\n    cluster(clusterName).database('Samples').StormEvents | count\r\n};\r\nfoo('help')\r\n```\r\n\r\n|Count|\r\n|---|\r\n|59066|\r\n\r\n### Use cluster() inside Functions \r\n\r\nThe same query as above can be rewritten to be used in a function that \r\nreceives a parameter `clusterName` - which is passed into the cluster() function.\r\n\r\n```\r\n.create function foo(clusterName:string)\r\n{\r\n    cluster(clusterName).database('Samples').StormEvents | count\r\n};\r\n```\r\n\r\n**Note:** such functions can be used only locally and not in the cross-cluster query.", "https://docs.microsoft.com/azure/data-explorer/kusto/query/clusterfunction", "clusterfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "coalesce", "Evaluates a list of expressions and returns the first non-null (or non-empty for string) expression.", "```\r\ncoalesce(tolong(\"not a number\"), tolong(\"42\"), 33) == 42\r\n```\r\n\r\n## Syntax\r\n\r\n`coalesce(`*expr_1*`, `*expr_2*`,` ...)\r\n\r\n## Arguments\r\n\r\n* *expr_i*: A scalar expression, to be evaluated.\r\n\r\n- All arguments must be of the same type.\r\n- Maximum of 64 arguments is supported.\r\n\r\n\r\n## Returns\r\n\r\nThe value of the first *expr_i* whose value is not null (or not-empty for string expressions).", "```\r\nprint result=coalesce(tolong(\"not a number\"), tolong(\"42\"), 33)\r\n```\r\n\r\n|result|\r\n|---|\r\n|42|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/coalescefunction", "coalescefunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "consume", "Consumes the tabular data stream handed to the operator.", "The `consume` operator is mostly used for triggering the query side-effect without actually returning\r\nthe results back to the caller.\r\n\r\n```\r\nT | consume\r\n```\r\n\r\n## Syntax\r\n\r\n`consume` [`decodeblocks` `=` *DecodeBlocks*]\r\n\r\n## Arguments\r\n\r\n* *DecodeBlocks*: A constant Boolean value. If set to `true`, or if the request\r\n  property `perftrace` is set to `true`, the `consume` operator will not just\r\n  enumerate the records at its input, but actually force each value in those\r\n  records to be decompressed and decoded.\r\n\r\nThe `consume` operator can be used for estimating the\r\ncost of a query without actually delivering the results back to the client.\r\n(The estimation is not exact for a variety of reasons; for example, `consume`\r\nis calculated distributively, so `T | consume` will not transmit the table's\r\ndata between the nodes of the cluster.)", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/consumeoperator", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "contains", "Filters a record set for data containing a case-insensitive string. `contains` searches for characters rather than [terms](datatypes-string-operators.md#what-is-a-term) of three or more characters. The query scans the values in the column, which is slower than looking up a term in a term index.", "The following table provides a comparison of the `contains` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`contains`](contains-operator.md) |RHS occurs as a subsequence of LHS |No |`\"FabriKam\" contains \"BRik\"`|\r\n|[`!contains`](not-contains-operator.md) |RHS doesn't occur in LHS |No |`\"Fabrikam\" !contains \"xyz\"`|\r\n|[`contains_cs`](contains-cs-operator.md) |RHS occurs as a subsequence of LHS |Yes |`\"FabriKam\" contains_cs \"Kam\"`|\r\n|[`!contains_cs`](not-contains-cs-operator.md)   |RHS doesn't occur in LHS |Yes |`\"Fabrikam\" !contains_cs \"Kam\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\nCase-insensitive operators are currently supported only for ASCII-text. For non-ASCII comparison, use the [tolower()](tolowerfunction.md) function.\r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor better performance, try the case-sensitive version of an operator, for example, `contains_cs`, not `contains`. \r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for better performance, try `has` or `in`. Also, `has` works faster than `contains`, `startswith`, or `endswith`, however it is not as precise and could provide unwanted records.\r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `contains` `(`*expression*`)`   \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State contains \"enn\"\r\n    | where event_count > 10\r\n    | project State, event_count\r\n    | render table\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|PENNSYLVANIA|1687|\r\n|TENNESSEE|1125|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/contains-operator", "contains-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "contains_cs", "Filters a record set for data containing a case-sensitive string. `contains` searches for characters rather than [terms](datatypes-string-operators.md#what-is-a-term) of three or more characters. The query scans the values in the column, which is slower than looking up a term in a term index.", "The following table provides a comparison of the `contains` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`contains`](contains-operator.md) |RHS occurs as a subsequence of LHS |No |`\"FabriKam\" contains \"BRik\"`|\r\n|[`!contains`](not-contains-operator.md) |RHS doesn't occur in LHS |No |`\"Fabrikam\" !contains \"xyz\"`|\r\n|[`contains_cs`](contains-cs-operator.md) |RHS occurs as a subsequence of LHS |Yes |`\"FabriKam\" contains_cs \"Kam\"`|\r\n|[`!contains_cs`](not-contains-cs-operator.md)   |RHS doesn't occur in LHS |Yes |`\"Fabrikam\" !contains_cs \"Kam\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\nFor faster results, use the case-sensitive version of an operator, for example, `contains_cs`, not `contains`.\r\n\r\nIf you're testing for the presence of a symbol or alphanumeric word that is bound by non-alphanumeric characters at the start or end of a field, for faster results use `has` or `in`. Also, `has` works faster than `contains`, `startswith`, or `endswith`, however it is not as precise and could provide unwanted records.\r\n\r\nFor best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `contains_cs` `(`*expression*`)`   \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State contains_cs \"AS\"\r\n    | count\r\n```\r\n\r\n**Output**\r\n\r\n|Count|\r\n|-----|\r\n|8|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/contains-cs-operator", "contains-cs-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "cos", "Returns the cosine function.", "## Syntax\r\n\r\n`cos(`*x*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: A real number.\r\n\r\n## Returns\r\n\r\n* The result of `cos(x)`", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/cosfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "cosmosdb_sql_request", "The `cosmosdb_sql_request` plugin sends a SQL query to a Cosmos DB SQL network endpoint and returns the results of the query. This plugin is primarily designed for querying small datasets, for example, enriching data with reference data stored in [Azure Cosmos DB](https://docs.microsoft.com/azure/cosmos-db/). The plugin is invoked with the [`evaluate`](evaluateoperator.md) operator.", "## Syntax\r\n\r\n`evaluate` `cosmosdb_sql_request` `(` *ConnectionString* `,` *SqlQuery* [`,` *SqlParameters* [`,` *Options*]] `)`\r\n\r\n## Arguments\r\n\r\n|Argument name | Description | Required/optional | \r\n|---|---|---|\r\n| *ConnectionString* | A `string` literal indicating the connection string that points to the Cosmos DB collection to query. It must include *AccountEndpoint*, *Database*, and *Collection*. It may include *AccountKey* if a master key is used for authentication. <br> **Example:** `'AccountEndpoint=https://cosmosdbacc.documents.azure.com/ ;Database=MyDatabase;Collection=MyCollection;AccountKey=' h'R8PM...;'`| Required |\r\n| *SqlQuery*| A `string` literal indicating the query to execute. | Required |\r\n| *SqlParameters* | A constant value of type `dynamic` that holds key-value pairs to pass as parameters along with the query. Parameter names must begin with `@`. | Optional |\r\n| *Options* | A constant value of type `dynamic` that holds more advanced settings as key-value pairs. | Optional |\r\n|| ----*Supported Options settings include:*-----\r\n|      `armResourceId` | Retrieve the API key from the Azure Resource Manager <br> **Example:** `/subscriptions/a0cd6542-7eaf-43d2-bbdd-b678a869aad1/resourceGroups/ cosmoddbresourcegrouput/providers/Microsoft.DocumentDb/databaseAccounts/cosmosdbacc`| \r\n|  `token` | Provide the Azure AD access token used to authenticate with the Azure Resource Manager.\r\n| `preferredLocations` | Control which region the data is queried from. <br> **Example:** `['East US']` | |  \r\n\r\n## Set callout policy\r\n\r\nThe plugin makes callouts to the Cosmos DB. Make sure that the cluster's [callout policy](../management/calloutpolicy.md) enables calls of type `cosmosdb` to the target *CosmosDbUri*.\r\n\r\nThe following example shows how to define the callout policy for Cosmos DB. It's recommended to restrict it to specific endpoints (`my_endpoint1`, `my_endpoint2`).\r\n\r\n```\r\n[\r\n  {\r\n    \"CalloutType\": \"CosmosDB\",\r\n    \"CalloutUriRegex\": \"my_endpoint1\\\\.documents\\\\.azure\\\\.com\",\r\n    \"CanCall\": true\r\n  },\r\n  {\r\n    \"CalloutType\": \"CosmosDB\",\r\n    \"CalloutUriRegex\": \"my_endpoint2\\\\.documents\\\\.azure\\\\.com\",\r\n    \"CanCall\": true\r\n  }\r\n]\r\n```\r\n\r\nThe following example shows an alter callout policy command for `cosmosdb` *CalloutType*\r\n\r\n```\r\n.alter cluster policy callout @'[{\"CalloutType\": \"cosmosdb\", \"CalloutUriRegex\": \"\\\\.documents\\\\.azure\\\\.com\", \"CanCall\": true}]'\r\n```", "### Query Cosmos DB\r\n\r\nThe following example uses the *cosmosdb_sql_request* plugin to send a SQL query to fetch data from Cosmos DB using its SQL API.\r\n\r\n```\r\nevaluate cosmosdb_sql_request(\r\n  'AccountEndpoint=https://cosmosdbacc.documents.azure.com/;Database=MyDatabase;Collection=MyCollection;AccountKey=' h'R8PM...;',\r\n  'SELECT * from c')\r\n```\r\n\r\n### Query Cosmos DB with parameters\r\n\r\nThe following example uses SQL query parameters and queries the data from an alternate region. For more information, see [`preferredLocations`](https://docs.microsoft.com/azure/cosmos-db/tutorial-global-distribution-sql-api?tabs=dotnetv2%2Capi-async#preferred-locations).\r\n\r\n```\r\nevaluate cosmosdb_sql_request(\r\n    'AccountEndpoint=https://cosmosdbacc.documents.azure.com/;Database=MyDatabase;Collection=MyCollection;AccountKey=' h'R8PM...;',\r\n    \"SELECT c.id, c.lastName, @param0 as Column0 FROM c WHERE c.dob >= '1970-01-01T00:00:00Z'\",\r\n    dynamic({'@param0': datetime(2019-04-16 16:47:26.7423305)}),\r\n    dynamic({'preferredLocations': ['East US']}))\r\n| where lastName == 'Smith'\r\n```\r\n\r\n---\r\n\r\n> [!IMPORTANT]\r\n> The content below is Microsoft confidential(not available in the [public site](https://docs.microsoft.com/en-us/azure/data-explorer/))\r\n\r\n# cosmosdb_sql_request plugin - Internal\r\n\r\nThe `cosmosdb_sql_request` plugin sends an SQL query to a Cosmos DB SQL network endpoint.\r\n\r\n- `evaluate` `cosmosdb_sql_request` `(` *ConnectionString* `,` *SqlQuery* [`,` *SqlParameters* [`,` *Options*]] `)`\r\n- `evaluate` `cosmosdb_sql_request` `(` *CosmosDbUri* `,` *authorization_key* `,` *database_name* `,` *collection_name* `,` *SqlQuery* `)`\r\n\r\n\r\n> [!NOTE]\r\n> It's recommended to use the syntax with `ConnectionString`. The other syntax is kept for backward-compatibility purposes.\r\n\r\n**Arguments**\r\n\r\n* *ConnectionString*: A `string` literal indicating the connection string that\r\n  points to the Cosmos DB collection to query. It must include `AccountEndpoint`, `Database`, `Collection`\r\n  and may include `AccountKey` if an API key is used for authentication. \r\n\r\n  Example:\r\n  `'AccountEndpoint=https://cosmosdbut.documents.azure.com:443/;Database=MyDatabase;Collection=MyCollection;AccountKey=' h'R8PM...;'`,\r\n  where the key is an [obfuscated string literal](./scalar-data-types/string.md#obfuscated-string-literals)\r\n  to prevent secret leakage.\r\n\r\n* *SqlQuery*: A `string` literal indicating the query to execute.\r\n\r\n* *SqlParameters*: A constant value of type `dynamic` that holds key-value pairs\r\n  to pass as parameters along with the query. Parameter names must begin with `@`. Optional.\r\n  \r\n* *Options*: A constant value of type `dynamic` that holds more advanced settings\r\n  as key-value pairs. Supported settings include `armResourceId` (Example: `/subscriptions/a0cd6542-7eaf-43d2-bbdd-b678a869aad1/resourceGroups/cosmoddbresourcegrouput/providers/Microsoft.DocumentDb/databaseAccounts/cosmosdbut`)\r\n  to retrieve the API key from ARM, `token` to provide the AAD access token used to authenticate with ARM, and `preferredLocations` (Example: `['East US']`)\r\n  to control which region the data is queried from. Optional.\r\n\r\n* *CosmosDbUri*: A `string` literal indicating the URI of the Cosmos DB endpoint\r\n  to connect to.\r\n\r\n* *authorization_key*: A `string` literal specifying CosmosDB resource or master token.\r\n  It is **strongly recommended** that this argument use the\r\n  [obfuscated string literals](./scalar-data-types/string.md#obfuscated-string-literals)\r\n  to prevent secret leakage.\r\n\r\n* *database_name*: A `string` literal specifying CosmosDB Database name.\r\n\r\n* *collection_name*: A `string` literal specifying CosmosDB collection name.\r\n\r\n\r\n**Restrictions**\r\n\r\nThe plugin makes callouts to Cosmos DB and so make sure that the\r\ncluster's [Callout policy](../management/calloutpolicy.md) enables calls of type\r\n`cosmosdb` to the target *CosmosDbUri*.\r\n\r\nBelow is an example of defining call-out policy for CosmosDB. It's recommended to restrict it to specific endpoints (`my_endpoint1`, `my_endpoint2`).\r\n\r\n```json\r\n[\r\n  {\r\n    \"CalloutType\": \"CosmosDB\",\r\n    \"CalloutUriRegex\": \"my_endpoint1.documents.azure.com\",\r\n    \"CanCall\": true\r\n  },\r\n  {\r\n    \"CalloutType\": \"CosmosDB\",\r\n    \"CalloutUriRegex\": \"my_endpoint2.documents.azure.com\",\r\n    \"CanCall\": true\r\n  }\r\n]\r\n```\r\n\r\n**Examples**\r\n\r\nUsing SQL query to fetch all rows from Cosmos DB:\r\n\r\n```\r\nevaluate cosmosdb_sql_request(\r\n  'AccountEndpoint=https://cosmosdbut.documents.azure.com:443/;Database=MyDatabase;Collection=MyCollection;AccountKey=' h'R8PM...;',\r\n  'SELECT * from c')\r\n```\r\n\r\nUsing SQL query parameters and querying the data from an alternate region:\r\n\r\n```\r\nevaluate cosmosdb_sql_request(\r\n    'AccountEndpoint=https://cosmosdbut.documents.azure.com:443/;Database=MyDatabase;Collection=MyCollection;AccountKey=' h'R8PM...;',\r\n    \"SELECT c.id, c.lastName, @param0 as Column0 FROM c WHERE c.dob >= '1970-01-01T00:00:00Z'\",\r\n    dynamic({'@param0': datetime(2019-04-16 16:47:26.7423305)}),\r\n    dynamic({'preferredLocations': ['East US']}))\r\n| where lastName == 'Smith'\r\n```", "https://docs.microsoft.com/azure/data-explorer/kusto/query/cosmosdb-plugin", "cosmosdb-plugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "cot", "Calculates the trigonometric cotangent of the specified angle, in radians.", "## Syntax\r\n\r\n`cot(`*x*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: A real number.\r\n\r\n## Returns\r\n\r\n* The cotangent function value for `x`", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/cotfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "count", "Returns a count of the records per summarization group (or in total, if summarization is done without grouping).", "* Can be used only in context of aggregation inside [summarize](summarizeoperator.md)\r\n* Use the [countif](countif-aggfunction.md) aggregation function\r\n  to count only records for which some predicate returns `true`.\r\n\r\n## Syntax\r\n\r\n`count` `(` `)`\r\n\r\n## Returns\r\n\r\nReturns a count of the records per summarization group (or in total, if summarization is done without grouping).", "Counting events in states starting with letter `W`:\r\n\r\n```\r\nStormEvents\r\n| where State startswith \"W\"\r\n| summarize Count=count() by State\r\n```\r\n\r\n|State|Count|\r\n|---|---|\r\n|WEST VIRGINIA|757|\r\n|WYOMING|396|\r\n|WASHINGTON|261|\r\n|WISCONSIN|1850|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/count-aggfunction", "count-aggfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "count", "Returns the number of records in the input record set.", "## Syntax\r\n\r\n`T | count`\r\n\r\n## Arguments\r\n\r\n*T*: The tabular data whose records are to be counted.\r\n\r\n## Returns\r\n\r\nThis function returns a table with a single record and column of type\r\n`long`. The value of the only cell is the number of records in *T*.", "```\r\nStormEvents | count\r\n```\r\n\r\n## See also\r\n\r\nFor information about the count() aggregation function, see [count() (aggregation function)](count-aggfunction.md).", "https://docs.microsoft.com/azure/data-explorer/kusto/query/countoperator", "countoperator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "countif", "Returns a count of rows for which *Predicate* evaluates to `true`. Can only be used only in context of aggregation inside [summarize](summarizeoperator.md).", "## Syntax\r\n\r\n`countif` `(`*Predicate*`)`\r\n\r\n## Arguments\r\n\r\n*Predicate*: Expression that will be used for aggregation calculation. *Predicate* can be any scalar expression with return type of bool (evaluating to true/false).\r\n\r\n## Returns\r\n\r\nReturns a count of rows for which *Predicate* evaluates to `true`.", "```\r\nlet T = datatable(name:string, day_of_birth:long)\r\n[\r\n   \"John\", 9,\r\n   \"Paul\", 18,\r\n   \"George\", 25,\r\n   \"Ringo\", 7\r\n];\r\nT\r\n| summarize countif(strlen(name) > 4)\r\n```\r\n\r\n|countif_|\r\n|----|\r\n|2|\r\n\r\n## See also\r\n\r\n[count()](count-aggfunction.md) function, which counts rows without predicate expression.", "https://docs.microsoft.com/azure/data-explorer/kusto/query/countif-aggfunction", "countif-aggfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "countof", "Counts occurrences of a substring in a string. Plain string matches may overlap; regex matches do not.", "```\r\ncountof(\"The cat sat on the mat\", \"at\") == 3\r\ncountof(\"The cat sat on the mat\", @\"\\b.at\\b\", \"regex\") == 3\r\n```\r\n\r\n## Syntax\r\n\r\n`countof(`*source*`,` *search* [`,` *kind*]`)`\r\n\r\n## Arguments\r\n\r\n* *source*: A string.\r\n* *search*: The plain string or [regular expression](./re2.md) to match inside *source*.\r\n* *kind*: `\"normal\"|\"regex\"` Default `normal`. \r\n\r\n## Returns\r\n\r\nThe number of times that the search string can be matched in the container. Plain string matches may overlap; regex matches do not.", "|Function call|Result|\r\n|---|---\r\n|`countof(\"aaa\", \"a\")`| 3 \r\n|`countof(\"aaaa\", \"aa\")`| 3 (not 2!)\r\n|`countof(\"ababa\", \"ab\", \"normal\")`| 2\r\n|`countof(\"ababa\", \"aba\")`| 2\r\n|`countof(\"ababa\", \"aba\", \"regex\")`| 1\r\n|`countof(\"abcabc\", \"a.c\", \"regex\")`| 2", "https://docs.microsoft.com/azure/data-explorer/kusto/query/countoffunction", "countoffunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "current_cluster_endpoint", "Returns the network endpoint (DNS name) of the current cluster being queried.", "## Syntax\r\n\r\n`current_cluster_endpoint()`\r\n\r\n## Returns\r\n\r\nThe network endpoint (DNS name) of the current cluster being queried,\r\nas a value of type `string`.", "```\r\nprint strcat(\"This query executed on: \", current_cluster_endpoint())\r\n```", "https://docs.microsoft.com/azure/data-explorer/kusto/query/current-cluster-endpoint-function", "current-cluster-endpoint-function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "current_database", "Returns the name of the database in scope (database that all query\r\nentities are resolved against if no other database is specified).", "## Syntax\r\n\r\n`current_database()`\r\n\r\n## Returns\r\n\r\nThe name of the database in scope as a value of type `string`.", "```\r\nprint strcat(\"Database in scope: \", current_database())\r\n```", "https://docs.microsoft.com/azure/data-explorer/kusto/query/current-database-function", "current-database-function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "current_principal", "Returns the current principal name that runs the query.", "## Syntax\r\n\r\n`current_principal()`\r\n\r\n## Returns\r\n\r\nThe current principal fully qualified name (FQN) as a `string`.  \r\nThe string format is:  \r\n*PrinciplaType*`=`*PrincipalId*`;`*TenantId*", "```\r\nprint fqn=current_principal()\r\n```\r\n\r\n|fqn|\r\n|---|\r\n|aaduser=346e950e-4a62-42bf-96f5-4cf4eac3f11e;72f988bf-86f1-41af-91ab-2d7cd011db47|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/current-principalfunction", "current-principalfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "current_principal_details", "Returns details of the principal running the query.", "## Syntax\r\n\r\n`current_principal_details()`\r\n\r\n## Returns\r\n\r\nThe details of the current principal as a `dynamic`.", "```\r\nprint d=current_principal_details()\r\n```\r\n\r\n|d|\r\n|---|\r\n|{<br>  \"UserPrincipalName\": \"user@fabrikam.com\",<br>  \"IdentityProvider\": \"https://sts.windows.net\",<br>  \"Authority\": \"72f988bf-86f1-41af-91ab-2d7cd011db47\",<br>  \"Mfa\": \"True\",<br>  \"Type\": \"AadUser\",<br>  \"DisplayName\": \"James Smith (upn: user@fabrikam.com)\",<br>  \"ObjectId\": \"346e950e-4a62-42bf-96f5-4cf4eac3f11e\",<br>  \"FQN\": null,<br>  \"Notes\": null<br>}|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/current-principal-detailsfunction", "current-principal-detailsfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "current_principal_is_member_of", "Checks group membership or principal identity of the current principal running the query.", "## Syntax\r\n\r\n`current_principal_is_member_of`(`*list of string literals*`)\r\n\r\n## Arguments\r\n\r\n* *list of expressions* - a comma-separated list of string literals, where each literal represents an AAD principal. See [examples for AAD principals](../management/access-control/principals-and-identity-providers.md#aad-tenants).\r\n\r\n## Returns\r\n\r\nThe function returns:\r\n\r\n* `true`: if the current principal running the query was successfully matched for at least one input argument.\r\n* `false`: otherwise", "```\r\nprint result=current_principal_is_member_of(\r\n    'aaduser=user1@fabrikam.com', \r\n    'aadgroup=group1@fabrikam.com',\r\n    'aadapp=66ad1332-3a94-4a69-9fa2-17732f093664;72f988bf-86f1-41af-91ab-2d7cd011db47'\r\n    )\r\n```\r\n\r\n| result |\r\n|--------|\r\n| false  |\r\n\r\nUsing dynamic array instead of multiple arguments:\r\n\r\n```\r\nprint result=current_principal_is_member_of(\r\n    dynamic([\r\n    'aaduser=user1@fabrikam.com', \r\n    'aadgroup=group1@fabrikam.com',\r\n    'aadapp=66ad1332-3a94-4a69-9fa2-17732f093664;72f988bf-86f1-41af-91ab-2d7cd011db47'\r\n    ]))\r\n```\r\n\r\n| result |\r\n|--------|\r\n| false  |", "https://docs.microsoft.com/azure/data-explorer/kusto/query/current-principal-ismemberoffunction", "current-principal-ismemberoffunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "cursor_after", "A predicate over the records of a table to compare their ingestion time\r\nagainst a database cursor.", "## Syntax\r\n\r\n`cursor_after` `(` *RHS* `)`\r\n\r\n## Arguments\r\n\r\n* *RHS*: Either an empty string literal, or a valid database cursor value.\r\n\r\n## Returns\r\n\r\nA scalar value of type `bool` that indicates whether the record was ingested\r\nafter the database cursor *RHS* (`true`) or not (`false`).\r\n\r\n**Notes**\r\n\r\nSee [database cursors](../management/databasecursor.md) for additional\r\ndetails on database cursors.\r\n\r\nThis function can only be invoked on records of a table which has the\r\n[IngestionTime policy](../management/ingestiontimepolicy.md) enabled.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/cursorafterfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "cursor_before_or_at", "A predicate over the records of a table to compare their ingestion time\r\nagainst a database cursor.", "## Syntax\r\n\r\n`cursor_before_or_at` `(` *RHS* `)`\r\n\r\n## Arguments\r\n\r\n* *RHS*: Either an empty string literal, or a valid database cursor value.\r\n\r\n## Returns\r\n\r\nA scalar value of type `bool` that indicates whether the record was ingested\r\nbefore or at the database cursor *RHS* (`true`) or not (`false`).\r\n\r\n**Notes**\r\n\r\nSee [database cursors](../management/databasecursor.md) for additional\r\ndetails on database cursors.\r\n\r\nThis function can only be invoked on records of a table which has the\r\n[IngestionTime policy](../management/ingestiontimepolicy.md) enabled.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/cursorbeforeoratfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "database", "Changes the reference of the query to a specific database within the cluster scope.", "```\r\ndatabase('Sample').StormEvents\r\ncluster('help').database('Sample').StormEvents\r\n```\r\n\r\n> [!NOTE]\r\n> * For more information, see [cross-database and cross-cluster queries](cross-cluster-or-database-queries.md).\r\n> * For accessing remote cluster and remote database, see [cluster()](clusterfunction.md) scope function.\r\n\r\n## Syntax\r\n\r\n`database(`*stringConstant*`)`\r\n\r\n## Arguments\r\n\r\n* *stringConstant*: Name of the database that is referenced. Database identified can be either `DatabaseName` or `PrettyName`. Argument has to be _constant_ prior of query execution, i.e. cannot come from sub-query evaluation.", "### Use database() to access table of other database\r\n\r\n```\r\ndatabase('Samples').StormEvents | count\r\n```\r\n\r\n|Count|\r\n|---|\r\n|59066|\r\n\r\n### Use database() inside let statements \r\n\r\nThe same query as above can be rewritten to use inline function (let statement) that \r\nreceives a parameter `dbName` - which is passed into the database() function.\r\n\r\n```\r\nlet foo = (dbName:string)\r\n{\r\n    database(dbName).StormEvents | count\r\n};\r\nfoo('help')\r\n```\r\n\r\n|Count|\r\n|---|\r\n|59066|\r\n\r\n### Use database() inside functions \r\n\r\nThe same query as above can be rewritten to be used in a function that \r\nreceives a parameter `dbName` - which is passed into the database() function.\r\n\r\n```\r\n.create function foo(dbName:string)\r\n{\r\n    database(dbName).StormEvents | count\r\n};\r\n```\r\n\r\n> [!NOTE]\r\n> Such functions can be used only locally and not in the cross-cluster query.", "https://docs.microsoft.com/azure/data-explorer/kusto/query/databasefunction", "databasefunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "datatable", "Returns a table whose schema and values are defined in the query itself.", "> [!NOTE]\r\n> This operator doesn't have a pipeline input.\r\n\r\n## Syntax\r\n\r\n`datatable` `(` *ColumnName* `:` *ColumnType* [`,` ...] `)` `[` *ScalarValue* [`,` *ScalarValue* ...] `]`\r\n\r\n## Arguments\r\n\r\n* *ColumnName*, *ColumnType*: These arguments define the schema of the table. The arguments use the same syntax as used when defining a table.\r\n  For more information, see [.create table](../management/create-table-command.md)).\r\n* *ScalarValue*: A constant scalar value to insert into the table. The number of values\r\n  must be an integer multiple of the columns in the table. The *n*'th value\r\n  must have a type that corresponds to column *n* % *NumColumns*.\r\n\r\n\r\n## Returns\r\n\r\nThis operator returns a data table of the given schema and data.", "```\r\ndatatable(Date:datetime, Event:string, MoreData:dynamic) [\r\n    datetime(1910-06-11), \"Born\", dynamic({\"key1\":\"value1\", \"key2\":\"value2\"}),\r\n    datetime(1930-01-01), \"Enters Ecole Navale\", dynamic({\"key1\":\"value3\", \"key2\":\"value4\"}),\r\n    datetime(1953-01-01), \"Published first book\", dynamic({\"key1\":\"value5\", \"key2\":\"value6\"}),\r\n    datetime(1997-06-25), \"Died\", dynamic({\"key1\":\"value7\", \"key2\":\"value8\"}),\r\n]\r\n| where strlen(Event) > 4\r\n| extend key2 = MoreData.key2\r\n```\r\n\r\nResult:\r\n\r\n|Date|Event|MoreData|key2|\r\n|---|---|---|---|\r\n|1930-01-01 00:00:00.0000000|Enters Ecole Navale|{<br>  \"key1\": \"value3\",<br>  \"key2\": \"value4\"<br>}|value4|\r\n|1953-01-01 00:00:00.0000000|Published first book|{<br>  \"key1\": \"value5\",<br>  \"key2\": \"value6\"<br>}|value6|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/datatableoperator", "datatableoperator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "datetime_add", "Calculates a new [datetime](./scalar-data-types/datetime.md) from a specified datepart multiplied by a specified amount, added to a specified [datetime](./scalar-data-types/datetime.md).", "## Syntax\r\n\r\n`datetime_add(`*period*`,`*amount*`,`*datetime*`)`\r\n\r\n## Arguments\r\n\r\n* `period`: [string](./scalar-data-types/string.md). \r\n* `amount`: [integer](./scalar-data-types/int.md).\r\n* `datetime`: [datetime](./scalar-data-types/datetime.md) value.\r\n\r\nPossible values of *period*: \r\n\r\n- Year\r\n- Quarter\r\n- Month\r\n- Week\r\n- Day\r\n- Hour\r\n- Minute\r\n- Second\r\n- Millisecond\r\n- Microsecond\r\n- Nanosecond\r\n\r\n## Returns\r\n\r\nA date after a certain time/date interval has been added.", "```\r\nprint  year = datetime_add('year',1,make_datetime(2017,1,1)),\r\nquarter = datetime_add('quarter',1,make_datetime(2017,1,1)),\r\nmonth = datetime_add('month',1,make_datetime(2017,1,1)),\r\nweek = datetime_add('week',1,make_datetime(2017,1,1)),\r\nday = datetime_add('day',1,make_datetime(2017,1,1)),\r\nhour = datetime_add('hour',1,make_datetime(2017,1,1)),\r\nminute = datetime_add('minute',1,make_datetime(2017,1,1)),\r\nsecond = datetime_add('second',1,make_datetime(2017,1,1))\r\n\r\n```\r\n\r\n|year|quarter|month|week|day|hour|minute|second|\r\n|---|---|---|---|---|---|---|---|\r\n|2018-01-01 00:00:00.0000000|2017-04-01 00:00:00.0000000|2017-02-01 00:00:00.0000000|2017-01-08 00:00:00.0000000|2017-01-02 00:00:00.0000000|2017-01-01 01:00:00.0000000|2017-01-01 00:01:00.0000000|2017-01-01 00:00:01.0000000|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/datetime-addfunction", "datetime-addfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "datetime_diff", "Calculates calendarian difference between two [datetime](./scalar-data-types/datetime.md) values.", "## Syntax\r\n\r\n`datetime_diff(`*period*`,`*datetime_1*`,`*datetime_2*`)`\r\n\r\n## Arguments\r\n\r\n* `period`: `string`. \r\n* `datetime_1`: [datetime](./scalar-data-types/datetime.md) value.\r\n* `datetime_2`: [datetime](./scalar-data-types/datetime.md) value.\r\n\r\nPossible values of *period*: \r\n\r\n- Year\r\n- Quarter\r\n- Month\r\n- Week\r\n- Day\r\n- Hour\r\n- Minute\r\n- Second\r\n- Millisecond\r\n- Microsecond\r\n- Nanosecond\r\n\r\n## Returns\r\n\r\nAn integer, which represents amount of `periods` in the result of subtraction (`datetime_1` - `datetime_2`).", "```\r\nprint\r\nyear = datetime_diff('year',datetime(2017-01-01),datetime(2000-12-31)),\r\nquarter = datetime_diff('quarter',datetime(2017-07-01),datetime(2017-03-30)),\r\nmonth = datetime_diff('month',datetime(2017-01-01),datetime(2015-12-30)),\r\nweek = datetime_diff('week',datetime(2017-10-29 00:00),datetime(2017-09-30 23:59)),\r\nday = datetime_diff('day',datetime(2017-10-29 00:00),datetime(2017-09-30 23:59)),\r\nhour = datetime_diff('hour',datetime(2017-10-31 01:00),datetime(2017-10-30 23:59)),\r\nminute = datetime_diff('minute',datetime(2017-10-30 23:05:01),datetime(2017-10-30 23:00:59)),\r\nsecond = datetime_diff('second',datetime(2017-10-30 23:00:10.100),datetime(2017-10-30 23:00:00.900)),\r\nmillisecond = datetime_diff('millisecond',datetime(2017-10-30 23:00:00.200100),datetime(2017-10-30 23:00:00.100900)),\r\nmicrosecond = datetime_diff('microsecond',datetime(2017-10-30 23:00:00.1009001),datetime(2017-10-30 23:00:00.1008009)),\r\nnanosecond = datetime_diff('nanosecond',datetime(2017-10-30 23:00:00.0000000),datetime(2017-10-30 23:00:00.0000007))\r\n```\r\n\r\n|year|quarter|month|week|day|hour|minute|second|millisecond|microsecond|nanosecond|\r\n|---|---|---|---|---|---|---|---|---|---|---|\r\n|17|2|13|5|29|2|5|10|100|100|-700|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/datetime-difffunction", "datetime-difffunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "datetime_part", "Extracts the requested date part as an integer value.", "```\r\ndatetime_part(\"Day\",datetime(2015-12-14))\r\n```\r\n\r\n## Syntax\r\n\r\n`datetime_part(`*part*`,`*datetime*`)`\r\n\r\n## Arguments\r\n\r\n* `date`: `datetime`\r\n* `part`: `string`\r\n\r\nPossible values of `part`: \r\n\r\n* Year\r\n* Quarter\r\n* Month\r\n* week_of_year\r\n* Day\r\n* DayOfYear\r\n* Hour\r\n* Minute\r\n* Second\r\n* Millisecond\r\n* Microsecond\r\n* Nanosecond\r\n\r\n## Returns\r\n\r\nAn integer representing the extracted part.\r\n\r\n> [!NOTE]\r\n> `week_of_year` returns an integer which represents the week number. The week number is calculated from the first week of a year, which is the one that includes the first Thursday.", "```\r\nlet dt = datetime(2017-10-30 01:02:03.7654321); \r\nprint \r\nyear = datetime_part(\"year\", dt),\r\nquarter = datetime_part(\"quarter\", dt),\r\nmonth = datetime_part(\"month\", dt),\r\nweekOfYear = datetime_part(\"week_of_year\", dt),\r\nday = datetime_part(\"day\", dt),\r\ndayOfYear = datetime_part(\"dayOfYear\", dt),\r\nhour = datetime_part(\"hour\", dt),\r\nminute = datetime_part(\"minute\", dt),\r\nsecond = datetime_part(\"second\", dt),\r\nmillisecond = datetime_part(\"millisecond\", dt),\r\nmicrosecond = datetime_part(\"microsecond\", dt),\r\nnanosecond = datetime_part(\"nanosecond\", dt)\r\n\r\n```\r\n\r\n|year|quarter|month|weekOfYear|day|dayOfYear|hour|minute|second|millisecond|microsecond|nanosecond|\r\n|---|---|---|---|---|---|---|---|---|---|---|---|\r\n|2017|4|10|44|30|303|1|2|3|765|765432|765432100|\r\n\r\n> [!NOTE]\r\n> `weekofyear` is an obsolete variant of `week_of_year` part. `weekofyear` was not ISO 8601 compliant; the first week of a year was defined as the week with the year's first Wednesday in it.\r\n> `week_of_year` is ISO 8601 compliant; the first week of a year is defined as the week with the year's first Thursday in it. [For more information](https://en.wikipedia.org/wiki/ISO_8601#Week_dates).", "https://docs.microsoft.com/azure/data-explorer/kusto/query/datetime-partfunction", "datetime-partfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "dayofmonth", "Returns the integer number representing the day number of the given month", "```\r\ndayofmonth(datetime(2015-12-14)) == 14\r\n```\r\n\r\n## Syntax\r\n\r\n`dayofmonth(`*a_date*`)`\r\n\r\n## Arguments\r\n\r\n* `a_date`: A `datetime`.\r\n\r\n## Returns\r\n\r\n`day number` of the given month.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/dayofmonthfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "dayofweek", "Returns the integer number of days since the preceding Sunday, as a `timespan`.", "```\r\ndayofweek(datetime(2015-12-14)) == 1d  // Monday\r\n```\r\n\r\n## Syntax\r\n\r\n`dayofweek(`*a_date*`)`\r\n\r\n## Arguments\r\n\r\n* `a_date`: A `datetime`.\r\n\r\n## Returns\r\n\r\nThe `timespan` since midnight at the beginning of the preceding Sunday, rounded down to an integer number of days.", "```\r\ndayofweek(datetime(1947-11-30 10:00:05))  // time(0.00:00:00), indicating Sunday\r\ndayofweek(datetime(1970-05-11))           // time(1.00:00:00), indicating Monday\r\n```", "https://docs.microsoft.com/azure/data-explorer/kusto/query/dayofweekfunction", "dayofweekfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "dayofyear", "Returns the integer number represents the day number of the given year.", "```\r\ndayofyear(datetime(2015-12-14))\r\n```\r\n\r\n## Syntax\r\n\r\n`dayofyear(`*a_date*`)`\r\n\r\n## Arguments\r\n\r\n* `a_date`: A `datetime`.\r\n\r\n## Returns\r\n\r\n`day number` of the given year.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/dayofyearfunction", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "dcount", "Returns an estimate for the number of distinct values that are taken by a scalar expression in the summary group.", "> [!NOTE]\r\n> The `dcount()` aggregation function is primarily useful for estimating the cardinality of huge sets. It trades performance for accuracy, and may return a result that varies between executions. The order of inputs may have an effect on its output.\r\n\r\n## Syntax\r\n\r\n`dcount` `(`*Expr*[`,` *Accuracy*]`)`\r\n\r\n## Arguments\r\n\r\n* *Expr*: A scalar expression whose distinct values are to be counted.\r\n* *Accuracy*: An optional `int` literal that defines the requested estimation accuracy. See below for supported values. If unspecified, the default value\r\n  `1` is used.\r\n\r\n## Returns\r\n\r\nReturns an estimate of the number of distinct values of *`Expr`* in the group.", "```\r\nPageViewLog | summarize countries=dcount(country) by continent\r\n```\r\n\r\n![D count.](images/dcount-aggfunction/dcount.png)\r\nGet an exact count of distinct values of `V` grouped by `G`.\r\n\r\n```\r\nT | summarize by V, G | summarize count() by G\r\n```\r\n\r\nThis calculation requires a great amount of internal memory, since distinct values of `V` are multiplied by the number of distinct values of `G`.\r\nIt may result in memory errors or large execution times. \r\n`dcount()`provides a fast and reliable alternative:\r\n\r\n```\r\nT | summarize dcount(V) by G | count\r\n```\r\n\r\n## Estimation accuracy\r\n\r\nThe `dcount()` aggregate function uses a variant of the [HyperLogLog (HLL) algorithm](https://en.wikipedia.org/wiki/HyperLogLog),\r\nwhich does a stochastic estimation of set cardinality. The algorithm provides a \"knob\" that can be used to balance accuracy and execution time per memory size:\r\n\r\n|Accuracy|Error (%)|Entry count   |\r\n|--------|---------|--------------|\r\n|       0|      1.6|2<sup>12</sup>|\r\n|       1|      0.8|2<sup>14</sup>|\r\n|       2|      0.4|2<sup>16</sup>|\r\n|       3|     0.28|2<sup>17</sup>|\r\n|       4|      0.2|2<sup>18</sup>|\r\n\r\n> [!NOTE]\r\n> The \"entry count\" column is the number of 1-byte counters in the HLL implementation.\r\n\r\nThe algorithm includes some provisions for doing a perfect count (zero error), if the set cardinality is small enough:\r\n\r\n* When the accuracy level is `1`, 1000 values are returned\r\n* When the accuracy level is `2`, 8000 values are returned\r\n\r\n\r\nThe error bound is probabilistic, not a theoretical bound. The value is the standard deviation of error distribution (the sigma), and 99.7% of the estimations will have a relative error of under 3 x sigma.\r\n\r\nThe following image shows the probability distribution function of the relative\r\nestimation error, in percentages, for all supported accuracy settings:\r\n\r\n![hll error distribution.](images/dcount-aggfunction/hll-error-distribution.png)", "https://docs.microsoft.com/azure/data-explorer/kusto/query/dcount-aggfunction", "dcount-aggfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "dcount_hll", "Calculates the dcount from hll results (that were generated by [hll](hll-aggfunction.md) or [hll_merge](hll-merge-aggfunction.md)).", "Read about the [underlying algorithm (*H*yper*L*og*L*og) and estimation accuracy](dcount-aggfunction.md#estimation-accuracy).\r\n\r\n## Syntax\r\n\r\n`dcount_hll(`*Expr*`)`\r\n\r\n## Arguments\r\n\r\n* *Expr*: Expression that was generated by [hll](hll-aggfunction.md) or [hll-merge](hll-merge-aggfunction.md)\r\n\r\n## Returns\r\n\r\nThe distinct count of each value in *Expr*", "```\r\nStormEvents\r\n| summarize hllRes = hll(DamageProperty) by bin(StartTime,10m)\r\n| summarize hllMerged = hll_merge(hllRes)\r\n| project dcount_hll(hllMerged)\r\n```\r\n\r\n|dcount_hll_hllMerged|\r\n|---|\r\n|315|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/dcount-hllfunction", "dcount-hllfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "dcount_intersect", "Calculates intersection between N sets based on `hll` values (N in range of [2..16]), and returns N `dcount` values. The plugin is invoked with the [`evaluate`](evaluateoperator.md) operator.", "Given sets S<sub>1</sub>, S<sub>2</sub>, .. S<sub>n</sub> - returns values will be representing distinct counts of:  \r\nS<sub>1</sub>,  \r\nS<sub>1</sub> \u2229 S<sub>2</sub>,  \r\nS<sub>1</sub> \u2229 S<sub>2</sub> \u2229 S<sub>3</sub>,  \r\n... ,  \r\nS<sub>1</sub> \u2229 S<sub>2</sub> \u2229 ... \u2229 S<sub>n</sub>\r\n\r\n```\r\nT | evaluate dcount_intersect(hll_1, hll_2, hll_3)\r\n```\r\n\r\n## Syntax\r\n\r\n*T* `| evaluate` `dcount_intersect(`*hll_1*, *hll_2*, [`,` *hll_3*`,` ...]`)`\r\n\r\n## Arguments\r\n\r\n* *T*: The input tabular expression.\r\n* *hll_i*: the values of set S<sub>i</sub> calculated with [`hll()`](./hll-aggfunction.md) function.\r\n\r\n## Returns\r\n\r\nReturns a table with N `dcount` values (per column, representing set intersections).\r\nColumn names are s0, s1, ... (until n-1).", "```\r\n// Generate numbers from 1 to 100\r\nrange x from 1 to 100 step 1\r\n| extend isEven = (x % 2 == 0), isMod3 = (x % 3 == 0), isMod5 = (x % 5 == 0)\r\n// Calculate conditional HLL values (note that '0' is included in each of them as additional value, so we will subtract it later)\r\n| summarize hll_even = hll(iif(isEven, x, 0), 2),\r\n            hll_mod3 = hll(iif(isMod3, x, 0), 2),\r\n            hll_mod5 = hll(iif(isMod5, x, 0), 2) \r\n// Invoke the plugin that calculates dcount intersections         \r\n| evaluate dcount_intersect(hll_even, hll_mod3, hll_mod5)\r\n| project evenNumbers = s0 - 1,             //                             100 / 2 = 50\r\n          even_and_mod3 = s1 - 1,           // gcd(2,3) = 6, therefor:     100 / 6 = 16\r\n          even_and_mod3_and_mod5 = s2 - 1   // gcd(2,3,5) is 30, therefore: 100 / 30 = 3 \r\n```\r\n\r\n|evenNumbers|even_and_mod3|even_and_mod3_and_mod5|\r\n|---|---|---|\r\n|50|16|3|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/dcount-intersect-plugin", "dcount-intersect-plugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "dcountif", "Returns an estimate of the number of distinct values of *Expr* of rows for which *Predicate* evaluates to `true`.", "* Can be used only in context of aggregation inside [summarize](summarizeoperator.md).\r\n\r\nRead about the [estimation accuracy](dcount-aggfunction.md#estimation-accuracy).\r\n\r\n## Syntax\r\n\r\n`dcountif` `(`*Expr*, *Predicate*, [`,` *Accuracy*]`)`\r\n\r\n## Arguments\r\n\r\n* *Expr*: Expression that will be used for aggregation calculation.\r\n* *Predicate*: Expression that will be used to filter rows.\r\n* *Accuracy*, if specified, controls the balance between speed and accuracy.\r\n    * `0` = the least accurate and fastest calculation. 1.6% error\r\n    * `1` = the default, which balances accuracy and calculation time; about 0.8% error.\r\n    * `2` = accurate and slow calculation; about 0.4% error.\r\n    * `3` = extra accurate and slow calculation; about 0.28% error.\r\n    * `4` = super accurate and slowest calculation; about 0.2% error.\r\n\t\r\n\r\n## Returns\r\n\r\nReturns an estimate of the number of distinct values of *Expr*  of rows for which *Predicate* evaluates to `true` in the group.", "```\r\nPageViewLog | summarize countries=dcountif(country, country startswith \"United\") by continent\r\n```\r\n\r\n**Tip: Offset error**\r\n\r\n`dcountif()` might result in a one-off error in the edge cases where all rows\r\npass, or none of the rows pass, the `Predicate` expression", "https://docs.microsoft.com/azure/data-explorer/kusto/query/dcountif-aggfunction", "dcountif-aggfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "degrees", "Converts angle value in radians into value in degrees, using formula `degrees = (180 / PI ) * angle_in_radians`", "## Syntax\r\n\r\n`degrees(`*a*`)`\r\n\r\n## Arguments\r\n\r\n* *a*: Angle in radians (a real number).\r\n\r\n## Returns\r\n\r\n* The corresponding angle in degrees for an angle specified in radians.", "```\r\nprint degrees0 = degrees(pi()/4), degrees1 = degrees(pi()*1.5), degrees2 = degrees(0)\r\n\r\n```\r\n\r\n|degrees0|degrees1|degrees2|\r\n|---|---|---|\r\n|45|270|0|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/degreesfunction", "degreesfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "diff", "Compares two data sets of the same structure and finds patterns of discrete attributes (dimensions) that characterize differences between the two data sets. The plugin is invoked with the [`evaluate`](evaluateoperator.md) operator.", "`Diffpatterns` was developed to help analyze failures (for example, by comparing failures to non-failures in a given time frame), but can potentially find differences between any two data sets of the same structure. \r\n\r\n> [!NOTE]\r\n> `diffpatterns` aims to find significant patterns (that capture portions of the data difference between the sets) and isn't meant for row-by-row differences.\r\n\r\n## Syntax\r\n\r\n`T | evaluate diffpatterns(SplitColumn, SplitValueA, SplitValueB [,\u00a0WeightColumn,\u00a0Threshold, MaxDimensions,\u00a0CustomWildcard, ...])` \r\n\r\n## Arguments \r\n\r\n### Required arguments\r\n\r\n* SplitColumn - *column_name*\r\n    Tells the algorithm how to split the query into data sets. According to the specified values for the SplitValueA and SplitValueB arguments (see below), the algorithm splits the query into two data sets, \u201cA\u201d and \u201cB\u201d, and analyze the differences between them. As such, the split column must have at least two distinct values.\r\n* SplitValueA - *string*\r\n    A string representation of one of the values in the SplitColumn that was specified. All the rows that have this value in their SplitColumn considered as data set \u201cA\u201d.\r\n* SplitValueB - *string*\r\n    A string representation of one of the values in the SplitColumn that was specified. All the rows that have this value in their SplitColumn considered as data set  \u201cB\u201d.\r\n    Example: `T | extend splitColumn=iff(request_responseCode == 200, \"Success\" , \"Failure\") | evaluate diffpatterns(splitColumn, \"Success\",\"Failure\") `\r\n\r\n### Optional arguments\r\n\r\nAll other arguments are optional, but they must be ordered as below. To indicate that the default value should be used, put the string tilde value - '~' (see examples below).\r\n\r\n* WeightColumn - *column_name*\r\n    Considers each row in the input according to the specified weight (by default each row has a weight of '1'). The argument must be a name of a numeric column (for example, `int`, `long`, `real`).\r\n    A common usage of a weight column is to take into account sampling or bucketing/aggregation of the data that is already embedded into each row.\r\n    \r\n    Example: `T | extend splitColumn=iff(request_responseCode == 200, \"Success\" , \"Failure\") | evaluate diffpatterns(splitColumn, \"Success\",\"Failure\", sample_Count) `\r\n* Threshold - 0.015 < *double* < 1 [default: 0.05]\r\n    Sets the minimal pattern (ratio) difference between the two sets.\r\n    Example:  `T | extend splitColumn = iff(request-responseCode == 200, \"Success\" , \"Failure\") | evaluate diffpatterns(splitColumn, \"Success\",\"Failure\", \"~\", 0.04)`\r\n* MaxDimensions  - 0 < *int* [default: unlimited]\r\n    Sets the maximum number of uncorrelated dimensions per result pattern. By specifying a limit, you decrease the query runtime.\r\n    Example:  `T | extend splitColumn = iff(request-responseCode == 200, \"Success\" , \"Failure\") | evaluate diffpatterns(splitColumn, \"Success\",\"Failure\", \"~\", \"~\", 3)`\r\n* CustomWildcard - *\"any-value-per-type\"*\r\n    Sets the wildcard value for a specific type in the result table that will indicate that the current pattern doesn't have a restriction on this column.\r\n    Default is null, for string default is an empty string. If the default is a viable value in the data, a different wildcard value should be used (for example, `*`).\r\n    See an example below.\r\n    Example: `T | extend splitColumn = iff(request-responseCode == 200, \"Success\" , \"Failure\") | evaluate diffpatterns(splitColumn, \"Success\",\"Failure\", \"~\", \"~\", \"~\", int(-1), double(-1), long(0), datetime(1900-1-1))`\r\n\r\n## Returns\r\n\r\n`Diffpatterns` returns a small set of patterns that capture different portions of the data in the two sets (that is, a pattern capturing a large percentage of the rows in the first data set and low percentage of the rows in the second set). Each pattern is represented by a row in the results.\r\n\r\nThe result of `diffpatterns` returns the following columns:\r\n\r\n* SegmentId: the identity assigned to the pattern in the current query (note: IDs are not guaranteed to be the same in repeating queries).\r\n* CountA: the number of rows captured by the pattern in Set A (Set A is the equivalent of `where tostring(splitColumn) == SplitValueA`).\r\n* CountB: the number of rows captured by the pattern in Set B (Set B is the equivalent of `where tostring(splitColumn) == SplitValueB`).\r\n* PercentA: the percentage of rows in Set A captured by the pattern (100.0 * CountA / count(SetA)).\r\n* PercentB: the percentage of rows in Set B captured by the pattern (100.0 * CountB / count(SetB)).\r\n* PercentDiffAB: the absolute percentage point difference between A and B (|PercentA - PercentB|) is the main measure of significance of patterns in describing the difference between the two sets.\r\n* Rest of the columns: are the original schema of the input and describe the pattern, each row (pattern) reresents the intersection of the non-wildcard values of the columns (equivalent of `where col1==val1 and col2==val2 and ... colN=valN` for each non-wildcard value in the row).\r\n\r\nFor each pattern, columns that are not set in the pattern (that is, without restriction on a specific value) will contain a wildcard value, which is null by default. See in the Arguments section below how wildcards can be manually changed.\r\n\r\n* Note: the patterns are often not distinct. They may be overlapping, and usually do not cover all the original rows. Some rows may not fall under any pattern.\r\n\r\n> [!TIP]\r\n> * Use [where](./whereoperator.md) and [project](./projectoperator.md) in the input pipe to reduce the data to just what you're interested in.\r\n> * When you find an interesting row, you might want to drill into it further by adding its specific values to your `where` filter.", "```\r\nStormEvents \r\n| where monthofyear(StartTime) == 5\r\n| extend Damage = iff(DamageCrops + DamageProperty > 0 , 1 , 0)\r\n| project State , EventType , Source , Damage, DamageCrops\r\n| evaluate diffpatterns(Damage, \"0\", \"1\" )\r\n```\r\n\r\n|SegmentId|CountA|CountB|PercentA|PercentB|PercentDiffAB|State|EventType|Source|DamageCrops|\r\n|---|---|---|---|---|---|---|---|---|---|\r\n|0|2278|93|49.8|7.1|42.7||Hail||0|\r\n|1|779|512|17.03|39.08|22.05||Thunderstorm Wind|||\r\n|2|1098|118|24.01|9.01|15|||Trained Spotter|0|\r\n|3|136|158|2.97|12.06|9.09|||Newspaper||\r\n|4|359|214|7.85|16.34|8.49||Flash Flood|||\r\n|5|50|122|1.09|9.31|8.22|IOWA||||\r\n|6|655|279|14.32|21.3|6.98|||Law Enforcement||\r\n|7|150|117|3.28|8.93|5.65||Flood|||\r\n|8|362|176|7.91|13.44|5.52|||Emergency Manager||\r\n\r\n---\r\n\r\n> [!IMPORTANT]\r\n> The content below is Microsoft confidential(not available in the [public site](https://docs.microsoft.com/en-us/azure/data-explorer/))\r\n\r\n# diffpatterns plugin - Internal\r\n\r\nIf you have questions about diffpatterns or other ML related workflows in Kusto you can post them on the following DL: [KustoML](mailto:kustoML@microsoft.com).", "https://docs.microsoft.com/azure/data-explorer/kusto/query/diffpatternsplugin", "diffpatternsplugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "diffpatterns_text", "Compares two data sets of string values and finds text patterns that characterize differences between the two data sets. The plugin is invoked with the [`evaluate`](evaluateoperator.md) operator.", "The `diffpatterns_text` returns a set of text patterns that capture different portions of the data in the two sets (i.e. a pattern capturing a large percentage of the rows when the condition is `true` and low percentage of the rows when the condition is `false`). The patterns are built from consecutive tokens (separated by white space), with a token from the text column or a `*` representing a wildcard. Each pattern is represented by a row in the results.\r\n\r\n## Syntax\r\n\r\n`T | evaluate diffpatterns_text(`TextColumn, BooleanCondition [, MinTokens,\u00a0Threshold , MaxTokens]`)` \r\n\r\n## Arguments\r\n\r\n### Required arguments\r\n\r\n* TextColumn - *column_name*\r\n    The text column to analyze, must be of type string.\r\n    \r\n* BooleanCondition - *Boolean expression*\r\n    Defines how to generate the two record subsets to compare to the input table. The algorithm splits the query into two data sets, \u201cTrue\u201d and \u201cFalse\u201d according to the condition, then analyzes the (text) differences between them. \r\n\r\n### Optional arguments\r\n\r\nAll other arguments are optional, but they must be ordered as below. \r\n\r\n* MinTokens  - 0 < *int* < 200 [default: 1]\r\n    Sets the minimal number of non-wildcard tokens per result pattern.\r\n* Threshold - 0.015 < *double* < 1 [default: 0.05]\r\n    Sets the minimal pattern (ratio) difference between the two sets (see [diffpatterns](diffpatternsplugin.md)).\r\n* MaxTokens  - 0 < *int* [default: 20]\r\n    Sets the maximal number of tokens (from the beginning) per result pattern, specifying a lower limit decreases the query runtime.\r\n\r\n## Returns\r\n\r\nThe result of diffpatterns_text returns the following columns:\r\n\r\n* Count_of_True: The number of rows matching the pattern when the condition is `true`.\r\n* Count_of_False: The number of rows matching the pattern when the condition is `false`.\r\n* Percent_of_True: The percentage of rows matching the pattern from the rows when the condition is `true`.\r\n* Percent_of_False: The percentage of rows matching the pattern from the rows when the condition is `false`.\r\n* Pattern: The text pattern containing tokens from the text string and '`*`' for wildcards. \r\n\r\n> [!NOTE]\r\n> The patterns aren't necessarily distinct and may not provide full coverage of the data set. The patterns may be overlapping and some rows may not match any pattern.", "The following example uses data from the StormEvents table in the help cluster. To access this data, sign in to [https://dataexplorer.azure.com/clusters/help/databases/Samples](https://dataexplorer.azure.com/clusters/help/databases/Samples). In the left menu, browse to **help** > **Samples** > **Tables** > **Storm_Events**.\r\n\r\n```\r\nStormEvents     \r\n| where EventNarrative != \"\" and monthofyear(StartTime) > 1 and monthofyear(StartTime) < 9\r\n| where EventType == \"Drought\" or EventType == \"Extreme Cold/Wind Chill\"\r\n| evaluate diffpatterns_text(EpisodeNarrative, EventType == \"Extreme Cold/Wind Chill\", 2)\r\n```\r\n\r\n|Count_of_True|Count_of_False|Percent_of_True|Percent_of_False|Pattern|\r\n|---|---|---|---|---|\r\n|11|0|6.29|0|Winds shifting northwest in * wake * a surface trough brought heavy lake effect snowfall downwind * Lake Superior from|\r\n|9|0|5.14|0|Canadian high pressure settled * * region * produced the coldest temperatures since February * 2006. Durations * freezing temperatures|\r\n|0|34|0|6.24|* * * * * * * * * * * * * * * * * * West Tennessee,|\r\n|0|42|0|7.71|* * * * * * caused * * * * * * * * across western Colorado. *|\r\n|0|45|0|8.26|* * below normal *|\r\n|0|110|0|20.18|Below normal *|\r\n\r\n---\r\n\r\n> [!IMPORTANT]\r\n> The content below is Microsoft confidential(not available in the [public site](https://docs.microsoft.com/en-us/azure/data-explorer/))\r\n\r\n# diffpatterns_text plugin - Internal\r\n\r\nIf you have questions about diffpatterns_text or other ML related workflows in Kusto you can post them on the following DL: [KustoML](mailto:kustoML@microsoft.com).", "https://docs.microsoft.com/azure/data-explorer/kusto/query/diffpatterns-textplugin", "diffpatterns-textplugin.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "distinct", "Produces a table with the distinct combination of the provided columns of the input table.", "```\r\nT | distinct Column1, Column2\r\n```\r\n\r\nProduces a table with the distinct combination of all columns in the input table.\r\n\r\n```\r\nT | distinct *\r\n```", "Shows the distinct combination of fruit and price.\r\n\r\n```\r\nTable | distinct fruit, price\r\n```\r\n\r\n![Two tables. One has suppliers, fruit types, and prices, with some fruit-price combinations repeated. The second table lists only unique combinations.](images/distinctoperator/distinct.PNG)\r\n**Notes**\r\n\r\n* Unlike `summarize by ...`, the `distinct` operator supports providing an asterisk (`*`) as the group key, making it easier to use for wide tables.\r\n* If the group by keys are of high cardinalities, using `summarize by ...` with the [shuffle strategy](shufflequery.md) could be useful.", "https://docs.microsoft.com/azure/data-explorer/kusto/query/distinctoperator", "distinctoperator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "dynamic_to_json", "Converts a scalar value of type `dynamic` to a canonical `string` representation.", "## Syntax\r\n\r\n`dynamic_to_json(Expr)`\r\n\r\n## Arguments\r\n\r\n* *Expr*: Expression of `dynamic` type. The function accepts one argument.\r\n\r\n## Returns\r\n\r\nReturns a canonical representation of the input as a value of type `string`,\r\naccording to the following rules:\r\n\r\n* If the input is a scalar value of type other than `dynamic`,\r\n   the output is the application of `tostring()` to that value.\r\n* If the input in an array of values, the output is composed of the\r\n   characters `[`, `,`, and `]` interspersed with the canonical representation\r\n   described here of each array element.\r\n* If the input is a property bag, the output is composed of the characters\r\n   `{`, `,`, and `}` interspersed with the colon (`:`)-delimited name/value pairs\r\n   of the properties. The pairs are sorted by the names, and the values\r\n   are in the canonical representation described here of each array element.", "Expression:\r\n\r\n```\r\nlet bag1 = dynamic_to_json(\r\n  dynamic({\r\n    'Y10':dynamic({}),\r\n    'X8': dynamic({\r\n      'c3':1,\r\n      'd8':5,\r\n      'a4':6\r\n    }),\r\n    'D1':114,\r\n    'A1':12,\r\n    'B1':2,\r\n    'C1':3,\r\n    'A14':[15, 13, 18]\r\n}));\r\nlet bag2 = dynamic_to_json(\r\n  dynamic({\r\n    'X8': dynamic({\r\n      'a4':6,\r\n      'c3':1,\r\n      'd8':5\r\n    }),\r\n    'A14':[15, 13, 18],\r\n    'C1':3,\r\n    'B1':2,\r\n    'Y10': dynamic({}),\r\n    'A1':12, 'D1':114\r\n  }));\r\nprint AreEqual=bag1 == bag2, Result=bag1\r\n```\r\n\r\nResult:\r\n\r\n|AreEqual|Result|\r\n|---|---|\r\n|true|{\"A1\":12,\"A14\":[15,13,18],\"B1\":2,\"C1\":3,\"D1\":114,\"X8\":{\"a4\":6,\"c3\":1,\"d8\":5},\"Y10\":{}}|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/dynamic-to-json-function", "dynamic-to-json-function.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "endofday", "Returns the end of the day containing the date, shifted by an offset, if provided.", "## Syntax\r\n\r\n`endofday(`*date* [`,`*offset*]`)`\r\n\r\n## Arguments\r\n\r\n* `date`: The input date.\r\n* `offset`: An optional number of offset days from the input date (integer, default - 0).\r\n\r\n## Returns\r\n\r\nA datetime representing the end of the day for the given *date* value, with the offset, if specified.", "```\r\n  range offset from -1 to 1 step 1\r\n | project dayEnd = endofday(datetime(2017-01-01 10:10:17), offset) \r\n```\r\n\r\n|dayEnd|\r\n|---|\r\n|2016-12-31 23:59:59.9999999|\r\n|2017-01-01 23:59:59.9999999|\r\n|2017-01-02 23:59:59.9999999|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/endofdayfunction", "endofdayfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "endofmonth", "Returns the end of the month containing the date, shifted by an offset, if provided.", "## Syntax\r\n\r\n`endofmonth(`*date* [`,`*offset*]`)`\r\n\r\n## Arguments\r\n\r\n* `date`: The input date.\r\n* `offset`: An optional number of offset months from the input date (integer, default - 0).\r\n\r\n## Returns\r\n\r\nA datetime representing the end of the month for the given *date* value, with the offset, if specified.", "```\r\n  range offset from -1 to 1 step 1\r\n | project monthEnd = endofmonth(datetime(2017-01-01 10:10:17), offset) \r\n```\r\n\r\n|monthEnd|\r\n|---|\r\n|2016-12-31 23:59:59.9999999|\r\n|2017-01-31 23:59:59.9999999|\r\n|2017-02-28 23:59:59.9999999|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/endofmonthfunction", "endofmonthfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "endofweek", "Returns the end of the week containing the date, shifted by an offset, if provided.", "Last day of the week is considered to be a Saturday.\r\n\r\n## Syntax\r\n\r\n`endofweek(`*date* [`,`*offset*]`)`\r\n\r\n## Arguments\r\n\r\n* `date`: The input date.\r\n* `offset`: An optional number of offset weeks from the input date (integer, default - 0).\r\n\r\n## Returns\r\n\r\nA datetime representing the end of the week for the given *date* value, with the offset, if specified.", "```\r\n  range offset from -1 to 1 step 1\r\n | project weekEnd = endofweek(datetime(2017-01-01 10:10:17), offset)  \r\n\r\n```\r\n\r\n|weekEnd|\r\n|---|\r\n|2016-12-31 23:59:59.9999999|\r\n|2017-01-07 23:59:59.9999999|\r\n|2017-01-14 23:59:59.9999999|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/endofweekfunction", "endofweekfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "endofyear", "Returns the end of the year containing the date, shifted by an offset, if provided.", "## Syntax\r\n\r\n`endofyear(`*date* [`,`*offset*]`)`\r\n\r\n## Arguments\r\n\r\n* `date`: The input date.\r\n* `offset`: An optional number of offset years from the input date (integer, default - 0).\r\n\r\n## Returns\r\n\r\nA datetime representing the end of the year for the given *date* value, with the offset, if specified.", "```\r\n  range offset from -1 to 1 step 1\r\n | project yearEnd = endofyear(datetime(2017-01-01 10:10:17), offset) \r\n```\r\n\r\n|yearEnd|\r\n|---|\r\n|2016-12-31 23:59:59.9999999|\r\n|2017-12-31 23:59:59.9999999|\r\n|2018-12-31 23:59:59.9999999|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/endofyearfunction", "endofyearfunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "endswith", "Filters a record set for data with a case-insensitive ending string.", "The following table provides a comparison of the `endswith` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`endswith`](endswith-operator.md) |RHS is a closing subsequence of LHS |No |`\"Fabrikam\" endswith \"Kam\"`|\r\n|[`!endswith`](not-endswith-operator.md) |RHS isn't a closing subsequence of LHS |No |`\"Fabrikam\" !endswith \"brik\"`|\r\n|[`endswith_cs`](endswith-cs-operator.md) |RHS is a closing subsequence of LHS |Yes |`\"Fabrikam\" endswith_cs \"kam\"`|\r\n|[`!endswith_cs`](not-endswith-cs-operator.md) |RHS isn't a closing subsequence of LHS |Yes |`\"Fabrikam\" !endswith_cs \"brik\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\nCase-insensitive operators are currently supported only for ASCII-text. For non-ASCII comparison, use the [tolower()](tolowerfunction.md) function.\r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `endswith_cs`, not `endswith`. For best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `endswith` `(`*expression*`)`   \r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State endswith \"sas\"\r\n    | where event_count > 10\r\n    | project State, event_count\r\n```\r\n\r\n**Output**\r\n\r\n|State|event_count|\r\n|-----|-----------|\r\n|KANSAS|3166|\r\n|ARKANSAS|1028|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/endswith-operator", "endswith-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "endswith_cs", "Filters a record set for data with a case-sensitive ending string.", "The following table provides a comparison of the `endswith` operators:\r\n\r\n|Operator   |Description   |Case-Sensitive  |Example (yields `true`)  |\r\n|-----------|--------------|----------------|-------------------------|\r\n|[`endswith`](endswith-operator.md) |RHS is a closing subsequence of LHS |No |`\"Fabrikam\" endswith \"Kam\"`|\r\n|[`!endswith`](not-endswith-operator.md) |RHS isn't a closing subsequence of LHS |No |`\"Fabrikam\" !endswith \"brik\"`|\r\n|[`endswith_cs`](endswith-cs-operator.md) |RHS is a closing subsequence of LHS |Yes |`\"Fabrikam\" endswith_cs \"kam\"`|\r\n|[`!endswith_cs`](not-endswith-cs-operator.md) |RHS isn't a closing subsequence of LHS |Yes |`\"Fabrikam\" !endswith_cs \"brik\"`|\r\n\r\n> [!NOTE]\r\n> The following abbreviations are used in the table above:\r\n>\r\n> * RHS = right hand side of the expression\r\n> * LHS = left hand side of the expression\r\n\r\nFor further information about other operators and to determine which operator is most appropriate for your query, see [datatype string operators](datatypes-string-operators.md). \r\n\r\n## Performance tips\r\n\r\n> [!NOTE]\r\n> Performance depends on the type of search and the structure of the data.\r\n\r\nFor faster results, use the case-sensitive version of an operator, for example, `endswith_cs`, not `endswith`. For best practices, see [Query best practices](best-practices.md).\r\n\r\n## Syntax\r\n\r\n*T* `|` `where` *col* `endswith_cs` `(`*expression*`)`\r\n\r\n## Arguments\r\n\r\n* *T* - The tabular input whose records are to be filtered.\r\n* *col* - The column to filter.\r\n* *expression* - Scalar or literal expression.\r\n\r\n## Returns\r\n\r\nRows in *T* for which the predicate is `true`.", "```\r\nStormEvents\r\n    | summarize event_count=count() by State\r\n    | where State endswith_cs \"IDA\"\r\n    | count\r\n```\r\n\r\n**Output**\r\n\r\n|Count|\r\n|-----|\r\n|1|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/endswith-cs-operator", "endswith-cs-operator.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "estimate_data_size", "Returns an estimated data size in bytes of the selected columns of the tabular expression.", "```\r\nestimate_data_size(*)\r\nestimate_data_size(Col1, Col2, Col3)\r\n```\r\n\r\n## Syntax\r\n\r\n`estimate_data_size(*)`\r\n\r\n`estimate_data_size(`*col1*`, `*col2*`, `...`)`\r\n\r\n## Arguments\r\n\r\n* *col1*, *col2*: Selection of column references in the source tabular expression that are used for data size estimation. To include all columns, use `*` (asterisk) syntax.\r\n\r\n## Returns\r\n\r\n* The estimated data size  in bytes of the record size. Estimation is based on data types and values lengths.", "Calculating total data size using `estimate_data_size()`:\r\n\r\n```\r\nrange x from 1 to 10 step 1                    // x (long) is 8 \r\n| extend Text = '1234567890'                   // Text length is 10  \r\n| summarize Total=sum(estimate_data_size(*))   // (8+10)x10 = 180\r\n```\r\n\r\n|Total|\r\n|---|\r\n|180|", "https://docs.microsoft.com/azure/data-explorer/kusto/query/estimate-data-sizefunction", "estimate-data-sizefunction.md"));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.OperatorToken, "evaluate", "Invokes a service-side query extension (plugin).", "The `evaluate` operator is a tabular operator that provides the ability to\r\ninvoke query language extensions known as **plugins**. Plugins can be enabled\r\nor disabled (unlike other language constructs, which are always available),\r\nand aren't \"bound\" by the relational nature of the language (for example, they may\r\nnot have a predefined, statically determined, output schema).\r\n\r\n> [!NOTE]\r\n> * Syntactically, `evaluate` behaves similarly to the [invoke operator](./invokeoperator.md), which invokes tabular functions.\r\n> * Plugins provided through the evaluate operator aren't bound by the regular rules of query execution or argument evaluation.\r\n> * Specific plugins may have specific restrictions. For example, plugins whose output schema depends on the data (for example, [bag_unpack plugin](./bag-unpackplugin.md) and [pivot plugin](./pivotplugin.md)) can't be used when performing cross-cluster queries.\r\n\r\n## Syntax \r\n\r\n[*T* `|`] `evaluate` [ *evaluateParameters* ] *PluginName* `(` [*PluginArg1* [`,` *PluginArg2*]... `)`\r\n\r\n## Arguments\r\n\r\n* *T* is an optional tabular input to the plugin. (Some plugins don't take\r\n  any input, and act as a tabular data source.)\r\n* *PluginName* is the mandatory name of the plugin being invoked.\r\n* *PluginArg1*, ... are the optional arguments to the plugin.\r\n* *evaluateParameters*: Zero or more (space-separated) parameters in the form of\r\n  *Name* `=` *Value* that control the behavior of the evaluate operation and execution plan. Each plugin may decide differently how to handle each parameter. Refer to each plugin's documentation for specific behavior.  \r\n\r\n## Parameters\r\n\r\nThe following parameters are supported: \r\n\r\n|Name                |Values                           |Description                                |\r\n  |--------------------|---------------------------------|-------------------------------------------|\r\n  |`hint.distribution` |`single`, `per_node`, `per_shard`| [Distribution hints](#distribution-hints) |\r\n  |`hint.pass_filters` |`true`, `false`| Allow `evaluate` operator to passthrough any matching filters before the plugin. Filter is considered as 'matched' if it refers to a column existing before the `evaluate` operator. Default: `false` |\r\n  |`hint.pass_filters_column` |*column_name*| Allow plugin operator to passthrough filters referring to *column_name* before the plugin. Parameter can be used multiple times with different column names. |\r\n\r\n## Plugins\r\n\r\nThe following plugins are supported:\r\n\r\n- [autocluster plugin](autoclusterplugin.md)\r\n- [azure-digital-twins-query-request plugin](azure-digital-twins-query-request-plugin.md)\r\n- [bag-unpack plugin](bag-unpackplugin.md)\r\n- [basket plugin](basketplugin.md)\r\n- [cosmosdb-sql-request plugin](cosmosdb-plugin.md)\r\n- [dcount-intersect plugin](dcount-intersect-plugin.md)\r\n- [diffpatterns plugin](diffpatternsplugin.md)\r\n- [diffpatterns-text plugin](diffpatterns-textplugin.md)\r\n- [infer-storage-schema plugin](inferstorageschemaplugin.md)\r\n- [ipv4-lookup plugin](ipv4-lookup-plugin.md)\r\n- [mysql-request-plugin](mysqlrequest-plugin.md)\r\n- [narrow plugin](narrowplugin.md)\r\n- [pivot plugin](pivotplugin.md)\r\n- [preview plugin](previewplugin.md)\r\n- [R plugin](rplugin.md)\r\n- [rolling-percentile plugin](rolling-percentile-plugin.md)\r\n- [rows-near plugin](rows-near-plugin.md)\r\n- [schema-merge plugin](schemamergeplugin.md)\r\n- [sql-request plugin](mysqlrequest-plugin.md)\r\n- [sequence-detect plugin](sequence-detect-plugin.md)\r\n\r\n## Distribution hints\r\n\r\nDistribution hints specify how the plugin execution will be distributed across multiple cluster nodes. Each plugin may implement a different support for the distribution. The plugin's documentation specifies the distribution options supported by the plugin.\r\n\r\nPossible values:\r\n\r\n* `single`: A single instance of the plugin will run over the entire query data.\r\n* `per_node`: If the query before the plugin call is distributed across nodes, then an instance of the plugin will run on each node over the data that it contains.\r\n* `per_shard`: If the data before the plugin call is distributed across shards, then an instance of the plugin will run over each shard of the data.", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/evaluateoperator", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "exp", "The base-e exponential function of x, which is e raised to the power x: e^x.", "## Syntax\r\n\r\n`exp(`*x*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: A real number, value of the exponent.\r\n\r\n## Returns\r\n\r\n* Exponential value of x.\r\n* For natural (base-e) logarithms, see [log()](log-function.md).\r\n* For exponential functions of base-2 and base-10 logarithms, see [exp2()](exp2-function.md), [exp10()](exp10-function.md)", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/exp-function", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "exp10", "The base-10 exponential function of x, which is 10 raised to the power x: 10^x.", "## Syntax\r\n\r\n`exp10(`*x*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: A real number, value of the exponent.\r\n\r\n## Returns\r\n\r\n* Exponential value of x.\r\n* For natural (base-10) logarithms, see [log10()](log10-function.md).\r\n* For exponential functions of base-e and base-2 logarithms, see [exp()](exp-function.md), [exp2()](exp2-function.md)", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/exp10-function", ""));
                this.AddTopic(new Kusto.Data.IntelliSense.CslTopicDocumentation(Kusto.Data.IntelliSense.CslCommandToken.Kind.FunctionNameToken, "exp2", "The base-2 exponential function of x, which is 2 raised to the power x: 2^x.", "## Syntax\r\n\r\n`exp2(`*x*`)`\r\n\r\n## Arguments\r\n\r\n* *x*: A real number, value of the exponent.\r\n\r\n## Returns\r\n\r\n* Exponential value of x.\r\n* For natural (base-2) logarithms, see [log2()](log2-function.md).\r\n* For exponential functions of base-e and base-10 logarithms, see [exp()](exp-function.md), [exp10()](exp10-function.md)", "", "https://docs.microsoft.com/azure/data-explorer/kusto/query/exp2-function", ""));
                this.AddTopic(