"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DebuggerΩ = void 0;
var tslib_1 = require("tslib");
var esprima_1 = require("esprima");
var esquery_1 = require("esquery");
var module_1 = tslib_1.__importDefault(require("module"));
var perf_hooks_1 = require("perf_hooks");
var util_1 = require("util");
var logger_1 = require("./logger");
var DebuggerΩ = /** @class */ (function () {
    function DebuggerΩ(options) {
        var _a = options.ignore, ignore = _a === void 0 ? [] : _a, _b = options.include, include = _b === void 0 ? [] : _b;
        this._include = include;
        this._ignore = ignore;
        this._logger = new logger_1.DebugLoggerΩ(options);
    }
    DebuggerΩ.prototype.shouldWrap = function (requirePath) {
        var isNodeModule = module_1.default.builtinModules.includes(requirePath) || requirePath.includes('node_modules');
        var isIncludedModule = this._include.some(function (regexp) { return regexp.test(requirePath); });
        var isIgnoredModule = this._ignore.some(function (ignore) { return ignore === requirePath; });
        var isDebuggerModule = requirePath.includes(__dirname);
        return !(isDebuggerModule || (isNodeModule && !isIncludedModule) || isIgnoredModule);
    };
    DebuggerΩ.prototype.wrap = function (module) {
        var _this = this;
        var exports = module;
        var exportFunctions = this._getFunctions(exports);
        Object.keys(exportFunctions).forEach(function (functionName) {
            Object.defineProperty(exports, functionName, {
                value: new Proxy(exports[functionName], {
                    apply: _this._createFunctionCallWrap(functionName),
                    construct: _this._createConstructorCallWrap(functionName)
                })
            });
        });
        return exports;
    };
    DebuggerΩ.prototype._createFunctionCallWrap = function (name) {
        var _this = this;
        return function (target, thisArg, args) {
            var startTime = perf_hooks_1.performance.now();
            _this._logger.start(name, args);
            var argNames = _this._getArgNames(target);
            var result = target.apply(thisArg, _this._wrapArgs(argNames, args));
            if (isPromise(result)) {
                return result.then(function (result) {
                    var endTime = perf_hooks_1.performance.now();
                    _this._logger.end(name, startTime, endTime, result);
                    return result;
                });
            }
            var endTime = perf_hooks_1.performance.now();
            _this._logger.end(name, startTime, endTime, result);
            return result;
        };
    };
    DebuggerΩ.prototype._createConstructorCallWrap = function (name) {
        var _this = this;
        return function (target, args) {
            var startTime = perf_hooks_1.performance.now();
            _this._logger.start(name, args);
            var proto = target.prototype;
            var prototypeFunctions = _this._getFunctions(proto);
            Object.keys(prototypeFunctions).forEach(function (functionName) {
                Object.defineProperty(proto, functionName, {
                    value: new Proxy(proto[functionName], {
                        apply: _this._createFunctionCallWrap(name + "." + functionName)
                    })
                });
            });
            var argNames = _this._getArgNames(target);
            var instance = new (target.bind.apply(target, tslib_1.__spreadArrays([void 0], _this._wrapArgs(argNames, args))))();
            var endTime = perf_hooks_1.performance.now();
            _this._logger.end(name, startTime, endTime, instance);
            return instance;
        };
    };
    DebuggerΩ.prototype._wrapArgs = function (argNames, args) {
        var _this = this;
        return args.map(function (arg, index) {
            if (!isFunction(arg)) {
                return arg;
            }
            return new Proxy(arg, {
                apply: _this._createFunctionCallWrap(argNames[index])
            });
        });
    };
    DebuggerΩ.prototype._getArgNames = function (target) {
        var parsed;
        try {
            parsed = esprima_1.parseScript("const f = " + target.toString());
        }
        catch (_a) {
            parsed = esprima_1.parseScript("class F { " + target.toString() + " }");
        }
        var func = esquery_1.query(parsed, '[type=/Function/]')[0];
        return func.params.map(function (param) {
            var identifier = esquery_1.query(param, 'Identifier')[0];
            return identifier.name;
        });
    };
    DebuggerΩ.prototype._getFunctions = function (map) {
        var functions = {};
        // Get all property descriptors:
        var descriptors = Object.getOwnPropertyDescriptors(map);
        // Filter out getters and setters:
        var functionNames = Object.keys(descriptors).filter(function (d) { return descriptors[d].value; });
        functionNames
            .filter(function (functionName) { return isFunction(map[functionName]) && !util_1.types.isProxy(map[functionName]); })
            .forEach(function (functionName) {
            functions[functionName] = map[functionName];
        });
        return functions;
    };
    return DebuggerΩ;
}());
exports.DebuggerΩ = DebuggerΩ;
function isPromise(value) {
    return util_1.types.isPromise(value);
}
function isFunction(value) {
    return typeof value === 'function';
}
//# sourceMappingURL=debugger.js.map