import "core-js/modules/es.array.reduce.js";
import deprecate from 'util-deprecate';
import dedent from 'ts-dedent';
import global from 'global';
import { SynchronousPromise } from 'synchronous-promise';
import Events, { IGNORED_EXCEPTION } from '@storybook/core-events';
import { logger } from '@storybook/client-logger';
import { addons } from '@storybook/addons';
import { StoryStore } from '@storybook/store';
import { UrlStore } from './UrlStore';
import { WebView } from './WebView';
const {
  window: globalWindow,
  AbortController,
  fetch
} = global;

function focusInInput(event) {
  const target = event.target;
  return /input|textarea/i.test(target.tagName) || target.getAttribute('contenteditable') !== null;
}

function createController() {
  if (AbortController) return new AbortController(); // Polyfill for IE11

  return {
    signal: {
      aborted: false
    },

    abort() {
      this.signal.aborted = true;
    }

  };
}

const STORY_INDEX_PATH = './stories.json';
export class PreviewWeb {
  constructor() {
    var _global$FEATURES;

    this.channel = void 0;
    this.serverChannel = void 0;
    this.urlStore = void 0;
    this.storyStore = void 0;
    this.view = void 0;
    this.getStoryIndex = void 0;
    this.importFn = void 0;
    this.renderToDOM = void 0;
    this.previewEntryError = void 0;
    this.previousSelection = void 0;
    this.previousStory = void 0;
    this.previousCleanup = void 0;
    this.abortController = void 0;
    this.disableKeyListeners = void 0;
    this.channel = addons.getChannel();

    if ((_global$FEATURES = global.FEATURES) !== null && _global$FEATURES !== void 0 && _global$FEATURES.storyStoreV7 && addons.hasServerChannel()) {
      this.serverChannel = addons.getServerChannel();
    }

    this.view = new WebView();
    this.urlStore = new UrlStore();
    this.storyStore = new StoryStore(); // Add deprecated APIs for back-compat
    // @ts-ignore

    this.storyStore.getSelection = deprecate(() => this.urlStore.selection, dedent`
        \`__STORYBOOK_STORY_STORE__.getSelection()\` is deprecated and will be removed in 7.0.
  
        To get the current selection, use the \`useStoryContext()\` hook from \`@storybook/addons\`.
      `);
  } // INITIALIZATION
  // NOTE: the reason that the preview and store's initialization code is written in a promise
  // style and not `async-await`, and the use of `SynchronousPromise`s is in order to allow
  // storyshots to immediately call `raw()` on the store without waiting for a later tick.
  // (Even simple things like `Promise.resolve()` and `await` involve the callback happening
  // in the next promise "tick").
  // See the comment in `storyshots-core/src/api/index.ts` for more detail.


  initialize({
    getStoryIndex,
    importFn,
    getProjectAnnotations
  }) {
    // We save these two on initialization in case `getProjectAnnotations` errors,
    // in which case we may need them later when we recover.
    this.getStoryIndex = getStoryIndex;
    this.importFn = importFn;
    this.setupListeners();
    return this.getProjectAnnotationsOrRenderError(getProjectAnnotations).then(projectAnnotations => this.initializeWithProjectAnnotations(projectAnnotations));
  }

  setupListeners() {
    var _this$serverChannel;

    globalWindow.onkeydown = this.onKeydown.bind(this);
    (_this$serverChannel = this.serverChannel) === null || _this$serverChannel === void 0 ? void 0 : _this$serverChannel.on(Events.STORY_INDEX_INVALIDATED, this.onStoryIndexChanged.bind(this));
    this.channel.on(Events.SET_CURRENT_STORY, this.onSetCurrentStory.bind(this));
    this.channel.on(Events.UPDATE_QUERY_PARAMS, this.onUpdateQueryParams.bind(this));
    this.channel.on(Events.UPDATE_GLOBALS, this.onUpdateGlobals.bind(this));
    this.channel.on(Events.UPDATE_STORY_ARGS, this.onUpdateArgs.bind(this));
    this.channel.on(Events.RESET_STORY_ARGS, this.onResetArgs.bind(this));
  }

  getProjectAnnotationsOrRenderError(getProjectAnnotations) {
    return SynchronousPromise.resolve().then(getProjectAnnotations).then(projectAnnotations => {
      this.renderToDOM = projectAnnotations.renderToDOM;

      if (!this.renderToDOM) {
        throw new Error(dedent`
            Expected your framework's preset to export a \`renderToDOM\` field.

            Perhaps it needs to be upgraded for Storybook 6.4?

            More info: https://github.com/storybookjs/storybook/blob/next/MIGRATION.md#mainjs-framework-field          
          `);
      }

      return projectAnnotations;
    }).catch(err => {
      // This is an error extracting the projectAnnotations (i.e. evaluating the previewEntries) and
      // needs to be show to the user as a simple error
      this.renderPreviewEntryError('Error reading preview.js:', err);
      throw err;
    });
  } // If initialization gets as far as project annotations, this function runs.


  initializeWithProjectAnnotations(projectAnnotations) {
    var _global$FEATURES2;

    this.storyStore.setProjectAnnotations(projectAnnotations);
    this.setInitialGlobals();
    let storyIndexPromise;

    if ((_global$FEATURES2 = global.FEATURES) !== null && _global$FEATURES2 !== void 0 && _global$FEATURES2.storyStoreV7) {
      storyIndexPromise = this.getStoryIndexFromServer();
    } else {
      if (!this.getStoryIndex) {
        throw new Error('No `getStoryIndex` passed defined in v6 mode');
      }

      storyIndexPromise = SynchronousPromise.resolve().then(this.getStoryIndex);
    }

    return storyIndexPromise.then(storyIndex => this.initializeWithStoryIndex(storyIndex)).catch(err => {
      this.renderPreviewEntryError('Error loading story index:', err);
      throw err;
    });
  }

  async setInitialGlobals() {
    const {
      globals
    } = this.urlStore.selectionSpecifier || {};

    if (globals) {
      this.storyStore.globals.updateFromPersisted(globals);
    }

    this.emitGlobals();
  }

  emitGlobals() {
    this.channel.emit(Events.SET_GLOBALS, {
      globals: this.storyStore.globals.get() || {},
      globalTypes: this.storyStore.projectAnnotations.globalTypes || {}
    });
  }

  async getStoryIndexFromServer() {
    const result = await fetch(STORY_INDEX_PATH);
    if (result.status === 200) return result.json();
    throw new Error(await result.text());
  } // If initialization gets as far as the story index, this function runs.


  initializeWithStoryIndex(storyIndex) {
    var _global$FEATURES3;

    return this.storyStore.initialize({
      storyIndex,
      importFn: this.importFn,
      cache: !((_global$FEATURES3 = global.FEATURES) !== null && _global$FEATURES3 !== void 0 && _global$FEATURES3.storyStoreV7)
    }).then(() => {
      var _global$FEATURES4;

      if (!((_global$FEATURES4 = global.FEATURES) !== null && _global$FEATURES4 !== void 0 && _global$FEATURES4.storyStoreV7)) {
        this.channel.emit(Events.SET_STORIES, this.storyStore.getSetStoriesPayload());
      }

      return this.selectSpecifiedStory();
    });
  } // Use the selection specifier to choose a story, then render it


  async selectSpecifiedStory() {
    if (!this.urlStore.selectionSpecifier) {
      this.renderMissingStory();
      return;
    }

    const {
      storySpecifier,
      viewMode,
      args
    } = this.urlStore.selectionSpecifier;
    const storyId = this.storyStore.storyIndex.storyIdFromSpecifier(storySpecifier);

    if (!storyId) {
      if (storySpecifier === '*') {
        this.renderStoryLoadingException(storySpecifier, new Error(dedent`
            Couldn't find any stories in your Storybook.
            - Please check your stories field of your main.js config.
            - Also check the browser console and terminal for error messages.
          `));
      } else {
        this.renderStoryLoadingException(storySpecifier, new Error(dedent`
            Couldn't find story matching '${storySpecifier}'.
            - Are you sure a story with that id exists?
            - Please check your stories field of your main.js config.
            - Also check the browser console and terminal for error messages.
          `));
      }

      return;
    }

    this.urlStore.setSelection({
      storyId,
      viewMode
    });
    this.channel.emit(Events.STORY_SPECIFIED, this.urlStore.selection);
    this.channel.emit(Events.CURRENT_STORY_WAS_SET, this.urlStore.selection);
    await this.renderSelection({
      persistedArgs: args
    });
  } // EVENT HANDLERS
  // This happens when a config file gets reloaded


  async onGetProjectAnnotationsChanged({
    getProjectAnnotations
  }) {
    delete this.previewEntryError;
    const projectAnnotations = await this.getProjectAnnotationsOrRenderError(getProjectAnnotations);

    if (!this.storyStore.projectAnnotations) {
      await this.initializeWithProjectAnnotations(projectAnnotations);
      return;
    }

    await this.storyStore.setProjectAnnotations(projectAnnotations);
    this.emitGlobals();
    this.renderSelection();
  }

  async onStoryIndexChanged() {
    delete this.previewEntryError;

    if (!this.storyStore.projectAnnotations) {
      // We haven't successfully set project annotations yet,
      // we need to do that before we can do anything else.
      return;
    }

    try {
      const storyIndex = await this.getStoryIndexFromServer(); // This is the first time the story index worked, let's load it into the store

      if (!this.storyStore.storyIndex) {
        await this.initializeWithStoryIndex(storyIndex);
      } // Update the store with the new stories.


      await this.onStoriesChanged({
        storyIndex
      });
    } catch (err) {
      this.renderPreviewEntryError('Error loading story index:', err);
      throw err;
    }
  } // This happens when a glob gets HMR-ed


  async onStoriesChanged({
    importFn,
    storyIndex
  }) {
    var _global$FEATURES5;

    await this.storyStore.onStoriesChanged({
      importFn,
      storyIndex
    });

    if (!((_global$FEATURES5 = global.FEATURES) !== null && _global$FEATURES5 !== void 0 && _global$FEATURES5.storyStoreV7)) {
      this.channel.emit(Events.SET_STORIES, await this.storyStore.getSetStoriesPayload());
    }

    if (this.urlStore.selection) {
      await this.renderSelection();
    } else {
      // Our selection has never applied before, but maybe it does now, let's try!
      await this.selectSpecifiedStory();
    }
  }

  onKeydown(event) {
    if (!this.disableKeyListeners && !focusInInput(event)) {
      // We have to pick off the keys of the event that we need on the other side
      const {
        altKey,
        ctrlKey,
        metaKey,
        shiftKey,
        key,
        code,
        keyCode
      } = event;
      this.channel.emit(Events.PREVIEW_KEYDOWN, {
        event: {
          altKey,
          ctrlKey,
          metaKey,
          shiftKey,
          key,
          code,
          keyCode
        }
      });
    }
  }

  onSetCurrentStory(selection) {
    this.urlStore.setSelection(selection);
    this.channel.emit(Events.CURRENT_STORY_WAS_SET, this.urlStore.selection);
    this.renderSelection();
  }

  onUpdateQueryParams(queryParams) {
    this.urlStore.setQueryParams(queryParams);
  }

  onUpdateGlobals({
    globals
  }) {
    this.storyStore.globals.update(globals);
    this.channel.emit(Events.GLOBALS_UPDATED, {
      globals: this.storyStore.globals.get(),
      initialGlobals: this.storyStore.globals.initialGlobals
    });
  }

  onUpdateArgs({
    storyId,
    updatedArgs
  }) {
    this.storyStore.args.update(storyId, updatedArgs);
    this.channel.emit(Events.STORY_ARGS_UPDATED, {
      storyId,
      args: this.storyStore.args.get(storyId)
    });
  }

  async onResetArgs({
    storyId,
    argNames
  }) {
    // NOTE: we have to be careful here and avoid await-ing when updating the current story's args.
    // That's because below in `renderStoryToElement` we have also bound to this event and will
    // render the story in the same tick.
    // However, we can do that safely as the current story is available in `this.previousStory`
    const {
      initialArgs
    } = storyId === this.previousStory.id ? this.previousStory : await this.storyStore.loadStory({
      storyId
    });
    const argNamesToReset = argNames || Object.keys(this.storyStore.args.get(storyId));
    const updatedArgs = argNamesToReset.reduce((acc, argName) => {
      acc[argName] = initialArgs[argName];
      return acc;
    }, {});
    this.onUpdateArgs({
      storyId,
      updatedArgs
    });
  } // RENDERING
  // We can either have:
  // - a story selected in "story" viewMode,
  //     in which case we render it to the root element, OR
  // - a story selected in "docs" viewMode,
  //     in which case we render the docsPage for that story


  async renderSelection({
    persistedArgs
  } = {}) {
    var _this$previousSelecti, _this$previousSelecti2, _global$FEATURES6;

    const {
      selection
    } = this.urlStore;

    if (!selection) {
      throw new Error('Cannot render story as no selection was made');
    }

    const {
      storyId
    } = selection;
    const storyIdChanged = ((_this$previousSelecti = this.previousSelection) === null || _this$previousSelecti === void 0 ? void 0 : _this$previousSelecti.storyId) !== storyId;
    const viewModeChanged = ((_this$previousSelecti2 = this.previousSelection) === null || _this$previousSelecti2 === void 0 ? void 0 : _this$previousSelecti2.viewMode) !== selection.viewMode; // Show a spinner while we load the next story

    if (selection.viewMode === 'story') {
      this.view.showPreparingStory();
    } else {
      this.view.showPreparingDocs();
    }

    let story;

    try {
      story = await this.storyStore.loadStory({
        storyId
      });
    } catch (err) {
      await this.cleanupPreviousRender();
      this.previousStory = null;
      this.renderStoryLoadingException(storyId, err);
      return;
    }

    const implementationChanged = !storyIdChanged && this.previousStory && story !== this.previousStory;

    if (persistedArgs) {
      this.storyStore.args.updateFromPersisted(story, persistedArgs);
    } // Don't re-render the story if nothing has changed to justify it


    if (this.previousStory && !storyIdChanged && !implementationChanged && !viewModeChanged) {
      this.channel.emit(Events.STORY_UNCHANGED, storyId);
      this.view.showMain();
      return;
    }

    await this.cleanupPreviousRender({
      unmountDocs: viewModeChanged
    }); // If we are rendering something new (as opposed to re-rendering the same or first story), emit

    if (this.previousSelection && (storyIdChanged || viewModeChanged)) {
      this.channel.emit(Events.STORY_CHANGED, storyId);
    } // Record the previous selection *before* awaiting the rendering, in cases things change before it is done.


    this.previousSelection = selection;
    this.previousStory = story;
    const {
      parameters,
      initialArgs,
      argTypes,
      args
    } = this.storyStore.getStoryContext(story);

    if ((_global$FEATURES6 = global.FEATURES) !== null && _global$FEATURES6 !== void 0 && _global$FEATURES6.storyStoreV7) {
      this.channel.emit(Events.STORY_PREPARED, {
        id: storyId,
        parameters,
        initialArgs,
        argTypes,
        args
      });
    } // For v6 mode / compatibility
    // If the implementation changed, or args were persisted, the args may have changed,
    // and the STORY_PREPARED event above may not be respected.


    if (implementationChanged || persistedArgs) {
      this.channel.emit(Events.STORY_ARGS_UPDATED, {
        storyId,
        args
      });
    }

    if (selection.viewMode === 'docs' || story.parameters.docsOnly) {
      this.previousCleanup = await this.renderDocs({
        story
      });
    } else {
      this.previousCleanup = this.renderStory({
        story
      });
    }
  }

  async renderDocs({
    story
  }) {
    var _global$FEATURES8;

    const {
      id,
      title,
      name
    } = story;
    const csfFile = await this.storyStore.loadCSFFileByStoryId(id);
    const docsContext = {
      id,
      title,
      name,
      // NOTE: these two functions are *sync* so cannot access stories from other CSF files
      storyById: storyId => this.storyStore.storyFromCSFFile({
        storyId,
        csfFile
      }),
      componentStories: () => this.storyStore.componentStoriesFromCSFFile({
        csfFile
      }),
      loadStory: storyId => this.storyStore.loadStory({
        storyId
      }),
      renderStoryToElement: this.renderStoryToElement.bind(this),
      getStoryContext: renderedStory => Object.assign({}, this.storyStore.getStoryContext(renderedStory), {
        viewMode: 'docs'
      })
    };

    const render = async () => {
      var _global$FEATURES7;

      const fullDocsContext = Object.assign({}, docsContext, !((_global$FEATURES7 = global.FEATURES) !== null && _global$FEATURES7 !== void 0 && _global$FEATURES7.breakingChangesV7) && this.storyStore.getStoryContext(story));
      const renderer = await import('./renderDocs');
      const element = this.view.prepareForDocs();
      renderer.renderDocs(story, fullDocsContext, element, () => this.channel.emit(Events.DOCS_RENDERED, id));
    }; // Initially render right away


    render(); // Listen to events and re-render
    // NOTE: we aren't checking to see the story args are targetted at the "right" story.
    // This is because we may render >1 story on the page and there is no easy way to keep track
    // of which ones were rendered by the docs page.
    // However, in `modernInlineRender`, the individual stories track their own events as they
    // each call `renderStoryToElement` below.

    if (!((_global$FEATURES8 = global.FEATURES) !== null && _global$FEATURES8 !== void 0 && _global$FEATURES8.modernInlineRender)) {
      this.channel.on(Events.UPDATE_GLOBALS, render);
      this.channel.on(Events.UPDATE_STORY_ARGS, render);
      this.channel.on(Events.RESET_STORY_ARGS, render);
    }

    return async () => {
      var _global$FEATURES9;

      if (!((_global$FEATURES9 = global.FEATURES) !== null && _global$FEATURES9 !== void 0 && _global$FEATURES9.modernInlineRender)) {
        this.channel.off(Events.UPDATE_GLOBALS, render);
        this.channel.off(Events.UPDATE_STORY_ARGS, render);
        this.channel.off(Events.RESET_STORY_ARGS, render);
      }
    };
  }

  renderStory({
    story
  }) {
    const element = this.view.prepareForStory(story);
    const {
      id,
      componentId,
      title,
      name
    } = story;
    const renderContext = {
      componentId,
      title,
      kind: title,
      id,
      name,
      story: name,
      showMain: () => this.view.showMain(),
      showError: err => this.renderError(id, err),
      showException: err => this.renderException(id, err)
    };
    return this.renderStoryToElement({
      story,
      renderContext,
      element,
      viewMode: 'story'
    });
  } // Render a story into a given element and watch for the events that would trigger us
  // to re-render it (plus deal sensibly with things like changing story mid-way through).


  renderStoryToElement({
    story,
    renderContext: renderContextWithoutStoryContext,
    element: canvasElement,
    viewMode
  }) {
    const {
      id,
      applyLoaders,
      unboundStoryFn,
      playFunction
    } = story;
    let notYetRendered = true;
    let phase;

    const isPending = () => ['rendering', 'playing'].includes(phase);

    this.abortController = createController();

    const render = async ({
      initial = false,
      forceRemount = false
    } = {}) => {
      if (forceRemount && !initial) {
        this.abortController.abort();
        this.abortController = createController();
      }

      const abortSignal = this.abortController.signal; // we need a stable reference to the signal

      const runPhase = async (phaseName, phaseFn) => {
        phase = phaseName;
        this.channel.emit(Events.STORY_RENDER_PHASE_CHANGED, {
          newPhase: phase,
          storyId: id
        });
        if (phaseFn) await phaseFn();

        if (abortSignal.aborted) {
          phase = 'aborted';
          this.channel.emit(Events.STORY_RENDER_PHASE_CHANGED, {
            newPhase: phase,
            storyId: id
          });
        }
      };

      try {
        let loadedContext;
        await runPhase('loading', async () => {
          loadedContext = await applyLoaders(Object.assign({}, this.storyStore.getStoryContext(story), {
            viewMode
          }));
        });
        if (abortSignal.aborted) return;
        const renderStoryContext = Object.assign({}, loadedContext, this.storyStore.getStoryContext(story), {
          abortSignal,
          canvasElement
        });
        const renderContext = Object.assign({}, renderContextWithoutStoryContext, {
          forceRemount: forceRemount || notYetRendered,
          storyContext: renderStoryContext,
          storyFn: () => unboundStoryFn(renderStoryContext),
          unboundStoryFn
        });
        await runPhase('rendering', () => this.renderToDOM(renderContext, canvasElement));
        notYetRendered = false;
        if (abortSignal.aborted) return;

        if (forceRemount && playFunction) {
          this.disableKeyListeners = true;
          await runPhase('playing', () => playFunction(renderContext.storyContext));
          await runPhase('played');
          this.disableKeyListeners = false;
          if (abortSignal.aborted) return;
        }

        await runPhase('completed', () => this.channel.emit(Events.STORY_RENDERED, id));
      } catch (err) {
        renderContextWithoutStoryContext.showException(err);
      }
    }; // Start the first (initial) render. We don't await here because we need to return the "cleanup"
    // function below right away, so if the user changes story during the first render we can cancel
    // it without having to first wait for it to finish.
    // Whenever the selection changes we want to force the component to be remounted.


    render({
      initial: true,
      forceRemount: true
    });

    const remountStoryIfMatches = ({
      storyId
    }) => {
      if (storyId === story.id) render({
        forceRemount: true
      });
    };

    const rerenderStoryIfMatches = ({
      storyId
    }) => {
      if (storyId === story.id) render();
    }; // Listen to events and re-render story
    // Don't forget to unsubscribe on cleanup


    this.channel.on(Events.UPDATE_GLOBALS, render);
    this.channel.on(Events.FORCE_RE_RENDER, render);
    this.channel.on(Events.FORCE_REMOUNT, remountStoryIfMatches);
    this.channel.on(Events.UPDATE_STORY_ARGS, rerenderStoryIfMatches);
    this.channel.on(Events.RESET_STORY_ARGS, rerenderStoryIfMatches); // Cleanup / teardown function invoked on next render (via `cleanupPreviousRender`)

    return async () => {
      // If the story is torn down (either a new story is rendered or the docs page removes it)
      // we need to consider the fact that the initial render may not be finished
      // (possibly the loaders or the play function are still running). We use the controller
      // as a method to abort them, ASAP, but this is not foolproof as we cannot control what
      // happens inside the user's code.
      this.abortController.abort();
      this.storyStore.cleanupStory(story);
      this.channel.off(Events.UPDATE_GLOBALS, render);
      this.channel.off(Events.FORCE_RE_RENDER, render);
      this.channel.off(Events.FORCE_REMOUNT, remountStoryIfMatches);
      this.channel.off(Events.UPDATE_STORY_ARGS, rerenderStoryIfMatches);
      this.channel.off(Events.RESET_STORY_ARGS, rerenderStoryIfMatches); // Check if we're done rendering/playing. If not, we may have to reload the page.

      if (!isPending()) return; // Wait several ticks that may be needed to handle the abort, then try again.
      // Note that there's a max of 5 nested timeouts before they're no longer "instant".

      await new Promise(resolve => setTimeout(resolve, 0));
      if (!isPending()) return;
      await new Promise(resolve => setTimeout(resolve, 0));
      if (!isPending()) return;
      await new Promise(resolve => setTimeout(resolve, 0));
      if (!isPending()) return; // If we still haven't completed, reload the page (iframe) to ensure we have a clean slate
      // for the next render. Since the reload can take a brief moment to happen, we want to stop
      // further rendering by awaiting a never-resolving promise (which is destroyed on reload).

      global.window.location.reload();
      await new Promise(() => {});
    };
  } // API


  async extract(options) {
    var _global$FEATURES10;

    if (this.previewEntryError) {
      throw this.previewEntryError;
    }

    if (!this.storyStore.projectAnnotations) {
      // In v6 mode, if your preview.js throws, we never get a chance to initialize the preview
      // or store, and the error is simply logged to the browser console. This is the best we can do
      throw new Error(dedent`Failed to initialize Storybook.
      
      Do you have an error in your \`preview.js\`? Check your Storybook's browser console for errors.`);
    }

    if ((_global$FEATURES10 = global.FEATURES) !== null && _global$FEATURES10 !== void 0 && _global$FEATURES10.storyStoreV7) {
      await this.storyStore.cacheAllCSFFiles();
    }

    return this.storyStore.extract(options);
  } // UTILITIES


  async cleanupPreviousRender({
    unmountDocs = true
  } = {}) {
    var _this$previousStory, _this$previousStory$p, _this$previousSelecti3;

    const previousViewMode = (_this$previousStory = this.previousStory) !== null && _this$previousStory !== void 0 && (_this$previousStory$p = _this$previousStory.parameters) !== null && _this$previousStory$p !== void 0 && _this$previousStory$p.docsOnly ? 'docs' : (_this$previousSelecti3 = this.previousSelection) === null || _this$previousSelecti3 === void 0 ? void 0 : _this$previousSelecti3.viewMode;

    if (unmountDocs && previousViewMode === 'docs') {
      (await import('./renderDocs')).unmountDocs(this.view.docsRoot());
    }

    if (this.previousCleanup) {
      await this.previousCleanup();
    }
  }

  renderPreviewEntryError(reason, err) {
    this.previewEntryError = err;
    logger.error(reason);
    logger.error(err);
    this.view.showErrorDisplay(err);
    this.channel.emit(Events.CONFIG_ERROR, err);
  }

  renderMissingStory() {
    this.view.showNoPreview();
    this.channel.emit(Events.STORY_MISSING);
  }

  renderStoryLoadingException(storySpecifier, err) {
    logger.error(`Unable to load story '${storySpecifier}':`);
    logger.error(err);
    this.view.showErrorDisplay(err);
    this.channel.emit(Events.STORY_MISSING, storySpecifier);
  } // renderException is used if we fail to render the story and it is uncaught by the app layer


  renderException(storyId, err) {
    this.channel.emit(Events.STORY_THREW_EXCEPTION, err);
    this.channel.emit(Events.STORY_RENDER_PHASE_CHANGED, {
      newPhase: 'errored',
      storyId
    }); // Ignored exceptions exist for control flow purposes, and are typically handled elsewhere.

    if (err !== IGNORED_EXCEPTION) {
      this.view.showErrorDisplay(err);
      logger.error(`Error rendering story '${storyId}':`);
      logger.error(err);
    }
  } // renderError is used by the various app layers to inform the user they have done something
  // wrong -- for instance returned the wrong thing from a story


  renderError(storyId, {
    title,
    description
  }) {
    logger.error(`Error rendering story ${title}: ${description}`);
    this.channel.emit(Events.STORY_ERRORED, {
      title,
      description
    });
    this.channel.emit(Events.STORY_RENDER_PHASE_CHANGED, {
      newPhase: 'errored',
      storyId
    });
    this.view.showErrorDisplay({
      message: title,
      stack: description
    });
  }

}