// Copyright 2023 Joseph Birr-Pixton.
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHORS DISCLAIM ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

#![cfg(any(feature = "ring", feature = "aws_lc_rs"))]

use pki_types::{CertificateDer, SignatureVerificationAlgorithm};
#[cfg(feature = "ring")]
use webpki::ring::{
    ECDSA_P256_SHA256, ECDSA_P256_SHA384, ECDSA_P384_SHA256, ECDSA_P384_SHA384, ED25519,
};
#[cfg(all(feature = "ring", feature = "alloc"))]
use webpki::ring::{
    RSA_PKCS1_2048_8192_SHA256, RSA_PKCS1_2048_8192_SHA384, RSA_PKCS1_2048_8192_SHA512,
    RSA_PKCS1_3072_8192_SHA384, RSA_PSS_2048_8192_SHA256_LEGACY_KEY,
    RSA_PSS_2048_8192_SHA384_LEGACY_KEY, RSA_PSS_2048_8192_SHA512_LEGACY_KEY,
};

#[cfg(all(not(feature = "ring"), feature = "aws_lc_rs"))]
use webpki::aws_lc_rs::{
    ECDSA_P256_SHA256, ECDSA_P256_SHA384, ECDSA_P384_SHA256, ECDSA_P384_SHA384, ECDSA_P521_SHA512,
    ED25519, RSA_PKCS1_2048_8192_SHA256, RSA_PKCS1_2048_8192_SHA384, RSA_PKCS1_2048_8192_SHA512,
    RSA_PKCS1_3072_8192_SHA384, RSA_PSS_2048_8192_SHA256_LEGACY_KEY,
    RSA_PSS_2048_8192_SHA384_LEGACY_KEY, RSA_PSS_2048_8192_SHA512_LEGACY_KEY,
};

#[cfg(feature = "alloc")]
fn check_sig(
    ee: &[u8],
    alg: &dyn SignatureVerificationAlgorithm,
    message: &[u8],
    signature: &[u8],
) -> Result<(), webpki::Error> {
    let ee = CertificateDer::from(ee);
    let cert = webpki::EndEntityCert::try_from(&ee).unwrap();
    cert.verify_signature(alg, message, signature)
}

// DO NOT EDIT BELOW: generated by tests/generate.py

#[test]
#[cfg(feature = "alloc")]
fn ed25519_key_and_ed25519_good_signature() {
    let ee = include_bytes!("signatures/ed25519.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!("signatures/ed25519_key_and_ed25519_good_signature.sig.bin");
    assert_eq!(check_sig(ee, ED25519, message, signature), Ok(()));
}

#[test]
#[cfg(feature = "alloc")]
fn ed25519_key_and_ed25519_detects_bad_signature() {
    let ee = include_bytes!("signatures/ed25519.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature =
        include_bytes!("signatures/ed25519_key_and_ed25519_detects_bad_signature.sig.bin");
    assert_eq!(
        check_sig(ee, ED25519, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn ed25519_key_rejected_by_other_algorithms() {
    let ee = include_bytes!("signatures/ed25519.ee.der");
    for algorithm in &[
        #[cfg(all(not(feature = "ring"), feature = "aws_lc_rs"))]
        ECDSA_P521_SHA512,
        ECDSA_P256_SHA256,
        ECDSA_P256_SHA384,
        ECDSA_P384_SHA256,
        ECDSA_P384_SHA384,
        RSA_PKCS1_2048_8192_SHA256,
        RSA_PKCS1_2048_8192_SHA384,
        RSA_PKCS1_2048_8192_SHA512,
        RSA_PKCS1_3072_8192_SHA384,
        RSA_PSS_2048_8192_SHA256_LEGACY_KEY,
        RSA_PSS_2048_8192_SHA384_LEGACY_KEY,
        RSA_PSS_2048_8192_SHA512_LEGACY_KEY,
    ] {
        assert_eq!(
            check_sig(ee, *algorithm, b"", b""),
            Err(webpki::Error::UnsupportedSignatureAlgorithmForPublicKey)
        );
    }
}

#[test]
#[cfg(feature = "alloc")]
fn ecdsa_p256_key_and_ecdsa_p256_sha384_good_signature() {
    let ee = include_bytes!("signatures/ecdsa_p256.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature =
        include_bytes!("signatures/ecdsa_p256_key_and_ecdsa_p256_sha384_good_signature.sig.bin");
    assert_eq!(check_sig(ee, ECDSA_P256_SHA384, message, signature), Ok(()));
}

#[test]
#[cfg(feature = "alloc")]
fn ecdsa_p256_key_and_ecdsa_p256_sha384_detects_bad_signature() {
    let ee = include_bytes!("signatures/ecdsa_p256.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/ecdsa_p256_key_and_ecdsa_p256_sha384_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, ECDSA_P256_SHA384, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn ecdsa_p256_key_and_ecdsa_p256_sha256_good_signature() {
    let ee = include_bytes!("signatures/ecdsa_p256.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature =
        include_bytes!("signatures/ecdsa_p256_key_and_ecdsa_p256_sha256_good_signature.sig.bin");
    assert_eq!(check_sig(ee, ECDSA_P256_SHA256, message, signature), Ok(()));
}

#[test]
#[cfg(feature = "alloc")]
fn ecdsa_p256_key_and_ecdsa_p256_sha256_detects_bad_signature() {
    let ee = include_bytes!("signatures/ecdsa_p256.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/ecdsa_p256_key_and_ecdsa_p256_sha256_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, ECDSA_P256_SHA256, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn ecdsa_p256_key_rejected_by_other_algorithms() {
    let ee = include_bytes!("signatures/ecdsa_p256.ee.der");
    for algorithm in &[
        #[cfg(all(not(feature = "ring"), feature = "aws_lc_rs"))]
        ECDSA_P521_SHA512,
        ECDSA_P384_SHA256,
        ECDSA_P384_SHA384,
        ED25519,
        RSA_PKCS1_2048_8192_SHA256,
        RSA_PKCS1_2048_8192_SHA384,
        RSA_PKCS1_2048_8192_SHA512,
        RSA_PKCS1_3072_8192_SHA384,
        RSA_PSS_2048_8192_SHA256_LEGACY_KEY,
        RSA_PSS_2048_8192_SHA384_LEGACY_KEY,
        RSA_PSS_2048_8192_SHA512_LEGACY_KEY,
    ] {
        assert_eq!(
            check_sig(ee, *algorithm, b"", b""),
            Err(webpki::Error::UnsupportedSignatureAlgorithmForPublicKey)
        );
    }
}

#[test]
#[cfg(feature = "alloc")]
fn ecdsa_p384_key_and_ecdsa_p384_sha384_good_signature() {
    let ee = include_bytes!("signatures/ecdsa_p384.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature =
        include_bytes!("signatures/ecdsa_p384_key_and_ecdsa_p384_sha384_good_signature.sig.bin");
    assert_eq!(check_sig(ee, ECDSA_P384_SHA384, message, signature), Ok(()));
}

#[test]
#[cfg(feature = "alloc")]
fn ecdsa_p384_key_and_ecdsa_p384_sha384_detects_bad_signature() {
    let ee = include_bytes!("signatures/ecdsa_p384.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/ecdsa_p384_key_and_ecdsa_p384_sha384_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, ECDSA_P384_SHA384, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn ecdsa_p384_key_and_ecdsa_p384_sha256_good_signature() {
    let ee = include_bytes!("signatures/ecdsa_p384.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature =
        include_bytes!("signatures/ecdsa_p384_key_and_ecdsa_p384_sha256_good_signature.sig.bin");
    assert_eq!(check_sig(ee, ECDSA_P384_SHA256, message, signature), Ok(()));
}

#[test]
#[cfg(feature = "alloc")]
fn ecdsa_p384_key_and_ecdsa_p384_sha256_detects_bad_signature() {
    let ee = include_bytes!("signatures/ecdsa_p384.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/ecdsa_p384_key_and_ecdsa_p384_sha256_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, ECDSA_P384_SHA256, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn ecdsa_p384_key_rejected_by_other_algorithms() {
    let ee = include_bytes!("signatures/ecdsa_p384.ee.der");
    for algorithm in &[
        #[cfg(all(not(feature = "ring"), feature = "aws_lc_rs"))]
        ECDSA_P521_SHA512,
        ECDSA_P256_SHA256,
        ECDSA_P256_SHA384,
        ED25519,
        RSA_PKCS1_2048_8192_SHA256,
        RSA_PKCS1_2048_8192_SHA384,
        RSA_PKCS1_2048_8192_SHA512,
        RSA_PKCS1_3072_8192_SHA384,
        RSA_PSS_2048_8192_SHA256_LEGACY_KEY,
        RSA_PSS_2048_8192_SHA384_LEGACY_KEY,
        RSA_PSS_2048_8192_SHA512_LEGACY_KEY,
    ] {
        assert_eq!(
            check_sig(ee, *algorithm, b"", b""),
            Err(webpki::Error::UnsupportedSignatureAlgorithmForPublicKey)
        );
    }
}

#[test]
#[cfg(all(not(feature = "ring"), feature = "aws_lc_rs"))]
fn ecdsa_p521_key_and_ecdsa_p521_sha512_good_signature() {
    let ee = include_bytes!("signatures/ecdsa_p521.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature =
        include_bytes!("signatures/ecdsa_p521_key_and_ecdsa_p521_sha512_good_signature.sig.bin");
    assert_eq!(check_sig(ee, ECDSA_P521_SHA512, message, signature), Ok(()));
}

#[test]
#[cfg(all(not(feature = "ring"), feature = "aws_lc_rs"))]
fn ecdsa_p521_key_and_ecdsa_p521_sha512_detects_bad_signature() {
    let ee = include_bytes!("signatures/ecdsa_p521.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/ecdsa_p521_key_and_ecdsa_p521_sha512_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, ECDSA_P521_SHA512, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn ecdsa_p521_key_rejected_by_other_algorithms() {
    let ee = include_bytes!("signatures/ecdsa_p521.ee.der");
    for algorithm in &[
        ECDSA_P256_SHA256,
        ECDSA_P256_SHA384,
        ECDSA_P384_SHA256,
        ECDSA_P384_SHA384,
        ED25519,
        RSA_PKCS1_2048_8192_SHA256,
        RSA_PKCS1_2048_8192_SHA384,
        RSA_PKCS1_2048_8192_SHA512,
        RSA_PKCS1_3072_8192_SHA384,
        RSA_PSS_2048_8192_SHA256_LEGACY_KEY,
        RSA_PSS_2048_8192_SHA384_LEGACY_KEY,
        RSA_PSS_2048_8192_SHA512_LEGACY_KEY,
    ] {
        assert_eq!(
            check_sig(ee, *algorithm, b"", b""),
            Err(webpki::Error::UnsupportedSignatureAlgorithmForPublicKey)
        );
    }
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pkcs1_2048_8192_sha256_good_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_2048_key_and_rsa_pkcs1_2048_8192_sha256_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA256, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pkcs1_2048_8192_sha256_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_2048_key_and_rsa_pkcs1_2048_8192_sha256_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA256, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pkcs1_2048_8192_sha384_good_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_2048_key_and_rsa_pkcs1_2048_8192_sha384_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA384, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pkcs1_2048_8192_sha384_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_2048_key_and_rsa_pkcs1_2048_8192_sha384_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA384, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pkcs1_2048_8192_sha512_good_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_2048_key_and_rsa_pkcs1_2048_8192_sha512_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA512, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pkcs1_2048_8192_sha512_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_2048_key_and_rsa_pkcs1_2048_8192_sha512_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA512, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pss_2048_8192_sha256_legacy_key_good_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_2048_key_and_rsa_pss_2048_8192_sha256_legacy_key_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA256_LEGACY_KEY, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pss_2048_8192_sha256_legacy_key_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!("signatures/rsa_2048_key_and_rsa_pss_2048_8192_sha256_legacy_key_detects_bad_signature.sig.bin");
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA256_LEGACY_KEY, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pss_2048_8192_sha384_legacy_key_good_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_2048_key_and_rsa_pss_2048_8192_sha384_legacy_key_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA384_LEGACY_KEY, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pss_2048_8192_sha384_legacy_key_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!("signatures/rsa_2048_key_and_rsa_pss_2048_8192_sha384_legacy_key_detects_bad_signature.sig.bin");
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA384_LEGACY_KEY, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pss_2048_8192_sha512_legacy_key_good_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_2048_key_and_rsa_pss_2048_8192_sha512_legacy_key_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA512_LEGACY_KEY, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_and_rsa_pss_2048_8192_sha512_legacy_key_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!("signatures/rsa_2048_key_and_rsa_pss_2048_8192_sha512_legacy_key_detects_bad_signature.sig.bin");
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA512_LEGACY_KEY, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_rejected_by_other_algorithms() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    for algorithm in &[
        #[cfg(all(not(feature = "ring"), feature = "aws_lc_rs"))]
        ECDSA_P521_SHA512,
        ECDSA_P256_SHA256,
        ECDSA_P256_SHA384,
        ECDSA_P384_SHA256,
        ECDSA_P384_SHA384,
        ED25519,
    ] {
        assert_eq!(
            check_sig(ee, *algorithm, b"", b""),
            Err(webpki::Error::UnsupportedSignatureAlgorithmForPublicKey)
        );
    }
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pkcs1_2048_8192_sha256_good_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_3072_key_and_rsa_pkcs1_2048_8192_sha256_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA256, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pkcs1_2048_8192_sha256_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_3072_key_and_rsa_pkcs1_2048_8192_sha256_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA256, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pkcs1_2048_8192_sha384_good_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_3072_key_and_rsa_pkcs1_2048_8192_sha384_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA384, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pkcs1_2048_8192_sha384_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_3072_key_and_rsa_pkcs1_2048_8192_sha384_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA384, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pkcs1_2048_8192_sha512_good_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_3072_key_and_rsa_pkcs1_2048_8192_sha512_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA512, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pkcs1_2048_8192_sha512_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_3072_key_and_rsa_pkcs1_2048_8192_sha512_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA512, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pss_2048_8192_sha256_legacy_key_good_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_3072_key_and_rsa_pss_2048_8192_sha256_legacy_key_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA256_LEGACY_KEY, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pss_2048_8192_sha256_legacy_key_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!("signatures/rsa_3072_key_and_rsa_pss_2048_8192_sha256_legacy_key_detects_bad_signature.sig.bin");
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA256_LEGACY_KEY, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pss_2048_8192_sha384_legacy_key_good_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_3072_key_and_rsa_pss_2048_8192_sha384_legacy_key_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA384_LEGACY_KEY, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pss_2048_8192_sha384_legacy_key_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!("signatures/rsa_3072_key_and_rsa_pss_2048_8192_sha384_legacy_key_detects_bad_signature.sig.bin");
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA384_LEGACY_KEY, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pss_2048_8192_sha512_legacy_key_good_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_3072_key_and_rsa_pss_2048_8192_sha512_legacy_key_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA512_LEGACY_KEY, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pss_2048_8192_sha512_legacy_key_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!("signatures/rsa_3072_key_and_rsa_pss_2048_8192_sha512_legacy_key_detects_bad_signature.sig.bin");
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA512_LEGACY_KEY, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pkcs1_3072_8192_sha384_good_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_3072_key_and_rsa_pkcs1_3072_8192_sha384_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_3072_8192_SHA384, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_and_rsa_pkcs1_3072_8192_sha384_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_3072_key_and_rsa_pkcs1_3072_8192_sha384_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_3072_8192_SHA384, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_3072_key_rejected_by_other_algorithms() {
    let ee = include_bytes!("signatures/rsa_3072.ee.der");
    for algorithm in &[
        #[cfg(all(not(feature = "ring"), feature = "aws_lc_rs"))]
        ECDSA_P521_SHA512,
        ECDSA_P256_SHA256,
        ECDSA_P256_SHA384,
        ECDSA_P384_SHA256,
        ECDSA_P384_SHA384,
        ED25519,
    ] {
        assert_eq!(
            check_sig(ee, *algorithm, b"", b""),
            Err(webpki::Error::UnsupportedSignatureAlgorithmForPublicKey)
        );
    }
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pkcs1_2048_8192_sha256_good_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_4096_key_and_rsa_pkcs1_2048_8192_sha256_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA256, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pkcs1_2048_8192_sha256_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_4096_key_and_rsa_pkcs1_2048_8192_sha256_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA256, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pkcs1_2048_8192_sha384_good_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_4096_key_and_rsa_pkcs1_2048_8192_sha384_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA384, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pkcs1_2048_8192_sha384_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_4096_key_and_rsa_pkcs1_2048_8192_sha384_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA384, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pkcs1_2048_8192_sha512_good_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_4096_key_and_rsa_pkcs1_2048_8192_sha512_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA512, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pkcs1_2048_8192_sha512_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_4096_key_and_rsa_pkcs1_2048_8192_sha512_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_2048_8192_SHA512, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pss_2048_8192_sha256_legacy_key_good_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_4096_key_and_rsa_pss_2048_8192_sha256_legacy_key_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA256_LEGACY_KEY, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pss_2048_8192_sha256_legacy_key_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!("signatures/rsa_4096_key_and_rsa_pss_2048_8192_sha256_legacy_key_detects_bad_signature.sig.bin");
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA256_LEGACY_KEY, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pss_2048_8192_sha384_legacy_key_good_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_4096_key_and_rsa_pss_2048_8192_sha384_legacy_key_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA384_LEGACY_KEY, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pss_2048_8192_sha384_legacy_key_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!("signatures/rsa_4096_key_and_rsa_pss_2048_8192_sha384_legacy_key_detects_bad_signature.sig.bin");
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA384_LEGACY_KEY, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pss_2048_8192_sha512_legacy_key_good_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_4096_key_and_rsa_pss_2048_8192_sha512_legacy_key_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA512_LEGACY_KEY, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pss_2048_8192_sha512_legacy_key_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!("signatures/rsa_4096_key_and_rsa_pss_2048_8192_sha512_legacy_key_detects_bad_signature.sig.bin");
    assert_eq!(
        check_sig(ee, RSA_PSS_2048_8192_SHA512_LEGACY_KEY, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pkcs1_3072_8192_sha384_good_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_4096_key_and_rsa_pkcs1_3072_8192_sha384_good_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_3072_8192_SHA384, message, signature),
        Ok(())
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_and_rsa_pkcs1_3072_8192_sha384_detects_bad_signature() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature = include_bytes!(
        "signatures/rsa_4096_key_and_rsa_pkcs1_3072_8192_sha384_detects_bad_signature.sig.bin"
    );
    assert_eq!(
        check_sig(ee, RSA_PKCS1_3072_8192_SHA384, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_4096_key_rejected_by_other_algorithms() {
    let ee = include_bytes!("signatures/rsa_4096.ee.der");
    for algorithm in &[
        #[cfg(all(not(feature = "ring"), feature = "aws_lc_rs"))]
        ECDSA_P521_SHA512,
        ECDSA_P256_SHA256,
        ECDSA_P256_SHA384,
        ECDSA_P384_SHA256,
        ECDSA_P384_SHA384,
        ED25519,
    ] {
        assert_eq!(
            check_sig(ee, *algorithm, b"", b""),
            Err(webpki::Error::UnsupportedSignatureAlgorithmForPublicKey)
        );
    }
}

#[test]
#[cfg(feature = "alloc")]
fn rsa_2048_key_rejected_by_rsa_pkcs1_3072_8192_sha384() {
    let ee = include_bytes!("signatures/rsa_2048.ee.der");
    let message = include_bytes!("signatures/message.bin");
    let signature =
        include_bytes!("signatures/rsa_2048_key_rejected_by_rsa_pkcs1_3072_8192_sha384.sig.bin");
    assert_eq!(
        check_sig(ee, RSA_PKCS1_3072_8192_SHA384, message, signature),
        Err(webpki::Error::InvalidSignatureForPublicKey)
    );
}
