// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2022 Alibaba Cloud. or its affiliates. All Rights Reserved.
 *
 * This module introduce pio2mmio driver to hide the difference of IO operations
 * between x86_64 and aarch64. It enables aarch64 guest kernel to use port IO
 * operations in dragonball features.
 *
 */

#include <linux/io.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/kconfig.h>

/* MMIO space base address, defined in dragonball */
static unsigned long PIO2MMIO_MMIO_BASE = 0;
/* MMIO space size, defined in dragonball */
static unsigned long PIO2MMIO_MMIO_SIZE = 0;

/* Get base address and space size from cmdline. */
#if IS_BUILTIN(CONFIG_DRAGONBALL_PIO2MMIO)
static int __init pio2mmio_setup(char *str)
{
	if (strcmp(str, ""))
		sscanf(str, "%ldk@0x%lx", &PIO2MMIO_MMIO_SIZE, &PIO2MMIO_MMIO_BASE);

	return 1;
}
__setup("pio2mmio.device=", pio2mmio_setup);
#endif

/* Get base address and space size from extern. */
#if IS_MODULE(CONFIG_DRAGONBALL_PIO2MMIO)
module_param_named(pio2mmio_base, PIO2MMIO_MMIO_BASE, ulong, S_IRUGO);
MODULE_PARM_DESC(pio2mmio_base, "MMIO base address for pio2mmio device.");
module_param_named(pio2mmio_size, PIO2MMIO_MMIO_SIZE, ulong, S_IRUGO);
MODULE_PARM_DESC(pio2mmio_size, "MMIO region size for pio2mmio device.");
#endif

/*
 * Pio2mmio structure, saves mmaped virtual address of mmio space and mmio space size.
 */
struct pio2mmio_device {
	/* Virtual address of mmio base address */
	void *mmio_base;
	/* mmio space size */
	uint64_t mmio_size;
	/* IO resource */
	struct resource *res;
} dev;

#if defined(CONFIG_ARM64)
#define BUILD_ARCHIO(bwl, type)                      \
inline void arch_out##bwl(type value, int addr)      \
{                                                    \
	if (addr >= PIO2MMIO_MMIO_SIZE) {                \
		pr_err("pio2mmio: Invalid pio address!\n");  \
		return;                                      \
	}                                                \
\
	write##bwl(value, dev.mmio_base + addr);         \
}                                                    \
EXPORT_SYMBOL_GPL(arch_out##bwl);                    \
\
inline type arch_in##bwl(int addr)                   \
{                                                    \
	if (addr >= PIO2MMIO_MMIO_SIZE) {                \
		pr_err("pio2mmio: Invalid pio address!");    \
		return -EINVAL;                              \
	}                                                \
\
	return read##bwl(dev.mmio_base + addr);          \
}                                                    \
EXPORT_SYMBOL_GPL(arch_in##bwl)
#elif defined(CONFIG_X86_64)
#define BUILD_ARCHIO(bwl, type)                      \
inline void arch_out##bwl(type value, int addr)      \
{                                                    \
	out##bwl(value, addr);                           \
}                                                    \
EXPORT_SYMBOL_GPL(arch_out##bwl);                    \
\
inline type arch_in##bwl(int addr)                   \
{                                                    \
	return in##bwl(addr);                            \
}                                                    \
EXPORT_SYMBOL_GPL(arch_in##bwl)
#endif

BUILD_ARCHIO(b, unsigned char);
BUILD_ARCHIO(w, unsigned short);
BUILD_ARCHIO(l, unsigned int);

static int __init pio2mmio_init(void)
{
	if (PIO2MMIO_MMIO_BASE == 0 || PIO2MMIO_MMIO_SIZE == 0) {
		pr_err("pio2mmio: Please pass \"pio2mmio_base\" and \"pio2mmio_size\" parameters for pio2mmio module!");
		return -EINVAL;
	}

	PIO2MMIO_MMIO_SIZE <<= 10;
	pr_info
	    ("pio2mmio: Initialize pio2mmio driver: start: 0x%lx, size: 0x%lx\n",
	     PIO2MMIO_MMIO_BASE, PIO2MMIO_MMIO_SIZE);
	dev.res =
	    request_mem_region(PIO2MMIO_MMIO_BASE, PIO2MMIO_MMIO_SIZE,
			       "pio2mmio");
	if (!dev.res) {
		pr_err("pio2mmio: Allocate IO resource failed!");
		return -ENOMEM;
	}

	dev.mmio_size = PIO2MMIO_MMIO_SIZE;
	dev.mmio_base = ioremap(PIO2MMIO_MMIO_BASE, PIO2MMIO_MMIO_SIZE);
	if (!dev.mmio_base) {
		pr_err("pio2mmio: Map the mmio space failed!");
		goto free_io_resource;
	}

	pr_info("pio2mmio module init!\n");

	return 0;

free_io_resource:
	release_mem_region(dev.res->start, PIO2MMIO_MMIO_SIZE);
	return -ENOMEM;
}

static void __init pio2mmio_exit(void)
{
	iounmap(dev.mmio_base);
	release_mem_region(dev.res->start, PIO2MMIO_MMIO_SIZE);
	pr_info("pio2mmio module exit!\n");
}

module_init(pio2mmio_init);
module_exit(pio2mmio_exit);

MODULE_LICENSE("GPL");
