/* SPDX-License-Identifier: GPL-2.0 */
/*
 * Copyright (C) 2021 Alibaba, Inc
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 */

#ifndef _DRAGONBALL_H
#define _DRAGONBALL_H

#include <linux/init.h>
#include <linux/types.h>
#include <linux/io.h>
#include <linux/device.h>
#include "dragonball_defs.h"
#ifdef CONFIG_X86_64
#include <asm/cmdline.h>
#endif

#if defined(CONFIG_X86_64)
	#define SHARED_IRQ_NO			5
#elif defined(CONFIG_ARM64)
	#define SHARED_IRQ_NO			32
#endif

#define DB_IDENTITY(x) x

#define DRAGONBALL_FEATURES						\
	DB_IDENTITY(DB_ADD_FEATURE(VSOCKSRV, vsocksrv)			\
		    DB_ADD_FEATURE(DEVMGR, devmgr)			\
		    DB_ADD_FEATURE(PCI_PASSTHROUGH, pci_passthrough)	\
		    DB_ADD_FEATURE(PVDMA, pvdma)			\
		    DB_ADD_FEATURE(NUMA, numa)				\
		    DB_ADD_FEATURE(VEBPF, vebpf)			\
		    DB_ADD_FEATURE(PVPANIC, pvpanic)			\
		    DB_ADD_FEATURE(PVDUMP, pvdump)			\
		    DB_ADD_FEATURE(QNC, qnc))

#define DB_ADD_FEATURE(_feat, _name) DB_FEAT_##_feat,
enum dragonball_features {
	DRAGONBALL_FEATURES
};
#undef DB_ADD_FEATURE

struct dragonball_device {
	/* platform device */
	struct platform_device *pdev;

	/* io mem's base */
	void __iomem *base;
	/* platform device's version */
	u32 version;
	/* the features supported by driver */
	u64 driver_features;
	/* the features supported by both driver and device */
	u64 acked_features;

	/* base id of irq to assign */
	u32 base_irq;
	/* next id of irq to assign */
	u32 next_irq;
	/* max irq id of irq to assign */
	u32 max_irq;
};

#ifdef CONFIG_DRAGONBALL_DRIVERS
extern bool __ref is_dragonball_platform(void);
extern void enable_dragonball_platform(void);
extern void disable_dragonball_platform(void);
extern bool __ref has_dragonball_feature(enum dragonball_features);
extern void enable_dragonball_feature(enum dragonball_features);
extern void disable_dragonball_feature(enum dragonball_features);
extern int dragonball_add_device(struct resource *resources, size_t res_size);
#else
static inline bool is_dragonball_platform(void) { return false; }
static inline void enable_dragonball_platform(void) {}
static inline void disable_dragonball_platform(void) {}
static inline bool has_dragonball_feature(enum dragonball_features f) { return false; }
static inline void enable_dragonball_feature(enum dragonball_features f) {}
static inline void disable_dragonball_feature(enum dragonball_features f) {}
static inline int dragonball_add_device(struct resource *resources, size_t res_size) { return 0; }
#endif

#if defined(CONFIG_VIRTIO_MMIO_DRAGONBALL) && defined(CONFIG_ARM64)
struct irq_domain *platform_msi_get_def_irq_domain(void);
#endif

#if defined(CONFIG_PVDMA) && defined(CONFIG_ARM64)
extern const struct dma_map_ops pv_dma_pci_ops;
extern bool pvdma_is_enabled;

static inline void pvdma_update_pci_dma_ops(struct device *dev)
{
	if (has_dragonball_feature(DB_FEAT_PVDMA) && pvdma_is_enabled
		&& !dev->dma_ops  && dev->bus
		&& dev->bus->name && !strcmp(dev->bus->name, "pci")) {
		dev->dma_ops = &pv_dma_pci_ops;
	}
}
#endif

uint32_t get_device_virq(uint32_t pin);

#endif	/* _DRAGONBALL_H */
