// SPDX-License-Identifier: GPL-2.0
#ifndef __PKE_H
#define __PKE_H

#include <stdint.h>

struct rsa_key {
	const unsigned char *n;
	const unsigned char *e;
	const unsigned char *d;
	const unsigned char *p;
	const unsigned char *q;
	const unsigned char *dp;
	const unsigned char *dq;
	const unsigned char *qinv;
	uint32_t n_sz;
	uint32_t e_sz;
	uint32_t d_sz;
	uint32_t p_sz;
	uint32_t q_sz;
	uint32_t dp_sz;
	uint32_t dq_sz;
	uint32_t qinv_sz;
};

struct akcipher_ctx {
	uint32_t flags;

	struct akcipher_alg *alg;

	void *__ctx[];
};

struct akcipher_req;
typedef void (*akcipher_completion_t)(struct akcipher_req *req, int err);

struct akcipher_req {
	unsigned char *src;
	unsigned char *dst;
	uint32_t src_len;
	uint32_t dst_len;

	int padding;

	akcipher_completion_t complete;
	void *data;

	struct akcipher_ctx *ctx;
	void *__req[];
};

struct akcipher_alg {
	const char *name;

	int (*sign)(struct akcipher_req *req);
	int (*verify)(struct akcipher_req *req);
	int (*encrypt)(struct akcipher_req *req);
	int (*decrypt)(struct akcipher_req *req);
	int (*set_pub_key)(struct akcipher_ctx *ctx, void *key,
			   unsigned int keylen);
	int (*set_priv_key)(struct akcipher_ctx *ctx, void *key,
			    unsigned int keylen);
	unsigned int (*max_size)(struct akcipher_ctx *ctx);
	int (*init)(struct akcipher_ctx *ctx);
	void (*exit)(struct akcipher_ctx *ctx);

	uint32_t ctxsize;
	unsigned int reqsize;
};

struct akcipher_alg_entry {
	struct akcipher_alg *alg;
	struct akcipher_alg_entry *next;
};

struct akcipher_ctx *akcipher_alloc_ctx(const char *alg_name, uint32_t flag);
void akcipher_free_ctx(struct akcipher_ctx *ctx);
struct akcipher_req *akcipher_alloc_req(struct akcipher_ctx *ctx);
void akcipher_free_req(struct akcipher_req *req);
void akcipher_req_set_callback(struct akcipher_req *req, akcipher_completion_t complete,
			       void *data);
void akcipher_set_req(struct akcipher_req *req, unsigned char *src, unsigned char *dst,
		      unsigned int src_len, unsigned int dst_len, int padding);
static inline int akcipher_get_outlen(struct akcipher_req *req)
{
	return req->dst_len;
}
int akcipher_sign(struct akcipher_req *req);
int akcipher_verify(struct akcipher_req *req);
int akcipher_set_pub_key(struct akcipher_ctx *ctx, void *key,
			 unsigned int keylen);
int akcipher_set_priv_key(struct akcipher_ctx *ctx, void *key,
			  unsigned int keylen);
unsigned int akcipher_maxsize(struct akcipher_ctx *ctx);
int akcipher_encrypt(struct akcipher_req *req);
int akcipher_decrypt(struct akcipher_req *req);
void akcipher_register_algs(void);
void akcipher_unregister_algs(void);

#endif
