// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2022 nebula-matrix Limited.
 * Author: Bennie Yan <bennie@nebula-matrix.com>
 */
#include "nbl_ethtool.h"
#include "nbl_ktls.h"
#include "nbl_ipsec.h"
#include "nbl_tc.h"

static void nbl_serv_set_link_state(struct nbl_service_mgt *serv_mgt, struct net_device *netdev);
static int nbl_serv_update_default_vlan(struct nbl_service_mgt *serv_mgt, u16 vid);

static void nbl_serv_set_queue_param(struct nbl_serv_ring *ring, u16 desc_num,
				     struct nbl_txrx_queue_param *param, u16 vsi_id,
				     u16 global_vector_id)
{
	param->vsi_id = vsi_id;
	param->dma = ring->dma;
	param->desc_num = desc_num;
	param->local_queue_id = ring->local_queue_id / 2;
	param->global_vector_id = global_vector_id;
	param->intr_en = 1;
	param->intr_mask = 1;
	param->extend_header = 1;
	param->rxcsum = 1;
	param->split = 0;
}

/**
 * In virtio mode, the emulator triggers the configuration of
 * txrx_registers only based on tx_ring, so the rx_info needs
 * to be delivered first before the tx_info can be delivered.
 */
static int
nbl_serv_setup_queues(struct nbl_service_mgt *serv_mgt, struct nbl_serv_ring_vsi_info *vsi_info)
{
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_txrx_queue_param param = {0};
	struct nbl_serv_ring *ring;
	struct nbl_serv_vector *vector;
	u16 start = vsi_info->ring_offset, end = vsi_info->ring_offset + vsi_info->ring_num;
	int vector_offset = 0;
	int i, ret = 0;

	if (vsi_info->vsi_index == NBL_VSI_XDP)
		vector_offset = ring_mgt->xdp_ring_offset;

	for (i = start; i < end; i++) {
		vector = &ring_mgt->vectors[i - vector_offset];
		ring = &ring_mgt->rx_rings[i];
		nbl_serv_set_queue_param(ring, ring_mgt->rx_desc_num, &param,
					 vsi_info->vsi_id, vector->global_vector_id);

		ret = disp_ops->setup_queue(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), &param, false);
		if (ret)
			return ret;
	}

	for (i = start; i < end; i++) {
		vector = &ring_mgt->vectors[i - vector_offset];
		ring = &ring_mgt->tx_rings[i];

		nbl_serv_set_queue_param(ring, ring_mgt->tx_desc_num, &param,
					 vsi_info->vsi_id, vector->global_vector_id);

		ret = disp_ops->setup_queue(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), &param, true);
		if (ret)
			return ret;
	}

	return 0;
}

static void
nbl_serv_flush_rx_queues(struct nbl_service_mgt *serv_mgt, u16 ring_offset, u16 ring_num)
{
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	int i;

	for (i = ring_offset; i < ring_offset + ring_num; i++)
		disp_ops->kick_rx_ring(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), i);
}

static int nbl_serv_setup_rings(struct nbl_service_mgt *serv_mgt, struct net_device *netdev,
				struct nbl_serv_ring_vsi_info *vsi_info, bool use_napi)
{
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	u16 start = vsi_info->ring_offset, end = vsi_info->ring_offset + vsi_info->ring_num;
	int i, ret = 0;

	for (i = start; i < end; i++) {
		ring_mgt->tx_rings[i].dma =
			disp_ops->start_tx_ring(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), i);
		if (!ring_mgt->tx_rings[i].dma) {
			netdev_err(netdev, "Fail to start tx ring %d", i);
			ret = -EFAULT;
			break;
		}
	}
	if (i != end) {
		while (--i + 1 > start)
			disp_ops->stop_tx_ring(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), i);
		goto tx_err;
	}

	for (i = start; i < end; i++) {
		ring_mgt->rx_rings[i].dma =
			disp_ops->start_rx_ring(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), i, use_napi);
		if (!ring_mgt->rx_rings[i].dma) {
			netdev_err(netdev, "Fail to start rx ring %d", i);
			ret = -EFAULT;
			break;
		}
	}
	if (i != end) {
		while (--i + 1 > start)
			disp_ops->stop_rx_ring(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), i);
		goto rx_err;
	}

	return 0;

rx_err:
	for (i = start; i < end; i++)
		disp_ops->stop_tx_ring(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), i);
tx_err:
	return ret;
}

static void nbl_serv_stop_rings(struct nbl_service_mgt *serv_mgt,
				struct nbl_serv_ring_vsi_info *vsi_info)
{
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	u16 start = vsi_info->ring_offset, end = vsi_info->ring_offset + vsi_info->ring_num;
	int i;

	for (i = start; i < end; i++)
		disp_ops->stop_tx_ring(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), i);

	for (i = start; i < end; i++)
		disp_ops->stop_rx_ring(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), i);
}

static int nbl_serv_set_tx_rings(struct nbl_serv_ring_mgt *ring_mgt,
				 struct net_device *netdev, struct device *dev)
{
	int i;
	u16 ring_num = ring_mgt->tx_ring_num;

	ring_mgt->tx_rings = devm_kcalloc(dev, ring_num, sizeof(*ring_mgt->tx_rings), GFP_KERNEL);
	if (!ring_mgt->tx_rings)
		return -ENOMEM;

	for (i = 0; i < ring_num; i++)
		ring_mgt->tx_rings[i].index = i;

	return 0;
}

static void nbl_serv_remove_tx_ring(struct nbl_serv_ring_mgt *ring_mgt, struct device *dev)
{
	devm_kfree(dev, ring_mgt->tx_rings);
	ring_mgt->tx_rings = NULL;
}

static int nbl_serv_set_rx_rings(struct nbl_serv_ring_mgt *ring_mgt,
				 struct net_device *netdev, struct device *dev)
{
	int i;
	u16 ring_num = ring_mgt->rx_ring_num;

	ring_mgt->rx_rings = devm_kcalloc(dev, ring_num, sizeof(*ring_mgt->rx_rings), GFP_KERNEL);
	if (!ring_mgt->rx_rings)
		return -ENOMEM;

	for (i = 0; i < ring_num; i++)
		ring_mgt->rx_rings[i].index = i;

	return 0;
}

static void nbl_serv_remove_rx_ring(struct nbl_serv_ring_mgt *ring_mgt, struct device *dev)
{
	devm_kfree(dev, ring_mgt->rx_rings);
	ring_mgt->rx_rings = NULL;
}

static int nbl_serv_register_xdp_rxq(struct nbl_service_mgt *serv_mgt,
				     struct nbl_serv_ring_mgt *ring_mgt)
{
	u16 ring_num;
	int i, j;
	int ret;
	struct nbl_dispatch_ops *disp_ops;
	struct nbl_serv_ring_vsi_info *vsi_info;

	if (ring_mgt->xdp_ring_offset == ring_mgt->tx_ring_num)
		return 0;

	vsi_info = &ring_mgt->vsi_info[NBL_VSI_DATA];
	ring_num = vsi_info->ring_num;
	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	for (i = 0; i < ring_num; i++) {
		ret = disp_ops->register_xdp_rxq(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), i);
		if (ret)
			goto register_xdp_err;
	}

	return 0;
register_xdp_err:
	for (j = 0; j < i; j++)
		disp_ops->unregister_xdp_rxq(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), j);

	return -1;
}

static void nbl_serv_unregister_xdp_rxq(struct nbl_service_mgt *serv_mgt,
					struct nbl_serv_ring_mgt *ring_mgt)
{
	u16 ring_num;
	int i;
	struct nbl_dispatch_ops *disp_ops;
	struct nbl_serv_ring_vsi_info *vsi_info;

	if (ring_mgt->xdp_ring_offset == ring_mgt->tx_ring_num)
		return;

	vsi_info = &ring_mgt->vsi_info[NBL_VSI_DATA];
	ring_num = vsi_info->ring_num;
	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	for (i = 0; i < ring_num; i++)
		disp_ops->unregister_xdp_rxq(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), i);
}

static int nbl_serv_set_vectors(struct nbl_service_mgt *serv_mgt,
				struct net_device *netdev, struct device *dev)
{
	struct nbl_adapter *adapter = NBL_NETDEV_TO_ADAPTER(netdev);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_resource_pt_ops *pt_ops = NBL_ADAPTER_TO_RES_PT_OPS(adapter);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	int i;
	u16 ring_num = ring_mgt->xdp_ring_offset;

	ring_mgt->vectors = devm_kcalloc(dev, ring_num, sizeof(*ring_mgt->vectors), GFP_KERNEL);
	if (!ring_mgt->vectors)
		return -ENOMEM;

	for (i = 0; i < ring_num; i++) {
		ring_mgt->vectors[i].napi =
			disp_ops->get_vector_napi(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), i);
		netif_napi_add(netdev, ring_mgt->vectors[i].napi, pt_ops->napi_poll);

		ring_mgt->vectors[i].netdev = netdev;
		cpumask_clear(&ring_mgt->vectors[i].cpumask);
	}

	return 0;
}

static void nbl_serv_remove_vectors(struct nbl_serv_ring_mgt *ring_mgt, struct device *dev)
{
	int i;
	u16 ring_num = ring_mgt->xdp_ring_offset;

	for (i = 0; i < ring_num; i++)
		netif_napi_del(ring_mgt->vectors[i].napi);

	devm_kfree(dev, ring_mgt->vectors);
	ring_mgt->vectors = NULL;
}

static struct nbl_serv_vlan_node *nbl_serv_alloc_vlan_node(void)
{
	struct nbl_serv_vlan_node *vlan_node = NULL;

	vlan_node = kzalloc(sizeof(*vlan_node), GFP_ATOMIC);
	if (!vlan_node)
		return NULL;

	INIT_LIST_HEAD(&vlan_node->node);
	vlan_node->ref_cnt = 1;

	return vlan_node;
}

static void nbl_serv_free_vlan_node(struct nbl_serv_vlan_node *vlan_node)
{
	kfree(vlan_node);
}

static struct nbl_serv_submac_node *nbl_serv_alloc_submac_node(void)
{
	struct nbl_serv_submac_node *submac_node = NULL;

	submac_node = kzalloc(sizeof(*submac_node), GFP_ATOMIC);
	if (!submac_node)
		return NULL;

	INIT_LIST_HEAD(&submac_node->node);
	return submac_node;
}

static void nbl_serv_free_submac_node(struct nbl_serv_submac_node *submac_node)
{
	kfree(submac_node);
}

static void nbl_serv_del_all_vlans(struct nbl_service_mgt *serv_mgt)
{
	struct nbl_serv_flow_mgt *flow_mgt = NBL_SERV_MGT_TO_FLOW_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_serv_vlan_node *vlan_node, *vlan_node_safe;

	list_for_each_entry_safe(vlan_node, vlan_node_safe, &flow_mgt->vlan_list, node) {
		disp_ops->del_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), flow_mgt->mac,
				      vlan_node->vid, NBL_COMMON_TO_VSI_ID(common));

		list_del(&vlan_node->node);
		nbl_serv_free_vlan_node(vlan_node);
	}
}

static void nbl_serv_del_all_submacs(struct nbl_service_mgt *serv_mgt, u16 vsi)
{
	struct nbl_serv_flow_mgt *flow_mgt = NBL_SERV_MGT_TO_FLOW_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_submac_node *submac_node, *submac_node_safe;

	list_for_each_entry_safe(submac_node, submac_node_safe, &flow_mgt->submac_list, node) {
		disp_ops->del_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), submac_node->mac,
				      NBL_DEFAULT_VLAN_ID, vsi);

		list_del(&submac_node->node);
		nbl_serv_free_submac_node(submac_node);
	}
}

static int nbl_serv_validate_tc_config(struct tc_mqprio_qopt_offload *mqprio_qopt,
				       struct nbl_common_info *common, u16 num_active_queues)
{
	u64 tx_rate = 0;
	int i, num_qps = 0;

	if (mqprio_qopt->qopt.num_tc > NBL_MAX_QUEUE_TC_NUM || mqprio_qopt->qopt.num_tc < 1) {
		nbl_err(common, NBL_DEBUG_QUEUE, "Invalid num_tc %u", mqprio_qopt->qopt.num_tc);
		return -EINVAL;
	}

	for (i = 0; i < mqprio_qopt->qopt.num_tc; i++) {
		if (!mqprio_qopt->qopt.count[i] || mqprio_qopt->qopt.offset[i] != num_qps) {
			nbl_err(common, NBL_DEBUG_QUEUE, "Invalid offset%u, num_qps:%u for tc %d",
				mqprio_qopt->qopt.offset[i], num_qps, i);
			return -EINVAL;
		}

		if (mqprio_qopt->min_rate[i]) {
			nbl_err(common, NBL_DEBUG_QUEUE,
				"Invalid min tx rate (greater than 0) specified for TC %d", i);
			return -EINVAL;
		}

		tx_rate = div_u64(mqprio_qopt->max_rate[i], NBL_TC_MBPS_DIVSIOR);

		if (mqprio_qopt->max_rate[i] && tx_rate < NBL_TC_WEIGHT_GRAVITY) {
			nbl_err(common, NBL_DEBUG_QUEUE,
				"Invalid max tx rate for TC %d, minimum %d Mbps",
				i, NBL_TC_MBPS_DIVSIOR);
			return -EINVAL;
		}

		if (tx_rate % NBL_TC_WEIGHT_GRAVITY != 0) {
			nbl_err(common, NBL_DEBUG_QUEUE,
				"Invalid max tx rate for TC %d, not divisible by %d",
				i, NBL_TC_MBPS_DIVSIOR);
			return -EINVAL;
		}

		num_qps += mqprio_qopt->qopt.count[i];
	}

	if (num_qps > num_active_queues) {
		nbl_err(common, NBL_DEBUG_QUEUE, "Cannot support requested number of queues");
		return -EINVAL;
	}

	return 0;
}

void nbl_serv_cpu_affinity_init(void *priv, u16 rings_num)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	int i;

	for (i = 0; i < rings_num; i++) {
		cpumask_set_cpu(cpumask_local_spread(i, NBL_COMMON_TO_DEV(common)->numa_node),
				&ring_mgt->vectors[i].cpumask);
		netif_set_xps_queue(ring_mgt->vectors[i].netdev, &ring_mgt->vectors[i].cpumask, i);
	}
}

static int nbl_serv_setup_tc_mqprio(struct net_device *netdev, void *type_data)
{
	struct nbl_netdev_priv *priv = netdev_priv(netdev);
	struct nbl_adapter *adapter = NBL_NETDEV_PRIV_TO_ADAPTER(priv);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_ADAPTER_TO_COMMON(adapter);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_tc_mgt *tc_mgt = NBL_SERV_MGT_TO_TC_MGT(serv_mgt);
	struct tc_mqprio_qopt_offload *mqprio_qopt = type_data;
	struct nbl_tc_qidsc_param param;
	u8 num_tc = mqprio_qopt->qopt.num_tc, total_qps = 0;
	struct nbl_serv_ring_vsi_info *vsi_info;
	int i, ret = 0;

	memset(&param, 0, sizeof(param));

	vsi_info = &ring_mgt->vsi_info[NBL_VSI_DATA];

	param.vsi_id = vsi_info->vsi_id;

	if (!mqprio_qopt->qopt.hw) {
		/*hw 1 to hw 0*/
		if (tc_mgt->state == NBL_TC_RUNNING) {
			/*reset the tc configuration*/
			netdev_reset_tc(netdev);
			netif_tx_stop_all_queues(netdev);
			netif_carrier_off(netdev);
			netif_tx_disable(netdev);

			nbl_serv_vsi_stop(serv_mgt, NBL_VSI_DATA);

			param.origin_qps = tc_mgt->total_qps;
			disp_ops->cfg_qdisc_mqprio(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), &param);

			total_qps = tc_mgt->orig_num_active_queues;
			tc_mgt->num_tc = 0;
			tc_mgt->state = NBL_TC_INVALID;

			goto exit;
		} else {
			return -EINVAL;
		}
	}

	if (mqprio_qopt->mode != TC_MQPRIO_MODE_CHANNEL)
		return -EOPNOTSUPP;

	if (tc_mgt->state != NBL_TC_INVALID) {
		netdev_err(netdev, "TC configuration already exists");
		return -EINVAL;
	}

	ret = nbl_serv_validate_tc_config(mqprio_qopt, common, vsi_info->ring_num);
	if (ret) {
		netdev_err(netdev, "TC config invalid");
		return ret;
	}

	if (tc_mgt->num_tc == num_tc)
		return 0;

	if (num_tc > NBL_MAX_TC_NUM) {
		netdev_err(netdev, "num_tc max to 8 but set %d\n", num_tc);
		return -EINVAL;
	}

	for (i = 0; i < num_tc; i++) {
		total_qps += mqprio_qopt->qopt.count[i];
		param.info[i].count = mqprio_qopt->qopt.count[i];
		param.info[i].offset = mqprio_qopt->qopt.offset[i];
		param.info[i].max_tx_rate = div_u64(mqprio_qopt->max_rate[i], NBL_TC_MBPS_DIVSIOR);
	}

	tc_mgt->num_tc = num_tc;
	tc_mgt->orig_num_active_queues = vsi_info->active_ring_num;

	netif_tx_stop_all_queues(netdev);
	netif_carrier_off(netdev);
	netif_tx_disable(netdev);

	nbl_serv_vsi_stop(serv_mgt, NBL_VSI_DATA);

	param.num_tc = num_tc;
	param.enable = true;
	param.origin_qps = tc_mgt->orig_num_active_queues;
	param.gravity = NBL_TC_WEIGHT_GRAVITY;
	ret = disp_ops->cfg_qdisc_mqprio(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), &param);
	if (ret) {
		netdev_err(netdev, "Fail to cfg qdisc mqprio");
		tc_mgt->num_tc = 0;
		return ret;
	}

	netdev_reset_tc(netdev);
	/* Report the tc mapping up the stack */
	netdev_set_num_tc(netdev, num_tc);
	for (i = 0; i < num_tc; i++)
		netdev_set_tc_queue(netdev, i, mqprio_qopt->qopt.count[i],
				    mqprio_qopt->qopt.offset[i]);

	tc_mgt->state = NBL_TC_RUNNING;
exit:
	/* If the device are unregistering, we cannot set queue nums or start them,
	 * otherwise we will hold the refcnt forever and block the unregistering process.
	 *
	 * Note: ndo_stop will not help, cause ndo_stop(in dev_close_many) execs
	 * before ndo_setup_tc(in dev_shutdown) when unregistering
	 */
	if (total_qps && NETREG_REGISTERED == netdev->reg_state &&
	    !test_bit(NBL_DOWN, adapter->state)) {
		nbl_serv_cpu_affinity_init(serv_mgt, total_qps);
		netif_set_real_num_rx_queues(netdev, total_qps);
		netif_set_real_num_tx_queues(netdev, total_qps);

		nbl_serv_vsi_open(serv_mgt, netdev, NBL_VSI_DATA, total_qps, 1);

		netif_tx_start_all_queues(netdev);
		nbl_serv_set_link_state(serv_mgt, netdev);
	}

	tc_mgt->total_qps = total_qps;
	return ret;
}

static int nbl_serv_ipv6_exthdr_num(struct sk_buff *skb, int start, u8 nexthdr)
{
	int exthdr_num = 0;
	struct ipv6_opt_hdr _hdr, *hp;
	unsigned int hdrlen;

	while (ipv6_ext_hdr(nexthdr)) {
		if (nexthdr == NEXTHDR_NONE)
			return -1;

		hp = skb_header_pointer(skb, start, sizeof(_hdr), &_hdr);
		if (!hp)
			return -1;

		exthdr_num++;

		if (nexthdr == NEXTHDR_FRAGMENT)
			hdrlen = 8;
		else if (nexthdr == NEXTHDR_AUTH)
			hdrlen = ipv6_authlen(hp);
		else
			hdrlen = ipv6_optlen(hp);

		nexthdr = hp->nexthdr;
		start += hdrlen;
	}

	return exthdr_num;
}

static void nbl_serv_set_sfp_state(void *priv, struct net_device *netdev, u8 eth_id,
				   bool open, bool is_force)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	int ret = 0;

	if (test_bit(NBL_FLAG_LINK_DOWN_ON_CLOSE, serv_mgt->flags) || is_force) {
		if (open) {
			ret = disp_ops->set_sfp_state(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						      eth_id, NBL_SFP_MODULE_ON);
			if (ret)
				netdev_info(netdev, "Fail to open sfp\n");
			else
				netdev_info(netdev, "open sfp\n");
		} else {
			ret = disp_ops->set_sfp_state(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						      eth_id, NBL_SFP_MODULE_OFF);
			if (ret)
				netdev_info(netdev, "Fail to close sfp\n");
			else
				netdev_info(netdev, "close sfp\n");
		}
	}
}

static void nbl_serv_set_netdev_carrier_state(void *priv, struct net_device *netdev, u8 link_state)
{
	struct nbl_adapter *adapter = NBL_NETDEV_TO_ADAPTER(netdev);
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = serv_mgt->net_resource_mgt;

	if (test_bit(NBL_DOWN, adapter->state))
		return;

	switch (net_resource_mgt->link_forced) {
	case IFLA_VF_LINK_STATE_AUTO:
		if (link_state) {
			if (!netif_carrier_ok(netdev)) {
				netif_carrier_on(netdev);
				netdev_info(netdev, "Set nic link up\n");
			}
		} else {
			if (netif_carrier_ok(netdev)) {
				netif_carrier_off(netdev);
				netdev_info(netdev, "Set nic link down\n");
			}
		}
		return;
	case IFLA_VF_LINK_STATE_ENABLE:
		netif_carrier_on(netdev);
		return;
	case IFLA_VF_LINK_STATE_DISABLE:
		netif_carrier_off(netdev);
		return;
	default:
		netif_carrier_on(netdev);
		return;
	}
}

static void nbl_serv_set_link_state(struct nbl_service_mgt *serv_mgt, struct net_device *netdev)
{
	struct nbl_serv_net_resource_mgt *net_resource_mgt = serv_mgt->net_resource_mgt;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	u16 vsi_id = NBL_COMMON_TO_VSI_ID(common);
	u8 eth_id = NBL_COMMON_TO_ETH_ID(common);
	struct nbl_eth_link_info eth_link_info = {0};
	int ret = 0;

	net_resource_mgt->link_forced =
		disp_ops->get_link_forced(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);

	if (net_resource_mgt->link_forced == IFLA_VF_LINK_STATE_AUTO) {
		ret = disp_ops->get_link_state(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					       eth_id, &eth_link_info);
		if (ret) {
			netdev_err(netdev, "Fail to get_link_state err %d\n", ret);
			eth_link_info.link_status = 1;
		}
	}

	nbl_serv_set_netdev_carrier_state(serv_mgt, netdev, eth_link_info.link_status);
}

static int nbl_serv_rep_netdev_open(struct net_device *netdev)
{
	int ret = 0;

	netdev_info(netdev, "Nbl rep open\n");
	ret = netif_set_real_num_tx_queues(netdev, 1);
	if (ret)
		goto fail;
	ret = netif_set_real_num_rx_queues(netdev, 1);
	if (ret)
		goto fail;

	netif_tx_start_all_queues(netdev);
	netif_carrier_on(netdev);
	netdev_info(netdev, "Nbl rep open ok!\n");

	return 0;

fail:
	return ret;
}

static int nbl_serv_rep_netdev_stop(struct net_device *netdev)
{
	netdev_info(netdev, "Nbl rep stop\n");
	netif_tx_stop_all_queues(netdev);
	netif_carrier_off(netdev);
	netif_tx_disable(netdev);
	netdev_info(netdev, "Nbl rep stop ok!\n");

	return 0;
}

int nbl_serv_vsi_open(void *priv, struct net_device *netdev, u16 vsi_index,
		      u16 real_qps, bool use_napi)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_ring_vsi_info *vsi_info = &ring_mgt->vsi_info[vsi_index];
	int ret = 0;

	if (vsi_info->started)
		return 0;

	ret = nbl_serv_setup_rings(serv_mgt, netdev, vsi_info, use_napi);
	if (ret) {
		netdev_err(netdev, "Fail to setup rings\n");
		goto setup_rings_fail;
	}

	ret = nbl_serv_setup_queues(serv_mgt, vsi_info);
	if (ret) {
		netdev_err(netdev, "Fail to setup queues\n");
		goto setup_queue_fail;
	}
	nbl_serv_flush_rx_queues(serv_mgt, vsi_info->ring_offset, vsi_info->ring_num);

	if (vsi_index == NBL_VSI_DATA)
		disp_ops->cfg_txrx_vlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					net_resource_mgt->vlan_tci, net_resource_mgt->vlan_proto,
					vsi_index);

	ret = disp_ops->cfg_dsch(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				 vsi_info->vsi_id, true);
	if (ret) {
		netdev_err(netdev, "Fail to setup dsch\n");
		goto setup_dsch_fail;
	}

	vsi_info->active_ring_num = real_qps;
	ret = disp_ops->setup_cqs(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_info->vsi_id, real_qps);
	if (ret)
		goto setup_cqs_fail;

	vsi_info->started = true;
	return 0;

setup_cqs_fail:
	disp_ops->cfg_dsch(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
			   NBL_COMMON_TO_VSI_ID(common), false);
setup_dsch_fail:
	disp_ops->remove_all_queues(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				    NBL_COMMON_TO_VSI_ID(common));
setup_queue_fail:
	nbl_serv_stop_rings(serv_mgt, vsi_info);
setup_rings_fail:
	return ret;
}

int nbl_serv_vsi_stop(void *priv, u16 vsi_index)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_ring_vsi_info *vsi_info = &ring_mgt->vsi_info[vsi_index];

	if (!vsi_info->started)
		return 0;

	vsi_info->started = false;
	/* modify defalt action and rss configuration */
	disp_ops->remove_cqs(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_info->vsi_id);

	/* disable and rest tx/rx logic queue */
	disp_ops->remove_all_queues(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_info->vsi_id);

	/* clear dsch config */
	disp_ops->cfg_dsch(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_info->vsi_id, false);
	/* free tx and rx bufs */
	nbl_serv_stop_rings(serv_mgt, vsi_info);

	return 0;
}

static int nbl_serv_switch_traffic_default_dest(void *priv, struct nbl_service_traffic_switch *info)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct net_device *dev = net_resource_mgt->netdev;
	struct nbl_netdev_priv *net_priv = netdev_priv(dev);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_flow_mgt *flow_mgt = NBL_SERV_MGT_TO_FLOW_MGT(serv_mgt);
	struct nbl_serv_vlan_node *vlan_node;
	int ret;
	u16 from_vsi, to_vsi;

	list_for_each_entry(vlan_node, &flow_mgt->vlan_list, node) {
		if (!vlan_node->vid) {
			from_vsi = net_priv->normal_vsi;
			to_vsi = info->normal_vsi;
		} else {
			from_vsi = net_priv->other_vsi;
			to_vsi = info->sync_other_vsi;
		}
		disp_ops->del_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), flow_mgt->mac,
				      vlan_node->vid, from_vsi);
		ret = disp_ops->add_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), flow_mgt->mac,
					    vlan_node->vid, to_vsi);
		if (ret) {
			netdev_err(dev, "Fail to cfg macvlan on vid %u in vsi switch",
				   vlan_node->vid);
			goto fail;
		}
	}
	/* arp/nd traffic */
	disp_ops->del_multi_rule(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), net_priv->normal_vsi);
	ret = disp_ops->add_multi_rule(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), info->normal_vsi);
	if (ret)
		goto add_multi_fail;

	/* lldp/lacp switch */
	if (info->has_lldp) {
		disp_ops->del_lldp_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), net_priv->other_vsi);
		ret = disp_ops->add_lldp_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					      info->sync_other_vsi);
		if (ret)
			goto add_lldp_fail;
	}

	if (info->has_lacp) {
		disp_ops->del_lag_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), net_priv->other_vsi);
		ret = disp_ops->add_lag_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					     info->sync_other_vsi);
		if (ret)
			goto add_lacp_fail;
	}

	net_priv->normal_vsi = info->normal_vsi;
	net_priv->other_vsi = info->sync_other_vsi;
	net_priv->async_pending_vsi = info->async_other_vsi;

	/* trigger submac update */
	net_resource_mgt->user_promisc_mode = info->promisc;
	net_resource_mgt->rxmode_set_required |= NBL_FLAG_AQ_MODIFY_MAC_FILTER;
	net_resource_mgt->rxmode_set_required |= NBL_FLAG_AQ_CONFIGURE_PROMISC_MODE;
	nbl_common_queue_work(&net_resource_mgt->rx_mode_async, false, false);

	return 0;

add_lacp_fail:
	disp_ops->add_lag_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), net_priv->other_vsi);
	if (info->has_lldp)
		disp_ops->del_lldp_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), info->sync_other_vsi);
add_lldp_fail:
	if (info->has_lldp)
		disp_ops->add_lldp_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), net_priv->other_vsi);
	disp_ops->del_multi_rule(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), info->normal_vsi);
add_multi_fail:
	disp_ops->add_multi_rule(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), net_priv->normal_vsi);
fail:
	list_for_each_entry(vlan_node, &flow_mgt->vlan_list, node) {
		if (!vlan_node->vid) {
			from_vsi = net_priv->normal_vsi;
			to_vsi = info->normal_vsi;
		} else {
			from_vsi = net_priv->other_vsi;
			to_vsi = info->sync_other_vsi;
		}
		disp_ops->del_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), flow_mgt->mac,
				      vlan_node->vid, to_vsi);
		disp_ops->add_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), flow_mgt->mac,
				      vlan_node->vid, from_vsi);
	}

	return -EINVAL;
}

static int nbl_serv_abnormal_event_to_queue(int event_type)
{
	switch (event_type) {
	case NBL_ABNORMAL_EVENT_DVN:
		return NBL_TX;
	case NBL_ABNORMAL_EVENT_UVN:
		return NBL_RX;
	default:
		return event_type;
	}
}

static int nbl_serv_stop_abnormal_sw_queue(struct nbl_service_mgt *serv_mgt,
					   u16 local_queue_id, int type)
{
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->stop_abnormal_sw_queue(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						local_queue_id, type);
}

static int nbl_serv_chan_stop_abnormal_sw_queue_req(struct nbl_service_mgt *serv_mgt,
						    u16 local_queue_id, u16 func_id, int type)
{
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);
	struct nbl_chan_param_stop_abnormal_sw_queue param = {0};
	struct nbl_chan_send_info chan_send = {0};
	int ret = 0;

	param.local_queue_id = local_queue_id;
	param.type = type;

	NBL_CHAN_SEND(chan_send, func_id, NBL_CHAN_MSG_STOP_ABNORMAL_SW_QUEUE,
		      &param, sizeof(param), NULL, 0, 1);
	ret = chan_ops->send_msg(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt), &chan_send);

	return ret;
}

static void nbl_serv_chan_stop_abnormal_sw_queue_resp(void *priv, u16 src_id, u16 msg_id,
						      void *data, u32 data_len)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);
	struct nbl_chan_param_stop_abnormal_sw_queue *param =
					(struct nbl_chan_param_stop_abnormal_sw_queue *)data;
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_serv_ring_vsi_info *vsi_info;
	struct nbl_chan_ack_info chan_ack;
	int ret = 0;

	vsi_info = &ring_mgt->vsi_info[NBL_VSI_DATA];
	if (param->local_queue_id < vsi_info->ring_offset ||
	    param->local_queue_id >= vsi_info->ring_offset + vsi_info->ring_num ||
	    !vsi_info->ring_num) {
		ret = -EINVAL;
		goto send_ack;
	}

	ret = nbl_serv_stop_abnormal_sw_queue(serv_mgt, param->local_queue_id, param->type);

send_ack:
	NBL_CHAN_ACK(chan_ack, src_id, NBL_CHAN_MSG_STOP_ABNORMAL_SW_QUEUE, msg_id,
		     ret, NULL, 0);
	chan_ops->send_ack(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt), &chan_ack);
}

static dma_addr_t nbl_serv_netdev_queue_restore(struct nbl_service_mgt *serv_mgt,
						u16 local_queue_id, int type)
{
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->restore_abnormal_ring(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					       local_queue_id, type);
}

static int nbl_serv_netdev_queue_restart(struct nbl_service_mgt *serv_mgt,
					 u16 local_queue_id, int type)
{
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->restart_abnormal_ring(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					       local_queue_id, type);
}

static dma_addr_t nbl_serv_chan_restore_netdev_queue_req(struct nbl_service_mgt *serv_mgt,
							 u16 local_queue_id, u16 func_id, int type)
{
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);
	struct nbl_chan_param_restore_queue param = {0};
	struct nbl_chan_send_info chan_send = {0};
	dma_addr_t dma = 0;
	int ret = 0;

	param.local_queue_id = local_queue_id;
	param.type = type;

	NBL_CHAN_SEND(chan_send, func_id, NBL_CHAN_MSG_RESTORE_NETDEV_QUEUE,
		      &param, sizeof(param), &dma, sizeof(dma), 1);
	ret = chan_ops->send_msg(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt), &chan_send);
	if (ret)
		return 0;

	return dma;
}

static void nbl_serv_chan_restore_netdev_queue_resp(void *priv, u16 src_id, u16 msg_id,
						    void *data, u32 data_len)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);
	struct nbl_chan_param_restore_queue *param = (struct nbl_chan_param_restore_queue *)data;
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_serv_ring_vsi_info *vsi_info;
	struct nbl_chan_ack_info chan_ack;
	dma_addr_t dma = 0;
	int ret = NBL_CHAN_RESP_OK;

	vsi_info = &ring_mgt->vsi_info[NBL_VSI_DATA];
	if (param->local_queue_id < vsi_info->ring_offset ||
	    param->local_queue_id >= vsi_info->ring_offset + vsi_info->ring_num ||
	    !vsi_info->ring_num) {
		ret = -EINVAL;
		goto send_ack;
	}

	dma = nbl_serv_netdev_queue_restore(serv_mgt, param->local_queue_id, param->type);

send_ack:
	NBL_CHAN_ACK(chan_ack, src_id, NBL_CHAN_MSG_RESTORE_NETDEV_QUEUE, msg_id,
		     ret, &dma, sizeof(dma));
	chan_ops->send_ack(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt), &chan_ack);
}

static int nbl_serv_chan_restart_netdev_queue_req(struct nbl_service_mgt *serv_mgt,
						  u16 local_queue_id, u16 func_id, int type)
{
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);
	struct nbl_chan_param_restart_queue param = {0};
	struct nbl_chan_send_info chan_send = {0};

	param.local_queue_id = local_queue_id;
	param.type = type;

	NBL_CHAN_SEND(chan_send, func_id, NBL_CHAN_MSG_RESTART_NETDEV_QUEUE,
		      &param, sizeof(param), NULL, 0, 1);
	return chan_ops->send_msg(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt), &chan_send);
}

static void nbl_serv_chan_restart_netdev_queue_resp(void *priv, u16 src_id, u16 msg_id,
						    void *data, u32 data_len)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);
	struct nbl_chan_param_restart_queue *param = (struct nbl_chan_param_restart_queue *)data;
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_serv_ring_vsi_info *vsi_info;
	struct nbl_chan_ack_info chan_ack;
	int ret = 0;

	vsi_info = &ring_mgt->vsi_info[NBL_VSI_DATA];
	if (param->local_queue_id < vsi_info->ring_offset ||
	    param->local_queue_id >= vsi_info->ring_offset + vsi_info->ring_num ||
	    !vsi_info->ring_num) {
		ret = -EINVAL;
		goto send_ack;
	}

	ret = nbl_serv_netdev_queue_restart(serv_mgt, param->local_queue_id, param->type);

send_ack:
	NBL_CHAN_ACK(chan_ack, src_id, NBL_CHAN_MSG_RESTART_NETDEV_QUEUE, msg_id,
		     ret, NULL, 0);
	chan_ops->send_ack(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt), &chan_ack);
}

static int
nbl_serv_start_abnormal_hw_queue(struct nbl_service_mgt *serv_mgt, u16 vsi_id, u16 local_queue_id,
				 dma_addr_t dma, int type)
{
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_txrx_queue_param param = {0};
	struct nbl_serv_vector *vector;
	struct nbl_serv_ring *ring;
	int ret = 0;

	switch (type) {
	case NBL_TX:
		vector = &ring_mgt->vectors[local_queue_id];
		ring = &ring_mgt->tx_rings[local_queue_id];
		ring->dma = dma;
		nbl_serv_set_queue_param(ring, ring_mgt->tx_desc_num, &param,
					 vsi_id, vector->global_vector_id);
		ret = disp_ops->setup_queue(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), &param, true);
		return ret;
	case NBL_RX:
		vector = &ring_mgt->vectors[local_queue_id];
		ring = &ring_mgt->rx_rings[local_queue_id];
		ring->dma = dma;

		nbl_serv_set_queue_param(ring, ring_mgt->rx_desc_num, &param,
					 vsi_id, vector->global_vector_id);
		ret = disp_ops->setup_queue(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), &param, false);
		return 0;
	default:
		break;
	}

	return -EINVAL;
}

static void nbl_serv_restore_queue(struct nbl_service_mgt *serv_mgt, u16 vsi_id,
				   u16 local_queue_id, u16 type, bool dif_err)
{
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	u16 func_id = disp_ops->get_function_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	u16 global_queue_id;
	dma_addr_t dma = 0;
	int ret = 0;

	while (!rtnl_trylock())
		msleep(20);

	ret = nbl_serv_chan_stop_abnormal_sw_queue_req(serv_mgt, local_queue_id, func_id, type);
	if (ret)
		goto unlock;

	ret = disp_ops->stop_abnormal_hw_queue(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id,
					       local_queue_id, type);
	if (ret)
		goto unlock;

	dma = nbl_serv_chan_restore_netdev_queue_req(serv_mgt, local_queue_id, func_id, type);
	if (!dma)
		goto unlock;

	ret = nbl_serv_start_abnormal_hw_queue(serv_mgt, vsi_id, local_queue_id, dma, type);
	if (ret)
		goto unlock;

	ret = nbl_serv_chan_restart_netdev_queue_req(serv_mgt, local_queue_id, func_id, type);
	if (ret)
		goto unlock;

	if (dif_err && type == NBL_TX) {
		global_queue_id =
			disp_ops->get_vsi_global_queue_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
							  vsi_id, local_queue_id);
		nbl_info(common, NBL_DEBUG_COMMON,
			 "dvn int_status:0, queue_id:%d\n", global_queue_id);
	}

unlock:
	rtnl_unlock();
}

static void nbl_serv_handle_tx_timeout(struct work_struct *work)
{
	struct nbl_serv_net_resource_mgt *serv_net_resource_mgt =
		container_of(work, struct nbl_serv_net_resource_mgt, tx_timeout);
	struct nbl_service_mgt *serv_mgt = serv_net_resource_mgt->serv_mgt;
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_serv_ring_vsi_info *vsi_info;
	struct nbl_serv_vector *vector;
	int i = 0;

	vsi_info = &ring_mgt->vsi_info[NBL_VSI_DATA];

	for (i = vsi_info->ring_offset; i < vsi_info->ring_offset + vsi_info->ring_num; i++) {
		if (ring_mgt->tx_rings[i].need_recovery) {
			vector = &ring_mgt->vectors[i];
			nbl_serv_restore_queue(serv_mgt, vsi_info->vsi_id, i, NBL_TX, false);
			ring_mgt->tx_rings[i].need_recovery = false;
		}
	}
}

static void nbl_serv_update_link_state(struct work_struct *work)
{
	struct nbl_serv_net_resource_mgt *serv_net_resource_mgt =
		container_of(work, struct nbl_serv_net_resource_mgt, update_link_state);
	struct nbl_service_mgt *serv_mgt = serv_net_resource_mgt->serv_mgt;

	nbl_serv_set_link_state(serv_mgt, serv_net_resource_mgt->netdev);
}

static int nbl_serv_chan_notify_link_forced_req(struct nbl_service_mgt *serv_mgt, u16 func_id)
{
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);
	struct nbl_chan_send_info chan_send = {0};

	NBL_CHAN_SEND(chan_send, func_id, NBL_CHAN_MSG_NOTIFY_LINK_FORCED, NULL, 0, NULL, 0, 1);
	return chan_ops->send_msg(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt), &chan_send);
}

static void nbl_serv_chan_notify_link_forced_resp(void *priv, u16 src_id, u16 msg_id,
						  void *data, u32 data_len)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = serv_mgt->net_resource_mgt;
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);
	struct nbl_chan_ack_info chan_ack;

	if (!net_resource_mgt)
		return;

	NBL_CHAN_ACK(chan_ack, src_id, NBL_CHAN_MSG_NOTIFY_LINK_FORCED, msg_id,
		     NBL_CHAN_RESP_OK, NULL, 0);
	chan_ops->send_ack(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt), &chan_ack);

	nbl_common_queue_work(&net_resource_mgt->update_link_state, false, false);
}

static void nbl_serv_register_link_forced_notify(struct nbl_service_mgt *serv_mgt)
{
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);

	if (!chan_ops->check_queue_exist(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt),
					 NBL_CHAN_TYPE_MAILBOX))
		return;

	chan_ops->register_msg(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt),
			       NBL_CHAN_MSG_NOTIFY_LINK_FORCED,
			       nbl_serv_chan_notify_link_forced_resp, serv_mgt);
}

static void nbl_serv_update_vlan(struct work_struct *work)
{
	struct nbl_serv_net_resource_mgt *net_resource_mgt =
		container_of(work, struct nbl_serv_net_resource_mgt, update_vlan);
	struct nbl_service_mgt *serv_mgt = net_resource_mgt->serv_mgt;
	struct net_device *netdev = net_resource_mgt->netdev;
	int was_running, err;
	u16 vid;

	vid = net_resource_mgt->vlan_tci & VLAN_VID_MASK;
	nbl_serv_update_default_vlan(serv_mgt, vid);

	rtnl_lock();
	was_running = netif_running(netdev);

	if (was_running) {
		err = nbl_serv_netdev_stop(netdev);
		if (err) {
			netdev_err(netdev, "Netdev stop failed while update_vlan\n");
			goto netdev_stop_fail;
		}

		err = nbl_serv_netdev_open(netdev);
		if (err) {
			netdev_err(netdev, "Netdev open failed after setting ringparam\n");
			goto netdev_open_fail;
		}
	}

netdev_stop_fail:
netdev_open_fail:
	rtnl_unlock();
}

static int nbl_serv_chan_notify_vlan_req(struct nbl_service_mgt *serv_mgt, u16 func_id,
					 struct nbl_serv_notify_vlan_param *param)
{
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);
	struct nbl_chan_send_info chan_send = {0};

	NBL_CHAN_SEND(chan_send, func_id, NBL_CHAN_MSG_NOTIFY_VLAN,
		      param, sizeof(*param), NULL, 0, 1);
	return chan_ops->send_msg(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt), &chan_send);
}

static void nbl_serv_chan_notify_vlan_resp(void *priv, u16 src_id, u16 msg_id,
					   void *data, u32 data_len)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);
	struct nbl_serv_notify_vlan_param *param = (struct nbl_serv_notify_vlan_param *)data;
	struct nbl_chan_ack_info chan_ack;

	net_resource_mgt->vlan_tci = param->vlan_tci;
	net_resource_mgt->vlan_proto = param->vlan_proto;

	nbl_common_queue_work(&net_resource_mgt->update_vlan, false, false);

	NBL_CHAN_ACK(chan_ack, src_id, NBL_CHAN_MSG_NOTIFY_VLAN, msg_id,
		     NBL_CHAN_RESP_OK, NULL, 0);
	chan_ops->send_ack(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt), &chan_ack);
}

static void nbl_serv_register_vlan_notify(struct nbl_service_mgt *serv_mgt)
{
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);

	if (!chan_ops->check_queue_exist(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt),
					 NBL_CHAN_TYPE_MAILBOX))
		return;

	chan_ops->register_msg(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt), NBL_CHAN_MSG_NOTIFY_VLAN,
			       nbl_serv_chan_notify_vlan_resp, serv_mgt);
}

int nbl_serv_netdev_open(struct net_device *netdev)
{
	struct nbl_adapter *adapter = NBL_NETDEV_TO_ADAPTER(netdev);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_serv_tc_mgt *tc_mgt = NBL_SERV_MGT_TO_TC_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_ADAPTER_TO_COMMON(adapter);
	struct nbl_serv_ring_vsi_info *vsi_info;
	int num_cpus, real_qps, ret = 0;
	bool netdev_open = true;

	if (!test_bit(NBL_DOWN, adapter->state))
		return -EBUSY;

	netdev_info(netdev, "Nbl open\n");

	if (ring_mgt->xdp_prog)
		nbl_event_notify(NBL_EVENT_NETDEV_STATE_CHANGE, &netdev_open,
				 NBL_COMMON_TO_VSI_ID(common), NBL_COMMON_TO_BOARD_ID(common));

	netif_carrier_off(netdev);
	nbl_serv_set_sfp_state(serv_mgt, netdev, NBL_COMMON_TO_ETH_ID(common), true, false);
	vsi_info = &ring_mgt->vsi_info[NBL_VSI_DATA];

	if (tc_mgt->num_tc) {
		real_qps = tc_mgt->total_qps;
	} else if (vsi_info->active_ring_num) {
		real_qps = vsi_info->active_ring_num;
	} else {
		num_cpus = num_online_cpus();
		real_qps = num_cpus > vsi_info->ring_num ? vsi_info->ring_num : num_cpus;
	}

	ret = nbl_serv_vsi_open(serv_mgt, netdev, NBL_VSI_DATA, real_qps, 1);
	if (ret)
		goto vsi_open_fail;

	ret = netif_set_real_num_tx_queues(netdev, real_qps);
	if (ret)
		goto setup_real_qps_fail;
	ret = netif_set_real_num_rx_queues(netdev, real_qps);
	if (ret)
		goto setup_real_qps_fail;

	netif_tx_start_all_queues(netdev);
	clear_bit(NBL_DOWN, adapter->state);
	set_bit(NBL_RUNNING, adapter->state);
	nbl_serv_set_link_state(serv_mgt, netdev);

	netdev_info(netdev, "Nbl open ok!\n");

	return 0;

setup_real_qps_fail:
	nbl_serv_vsi_stop(serv_mgt, NBL_VSI_DATA);
vsi_open_fail:
	netdev_open = false;
	if (ring_mgt->xdp_prog)
		nbl_event_notify(NBL_EVENT_NETDEV_STATE_CHANGE, &netdev_open,
				 NBL_COMMON_TO_VSI_ID(common), NBL_COMMON_TO_BOARD_ID(common));
	return ret;
}

int nbl_serv_netdev_stop(struct net_device *netdev)
{
	struct nbl_adapter *adapter = NBL_NETDEV_TO_ADAPTER(netdev);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_ADAPTER_TO_COMMON(adapter);
	bool netdev_open = false;

	if (!test_bit(NBL_RUNNING, adapter->state))
		return -EBUSY;

	netdev_info(netdev, "Nbl stop\n");
	set_bit(NBL_DOWN, adapter->state);
	clear_bit(NBL_RUNNING, adapter->state);

	nbl_serv_set_sfp_state(serv_mgt, netdev, NBL_COMMON_TO_ETH_ID(common), false, false);

	netif_tx_stop_all_queues(netdev);
	netif_carrier_off(netdev);
	netif_tx_disable(netdev);
	nbl_serv_vsi_stop(serv_mgt, NBL_VSI_DATA);

	if (ring_mgt->xdp_prog)
		nbl_event_notify(NBL_EVENT_NETDEV_STATE_CHANGE, &netdev_open,
				 NBL_COMMON_TO_VSI_ID(common), NBL_COMMON_TO_BOARD_ID(common));

	netdev_info(netdev, "Nbl stop ok!\n");

	return 0;
}

static int nbl_serv_change_mtu(struct net_device *netdev, int new_mtu)
{
	netdev->mtu = new_mtu;
	return 0;
}

static int nbl_serv_set_mac(struct net_device *dev, void *p)
{
	struct nbl_adapter *adapter = NBL_NETDEV_TO_ADAPTER(dev);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_ADAPTER_TO_COMMON(adapter);
	struct nbl_serv_flow_mgt *flow_mgt = NBL_SERV_MGT_TO_FLOW_MGT(serv_mgt);
	struct nbl_serv_vlan_node *vlan_node;
	struct sockaddr *addr = p;
	struct nbl_netdev_priv *priv = netdev_priv(dev);
	u16 vsi_id;
	int ret = 0;

	if (!is_valid_ether_addr(addr->sa_data)) {
		netdev_err(dev, "Temp to change a invalid mac address %pM\n", addr->sa_data);
		return -EADDRNOTAVAIL;
	}

	if (ether_addr_equal(dev->dev_addr, addr->sa_data))
		return 0;

	list_for_each_entry(vlan_node, &flow_mgt->vlan_list, node) {
		if (vlan_node->vid == 0)
			vsi_id = priv->normal_vsi;
		else
			vsi_id = priv->other_vsi;
		disp_ops->del_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), flow_mgt->mac,
				      vlan_node->vid, vsi_id);
		ret = disp_ops->add_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), addr->sa_data,
					    vlan_node->vid, vsi_id);
		if (ret) {
			netdev_err(dev, "Fail to cfg macvlan on vid %u", vlan_node->vid);
			goto fail;
		}
	}

	disp_ops->set_spoof_check_addr(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				       priv->normal_vsi, addr->sa_data);

	ether_addr_copy(flow_mgt->mac, addr->sa_data);
	eth_hw_addr_set(dev, addr->sa_data);

	if (!NBL_COMMON_TO_VF_CAP(common))
		disp_ops->set_eth_mac_addr(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					   addr->sa_data, NBL_COMMON_TO_ETH_ID(common));

	return 0;
fail:
	list_for_each_entry(vlan_node, &flow_mgt->vlan_list, node) {
		if (vlan_node->vid == 0)
			vsi_id = priv->normal_vsi;
		else
			vsi_id = priv->other_vsi;
		disp_ops->del_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), addr->sa_data,
				      vlan_node->vid, vsi_id);
		disp_ops->add_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), flow_mgt->mac,
				      vlan_node->vid, vsi_id);
	}
	return -EAGAIN;
}

static int nbl_serv_rx_add_vid(struct net_device *dev, __be16 proto, u16 vid)
{
	struct nbl_adapter *adapter = NBL_NETDEV_TO_ADAPTER(dev);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_ADAPTER_TO_COMMON(adapter);
	struct nbl_serv_flow_mgt *flow_mgt = NBL_SERV_MGT_TO_FLOW_MGT(serv_mgt);
	struct nbl_netdev_priv *priv = netdev_priv(dev);
	struct nbl_serv_vlan_node *vlan_node;
	u16 vsi_id = priv->other_vsi;
	int ret = 0;

	if (vid == NBL_DEFAULT_VLAN_ID)
		return 0;

	nbl_debug(common, NBL_DEBUG_COMMON, "add mac-vlan dev for proto 0x%04x, vid %u.",
		  be16_to_cpu(proto), vid);

	list_for_each_entry(vlan_node, &flow_mgt->vlan_list, node) {
		nbl_debug(common, NBL_DEBUG_COMMON, "add mac-vlan dev vid %u.", vlan_node->vid);
		if (vlan_node->vid == vid) {
			vlan_node->ref_cnt++;
			return 0;
		}
	}

	vlan_node = nbl_serv_alloc_vlan_node();
	if (!vlan_node)
		return -EAGAIN;

	ret = disp_ops->add_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				    flow_mgt->mac, vid, vsi_id);
	if (ret) {
		nbl_serv_free_vlan_node(vlan_node);
		return -EAGAIN;
	}

	vlan_node->vid = vid;
	list_add(&vlan_node->node, &flow_mgt->vlan_list);

	return 0;
}

static int nbl_serv_rx_kill_vid(struct net_device *dev, __be16 proto, u16 vid)
{
	struct nbl_adapter *adapter = NBL_NETDEV_TO_ADAPTER(dev);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_ADAPTER_TO_COMMON(adapter);
	struct nbl_serv_flow_mgt *flow_mgt = NBL_SERV_MGT_TO_FLOW_MGT(serv_mgt);
	struct nbl_netdev_priv *priv = netdev_priv(dev);
	struct nbl_serv_vlan_node *vlan_node;
	u16 vsi_id = priv->other_vsi;

	if (vid == NBL_DEFAULT_VLAN_ID)
		return 0;

	nbl_debug(common, NBL_DEBUG_COMMON, "del mac-vlan dev for proto 0x%04x, vid %u.",
		  be16_to_cpu(proto), vid);

	list_for_each_entry(vlan_node, &flow_mgt->vlan_list, node) {
		nbl_debug(common, NBL_DEBUG_COMMON, "del mac-vlan dev vid %u.", vlan_node->vid);
		if (vlan_node->vid == vid) {
			vlan_node->ref_cnt--;
			if (!vlan_node->ref_cnt) {
				disp_ops->del_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						      flow_mgt->mac, vid, vsi_id);
				list_del(&vlan_node->node);
				nbl_serv_free_vlan_node(vlan_node);
			}
			break;
		}
	}

	return 0;
}

static int nbl_serv_update_default_vlan(struct nbl_service_mgt *serv_mgt, u16 vid)
{
	struct nbl_serv_flow_mgt *flow_mgt = NBL_SERV_MGT_TO_FLOW_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_vlan_node *vlan_node;
	struct nbl_serv_vlan_node *node, *tmp;
	struct nbl_serv_submac_node *submac_node;
	struct nbl_common_info *common;
	u16 vsi;
	int ret;

	if (flow_mgt->vid == vid)
		return 0;

	common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	vsi = NBL_COMMON_TO_VSI_ID(common);
	rtnl_lock();

	/* update mac sub-interface */
	list_for_each_entry(submac_node, &flow_mgt->submac_list, node) {
		disp_ops->del_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), submac_node->mac,
				      flow_mgt->vid, vsi);
		ret = disp_ops->add_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), submac_node->mac,
					    vid, vsi);
		if (ret) {
			nbl_err(common, NBL_DEBUG_COMMON, "update vlan %u, submac %pM failed\n",
				vid, submac_node->mac);
			goto update_submac_if_failed;
		}
	}

	list_for_each_entry(vlan_node, &flow_mgt->vlan_list, node) {
		if (vlan_node->vid == vid) {
			vlan_node->ref_cnt++;
			goto free_old_vlan;
		}
	}

	/* new vlan node */
	vlan_node = nbl_serv_alloc_vlan_node();
	if (!vlan_node) {
		ret = -ENOMEM;
		goto alloc_node_failed;
	}

	ret = disp_ops->add_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), flow_mgt->mac, vid, vsi);
	if (ret)
		goto add_macvlan_failed;
	vlan_node->vid = vid;
	list_add(&vlan_node->node, &flow_mgt->vlan_list);

free_old_vlan:
	list_for_each_entry_safe(node, tmp, &flow_mgt->vlan_list, node) {
		if (node->vid == flow_mgt->vid) {
			node->ref_cnt--;
			if (!node->ref_cnt) {
				disp_ops->del_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						      flow_mgt->mac, node->vid, vsi);
				list_del(&node->node);
				nbl_serv_free_vlan_node(node);
			}
			break;
		}
	}

	flow_mgt->vid = vid;
	rtnl_unlock();

	return 0;

add_macvlan_failed:
	nbl_serv_free_vlan_node(vlan_node);
alloc_node_failed:
update_submac_if_failed:
	list_for_each_entry(submac_node, &flow_mgt->submac_list, node) {
		disp_ops->del_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), submac_node->mac,
				      vid, vsi);
		disp_ops->add_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), submac_node->mac,
				      flow_mgt->vid, vsi);
	}
	rtnl_unlock();

	return ret;
}

static void nbl_serv_get_stats64(struct net_device *netdev, struct rtnl_link_stats64 *stats)
{
	struct nbl_queue_stats queue_stats = { 0 };
	struct nbl_adapter *adapter = NBL_NETDEV_TO_ADAPTER(netdev);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_ring_vsi_info *vsi_info;
	u16 start, end;
	int i;

	vsi_info = &ring_mgt->vsi_info[NBL_VSI_DATA];
	start = vsi_info->ring_offset;
	end = vsi_info->ring_offset + vsi_info->ring_num;

	if (!stats) {
		pr_err("get_stats64 is null\n");
		return;
	}

	for (i = start; i < end; i++) {
		disp_ops->get_queue_stats(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					  i, &queue_stats, true);
		stats->tx_packets += queue_stats.packets;
		stats->tx_bytes += queue_stats.bytes;
	}

	for (i = start; i < end; i++) {
		disp_ops->get_queue_stats(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					  i, &queue_stats, false);
		stats->rx_packets += queue_stats.packets;
		stats->rx_bytes += queue_stats.bytes;
	}

	stats->multicast = 0;
	stats->rx_errors = 0;
	stats->tx_errors = 0;
	stats->rx_length_errors = 0;
	stats->rx_crc_errors = 0;
	stats->rx_frame_errors = 0;
	stats->rx_dropped = 0;
	stats->tx_dropped = 0;
}

static void nbl_modify_submacs(struct nbl_serv_net_resource_mgt *net_resource_mgt)
{
	struct netdev_hw_addr *ha;
	struct nbl_service_mgt *serv_mgt = net_resource_mgt->serv_mgt;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_flow_mgt *flow_mgt = NBL_SERV_MGT_TO_FLOW_MGT(serv_mgt);
	struct nbl_netdev_priv *priv = netdev_priv(net_resource_mgt->netdev);
	struct nbl_serv_submac_node *submac_node;
	int uc_count, i, ret = 0;
	u8 *buf = NULL;
	u16 len;

	spin_lock_bh(&net_resource_mgt->mac_vlan_list_lock);
	uc_count = netdev_uc_count(net_resource_mgt->netdev);

	if (uc_count) {
		len = uc_count * ETH_ALEN;
		buf = kzalloc(len, GFP_ATOMIC);

		if (!buf) {
			spin_unlock_bh(&net_resource_mgt->mac_vlan_list_lock);
			return;
		}

		i = 0;
		netdev_hw_addr_list_for_each(ha, &net_resource_mgt->netdev->uc) {
			if (i >= len)
				break;
			memcpy(&buf[i], ha->addr, ETH_ALEN);
			i += ETH_ALEN;
		}

		net_resource_mgt->rxmode_set_required &= ~NBL_FLAG_AQ_MODIFY_MAC_FILTER;
	}
	spin_unlock_bh(&net_resource_mgt->mac_vlan_list_lock);

	nbl_serv_del_all_submacs(serv_mgt, priv->async_other_vsi);

	for (i = 0; i < uc_count; i++) {
		submac_node = nbl_serv_alloc_submac_node();
		if (!submac_node)
			break;

		ret = disp_ops->add_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), &buf[i * ETH_ALEN],
					    flow_mgt->vid, priv->async_pending_vsi);
		if (ret) {
			nbl_serv_free_submac_node(submac_node);
			break;
		}

		ether_addr_copy(submac_node->mac, &buf[i * ETH_ALEN]);
		list_add(&submac_node->node, &flow_mgt->submac_list);
	}

	kfree(buf);
	priv->async_other_vsi = priv->async_pending_vsi;
}

static void nbl_modify_promisc_mode(struct nbl_serv_net_resource_mgt *net_resource_mgt)
{
	struct nbl_netdev_priv *priv = netdev_priv(net_resource_mgt->netdev);
	struct nbl_service_mgt *serv_mgt = net_resource_mgt->serv_mgt;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	u16 mode = 0;

	spin_lock_bh(&net_resource_mgt->current_netdev_promisc_flags_lock);
	if (net_resource_mgt->curr_promiscuout_mode & (IFF_PROMISC | IFF_ALLMULTI))
		mode = 1;

	if (net_resource_mgt->user_promisc_mode)
		mode = 1;

	net_resource_mgt->rxmode_set_required &= ~NBL_FLAG_AQ_CONFIGURE_PROMISC_MODE;
	spin_unlock_bh(&net_resource_mgt->current_netdev_promisc_flags_lock);

	disp_ops->set_promisc_mode(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				   priv->async_other_vsi, mode);
}

static struct nbl_mac_filter *nbl_find_filter(struct nbl_adapter *adapter, const u8 *macaddr)
{
	struct nbl_service_mgt *serv_mgt;
	struct nbl_serv_net_resource_mgt *net_resource_mgt;
	struct nbl_mac_filter *f;

	if (!macaddr)
		return NULL;

	serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	list_for_each_entry(f, &net_resource_mgt->mac_filter_list, list) {
		if (ether_addr_equal(macaddr, f->macaddr))
			return f;
	}

	return NULL;
}

static void nbl_free_filter(struct nbl_serv_net_resource_mgt *net_resource_mgt)
{
	struct nbl_mac_filter *f;
	struct list_head *pos, *n;

	list_for_each_safe(pos, n, &net_resource_mgt->mac_filter_list) {
		f = list_entry(pos, struct nbl_mac_filter, list);
		list_del(&f->list);
		kfree(f);
	}
}

static struct nbl_mac_filter *nbl_add_filter(struct nbl_adapter *adapter, const u8 *macaddr)
{
	struct nbl_mac_filter *f;
	struct nbl_service_mgt *serv_mgt;
	struct nbl_serv_net_resource_mgt *net_resource_mgt;

	if (!macaddr)
		return NULL;

	serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);

	f = nbl_find_filter(adapter, macaddr);
	if (!f) {
		f = kzalloc(sizeof(*f), GFP_ATOMIC);
		if (!f)
			return f;

		ether_addr_copy(f->macaddr, macaddr);
		list_add_tail(&f->list, &net_resource_mgt->mac_filter_list);
		net_resource_mgt->rxmode_set_required |= NBL_FLAG_AQ_MODIFY_MAC_FILTER;
	}

	return f;
}

static int nbl_addr_unsync(struct net_device *netdev, const u8 *addr)
{
	struct nbl_adapter *adapter;
	struct nbl_mac_filter *f;
	struct nbl_service_mgt *serv_mgt;
	struct nbl_serv_net_resource_mgt *net_resource_mgt;

	adapter = NBL_NETDEV_TO_ADAPTER(netdev);
	serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);

	if (ether_addr_equal(addr, netdev->dev_addr))
		return 0;

	f = nbl_find_filter(adapter, addr);
	if (f) {
		list_del(&f->list);
		kfree(f);
		net_resource_mgt->rxmode_set_required |= NBL_FLAG_AQ_MODIFY_MAC_FILTER;
	}

	return 0;
}

static int nbl_addr_sync(struct net_device *netdev, const u8 *addr)
{
	struct nbl_adapter *adapter;

	adapter = NBL_NETDEV_TO_ADAPTER(netdev);
	if (ether_addr_equal(addr, netdev->dev_addr))
		return 0;

	if (nbl_add_filter(adapter, addr))
		return 0;
	else
		return -ENOMEM;
}

static bool nbl_serv_promisc_mode_changed(struct net_device *dev)
{
	struct nbl_adapter *adapter;
	struct nbl_service_mgt *serv_mgt;
	struct nbl_serv_net_resource_mgt *net_resource_mgt;

	adapter = NBL_NETDEV_TO_ADAPTER(dev);
	serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);

	return (net_resource_mgt->curr_promiscuout_mode ^ dev->flags)
		& (IFF_PROMISC | IFF_ALLMULTI);
}

static void nbl_serv_set_rx_mode(struct net_device *dev)
{
	struct nbl_adapter *adapter;
	struct nbl_service_mgt *serv_mgt;
	struct nbl_serv_net_resource_mgt *net_resource_mgt;

	adapter = NBL_NETDEV_TO_ADAPTER(dev);
	serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);

	spin_lock_bh(&net_resource_mgt->mac_vlan_list_lock);
	__dev_uc_sync(dev, nbl_addr_sync, nbl_addr_unsync);
	spin_unlock_bh(&net_resource_mgt->mac_vlan_list_lock);

	if (!NBL_COMMON_TO_VF_CAP(NBL_SERV_MGT_TO_COMMON(serv_mgt))) { /* only pf support */
		spin_lock_bh(&net_resource_mgt->current_netdev_promisc_flags_lock);
		if (nbl_serv_promisc_mode_changed(dev)) {
			net_resource_mgt->rxmode_set_required |= NBL_FLAG_AQ_CONFIGURE_PROMISC_MODE;
			net_resource_mgt->curr_promiscuout_mode = dev->flags;
		}
		spin_unlock_bh(&net_resource_mgt->current_netdev_promisc_flags_lock);
	}

	nbl_common_queue_work(&net_resource_mgt->rx_mode_async, false, false);
}

static void nbl_serv_change_rx_flags(struct net_device *dev, int flag)
{
	struct nbl_adapter *adapter;
	struct nbl_service_mgt *serv_mgt;
	struct nbl_serv_net_resource_mgt *net_resource_mgt;

	adapter = NBL_NETDEV_TO_ADAPTER(dev);
	serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);

	spin_lock_bh(&net_resource_mgt->current_netdev_promisc_flags_lock);
	if (nbl_serv_promisc_mode_changed(dev)) {
		net_resource_mgt->rxmode_set_required |= NBL_FLAG_AQ_CONFIGURE_PROMISC_MODE;
		net_resource_mgt->curr_promiscuout_mode = dev->flags;
	}
	spin_unlock_bh(&net_resource_mgt->current_netdev_promisc_flags_lock);

	nbl_common_queue_work(&net_resource_mgt->rx_mode_async, false, false);
}

static netdev_features_t
nbl_serv_features_check(struct sk_buff *skb, struct net_device *dev, netdev_features_t features)
{
	u32 l2_l3_hrd_len = 0, l4_hrd_len = 0, total_hrd_len = 0;
	u8 l4_proto = 0;
	__be16 protocol, frag_off;
	int ret;
	unsigned char *exthdr;
	unsigned int offset = 0;
	int nexthdr = 0;
	int exthdr_num = 0;
	union {
		struct iphdr *v4;
		struct ipv6hdr *v6;
		unsigned char *hdr;
	} ip;
	union {
		struct tcphdr *tcp;
		struct udphdr *udp;
		unsigned char *hdr;
	} l4;

	/* No point in doing any of this if neither checksum nor GSO are
	 * being requested for this frame. We can rule out both by just
	 * checking for CHECKSUM_PARTIAL.
	 */
	if (skb->ip_summed != CHECKSUM_PARTIAL)
		return features;

	/* We cannot support GSO if the MSS is going to be less than
	 * 256 bytes or bigger than 16383 bytes. If it is then we need to drop support for GSO.
	 */
	if (skb_is_gso(skb) && (skb_shinfo(skb)->gso_size < NBL_TX_TSO_MSS_MIN ||
				skb_shinfo(skb)->gso_size > NBL_TX_TSO_MSS_MAX))
		features &= ~NETIF_F_GSO_MASK;

	l2_l3_hrd_len = (u32)(skb_transport_header(skb) - skb->data);

	ip.hdr = skb_network_header(skb);
	l4.hdr = skb_transport_header(skb);
	protocol = vlan_get_protocol(skb);

	if (protocol == htons(ETH_P_IP)) {
		l4_proto = ip.v4->protocol;
	} else if (protocol == htons(ETH_P_IPV6)) {
		exthdr = ip.hdr + sizeof(*ip.v6);
		l4_proto = ip.v6->nexthdr;
		if (l4.hdr != exthdr) {
			ret = ipv6_skip_exthdr(skb, exthdr - skb->data, &l4_proto, &frag_off);
			if (ret < 0)
				goto out_rm_features;
		}

		/* IPV6 extension headers
		 * (1) donot support routing and destination extension headers
		 * (2) support 2 extension headers mostly
		 */
		nexthdr = ipv6_find_hdr(skb, &offset, NEXTHDR_ROUTING, NULL, NULL);
		if (nexthdr == NEXTHDR_ROUTING) {
			netdev_info(dev, "skb contain ipv6 routing ext header\n");
			goto out_rm_features;
		}

		nexthdr = ipv6_find_hdr(skb, &offset, NEXTHDR_DEST, NULL, NULL);
		if (nexthdr == NEXTHDR_DEST) {
			netdev_info(dev, "skb contain ipv6 routing dest header\n");
			goto out_rm_features;
		}

		exthdr_num = nbl_serv_ipv6_exthdr_num(skb, exthdr - skb->data, ip.v6->nexthdr);
		if (exthdr_num < 0 || exthdr_num > 2) {
			netdev_info(dev, "skb ipv6 exthdr_num:%d\n", exthdr_num);
			goto out_rm_features;
		}
	} else {
		goto out_rm_features;
	}

	switch (l4_proto) {
	case IPPROTO_TCP:
		l4_hrd_len = (l4.tcp->doff) * 4;
		break;
	case IPPROTO_UDP:
		l4_hrd_len = sizeof(struct udphdr);
		break;
	case IPPROTO_SCTP:
		l4_hrd_len = sizeof(struct sctphdr);
		break;
	default:
		goto out_rm_features;
	}

	total_hrd_len = l2_l3_hrd_len + l4_hrd_len;

	// TX checksum offload support total header len is [0, 255]
	if (total_hrd_len > NBL_TX_CHECKSUM_OFFLOAD_L2L3L4_HDR_LEN_MAX)
		goto out_rm_features;

	// TSO support total header len is [42, 128]
	if (total_hrd_len < NBL_TX_TSO_L2L3L4_HDR_LEN_MIN ||
	    total_hrd_len > NBL_TX_TSO_L2L3L4_HDR_LEN_MAX)
		features &= ~NETIF_F_GSO_MASK;

	if (skb->encapsulation)
		goto out_rm_features;

	return features;

out_rm_features:
	return features & ~(NETIF_F_IP_CSUM |
			    NETIF_F_IPV6_CSUM |
			    NETIF_F_SCTP_CRC |
			    NETIF_F_GSO_MASK);
}

static int nbl_serv_set_features(struct net_device *netdev, netdev_features_t features)
{
	struct nbl_netdev_priv *priv = netdev_priv(netdev);
	struct nbl_adapter *adapter = NBL_NETDEV_PRIV_TO_ADAPTER(priv);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	netdev_features_t changed = netdev->features ^ features;
	u16 vsi_id = NBL_COMMON_TO_VSI_ID(common);

	if (changed & NETIF_F_NTUPLE) {
		bool ena = !!(features & NETIF_F_NTUPLE);

		disp_ops->config_fd_flow_state(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					       NBL_CHAN_FDIR_RULE_NORMAL, vsi_id, ena);
	}

	return 0;
}

static int nbl_serv_config_fd_flow_state(void *priv, enum nbl_chan_fdir_rule_type type, u32 state)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	u16 vsi_id = NBL_COMMON_TO_VSI_ID(common);

	return disp_ops->config_fd_flow_state(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					      type, vsi_id, state);
}

static LIST_HEAD(nbl_serv_block_cb_list);

static int nbl_serv_setup_tc(struct net_device *dev, enum tc_setup_type type, void *type_data)
{
	struct nbl_netdev_priv *priv = netdev_priv(dev);

	switch (type) {
	case TC_SETUP_BLOCK: {
		return flow_block_cb_setup_simple((struct flow_block_offload *)type_data,
						  &nbl_serv_block_cb_list,
						  nbl_serv_setup_tc_block_cb,
						  priv, priv, true);
	}
	case TC_SETUP_QDISC_MQPRIO:
		return nbl_serv_setup_tc_mqprio(dev, type_data);
	default:
		return -EOPNOTSUPP;
	}
}

static int nbl_serv_set_vf_mac(struct net_device *dev, int vf_id, u8 *mac)
{
	struct nbl_service_mgt *serv_mgt = NBL_NETDEV_TO_SERV_MGT(dev);
	struct nbl_serv_net_resource_mgt *net_resource_mgt =
					NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	u16 function_id = U16_MAX;

	if (vf_id >= net_resource_mgt->total_vfs || !net_resource_mgt->vf_info)
		return -EINVAL;

	function_id = disp_ops->get_vf_function_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						   NBL_COMMON_TO_VSI_ID(common), vf_id);

	if (function_id == U16_MAX) {
		netdev_info(dev, "vf id %d invalid\n", vf_id);
		return -EINVAL;
	}

	ether_addr_copy(net_resource_mgt->vf_info[vf_id].mac, mac);

	disp_ops->register_func_mac(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), mac, function_id);

	return 0;
}

static int nbl_serv_set_vf_rate(struct net_device *dev, int vf_id, int min_rate, int max_rate)
{
	struct nbl_service_mgt *serv_mgt = NBL_NETDEV_TO_SERV_MGT(dev);
	struct nbl_serv_net_resource_mgt *net_resource_mgt =
					NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	u16 function_id = U16_MAX;
	int ret = 0;

	if (vf_id >= net_resource_mgt->total_vfs || !net_resource_mgt->vf_info || min_rate > 0)
		return -EINVAL;

	if (vf_id < net_resource_mgt->num_vfs) {
		function_id = disp_ops->get_vf_function_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
							   NBL_COMMON_TO_VSI_ID(common), vf_id);

		if (function_id == U16_MAX) {
			netdev_info(dev, "vf id %d invalid\n", vf_id);
			return -EINVAL;
		}

		ret = disp_ops->set_tx_rate(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					    function_id, max_rate);
	}

	if (!ret)
		net_resource_mgt->vf_info[vf_id].max_tx_rate = max_rate;

	ret = disp_ops->register_func_rate(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					   function_id, max_rate);

	return ret;
}

static int nbl_serv_set_vf_spoofchk(struct net_device *dev, int vf_id, bool ena)
{
	struct nbl_service_mgt *serv_mgt = NBL_NETDEV_TO_SERV_MGT(dev);
	struct nbl_serv_net_resource_mgt *net_resource_mgt =
					NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	int ret = 0;

	if (vf_id >= net_resource_mgt->total_vfs || !net_resource_mgt->vf_info)
		return -EINVAL;

	if (vf_id < net_resource_mgt->num_vfs)
		ret = disp_ops->set_vf_spoof_check(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						   NBL_COMMON_TO_VSI_ID(common), vf_id, ena);

	if (!ret)
		net_resource_mgt->vf_info[vf_id].spoof_check = ena;

	return ret;
}

static int nbl_serv_set_vf_link_state(struct net_device *dev, int vf_id, int link_state)
{
	struct nbl_service_mgt *serv_mgt = NBL_NETDEV_TO_SERV_MGT(dev);
	struct nbl_serv_net_resource_mgt *net_resource_mgt =
					NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	u16 function_id = U16_MAX;
	bool should_notify = false;
	int ret = 0;

	if (vf_id >= net_resource_mgt->total_vfs || !net_resource_mgt->vf_info)
		return -EINVAL;

	function_id = disp_ops->get_vf_function_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						   NBL_COMMON_TO_VSI_ID(common), vf_id);

	if (function_id == U16_MAX) {
		netdev_info(dev, "vf id %d invalid\n", vf_id);
		return -EINVAL;
	}

	ret = disp_ops->register_func_link_forced(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						  function_id, link_state, &should_notify);
	if (!ret && should_notify)
		nbl_serv_chan_notify_link_forced_req(serv_mgt, function_id);

	if (!ret)
		net_resource_mgt->vf_info[vf_id].state = link_state;

	return ret;
}

static int nbl_serv_set_vf_vlan(struct net_device *dev, int vf_id, u16 vlan, u8 qos, __be16 proto)
{
	struct nbl_service_mgt *serv_mgt = NBL_NETDEV_TO_SERV_MGT(dev);
	struct nbl_serv_net_resource_mgt *net_resource_mgt =
					NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_notify_vlan_param param = {0};
	int ret = 0;
	u16 function_id = U16_MAX;
	bool should_notify = false;

	if (vf_id >= net_resource_mgt->total_vfs || !net_resource_mgt->vf_info)
		return -EINVAL;

	if (vlan > 4095 || qos > 7)
		return -EINVAL;

	function_id = disp_ops->get_vf_function_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						   NBL_COMMON_TO_VSI_ID(common), vf_id);

	if (function_id == U16_MAX) {
		netdev_info(dev, "vf id %d invalid\n", vf_id);
		return -EINVAL;
	}

	if (vlan) {
		param.vlan_tci = (vlan & VLAN_VID_MASK) | (qos << VLAN_PRIO_SHIFT);
		param.vlan_proto = ntohs(proto);
	} else {
		proto = 0;
		qos = 0;
	}

	ret = disp_ops->register_func_vlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), function_id,
					   param.vlan_tci, param.vlan_proto,
					   &should_notify);
	if (should_notify && !ret) {
		ret = nbl_serv_chan_notify_vlan_req(serv_mgt, function_id, &param);
		if (ret)
			disp_ops->register_func_vlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						     function_id, 0, 0, &should_notify);
	}
	if (!ret) {
		net_resource_mgt->vf_info[vf_id].vlan = vlan;
		net_resource_mgt->vf_info[vf_id].vlan_proto = ntohs(proto);
		net_resource_mgt->vf_info[vf_id].vlan_qos = qos;
	}

	return ret;
}

static int nbl_serv_get_vf_config(struct net_device *dev, int vf_id, struct ifla_vf_info *ivi)
{
	struct nbl_netdev_priv *priv = netdev_priv(dev);
	struct nbl_adapter *adapter = NBL_NETDEV_PRIV_TO_ADAPTER(priv);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_serv_net_resource_mgt *net_resource_mgt =
					NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_serv_vf_info *vf_info = net_resource_mgt->vf_info;

	if (vf_id >= net_resource_mgt->total_vfs || !net_resource_mgt->vf_info)
		return -EINVAL;

	ivi->vf = vf_id;
	ivi->spoofchk = vf_info[vf_id].spoof_check;
	ivi->linkstate = vf_info[vf_id].state;
	ivi->max_tx_rate = vf_info[vf_id].max_tx_rate;
	ivi->vlan = vf_info[vf_id].vlan;
	ivi->vlan_proto = htons(vf_info[vf_id].vlan_proto);
	ivi->qos = vf_info[vf_id].vlan_qos;
	ether_addr_copy(ivi->mac, vf_info[vf_id].mac);

	return 0;
}

static u8 nbl_get_dscp_up(struct nbl_serv_net_resource_mgt *net_resource_mgt, struct sk_buff *skb)
{
	u8 dscp = 0;

	if (skb->protocol == htons(ETH_P_IP))
		dscp = ipv4_get_dsfield(ip_hdr(skb)) >> 2;
	else if (skb->protocol == htons(ETH_P_IPV6))
		dscp = ipv6_get_dsfield(ipv6_hdr(skb)) >> 2;

	return net_resource_mgt->dscp2prio_map[dscp];
}

static u16
nbl_serv_select_queue(struct net_device *netdev, struct sk_buff *skb,
		      struct net_device *sb_dev)
{
	struct nbl_netdev_priv *priv = netdev_priv(netdev);
	struct nbl_adapter *adapter = NBL_NETDEV_PRIV_TO_ADAPTER(priv);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_serv_net_resource_mgt *net_resource_mgt =
					NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);

	if (net_resource_mgt->pfc_mode == NBL_TRUST_MODE_DSCP)
		skb->priority = nbl_get_dscp_up(net_resource_mgt, skb);

	return netdev_pick_tx(netdev, skb, sb_dev);
}

static void nbl_serv_tx_timeout(struct net_device *netdev, unsigned int txqueue)
{
	struct nbl_netdev_priv *priv = netdev_priv(netdev);
	struct nbl_adapter *adapter = NBL_NETDEV_PRIV_TO_ADAPTER(priv);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_serv_ring_vsi_info *vsi_info;

	vsi_info = &ring_mgt->vsi_info[NBL_VSI_DATA];

	ring_mgt->tx_rings[vsi_info->ring_offset + txqueue].need_recovery = true;
	ring_mgt->tx_rings[vsi_info->ring_offset + txqueue].tx_timeout_count++;

	nbl_warn(common, NBL_DEBUG_QUEUE, "TX timeout on queue %d", txqueue);

	nbl_common_queue_work(&NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt)->tx_timeout, false, false);
}

static int nbl_serv_bridge_getlink(struct sk_buff *skb, u32 pid, u32 seq,
				   struct net_device *netdev, u32 filter_mask, int nlflags)
{
	struct nbl_netdev_priv *priv = netdev_priv(netdev);
	struct nbl_adapter *adapter = NBL_NETDEV_PRIV_TO_ADAPTER(priv);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_serv_net_resource_mgt *net_resource_mgt = serv_mgt->net_resource_mgt;
	u16 bmode;

	bmode = net_resource_mgt->bridge_mode;

	return ndo_dflt_bridge_getlink(skb, pid, seq, netdev, bmode, 0, 0, nlflags,
				       filter_mask, NULL);
}

static int nbl_serv_bridge_setlink(struct net_device *netdev, struct nlmsghdr *nlh,
				   u16 flags, struct netlink_ext_ack *extack)
{
	struct nbl_netdev_priv *priv = netdev_priv(netdev);
	struct nbl_adapter *adapter = NBL_NETDEV_PRIV_TO_ADAPTER(priv);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_serv_net_resource_mgt *net_resource_mgt = serv_mgt->net_resource_mgt;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nlattr *attr, *br_spec;
	u16 mode;
	int ret, rem;

	br_spec = nlmsg_find_attr(nlh, sizeof(struct ifinfomsg), IFLA_AF_SPEC);

	nla_for_each_nested(attr, br_spec, rem) {
		if (nla_type(attr) != IFLA_BRIDGE_MODE)
			continue;

		mode = nla_get_u16(attr);
		if (mode != BRIDGE_MODE_VEPA && mode != BRIDGE_MODE_VEB)
			return -EINVAL;

		if (mode == net_resource_mgt->bridge_mode)
			continue;

		ret = disp_ops->set_bridge_mode(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), mode);
		if (ret) {
			netdev_info(netdev, "bridge_setlink failed 0x%x", ret);
			return ret;
		}

		net_resource_mgt->bridge_mode = mode;
	}

	return 0;
}

static int nbl_serv_get_phys_port_name(struct net_device *dev, char *name, size_t len)
{
	struct nbl_common_info *common = NBL_NETDEV_TO_COMMON(dev);
	u8 pf_id;

	pf_id = common->eth_id;
	if ((NBL_COMMON_TO_ETH_MODE(common) == NBL_TWO_ETHERNET_PORT) && common->eth_id == 2)
		pf_id = 1;

	if (snprintf(name, len, "p%u", pf_id) >= len)
		return -EINVAL;
	return 0;
}

static int nbl_serv_get_port_parent_id(struct net_device *dev, struct netdev_phys_item_id *ppid)
{
	struct nbl_netdev_priv *priv = netdev_priv(dev);
	struct nbl_adapter *adapter = NBL_NETDEV_PRIV_TO_ADAPTER(priv);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	u8 mac[ETH_ALEN];

	/* return success to avoid linkwatch_do_dev report warnning */
	if (test_bit(NBL_FATAL_ERR, adapter->state))
		return 0;

	disp_ops->get_base_mac_addr(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), mac);

	ppid->id_len = ETH_ALEN;
	memcpy(&ppid->id, mac, ppid->id_len);

	return 0;
}

static int nbl_serv_register_net(void *priv, struct nbl_register_net_param *register_param,
				 struct nbl_register_net_result *register_result)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	int p4_type, ret = 0;

	ret = disp_ops->register_net(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				     register_param, register_result);
	if (ret)
		return ret;

	p4_type = disp_ops->get_p4_used(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	switch (p4_type) {
	case NBL_P4_DEFAULT:
		set_bit(NBL_FLAG_P4_DEFAULT, serv_mgt->flags);
		break;
	default:
		nbl_warn(NBL_SERV_MGT_TO_COMMON(serv_mgt), NBL_DEBUG_CUSTOMIZED_P4,
			 "Unknown P4 type %d", p4_type);
	}

	return 0;
}

static int nbl_serv_unregister_net(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->unregister_net(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_setup_txrx_queues(void *priv, u16 vsi_id, u16 queue_num, u16 net_vector_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_vector *vector;
	int i, ret = 0;

	/* Clear cfgs, in case this function exited abnormaly last time */
	disp_ops->clear_queues(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);

	/* queue_num include user&kernel queue */
	ret = disp_ops->alloc_txrx_queues(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id, queue_num);
	if (ret)
		return -EFAULT;

	/* ring_mgt->tx_ring_number only for kernel use */
	for (i = 0; i < ring_mgt->tx_ring_num; i++) {
		ring_mgt->tx_rings[i].local_queue_id = NBL_PAIR_ID_GET_TX(i);
		ring_mgt->rx_rings[i].local_queue_id = NBL_PAIR_ID_GET_RX(i);
	}

	for (i = 0; i < ring_mgt->xdp_ring_offset; i++) {
		vector = &ring_mgt->vectors[i];
		vector->local_vector_id = i + net_vector_id;
		vector->global_vector_id =
			disp_ops->get_global_vector(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						    vsi_id, vector->local_vector_id);
		vector->irq_enable_base =
			disp_ops->get_msix_irq_enable_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
							   vector->global_vector_id,
							   &vector->irq_data);

		disp_ops->set_vector_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					  vector->irq_enable_base,
					  vector->irq_data, i,
					  ring_mgt->net_msix_mask_en);
	}

	return 0;
}

static void nbl_serv_remove_txrx_queues(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_ring_mgt *ring_mgt;
	struct nbl_dispatch_ops *disp_ops;

	ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->free_txrx_queues(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
}

static int nbl_serv_init_tx_rate(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	u16 func_id;
	int ret = 0;

	if (net_resource_mgt->max_tx_rate) {
		func_id = disp_ops->get_function_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
		ret = disp_ops->set_tx_rate(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					    func_id, net_resource_mgt->max_tx_rate);
	}

	return ret;
}

static int nbl_serv_setup_q2vsi(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->setup_q2vsi(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
}

static void nbl_serv_remove_q2vsi(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->remove_q2vsi(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
}

static int nbl_serv_setup_rss(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->setup_rss(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
}

static void nbl_serv_remove_rss(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->remove_rss(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
}

static int nbl_serv_alloc_rings(void *priv, struct net_device *netdev, struct nbl_ring_param *param)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct device *dev;
	struct nbl_serv_ring_mgt *ring_mgt;
	struct nbl_dispatch_ops *disp_ops;
	int ret = 0;

	dev = NBL_SERV_MGT_TO_DEV(serv_mgt);
	ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	ring_mgt->tx_ring_num = param->tx_ring_num;
	ring_mgt->rx_ring_num = param->rx_ring_num;
	ring_mgt->tx_desc_num = param->queue_size;
	ring_mgt->rx_desc_num = param->queue_size;
	ring_mgt->xdp_ring_offset = param->xdp_ring_offset;

	ret = disp_ops->alloc_rings(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), netdev, param);
	if (ret)
		goto alloc_rings_fail;

	ret = nbl_serv_set_tx_rings(ring_mgt, netdev, dev);
	if (ret)
		goto set_tx_fail;

	ret = nbl_serv_set_rx_rings(ring_mgt, netdev, dev);
	if (ret)
		goto set_rx_fail;

	ret = nbl_serv_set_vectors(serv_mgt, netdev, dev);
	if (ret)
		goto set_vectors_fail;

	ret = nbl_serv_register_xdp_rxq(serv_mgt, ring_mgt);
	if (ret)
		goto register_xdp_err;

	return 0;

register_xdp_err:
	nbl_serv_remove_vectors(ring_mgt, dev);
set_vectors_fail:
	nbl_serv_remove_rx_ring(ring_mgt, dev);
set_rx_fail:
	nbl_serv_remove_tx_ring(ring_mgt, dev);
set_tx_fail:
	disp_ops->remove_rings(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
alloc_rings_fail:
	return ret;
}

static void nbl_serv_free_rings(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct device *dev;
	struct nbl_serv_ring_mgt *ring_mgt;
	struct nbl_dispatch_ops *disp_ops;

	dev = NBL_SERV_MGT_TO_DEV(serv_mgt);
	ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	nbl_serv_unregister_xdp_rxq(serv_mgt, ring_mgt);
	nbl_serv_remove_vectors(ring_mgt, dev);
	nbl_serv_remove_rx_ring(ring_mgt, dev);
	nbl_serv_remove_tx_ring(ring_mgt, dev);

	disp_ops->remove_rings(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_enable_napis(void *priv, u16 vsi_index)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_serv_ring_vsi_info *vsi_info = &ring_mgt->vsi_info[vsi_index];
	u16 start = vsi_info->ring_offset, end = vsi_info->ring_offset + vsi_info->ring_num;
	int i;

	for (i = start; i < end; i++)
		napi_enable(ring_mgt->vectors[i].napi);

	return 0;
}

static void nbl_serv_disable_napis(void *priv, u16 vsi_index)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_serv_ring_vsi_info *vsi_info = &ring_mgt->vsi_info[vsi_index];
	u16 start = vsi_info->ring_offset, end = vsi_info->ring_offset + vsi_info->ring_num;
	int i;

	for (i = start; i < end; i++)
		napi_disable(ring_mgt->vectors[i].napi);
}

static void nbl_serv_set_mask_en(void *priv, bool enable)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_ring_mgt *ring_mgt;

	ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);

	ring_mgt->net_msix_mask_en = enable;
}

static int nbl_serv_start_net_flow(void *priv, struct net_device *netdev, u16 vsi_id, u16 vid)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_flow_mgt *flow_mgt = NBL_SERV_MGT_TO_FLOW_MGT(serv_mgt);
	struct nbl_serv_vlan_node *vlan_node;
	int ret = 0;

	/* Clear cfgs, in case this function exited abnormaly last time */
	disp_ops->clear_accel_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
	disp_ops->clear_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
	if (!common->is_vf)
		disp_ops->config_fd_flow_state(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					       NBL_CHAN_FDIR_RULE_NORMAL, vsi_id, 1);

	if (!list_empty(&flow_mgt->vlan_list))
		return -ECONNRESET;

	ret = disp_ops->add_multi_rule(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
	if (ret)
		goto add_multi_fail;

	vlan_node = nbl_serv_alloc_vlan_node();
	if (!vlan_node)
		goto alloc_fail;

	flow_mgt->vid = vid;
	ether_addr_copy(flow_mgt->mac, netdev->dev_addr);
	ret = disp_ops->add_macvlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), flow_mgt->mac,
				    vid, vsi_id);
	if (ret)
		goto add_macvlan_fail;

	vlan_node->vid = vid;
	list_add(&vlan_node->node, &flow_mgt->vlan_list);

	return 0;

add_macvlan_fail:
	nbl_serv_free_vlan_node(vlan_node);
alloc_fail:
	disp_ops->del_multi_rule(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
add_multi_fail:
	return ret;
}

static void nbl_serv_stop_net_flow(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct net_device *dev = net_resource_mgt->netdev;
	struct nbl_netdev_priv *net_priv = netdev_priv(dev);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_flow_mgt *flow_mgt = NBL_SERV_MGT_TO_FLOW_MGT(serv_mgt);

	nbl_serv_del_all_vlans(serv_mgt);
	nbl_serv_del_all_submacs(serv_mgt, net_priv->async_other_vsi);
	if (!common->is_vf)
		disp_ops->config_fd_flow_state(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					       NBL_CHAN_FDIR_RULE_NORMAL, vsi_id, 0);

	disp_ops->del_multi_rule(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);

	disp_ops->set_vf_spoof_check(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				     vsi_id, -1, false);
	memset(flow_mgt->mac, 0, sizeof(flow_mgt->mac));
}

static int nbl_serv_set_lldp_flow(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->add_lldp_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
}

static void nbl_serv_remove_lldp_flow(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->del_lldp_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
}

static int nbl_serv_start_mgt_flow(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->setup_multi_group(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static void nbl_serv_stop_mgt_flow(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->remove_multi_group(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static u32 nbl_serv_get_tx_headroom(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_tx_headroom(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

/**
 * This ops get flexible product capability from ctrl device, if the device has not manager cap, it
 * need get capability from ctr device by channel
 */
static bool nbl_serv_get_product_flex_cap(void *priv, enum nbl_flex_cap_type cap_type)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_product_flex_cap(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						       cap_type);
}

/**
 * This ops get fix product capability from resource layer, this capability fix by product_type, no
 * need get from ctrl device
 */
static bool nbl_serv_get_product_fix_cap(void *priv, enum nbl_fix_cap_type cap_type)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_product_fix_cap(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						       cap_type);
}

static int nbl_serv_init_chip_factory(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct device *dev = NBL_COMMON_TO_DEV(common);
	int ret = 0;

	ret = disp_ops->init_chip_module(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (ret) {
		dev_err(dev, "init_chip_module failed\n");
		goto module_init_fail;
	}

	return 0;

module_init_fail:
	return ret;
}

static int nbl_serv_destroy_chip_factory(void *p)
{
	return 0;
}

static int nbl_serv_init_chip(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;
	struct nbl_common_info *common;
	struct device *dev;
	int ret = 0;

	common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	dev = NBL_COMMON_TO_DEV(common);

	ret = disp_ops->init_chip_module(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (ret) {
		dev_err(dev, "init_chip_module failed\n");
		goto module_init_fail;
	}

	ret = disp_ops->queue_init(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (ret) {
		dev_err(dev, "queue_init failed\n");
		goto queue_init_fail;
	}

	ret = disp_ops->vsi_init(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (ret) {
		dev_err(dev, "vsi_init failed\n");
		goto vsi_init_fail;
	}

	return 0;

vsi_init_fail:
queue_init_fail:
module_init_fail:
	return ret;
}

static int nbl_serv_destroy_chip(void *p)
{
	return 0;
}

static int nbl_serv_configure_msix_map(void *priv, u16 num_net_msix, u16 num_others_msix,
				       bool net_msix_mask_en)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;
	int ret = 0;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	ret = disp_ops->configure_msix_map(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), num_net_msix,
					   num_others_msix, net_msix_mask_en);
	if (ret)
		return -EIO;

	return 0;
}

static int nbl_serv_destroy_msix_map(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;
	int ret = 0;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	ret = disp_ops->destroy_msix_map(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (ret)
		return -EIO;

	return 0;
}

static int nbl_serv_enable_mailbox_irq(void *priv, u16 vector_id, bool enable_msix)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;
	int ret = 0;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	ret = disp_ops->enable_mailbox_irq(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					   vector_id, enable_msix);
	if (ret)
		return -EIO;

	return 0;
}

static int nbl_serv_enable_abnormal_irq(void *priv, u16 vector_id, bool enable_msix)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;
	int ret = 0;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	ret = disp_ops->enable_abnormal_irq(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					    vector_id, enable_msix);
	if (ret)
		return -EIO;

	return 0;
}

static irqreturn_t nbl_serv_clean_rings(int __always_unused irq, void *data)
{
	struct nbl_serv_vector *vector = (struct nbl_serv_vector *)data;

	napi_schedule_irqoff(vector->napi);

	return IRQ_HANDLED;
}

static int nbl_serv_request_net_irq(void *priv, struct nbl_msix_info_param *msix_info)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct device *dev = NBL_COMMON_TO_DEV(common);
	struct nbl_serv_ring *tx_ring, *rx_ring;
	struct nbl_serv_vector *vector;
	u32 irq_num;
	int i, ret = 0;

	for (i = 0; i < ring_mgt->xdp_ring_offset; i++) {
		tx_ring = &ring_mgt->tx_rings[i];
		rx_ring = &ring_mgt->rx_rings[i];
		vector = &ring_mgt->vectors[i];
		vector->tx_ring = tx_ring;
		vector->rx_ring = rx_ring;

		irq_num = msix_info->msix_entries[i].vector;
		snprintf(vector->name, sizeof(vector->name) - 1, "%s%03d-%s-%02u", "NBL",
			 NBL_COMMON_TO_VSI_ID(common), "TxRx", i);
		ret = devm_request_irq(dev, irq_num, nbl_serv_clean_rings, 0,
				       vector->name, vector);
		if (ret) {
			nbl_err(common, NBL_DEBUG_INTR, "TxRx Queue %u requests MSIX irq failed "
				"with error %d", i, ret);
			goto request_irq_err;
		}
		if (!cpumask_empty(&vector->cpumask))
			irq_set_affinity_hint(irq_num, &vector->cpumask);
	}

	net_resource_mgt->num_net_msix = msix_info->msix_num;

	return 0;

request_irq_err:
	while (--i + 1) {
		vector = &ring_mgt->vectors[i];

		irq_num = msix_info->msix_entries[i].vector;
		irq_set_affinity_hint(irq_num, NULL);
		devm_free_irq(dev, irq_num, vector);
	}
	return ret;
}

static void nbl_serv_free_net_irq(void *priv, struct nbl_msix_info_param *msix_info)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct device *dev = NBL_COMMON_TO_DEV(common);
	struct nbl_serv_vector *vector;
	u32 irq_num;
	int i;

	for (i = 0; i < ring_mgt->xdp_ring_offset; i++) {
		vector = &ring_mgt->vectors[i];

		irq_num = msix_info->msix_entries[i].vector;
		irq_set_affinity_hint(irq_num, NULL);
		devm_free_irq(dev, irq_num, vector);
	}
}

static u16 nbl_serv_get_global_vector(void *priv, u16 local_vector_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_global_vector(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					   NBL_COMMON_TO_VSI_ID(common), local_vector_id);
}

static u16 nbl_serv_get_msix_entry_id(void *priv, u16 local_vector_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_msix_entry_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					   NBL_COMMON_TO_VSI_ID(common), local_vector_id);
}

static u16 nbl_serv_get_vsi_id(void *priv, u16 func_id, u16 type)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_vsi_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), func_id, type);
}

static void nbl_serv_get_eth_id(void *priv, u16 vsi_id, u8 *eth_mode, u8 *eth_id, u8 *logic_eth_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_eth_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id,
				    eth_mode, eth_id, logic_eth_id);
}

void nbl_serv_get_rep_drop_stats(struct nbl_service_mgt *serv_mgt, u16 rep_vsi_id,
				 struct nbl_rep_stats *rep_stats)
{
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_net_resource_mgt *net_resource_mgt;
	struct nbl_serv_rep_drop *rep_drop;
	u16 rep_data_index;
	unsigned int start;

	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	rep_data_index = disp_ops->get_rep_index(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), rep_vsi_id);
	if (rep_data_index >= net_resource_mgt->num_vfs)
		return;

	rep_drop = &net_resource_mgt->rep_drop[rep_data_index];
	do {
		start = u64_stats_fetch_begin(&rep_drop->rep_drop_syncp);
		rep_stats->dropped = rep_drop->tx_dropped;
	} while (u64_stats_fetch_retry(&rep_drop->rep_drop_syncp, start));
}

static void nbl_serv_rep_get_stats64(struct net_device *netdev, struct rtnl_link_stats64 *stats)
{
	struct nbl_netdev_priv *rep_priv = netdev_priv(netdev);
	struct nbl_rep_stats rep_stats = { 0 };
	struct nbl_adapter *adapter;
	struct nbl_service_mgt *serv_mgt;
	struct nbl_dispatch_ops *disp_ops;

	adapter = NBL_NETDEV_TO_ADAPTER(netdev);
	if (!adapter) {
		netdev_err(netdev, "rep get stats, adapter is null\n");
		return;
	}
	serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	if (!stats) {
		netdev_err(netdev, "rep get stats, stats is null\n");
		return;
	}

	disp_ops->get_rep_stats(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				rep_priv->rep->rep_vsi_id, &rep_stats, true);
	stats->tx_packets += rep_stats.packets;
	stats->tx_bytes += rep_stats.bytes;

	disp_ops->get_rep_stats(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				rep_priv->rep->rep_vsi_id, &rep_stats, false);
	stats->rx_packets += rep_stats.packets;
	stats->rx_bytes += rep_stats.bytes;

	nbl_serv_get_rep_drop_stats(serv_mgt, rep_priv->rep->rep_vsi_id, &rep_stats);
	stats->tx_dropped += rep_stats.dropped;
	stats->rx_dropped = 0;
	stats->multicast = 0;
	stats->rx_errors = 0;
	stats->tx_errors = 0;
	stats->rx_length_errors = 0;
	stats->rx_crc_errors = 0;
	stats->rx_frame_errors = 0;
}

static void nbl_serv_rep_set_rx_mode(struct net_device *dev)
{
}

static int nbl_serv_rep_set_mac(struct net_device *dev, void *p)
{
	struct sockaddr *addr = p;

	if (!is_valid_ether_addr(addr->sa_data)) {
		netdev_err(dev, "Temp to change a invalid mac address %pM\n", addr->sa_data);
		return -EADDRNOTAVAIL;
	}

	if (ether_addr_equal(dev->dev_addr, addr->sa_data))
		return 0;

	return -EOPNOTSUPP;
}

static int nbl_serv_rep_rx_add_vid(struct net_device *dev, __be16 proto, u16 vid)
{
	if (vid == NBL_DEFAULT_VLAN_ID)
		return 0;

	return -EAGAIN;
}

static int nbl_serv_rep_rx_kill_vid(struct net_device *dev, __be16 proto, u16 vid)
{
	if (vid == NBL_DEFAULT_VLAN_ID)
		return 0;

	return -EAGAIN;
}

static LIST_HEAD(nbl_serv_rep_block_cb_list);

static int nbl_serv_rep_setup_tc(struct net_device *dev, enum tc_setup_type type, void *type_data)
{
	struct nbl_netdev_priv *priv = netdev_priv(dev);

	switch (type) {
	case TC_SETUP_BLOCK: {
		return flow_block_cb_setup_simple((struct flow_block_offload *)type_data,
						  &nbl_serv_rep_block_cb_list,
						  nbl_serv_setup_tc_block_cb,
						  priv, priv, true);
	}
	default:
		return -EOPNOTSUPP;
	}
}

static int nbl_serv_rep_get_phys_port_name(struct net_device *dev, char *name, size_t len)
{
	struct nbl_netdev_priv *priv = netdev_priv(dev);
	struct nbl_adapter *adapter = NBL_NETDEV_TO_ADAPTER(dev);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_common_info *common = NBL_ADAPTER_TO_COMMON(adapter);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	u16 vf_base_vsi_id;
	u16 vf_id;
	u8 pf_id;

	pf_id = common->eth_id;
	if ((NBL_COMMON_TO_ETH_MODE(common) == NBL_TWO_ETHERNET_PORT) && common->eth_id == 2)
		pf_id = 1;

	vf_base_vsi_id = disp_ops->get_vf_base_vsi_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						      NBL_COMMON_TO_MGT_PF(common));
	vf_id =  priv->rep->rep_vsi_id - vf_base_vsi_id;
	if (snprintf(name, len, "pf%uvf%u", pf_id, vf_id) >= len)
		return -EINVAL;
	return 0;
}

static int nbl_serv_rep_get_port_parent_id(struct net_device *dev, struct netdev_phys_item_id *ppid)
{
	struct nbl_netdev_priv *priv = netdev_priv(dev);
	struct nbl_adapter *adapter = NBL_NETDEV_PRIV_TO_ADAPTER(priv);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	u8 mac[ETH_ALEN];

	disp_ops->get_base_mac_addr(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), mac);

	ppid->id_len = ETH_ALEN;
	memcpy(&ppid->id, mac, ppid->id_len);

	return 0;
}

static struct nbl_indr_dev_priv *nbl_find_indr_dev_priv(void *priv, struct net_device *netdev,
							int binder_type)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt;
	struct nbl_indr_dev_priv *indr_priv;

	if (!netdev)
		return NULL;

	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	list_for_each_entry(indr_priv, &net_resource_mgt->indr_dev_priv_list, list)
		if (indr_priv->indr_dev == netdev && indr_priv->binder_type == binder_type)
			return indr_priv;

	return NULL;
}

static void nbl_serv_indr_dev_block_unbind(void *priv)
{
	struct nbl_indr_dev_priv *indr_priv = priv;
	struct nbl_adapter *adapter = NBL_NETDEV_PRIV_TO_ADAPTER(indr_priv->dev_priv);
	struct device *dev = NBL_ADAPTER_TO_DEV(adapter);

	list_del(&indr_priv->list);
	devm_kfree(dev, indr_priv);
}

static LIST_HEAD(nbl_serv_indr_block_cb_list);

static int nbl_serv_indr_dev_setup_block(struct net_device *netdev, struct Qdisc *sch,
					 struct nbl_netdev_priv *dev_priv,
					 struct flow_block_offload *flow_bo,
					 flow_setup_cb_t *setup_cb, void *data,
					 void (*cleanup)(struct flow_block_cb *block_cb))
{
	struct nbl_adapter *adapter = NBL_NETDEV_PRIV_TO_ADAPTER(dev_priv);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct device *dev = NBL_ADAPTER_TO_DEV(adapter);
	struct nbl_indr_dev_priv *indr_priv = NULL;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NULL;
	struct flow_block_cb *block_cb = NULL;

	if (flow_bo->binder_type != FLOW_BLOCK_BINDER_TYPE_CLSACT_INGRESS &&
	    (flow_bo->binder_type == FLOW_BLOCK_BINDER_TYPE_CLSACT_EGRESS &&
	    !netif_is_ovs_master(netdev)))
		return -EOPNOTSUPP;

	flow_bo->unlocked_driver_cb = true;
	flow_bo->driver_block_list = &nbl_serv_indr_block_cb_list;
	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);

	switch (flow_bo->command) {
	case FLOW_BLOCK_BIND:
		indr_priv = nbl_find_indr_dev_priv(serv_mgt, netdev, flow_bo->binder_type);
		if (indr_priv)
			return -EEXIST;

		indr_priv = devm_kzalloc(dev, sizeof(struct nbl_indr_dev_priv), GFP_KERNEL);
		if (!indr_priv)
			return -ENOMEM;

		indr_priv->indr_dev = netdev;
		indr_priv->dev_priv = dev_priv;
		indr_priv->binder_type = flow_bo->binder_type;
		list_add_tail(&indr_priv->list, &net_resource_mgt->indr_dev_priv_list);

		block_cb = flow_indr_block_cb_alloc(setup_cb, indr_priv, indr_priv,
						    nbl_serv_indr_dev_block_unbind, flow_bo,
						    netdev, sch, data, dev_priv, cleanup);
		if (IS_ERR(block_cb)) {
			netdev_err(netdev, "indr block cb alloc fail\n");
			list_del(&indr_priv->list);
			devm_kfree(dev, indr_priv);
			return PTR_ERR(block_cb);
		}
		flow_block_cb_add(block_cb, flow_bo);
		list_add_tail(&block_cb->driver_list, &nbl_serv_indr_block_cb_list);
		break;
	case FLOW_BLOCK_UNBIND:
		indr_priv = nbl_find_indr_dev_priv(serv_mgt, netdev, flow_bo->binder_type);
		if (!indr_priv)
			return -ENOENT;

		block_cb = flow_block_cb_lookup(flow_bo->block, setup_cb, indr_priv);
		if (!block_cb)
			return -ENOENT;
		flow_indr_block_cb_remove(block_cb, flow_bo);
		list_del(&block_cb->driver_list);
		break;
	default:
		return -EOPNOTSUPP;
	}
	return 0;
}

static int nbl_serv_indr_dev_setup_tc(struct net_device *dev, struct Qdisc *sch, void *cb_priv,
				      enum tc_setup_type type, void *type_data, void *data,
				      void (*cleanup)(struct flow_block_cb *block_cb))
{
	struct nbl_netdev_priv *priv = cb_priv;

	switch (type) {
	case TC_SETUP_BLOCK:
		return nbl_serv_indr_dev_setup_block(dev, sch, priv, type_data,
						     nbl_serv_indr_setup_tc_block_cb,
						     data, cleanup);
	default:
		return -EOPNOTSUPP;
	}
}

static void nbl_serv_get_rep_feature(void *priv, struct nbl_register_net_result *register_result)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->get_rep_feature(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), register_result);
}

static void nbl_serv_get_rep_queue_num(void *priv, u8 *base_queue_id, u8 *rep_queue_num)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);

	*base_queue_id = (u8)ring_mgt->vsi_info[NBL_VSI_CTRL].ring_offset;
	*rep_queue_num = (u8)ring_mgt->vsi_info[NBL_VSI_CTRL].ring_num;
}

static void nbl_serv_get_rep_queue_info(void *priv, u16 *queue_num, u16 *queue_size)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->get_rep_queue_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				     queue_num, queue_size);
}

static void nbl_serv_get_user_queue_info(void *priv, u16 *queue_num, u16 *queue_size, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->get_user_queue_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				      queue_num, queue_size, vsi_id);
}

static int nbl_serv_rep_enqueue(struct sk_buff *skb,
				struct nbl_serv_rep_queue_mgt *rep_queue_mgt)
{
	if (rep_queue_mgt->size == 0)
		return -EINVAL;

	return ptr_ring_produce(&rep_queue_mgt->ring, skb);
}

static struct sk_buff *nbl_serv_rep_dequeue(struct nbl_serv_rep_queue_mgt *rep_queue_mgt)
{
	struct sk_buff *skb;

	if (rep_queue_mgt->size == 0)
		return NULL;

	if (__ptr_ring_empty(&rep_queue_mgt->ring))
		skb = NULL;
	else
		skb = __ptr_ring_consume(&rep_queue_mgt->ring);

	if (unlikely(!skb)) {
		/* smp_mb for dequeue */
		smp_mb__after_atomic();
		if (!__ptr_ring_empty(&rep_queue_mgt->ring))
			skb = __ptr_ring_consume(&rep_queue_mgt->ring);
	}

	return skb;
}

static inline bool nbl_serv_rep_queue_mgt_start(struct nbl_serv_rep_queue_mgt *rep_queue_mgt)
{
	return spin_trylock(&rep_queue_mgt->seq_lock);
}

static inline void nbl_serv_rep_queue_mgt_end(struct nbl_serv_rep_queue_mgt *rep_queue_mgt)
{
	spin_unlock(&rep_queue_mgt->seq_lock);
}

static void nbl_serv_rep_update_drop_stats(void *priv, struct sk_buff *skb)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_net_resource_mgt *net_resource_mgt;
	u16 rep_vsi_id;
	u16 rep_data_index;

	rep_vsi_id = *(u16 *)&skb->cb[NBL_SKB_FILL_VSI_ID_OFF];
	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	rep_data_index = disp_ops->get_rep_index(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), rep_vsi_id);
	dev_kfree_skb_any(skb);
	if (rep_data_index >= net_resource_mgt->num_vfs)
		return;

	u64_stats_update_begin(&net_resource_mgt->rep_drop[rep_data_index].rep_drop_syncp);
	net_resource_mgt->rep_drop[rep_data_index].tx_dropped++;
	u64_stats_update_end(&net_resource_mgt->rep_drop[rep_data_index].rep_drop_syncp);
}

static void nbl_serv_rep_queue_mgt_run(struct nbl_serv_rep_queue_mgt *rep_queue_mgt,
				       struct net_device *netdev)
{
	struct nbl_adapter *adapter = NBL_NETDEV_TO_ADAPTER(netdev);
	struct nbl_service_mgt *serv_mgt = NBL_ADAPTER_TO_SERV_MGT(adapter);
	struct nbl_resource_pt_ops *pt_ops = NBL_ADAPTER_TO_RES_PT_OPS(adapter);
	struct sk_buff *skb;
	netdev_tx_t ret = NETDEV_TX_OK;
	int i = 0;

	skb = nbl_serv_rep_dequeue(rep_queue_mgt);
	if (!skb)
		return;
	for (; skb; skb = nbl_serv_rep_dequeue(rep_queue_mgt)) {
		ret = pt_ops->rep_xmit(skb, rep_queue_mgt->netdev);
		if (ret == NETDEV_TX_BUSY) {
			if (net_ratelimit())
				netdev_dbg(netdev, "dequeue skb tx busy!\n");
			/* never hang in sirq too long, so if a tx_busy is returned, drop it */
			nbl_serv_rep_update_drop_stats(serv_mgt, skb);
		}
		if (i++ >= NBL_DEFAULT_REP_TX_MAX_NUM)
			return;
	}
}

static netdev_tx_t nbl_serv_rep_start_xmit(struct sk_buff *skb, struct net_device *netdev)
{
	struct nbl_netdev_priv *rep_priv = netdev_priv(netdev);
	struct nbl_adapter *adapter;
	struct nbl_service_mgt *serv_mgt;
	struct nbl_serv_rep_queue_mgt *rep_queue_mgt;
	int ret;
	u8 rep_queue_idx;
	u8 i;
	bool has_locked_flag = false;

	adapter = NBL_NETDEV_TO_ADAPTER(netdev);
	serv_mgt = NBL_NETDEV_TO_SERV_MGT(netdev);

	rep_queue_idx = (rep_priv->rep->rep_vsi_id - 1) % rep_priv->rep->rep_queue_num;
	rep_queue_mgt = &serv_mgt->rep_queue_mgt[rep_queue_idx];
	skb->queue_mapping = rep_queue_idx + rep_priv->rep->base_queue_id;
	*(u16 *)(&skb->cb[NBL_SKB_FILL_VSI_ID_OFF]) = rep_priv->rep->rep_vsi_id;
	skb->cb[NBL_SKB_FILL_EXT_HDR_OFF] = NBL_REP_FILL_EXT_HDR;
	ret = nbl_serv_rep_enqueue(skb, rep_queue_mgt);

	if (unlikely(ret)) {
		if (net_ratelimit())
			netdev_info(netdev, "rep enqueue fail, size:%d, rep_vsi_id:%d!!\n",
				    rep_queue_mgt->size, rep_priv->rep->rep_vsi_id);
	}
	for (i = 0; i < NBL_DEFAULT_REP_TX_RETRY_NUM; i++) {
		if (nbl_serv_rep_queue_mgt_start(rep_queue_mgt)) {
			has_locked_flag = true;
			nbl_serv_rep_queue_mgt_run(rep_queue_mgt, netdev);
			nbl_serv_rep_queue_mgt_end(rep_queue_mgt);
		}
	}

	if (has_locked_flag) {
		if (ret)
			ret = NET_XMIT_CN;
		else
			ret = NET_XMIT_SUCCESS;
	}

	if (likely(ret)) {
		/* enqueue failed but get lock succ, need a retry */
		if (ret == NET_XMIT_CN) {
			return NETDEV_TX_BUSY;
		} else if (ret == NET_XMIT_SUCCESS) {
			/* enqueue succ and get lock succ, rep_xmit regard as a ok */
			return NETDEV_TX_OK;
		}
		/* enqueue and get lock failed, free skb and no need a retry */
		nbl_serv_rep_update_drop_stats(serv_mgt, skb);
		return NETDEV_TX_OK;
	}

	/* enqueue succ but get lock failed, rep_xmit regard as a ok */
	return NETDEV_TX_OK;
}

static int nbl_serv_alloc_rep_queue_mgt(void *priv, struct net_device *netdev)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct device *dev;
	int i, ret;
	u8 base_queue_id;
	u8 rep_queue_num;

	if (!serv_mgt)
		return -EINVAL;

	dev = NBL_SERV_MGT_TO_DEV(serv_mgt);

	dev_info(dev, "nbl serv alloc rep queue mgt start\n");
	nbl_serv_get_rep_queue_num(serv_mgt, &base_queue_id, &rep_queue_num);
	serv_mgt->rep_queue_mgt = devm_kcalloc(dev, rep_queue_num,
					       sizeof(struct nbl_serv_rep_queue_mgt), GFP_KERNEL);
	if (!serv_mgt->rep_queue_mgt)
		return -ENOMEM;
	for (i = 0; i < rep_queue_num; i++) {
		ret = ptr_ring_init(&serv_mgt->rep_queue_mgt[i].ring,
				    NBL_REP_QUEUE_MGT_DESC_NUM, GFP_KERNEL);
		if (ret) {
			dev_err(dev, "ptr ring init failed\n");
			goto free_ptr_ring;
		}

		spin_lock_init(&serv_mgt->rep_queue_mgt[i].seq_lock);
		serv_mgt->rep_queue_mgt[i].size = NBL_REP_QUEUE_MGT_DESC_NUM;
		serv_mgt->rep_queue_mgt[i].netdev = netdev;
		dev_info(dev, "rep_queue_mgt init success\n");
	}
	dev_info(dev, "nbl serv alloc rep queue mgt end\n");

	return 0;

free_ptr_ring:
	for (; i >= 0; i--)
		ptr_ring_cleanup(&serv_mgt->rep_queue_mgt[i].ring, 0);

	devm_kfree(dev, serv_mgt->rep_queue_mgt);
	serv_mgt->rep_queue_mgt = NULL;
	return -ENOMEM;
}

static int nbl_serv_free_rep_queue_mgt(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct device *dev;
	int i;
	u8 base_queue_id;
	u8 rep_queue_num;

	if (!serv_mgt)
		return -EINVAL;

	dev = NBL_SERV_MGT_TO_DEV(serv_mgt);
	if (!serv_mgt->rep_queue_mgt)
		return -EINVAL;

	nbl_serv_get_rep_queue_num(serv_mgt, &base_queue_id, &rep_queue_num);
	for (i = 0; i < rep_queue_num; i++)
		ptr_ring_cleanup(&serv_mgt->rep_queue_mgt[i].ring, 0);

	dev_info(dev, "ptr ring cleanup\n");
	devm_kfree(dev, serv_mgt->rep_queue_mgt);
	serv_mgt->rep_queue_mgt = NULL;

	return 0;
}

static void nbl_serv_set_eswitch_mode(void *priv, u16 eswitch_mode)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common =  NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_serv_net_resource_mgt *net_resource_mgt;

	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);

	if (eswitch_mode == NBL_ESWITCH_OFFLOADS) {
		disp_ops->set_dport_fc_th_vld(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					      common->eth_id, false);
		if (net_resource_mgt->lag_info && net_resource_mgt->lag_info->lag_num > 1)
			disp_ops->set_shaping_dport_vld(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
							common->eth_id, false);
	} else {
		disp_ops->set_dport_fc_th_vld(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					      common->eth_id, true);
		if (net_resource_mgt->lag_info && net_resource_mgt->lag_info->lag_num > 1)
			disp_ops->set_shaping_dport_vld(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
							common->eth_id, true);
	}
	disp_ops->set_eswitch_mode(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), eswitch_mode);
}

static u16 nbl_serv_get_eswitch_mode(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_eswitch_mode(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_alloc_rep_data(void *priv, int num_vfs, u16 vf_base_vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt;
	struct nbl_dispatch_ops *disp_ops;
	struct device *dev;

	if (!serv_mgt)
		return -EINVAL;

	dev = NBL_SERV_MGT_TO_DEV(serv_mgt);
	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	net_resource_mgt->rep_drop = devm_kcalloc(dev, num_vfs,
						  sizeof(struct nbl_serv_rep_drop),
						  GFP_KERNEL);
	if (!net_resource_mgt->rep_drop)
		return -ENOMEM;

	net_resource_mgt->num_vfs = num_vfs;
	return disp_ops->alloc_rep_data(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), num_vfs,
					vf_base_vsi_id);
}

static void nbl_serv_free_rep_data(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt;
	struct nbl_dispatch_ops *disp_ops;
	struct device *dev;

	if (!serv_mgt)
		return;

	dev = NBL_SERV_MGT_TO_DEV(serv_mgt);
	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	if (net_resource_mgt->rep_drop)
		devm_kfree(dev, net_resource_mgt->rep_drop);
	disp_ops->free_rep_data(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static void nbl_serv_set_rep_netdev_info(void *priv, void *rep_data)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->set_rep_netdev_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), rep_data);
}

static void nbl_serv_unset_rep_netdev_info(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->unset_rep_netdev_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_disable_phy_flow(void *priv, u8 eth_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->disable_phy_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), eth_id);
}

static int nbl_serv_enable_phy_flow(void *priv, u8 eth_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->enable_phy_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), eth_id);
}

static void nbl_serv_init_acl(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->init_acl(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static void nbl_serv_uninit_acl(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->uninit_acl(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_set_upcall_rule(void *priv, u8 eth_id, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->add_nd_upcall_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id, 0);

	return disp_ops->set_upcall_rule(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), eth_id, vsi_id);
}

static int nbl_serv_unset_upcall_rule(void *priv, u8 eth_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->del_nd_upcall_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));

	return disp_ops->unset_upcall_rule(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), eth_id);
}

static int nbl_serv_switchdev_init_cmdq(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	return disp_ops->switchdev_init_cmdq(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_switchdev_deinit_cmdq(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	return disp_ops->switchdev_deinit_cmdq(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_set_tc_flow_info(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->set_tc_flow_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_unset_tc_flow_info(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	return disp_ops->unset_tc_flow_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_get_tc_flow_info(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_tc_flow_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_register_indr_dev_tc_offload(void *priv, struct net_device *netdev)
{
	struct nbl_netdev_priv *dev_priv = netdev_priv(netdev);

	return flow_indr_dev_register(nbl_serv_indr_dev_setup_tc, dev_priv);
}

static void nbl_serv_unregister_indr_dev_tc_offload(void *priv, struct net_device *netdev)
{
	struct nbl_netdev_priv *dev_priv = netdev_priv(netdev);

	flow_indr_dev_unregister(nbl_serv_indr_dev_setup_tc, dev_priv,
				 nbl_serv_indr_dev_block_unbind);
}

static void nbl_serv_set_lag_info(void *priv, struct net_device *bond_netdev, u8 lag_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct device *dev = NBL_COMMON_TO_DEV(common);

	net_resource_mgt->lag_info = devm_kzalloc(dev, sizeof(struct nbl_serv_lag_info),
						  GFP_KERNEL);
	if (!net_resource_mgt->lag_info)
		return;
	net_resource_mgt->lag_info->bond_netdev = bond_netdev;
	net_resource_mgt->lag_info->lag_id = lag_id;

	dev_info(dev, "set lag info, bond_netdev:%p, lag_id:%d\n", bond_netdev, lag_id);
}

static void nbl_serv_unset_lag_info(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct device *dev = NBL_COMMON_TO_DEV(common);

	if (net_resource_mgt->lag_info) {
		devm_kfree(dev, net_resource_mgt->lag_info);
		net_resource_mgt->lag_info = NULL;
	}
}

static void
nbl_serv_set_netdev_ops(void *priv, const struct net_device_ops *net_device_ops, bool is_pf)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct device *dev = NBL_SERV_MGT_TO_DEV(serv_mgt);

	dev_info(dev, "set netdev ops:%p is_pf:%d\n", net_device_ops, is_pf);
	if (is_pf)
		net_resource_mgt->netdev_ops.pf_netdev_ops = (void *)net_device_ops;
	else
		net_resource_mgt->netdev_ops.rep_netdev_ops = (void *)net_device_ops;
}

static int nbl_serv_enable_lag_protocol(void *priv, u16 eth_id, bool lag_en)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct net_device *dev = net_resource_mgt->netdev;
	struct nbl_netdev_priv *net_priv = netdev_priv(dev);
	int ret = 0;

	ret = disp_ops->enable_lag_protocol(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), eth_id, lag_en);
	if (lag_en)
		ret = disp_ops->add_lag_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					     net_priv->other_vsi);
	else
		disp_ops->del_lag_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), net_priv->other_vsi);

	return ret;
}

static int nbl_serv_cfg_lag_hash_algorithm(void *priv, u16 eth_id, u16 lag_id,
					   enum netdev_lag_hash hash_type)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->cfg_lag_hash_algorithm(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						eth_id, lag_id, hash_type);
}

static int nbl_serv_cfg_lag_member_fwd(void *priv, u16 eth_id, u16 lag_id, u8 fwd)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	if (net_resource_mgt->lag_info)
		net_resource_mgt->lag_info->lag_id = lag_id;

	disp_ops->cfg_lag_mcc(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), eth_id, lag_id, fwd);

	return disp_ops->cfg_lag_member_fwd(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					    eth_id, lag_id, fwd);
}

static int nbl_serv_cfg_lag_member_list(void *priv, struct nbl_lag_member_list_param *param)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common =  NBL_SERV_MGT_TO_COMMON(serv_mgt);
	int ret = 0;
	u16 cur_eswitch_mode = NBL_ESWITCH_NONE;
	bool shaping_vld = true;

	ret = disp_ops->cfg_lag_member_list(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), param);
	if (ret)
		return ret;

	ret = disp_ops->cfg_duppkt_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), param);
	if (ret)
		return ret;

	if (net_resource_mgt->lag_info)
		net_resource_mgt->lag_info->lag_num = param->lag_num;

	cur_eswitch_mode = disp_ops->get_eswitch_mode(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (cur_eswitch_mode == NBL_ESWITCH_OFFLOADS) {
		shaping_vld = param->lag_num > 1 ? false : true;
		disp_ops->set_shaping_dport_vld(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						common->eth_id, shaping_vld);
	}

	ret = disp_ops->cfg_eth_bond_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), param);
	if (ret)
		return ret;

	ret = disp_ops->cfg_duppkt_mcc(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), param);

	return ret;
}

static int nbl_serv_cfg_lag_member_up_attr(void *priv, u16 eth_id, u16 lag_id, bool enable)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->cfg_lag_member_up_attr(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						eth_id, lag_id, enable);
}

static void nbl_serv_net_stats_update_task(struct work_struct *work)
{
	struct nbl_serv_net_resource_mgt *serv_net_resource_mgt =
		container_of(work, struct nbl_serv_net_resource_mgt, net_stats_update);
	struct nbl_service_mgt *serv_mgt;

	serv_mgt = serv_net_resource_mgt->serv_mgt;

	nbl_serv_update_stats(serv_mgt, false);
}

static void nbl_serv_rx_mode_async_task(struct work_struct *work)
{
	struct nbl_serv_net_resource_mgt *serv_net_resource_mgt =
		container_of(work, struct nbl_serv_net_resource_mgt, rx_mode_async);

	if (serv_net_resource_mgt->rxmode_set_required & NBL_FLAG_AQ_MODIFY_MAC_FILTER)
		nbl_modify_submacs(serv_net_resource_mgt);

	if (serv_net_resource_mgt->rxmode_set_required & NBL_FLAG_AQ_CONFIGURE_PROMISC_MODE)
		nbl_modify_promisc_mode(serv_net_resource_mgt);
}

static void nbl_serv_net_task_service_timer(struct timer_list *t)
{
	struct nbl_serv_net_resource_mgt *net_resource_mgt =
					from_timer(net_resource_mgt, t, serv_timer);

	mod_timer(&net_resource_mgt->serv_timer,
		  round_jiffies(net_resource_mgt->serv_timer_period + jiffies));
	nbl_common_queue_work(&net_resource_mgt->net_stats_update, false, false);
}

static void nbl_serv_setup_flow_mgt(struct nbl_serv_flow_mgt *flow_mgt)
{
	INIT_LIST_HEAD(&flow_mgt->vlan_list);
	INIT_LIST_HEAD(&flow_mgt->submac_list);
}

static void nbl_serv_register_restore_netdev_queue(struct nbl_service_mgt *serv_mgt)
{
	struct nbl_channel_ops *chan_ops = NBL_SERV_MGT_TO_CHAN_OPS(serv_mgt);

	if (!chan_ops->check_queue_exist(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt),
					 NBL_CHAN_TYPE_MAILBOX))
		return;

	chan_ops->register_msg(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt),
			       NBL_CHAN_MSG_STOP_ABNORMAL_SW_QUEUE,
			       nbl_serv_chan_stop_abnormal_sw_queue_resp, serv_mgt);

	chan_ops->register_msg(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt),
			       NBL_CHAN_MSG_RESTORE_NETDEV_QUEUE,
			       nbl_serv_chan_restore_netdev_queue_resp, serv_mgt);

	chan_ops->register_msg(NBL_SERV_MGT_TO_CHAN_PRIV(serv_mgt),
			       NBL_CHAN_MSG_RESTART_NETDEV_QUEUE,
			       nbl_serv_chan_restart_netdev_queue_resp, serv_mgt);
}

static void nbl_serv_remove_net_resource_mgt(void *priv)
{
	struct device *dev;
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);

	net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	dev = NBL_COMMON_TO_DEV(common);

	if (net_resource_mgt) {
		del_timer_sync(&net_resource_mgt->serv_timer);
		nbl_common_release_task(&net_resource_mgt->rx_mode_async);
		nbl_common_release_task(&net_resource_mgt->net_stats_update);
		nbl_common_release_task(&net_resource_mgt->tx_timeout);
		if (common->is_vf) {
			nbl_common_release_task(&net_resource_mgt->update_link_state);
			nbl_common_release_task(&net_resource_mgt->update_vlan);
		}
		nbl_free_filter(net_resource_mgt);
		devm_kfree(dev, net_resource_mgt);
		NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt) = NULL;
	}
}

static int nbl_serv_phy_init(struct nbl_serv_net_resource_mgt *net_resource_mgt)
{
	struct nbl_service_mgt *serv_mgt = net_resource_mgt->serv_mgt;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	u8 eth_id = NBL_COMMON_TO_ETH_ID(common);
	struct nbl_dispatch_ops *disp_ops;
	int ret = 0;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->get_phy_caps(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
			       eth_id, &net_resource_mgt->phy_caps);

	return ret;
}

static int nbl_serv_setup_net_resource_mgt(void *priv, struct net_device *netdev,
					   u16 vlan_proto, u16 vlan_tci, u32 rate)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct device *dev = NBL_COMMON_TO_DEV(common);
	struct nbl_serv_net_resource_mgt *net_resource_mgt;

	net_resource_mgt = devm_kzalloc(dev, sizeof(struct nbl_serv_net_resource_mgt), GFP_KERNEL);
	if (!net_resource_mgt)
		return -ENOMEM;

	net_resource_mgt->netdev = netdev;
	net_resource_mgt->serv_mgt = serv_mgt;
	net_resource_mgt->vlan_proto = vlan_proto;
	net_resource_mgt->vlan_tci = vlan_tci;
	net_resource_mgt->max_tx_rate = rate;
	NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt) = net_resource_mgt;

	nbl_serv_phy_init(net_resource_mgt);
	nbl_serv_register_restore_netdev_queue(serv_mgt);
	if (common->is_vf) {
		nbl_serv_register_link_forced_notify(serv_mgt);
		nbl_serv_register_vlan_notify(serv_mgt);
	}
	timer_setup(&net_resource_mgt->serv_timer, nbl_serv_net_task_service_timer, 0);

	net_resource_mgt->serv_timer_period = HZ;
	nbl_common_alloc_task(&net_resource_mgt->rx_mode_async, nbl_serv_rx_mode_async_task);
	nbl_common_alloc_task(&net_resource_mgt->net_stats_update, nbl_serv_net_stats_update_task);
	nbl_common_alloc_task(&net_resource_mgt->tx_timeout, nbl_serv_handle_tx_timeout);
	if (common->is_vf) {
		nbl_common_alloc_task(&net_resource_mgt->update_link_state,
				      nbl_serv_update_link_state);
		nbl_common_alloc_task(&net_resource_mgt->update_vlan,
				      nbl_serv_update_vlan);
	}

	INIT_LIST_HEAD(&net_resource_mgt->mac_filter_list);
	INIT_LIST_HEAD(&net_resource_mgt->indr_dev_priv_list);
	spin_lock_init(&net_resource_mgt->mac_vlan_list_lock);
	spin_lock_init(&net_resource_mgt->current_netdev_promisc_flags_lock);
	net_resource_mgt->get_stats_jiffies = jiffies;

	mod_timer(&net_resource_mgt->serv_timer,
		  round_jiffies(jiffies + net_resource_mgt->serv_timer_period));

	return 0;
}

static int nbl_serv_enable_adminq_irq(void *priv, u16 vector_id, bool enable_msix)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;
	int ret = 0;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	ret = disp_ops->enable_adminq_irq(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					    vector_id, enable_msix);
	if (ret)
		return -EIO;

	return 0;
}

static u16 nbl_serv_get_rdma_cap_num(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_rdma_cap_num(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static void nbl_serv_setup_rdma_id(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->setup_rdma_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static void nbl_serv_remove_rdma_id(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->remove_rdma_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static void nbl_serv_register_rdma(void *priv, u16 vsi_id, struct nbl_rdma_register_param *param)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->register_rdma(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id, param);
}

static void nbl_serv_unregister_rdma(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->unregister_rdma(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
}

static void nbl_serv_register_rdma_bond(void *priv, struct nbl_lag_member_list_param *list_param,
					struct nbl_rdma_register_param *register_param)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->register_rdma_bond(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				     list_param, register_param);
}

static void nbl_serv_unregister_rdma_bond(void *priv, u16 lag_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->unregister_rdma_bond(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), lag_id);
}

static u8 __iomem *nbl_serv_get_hw_addr(void *priv, size_t *size)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_hw_addr(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), size);
}

static u64 nbl_serv_get_real_hw_addr(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_real_hw_addr(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
}

static u16 nbl_serv_get_function_id(void *priv, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_function_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id);
}

static void nbl_serv_get_real_bdf(void *priv, u16 vsi_id, u8 *bus, u8 *dev, u8 *function)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_real_bdf(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_id,
				      bus, dev, function);
}

static int nbl_serv_get_devlink_info(struct devlink *devlink, struct devlink_info_req *req,
				     struct netlink_ext_ack *extack)
{
	struct nbl_devlink_priv *priv = devlink_priv(devlink);
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv->priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	char firmware_version[NBL_DEVLINK_INFO_FRIMWARE_VERSION_LEN] = {0};
	int ret = 0;

	disp_ops->get_firmware_version(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				       firmware_version, sizeof(firmware_version));
	if (ret)
		return ret;

	ret = devlink_info_version_fixed_put(req, "FW Version:", firmware_version);
	if (ret)
		return ret;

	return ret;
}

/* Why do we need this?
 * Because the original function in kernel cannot handle when we set subvendor and subdevice
 * to be 0xFFFF, so write a correct one.
 */
static bool
nbl_serv_pldmfw_op_pci_match_record(struct pldmfw *context, struct pldmfw_record *record)
{
	struct pci_dev *pdev = to_pci_dev(context->dev);
	struct nbl_serv_pldm_pci_record_id id = {
		.vendor = PCI_ANY_ID,
		.device = PCI_ANY_ID,
		.subsystem_vendor = PCI_ANY_ID,
		.subsystem_device = PCI_ANY_ID,
	};
	struct pldmfw_desc_tlv *desc;
	bool ret;

	list_for_each_entry(desc, &record->descs, entry) {
		u16 value;
		u16 *ptr;

		switch (desc->type) {
		case PLDM_DESC_ID_PCI_VENDOR_ID:
			ptr = &id.vendor;
			break;
		case PLDM_DESC_ID_PCI_DEVICE_ID:
			ptr = &id.device;
			break;
		case PLDM_DESC_ID_PCI_SUBVENDOR_ID:
			ptr = &id.subsystem_vendor;
			break;
		case PLDM_DESC_ID_PCI_SUBDEV_ID:
			ptr = &id.subsystem_device;
			break;
		default:
			/* Skip unrelated TLVs */
			continue;
		}

		value = get_unaligned_le16(desc->data);
		/* A value of zero for one of the descriptors is sometimes
		 * used when the record should ignore this field when matching
		 * device. For example if the record applies to any subsystem
		 * device or vendor.
		 */
		if (value)
			*ptr = (int)value;
		else
			*ptr = PCI_ANY_ID;
	}

	if ((id.vendor == (u16)PCI_ANY_ID || id.vendor == pdev->vendor) &&
	    (id.device == (u16)PCI_ANY_ID || id.device == pdev->device) &&
	    (id.subsystem_vendor == (u16)PCI_ANY_ID ||
	     id.subsystem_vendor == pdev->subsystem_vendor) &&
	    (id.subsystem_device == (u16)PCI_ANY_ID ||
	     id.subsystem_device == pdev->subsystem_device))
		ret = true;
	else
		ret = false;

	return ret;
}

static int nbl_serv_send_package_data(struct pldmfw *context, const u8 *data, u16 length)
{
	struct nbl_serv_update_fw_priv *priv = container_of(context, struct nbl_serv_update_fw_priv,
							    context);
	struct nbl_service_mgt *serv_mgt = priv->serv_mgt;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	int ret = 0;

	nbl_info(common, NBL_DEBUG_DEVLINK, "Send package data");

	ret = disp_ops->flash_lock(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (ret)
		return ret;

	ret = disp_ops->flash_prepare(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (ret)
		disp_ops->flash_unlock(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));

	return 0;
}

static int nbl_serv_send_component_table(struct pldmfw *context, struct pldmfw_component *component,
					 u8 transfer_flags)
{
	struct nbl_serv_update_fw_priv *priv = container_of(context, struct nbl_serv_update_fw_priv,
							    context);
	struct nbl_service_mgt *serv_mgt = priv->serv_mgt;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);

	nbl_info(common, NBL_DEBUG_DEVLINK, "Send component table, id %d", component->identifier);

	return 0;
}

static int nbl_serv_flash_component(struct pldmfw *context, struct pldmfw_component *component)
{
	struct nbl_serv_update_fw_priv *priv = container_of(context, struct nbl_serv_update_fw_priv,
							    context);
	struct nbl_service_mgt *serv_mgt = priv->serv_mgt;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	u32 component_crc, calculated_crc;
	size_t data_len = component->component_size - NBL_DEVLINK_FLASH_COMPONENT_CRC_SIZE;
	int ret = 0;

	nbl_info(common, NBL_DEBUG_DEVLINK, "Flash component table, id %d", component->identifier);

	component_crc = *(u32 *)((u8 *)component->component_data + data_len);
	calculated_crc = crc32_le(~0, component->component_data, data_len) ^ ~0;
	if (component_crc != calculated_crc) {
		nbl_err(common, NBL_DEBUG_DEVLINK, "Flash component crc error");
		disp_ops->flash_unlock(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
		return -EFAULT;
	}

	ret = disp_ops->flash_image(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), component->identifier,
				    component->component_data, data_len);
	if (ret)
		disp_ops->flash_unlock(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));

	return ret;
}

static int nbl_serv_finalize_update(struct pldmfw *context)
{
	struct nbl_serv_update_fw_priv *priv = container_of(context, struct nbl_serv_update_fw_priv,
							    context);
	struct nbl_service_mgt *serv_mgt = priv->serv_mgt;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	int ret = 0;

	nbl_info(common, NBL_DEBUG_DEVLINK, "Flash activate");

	ret = disp_ops->flash_activate(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));

	disp_ops->flash_unlock(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	return ret;
}

static const struct pldmfw_ops nbl_update_fw_ops = {
	.match_record = nbl_serv_pldmfw_op_pci_match_record,
	.send_package_data = nbl_serv_send_package_data,
	.send_component_table = nbl_serv_send_component_table,
	.flash_component = nbl_serv_flash_component,
	.finalize_update = nbl_serv_finalize_update,
};

static int nbl_serv_update_firmware(struct nbl_service_mgt *serv_mgt, const struct firmware *fw,
				    struct netlink_ext_ack *extack)
{
	struct nbl_serv_update_fw_priv priv = {0};
	int ret = 0;

	priv.context.ops = &nbl_update_fw_ops;
	priv.context.dev = NBL_SERV_MGT_TO_DEV(serv_mgt);
	priv.extack = extack;
	priv.serv_mgt = serv_mgt;

	ret = pldmfw_flash_image(&priv.context, fw);

	return ret;
}

static int nbl_serv_update_devlink_flash(struct devlink *devlink,
					 struct devlink_flash_update_params *params,
					 struct netlink_ext_ack *extack)
{
	struct nbl_devlink_priv *priv = devlink_priv(devlink);
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv->priv;
	int ret = 0;

	devlink_flash_update_status_notify(devlink, "Flash start", NULL, 0, 0);

	ret = nbl_serv_update_firmware(serv_mgt, params->fw, extack);
	if (ret)
		devlink_flash_update_status_notify(devlink, "Flash failed", NULL, 0, 0);
	else
		devlink_flash_update_status_notify(devlink,
						   "Flash finished, please reboot to take effect",
						   NULL, 0, 0);
	return ret;
}

static u32 nbl_serv_get_adminq_tx_buf_size(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_adminq_tx_buf_size(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_emp_console_write(void *priv, char *buf, size_t count)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->emp_console_write(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), buf, count);
}

static bool nbl_serv_check_fw_heartbeat(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->check_fw_heartbeat(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static bool nbl_serv_check_fw_reset(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->check_fw_reset(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static void nbl_serv_get_common_irq_num(void *priv, struct nbl_common_irq_num *irq_num)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	irq_num->mbx_irq_num = disp_ops->get_mbx_irq_num(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static void nbl_serv_get_ctrl_irq_num(void *priv, struct nbl_ctrl_irq_num *irq_num)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	irq_num->adminq_irq_num = disp_ops->get_adminq_irq_num(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	irq_num->abnormal_irq_num =
		disp_ops->get_abnormal_irq_num(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_check_offload_status(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;
	bool is_down = false;
	int ret;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	ret = disp_ops->check_offload_status(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), &is_down);

	/* ovs down, need to delete related pmd flow rules */
	if (is_down)
		disp_ops->del_nd_upcall_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));

	return ret;
}

static u32 nbl_serv_get_chip_temperature(void *priv, enum nbl_hwmon_type type, u32 senser_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_chip_temperature(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), type, senser_id);
}

static int nbl_serv_get_module_temperature(void *priv, u8 eth_id, enum nbl_hwmon_type type)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_module_temperature(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), eth_id, type);
}

static int nbl_serv_get_port_attributes(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;
	int ret = 0;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	ret = disp_ops->get_port_attributes(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (ret)
		return -EIO;

	return 0;
}

static int nbl_serv_update_template_config(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	int ret = 0;

	ret = disp_ops->update_ring_num(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (ret)
		return ret;

	ret = disp_ops->update_rdma_cap(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (ret)
		return ret;

	ret = disp_ops->update_rdma_mem_type(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
	if (ret)
		return ret;

	return 0;
}

static int nbl_serv_enable_port(void *priv, bool enable)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;
	int ret = 0;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	ret = disp_ops->enable_port(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), enable);
	if (ret)
		return -EIO;

	return 0;
}

static void nbl_serv_init_port(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops;

	disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->init_port(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static void nbl_serv_configure_virtio_dev_msix(void *priv, u16 vector)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->configure_virtio_dev_msix(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vector);
}

static void nbl_serv_configure_rdma_msix_off(void *priv, u16 vector)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->configure_rdma_msix_off(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vector);
}

static void nbl_serv_configure_virtio_dev_ready(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->configure_virtio_dev_ready(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_set_eth_mac_addr(void *priv, u8 *mac, u8 eth_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);

	if (NBL_COMMON_TO_VF_CAP(common))
		return 0;
	else
		return disp_ops->set_eth_mac_addr(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						  mac, eth_id);
}

static void nbl_serv_adapt_desc_gother(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->adapt_desc_gother(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static void nbl_serv_process_flr(void *priv, u16 vfid)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->flr_clear_queues(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vfid);
	disp_ops->flr_clear_accel_flow(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vfid);
	disp_ops->flr_clear_flows(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vfid);
	disp_ops->flr_clear_accel(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vfid);
	disp_ops->flr_clear_interrupt(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vfid);
	disp_ops->flr_clear_rdma(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vfid);
	disp_ops->flr_clear_net(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vfid);
}

static u16 nbl_serv_covert_vfid_to_vsi_id(void *priv, u16 vfid)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->covert_vfid_to_vsi_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vfid);
}

static void nbl_serv_recovery_abnormal(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->unmask_all_interrupts(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static void nbl_serv_keep_alive(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->keep_alive(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_register_vsi_info(void *priv, struct nbl_vsi_param *vsi_param)
{
	u16 vsi_index = vsi_param->index;
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_ring_mgt *ring_mgt = NBL_SERV_MGT_TO_RING_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	u32 num_cpus;

	ring_mgt->vsi_info[vsi_index].vsi_index = vsi_index;
	ring_mgt->vsi_info[vsi_index].vsi_id = vsi_param->vsi_id;
	ring_mgt->vsi_info[vsi_index].ring_offset = vsi_param->queue_offset;
	ring_mgt->vsi_info[vsi_index].ring_num = vsi_param->queue_num;

	/* init active ring number before first open, guarantee fd direct config check success. */
	num_cpus = num_online_cpus();
	ring_mgt->vsi_info[vsi_index].active_ring_num = (u16)num_cpus > vsi_param->queue_num ?
							vsi_param->queue_num : (u16)num_cpus;

	if (disp_ops->get_product_fix_cap(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					  NBL_ITR_DYNAMIC))
		ring_mgt->vsi_info[vsi_index].itr_dynamic = true;

	disp_ops->register_vsi_ring(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_index,
				    vsi_param->queue_offset, vsi_param->queue_num);

	return disp_ops->register_vsi2q(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), vsi_index,
					vsi_param->vsi_id, vsi_param->queue_offset,
					vsi_param->queue_num);
}

static int nbl_serv_st_open(struct inode *inode, struct file *filep)
{
	struct nbl_serv_st_mgt *p = container_of(inode->i_cdev, struct nbl_serv_st_mgt, cdev);

	filep->private_data = p;

	return 0;
}

static ssize_t nbl_serv_st_write(struct file *file, const char __user *ubuf,
				 size_t size, loff_t *ppos)
{
	return 0;
}

static ssize_t nbl_serv_st_read(struct file *file, char __user *ubuf, size_t size, loff_t *ppos)
{
	return 0;
}

static int nbl_serv_st_release(struct inode *inode, struct file *filp)
{
	return 0;
}

static int nbl_serv_process_passthrough(struct nbl_service_mgt *serv_mgt,
					unsigned int cmd, unsigned long arg)
{
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_passthrough_fw_cmd_param *param = NULL, *result = NULL;
	int ret = 0;

	param = kzalloc(sizeof(*param), GFP_KERNEL);
	if (!param)
		goto alloc_param_fail;

	result = kzalloc(sizeof(*result), GFP_KERNEL);
	if (!result)
		goto alloc_result_fail;

	ret = copy_from_user(param, (void *)arg, _IOC_SIZE(cmd));
	if (ret) {
		nbl_err(common, NBL_DEBUG_ST, "Bad access %d.\n", ret);
		return ret;
	}

	nbl_debug(common, NBL_DEBUG_ST, "Passthough opcode: %d\n", param->opcode);

	ret = disp_ops->passthrough_fw_cmd(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), param, result);
	if (ret)
		goto passthrough_fail;

	ret = copy_to_user((void *)arg, result, _IOC_SIZE(cmd));

passthrough_fail:
	kfree(result);
alloc_result_fail:
	kfree(param);
alloc_param_fail:
	return ret;
}

static int nbl_serv_process_st_info(struct nbl_service_mgt *serv_mgt,
				    unsigned int cmd, unsigned long arg)
{
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_st_info_param *param = NULL;
	int ret = 0;

	nbl_debug(common, NBL_DEBUG_ST, "Get st info\n");

	param = kzalloc(sizeof(*param), GFP_KERNEL);
	if (!param)
		return -ENOMEM;

	strscpy(param->driver_name, NBL_DRIVER_NAME, sizeof(param->driver_name));
	if (net_resource_mgt->netdev)
		strscpy(param->netdev_name[0], net_resource_mgt->netdev->name,
			sizeof(param->netdev_name[0]));

	disp_ops->get_driver_version(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), param->driver_ver,
				     sizeof(param->driver_ver));

	param->bus = common->bus;
	param->devid = common->devid;
	param->function = common->function;
	param->domain = pci_domain_nr(NBL_COMMON_TO_PDEV(common)->bus);

	param->version = IOCTL_ST_INFO_VERSION;

	ret = copy_to_user((void *)arg, param, _IOC_SIZE(cmd));

	kfree(param);
	return ret;
}

static long nbl_serv_st_unlock_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct nbl_serv_st_mgt *st_mgt = file->private_data;
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)st_mgt->serv_mgt;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	int ret = 0;

	if (_IOC_TYPE(cmd) != IOCTL_TYPE) {
		nbl_err(common, NBL_DEBUG_ST, "cmd %u, bad magic 0x%x/0x%x.\n",
			cmd, _IOC_TYPE(cmd), IOCTL_TYPE);
		return -ENOTTY;
	}

	if (_IOC_DIR(cmd) & _IOC_READ)
		ret = !access_ok((void __user *)arg, _IOC_SIZE(cmd));
	else if (_IOC_DIR(cmd) & _IOC_WRITE)
		ret = !access_ok((void __user *)arg, _IOC_SIZE(cmd));

	if (ret) {
		nbl_err(common, NBL_DEBUG_ST, "Bad access.\n");
		return ret;
	}

	switch (cmd) {
	case IOCTL_PASSTHROUGH:
		ret = nbl_serv_process_passthrough(serv_mgt, cmd, arg);
		break;
	case IOCTL_ST_INFO:
		ret = nbl_serv_process_st_info(serv_mgt, cmd, arg);
		break;
	default:
		nbl_err(common, NBL_DEBUG_ST, "Unknown cmd %d.\n", cmd);
		return -EFAULT;
	}

	return ret;
}

static const struct file_operations st_ops = {
	.owner = THIS_MODULE,
	.open = nbl_serv_st_open,
	.write = nbl_serv_st_write,
	.read = nbl_serv_st_read,
	.unlocked_ioctl = nbl_serv_st_unlock_ioctl,
	.release = nbl_serv_st_release,
};

static int nbl_serv_alloc_subdev_id(struct nbl_software_tool_table *st_table)
{
	int subdev_id;

	subdev_id = find_first_zero_bit(st_table->devid, NBL_ST_MAX_DEVICE_NUM);
	if (subdev_id == NBL_ST_MAX_DEVICE_NUM)
		return -ENOSPC;
	set_bit(subdev_id, st_table->devid);

	return subdev_id;
}

static void nbl_serv_free_subdev_id(struct nbl_software_tool_table *st_table, int id)
{
	clear_bit(id, st_table->devid);
}

static int nbl_serv_setup_st(void *priv, void *st_table_param)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_software_tool_table *st_table = (struct nbl_software_tool_table *)st_table_param;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	struct nbl_serv_st_mgt *st_mgt = NBL_SERV_MGT_TO_ST_MGT(serv_mgt);
	struct device *test_device;
	char name[NBL_RESTOOL_NAME_LEN] = {0};
	dev_t devid;
	int id, subdev_id, ret = 0;

	id = NBL_COMMON_TO_BOARD_ID(common);

	subdev_id = nbl_serv_alloc_subdev_id(st_table);
	if (subdev_id < 0)
		goto alloc_subdev_id_fail;

	devid = MKDEV(st_table->major, subdev_id);

	if (!NBL_COMMON_TO_PCI_FUNC_ID(common))
		snprintf(name, sizeof(name), "nblst%04x_conf%d",
			 NBL_COMMON_TO_PDEV(common)->device, id);
	else
		snprintf(name, sizeof(name), "nblst%04x_conf%d.%d",
			 NBL_COMMON_TO_PDEV(common)->device, id, NBL_COMMON_TO_PCI_FUNC_ID(common));

	st_mgt = devm_kzalloc(NBL_COMMON_TO_DEV(common), sizeof(*st_mgt), GFP_KERNEL);
	if (!st_mgt)
		goto malloc_fail;

	st_mgt->serv_mgt = serv_mgt;

	st_mgt->major = MAJOR(devid);
	st_mgt->minor = MINOR(devid);
	st_mgt->devno = devid;
	st_mgt->subdev_id = subdev_id;

	cdev_init(&st_mgt->cdev, &st_ops);
	ret = cdev_add(&st_mgt->cdev, devid, 1);
	if (ret)
		goto cdev_add_fail;

	test_device = device_create(st_table->cls, NULL, st_mgt->devno, NULL, name);
	if (IS_ERR(test_device)) {
		ret = -EBUSY;
		goto device_create_fail;
	}

	NBL_SERV_MGT_TO_ST_MGT(serv_mgt) = st_mgt;
	return 0;

device_create_fail:
	cdev_del(&st_mgt->cdev);
cdev_add_fail:
	devm_kfree(NBL_COMMON_TO_DEV(common), st_mgt);
malloc_fail:
	nbl_serv_free_subdev_id(st_table, subdev_id);
alloc_subdev_id_fail:
	return ret;
}

static void nbl_serv_remove_st(void *priv, void *st_table_param)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_software_tool_table *st_table = (struct nbl_software_tool_table *)st_table_param;
	struct nbl_serv_st_mgt *st_mgt = NBL_SERV_MGT_TO_ST_MGT(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);

	if (!st_mgt)
		return;

	device_destroy(st_table->cls, st_mgt->devno);
	cdev_del(&st_mgt->cdev);

	nbl_serv_free_subdev_id(st_table, st_mgt->subdev_id);

	NBL_SERV_MGT_TO_ST_MGT(serv_mgt) = NULL;
	devm_kfree(NBL_COMMON_TO_DEV(common), st_mgt);
}

static int nbl_serv_set_spoof_check_addr(void *priv, u8 *mac)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);

	return disp_ops->set_spoof_check_addr(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					      NBL_COMMON_TO_VSI_ID(common), mac);
}

static u16 nbl_serv_get_vf_base_vsi_id(void *priv, u16 func_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_vf_base_vsi_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), func_id);
}

static int nbl_serv_get_board_id(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->get_board_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt));
}

static int nbl_serv_process_abnormal_event(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_abnormal_event_info abnomal_info;
	struct nbl_abnormal_details *detail;
	u16 local_queue_id;
	int type, i, ret = 0;

	memset(&abnomal_info, 0, sizeof(abnomal_info));

	ret = disp_ops->process_abnormal_event(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), &abnomal_info);
	if (!ret)
		return ret;

	for (i = 0; i < NBL_ABNORMAL_EVENT_MAX; i++) {
		detail = &abnomal_info.details[i];

		if (!detail->abnormal)
			continue;

		type = nbl_serv_abnormal_event_to_queue(i);
		local_queue_id = disp_ops->get_local_queue_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
							      detail->vsi_id, detail->qid);
		if (local_queue_id == U16_MAX)
			return 0;

		nbl_serv_restore_queue(serv_mgt, detail->vsi_id, local_queue_id, type, true);
	}

	return 0;
}

static int nbl_serv_cfg_bond_shaping(void *priv, u8 eth_id, bool enable)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	return disp_ops->cfg_bond_shaping(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), eth_id, enable);
}

static void nbl_serv_cfg_bgid_back_pressure(void *priv, u8 main_eth_id,
					    u8 other_eth_id, bool enable)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->cfg_bgid_back_pressure(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), main_eth_id,
					 other_eth_id, enable);
}

static void nbl_serv_cfg_eth_bond_event(void *priv, bool enable)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->cfg_eth_bond_event(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), enable);
}

static ssize_t nbl_serv_vf_mac_show(struct kobject *kobj, struct kobj_attribute *attr, char *buf)
{
	return sprintf(buf, "usage: write MAC ADDR to set mac address\n");
}

static ssize_t nbl_serv_vf_mac_store(struct kobject *kobj, struct kobj_attribute *attr,
				     const char *buf, size_t count)
{
	struct nbl_serv_vf_info *vf_info = container_of(kobj, struct nbl_serv_vf_info, kobj);
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)vf_info->priv;
	u8 mac[ETH_ALEN];
	int ret = 0;

	ret = sscanf(buf, "%hhx:%hhx:%hhx:%hhx:%hhx:%hhx",
		     &mac[0], &mac[1], &mac[2], &mac[3], &mac[4], &mac[5]);
	if (ret != ETH_ALEN)
		return -EINVAL;

	ret = nbl_serv_set_vf_mac(NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt)->netdev,
				  vf_info->vf_id, mac);
	return ret ? ret : count;
}

static ssize_t nbl_serv_vf_vlan_show(struct kobject *kobj, struct kobj_attribute *attr, char *buf)
{
	return sprintf(buf, "usage: write <Vlan:Qos[:Proto]> to set VF Vlan,"
		       " Qos, and optionally Vlan Protocol (default 802.1Q)\n");
}

static ssize_t nbl_serv_vf_vlan_store(struct kobject *kobj, struct kobj_attribute *attr,
				      const char *buf, size_t count)
{
	struct nbl_serv_vf_info *vf_info = container_of(kobj, struct nbl_serv_vf_info, kobj);
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)vf_info->priv;
	char vproto_ext[5] = {'\0'};
	__be16 vlan_proto;
	u16 vlan_id;
	u8 qos;
	int ret = 0;

	ret = sscanf(buf, "%hu:%hhu:802.%4s", &vlan_id, &qos, vproto_ext);
	if (ret == 3) {
		if ((strcmp(vproto_ext, "1AD") == 0) ||
		    (strcmp(vproto_ext, "1ad") == 0))
			vlan_proto = htons(ETH_P_8021AD);
		else if ((strcmp(vproto_ext, "1Q") == 0) ||
			 (strcmp(vproto_ext, "1q") == 0))
			vlan_proto = htons(ETH_P_8021Q);
		else
			return -EINVAL;
	} else {
		ret = sscanf(buf, "%hu:%hhu", &vlan_id, &qos);
		if (ret != 2)
			return -EINVAL;
		vlan_proto = htons(ETH_P_8021Q);
	}

	ret = nbl_serv_set_vf_vlan(NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt)->netdev,
				   vf_info->vf_id, vlan_id, qos, vlan_proto);
	return ret ? ret : count;
}

static ssize_t nbl_serv_vf_max_tx_rate_show(struct kobject *kobj, struct kobj_attribute *attr,
					    char *buf)
{
	return sprintf(buf, "usage: write RATE to set max_tx_rate(Mbps)\n");
}

static ssize_t nbl_serv_vf_max_tx_rate_store(struct kobject *kobj, struct kobj_attribute *attr,
					     const char *buf, size_t count)
{
	struct nbl_serv_vf_info *vf_info = container_of(kobj, struct nbl_serv_vf_info, kobj);
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)vf_info->priv;
	int max_tx_rate = 0, ret = 0;

	ret = kstrtos32(buf, 0, &max_tx_rate);
	if (ret)
		return -EINVAL;

	ret = nbl_serv_set_vf_rate(NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt)->netdev,
				   vf_info->vf_id, 0, max_tx_rate);
	return ret ? ret : count;
}

static ssize_t nbl_serv_vf_spoofchk_show(struct kobject *kobj, struct kobj_attribute *attr,
					 char *buf)
{
	return sprintf(buf, "usage: write <ON|OFF> to set vf spoof check\n");
}

static ssize_t nbl_serv_vf_spoofchk_store(struct kobject *kobj, struct kobj_attribute *attr,
					  const char *buf, size_t count)
{
	struct nbl_serv_vf_info *vf_info = container_of(kobj, struct nbl_serv_vf_info, kobj);
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)vf_info->priv;
	bool enable = false;
	int ret = 0;

	if (sysfs_streq(buf, "ON"))
		enable = true;
	else if (sysfs_streq(buf, "OFF"))
		enable = false;
	else
		return -EINVAL;

	ret = nbl_serv_set_vf_spoofchk(NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt)->netdev,
				       vf_info->vf_id, enable);
	return ret ? ret : count;
}

static ssize_t nbl_serv_vf_link_state_show(struct kobject *kobj, struct kobj_attribute *attr,
					   char *buf)
{
	return sprintf(buf, "usage: write <AUTO|ENABLE|DISABLE> to set vf link state\n");
}

static ssize_t nbl_serv_vf_link_state_store(struct kobject *kobj, struct kobj_attribute *attr,
					    const char *buf, size_t count)
{
	struct nbl_serv_vf_info *vf_info = container_of(kobj, struct nbl_serv_vf_info, kobj);
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)vf_info->priv;
	int state = 0, ret = 0;

	if (sysfs_streq(buf, "AUTO"))
		state = IFLA_VF_LINK_STATE_AUTO;
	else if (sysfs_streq(buf, "ENABLE"))
		state = IFLA_VF_LINK_STATE_ENABLE;
	else if (sysfs_streq(buf, "DISABLE"))
		state = IFLA_VF_LINK_STATE_DISABLE;
	else
		return -EINVAL;

	ret = nbl_serv_set_vf_link_state(NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt)->netdev,
					 vf_info->vf_id, state);
	return ret ? ret : count;
}

static ssize_t nbl_serv_vf_config_show(struct kobject *kobj, struct attribute *attr, char *buf)
{
	struct kobj_attribute *kattr = container_of(attr, struct kobj_attribute, attr);

	if (kattr->show)
		return kattr->show(kobj, kattr, buf);

	return -EIO;
}

static ssize_t nbl_serv_vf_config_store(struct kobject *kobj, struct attribute *attr,
					const char *buf, size_t count)
{
	struct kobj_attribute *kattr = container_of(attr, struct kobj_attribute, attr);

	if (kattr->show)
		return kattr->store(kobj, kattr, buf, count);

	return -EIO;
}

static struct kobj_attribute nbl_attr_vf_mac = {
	.attr = {.name = "mac",
		 .mode = 0644},
	.show = nbl_serv_vf_mac_show,
	.store = nbl_serv_vf_mac_store,
};

static struct kobj_attribute nbl_attr_vf_vlan = {
	.attr = {.name = "vlan",
		 .mode = 0644},
	.show = nbl_serv_vf_vlan_show,
	.store = nbl_serv_vf_vlan_store,
};

static struct kobj_attribute nbl_attr_vf_max_tx_rate = {
	.attr = {.name = "max_tx_rate",
		 .mode = 0644},
	.show = nbl_serv_vf_max_tx_rate_show,
	.store = nbl_serv_vf_max_tx_rate_store,
};

static struct kobj_attribute nbl_attr_vf_spoofchk = {
	.attr = {.name = "spoofchk",
		 .mode = 0644},
	.show = nbl_serv_vf_spoofchk_show,
	.store = nbl_serv_vf_spoofchk_store,
};

static struct kobj_attribute nbl_attr_vf_link_state = {
	.attr = {.name = "link_state",
		 .mode = 0644},
	.show = nbl_serv_vf_link_state_show,
	.store = nbl_serv_vf_link_state_store,
};

static struct attribute *nbl_vf_config_attrs[] = {
	&nbl_attr_vf_mac.attr,
	&nbl_attr_vf_vlan.attr,
	&nbl_attr_vf_max_tx_rate.attr,
	&nbl_attr_vf_spoofchk.attr,
	&nbl_attr_vf_link_state.attr,
	NULL,
};

ATTRIBUTE_GROUPS(nbl_vf_config);

static const struct sysfs_ops nbl_sysfs_ops_vf = {
	.show = nbl_serv_vf_config_show,
	.store = nbl_serv_vf_config_store,
};

static const struct kobj_type nbl_kobj_vf_type = {
	.sysfs_ops = &nbl_sysfs_ops_vf,
	.default_groups = nbl_vf_config_groups,
};

static int nbl_serv_setup_vf_sysfs(struct nbl_service_mgt *serv_mgt)
{
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_serv_vf_info *vf_info = net_resource_mgt->vf_info;
	int i = 0, ret = 0;

	for (i = 0; i < net_resource_mgt->num_vfs; i++) {
		vf_info[i].priv = serv_mgt;
		vf_info[i].vf_id = (u16)i;

		ret = kobject_init_and_add(&vf_info[i].kobj, &nbl_kobj_vf_type,
					   net_resource_mgt->sriov_kobj, "vf%d", i);
		if (ret)
			goto err;
	}

	return 0;

err:
	while (--i + 1)
		kobject_put(&vf_info[i].kobj);

	return ret;
}

static void nbl_serv_remove_vf_sysfs(struct nbl_service_mgt *serv_mgt)
{
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_serv_vf_info *vf_info = net_resource_mgt->vf_info;
	int i = 0;

	for (i = 0; i < net_resource_mgt->num_vfs; i++)
		kobject_put(&vf_info[i].kobj);
}

static int nbl_serv_setup_vf_config(void *priv, int num_vfs, bool is_flush)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	struct nbl_serv_vf_info *vf_info = net_resource_mgt->vf_info;
	struct nbl_common_info *common = NBL_SERV_MGT_TO_COMMON(serv_mgt);
	u16 func_id = U16_MAX;
	u16 vlan_tci;
	bool should_notify;
	int i, ret = 0;

	net_resource_mgt->num_vfs = num_vfs;

	for (i = 0; i < net_resource_mgt->num_vfs; i++) {
		func_id = disp_ops->get_vf_function_id(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						       NBL_COMMON_TO_VSI_ID(common), i);

		if (func_id == U16_MAX) {
			nbl_err(common, NBL_DEBUG_MAIN, "vf id %d invalid\n", i);
			return -EINVAL;
		}

		disp_ops->register_func_mac(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					    vf_info[i].mac, func_id);

		vlan_tci = vf_info[i].vlan | (u16)(vf_info[i].vlan_qos << VLAN_PRIO_SHIFT);
		ret = disp_ops->register_func_vlan(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), func_id,
						   vlan_tci, vf_info[i].vlan_proto,
						   &should_notify);
		if (ret)
			break;

		ret = disp_ops->register_func_rate(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), func_id,
						   vf_info[i].max_tx_rate);
		if (ret)
			break;

		ret = disp_ops->set_tx_rate(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					    func_id, vf_info[i].max_tx_rate);
		if (ret)
			break;

		ret = disp_ops->set_vf_spoof_check(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
						   NBL_COMMON_TO_VSI_ID(common), i,
						   vf_info[i].spoof_check);
		if (ret)
			break;

		/* No need to notify vf, vf will get link forced when probe,
		 * Here we only flush the config.
		 */
		ret = disp_ops->register_func_link_forced(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
							  func_id, vf_info[i].state,
							  &should_notify);
		if (ret)
			break;
	}

	if (!ret && net_resource_mgt->sriov_kobj && !is_flush)
		ret = nbl_serv_setup_vf_sysfs(serv_mgt);

	if (ret)
		net_resource_mgt->num_vfs = 0;

	return ret;
}

static void nbl_serv_remove_vf_config(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_serv_vf_info *vf_info = net_resource_mgt->vf_info;
	int i;

	nbl_serv_remove_vf_sysfs(serv_mgt);

	for (i = 0; i < net_resource_mgt->num_vfs; i++)
		memset(&vf_info[i], 0, sizeof(vf_info[i]));

	nbl_serv_setup_vf_config(priv, net_resource_mgt->num_vfs, true);

	net_resource_mgt->num_vfs = 0;
}

static int nbl_serv_setup_vf_resource(void *priv, int num_vfs)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct nbl_serv_vf_info *vf_info;
	struct device *dev = NBL_SERV_MGT_TO_DEV(serv_mgt);
	int i;

	net_resource_mgt->total_vfs = num_vfs;

	net_resource_mgt->vf_info = devm_kcalloc(dev, net_resource_mgt->total_vfs,
						 sizeof(struct nbl_serv_vf_info), GFP_KERNEL);
	if (!net_resource_mgt->vf_info)
		return -ENOMEM;

	vf_info = net_resource_mgt->vf_info;
	for (i = 0; i < net_resource_mgt->total_vfs; i++) {
		vf_info[i].state = IFLA_VF_LINK_STATE_AUTO;
		vf_info[i].spoof_check = false;
	}

	net_resource_mgt->sriov_kobj = kobject_create_and_add("SRIOV", &dev->kobj);
	if (!net_resource_mgt->sriov_kobj)
		nbl_warn(NBL_SERV_MGT_TO_COMMON(serv_mgt), NBL_DEBUG_MAIN,
			 "Fail to create sriov sysfs");

	return 0;
}

static void nbl_serv_remove_vf_resource(void *priv)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = NBL_SERV_MGT_TO_NET_RES_MGT(serv_mgt);
	struct device *dev = NBL_SERV_MGT_TO_DEV(serv_mgt);

	nbl_serv_remove_vf_config(priv);

	kobject_put(net_resource_mgt->sriov_kobj);

	if (net_resource_mgt->vf_info) {
		devm_kfree(dev, net_resource_mgt->vf_info);
		net_resource_mgt->vf_info = NULL;
	}
}

static void nbl_serv_cfg_fd_update_event(void *priv, bool enable)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->cfg_fd_update_event(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), enable);
}

static void nbl_serv_get_xdp_queue_info(void *priv, u16 *queue_num, u16 *queue_size, u16 vsi_id)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->get_xdp_queue_info(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), queue_num, queue_size,
				     vsi_id);
}

static void nbl_serv_set_hw_status(void *priv, enum nbl_hw_status hw_status)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->set_hw_status(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), hw_status);
}

static void nbl_serv_get_active_func_bitmaps(void *priv, unsigned long *bitmap, int max_func)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);

	disp_ops->get_active_func_bitmaps(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt), bitmap, max_func);
}

static int nbl_serv_configure_qos(void *priv, u8 eth_id, u8 *pfc, u8 trust, u8 *dscp2prio_map)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_serv_net_resource_mgt *net_resource_mgt = serv_mgt->net_resource_mgt;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	int ret;

	ret = disp_ops->configure_qos(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
				      eth_id, pfc, trust, dscp2prio_map);

	net_resource_mgt->pfc_mode = trust;
	memcpy(net_resource_mgt->dscp2prio_map, dscp2prio_map, NBL_DSCP_MAX);

	return ret;
}

static int nbl_serv_set_pfc_buffer_size(void *priv, u8 eth_id, u8 prio, int xoff, int xon)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	int ret;

	ret = disp_ops->set_pfc_buffer_size(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					    eth_id, prio, xoff, xon);

	return ret;
}

static int nbl_serv_get_pfc_buffer_size(void *priv, u8 eth_id, u8 prio, int *xoff, int *xon)
{
	struct nbl_service_mgt *serv_mgt = (struct nbl_service_mgt *)priv;
	struct nbl_dispatch_ops *disp_ops = NBL_SERV_MGT_TO_DISP_OPS(serv_mgt);
	int ret;

	ret = disp_ops->get_pfc_buffer_size(NBL_SERV_MGT_TO_DISP_PRIV(serv_mgt),
					    eth_id, prio, xoff, xon);

	return ret;
}

static struct nbl_service_ops serv_ops = {
	.init_chip_factory = nbl_serv_init_chip_factory,
	.destroy_chip_factory = nbl_serv_destroy_chip_factory,
	.init_chip = nbl_serv_init_chip,
	.destroy_chip = nbl_serv_destroy_chip,

	.configure_msix_map = nbl_serv_configure_msix_map,
	.destroy_msix_map = nbl_serv_destroy_msix_map,
	.enable_mailbox_irq = nbl_serv_enable_mailbox_irq,
	.enable_abnormal_irq = nbl_serv_enable_abnormal_irq,
	.enable_adminq_irq = nbl_serv_enable_adminq_irq,
	.request_net_irq = nbl_serv_request_net_irq,
	.free_net_irq = nbl_serv_free_net_irq,
	.get_global_vector = nbl_serv_get_global_vector,
	.get_msix_entry_id = nbl_serv_get_msix_entry_id,
	.get_common_irq_num = nbl_serv_get_common_irq_num,
	.get_ctrl_irq_num = nbl_serv_get_ctrl_irq_num,
	.get_chip_temperature = nbl_serv_get_chip_temperature,
	.get_module_temperature = nbl_serv_get_module_temperature,
	.get_port_attributes = nbl_serv_get_port_attributes,
	.update_template_config = nbl_serv_update_template_config,
	.enable_port = nbl_serv_enable_port,
	.init_port = nbl_serv_init_port,
	.set_sfp_state = nbl_serv_set_sfp_state,

	.register_net = nbl_serv_register_net,
	.unregister_net = nbl_serv_unregister_net,
	.setup_txrx_queues = nbl_serv_setup_txrx_queues,
	.remove_txrx_queues = nbl_serv_remove_txrx_queues,
	.check_offload_status = nbl_serv_check_offload_status,
	.init_tx_rate = nbl_serv_init_tx_rate,
	.setup_q2vsi = nbl_serv_setup_q2vsi,
	.remove_q2vsi = nbl_serv_remove_q2vsi,
	.setup_rss = nbl_serv_setup_rss,
	.remove_rss = nbl_serv_remove_rss,
	.register_vsi_info = nbl_serv_register_vsi_info,

	.alloc_rings = nbl_serv_alloc_rings,
	.cpu_affinity_init = nbl_serv_cpu_affinity_init,
	.free_rings = nbl_serv_free_rings,
	.enable_napis = nbl_serv_enable_napis,
	.disable_napis = nbl_serv_disable_napis,
	.set_mask_en = nbl_serv_set_mask_en,
	.start_net_flow = nbl_serv_start_net_flow,
	.stop_net_flow = nbl_serv_stop_net_flow,
	.set_lldp_flow = nbl_serv_set_lldp_flow,
	.remove_lldp_flow = nbl_serv_remove_lldp_flow,
	.start_mgt_flow = nbl_serv_start_mgt_flow,
	.stop_mgt_flow = nbl_serv_stop_mgt_flow,
	.get_tx_headroom = nbl_serv_get_tx_headroom,
	.get_product_flex_cap	= nbl_serv_get_product_flex_cap,
	.get_product_fix_cap	= nbl_serv_get_product_fix_cap,
	.set_spoof_check_addr = nbl_serv_set_spoof_check_addr,

	.vsi_open = nbl_serv_vsi_open,
	.vsi_stop = nbl_serv_vsi_stop,
	.switch_traffic_default_dest = nbl_serv_switch_traffic_default_dest,
	.config_fd_flow_state = nbl_serv_config_fd_flow_state,

	/* For netdev ops */
	.netdev_open = nbl_serv_netdev_open,
	.netdev_stop = nbl_serv_netdev_stop,
	.change_mtu = nbl_serv_change_mtu,
	.set_mac = nbl_serv_set_mac,
	.rx_add_vid = nbl_serv_rx_add_vid,
	.rx_kill_vid = nbl_serv_rx_kill_vid,
	.get_stats64 = nbl_serv_get_stats64,
	.set_rx_mode = nbl_serv_set_rx_mode,
	.change_rx_flags = nbl_serv_change_rx_flags,
	.set_features = nbl_serv_set_features,
	.features_check = nbl_serv_features_check,
	.setup_tc = nbl_serv_setup_tc,
	.set_vf_spoofchk = nbl_serv_set_vf_spoofchk,
	.get_phys_port_name = nbl_serv_get_phys_port_name,
	.get_port_parent_id = nbl_serv_get_port_parent_id,
	.tx_timeout = nbl_serv_tx_timeout,
	.bridge_setlink = nbl_serv_bridge_setlink,
	.bridge_getlink = nbl_serv_bridge_getlink,
	.set_vf_link_state = nbl_serv_set_vf_link_state,
	.set_vf_mac = nbl_serv_set_vf_mac,
	.set_vf_rate = nbl_serv_set_vf_rate,
	.set_vf_vlan = nbl_serv_set_vf_vlan,
	.get_vf_config = nbl_serv_get_vf_config,
	.select_queue = nbl_serv_select_queue,

	/* For rep associated */
	.rep_netdev_open = nbl_serv_rep_netdev_open,
	.rep_netdev_stop = nbl_serv_rep_netdev_stop,
	.rep_start_xmit = nbl_serv_rep_start_xmit,
	.rep_get_stats64 = nbl_serv_rep_get_stats64,
	.rep_set_rx_mode = nbl_serv_rep_set_rx_mode,
	.rep_set_mac = nbl_serv_rep_set_mac,
	.rep_rx_add_vid = nbl_serv_rep_rx_add_vid,
	.rep_rx_kill_vid = nbl_serv_rep_rx_kill_vid,
	.rep_setup_tc = nbl_serv_rep_setup_tc,
	.rep_get_phys_port_name = nbl_serv_rep_get_phys_port_name,
	.rep_get_port_parent_id = nbl_serv_rep_get_port_parent_id,
	.get_rep_feature = nbl_serv_get_rep_feature,
	.get_rep_queue_num = nbl_serv_get_rep_queue_num,
	.get_rep_queue_info = nbl_serv_get_rep_queue_info,
	.get_user_queue_info = nbl_serv_get_user_queue_info,
	.alloc_rep_queue_mgt = nbl_serv_alloc_rep_queue_mgt,
	.free_rep_queue_mgt = nbl_serv_free_rep_queue_mgt,
	.set_eswitch_mode = nbl_serv_set_eswitch_mode,
	.get_eswitch_mode = nbl_serv_get_eswitch_mode,
	.alloc_rep_data = nbl_serv_alloc_rep_data,
	.free_rep_data = nbl_serv_free_rep_data,
	.set_rep_netdev_info = nbl_serv_set_rep_netdev_info,
	.unset_rep_netdev_info = nbl_serv_unset_rep_netdev_info,
	.disable_phy_flow = nbl_serv_disable_phy_flow,
	.enable_phy_flow = nbl_serv_enable_phy_flow,
	.init_acl = nbl_serv_init_acl,
	.uninit_acl = nbl_serv_uninit_acl,
	.set_upcall_rule = nbl_serv_set_upcall_rule,
	.unset_upcall_rule = nbl_serv_unset_upcall_rule,
	.switchdev_init_cmdq = nbl_serv_switchdev_init_cmdq,
	.switchdev_deinit_cmdq = nbl_serv_switchdev_deinit_cmdq,
	.set_tc_flow_info = nbl_serv_set_tc_flow_info,
	.unset_tc_flow_info = nbl_serv_unset_tc_flow_info,
	.get_tc_flow_info = nbl_serv_get_tc_flow_info,
	.register_indr_dev_tc_offload = nbl_serv_register_indr_dev_tc_offload,
	.unregister_indr_dev_tc_offload = nbl_serv_unregister_indr_dev_tc_offload,
	.set_lag_info = nbl_serv_set_lag_info,
	.unset_lag_info = nbl_serv_unset_lag_info,
	.set_netdev_ops = nbl_serv_set_netdev_ops,

	.get_vsi_id = nbl_serv_get_vsi_id,
	.get_eth_id = nbl_serv_get_eth_id,
	.setup_net_resource_mgt = nbl_serv_setup_net_resource_mgt,
	.remove_net_resource_mgt = nbl_serv_remove_net_resource_mgt,
	.enable_lag_protocol = nbl_serv_enable_lag_protocol,
	.cfg_lag_hash_algorithm = nbl_serv_cfg_lag_hash_algorithm,
	.cfg_lag_member_fwd = nbl_serv_cfg_lag_member_fwd,
	.cfg_lag_member_list = nbl_serv_cfg_lag_member_list,
	.cfg_lag_member_up_attr = nbl_serv_cfg_lag_member_up_attr,
	.cfg_bond_shaping = nbl_serv_cfg_bond_shaping,
	.cfg_bgid_back_pressure = nbl_serv_cfg_bgid_back_pressure,

	.get_rdma_cap_num = nbl_serv_get_rdma_cap_num,
	.setup_rdma_id = nbl_serv_setup_rdma_id,
	.remove_rdma_id = nbl_serv_remove_rdma_id,
	.register_rdma = nbl_serv_register_rdma,
	.unregister_rdma = nbl_serv_unregister_rdma,
	.register_rdma_bond = nbl_serv_register_rdma_bond,
	.unregister_rdma_bond = nbl_serv_unregister_rdma_bond,
	.get_hw_addr = nbl_serv_get_hw_addr,
	.get_real_hw_addr = nbl_serv_get_real_hw_addr,
	.get_function_id = nbl_serv_get_function_id,
	.get_real_bdf = nbl_serv_get_real_bdf,
	.set_eth_mac_addr = nbl_serv_set_eth_mac_addr,
	.process_abnormal_event = nbl_serv_process_abnormal_event,
	.adapt_desc_gother = nbl_serv_adapt_desc_gother,
	.process_flr = nbl_serv_process_flr,
	.get_board_id = nbl_serv_get_board_id,
	.covert_vfid_to_vsi_id = nbl_serv_covert_vfid_to_vsi_id,
	.recovery_abnormal = nbl_serv_recovery_abnormal,
	.keep_alive = nbl_serv_keep_alive,

	.get_devlink_info = nbl_serv_get_devlink_info,
	.update_devlink_flash = nbl_serv_update_devlink_flash,
	.get_adminq_tx_buf_size = nbl_serv_get_adminq_tx_buf_size,
	.emp_console_write = nbl_serv_emp_console_write,

	.check_fw_heartbeat = nbl_serv_check_fw_heartbeat,
	.check_fw_reset = nbl_serv_check_fw_reset,
	.set_netdev_carrier_state = nbl_serv_set_netdev_carrier_state,
	.cfg_eth_bond_event = nbl_serv_cfg_eth_bond_event,
	.cfg_fd_update_event = nbl_serv_cfg_fd_update_event,

	/* For virtio */
	.configure_virtio_dev_msix = nbl_serv_configure_virtio_dev_msix,
	.configure_rdma_msix_off = nbl_serv_configure_rdma_msix_off,
	.configure_virtio_dev_ready = nbl_serv_configure_virtio_dev_ready,

	.setup_st = nbl_serv_setup_st,
	.remove_st = nbl_serv_remove_st,
	.get_vf_base_vsi_id = nbl_serv_get_vf_base_vsi_id,

	.setup_vf_config = nbl_serv_setup_vf_config,
	.remove_vf_config = nbl_serv_remove_vf_config,
	.setup_vf_resource = nbl_serv_setup_vf_resource,
	.remove_vf_resource = nbl_serv_remove_vf_resource,

	.get_xdp_queue_info = nbl_serv_get_xdp_queue_info,
	.set_hw_status = nbl_serv_set_hw_status,
	.get_active_func_bitmaps = nbl_serv_get_active_func_bitmaps,
	.configure_qos = nbl_serv_configure_qos,
	.set_pfc_buffer_size = nbl_serv_set_pfc_buffer_size,
	.get_pfc_buffer_size = nbl_serv_get_pfc_buffer_size,
};

/* Structure starts here, adding an op should not modify anything below */
static int nbl_serv_setup_serv_mgt(struct nbl_common_info *common,
				   struct nbl_service_mgt **serv_mgt)
{
	struct device *dev;

	dev = NBL_COMMON_TO_DEV(common);
	*serv_mgt = devm_kzalloc(dev, sizeof(struct nbl_service_mgt), GFP_KERNEL);
	if (!*serv_mgt)
		return -ENOMEM;

	NBL_SERV_MGT_TO_COMMON(*serv_mgt) = common;
	nbl_serv_setup_flow_mgt(NBL_SERV_MGT_TO_FLOW_MGT(*serv_mgt));

	return 0;
}

static void nbl_serv_remove_serv_mgt(struct nbl_common_info *common,
				     struct nbl_service_mgt **serv_mgt)
{
	struct device *dev;

	dev = NBL_COMMON_TO_DEV(common);
	devm_kfree(dev, *serv_mgt);
	*serv_mgt = NULL;
}

static void nbl_serv_remove_ops(struct device *dev, struct nbl_service_ops_tbl **serv_ops_tbl)
{
	devm_kfree(dev, *serv_ops_tbl);
	*serv_ops_tbl = NULL;
}

static int nbl_serv_setup_ops(struct device *dev, struct nbl_service_ops_tbl **serv_ops_tbl,
			      struct nbl_service_mgt *serv_mgt)
{
	*serv_ops_tbl = devm_kzalloc(dev, sizeof(struct nbl_service_ops_tbl), GFP_KERNEL);
	if (!*serv_ops_tbl)
		return -ENOMEM;

	NBL_SERV_OPS_TBL_TO_OPS(*serv_ops_tbl) = &serv_ops;
	nbl_serv_setup_ethtool_ops(&serv_ops);
	nbl_serv_setup_ktls_ops(&serv_ops);
	nbl_serv_setup_xfrm_ops(&serv_ops);
	NBL_SERV_OPS_TBL_TO_PRIV(*serv_ops_tbl) = serv_mgt;

	return 0;
}

int nbl_serv_init(void *p, struct nbl_init_param *param)
{
	struct nbl_adapter *adapter = (struct nbl_adapter *)p;
	struct device *dev;
	struct nbl_common_info *common;
	struct nbl_service_mgt **serv_mgt;
	struct nbl_service_ops_tbl **serv_ops_tbl;
	struct nbl_dispatch_ops_tbl *disp_ops_tbl;
	struct nbl_dispatch_ops *disp_ops;
	struct nbl_channel_ops_tbl *chan_ops_tbl;
	int ret = 0;

	dev = NBL_ADAPTER_TO_DEV(adapter);
	common = NBL_ADAPTER_TO_COMMON(adapter);
	serv_mgt = (struct nbl_service_mgt **)&NBL_ADAPTER_TO_SERV_MGT(adapter);
	serv_ops_tbl = &NBL_ADAPTER_TO_SERV_OPS_TBL(adapter);
	disp_ops_tbl = NBL_ADAPTER_TO_DISP_OPS_TBL(adapter);
	chan_ops_tbl = NBL_ADAPTER_TO_CHAN_OPS_TBL(adapter);
	disp_ops = disp_ops_tbl->ops;

	ret = nbl_serv_setup_serv_mgt(common, serv_mgt);
	if (ret)
		goto setup_mgt_fail;

	ret = nbl_serv_setup_ops(dev, serv_ops_tbl, *serv_mgt);
	if (ret)
		goto setup_ops_fail;

	NBL_SERV_MGT_TO_DISP_OPS_TBL(*serv_mgt) = disp_ops_tbl;
	NBL_SERV_MGT_TO_CHAN_OPS_TBL(*serv_mgt) = chan_ops_tbl;
	disp_ops->get_resource_pt_ops(disp_ops_tbl->priv, &(*serv_ops_tbl)->pt_ops);

	return 0;

setup_ops_fail:
	nbl_serv_remove_serv_mgt(common, serv_mgt);
setup_mgt_fail:
	return ret;
}

void nbl_serv_remove(void *p)
{
	struct nbl_adapter *adapter = (struct nbl_adapter *)p;
	struct device *dev;
	struct nbl_common_info *common;
	struct nbl_service_mgt **serv_mgt;
	struct nbl_service_ops_tbl **serv_ops_tbl;

	if (!adapter)
		return;

	dev = NBL_ADAPTER_TO_DEV(adapter);
	common = NBL_ADAPTER_TO_COMMON(adapter);
	serv_mgt = (struct nbl_service_mgt **)&NBL_ADAPTER_TO_SERV_MGT(adapter);
	serv_ops_tbl = &NBL_ADAPTER_TO_SERV_OPS_TBL(adapter);

	nbl_serv_remove_ops(dev, serv_ops_tbl);
	nbl_serv_remove_serv_mgt(common, serv_mgt);
}
