use indexmap::map::IndexMap;

/// # Claims
///
/// In `rats-rs`, the concept of `Claims` aligns with that in the Remote ATtestation ProcedureS (RATS), embodying a collection of assertions conveying the authenticity and properties of an attestation report. These `Claims` serve as a critical bridge in establishing trust between an attester and a verifier.
///
/// The `Claims` within `rats-rs` is designed to accommodate both **Built-in Claims** and **Custom Claims**:
///
/// - **Built-in Claims**: These are predefined claims that come bundled with `rats-rs`, divided into two categories:
///   - **Common Built-in Claims**: Universally applicable across all TEE types, these include fundamental assertions like:
///     - `BUILT_IN_CLAIM_COMMON_TEE_TYPE`: the type string of TEE.
///     - `BUILT_IN_CLAIM_COMMON_EVIDENCE`: raw evidence data struct generated by attester.
///   - **TEE-specific Built-in Claims**: These claims are derived specifically from the evidence corresponding to each TEE type, which reflect the distinct attributes of their respective TEE instances. Verifiers can rely on examining these fields to ascertain the trustworthiness of the attester's TEE. Their definitions can be located in the respective source files: 
///     - SGX-DCAP: `rats-rs/src/tee/sgx_dcap/claims.rs`
///     - TDX: `rats-rs/src/tee/tdx/claims.rs`
///
/// - **Custom Claims**: Offered as a flexible mechanism for the attester's TEE to include any additional information during the attestation process. Unlike built-in claims that are generated and endorsed by the hardware TEE with cryptographic guarantees, custom claims are endorsed by the attester's TEE instance itself. Hence, the veracity of custom claims inherently relies on the trust established through the verification of built-in claims, underscoring the importance of a secure chain of trust from the hardware up to the application layer.
pub type Claims = IndexMap<String, Vec<u8>>;

/* Common built-in claims */
pub const BUILT_IN_CLAIM_COMMON_EVIDENCE: &'static str = "common_evidence";
pub const BUILT_IN_CLAIM_COMMON_TEE_TYPE: &'static str = "common_tee_type";
