// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters;
using System.Runtime.Serialization.Formatters.Binary;
using System.Runtime.CompilerServices;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;

// The values represent platforms where there was change in the serialization for one or more types.
public enum TargetFrameworkMoniker
{
    netfx461,
    netfx471,
    netfx472,
    netfx472_3260,
    netcoreapp20,
    netcoreapp21,
    netcoreapp30
}

class GitHub_25468
{
    public readonly struct TypeSerializableValue
    {
        public readonly string Base64Blob;

        // This is the minimum version, when the blob changed.
        public readonly TargetFrameworkMoniker Platform;

        public TypeSerializableValue(string base64Blob, TargetFrameworkMoniker platform)
        {
            Base64Blob = base64Blob;
            Platform = platform;
        }
    }

    public static object[] GetObject()
    {
        // System.Drawing.Common
        // In UAP System.Drawing.Common is a not supported assembly, so this types throw PNSE.
        // Since this code paths use GDI+ we need to validate it is available
        // if (!PlatformDetection.IsUap && PlatformDetection.IsDrawingSupported)
        const string InlineJpegImage = "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";
        byte[] rawInlineImageBytes = Convert.FromBase64String(InlineJpegImage);
        // Stream is disposed by the save operation on Bitmap.
        MemoryStream ms = new MemoryStream(rawInlineImageBytes);
        // Bitmap
        Bitmap bitmapImage = new Bitmap(ms);
        return new object[] { bitmapImage, new TypeSerializableValue[] { new TypeSerializableValue("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", TargetFrameworkMoniker.netcoreapp30), new TypeSerializableValue("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", TargetFrameworkMoniker.netfx461) } };
    }

#pragma warning disable SYSLIB0011 // BinaryFormatter is obsolete
    public static int Roundtrip()
    {
        int returnVal = 100;
        object[] obj;
        try
        {
            obj = GetObject();
        }
        catch (Exception e)
        {
            // If this fires then this is probably not supported on this platform.
            Console.WriteLine("Not supported" + e.Message);
            return returnVal;
        }
        var s = new MemoryStream();
        var f = new BinaryFormatter();

        Console.WriteLine("Serializing " + obj[0].GetType().ToString());
        Bitmap original = (Bitmap)obj[0];
        Console.WriteLine("  {" + original.Width + "," + original.Height + "," + original.Flags + "," +
                          original.HorizontalResolution + "," + original.VerticalResolution + "}");
        f.Serialize(s, obj[0]);
        s.Position = 0;
        Console.WriteLine("Deserializing " + obj[0].GetType().ToString());
        object clone = f.Deserialize(s);
        Bitmap other = (Bitmap)clone;
        if (original == null || other == null)
        {
            returnVal = -1;
        }
        if ((original.Width != other.Width) || (original.Height != other.Height) || (original.Flags != other.Flags) ||
            (original.HorizontalResolution != other.HorizontalResolution) ||
            (original.VerticalResolution != other.VerticalResolution))
        {
            Console.WriteLine("Failed:");
            Console.WriteLine("  {" + other.Width + "," + other.Height + "," + other.Flags + "," +
                               other.HorizontalResolution + "," + other.VerticalResolution + "}");
            returnVal = -1;
        }

        return returnVal;
    }
#pragma warning restore SYSLIB0011

    public static int Main()
    {
        try
        {
            return Roundtrip();
        }
        catch (Exception e)
        {
            Console.WriteLine("Failed with exception " + e.Message);
            return -1;
        }
        return 100;
    }
}
