﻿/*
using System;
using System.Collections;
using System.IO;
using System.Text;
using System.Xml;
using Microsoft.Build.Framework;
using Microsoft.Build.Utilities;
using Microsoft.Build.Tasks;


#nullable disable

namespace Microsoft.Build.UnitTests
{
    [TestFixture]
    sealed public class ResolveVCProjectOutput_Tests
    {
        internal void VerifyVCProjectOutputs(string vcProjectContents, bool useOverride, string configuration,
            string expectedAssemblyOutput, string expectedImportLibOutput)
        {
            ObjectModelHelpers.DeleteTempProjectDirectory();

            ResolveVCProjectOutput resolveVCProjectOutput = new ResolveVCProjectOutput();
            resolveVCProjectOutput.BuildEngine = new MockEngine();

            ObjectModelHelpers.CreateFileInTempProjectDirectory("cppproj.vcproj", vcProjectContents);

            if (useOverride)
            {
                string overrideContents = @"<?xml version=`1.0`?>
                <VisualStudioPropertySheet 
	                ProjectType=`Visual C++` 
	                Version=`8.00` 
	                Name=`VCOverride` OutputDirectory=`Outputs\$(ConfigurationName)`
	                >
                </VisualStudioPropertySheet>";

                ObjectModelHelpers.CreateFileInTempProjectDirectory("VCOverride.vsprops", overrideContents);

                resolveVCProjectOutput.Override = new TaskItem(Path.Combine(ObjectModelHelpers.TempProjectDir, "VCOverride.vsprops"));
            }

            resolveVCProjectOutput.ProjectReferences = new TaskItem[] { new TaskItem(Path.Combine(ObjectModelHelpers.TempProjectDir, "cppproj.vcproj")) };
            resolveVCProjectOutput.Configuration = configuration;

            Assertion.AssertEquals(true, resolveVCProjectOutput.Execute());

            if (expectedAssemblyOutput != null)
            {
                Assertion.AssertEquals(1, resolveVCProjectOutput.ResolvedOutputPaths.GetLength(0));

                Assertion.Assert(resolveVCProjectOutput.ResolvedOutputPaths[0].ItemSpec,
                    string.Compare(Path.Combine(ObjectModelHelpers.TempProjectDir, expectedAssemblyOutput),
                    resolveVCProjectOutput.ResolvedOutputPaths[0].ItemSpec, StringComparison.OrdinalIgnoreCase) == 0);
            }
            else
            {
                Assertion.AssertEquals(0, resolveVCProjectOutput.ResolvedOutputPaths.GetLength(0));
            }

            if (expectedImportLibOutput != null)
            {
                Assertion.AssertEquals(1, resolveVCProjectOutput.ResolvedImportLibraryPaths.GetLength(0));

                Assertion.Assert(resolveVCProjectOutput.ResolvedImportLibraryPaths[0].ItemSpec,
                    string.Compare(Path.Combine(ObjectModelHelpers.TempProjectDir, expectedImportLibOutput),
                    resolveVCProjectOutput.ResolvedImportLibraryPaths[0].ItemSpec, StringComparison.OrdinalIgnoreCase) == 0);
            }
            else
            {
                Assertion.AssertEquals(0, resolveVCProjectOutput.ResolvedImportLibraryPaths.GetLength(0));
            }
        }

        [Test]
        public void OverrideOutputDirectory()
        {
            string vcProjectContents =  
            #region VC project contents
            @"<?xml version=`1.0` encoding=`Windows-1252`?>
                <VisualStudioProject
	                ProjectType=`Visual C++`
	                Version=`8.00`
	                Name=`CppProj`
	                ProjectGUID=`{AF346FA3-1FCD-4EA6-8859-B934B754B5F7}`
	                RootNamespace=`CppProj`
	                SccProjectName=`SAK`
	                SccAuxPath=`SAK`
	                SccLocalPath=`SAK`
	                SccProvider=`SAK`
	                Keyword=`ManagedCProj`
	                >
	                <Platforms>
		                <Platform
			                Name=`Win32`
		                />
	                </Platforms>
	                <ToolFiles>
	                </ToolFiles>
	                <Configurations>
		                <Configuration
			                Name=`Debug|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                ConfigurationType=`2`
			                CharacterSet=`1`
			                ManagedExtensions=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                Optimization=`0`
				                PreprocessorDefinitions=`WIN32;_DEBUG`
				                RuntimeLibrary=`3`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                DebugInformationFormat=`3`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLinkerTool`
				                AdditionalDependencies=`$(NoInherit)`
				                LinkIncremental=`2`
				                GenerateDebugInformation=`true`
				                AssemblyDebug=`1`
				                TargetMachine=`1`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCManifestTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCAppVerifierTool`
			                />
			                <Tool
				                Name=`VCWebDeploymentTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
		                <Configuration
			                Name=`Release|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                ConfigurationType=`2`
			                CharacterSet=`1`
			                ManagedExtensions=`1`
			                WholeProgramOptimization=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                PreprocessorDefinitions=`WIN32;NDEBUG`
				                RuntimeLibrary=`2`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                DebugInformationFormat=`3`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLinkerTool`
				                AdditionalDependencies=`$(NoInherit)`
				                LinkIncremental=`1`
				                GenerateDebugInformation=`true`
				                TargetMachine=`1`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCManifestTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCAppVerifierTool`
			                />
			                <Tool
				                Name=`VCWebDeploymentTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
	                </Configurations>
	                <References>
		                <AssemblyReference
			                RelativePath=`System.dll`
			                AssemblyName=`System, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=MSIL`
		                />
		                <AssemblyReference
			                RelativePath=`System.Data.dll`
			                AssemblyName=`System.Data, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=x86`
		                />
		                <AssemblyReference
			                RelativePath=`System.XML.dll`
			                AssemblyName=`System.Xml, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=MSIL`
		                />
	                </References>
	                <Files>
		                <Filter
			                Name=`Source Files`
			                Filter=`cpp;c;cc;cxx;def;odl;idl;hpj;bat;asm;asmx`
			                UniqueIdentifier=`{4FC737F1-C7A5-4376-A066-2A32D752A2FF}`
			                >
			                <File
				                RelativePath=`.\AssemblyInfo.cpp`
				                >
			                </File>
			                <File
				                RelativePath=`.\CppProj.cpp`
				                >
			                </File>
			                <File
				                RelativePath=`.\Stdafx.cpp`
				                >
				                <FileConfiguration
					                Name=`Debug|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
				                <FileConfiguration
					                Name=`Release|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
			                </File>
		                </Filter>
		                <Filter
			                Name=`Header Files`
			                Filter=`h;hpp;hxx;hm;inl;inc;xsd`
			                UniqueIdentifier=`{93995380-89BD-4b04-88EB-625FBE52EBFB}`
			                >
			                <File
				                RelativePath=`.\CppProj.h`
				                >
			                </File>
			                <File
				                RelativePath=`.\resource.h`
				                >
			                </File>
			                <File
				                RelativePath=`.\Stdafx.h`
				                >
			                </File>
		                </Filter>
		                <Filter
			                Name=`Resource Files`
			                Filter=`rc;ico;cur;bmp;dlg;rc2;rct;bin;rgs;gif;jpg;jpeg;jpe;resx;tiff;tif;png;wav`
			                UniqueIdentifier=`{67DA6AB6-F800-4c08-8B7A-83BB121AAD01}`
			                >
			                <File
				                RelativePath=`.\app.ico`
				                >
			                </File>
			                <File
				                RelativePath=`.\app.rc`
				                >
			                </File>
		                </Filter>
		                <File
			                RelativePath=`.\ReadMe.txt`
			                >
		                </File>
		                <File
			                RelativePath=`.\VCOverride.vsprops`
			                >
		                </File>
	                </Files>
	                <Globals>
	                </Globals>
                </VisualStudioProject>";
            #endregion

            VerifyVCProjectOutputs(vcProjectContents, true, "Debug|Win32", 
                @"Outputs\Debug\CppProj.dll", null);
        }

        [Test]
        public void ResolveNativeDllOutputs()
        {
            string vcProjectContents =
            #region VC project contents
            @"<?xml version=`1.0` encoding=`Windows-1252`?>
                <VisualStudioProject
	                ProjectType=`Visual C++`
	                Version=`8.00`
	                Name=`NativeDll`
	                ProjectGUID=`{E2F5F366-6B25-4897-BC2A-AC57A8693F14}`
	                RootNamespace=`NativeDll`
	                Keyword=`Win32Proj`
	                >
	                <Platforms>
		                <Platform
			                Name=`Win32`
		                />
	                </Platforms>
	                <ToolFiles>
	                </ToolFiles>
	                <Configurations>
		                <Configuration
			                Name=`Debug|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                ConfigurationType=`2`
			                CharacterSet=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                Optimization=`0`
				                PreprocessorDefinitions=`WIN32;_DEBUG;_WINDOWS;_USRDLL;NATIVEDLL_EXPORTS`
				                MinimalRebuild=`true`
				                BasicRuntimeChecks=`3`
				                RuntimeLibrary=`3`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                Detect64BitPortabilityProblems=`true`
				                DebugInformationFormat=`4`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLinkerTool`
				                LinkIncremental=`2`
				                GenerateDebugInformation=`true`
				                SubSystem=`2`
				                TargetMachine=`1`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCManifestTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCAppVerifierTool`
			                />
			                <Tool
				                Name=`VCWebDeploymentTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
		                <Configuration
			                Name=`Release|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                ConfigurationType=`2`
			                CharacterSet=`1`
			                WholeProgramOptimization=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                PreprocessorDefinitions=`WIN32;NDEBUG;_WINDOWS;_USRDLL;NATIVEDLL_EXPORTS`
				                RuntimeLibrary=`2`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                Detect64BitPortabilityProblems=`true`
				                DebugInformationFormat=`3`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLinkerTool`
				                IgnoreImportLibrary=`true`
				                LinkIncremental=`1`
				                GenerateDebugInformation=`true`
				                SubSystem=`2`
				                OptimizeReferences=`2`
				                EnableCOMDATFolding=`2`
				                TargetMachine=`1`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCManifestTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCAppVerifierTool`
			                />
			                <Tool
				                Name=`VCWebDeploymentTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
	                </Configurations>
	                <References>
	                </References>
	                <Files>
		                <Filter
			                Name=`Source Files`
			                Filter=`cpp;c;cc;cxx;def;odl;idl;hpj;bat;asm;asmx`
			                UniqueIdentifier=`{4FC737F1-C7A5-4376-A066-2A32D752A2FF}`
			                >
			                <File
				                RelativePath=`.\NativeDll.cpp`
				                >
			                </File>
			                <File
				                RelativePath=`.\stdafx.cpp`
				                >
				                <FileConfiguration
					                Name=`Debug|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
				                <FileConfiguration
					                Name=`Release|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
			                </File>
		                </Filter>
		                <Filter
			                Name=`Header Files`
			                Filter=`h;hpp;hxx;hm;inl;inc;xsd`
			                UniqueIdentifier=`{93995380-89BD-4b04-88EB-625FBE52EBFB}`
			                >
			                <File
				                RelativePath=`.\stdafx.h`
				                >
			                </File>
		                </Filter>
		                <Filter
			                Name=`Resource Files`
			                Filter=`rc;ico;cur;bmp;dlg;rc2;rct;bin;rgs;gif;jpg;jpeg;jpe;resx;tiff;tif;png;wav`
			                UniqueIdentifier=`{67DA6AB6-F800-4c08-8B7A-83BB121AAD01}`
			                >
		                </Filter>
		                <File
			                RelativePath=`.\ReadMe.txt`
			                >
		                </File>
	                </Files>
	                <Globals>
	                </Globals>
                </VisualStudioProject>";
            #endregion

            // Standard import lib for VC native dll
            VerifyVCProjectOutputs(vcProjectContents, false, "Debug|Win32",
                null, @"Debug\NativeDll.lib");

            // No import lib because IgnoreImportLibrary is set to Yes for the Release configuration
            VerifyVCProjectOutputs(vcProjectContents, false, "Release|Win32",
                null, null);
        }

        [Test]
        public void ResolveStaticLibOutputs()
        {
            string vcProjectContents =
            #region VC project contents
             @"<?xml version=`1.0` encoding=`Windows-1252`?>
                <VisualStudioProject
	                ProjectType=`Visual C++`
	                Version=`8.00`
	                Name=`StaticLib`
	                ProjectGUID=`{E5127FDE-D9BE-4408-AC5E-2FF43B05FC3F}`
	                RootNamespace=`StaticLib`
	                Keyword=`Win32Proj`
	                >
	                <Platforms>
		                <Platform
			                Name=`Win32`
		                />
	                </Platforms>
	                <ToolFiles>
	                </ToolFiles>
	                <Configurations>
		                <Configuration
			                Name=`Debug|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                ConfigurationType=`4`
			                CharacterSet=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                Optimization=`0`
				                PreprocessorDefinitions=`WIN32;_DEBUG;_LIB`
				                MinimalRebuild=`true`
				                BasicRuntimeChecks=`3`
				                RuntimeLibrary=`3`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                Detect64BitPortabilityProblems=`true`
				                DebugInformationFormat=`4`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLibrarianTool`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
		                <Configuration
			                Name=`Release|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                ConfigurationType=`4`
			                CharacterSet=`1`
			                WholeProgramOptimization=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                PreprocessorDefinitions=`WIN32;NDEBUG;_LIB`
				                RuntimeLibrary=`2`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                Detect64BitPortabilityProblems=`true`
				                DebugInformationFormat=`3`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLibrarianTool`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
	                </Configurations>
	                <References>
	                </References>
	                <Files>
		                <Filter
			                Name=`Source Files`
			                Filter=`cpp;c;cc;cxx;def;odl;idl;hpj;bat;asm;asmx`
			                UniqueIdentifier=`{4FC737F1-C7A5-4376-A066-2A32D752A2FF}`
			                >
			                <File
				                RelativePath=`.\stdafx.cpp`
				                >
				                <FileConfiguration
					                Name=`Debug|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
				                <FileConfiguration
					                Name=`Release|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
			                </File>
		                </Filter>
		                <Filter
			                Name=`Header Files`
			                Filter=`h;hpp;hxx;hm;inl;inc;xsd`
			                UniqueIdentifier=`{93995380-89BD-4b04-88EB-625FBE52EBFB}`
			                >
			                <File
				                RelativePath=`.\stdafx.h`
				                >
			                </File>
		                </Filter>
		                <Filter
			                Name=`Resource Files`
			                Filter=`rc;ico;cur;bmp;dlg;rc2;rct;bin;rgs;gif;jpg;jpeg;jpe;resx;tiff;tif;png;wav`
			                UniqueIdentifier=`{67DA6AB6-F800-4c08-8B7A-83BB121AAD01}`
			                >
		                </Filter>
		                <File
			                RelativePath=`.\ReadMe.txt`
			                >
		                </File>
	                </Files>
	                <Globals>
	                </Globals>
                </VisualStudioProject>";
            #endregion

            // Primary output for VC native static lib is an import library
            VerifyVCProjectOutputs(vcProjectContents, false, "Debug|Win32",
                null, @"Debug\StaticLib.lib");

            VerifyVCProjectOutputs(vcProjectContents, true, "Debug|Win32",
                null, @"Outputs\Debug\StaticLib.lib");
        }

        /// <summary>
        /// Regress DDB #81422
        /// </summary>
        [Test]
        public void ResolveStaticLibOutputsWherePathIsQuoted()
        {
            string vcProjectContents =
            #region VC project contents
 @"<?xml version=""1.0"" encoding=""Windows-1252""?>
<VisualStudioProject
	ProjectType=""Visual C++""
	Version=""9.00""
	Name=""w32staticlib""
	ProjectGUID=""{E611A0C2-7D43-4E03-B0E6-2E91C3E870A5}""
	RootNamespace=""w32staticlib""
	Keyword=""Win32Proj""
	TargetFrameworkVersion=""196613""
	>
	<Platforms>
		<Platform
			Name=""Win32""
		/>
	</Platforms>
	<ToolFiles>
	</ToolFiles>
	<Configurations>
		<Configuration
			Name=""Debug|Win32""
			OutputDirectory=""$(SolutionDir)$(ConfigurationName)""
			IntermediateDirectory=""$(ConfigurationName)""
			ConfigurationType=""4""
			CharacterSet=""1""
			>
			<Tool
				Name=""VCPreBuildEventTool""
			/>
			<Tool
				Name=""VCCustomBuildTool""
			/>
			<Tool
				Name=""VCXMLDataGeneratorTool""
			/>
			<Tool
				Name=""VCWebServiceProxyGeneratorTool""
			/>
			<Tool
				Name=""VCMIDLTool""
			/>
			<Tool
				Name=""VCCLCompilerTool""
				Optimization=""0""
				PreprocessorDefinitions=""WIN32;_DEBUG;_LIB""
				MinimalRebuild=""true""
				BasicRuntimeChecks=""3""
				RuntimeLibrary=""3""
				UsePrecompiledHeader=""2""
				WarningLevel=""3""
				Detect64BitPortabilityProblems=""true""
				DebugInformationFormat=""4""
			/>
			<Tool
				Name=""VCManagedResourceCompilerTool""
			/>
			<Tool
				Name=""VCResourceCompilerTool""
			/>
			<Tool
				Name=""VCPreLinkEventTool""
			/>
			<Tool
				Name=""VCLibrarianTool""
				OutputFile=""&quot;$(OutDir)\$(ProjectName).lib&quot;""
			/>
			<Tool
				Name=""VCALinkTool""
			/>
			<Tool
				Name=""VCXDCMakeTool""
			/>
			<Tool
				Name=""VCBscMakeTool""
			/>
			<Tool
				Name=""VCFxCopTool""
			/>
			<Tool
				Name=""VCPostBuildEventTool""
			/>
		</Configuration>
		<Configuration
			Name=""Release|Win32""
			OutputDirectory=""$(SolutionDir)$(ConfigurationName)""
			IntermediateDirectory=""$(ConfigurationName)""
			ConfigurationType=""4""
			CharacterSet=""1""
			WholeProgramOptimization=""1""
			>
			<Tool
				Name=""VCPreBuildEventTool""
			/>
			<Tool
				Name=""VCCustomBuildTool""
			/>
			<Tool
				Name=""VCXMLDataGeneratorTool""
			/>
			<Tool
				Name=""VCWebServiceProxyGeneratorTool""
			/>
			<Tool
				Name=""VCMIDLTool""
			/>
			<Tool
				Name=""VCCLCompilerTool""
				Optimization=""2""
				EnableIntrinsicFunctions=""true""
				PreprocessorDefinitions=""WIN32;NDEBUG;_LIB""
				RuntimeLibrary=""2""
				EnableFunctionLevelLinking=""true""
				UsePrecompiledHeader=""2""
				WarningLevel=""3""
				Detect64BitPortabilityProblems=""true""
				DebugInformationFormat=""3""
			/>
			<Tool
				Name=""VCManagedResourceCompilerTool""
			/>
			<Tool
				Name=""VCResourceCompilerTool""
			/>
			<Tool
				Name=""VCPreLinkEventTool""
			/>
			<Tool
				Name=""VCLibrarianTool""
			/>
			<Tool
				Name=""VCALinkTool""
			/>
			<Tool
				Name=""VCXDCMakeTool""
			/>
			<Tool
				Name=""VCBscMakeTool""
			/>
			<Tool
				Name=""VCFxCopTool""
			/>
			<Tool
				Name=""VCPostBuildEventTool""
			/>
		</Configuration>
	</Configurations>
	<References>
	</References>
	<Files>
		<Filter
			Name=""Source Files""
			Filter=""cpp;c;cc;cxx;def;odl;idl;hpj;bat;asm;asmx""
			UniqueIdentifier=""{4FC737F1-C7A5-4376-A066-2A32D752A2FF}""
			>
			<File
				RelativePath="".\stdafx.cpp""
				>
				<FileConfiguration
					Name=""Debug|Win32""
					>
					<Tool
						Name=""VCCLCompilerTool""
						UsePrecompiledHeader=""1""
					/>
				</FileConfiguration>
				<FileConfiguration
					Name=""Release|Win32""
					>
					<Tool
						Name=""VCCLCompilerTool""
						UsePrecompiledHeader=""1""
					/>
				</FileConfiguration>
			</File>
		</Filter>
		<Filter
			Name=""Header Files""
			Filter=""h;hpp;hxx;hm;inl;inc;xsd""
			UniqueIdentifier=""{93995380-89BD-4b04-88EB-625FBE52EBFB}""
			>
			<File
				RelativePath="".\stdafx.h""
				>
			</File>
			<File
				RelativePath="".\targetver.h""
				>
			</File>
		</Filter>
		<Filter
			Name=""Resource Files""
			Filter=""rc;ico;cur;bmp;dlg;rc2;rct;bin;rgs;gif;jpg;jpeg;jpe;resx;tiff;tif;png;wav""
			UniqueIdentifier=""{67DA6AB6-F800-4c08-8B7A-83BB121AAD01}""
			>
		</Filter>
		<File
			RelativePath="".\ReadMe.txt""
			>
		</File>
	</Files>
	<Globals>
	</Globals>
</VisualStudioProject>
";
            #endregion

            VerifyVCProjectOutputs(vcProjectContents, false, "Debug|Win32",
                null, @"Debug\w32staticlib.lib");
        }

        [Test]
        public void ResolveNativeExeOutputs()
        {
            string vcProjectContents =
            #region VC project contents
             @"<?xml version=`1.0` encoding=`Windows-1252`?>
                <VisualStudioProject
	                ProjectType=`Visual C++`
	                Version=`8.00`
	                Name=`NativeExe`
	                ProjectGUID=`{7A309709-D500-4A2C-B3DC-F9FEFA0B7D08}`
	                RootNamespace=`NativeExe`
	                Keyword=`Win32Proj`
	                >
	                <Platforms>
		                <Platform
			                Name=`Win32`
		                />
	                </Platforms>
	                <ToolFiles>
	                </ToolFiles>
	                <Configurations>
		                <Configuration
			                Name=`Debug|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                ConfigurationType=`1`
			                CharacterSet=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                Optimization=`0`
				                PreprocessorDefinitions=`WIN32;_DEBUG;_WINDOWS`
				                MinimalRebuild=`true`
				                BasicRuntimeChecks=`3`
				                RuntimeLibrary=`3`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                Detect64BitPortabilityProblems=`true`
				                DebugInformationFormat=`4`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLinkerTool`
				                LinkIncremental=`2`
				                GenerateDebugInformation=`true`
				                SubSystem=`2`
				                TargetMachine=`1`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCManifestTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCAppVerifierTool`
			                />
			                <Tool
				                Name=`VCWebDeploymentTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
		                <Configuration
			                Name=`Release|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                CharacterSet=`1`
			                WholeProgramOptimization=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                PreprocessorDefinitions=`WIN32;NDEBUG;_WINDOWS`
				                RuntimeLibrary=`2`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                Detect64BitPortabilityProblems=`true`
				                DebugInformationFormat=`3`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLinkerTool`
				                LinkIncremental=`1`
				                GenerateDebugInformation=`true`
				                SubSystem=`2`
				                OptimizeReferences=`2`
				                EnableCOMDATFolding=`2`
				                TargetMachine=`1`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCManifestTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCAppVerifierTool`
			                />
			                <Tool
				                Name=`VCWebDeploymentTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
	                </Configurations>
	                <References>
	                </References>
	                <Files>
		                <Filter
			                Name=`Source Files`
			                Filter=`cpp;c;cc;cxx;def;odl;idl;hpj;bat;asm;asmx`
			                UniqueIdentifier=`{4FC737F1-C7A5-4376-A066-2A32D752A2FF}`
			                >
			                <File
				                RelativePath=`.\NativeExe.cpp`
				                >
			                </File>
			                <File
				                RelativePath=`.\stdafx.cpp`
				                >
				                <FileConfiguration
					                Name=`Debug|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
				                <FileConfiguration
					                Name=`Release|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
			                </File>
		                </Filter>
		                <Filter
			                Name=`Header Files`
			                Filter=`h;hpp;hxx;hm;inl;inc;xsd`
			                UniqueIdentifier=`{93995380-89BD-4b04-88EB-625FBE52EBFB}`
			                >
			                <File
				                RelativePath=`.\NativeExe.h`
				                >
			                </File>
			                <File
				                RelativePath=`.\Resource.h`
				                >
			                </File>
			                <File
				                RelativePath=`.\stdafx.h`
				                >
			                </File>
		                </Filter>
		                <Filter
			                Name=`Resource Files`
			                Filter=`rc;ico;cur;bmp;dlg;rc2;rct;bin;rgs;gif;jpg;jpeg;jpe;resx;tiff;tif;png;wav`
			                UniqueIdentifier=`{67DA6AB6-F800-4c08-8B7A-83BB121AAD01}`
			                >
			                <File
				                RelativePath=`.\NativeExe.ico`
				                >
			                </File>
			                <File
				                RelativePath=`.\NativeExe.rc`
				                >
			                </File>
			                <File
				                RelativePath=`.\small.ico`
				                >
			                </File>
		                </Filter>
		                <File
			                RelativePath=`.\ReadMe.txt`
			                >
		                </File>
	                </Files>
	                <Globals>
	                </Globals>
                </VisualStudioProject>";
            #endregion

            // No consumable outputs from an Exe project
            VerifyVCProjectOutputs(vcProjectContents, false, "Debug|Win32",
                null, null);

            // Release has no values for ConfigurationType and ManagedExtensions, which should default to a native .exe
            VerifyVCProjectOutputs(vcProjectContents, false, "Release|Win32",
                null, null);
        }

        [Test]
        public void ResolveManagedDllOutputs()
        {
            string vcProjectContents =
            #region VC project contents
             @"<?xml version=`1.0` encoding=`Windows-1252`?>
                <VisualStudioProject
	                ProjectType=`Visual C++`
	                Version=`8.00`
	                Name=`ManagedDll`
	                ProjectGUID=`{8E37FC7D-4540-4778-8E8E-DE1226B4BA14}`
	                RootNamespace=`ManagedDll`
	                Keyword=`ManagedCProj`
	                >
	                <Platforms>
		                <Platform
			                Name=`Win32`
		                />
	                </Platforms>
	                <ToolFiles>
	                </ToolFiles>
	                <Configurations>
		                <Configuration
			                Name=`Debug|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                ConfigurationType=`2`
			                CharacterSet=`1`
			                ManagedExtensions=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                Optimization=`0`
				                PreprocessorDefinitions=`WIN32;_DEBUG`
				                RuntimeLibrary=`3`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                DebugInformationFormat=`3`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLinkerTool`
				                AdditionalDependencies=`$(NoInherit)`
				                LinkIncremental=`2`
				                ModuleDefinitionFile=`ManagedDef.def`
				                GenerateDebugInformation=`true`
				                AssemblyDebug=`1`
				                TargetMachine=`1`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCManifestTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCAppVerifierTool`
			                />
			                <Tool
				                Name=`VCWebDeploymentTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
		                <Configuration
			                Name=`Release|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                ConfigurationType=`2`
			                CharacterSet=`1`
			                ManagedExtensions=`1`
			                WholeProgramOptimization=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                PreprocessorDefinitions=`WIN32;NDEBUG`
				                RuntimeLibrary=`2`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                DebugInformationFormat=`3`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLinkerTool`
				                IgnoreImportLibrary=`false`
				                AdditionalDependencies=`$(NoInherit)`
				                LinkIncremental=`1`
				                ModuleDefinitionFile=`ManagedDef.def`
				                GenerateDebugInformation=`true`
				                TargetMachine=`1`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCManifestTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCAppVerifierTool`
			                />
			                <Tool
				                Name=`VCWebDeploymentTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
	                </Configurations>
	                <References>
		                <AssemblyReference
			                RelativePath=`System.dll`
			                AssemblyName=`System, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=MSIL`
		                />
		                <AssemblyReference
			                RelativePath=`System.Data.dll`
			                AssemblyName=`System.Data, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=x86`
		                />
		                <AssemblyReference
			                RelativePath=`System.Xml.dll`
			                AssemblyName=`System.Xml, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=MSIL`
		                />
	                </References>
	                <Files>
		                <Filter
			                Name=`Source Files`
			                Filter=`cpp;c;cc;cxx;def;odl;idl;hpj;bat;asm;asmx`
			                UniqueIdentifier=`{4FC737F1-C7A5-4376-A066-2A32D752A2FF}`
			                >
			                <File
				                RelativePath=`.\AssemblyInfo.cpp`
				                >
			                </File>
			                <File
				                RelativePath=`.\ManagedDef.def`
				                >
			                </File>
			                <File
				                RelativePath=`.\ManagedDll.cpp`
				                >
			                </File>
			                <File
				                RelativePath=`.\Stdafx.cpp`
				                >
				                <FileConfiguration
					                Name=`Debug|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
				                <FileConfiguration
					                Name=`Release|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
			                </File>
		                </Filter>
		                <Filter
			                Name=`Header Files`
			                Filter=`h;hpp;hxx;hm;inl;inc;xsd`
			                UniqueIdentifier=`{93995380-89BD-4b04-88EB-625FBE52EBFB}`
			                >
			                <File
				                RelativePath=`.\ManagedDll.h`
				                >
			                </File>
			                <File
				                RelativePath=`.\resource.h`
				                >
			                </File>
			                <File
				                RelativePath=`.\Stdafx.h`
				                >
			                </File>
		                </Filter>
		                <Filter
			                Name=`Resource Files`
			                Filter=`rc;ico;cur;bmp;dlg;rc2;rct;bin;rgs;gif;jpg;jpeg;jpe;resx;tiff;tif;png;wav`
			                UniqueIdentifier=`{67DA6AB6-F800-4c08-8B7A-83BB121AAD01}`
			                >
			                <File
				                RelativePath=`.\app.ico`
				                >
			                </File>
			                <File
				                RelativePath=`.\app.rc`
				                >
			                </File>
		                </Filter>
		                <File
			                RelativePath=`.\ReadMe.txt`
			                >
		                </File>
	                </Files>
	                <Globals>
	                </Globals>
                </VisualStudioProject>
                ";
            #endregion

            // For normal managed dll, only primary output is consumable as an assembly ref
            VerifyVCProjectOutputs(vcProjectContents, false, "Debug|Win32",
                @"Debug\ManagedDll.dll", null);

            // The Release configuration is changed to generate an import lib (yes, it's possible even for a managed project)
            // It's enough to set IgnoreImportLibrary to No to achieve this
            VerifyVCProjectOutputs(vcProjectContents, false, "Release|Win32",
                @"Release\ManagedDll.dll", @"Release\ManagedDll.lib");
        }

        [Test]
        public void ResolveManagedExeOutputs()
        {
            string vcProjectContents =
            #region VC project contents
             @"<?xml version=`1.0` encoding=`Windows-1252`?>
                <VisualStudioProject
	                ProjectType=`Visual C++`
	                Version=`8.00`
	                Name=`ManagedExe`
	                ProjectGUID=`{87DA3F70-B414-4868-817A-5781824F88EF}`
	                RootNamespace=`ManagedExe`
	                Keyword=`ManagedCProj`
	                >
	                <Platforms>
		                <Platform
			                Name=`Win32`
		                />
	                </Platforms>
	                <ToolFiles>
	                </ToolFiles>
	                <Configurations>
		                <Configuration
			                Name=`Debug|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                ConfigurationType=`1`
			                CharacterSet=`1`
			                ManagedExtensions=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                Optimization=`0`
				                PreprocessorDefinitions=`WIN32;_DEBUG`
				                RuntimeLibrary=`3`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                DebugInformationFormat=`3`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLinkerTool`
				                AdditionalDependencies=`$(NoInherit)`
				                LinkIncremental=`2`
				                ModuleDefinitionFile=`ManagedExe.def`
				                GenerateDebugInformation=`true`
				                AssemblyDebug=`1`
				                TargetMachine=`1`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCManifestTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCAppVerifierTool`
			                />
			                <Tool
				                Name=`VCWebDeploymentTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
		                <Configuration
			                Name=`Release|Win32`
			                OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                IntermediateDirectory=`$(ConfigurationName)`
			                ConfigurationType=`1`
			                CharacterSet=`1`
			                ManagedExtensions=`1`
			                WholeProgramOptimization=`1`
			                >
			                <Tool
				                Name=`VCPreBuildEventTool`
			                />
			                <Tool
				                Name=`VCCustomBuildTool`
			                />
			                <Tool
				                Name=`VCXMLDataGeneratorTool`
			                />
			                <Tool
				                Name=`VCWebServiceProxyGeneratorTool`
			                />
			                <Tool
				                Name=`VCMIDLTool`
			                />
			                <Tool
				                Name=`VCCLCompilerTool`
				                PreprocessorDefinitions=`WIN32;NDEBUG`
				                RuntimeLibrary=`2`
				                UsePrecompiledHeader=`2`
				                WarningLevel=`3`
				                DebugInformationFormat=`3`
			                />
			                <Tool
				                Name=`VCManagedResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCResourceCompilerTool`
			                />
			                <Tool
				                Name=`VCPreLinkEventTool`
			                />
			                <Tool
				                Name=`VCLinkerTool`
				                IgnoreImportLibrary=`false`
				                AdditionalDependencies=`$(NoInherit)`
				                LinkIncremental=`1`
				                ModuleDefinitionFile=`ManagedExe.def`
				                GenerateDebugInformation=`true`
				                TargetMachine=`1`
			                />
			                <Tool
				                Name=`VCALinkTool`
			                />
			                <Tool
				                Name=`VCManifestTool`
			                />
			                <Tool
				                Name=`VCXDCMakeTool`
			                />
			                <Tool
				                Name=`VCBscMakeTool`
			                />
			                <Tool
				                Name=`VCFxCopTool`
			                />
			                <Tool
				                Name=`VCAppVerifierTool`
			                />
			                <Tool
				                Name=`VCWebDeploymentTool`
			                />
			                <Tool
				                Name=`VCPostBuildEventTool`
			                />
		                </Configuration>
	                </Configurations>
	                <References>
		                <AssemblyReference
			                RelativePath=`System.dll`
			                AssemblyName=`System, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=MSIL`
		                />
		                <AssemblyReference
			                RelativePath=`System.Data.dll`
			                AssemblyName=`System.Data, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=x86`
		                />
		                <AssemblyReference
			                RelativePath=`System.Xml.dll`
			                AssemblyName=`System.Xml, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=MSIL`
		                />
	                </References>
	                <Files>
		                <Filter
			                Name=`Source Files`
			                Filter=`cpp;c;cc;cxx;def;odl;idl;hpj;bat;asm;asmx`
			                UniqueIdentifier=`{4FC737F1-C7A5-4376-A066-2A32D752A2FF}`
			                >
			                <File
				                RelativePath=`.\AssemblyInfo.cpp`
				                >
			                </File>
			                <File
				                RelativePath=`.\ManagedExe.cpp`
				                >
			                </File>
			                <File
				                RelativePath=`.\ManagedExe.def`
				                >
			                </File>
			                <File
				                RelativePath=`.\stdafx.cpp`
				                >
				                <FileConfiguration
					                Name=`Debug|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
				                <FileConfiguration
					                Name=`Release|Win32`
					                >
					                <Tool
						                Name=`VCCLCompilerTool`
						                UsePrecompiledHeader=`1`
					                />
				                </FileConfiguration>
			                </File>
		                </Filter>
		                <Filter
			                Name=`Header Files`
			                Filter=`h;hpp;hxx;hm;inl;inc;xsd`
			                UniqueIdentifier=`{93995380-89BD-4b04-88EB-625FBE52EBFB}`
			                >
			                <File
				                RelativePath=`.\resource.h`
				                >
			                </File>
			                <File
				                RelativePath=`.\stdafx.h`
				                >
			                </File>
		                </Filter>
		                <Filter
			                Name=`Resource Files`
			                Filter=`rc;ico;cur;bmp;dlg;rc2;rct;bin;rgs;gif;jpg;jpeg;jpe;resx;tiff;tif;png;wav`
			                UniqueIdentifier=`{67DA6AB6-F800-4c08-8B7A-83BB121AAD01}`
			                >
			                <File
				                RelativePath=`.\app.ico`
				                >
			                </File>
			                <File
				                RelativePath=`.\app.rc`
				                >
			                </File>
		                </Filter>
		                <File
			                RelativePath=`.\ReadMe.txt`
			                >
		                </File>
	                </Files>
	                <Globals>
	                </Globals>
                </VisualStudioProject>";
            #endregion

            // For normal managed exe, only primary output is consumable as an assembly ref
            VerifyVCProjectOutputs(vcProjectContents, false, "Debug|Win32",
                @"Debug\ManagedExe.exe", null);

            // The Release configuration is changed to generate an import lib (yes, it's possible even for a managed project)
            // It's enough to set IgnoreImportLibrary to No to achieve this
            // However, we should always ignore import libs for .exe projects
            VerifyVCProjectOutputs(vcProjectContents, false, "Release|Win32",
                @"Release\ManagedExe.exe", null);

            // For normal managed exe, only primary output is consumable as an assembly ref
            VerifyVCProjectOutputs(vcProjectContents, true, "Debug|Win32",
                @"Outputs\Debug\ManagedExe.exe", null);
        }

        [Test]
        public void ResolveManagedDllOutputsManagedExtensionsOverridenInFiles()
        {
            string vcProjectContents =
            #region VC project contents
                @"<?xml version=`1.0` encoding=`Windows-1252`?>
                    <VisualStudioProject
	                    ProjectType=`Visual C++`
	                    Version=`9.00`
	                    Name=`ClassLib`
	                    ProjectGUID=`{06F64E5A-423E-4835-A6BC-8A9540536D9A}`
	                    RootNamespace=`ClassLib`
	                    Keyword=`ManagedCProj`
	                    TargetFrameworkVersion=`196613`
	                    >
	                    <Platforms>
		                    <Platform
			                    Name=`Win32`
		                    />
	                    </Platforms>
	                    <ToolFiles>
	                    </ToolFiles>
	                    <Configurations>
		                    <Configuration
			                    Name=`Debug|Win32`
			                    OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                    IntermediateDirectory=`$(ConfigurationName)`
			                    ConfigurationType=`2`
			                    CharacterSet=`1`
			                    ManagedExtensions=`0`
			                    >
			                    <Tool
				                    Name=`VCPreBuildEventTool`
			                    />
			                    <Tool
				                    Name=`VCCustomBuildTool`
			                    />
			                    <Tool
				                    Name=`VCXMLDataGeneratorTool`
			                    />
			                    <Tool
				                    Name=`VCWebServiceProxyGeneratorTool`
			                    />
			                    <Tool
				                    Name=`VCMIDLTool`
			                    />
			                    <Tool
				                    Name=`VCCLCompilerTool`
				                    Optimization=`0`
				                    PreprocessorDefinitions=`WIN32;_DEBUG`
				                    RuntimeLibrary=`3`
				                    UsePrecompiledHeader=`2`
				                    WarningLevel=`3`
				                    DebugInformationFormat=`3`
			                    />
			                    <Tool
				                    Name=`VCManagedResourceCompilerTool`
			                    />
			                    <Tool
				                    Name=`VCResourceCompilerTool`
			                    />
			                    <Tool
				                    Name=`VCPreLinkEventTool`
			                    />
			                    <Tool
				                    Name=`VCLinkerTool`
				                    AdditionalDependencies=`$(NoInherit)`
				                    LinkIncremental=`2`
				                    GenerateDebugInformation=`true`
				                    AssemblyDebug=`1`
				                    TargetMachine=`1`
			                    />
			                    <Tool
				                    Name=`VCALinkTool`
			                    />
			                    <Tool
				                    Name=`VCManifestTool`
			                    />
			                    <Tool
				                    Name=`VCXDCMakeTool`
			                    />
			                    <Tool
				                    Name=`VCBscMakeTool`
			                    />
			                    <Tool
				                    Name=`VCFxCopTool`
			                    />
			                    <Tool
				                    Name=`VCAppVerifierTool`
			                    />
			                    <Tool
				                    Name=`VCWebDeploymentTool`
			                    />
			                    <Tool
				                    Name=`VCPostBuildEventTool`
			                    />
		                    </Configuration>
		                    <Configuration
			                    Name=`Release|Win32`
			                    OutputDirectory=`$(SolutionDir)$(ConfigurationName)`
			                    IntermediateDirectory=`$(ConfigurationName)`
			                    ConfigurationType=`2`
			                    CharacterSet=`1`
			                    ManagedExtensions=`0`
			                    WholeProgramOptimization=`1`
			                    >
			                    <Tool
				                    Name=`VCPreBuildEventTool`
			                    />
			                    <Tool
				                    Name=`VCCustomBuildTool`
			                    />
			                    <Tool
				                    Name=`VCXMLDataGeneratorTool`
			                    />
			                    <Tool
				                    Name=`VCWebServiceProxyGeneratorTool`
			                    />
			                    <Tool
				                    Name=`VCMIDLTool`
			                    />
			                    <Tool
				                    Name=`VCCLCompilerTool`
				                    PreprocessorDefinitions=`WIN32;NDEBUG`
				                    RuntimeLibrary=`2`
				                    UsePrecompiledHeader=`2`
				                    WarningLevel=`3`
				                    DebugInformationFormat=`3`
			                    />
			                    <Tool
				                    Name=`VCManagedResourceCompilerTool`
			                    />
			                    <Tool
				                    Name=`VCResourceCompilerTool`
			                    />
			                    <Tool
				                    Name=`VCPreLinkEventTool`
			                    />
			                    <Tool
				                    Name=`VCLinkerTool`
				                    AdditionalDependencies=`$(NoInherit)`
				                    LinkIncremental=`1`
				                    GenerateDebugInformation=`true`
				                    TargetMachine=`1`
			                    />
			                    <Tool
				                    Name=`VCALinkTool`
			                    />
			                    <Tool
				                    Name=`VCManifestTool`
			                    />
			                    <Tool
				                    Name=`VCXDCMakeTool`
			                    />
			                    <Tool
				                    Name=`VCBscMakeTool`
			                    />
			                    <Tool
				                    Name=`VCFxCopTool`
			                    />
			                    <Tool
				                    Name=`VCAppVerifierTool`
			                    />
			                    <Tool
				                    Name=`VCWebDeploymentTool`
			                    />
			                    <Tool
				                    Name=`VCPostBuildEventTool`
			                    />
		                    </Configuration>
	                    </Configurations>
	                    <References>
		                    <AssemblyReference
			                    RelativePath=`System.dll`
			                    AssemblyName=`System, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=MSIL`
			                    MinFrameworkVersion=`131072`
		                    />
		                    <AssemblyReference
			                    RelativePath=`System.Data.dll`
			                    AssemblyName=`System.Data, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=x86`
			                    MinFrameworkVersion=`131072`
		                    />
		                    <AssemblyReference
			                    RelativePath=`System.XML.dll`
			                    AssemblyName=`System.Xml, Version=2.0.0.0, PublicKeyToken=b77a5c561934e089, processorArchitecture=MSIL`
			                    MinFrameworkVersion=`131072`
		                    />
	                    </References>
	                    <Files>
		                    <Filter
			                    Name=`Source Files`
			                    Filter=`cpp;c;cc;cxx;def;odl;idl;hpj;bat;asm;asmx`
			                    UniqueIdentifier=`{4FC737F1-C7A5-4376-A066-2A32D752A2FF}`
			                    >
			                    <File
				                    RelativePath=`.\AssemblyInfo.cpp`
				                    >
				                    <FileConfiguration
					                    Name=`Debug|Win32`
					                    >
					                    <Tool
						                    Name=`VCCLCompilerTool`
						                    ExceptionHandling=`0`
						                    CompileAsManaged=`1`
					                    />
				                    </FileConfiguration>
			                    </File>
			                    <File
				                    RelativePath=`.\ClassLib.cpp`
				                    >
				                    <FileConfiguration
					                    Name=`Debug|Win32`
					                    >
					                    <Tool
						                    Name=`VCCLCompilerTool`
						                    ExceptionHandling=`0`
						                    CompileAsManaged=`1`
					                    />
				                    </FileConfiguration>
			                    </File>
			                    <File
				                    RelativePath=`.\Stdafx.cpp`
				                    >
				                    <FileConfiguration
					                    Name=`Debug|Win32`
					                    >
					                    <Tool
						                    Name=`VCCLCompilerTool`
						                    ExceptionHandling=`0`
						                    UsePrecompiledHeader=`1`
						                    CompileAsManaged=`1`
					                    />
				                    </FileConfiguration>
				                    <FileConfiguration
					                    Name=`Release|Win32`
					                    >
					                    <Tool
						                    Name=`VCCLCompilerTool`
						                    UsePrecompiledHeader=`1`
					                    />
				                    </FileConfiguration>
			                    </File>
		                    </Filter>
		                    <Filter
			                    Name=`Header Files`
			                    Filter=`h;hpp;hxx;hm;inl;inc;xsd`
			                    UniqueIdentifier=`{93995380-89BD-4b04-88EB-625FBE52EBFB}`
			                    >
			                    <File
				                    RelativePath=`.\ClassLib.h`
				                    >
			                    </File>
			                    <File
				                    RelativePath=`.\resource.h`
				                    >
			                    </File>
			                    <File
				                    RelativePath=`.\Stdafx.h`
				                    >
			                    </File>
		                    </Filter>
		                    <Filter
			                    Name=`Resource Files`
			                    Filter=`rc;ico;cur;bmp;dlg;rc2;rct;bin;rgs;gif;jpg;jpeg;jpe;resx;tiff;tif;png;wav`
			                    UniqueIdentifier=`{67DA6AB6-F800-4c08-8B7A-83BB121AAD01}`
			                    >
			                    <File
				                    RelativePath=`.\app.ico`
				                    >
			                    </File>
			                    <File
				                    RelativePath=`.\app.rc`
				                    >
			                    </File>
		                    </Filter>
		                    <File
			                    RelativePath=`.\ReadMe.txt`
			                    >
		                    </File>
	                    </Files>
	                    <Globals>
	                    </Globals>
                    </VisualStudioProject>";

#endregion

            // The project has its main ManagedExtensions property set to No, but the Debug configuration overrides it
            // for individual files.
            VerifyVCProjectOutputs(vcProjectContents, false, "Debug|Win32",
                @"Debug\ClassLib.dll", @"Debug\ClassLib.lib");

            // The Release configuration does not override the ManagedExtensions property.
            VerifyVCProjectOutputs(vcProjectContents, false, "Release|Win32",
                null, @"Release\ClassLib.lib");
        }
    }
}
*/
