/*
 * Assistant APIs for VMEM and relative modules
 *
 * Author:
 *	Naixuan Guan	<guannaixuan@linux.alibaba.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 */

#include <linux/types.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/memblock.h>
#include <asm/vmem_assist.h>

struct vmem_rsv_range {
	uint64_t start;
	uint64_t end;
	struct list_head next;
};

static unsigned long sys_max_pfn;
static int vmem_rsv_ranges;
static LIST_HEAD(vmem_rsv_range_list);

static int vmem_assist_add_merge_reserve_range(uint64_t start,
	uint64_t len);

static int vmem_assist_parse_memmap_one(char *p)
{
	char *oldp;
	u64 start_at, mem_size;

	if (!p)
		return -EINVAL;

	oldp = p;
	mem_size = memparse(p, &p);
	if (p == oldp)
		return -EINVAL;

	if (*p != '$')
		return -EINVAL;

	start_at = memparse(p+1, &p);

	return vmem_assist_add_merge_reserve_range(start_at, mem_size);
}

static int vmem_assist_parse_memmap_intern(char *str)
{
	int ret = 0;

	while (str) {
		char *k = strchr(str, ',');

		while (k && (*k == ','))
			k++;

		ret = vmem_assist_parse_memmap_one(str);
		if (ret)
			break;

		str = k;
	}

	return ret;
}

static int vmem_assist_dup_sub_str(char *str, char **dup)
{
	char *p, *buf;
	int len;

	if (!str)
		return -EINVAL;

	p = str;
	while ((*p != ' ') && (*p != '\0'))
		p++;

	len = p - str;
	if (!len)
		return -EINVAL;

	buf = kmalloc(len + 1, GFP_KERNEL);
	if (!buf)
		return -ENOMEM;

	strncpy(buf, str, len);
	buf[len] = '\0';
	if (dup)
		*dup = buf;

	return 0;
}

static int vmem_assist_parse_memmap_opt(char *str)
{
	int ret = 0;
	char *p;
	char *buf;
	int len = strlen("memmap=");

	while ((p = strstr(str, "memmap=")) != NULL) {
		p += len;
		ret = vmem_assist_dup_sub_str(p, &buf);
		if (ret)
			break;

		ret = vmem_assist_parse_memmap_intern(buf);
		kfree(buf);
		if (ret)
			break;

		str = p;
	}

	return ret;
}

static int vmem_assist_parse_mem_intern(char *p)
{
	u64 mem_size;

	mem_size = memparse(p, &p);
	if (mem_size == 0)
		return -EINVAL;

	sys_max_pfn = mem_size >> PAGE_SHIFT;

	return 0;
}

static int vmem_assist_parse_mem_opt(char *str)
{
	int ret = 0;
	char *p;
	char *buf;
	int len = strlen("mem=");

	p = strstr(str, "mem=");
	if (p != NULL) {
		p += len;
		ret = vmem_assist_dup_sub_str(p, &buf);
		if (ret)
			goto end;

		ret = vmem_assist_parse_mem_intern(buf);
		kfree(buf);
		str = p;
	}

end:
	return ret;
}

static int vmem_assist_parse_cmdline(void)
{
	char *cmdline;
	int ret = 0;

	cmdline = kstrdup(saved_command_line, GFP_KERNEL);
	if (!cmdline) {
		ret = -ENOMEM;
		goto end;
	}

	ret = vmem_assist_parse_memmap_opt(cmdline);
	if (ret)
		goto parse_fail;

	ret = vmem_assist_parse_mem_opt(cmdline);
	if (ret)
		goto parse_fail;

	if (e820_max_pfn > sys_max_pfn)
		ret = vmem_assist_add_merge_reserve_range(
			sys_max_pfn << PAGE_SHIFT,
			(e820_max_pfn - sys_max_pfn) << PAGE_SHIFT);

parse_fail:
	kfree(cmdline);

end:
	return ret;
}

static int vmem_assist_add_reserve_range(uint64_t start, uint64_t len)
{
	struct vmem_rsv_range *range;

	if (start == 0 || len == 0) {
		pr_warn("vmem_assist: Invalid reserve start %#llx len %#llx\n",
				start, len);
		return -EINVAL;
	}

	range = kzalloc(sizeof(*range), GFP_KERNEL);
	if (!range)
		return -ENOMEM;

	range->start = start;
	range->end = start + len;
	list_add(&range->next, &vmem_rsv_range_list);
	vmem_rsv_ranges++;

	return 0;
}

static void vmem_assist_del_reserve_range(struct vmem_rsv_range *range)
{
	if (!range)
		return;

	list_del(&range->next);
	kfree(range);
	vmem_rsv_ranges--;
}

static int vmem_assist_add_merge_reserve_range(uint64_t start,
	uint64_t len)
{
	struct vmem_rsv_range *range, *tmp;
	uint64_t end;
	bool overlap;

	end = start + len;
	list_for_each_entry_safe(range, tmp, &vmem_rsv_range_list, next) {
		overlap = false;

		/* extend the orig range if overlapped */
		if (start < range->start && end > range->end) {
			overlap = true;
		} else if (start >= range->start && start < range->end) {
			if (end > range->end)
				start = range->start;
			overlap = true;
		} else if (end > range->start && end <= range->end) {
			if (start < range->start)
				end = range->end;
			overlap = true;
		}

		if (overlap)
			vmem_assist_del_reserve_range(range);
	}

	return vmem_assist_add_reserve_range(start, end - start);
}

static void vmem_assist_dump_reserve_range(void)
{
	struct vmem_rsv_range *range;

	pr_info("vmem_assist: Ranges cnt %d\n", vmem_rsv_ranges);
	list_for_each_entry(range, &vmem_rsv_range_list, next) {
		pr_info("vmem_assist: Range start %#llx end %#llx\n",
			range->start, range->end);
	}
}

static int __init vmem_assist_init_reserve_range(void)
{
	if (vmem_assist_parse_cmdline())
		return -EINVAL;

	vmem_assist_dump_reserve_range();

	return 0;
}
late_initcall(vmem_assist_init_reserve_range);

/*
 * The caller should make sure the pfn belongs [s,e)
 */
static bool vmem_assist_is_borrowed(uint64_t pa, uint64_t s, uint64_t e)
{
	uint64_t sz = 1UL << SECTION_SIZE_BITS;
	uint64_t sec_start, sec_end;

	sec_start =  pa < sz ? 0 : ALIGN_DOWN(pa, sz);
	sec_end = ALIGN(pa + 1, sz);

	if (sec_start >= s && sec_end <= e &&
		pfn_valid(sec_start >> PAGE_SHIFT))
		return true;

	return false;
}

/*
 * Check whether a pfn is pass to vmem from cmdline
 */
bool vmem_assist_is_cmd_reserved_pfn(unsigned long pfn)
{
	struct vmem_rsv_range *range;
	uint64_t pa = pfn << PAGE_SHIFT;

	list_for_each_entry(range, &vmem_rsv_range_list, next) {
		if (pa >= range->start && pa < range->end)
			return true;
	}

	return false;
}
EXPORT_SYMBOL(vmem_assist_is_cmd_reserved_pfn);

/*
 * Check whether a pfn is reserved or managed by vmem
 */
bool vmem_assist_is_reserved_pfn(unsigned long pfn)
{
	struct vmem_rsv_range *range;
	uint64_t pa = pfn << PAGE_SHIFT;

	list_for_each_entry(range, &vmem_rsv_range_list, next) {
		if (pa >= range->start && pa < range->end &&
			!vmem_assist_is_borrowed(pa, range->start, range->end))
			return true;
	}

	return false;
}
EXPORT_SYMBOL(vmem_assist_is_reserved_pfn);
