// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (C) 2021 Alibaba Cloud. or its affiliates. All Rights Reserved.
 *
 * Since many features in dragonball require the support of drivers in guest
 * kernel, but they are not traditional device drivers in common sense, and
 * more similar to the features that require the coordination of the
 * underlying vmm.
 *
 * Therefore, we abstracted the dragonball platform driver in guest kernel,
 * and used it as a driver for the dragonball architecture, which is similar
 * to an enhanced x86-64/aarch64 architecture. In this way, other dragonball
 * functions that require the cooperation of guest can be added into this
 * driver as one of the features, and the driver will provides basic services
 * for them, such as feature probe, resource assign, etc.
 *
 * The dragonball virtual platform device based on a memory mapped area, and
 * it  may be instantiated in one of two equivalent ways:
 *
 * 1. Device Tree node, eg.:
 *
 *		dragonball@1e000 {
 *			compatible = "dragonball";
 *			reg = <0x1e000 0x100>;
 *			interrupts = <42>;
 *		}
 *
 * 2. Kernel module (or command line) parameter. Can be used more than once -
 *    one device will be created for each one. Syntax:
 *
 *		[dragonball.]device=<size>@<baseaddr>:<irq>
 *    where:
 *		<size>     := size (can use standard suffixes like K, M or G)
 *		<baseaddr> := physical base address
 *		<irq>      := interrupt number (as passed to request_irq())
 *    eg.:
 *		dragonball.device=0x100@0x100b0000:12
 *
 * Based on Virtio MMIO driver.
 */

#define pr_fmt(fmt) "dragonball: " fmt

#include <asm/setup.h>
#ifdef CONFIG_X86_64
#include <asm/cmdline.h>
#endif
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/msi.h>
#include <linux/platform_device.h>
#include <dragonball/dragonball.h>
#include <dragonball/platform_feature.h>
#ifdef CONFIG_ARM64
#include <linux/irqdomain.h>
#include <linux/irq.h>
#endif

#define DB_ADD_FEATURE(_feat, _name)                                           \
	[DB_FEAT_##_feat] = {                                                  \
		.name = "dragonball.no" #_name,                                \
		.val = (1 << DB_FEAT_##_feat),                                 \
	},

/*
 * The feature instance is mainly used to judge and set whether some features
 * in the guest kernel are enabled.
 *
 * @name: if this parameter exists in cmdline, the feature is turned off.
 * @val: the offset of this feature in the feature switch.
 */
static const struct {
	char *name;
	u64 val;
} features[] = { DRAGONBALL_FEATURES };
#undef DB_ADD_FEATURE

/*
 * Since some of the guest kernel features need to be confirmed at the
 * beginning of the kernel startup, dirvers have not yet started to load.
 * Therefore, early_param is used to match the cmdline parameter. If the
 * match is successful, the corresponding feature is turned off.
 */
#define MACRO_NOOP
#define DB_ADD_FEATURE(_feat, _name)                                           \
	static int __init dragonball_no##_name##_setup(char *str)              \
	{                                                                      \
		disable_dragonball_feature(DB_FEAT_##_feat);                   \
		return 0;                                                      \
	}                                                                      \
	early_param("dragonball.no" #_name, dragonball_no##_name##_setup);     \
	MACRO_NOOP
DRAGONBALL_FEATURES
#undef DB_ADD_FEATURE

/*
 * dragonball_priv is used to confirm the status of guest kernel related
 * features.
 *
 * @stable: set to true if the dragonball_platform driver is already loaded.
 * @state: used to confirm whether it is dragonball guest kernel.
 * @features: used to confirm the status of each feature in the guest kernel.
 */
static struct {
	bool stable;
	bool state;
	u64 features;
} dragonball_priv __read_mostly;

static int __init nodragonball_setup(char *str)
{
	disable_dragonball_platform();
	return 0;
}
early_param("nodragonball", nodragonball_setup);

/*
 * The function get_device_virq is used for getting the virq which is
 * corresponding to the pin. The relationship between pin and virq is saved
 * in struct irq_desc.
 *
 * For x86, the pin and virq are the same for legacy irq, and the irq_desc
 * is built during APIC initialization.
 *
 * For ARM, we need to create mapping between pin and virq when device is
 * hot plugged.
 */
#ifdef CONFIG_ARM64
/*
 * To build a new irq_desc we need to provide irq domain and irq.
 *
 * 1. irq domain: it contains the virq allocation routine, which is saved in
 * interrupt controller's device node. Besides the interrupt controller is
 * named "intc" in device tree which is passed from dragonball.
 *
 * 2. irq information: it contains 3 parameter {irq type, pin, irq trigger
 * type}. The irq type stands for SPI or PPI etc. Besides the SPI is
 * 0, and PPI is 1. The irq trigger type stands for edge trigger or level
 * trigger etc.
 */
uint32_t get_device_virq(uint32_t pin)
{
	uint32_t virq;
	struct device_node *node;
	struct irq_fwspec dummy_fwspec = {
		.param_count = 3,
		.param = {0, 0, IRQ_TYPE_EDGE_RISING}
	};

	node = of_find_node_by_name(NULL, "intc");
	if (!node) {
		pr_err("interrupt controller device node not found.\n");
		return 0;
	}
	dummy_fwspec.param[1] = pin;
	dummy_fwspec.fwnode = of_node_to_fwnode(node);
	virq = irq_create_fwspec_mapping(&dummy_fwspec);
	of_node_put(node);
	return virq;
}
#elif defined(CONFIG_X86_64)
uint32_t get_device_virq(uint32_t pin)
{
	return pin;
}
#endif

static inline char *get_cmdline(void)
{
	char *cmdline = saved_command_line;

	if (!cmdline)
		cmdline = boot_command_line;

	return cmdline;
}

#ifdef CONFIG_X86_64
inline bool arch_cmdline_find_option(char *cmdline, char *option)
{
	return cmdline_find_option_bool(cmdline, option);
}
#else
inline bool arch_cmdline_find_option(char *cmdline, char *option)
{
	return strstr(cmdline, option);
}
#endif

bool __ref is_dragonball_platform(void)
{
	if (unlikely(!dragonball_priv.stable)
		&& arch_cmdline_find_option(get_cmdline(), "nodragonball"))
		return false;

	return dragonball_priv.state;
}
EXPORT_SYMBOL_GPL(is_dragonball_platform);

void enable_dragonball_platform(void)
{
	dragonball_priv.state = true;
}
EXPORT_SYMBOL_GPL(enable_dragonball_platform);

void disable_dragonball_platform(void)
{
	dragonball_priv.state = false;
}
EXPORT_SYMBOL_GPL(disable_dragonball_platform);

bool __ref has_dragonball_feature(enum dragonball_features f)
{
	if (!is_dragonball_platform())
		return false;

	if (unlikely(!dragonball_priv.stable)
		&& arch_cmdline_find_option(get_cmdline(), features[f].name))
		return false;

	return !!(dragonball_priv.features & features[f].val);
}
EXPORT_SYMBOL_GPL(has_dragonball_feature);

void enable_dragonball_feature(enum dragonball_features f)
{
	dragonball_priv.features |= features[f].val;
}
EXPORT_SYMBOL_GPL(enable_dragonball_feature);

void disable_dragonball_feature(enum dragonball_features f)
{
	dragonball_priv.features &= ~features[f].val;
}
EXPORT_SYMBOL(disable_dragonball_feature);

static u64 dragonball_enabled_features(void)
{
	return dragonball_priv.features;
}

int dragonball_add_device(struct resource *resources, size_t res_size)
{
	struct platform_device *pdev;

	pr_info("Registering dragonball platform device at %pR, %pR.\n",
		&resources[0], &resources[1]);

	pdev = platform_device_register_resndata(NULL, "dragonball", 0,
						 resources, res_size, NULL, 0);

	return PTR_ERR_OR_ZERO(pdev);
}
EXPORT_SYMBOL(dragonball_add_device);

/*
 * Build dragonball_platform device according to cmdline parameters. In theory,
 * only x86 follows this process, and arm prefers to construct this device by fdt.
 */
static int db_cmdline_set(const char *device, const struct kernel_param *kp)
{
	struct resource resources[2] = {};
	char *str;
	resource_size_t base, size;
	unsigned int irq;
	int processed, consumed = 0;

	/* Consume "size" part of the command line parameter */
	size = memparse(device, &str);

	/* Get "@<base>:<irq>" chunks */
	processed = sscanf(str, "@%lli:%u%n", &base, &irq, &consumed);

	/*
	 * sscanf() must processes at least 2 chunks; also there
	 * must be no extra characters after the last chunk, so
	 * str[consumed] must be '\0'
	 */
	if (processed < 2 || str[consumed])
		return -EINVAL;

	resources[0].flags = IORESOURCE_MEM;
	resources[0].start = base;
	resources[0].end = base + size - 1;

	resources[1].flags = IORESOURCE_IRQ;
	resources[1].start = resources[1].end = irq;

	return dragonball_add_device(resources, ARRAY_SIZE(resources));
}

/*
 * postcore_param_cb is used to abstract the dragonball device according to the
 * cmdline, and the @device parameter will be spliced with the MODULE_PARAM_PREFIX
 * macro definition into a "dragonball.device" string, which is used by the
 * cmdline to identify the dragonball device.
 */
static const struct kernel_param_ops db_cmdline_param_ops = {
	.set = db_cmdline_set,
};

postcore_param_cb(device, &db_cmdline_param_ops, NULL, 0400);

static irqreturn_t interrupt_handler(int irq, void *opaque)
{
	struct dragonball_device *db_dev = opaque;

	return db_features_handle_interrupt(db_dev);
}

/*
 * When the driver enables or disables the msi, this function is called
 * to inform the device of the current state of the msi by writing the mmio
 * space.
 */
static void msi_write_msg(struct msi_desc *desc, struct msi_msg *msg)
{
	struct device *dev;
	struct dragonball_device *db_dev;

	dev = desc->dev;
	db_dev = (struct dragonball_device *)dev->driver_data;

	writel(msg->address_lo, db_dev->base + DB_DEVICE_MSI_ADDR_LO_OFF);
	writel(msg->address_hi, db_dev->base + DB_DEVICE_MSI_ADDR_HI_OFF);
	writel(msg->data, db_dev->base + DB_DEVICE_MSI_DATA_OFF);
	writel(desc->platform.msi_index, db_dev->base + DB_DEVICE_MSI_ARG_OFF);
	writel(DB_DEVICE_MSI_CMD_UPDATE, db_dev->base + DB_DEVICE_MSI_CMD_OFF);
}

static int setup_msi_irq(struct dragonball_device *db_dev)
{
	int err;
	u32 msi_irq_nr;
	struct msi_desc *entry;

	msi_irq_nr = readl(db_dev->base + DB_DEVICE_MSI_NR_OFF);
	if (!msi_irq_nr)
		return 0;

	if (!db_dev->pdev->dev.msi_domain) {
		db_dev->pdev->dev.msi_domain =
			platform_msi_get_def_irq_domain();
	}
	if (!db_dev->pdev->dev.msi_domain)
		return -ENOENT;

	err = platform_msi_domain_alloc_irqs(&db_dev->pdev->dev, msi_irq_nr,
					     msi_write_msg);
	if (err)
		return err;

	entry = first_msi_entry(&db_dev->pdev->dev);
	db_dev->base_irq = entry->irq;
	db_dev->next_irq = entry->irq;
	db_dev->max_irq = entry->irq + msi_irq_nr - 1;

	return 0;
}

static int dragonball_platform_probe(struct dragonball_device *db_dev)
{
	int err;
	u32 magic;
	u64 device_features;

	/* Check the magin value */
	magic = readl(db_dev->base + DB_PLATFORM_MAGIC_OFF);
	if (magic != ('D' | 'B' << 8 | 'P' << 16 | 'F' << 24)) {
		pr_err("wrong magic value %#x!\n", magic);
		return -ENODEV;
	}

	/* Send status [RESET] */
	writel(DB_PLATFORM_DRIVER_RESET, db_dev->base + DB_DEVICE_STATUS_OFF);

	/* Send status [ACK] */
	writel(DB_PLATFORM_DRIVER_ACK, db_dev->base + DB_DEVICE_STATUS_OFF);

	/* Send driver version */
	writel(DB_PLATFORM_DRIVER_VER, db_dev->base + DB_DRIVER_VER_OFF);

	/* Get and check device's version */
	db_dev->version = readl(db_dev->base + DB_DEVICE_VER_OFF);
	if (db_dev->version != 1) {
		pr_err("version %d is not supported!\n", db_dev->version);
		err = -ENXIO;
		goto error;
	}

	/* Send driver features */
	writel(1, db_dev->base + DB_DRIVER_FEAT_SEL_OFF);
	writel((u32)(db_dev->driver_features >> 32),
	       db_dev->base + DB_DRIVER_FEAT_OFF);
	writel(0, db_dev->base + DB_DRIVER_FEAT_SEL_OFF);
	writel((u32)db_dev->driver_features, db_dev->base + DB_DRIVER_FEAT_OFF);

	/* Get device's feature */
	writel(1, db_dev->base + DB_DEVICE_FEAT_SEL_OFF);
	device_features = readl(db_dev->base + DB_DEVICE_FEAT_OFF);
	device_features <<= 32;
	writel(0, db_dev->base + DB_DEVICE_FEAT_SEL_OFF);
	device_features |= readl(db_dev->base + DB_DEVICE_FEAT_OFF);

	/* Check device's feature */
	device_features &= db_dev->driver_features;
	db_dev->acked_features = device_features;

	/* Send status [FEATURE_OK] */
	writel(DB_PLATFORM_FEAT_OK, db_dev->base + DB_DEVICE_STATUS_OFF);

	/* Setup msi irq */
	err = setup_msi_irq(db_dev);
	if (err) {
		pr_err("setup msi irq failed: %d\n", err);
		goto error;
	}

	/* Setup features */
	err = probe_db_platform_features(db_dev);
	if (err) {
		pr_err("probe platform feature failed: %d\n", err);
		goto error;
	}

	/* Send status [DRIVER_OK] */
	writel(DB_PLATFORM_DRIVER_OK, db_dev->base + DB_DEVICE_STATUS_OFF);
	return 0;
error:
	/* Send status [DRIVER_FAIL] */
	writel(DB_PLATFORM_DRIVER_FAIL, db_dev->base + DB_DEVICE_STATUS_OFF);
	return err;
}

static int dragonball_probe(struct platform_device *pdev)
{
	struct dragonball_device *db_dev;
	struct resource *mem, *irq;
	int virq, ret;
	unsigned long flag = 0;

	dev_info(&pdev->dev, "dragonball platform driver probe");

	mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!mem)
		return -EINVAL;

	irq = platform_get_resource(pdev, IORESOURCE_IRQ, 0);
	if (!irq)
		return -EINVAL;

	if (irq->start == SHARED_IRQ_NO)
		flag |= IRQF_SHARED;

	if (!devm_request_mem_region(&pdev->dev, mem->start, resource_size(mem),
				     pdev->name))
		return -EBUSY;

	db_dev = devm_kzalloc(&pdev->dev, sizeof(*db_dev), GFP_KERNEL);
	if (!db_dev)
		return -ENOMEM;

	db_dev->pdev = pdev;
	db_dev->driver_features = dragonball_enabled_features();
	db_dev->base = devm_ioremap(&pdev->dev, mem->start, resource_size(mem));
	if (db_dev->base == NULL) {
		ret = -EFAULT;
		goto err_alloc_db_dev;
	}

	/* The pin number of arm and virq number are not equal, and virq needs to apply to gic. */
	virq = get_device_virq(irq->start);
	ret = devm_request_irq(&pdev->dev, virq, interrupt_handler, flag,
			       dev_name(&pdev->dev), db_dev);
	if (ret) {
		dev_err(&pdev->dev, "request irq failed with %d\n", ret);
		goto err_ioremap;
	}

	platform_set_drvdata(pdev, db_dev);

	ret = dragonball_platform_probe(db_dev);
	if (ret) {
		dev_err(&pdev->dev, "probe failed with %d\n", ret);
		goto err_dev_irq;
	}

	return 0;

err_dev_irq:
	devm_free_irq(&pdev->dev, irq->start, db_dev);
err_ioremap:
	devm_iounmap(&pdev->dev, db_dev->base);
err_alloc_db_dev:
	devm_kfree(&pdev->dev, db_dev);
	return ret;
}

/*
 * MODULE_DEVICE_TABLE can generate device_table for mod-related scripts.
 * For details, please refer to the ./scripts/mod/file2alias.c file.
 */
static const struct of_device_id dragonball_match[] = {
	{
		.compatible = "dragonball",
	},
	{},
};
MODULE_DEVICE_TABLE(of, dragonball_match);

/*
 * The builtin_platform_driver function registers the dragonball
 * platform driver to the kernel.
 */
static struct platform_driver dragonball_platform_driver = {
	.driver = {
		.name = "dragonball",
		.suppress_bind_attrs = true,
		.of_match_table = of_match_ptr(dragonball_match),
	},
	.probe = dragonball_probe,
};
builtin_platform_driver(dragonball_platform_driver);

/*
 * When the dragonball platform driver is successfully registered,
 * set dragonball_priv.stable to true.
 */
static int __init dragonball_init(void)
{
	dragonball_priv.stable = true;

	return 0;
}
core_initcall(dragonball_init);
